(*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *)

open Type
open Flow_js_utils

let rec default_resolve_touts ~flow ?resolve_callee cx loc u =
  let _TODO = () in
  let any = AnyT.at (AnyError None) loc in
  let resolve t = flow (any, t) in
  let resolve_tvar t = flow (any, OpenT t) in
  let map_opt f t = Base.Option.iter ~f t in
  let resolve_specialized_callee x =
    let resolve_callee =
      match resolve_callee with
      | None -> any
      | Some (r, []) -> DefT (r, MixedT Mixed_everything)
      | Some (_, [t]) -> t
      | Some (r, t1 :: t2 :: ts) -> IntersectionT (r, InterRep.make t1 t2 ts)
    in
    CalleeRecorder.add_callee cx CalleeRecorder.All resolve_callee x
  in
  let resolve_method_action action =
    match action with
    | ChainM
        {
          exp_reason = _;
          lhs_reason = _;
          methodcalltype = { meth_tout; _ };
          voided_out_collector = _;
          return_hint = _;
          specialized_callee;
        } ->
      resolve_tvar meth_tout;
      resolve_specialized_callee specialized_callee
    | CallM { methodcalltype = { meth_tout; _ }; return_hint = _; specialized_callee } ->
      resolve_tvar meth_tout;
      resolve_specialized_callee specialized_callee
    | NoMethodAction tout -> resolve tout
  in
  let resolve_lookup_action action =
    match action with
    | ReadProp { tout; _ } -> resolve_tvar tout
    | WriteProp { prop_tout; _ } -> map_opt resolve prop_tout
    | LookupPropForTvarPopulation _
    | LookupPropForSubtyping _
    | SuperProp _
    | MatchProp _ ->
      ()
  in
  let resolve_elem_action action =
    match action with
    | ReadElem { tout; _ } -> resolve_tvar tout
    | WriteElem { tout; _ } -> map_opt resolve tout
    | CallElem (_, action) -> resolve_method_action action
  in
  let resolve_react_tool tool =
    let open React in
    match tool with
    | CreateElement { tout; _ }
    | ConfigCheck { props = tout }
    | GetConfig { tout } ->
      resolve tout
  in
  let resolve_spread_resolve resolve_tool =
    match resolve_tool with
    | ResolveSpreadsToTupleType { tout = t; _ }
    | ResolveSpreadsToArrayLiteral { tout = t; _ }
    | ResolveSpreadsToArray (_, t) ->
      resolve t
    | ResolveSpreadsToMultiflowCallFull _ -> _TODO
    | ResolveSpreadsToMultiflowPartial (_, _, _, t) -> resolve t
    | ResolveSpreadsToMultiflowSubtypeFull _ -> _TODO
  in
  let resolve_cont cont =
    match cont with
    | Upper use -> default_resolve_touts ~flow cx loc use
    | Lower _ -> _TODO
  in
  match u with
  | UseT _ -> _TODO
  | BindT (_, _, { call_tout; _ }) -> resolve_tvar call_tout
  | CallT
      {
        use_op = _;
        reason = _;
        call_action = Funcalltype { call_tout; call_specialized_callee; _ };
        return_hint = _;
      } ->
    resolve_tvar call_tout;
    resolve_specialized_callee call_specialized_callee
  | CallT { use_op = _; reason = _; call_action = ConcretizeCallee tout; return_hint = _ } ->
    resolve_tvar tout
  | ConditionalT { tout; _ } -> resolve_tvar tout
  | MethodT (_, _, _, _, action)
  | PrivateMethodT (_, _, _, _, _, _, action) ->
    resolve_method_action action
  | SetPropT (_, _, _, _, _, _, topt)
  | SetPrivatePropT (_, _, _, _, _, _, _, _, topt) ->
    map_opt resolve topt
  | GetTypeFromNamespaceT { tout = tvar; _ }
  | GetPrivatePropT (_, _, _, _, _, tvar)
  | GetPropT
      {
        use_op = _;
        reason = _;
        id = _;
        from_annot = _;
        skip_optional = _;
        propref = _;
        tout = tvar;
        hint = _;
      }
  | TestPropT { use_op = _; reason = _; id = _; propref = _; tout = tvar; hint = _ } ->
    resolve_tvar tvar
  | SetElemT (_, _, _, _, _, topt) -> map_opt resolve topt
  | GetElemT { tout; _ } -> resolve_tvar tout
  | CallElemT (_, _, _, _, action) -> resolve_method_action action
  | GetStaticsT tvar
  | GetProtoT (_, tvar) ->
    resolve_tvar tvar
  | SetProtoT _ -> ()
  | ReposLowerT _
  | ReposUseT _ ->
    _TODO
  | ConstructorT
      { use_op = _; reason = _; targs = _; args = _; tout; return_hint = _; specialized_ctor = _ }
    ->
    resolve tout
  | SuperT _ -> ()
  | ImplementsT _ -> ()
  | MixinT (_, t) -> resolve t
  | ToStringT { t_out; _ } -> default_resolve_touts ~flow cx loc t_out
  | SpecializeT (_, _, _, _, tout) -> resolve tout
  | ThisSpecializeT (_, _, k) -> resolve_cont k
  | ValueToTypeReferenceT (_, _, _, t) -> resolve t
  | ConcretizeTypeAppsT _ -> _TODO
  | LookupT { lookup_action; _ } -> resolve_lookup_action lookup_action
  | ObjRestT (_, _, t, _)
  | ObjTestProtoT (_, t)
  | ObjTestT (_, _, t)
  | ArrRestT (_, _, _, t) ->
    resolve t
  | GetDictValuesT (_, use) -> default_resolve_touts ~flow cx loc use
  | GetKeysT (_, use) -> default_resolve_touts ~flow cx loc use
  | HasOwnPropT _ -> ()
  | GetValuesT (_, t) -> resolve t
  | ElemT { action; _ } -> resolve_elem_action action
  | MapTypeT (_, _, _, t)
  | ObjKitT (_, _, _, _, t) ->
    resolve t
  | ReactKitT (_, _, tool) -> resolve_react_tool tool
  | ConcretizeT _ -> _TODO
  | ResolveSpreadT (_, _, { rrt_resolve_to; _ }) -> resolve_spread_resolve rrt_resolve_to
  | CondT (_, _, _, t) -> resolve t
  | ExtendsUseT _ -> ()
  | DestructuringT (_, _, _, tvar, _) -> resolve_tvar tvar
  | ResolveUnionT { upper; _ } -> default_resolve_touts ~flow cx loc upper
  | TypeCastT (_, t) -> resolve t
  | GetEnumT { tout; _ } -> resolve tout
  | EnumCastT _
  | EnumExhaustiveCheckT _ ->
    ()
  | HooklikeT tvar
  | DeepReadOnlyT (tvar, _) ->
    resolve_tvar tvar
  | FilterOptionalT (_, t)
  | FilterMaybeT (_, t) ->
    resolve t
  | SealGenericT { cont; _ } -> resolve_cont cont
  | OptionalIndexedAccessT { tout_tvar; _ } -> resolve_tvar tout_tvar
  | CheckUnusedPromiseT _ -> _TODO
  | WriteComputedObjPropCheckT _ -> ()
  | ConvertEmptyPropsToMixedT (_, tout) -> resolve tout
  | ExitRendersT { renders_reason = _; u } -> default_resolve_touts ~flow cx loc u
  | EvalTypeDestructorT _ -> ()
