(*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *)

type module_system =
  | Node
  | Haste

type supported_os = CentOS

type jsx_mode =
  (* JSX desugars into a `React.createElement(name, props, ...children)` call *)
  | Jsx_react
  (*
   * Specifies a function that should be invoked instead of React.createElement
   * when interpreting JSX syntax. Otherwise, the usual rules of JSX are
   * followed: children are varargs after a props argument.
   *)
  | Jsx_pragma of (string * (ALoc.t, ALoc.t) Flow_ast.Expression.t)

type saved_state_fetcher =
  | Dummy_fetcher
  | Local_fetcher
  | Scm_fetcher
  | Fb_fetcher

module ReactRefAsProp = struct
  type t =
    | Legacy  (** Only in component syntax: ban ref prop in spread *)
    | FullSupport  (** Implement full React 19 behavior *)
end

type react_runtime =
  | ReactRuntimeAutomatic
  | ReactRuntimeClassic

type react_rules =
  | ValidateRefAccessDuringRender
  | DeepReadOnlyProps
  | DeepReadOnlyHookReturns
  | RulesOfHooks

module AssertOperator = struct
  type t =
    | Enabled
    | Specialized
    | Disabled
    | Unparsed

  let parse = function
    | Unparsed -> false
    | Enabled
    | Specialized
    | Disabled ->
      true

  let usable = function
    | Unparsed
    | Disabled ->
      false
    | Enabled
    | Specialized ->
      true

  let specialized = function
    | Specialized -> true
    | Unparsed
    | Disabled
    | Enabled ->
      false
end

type format = {
  opt_bracket_spacing: bool;
  opt_single_quotes: bool;
}

type gc_control = {
  gc_minor_heap_size: int option;
  gc_major_heap_increment: int option;
  gc_space_overhead: int option;
  gc_window_size: int option;
  gc_custom_major_ratio: int option;
  gc_custom_minor_ratio: int option;
  gc_custom_minor_max_size: int option;
}

type log_saving = {
  threshold_time_ms: int;
  limit: int option;
  rate: float;
}

module CastingSyntax = struct
  type t =
    | As
    | Both
end

type t = {
  opt_all: bool;
  opt_assert_operator: AssertOperator.t;
  opt_autoimports: bool;
  opt_autoimports_min_characters: int;
  opt_autoimports_ranked_by_usage: bool;
  opt_autoimports_ranked_by_usage_boost_exact_match_min_length: int;
  opt_automatic_require_default: bool;
  opt_babel_loose_array_spread: bool;
  opt_ban_spread_key_props: bool;
  opt_casting_syntax: CastingSyntax.t;
  opt_casting_syntax_only_support_as_excludes: Str.regexp list;
  opt_channel_mode: [ `pipe | `socket ];
  opt_component_syntax: bool;
  opt_debug: bool;
  opt_deprecated_utilities: string list SMap.t;
  opt_deprecated_utilities_excludes: Str.regexp list;
  opt_dev_only_refinement_info_as_errors: bool;
  opt_distributed: bool;
  opt_enable_const_params: bool;
  opt_enable_jest_integration: bool;
  opt_enable_pattern_matching: bool;
  opt_enable_pattern_matching_instance_patterns: bool;
  opt_enable_records: bool;
  opt_enable_relay_integration: bool;
  opt_enable_custom_error: bool;
  opt_enabled_rollouts: string SMap.t;
  opt_enums: bool;
  opt_estimate_recheck_time: bool;
  opt_exact_by_default: bool;
  opt_facebook_fbs: string option;
  opt_facebook_fbt: string option;
  opt_facebook_module_interop: bool;
  opt_file_options: Files.options;
  opt_flowconfig_hash: string;
  opt_flowconfig_name: string;
  opt_format: format;
  opt_gc_worker: gc_control;
  opt_haste_module_ref_prefix: string option;
  opt_hook_compatibility: bool;
  opt_hook_compatibility_excludes: Str.regexp list;
  opt_hook_compatibility_includes: Str.regexp list;
  opt_ignore_non_literal_requires: bool;
  opt_include_suppressions: bool;
  opt_include_warnings: bool;
  opt_instance_t_objkit_fix: bool;
  opt_lazy_mode: bool;
  opt_lint_severities: Severity.severity LintSettings.t;
  opt_log_file: File_path.t;
  opt_log_saving: log_saving SMap.t;
  opt_long_lived_workers: bool;
  opt_max_files_checked_per_worker: int;
  opt_max_header_tokens: int;
  opt_max_seconds_for_check_per_worker: float;
  opt_max_workers: int;
  opt_merge_timeout: float option;
  opt_missing_module_generators: (Str.regexp * string) list;
  opt_module: module_system;
  opt_module_name_mappers: (Str.regexp * string) list;
  opt_modules_are_use_strict: bool;
  opt_munge_underscores: bool;
  opt_no_unchecked_indexed_access: bool;
  opt_node_main_fields: string list;
  opt_node_package_export_conditions: string list;
  opt_node_resolver_allow_root_relative: bool;
  opt_node_resolver_root_relative_dirnames: (string option * string) list;
  opt_opaque_type_new_bound_syntax: bool;
  opt_profile: bool;
  opt_records_includes: string list;
  opt_projects_options: Flow_projects.options;
  opt_quiet: bool;
  opt_react_custom_jsx_typing: bool;
  opt_react_ref_as_prop: ReactRefAsProp.t;
  opt_react_rules: react_rules list;
  opt_react_runtime: react_runtime;
  opt_recursion_limit: int;
  opt_relay_integration_esmodules: bool;
  opt_relay_integration_excludes: Str.regexp list;
  opt_relay_integration_module_prefix: string option;
  opt_relay_integration_module_prefix_includes: Str.regexp list;
  opt_root: File_path.t;
  opt_root_name: string option;
  opt_saved_state_fetcher: saved_state_fetcher;
  opt_saved_state_force_recheck: bool;
  opt_saved_state_no_fallback: bool;
  opt_saved_state_skip_version_check: bool;
  opt_saved_state_verify: bool;
  opt_slow_to_check_logging: Slow_to_check_logging.t;
  opt_strict_es6_import_export: bool;
  opt_strict_mode: StrictModeSettings.t;
  opt_strip_root: bool;
  opt_supported_operating_systems: supported_os list;
  opt_temp_dir: string;
  opt_ts_syntax: bool;
  opt_ts_utility_syntax: bool;
  opt_type_expansion_recursion_limit: int;
  opt_unsuppressable_error_codes: SSet.t;
  opt_use_unknown_in_catch_variables: bool;
  opt_verbose: Verbose.t option;
  opt_vpn_less: bool;
  opt_wait_for_recheck: bool;
}

let all opts = opts.opt_all

let assert_operator opts = opts.opt_assert_operator

let autoimports opts = opts.opt_autoimports

let autoimports_min_characters opts = opts.opt_autoimports_min_characters

let autoimports_ranked_by_usage opts = opts.opt_autoimports_ranked_by_usage

let autoimports_ranked_by_usage_boost_exact_match_min_length opts =
  opts.opt_autoimports_ranked_by_usage_boost_exact_match_min_length

let automatic_require_default opts = opts.opt_automatic_require_default

let babel_loose_array_spread opts = opts.opt_babel_loose_array_spread

let ban_spread_key_props opts = opts.opt_ban_spread_key_props

let casting_syntax opts = opts.opt_casting_syntax

let casting_syntax_only_support_as_excludes opts = opts.opt_casting_syntax_only_support_as_excludes

let channel_mode opts = opts.opt_channel_mode

let component_syntax opts = opts.opt_component_syntax

let deprecated_utilities opts = opts.opt_deprecated_utilities

let deprecated_utilities_excludes opts = opts.opt_deprecated_utilities_excludes

let dev_only_refinement_info_as_errors opts = opts.opt_dev_only_refinement_info_as_errors

let distributed opts = opts.opt_distributed

let enable_const_params opts = opts.opt_enable_const_params

let enable_jest_integration opts = opts.opt_enable_jest_integration

let enable_pattern_matching opts = opts.opt_enable_pattern_matching

let enable_pattern_matching_instance_patterns opts =
  opts.opt_enable_pattern_matching_instance_patterns

let enable_records opts = opts.opt_enable_records

let enable_relay_integration opts = opts.opt_enable_relay_integration

let enable_custom_error opts = opts.opt_enable_custom_error

let enabled_rollouts opts = opts.opt_enabled_rollouts

let enums opts = opts.opt_enums

let estimate_recheck_time opts = opts.opt_estimate_recheck_time

let exact_by_default opts = opts.opt_exact_by_default

let facebook_fbs opts = opts.opt_facebook_fbs

let facebook_fbt opts = opts.opt_facebook_fbt

let facebook_module_interop opts = opts.opt_facebook_module_interop

let file_options opts = opts.opt_file_options

let flowconfig_hash opts = opts.opt_flowconfig_hash

let flowconfig_name opts = opts.opt_flowconfig_name

let format_bracket_spacing opts = opts.opt_format.opt_bracket_spacing

let format_single_quotes opts = opts.opt_format.opt_single_quotes

let gc_worker opts = opts.opt_gc_worker

let haste_module_ref_prefix opts = opts.opt_haste_module_ref_prefix

let hook_compatibility opts = opts.opt_hook_compatibility

let hook_compatibility_excludes opts = opts.opt_hook_compatibility_excludes

let hook_compatibility_includes opts = opts.opt_hook_compatibility_includes

let hook_compatibility_in_file opts file =
  let enabled = hook_compatibility opts in
  let included =
    match hook_compatibility_includes opts with
    | [] -> false
    | dirs ->
      let filename = File_key.to_string file in
      let normalized_filename = Sys_utils.normalize_filename_dir_sep filename in
      List.exists (fun r -> Str.string_match r normalized_filename 0) dirs
  in
  let excluded =
    match hook_compatibility_excludes opts with
    | [] -> false
    | dirs ->
      let filename = File_key.to_string file in
      let normalized_filename = Sys_utils.normalize_filename_dir_sep filename in
      List.exists (fun r -> Str.string_match r normalized_filename 0) dirs
  in
  included || (enabled && not excluded)

let include_suppressions opts = opts.opt_include_suppressions

let instance_t_objkit_fix opts = opts.opt_instance_t_objkit_fix

let is_debug_mode opts = opts.opt_debug

let is_quiet opts = opts.opt_quiet

let lazy_mode opts = opts.opt_lazy_mode

let lint_severities opts = opts.opt_lint_severities

let log_file opts = opts.opt_log_file

let log_saving opts = opts.opt_log_saving

let long_lived_workers opts = opts.opt_long_lived_workers

let max_files_checked_per_worker opts = opts.opt_max_files_checked_per_worker

let max_header_tokens opts = opts.opt_max_header_tokens

let max_seconds_for_check_per_worker opts = opts.opt_max_seconds_for_check_per_worker

let max_workers opts = opts.opt_max_workers

let merge_timeout opts = opts.opt_merge_timeout

let missing_module_generators opts = opts.opt_missing_module_generators

let module_name_mappers opts = opts.opt_module_name_mappers

let module_system opts = opts.opt_module

let modules_are_use_strict opts = opts.opt_modules_are_use_strict

let no_unchecked_indexed_access opts = opts.opt_no_unchecked_indexed_access

let node_main_fields opts = opts.opt_node_main_fields

let node_package_export_conditions opts = opts.opt_node_package_export_conditions

let node_resolver_allow_root_relative opts = opts.opt_node_resolver_allow_root_relative

let node_resolver_root_relative_dirnames opts = opts.opt_node_resolver_root_relative_dirnames

let opaque_type_new_bound_syntax opts = opts.opt_opaque_type_new_bound_syntax

let projects_options opts = opts.opt_projects_options

let records_includes opts = opts.opt_records_includes

let react_custom_jsx_typing opts = opts.opt_react_custom_jsx_typing

let react_ref_as_prop opts = opts.opt_react_ref_as_prop

let react_rules opts = opts.opt_react_rules

let react_runtime opts = opts.opt_react_runtime

let recursion_limit opts = opts.opt_recursion_limit

let relay_integration_esmodules opts = opts.opt_relay_integration_esmodules

let relay_integration_excludes opts = opts.opt_relay_integration_excludes

let relay_integration_module_prefix opts = opts.opt_relay_integration_module_prefix

let relay_integration_module_prefix_includes opts =
  opts.opt_relay_integration_module_prefix_includes

let root opts = opts.opt_root

let root_name opts = opts.opt_root_name

let saved_state_fetcher opts = opts.opt_saved_state_fetcher

let saved_state_force_recheck opts = opts.opt_saved_state_force_recheck

let saved_state_no_fallback opts = opts.opt_saved_state_no_fallback

let saved_state_skip_version_check opts = opts.opt_saved_state_skip_version_check

let saved_state_verify opts = opts.opt_saved_state_verify

let should_ignore_non_literal_requires opts = opts.opt_ignore_non_literal_requires

let should_include_warnings opts = opts.opt_include_warnings

let should_munge_underscores opts = opts.opt_munge_underscores

let should_profile opts = opts.opt_profile && not opts.opt_quiet

let should_strip_root opts = opts.opt_strip_root

let slow_to_check_logging opts = opts.opt_slow_to_check_logging

let strict_es6_import_export opts = opts.opt_strict_es6_import_export

let strict_mode opts = opts.opt_strict_mode

let supported_operating_systems opts = opts.opt_supported_operating_systems

let temp_dir opts = opts.opt_temp_dir

let ts_syntax opts = opts.opt_ts_syntax

let ts_utility_syntax opts = opts.opt_ts_utility_syntax

let type_expansion_recursion_limit opts = opts.opt_type_expansion_recursion_limit

let typecheck_component_syntax_in_file opts file =
  component_syntax opts || File_key.is_lib_file file

let unsuppressable_error_codes opts = opts.opt_unsuppressable_error_codes

let use_unknown_in_catch_variables opts = opts.opt_use_unknown_in_catch_variables

let verbose opts = opts.opt_verbose

let vpn_less opts = opts.opt_vpn_less

let wait_for_recheck opts = opts.opt_wait_for_recheck
