/*
 * @Date: 2024-01-31 15:00:11
 * @LastEditors: maggieyyy
 * @LastEditTime: 2024-06-04 10:49:35
 * @FilePath: \frontend\packages\common\src\components\postcat\api\MoreSetting\components\ParamLimit.tsx
 */
import { ChangeEvent, useEffect, useState } from 'react'
import { FormControl, TextField, Box } from '@mui/material'

interface ParamLimitProps {
  min: number | null
  max: number | null
  onChange: ({ min, max }: { min: number; max: number }) => void
  minLabel?: string
  maxLabel?: string
}

export function ParamLimit({ min, max, onChange, minLabel = 'Minimum', maxLabel = 'Maximum' }: ParamLimitProps) {
  const [minValue, setMinValue] = useState<number>(min ?? 0)
  const [maxValue, setMaxValue] = useState<number>(max ?? 0)
  const [error, setError] = useState<string | null>(null)

  const validate = (minVal: number, maxVal: number) => {
    if (isNaN(minVal) || minVal < 0) {
      return `The ${minLabel} must not be negative.`
    }

    if (isNaN(maxVal) || maxVal < 0) {
      return `The ${maxLabel} must not be negative.`
    }

    if (minVal > maxVal) {
      return `The ${maxLabel} must be greater than or equal to the ${minLabel}.`
    }

    return null
  }

  useEffect(() => {
    onChange?.({
      min: minValue,
      max: maxValue
    })
  }, [minValue, maxValue, onChange])

  const handleMinChange = (event: ChangeEvent<HTMLInputElement>) => {
    const newMinValue = parseFloat(event.target.value)
    setMinValue(newMinValue)
    setError(validate(newMinValue, maxValue))
  }

  const handleMaxChange = (event: ChangeEvent<HTMLInputElement>) => {
    const newMaxValue = parseFloat(event.target.value)
    setMaxValue(newMaxValue)
    setError(validate(minValue, newMaxValue))
  }

  return (
    <Box width="100%">
      <form>
        <FormControl error={!!error} variant="standard" fullWidth>
          <Box display="flex" width="100%" gap={2}>
            <Box width="100%">
              <TextField
                fullWidth
                label={minLabel}
                sx={{
                  '& .MuiOutlinedInput-notchedOutline': {
                    border: '1px solid #EDEDED'
                  },
                  '& .Mui-focused .MuiOutlinedInput-notchedOutline':{
                    border:'1px solid #7371fc'
                  },
                  '& .MuiInputLabel-root.Mui-focused':{
                    color:'#7371fc'
                  },
                  '& .MuiOutlinedInput-root:hover .MuiOutlinedInput-notchedOutline':{
                    borderColor:'#7371fc'
                    }
                }}
                value={minValue}
                onChange={handleMinChange}
                type="number"
                error={!!error && (isNaN(minValue) || minValue < 0)}
                helperText={!!error && (isNaN(minValue) || minValue < 0) ? error : ''}
              />
            </Box>
            <Box width="100%">
              <TextField
                fullWidth
                label={maxLabel}
                value={maxValue}
                sx={{
                  '& .MuiOutlinedInput-notchedOutline': {
                    border: '1px solid #EDEDED'
                  },
                  '& .Mui-focused .MuiOutlinedInput-notchedOutline':{
                    border:'1px solid #7371fc'
                  },
                  '& .MuiInputLabel-root.Mui-focused':{
                    color:'#7371fc'
                  },
                  '& .MuiOutlinedInput-root:hover .MuiOutlinedInput-notchedOutline':{
                    borderColor:'#7371fc'
                    }
                }}
                onChange={handleMaxChange}
                type="number"
                error={!!error && (isNaN(maxValue) || maxValue < 0 || maxValue < minValue)}
                helperText={!!error && (isNaN(maxValue) || maxValue < 0 || maxValue < minValue) ? error : ''}
              />
            </Box>
          </Box>
        </FormControl>
      </form>
    </Box>
  )
}
