/*
 * Copyright (c) 2013, 2015 Oracle and/or its affiliates. All rights reserved. This
 * code is released under a tri EPL/GPL/LGPL license. You can use it,
 * redistribute it and/or modify it under the terms of the:
 *
 * Eclipse Public License version 1.0
 * GNU General Public License version 2
 * GNU Lesser General Public License version 2.1
 */
package org.jruby.truffle.runtime.object;

import com.oracle.truffle.api.Truffle;
import com.oracle.truffle.api.frame.Frame;
import com.oracle.truffle.api.frame.FrameInstance;
import com.oracle.truffle.api.frame.FrameInstanceVisitor;
import com.oracle.truffle.api.frame.FrameSlot;
import com.oracle.truffle.api.nodes.Node;
import com.oracle.truffle.api.object.DynamicObject;
import com.oracle.truffle.api.object.Property;
import org.jruby.truffle.runtime.RubyArguments;
import org.jruby.truffle.runtime.RubyContext;
import org.jruby.truffle.runtime.hash.Entry;
import org.jruby.truffle.runtime.layouts.Layouts;
import org.jruby.truffle.runtime.subsystems.SafepointAction;

import java.util.*;

public abstract class ObjectGraph {

    public static Set<DynamicObject> stopAndGetAllObjects(
            Node currentNode, final RubyContext context) {
        final Set<DynamicObject> visited = new HashSet<>();

        final Thread stoppingThread = Thread.currentThread();

        context.getSafepointManager().pauseAllThreadsAndExecute(currentNode, false, new SafepointAction() {

            @Override
            public void run(DynamicObject thread, Node currentNode) {
                synchronized (visited) {
                    final Deque<DynamicObject> stack = new ArrayDeque<>();

                    stack.add(thread);

                    if (Thread.currentThread() == stoppingThread) {
                        visitContextRoots(context, stack);
                    }

                    final FrameInstance currentFrame = Truffle.getRuntime().getCurrentFrame();
                    if (currentFrame != null) {
                        stack.addAll(getObjectsInFrame(currentFrame.getFrame(FrameInstance.FrameAccess.READ_ONLY, true)));
                    }

                    Truffle.getRuntime().iterateFrames(new FrameInstanceVisitor<Object>() {
                        @Override
                        public Object visitFrame(FrameInstance frameInstance) {
                            stack.addAll(getObjectsInFrame(frameInstance.getFrame(FrameInstance.FrameAccess.READ_ONLY, true)));
                            return null;
                        }
                    });

                    while (!stack.isEmpty()) {
                        final DynamicObject object = stack.pop();

                        if (visited.add(object)) {
                            stack.addAll(ObjectGraph.getAdjacentObjects(object));
                        }
                    }
                }
            }
        });

        return visited;
    }

    public static Set<DynamicObject> stopAndGetRootObjects(Node currentNode, final RubyContext context) {
        final Set<DynamicObject> objects = new HashSet<>();

        final Thread stoppingThread = Thread.currentThread();

        context.getSafepointManager().pauseAllThreadsAndExecute(currentNode, false, new SafepointAction() {
            @Override
            public void run(DynamicObject thread, Node currentNode) {
                objects.add(thread);

                if (Thread.currentThread() == stoppingThread) {
                    visitContextRoots(context, objects);
                }
            }
        });

        return objects;
    }

    public static void visitContextRoots(RubyContext context, Collection<DynamicObject> stack) {
        // We do not want to expose the global object
        stack.addAll(ObjectGraph.getAdjacentObjects(context.getCoreLibrary().getGlobalVariablesObject()));

        stack.addAll(context.getAtExitManager().getHandlers());
        stack.addAll(context.getObjectSpaceManager().getFinalizerHandlers());
    }

    public static Set<DynamicObject> getAdjacentObjects(DynamicObject object) {
        final Set<DynamicObject> reachable = new HashSet<>();

        reachable.add(Layouts.BASIC_OBJECT.getLogicalClass(object));
        reachable.add(Layouts.BASIC_OBJECT.getMetaClass(object));

        for (Property property : object.getShape().getPropertyListInternal(false)) {
            final Object propertyValue = property.get(object, object.getShape());

            if (propertyValue instanceof DynamicObject) {
                reachable.add((DynamicObject) propertyValue);
            } else if (propertyValue instanceof Entry[]) {
                for (Entry bucket : (Entry[]) propertyValue) {
                    while (bucket != null) {
                        if (bucket.getKey() instanceof DynamicObject) {
                            reachable.add((DynamicObject) bucket.getKey());
                        }

                        if (bucket.getValue() instanceof DynamicObject) {
                            reachable.add((DynamicObject) bucket.getValue());
                        }

                        bucket = bucket.getNextInLookup();
                    }
                }
            } else if (propertyValue instanceof Object[]) {
                for (Object element : (Object[]) propertyValue) {
                    if (element instanceof DynamicObject) {
                        reachable.add((DynamicObject) element);
                    }
                }
            } else if (propertyValue instanceof Collection<?>) {
                for (Object element : ((Collection<?>) propertyValue)) {
                    if (element instanceof DynamicObject) {
                        reachable.add((DynamicObject) element);
                    }
                }
            } else if (propertyValue instanceof Frame) {
                reachable.addAll(getObjectsInFrame((Frame) propertyValue));
            } else if (propertyValue instanceof ObjectGraphNode) {
                reachable.addAll(((ObjectGraphNode) propertyValue).getAdjacentObjects());
            }
        }

        return reachable;
    }

    public static Set<DynamicObject> getObjectsInFrame(Frame frame) {
        final Set<DynamicObject> objects = new HashSet<>();

        final Object[] arguments = frame.getArguments();
        final Frame lexicalParentFrame = RubyArguments.tryGetDeclarationFrame(arguments);
        if (lexicalParentFrame != null) {
            objects.addAll(getObjectsInFrame(lexicalParentFrame));
        }

        final Object self = RubyArguments.tryGetSelf(arguments);
        if (self instanceof DynamicObject) {
            objects.add((DynamicObject) self);
        }

        final DynamicObject block = RubyArguments.tryGetBlock(arguments);
        if (block != null) {
            objects.add(block);
        }

        // Other frame arguments are either only internal or user arguments which appear in slots.

        for (FrameSlot slot : frame.getFrameDescriptor().getSlots()) {
            final Object slotValue = frame.getValue(slot);

            if (slotValue instanceof DynamicObject) {
                objects.add((DynamicObject) slotValue);
            }
        }

        return objects;
    }

}
