/*
 * Copyright (c) 2013, 2016 Oracle and/or its affiliates. All rights reserved. This
 * code is released under a tri EPL/GPL/LGPL license. You can use it,
 * redistribute it and/or modify it under the terms of the:
 *
 * Eclipse Public License version 1.0
 * GNU General Public License version 2
 * GNU Lesser General Public License version 2.1
 */
package org.jruby.truffle.nodes.core.array;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.dsl.ImportStatic;
import com.oracle.truffle.api.dsl.NodeChild;
import com.oracle.truffle.api.dsl.NodeChildren;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.object.DynamicObject;
import com.oracle.truffle.api.source.SourceSection;
import org.jruby.truffle.nodes.RubyNode;
import org.jruby.truffle.runtime.RubyContext;
import org.jruby.truffle.runtime.array.ArrayUtils;
import org.jruby.truffle.runtime.layouts.Layouts;

@NodeChildren({@NodeChild(value = "array", type = RubyNode.class)})
@ImportStatic(ArrayGuards.class)
public abstract class ArrayDropTailNode extends RubyNode {

    final int index;

    public ArrayDropTailNode(RubyContext context, SourceSection sourceSection, int index) {
        super(context, sourceSection);
        this.index = index;
    }

    @Specialization(guards = "isNullArray(array)")
    public DynamicObject getHeadNull(DynamicObject array) {
        CompilerDirectives.transferToInterpreter();

        return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), null, 0);
    }

    @Specialization(guards = "isIntArray(array)")
    public DynamicObject getHeadIntegerFixnum(DynamicObject array) {
        CompilerDirectives.transferToInterpreter();

        if (index >= Layouts.ARRAY.getSize(array)) {
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), null, 0);
        } else {
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), ArrayUtils.extractRange((int[]) Layouts.ARRAY.getStore(array), 0, Layouts.ARRAY.getSize(array) - index), Layouts.ARRAY.getSize(array) - index);
        }
    }

    @Specialization(guards = "isLongArray(array)")
    public DynamicObject geHeadLongFixnum(DynamicObject array) {
        CompilerDirectives.transferToInterpreter();

        if (index >= Layouts.ARRAY.getSize(array)) {
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), null, 0);
        } else {
            final int size = Layouts.ARRAY.getSize(array) - index;
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), ArrayUtils.extractRange((long[]) Layouts.ARRAY.getStore(array), 0, size), size);
        }
    }

    @Specialization(guards = "isDoubleArray(array)")
    public DynamicObject getHeadFloat(DynamicObject array) {
        CompilerDirectives.transferToInterpreter();

        if (index >= Layouts.ARRAY.getSize(array)) {
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), null, 0);
        } else {
            final int size = Layouts.ARRAY.getSize(array) - index;
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), ArrayUtils.extractRange((double[]) Layouts.ARRAY.getStore(array), 0, size), size);
        }
    }

    @Specialization(guards = "isObjectArray(array)")
    public DynamicObject getHeadObject(DynamicObject array) {
        CompilerDirectives.transferToInterpreter();

        if (index >= Layouts.ARRAY.getSize(array)) {
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), null, 0);
        } else {
            final int size = Layouts.ARRAY.getSize(array) - index;
            return Layouts.ARRAY.createArray(getContext().getCoreLibrary().getArrayFactory(), ArrayUtils.extractRange((Object[]) Layouts.ARRAY.getStore(array), 0, size), size);
        }
    }

}
