require File.dirname(__FILE__) + "/../spec_helper"

java_import "java_integration.fixtures.ClassWithEnums"
java_import "java_integration.fixtures.JavaFields"
java_import "java_integration.fixtures.InnerClasses"

describe "Kernel\#java_import" do
  describe "given a default-package class" do
    it "imports the class appropriately" do
      m = Module.new do
        java_import Java::DefaultPackageClass
      end
      expect(m::DefaultPackageClass).to eq(Java::DefaultPackageClass)
    end
  end
end

describe "Java::JavaClass.for_name" do
  it "should return primitive classes for Java primitive type names" do
    expect(Java::JavaClass.for_name("byte")).to eq(Java::byte.java_class)
    expect(Java::JavaClass.for_name("boolean")).to eq(Java::boolean.java_class)
    expect(Java::JavaClass.for_name("short")).to eq(Java::short.java_class)
    expect(Java::JavaClass.for_name("char")).to eq(Java::char.java_class)
    expect(Java::JavaClass.for_name("int")).to eq(Java::int.java_class)
    expect(Java::JavaClass.for_name("long")).to eq(Java::long.java_class)
    expect(Java::JavaClass.for_name("float")).to eq(Java::float.java_class)
    expect(Java::JavaClass.for_name("double")).to eq(Java::double.java_class)
  end
end

describe "Java classes with nested enums" do
  it "should allow access to the values() method on the enum" do
    expect(ClassWithEnums::Enums.values.map{|e|e.to_s}).to eq(["A", "B", "C"]);
  end
end

describe "A Java class" do
  describe "in a package with a leading underscore" do
    it "can be accessed directly using the Java:: prefix" do
      myclass = Java::java_integration.fixtures._funky.MyClass
      expect(myclass.new.foo).to eq("MyClass")
    end
  end
end

describe "A JavaClass wrapper around a java.lang.Class" do
  it "provides a nice String output for inspect" do
    myclass = java.lang.String.java_class
    expect(myclass.inspect).to eq("class java.lang.String")
  end
end

describe "A JavaClass with fields containing leading and trailing $" do
  it "should be accessible" do
    expect(JavaFields.send('$LEADING')).to eq("leading")
    expect(JavaFields.send('TRAILING$')).to eq(true)
  end
end

describe "A Java class with inner classes" do
  it "should define constants for constantable classes" do
    expect(InnerClasses.constants).to have_strings_or_symbols 'CapsInnerClass'
    expect(InnerClasses::CapsInnerClass.value).to eq(1)

    expect(InnerClasses::CapsInnerClass.constants).to have_strings_or_symbols "CapsInnerClass2"
    expect(InnerClasses::CapsInnerClass::CapsInnerClass2.value).to eq(1)

    expect(InnerClasses::CapsInnerClass.constants).to have_strings_or_symbols "CapsInnerInterface2"

    expect(InnerClasses::CapsInnerClass.constants).not_to have_strings_or_symbols 'lowerInnerClass2'
    expect(InnerClasses::CapsInnerClass.constants).not_to have_strings_or_symbols 'lowerInnerInterface2'

    expect(InnerClasses.constants).to have_strings_or_symbols 'CapsInnerInterface'

    expect(InnerClasses::CapsInnerInterface.constants).to have_strings_or_symbols "CapsInnerClass4"
    expect(InnerClasses::CapsInnerInterface::CapsInnerClass4.value).to eq(1)

    expect(InnerClasses::CapsInnerInterface.constants).to have_strings_or_symbols "CapsInnerInterface4"

    expect(InnerClasses::CapsInnerInterface.constants).not_to have_strings_or_symbols 'lowerInnerClass4'
    expect(InnerClasses::CapsInnerInterface.constants).not_to have_strings_or_symbols 'lowerInnerInterface4'
  end

  it "should define methods for lower-case classes" do
    expect(InnerClasses.methods).to have_strings_or_symbols 'lowerInnerClass'
    expect(InnerClasses::lowerInnerClass.value).to eq(1)
    expect(InnerClasses.lowerInnerClass.value).to eq(1)
    expect(InnerClasses.lowerInnerClass).to eq(InnerClasses::lowerInnerClass)

    expect(InnerClasses.lowerInnerClass.constants).to have_strings_or_symbols 'CapsInnerClass3'
    expect(InnerClasses.lowerInnerClass.constants).to have_strings_or_symbols 'CapsInnerInterface3'

    expect(InnerClasses.lowerInnerClass::CapsInnerClass3.value).to eq(1)

    expect(InnerClasses.lowerInnerClass.methods).to have_strings_or_symbols 'lowerInnerInterface3'
    expect(InnerClasses.lowerInnerClass.methods).to have_strings_or_symbols 'lowerInnerClass3'

    expect(InnerClasses.lowerInnerClass::lowerInnerClass3.value).to eq(1)
    expect(InnerClasses.lowerInnerClass.lowerInnerClass3.value).to eq(1)

    expect(InnerClasses.methods).to have_strings_or_symbols 'lowerInnerInterface'
    expect(InnerClasses.lowerInnerInterface).to eq(InnerClasses::lowerInnerInterface)

    expect(InnerClasses.lowerInnerInterface.constants).to have_strings_or_symbols 'CapsInnerClass5'
    expect(InnerClasses.lowerInnerInterface.constants).to have_strings_or_symbols 'CapsInnerInterface5'

    expect(InnerClasses.lowerInnerInterface::CapsInnerClass5.value).to eq(1)

    expect(InnerClasses.lowerInnerInterface.methods).to have_strings_or_symbols 'lowerInnerInterface5'
    expect(InnerClasses.lowerInnerInterface.methods).to have_strings_or_symbols 'lowerInnerClass5'

    expect(InnerClasses.lowerInnerInterface::lowerInnerClass5.value).to eq(1)
    expect(InnerClasses.lowerInnerInterface.lowerInnerClass5.value).to eq(1)
  end

  it "defines constant for public inner classes" do
    expect( java.awt.font.TextLayout.constants.map(&:to_sym) ).to include :CaretPolicy
    java.awt.font.TextLayout::CaretPolicy
  end

  it "does not define constants for non-public inner classes" do
    constants = InnerClasses.constants.map(&:to_sym)
    expect( constants ).to_not include :PackageInner
    expect( constants ).to_not include :ProtectedInner
    expect( constants ).to_not include :PrivateInner
  end

  it "allows to retrieve non-public inner classes" do
    # InnerClasses::PackageInner
    expect( InnerClasses.const_get :PackageInner ).to_not be nil
    class InnerClasses
      PACKAGE_INNER = PackageInner
    end
    expect( InnerClasses.constants ).to_not include :PackageInner
    expect( InnerClasses::PACKAGE_INNER ).to eql JavaUtilities.get_proxy_class('java_integration.fixtures.InnerClasses$PackageInner')

    expect( InnerClasses.const_get :PrivateInner ).to_not be nil
    expect( InnerClasses.const_get :ProtectedInner ).to_not be nil

    class InnerClasses
      PROTECTED_INNER = ProtectedInner
    end

    expect( InnerClasses::PROTECTED_INNER.const_get :Nested ).to_not be nil

    expect { InnerClasses::MissingInner }.to raise_error(NameError)
    begin
      InnerClasses.const_get :MissingInner
    rescue NameError => e
      expect( e.message ).to start_with 'uninitialized constant Java::Java_integrationFixtures::InnerClasses::MissingInner'
    else
      fail 'did not raise!'
    end
  end

  it "raises error importing lower-case names" do
    expect do
      java_import InnerClasses::lowerInnerClass
    end.to raise_error(ArgumentError)
  end

  it "imports upper-case names successfully" do
    expect do
      java_import InnerClasses::CapsInnerClass
    end.not_to raise_error
    expect(CapsInnerClass).to eq(InnerClasses::CapsInnerClass)
  end
end
