// SPDX-License-Identifier: Apache-2.0

use crate::drawbridge::{client, RepoSpec};

use std::ffi::OsString;

use anyhow::Context;
use camino::Utf8PathBuf;
use clap::Args;
use drawbridge_client::types::{RepositoryConfig, TagName};
use oauth2::url::Url;
use serde::Serialize;

#[derive(Serialize)]
struct RepoInfo {
    config: RepositoryConfig,
    tags: Vec<TagName>,
}

/// List all tags associated with a repository.
#[derive(Args, Debug)]
pub struct Options {
    #[clap(long, env = "ENARX_CA_BUNDLE")]
    ca_bundle: Option<Utf8PathBuf>,
    #[clap(long, default_value = "https://auth.profian.com/")]
    oidc_domain: Url,
    #[clap(long, env = "ENARX_INSECURE_AUTH_TOKEN")]
    insecure_auth_token: Option<String>,
    #[clap(long, env = "ENARX_CREDENTIAL_HELPER")]
    credential_helper: Option<OsString>,
    spec: RepoSpec,
}

impl Options {
    pub fn execute(self) -> anyhow::Result<()> {
        let cl = client(
            &self.spec.host,
            &self.oidc_domain,
            &self.insecure_auth_token,
            &self.ca_bundle,
            &self.credential_helper,
        )?;
        let repo = cl.repository(&self.spec.ctx);
        let config = repo
            .get()
            .context("Failed to retrieve repository information")?;
        let tags = repo.tags().context("Failed to retrieve repository tags")?;
        let info = RepoInfo { config, tags };
        println!("{}", serde_json::to_string_pretty(&info)?);
        Ok(())
    }
}
