// Copyright (c) 2016 GitHub, Inc.
// Use of this source code is governed by the MIT license that can be
// found in the LICENSE file.

#ifndef ELECTRON_SHELL_BROWSER_API_ELECTRON_API_SYSTEM_PREFERENCES_H_
#define ELECTRON_SHELL_BROWSER_API_ELECTRON_API_SYSTEM_PREFERENCES_H_

#include <memory>
#include <string>

#include "base/functional/bind.h"
#include "base/values.h"
#include "shell/browser/event_emitter_mixin.h"
#include "shell/common/gin_helper/wrappable.h"

#if BUILDFLAG(IS_WIN)
#include "base/callback_list.h"
#include "shell/browser/browser.h"
#include "shell/browser/browser_observer.h"
#endif
#if BUILDFLAG(IS_LINUX)
#include "base/memory/raw_ptr.h"
#include "ui/native_theme/native_theme.h"
#include "ui/native_theme/native_theme_observer.h"
#endif

namespace gin_helper {
template <typename T>
class Handle;
}  // namespace gin_helper

namespace gin_helper {
class ErrorThrower;
}  // namespace gin_helper

namespace electron::api {

#if BUILDFLAG(IS_MAC)
enum class NotificationCenterKind {
  kNSDistributedNotificationCenter = 0,
  kNSNotificationCenter,
  kNSWorkspaceNotificationCenter,
};
#endif

class SystemPreferences final
    : public gin_helper::DeprecatedWrappable<SystemPreferences>,
      public gin_helper::EventEmitterMixin<SystemPreferences>
#if BUILDFLAG(IS_WIN)
    ,
      public BrowserObserver
#elif BUILDFLAG(IS_LINUX)
    ,
      public ui::NativeThemeObserver
#endif
{
 public:
  static gin_helper::Handle<SystemPreferences> Create(v8::Isolate* isolate);

  // gin_helper::Wrappable
  static gin::DeprecatedWrapperInfo kWrapperInfo;
  gin::ObjectTemplateBuilder GetObjectTemplateBuilder(
      v8::Isolate* isolate) override;
  const char* GetTypeName() override;

  std::string GetAccentColor();
#if BUILDFLAG(IS_WIN) || BUILDFLAG(IS_MAC)
  std::string GetColor(gin_helper::ErrorThrower thrower,
                       const std::string& color);
  std::string GetMediaAccessStatus(gin_helper::ErrorThrower thrower,
                                   const std::string& media_type);
#endif
#if BUILDFLAG(IS_WIN)
  void InitializeWindow();

  // Called by `hwnd_subscription_`.
  void OnWndProc(HWND hwnd, UINT message, WPARAM wparam, LPARAM lparam);

  // BrowserObserver:
  void OnFinishLaunching(base::Value::Dict launch_info) override;

#elif BUILDFLAG(IS_MAC)
  using NotificationCallback = base::RepeatingCallback<
      void(const std::string&, base::Value, const std::string&)>;

  void PostNotification(const std::string& name,
                        base::Value::Dict user_info,
                        gin::Arguments* args);
  int SubscribeNotification(v8::Local<v8::Value> maybe_name,
                            const NotificationCallback& callback);
  void UnsubscribeNotification(int id);
  void PostLocalNotification(const std::string& name,
                             base::Value::Dict user_info);
  int SubscribeLocalNotification(v8::Local<v8::Value> maybe_name,
                                 const NotificationCallback& callback);
  void UnsubscribeLocalNotification(int request_id);
  void PostWorkspaceNotification(const std::string& name,
                                 base::Value::Dict user_info);
  int SubscribeWorkspaceNotification(v8::Local<v8::Value> maybe_name,
                                     const NotificationCallback& callback);
  void UnsubscribeWorkspaceNotification(int request_id);
  v8::Local<v8::Value> GetUserDefault(v8::Isolate* isolate,
                                      const std::string& name,
                                      const std::string& type);
  void RegisterDefaults(gin::Arguments* args);
  void SetUserDefault(const std::string& name,
                      const std::string& type,
                      gin::Arguments* args);
  void RemoveUserDefault(const std::string& name);
  bool IsSwipeTrackingFromScrollEventsEnabled();
  bool AccessibilityDisplayShouldReduceTransparency();

  std::string GetSystemColor(gin_helper::ErrorThrower thrower,
                             const std::string& color);

  bool CanPromptTouchID();
  v8::Local<v8::Promise> PromptTouchID(v8::Isolate* isolate,
                                       const std::string& reason);

  static bool IsTrustedAccessibilityClient(bool prompt);

  v8::Local<v8::Promise> AskForMediaAccess(v8::Isolate* isolate,
                                           const std::string& media_type);

  // TODO(MarshallOfSound): Write tests for these methods once we
  // are running tests on a Mojave machine
  v8::Local<v8::Value> GetEffectiveAppearance(v8::Isolate* isolate);

#elif BUILDFLAG(IS_LINUX)
  // ui::NativeThemeObserver:
  void OnNativeThemeUpdated(ui::NativeTheme* theme) override;
#endif
  v8::Local<v8::Value> GetAnimationSettings(v8::Isolate* isolate);

  // disable copy
  SystemPreferences(const SystemPreferences&) = delete;
  SystemPreferences& operator=(const SystemPreferences&) = delete;

 protected:
  SystemPreferences();
  ~SystemPreferences() override;

#if BUILDFLAG(IS_MAC)
  int DoSubscribeNotification(v8::Local<v8::Value> maybe_name,
                              const NotificationCallback& callback,
                              NotificationCenterKind kind);
  void DoUnsubscribeNotification(int request_id, NotificationCenterKind kind);
#endif

 private:
#if BUILDFLAG(IS_WIN)
  // Static callback invoked when a message comes in to our messaging window.
  static LRESULT CALLBACK WndProcStatic(HWND hwnd,
                                        UINT message,
                                        WPARAM wparam,
                                        LPARAM lparam);

  LRESULT CALLBACK WndProc(HWND hwnd,
                           UINT message,
                           WPARAM wparam,
                           LPARAM lparam);

  // The window class of |window_|.
  ATOM atom_;

  // The handle of the module that contains the window procedure of |window_|.
  HMODULE instance_;

  // The window used for processing events.
  HWND window_;

  std::string current_color_;

  // Color/high contrast mode change observer.
  base::CallbackListSubscription hwnd_subscription_;
#endif
#if BUILDFLAG(IS_LINUX)
  void OnNativeThemeUpdatedOnUI();

  raw_ptr<ui::NativeTheme> ui_theme_;
  std::string current_accent_color_;
#endif
};

}  // namespace electron::api

#endif  // ELECTRON_SHELL_BROWSER_API_ELECTRON_API_SYSTEM_PREFERENCES_H_
