#!/bin/sh
#===----------------------------------------------------------------------
#
# This source file is part of the Swift Collections open source project
#
# Copyright (c) 2022 Apple Inc. and the Swift project authors
# Licensed under Apache License v2.0 with Runtime Library Exception
#
# See https://swift.org/LICENSE.txt for license information
#
#===----------------------------------------------------------------------

set -eu

shopt -s nullglob
cd "$(dirname $0)/.."

if [ "$(uname)" = "Darwin" ]; then
    swift="xcrun swift"
    export DOCC_HTML_DIR="$(dirname $(xcrun --find docc))/../share/docc/render"
else
    swift="swift"
fi

build_dir="$(mktemp -d "/tmp/$(basename $0).XXXXX")"
echo "Build directory: $build_dir"

output_dir_base=/tmp/foo

rm -rf "$output_dir_base"

components="DequeModule OrderedCollections"

if [ $# -eq 0 ]; then
    targets="Collections $components"
else
    targets="$@"
fi

for target in $targets; do
    if [ "$target" = "Collections" ]; then
        # Update extensions under Collections.docc
        rm -rf Sources/Collections/Collections.docc/Extensions/*.md
        for component in $components; do
            for file in Sources/"$component"/"$component".docc/Extensions/*.md; do
                output="Sources/Collections/Collections.docc/Extensions/$(basename $file)"
                blurb="<!-- DO NOT EDIT THIS FILE; IT'S AUTOMATICALLY GENERATED -->"
                sed 's?^# ``[^/]*/\(.*\)``?# ``Collections/\1``\n\n'"$blurb"'\n?' "$file" > "$output"
            done
        done
    fi

    mkdir -p "$build_dir/$target/build"
    mkdir -p "$build_dir/$target/symbol-graphs"
    mkdir -p "$build_dir/$target/docc"
    $swift build --target "$target" \
           --build-path "$build_dir/$target/build" \
           -Xswiftc -emit-symbol-graph \
           -Xswiftc -emit-symbol-graph-dir \
           -Xswiftc "$build_dir/$target/symbol-graphs"

    # Prevent DocC from getting confused by too much data :-/
    cp \
        "$build_dir/$target/symbol-graphs/$target.symbols.json" \
        "$build_dir/$target/docc"
    extrafile="$build_dir/$target/symbol-graphs/${target}@Swift.symbols.json"
    if [ -r "$extrafile" ]; then
        cp "$extrafile" "$build_dir/$target/docc"
    fi

    mkdir -p "$output_dir_base/$target"
    xcrun docc convert \
          --analyze \
          --fallback-display-name "$target" \
          --fallback-bundle-identifier "org.swift.swift-collections.$target" \
          --fallback-bundle-version 1.1.0 \
          --default-code-listing-language swift \
          --additional-symbol-graph-dir "$build_dir/$target/docc" \
          --transform-for-static-hosting \
          --output-path "$output_dir_base/$target" \
          --hosting-base-path "/swift-collections/$target" \
          "Sources/$target/$target.docc"
done

if [ $# -eq 1 ]; then
    module="$1"
    xcrun docc preview "Sources/$module/$module.docc" \
          --fallback-display-name "Swift Collections" \
          --fallback-bundle-identifier "org.swift.swift-collections.$module" \
          --fallback-bundle-version 1.1.0 \
          --additional-symbol-graph-dir "$build_dir/$module/docc"
fi
