// Copyright (c) 2021 Target Brands, Inc. All rights reserved.
//
// Use of this source code is governed by the LICENSE file in this repository.

package database

import (
	"database/sql"
	"errors"

	"github.com/go-vela/types/library"
)

var (
	// ErrEmptyHookNumber defines the error type when a
	// Hook type has an empty Number field provided.
	ErrEmptyHookNumber = errors.New("empty webhook number provided")

	// ErrEmptyHookRepoID defines the error type when a
	// Hook type has an empty RepoID field provided.
	ErrEmptyHookRepoID = errors.New("empty webhook repo_id provided")

	// ErrEmptyHookSourceID defines the error type when a
	// Hook type has an empty SourceID field provided.
	ErrEmptyHookSourceID = errors.New("empty webhook source_id provided")
)

// Hook is the database representation of a webhook for a repo.
type Hook struct {
	ID       sql.NullInt64  `sql:"id"`
	RepoID   sql.NullInt64  `sql:"repo_id"`
	BuildID  sql.NullInt64  `sql:"build_id"`
	Number   sql.NullInt32  `sql:"number"`
	SourceID sql.NullString `sql:"source_id"`
	Created  sql.NullInt64  `sql:"created"`
	Host     sql.NullString `sql:"host"`
	Event    sql.NullString `sql:"event"`
	Branch   sql.NullString `sql:"branch"`
	Error    sql.NullString `sql:"error"`
	Status   sql.NullString `sql:"status"`
	Link     sql.NullString `sql:"link"`
}

// Nullify ensures the valid flag for
// the sql.Null types are properly set.
//
// When a field within the Hook type is the zero
// value for the field, the valid flag is set to
// false causing it to be NULL in the database.
func (h *Hook) Nullify() *Hook {
	if h == nil {
		return nil
	}

	// check if the ID field should be false
	if h.ID.Int64 == 0 {
		h.ID.Valid = false
	}

	// check if the RepoID field should be false
	if h.RepoID.Int64 == 0 {
		h.RepoID.Valid = false
	}

	// check if the BuildID field should be false
	if h.BuildID.Int64 == 0 {
		h.BuildID.Valid = false
	}

	// check if the Number field should be false
	if h.Number.Int32 == 0 {
		h.Number.Valid = false
	}

	// check if the SourceID field should be false
	if len(h.SourceID.String) == 0 {
		h.SourceID.Valid = false
	}

	// check if the Created field should be false
	if h.Created.Int64 == 0 {
		h.Created.Valid = false
	}

	// check if the Host field should be false
	if len(h.Host.String) == 0 {
		h.Host.Valid = false
	}

	// check if the Event field should be false
	if len(h.Event.String) == 0 {
		h.Event.Valid = false
	}

	// check if the Branch field should be false
	if len(h.Branch.String) == 0 {
		h.Branch.Valid = false
	}

	// check if the Error field should be false
	if len(h.Error.String) == 0 {
		h.Error.Valid = false
	}

	// check if the Status field should be false
	if len(h.Status.String) == 0 {
		h.Status.Valid = false
	}

	// check if the Link field should be false
	if len(h.Link.String) == 0 {
		h.Link.Valid = false
	}

	return h
}

// ToLibrary converts the Hook type
// to a library Hook type.
func (h *Hook) ToLibrary() *library.Hook {
	hook := new(library.Hook)

	hook.SetID(h.ID.Int64)
	hook.SetRepoID(h.RepoID.Int64)
	hook.SetBuildID(h.BuildID.Int64)
	hook.SetNumber(int(h.Number.Int32))
	hook.SetSourceID(h.SourceID.String)
	hook.SetCreated(h.Created.Int64)
	hook.SetHost(h.Host.String)
	hook.SetEvent(h.Event.String)
	hook.SetBranch(h.Branch.String)
	hook.SetError(h.Error.String)
	hook.SetStatus(h.Status.String)
	hook.SetLink(h.Link.String)

	return hook
}

// Validate verifies the necessary fields for
// the Hook type are populated correctly.
func (h *Hook) Validate() error {
	// verify the RepoID field is populated
	if h.RepoID.Int64 <= 0 {
		return ErrEmptyHookRepoID
	}

	// verify the Number field is populated
	if h.Number.Int32 <= 0 {
		return ErrEmptyHookNumber
	}

	// verify the SourceID field is populated
	if len(h.SourceID.String) <= 0 {
		return ErrEmptyHookSourceID
	}

	// ensure that all Hook string fields
	// that can be returned as JSON are sanitized
	// to avoid unsafe HTML content
	h.SourceID = sql.NullString{String: sanitize(h.SourceID.String), Valid: true}
	h.Host = sql.NullString{String: sanitize(h.Host.String), Valid: true}
	h.Event = sql.NullString{String: sanitize(h.Event.String), Valid: true}
	h.Branch = sql.NullString{String: sanitize(h.Branch.String), Valid: true}
	h.Error = sql.NullString{String: sanitize(h.Error.String), Valid: true}
	h.Status = sql.NullString{String: sanitize(h.Status.String), Valid: true}
	h.Link = sql.NullString{String: sanitize(h.Link.String), Valid: true}

	return nil
}

// HookFromLibrary converts the Hook type
// to a library Hook type.
func HookFromLibrary(h *library.Hook) *Hook {
	hook := &Hook{
		ID:       sql.NullInt64{Int64: h.GetID(), Valid: true},
		RepoID:   sql.NullInt64{Int64: h.GetRepoID(), Valid: true},
		BuildID:  sql.NullInt64{Int64: h.GetBuildID(), Valid: true},
		Number:   sql.NullInt32{Int32: int32(h.GetNumber()), Valid: true},
		SourceID: sql.NullString{String: h.GetSourceID(), Valid: true},
		Created:  sql.NullInt64{Int64: h.GetCreated(), Valid: true},
		Host:     sql.NullString{String: h.GetHost(), Valid: true},
		Event:    sql.NullString{String: h.GetEvent(), Valid: true},
		Branch:   sql.NullString{String: h.GetBranch(), Valid: true},
		Error:    sql.NullString{String: h.GetError(), Valid: true},
		Status:   sql.NullString{String: h.GetStatus(), Valid: true},
		Link:     sql.NullString{String: h.GetLink(), Valid: true},
	}

	return hook.Nullify()
}
