/*
 * Copyright 2022 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.keycloak.config;

import org.keycloak.models.map.storage.hotRod.common.AutogeneratedHotRodDescriptors;
import org.keycloak.models.map.storage.hotRod.common.HotRodEntityDescriptor;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class StorageOptions {

    public enum StorageType {

        jpa("jpa"),
        chm("concurrenthashmap"),
        hotrod("hotrod");

        private final String provider;

        StorageType(String provider) {
            this.provider = provider;
        }

        public String getProvider() {
            return provider;
        }
    }

    public static final Option<StorageType> STORAGE = new OptionBuilder<>("storage", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(String.format("Sets the default storage mechanism for all areas. Possible values are: %s.", storageAreas()))
            .expectedValues(StorageType.values())
            .defaultValue(Optional.empty())
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_PROVIDER = new OptionBuilder<>("storage-provider", StorageType.class)
            .category(OptionCategory.STORAGE)
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_EVENT_STORE_PROVIDER = new OptionBuilder<>("storage-event-store-provider", String.class)
            .category(OptionCategory.STORAGE)
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_EVENT_ADMIN_STORE = new OptionBuilder<>("storage-area-event-admin", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("admin events"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_EVENT_AUTH_STORE = new OptionBuilder<>("storage-area-event-auth", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("authentication and authorization events"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_EXCEPTION_CONVERTER = new OptionBuilder<>("storage-exception-converter", StorageType.class)
            .category(OptionCategory.STORAGE)
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_REALM_PROVIDER = new OptionBuilder<>("storage-realm-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_REALM_STORE = new OptionBuilder<>("storage-area-realm", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("realms"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_CLIENT_PROVIDER = new OptionBuilder<>("storage-client-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_CLIENT_STORE = new OptionBuilder<>("storage-area-client", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("clients"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_CLIENT_SCOPE_PROVIDER = new OptionBuilder<>("storage-client-scope-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_CLIENT_SCOPE_STORE = new OptionBuilder<>("storage-area-client-scope", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("client scopes"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_GROUP_PROVIDER = new OptionBuilder<>("storage-group-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_GROUP_STORE = new OptionBuilder<>("storage-area-group", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("groups"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ROLE_PROVIDER = new OptionBuilder<>("storage-role-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_ROLE_STORE = new OptionBuilder<>("storage-area-role", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("roles"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_USER_PROVIDER = new OptionBuilder<>("storage-user-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_USER_STORE = new OptionBuilder<>("storage-area-user", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("users"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_DEPLOYMENT_STATE_PROVIDER = new OptionBuilder<>("storage-deployment-state-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_AUTH_SESSION_PROVIDER = new OptionBuilder<>("storage-auth-session-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_AUTH_SESSION_STORE = new OptionBuilder<>("storage-area-auth-session", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("authentication sessions"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_USER_SESSION_PROVIDER = new OptionBuilder<>("storage-user-session-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_USER_SESSION_STORE = new OptionBuilder<>("storage-area-user-session", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("user and client sessions"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_LOGIN_FAILURE_PROVIDER = new OptionBuilder<>("storage-login-failure-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_LOGIN_FAILURE_STORE = new OptionBuilder<>("storage-area-login-failure", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("login failures"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_AUTHORIZATION_PROVIDER = new OptionBuilder<>("storage-authorization-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_AUTHORIZATION_STORE = new OptionBuilder<>("storage-area-authorization", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("authorizations"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_USER_SESSION_PERSISTER = new OptionBuilder<>("storage-user-session-persister", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ACTION_TOKEN_PROVIDER = new OptionBuilder<>("storage-action-token-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_ACTION_TOKEN_STORE = new OptionBuilder<>("storage-area-action-token", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("action tokens"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_DBLOCK = new OptionBuilder<>("storage-dblock", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_CACHE_REALM_ENABLED = new OptionBuilder<>("cache-realm-enabled", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_CACHE_USER_ENABLED = new OptionBuilder<>("cache-user-enabled", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ADMIN_CACHE_CLEAR_USER = new OptionBuilder<>("cache-clear-user", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ADMIN_CACHE_CLEAR_REALM = new OptionBuilder<>("cache-clear-realm", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ADMIN_CACHE_CLEAR_KEYS = new OptionBuilder<>("cache-clear-keys", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_SINGLE_USE_OBJECT_PROVIDER = new OptionBuilder<>("storage-single-use-object-provider", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<StorageType> STORAGE_SINGLE_USE_OBJECT_STORE = new OptionBuilder<>("storage-area-single-use-object", StorageType.class)
            .category(OptionCategory.STORAGE)
            .description(descriptionForStorageAreas("single use objects"))
            .expectedValues(StorageType.values())
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_PUBLIC_KEY_STORAGE_STORE = new OptionBuilder<>("storage-public-key-storage", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_CACHE_AUTHORIZATION_ENABLED = new OptionBuilder<>("cache-authorization-enabled", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_LEGACY_SESSION_SUPPORT = new OptionBuilder<>("storage-legacy-session-support", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_ADMIN_USER_STORAGE = new OptionBuilder<>("storage-admin-user-storage", String.class)
            .category(OptionCategory.STORAGE)
            .hidden()
            .buildTime(true)
            .build();

    public static final Option<String> STORAGE_HOTROD_HOST = new OptionBuilder<>("storage-hotrod-host", String.class)
            .category(OptionCategory.STORAGE)
            .description("Sets the host of the Infinispan server.")
            .hidden()
            .build();

    public static final Option<Integer> STORAGE_HOTROD_PORT = new OptionBuilder<>("storage-hotrod-port", Integer.class)
            .category(OptionCategory.STORAGE)
            .description("Sets the port of the Infinispan server.")
            .hidden()
            .build();

    public static final Option<String> STORAGE_HOTROD_USERNAME = new OptionBuilder<>("storage-hotrod-username", String.class)
            .category(OptionCategory.STORAGE)
            .description("Sets the username of the Infinispan user.")
            .hidden()
            .build();

    public static final Option<String> STORAGE_HOTROD_PASSWORD = new OptionBuilder<>("storage-hotrod-password", String.class)
            .category(OptionCategory.STORAGE)
            .description("Sets the password of the Infinispan user.")
            .hidden()
            .build();

    public static final Option<Boolean> STORAGE_HOTROD_CACHE_CONFIGURE = new OptionBuilder<>("storage-hotrod-cache-configure", Boolean.class)
            .category(OptionCategory.STORAGE)
            .defaultValue(true)
            .description("When set to true, Keycloak will create and configure Infinispan caches on startup.")
            .hidden()
            .build();

    public static final Option<String> STORAGE_HOTROD_CACHE_REINDEX = new OptionBuilder<>("storage-hotrod-cache-reindex", String.class)
            .category(OptionCategory.STORAGE)
            .defaultValue("all")
            .expectedValues(Stream.concat(Stream.of("all"), AutogeneratedHotRodDescriptors.ENTITY_DESCRIPTOR_MAP.values().stream().map(HotRodEntityDescriptor::getCacheName).distinct()).collect(Collectors.toList()))
            .description("List of cache names that should be indexed on Keycloak startup. Defaulting to `all` which means all caches are reindexed.")
            .hidden()
            .build();


    public static final List<Option<?>> ALL_OPTIONS = List.of(
            STORAGE,
            STORAGE_EVENT_ADMIN_STORE,
            STORAGE_EVENT_AUTH_STORE,
            STORAGE_REALM_STORE,
            STORAGE_CLIENT_STORE,
            STORAGE_CLIENT_SCOPE_STORE,
            STORAGE_GROUP_STORE,
            STORAGE_ROLE_STORE,
            STORAGE_USER_STORE,
            STORAGE_AUTH_SESSION_STORE,
            STORAGE_USER_SESSION_STORE,
            STORAGE_LOGIN_FAILURE_STORE,
            STORAGE_AUTHORIZATION_STORE,
            STORAGE_ACTION_TOKEN_STORE,
            STORAGE_SINGLE_USE_OBJECT_STORE,
            STORAGE_HOTROD_HOST,
            STORAGE_HOTROD_PORT,
            STORAGE_HOTROD_USERNAME,
            STORAGE_HOTROD_PASSWORD,
            STORAGE_HOTROD_CACHE_CONFIGURE,
            STORAGE_HOTROD_CACHE_REINDEX
    );

    private static String descriptionForStorageAreas(String areaAsText) {
        return String.format("Sets a storage mechanism for %s. Possible values are: %s.", areaAsText, storageAreas());
    }

    private static String storageAreas() {
        return String.join(",", Arrays.stream(StorageType.values()).map(StorageType::name).collect(Collectors.joining(", ")));
    }
}
