/// <reference path="../../node_modules/@types/mocha/index.d.ts" />

import { getRxStorageLoki } from '../../plugins/storage-lokijs/index.mjs';
import {
    getRxStorageDexie
} from '../../plugins/storage-dexie/index.mjs';
import { getRxStorageRemoteWebsocket } from '../../plugins/storage-remote-websocket/index.mjs';
import { getRxStorageMemory } from '../../plugins/storage-memory/index.mjs';
import { getRxStorageDenoKV } from '../../plugins/storage-denokv/index.mjs';
import { CUSTOM_STORAGE } from './custom-storage.ts';
import { wrappedValidateAjvStorage } from '../../plugins/validate-ajv/index.mjs';
import { randomNumber } from 'async-test-util';

import {
    RxTestStorage,
    randomDelayStorage
} from '../../plugins/core/index.mjs';

import {
    indexedDB as fakeIndexedDB,
    IDBKeyRange as fakeIDBKeyRange
} from 'fake-indexeddb';
import LokiFsStructuredAdapter from 'lokijs/src/loki-fs-structured-adapter.js';
import LokiIncrementalIndexedDBAdapter from 'lokijs/src/incremental-indexeddb-adapter.js';
import parallel from 'mocha.parallel';

import { createRequire } from 'node:module';
import {
    DEFAULT_STORAGE,
    ENV_VARIABLES,
    getConfig,
    isDeno,
    isFastMode,
    isNode,
    setConfig
} from '../../plugins/test-utils/index.mjs';

function nodeRequire(filePath: string) {
    const require = createRequire(import.meta.url);
    return require(filePath);
}

export const describeParallel: typeof describe = ENV_VARIABLES.NODE_ENV === 'fast' ? parallel : describe;


export function getStorage(storageKey: string): RxTestStorage {
    if (storageKey === CUSTOM_STORAGE.name || storageKey === 'custom') {
        return CUSTOM_STORAGE;
    }

    switch (storageKey) {
        case 'memory':
            return {
                name: storageKey,
                getStorage: () => getRxStorageMemory(),
                getPerformanceStorage() {
                    return {
                        description: 'memory',
                        storage: getRxStorageMemory()
                    };
                },
                hasPersistence: true,
                hasMultiInstance: false,
                hasAttachments: true,
                hasReplication: true
            };
            break;
        /**
         * We run the tests once together
         * with a validation plugin
         * to ensure we do not accidentally use non-valid data
         * in the tests.
         */
        case 'memory-validation':
            return {
                name: storageKey,
                getStorage: () => getRxStorageMemory(),
                getPerformanceStorage() {
                    return {
                        description: 'memory',
                        storage: wrappedValidateAjvStorage({
                            storage: getRxStorageMemory()
                        })
                    };
                },
                hasPersistence: true,
                hasMultiInstance: false,
                hasAttachments: true,
                hasReplication: true
            };
            break;
        /**
         * We run the tests once with random delays
         * on reads and writes. Used to easier detect flaky tests.
         */
        case 'memory-random-delay':

            const delayFn = () => randomNumber(10, 50);
            // const delayFn = () => 150;

            return {
                name: storageKey,
                getStorage: () => randomDelayStorage({
                    storage: getRxStorageDexie({
                        indexedDB: fakeIndexedDB,
                        IDBKeyRange: fakeIDBKeyRange
                    }),
                    delayTimeBefore: delayFn,
                    delayTimeAfter: delayFn
                }),
                getPerformanceStorage() {
                    return {
                        description: 'memory-random-delay',
                        storage: randomDelayStorage({
                            storage: getRxStorageDexie({
                                indexedDB: fakeIndexedDB,
                                IDBKeyRange: fakeIDBKeyRange
                            }),
                            delayTimeBefore: delayFn,
                            delayTimeAfter: delayFn
                        })
                    };
                },
                hasPersistence: true,
                hasMultiInstance: true,
                hasAttachments: false,
                hasReplication: true
            };
            break;
        case 'lokijs':
            return {
                name: storageKey,
                getStorage: () => getRxStorageLoki(),
                getPerformanceStorage() {
                    if (isNode) {
                        // Node.js
                        return {
                            storage: getRxStorageLoki({
                                adapter: new LokiFsStructuredAdapter()
                            }),
                            description: 'loki+fs-structured-adapter'
                        };
                    } else {
                        // browser
                        return {
                            storage: getRxStorageLoki({
                                adapter: new LokiIncrementalIndexedDBAdapter()
                            }),
                            description: 'loki+incremental-indexeddb'
                        };
                    }
                },
                hasPersistence: true,
                hasMultiInstance: true,
                hasAttachments: false,
                hasReplication: true
            };
            break;
        case 'dexie':
            return {
                name: storageKey,
                getStorage: () => {
                    if (
                        isNode ||
                        isDeno ||
                        isFastMode()
                    ) {
                        return getRxStorageDexie({
                            indexedDB: fakeIndexedDB,
                            IDBKeyRange: fakeIDBKeyRange
                        });
                    } else {
                        return getRxStorageDexie({});
                    }
                },
                getPerformanceStorage() {
                    if (isNode) {
                        return {
                            storage: getRxStorageDexie({
                                indexedDB,
                                IDBKeyRange
                            }),
                            description: 'dexie+fake-indexeddb'
                        };
                    } else {
                        return {
                            storage: getRxStorageDexie({}),
                            description: 'dexie+native-indexeddb'
                        };
                    }
                },
                hasPersistence: true,
                hasMultiInstance: true,
                hasAttachments: true,
                hasReplication: true
            };
            break;
        case 'foundationdb':
            const foundationDBAPIVersion = 630;


            let getStorageFnFoundation: any;
            return {
                async init() {
                    // use a dynamic import so it does not break browser bundling
                    const { getRxStorageFoundationDB } = await nodeRequire('../../plugins/storage-foundationdb/index.cjs');
                    getStorageFnFoundation = getRxStorageFoundationDB;
                },
                name: storageKey,
                getStorage: () => {
                    return getStorageFnFoundation({
                        apiVersion: foundationDBAPIVersion
                    });
                },
                getPerformanceStorage() {
                    return {
                        description: 'foundationdb-native',
                        storage: getStorageFnFoundation({
                            apiVersion: foundationDBAPIVersion
                        })
                    };
                },
                hasPersistence: true,
                hasMultiInstance: false,
                hasAttachments: true,
                hasReplication: true
            };
            break;
        case 'mongodb':

            // use a dynamic import so it does not break browser bundling

            const mongoConnectionString = 'mongodb://localhost:27017';
            let getStorageFnMongo: any;
            return {
                async init() {
                    const { getRxStorageMongoDB } = await nodeRequire('../../plugins/storage-mongodb/index.cjs');
                    getStorageFnMongo = getRxStorageMongoDB;
                },
                name: storageKey,
                getStorage: () => {
                    return getStorageFnMongo({
                        connection: mongoConnectionString
                    });
                },
                getPerformanceStorage() {
                    return {
                        description: 'mongodb-native',
                        storage: getStorageFnMongo({
                            connection: mongoConnectionString
                        })
                    };
                },
                hasPersistence: true,
                hasMultiInstance: false,
                hasAttachments: false,
                hasReplication: true
            };
            break;
        case 'remote':
            return {
                name: storageKey,
                getStorage: () => {
                    return getRxStorageRemoteWebsocket({
                        url: 'ws://localhost:18007',
                        mode: 'storage'
                    });
                },
                getPerformanceStorage() {
                    return {
                        storage: getRxStorageRemoteWebsocket({
                            url: 'ws://localhost:18007',
                            mode: 'storage'
                        }),
                        description: 'remote+dexie+fake-indexeddb'
                    };
                },
                hasPersistence: true,
                hasMultiInstance: true,
                hasAttachments: true,
                hasReplication: true
            };
            break;
        case 'denokv':
            return {
                name: storageKey,
                getStorage: () => getRxStorageDenoKV(),
                getPerformanceStorage() {
                    return {
                        description: 'denokv',
                        storage: getRxStorageDenoKV()
                    };
                },
                hasPersistence: true,
                hasMultiInstance: true,
                hasAttachments: false,
                hasReplication: true
            };
            break;
        default:
            throw new Error('no DEFAULT_STORAGE set');
    }
}


const config = (() => {
    setConfig({
        storage: getStorage(DEFAULT_STORAGE) as any
    });
    console.log('# use RxStorage: ' + getConfig().storage.name);
    return getConfig();
})();
export default config;
