package sdk

import (
	"context"
	"errors"
	"fmt"
	"os"
	"strings"

	"dagger.io/dagger"
	"github.com/dagger/dagger/internal/mage/util"
	"github.com/magefile/mage/mg"
)

var (
	nodejsGeneratedAPIPath = "sdk/nodejs/api/client.gen.ts"
)

var _ SDK = Nodejs{}

type Nodejs mg.Namespace

// Lint lints the Node.js SDK
func (t Nodejs) Lint(ctx context.Context) error {
	c, err := dagger.Connect(ctx, dagger.WithLogOutput(os.Stderr))
	if err != nil {
		return err
	}
	defer c.Close()

	return util.WithDevEngine(ctx, c, func(ctx context.Context, c *dagger.Client) error {
		_, err = nodeJsBase(c).
			Exec(dagger.ContainerExecOpts{
				Args:                          []string{"yarn", "lint"},
				ExperimentalPrivilegedNesting: true,
			}).
			ExitCode(ctx)
		return err
	})
}

// Generateandcheck checks generated code
func (t Nodejs) Generateandcheck(ctx context.Context) error {
	return lintGeneratedCode(func() error {
		return t.Generate(ctx)
	}, nodejsGeneratedAPIPath)
}

// Test tests the Node.js SDK
func (t Nodejs) Test(ctx context.Context) error {
	c, err := dagger.Connect(ctx, dagger.WithLogOutput(os.Stderr))
	if err != nil {
		return err
	}
	defer c.Close()

	return util.WithDevEngine(ctx, c, func(ctx context.Context, c *dagger.Client) error {
		_, err = nodeJsBase(c).
			Exec(dagger.ContainerExecOpts{
				Args:                          []string{"yarn", "test"},
				ExperimentalPrivilegedNesting: true,
			}).
			ExitCode(ctx)
		return err
	})
}

// Generate re-generates the SDK API
func (t Nodejs) Generate(ctx context.Context) error {
	c, err := dagger.Connect(ctx, dagger.WithLogOutput(os.Stderr))
	if err != nil {
		return err
	}
	defer c.Close()

	return util.WithDevEngine(ctx, c, func(ctx context.Context, c *dagger.Client) error {
		generated, err := util.GoBase(c).
			WithMountedFile("/usr/local/bin/cloak", util.DaggerBinary(c)).
			Exec(dagger.ContainerExecOpts{
				Args:                          []string{"cloak", "client-gen", "--lang", "nodejs", "-o", nodejsGeneratedAPIPath},
				ExperimentalPrivilegedNesting: true,
			}).
			File(nodejsGeneratedAPIPath).
			Contents(ctx)
		if err != nil {
			return err
		}
		return os.WriteFile(nodejsGeneratedAPIPath, []byte(generated), 0o600)
	})
}

// Publish publishes the Node.js SDK
func (t Nodejs) Publish(ctx context.Context, tag string) error {
	c, err := dagger.Connect(ctx, dagger.WithLogOutput(os.Stderr))
	if err != nil {
		return err
	}
	defer c.Close()

	var (
		version     = strings.TrimPrefix(tag, "sdk/nodejs/v")
		tokenSecret = c.Host().EnvVariable("NPM_TOKEN").Secret()
	)
	token, err := tokenSecret.Plaintext(ctx)
	if err != nil {
		return err
	}

	if token == "" {
		return errors.New("NPM_TOKEN environment variable must be set")
	}

	build := nodeJsBase(c).Exec(dagger.ContainerExecOpts{
		Args: []string{"npm", "run", "build"},
	})

	// configure .npmrc
	npmrc := fmt.Sprintf(`//registry.npmjs.org/:_authToken=%s
registry=https://registry.npmjs.org/
always-auth=true`, token)
	if err := os.WriteFile("sdk/nodejs/.npmrc", []byte(npmrc), 0600); err != nil {
		return err
	}

	// set version & publish
	_, err = build.
		Exec(dagger.ContainerExecOpts{
			Args: []string{"npm", "version", version},
		}).
		Exec(dagger.ContainerExecOpts{
			Args: []string{"npm", "publish", "--access", "public"},
		}).
		ExitCode(ctx)

	return err
}

// Bump the Node.js SDK's Engine dependency
func (t Nodejs) Bump(ctx context.Context, version string) error {
	engineReference := fmt.Sprintf("// Code generated by dagger. DO NOT EDIT.\n"+
		"const DEFAULT_IMAGE_REF = %q;\n\n"+

		"export const DEFAULT_HOST = `docker-image://${DEFAULT_IMAGE_REF}`\n", version)

	return os.WriteFile("sdk/nodejs/provisioning/default.ts", []byte(engineReference), 0600)
}

func nodeJsBase(c *dagger.Client) *dagger.Container {
	workdir := c.Directory().WithDirectory("/", util.Repository(c).Directory("sdk/nodejs"))

	base := c.Container().
		// ⚠️  Keep this in sync with the engine version defined in package.json
		From("node:16-alpine").
		WithWorkdir("/workdir")

	deps := base.WithFS(
		base.
			FS().
			WithFile("/workdir/package.json", workdir.File("package.json")).
			WithFile("/workdir/yarn.lock", workdir.File("yarn.lock")),
	).
		Exec(dagger.ContainerExecOpts{
			Args: []string{"yarn", "install"},
		})

	src := deps.WithFS(
		deps.
			FS().
			WithDirectory("/workdir", workdir),
	)

	return src
}
