/*
 *    Copyright 2017 OICR
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package io.dockstore.webservice.helpers;

import static io.dockstore.webservice.Constants.SKIP_COMMIT_ID;

import com.google.common.base.Strings;
import io.dockstore.common.DescriptorLanguage;
import io.dockstore.common.SourceControl;
import io.dockstore.webservice.CustomWebApplicationException;
import io.dockstore.webservice.DockstoreWebserviceApplication;
import io.dockstore.webservice.core.Entry;
import io.dockstore.webservice.core.SourceFile;
import io.dockstore.webservice.core.Version;
import io.dockstore.webservice.core.Workflow;
import io.dockstore.webservice.core.WorkflowVersion;
import io.swagger.bitbucket.client.ApiClient;
import io.swagger.bitbucket.client.ApiException;
import io.swagger.bitbucket.client.Configuration;
import io.swagger.bitbucket.client.api.RefsApi;
import io.swagger.bitbucket.client.api.RepositoriesApi;
import io.swagger.bitbucket.client.model.Branch;
import io.swagger.bitbucket.client.model.PaginatedRefs;
import io.swagger.bitbucket.client.model.PaginatedRepositories;
import io.swagger.bitbucket.client.model.PaginatedTreeentries;
import io.swagger.bitbucket.client.model.Repository;
import io.swagger.bitbucket.client.model.Tag;
import jakarta.ws.rs.core.GenericType;
import jakarta.ws.rs.core.Response.Status;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author dyuen
 */
public class BitBucketSourceCodeRepo extends SourceCodeRepoInterface {
    /**
     * should use the java api, but I can't make heads or tails of the documentation
     * <a href="https://docs.atlassian.com/bitbucket-server/javadoc/5.11.1/api/reference/packages.html">...</a>
     */
    private static final String BITBUCKET_V2_API_URL = "https://api.bitbucket.org/2.0/";
    private static final String BITBUCKET_GIT_URL_PREFIX = "git@bitbucket.org:";
    private static final String BITBUCKET_GIT_URL_SUFFIX = ".git";

    private static final long ONE_MINUTE_IN_MS = 60_000L;

    private static final Logger LOG = LoggerFactory.getLogger(BitBucketSourceCodeRepo.class);
    private final ApiClient apiClient;

    /**
     * @param gitUsername           username that owns the bitbucket token
     * @param bitbucketTokenContent bitbucket token
     */
    public BitBucketSourceCodeRepo(String gitUsername, String bitbucketTokenContent) {
        this.gitUsername = gitUsername;

        apiClient = Configuration.getDefaultApiClient();
        apiClient.addDefaultHeader("Authorization", "Bearer " + bitbucketTokenContent);
    }

    @Override
    public String readFile(String repositoryId, String fileName, String reference) {
        if (fileName.startsWith("/")) {
            fileName = fileName.substring(1);
        }
        if (fileName.isEmpty()) {
            LOG.info(gitUsername + ": no file path provided for " + repositoryId);
            return null;
        }
        try {
            String fileContent = this
                .getArbitraryURL(BITBUCKET_V2_API_URL + "repositories/" + repositoryId + "/src/" + reference + '/' + fileName,
                    new GenericType<String>() {
                    });
            LOG.info(gitUsername + ": FOUND: {}", fileName);
            return fileContent;
        } catch (ApiException e) {
            LOG.error(gitUsername + ": ApiException on readFile " + fileName + " from repository " + repositoryId +  ":" + reference + ", " + e.getMessage(), e);
            return null;
        }
    }

    @Override
    public List<String> listFiles(String repositoryId, String pathToDirectory, String reference) {
        RepositoriesApi repositoriesApi = new RepositoriesApi(apiClient);
        try {
            List<String> files = new ArrayList<>();

            PaginatedTreeentries paginatedTreeentries = null;
            boolean rateLimited;
            do {
                rateLimited = false;
                try {
                    paginatedTreeentries = repositoriesApi
                        .repositoriesUsernameRepoSlugSrcNodePathGet(repositoryId.split("/")[0], reference, pathToDirectory,
                            repositoryId.split("/")[1], null, null, null);
                } catch (ApiException e) {
                    rateLimited = isRateLimited(false, e, "listing files");
                }
            } while (rateLimited);

            // TODO: this pagination pattern happens a lot with Bitbucket, a future exercise would clean this up
            while (paginatedTreeentries != null) {
                files.addAll(
                    paginatedTreeentries.getValues().stream().map(entry -> StringUtils.removeStart(entry.getPath(), pathToDirectory + "/"))
                        .toList());
                if (paginatedTreeentries.getNext() != null) {
                    paginatedTreeentries = getArbitraryURL(paginatedTreeentries.getNext(), new GenericType<PaginatedTreeentries>() {
                    });
                } else {
                    paginatedTreeentries = null;
                }
            }
            return files;
        } catch (ApiException e) {
            LOG.error(gitUsername + ": IOException on listFiles in " + pathToDirectory + " for repository " + repositoryId +  ":" + reference + ", " + e.getMessage(), e);
            return null;
        }
    }

    @Override
    public Map<String, String> getWorkflowGitUrl2RepositoryId() {
        RepositoriesApi repositoriesApi = new RepositoriesApi(apiClient);
        try {
            Map<String, String> collect = new HashMap<>();
            PaginatedRepositories contributor = null;
            boolean rateLimited;
            do {
                rateLimited = false;
                try {
                    contributor = repositoriesApi.repositoriesUsernameGet(gitUsername, "contributor");
                } catch (ApiException e) {
                    rateLimited = isRateLimited(false, e, "getting repositories");
                }
            } while (rateLimited);

            while (contributor != null) {
                collect.putAll(contributor.getValues().stream().collect(Collectors
                    .toMap(object -> BITBUCKET_GIT_URL_PREFIX + object.getFullName() + BITBUCKET_GIT_URL_SUFFIX, Repository::getFullName)));
                if (contributor.getNext() != null) {
                    contributor = getArbitraryURL(contributor.getNext(), new GenericType<PaginatedRepositories>() {
                    });
                } else {
                    contributor = null;
                }
            }
            return collect;
        } catch (ApiException e) {
            return this.handleGetWorkflowGitUrl2RepositoryIdError(e);
        }
    }

    @Override
    public String getName() {
        return "Bitbucket";
    }

    @Override
    public void setLicenseInformation(Entry entry, String gitRepository) {

    }

    /**
     * Gets arbitrary URLs that Bitbucket seems to use for pagination
     *
     * @param url full URL coming back from a self link in Bitbucket
     * @return the typed result
     * @throws ApiException swagger classes throw this exception
     */
    private <T> T getArbitraryURL(String url, GenericType<T> type) throws ApiException {
        String substring = url.substring(BITBUCKET_V2_API_URL.length() - 1);
        T result = null;
        boolean rateLimited;
        do {
            rateLimited = false;
            try {
                result = apiClient
                    .invokeAPI(substring, "GET", new ArrayList<>(), null, new HashMap<>(), new HashMap<>(), "application/json", "application/json",
                        new String[]{"api_key", "basic", "oauth2"}, type).getData();
            } catch (ApiException e) {
                rateLimited = isRateLimited(false, e, "getting pagination");
            }
        } while (rateLimited);

        return result;
    }

    /**
     * Uses Bitbucket API to grab a raw source file and return it; Return null if nothing found
     *
     * @param path         path to the file
     * @param repositoryId id in the format of "name/repo"
     * @param branch       branch name
     * @param type         type of file to create
     * @return source file
     */
    @Override
    public SourceFile getSourceFile(String path, String repositoryId, String branch, DescriptorLanguage.FileType type) {
        // TODO: should we even be creating a sourcefile before checking that it is valid?
        // I think it is fine since in the next part we just check that source file has content or not (no content is like null)
        String content = this.readFile(repositoryId, path, branch);

        if (Strings.isNullOrEmpty(content)) {
            return null;
        }
        // Grab content from found file
        // do not censor invalid versions to match github expected behaviour
        return SourceFile.limitedBuilder().type(type).content(content).paths(path).build();
    }
    @Override
    public void updateReferenceType(String repositoryId, Version version) {
        if (version.getReferenceType() != Version.ReferenceType.UNSET) {
            return;
        }
        String workspace = repositoryId.split("/")[0];
        String repoSlug = repositoryId.split("/")[1];
        String name = version.getReference();
        RefsApi refsApi = new RefsApi(apiClient);
        // There isn't exactly a single Bitbucket endpoint to get a version which then allows us to determine if it's a branch or tag.
        // This code checks two endpoints (branches and tags) to see if it belongs in which.
        boolean rateLimited;
        do {
            try {
                refsApi.repositoriesUsernameRepoSlugRefsBranchesNameGet(workspace, name, repoSlug);
                version.setReferenceType(Version.ReferenceType.BRANCH);
                return;
            } catch (ApiException e) {
                rateLimited = isRateLimited(false, e, "reading branches");
            }

            try {
                refsApi.repositoriesUsernameRepoSlugRefsTagsNameGet(workspace, name, repoSlug);
                version.setReferenceType(Version.ReferenceType.TAG);
                return;
            } catch (ApiException e) {
                // this is not so critical to warrant a http error code
                rateLimited = isRateLimited(rateLimited, e, "reading tags");
            }
        } while (rateLimited);

        throw new CustomWebApplicationException(name + " is not a Bitbucket branch or tag in " + repositoryId, HttpStatus.SC_INTERNAL_SERVER_ERROR);
    }

    /**
     * determine whether a request is rate-limited, output logging messages, and potentially sleep past rate-limiting
     * @param rateLimited
     * @param e
     * @param message
     * @return
     */
    private boolean isRateLimited(final boolean rateLimited, ApiException e, String message) {
        if (LOG.isWarnEnabled()) {
            LOG.warn("%s: apiexception on %s%s".formatted(gitUsername, message, e.getMessage()), e);
        }
        // this is not so critical to warrant a http error code
        boolean newlyRateLimited = false;
        if (e.getCode() == Status.TOO_MANY_REQUESTS.getStatusCode()) {
            newlyRateLimited = true;
            LOG.error("%s: rate-limited on %s".formatted(gitUsername, message));
            if (DockstoreWebserviceApplication.runningOnCircleCI()) {
                try {
                    LOG.error("We sleep");
                    Thread.sleep(ONE_MINUTE_IN_MS); // one minute, should be exponential back-off
                } catch (InterruptedException ex) {
                    LOG.error("Rate-limit wait interrupted!", e);
                    // Restore interrupted state...
                    Thread.currentThread().interrupt();
                }
            } else {
                throw new CustomWebApplicationException("rate limited by bitbucket, please wait for up to one hour for quota to re-generate", HttpStatus.SC_BAD_REQUEST);
            }
        }
        return rateLimited || newlyRateLimited;
    }

    @Override
    protected String getCommitID(String repositoryId, Version version) {
        RefsApi refsApi = new RefsApi(apiClient);
        try {
            Branch branch = refsApi.repositoriesUsernameRepoSlugRefsBranchesNameGet(repositoryId.split("/")[0], version.getReference(),
                repositoryId.split("/")[1]);
            if (branch != null) {
                return branch.getTarget().getHash();
            }
        } catch (ApiException ex) {
            LOG.error(gitUsername + ": apiexception on reading branch commitid", ex);
            // this is not so critical to warrant a http error code
        }

        try {
            Tag tag = refsApi.repositoriesUsernameRepoSlugRefsTagsNameGet(repositoryId.split("/")[0], version.getReference(),
                    repositoryId.split("/")[1]);
            if (tag != null) {
                return tag.getTarget().getHash();
            }
        } catch (ApiException ex) {
            LOG.error(gitUsername + ": apiexception on reading tag commitid", ex);
            // this is not so critical to warrant a http error code
        }
        return null;
    }

    @Override
    public Workflow initializeWorkflow(String repositoryId, Workflow workflow) {
        // Does this split not work if name has a slash?
        String[] id = repositoryId.split("/");
        String owner = id[0];
        String name = id[1];

        // Setup workflow
        workflow.setOrganization(owner);
        workflow.setRepository(name);
        workflow.setSourceControl(SourceControl.BITBUCKET);

        final String gitUrl = BITBUCKET_GIT_URL_PREFIX + repositoryId + BITBUCKET_GIT_URL_SUFFIX;
        workflow.setGitUrl(gitUrl);
        workflow.setLastUpdated(new Date());

        return workflow;
    }

    @Override
    public Workflow setupWorkflowVersions(String repositoryId, Workflow workflow, Optional<Workflow> existingWorkflow,
        Map<String, WorkflowVersion> existingDefaults, Optional<String> versionName, boolean hardRefresh) {
        RefsApi refsApi = new RefsApi(apiClient);
        try {

            PaginatedRefs paginatedRefs = null;
            boolean rateLimited;
            do {
                rateLimited = false;
                try {
                    paginatedRefs = refsApi.repositoriesUsernameRepoSlugRefsGet(repositoryId.split("/")[0], repositoryId.split("/")[1]);
                } catch (ApiException e) {
                    rateLimited = isRateLimited(false, e, "getting pagination");
                }
            } while (rateLimited);
            // this pagination structure is repetitive and should be refactored
            while (paginatedRefs != null) {
                paginatedRefs.getValues().forEach(ref -> {
                    final String branchName = ref.getName();
                    if (versionName.isEmpty() || Objects.equals(branchName, versionName.get())) {
                        WorkflowVersion version = new WorkflowVersion();
                        version.setName(branchName);
                        version.setReference(branchName);
                        final OffsetDateTime date = ref.getTarget().getDate();
                        final Date lastModifiedDate = Date.from(date.toInstant());
                        version.setLastModified(lastModifiedDate);
                        final String commitId = getCommitID(repositoryId, version);

                        if (toRefreshVersion(commitId, existingDefaults.get(branchName), hardRefresh)) {
                            LOG.info(gitUsername + ": Looking at Bitbucket reference: " + branchName);
                            version = initializeWorkflowVersion(branchName, existingWorkflow, existingDefaults);

                            version.setLastModified(lastModifiedDate);
                            String calculatedPath = version.getWorkflowPath();
                            // Now grab source files
                            DescriptorLanguage.FileType identifiedType = workflow.getFileType();
                            // TODO: No exceptions are caught here in the event of a failed call
                            SourceFile sourceFile = getSourceFile(calculatedPath, repositoryId, branchName, identifiedType);

                            // Use default test parameter file if either new version or existing version that hasn't been edited
                            createTestParameterFiles(workflow, repositoryId, branchName, version, identifiedType);
                            workflow.addWorkflowVersion(combineVersionAndSourcefile(repositoryId, sourceFile, workflow, identifiedType, version, existingDefaults));

                            version.setCommitID(getCommitID(repositoryId, version));

                            version = versionValidation(version, workflow, calculatedPath);
                            if (version != null) {
                                workflow.addWorkflowVersion(version);
                            }
                        } else {
                            // Version didn't change, but we don't want to delete
                            // Add a stub version with commit ID set to an ignore value so that the version isn't deleted
                            LOG.info(gitUsername + ": Skipping Bitbucket reference: " + branchName);
                            version.setCommitID(SKIP_COMMIT_ID);
                            workflow.addWorkflowVersion(version);
                        }
                    }
                });

                if (paginatedRefs.getNext() != null) {
                    paginatedRefs = getArbitraryURL(paginatedRefs.getNext(), new GenericType<PaginatedRefs>() {
                    });
                } else {
                    paginatedRefs = null;
                }
            }
        } catch (ApiException e) {
            LOG.error("Could not find Bitbucket repository " + repositoryId + " for user.", e);
            throw new CustomWebApplicationException("Could not reach Bitbucket", HttpStatus.SC_SERVICE_UNAVAILABLE);
        }
        return workflow;
    }

    @Override
    public String getRepositoryId(Entry entry) {
        String repositoryId;
        String giturl = entry.getGitUrl();
        Optional<Map<String, String>> gitMap = SourceCodeRepoFactory.parseGitUrl(giturl, Optional.of("bitbucket.org"));
        LOG.info(gitUsername + ": " + giturl);
        if (gitMap.isEmpty()) {
            LOG.info(gitUsername + ": Namespace and/or repository name could not be found from tool's giturl");
            return null;
        }
        repositoryId = gitMap.get().get(SourceCodeRepoFactory.GIT_URL_USER_KEY) + "/"
                + gitMap.get().get(SourceCodeRepoFactory.GIT_URL_REPOSITORY_KEY);
        return repositoryId;
    }

    @Override
    public String getMainBranch(Entry entry, String repositoryId) {
        // Is default version set?
        if (entry.getDefaultVersion() != null) {
            return getBranchNameFromDefaultVersion(entry);
        } else {
            // If default version is not set, need to find the main branch
            return getDefaultBranch(repositoryId);
        }
    }

    @Override
    public String getDefaultBranch(String repositoryId) {
        RepositoriesApi api = new RepositoriesApi(apiClient);
        Repository repository;
        try {
            repository = api.repositoriesUsernameRepoSlugGet(repositoryId.split("/")[0], repositoryId.split("/")[1]);
            if (repository != null && repository.getMainbranch() != null) {
                return repository.getMainbranch().getName();
            }
        } catch (ApiException e) {
            LOG.error("Unable to retrieve default branch for repository " + repositoryId, e);
            return null;
        }
        return null;
    }

    @Override
    public boolean checkSourceControlTokenValidity() {
        //TODO
        return true;
    }
}
