package io.dockstore.client.cli;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;

import io.dockstore.client.cli.BaseIT.TestStatus;
import io.dockstore.common.CommonTestUtilities;
import io.dockstore.common.MuteForSuccessfulTests;
import io.dockstore.openapi.client.ApiClient;
import io.dockstore.openapi.client.ApiException;
import io.dockstore.openapi.client.api.CloudInstancesApi;
import io.dockstore.openapi.client.api.UsersApi;
import io.dockstore.openapi.client.model.CloudInstance;
import io.dockstore.openapi.client.model.CloudInstance.PartnerEnum;
import io.dockstore.openapi.client.model.Language;
import io.dockstore.openapi.client.model.User;
import java.util.ArrayList;
import java.util.List;
import org.apache.http.HttpStatus;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import uk.org.webcompere.systemstubs.jupiter.SystemStub;
import uk.org.webcompere.systemstubs.jupiter.SystemStubsExtension;
import uk.org.webcompere.systemstubs.stream.SystemErr;
import uk.org.webcompere.systemstubs.stream.SystemOut;

@ExtendWith(SystemStubsExtension.class)
@ExtendWith(MuteForSuccessfulTests.class)
@ExtendWith(TestStatus.class)
class CloudInstanceIT extends BaseIT {
    public static final CloudInstance.PartnerEnum MODIFIED_MEMBER_PARTNER_1 = CloudInstance.PartnerEnum.DNA_STACK;
    public static final CloudInstance.PartnerEnum MEMBER_PARTNER_2 = CloudInstance.PartnerEnum.DNA_NEXUS;
    public static final CloudInstance.PartnerEnum MODIFIED_ADMIN_PARTNER_1 = CloudInstance.PartnerEnum.CGC;
    public static final CloudInstance.PartnerEnum ADMIN_PARTNER_2 = CloudInstance.PartnerEnum.ANVIL;
    public static final CloudInstance.PartnerEnum MEMBER_PARTNER_1 = CloudInstance.PartnerEnum.GALAXY;
    public static final CloudInstance.PartnerEnum ADMIN_PARTNER_1 = CloudInstance.PartnerEnum.TERRA;

    @SystemStub
    public final SystemOut systemOut = new SystemOut();
    @SystemStub
    public final SystemErr systemErr = new SystemErr();

    @BeforeEach
    @Override
    public void resetDBBetweenTests() throws Exception {
        CommonTestUtilities.addAdditionalToolsWithPrivate2(SUPPORT, false, testingPostgres);
    }

    @Test
    void cloudInstanceResourceTest() {
        ApiClient adminApiClient = getOpenAPIWebClient(ADMIN_USERNAME, testingPostgres);
        ApiClient memberApiClient = getOpenAPIWebClient(OTHER_USERNAME, testingPostgres);
        ApiClient anonymousApiClient = getAnonymousOpenAPIWebClient();
        CloudInstancesApi adminCloudInstancesApi = new CloudInstancesApi(adminApiClient);
        CloudInstancesApi memberCloudInstancesApi = new CloudInstancesApi(memberApiClient);
        CloudInstancesApi anonymousCloudInstancesApi = new CloudInstancesApi(anonymousApiClient);
        List<CloudInstance> adminCloudInstances = adminCloudInstancesApi.getCloudInstances();
        List<CloudInstance> memberCloudInstances = memberCloudInstancesApi.getCloudInstances();
        List<CloudInstance> anonymousCloudInstances = anonymousCloudInstancesApi.getCloudInstances();
        assertEquals(0, adminCloudInstances.size());
        assertEquals(0, memberCloudInstances.size());
        assertEquals(0, anonymousCloudInstances.size());
        CloudInstance newCloudInstance = new CloudInstance();
        // This should not do anything
        Long ignoredId = 9001L;
        newCloudInstance.setId(ignoredId);
        newCloudInstance.setPartner(CloudInstance.PartnerEnum.DNA_STACK);
        newCloudInstance.setUrl("www.google.ca");
        newCloudInstance.setDisplayName("google.ca");
        newCloudInstance.setSupportsFileImports(null);
        newCloudInstance.setSupportsHttpImports(null);
        newCloudInstance.setSupportedLanguages(new ArrayList<>());
        // testing out languages
        Language language = new Language();
        language.setLanguage(Language.LanguageEnum.WDL);
        language.setVersion("draft-1.0");
        newCloudInstance.getSupportedLanguages().add(language);
        try {
            anonymousCloudInstancesApi.postCloudInstance(newCloudInstance);
            fail("Only admins can create a new cloud instance");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }

        adminCloudInstancesApi.postCloudInstance(newCloudInstance);

        newCloudInstance.setSupportsFileImports(true);
        try {
            adminCloudInstancesApi.postCloudInstance(newCloudInstance);
            fail("Cannot create a new global launch with partner with the same URL even if slightly different");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_CONFLICT, e.getCode());
        }

        adminCloudInstances = adminCloudInstancesApi.getCloudInstances();
        memberCloudInstances = memberCloudInstancesApi.getCloudInstances();
        anonymousCloudInstances = anonymousCloudInstancesApi.getCloudInstances();
        assertEquals(1, adminCloudInstances.size());
        assertEquals(1, memberCloudInstances.size());
        assertEquals(1, anonymousCloudInstances.size());
        Long dnaNexusId = anonymousCloudInstances.get(0).getId();
        assertNotEquals(ignoredId, dnaNexusId, "Should have ignored the ID passed in");
        newCloudInstance.setPartner(CloudInstance.PartnerEnum.DNA_NEXUS);
        newCloudInstance.setUrl("www.google.com");
        newCloudInstance.setDisplayName("google.com");
        adminCloudInstancesApi.postCloudInstance(newCloudInstance);
        adminCloudInstances = adminCloudInstancesApi.getCloudInstances();
        memberCloudInstances = memberCloudInstancesApi.getCloudInstances();
        anonymousCloudInstances = anonymousCloudInstancesApi.getCloudInstances();
        assertEquals(2, adminCloudInstances.size());
        assertEquals(2, memberCloudInstances.size());
        assertEquals(2, anonymousCloudInstances.size());
        try {
            anonymousCloudInstancesApi.deleteCloudInstance(dnaNexusId);
            fail("Only admins can create a new cloud instance");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }

        try {
            memberCloudInstancesApi.deleteCloudInstance(dnaNexusId);
            fail("Only admins can create a new cloud instance");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }
        adminCloudInstancesApi.deleteCloudInstance(dnaNexusId);
        adminCloudInstances = adminCloudInstancesApi.getCloudInstances();
        memberCloudInstances = memberCloudInstancesApi.getCloudInstances();
        anonymousCloudInstances = anonymousCloudInstancesApi.getCloudInstances();
        assertEquals(1, adminCloudInstances.size());
        assertEquals(1, memberCloudInstances.size());
        assertEquals(1, anonymousCloudInstances.size());
        assertEquals(PartnerEnum.DNA_NEXUS, anonymousCloudInstances.get(0).getPartner(), "The DNAstack cloud instance should be deleted, not DNAnexus");

        testCloudInstancesInUserResource(adminApiClient, memberApiClient, anonymousApiClient, newCloudInstance);

        List<CloudInstance> cloudInstances = anonymousCloudInstancesApi.getCloudInstances();
        assertEquals(1, cloudInstances.size(), "After all the user cloud instance tests, public cloud instances should remain unchanged");
    }

    private void testCloudInstancesInUserResource(ApiClient adminApiClient, ApiClient memberApiClient, ApiClient anonymousApiClient,
            CloudInstance newCloudInstance) {
        UsersApi adminUsersApi = new UsersApi(adminApiClient);
        UsersApi memberUsersApi = new UsersApi(memberApiClient);
        UsersApi anonymousUsersApi = new UsersApi(anonymousApiClient);

        // Check get works
        User adminUser = adminUsersApi.getUser();
        Long adminUserId = adminUser.getId();
        List<CloudInstance> adminUserCloudInstances = adminUsersApi.getUserCloudInstances(adminUserId);
        assertEquals(0, adminUserCloudInstances.size());
        User memberUser = memberUsersApi.getUser();
        Long memberUserId = memberUser.getId();
        List<CloudInstance> memberUserCloudInstances = memberUsersApi.getUserCloudInstances(memberUserId);
        assertEquals(0, memberUserCloudInstances.size());
        anonymousUserCannotDoAnything(anonymousUsersApi);

        try {
            memberUsersApi.getUserCloudInstances(adminUserId);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }

        try {
            adminUsersApi.getUserCloudInstances(memberUserId);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }

        // Check post works
        newCloudInstance.setPartner(ADMIN_PARTNER_1);
        adminUsersApi.postUserCloudInstance(newCloudInstance, adminUserId);
        newCloudInstance.setPartner(MEMBER_PARTNER_1);
        memberUsersApi.postUserCloudInstance(newCloudInstance, memberUserId);
        newCloudInstance.setPartner(ADMIN_PARTNER_2);
        adminUsersApi.postUserCloudInstance(newCloudInstance, adminUserId);
        newCloudInstance.setPartner(MEMBER_PARTNER_2);
        memberUsersApi.postUserCloudInstance(newCloudInstance, memberUserId);

        newCloudInstance.setSupportsFileImports(true);
        try {
            memberUsersApi.postUserCloudInstance(newCloudInstance, memberUserId);
            fail("Cannot create a new user launch with partner with the same URL even if slightly different");
        } catch (ApiException e) {
            assertTrue(e.getMessage().contains("constraint"));
            //TODO: catch and return a proper error code
            //Assert.assertEquals(HttpStatus.SC_BAD_REQUEST, e.getCode());
        }

        memberUserCloudInstances = memberUsersApi.getUserCloudInstances(memberUserId);
        adminUserCloudInstances = adminUsersApi.getUserCloudInstances(adminUserId);
        assertEquals(2, memberUserCloudInstances.size());
        Long memberPartner1Id = memberUserCloudInstances.stream()
                .filter(cloudInstance -> cloudInstance.getPartner().equals(MEMBER_PARTNER_1)).findFirst().get().getId();
        Long memberPartner2Id = memberUserCloudInstances.stream()
                .filter(cloudInstance -> cloudInstance.getPartner().equals(MEMBER_PARTNER_2)).findFirst().get().getId();
        assertEquals(2, adminUserCloudInstances.size());
        Long adminPartner1Id = adminUserCloudInstances.stream().filter(cloudInstance -> cloudInstance.getPartner().equals(ADMIN_PARTNER_1))
                .findFirst().get().getId();
        Long adminPartner2Id = adminUserCloudInstances.stream().filter(cloudInstance -> cloudInstance.getPartner().equals(ADMIN_PARTNER_2))
                .findFirst().get().getId();
        try {
            memberUsersApi.postUserCloudInstance(newCloudInstance, adminUserId);
            fail("Should not be create a cloud instances on a different user");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }

        // Check put works
        newCloudInstance.setPartner(MODIFIED_MEMBER_PARTNER_1);
        memberUsersApi.putUserCloudInstance(newCloudInstance, memberUserId, memberPartner1Id);
        memberUserCloudInstances = memberUsersApi.getUserCloudInstances(memberUserId);
        newCloudInstance.setPartner(MODIFIED_ADMIN_PARTNER_1);
        adminUsersApi.putUserCloudInstance(newCloudInstance, adminUserId, adminPartner1Id);
        adminUserCloudInstances = adminUsersApi.getUserCloudInstances(adminUserId);
        try {
            memberUsersApi.putUserCloudInstance(newCloudInstance, adminUserId, memberPartner1Id);
            fail("Should not be update a cloud instances on a different user");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }
        try {
            memberUsersApi.putUserCloudInstance(newCloudInstance, memberUserId, 9001L);
            fail("Should not be update a cloud instances that doesn't exist");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_NOT_FOUND, e.getCode());
        }
        assertEquals(2, memberUserCloudInstances.size());
        memberPartner1Id = memberUserCloudInstances.stream()
                .filter(cloudInstance -> cloudInstance.getPartner().equals(MODIFIED_MEMBER_PARTNER_1)).findFirst().get().getId();
        memberPartner2Id = memberUserCloudInstances.stream().filter(cloudInstance -> cloudInstance.getPartner().equals(MEMBER_PARTNER_2))
                .findFirst().get().getId();
        assertEquals(2, adminUserCloudInstances.size());
        adminPartner1Id = adminUserCloudInstances.stream()
                .filter(cloudInstance -> cloudInstance.getPartner().equals(MODIFIED_ADMIN_PARTNER_1)).findFirst().get().getId();
        adminPartner2Id = adminUserCloudInstances.stream().filter(cloudInstance -> cloudInstance.getPartner().equals(ADMIN_PARTNER_2))
                .findFirst().get().getId();

        // Check delete works
        adminUsersApi.deleteUserCloudInstance(adminUserId, adminPartner1Id);
        memberUsersApi.deleteUserCloudInstance(memberUserId, memberPartner1Id);
        try {
            memberUsersApi.deleteUserCloudInstance(adminUserId, adminPartner1Id);
            fail("Should not be delete a cloud instances on a different user");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_FORBIDDEN, e.getCode());
        }
        try {
            memberUsersApi.deleteUserCloudInstance(memberUserId, 9001L);
            fail("Should not be delete a cloud instances that doesn't exist");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_NOT_FOUND, e.getCode());
        }
        adminUserCloudInstances = adminUsersApi.getUserCloudInstances(adminUserId);
        assertEquals(1, adminUserCloudInstances.size());
        assertEquals(ADMIN_PARTNER_2, adminUserCloudInstances.get(0).getPartner());
        assertEquals(adminPartner2Id, adminUserCloudInstances.get(0).getId());
        memberUserCloudInstances = memberUsersApi.getUserCloudInstances(memberUserId);
        assertEquals(1, memberUserCloudInstances.size());
        assertEquals(MEMBER_PARTNER_2, memberUserCloudInstances.get(0).getPartner());
        assertEquals(memberPartner2Id, memberUserCloudInstances.get(0).getId());
    }

    private void anonymousUserCannotDoAnything(UsersApi anonymousUsersApi) {
        try {
            anonymousUsersApi.getUserCloudInstances(1L);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }
        try {
            anonymousUsersApi.postUserCloudInstance(new CloudInstance(), 1L);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }
        try {
            anonymousUsersApi.putUserCloudInstance(new CloudInstance(), 1L, 1L);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }
        try {
            anonymousUsersApi.deleteUserCloudInstance(1L, 1L);
            fail("Should not be able to get a different user's cloud instances");
        } catch (ApiException e) {
            assertEquals(HttpStatus.SC_UNAUTHORIZED, e.getCode());
        }
    }
}
