# Null-Safe Clang: An experimental C/C++ compiler

[![Test Null-Safety](https://github.com/cs01/llvm-project/actions/workflows/test-null-safety.yml/badge.svg)](https://github.com/cs01/llvm-project/actions/workflows/test-null-safety.yml)

An experimental Clang fork that adds flow-sensitive null safety to C and C++, inspired by modern languages like TypeScript, Kotlin, and Rust.

**Try it online:** [Interactive Playground](https://cs01.github.io/llvm-project/) - See null-safety warnings in real-time in your browser!

## What This Adds

This compiler adds two key features to prevent null pointer crashes:

1. Nullable-by-default pointers - All pointers are assumed nullable unless explicitly marked `_Nonnull`
2. Type narrowing - The compiler tracks when you've null-checked a pointer and knows it's safe to use

In standard C/C++, pointers have no nullability information and the compiler can't tell if a pointer might be null. This compiler treats all unmarked pointers as nullable by default and uses flow-sensitive analysis to track null checks. When you write `if (p)`, the type system understands `p` is non-null in that branch, just like TypeScript, Kotlin, and Rust. This catches null pointer dereferences at compile time instead of crashing at runtime.

## Example

```c
void process(int* data) {
    if (data) {
        *data = 42;        // ✓ OK - strict nullability knows data is non-null here
    }
}

void unsafe(int* data) {
    *data = 42;            // ⚠️  Warning - data might be null!
}
```

Standard Clang/GCC: Both functions compile without warnings.
This fork: The `unsafe` function warns you about the potential null dereference.

This experimental fork of Clang adds flow-sensitive nullability analysis while remaining 100% compatible with standard C. It includes all of Clang's features plus enhanced nullability checking in both the compiler and the `clangd` language server.

By default, strict nullability is enabled and issues warnings. You can promote warnings to errors with `-Werror=nullability`, or disable the feature entirely with `-fno-strict-nullability`.

## Installation

### Quick Install (Linux/macOS)

Download and extract the latest release for your platform:

```bash
# Auto-detect your platform and install
curl -fsSL https://raw.githubusercontent.com/cs01/llvm-project/null-safe-c-dev/install.sh | sh

# Or manually download:
# Linux x86_64:
curl -L https://github.com/cs01/llvm-project/releases/latest/download/clang-nullsafe-linux-x86_64.tar.gz | tar xz

# macOS (Intel/Apple Silicon):
curl -L https://github.com/cs01/llvm-project/releases/latest/download/clang-nullsafe-macos-universal.tar.gz | tar xz

# Add to PATH
export PATH="$PWD/bin:$PATH"
clang --version
```

### Windows

Download the Windows release:
```bash
curl -L https://github.com/cs01/llvm-project/releases/latest/download/clang-nullsafe-windows-x86_64.tar.gz -o clang-nullsafe.tar.gz
tar -xzf clang-nullsafe.tar.gz
# Add bin\ directory to your PATH
clang --version
```

### What's Included

Each release includes:
- **`clang`** - The Null-Safe C compiler with flow-sensitive null checking
- **`clangd`** - Language server for IDE integration (VSCode, vim, Neovim, Emacs, etc.)

### IDE Integration

Once installed, configure your editor to use the null-safe `clangd`:

**VSCode:**
```json
// settings.json
{
  "clangd.path": "/path/to/null-safe-clang/bin/clangd"
}
```

**Neovim/vim:**
```lua
require('lspconfig').clangd.setup({
  cmd = { '/path/to/null-safe-clang/bin/clangd' }
})
```

This gives you real-time null-safety warnings as you type!


## Memory Safety in General

Null pointer dereferences are just one category of memory safety bugs. Here's how different approaches compare:

### What Gets Fixed

| Safety Issue            | Standard C | **Null-Safe Clang** (null checking) |
|-------------------------|------------|:-----------------------------------:|
| Null pointer dereferences | ❌ Unsafe | **✅ Fixed**                        |
| Buffer overflows        | ❌ Unsafe  | ❌ Unsafe                            |
| Use-after-free          | ❌ Unsafe  | ❌ Unsafe                            |
| Double-free             | ❌ Unsafe  | ❌ Unsafe                            |
| Uninitialized memory    | ❌ Unsafe  | ❌ Unsafe                            |


### Why you still might want to try this

While Null-Safe Clang doesn't solve all memory safety issues, null pointer dereferences are a significant problem:

- Many memory safety bugs involve null pointer dereferences
- Easier to adopt than rewriting in Rust (100% compatible with existing C code)
- Complements other efforts (combine with `-fbounds-safety` for buffer safety)
- Incremental deployment (warnings by default, can enable per-file)


## Usage

Basic usage (warnings enabled by default):
```bash
clang mycode.c                          # Warnings for nullable dereferences
clang -Werror=nullability mycode.c      # Treat nullability issues as errors
clang -fno-strict-nullability mycode.c  # Turn off nullability checking
```

Gradual adoption (per-file or per-function):
```c
// Disable warnings for specific files
#pragma clang diagnostic ignored "-Wnullability"

// Or per-function
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wnullability"
void legacy_function(int* p) { ... }
#pragma clang diagnostic pop
```

## Features

- Nullable-by-default: All pointers are `_Nullable` unless marked `_Nonnull`
- Flow-sensitive narrowing: `if (p)` proves `p` is non-null in that scope
- Early-exit patterns: Understands `return`, `goto`, `break`, `continue`
- Pointer arithmetic: `q = p + 1` preserves narrowing from `p`
- Type checking through function calls, returns, and assignments
- Works with Typedefs
- Assumes functions have side effects (use `__attribute__((pure))` or `__attribute__((const))` to preserve narrowing)
- Null-safe headers: Annotated C standard library in `clang/nullsafe-headers/`
- IDE integration: `clangd` built from this fork has the same logic and warnings as clang

## How It Works

Strict nullability adds flow-sensitive analysis to Clang's semantic analyzer. When you write `if (p)`, the compiler tracks that `p` is non-null within that branch—just like TypeScript, Swift, or Kotlin do. The difference is we add this to C without changing the language itself.

### Function Call Invalidation

By default, function calls invalidate narrowing because they may have side effects:

```c
void some_function(void);

void example(int* p) {
    if (p) {
        *p = 1;           // OK - p is narrowed to non-null
        some_function();  // Invalidates narrowing
        *p = 2;           // Warning: p is nullable again
    }
}
```

This is conservative but safe—functions can modify global state or escaped pointers.

Preserve narrowing with pure functions:

Mark side-effect-free functions with `__attribute__((pure))` or `__attribute__((const))`:

```c
int tolower(int c) __attribute__((const));  // No side effects

void example(char* p) {
    if (p) {
        tolower(*p);  // Pure function - doesn't invalidate
        *p = 'x';     // OK - p is still narrowed
    }
}
```

The difference:
- `__attribute__((const))`: Only uses its arguments (e.g., `tolower`, `abs`)
- `__attribute__((pure))`: May read globals but doesn't modify anything (e.g., `strlen`)

Many standard library functions already have these attributes in GNU libc headers, so strict nullability automatically recognizes them.

## Null-Safe C Standard Library

Nullability-annotated headers for `string.h`, `stdlib.h`, and `stdio.h` are available in `clang/nullsafe-headers/`. These tell the compiler which functions return nullable pointers and which parameters must be non-null.

```bash
# Compile with null-safe headers
clang -Iclang/nullsafe-headers/include mycode.c
```

```c
#include "string.h"
#include "stdlib.h"

void example(const char* input) {
    if (!input) return;
    char* copy = malloc(strlen(input) + 1);  // malloc returns _Nullable
    if (copy) {
        strcpy(copy, input);  // Both parameters narrowed to non-null
        free(copy);
    }
}
```

See [`clang/nullsafe-headers/README.md`](clang/nullsafe-headers/README.md) for details.
