// Copyright 2018-2024 CERN
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// In applying this license, CERN does not waive the privileges and immunities
// granted to it by virtue of its status as an Intergovernmental Organization
// or submit itself to any jurisdiction.

package shares

import (
	"context"
	"net/http"
	"path/filepath"

	providerv1beta1 "github.com/cs3org/go-cs3apis/cs3/app/provider/v1beta1"
	gatewayv1beta1 "github.com/cs3org/go-cs3apis/cs3/gateway/v1beta1"
	userpb "github.com/cs3org/go-cs3apis/cs3/identity/user/v1beta1"
	invitepb "github.com/cs3org/go-cs3apis/cs3/ocm/invite/v1beta1"
	providerpb "github.com/cs3org/go-cs3apis/cs3/ocm/provider/v1beta1"
	rpc "github.com/cs3org/go-cs3apis/cs3/rpc/v1beta1"
	ocm "github.com/cs3org/go-cs3apis/cs3/sharing/ocm/v1beta1"
	ocmd "github.com/cs3org/reva/v3/internal/http/services/opencloudmesh/ocmd"
	provider "github.com/cs3org/go-cs3apis/cs3/storage/provider/v1beta1"
	types "github.com/cs3org/go-cs3apis/cs3/types/v1beta1"
	"github.com/cs3org/reva/v3/internal/http/services/owncloud/ocs/conversions"
	"github.com/cs3org/reva/v3/internal/http/services/owncloud/ocs/response"
	"github.com/cs3org/reva/v3/pkg/appctx"
	"github.com/cs3org/reva/v3/pkg/ocm/share"
	"github.com/cs3org/reva/v3/pkg/rgrpc/todo/pool"
	"github.com/cs3org/reva/v3/pkg/utils"
	"github.com/go-chi/chi/v5"
	"github.com/pkg/errors"
)

func (h *Handler) createFederatedCloudShare(w http.ResponseWriter, r *http.Request, resource *provider.ResourceInfo, role *conversions.Role, roleVal []byte) {
	ctx := r.Context()

	c, err := pool.GetGatewayServiceClient(pool.Endpoint(h.gatewayAddr))
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error getting grpc gateway client", err)
		return
	}

	shareWithUser, shareWithProvider := r.FormValue("shareWithUser"), r.FormValue("shareWithProvider")
	if shareWithUser == "" || shareWithProvider == "" {
		response.WriteOCSError(w, r, response.MetaBadRequest.StatusCode, "missing shareWith parameters", nil)
		return
	}

	providerInfoResp, err := c.GetInfoByDomain(ctx, &providerpb.GetInfoByDomainRequest{
		Domain: shareWithProvider,
	})
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error sending a grpc get invite by domain info request", err)
		return
	}

	if providerInfoResp.Status.Code != rpc.Code_CODE_OK {
		// return proper error
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error from provider info response", errors.New(providerInfoResp.Status.Message))
		return
	}

	user := appctx.ContextMustGetUser(ctx)
	d, err := utils.MarshalProtoV1ToJSON(user.Id)
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, err.Error(), errors.New(providerInfoResp.Status.Message))
		return
	}

	o := &types.Opaque{
		Map: map[string]*types.OpaqueEntry{
			"user-filter": {
				Decoder: "json",
				Value:   d,
			},
		},
	}

	remoteUserRes, err := c.GetAcceptedUser(ctx, &invitepb.GetAcceptedUserRequest{
		RemoteUserId: &userpb.UserId{OpaqueId: shareWithUser, Idp: shareWithProvider, Type: userpb.UserType_USER_TYPE_FEDERATED},
		Opaque:       o,
	})
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error searching recipient", err)
		return
	}
	if remoteUserRes.Status.Code != rpc.Code_CODE_OK {
		response.WriteOCSError(w, r, response.MetaNotFound.StatusCode, "user not found", err)
		return
	}

	createShareResponse, err := c.CreateOCMShare(ctx, &ocm.CreateOCMShareRequest{
		ResourceId: resource.Id,
		Grantee: &provider.Grantee{
			Type: provider.GranteeType_GRANTEE_TYPE_USER,
			Id: &provider.Grantee_UserId{
				UserId: remoteUserRes.RemoteUser.Id,
			},
		},
		RecipientMeshProvider: providerInfoResp.ProviderInfo,
		AccessMethods: []*ocm.AccessMethod{
			share.NewWebDavAccessMethod(role.CS3ResourcePermissions(), []string{}),
			share.NewWebappAccessMethod(getViewModeFromRole(role)),
		},
	})
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error sending a grpc create ocm share request", err)
		return
	}
	if createShareResponse.Status.Code != rpc.Code_CODE_OK {
		if createShareResponse.Status.Code == rpc.Code_CODE_NOT_FOUND {
			response.WriteOCSError(w, r, response.MetaNotFound.StatusCode, "not found", nil)
			return
		}
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "grpc create ocm share request failed", err)
		return
	}

	s := createShareResponse.Share
	data, err := conversions.OCMShare2ShareData(s)
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error converting share", err)
		return
	}
	h.mapUserIdsFederatedShare(ctx, c, data)

	info, status, err := h.getResourceInfoByID(ctx, c, s.ResourceId)
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error statting resource id", err)
		return
	}
	if status.Code != rpc.Code_CODE_OK {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error statting resource id", errors.New(status.Message))
		return
	}

	err = h.addFileInfo(ctx, data, info)
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error statting resource id", err)
		return
	}

	response.WriteOCSSuccess(w, r, data)
}

func getViewModeFromRole(role *conversions.Role) providerv1beta1.ViewMode {
	switch role.Name {
	case conversions.RoleViewer:
		return providerv1beta1.ViewMode_VIEW_MODE_READ_ONLY
	case conversions.RoleEditor:
		return providerv1beta1.ViewMode_VIEW_MODE_READ_WRITE
	}
	return providerv1beta1.ViewMode_VIEW_MODE_INVALID
}

// GetFederatedShare handles GET requests on /apps/files_sharing/api/v1/shares/remote_shares/{shareid}.
func (h *Handler) GetFederatedShare(w http.ResponseWriter, r *http.Request) {
	// TODO: Implement response with HAL schemating
	ctx := r.Context()

	shareID := chi.URLParam(r, "shareid")
	gatewayClient, err := pool.GetGatewayServiceClient(pool.Endpoint(h.gatewayAddr))
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error getting grpc gateway client", err)
		return
	}

	listOCMSharesRequest := &ocm.GetOCMShareRequest{
		Ref: &ocm.ShareReference{
			Spec: &ocm.ShareReference_Id{
				Id: &ocm.ShareId{
					OpaqueId: shareID,
				},
			},
		},
	}
	ocmShareResponse, err := gatewayClient.GetOCMShare(ctx, listOCMSharesRequest)
	if err != nil {
		response.WriteOCSError(w, r, response.MetaServerError.StatusCode, "error sending a grpc get ocm share request", err)
		return
	}

	share := ocmShareResponse.GetShare()
	if share == nil {
		response.WriteOCSError(w, r, response.MetaNotFound.StatusCode, "share not found", err)
		return
	}
	response.WriteOCSSuccess(w, r, share)
}

// ListFederatedShares handles GET requests on /apps/files_sharing/api/v1/shares/remote_shares.
func (h *Handler) ListFederatedShares(w http.ResponseWriter, r *http.Request) {
	// TODO Implement pagination.
	// TODO Implement response with HAL schemating
}

func (h *Handler) listReceivedFederatedShares(ctx context.Context, gw gatewayv1beta1.GatewayAPIClient, state ocm.ShareState) ([]*conversions.ShareData, error) {
	listRes, err := gw.ListReceivedOCMShares(ctx, &ocm.ListReceivedOCMSharesRequest{})
	if err != nil {
		return nil, err
	}

	shares := []*conversions.ShareData{}
	for _, s := range listRes.Shares {
		if state != ocsStateUnknown && s.State != state {
			continue
		}
		sd, err := conversions.ReceivedOCMShare2ShareData(s, h.ocmLocalMount(s))
		if err != nil {
			continue
		}
		h.mapUserIdsReceivedFederatedShare(ctx, gw, sd)
		sd.State = mapOCMState(s.State)
		shares = append(shares, sd)
	}
	return shares, nil
}

func (h *Handler) ocmLocalMount(share *ocm.ReceivedShare) string {
	return filepath.Join("/", h.ocmMountPoint, share.Id.OpaqueId)
}

func (h *Handler) mapUserIdsReceivedFederatedShare(ctx context.Context, gw gatewayv1beta1.GatewayAPIClient, sd *conversions.ShareData) {
	if sd.ShareWith != "" {
		user := h.mustGetIdentifiers(ctx, gw, sd.ShareWith, false)
		sd.ShareWith = user.Username
		sd.ShareWithDisplayname = user.DisplayName
	}

	if sd.UIDOwner != "" {
		user := h.mustGetRemoteUser(ctx, gw, sd.UIDOwner)
		sd.DisplaynameOwner = user.DisplayName
	}

	if sd.UIDFileOwner != "" {
		user := h.mustGetRemoteUser(ctx, gw, sd.UIDFileOwner)
		sd.DisplaynameFileOwner = user.DisplayName
	}
}

func (h *Handler) mapUserIdsFederatedShare(ctx context.Context, gw gatewayv1beta1.GatewayAPIClient, sd *conversions.ShareData) {
	if sd.ShareWith != "" {
		user := h.mustGetRemoteUser(ctx, gw, sd.ShareWith)
		sd.ShareWith = user.Username
		sd.ShareWithDisplayname = user.DisplayName
	}

	if sd.UIDOwner != "" {
		user := h.mustGetIdentifiers(ctx, gw, sd.UIDOwner, false)
		sd.DisplaynameOwner = user.DisplayName
	}

	if sd.UIDFileOwner != "" {
		user := h.mustGetIdentifiers(ctx, gw, sd.UIDFileOwner, false)
		sd.DisplaynameFileOwner = user.DisplayName
	}
}

func (h *Handler) mustGetRemoteUser(ctx context.Context, gw gatewayv1beta1.GatewayAPIClient, ocmAddress string) *userIdentifiers {
	user := appctx.ContextMustGetUser(ctx)
	d, err := utils.MarshalProtoV1ToJSON(user.Id)
	if err != nil {
		return &userIdentifiers{}
	}

	o := &types.Opaque{
		Map: map[string]*types.OpaqueEntry{
			"user-filter": {
				Decoder: "json",
				Value:   d,
			},
		},
	}

	remoteUserId, err := ocmd.GetUserIdFromOCMAddress(ocmAddress)
	if err != nil {
		return &userIdentifiers{}
	}
	userRes, err := gw.GetAcceptedUser(ctx, &invitepb.GetAcceptedUserRequest{
		RemoteUserId: remoteUserId,
		Opaque: o,
	})
	if err != nil {
		return &userIdentifiers{}
	}
	if userRes.Status.Code != rpc.Code_CODE_OK {
		return &userIdentifiers{}
	}

	remote := userRes.RemoteUser
	return &userIdentifiers{
		DisplayName: remote.DisplayName,
		Username:    remote.Username,
		Mail:        remote.Mail,
	}
}

func (h *Handler) listOutcomingFederatedShares(ctx context.Context, gw gatewayv1beta1.GatewayAPIClient, filters []*ocm.ListOCMSharesRequest_Filter) ([]*conversions.ShareData, error) {
	listRes, err := gw.ListOCMShares(ctx, &ocm.ListOCMSharesRequest{
		Filters: filters,
	})
	if err != nil {
		return nil, err
	}

	shares := []*conversions.ShareData{}
	for _, s := range listRes.Shares {
		sd, err := conversions.OCMShare2ShareData(s)
		if err != nil {
			continue
		}
		h.mapUserIdsFederatedShare(ctx, gw, sd)

		info, status, err := h.getResourceInfoByID(ctx, gw, s.ResourceId)
		if err != nil {
			return nil, err
		}

		if status.Code != rpc.Code_CODE_OK {
			return nil, err
		}

		err = h.addFileInfo(ctx, sd, info)
		if err != nil {
			return nil, err
		}
		shares = append(shares, sd)
	}
	return shares, nil
}
