// Copyright 2018-2024 CERN
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// In applying this license, CERN does not waive the privileges and immunities
// granted to it by virtue of its status as an Intergovernmental Organization
// or submit itself to any jurisdiction.

package appctx

import (
	"context"
	"strings"

	ua "github.com/mileusna/useragent"
	"google.golang.org/grpc/metadata"
)

// UserAgentHeader is the header used for the user agent.
const (
	UserAgentHeader = "x-user-agent"

	WebUserAgent     = "web"
	GrpcUserAgent    = "grpc"
	MobileUserAgent  = "mobile"
	DesktopUserAgent = "desktop"
)

// ContextGetUserAgent returns the user agent if set in the given context.
// see https://github.com/grpc/grpc-go/issues/1100
func ContextGetUserAgent(ctx context.Context) (*ua.UserAgent, bool) {
	if userAgentStr, ok := ContextGetUserAgentString(ctx); ok {
		userAgent := ua.Parse(userAgentStr)
		return &userAgent, true
	}
	return nil, false
}

// ContextGetUserAgentString returns the user agent string if set in the given context.
func ContextGetUserAgentString(ctx context.Context) (string, bool) {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		return "", false
	}
	userAgentLst, ok := md[UserAgentHeader]
	if !ok {
		userAgentLst, ok = md["user-agent"]
		if !ok {
			return "", false
		}
	}
	if len(userAgentLst) == 0 {
		return "", false
	}
	return userAgentLst[0], true
}

// ContextGetUserAgentCategory returns the category of the user agent
// (i.e. if it is a web, mobile, desktop or grpc user agent).
func ContextGetUserAgentCategory(ctx context.Context) (string, bool) {
	agent, ok := ContextGetUserAgent(ctx)
	if !ok {
		return "", false
	}
	switch {
	case isWeb(agent):
		return WebUserAgent, true
	case isMobile(agent):
		return MobileUserAgent, true
	case isDesktop(agent):
		return DesktopUserAgent, true
	case isGRPC(agent):
		return GrpcUserAgent, true
	default:
		return "", false
	}
}

func isWeb(ua *ua.UserAgent) bool {
	return ua.IsChrome() || ua.IsEdge() || ua.IsFirefox() || ua.IsSafari() ||
		ua.IsInternetExplorer() || ua.IsOpera() || ua.IsOperaMini()
}

// isMobile returns true if the useragent is generated by the mobile.
func isMobile(ua *ua.UserAgent) bool {
	// workaround as the library does not recognise iOS string inside the user agent
	isIOS := ua.IsIOS() || strings.Contains(ua.String, "iOS")
	return !isWeb(ua) && (ua.IsAndroid() || isIOS)
}

// isDesktop returns true if the useragent is generated by a desktop application.
func isDesktop(ua *ua.UserAgent) bool {
	return ua.Desktop && !isWeb(ua)
}

// isGRPC returns true if the useragent is generated by a grpc client.
func isGRPC(ua *ua.UserAgent) bool {
	return strings.HasPrefix(ua.Name, "grpc")
}
