package preflight

import (
	"fmt"
	"os"
	"path/filepath"

	"github.com/crc-org/crc/pkg/crc/adminhelper"
	"github.com/crc-org/crc/pkg/crc/cluster"
	"github.com/crc-org/crc/pkg/crc/constants"
	"github.com/crc-org/crc/pkg/crc/logging"
	"github.com/crc-org/crc/pkg/crc/machine/bundle"
	crcpreset "github.com/crc-org/crc/pkg/crc/preset"
	"github.com/crc-org/crc/pkg/crc/ssh"
	"github.com/crc-org/crc/pkg/crc/validation"
	"github.com/docker/go-units"
	"github.com/pkg/errors"
)

func bundleCheck(bundlePath string, preset crcpreset.Preset) Check {
	return Check{
		configKeySuffix:  "check-bundle-extracted",
		checkDescription: "Checking if CRC bundle is extracted in '$HOME/.crc'",
		check:            checkBundleExtracted(bundlePath),
		fixDescription:   "Getting bundle for the CRC executable",
		fix:              fixBundleExtracted(bundlePath, preset),
		flags:            SetupOnly,

		labels: None,
	}
}

func memoryCheck(preset crcpreset.Preset) Check {
	return Check{
		configKeySuffix:  "check-ram",
		checkDescription: "Checking minimum RAM requirements",
		check: func() error {
			return validation.ValidateEnoughMemory(constants.GetDefaultMemory(preset))
		},
		fixDescription: fmt.Sprintf("crc requires at least %s to run", units.HumanSize(float64(constants.GetDefaultMemory(preset)*1024*1024))),
		flags:          NoFix,

		labels: None,
	}
}

var genericCleanupChecks = []Check{
	{
		cleanupDescription: "Removing CRC Machine Instance directory",
		cleanup:            removeCRCMachinesDir,
		flags:              CleanUpOnly,

		labels: None,
	},
	{
		cleanupDescription: "Removing older logs",
		cleanup:            removeOldLogs,
		flags:              CleanUpOnly,

		labels: None,
	},
	{
		cleanupDescription: "Removing pull secret from the keyring",
		cleanup:            cluster.ForgetPullSecret,
		flags:              CleanUpOnly,

		labels: None,
	},
	{
		cleanupDescription: "Removing hosts file records added by CRC",
		cleanup:            removeHostsFileEntry,
		flags:              CleanUpOnly,

		labels: None,
	},
	{
		cleanupDescription: "Removing CRC Specific entries from user's known_hosts file",
		cleanup:            removeCRCHostEntriesFromKnownHosts,
		flags:              CleanUpOnly,

		labels: None,
	},
}

func checkBundleExtracted(bundlePath string) func() error {
	return func() error {
		logging.Infof("Checking if %s exists", bundlePath)
		bundleName := bundle.GetBundleNameFromURI(bundlePath)
		if _, err := bundle.Get(bundleName); err != nil {
			logging.Debugf("error getting bundle info for %s: %v", bundleName, err)
			return err
		}
		logging.Debugf("%s exists", bundlePath)
		return nil
	}
}

func fixBundleExtracted(bundlePath string, preset crcpreset.Preset) func() error {
	// Should be removed after 1.19 release
	// This check will ensure correct mode for `~/.crc/cache` directory
	// in case it exists.
	if err := os.Chmod(constants.MachineCacheDir, 0775); err != nil {
		logging.Debugf("Error changing %s permissions to 0775", constants.MachineCacheDir)
	}

	return func() error {
		bundleDir := filepath.Dir(constants.GetDefaultBundlePath(preset))
		logging.Debugf("Ensuring directory %s exists", bundleDir)
		if err := os.MkdirAll(bundleDir, 0775); err != nil {
			return fmt.Errorf("Cannot create directory %s: %v", bundleDir, err)
		}
		if err := validation.ValidateBundle(bundlePath, preset); err != nil {
			var e *validation.InvalidPath
			if !errors.As(err, &e) {
				return err
			}
			if bundlePath != constants.GetDefaultBundlePath(preset) {
				/* This message needs to be improved when the bundle has been set in crc config for example */
				return fmt.Errorf("%s is invalid or missing, run 'crc setup' to download the bundle", bundlePath)
			}
		}

		var err error
		logging.Infof("Downloading bundle: %s...", bundlePath)
		if bundlePath, err = bundle.Download(preset, bundlePath); err != nil {
			return err
		}

		logging.Infof("Uncompressing %s", bundlePath)
		if _, err := bundle.Extract(bundlePath); err != nil {
			if errors.Is(err, os.ErrNotExist) {
				return errors.Wrap(err, "Use `crc setup -b <bundle-path>`")
			}
			return err
		}
		return nil
	}
}

func removeHostsFileEntry() error {
	err := adminhelper.CleanHostsFile()
	if errors.Is(err, os.ErrNotExist) {
		return nil
	}
	return err
}

func removeCRCMachinesDir() error {
	logging.Debug("Deleting machines directory")
	if err := os.RemoveAll(constants.MachineInstanceDir); err != nil {
		return fmt.Errorf("Failed to delete crc machines directory: %w", err)
	}
	return nil
}

func removeOldLogs() error {
	logFiles, err := filepath.Glob(filepath.Join(constants.CrcBaseDir, "*.log_*"))
	if err != nil {
		return fmt.Errorf("Failed to get old logs: %w", err)
	}
	for _, f := range logFiles {
		logging.Debugf("Deleting %s log file", f)
		if err := os.RemoveAll(f); err != nil {
			return fmt.Errorf("Failed to delete %s: %w", f, err)
		}
	}
	return nil
}

func removeCRCHostEntriesFromKnownHosts() error {
	return ssh.RemoveCRCHostEntriesFromKnownHosts()
}
