package client

import (
	"fmt"
	"strings"
)

// Project represents an interface for managing Java classes within a project.
type Project interface {
	// Classes retrieves all Java classes within the project.
	Classes() ([]JavaClass, error)
}

// JavaClass represents an interface for a single Java class.
type JavaClass interface {
	// Name returns the name of the Java class.
	Name() string
	// Content retrieves the content of the Java class.
	Content() string
	// SetContent sets the content of the Java class.
	SetContent(content string) error
}

// InMemoryProject is an implementation of Project that stores its classes in memory.
type InMemoryProject struct {
	files map[string]JavaClass
}

// InMemoryJavaClass is an implementation of JavaClass that stores its data in memory.
type InMemoryJavaClass struct {
	name    string
	content string
}

// NewMockProject creates a mock project with predefined content for testing purposes.
func NewMockProject() Project {
	mapping := map[string]string{
		"Main.java": "public class Main {\n\tpublic static void main(String[] args) {\n\t\tString m = \"Hello, World\";\n\t\tSystem.out.println(m);\n\t}\n}\n",
	}
	return NewInMemoryProject(mapping)
}

// SingleClassProject creates a project containing a single Java class with the provided name and content.
func SingleClassProject(name, content string) Project {
	mapping := map[string]string{
		name: content,
	}
	return NewInMemoryProject(mapping)
}

// NewInMemoryProject creates a new in-memory project with the given map of file names to Java class content.
func NewInMemoryProject(files map[string]string) Project {
	res := make(map[string]JavaClass, len(files))
	for name, content := range files {
		res[name] = &InMemoryJavaClass{
			name:    name,
			content: content,
		}
	}
	return &InMemoryProject{
		files: res,
	}
}

// Classes retrieves all Java classes in the in-memory project.
func (i *InMemoryProject) Classes() ([]JavaClass, error) {
	res := make([]JavaClass, 0)
	for _, class := range i.files {
		res = append(res, class)
	}
	return res, nil
}

// SetContent updates the content of the in-memory Java class.
func (i *InMemoryJavaClass) SetContent(content string) error {
	i.content = content
	return nil
}

// Content retrieves the content of the in-memory Java class.
func (i *InMemoryJavaClass) Content() string {
	return i.content
}

// Name retrieves the name of the in-memory Java class.
func (i *InMemoryJavaClass) Name() string {
	return i.name
}

// String returns a string representation of the in-memory project.
func (i *InMemoryProject) String() string {
	names := make([]string, 0, len(i.files))
	for name := range i.files {
		names = append(names, name)
	}
	if len(names) == 0 {
		return "[empty project]"
	}
	return fmt.Sprintf("[%s]", strings.Join(names, ", "))
}
