import React from 'react';
import { DemoComponentProps, demoData } from '@epam/uui-docs';
import { ArrayDataSource, DataColumnProps, Metadata } from '@epam/uui-core';
import {
    Panel, DataTable, DataTableCell, useForm, FlexRow, Button, FlexSpacer, FlexCell,
} from '@epam/loveship';
import { Text } from '@epam/promo';

interface Person {
    id: number;
    name: string;
    yearsInCompany: number;
    personType: string;
    avatarUrl: string;
    phoneNumber: string;
    jobTitle: string;
    birthDate: string;
    gender: string;
    hireDate: string;
    departmentId: number;
    departmentName: string;
}

interface FormState {
    items: Record<number, Person>;
}

const metadata: Metadata<FormState> = {
    props: {
        items: {
            all: {
                props: {
                    name: { isRequired: true },
                    yearsInCompany: { isRequired: true },
                    departmentName: { isRequired: true },
                    birthDate: { isRequired: true },
                },
            },
        },
    },
};

let savedValue: FormState = { items: {} };

export function TableContext(contextProps: DemoComponentProps) {
    const { DemoComponent } = contextProps;

    const { lens, save } = useForm<FormState>({
        value: savedValue,
        onSave: async (value) => {
            savedValue = value;
        },
        getMetadata: () => metadata,
    });

    const [tableState, setTableState] = React.useState({});

    const dataSource = new ArrayDataSource<Person, number, unknown>({
        items: demoData.personDemoData,
        getId: (p) => p.id,
    });

    const dataView = dataSource.useView(tableState, setTableState, {
        getRowOptions: (person) => ({
            ...lens.prop('items').prop(person.id).toProps(),
            checkbox: { isVisible: true },
        }),
    });

    const personColumns: DataColumnProps<Person>[] = [
        {
            key: 'name',
            caption: 'Name',
            renderCell: (props) => (
                <DataTableCell
                    { ...props.rowLens.prop('name').toProps() }
                    renderEditor={ (props) => <DemoComponent valueType="id" selectionMode="single" dataSource={ dataSource } { ...props } /> }
                    { ...props }
                />
            ),
            isSortable: true,
            isAlwaysVisible: true,
            width: 200,
            fix: 'left',
        }, {
            key: 'yearsInCompany',
            caption: 'Years In Company',
            renderCell: (props) => (
                <DataTableCell
                    { ...props.rowLens.prop('yearsInCompany').toProps() }
                    renderEditor={ (props) => <DemoComponent valueType="id" selectionMode="single" dataSource={ dataSource } { ...props } /> }
                    { ...props }
                />
            ),
            isSortable: true,
            isAlwaysVisible: true,
            width: 120,
        }, {
            key: 'departmentName',
            caption: 'Department Name',
            render: (props) => <Text>{props.departmentName}</Text>,
            isSortable: true,
            isAlwaysVisible: true,
            width: 200,
        }, {
            key: 'birthDate',
            caption: 'Birth Date',
            renderCell: (props) => (
                <DataTableCell
                    { ...props.rowLens.prop('birthDate').toProps() }
                    renderEditor={ (props) => <DemoComponent valueType="id" selectionMode="single" dataSource={ dataSource } { ...props } /> }
                    { ...props }
                />
            ),
            isSortable: true,
            isAlwaysVisible: true,
            width: 180,
        },
    ];

    return (
        <>
            <Panel style={ { width: '100%', paddingLeft: '3px' } }>
                <DataTable
                    headerTextCase="upper"
                    getRows={ dataView.getVisibleRows }
                    columns={ personColumns }
                    value={ tableState }
                    onValueChange={ setTableState }
                    showColumnsConfig
                    allowColumnsResizing
                    allowColumnsReordering
                    { ...dataView.getListProps() }
                />
            </Panel>
            <FlexRow spacing="12" margin="12">
                <FlexSpacer />
                <FlexCell width="auto">
                    <Button color="grass" caption="Save" onClick={ save } />
                </FlexCell>
            </FlexRow>
        </>
    );
}

TableContext.displayName = 'Table';
