/* SPDX-FileCopyrightText: 2024 Google LLC */
/* SPDX-License-Identifier: Apache-2.0 */

#include <stdint.h>

#include "resource/timeline_resource_ids.auto.h"
#include "util/attributes.h"

// Chunked dictionary 1/2 sent by ANCS (w/header)
// 00 97 00 00 00 00 13 00  63 6f 6d 2e 61 70 70 6c   ........ com.appl
// 65 2e 4d 6f 62 69 6c 65  53 4d 53 01 10 00 72 6a   e.Mobile SMS...rj
// 63 61 73 65 40 67 6d 61  69 6c 2e 63 6f 6d 02 00   case@gma il.com..
// 00 03 c8 00 54 68 69 73  20 69 73 20 61 20 76 65   ....This  is a ve
// 72 79 20 63 6f 6d 70 6c  69 63 61 74 65 64 20 63   ry compl icated c
// 61 73 65 2c 20 4d 61 75  64 65 2e 20 59 6f 75 20   ase, Mau de. You
// 6b 6e 6f 77 2c 20 61 20  6c 6f 74 74 61 20 69 6e   know, a  lotta in
// 73 2c 20 6c 6f 74 74 61  20 6f 75 74 73 2c 20 6c   s, lotta  outs, l
// 6f 74 74 61 20 77 68 61  74 2d 68 61 76 65 2d 79   otta wha t-have-y
// 6f 75 27 73 2e 20 41 6e  64 2c 20                  ou's. An d,
// NB header has been removed
static const uint8_t s_chunked_dict_part_one[] = {
  0x00, 0x97, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c, 0x65, 0x53, 0x4d, 0x53, 0x01, 0x10, 0x00, 0x72, 0x6a,
  0x63, 0x61, 0x73, 0x65, 0x40, 0x67, 0x6d, 0x61, 0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x02, 0x00,
  0x00, 0x03, 0xc8, 0x00, 0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x76, 0x65,
  0x72, 0x79, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x65, 0x64, 0x20, 0x63,
  0x61, 0x73, 0x65, 0x2c, 0x20, 0x4d, 0x61, 0x75, 0x64, 0x65, 0x2e, 0x20, 0x59, 0x6f, 0x75, 0x20,
  0x6b, 0x6e, 0x6f, 0x77, 0x2c, 0x20, 0x61, 0x20, 0x6c, 0x6f, 0x74, 0x74, 0x61, 0x20, 0x69, 0x6e,
  0x73, 0x2c, 0x20, 0x6c, 0x6f, 0x74, 0x74, 0x61, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x2c, 0x20, 0x6c,
  0x6f, 0x74, 0x74, 0x61, 0x20, 0x77, 0x68, 0x61, 0x74, 0x2d, 0x68, 0x61, 0x76, 0x65, 0x2d, 0x79,
  0x6f, 0x75, 0x27, 0x73, 0x2e, 0x20, 0x41, 0x6e, 0x64, 0x2c, 0x20
};

// Chunked dictionary 2/2 sent by ANCS
// 75 68 2c 20 6c 6f 74 74  61 20 73 74 72 61 6e 64   uh, lott a strand
// 73 20 74 6f 20 6b 65 65  70 20 69 6e 20 6d 79 20   s to kee p in my
// 68 65 61 64 2c 20 6d 61  6e 2e 20 4c 6f 74 74 61   head, ma n. Lotta
// 20 73 74 72 61 6e 64 73  20 69 6e 20 6f 6c 64 20    strands  in old
// 44 75 64 65 72 27 73 20  68 65 61 64 2e 20 4c 75   Duder's  head. Lu
// 63 6b 69 6c 79 20 49 27  6d 20 61 64 68 65 72 69   ckily I' m adheri
// 6e                                                 n
static const uint8_t s_chunked_dict_part_two[] = {
  0x75, 0x68, 0x2c, 0x20, 0x6c, 0x6f, 0x74, 0x74,  0x61, 0x20, 0x73, 0x74, 0x72, 0x61, 0x6e, 0x64,
  0x73, 0x20, 0x74, 0x6f, 0x20, 0x6b, 0x65, 0x65,  0x70, 0x20, 0x69, 0x6e, 0x20, 0x6d, 0x79, 0x20,
  0x68, 0x65, 0x61, 0x64, 0x2c, 0x20, 0x6d, 0x61,  0x6e, 0x2e, 0x20, 0x4c, 0x6f, 0x74, 0x74, 0x61,
  0x20, 0x73, 0x74, 0x72, 0x61, 0x6e, 0x64, 0x73,  0x20, 0x69, 0x6e, 0x20, 0x6f, 0x6c, 0x64, 0x20,
  0x44, 0x75, 0x64, 0x65, 0x72, 0x27, 0x73, 0x20,  0x68, 0x65, 0x61, 0x64, 0x2e, 0x20, 0x4c, 0x75,
  0x63, 0x6b, 0x69, 0x6c, 0x79, 0x20, 0x49, 0x27,  0x6d, 0x20, 0x61, 0x64, 0x68, 0x65, 0x72, 0x69,
  0x6e, /* date */ 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31,
  0x33, 0x31, 0x31, 0x30, 0x35, 0x54, 0x31, 0x38, 0x31, 0x39, 0x32, 0x38
};

// Complete dictionary sent by ANCS
// 00 98 00 00 00 00 13 00  63 6f 6d 2e 61 70 70 6c   ........ com.appl
// 65 2e 4d 6f 62 69 6c 65  53 4d 53 01 10 00 72 6a   e.Mobile SMS...rj
// 63 61 73 65 40 67 6d 61  69 6c 2e 63 6f 6d 02 00   case@gma il.com..
// 00 03 26 00 43 61 72 65  66 75 6c 2c 20 6d 61 6e   ..&.Care ful, man
// 2c 20 74 68 65 72 65 27  73 20 61 20 62 65 76 65   , there' s a beve
// 72 61 67 65 20 68 65 72  65 21                     rage her e! // FIXME 4th
// attr has been appended
// Has header removed:
static const uint8_t s_complete_dict[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00,  0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c, 0x65,  0x53, 0x4d, 0x53, 0x01, 0x10, 0x00, 0x72, 0x6a,
  0x63, 0x61, 0x73, 0x65, 0x40, 0x67, 0x6d, 0x61,  0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x02, 0x00,
  0x00, 0x03, 0x26, 0x00, 0x43, 0x61, 0x72, 0x65,  0x66, 0x75, 0x6c, 0x2c, 0x20, 0x6d, 0x61, 0x6e,
  0x2c, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x27,  0x73, 0x20, 0x61, 0x20, 0x62, 0x65, 0x76, 0x65,
  0x72, 0x61, 0x67, 0x65, 0x20, 0x68, 0x65, 0x72,  0x65, 0x21, /* date */ 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31,
  0x33, 0x31, 0x31, 0x30, 0x35, 0x54, 0x31, 0x38, 0x31, 0x39, 0x32, 0x38
};


//! 00 3c 00 00 00 05 0f 00  32 30 31 36 30 33 30 31   ........ 20160301
//! 54 31 34 31 35 32 33 01  08 00 4f 76 65 72 63 61   T141523. ..Overca
//! 73 74 02 00 00 03 2b 00  54 68 65 20 4d 6f 74 68   st....+. The Moth
//! 20 50 6f 64 63 61 73 74  3a 20 4d 6f 6e 69 63 61    Podcast : Monica
//! 20 43 68 65 72 75 2d 4d  70 61 6d 62 61 77 61 73    Cheru-M pambawas
//! e2 80 a6 04 03 00 31 31  33                        ......11 3
static const uint8_t s_message_size_attr_dict[] = {
  0x00, 0x3c, 0x00, 0x00, 0x00, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x36, 0x30, 0x33, 0x30, 0x31,
  0x54, 0x31, 0x34, 0x31, 0x35, 0x32, 0x33, 0x01, 0x08, 0x00, 0x4f, 0x76, 0x65, 0x72, 0x63, 0x61,
  0x73, 0x74, 0x02, 0x00, 0x00, 0x03, 0x2b, 0x00, 0x54, 0x68, 0x65, 0x20, 0x4d, 0x6f, 0x74, 0x68,
  0x20, 0x50, 0x6f, 0x64, 0x63, 0x61, 0x73, 0x74, 0x3a, 0x20, 0x4d, 0x6f, 0x6e, 0x69, 0x63, 0x61,
  0x20, 0x43, 0x68, 0x65, 0x72, 0x75, 0x2d, 0x4d, 0x70, 0x61, 0x6d, 0x62, 0x61, 0x77, 0x61, 0x73,
  0xe2, 0x80, 0xa6, 0x04, 0x03, 0x00, 0x31, 0x31, 0x33,
  /* adding an app_id because it didnt' come with one */
  0x00, 0x01, 0x00, 0x50,
};

//! Complete dict for a messages app notification
//! 00 1a 00 00 00 06 00 00  07 05 00 43 6c 65 61 72   ........ ...Clear
//! 00 13 00 63 6f 6d 2e 61  70 70 6c 65 2e 4d 6f 62   ...com.a pple.Mob
//! 69 6c 65 53 4d 53 01 13  00 46 72 61 6e 63 6f 69   ileSMS.. .Francoi
//! 73 20 42 61 6c 64 61 73  73 61 72 69 02 00 00 03   s Baldas sari....
//! 1b 00 74 68 69 73 20 69  73 20 61 20 74 65 73 74   ..this i s a test
//! 20 6e 6f 74 69 66 69 63  61 74 69 6f 6e 05 0f 00    notific ation...
//! 32 30 31 35 30 36 31 35  54 31 36 31 38 33 37      20150615 T161837
static const uint8_t s_message_dict[] = {
  0x00, 0x1a, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x05, 0x00, 0x43, 0x6c, 0x65, 0x61, 0x72,
  0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x4d, 0x6f, 0x62,
  0x69, 0x6c, 0x65, 0x53, 0x4d, 0x53, 0x01, 0x13, 0x00, 0x46, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x69,
  0x73, 0x20, 0x42, 0x61, 0x6c, 0x64, 0x61, 0x73, 0x73, 0x61, 0x72, 0x69, 0x02, 0x00, 0x00, 0x03,
  0x1b, 0x00, 0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x74, 0x65, 0x73, 0x74,
  0x20, 0x6e, 0x6f, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x05, 0x0f, 0x00,
  0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x31, 0x35, 0x54, 0x31, 0x36, 0x31, 0x38, 0x33, 0x37
};

static TimelineItem s_message_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x1a,
    .timestamp = 1434385117,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 6,
    .attributes = (Attribute[6]) {
      { .id = AttributeIdSender, .cstring = "Francois Baldassari" },
      { .id = AttributeIdTitle, .cstring = "Francois Baldassari" },
      { .id = AttributeIdBody, .cstring = "this is a test notification" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.apple.MobileSMS" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_GENERIC_SMS },
      { .id = AttributeIdBgColor, .uint8 = GColorIslamicGreenARGB8 }
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 1,
    .actions = (TimelineItemAction[1]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative},
            { .id = AttributeIdTitle, .cstring = "Dismiss" }}
         }
      }
    }
  }
};

//! Complete dict for the messages app attributes
//! 01 63 6f 6d 2e 61 70 70  6c 65 2e 4d 6f 62 69 6c   .com.app le.Mobil
//! 65 53 4d 53 00 00 08 00  4d 65 73 73 61 67 65 73   eSMS.... Messages

static const uint8_t s_message_app_info_dict[] = {
  0x01, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c,
  0x65, 0x53, 0x4d, 0x53, 0x00, 0x00, 0x08, 0x00, 0x4d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73
};

//! Complete dict for the unrecognized fake app attributes
//! 01 63 6f 6d 2e 74 65 73  74 73 2e 4e 6f 74 41 6e   .com.tes ts.NotAn
//! 41 70 70 00 00 07 00 46  61 6b 65 41 70 70         App....F akeApp

static const uint8_t s_fake_app_info_dict[] = {
  0x01, 0x63, 0x6f, 0x6d, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x73, 0x2e, 0x4e, 0x6f, 0x74, 0x41, 0x6e,
  0x41, 0x70, 0x70, 0x00, 0x00, 0x07, 0x00, 0x46, 0x61, 0x6b, 0x65, 0x41, 0x70, 0x70,
};

// Complete app dictionary sent by ANCS
// 01 63 6f 6d 2e 67 65 74  70 65 62 62 6c 65 2e 70   .com.get pebble.p
// 65 62 62 6c 65 74 69 6d  65 2e 65 6e 74 65 72 70   ebbletim e.enterp
// 72 69 73 65 00 00 0e 00  50 65 62 62 6c 65 20 54   rise.... Pebble T
// 69 6d 65 20 ce b2                                  ime ..
static const uint8_t s_app_attr_dict[] = {
  0x01, 0x63, 0x6f, 0x6d, 0x2e, 0x67, 0x65, 0x74, 0x70, 0x65, 0x62, 0x62, 0x6c, 0x65, 0x2e, 0x70,
  0x65, 0x62, 0x62, 0x6c, 0x65, 0x74, 0x69, 0x6d, 0x65, 0x2e, 0x65, 0x6e, 0x74, 0x65, 0x72, 0x70,
  0x72, 0x69, 0x73, 0x65, 0x00, 0x00, 0x0e, 0x00, 0x50, 0x65, 0x62, 0x62, 0x6c, 0x65, 0x20, 0x54,
  0x69, 0x6d, 0x65, 0x20, 0xce, 0xb2
};

static const uint8_t s_missing_last_attribute[] = {
  0x00, 0x95, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00,  0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c, 0x65,  0x53, 0x4d, 0x53, 0x01, 0x10, 0x00, 0x72, 0x6a,
  0x63, 0x61, 0x73, 0x65, 0x40, 0x67, 0x6d, 0x61,  0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x02, 0x00,
  0x00, 0x03, 0x26, 0x00, 0x43, 0x61, 0x72, 0x65,  0x66, 0x75, 0x6c, 0x2c, 0x20, 0x6d, 0x61, 0x6e,
  0x2c, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x27,  0x73, 0x20, 0x61, 0x20, 0x62, 0x65, 0x76, 0x65,
  0x72, 0x61, 0x67, 0x65, 0x20, 0x68, 0x65, 0x72,  0x65, 0x21
};

// Length of attribute too long
static const uint8_t s_invalid_attribute_length[] = {
  0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00,  0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c, 0x65,  0x53, 0x4d, 0x53, 0x01, 0x29, 0x00, 0x72, 0x6a,
  0x63, 0x61, 0x73, 0x65, 0x40, 0x67, 0x6d, 0x61,  0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x20, 0x20,
  0x20, 0x20, 0x20, 0x20, 0x43, 0x61, 0x72, 0x65,  0x66, 0x75, 0x6c, 0x2c, 0x20, 0x6d, 0x61, 0x6e,
  0x2c, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x00,  0x73, 0x20, 0x61, 0x20, 0x62, 0x65, 0x76, 0x65,
  0x72, 0x61, 0x67, 0x65, 0x20, 0x68, 0x65, 0x72,  0x65, 0x21, 0x02, 0x00, 0x00, 0x03, 0x26, 0x00,
  0x43, 0x61, 0x72, 0x65, 0x66, 0x75, 0x6c, 0x2c,  0x20, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x74, 0x68,
  0x65, 0x72, 0x65, 0x27, 0x73, 0x20, 0x61, 0x20,  0x62, 0x65, 0x76, 0x65, 0x72, 0x61, 0x67, 0x65,
  0x20, 0x68, 0x65, 0x72, 0x65, 0x21, 0x05, 0x0f,  0x00, 0x32, 0x30, 0x31, 0x33, 0x31, 0x31, 0x30,
  0x35, 0x54, 0x31, 0x38, 0x31, 0x39, 0x32, 0x38
};

// Chunked dictionary 1/2 sent by ANCS (w/header)
// 00 0d 00 00 00 00 1b 00  63 6f 6d 2e 6b 69 63 6b   ........ com.kick
// 73 74 61 72 74 65 72 2e  6b 69 63 6b 73 74 61 72   starter. kickstar
// 74 65 72 01 0b 00 4b 69  63 6b 73 74 61 72 74 65   ter...Ki ckstarte
// 72 02 00 00 03 56 00 52  65 73 70 6f 6e 73 65 20   r....V.R esponse
// 6e 65 65 64 65 64 21 20  47 65 74 20 79 6f 75 72   needed!  Get your
// 20 72 65 77 61 72 64 20  66 6f 72 20 62 61 63 6b    reward  for back
// 69 6e 67 20 50 65 62 62  6c 65 3a 20 45 2d 50 61   ing Pebb le: E-Pa
// 70 65 72 79 20 57 61 74  63 68 20 66 6f 72 20 69   pery Wat ch for i
// 4f 53 20 61 6e 64 20 44  72 6f 69 64 2e 05 0f 00   OS and D roid....
// 32 30 31 34 30 36 31 30  54 32 32                  20140610 T22
static const uint8_t s_split_timestamp_dict_part_one[] = {
  0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00,  0x63, 0x6f, 0x6d, 0x2e, 0x6b, 0x69, 0x63, 0x6b,
  0x73, 0x74, 0x61, 0x72, 0x74, 0x65, 0x72, 0x2e,  0x6b, 0x69, 0x63, 0x6b, 0x73, 0x74, 0x61, 0x72,
  0x74, 0x65, 0x72, 0x01, 0x0b, 0x00, 0x4b, 0x69,  0x63, 0x6b, 0x73, 0x74, 0x61, 0x72, 0x74, 0x65,
  0x72, 0x02, 0x00, 0x00, 0x03, 0x56, 0x00, 0x52,  0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73, 0x65, 0x20,
  0x6e, 0x65, 0x65, 0x64, 0x65, 0x64, 0x21, 0x20,  0x47, 0x65, 0x74, 0x20, 0x79, 0x6f, 0x75, 0x72,
  0x20, 0x72, 0x65, 0x77, 0x61, 0x72, 0x64, 0x20,  0x66, 0x6f, 0x72, 0x20, 0x62, 0x61, 0x63, 0x6b,
  0x69, 0x6e, 0x67, 0x20, 0x50, 0x65, 0x62, 0x62,  0x6c, 0x65, 0x3a, 0x20, 0x45, 0x2d, 0x50, 0x61,
  0x70, 0x65, 0x72, 0x79, 0x20, 0x57, 0x61, 0x74,  0x63, 0x68, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x69,
  0x4f, 0x53, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x44,  0x72, 0x6f, 0x69, 0x64, 0x2e, 0x05, 0x0f, 0x00,
  0x32, 0x30, 0x31, 0x34, 0x30, 0x36, 0x31, 0x30,  0x54, 0x32, 0x32
};

// Chunked dictionary 2/2 sent by ANCS
// 32 30 34 38                                        2048
static const uint8_t s_split_timestamp_dict_part_two[] = {
  0x32, 0x30, 0x34, 0x38
};

static const uint8_t s_invalid_attribute_length_fixed[] = {
  0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00,  0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x4d, 0x6f, 0x62, 0x69, 0x6c, 0x65,  0x53, 0x4d, 0x53, 0x01, 0x10, 0x00, 0x72, 0x6a,
  0x63, 0x61, 0x73, 0x65, 0x40, 0x67, 0x6d, 0x61,  0x69, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x02, 0x00,
  0x00, 0x03, 0x26, 0x00, 0x43, 0x61, 0x72, 0x65,  0x66, 0x75, 0x6c, 0x2c, 0x20, 0x6d, 0x61, 0x6e,
  0x2c, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x27,  0x73, 0x20, 0x61, 0x20, 0x62, 0x65, 0x76, 0x65,
  0x72, 0x61, 0x67, 0x65, 0x20, 0x68, 0x65, 0x72,  0x65, 0x21, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31,
  0x33, 0x31, 0x31, 0x30, 0x35, 0x54, 0x31, 0x38, 0x31, 0x39, 0x32, 0x38
};


// 00 22 00 00 00 07 07 00  44 69 73 6d 69 73 73 00   ."...... Dismiss.
// 14 00 63 6f 6d 2e 61 70  70 6c 65 2e 6d 6f 62 69   ..com.ap ple.mobi
// 6c 65 6d 61 69 6c 01 15  00 41 61 72 6f 6e 20 48   lemail.. .Aaron H
// 65 63 6b 6d 61 6e 6e 20  28 4a 49 52 41 29 02 28   eckmann  (JIRA).(
// 00 5b 4a 49 52 41 5d 20  28 50 42 4c 2d 38 37 37   .[JIRA]  (PBL-877
// 36 29 20 45 72 72 6f 72  20 73 63 72 65 65 6e 20   6) Error  screen
// 61 66 74 65 72 20 6c 6f  67 03 1c 00 54 68 69 73   after lo g...This
// 20 6d 65 73 73 61 67 65  20 68 61 73 20 6e 6f 20    message  has no
// 63 6f 6e 74 65 6e 74 2e  05 0f 00 32 30 31 34 30   content. ...20140
// 39 32 36 54 31 30 34 33  30 30                     926T1043 00
static const uint8_t s_this_message_has_no_content_response[] = {
  0x00, 0x22, 0x00, 0x00, 0x00, 0x07, 0x07, 0x00,  0x44, 0x69, 0x73, 0x6d, 0x69, 0x73, 0x73, 0x00,
  0x14, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70,  0x70, 0x6c, 0x65, 0x2e, 0x6d, 0x6f, 0x62, 0x69,
  0x6c, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x01, 0x15,  0x00, 0x41, 0x61, 0x72, 0x6f, 0x6e, 0x20, 0x48,
  0x65, 0x63, 0x6b, 0x6d, 0x61, 0x6e, 0x6e, 0x20,  0x28, 0x4a, 0x49, 0x52, 0x41, 0x29, 0x02, 0x28,
  0x00, 0x5b, 0x4a, 0x49, 0x52, 0x41, 0x5d, 0x20,  0x28, 0x50, 0x42, 0x4c, 0x2d, 0x38, 0x37, 0x37,
  0x36, 0x29, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72,  0x20, 0x73, 0x63, 0x72, 0x65, 0x65, 0x6e, 0x20,
  0x61, 0x66, 0x74, 0x65, 0x72, 0x20, 0x6c, 0x6f,  0x67, 0x03, 0x1c, 0x00, 0x54, 0x68, 0x69, 0x73,
  0x20, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65,  0x20, 0x68, 0x61, 0x73, 0x20, 0x6e, 0x6f, 0x20,
  0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2e,  0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x34, 0x30,
  0x39, 0x32, 0x36, 0x54, 0x31, 0x30, 0x34, 0x33,  0x30, 0x30
};

// 00 19 00 00 00 07 07 00  44 69 73 6d 69 73 73 00   ........ Dismiss.
// 14 00 63 6f 6d 2e 61 70  70 6c 65 2e 6d 6f 62 69   ..com.ap ple.mobi
// 6c 65 6d 61 69 6c 01 05  00 4d 61 72 74 79 02 04   lemail.. .Marty..
// 00 74 65 73 74 03 0a 00  4c 6f 61 64 69 6e 67 e2   .test... Loading.
// 80 a6 05 0f 00 32 30 31  34 30 39 32 39 54 31 35   .....201 40929T15
// 31 35 31 30                                        1510
static const uint8_t s_loading_response[] = {
  0x00, 0x19, 0x00, 0x00, 0x00, 0x07, 0x07, 0x00,  0x44, 0x69, 0x73, 0x6d, 0x69, 0x73, 0x73, 0x00,
  0x14, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70,  0x70, 0x6c, 0x65, 0x2e, 0x6d, 0x6f, 0x62, 0x69,
  0x6c, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x01, 0x05,  0x00, 0x4d, 0x61, 0x72, 0x74, 0x79, 0x02, 0x04,
  0x00, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x00,  0x4c, 0x6f, 0x61, 0x64, 0x69, 0x6e, 0x67, 0xe2,
  0x80, 0xa6, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31,  0x34, 0x30, 0x39, 0x32, 0x39, 0x54, 0x31, 0x35,
  0x31, 0x35, 0x31, 0x30,
};

// 00 aa 00 00 00 07 07 00 44 69 73 6d 69 73 73 00 ........ Dismiss.
// 14 00 63 6f 6d 2e 61 70 70 6c 65 2e 6d 6f 62 69 ..com.ap ple.mobi
// 6c 65 6d 61 69 6c 01 16 00 63 61 6c 65 6e 64 61 lemail.. .calenda
// 72 40 63 73 61 69 6c 2e 6d 69 74 2e 65 64 75 02 r@csail. mit.edu.
// 28 00 54 41 4c 4b 3a 20 54 75 65 73 64 61 79 20 (.TALK: Tuesday
// 31 30 2d 32 38 2d 32 30 31 34 20 53 63 69 65 6e 10-28-20 14 Scien
// 74 69 66 69 63 20 55 74 6f 70 03 c8 00 53 63 69 tific Ut op...Sci
// 65 6e 74 69 66 69 63 20 55 74 6f 70 69 61 3a 20 entific Utopia:
// 49 6d 70 72 6f 76 69 6e 67 20 4f 70 65 6e 6e 65 Improvin g Openne
// 73 73 20 61 6e 64 20 52 65 70 72 6f 64 75 63 69 ss and R eproduci
// 62 69 6c 69 74 79 20 69 6e 20 53 63 69 65 6e 74 bility i n Scient
// 69 66 69 63 20 52 65 73 65 61 72 63 68 20 53 70 ific Res earch Sp
// 65 61 6b 65 72 3a 20 42 72 69 61 6e 20 4e 6f 73 eaker: B rian Nos
// 65 6b 2c 20 55 6e 69 76 65 72 73 69 74 79 20 6f ek, Univ ersity o
// 66 20 56 69 72 67 69 6e 69 61 20 61 6e 64 20 43 f Virgin ia and C
// 65 6e 74 65 72 20 66 6f 72 20 4f 70 65 6e 20 53 enter fo r Open S
// 63 69 65 6e 63 65 20 53 70 65 61 6b 65 72 20 41 cience S peaker A
// 66 66 69 6c 69 61 74 69 6f 6e 3a 20 55 6e 69 76 ffiliati on: Univ
// 65 72 73 69 74 79 20 6f 66 20 56 69 72 67 69 6e ersity o f Virgin
// 69 61 20 48 6f 05 ia Ho.
typedef struct PACKED s_sample_memory_layout {
  const uint8_t attribute_data[310];
  const uint8_t out_of_bounds_memory[6];
} s_sample_memory_layout;

s_sample_memory_layout memory_with_attribute_id_at_end = {
  .attribute_data = {
    0x00, 0xaa, 0x00, 0x00, 0x00, 0x07, 0x07, 0x00, 0x44, 0x69, 0x73, 0x6d, 0x69, 0x73, 0x73, 0x00,
    0x14, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x6d, 0x6f, 0x62, 0x69,
    0x6c, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x01, 0x16, 0x00, 0x63, 0x61, 0x6c, 0x65, 0x6e, 0x64, 0x61,
    0x72, 0x40, 0x63, 0x73, 0x61, 0x69, 0x6c, 0x2e, 0x6d, 0x69, 0x74, 0x2e, 0x65, 0x64, 0x75, 0x02,
    0x28, 0x00, 0x54, 0x41, 0x4c, 0x4b, 0x3a, 0x20, 0x54, 0x75, 0x65, 0x73, 0x64, 0x61, 0x79, 0x20,
    0x31, 0x30, 0x2d, 0x32, 0x38, 0x2d, 0x32, 0x30, 0x31, 0x34, 0x20, 0x53, 0x63, 0x69, 0x65, 0x6e,
    0x74, 0x69, 0x66, 0x69, 0x63, 0x20, 0x55, 0x74, 0x6f, 0x70, 0x03, 0xc8, 0x00, 0x53, 0x63, 0x69,
    0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x63, 0x20, 0x55, 0x74, 0x6f, 0x70, 0x69, 0x61, 0x3a, 0x20,
    0x49, 0x6d, 0x70, 0x72, 0x6f, 0x76, 0x69, 0x6e, 0x67, 0x20, 0x4f, 0x70, 0x65, 0x6e, 0x6e, 0x65,
    0x73, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x52, 0x65, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x69,
    0x62, 0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x69, 0x6e, 0x20, 0x53, 0x63, 0x69, 0x65, 0x6e, 0x74,
    0x69, 0x66, 0x69, 0x63, 0x20, 0x52, 0x65, 0x73, 0x65, 0x61, 0x72, 0x63, 0x68, 0x20, 0x53, 0x70,
    0x65, 0x61, 0x6b, 0x65, 0x72, 0x3a, 0x20, 0x42, 0x72, 0x69, 0x61, 0x6e, 0x20, 0x4e, 0x6f, 0x73,
    0x65, 0x6b, 0x2c, 0x20, 0x55, 0x6e, 0x69, 0x76, 0x65, 0x72, 0x73, 0x69, 0x74, 0x79, 0x20, 0x6f,
    0x66, 0x20, 0x56, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x69, 0x61, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x43,
    0x65, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x4f, 0x70, 0x65, 0x6e, 0x20, 0x53,
    0x63, 0x69, 0x65, 0x6e, 0x63, 0x65, 0x20, 0x53, 0x70, 0x65, 0x61, 0x6b, 0x65, 0x72, 0x20, 0x41,
    0x66, 0x66, 0x69, 0x6c, 0x69, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x55, 0x6e, 0x69, 0x76,
    0x65, 0x72, 0x73, 0x69, 0x74, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x56, 0x69, 0x72, 0x67, 0x69, 0x6e,
    0x69, 0x61, 0x20, 0x48, 0x6f, 0x05,
  },
  .out_of_bounds_memory = {
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  },
};

static const uint8_t memory_with_attribute_id_at_end_p2[] = {
 0x0f, 0x00, 0x32, 0x30, 0x31, 0x33, 0x31, 0x31, 0x30, 0x35, 0x54, 0x31, 0x38, 0x31, 0x39, 0x32, 0x38
};


// 00 74 00 00 00 06 06 00 41 6e 73 77 65 72 07 07 .t...... Answer..
// 00 44 65 63 6c 69 6e 65 00 15 00 63 6f 6d 2e 61 .Decline ...com.a
// 70 70 6c 65 2e 6d 6f 62 69 6c 65 70 68 6f 6e 65 pple.mob ilephone
// 01 03 00 49 6d 6d 02 06 00 6d 6f 62 69 6c 65 03 ...Imm.. .mobile.
// 0d 00 49 6e 63 6f 6d 69 6e 67 20 43 61 6c 6c 05 ..Incomi ng Call.
// 00 00 00 75 00 00 00 07 07 00 44 69 73 6d 69 73 ...u.... ..Dismis
// 73 00 13 00 63 6f 6d 2e 61 70 70 6c 65 2e 6d 6f s...com. apple.mo
// 62 69 6c 65 63 61 6c 01 11 00 48 61 69 6c 65 79 bilecal. ..Hailey
// 27 73 20 70 72 61 63 74 69 63 65 02 00 00 03 0a 's pract ice.....
// 00 52 65 64 20 4d 6f 72 74 6f 6e 05 0f 00 32 30 .Red Mor ton...20
// 31 34 31 30 32 38 54 31 36 30 30 30 30 00 76 00 141028T1 60000.v.
// 00 00 06 06 00 41 6e 73 77 65 72 07 07 00 44 65 .....Ans wer...De
// 63 6c 69 6e 65 00 15 00 63 6f 6d 2e 61 70 70 6c cline... com.appl
// 65 2e 6d 6f 62 69 6c 65 70 68 6f 6e 58 01 03 00 e.mobile phonX...  //phone -> phonX for testing
// 49 6d 6d 02 06 00 6d 6f 62 69 6c 65 03 0d 00 49 Imm...mo bile...I
// 6e 63 6f 6d 69 6e 67 20 43 61 6c 6c 05 00 00 ncoming Call...

static const uint8_t s_multiple_complete_dicts[] = {
  // Dict 1
  0x00, 0x74, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x41, 0x6e, 0x73, 0x77, 0x65, 0x72, 0x07, 0x07,
  0x00, 0x44, 0x65, 0x63, 0x6c, 0x69, 0x6e, 0x65, 0x00, 0x15, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61,
  0x70, 0x70, 0x6c, 0x65, 0x2e, 0x6d, 0x6f, 0x62, 0x69, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x65,
  0x01, 0x03, 0x00, 0x49, 0x6d, 0x6d, 0x02, 0x06, 0x00, 0x6d, 0x6f, 0x62, 0x69, 0x6c, 0x65, 0x03,
  0x0d, 0x00, 0x49, 0x6e, 0x63, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x20, 0x43, 0x61, 0x6c, 0x6c, 0x05,
  0x00, 0x00,

  // Dict 2
  0x00, 0x75, 0x00, 0x00, 0x00, 0x07, 0x07, 0x00, 0x44, 0x69, 0x73, 0x6d, 0x69, 0x73,
  0x73, 0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x6d, 0x6f,
  0x62, 0x69, 0x6c, 0x65, 0x63, 0x61, 0x6c, 0x01, 0x11, 0x00, 0x48, 0x61, 0x69, 0x6c, 0x65, 0x79,
  0x27, 0x73, 0x20, 0x70, 0x72, 0x61, 0x63, 0x74, 0x69, 0x63, 0x65, 0x02, 0x00, 0x00, 0x03, 0x0a,
  0x00, 0x52, 0x65, 0x64, 0x20, 0x4d, 0x6f, 0x72, 0x74, 0x6f, 0x6e, 0x05, 0x0f, 0x00, 0x32, 0x30,
  0x31, 0x34, 0x31, 0x30, 0x32, 0x38, 0x54, 0x31, 0x36, 0x30, 0x30, 0x30, 0x30,

  //Dict 3
  0x00, 0x76, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x41, 0x6e, 0x73, 0x77, 0x65, 0x72, 0x07, 0x07, 0x00, 0x44, 0x65,
  0x63, 0x6c, 0x69, 0x6e, 0x65, 0x00, 0x15, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c,
  0x65, 0x2e, 0x6d, 0x6f, 0x62, 0x69, 0x6c, 0x65, 0x70, 0x68, 0x6f, 0x6e, 0x64, 0x01, 0x03, 0x00,
  0x49, 0x6d, 0x6d, 0x02, 0x06, 0x00, 0x6d, 0x6f, 0x62, 0x69, 0x6c, 0x65, 0x03, 0x0d, 0x00, 0x49,
  0x6e, 0x63, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x20, 0x43, 0x61, 0x6c, 0x6c, 0x05, 0x00, 0x00
};

//! Complete dict for a messages app notification with a duplicated title
//! 00 1b 00 00 00 06 00 00  07 05 00 43 6c 65 61 72   ........ ...Clear
//! 00 13 00 63 6f 6d 2e 61  70 70 6c 65 2e 4d 6f 62   ...com.a pple.Mob
//! 69 6c 65 53 4d 53 01 08  00 4D 65 73 73 61 67 65   ileSMS.. .Message
//! 73 02 00 00 03 1b 00 74  68 69 73 20 69 73 20 61   s......t his is a
//! 20 74 65 73 74 20 6e 6f  74 69 66 69 63 61 74 69    test no tificati
//! 6f 6e 05 0f 00 32 30 31  35 30 36 31 35 54 31 36   on...201 50615T16
//! 31 38 33 37                                        1837

static const uint8_t s_app_name_title_dict[] = {
  0x00, 0x1b, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x05, 0x00, 0x43, 0x6c, 0x65, 0x61, 0x72,
  0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x4d, 0x6f, 0x62,
  0x69, 0x6c, 0x65, 0x53, 0x4d, 0x53, 0x01, 0x08, 0x00, 0x4D, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65,
  0x73, 0x02, 0x00, 0x00, 0x03, 0x1b, 0x00, 0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61,
  0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69,
  0x6f, 0x6e, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x31, 0x35, 0x54, 0x31, 0x36,
  0x31, 0x38, 0x33, 0x37
};

static TimelineItem s_app_name_title_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x1b,
    .timestamp = 1434385117,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 4,
    .attributes = (Attribute[4]) {
      { .id = AttributeIdBody, .cstring = "this is a test notification" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.apple.MobileSMS" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_GENERIC_SMS },
      { .id = AttributeIdBgColor, .uint8 = GColorIslamicGreenARGB8 },
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 1,
    .actions = (TimelineItemAction[1]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative},
            { .id = AttributeIdTitle, .cstring = "Dismiss" }}
         }
      }
    }
  }
};

//! Complete dict for a notification from an unrecognized app with a duplicated title
//! 00 1c 00 00 00 06 04 00  4c 69 6b 65 07 05 00 43   ........ Like...C
//! 6c 65 61 72 00 12 00 63  6f 6d 2e 74 65 73 74 73   lear...com.t ests
//! 2e 4e 6f 74 41 6e 41 70  70 01 07 00 46 61 6b 65   .NotAnApp... Fake
//! 41 70 70 02 00 00 03 1b  00 74 68 69 73 20 69 73   App......thi s is
//! 20 61 20 74 65 73 74 20  6e 6f 74 69 66 69 63 61    a test noti fica
//! 74 69 6f 6e 05 0f 00 32  30 31 35 30 36 31 35 54   tion...20150 615T
//! 31 36 31 38 33 37                                  161837

static const uint8_t s_unknown_app_dict[] = {
 0x00, 0x1c, 0x00, 0x00, 0x00, 0x06, 0x04, 0x00, 0x4c, 0x69, 0x6b, 0x65, 0x07, 0x05, 0x00, 0x43,
 0x6c, 0x65, 0x61, 0x72, 0x00, 0x12, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x73,
 0x2e, 0x4e, 0x6f, 0x74, 0x41, 0x6e, 0x41, 0x70, 0x70, 0x01, 0x07, 0x00, 0x46, 0x61, 0x6b, 0x65,
 0x41, 0x70, 0x70, 0x02, 0x00, 0x00, 0x03, 0x1b, 0x00, 0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73,
 0x20, 0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61,
 0x74, 0x69, 0x6f, 0x6e, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x31, 0x35, 0x54,
 0x31, 0x36, 0x31, 0x38, 0x33, 0x37
};

static TimelineItem s_unknown_app_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x1c,
    .timestamp = 1434385117,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 4,
    .attributes = (Attribute[4]) {
      { .id = AttributeIdAppName, .cstring = "FakeApp" },
      { .id = AttributeIdBody, .cstring = "this is a test notification" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.tests.NotAnApp" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_NOTIFICATION_GENERIC },
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 2,
    .actions = (TimelineItemAction[2]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsPositive,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDPositive},
            { .id = AttributeIdTitle, .cstring = "Like" }}
         }
      },
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative},
            { .id = AttributeIdTitle, .cstring = "Dismiss" }}
         }
      }
    }
  }
};

//! Complete dict for a notification from an unrecognized app with a unique title
//! 00 1d 00 00 00 06 00 00  07 05 00 43 6c 65 61 72   ........ ...Clear
//! 00 12 00 63 6f 6d 2e 74  65 73 74 73 2e 4e 6f 74   ...com.t ests.Not
//! 41 6e 41 70 70 01 0d 00  4d 65 73 73 61 67 65 20   AnApp... Message
//! 54 69 74 6c 65 02 00 00  03 1b 00 74 68 69 73 20   Title... ...this
//! 69 73 20 61 20 74 65 73  74 20 6e 6f 74 69 66 69   is a tes t notifi
//! 63 61 74 69 6f 6e 05 0f  00 32 30 31 35 30 36 31   cation.. .2015061
//! 35 54 31 36 31 38 33 37                            5T161837

static const uint8_t s_unknown_app_unique_title_dict[] = {
 0x00, 0x1d, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x05, 0x00, 0x43, 0x6c, 0x65, 0x61, 0x72,
 0x00, 0x12, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x74, 0x65, 0x73, 0x74, 0x73, 0x2e, 0x4e, 0x6f, 0x74,
 0x41, 0x6e, 0x41, 0x70, 0x70, 0x01, 0x0d, 0x00, 0x4d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20,
 0x54, 0x69, 0x74, 0x6c, 0x65, 0x02, 0x00, 0x00, 0x03, 0x1b, 0x00, 0x74, 0x68, 0x69, 0x73, 0x20,
 0x69, 0x73, 0x20, 0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x69, 0x66, 0x69,
 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x31,
 0x35, 0x54, 0x31, 0x36, 0x31, 0x38, 0x33, 0x37
};

static TimelineItem s_unknown_app_unique_title_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x1d,
    .timestamp = 1434385117,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 5,
    .attributes = (Attribute[5]) {
      { .id = AttributeIdTitle, .cstring = "Message Title" },
      { .id = AttributeIdAppName, .cstring = "FakeApp" },
      { .id = AttributeIdBody, .cstring = "this is a test notification" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.tests.NotAnApp" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_NOTIFICATION_GENERIC },
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 1,
    .actions = (TimelineItemAction[1]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative},
            { .id = AttributeIdTitle, .cstring = "Dismiss" }}
         }
      }
    }
  }
};

//! Complete dict for an MMS with an image
//! 00 1f 00 00 00 06 00 00  07 05 00 43 6c 65 61 72   .4.........Clear
//! 00 13 00 63 6f 6d 2e 61  70 70 6c 65 2e 4d 6f 62   ...com.apple.Mob
//! 69 6c 65 53 4d 53 01 08  00 48 75 79 20 54 72 61   ileSMS...Huy Tra
//! 6e 02 00 00 03 00 00 05  0f 00 32 30 31 36 30 31   n.........201601
//! 31 31 54 31 38 34 38 35  32                        11T184852

static const uint8_t s_mms_no_caption_dict[] = {
  0x00, 0x1f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x05, 0x00, 0x43, 0x6c, 0x65, 0x61, 0x72,
  0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x4d, 0x6f, 0x62,
  0x69, 0x6c, 0x65, 0x53, 0x4d, 0x53, 0x01, 0x08, 0x00, 0x48, 0x75, 0x79, 0x20, 0x54, 0x72, 0x61,
  0x6e, 0x02, 0x00, 0x00, 0x03, 0x00, 0x00, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x36, 0x30, 0x31,
  0x31, 0x31, 0x54, 0x31, 0x38, 0x34, 0x38, 0x35, 0x32
};

static TimelineItem s_mms_no_caption_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x1f,
    .timestamp = 1452538132,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 5,
    .attributes = (Attribute[5]) {
      { .id = AttributeIdSender, .cstring = "Huy Tran" },
      { .id = AttributeIdBody, .cstring = "Huy Tran sent an attachment 🎁" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.apple.MobileSMS" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_GENERIC_SMS },
      { .id = AttributeIdBgColor, .uint8 = GColorIslamicGreenARGB8 }
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 1,
    .actions = (TimelineItemAction[1]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative },
            { .id = AttributeIdTitle, .cstring = "Dismiss" }
          }
        }
      }
    }
  }
};

//! Complete dict for an MMS with an image and caption
//! 00 20 00 00 00 06 00 00  07 05 00 43 6c 65 61 72   .'......  ...Clear
//! 00 13 00 63 6f 6d 2e 61  70 70 6c 65 2e 4d 6f 62   ...com.a pple.Mob
//! 69 6c 65 53 4d 53 01 08  00 48 75 79 20 54 72 61   ileSMS..  .Huy Tra
//! 6e 02 00 00 03 10 00 20  43 68 65 63 6b 20 74 68   n......  Check th
//! 69 73 20 6f 75 74 21 05  0f 00 32 30 31 36 30 31   is out!.  ..201601
//! 31 31 54 32 31 30 34 31  35                        11T21041 5

static const uint8_t s_mms_with_caption_dict[] = {
  0x00, 0x20, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x05, 0x00, 0x43, 0x6c, 0x65, 0x61, 0x72,
  0x00, 0x13, 0x00, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x2e, 0x4d, 0x6f, 0x62,
  0x69, 0x6c, 0x65, 0x53, 0x4d, 0x53, 0x01, 0x08, 0x00, 0x48, 0x75, 0x79, 0x20, 0x54, 0x72, 0x61,
  0x6e, 0x02, 0x00, 0x00, 0x03, 0x10, 0x00, 0x20, 0x43, 0x68, 0x65, 0x63, 0x6b, 0x20, 0x74, 0x68,
  0x69, 0x73, 0x20, 0x6f, 0x75, 0x74, 0x21, 0x05, 0x0f, 0x00, 0x32, 0x30, 0x31, 0x36, 0x30, 0x31,
  0x31, 0x31, 0x54, 0x32, 0x31, 0x30, 0x34, 0x31, 0x35
};

static TimelineItem s_mms_with_caption_parsed_item = {
  .header = {
    .id = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    .ancs_uid = 0x20,
    .timestamp = 1452546255,
    .duration = 0,
    .type = TimelineItemTypeNotification,
    .layout = LayoutIdNotification,
    .ancs_notif = 1
  },
  .attr_list = (AttributeList) {
    .num_attributes = 5,
    .attributes = (Attribute[5]) {
      { .id = AttributeIdSender, .cstring = "Huy Tran" },
      { .id = AttributeIdBody, .cstring = "Huy Tran sent an attachment 🎁\n\"Check this out!\"" },
      { .id = AttributeIdiOSAppIdentifier, .cstring = "com.apple.MobileSMS" },
      { .id = AttributeIdIconTiny, .uint32 = TIMELINE_RESOURCE_GENERIC_SMS },
      { .id = AttributeIdBgColor, .uint8 = GColorIslamicGreenARGB8 }
    }
  },
  .action_group = (TimelineItemActionGroup) {
    .num_actions = 1,
    .actions = (TimelineItemAction[1]) {
      {
        .id = 0,
        .type = TimelineItemActionTypeAncsNegative,
        .attr_list = (AttributeList) {
          .num_attributes = 2,
          .attributes = (Attribute[2]) {
            { .id = AttributeIdAncsAction, .uint8 = ActionIDNegative },
            { .id = AttributeIdTitle, .cstring = "Dismiss" }
          }
        }
      }
    }
  }
};

