#include "util/crc32.h"

// Nybble-wide table driven CRC-32 algorithm
//
// A compromise between speed and size, this algorithm uses a lookup table to
// calculate the CRC four bits at a time with a size cost of only 64 bytes. By
// contrast, a byte-wide algorithm requires a lookup table sixteen times larger!
//
// The lookup table is generated by the crc32_lut.py

static const uint32_t s_lookup_table[] = {
  0x00000000, 0x1db71064, 0x3b6e20c8, 0x26d930ac,
  0x76dc4190, 0x6b6b51f4, 0x4db26158, 0x5005713c,
  0xedb88320, 0xf00f9344, 0xd6d6a3e8, 0xcb61b38c,
  0x9b64c2b0, 0x86d3d2d4, 0xa00ae278, 0xbdbdf21c,
};

uint32_t crc32(uint32_t crc, const void * restrict data, size_t length) {
  if (data == 0) {
    return 0;
  }
  const uint8_t * restrict bytes = data;

  crc ^= 0xffffffff;
  while (length--) {
    crc = (crc >> 4) ^ s_lookup_table[(crc ^ *bytes) & 0xf];
    crc = (crc >> 4) ^ s_lookup_table[(crc ^ (*bytes >> 4)) & 0xf];
    bytes++;
  }
  crc ^= 0xffffffff;
  return crc;
}
