/* SPDX-FileCopyrightText: 2016 Samsung Electronics Co., Ltd. */
/* SPDX-FileCopyrightText: 2016 University of Szeged. */
/* SPDX-License-Identifier: Apache-2.0 */

/*
 * Unit test generator for jerry-libm.
 * To be compiled separately from the rest of jerry and to be linked to a trusted libm.
 * Its output should be redirected to test-libm.inc.h.
 *
 * Example:
 * gcc gen-test-libm.c -o gen-test-libm -lm
 * ./gen-test-libm >test-libm.inc.h
 */

#include <math.h>
#include <stdio.h>

#define GEN_INT_TEST(EXPR) printf("check_int (\"%s\", %s, %d);\n", #EXPR, #EXPR, EXPR);
#define GEN_DBL_TEST(EXPR) printf("check_double (\"%s\", %s, %.20E);\n", #EXPR, #EXPR, EXPR);

int
main (int argc, char **args)
{
  printf ("/* Copyright 2016 Samsung Electronics Co., Ltd.\n"
          " * Copyright 2016 University of Szeged.\n"
          " *\n"
          " * Licensed under the Apache License, Version 2.0 (the \"License\");\n"
          " * you may not use this file except in compliance with the License.\n"
          " * You may obtain a copy of the License at\n"
          " *\n"
          " *     http://www.apache.org/licenses/LICENSE-2.0\n"
          " *\n"
          " * Unless required by applicable law or agreed to in writing, software\n"
          " * distributed under the License is distributed on an \"AS IS\" BASIS\n"
          " * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n"
          " * See the License for the specific language governing permissions and\n"
          " * limitations under the License.\n"
          " */\n"
          "\n"
          "/*\n"
          " * Generated by tools/gen-test-libm.sh\n"
          " * DO NOT EDIT!!!\n"
          " */\n");

  /* acos tests */
  GEN_DBL_TEST (acos (0.0));
  GEN_DBL_TEST (acos (-0.0));
  GEN_DBL_TEST (acos (1.0));
  GEN_DBL_TEST (acos (-1.0));
  GEN_DBL_TEST (acos (0.5));
  GEN_DBL_TEST (acos (-0.5));
  GEN_DBL_TEST (acos (INFINITY));
  GEN_DBL_TEST (acos (-INFINITY));
  GEN_DBL_TEST (acos (NAN));
  GEN_DBL_TEST (acos (6.9e-18));
  GEN_DBL_TEST (acos (-6.9e-18));
  GEN_DBL_TEST (acos (7.0e-18));
  GEN_DBL_TEST (acos (-7.0e-18));
  GEN_DBL_TEST (acos (7.4e-9));
  GEN_DBL_TEST (acos (-7.4e-9));
  GEN_DBL_TEST (acos (7.5e-9));
  GEN_DBL_TEST (acos (-7.5e-9));
  GEN_DBL_TEST (acos (0.1));
  GEN_DBL_TEST (acos (-0.1));
  GEN_DBL_TEST (acos (0.4));
  GEN_DBL_TEST (acos (-0.4));
  GEN_DBL_TEST (acos (0.6));
  GEN_DBL_TEST (acos (-0.6));
  GEN_DBL_TEST (acos (0.99));
  GEN_DBL_TEST (acos (-0.99));
  GEN_DBL_TEST (acos (1.1));
  GEN_DBL_TEST (acos (-1.1));
  GEN_DBL_TEST (acos (0.7));

  /* asin tests*/
  GEN_DBL_TEST (asin (0.0));
  GEN_DBL_TEST (asin (-0.0));
  GEN_DBL_TEST (asin (1.0));
  GEN_DBL_TEST (asin (-1.0));
  GEN_DBL_TEST (asin (0.5));
  GEN_DBL_TEST (asin (-0.5));
  GEN_DBL_TEST (asin (0.98));
  GEN_DBL_TEST (asin (-0.98));
  GEN_DBL_TEST (asin (INFINITY));
  GEN_DBL_TEST (asin (-INFINITY));
  GEN_DBL_TEST (asin (NAN));
  GEN_DBL_TEST (asin (6.9e-18));
  GEN_DBL_TEST (asin (-6.9e-18));
  GEN_DBL_TEST (asin (7.0e-18));
  GEN_DBL_TEST (asin (-7.0e-18));
  GEN_DBL_TEST (asin (7.4e-9));
  GEN_DBL_TEST (asin (-7.4e-9));
  GEN_DBL_TEST (asin (7.5e-9));
  GEN_DBL_TEST (asin (-7.5e-9));
  GEN_DBL_TEST (asin (0.1));
  GEN_DBL_TEST (asin (-0.1));
  GEN_DBL_TEST (asin (0.4));
  GEN_DBL_TEST (asin (-0.4));
  GEN_DBL_TEST (asin (0.6));
  GEN_DBL_TEST (asin (-0.6));
  GEN_DBL_TEST (asin (0.97));
  GEN_DBL_TEST (asin (-0.97));
  GEN_DBL_TEST (asin (0.99));
  GEN_DBL_TEST (asin (-0.99));
  GEN_DBL_TEST (asin (1.1));
  GEN_DBL_TEST (asin (-1.1));
  GEN_DBL_TEST (asin (0.7));

  /* atan tests*/
  GEN_DBL_TEST (atan (0.0));
  GEN_DBL_TEST (atan (-0.0));
  GEN_DBL_TEST (atan (7.0 / 16.0));
  GEN_DBL_TEST (atan (-7.0 / 16.0));
  GEN_DBL_TEST (atan (11.0 / 16.0));
  GEN_DBL_TEST (atan (-11.0 / 16.0));
  GEN_DBL_TEST (atan (19.0 / 16.0));
  GEN_DBL_TEST (atan (-19.0 / 16.0));
  GEN_DBL_TEST (atan (39.0 / 16.0));
  GEN_DBL_TEST (atan (-39.0 / 16.0));
  GEN_DBL_TEST (atan (1.0));
  GEN_DBL_TEST (atan (-1.0));
  GEN_DBL_TEST (atan (INFINITY));
  GEN_DBL_TEST (atan (-INFINITY));
  GEN_DBL_TEST (atan (NAN));
  GEN_DBL_TEST (atan (6.9 / 16.0));
  GEN_DBL_TEST (atan (-6.9 / 16.0));
  GEN_DBL_TEST (atan (7.1 / 16.0));
  GEN_DBL_TEST (atan (-7.1 / 16.0));
  GEN_DBL_TEST (atan (10.9 / 16.0));
  GEN_DBL_TEST (atan (-10.9 / 16.0));
  GEN_DBL_TEST (atan (11.1 / 16.0));
  GEN_DBL_TEST (atan (-11.1 / 16.0));
  GEN_DBL_TEST (atan (18.9 / 16.0));
  GEN_DBL_TEST (atan (-18.9 / 16.0));
  GEN_DBL_TEST (atan (19.1 / 16.0));
  GEN_DBL_TEST (atan (-19.1 / 16.0));
  GEN_DBL_TEST (atan (38.9 / 16.0));
  GEN_DBL_TEST (atan (-38.9 / 16.0));
  GEN_DBL_TEST (atan (39.1 / 16.0));
  GEN_DBL_TEST (atan (-39.1 / 16.0));
  GEN_DBL_TEST (atan (0.99));
  GEN_DBL_TEST (atan (-0.99));
  GEN_DBL_TEST (atan (1.1));
  GEN_DBL_TEST (atan (-1.1));
  GEN_DBL_TEST (atan (7.37e+19));
  GEN_DBL_TEST (atan (-7.37e+19));
  GEN_DBL_TEST (atan (7.38e+19));
  GEN_DBL_TEST (atan (-7.38e+19));
  GEN_DBL_TEST (atan (0.7));

  /* atan2 tests*/
  GEN_DBL_TEST (atan2 (NAN, NAN));
  GEN_DBL_TEST (atan2 (0.0, NAN));
  GEN_DBL_TEST (atan2 (-0.0, NAN));
  GEN_DBL_TEST (atan2 (1.0, NAN));
  GEN_DBL_TEST (atan2 (-1.0, NAN));
  GEN_DBL_TEST (atan2 (INFINITY, NAN));
  GEN_DBL_TEST (atan2 (-INFINITY, NAN));
  GEN_DBL_TEST (atan2 (NAN, 0.0));
  GEN_DBL_TEST (atan2 (NAN, -0.0));
  GEN_DBL_TEST (atan2 (NAN, 1.0));
  GEN_DBL_TEST (atan2 (NAN, -1.0));
  GEN_DBL_TEST (atan2 (NAN, INFINITY));
  GEN_DBL_TEST (atan2 (NAN, -INFINITY));
  GEN_DBL_TEST (atan2 (0.0, 0.0));
  GEN_DBL_TEST (atan2 (0.0, -0.0));
  GEN_DBL_TEST (atan2 (-0.0, 0.0));
  GEN_DBL_TEST (atan2 (-0.0, -0.0));
  GEN_DBL_TEST (atan2 (0.0, 1.0));
  GEN_DBL_TEST (atan2 (0.0, -1.0));
  GEN_DBL_TEST (atan2 (0.0, INFINITY));
  GEN_DBL_TEST (atan2 (0.0, -INFINITY));
  GEN_DBL_TEST (atan2 (-0.0, 1.0));
  GEN_DBL_TEST (atan2 (-0.0, -1.0));
  GEN_DBL_TEST (atan2 (-0.0, INFINITY));
  GEN_DBL_TEST (atan2 (-0.0, -INFINITY));
  GEN_DBL_TEST (atan2 (1.0, 0.0));
  GEN_DBL_TEST (atan2 (1.0, -0.0));
  GEN_DBL_TEST (atan2 (INFINITY, 0.0));
  GEN_DBL_TEST (atan2 (INFINITY, -0.0));
  GEN_DBL_TEST (atan2 (-1.0, 0.0));
  GEN_DBL_TEST (atan2 (-1.0, -0.0));
  GEN_DBL_TEST (atan2 (-INFINITY, 0.0));
  GEN_DBL_TEST (atan2 (-INFINITY, -0.0));
  GEN_DBL_TEST (atan2 (1.0, INFINITY));
  GEN_DBL_TEST (atan2 (-1.0, INFINITY));
  GEN_DBL_TEST (atan2 (1.0, -INFINITY));
  GEN_DBL_TEST (atan2 (-1.0, -INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, -INFINITY));
  GEN_DBL_TEST (atan2 (-INFINITY, INFINITY));
  GEN_DBL_TEST (atan2 (-INFINITY, -INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, 1.0));
  GEN_DBL_TEST (atan2 (INFINITY, -1.0));
  GEN_DBL_TEST (atan2 (-INFINITY, 1.0));
  GEN_DBL_TEST (atan2 (-INFINITY, -1.0));
  GEN_DBL_TEST (atan2 (0.7, 1.0));
  GEN_DBL_TEST (atan2 (-0.7, 1.0));
  GEN_DBL_TEST (atan2 (0.7, -1.0));
  GEN_DBL_TEST (atan2 (-0.7, -1.0));
  GEN_DBL_TEST (atan2 (0.4, 0.0003));
  GEN_DBL_TEST (atan2 (1.4, -0.93));

  /* ceil tests */
  GEN_DBL_TEST (ceil (0.0));
  GEN_DBL_TEST (ceil (-0.0));
  GEN_DBL_TEST (ceil (INFINITY));
  GEN_DBL_TEST (ceil (-INFINITY));
  GEN_DBL_TEST (ceil (NAN));
  GEN_DBL_TEST (ceil (3.14));
  GEN_DBL_TEST (ceil (-3.14));
  GEN_DBL_TEST (ceil (3.72e-09));
  GEN_DBL_TEST (ceil (-3.72e-09));
  GEN_DBL_TEST (ceil (7.37e+19));
  GEN_DBL_TEST (ceil (-7.37e+19));

  /* copysign tests */
  /* SKIPPED: not publicly declared in jerry-libm
  GEN_DBL_TEST (copysign (0.0, 0.0));
  GEN_DBL_TEST (copysign (0.0, -0.0));
  GEN_DBL_TEST (copysign (-0.0, 0.0));
  GEN_DBL_TEST (copysign (-0.0, -0.0));
  GEN_DBL_TEST (copysign (0.0, 1.0));
  GEN_DBL_TEST (copysign (0.0, -1.0));
  GEN_DBL_TEST (copysign (-0.0, 1.0));
  GEN_DBL_TEST (copysign (-0.0, -1.0));
  GEN_DBL_TEST (copysign (0.0, INFINITY));
  GEN_DBL_TEST (copysign (0.0, -INFINITY));
  GEN_DBL_TEST (copysign (-0.0, INFINITY));
  GEN_DBL_TEST (copysign (-0.0, -INFINITY));
  GEN_DBL_TEST (copysign (0.0, NAN));
  GEN_DBL_TEST (copysign (-0.0, NAN));
  GEN_DBL_TEST (copysign (1.0, 0.0));
  GEN_DBL_TEST (copysign (1.0, -0.0));
  GEN_DBL_TEST (copysign (-1.0, 0.0));
  GEN_DBL_TEST (copysign (-1.0, -0.0));
  GEN_DBL_TEST (copysign (1.0, 1.0));
  GEN_DBL_TEST (copysign (1.0, -1.0));
  GEN_DBL_TEST (copysign (-1.0, 1.0));
  GEN_DBL_TEST (copysign (-1.0, -1.0));
  GEN_DBL_TEST (copysign (1.0, INFINITY));
  GEN_DBL_TEST (copysign (1.0, -INFINITY));
  GEN_DBL_TEST (copysign (-1.0, INFINITY));
  GEN_DBL_TEST (copysign (-1.0, -INFINITY));
  GEN_DBL_TEST (copysign (1.0, NAN));
  GEN_DBL_TEST (copysign (-1.0, NAN));
  GEN_DBL_TEST (copysign (INFINITY, 0.0));
  GEN_DBL_TEST (copysign (INFINITY, -0.0));
  GEN_DBL_TEST (copysign (-INFINITY, 0.0));
  GEN_DBL_TEST (copysign (-INFINITY, -0.0));
  GEN_DBL_TEST (copysign (INFINITY, 1.0));
  GEN_DBL_TEST (copysign (INFINITY, -1.0));
  GEN_DBL_TEST (copysign (-INFINITY, 1.0));
  GEN_DBL_TEST (copysign (-INFINITY, -1.0));
  GEN_DBL_TEST (copysign (INFINITY, INFINITY));
  GEN_DBL_TEST (copysign (INFINITY, -INFINITY));
  GEN_DBL_TEST (copysign (-INFINITY, INFINITY));
  GEN_DBL_TEST (copysign (-INFINITY, -INFINITY));
  GEN_DBL_TEST (copysign (INFINITY, NAN));
  GEN_DBL_TEST (copysign (-INFINITY, NAN));
  GEN_DBL_TEST (copysign (NAN, 0.0));
  GEN_DBL_TEST (copysign (NAN, -0.0));
  GEN_DBL_TEST (copysign (NAN, 1.0));
  GEN_DBL_TEST (copysign (NAN, -1.0));
  GEN_DBL_TEST (copysign (NAN, INFINITY));
  GEN_DBL_TEST (copysign (NAN, -INFINITY));
  GEN_DBL_TEST (copysign (NAN, NAN));
  GEN_DBL_TEST (copysign (3.14, -1.0));
  GEN_DBL_TEST (copysign (-3.14, 1.0));
  GEN_DBL_TEST (copysign (1.0, -3.14));
  GEN_DBL_TEST (copysign (-1.0, 3.14));
  */

  /* exp tests */
  GEN_DBL_TEST (exp (0.0));
  GEN_DBL_TEST (exp (-0.0));
  GEN_DBL_TEST (exp (1.0));
  GEN_DBL_TEST (exp (-1.0));
  GEN_DBL_TEST (exp (INFINITY));
  GEN_DBL_TEST (exp (-INFINITY));
  GEN_DBL_TEST (exp (NAN));
  GEN_DBL_TEST (exp (7.08e+02));
  GEN_DBL_TEST (exp (7.10e+02));
  GEN_DBL_TEST (exp (-7.40e+02));
  GEN_DBL_TEST (exp (-7.50e+02));
  GEN_DBL_TEST (exp (0.34));
  GEN_DBL_TEST (exp (-0.34));
  GEN_DBL_TEST (exp (0.35));
  GEN_DBL_TEST (exp (-0.35));
  GEN_DBL_TEST (exp (1.03));
  GEN_DBL_TEST (exp (-1.03));
  GEN_DBL_TEST (exp (1.04));
  GEN_DBL_TEST (exp (-1.04));
  GEN_DBL_TEST (exp (3.72e-09));
  GEN_DBL_TEST (exp (-3.72e-09));
  GEN_DBL_TEST (exp (3.73e-09));
  GEN_DBL_TEST (exp (-3.73e-09));
  GEN_DBL_TEST (exp (2.0));
  GEN_DBL_TEST (exp (3.0));
  GEN_DBL_TEST (exp (0.7));
  GEN_DBL_TEST (exp (38.0));

  /* fabs tests */
  GEN_DBL_TEST (fabs (0.0));
  GEN_DBL_TEST (fabs (-0.0));
  GEN_DBL_TEST (fabs (1.0));
  GEN_DBL_TEST (fabs (-1.0));
  GEN_DBL_TEST (fabs (INFINITY));
  GEN_DBL_TEST (fabs (-INFINITY));
  GEN_DBL_TEST (fabs (NAN));
  GEN_DBL_TEST (fabs (3.14));
  GEN_DBL_TEST (fabs (-3.14));
  GEN_DBL_TEST (fabs (0.7));
  GEN_DBL_TEST (fabs (-0.7));
  GEN_DBL_TEST (fabs (3.72e-09));
  GEN_DBL_TEST (fabs (-3.72e-09));
  GEN_DBL_TEST (fabs (7.37e+19));
  GEN_DBL_TEST (fabs (-7.37e+19));

  /* finite tests */
  /* SKIPPED: not publicly declared in jerry-libm
  GEN_INT_TEST (finite (0.0));
  GEN_INT_TEST (finite (-0.0));
  GEN_INT_TEST (finite (1.0));
  GEN_INT_TEST (finite (-1.0));
  GEN_INT_TEST (finite (INFINITY));
  GEN_INT_TEST (finite (-INFINITY));
  GEN_INT_TEST (finite (NAN));
  GEN_INT_TEST (finite (3.14));
  GEN_INT_TEST (finite (-3.14));
  GEN_INT_TEST (finite (0.7));
  GEN_INT_TEST (finite (-0.7));
  GEN_INT_TEST (finite (3.72e-09));
  GEN_INT_TEST (finite (-3.72e-09));
  GEN_INT_TEST (finite (7.37e+19));
  GEN_INT_TEST (finite (-7.37e+19));
  */

  /* floor tests */
  GEN_DBL_TEST (floor (0.0));
  GEN_DBL_TEST (floor (-0.0));
  GEN_DBL_TEST (floor (INFINITY));
  GEN_DBL_TEST (floor (-INFINITY));
  GEN_DBL_TEST (floor (NAN));
  GEN_DBL_TEST (floor (3.14));
  GEN_DBL_TEST (floor (-3.14));
  GEN_DBL_TEST (floor (3.72e-09));
  GEN_DBL_TEST (floor (-3.72e-09));
  GEN_DBL_TEST (floor (7.37e+19));
  GEN_DBL_TEST (floor (-7.37e+19));

  /* fmod tests */
  GEN_DBL_TEST (fmod (0.0, 0.0));
  GEN_DBL_TEST (fmod (0.0, -0.0));
  GEN_DBL_TEST (fmod (-0.0, 0.0));
  GEN_DBL_TEST (fmod (-0.0, -0.0));
  GEN_DBL_TEST (fmod (0.0, 3.0));
  GEN_DBL_TEST (fmod (0.0, -3.0));
  GEN_DBL_TEST (fmod (-0.0, 3.0));
  GEN_DBL_TEST (fmod (-0.0, -3.0));
  GEN_DBL_TEST (fmod (0.0, INFINITY));
  GEN_DBL_TEST (fmod (0.0, -INFINITY));
  GEN_DBL_TEST (fmod (-0.0, INFINITY));
  GEN_DBL_TEST (fmod (-0.0, -INFINITY));
  GEN_DBL_TEST (fmod (0.0, NAN));
  GEN_DBL_TEST (fmod (-0.0, NAN));
  GEN_DBL_TEST (fmod (3.0, 0.0));
  GEN_DBL_TEST (fmod (3.0, -0.0));
  GEN_DBL_TEST (fmod (-3.0, 0.0));
  GEN_DBL_TEST (fmod (-3.0, -0.0));
  GEN_DBL_TEST (fmod (3.0, 3.0));
  GEN_DBL_TEST (fmod (3.0, -3.0));
  GEN_DBL_TEST (fmod (-3.0, 3.0));
  GEN_DBL_TEST (fmod (-3.0, -3.0));
  GEN_DBL_TEST (fmod (3.0, INFINITY));
  GEN_DBL_TEST (fmod (3.0, -INFINITY));
  GEN_DBL_TEST (fmod (-3.0, INFINITY));
  GEN_DBL_TEST (fmod (-3.0, -INFINITY));
  GEN_DBL_TEST (fmod (3.0, NAN));
  GEN_DBL_TEST (fmod (-3.0, NAN));
  GEN_DBL_TEST (fmod (INFINITY, 0.0));
  GEN_DBL_TEST (fmod (INFINITY, -0.0));
  GEN_DBL_TEST (fmod (-INFINITY, 0.0));
  GEN_DBL_TEST (fmod (-INFINITY, -0.0));
  GEN_DBL_TEST (fmod (INFINITY, 3.0));
  GEN_DBL_TEST (fmod (INFINITY, -3.0));
  GEN_DBL_TEST (fmod (-INFINITY, 3.0));
  GEN_DBL_TEST (fmod (-INFINITY, -3.0));
  GEN_DBL_TEST (fmod (INFINITY, INFINITY));
  GEN_DBL_TEST (fmod (INFINITY, -INFINITY));
  GEN_DBL_TEST (fmod (-INFINITY, INFINITY));
  GEN_DBL_TEST (fmod (-INFINITY, -INFINITY));
  GEN_DBL_TEST (fmod (INFINITY, NAN));
  GEN_DBL_TEST (fmod (-INFINITY, NAN));
  GEN_DBL_TEST (fmod (NAN, 0.0));
  GEN_DBL_TEST (fmod (NAN, -0.0));
  GEN_DBL_TEST (fmod (NAN, 3.0));
  GEN_DBL_TEST (fmod (NAN, -3.0));
  GEN_DBL_TEST (fmod (NAN, INFINITY));
  GEN_DBL_TEST (fmod (NAN, -INFINITY));
  GEN_DBL_TEST (fmod (NAN, NAN));
  GEN_DBL_TEST (fmod (3.0, 1.0));
  GEN_DBL_TEST (fmod (3.0, -1.0));
  GEN_DBL_TEST (fmod (-3.0, 1.0));
  GEN_DBL_TEST (fmod (-3.0, -1.0));
  GEN_DBL_TEST (fmod (6.5, 2.3));
  GEN_DBL_TEST (fmod (6.5, -2.3));
  GEN_DBL_TEST (fmod (-6.5, 2.3));
  GEN_DBL_TEST (fmod (-6.5, -2.3));

  /* isnan tests */
  GEN_INT_TEST (isnan (0.0));
  GEN_INT_TEST (isnan (-0.0));
  GEN_INT_TEST (isnan (1.0));
  GEN_INT_TEST (isnan (-1.0));
  GEN_INT_TEST (isnan (INFINITY));
  GEN_INT_TEST (isnan (-INFINITY));
  GEN_INT_TEST (isnan (NAN));
  GEN_INT_TEST (isnan (3.14));
  GEN_INT_TEST (isnan (-3.14));
  GEN_INT_TEST (isnan (0.7));
  GEN_INT_TEST (isnan (-0.7));
  GEN_INT_TEST (isnan (3.72e-09));
  GEN_INT_TEST (isnan (-3.72e-09));
  GEN_INT_TEST (isnan (7.37e+19));
  GEN_INT_TEST (isnan (-7.37e+19));

  /* log tests */
  GEN_DBL_TEST (log (0.0));
  GEN_DBL_TEST (log (-0.0));
  GEN_DBL_TEST (log (1.0));
  GEN_DBL_TEST (log (-1.0));
  GEN_DBL_TEST (log (INFINITY));
  GEN_DBL_TEST (log (-INFINITY));
  GEN_DBL_TEST (log (NAN));
  GEN_DBL_TEST (log (M_E));
  GEN_DBL_TEST (log (1.0 / M_E));
  GEN_DBL_TEST (log (2));
  GEN_DBL_TEST (log (10));
  GEN_DBL_TEST (log (0.7));
  GEN_DBL_TEST (log (2.22e-308));
  GEN_DBL_TEST (log (2.23e-308));
  GEN_DBL_TEST (log (0.17));
  GEN_DBL_TEST (log (0.18));
  GEN_DBL_TEST (log (1999.0));
  GEN_DBL_TEST (log (2000.0));
  GEN_DBL_TEST (log (2001.0));

  /* pow tests */
  GEN_DBL_TEST (pow (0.0, 0.0));
  GEN_DBL_TEST (pow (0.0, -0.0));
  GEN_DBL_TEST (pow (-0.0, 0.0));
  GEN_DBL_TEST (pow (-0.0, -0.0));
  GEN_DBL_TEST (pow (0.0, 1.0));
  GEN_DBL_TEST (pow (0.0, -1.0));
  GEN_DBL_TEST (pow (-0.0, 1.0));
  GEN_DBL_TEST (pow (-0.0, -1.0));
  GEN_DBL_TEST (pow (0.0, INFINITY));
  GEN_DBL_TEST (pow (0.0, -INFINITY));
  GEN_DBL_TEST (pow (-0.0, INFINITY));
  GEN_DBL_TEST (pow (-0.0, -INFINITY));
  GEN_DBL_TEST (pow (0.0, NAN));
  GEN_DBL_TEST (pow (-0.0, NAN));
  GEN_DBL_TEST (pow (1.0, 0.0));
  GEN_DBL_TEST (pow (1.0, -0.0));
  GEN_DBL_TEST (pow (-1.0, 0.0));
  GEN_DBL_TEST (pow (-1.0, -0.0));
  GEN_DBL_TEST (pow (1.0, 1.0));
  GEN_DBL_TEST (pow (1.0, -1.0));
  GEN_DBL_TEST (pow (-1.0, 1.0));
  GEN_DBL_TEST (pow (-1.0, -1.0));
  GEN_DBL_TEST (pow (1.0, INFINITY));
  GEN_DBL_TEST (pow (1.0, -INFINITY));
  GEN_DBL_TEST (pow (-1.0, INFINITY));
  GEN_DBL_TEST (pow (-1.0, -INFINITY));
  GEN_DBL_TEST (pow (1.0, NAN));
  GEN_DBL_TEST (pow (-1.0, NAN));
  GEN_DBL_TEST (pow (INFINITY, 0.0));
  GEN_DBL_TEST (pow (INFINITY, -0.0));
  GEN_DBL_TEST (pow (-INFINITY, 0.0));
  GEN_DBL_TEST (pow (-INFINITY, -0.0));
  GEN_DBL_TEST (pow (INFINITY, 1.0));
  GEN_DBL_TEST (pow (INFINITY, -1.0));
  GEN_DBL_TEST (pow (-INFINITY, 1.0));
  GEN_DBL_TEST (pow (-INFINITY, -1.0));
  GEN_DBL_TEST (pow (INFINITY, INFINITY));
  GEN_DBL_TEST (pow (INFINITY, -INFINITY));
  GEN_DBL_TEST (pow (-INFINITY, INFINITY));
  GEN_DBL_TEST (pow (-INFINITY, -INFINITY));
  GEN_DBL_TEST (pow (INFINITY, NAN));
  GEN_DBL_TEST (pow (-INFINITY, NAN));
  GEN_DBL_TEST (pow (NAN, 0.0));
  GEN_DBL_TEST (pow (NAN, -0.0));
  GEN_DBL_TEST (pow (NAN, 1.0));
  GEN_DBL_TEST (pow (NAN, -1.0));
  GEN_DBL_TEST (pow (NAN, INFINITY));
  GEN_DBL_TEST (pow (NAN, -INFINITY));
  GEN_DBL_TEST (pow (NAN, NAN));
  GEN_DBL_TEST (pow (0.9, INFINITY));
  GEN_DBL_TEST (pow (0.9, -INFINITY));
  GEN_DBL_TEST (pow (-0.9, INFINITY));
  GEN_DBL_TEST (pow (-0.9, -INFINITY));
  GEN_DBL_TEST (pow (1.1, INFINITY));
  GEN_DBL_TEST (pow (1.1, -INFINITY));
  GEN_DBL_TEST (pow (-1.1, INFINITY));
  GEN_DBL_TEST (pow (-1.1, -INFINITY));
  GEN_DBL_TEST (pow (0.0, 2.0));
  GEN_DBL_TEST (pow (0.0, -2.0));
  GEN_DBL_TEST (pow (-0.0, 2.0));
  GEN_DBL_TEST (pow (-0.0, -2.0));
  GEN_DBL_TEST (pow (0.0, 3.0));
  GEN_DBL_TEST (pow (0.0, -3.0));
  GEN_DBL_TEST (pow (-0.0, 3.0));
  GEN_DBL_TEST (pow (-0.0, -3.0));
  GEN_DBL_TEST (pow (0.0, 3.14));
  GEN_DBL_TEST (pow (0.0, -3.14));
  GEN_DBL_TEST (pow (-0.0, 3.14));
  GEN_DBL_TEST (pow (-0.0, -3.14));
  GEN_DBL_TEST (pow (1.0, 3.14));
  GEN_DBL_TEST (pow (1.0, -3.14));
  GEN_DBL_TEST (pow (-1.0, 3.14));
  GEN_DBL_TEST (pow (-1.0, -3.14));
  GEN_DBL_TEST (pow (3.14, 0.0));
  GEN_DBL_TEST (pow (3.14, -0.0));
  GEN_DBL_TEST (pow (-3.14, 0.0));
  GEN_DBL_TEST (pow (-3.14, -0.0));
  GEN_DBL_TEST (pow (3.14, 1.0));
  GEN_DBL_TEST (pow (3.14, -1.0));
  GEN_DBL_TEST (pow (-3.14, 1.0));
  GEN_DBL_TEST (pow (-3.14, -1.0));
  GEN_DBL_TEST (pow (3.14, 2.0));
  GEN_DBL_TEST (pow (3.14, -2.0));
  GEN_DBL_TEST (pow (-3.14, 2.0));
  GEN_DBL_TEST (pow (-3.14, -2.0));
  GEN_DBL_TEST (pow (3.14, 3.0));
  GEN_DBL_TEST (pow (3.14, -3.0));
  GEN_DBL_TEST (pow (-3.14, 3.0));
  GEN_DBL_TEST (pow (-3.14, -3.0));
  GEN_DBL_TEST (pow (3.14, 3.14));
  GEN_DBL_TEST (pow (3.14, -3.14));
  GEN_DBL_TEST (pow (-3.14, 3.14));
  GEN_DBL_TEST (pow (-3.14, -3.14));
  GEN_DBL_TEST (pow (INFINITY, 2.0));
  GEN_DBL_TEST (pow (INFINITY, -2.0));
  GEN_DBL_TEST (pow (-INFINITY, 2.0));
  GEN_DBL_TEST (pow (-INFINITY, -2.0));
  GEN_DBL_TEST (pow (INFINITY, 3.0));
  GEN_DBL_TEST (pow (INFINITY, -3.0));
  GEN_DBL_TEST (pow (-INFINITY, 3.0));
  GEN_DBL_TEST (pow (-INFINITY, -3.0));
  GEN_DBL_TEST (pow (INFINITY, 3.14));
  GEN_DBL_TEST (pow (INFINITY, -3.14));
  GEN_DBL_TEST (pow (-INFINITY, 3.14));
  GEN_DBL_TEST (pow (-INFINITY, -3.14));
  GEN_DBL_TEST (pow (0.7, 1.2));

  /* scalbn tests */
  /* SKIPPED: not publicly declared in jerry-libm
  GEN_DBL_TEST (scalbn (0.0, 0));
  GEN_DBL_TEST (scalbn (-0.0, 0));
  GEN_DBL_TEST (scalbn (0.0, 1));
  GEN_DBL_TEST (scalbn (0.0, -1));
  GEN_DBL_TEST (scalbn (-0.0, 1));
  GEN_DBL_TEST (scalbn (-0.0, -1));
  GEN_DBL_TEST (scalbn (1.0, 0));
  GEN_DBL_TEST (scalbn (-1.0, 0));
  GEN_DBL_TEST (scalbn (1.0, 1));
  GEN_DBL_TEST (scalbn (1.0, -1));
  GEN_DBL_TEST (scalbn (-1.0, 1));
  GEN_DBL_TEST (scalbn (-1.0, -1));
  GEN_DBL_TEST (scalbn (INFINITY, 0));
  GEN_DBL_TEST (scalbn (-INFINITY, 0));
  GEN_DBL_TEST (scalbn (INFINITY, 1));
  GEN_DBL_TEST (scalbn (INFINITY, -1));
  GEN_DBL_TEST (scalbn (-INFINITY, 1));
  GEN_DBL_TEST (scalbn (-INFINITY, -1));
  GEN_DBL_TEST (scalbn (NAN, 0));
  GEN_DBL_TEST (scalbn (NAN, 1));
  GEN_DBL_TEST (scalbn (NAN, -1));
  GEN_DBL_TEST (scalbn (3.14, -1));
  GEN_DBL_TEST (scalbn (-3.14, 1));
  GEN_DBL_TEST (scalbn (0.7, 4));
  GEN_DBL_TEST (scalbn (0.7, -4));
  GEN_DBL_TEST (scalbn (-0.7, 4));
  GEN_DBL_TEST (scalbn (-0.7, -4));
  GEN_DBL_TEST (scalbn (0.8, 5));
  GEN_DBL_TEST (scalbn (0.8, -5));
  GEN_DBL_TEST (scalbn (-0.8, 5));
  GEN_DBL_TEST (scalbn (-0.8, -5));
  GEN_DBL_TEST (scalbn (5.55e-18, 49999));
  GEN_DBL_TEST (scalbn (5.55e-18, 50000));
  GEN_DBL_TEST (scalbn (5.55e-18, 50001));
  GEN_DBL_TEST (scalbn (1.0, -49999));
  GEN_DBL_TEST (scalbn (1.0, -50000));
  GEN_DBL_TEST (scalbn (1.0, -50001));
  */

  /* sqrt tests */
  GEN_DBL_TEST (sqrt (0.0));
  GEN_DBL_TEST (sqrt (-0.0));
  GEN_DBL_TEST (sqrt (1.0));
  GEN_DBL_TEST (sqrt (-1.0));
  GEN_DBL_TEST (sqrt (INFINITY));
  GEN_DBL_TEST (sqrt (-INFINITY));
  GEN_DBL_TEST (sqrt (NAN));
  GEN_DBL_TEST (sqrt (0.7));
  GEN_DBL_TEST (sqrt (2));
  GEN_DBL_TEST (sqrt (10));
  GEN_DBL_TEST (sqrt (2.22e-308));
  GEN_DBL_TEST (sqrt (2.23e-308));
  GEN_DBL_TEST (sqrt (3.72e-09));
  GEN_DBL_TEST (sqrt (7.37e+19));
  GEN_DBL_TEST (sqrt (2209));
  GEN_DBL_TEST (sqrt (4));
  GEN_DBL_TEST (sqrt (0.25));
  GEN_DBL_TEST (sqrt (6642.25));
  GEN_DBL_TEST (sqrt (15239.9025));

  /* sin tests */
  GEN_DBL_TEST (sin (0.0));
  GEN_DBL_TEST (sin (-0.0));
  GEN_DBL_TEST (sin (1.0));
  GEN_DBL_TEST (sin (-1.0));
  GEN_DBL_TEST (sin (INFINITY));
  GEN_DBL_TEST (sin (-INFINITY));
  GEN_DBL_TEST (sin (NAN));
  GEN_DBL_TEST (sin (M_PI));
  GEN_DBL_TEST (sin (-M_PI));
  GEN_DBL_TEST (sin (2.0 * M_PI));
  GEN_DBL_TEST (sin (-2.0 * M_PI));
  GEN_DBL_TEST (sin (M_PI / 2.0));
  GEN_DBL_TEST (sin (-M_PI / 2.0));
  GEN_DBL_TEST (sin (M_PI / 3.0));
  GEN_DBL_TEST (sin (-M_PI / 3.0));
  GEN_DBL_TEST (sin (M_PI / 4.0));
  GEN_DBL_TEST (sin (-M_PI / 4.0));
  GEN_DBL_TEST (sin (M_PI / 6.0));
  GEN_DBL_TEST (sin (-M_PI / 6.0));
  GEN_DBL_TEST (sin (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sin (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sin (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sin (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sin (6.9e-18));
  GEN_DBL_TEST (sin (-6.9e-18));
  GEN_DBL_TEST (sin (7.0e-18));
  GEN_DBL_TEST (sin (-7.0e-18));
  GEN_DBL_TEST (sin (7.4e-9));
  GEN_DBL_TEST (sin (-7.4e-9));
  GEN_DBL_TEST (sin (7.5e-9));
  GEN_DBL_TEST (sin (-7.5e-9));
  GEN_DBL_TEST (sin (0.2));
  GEN_DBL_TEST (sin (-0.2));
  GEN_DBL_TEST (sin (0.4));
  GEN_DBL_TEST (sin (-0.4));
  GEN_DBL_TEST (sin (0.7));
  GEN_DBL_TEST (sin (-0.7));
  GEN_DBL_TEST (sin (0.8));
  GEN_DBL_TEST (sin (-0.8));
  GEN_DBL_TEST (sin (3.0));
  GEN_DBL_TEST (sin (-3.0));
  GEN_DBL_TEST (sin (4.0));
  GEN_DBL_TEST (sin (-4.0));
  GEN_DBL_TEST (sin (6.0));
  GEN_DBL_TEST (sin (-6.0));
  GEN_DBL_TEST (sin (7.0));
  GEN_DBL_TEST (sin (-7.0));

  /* cos tests */
  GEN_DBL_TEST (cos (0.0));
  GEN_DBL_TEST (cos (-0.0));
  GEN_DBL_TEST (cos (1.0));
  GEN_DBL_TEST (cos (-1.0));
  GEN_DBL_TEST (cos (INFINITY));
  GEN_DBL_TEST (cos (-INFINITY));
  GEN_DBL_TEST (cos (NAN));
  GEN_DBL_TEST (cos (M_PI));
  GEN_DBL_TEST (cos (-M_PI));
  GEN_DBL_TEST (cos (2.0 * M_PI));
  GEN_DBL_TEST (cos (-2.0 * M_PI));
  GEN_DBL_TEST (cos (M_PI / 2.0));
  GEN_DBL_TEST (cos (-M_PI / 2.0));
  GEN_DBL_TEST (cos (M_PI / 3.0));
  GEN_DBL_TEST (cos (-M_PI / 3.0));
  GEN_DBL_TEST (cos (M_PI / 4.0));
  GEN_DBL_TEST (cos (-M_PI / 4.0));
  GEN_DBL_TEST (cos (M_PI / 6.0));
  GEN_DBL_TEST (cos (-M_PI / 6.0));
  GEN_DBL_TEST (cos (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cos (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cos (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cos (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cos (6.9e-18));
  GEN_DBL_TEST (cos (-6.9e-18));
  GEN_DBL_TEST (cos (7.0e-18));
  GEN_DBL_TEST (cos (-7.0e-18));
  GEN_DBL_TEST (cos (7.4e-9));
  GEN_DBL_TEST (cos (-7.4e-9));
  GEN_DBL_TEST (cos (7.5e-9));
  GEN_DBL_TEST (cos (-7.5e-9));
  GEN_DBL_TEST (cos (0.2));
  GEN_DBL_TEST (cos (-0.2));
  GEN_DBL_TEST (cos (0.4));
  GEN_DBL_TEST (cos (-0.4));
  GEN_DBL_TEST (cos (0.7));
  GEN_DBL_TEST (cos (-0.7));
  GEN_DBL_TEST (cos (0.8));
  GEN_DBL_TEST (cos (-0.8));
  GEN_DBL_TEST (cos (3.0));
  GEN_DBL_TEST (cos (-3.0));
  GEN_DBL_TEST (cos (4.0));
  GEN_DBL_TEST (cos (-4.0));
  GEN_DBL_TEST (cos (6.0));
  GEN_DBL_TEST (cos (-6.0));
  GEN_DBL_TEST (cos (7.0));
  GEN_DBL_TEST (cos (-7.0));

  /* tan tests */
  GEN_DBL_TEST (tan (0.0));
  GEN_DBL_TEST (tan (-0.0));
  GEN_DBL_TEST (tan (1.0));
  GEN_DBL_TEST (tan (-1.0));
  GEN_DBL_TEST (tan (INFINITY));
  GEN_DBL_TEST (tan (-INFINITY));
  GEN_DBL_TEST (tan (NAN));
  GEN_DBL_TEST (tan (M_PI));
  GEN_DBL_TEST (tan (-M_PI));
  GEN_DBL_TEST (tan (2.0 * M_PI));
  GEN_DBL_TEST (tan (-2.0 * M_PI));
  GEN_DBL_TEST (tan (M_PI / 2.0));
  GEN_DBL_TEST (tan (-M_PI / 2.0));
  GEN_DBL_TEST (tan (M_PI / 3.0));
  GEN_DBL_TEST (tan (-M_PI / 3.0));
  GEN_DBL_TEST (tan (M_PI / 4.0));
  GEN_DBL_TEST (tan (-M_PI / 4.0));
  GEN_DBL_TEST (tan (M_PI / 6.0));
  GEN_DBL_TEST (tan (-M_PI / 6.0));
  GEN_DBL_TEST (tan (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tan (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tan (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tan (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tan (3.7e-9));
  GEN_DBL_TEST (tan (-3.7e-9));
  GEN_DBL_TEST (tan (3.8e-9));
  GEN_DBL_TEST (tan (-3.8e-9));
  GEN_DBL_TEST (tan (0.6));
  GEN_DBL_TEST (tan (-0.6));
  GEN_DBL_TEST (tan (0.7));
  GEN_DBL_TEST (tan (-0.7));
  GEN_DBL_TEST (tan (3.0));
  GEN_DBL_TEST (tan (-3.0));
  GEN_DBL_TEST (tan (4.0));
  GEN_DBL_TEST (tan (-4.0));
  GEN_DBL_TEST (tan (6.0));
  GEN_DBL_TEST (tan (-6.0));
  GEN_DBL_TEST (tan (7.0));
  GEN_DBL_TEST (tan (-7.0));
} /* main */
