/* SPDX-FileCopyrightText: 2016 ARM Limited */
/* SPDX-License-Identifier: Apache-2.0 */
#include "mbed.h"
#include "rtos.h"

#include "jerry-core/jerry-api.h"

#include "jerryscript-mbed-event-loop/EventLoop.h"

#include "jerryscript-mbed-util/js_source.h"
#include "jerryscript-mbed-library-registry/registry.h"

#include "jerryscript-mbed-launcher/launcher.h"
#include "jerryscript-mbed-launcher/setup.h"

#include "jerry_targetjs.h"

DECLARE_JS_CODES;

/**
 * load_javascript
 *
 * Parse and run javascript files specified in jerry_targetjs.h
 */
static int load_javascript() {
    for (int src = 0; js_codes[src].source; src++) {
        LOG_PRINT("running js file %s\r\n", js_codes[src].name);

        const jerry_char_t* code = reinterpret_cast<const jerry_char_t*>(js_codes[src].source);
        const size_t length = js_codes[src].length;

        jerry_value_t parsed_code = jerry_parse(code, length, false);

        if (jerry_value_has_error_flag(parsed_code)) {
            LOG_PRINT_ALWAYS("jerry_parse failed [%s]\r\n", js_codes[src].name);
            jsmbed_js_exit();
            return -1;
        }

        jerry_value_t returned_value = jerry_run(parsed_code);

        if (jerry_value_has_error_flag(returned_value)) {
            LOG_PRINT_ALWAYS("jerry_run failed [%s]\r\n", js_codes[src].name);
            jsmbed_js_exit();
            return -1;
        }

        jerry_release_value(parsed_code);
        jerry_release_value(returned_value);
    }

    return 0;
}

int jsmbed_js_init() {
    jerry_init_flag_t flags = JERRY_INIT_EMPTY;
    jerry_init(flags);

    jsmbed_js_load_magic_strings();
    mbed::js::LibraryRegistry::getInstance().register_all();

    return 0;
}

void jsmbed_js_exit() {
    jerry_cleanup();
}

void jsmbed_js_launch() {
    jsmbed_js_init();

    puts("   JerryScript in mbed\r\n");
    puts("   build date:  " __DATE__ " \r\n");

    if (load_javascript() == 0) {
        mbed::js::event_loop();
    }
}
