/* SPDX-FileCopyrightText: 2024 Google LLC */
/* SPDX-License-Identifier: Apache-2.0 */

#include "applib/pbl_std/pbl_std.h"
#include "applib/pbl_std/locale.h"
#include "services/common/i18n/i18n.h"
#include "util/size.h"

#include "clar.h"

// Stubs
//////////////////////////////////////////////////////////
#include "stubs_heap.h"
#include "stubs_hexdump.h"
#include "stubs_logging.h"
#include "stubs_mutex.h"
#include "stubs_passert.h"
#include "stubs_pbl_malloc.h"
#include "stubs_pebble_tasks.h"
#include "stubs_print.h"
#include "stubs_prompt.h"
#include "stubs_serial.h"
#include "stubs_sleep.h"
#include "stubs_syscall_internal.h"
#include "stubs_system_reset.h"
#include "stubs_task_watchdog.h"
#include "stubs_app_state.h"
#include "stubs_worker_state.h"

// Overrides
//////////////////////////////////////////////////////////
void sys_get_time_ms(time_t *t, uint16_t *out_ms) {}

time_t sys_time_utc_to_local(time_t t) {
  return t;
}

const char *get_timezone_abbr(void) {
  static const char s_timezone_abbr[] = "A";
  return s_timezone_abbr;
}

void sys_copy_timezone_abbr(char* timezone_abbr, time_t time) {
  const char* sys_tz = get_timezone_abbr();
  strncpy(timezone_abbr, sys_tz, TZ_LEN);
}

struct tm *sys_gmtime_r(const time_t *timep, struct tm *result) {
  return gmtime_r(timep, result);
}

struct tm *sys_localtime_r(const time_t *timep, struct tm *result) {
  return localtime_r(timep, result);
}

// i18n dummies
////////////////////////////////////
static bool s_i18n_translate = false;
static bool s_i18n_locale_other = false;

char *i18n_get_locale(void) {
  if (s_i18n_locale_other) {
    return "es_ES";
  } else {
    return "en_US";
  }
}
char *app_get_system_locale(void) {
  return "es_ES";
}

const char *i18n_get(const char *msgid, const void *owner) {
  if (s_i18n_translate) {
    return "Hola";
  } else {
    return msgid;
  }
}

void i18n_get_with_buffer(const char *string, char *buffer, size_t length) {
  strncpy(buffer, i18n_get(string, NULL), length);
}

size_t i18n_get_length(const char *string) {
  return strlen(i18n_get(string, NULL));
}

void i18n_free(const char *original, const void *owner) {
}

void i18n_free_all(const void *owner) {
}

void sys_i18n_get_with_buffer(const char *string, char *buffer, size_t length) {
  i18n_get_with_buffer(string, buffer, length);
}

size_t sys_i18n_get_length(const char *string) {
  return i18n_get_length(string);
}


// Tests
////////////////////////////////////

// (SU) = Single UNIX Specification extension
// (GNU)= glibc extension
// (TZ) = Olson's timezone package extension?
// (C99)= C99 extension
// We support them all~

#define EXTENSION_SU 1
#define EXTENSION_GNU 1
#define EXTENSION_TZ 1
#define EXTENSION_C99 1

// Our current strftime lacks a few things and has some bugs.
// It does not support %s, nor %+
// %z totally breaks down if the GMT offset is in the range -1s ~ -59m
//   This is because for minute it does an abs(), but doesn't change the sign of the output.
// We support the 0 and + (sorta) flag and minimum field width, but no other flag modifiers
// The code supports %O* and %E*, but we don't actually implement the data for them.
#define OUR_STRFTIME_BUGS 1

// newlib strftime was using really annoying failure string output.
// C99 spec says the result is undefined so it's fine if these tests don't pass.
#define SUPER_STRICT_FAILURE 0

void test_strftime__simple(void) {
  char tmbuf[512];

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");

#if EXTENSION_SU
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%C %h", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "20 Jan");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%D %e", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "01/02/15  2");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%n %r", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "\n 01:00:00 PM");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%R %t %T", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "13:00 \t 13:00:00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%u %V", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "5 01");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0000");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%d", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "02");

#if EXTENSION_C99
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%F", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "2015-01-02");
#endif

#if EXTENSION_TZ
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%G %g", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "2015 15");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%k %l", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "13  1");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%H %I", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "13 01");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%j %m %M %p", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "002 01 00 PM");

#if EXTENSION_GNU
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%P", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "pm");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%S %U", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "00 00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%w %W", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "5 00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%x %X", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "01/02/15 13:00:00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%y %Y", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "15 2015");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%Z %%", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "UTC %");
}

void test_strftime__return_value(void) {
  char tmbuf[512];

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  cl_assert_equal_i(strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00), 22);
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
}

void test_strftime__abusive(void) {
  char tmbuf[512];

  struct tm gobbledygook = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0+MONTHS_PER_YEAR, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5+DAYS_PER_WEEK, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%c", &gobbledygook);
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");
}

void test_strftime__small_buffer(void) {
  char tmbuf[8];

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

#if SUPER_STRICT_FAILURE
  memset(tmbuf, 0xFF, sizeof(tmbuf));
  cl_assert_equal_i(strftime(tmbuf, 4, "%a %A %b %B", &jan_2_2015__13_00_00), 0);
  // This is how our current strftime works, so we need to obey...
  cl_assert_equal_m(tmbuf, "Fri\xFF\xFF", 5);

  memset(tmbuf, 0xFF, sizeof(tmbuf));
  cl_assert_equal_i(strftime(tmbuf, 4, "%a ", &jan_2_2015__13_00_00), 0);
  // This is how our current strftime works, so we need to obey...
  cl_assert_equal_m(tmbuf, "Fri\xFF\xFF", 5);
#endif

  memset(tmbuf, 0xFF, sizeof(tmbuf));
  cl_assert_equal_i(strftime(tmbuf, 4, "%a", &jan_2_2015__13_00_00), 3);
  cl_assert_equal_m(tmbuf, "Fri\x00\xFF", 5);
}

#if EXTENSION_SU
typedef struct {
  time_t stamp;
  const char *str_V;
  const char *str_g;
} ISO8601TestData;

/* Generated with this ruby script:

require 'date'
d = DateTime.new(2000,1,1)
(0..(365*8)).each do |i|
  print "\n  " if (i % 7) == 0
  printf("{%d,\"%s\",\"%s\"},", d.to_time.to_i,
          d.strftime("%V"), d.strftime("%g"))
  d += 1
end
print "\n"
*/
static const ISO8601TestData s_iso8601_data[] = {
  {946684800,"52","99"},{946771200,"52","99"},{946857600,"01","00"},{946944000,"01","00"},{947030400,"01","00"},{947116800,"01","00"},{947203200,"01","00"},
  {947289600,"01","00"},{947376000,"01","00"},{947462400,"02","00"},{947548800,"02","00"},{947635200,"02","00"},{947721600,"02","00"},{947808000,"02","00"},
  {947894400,"02","00"},{947980800,"02","00"},{948067200,"03","00"},{948153600,"03","00"},{948240000,"03","00"},{948326400,"03","00"},{948412800,"03","00"},
  {948499200,"03","00"},{948585600,"03","00"},{948672000,"04","00"},{948758400,"04","00"},{948844800,"04","00"},{948931200,"04","00"},{949017600,"04","00"},
  {949104000,"04","00"},{949190400,"04","00"},{949276800,"05","00"},{949363200,"05","00"},{949449600,"05","00"},{949536000,"05","00"},{949622400,"05","00"},
  {949708800,"05","00"},{949795200,"05","00"},{949881600,"06","00"},{949968000,"06","00"},{950054400,"06","00"},{950140800,"06","00"},{950227200,"06","00"},
  {950313600,"06","00"},{950400000,"06","00"},{950486400,"07","00"},{950572800,"07","00"},{950659200,"07","00"},{950745600,"07","00"},{950832000,"07","00"},
  {950918400,"07","00"},{951004800,"07","00"},{951091200,"08","00"},{951177600,"08","00"},{951264000,"08","00"},{951350400,"08","00"},{951436800,"08","00"},
  {951523200,"08","00"},{951609600,"08","00"},{951696000,"09","00"},{951782400,"09","00"},{951868800,"09","00"},{951955200,"09","00"},{952041600,"09","00"},
  {952128000,"09","00"},{952214400,"09","00"},{952300800,"10","00"},{952387200,"10","00"},{952473600,"10","00"},{952560000,"10","00"},{952646400,"10","00"},
  {952732800,"10","00"},{952819200,"10","00"},{952905600,"11","00"},{952992000,"11","00"},{953078400,"11","00"},{953164800,"11","00"},{953251200,"11","00"},
  {953337600,"11","00"},{953424000,"11","00"},{953510400,"12","00"},{953596800,"12","00"},{953683200,"12","00"},{953769600,"12","00"},{953856000,"12","00"},
  {953942400,"12","00"},{954028800,"12","00"},{954115200,"13","00"},{954201600,"13","00"},{954288000,"13","00"},{954374400,"13","00"},{954460800,"13","00"},
  {954547200,"13","00"},{954633600,"13","00"},{954720000,"14","00"},{954806400,"14","00"},{954892800,"14","00"},{954979200,"14","00"},{955065600,"14","00"},
  {955152000,"14","00"},{955238400,"14","00"},{955324800,"15","00"},{955411200,"15","00"},{955497600,"15","00"},{955584000,"15","00"},{955670400,"15","00"},
  {955756800,"15","00"},{955843200,"15","00"},{955929600,"16","00"},{956016000,"16","00"},{956102400,"16","00"},{956188800,"16","00"},{956275200,"16","00"},
  {956361600,"16","00"},{956448000,"16","00"},{956534400,"17","00"},{956620800,"17","00"},{956707200,"17","00"},{956793600,"17","00"},{956880000,"17","00"},
  {956966400,"17","00"},{957052800,"17","00"},{957139200,"18","00"},{957225600,"18","00"},{957312000,"18","00"},{957398400,"18","00"},{957484800,"18","00"},
  {957571200,"18","00"},{957657600,"18","00"},{957744000,"19","00"},{957830400,"19","00"},{957916800,"19","00"},{958003200,"19","00"},{958089600,"19","00"},
  {958176000,"19","00"},{958262400,"19","00"},{958348800,"20","00"},{958435200,"20","00"},{958521600,"20","00"},{958608000,"20","00"},{958694400,"20","00"},
  {958780800,"20","00"},{958867200,"20","00"},{958953600,"21","00"},{959040000,"21","00"},{959126400,"21","00"},{959212800,"21","00"},{959299200,"21","00"},
  {959385600,"21","00"},{959472000,"21","00"},{959558400,"22","00"},{959644800,"22","00"},{959731200,"22","00"},{959817600,"22","00"},{959904000,"22","00"},
  {959990400,"22","00"},{960076800,"22","00"},{960163200,"23","00"},{960249600,"23","00"},{960336000,"23","00"},{960422400,"23","00"},{960508800,"23","00"},
  {960595200,"23","00"},{960681600,"23","00"},{960768000,"24","00"},{960854400,"24","00"},{960940800,"24","00"},{961027200,"24","00"},{961113600,"24","00"},
  {961200000,"24","00"},{961286400,"24","00"},{961372800,"25","00"},{961459200,"25","00"},{961545600,"25","00"},{961632000,"25","00"},{961718400,"25","00"},
  {961804800,"25","00"},{961891200,"25","00"},{961977600,"26","00"},{962064000,"26","00"},{962150400,"26","00"},{962236800,"26","00"},{962323200,"26","00"},
  {962409600,"26","00"},{962496000,"26","00"},{962582400,"27","00"},{962668800,"27","00"},{962755200,"27","00"},{962841600,"27","00"},{962928000,"27","00"},
  {963014400,"27","00"},{963100800,"27","00"},{963187200,"28","00"},{963273600,"28","00"},{963360000,"28","00"},{963446400,"28","00"},{963532800,"28","00"},
  {963619200,"28","00"},{963705600,"28","00"},{963792000,"29","00"},{963878400,"29","00"},{963964800,"29","00"},{964051200,"29","00"},{964137600,"29","00"},
  {964224000,"29","00"},{964310400,"29","00"},{964396800,"30","00"},{964483200,"30","00"},{964569600,"30","00"},{964656000,"30","00"},{964742400,"30","00"},
  {964828800,"30","00"},{964915200,"30","00"},{965001600,"31","00"},{965088000,"31","00"},{965174400,"31","00"},{965260800,"31","00"},{965347200,"31","00"},
  {965433600,"31","00"},{965520000,"31","00"},{965606400,"32","00"},{965692800,"32","00"},{965779200,"32","00"},{965865600,"32","00"},{965952000,"32","00"},
  {966038400,"32","00"},{966124800,"32","00"},{966211200,"33","00"},{966297600,"33","00"},{966384000,"33","00"},{966470400,"33","00"},{966556800,"33","00"},
  {966643200,"33","00"},{966729600,"33","00"},{966816000,"34","00"},{966902400,"34","00"},{966988800,"34","00"},{967075200,"34","00"},{967161600,"34","00"},
  {967248000,"34","00"},{967334400,"34","00"},{967420800,"35","00"},{967507200,"35","00"},{967593600,"35","00"},{967680000,"35","00"},{967766400,"35","00"},
  {967852800,"35","00"},{967939200,"35","00"},{968025600,"36","00"},{968112000,"36","00"},{968198400,"36","00"},{968284800,"36","00"},{968371200,"36","00"},
  {968457600,"36","00"},{968544000,"36","00"},{968630400,"37","00"},{968716800,"37","00"},{968803200,"37","00"},{968889600,"37","00"},{968976000,"37","00"},
  {969062400,"37","00"},{969148800,"37","00"},{969235200,"38","00"},{969321600,"38","00"},{969408000,"38","00"},{969494400,"38","00"},{969580800,"38","00"},
  {969667200,"38","00"},{969753600,"38","00"},{969840000,"39","00"},{969926400,"39","00"},{970012800,"39","00"},{970099200,"39","00"},{970185600,"39","00"},
  {970272000,"39","00"},{970358400,"39","00"},{970444800,"40","00"},{970531200,"40","00"},{970617600,"40","00"},{970704000,"40","00"},{970790400,"40","00"},
  {970876800,"40","00"},{970963200,"40","00"},{971049600,"41","00"},{971136000,"41","00"},{971222400,"41","00"},{971308800,"41","00"},{971395200,"41","00"},
  {971481600,"41","00"},{971568000,"41","00"},{971654400,"42","00"},{971740800,"42","00"},{971827200,"42","00"},{971913600,"42","00"},{972000000,"42","00"},
  {972086400,"42","00"},{972172800,"42","00"},{972259200,"43","00"},{972345600,"43","00"},{972432000,"43","00"},{972518400,"43","00"},{972604800,"43","00"},
  {972691200,"43","00"},{972777600,"43","00"},{972864000,"44","00"},{972950400,"44","00"},{973036800,"44","00"},{973123200,"44","00"},{973209600,"44","00"},
  {973296000,"44","00"},{973382400,"44","00"},{973468800,"45","00"},{973555200,"45","00"},{973641600,"45","00"},{973728000,"45","00"},{973814400,"45","00"},
  {973900800,"45","00"},{973987200,"45","00"},{974073600,"46","00"},{974160000,"46","00"},{974246400,"46","00"},{974332800,"46","00"},{974419200,"46","00"},
  {974505600,"46","00"},{974592000,"46","00"},{974678400,"47","00"},{974764800,"47","00"},{974851200,"47","00"},{974937600,"47","00"},{975024000,"47","00"},
  {975110400,"47","00"},{975196800,"47","00"},{975283200,"48","00"},{975369600,"48","00"},{975456000,"48","00"},{975542400,"48","00"},{975628800,"48","00"},
  {975715200,"48","00"},{975801600,"48","00"},{975888000,"49","00"},{975974400,"49","00"},{976060800,"49","00"},{976147200,"49","00"},{976233600,"49","00"},
  {976320000,"49","00"},{976406400,"49","00"},{976492800,"50","00"},{976579200,"50","00"},{976665600,"50","00"},{976752000,"50","00"},{976838400,"50","00"},
  {976924800,"50","00"},{977011200,"50","00"},{977097600,"51","00"},{977184000,"51","00"},{977270400,"51","00"},{977356800,"51","00"},{977443200,"51","00"},
  {977529600,"51","00"},{977616000,"51","00"},{977702400,"52","00"},{977788800,"52","00"},{977875200,"52","00"},{977961600,"52","00"},{978048000,"52","00"},
  {978134400,"52","00"},{978220800,"52","00"},{978307200,"01","01"},{978393600,"01","01"},{978480000,"01","01"},{978566400,"01","01"},{978652800,"01","01"},
  {978739200,"01","01"},{978825600,"01","01"},{978912000,"02","01"},{978998400,"02","01"},{979084800,"02","01"},{979171200,"02","01"},{979257600,"02","01"},
  {979344000,"02","01"},{979430400,"02","01"},{979516800,"03","01"},{979603200,"03","01"},{979689600,"03","01"},{979776000,"03","01"},{979862400,"03","01"},
  {979948800,"03","01"},{980035200,"03","01"},{980121600,"04","01"},{980208000,"04","01"},{980294400,"04","01"},{980380800,"04","01"},{980467200,"04","01"},
  {980553600,"04","01"},{980640000,"04","01"},{980726400,"05","01"},{980812800,"05","01"},{980899200,"05","01"},{980985600,"05","01"},{981072000,"05","01"},
  {981158400,"05","01"},{981244800,"05","01"},{981331200,"06","01"},{981417600,"06","01"},{981504000,"06","01"},{981590400,"06","01"},{981676800,"06","01"},
  {981763200,"06","01"},{981849600,"06","01"},{981936000,"07","01"},{982022400,"07","01"},{982108800,"07","01"},{982195200,"07","01"},{982281600,"07","01"},
  {982368000,"07","01"},{982454400,"07","01"},{982540800,"08","01"},{982627200,"08","01"},{982713600,"08","01"},{982800000,"08","01"},{982886400,"08","01"},
  {982972800,"08","01"},{983059200,"08","01"},{983145600,"09","01"},{983232000,"09","01"},{983318400,"09","01"},{983404800,"09","01"},{983491200,"09","01"},
  {983577600,"09","01"},{983664000,"09","01"},{983750400,"10","01"},{983836800,"10","01"},{983923200,"10","01"},{984009600,"10","01"},{984096000,"10","01"},
  {984182400,"10","01"},{984268800,"10","01"},{984355200,"11","01"},{984441600,"11","01"},{984528000,"11","01"},{984614400,"11","01"},{984700800,"11","01"},
  {984787200,"11","01"},{984873600,"11","01"},{984960000,"12","01"},{985046400,"12","01"},{985132800,"12","01"},{985219200,"12","01"},{985305600,"12","01"},
  {985392000,"12","01"},{985478400,"12","01"},{985564800,"13","01"},{985651200,"13","01"},{985737600,"13","01"},{985824000,"13","01"},{985910400,"13","01"},
  {985996800,"13","01"},{986083200,"13","01"},{986169600,"14","01"},{986256000,"14","01"},{986342400,"14","01"},{986428800,"14","01"},{986515200,"14","01"},
  {986601600,"14","01"},{986688000,"14","01"},{986774400,"15","01"},{986860800,"15","01"},{986947200,"15","01"},{987033600,"15","01"},{987120000,"15","01"},
  {987206400,"15","01"},{987292800,"15","01"},{987379200,"16","01"},{987465600,"16","01"},{987552000,"16","01"},{987638400,"16","01"},{987724800,"16","01"},
  {987811200,"16","01"},{987897600,"16","01"},{987984000,"17","01"},{988070400,"17","01"},{988156800,"17","01"},{988243200,"17","01"},{988329600,"17","01"},
  {988416000,"17","01"},{988502400,"17","01"},{988588800,"18","01"},{988675200,"18","01"},{988761600,"18","01"},{988848000,"18","01"},{988934400,"18","01"},
  {989020800,"18","01"},{989107200,"18","01"},{989193600,"19","01"},{989280000,"19","01"},{989366400,"19","01"},{989452800,"19","01"},{989539200,"19","01"},
  {989625600,"19","01"},{989712000,"19","01"},{989798400,"20","01"},{989884800,"20","01"},{989971200,"20","01"},{990057600,"20","01"},{990144000,"20","01"},
  {990230400,"20","01"},{990316800,"20","01"},{990403200,"21","01"},{990489600,"21","01"},{990576000,"21","01"},{990662400,"21","01"},{990748800,"21","01"},
  {990835200,"21","01"},{990921600,"21","01"},{991008000,"22","01"},{991094400,"22","01"},{991180800,"22","01"},{991267200,"22","01"},{991353600,"22","01"},
  {991440000,"22","01"},{991526400,"22","01"},{991612800,"23","01"},{991699200,"23","01"},{991785600,"23","01"},{991872000,"23","01"},{991958400,"23","01"},
  {992044800,"23","01"},{992131200,"23","01"},{992217600,"24","01"},{992304000,"24","01"},{992390400,"24","01"},{992476800,"24","01"},{992563200,"24","01"},
  {992649600,"24","01"},{992736000,"24","01"},{992822400,"25","01"},{992908800,"25","01"},{992995200,"25","01"},{993081600,"25","01"},{993168000,"25","01"},
  {993254400,"25","01"},{993340800,"25","01"},{993427200,"26","01"},{993513600,"26","01"},{993600000,"26","01"},{993686400,"26","01"},{993772800,"26","01"},
  {993859200,"26","01"},{993945600,"26","01"},{994032000,"27","01"},{994118400,"27","01"},{994204800,"27","01"},{994291200,"27","01"},{994377600,"27","01"},
  {994464000,"27","01"},{994550400,"27","01"},{994636800,"28","01"},{994723200,"28","01"},{994809600,"28","01"},{994896000,"28","01"},{994982400,"28","01"},
  {995068800,"28","01"},{995155200,"28","01"},{995241600,"29","01"},{995328000,"29","01"},{995414400,"29","01"},{995500800,"29","01"},{995587200,"29","01"},
  {995673600,"29","01"},{995760000,"29","01"},{995846400,"30","01"},{995932800,"30","01"},{996019200,"30","01"},{996105600,"30","01"},{996192000,"30","01"},
  {996278400,"30","01"},{996364800,"30","01"},{996451200,"31","01"},{996537600,"31","01"},{996624000,"31","01"},{996710400,"31","01"},{996796800,"31","01"},
  {996883200,"31","01"},{996969600,"31","01"},{997056000,"32","01"},{997142400,"32","01"},{997228800,"32","01"},{997315200,"32","01"},{997401600,"32","01"},
  {997488000,"32","01"},{997574400,"32","01"},{997660800,"33","01"},{997747200,"33","01"},{997833600,"33","01"},{997920000,"33","01"},{998006400,"33","01"},
  {998092800,"33","01"},{998179200,"33","01"},{998265600,"34","01"},{998352000,"34","01"},{998438400,"34","01"},{998524800,"34","01"},{998611200,"34","01"},
  {998697600,"34","01"},{998784000,"34","01"},{998870400,"35","01"},{998956800,"35","01"},{999043200,"35","01"},{999129600,"35","01"},{999216000,"35","01"},
  {999302400,"35","01"},{999388800,"35","01"},{999475200,"36","01"},{999561600,"36","01"},{999648000,"36","01"},{999734400,"36","01"},{999820800,"36","01"},
  {999907200,"36","01"},{999993600,"36","01"},{1000080000,"37","01"},{1000166400,"37","01"},{1000252800,"37","01"},{1000339200,"37","01"},{1000425600,"37","01"},
  {1000512000,"37","01"},{1000598400,"37","01"},{1000684800,"38","01"},{1000771200,"38","01"},{1000857600,"38","01"},{1000944000,"38","01"},{1001030400,"38","01"},
  {1001116800,"38","01"},{1001203200,"38","01"},{1001289600,"39","01"},{1001376000,"39","01"},{1001462400,"39","01"},{1001548800,"39","01"},{1001635200,"39","01"},
  {1001721600,"39","01"},{1001808000,"39","01"},{1001894400,"40","01"},{1001980800,"40","01"},{1002067200,"40","01"},{1002153600,"40","01"},{1002240000,"40","01"},
  {1002326400,"40","01"},{1002412800,"40","01"},{1002499200,"41","01"},{1002585600,"41","01"},{1002672000,"41","01"},{1002758400,"41","01"},{1002844800,"41","01"},
  {1002931200,"41","01"},{1003017600,"41","01"},{1003104000,"42","01"},{1003190400,"42","01"},{1003276800,"42","01"},{1003363200,"42","01"},{1003449600,"42","01"},
  {1003536000,"42","01"},{1003622400,"42","01"},{1003708800,"43","01"},{1003795200,"43","01"},{1003881600,"43","01"},{1003968000,"43","01"},{1004054400,"43","01"},
  {1004140800,"43","01"},{1004227200,"43","01"},{1004313600,"44","01"},{1004400000,"44","01"},{1004486400,"44","01"},{1004572800,"44","01"},{1004659200,"44","01"},
  {1004745600,"44","01"},{1004832000,"44","01"},{1004918400,"45","01"},{1005004800,"45","01"},{1005091200,"45","01"},{1005177600,"45","01"},{1005264000,"45","01"},
  {1005350400,"45","01"},{1005436800,"45","01"},{1005523200,"46","01"},{1005609600,"46","01"},{1005696000,"46","01"},{1005782400,"46","01"},{1005868800,"46","01"},
  {1005955200,"46","01"},{1006041600,"46","01"},{1006128000,"47","01"},{1006214400,"47","01"},{1006300800,"47","01"},{1006387200,"47","01"},{1006473600,"47","01"},
  {1006560000,"47","01"},{1006646400,"47","01"},{1006732800,"48","01"},{1006819200,"48","01"},{1006905600,"48","01"},{1006992000,"48","01"},{1007078400,"48","01"},
  {1007164800,"48","01"},{1007251200,"48","01"},{1007337600,"49","01"},{1007424000,"49","01"},{1007510400,"49","01"},{1007596800,"49","01"},{1007683200,"49","01"},
  {1007769600,"49","01"},{1007856000,"49","01"},{1007942400,"50","01"},{1008028800,"50","01"},{1008115200,"50","01"},{1008201600,"50","01"},{1008288000,"50","01"},
  {1008374400,"50","01"},{1008460800,"50","01"},{1008547200,"51","01"},{1008633600,"51","01"},{1008720000,"51","01"},{1008806400,"51","01"},{1008892800,"51","01"},
  {1008979200,"51","01"},{1009065600,"51","01"},{1009152000,"52","01"},{1009238400,"52","01"},{1009324800,"52","01"},{1009411200,"52","01"},{1009497600,"52","01"},
  {1009584000,"52","01"},{1009670400,"52","01"},{1009756800,"01","02"},{1009843200,"01","02"},{1009929600,"01","02"},{1010016000,"01","02"},{1010102400,"01","02"},
  {1010188800,"01","02"},{1010275200,"01","02"},{1010361600,"02","02"},{1010448000,"02","02"},{1010534400,"02","02"},{1010620800,"02","02"},{1010707200,"02","02"},
  {1010793600,"02","02"},{1010880000,"02","02"},{1010966400,"03","02"},{1011052800,"03","02"},{1011139200,"03","02"},{1011225600,"03","02"},{1011312000,"03","02"},
  {1011398400,"03","02"},{1011484800,"03","02"},{1011571200,"04","02"},{1011657600,"04","02"},{1011744000,"04","02"},{1011830400,"04","02"},{1011916800,"04","02"},
  {1012003200,"04","02"},{1012089600,"04","02"},{1012176000,"05","02"},{1012262400,"05","02"},{1012348800,"05","02"},{1012435200,"05","02"},{1012521600,"05","02"},
  {1012608000,"05","02"},{1012694400,"05","02"},{1012780800,"06","02"},{1012867200,"06","02"},{1012953600,"06","02"},{1013040000,"06","02"},{1013126400,"06","02"},
  {1013212800,"06","02"},{1013299200,"06","02"},{1013385600,"07","02"},{1013472000,"07","02"},{1013558400,"07","02"},{1013644800,"07","02"},{1013731200,"07","02"},
  {1013817600,"07","02"},{1013904000,"07","02"},{1013990400,"08","02"},{1014076800,"08","02"},{1014163200,"08","02"},{1014249600,"08","02"},{1014336000,"08","02"},
  {1014422400,"08","02"},{1014508800,"08","02"},{1014595200,"09","02"},{1014681600,"09","02"},{1014768000,"09","02"},{1014854400,"09","02"},{1014940800,"09","02"},
  {1015027200,"09","02"},{1015113600,"09","02"},{1015200000,"10","02"},{1015286400,"10","02"},{1015372800,"10","02"},{1015459200,"10","02"},{1015545600,"10","02"},
  {1015632000,"10","02"},{1015718400,"10","02"},{1015804800,"11","02"},{1015891200,"11","02"},{1015977600,"11","02"},{1016064000,"11","02"},{1016150400,"11","02"},
  {1016236800,"11","02"},{1016323200,"11","02"},{1016409600,"12","02"},{1016496000,"12","02"},{1016582400,"12","02"},{1016668800,"12","02"},{1016755200,"12","02"},
  {1016841600,"12","02"},{1016928000,"12","02"},{1017014400,"13","02"},{1017100800,"13","02"},{1017187200,"13","02"},{1017273600,"13","02"},{1017360000,"13","02"},
  {1017446400,"13","02"},{1017532800,"13","02"},{1017619200,"14","02"},{1017705600,"14","02"},{1017792000,"14","02"},{1017878400,"14","02"},{1017964800,"14","02"},
  {1018051200,"14","02"},{1018137600,"14","02"},{1018224000,"15","02"},{1018310400,"15","02"},{1018396800,"15","02"},{1018483200,"15","02"},{1018569600,"15","02"},
  {1018656000,"15","02"},{1018742400,"15","02"},{1018828800,"16","02"},{1018915200,"16","02"},{1019001600,"16","02"},{1019088000,"16","02"},{1019174400,"16","02"},
  {1019260800,"16","02"},{1019347200,"16","02"},{1019433600,"17","02"},{1019520000,"17","02"},{1019606400,"17","02"},{1019692800,"17","02"},{1019779200,"17","02"},
  {1019865600,"17","02"},{1019952000,"17","02"},{1020038400,"18","02"},{1020124800,"18","02"},{1020211200,"18","02"},{1020297600,"18","02"},{1020384000,"18","02"},
  {1020470400,"18","02"},{1020556800,"18","02"},{1020643200,"19","02"},{1020729600,"19","02"},{1020816000,"19","02"},{1020902400,"19","02"},{1020988800,"19","02"},
  {1021075200,"19","02"},{1021161600,"19","02"},{1021248000,"20","02"},{1021334400,"20","02"},{1021420800,"20","02"},{1021507200,"20","02"},{1021593600,"20","02"},
  {1021680000,"20","02"},{1021766400,"20","02"},{1021852800,"21","02"},{1021939200,"21","02"},{1022025600,"21","02"},{1022112000,"21","02"},{1022198400,"21","02"},
  {1022284800,"21","02"},{1022371200,"21","02"},{1022457600,"22","02"},{1022544000,"22","02"},{1022630400,"22","02"},{1022716800,"22","02"},{1022803200,"22","02"},
  {1022889600,"22","02"},{1022976000,"22","02"},{1023062400,"23","02"},{1023148800,"23","02"},{1023235200,"23","02"},{1023321600,"23","02"},{1023408000,"23","02"},
  {1023494400,"23","02"},{1023580800,"23","02"},{1023667200,"24","02"},{1023753600,"24","02"},{1023840000,"24","02"},{1023926400,"24","02"},{1024012800,"24","02"},
  {1024099200,"24","02"},{1024185600,"24","02"},{1024272000,"25","02"},{1024358400,"25","02"},{1024444800,"25","02"},{1024531200,"25","02"},{1024617600,"25","02"},
  {1024704000,"25","02"},{1024790400,"25","02"},{1024876800,"26","02"},{1024963200,"26","02"},{1025049600,"26","02"},{1025136000,"26","02"},{1025222400,"26","02"},
  {1025308800,"26","02"},{1025395200,"26","02"},{1025481600,"27","02"},{1025568000,"27","02"},{1025654400,"27","02"},{1025740800,"27","02"},{1025827200,"27","02"},
  {1025913600,"27","02"},{1026000000,"27","02"},{1026086400,"28","02"},{1026172800,"28","02"},{1026259200,"28","02"},{1026345600,"28","02"},{1026432000,"28","02"},
  {1026518400,"28","02"},{1026604800,"28","02"},{1026691200,"29","02"},{1026777600,"29","02"},{1026864000,"29","02"},{1026950400,"29","02"},{1027036800,"29","02"},
  {1027123200,"29","02"},{1027209600,"29","02"},{1027296000,"30","02"},{1027382400,"30","02"},{1027468800,"30","02"},{1027555200,"30","02"},{1027641600,"30","02"},
  {1027728000,"30","02"},{1027814400,"30","02"},{1027900800,"31","02"},{1027987200,"31","02"},{1028073600,"31","02"},{1028160000,"31","02"},{1028246400,"31","02"},
  {1028332800,"31","02"},{1028419200,"31","02"},{1028505600,"32","02"},{1028592000,"32","02"},{1028678400,"32","02"},{1028764800,"32","02"},{1028851200,"32","02"},
  {1028937600,"32","02"},{1029024000,"32","02"},{1029110400,"33","02"},{1029196800,"33","02"},{1029283200,"33","02"},{1029369600,"33","02"},{1029456000,"33","02"},
  {1029542400,"33","02"},{1029628800,"33","02"},{1029715200,"34","02"},{1029801600,"34","02"},{1029888000,"34","02"},{1029974400,"34","02"},{1030060800,"34","02"},
  {1030147200,"34","02"},{1030233600,"34","02"},{1030320000,"35","02"},{1030406400,"35","02"},{1030492800,"35","02"},{1030579200,"35","02"},{1030665600,"35","02"},
  {1030752000,"35","02"},{1030838400,"35","02"},{1030924800,"36","02"},{1031011200,"36","02"},{1031097600,"36","02"},{1031184000,"36","02"},{1031270400,"36","02"},
  {1031356800,"36","02"},{1031443200,"36","02"},{1031529600,"37","02"},{1031616000,"37","02"},{1031702400,"37","02"},{1031788800,"37","02"},{1031875200,"37","02"},
  {1031961600,"37","02"},{1032048000,"37","02"},{1032134400,"38","02"},{1032220800,"38","02"},{1032307200,"38","02"},{1032393600,"38","02"},{1032480000,"38","02"},
  {1032566400,"38","02"},{1032652800,"38","02"},{1032739200,"39","02"},{1032825600,"39","02"},{1032912000,"39","02"},{1032998400,"39","02"},{1033084800,"39","02"},
  {1033171200,"39","02"},{1033257600,"39","02"},{1033344000,"40","02"},{1033430400,"40","02"},{1033516800,"40","02"},{1033603200,"40","02"},{1033689600,"40","02"},
  {1033776000,"40","02"},{1033862400,"40","02"},{1033948800,"41","02"},{1034035200,"41","02"},{1034121600,"41","02"},{1034208000,"41","02"},{1034294400,"41","02"},
  {1034380800,"41","02"},{1034467200,"41","02"},{1034553600,"42","02"},{1034640000,"42","02"},{1034726400,"42","02"},{1034812800,"42","02"},{1034899200,"42","02"},
  {1034985600,"42","02"},{1035072000,"42","02"},{1035158400,"43","02"},{1035244800,"43","02"},{1035331200,"43","02"},{1035417600,"43","02"},{1035504000,"43","02"},
  {1035590400,"43","02"},{1035676800,"43","02"},{1035763200,"44","02"},{1035849600,"44","02"},{1035936000,"44","02"},{1036022400,"44","02"},{1036108800,"44","02"},
  {1036195200,"44","02"},{1036281600,"44","02"},{1036368000,"45","02"},{1036454400,"45","02"},{1036540800,"45","02"},{1036627200,"45","02"},{1036713600,"45","02"},
  {1036800000,"45","02"},{1036886400,"45","02"},{1036972800,"46","02"},{1037059200,"46","02"},{1037145600,"46","02"},{1037232000,"46","02"},{1037318400,"46","02"},
  {1037404800,"46","02"},{1037491200,"46","02"},{1037577600,"47","02"},{1037664000,"47","02"},{1037750400,"47","02"},{1037836800,"47","02"},{1037923200,"47","02"},
  {1038009600,"47","02"},{1038096000,"47","02"},{1038182400,"48","02"},{1038268800,"48","02"},{1038355200,"48","02"},{1038441600,"48","02"},{1038528000,"48","02"},
  {1038614400,"48","02"},{1038700800,"48","02"},{1038787200,"49","02"},{1038873600,"49","02"},{1038960000,"49","02"},{1039046400,"49","02"},{1039132800,"49","02"},
  {1039219200,"49","02"},{1039305600,"49","02"},{1039392000,"50","02"},{1039478400,"50","02"},{1039564800,"50","02"},{1039651200,"50","02"},{1039737600,"50","02"},
  {1039824000,"50","02"},{1039910400,"50","02"},{1039996800,"51","02"},{1040083200,"51","02"},{1040169600,"51","02"},{1040256000,"51","02"},{1040342400,"51","02"},
  {1040428800,"51","02"},{1040515200,"51","02"},{1040601600,"52","02"},{1040688000,"52","02"},{1040774400,"52","02"},{1040860800,"52","02"},{1040947200,"52","02"},
  {1041033600,"52","02"},{1041120000,"52","02"},{1041206400,"01","03"},{1041292800,"01","03"},{1041379200,"01","03"},{1041465600,"01","03"},{1041552000,"01","03"},
  {1041638400,"01","03"},{1041724800,"01","03"},{1041811200,"02","03"},{1041897600,"02","03"},{1041984000,"02","03"},{1042070400,"02","03"},{1042156800,"02","03"},
  {1042243200,"02","03"},{1042329600,"02","03"},{1042416000,"03","03"},{1042502400,"03","03"},{1042588800,"03","03"},{1042675200,"03","03"},{1042761600,"03","03"},
  {1042848000,"03","03"},{1042934400,"03","03"},{1043020800,"04","03"},{1043107200,"04","03"},{1043193600,"04","03"},{1043280000,"04","03"},{1043366400,"04","03"},
  {1043452800,"04","03"},{1043539200,"04","03"},{1043625600,"05","03"},{1043712000,"05","03"},{1043798400,"05","03"},{1043884800,"05","03"},{1043971200,"05","03"},
  {1044057600,"05","03"},{1044144000,"05","03"},{1044230400,"06","03"},{1044316800,"06","03"},{1044403200,"06","03"},{1044489600,"06","03"},{1044576000,"06","03"},
  {1044662400,"06","03"},{1044748800,"06","03"},{1044835200,"07","03"},{1044921600,"07","03"},{1045008000,"07","03"},{1045094400,"07","03"},{1045180800,"07","03"},
  {1045267200,"07","03"},{1045353600,"07","03"},{1045440000,"08","03"},{1045526400,"08","03"},{1045612800,"08","03"},{1045699200,"08","03"},{1045785600,"08","03"},
  {1045872000,"08","03"},{1045958400,"08","03"},{1046044800,"09","03"},{1046131200,"09","03"},{1046217600,"09","03"},{1046304000,"09","03"},{1046390400,"09","03"},
  {1046476800,"09","03"},{1046563200,"09","03"},{1046649600,"10","03"},{1046736000,"10","03"},{1046822400,"10","03"},{1046908800,"10","03"},{1046995200,"10","03"},
  {1047081600,"10","03"},{1047168000,"10","03"},{1047254400,"11","03"},{1047340800,"11","03"},{1047427200,"11","03"},{1047513600,"11","03"},{1047600000,"11","03"},
  {1047686400,"11","03"},{1047772800,"11","03"},{1047859200,"12","03"},{1047945600,"12","03"},{1048032000,"12","03"},{1048118400,"12","03"},{1048204800,"12","03"},
  {1048291200,"12","03"},{1048377600,"12","03"},{1048464000,"13","03"},{1048550400,"13","03"},{1048636800,"13","03"},{1048723200,"13","03"},{1048809600,"13","03"},
  {1048896000,"13","03"},{1048982400,"13","03"},{1049068800,"14","03"},{1049155200,"14","03"},{1049241600,"14","03"},{1049328000,"14","03"},{1049414400,"14","03"},
  {1049500800,"14","03"},{1049587200,"14","03"},{1049673600,"15","03"},{1049760000,"15","03"},{1049846400,"15","03"},{1049932800,"15","03"},{1050019200,"15","03"},
  {1050105600,"15","03"},{1050192000,"15","03"},{1050278400,"16","03"},{1050364800,"16","03"},{1050451200,"16","03"},{1050537600,"16","03"},{1050624000,"16","03"},
  {1050710400,"16","03"},{1050796800,"16","03"},{1050883200,"17","03"},{1050969600,"17","03"},{1051056000,"17","03"},{1051142400,"17","03"},{1051228800,"17","03"},
  {1051315200,"17","03"},{1051401600,"17","03"},{1051488000,"18","03"},{1051574400,"18","03"},{1051660800,"18","03"},{1051747200,"18","03"},{1051833600,"18","03"},
  {1051920000,"18","03"},{1052006400,"18","03"},{1052092800,"19","03"},{1052179200,"19","03"},{1052265600,"19","03"},{1052352000,"19","03"},{1052438400,"19","03"},
  {1052524800,"19","03"},{1052611200,"19","03"},{1052697600,"20","03"},{1052784000,"20","03"},{1052870400,"20","03"},{1052956800,"20","03"},{1053043200,"20","03"},
  {1053129600,"20","03"},{1053216000,"20","03"},{1053302400,"21","03"},{1053388800,"21","03"},{1053475200,"21","03"},{1053561600,"21","03"},{1053648000,"21","03"},
  {1053734400,"21","03"},{1053820800,"21","03"},{1053907200,"22","03"},{1053993600,"22","03"},{1054080000,"22","03"},{1054166400,"22","03"},{1054252800,"22","03"},
  {1054339200,"22","03"},{1054425600,"22","03"},{1054512000,"23","03"},{1054598400,"23","03"},{1054684800,"23","03"},{1054771200,"23","03"},{1054857600,"23","03"},
  {1054944000,"23","03"},{1055030400,"23","03"},{1055116800,"24","03"},{1055203200,"24","03"},{1055289600,"24","03"},{1055376000,"24","03"},{1055462400,"24","03"},
  {1055548800,"24","03"},{1055635200,"24","03"},{1055721600,"25","03"},{1055808000,"25","03"},{1055894400,"25","03"},{1055980800,"25","03"},{1056067200,"25","03"},
  {1056153600,"25","03"},{1056240000,"25","03"},{1056326400,"26","03"},{1056412800,"26","03"},{1056499200,"26","03"},{1056585600,"26","03"},{1056672000,"26","03"},
  {1056758400,"26","03"},{1056844800,"26","03"},{1056931200,"27","03"},{1057017600,"27","03"},{1057104000,"27","03"},{1057190400,"27","03"},{1057276800,"27","03"},
  {1057363200,"27","03"},{1057449600,"27","03"},{1057536000,"28","03"},{1057622400,"28","03"},{1057708800,"28","03"},{1057795200,"28","03"},{1057881600,"28","03"},
  {1057968000,"28","03"},{1058054400,"28","03"},{1058140800,"29","03"},{1058227200,"29","03"},{1058313600,"29","03"},{1058400000,"29","03"},{1058486400,"29","03"},
  {1058572800,"29","03"},{1058659200,"29","03"},{1058745600,"30","03"},{1058832000,"30","03"},{1058918400,"30","03"},{1059004800,"30","03"},{1059091200,"30","03"},
  {1059177600,"30","03"},{1059264000,"30","03"},{1059350400,"31","03"},{1059436800,"31","03"},{1059523200,"31","03"},{1059609600,"31","03"},{1059696000,"31","03"},
  {1059782400,"31","03"},{1059868800,"31","03"},{1059955200,"32","03"},{1060041600,"32","03"},{1060128000,"32","03"},{1060214400,"32","03"},{1060300800,"32","03"},
  {1060387200,"32","03"},{1060473600,"32","03"},{1060560000,"33","03"},{1060646400,"33","03"},{1060732800,"33","03"},{1060819200,"33","03"},{1060905600,"33","03"},
  {1060992000,"33","03"},{1061078400,"33","03"},{1061164800,"34","03"},{1061251200,"34","03"},{1061337600,"34","03"},{1061424000,"34","03"},{1061510400,"34","03"},
  {1061596800,"34","03"},{1061683200,"34","03"},{1061769600,"35","03"},{1061856000,"35","03"},{1061942400,"35","03"},{1062028800,"35","03"},{1062115200,"35","03"},
  {1062201600,"35","03"},{1062288000,"35","03"},{1062374400,"36","03"},{1062460800,"36","03"},{1062547200,"36","03"},{1062633600,"36","03"},{1062720000,"36","03"},
  {1062806400,"36","03"},{1062892800,"36","03"},{1062979200,"37","03"},{1063065600,"37","03"},{1063152000,"37","03"},{1063238400,"37","03"},{1063324800,"37","03"},
  {1063411200,"37","03"},{1063497600,"37","03"},{1063584000,"38","03"},{1063670400,"38","03"},{1063756800,"38","03"},{1063843200,"38","03"},{1063929600,"38","03"},
  {1064016000,"38","03"},{1064102400,"38","03"},{1064188800,"39","03"},{1064275200,"39","03"},{1064361600,"39","03"},{1064448000,"39","03"},{1064534400,"39","03"},
  {1064620800,"39","03"},{1064707200,"39","03"},{1064793600,"40","03"},{1064880000,"40","03"},{1064966400,"40","03"},{1065052800,"40","03"},{1065139200,"40","03"},
  {1065225600,"40","03"},{1065312000,"40","03"},{1065398400,"41","03"},{1065484800,"41","03"},{1065571200,"41","03"},{1065657600,"41","03"},{1065744000,"41","03"},
  {1065830400,"41","03"},{1065916800,"41","03"},{1066003200,"42","03"},{1066089600,"42","03"},{1066176000,"42","03"},{1066262400,"42","03"},{1066348800,"42","03"},
  {1066435200,"42","03"},{1066521600,"42","03"},{1066608000,"43","03"},{1066694400,"43","03"},{1066780800,"43","03"},{1066867200,"43","03"},{1066953600,"43","03"},
  {1067040000,"43","03"},{1067126400,"43","03"},{1067212800,"44","03"},{1067299200,"44","03"},{1067385600,"44","03"},{1067472000,"44","03"},{1067558400,"44","03"},
  {1067644800,"44","03"},{1067731200,"44","03"},{1067817600,"45","03"},{1067904000,"45","03"},{1067990400,"45","03"},{1068076800,"45","03"},{1068163200,"45","03"},
  {1068249600,"45","03"},{1068336000,"45","03"},{1068422400,"46","03"},{1068508800,"46","03"},{1068595200,"46","03"},{1068681600,"46","03"},{1068768000,"46","03"},
  {1068854400,"46","03"},{1068940800,"46","03"},{1069027200,"47","03"},{1069113600,"47","03"},{1069200000,"47","03"},{1069286400,"47","03"},{1069372800,"47","03"},
  {1069459200,"47","03"},{1069545600,"47","03"},{1069632000,"48","03"},{1069718400,"48","03"},{1069804800,"48","03"},{1069891200,"48","03"},{1069977600,"48","03"},
  {1070064000,"48","03"},{1070150400,"48","03"},{1070236800,"49","03"},{1070323200,"49","03"},{1070409600,"49","03"},{1070496000,"49","03"},{1070582400,"49","03"},
  {1070668800,"49","03"},{1070755200,"49","03"},{1070841600,"50","03"},{1070928000,"50","03"},{1071014400,"50","03"},{1071100800,"50","03"},{1071187200,"50","03"},
  {1071273600,"50","03"},{1071360000,"50","03"},{1071446400,"51","03"},{1071532800,"51","03"},{1071619200,"51","03"},{1071705600,"51","03"},{1071792000,"51","03"},
  {1071878400,"51","03"},{1071964800,"51","03"},{1072051200,"52","03"},{1072137600,"52","03"},{1072224000,"52","03"},{1072310400,"52","03"},{1072396800,"52","03"},
  {1072483200,"52","03"},{1072569600,"52","03"},{1072656000,"01","04"},{1072742400,"01","04"},{1072828800,"01","04"},{1072915200,"01","04"},{1073001600,"01","04"},
  {1073088000,"01","04"},{1073174400,"01","04"},{1073260800,"02","04"},{1073347200,"02","04"},{1073433600,"02","04"},{1073520000,"02","04"},{1073606400,"02","04"},
  {1073692800,"02","04"},{1073779200,"02","04"},{1073865600,"03","04"},{1073952000,"03","04"},{1074038400,"03","04"},{1074124800,"03","04"},{1074211200,"03","04"},
  {1074297600,"03","04"},{1074384000,"03","04"},{1074470400,"04","04"},{1074556800,"04","04"},{1074643200,"04","04"},{1074729600,"04","04"},{1074816000,"04","04"},
  {1074902400,"04","04"},{1074988800,"04","04"},{1075075200,"05","04"},{1075161600,"05","04"},{1075248000,"05","04"},{1075334400,"05","04"},{1075420800,"05","04"},
  {1075507200,"05","04"},{1075593600,"05","04"},{1075680000,"06","04"},{1075766400,"06","04"},{1075852800,"06","04"},{1075939200,"06","04"},{1076025600,"06","04"},
  {1076112000,"06","04"},{1076198400,"06","04"},{1076284800,"07","04"},{1076371200,"07","04"},{1076457600,"07","04"},{1076544000,"07","04"},{1076630400,"07","04"},
  {1076716800,"07","04"},{1076803200,"07","04"},{1076889600,"08","04"},{1076976000,"08","04"},{1077062400,"08","04"},{1077148800,"08","04"},{1077235200,"08","04"},
  {1077321600,"08","04"},{1077408000,"08","04"},{1077494400,"09","04"},{1077580800,"09","04"},{1077667200,"09","04"},{1077753600,"09","04"},{1077840000,"09","04"},
  {1077926400,"09","04"},{1078012800,"09","04"},{1078099200,"10","04"},{1078185600,"10","04"},{1078272000,"10","04"},{1078358400,"10","04"},{1078444800,"10","04"},
  {1078531200,"10","04"},{1078617600,"10","04"},{1078704000,"11","04"},{1078790400,"11","04"},{1078876800,"11","04"},{1078963200,"11","04"},{1079049600,"11","04"},
  {1079136000,"11","04"},{1079222400,"11","04"},{1079308800,"12","04"},{1079395200,"12","04"},{1079481600,"12","04"},{1079568000,"12","04"},{1079654400,"12","04"},
  {1079740800,"12","04"},{1079827200,"12","04"},{1079913600,"13","04"},{1080000000,"13","04"},{1080086400,"13","04"},{1080172800,"13","04"},{1080259200,"13","04"},
  {1080345600,"13","04"},{1080432000,"13","04"},{1080518400,"14","04"},{1080604800,"14","04"},{1080691200,"14","04"},{1080777600,"14","04"},{1080864000,"14","04"},
  {1080950400,"14","04"},{1081036800,"14","04"},{1081123200,"15","04"},{1081209600,"15","04"},{1081296000,"15","04"},{1081382400,"15","04"},{1081468800,"15","04"},
  {1081555200,"15","04"},{1081641600,"15","04"},{1081728000,"16","04"},{1081814400,"16","04"},{1081900800,"16","04"},{1081987200,"16","04"},{1082073600,"16","04"},
  {1082160000,"16","04"},{1082246400,"16","04"},{1082332800,"17","04"},{1082419200,"17","04"},{1082505600,"17","04"},{1082592000,"17","04"},{1082678400,"17","04"},
  {1082764800,"17","04"},{1082851200,"17","04"},{1082937600,"18","04"},{1083024000,"18","04"},{1083110400,"18","04"},{1083196800,"18","04"},{1083283200,"18","04"},
  {1083369600,"18","04"},{1083456000,"18","04"},{1083542400,"19","04"},{1083628800,"19","04"},{1083715200,"19","04"},{1083801600,"19","04"},{1083888000,"19","04"},
  {1083974400,"19","04"},{1084060800,"19","04"},{1084147200,"20","04"},{1084233600,"20","04"},{1084320000,"20","04"},{1084406400,"20","04"},{1084492800,"20","04"},
  {1084579200,"20","04"},{1084665600,"20","04"},{1084752000,"21","04"},{1084838400,"21","04"},{1084924800,"21","04"},{1085011200,"21","04"},{1085097600,"21","04"},
  {1085184000,"21","04"},{1085270400,"21","04"},{1085356800,"22","04"},{1085443200,"22","04"},{1085529600,"22","04"},{1085616000,"22","04"},{1085702400,"22","04"},
  {1085788800,"22","04"},{1085875200,"22","04"},{1085961600,"23","04"},{1086048000,"23","04"},{1086134400,"23","04"},{1086220800,"23","04"},{1086307200,"23","04"},
  {1086393600,"23","04"},{1086480000,"23","04"},{1086566400,"24","04"},{1086652800,"24","04"},{1086739200,"24","04"},{1086825600,"24","04"},{1086912000,"24","04"},
  {1086998400,"24","04"},{1087084800,"24","04"},{1087171200,"25","04"},{1087257600,"25","04"},{1087344000,"25","04"},{1087430400,"25","04"},{1087516800,"25","04"},
  {1087603200,"25","04"},{1087689600,"25","04"},{1087776000,"26","04"},{1087862400,"26","04"},{1087948800,"26","04"},{1088035200,"26","04"},{1088121600,"26","04"},
  {1088208000,"26","04"},{1088294400,"26","04"},{1088380800,"27","04"},{1088467200,"27","04"},{1088553600,"27","04"},{1088640000,"27","04"},{1088726400,"27","04"},
  {1088812800,"27","04"},{1088899200,"27","04"},{1088985600,"28","04"},{1089072000,"28","04"},{1089158400,"28","04"},{1089244800,"28","04"},{1089331200,"28","04"},
  {1089417600,"28","04"},{1089504000,"28","04"},{1089590400,"29","04"},{1089676800,"29","04"},{1089763200,"29","04"},{1089849600,"29","04"},{1089936000,"29","04"},
  {1090022400,"29","04"},{1090108800,"29","04"},{1090195200,"30","04"},{1090281600,"30","04"},{1090368000,"30","04"},{1090454400,"30","04"},{1090540800,"30","04"},
  {1090627200,"30","04"},{1090713600,"30","04"},{1090800000,"31","04"},{1090886400,"31","04"},{1090972800,"31","04"},{1091059200,"31","04"},{1091145600,"31","04"},
  {1091232000,"31","04"},{1091318400,"31","04"},{1091404800,"32","04"},{1091491200,"32","04"},{1091577600,"32","04"},{1091664000,"32","04"},{1091750400,"32","04"},
  {1091836800,"32","04"},{1091923200,"32","04"},{1092009600,"33","04"},{1092096000,"33","04"},{1092182400,"33","04"},{1092268800,"33","04"},{1092355200,"33","04"},
  {1092441600,"33","04"},{1092528000,"33","04"},{1092614400,"34","04"},{1092700800,"34","04"},{1092787200,"34","04"},{1092873600,"34","04"},{1092960000,"34","04"},
  {1093046400,"34","04"},{1093132800,"34","04"},{1093219200,"35","04"},{1093305600,"35","04"},{1093392000,"35","04"},{1093478400,"35","04"},{1093564800,"35","04"},
  {1093651200,"35","04"},{1093737600,"35","04"},{1093824000,"36","04"},{1093910400,"36","04"},{1093996800,"36","04"},{1094083200,"36","04"},{1094169600,"36","04"},
  {1094256000,"36","04"},{1094342400,"36","04"},{1094428800,"37","04"},{1094515200,"37","04"},{1094601600,"37","04"},{1094688000,"37","04"},{1094774400,"37","04"},
  {1094860800,"37","04"},{1094947200,"37","04"},{1095033600,"38","04"},{1095120000,"38","04"},{1095206400,"38","04"},{1095292800,"38","04"},{1095379200,"38","04"},
  {1095465600,"38","04"},{1095552000,"38","04"},{1095638400,"39","04"},{1095724800,"39","04"},{1095811200,"39","04"},{1095897600,"39","04"},{1095984000,"39","04"},
  {1096070400,"39","04"},{1096156800,"39","04"},{1096243200,"40","04"},{1096329600,"40","04"},{1096416000,"40","04"},{1096502400,"40","04"},{1096588800,"40","04"},
  {1096675200,"40","04"},{1096761600,"40","04"},{1096848000,"41","04"},{1096934400,"41","04"},{1097020800,"41","04"},{1097107200,"41","04"},{1097193600,"41","04"},
  {1097280000,"41","04"},{1097366400,"41","04"},{1097452800,"42","04"},{1097539200,"42","04"},{1097625600,"42","04"},{1097712000,"42","04"},{1097798400,"42","04"},
  {1097884800,"42","04"},{1097971200,"42","04"},{1098057600,"43","04"},{1098144000,"43","04"},{1098230400,"43","04"},{1098316800,"43","04"},{1098403200,"43","04"},
  {1098489600,"43","04"},{1098576000,"43","04"},{1098662400,"44","04"},{1098748800,"44","04"},{1098835200,"44","04"},{1098921600,"44","04"},{1099008000,"44","04"},
  {1099094400,"44","04"},{1099180800,"44","04"},{1099267200,"45","04"},{1099353600,"45","04"},{1099440000,"45","04"},{1099526400,"45","04"},{1099612800,"45","04"},
  {1099699200,"45","04"},{1099785600,"45","04"},{1099872000,"46","04"},{1099958400,"46","04"},{1100044800,"46","04"},{1100131200,"46","04"},{1100217600,"46","04"},
  {1100304000,"46","04"},{1100390400,"46","04"},{1100476800,"47","04"},{1100563200,"47","04"},{1100649600,"47","04"},{1100736000,"47","04"},{1100822400,"47","04"},
  {1100908800,"47","04"},{1100995200,"47","04"},{1101081600,"48","04"},{1101168000,"48","04"},{1101254400,"48","04"},{1101340800,"48","04"},{1101427200,"48","04"},
  {1101513600,"48","04"},{1101600000,"48","04"},{1101686400,"49","04"},{1101772800,"49","04"},{1101859200,"49","04"},{1101945600,"49","04"},{1102032000,"49","04"},
  {1102118400,"49","04"},{1102204800,"49","04"},{1102291200,"50","04"},{1102377600,"50","04"},{1102464000,"50","04"},{1102550400,"50","04"},{1102636800,"50","04"},
  {1102723200,"50","04"},{1102809600,"50","04"},{1102896000,"51","04"},{1102982400,"51","04"},{1103068800,"51","04"},{1103155200,"51","04"},{1103241600,"51","04"},
  {1103328000,"51","04"},{1103414400,"51","04"},{1103500800,"52","04"},{1103587200,"52","04"},{1103673600,"52","04"},{1103760000,"52","04"},{1103846400,"52","04"},
  {1103932800,"52","04"},{1104019200,"52","04"},{1104105600,"53","04"},{1104192000,"53","04"},{1104278400,"53","04"},{1104364800,"53","04"},{1104451200,"53","04"},
  {1104537600,"53","04"},{1104624000,"53","04"},{1104710400,"01","05"},{1104796800,"01","05"},{1104883200,"01","05"},{1104969600,"01","05"},{1105056000,"01","05"},
  {1105142400,"01","05"},{1105228800,"01","05"},{1105315200,"02","05"},{1105401600,"02","05"},{1105488000,"02","05"},{1105574400,"02","05"},{1105660800,"02","05"},
  {1105747200,"02","05"},{1105833600,"02","05"},{1105920000,"03","05"},{1106006400,"03","05"},{1106092800,"03","05"},{1106179200,"03","05"},{1106265600,"03","05"},
  {1106352000,"03","05"},{1106438400,"03","05"},{1106524800,"04","05"},{1106611200,"04","05"},{1106697600,"04","05"},{1106784000,"04","05"},{1106870400,"04","05"},
  {1106956800,"04","05"},{1107043200,"04","05"},{1107129600,"05","05"},{1107216000,"05","05"},{1107302400,"05","05"},{1107388800,"05","05"},{1107475200,"05","05"},
  {1107561600,"05","05"},{1107648000,"05","05"},{1107734400,"06","05"},{1107820800,"06","05"},{1107907200,"06","05"},{1107993600,"06","05"},{1108080000,"06","05"},
  {1108166400,"06","05"},{1108252800,"06","05"},{1108339200,"07","05"},{1108425600,"07","05"},{1108512000,"07","05"},{1108598400,"07","05"},{1108684800,"07","05"},
  {1108771200,"07","05"},{1108857600,"07","05"},{1108944000,"08","05"},{1109030400,"08","05"},{1109116800,"08","05"},{1109203200,"08","05"},{1109289600,"08","05"},
  {1109376000,"08","05"},{1109462400,"08","05"},{1109548800,"09","05"},{1109635200,"09","05"},{1109721600,"09","05"},{1109808000,"09","05"},{1109894400,"09","05"},
  {1109980800,"09","05"},{1110067200,"09","05"},{1110153600,"10","05"},{1110240000,"10","05"},{1110326400,"10","05"},{1110412800,"10","05"},{1110499200,"10","05"},
  {1110585600,"10","05"},{1110672000,"10","05"},{1110758400,"11","05"},{1110844800,"11","05"},{1110931200,"11","05"},{1111017600,"11","05"},{1111104000,"11","05"},
  {1111190400,"11","05"},{1111276800,"11","05"},{1111363200,"12","05"},{1111449600,"12","05"},{1111536000,"12","05"},{1111622400,"12","05"},{1111708800,"12","05"},
  {1111795200,"12","05"},{1111881600,"12","05"},{1111968000,"13","05"},{1112054400,"13","05"},{1112140800,"13","05"},{1112227200,"13","05"},{1112313600,"13","05"},
  {1112400000,"13","05"},{1112486400,"13","05"},{1112572800,"14","05"},{1112659200,"14","05"},{1112745600,"14","05"},{1112832000,"14","05"},{1112918400,"14","05"},
  {1113004800,"14","05"},{1113091200,"14","05"},{1113177600,"15","05"},{1113264000,"15","05"},{1113350400,"15","05"},{1113436800,"15","05"},{1113523200,"15","05"},
  {1113609600,"15","05"},{1113696000,"15","05"},{1113782400,"16","05"},{1113868800,"16","05"},{1113955200,"16","05"},{1114041600,"16","05"},{1114128000,"16","05"},
  {1114214400,"16","05"},{1114300800,"16","05"},{1114387200,"17","05"},{1114473600,"17","05"},{1114560000,"17","05"},{1114646400,"17","05"},{1114732800,"17","05"},
  {1114819200,"17","05"},{1114905600,"17","05"},{1114992000,"18","05"},{1115078400,"18","05"},{1115164800,"18","05"},{1115251200,"18","05"},{1115337600,"18","05"},
  {1115424000,"18","05"},{1115510400,"18","05"},{1115596800,"19","05"},{1115683200,"19","05"},{1115769600,"19","05"},{1115856000,"19","05"},{1115942400,"19","05"},
  {1116028800,"19","05"},{1116115200,"19","05"},{1116201600,"20","05"},{1116288000,"20","05"},{1116374400,"20","05"},{1116460800,"20","05"},{1116547200,"20","05"},
  {1116633600,"20","05"},{1116720000,"20","05"},{1116806400,"21","05"},{1116892800,"21","05"},{1116979200,"21","05"},{1117065600,"21","05"},{1117152000,"21","05"},
  {1117238400,"21","05"},{1117324800,"21","05"},{1117411200,"22","05"},{1117497600,"22","05"},{1117584000,"22","05"},{1117670400,"22","05"},{1117756800,"22","05"},
  {1117843200,"22","05"},{1117929600,"22","05"},{1118016000,"23","05"},{1118102400,"23","05"},{1118188800,"23","05"},{1118275200,"23","05"},{1118361600,"23","05"},
  {1118448000,"23","05"},{1118534400,"23","05"},{1118620800,"24","05"},{1118707200,"24","05"},{1118793600,"24","05"},{1118880000,"24","05"},{1118966400,"24","05"},
  {1119052800,"24","05"},{1119139200,"24","05"},{1119225600,"25","05"},{1119312000,"25","05"},{1119398400,"25","05"},{1119484800,"25","05"},{1119571200,"25","05"},
  {1119657600,"25","05"},{1119744000,"25","05"},{1119830400,"26","05"},{1119916800,"26","05"},{1120003200,"26","05"},{1120089600,"26","05"},{1120176000,"26","05"},
  {1120262400,"26","05"},{1120348800,"26","05"},{1120435200,"27","05"},{1120521600,"27","05"},{1120608000,"27","05"},{1120694400,"27","05"},{1120780800,"27","05"},
  {1120867200,"27","05"},{1120953600,"27","05"},{1121040000,"28","05"},{1121126400,"28","05"},{1121212800,"28","05"},{1121299200,"28","05"},{1121385600,"28","05"},
  {1121472000,"28","05"},{1121558400,"28","05"},{1121644800,"29","05"},{1121731200,"29","05"},{1121817600,"29","05"},{1121904000,"29","05"},{1121990400,"29","05"},
  {1122076800,"29","05"},{1122163200,"29","05"},{1122249600,"30","05"},{1122336000,"30","05"},{1122422400,"30","05"},{1122508800,"30","05"},{1122595200,"30","05"},
  {1122681600,"30","05"},{1122768000,"30","05"},{1122854400,"31","05"},{1122940800,"31","05"},{1123027200,"31","05"},{1123113600,"31","05"},{1123200000,"31","05"},
  {1123286400,"31","05"},{1123372800,"31","05"},{1123459200,"32","05"},{1123545600,"32","05"},{1123632000,"32","05"},{1123718400,"32","05"},{1123804800,"32","05"},
  {1123891200,"32","05"},{1123977600,"32","05"},{1124064000,"33","05"},{1124150400,"33","05"},{1124236800,"33","05"},{1124323200,"33","05"},{1124409600,"33","05"},
  {1124496000,"33","05"},{1124582400,"33","05"},{1124668800,"34","05"},{1124755200,"34","05"},{1124841600,"34","05"},{1124928000,"34","05"},{1125014400,"34","05"},
  {1125100800,"34","05"},{1125187200,"34","05"},{1125273600,"35","05"},{1125360000,"35","05"},{1125446400,"35","05"},{1125532800,"35","05"},{1125619200,"35","05"},
  {1125705600,"35","05"},{1125792000,"35","05"},{1125878400,"36","05"},{1125964800,"36","05"},{1126051200,"36","05"},{1126137600,"36","05"},{1126224000,"36","05"},
  {1126310400,"36","05"},{1126396800,"36","05"},{1126483200,"37","05"},{1126569600,"37","05"},{1126656000,"37","05"},{1126742400,"37","05"},{1126828800,"37","05"},
  {1126915200,"37","05"},{1127001600,"37","05"},{1127088000,"38","05"},{1127174400,"38","05"},{1127260800,"38","05"},{1127347200,"38","05"},{1127433600,"38","05"},
  {1127520000,"38","05"},{1127606400,"38","05"},{1127692800,"39","05"},{1127779200,"39","05"},{1127865600,"39","05"},{1127952000,"39","05"},{1128038400,"39","05"},
  {1128124800,"39","05"},{1128211200,"39","05"},{1128297600,"40","05"},{1128384000,"40","05"},{1128470400,"40","05"},{1128556800,"40","05"},{1128643200,"40","05"},
  {1128729600,"40","05"},{1128816000,"40","05"},{1128902400,"41","05"},{1128988800,"41","05"},{1129075200,"41","05"},{1129161600,"41","05"},{1129248000,"41","05"},
  {1129334400,"41","05"},{1129420800,"41","05"},{1129507200,"42","05"},{1129593600,"42","05"},{1129680000,"42","05"},{1129766400,"42","05"},{1129852800,"42","05"},
  {1129939200,"42","05"},{1130025600,"42","05"},{1130112000,"43","05"},{1130198400,"43","05"},{1130284800,"43","05"},{1130371200,"43","05"},{1130457600,"43","05"},
  {1130544000,"43","05"},{1130630400,"43","05"},{1130716800,"44","05"},{1130803200,"44","05"},{1130889600,"44","05"},{1130976000,"44","05"},{1131062400,"44","05"},
  {1131148800,"44","05"},{1131235200,"44","05"},{1131321600,"45","05"},{1131408000,"45","05"},{1131494400,"45","05"},{1131580800,"45","05"},{1131667200,"45","05"},
  {1131753600,"45","05"},{1131840000,"45","05"},{1131926400,"46","05"},{1132012800,"46","05"},{1132099200,"46","05"},{1132185600,"46","05"},{1132272000,"46","05"},
  {1132358400,"46","05"},{1132444800,"46","05"},{1132531200,"47","05"},{1132617600,"47","05"},{1132704000,"47","05"},{1132790400,"47","05"},{1132876800,"47","05"},
  {1132963200,"47","05"},{1133049600,"47","05"},{1133136000,"48","05"},{1133222400,"48","05"},{1133308800,"48","05"},{1133395200,"48","05"},{1133481600,"48","05"},
  {1133568000,"48","05"},{1133654400,"48","05"},{1133740800,"49","05"},{1133827200,"49","05"},{1133913600,"49","05"},{1134000000,"49","05"},{1134086400,"49","05"},
  {1134172800,"49","05"},{1134259200,"49","05"},{1134345600,"50","05"},{1134432000,"50","05"},{1134518400,"50","05"},{1134604800,"50","05"},{1134691200,"50","05"},
  {1134777600,"50","05"},{1134864000,"50","05"},{1134950400,"51","05"},{1135036800,"51","05"},{1135123200,"51","05"},{1135209600,"51","05"},{1135296000,"51","05"},
  {1135382400,"51","05"},{1135468800,"51","05"},{1135555200,"52","05"},{1135641600,"52","05"},{1135728000,"52","05"},{1135814400,"52","05"},{1135900800,"52","05"},
  {1135987200,"52","05"},{1136073600,"52","05"},{1136160000,"01","06"},{1136246400,"01","06"},{1136332800,"01","06"},{1136419200,"01","06"},{1136505600,"01","06"},
  {1136592000,"01","06"},{1136678400,"01","06"},{1136764800,"02","06"},{1136851200,"02","06"},{1136937600,"02","06"},{1137024000,"02","06"},{1137110400,"02","06"},
  {1137196800,"02","06"},{1137283200,"02","06"},{1137369600,"03","06"},{1137456000,"03","06"},{1137542400,"03","06"},{1137628800,"03","06"},{1137715200,"03","06"},
  {1137801600,"03","06"},{1137888000,"03","06"},{1137974400,"04","06"},{1138060800,"04","06"},{1138147200,"04","06"},{1138233600,"04","06"},{1138320000,"04","06"},
  {1138406400,"04","06"},{1138492800,"04","06"},{1138579200,"05","06"},{1138665600,"05","06"},{1138752000,"05","06"},{1138838400,"05","06"},{1138924800,"05","06"},
  {1139011200,"05","06"},{1139097600,"05","06"},{1139184000,"06","06"},{1139270400,"06","06"},{1139356800,"06","06"},{1139443200,"06","06"},{1139529600,"06","06"},
  {1139616000,"06","06"},{1139702400,"06","06"},{1139788800,"07","06"},{1139875200,"07","06"},{1139961600,"07","06"},{1140048000,"07","06"},{1140134400,"07","06"},
  {1140220800,"07","06"},{1140307200,"07","06"},{1140393600,"08","06"},{1140480000,"08","06"},{1140566400,"08","06"},{1140652800,"08","06"},{1140739200,"08","06"},
  {1140825600,"08","06"},{1140912000,"08","06"},{1140998400,"09","06"},{1141084800,"09","06"},{1141171200,"09","06"},{1141257600,"09","06"},{1141344000,"09","06"},
  {1141430400,"09","06"},{1141516800,"09","06"},{1141603200,"10","06"},{1141689600,"10","06"},{1141776000,"10","06"},{1141862400,"10","06"},{1141948800,"10","06"},
  {1142035200,"10","06"},{1142121600,"10","06"},{1142208000,"11","06"},{1142294400,"11","06"},{1142380800,"11","06"},{1142467200,"11","06"},{1142553600,"11","06"},
  {1142640000,"11","06"},{1142726400,"11","06"},{1142812800,"12","06"},{1142899200,"12","06"},{1142985600,"12","06"},{1143072000,"12","06"},{1143158400,"12","06"},
  {1143244800,"12","06"},{1143331200,"12","06"},{1143417600,"13","06"},{1143504000,"13","06"},{1143590400,"13","06"},{1143676800,"13","06"},{1143763200,"13","06"},
  {1143849600,"13","06"},{1143936000,"13","06"},{1144022400,"14","06"},{1144108800,"14","06"},{1144195200,"14","06"},{1144281600,"14","06"},{1144368000,"14","06"},
  {1144454400,"14","06"},{1144540800,"14","06"},{1144627200,"15","06"},{1144713600,"15","06"},{1144800000,"15","06"},{1144886400,"15","06"},{1144972800,"15","06"},
  {1145059200,"15","06"},{1145145600,"15","06"},{1145232000,"16","06"},{1145318400,"16","06"},{1145404800,"16","06"},{1145491200,"16","06"},{1145577600,"16","06"},
  {1145664000,"16","06"},{1145750400,"16","06"},{1145836800,"17","06"},{1145923200,"17","06"},{1146009600,"17","06"},{1146096000,"17","06"},{1146182400,"17","06"},
  {1146268800,"17","06"},{1146355200,"17","06"},{1146441600,"18","06"},{1146528000,"18","06"},{1146614400,"18","06"},{1146700800,"18","06"},{1146787200,"18","06"},
  {1146873600,"18","06"},{1146960000,"18","06"},{1147046400,"19","06"},{1147132800,"19","06"},{1147219200,"19","06"},{1147305600,"19","06"},{1147392000,"19","06"},
  {1147478400,"19","06"},{1147564800,"19","06"},{1147651200,"20","06"},{1147737600,"20","06"},{1147824000,"20","06"},{1147910400,"20","06"},{1147996800,"20","06"},
  {1148083200,"20","06"},{1148169600,"20","06"},{1148256000,"21","06"},{1148342400,"21","06"},{1148428800,"21","06"},{1148515200,"21","06"},{1148601600,"21","06"},
  {1148688000,"21","06"},{1148774400,"21","06"},{1148860800,"22","06"},{1148947200,"22","06"},{1149033600,"22","06"},{1149120000,"22","06"},{1149206400,"22","06"},
  {1149292800,"22","06"},{1149379200,"22","06"},{1149465600,"23","06"},{1149552000,"23","06"},{1149638400,"23","06"},{1149724800,"23","06"},{1149811200,"23","06"},
  {1149897600,"23","06"},{1149984000,"23","06"},{1150070400,"24","06"},{1150156800,"24","06"},{1150243200,"24","06"},{1150329600,"24","06"},{1150416000,"24","06"},
  {1150502400,"24","06"},{1150588800,"24","06"},{1150675200,"25","06"},{1150761600,"25","06"},{1150848000,"25","06"},{1150934400,"25","06"},{1151020800,"25","06"},
  {1151107200,"25","06"},{1151193600,"25","06"},{1151280000,"26","06"},{1151366400,"26","06"},{1151452800,"26","06"},{1151539200,"26","06"},{1151625600,"26","06"},
  {1151712000,"26","06"},{1151798400,"26","06"},{1151884800,"27","06"},{1151971200,"27","06"},{1152057600,"27","06"},{1152144000,"27","06"},{1152230400,"27","06"},
  {1152316800,"27","06"},{1152403200,"27","06"},{1152489600,"28","06"},{1152576000,"28","06"},{1152662400,"28","06"},{1152748800,"28","06"},{1152835200,"28","06"},
  {1152921600,"28","06"},{1153008000,"28","06"},{1153094400,"29","06"},{1153180800,"29","06"},{1153267200,"29","06"},{1153353600,"29","06"},{1153440000,"29","06"},
  {1153526400,"29","06"},{1153612800,"29","06"},{1153699200,"30","06"},{1153785600,"30","06"},{1153872000,"30","06"},{1153958400,"30","06"},{1154044800,"30","06"},
  {1154131200,"30","06"},{1154217600,"30","06"},{1154304000,"31","06"},{1154390400,"31","06"},{1154476800,"31","06"},{1154563200,"31","06"},{1154649600,"31","06"},
  {1154736000,"31","06"},{1154822400,"31","06"},{1154908800,"32","06"},{1154995200,"32","06"},{1155081600,"32","06"},{1155168000,"32","06"},{1155254400,"32","06"},
  {1155340800,"32","06"},{1155427200,"32","06"},{1155513600,"33","06"},{1155600000,"33","06"},{1155686400,"33","06"},{1155772800,"33","06"},{1155859200,"33","06"},
  {1155945600,"33","06"},{1156032000,"33","06"},{1156118400,"34","06"},{1156204800,"34","06"},{1156291200,"34","06"},{1156377600,"34","06"},{1156464000,"34","06"},
  {1156550400,"34","06"},{1156636800,"34","06"},{1156723200,"35","06"},{1156809600,"35","06"},{1156896000,"35","06"},{1156982400,"35","06"},{1157068800,"35","06"},
  {1157155200,"35","06"},{1157241600,"35","06"},{1157328000,"36","06"},{1157414400,"36","06"},{1157500800,"36","06"},{1157587200,"36","06"},{1157673600,"36","06"},
  {1157760000,"36","06"},{1157846400,"36","06"},{1157932800,"37","06"},{1158019200,"37","06"},{1158105600,"37","06"},{1158192000,"37","06"},{1158278400,"37","06"},
  {1158364800,"37","06"},{1158451200,"37","06"},{1158537600,"38","06"},{1158624000,"38","06"},{1158710400,"38","06"},{1158796800,"38","06"},{1158883200,"38","06"},
  {1158969600,"38","06"},{1159056000,"38","06"},{1159142400,"39","06"},{1159228800,"39","06"},{1159315200,"39","06"},{1159401600,"39","06"},{1159488000,"39","06"},
  {1159574400,"39","06"},{1159660800,"39","06"},{1159747200,"40","06"},{1159833600,"40","06"},{1159920000,"40","06"},{1160006400,"40","06"},{1160092800,"40","06"},
  {1160179200,"40","06"},{1160265600,"40","06"},{1160352000,"41","06"},{1160438400,"41","06"},{1160524800,"41","06"},{1160611200,"41","06"},{1160697600,"41","06"},
  {1160784000,"41","06"},{1160870400,"41","06"},{1160956800,"42","06"},{1161043200,"42","06"},{1161129600,"42","06"},{1161216000,"42","06"},{1161302400,"42","06"},
  {1161388800,"42","06"},{1161475200,"42","06"},{1161561600,"43","06"},{1161648000,"43","06"},{1161734400,"43","06"},{1161820800,"43","06"},{1161907200,"43","06"},
  {1161993600,"43","06"},{1162080000,"43","06"},{1162166400,"44","06"},{1162252800,"44","06"},{1162339200,"44","06"},{1162425600,"44","06"},{1162512000,"44","06"},
  {1162598400,"44","06"},{1162684800,"44","06"},{1162771200,"45","06"},{1162857600,"45","06"},{1162944000,"45","06"},{1163030400,"45","06"},{1163116800,"45","06"},
  {1163203200,"45","06"},{1163289600,"45","06"},{1163376000,"46","06"},{1163462400,"46","06"},{1163548800,"46","06"},{1163635200,"46","06"},{1163721600,"46","06"},
  {1163808000,"46","06"},{1163894400,"46","06"},{1163980800,"47","06"},{1164067200,"47","06"},{1164153600,"47","06"},{1164240000,"47","06"},{1164326400,"47","06"},
  {1164412800,"47","06"},{1164499200,"47","06"},{1164585600,"48","06"},{1164672000,"48","06"},{1164758400,"48","06"},{1164844800,"48","06"},{1164931200,"48","06"},
  {1165017600,"48","06"},{1165104000,"48","06"},{1165190400,"49","06"},{1165276800,"49","06"},{1165363200,"49","06"},{1165449600,"49","06"},{1165536000,"49","06"},
  {1165622400,"49","06"},{1165708800,"49","06"},{1165795200,"50","06"},{1165881600,"50","06"},{1165968000,"50","06"},{1166054400,"50","06"},{1166140800,"50","06"},
  {1166227200,"50","06"},{1166313600,"50","06"},{1166400000,"51","06"},{1166486400,"51","06"},{1166572800,"51","06"},{1166659200,"51","06"},{1166745600,"51","06"},
  {1166832000,"51","06"},{1166918400,"51","06"},{1167004800,"52","06"},{1167091200,"52","06"},{1167177600,"52","06"},{1167264000,"52","06"},{1167350400,"52","06"},
  {1167436800,"52","06"},{1167523200,"52","06"},{1167609600,"01","07"},{1167696000,"01","07"},{1167782400,"01","07"},{1167868800,"01","07"},{1167955200,"01","07"},
  {1168041600,"01","07"},{1168128000,"01","07"},{1168214400,"02","07"},{1168300800,"02","07"},{1168387200,"02","07"},{1168473600,"02","07"},{1168560000,"02","07"},
  {1168646400,"02","07"},{1168732800,"02","07"},{1168819200,"03","07"},{1168905600,"03","07"},{1168992000,"03","07"},{1169078400,"03","07"},{1169164800,"03","07"},
  {1169251200,"03","07"},{1169337600,"03","07"},{1169424000,"04","07"},{1169510400,"04","07"},{1169596800,"04","07"},{1169683200,"04","07"},{1169769600,"04","07"},
  {1169856000,"04","07"},{1169942400,"04","07"},{1170028800,"05","07"},{1170115200,"05","07"},{1170201600,"05","07"},{1170288000,"05","07"},{1170374400,"05","07"},
  {1170460800,"05","07"},{1170547200,"05","07"},{1170633600,"06","07"},{1170720000,"06","07"},{1170806400,"06","07"},{1170892800,"06","07"},{1170979200,"06","07"},
  {1171065600,"06","07"},{1171152000,"06","07"},{1171238400,"07","07"},{1171324800,"07","07"},{1171411200,"07","07"},{1171497600,"07","07"},{1171584000,"07","07"},
  {1171670400,"07","07"},{1171756800,"07","07"},{1171843200,"08","07"},{1171929600,"08","07"},{1172016000,"08","07"},{1172102400,"08","07"},{1172188800,"08","07"},
  {1172275200,"08","07"},{1172361600,"08","07"},{1172448000,"09","07"},{1172534400,"09","07"},{1172620800,"09","07"},{1172707200,"09","07"},{1172793600,"09","07"},
  {1172880000,"09","07"},{1172966400,"09","07"},{1173052800,"10","07"},{1173139200,"10","07"},{1173225600,"10","07"},{1173312000,"10","07"},{1173398400,"10","07"},
  {1173484800,"10","07"},{1173571200,"10","07"},{1173657600,"11","07"},{1173744000,"11","07"},{1173830400,"11","07"},{1173916800,"11","07"},{1174003200,"11","07"},
  {1174089600,"11","07"},{1174176000,"11","07"},{1174262400,"12","07"},{1174348800,"12","07"},{1174435200,"12","07"},{1174521600,"12","07"},{1174608000,"12","07"},
  {1174694400,"12","07"},{1174780800,"12","07"},{1174867200,"13","07"},{1174953600,"13","07"},{1175040000,"13","07"},{1175126400,"13","07"},{1175212800,"13","07"},
  {1175299200,"13","07"},{1175385600,"13","07"},{1175472000,"14","07"},{1175558400,"14","07"},{1175644800,"14","07"},{1175731200,"14","07"},{1175817600,"14","07"},
  {1175904000,"14","07"},{1175990400,"14","07"},{1176076800,"15","07"},{1176163200,"15","07"},{1176249600,"15","07"},{1176336000,"15","07"},{1176422400,"15","07"},
  {1176508800,"15","07"},{1176595200,"15","07"},{1176681600,"16","07"},{1176768000,"16","07"},{1176854400,"16","07"},{1176940800,"16","07"},{1177027200,"16","07"},
  {1177113600,"16","07"},{1177200000,"16","07"},{1177286400,"17","07"},{1177372800,"17","07"},{1177459200,"17","07"},{1177545600,"17","07"},{1177632000,"17","07"},
  {1177718400,"17","07"},{1177804800,"17","07"},{1177891200,"18","07"},{1177977600,"18","07"},{1178064000,"18","07"},{1178150400,"18","07"},{1178236800,"18","07"},
  {1178323200,"18","07"},{1178409600,"18","07"},{1178496000,"19","07"},{1178582400,"19","07"},{1178668800,"19","07"},{1178755200,"19","07"},{1178841600,"19","07"},
  {1178928000,"19","07"},{1179014400,"19","07"},{1179100800,"20","07"},{1179187200,"20","07"},{1179273600,"20","07"},{1179360000,"20","07"},{1179446400,"20","07"},
  {1179532800,"20","07"},{1179619200,"20","07"},{1179705600,"21","07"},{1179792000,"21","07"},{1179878400,"21","07"},{1179964800,"21","07"},{1180051200,"21","07"},
  {1180137600,"21","07"},{1180224000,"21","07"},{1180310400,"22","07"},{1180396800,"22","07"},{1180483200,"22","07"},{1180569600,"22","07"},{1180656000,"22","07"},
  {1180742400,"22","07"},{1180828800,"22","07"},{1180915200,"23","07"},{1181001600,"23","07"},{1181088000,"23","07"},{1181174400,"23","07"},{1181260800,"23","07"},
  {1181347200,"23","07"},{1181433600,"23","07"},{1181520000,"24","07"},{1181606400,"24","07"},{1181692800,"24","07"},{1181779200,"24","07"},{1181865600,"24","07"},
  {1181952000,"24","07"},{1182038400,"24","07"},{1182124800,"25","07"},{1182211200,"25","07"},{1182297600,"25","07"},{1182384000,"25","07"},{1182470400,"25","07"},
  {1182556800,"25","07"},{1182643200,"25","07"},{1182729600,"26","07"},{1182816000,"26","07"},{1182902400,"26","07"},{1182988800,"26","07"},{1183075200,"26","07"},
  {1183161600,"26","07"},{1183248000,"26","07"},{1183334400,"27","07"},{1183420800,"27","07"},{1183507200,"27","07"},{1183593600,"27","07"},{1183680000,"27","07"},
  {1183766400,"27","07"},{1183852800,"27","07"},{1183939200,"28","07"},{1184025600,"28","07"},{1184112000,"28","07"},{1184198400,"28","07"},{1184284800,"28","07"},
  {1184371200,"28","07"},{1184457600,"28","07"},{1184544000,"29","07"},{1184630400,"29","07"},{1184716800,"29","07"},{1184803200,"29","07"},{1184889600,"29","07"},
  {1184976000,"29","07"},{1185062400,"29","07"},{1185148800,"30","07"},{1185235200,"30","07"},{1185321600,"30","07"},{1185408000,"30","07"},{1185494400,"30","07"},
  {1185580800,"30","07"},{1185667200,"30","07"},{1185753600,"31","07"},{1185840000,"31","07"},{1185926400,"31","07"},{1186012800,"31","07"},{1186099200,"31","07"},
  {1186185600,"31","07"},{1186272000,"31","07"},{1186358400,"32","07"},{1186444800,"32","07"},{1186531200,"32","07"},{1186617600,"32","07"},{1186704000,"32","07"},
  {1186790400,"32","07"},{1186876800,"32","07"},{1186963200,"33","07"},{1187049600,"33","07"},{1187136000,"33","07"},{1187222400,"33","07"},{1187308800,"33","07"},
  {1187395200,"33","07"},{1187481600,"33","07"},{1187568000,"34","07"},{1187654400,"34","07"},{1187740800,"34","07"},{1187827200,"34","07"},{1187913600,"34","07"},
  {1188000000,"34","07"},{1188086400,"34","07"},{1188172800,"35","07"},{1188259200,"35","07"},{1188345600,"35","07"},{1188432000,"35","07"},{1188518400,"35","07"},
  {1188604800,"35","07"},{1188691200,"35","07"},{1188777600,"36","07"},{1188864000,"36","07"},{1188950400,"36","07"},{1189036800,"36","07"},{1189123200,"36","07"},
  {1189209600,"36","07"},{1189296000,"36","07"},{1189382400,"37","07"},{1189468800,"37","07"},{1189555200,"37","07"},{1189641600,"37","07"},{1189728000,"37","07"},
  {1189814400,"37","07"},{1189900800,"37","07"},{1189987200,"38","07"},{1190073600,"38","07"},{1190160000,"38","07"},{1190246400,"38","07"},{1190332800,"38","07"},
  {1190419200,"38","07"},{1190505600,"38","07"},{1190592000,"39","07"},{1190678400,"39","07"},{1190764800,"39","07"},{1190851200,"39","07"},{1190937600,"39","07"},
  {1191024000,"39","07"},{1191110400,"39","07"},{1191196800,"40","07"},{1191283200,"40","07"},{1191369600,"40","07"},{1191456000,"40","07"},{1191542400,"40","07"},
  {1191628800,"40","07"},{1191715200,"40","07"},{1191801600,"41","07"},{1191888000,"41","07"},{1191974400,"41","07"},{1192060800,"41","07"},{1192147200,"41","07"},
  {1192233600,"41","07"},{1192320000,"41","07"},{1192406400,"42","07"},{1192492800,"42","07"},{1192579200,"42","07"},{1192665600,"42","07"},{1192752000,"42","07"},
  {1192838400,"42","07"},{1192924800,"42","07"},{1193011200,"43","07"},{1193097600,"43","07"},{1193184000,"43","07"},{1193270400,"43","07"},{1193356800,"43","07"},
  {1193443200,"43","07"},{1193529600,"43","07"},{1193616000,"44","07"},{1193702400,"44","07"},{1193788800,"44","07"},{1193875200,"44","07"},{1193961600,"44","07"},
  {1194048000,"44","07"},{1194134400,"44","07"},{1194220800,"45","07"},{1194307200,"45","07"},{1194393600,"45","07"},{1194480000,"45","07"},{1194566400,"45","07"},
  {1194652800,"45","07"},{1194739200,"45","07"},{1194825600,"46","07"},{1194912000,"46","07"},{1194998400,"46","07"},{1195084800,"46","07"},{1195171200,"46","07"},
  {1195257600,"46","07"},{1195344000,"46","07"},{1195430400,"47","07"},{1195516800,"47","07"},{1195603200,"47","07"},{1195689600,"47","07"},{1195776000,"47","07"},
  {1195862400,"47","07"},{1195948800,"47","07"},{1196035200,"48","07"},{1196121600,"48","07"},{1196208000,"48","07"},{1196294400,"48","07"},{1196380800,"48","07"},
  {1196467200,"48","07"},{1196553600,"48","07"},{1196640000,"49","07"},{1196726400,"49","07"},{1196812800,"49","07"},{1196899200,"49","07"},{1196985600,"49","07"},
  {1197072000,"49","07"},{1197158400,"49","07"},{1197244800,"50","07"},{1197331200,"50","07"},{1197417600,"50","07"},{1197504000,"50","07"},{1197590400,"50","07"},
  {1197676800,"50","07"},{1197763200,"50","07"},{1197849600,"51","07"},{1197936000,"51","07"},{1198022400,"51","07"},{1198108800,"51","07"},{1198195200,"51","07"},
  {1198281600,"51","07"},{1198368000,"51","07"},{1198454400,"52","07"},{1198540800,"52","07"},{1198627200,"52","07"},{1198713600,"52","07"},{1198800000,"52","07"},
  {1198886400,"52","07"},{1198972800,"52","07"},
};

void test_strftime__full_iso8601(void) {
  char tmbuf[512];

  struct tm wrk = {0};

  for(int i = 0; i < ARRAY_LENGTH(s_iso8601_data); i++) {
    gmtime_r(&s_iso8601_data[i].stamp, &wrk);
    printf("%d: %ld(%d,%d,%d,%d), ", i, s_iso8601_data[i].stamp, wrk.tm_yday,wrk.tm_year,wrk.tm_mon,wrk.tm_mday);
    tmbuf[0] = '\0';
    strftime(tmbuf, sizeof(tmbuf), "%V", &wrk);
    cl_assert_equal_s(tmbuf, s_iso8601_data[i].str_V);
    printf("%s, ", tmbuf);
    tmbuf[0] = '\0';
    strftime(tmbuf, sizeof(tmbuf), "%g", &wrk);
    cl_assert_equal_s(tmbuf, s_iso8601_data[i].str_g);
    printf("%s\n", tmbuf);
  }
}
#endif

#if EXTENSION_SU
void test_strftime__full_percent_z(void) {
  char tmbuf[512];

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0000");

  jan_2_2015__13_00_00.tm_gmtoff = 60;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0001");

  jan_2_2015__13_00_00.tm_gmtoff = 600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0010");

  jan_2_2015__13_00_00.tm_gmtoff = 3600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0100");

  jan_2_2015__13_00_00.tm_gmtoff = 4200;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0110");

  jan_2_2015__13_00_00.tm_gmtoff = 36000;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+1000");

  jan_2_2015__13_00_00.tm_gmtoff = 39600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "+1100");

  jan_2_2015__13_00_00.tm_gmtoff = -60;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
#if !OUR_STRFTIME_BUGS
  cl_assert_equal_s(tmbuf, "-0001");
#else // BUG!!
  cl_assert_equal_s(tmbuf, "+0001");
#endif

  jan_2_2015__13_00_00.tm_gmtoff = -600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
#if !OUR_STRFTIME_BUGS
  cl_assert_equal_s(tmbuf, "-0010");
#else // BUG!!
  cl_assert_equal_s(tmbuf, "+0010");
#endif

  jan_2_2015__13_00_00.tm_gmtoff = -3600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "-0100");

  jan_2_2015__13_00_00.tm_gmtoff = -4200;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "-0110");

  jan_2_2015__13_00_00.tm_gmtoff = -36000;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "-1000");

  jan_2_2015__13_00_00.tm_gmtoff = -39600;
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%z", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "-1100");
}
#endif

void test_strftime__full_percent_U(void) {
  char tmbuf[512];

  struct tm dec_30_2014__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2014 - 1900, .tm_mon = 11, .tm_mday = 30, // 2014/12/30
    .tm_wday = 2, .tm_yday = 363, // Tuesday, 364th day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  struct tm jan_1_2016__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2016 - 1900, .tm_mon = 0, .tm_mday = 1, // 2016/01/01
    .tm_wday = 5, .tm_yday = 0, // Friday, 1st day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%U", &dec_30_2014__13_00_00);
  cl_assert_equal_s(tmbuf, "52");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%U", &jan_2_2015__13_00_00);
  cl_assert_equal_s(tmbuf, "00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%U", &jan_1_2016__13_00_00);
  cl_assert_equal_s(tmbuf, "00");
}

// Test the zero flag
void test_strftime__zeroflag(void) {
  char tmbuf[512];

  struct tm jan_2_10015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 10015 - 1900, .tm_mon = 0, .tm_mday = 2, // 10015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%02a %01A %06b %04B", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%06c", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 10015");

#if EXTENSION_SU
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%0C %01h", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "100 Jan");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%02D %04e", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "01/02/15 0002");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%05n %03r", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "\n 01:00:00 PM");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%02R %06t %04T", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "13:00 \t 13:00:00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%5u %3V", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "00005 001");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%9z", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "+0000");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%12d", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "000000000002");

#if EXTENSION_C99
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%123F", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "10015-01-02");
#endif

#if EXTENSION_TZ
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%7G %01g", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "0010015 15");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%4k %0l", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "  13 01");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%3H %4I", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "013 0001");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%01j %05m %03M %05p", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "002 00001 000 PM");

#if EXTENSION_GNU
  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%05P", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "pm");
#endif

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%4S %2U", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "0000 00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%3w %1W", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "005 00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%123x %432X", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "01/02/15 13:00:00");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%6Y %3y", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "010015 015");

  tmbuf[0] = '\0';
  strftime(tmbuf, sizeof(tmbuf), "%Z %%", &jan_2_10015__13_00_00);
  cl_assert_equal_s(tmbuf, "UTC %");
}

extern size_t localized_strftime(char* s,
      size_t maxsize, const char* format, const struct tm* tim_p, const char *locale);

void test_strftime__i18n(void) {
  char tmbuf[512];

  struct tm jan_2_2015__13_00_00 = {
    .tm_hour = 13, .tm_min = 0, .tm_sec = 0, // 13:00:00
    .tm_year = 2015 - 1900, .tm_mon = 0, .tm_mday = 2, // 2015/01/02
    .tm_wday = 5, .tm_yday = 1, // Friday, 2nd day of the year
    .tm_gmtoff = 0, .tm_isdst = 0, .tm_zone = "UTC", // No DST, UTC+0
  };

  s_i18n_translate = false;
  s_i18n_locale_other = false;
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");

  s_i18n_translate = true;
  s_i18n_locale_other = false;
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Hola Hola Hola Hola");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Hola");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");

  s_i18n_translate = false;
  s_i18n_locale_other = true;
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Friday Jan January");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Fri Jan  2 13:00:00 2015");

  s_i18n_translate = true;
  s_i18n_locale_other = true;
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Hola Hola Hola Hola");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, NULL);
  cl_assert_equal_s(tmbuf, "Hola");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%a %A %b %B", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Hola Hola Hola Hola");
  tmbuf[0] = '\0';
  localized_strftime(tmbuf, sizeof(tmbuf), "%c", &jan_2_2015__13_00_00, i18n_get_locale());
  cl_assert_equal_s(tmbuf, "Hola");

  s_i18n_translate = false;
  s_i18n_locale_other = false;
}
