# JerryScript types

## jerry_init_flag_t

Enum that contains the following elements:

 - JERRY_INIT_EMPTY - empty flag set
 - JERRY_INIT_SHOW_OPCODES - dump byte-code to log after parse
 - JERRY_INIT_SHOW_REGEXP_OPCODES - dump regexp byte-code to log after compilation
 - JERRY_INIT_MEM_STATS - dump memory statistics
 - JERRY_INIT_MEM_STATS_SEPARATE - dump memory statistics and reset peak values after parse

## jerry_error_t

Possible types of an error:

 - JERRY_ERROR_COMMON - common error
 - JERRY_ERROR_EVAL - eval error
 - JERRY_ERROR_RANGE - range error
 - JERRY_ERROR_REFERENCE - reference error
 - JERRY_ERROR_SYNTAX - syntax error
 - JERRY_ERROR_TYPE - type error
 - JERRY_ERROR_URI - URI error

## jerry_char_t

**Summary**

Jerry's char value

**Prototype**

```c
typedef uint8_t jerry_char_t;
```

## jerry_char_ptr_t

**Summary**

Pointer to an array of character values

**Prototype**

```c
typedef jerry_char_t *jerry_char_ptr_t;
```

## jerry_size_t

**Summary**

Jerry's size

**Prototype**

```c
typedef uint32_t jerry_size_t;
```

## jerry_length_t

**Summary**

Jerry's length

**Prototype**

```c
typedef uint32_t jerry_length_t;
```

## jerry_value_t

**Summary**

JerryScript value can be a boolean, number, null, object, string or undefined. The value has an error flag,
that indicates whether is an error or not. Every type has an error flag not only objects. The error flag should
be cleared before the value is passed as an argument, otherwise it can lead to a type error. The error objects
created by API functions has the error flag set.

**Prototype**

```c
typedef uint32_t jerry_value_t;
```

## jerry_property_descriptor_t

**Summary**

Description of ECMA property descriptor

**Prototype**

```c
typedef struct
{
  /** Is [[Value]] defined? */
  bool is_value_defined;

  /** Is [[Get]] defined? */
  bool is_get_defined;

  /** Is [[Set]] defined? */
  bool is_set_defined;

  /** Is [[Writable]] defined? */
  bool is_writable_defined;

  /** [[Writable]] */
  bool is_writable;

  /** Is [[Enumerable]] defined? */
  bool is_enumerable_defined;

  /** [[Enumerable]] */
  bool is_enumerable;

  /** Is [[Configurable]] defined? */
  bool is_configurable_defined;

  /** [[Configurable]] */
  bool is_configurable;

  /** [[Value]] */
  jerry_value_t value;

  /** [[Get]] */
  jerry_value_t getter;

  /** [[Set]] */
  jerry_value_t setter;
} jerry_property_descriptor_t;
```

## jerry_external_handler_t

**Summary**

Type of an external function handler

**Prototype**

```c
typedef jerry_value_t (*jerry_external_handler_t) (const jerry_value_t function_obj,
                                                   const jerry_value_t this_val,
                                                   const jerry_value_t args_p[],
                                                   const jerry_length_t args_count);
```

## jerry_object_free_callback_t

**Summary**

Native free callback of an object

**Prototype**

```c
typedef void (*jerry_object_free_callback_t) (const uintptr_t native_p);
```

## jerry_object_property_foreach_t

**Summary**

Function type applied for each data property of an object

**Prototype**

```c
typedef bool (*jerry_object_property_foreach_t) (const jerry_value_t property_name,
                                                 const jerry_value_t property_value,
                                                 void *user_data_p);
```

# General engine functions

## jerry_init

**Summary**

Initializes the JerryScript engine, making possible to run JavaScript code and perform operations on
JavaScript values.

**Prototype**

```c
void
jerry_init (jerry_init_flag_t flags)
```

`flags` - combination of various engine configuration flags:

- `JERRY_INIT_EMPTY` - no flags, just initialize in default configuration.
- `JERRY_INIT_SHOW_OPCODES` - print compiled byte-code.
- `JERRY_INIT_SHOW_REGEXP_OPCODES` - print compiled regexp byte-code.
- `JERRY_INIT_MEM_STATS` - dump memory statistics.
- `JERRY_INIT_MEM_STATS_SEPARATE` - dump memory statistics and reset peak values after parse.

**Example**

```c
{
  jerry_init (JERRY_INIT_SHOW_OPCODES | JERRY_INIT_SHOW_REGEXP_OPCODES);

  // ...

  jerry_cleanup ();
}
```

**See also**

- [jerry_cleanup](#jerry_cleanup)


## jerry_cleanup

**Summary**

Finish JavaScript engine execution, freeing memory and JavaScript values.

*Note*: JavaScript values, received from engine, will be inaccessible after the cleanup.

**Prototype**

```c
void
jerry_cleanup (void);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_register_magic_strings

**Summary**

Registers an external magic string array.

**Prototype**

```c
void
jerry_register_magic_strings  (const jerry_char_ptr_t *ex_str_items_p,
                               uint32_t count,
                               const jerry_length_t *str_lengths_p);
```

- `ex_str_items_p` - character arrays, representing external magic strings' contents
- `count` - number of the strings
- `str_lengths_p` - lengths of the strings

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_char_ptr_t magic_string_items[] = {
                                                         (const jerry_char_ptr_t) "magicstring1",
                                                         (const jerry_char_ptr_t) "magicstring2",
                                                         (const jerry_char_ptr_t) "magicstring3"
                                                        };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_ptr_t));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = {
                                                         (jerry_length_t)strlen (magic_string_items[0]),
                                                         (jerry_length_t)strlen (magic_string_items[1]),
                                                         (jerry_length_t)strlen (magic_string_items[2])
                                                       };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)


## jerry_get_memory_limits

**Summary**

Gets configured memory limits of JerryScript.

**Prototype**

```c
void
jerry_get_memory_limits (size_t *out_data_bss_brk_limit_p,
                         size_t *out_stack_limit_p);
```

- `out_data_bss_brk_limit_p` - out parameter, that gives the maximum size of data + bss + brk sections.
- `out_stack_limit_p` - out parameter, that gives the maximum size of the stack.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  size_t stack_limit;
  size_t data_bss_brk_limit;
  jerry_get_memory_limits (&stack_limit, &data_bss_brk_limit);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)


## jerry_gc

**Summary**

Performs garbage collection.

**Prototype**

```c
void
jerry_gc (void);
```

**Example**

```c
jerry_gc ();
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)

# Parser and executor functions

Functions to parse and run JavaScript source code.

## jerry_run_simple

**Summary**

The simplest way to run JavaScript.

**Prototype**

```c
bool
jerry_run_simple (const jerry_char_t *script_source_p,
                  size_t script_source_size,
                  jerry_init_flag_t flags);
```

- `script_source_p` - source code, it must be a valid utf8 string.
- `script_source_size` - size of source code buffer, in bytes.
- `jerry_init_flag_t` - combination of various engine configuration flags
- return value
  - true, if run was successful
  - false, otherwise

**Example**

```c
{
  const jerry_char_t *script = "print ('Hello, World!');";

  jerry_run_simple (script, strlen ((const char *) script), JERRY_INIT_EMPTY);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)


## jerry_parse

**Summary**

Parse script and construct an EcmaScript function. The
lexical environment is set to the global lexical environment.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse (const jerry_char_t *source_p,
             size_t source_size,
             bool is_strict);
```

- `source_p` - string, containing source code to parse. It must be a valid utf8 string.
- `source_size` - size of the string, in bytes.
- `is_strict` - defines strict mode.
- return value
  - function object value, if script was parsed successfully,
  - thrown error, otherwise

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "print ('Hello, World!');";
  size_t script_size = strlen ((const char *) script);

  jerry_value_t parsed_code = jerry_parse (script, script_size, false);
  jerry_release_value (parsed_code);

  jerry_cleanup ();
}
```

**See also**

- [jerry_run](#jerry_run)


## jerry_run

**Summary**

Run an EcmaScript function created by `jerry_parse`.

*Note*: The code should be previously parsed with `jerry_parse`.

**Prototype**

```c
jerry_value_t
jerry_run (const jerry_value_t func_val);
```

- `func_val` - function to run
- return value
  - result of bytecode, if run was successful
  - thrown error, otherwise

**Example**

```c
{
  const jerry_char_t script[] = "print ('Hello, World!');";
  size_t script_size = strlen ((const char *) script);

  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  /* Setup Global scope code */
  jerry_value_t parsed_code = jerry_parse (script, script_size, false);

  if (!jerry_value_has_error_flag (parsed_code))
  {
    /* Execute the parsed source code in the Global scope */
    jerry_value_t ret_value = jerry_run (parsed_code);

    /* Returned value must be freed */
    jerry_release_value (ret_value);
  }

  /* Parsed source code must be freed */
  jerry_release_value (parsed_code);

  /* Cleanup engine */
  jerry_cleanup ();
}
```

**See also**

- [jerry_parse](#jerry_parse)


## jerry_eval

**Summary**

Perform JavaScript `eval`.

**Prototype**

```c
jerry_value_t
jerry_eval (const jerry_char_t *source_p,
            size_t source_size,
            bool is_strict);
```

- `source_p` - source code to evaluate, it must be a valid utf8 string.
- `source_size` - length of the source code
- `is_strict` - perform `eval` as it is called from "strict mode" code.
- return value - result of eval, may be error value.

**Example**

```c
{
  jerry_value_t ret_val = jerry_eval (str_to_eval,
                                      strlen (str_to_eval),
                                      false);
}
```

**See also**

- [jerry_create_external_function](#jerry_create_external_function)
- [jerry_external_handler_t](#jerry_external_handler_t)


# Get the global context

## jerry_get_global_object

**Summary**

Get the Global object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_global_object (void);
```

- return value - api value of global object

**Example**

```c
{
  jerry_value_t glob_obj_val = jerry_get_global_object ();

  ... // Do something with global object, ex: add properties

  jerry_release_value (glob_obj_val);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_define_own_property](#jerry_define_own_property)

# Checker functions

Functions to check the type of an API value ([jerry_value_t](#jerry_value_t)).

## jerry_value_is_array

**Summary**

Returns whether the given `jerry_value_t` is an array.

**Prototype**

```c
bool
jerry_value_is_array (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an array
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_array (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_boolean

**Summary**

Returns whether the given `jerry_value_t` is a boolean value.

**Prototype**

```c
bool
jerry_value_is_boolean (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a boolean value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_constructor

**Summary**

Returns whether the given `jerry_value_t` is a constructor function.

**Prototype**

```c
bool
jerry_value_is_constructor (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a constructor
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_constructor (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_function

**Summary**

Returns whether the given `jerry_value_t` is a function.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a function
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_function (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_number

**Summary**

Returns whether the given `jerry_value_t` is a number.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a number
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_null

**Summary**

Returns whether the given `jerry_value_t` is a null value.

**Prototype**

```c
bool
jerry_value_is_null (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a null
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_null (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_object

**Summary**

Returns whether the given `jerry_value_t` is an object value.

**Prototype**

```c
bool
jerry_value_is_object (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an object
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_object (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_string

**Summary**

Returns whether the given `jerry_value_t` is a string value.

**Prototype**

```c
bool
jerry_value_is_string (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a string
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_string (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_value_is_undefined

**Summary**

Returns whether the given `jerry_value_t` is an undefined value.

**Prototype**

```c
bool
jerry_value_is_undefined (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an undefined value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_undefined (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


# Error flag manipulation functions

## jerry_value_has_error_flag

**Summary**

Returns whether the given `jerry_value_t` has the error flag set.

**Prototype**

```c
bool
jerry_value_has_error_flag (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` has the error flag set
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_has_error_flag (value))
  {
    ...
  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


## jerry_value_clear_error_flag

**Summary**

Clear the error flag.

**Prototype**

```c
void
jerry_value_clear_error_flag (jerry_value_t *value_p);
```

- `value_p` - pointer to an api value

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_clear_error_flag (&value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


## jerry_value_set_error_flag

**Summary**

Set the error flag.

**Prototype**

```c
void
jerry_value_set_error_flag (jerry_value_t *value_p);
```

- `value_p` - pointer to an api value

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_set_error_flag (&value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)


# Getter functions of 'jerry_value_t'

Get raw data from API values.

## jerry_get_boolean_value

**Summary**

Gets the raw bool value from a `jerry_value_t`.

**Prototype**

```c
bool
jerry_get_boolean_value (const jerry_value_t value);
```

- `value` - api value
- return value - boolean value represented by the argument.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_boolean (value))
  {
    bool raw_value = jerry_get_boolean_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_is_boolean](#jerry_value_is_boolean)
- [jerry_release_value](#jerry_release_value)


## jerry_get_number_value

**Summary**

Gets the number value of the given `jerry_value_t` parameter as a raw double.

**Prototype**

```c
double
jerry_get_number_value (const jerry_value_t value);
```

- `value` - api value
- return value - the number value of the given `jerry_value_t` parameter as a raw double.

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  if (jerry_value_is_number (value))
  {
    double raw_value = jerry_get_number_value (value);

    ... // usage of raw value

  }

  jerry_release_value (value);
}
```

**See also**

- [jerry_value_is_number](#jerry_value_is_number)
- [jerry_release_value](#jerry_release_value)


# Functions for string values

## jerry_get_string_size

**Summary**

Get the size of a string. Returns zero, if the value parameter is not a string.

**Prototype**

```c
jerry_size_t
jerry_get_string_size (const jerry_value_t value);
```
- `value` - api value
- return value - number of bytes in the buffer needed to represent the string.

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_size_t string_size = jerry_get_string_size (string);

  ... // usage of string_size

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_length](#jerry_get_string_length)


## jerry_get_string_length

**Summary**

Get the length of a string. Returns zero, if the value parameter is not a string.

**Prototype**

```c
jerry_length_t
jerry_get_string_length (const jerry_value_t value);
```

- `value` - api value
- return value - number of characters in the string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string = jerry_create_string (char_array);

  jerry_length_t string_length = jerry_get_string_length (string);

  ... // usage of string_length

  jerry_release_value (string);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)


## jerry_string_to_char_buffer

**Summary**

Copy the characters of a string into a specified buffer. The
'\0' character could occur in character buffer. Returns 0,
if the value parameter is not a string or the buffer is
not large enough for the whole string.

**Prototype**

```c
jerry_size_t
jerry_string_to_char_buffer (const jerry_value_t value,
                             jerry_char_t *buffer_p,
                             jerry_size_t buffer_size);
```

- `value` - input string value
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes, actually copied to the buffer

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_size_t req_sz = jerry_get_string_size (value);
  jerry_char_t str_buf_p[req_sz];

  jerry_string_to_char_buffer (value, str_buf_p, req_sz);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_string](#jerry_create_string)
- [jerry_get_string_size](#jerry_get_string_size)


# Functions for array object values

## jerry_get_array_length

**Summary**

Get length of an array object. Returns zero, if the given parameter is not an array object.

**Prototype**

```c
uint32_t
jerry_get_array_length (const jerry_value_t value);
```

- `value` - input array value
- return value - length of the given array

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  uint32_t len = jerry_get_array_length (value);

  jerry_release_value (value);
}
```

**See also**

- [jerry_create_array](#jerry_create_array)


# Converters of 'jerry_value_t'

Functions for converting API values to another value type.

## jerry_value_to_boolean

**Summary**

Call ToBoolean operation on the api value.

**Prototype**

```c
bool
jerry_value_to_boolean (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the logical value is true
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  bool b = jerry_value_to_boolean (value);

  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_number

**Summary**

Call ToNumber operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_number (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted number value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t number_value = jerry_value_to_number (value);

  jerry_release_value (number_value);
  jerry_release_value (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_object

**Summary**

Call ToObject operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_object (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t object_value = jerry_value_to_object (value);

  jerry_release_value (object_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_primitive

**Summary**

Call ToPrimitive operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_primitive (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted primitive value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t prim_value = jerry_value_to_primitive (value);

  jerry_release_value (prim_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)

## jerry_value_to_string

**Summary**

Call the ToString ecma builtin operation on the api value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_string (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted string value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t string_value = jerry_value_to_string (value);

  jerry_release_value (string_value);
  jerry_release_value (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)


# Acquire and release API values

## jerry_acquire_value

**Summary**

Acquires the specified Jerry API value.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_acquire_value (jerry_value_t value);
```

- `value` - api value
- return value - acquired value that may be used outside of the engine

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  jerry_value_t acquired_object = jerry_acquire_value (object_value);

  jerry_release_value (object_value);

  // acquired_object refers to the created object and makes it
  // available after the release of 'object_value'

  jerry_release_value (acquired_object);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_value_t](#jerry_value_t)


## jerry_release_value

**Summary**

Release specified Jerry API value.

**Prototype**

```c
void
jerry_release_value (jerry_value_t value);
```

- `value` - api value

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ...

  jerry_release_value (object_value);
}
```


# Create API values

Function for creating [API values](#jerry_value_t).

*Note*: Every created API value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

## jerry_create_array

**Summary**

Create an array object value.

**Prototype**

```c
jerry_value_t
jerry_create_array (uint32_t size);
```

 - `size` - size of array;
 - return value - value of the constructed array object

 **Example**

```c
{
    jerry_value_t array = jerry_create_array (10);

    ...

    jerry_release_value (array);
}
```

**See also**

- [jerry_set_property_by_index](#jerry_set_property_by_index)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_create_boolean

**Summary**

Create a jerry_value_t representing a boolean value from the given boolean parameter.

**Prototype**

```c
jerry_value_t
jerry_create_boolean (bool value);
```

- `value` - raw boolean value.
- return value - a `jerry_value_t` created from the given boolean argument.

**Example**

```c
{
  jerry_value_t boolean_value = jerry_create_boolean (true);

  ... // usage of the value

  jerry_release_value (boolean_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_error

**Summary**

Create new JavaScript error object.

**Prototype**

```c
jerry_value_t
jerry_create_error (jerry_error_t error_type,
                    const jerry_char_t *message_p);
```

- `error_type` - type of error
- `message_p` - value of 'message' property of constructed error object
- return value - value of the constructed error object

**Example**

```c
{
  jerry_value_t error_obj = jerry_create_error (JERRY_ERROR_TYPE,
                                                (const jerry_char_t *) "error");

  ... // usage of error_obj


  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_value_has_error_flag](#jerry_value_has_error_flag)
- [jerry_value_clear_error_flag](#jerry_value_clear_error_flag)
- [jerry_value_set_error_flag](#jerry_value_set_error_flag)


## jerry_create_error_sz

**Summary**

Create new JavaScript error object.

**Prototype**

```c
jerry_value_t
jerry_create_error_sz (jerry_error_t error_type,
                       const jerry_char_t *message_p,
                       jerry_size_t message_size);
```

- `error_type` - type of the error
- `message_p` - value of 'message' property of the constructed error object
- `message_size` - size of the message in bytes
- return value - value of the constructed error object

**Example**

```c
{
  const jerry_char_t *message = "error";
  jerry_value_t error_obj = jerry_create_error_sz (JERRY_ERROR_COMMON,
                                                   message,
                                                   strlen ((const char *) message));

  ... // usage of error_obj

  jerry_release_value (error_obj);
}
```

**See also**

- [jerry_create_error](#jerry_create_error)


## jerry_create_external_function

**Summary**

Create an external function object.

**Prototype**

```c
jerry_value_t
jerry_create_external_function (jerry_external_handler_t handler_p);
```

- `handler_p` - pointer to native handler of the function object
- return value - value of the constructed function object

**Example**

```c
static jerry_value_t
handler (const jerry_value_t function_obj,
         const jerry_value_t this_val,
         const jerry_value_t args_p[],
         const jerry_length_t args_cnt)
{
  printf ("native handler called!\n");

  return jerry_create_boolean (true);
}

{
  jerry_value_t func_val = jerry_create_external_function (handler);
  jerry_value_t glob_obj = jerry_get_global_object ();

  // after this, script can invoke the native handler through "handler_field (1, 2, 3);"
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");
  jerry_set_property (glob_obj, prop_name, func_val);
  jerry_release_value (prop_name);

  jerry_release_value (func_val);
  jerry_release_value (glob_obj);
}
```

**See also**

- [jerry_external_handler_t](#jerry_external_handler_t)
- [jerry_set_property](#jerry_set_property)
- [jerry_call_function](#jerry_call_function)


## jerry_create_number

**Summary**

Creates a `jerry_value_t` representing a number value.

**Prototype**

```c
jerry_value_t
jerry_create_number (double value);
```

- `value` - double value from which a `jerry_value_t` will be created
- return value - a `jerry_value_t` created from the given double argument

**Example**

```c
{
  jerry_value_t number_value = jerry_create_number (3.14);

  ... // usage of the value

  jerry_release_value (number_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number_infinity](#jerry_create_number_infinity)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_infinity

**Summary**

Creates a `jerry_value_t` representing a positive or negative infinity value.

**Prototype**

```c
jerry_value_t
jerry_create_number_infinity (bool sign);
```

- `sign` - true for negative Infinity and false for positive Infinity
- return value - a `jerry_value_t` representing the infinity value

**Example**

```c
{
  jerry_value_t positive_inf_value = jerry_create_number_infinity (false);

  ... // usage of the positive_inf_value

  jerry_release_value (positive_inf_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_nan](#jerry_create_number_nan)


## jerry_create_number_nan

**Summary**

Creates a `jerry_value_t` representing a not-a-number value.

**Prototype**

```c
jerry_value_t
jerry_create_number_nan (void);
```

- return value - a `jerry_value_t` representing the not-a-number value

**Example**

```c
{
  jerry_value_t nan_value = jerry_create_number_nan ();

  ... // usage of the nan_value

  jerry_release_value (nan_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)
- [jerry_create_number](#jerry_create_number)
- [jerry_create_number_infinity](#jerry_create_number_infinity)


## jerry_create_null

**Summary**

Creates and returns a `jerry_value_t` with type null object.

**Prototype**

```c
jerry_value_t
jerry_create_null (void);
```

- return value - a `jerry_value_t` representing null.

**Example**

```c
{
  jerry_value_t null_value = jerry_create_null ();

  ... // usage of the value

  jerry_release_value (null_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_object

**Summary**

Create new JavaScript object, like with new Object().

**Prototype**

```c
jerry_value_t
jerry_create_object (void);
```

- return value - value of the created object

**Example**

```c
{
  jerry_value_t object_value = jerry_create_object ();

  ... // usage of object_value

  jerry_release_value (object_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


## jerry_create_string

**Summary**

Create string from a valid CESU8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string (const jerry_char_t *str_p);
```

- `str_p` - pointer to string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string (char_array);

  ... // usage of string_value

  jerry_release_value (string_value);
}
```

**See also**

- [jerry_create_string_sz](#jerry_create_string_sz)


## jerry_create_string_sz

**Summary**

Create string from a valid CESU8 string.

**Prototype**

```c
jerry_value_t
jerry_create_string_sz (const jerry_char_t *str_p,
                        jerry_size_t str_size)
```

- `str_p` - pointer to string
- `str_size` - size of the string
- return value - value of the created string

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value  = jerry_create_string_sz (char_array,
                                                        strlen ((const char *) char_array));

  ... // usage of string_value

  jerry_release_value (string_value);
}

```

**See also**

- [jerry_create_string](#jerry_create_string)


## jerry_create_undefined

**Summary**

Creates a `jerry_value_t` representing an undefined value.

**Prototype**

```c
jerry_value_t
jerry_create_undefined (void);
```

- return value - value of undefined

**Example**

```c
{
  jerry_value_t undefined_value = jerry_create_undefined ();

  ... // usage of the value

  jerry_release_value (undefined_value);
}
```

**See also**

- [jerry_release_value](#jerry_release_value)


# General API functions of JS objects

## jerry_has_property

**Summary**

Checks whether the object or it's prototype objects have the given property.

**Prototype**

```c
bool
jerry_has_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if the property exists
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  bool has_prop = jerry_has_property (global_object, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_has_own_property

**Summary**

Checks whether the object has the given property.

**Prototype**

```c
bool
jerry_has_own_property (const jerry_value_t obj_val,
                        const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if the property exists
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "handler_field");

  bool has_prop = jerry_has_own_property (global_object, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_delete_property](#jerry_delete_property)


## jerry_delete_property

**Summary**

Delete a property from an object.

**Prototype**

```c
bool
jerry_delete_property (const jerry_value_t obj_val,
                       const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - false, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_delete_property (global_object, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_get_property](#jerry_get_property)


## jerry_get_property

**Summary**

Get value of a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t global_object = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_value_t prop_value = jerry_get_property (obj_val, prop_name);

  jerry_release_value (prop_name);
  jerry_release_value (global_object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_get_property_by_index

**Summary**

Get value by an index from the specified object.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_get_property_by_index (const jerry_value_t obj_val,
                             uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - stored value on the specified index, if success
  - thrown exception, otherwise.

**Example**

```c
{
  jerry_value_t object;

  ... // create or acquire object

  jerry_value_t value = jerry_get_property_by_index (object, 5);

  ...

  jerry_release_value (value);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_set_property

**Summary**

Set a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property (const jerry_value_t obj_val,
                    const jerry_value_t prop_name_val,
                    const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t value_to_set;

  ... // create or acquire value to set

  jerry_value_t glob_obj = jerry_get_global_object ();
  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");

  jerry_set_property (glob_obj, prop_name, value_to_set);

  jerry_release_value (prop_name);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (glob_obj);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)
- [jerry_set_property_by_index](#jerry_set_property_by_index)


## jerry_set_property_by_index

**Summary**

Set indexed value in the specified object

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_set_property_by_index (const jerry_value_t obj_val,
                             uint32_t index,
                             const jerry_value_t value_to_set);
```

- `obj_val` - object value
- `index` - index number
- `value_to_set` - value to set
- return value
  - true, if field value was set successfully
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t value_to_set;

  ... // create or acquire object and value to set

  jerry_value_t ret_val = jerry_set_property_by_index (object, 5, value_to_set);

  ...

  jerry_release_value (value_to_set);
  jerry_release_value (ret_val);
  jerry_release_value (object);
}
```

**See also**

- [jerry_has_property](#jerry_has_property)
- [jerry_has_own_property](#jerry_has_own_property)
- [jerry_delete_property](#jerry_delete_property)
- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)
- [jerry_get_property_by_index](#jerry_get_property_by_index)


## jerry_init_property_descriptor_fields

**Summary**

Initialize property descriptor.

**Prototype**

```c
void
jerry_init_property_descriptor_fields (jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of prop_desc

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

**See also**

- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_define_own_property

**Summary**

Define a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_define_own_property (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val,
                           const jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_get_global_object ();

  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t value_to_set;

  ... // create or acquire value to set

  prop_desc.is_value_defined = true;
  prop_desc.value = value_to_set;

  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_define_own_property (global_obj_val, prop_name, &prop_desc);
  jerry_release_value (prop_name);

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_get_own_property_descriptor

**Summary**

Construct property descriptor from specified property.

**Prototype**

```c
bool
jerry_get_own_property_descriptor (const jerry_value_t  obj_val,
                                   const jerry_value_t prop_name_val,
                                   jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_get_global_object ();

  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  jerry_value_t prop_name = jerry_create_string ((const jerry_char_t *) "my_prop");
  jerry_get_own_property_descriptor (global_obj_val, prop_name, &prop_desc);
  jerry_release_value (prop_name);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
  jerry_release_value (global_obj_val);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_free_property_descriptor_fields](#jerry_free_property_descriptor_fields)


## jerry_free_property_descriptor_fields

**Summary**

Free fields of property descriptor (setter, getter and value).

**Prototype**

```c
void
jerry_free_property_descriptor_fields (const jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

**Example**

```c
{
  jerry_property_descriptor_t prop_desc;
  jerry_init_property_descriptor_fields (&prop_desc);

  ... // usage of property descriptor

  jerry_free_property_descriptor_fields (&prop_desc);
}
```

**See also**

- [jerry_init_property_descriptor_fields](#jerry_init_property_descriptor_fields)
- [jerry_define_own_property](#jerry_define_own_property)
- [jerry_get_own_property_descriptor](#jerry_get_own_property_descriptor)


## jerry_call_function

**Summary**

Call function specified by a function value. Error flag
must not be set for any arguments of this function.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_call_function (const jerry_value_t func_obj_val,
                     const jerry_value_t this_val,
                     const jerry_value_t args_p[],
                     jerry_size_t args_count);
```

- `func_obj_val` - the function object to call
- `this_val` - object for 'this' binding
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned jerry value of the called function

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_value_is_function (val))
  {
    jerry_value_t this_val = jerry_create_undefined ();
    jerry_value_t ret_val = jerry_call_function (val, this_val, NULL, 0);

    if (!jerry_value_has_error_flag (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
    jerry_release_value (this_val);
  }
}
```

**See also**

- [jerry_is_function](#jerry_is_function)
- [jerry_create_external_function](#jerry_create_external_function)


## jerry_construct_object

**Summary**

Construct object, invoking specified function object as constructor.
Error flag must not be set for any arguments of this function.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_construct_object (const jerry_value_t func_obj_val,
                        const jerry_value_t args_p[],
                        jerry_size_t args_count);
```

- `func_obj_val` - function object to call
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned value of the invoked constructor

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_is_constructor (val))
  {
    jerry_value_t ret_val = jerry_construct_object (val, NULL, 0);

    if (!jerry_value_has_error_flag (ret_val))
    {
      ... // handle return value
    }

    jerry_release_value (ret_val);
  }
}
```

**See also**

 - [jerry_is_constructor](#jerry_is_constructor)


## jerry_get_object_keys

**Summary**

Get keys of the specified object value.

**Prototype**

```c
jerry_value_t
jerry_get_object_keys (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - array object value, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t keys_array = jerry_get_object_keys (object);

  ... // usage of keys_array

  jerry_release_value (keys_array);
}
```

**See also**

- [jerry_get_property](#jerry_get_property)
- [jerry_set_property](#jerry_set_property)


## jerry_get_prototype

**Summary**

Get the prototype of the specified object.

**Prototype**

```c
jerry_value_t
jerry_get_prototype (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - object value, if success
  - null or thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  ... // create or acquire object

  jerry_value_t prototype = jerry_get_prototype (object);

  ... // usage of prototype object

  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_set_prototype](#jerry_set_prototype)


## jerry_set_prototype

**Summary**

Set the prototype of the specified object.

**Prototype**

```c
jerry_value_t
jerry_get_prototype (const jerry_value_t obj_val,
                     const jerry_value_t proto_obj_val);
```

- `obj_val` - object value
- `proto_obj_val` - prototype object value
- return value
  - true, if success
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t prototype;

  ... // create or acquire object and prototype

  jerry_value_t ret_val = jerry_set_prototype (object, prototype);

  jerry_release_value (ret_val);
  jerry_release_value (prototype);
  jerry_release_value (object);
}
```

**See also**

- [jerry_get_prototype](#jerry_get_prototype)


## jerry_get_object_native_handle

**Summary**

Get native handle, previously associated with specified object.

**Prototype**

```c
bool
jerry_get_object_native_handle (const jerry_value_t obj_val,
                                uintptr_t *out_handle_p);
```

- `obj_val` - object value
- `out_handle_p` - handle value (output parameter).
- return value
  - true, if there is handle associated with the object
  - false, otherwise

**Example**

```c
{
  jerry_value_t object;
  uintptr_t handle_set;

  ... // receive or construct object and handle_set value

  jerry_set_object_native_handle (object, handle_set, NULL);

  ...

  uintptr_t handle_get;
  bool is_there_associated_handle = jerry_get_object_native_handle (object, &handle_get);
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_set_object_native_handle](#jerry_set_object_native_handle)

## jerry_set_object_native_handle

**Summary**

Set native handle and an optional free callback for the specified object

*Note*: If native handle was already set for the object, its value is updated.

*Note*: If a non-NULL free callback is specified, it will be called
        by the garbage collector when the object is freed. The free
        callback always overwrites the previous value, so passing
        a NULL value deletes the current free callback.

**Prototype**

```c
void
jerry_set_object_native_handle (const jerry_value_t obj_val,
                                uintptr_t handle_p,
                                jerry_object_free_callback_t freecb_p);
```

- `obj_val` - object value to set handle in
- `handle_p` - handle value
- `freecb_p` - pointer to "free" callback or NULL

**Example**

```c
{
  jerry_value_t object;
  uintptr_t handle_set;

  ... // receive or construct object and handle_set value

  jerry_set_object_native_handle (object, handle_set, NULL);

  ...

  uintptr_t handle_get;
  bool is_there_associated_handle = jerry_get_object_native_handle (object, &handle_get);
}
```

**See also**

- [jerry_create_object](#jerry_create_object)
- [jerry_get_object_native_handle](#jerry_get_object_native_handle)


## jerry_foreach_object_property

**Summary**

Applies the given function to every property in the given object.

**Prototype**

```c
bool
jerry_foreach_object_property (jerry_value_t obj_val,
                               jerry_object_property_foreach_t foreach_p,
                               void *user_data_p);
```

- `obj_val` - object value
- `foreach_p` - foreach function, that will be applied for each property
- `user_data_p` - user data for foreach function
- return value
  - true, if object fields traversal was performed successfully, i.e.:
    - no unhandled exceptions were thrown in object fields traversal
    - object fields traversal was stopped on callback that returned false
  - false, otherwise

**Example**

```c
bool foreach_function (const jerry_value_t prop_name,
                       const jerry_value_t prop_value,
                       void *user_data_p)
{

  ... // implementation of the foreach function

}

{
  jerry_value_t object;
  ... // receive or construct object

  double data = 3.14; // example data

  jerry_foreach_object_property (object, foreach_function, &data);

}
```

**See also**

- [jerry_object_property_foreach_t](#jerry_object_property_foreach_t)


# Snapshot functions

## jerry_parse_and_save_snapshot

**Summary**

Generate snapshot from the specified source code.

**Prototype**

```c
size_t
jerry_parse_and_save_snapshot (const jerry_char_t *source_p,
                               size_t source_size,
                               bool is_for_global,
                               bool is_strict,
                               uint8_t *buffer_p,
                               size_t buffer_size);
```

- `source_p` - script source, it must be a valid utf8 string.
- `source_size` - script source size, in bytes.
- `is_for_global` - snapshot would be executed as global (true) or eval (false).
- `is_strict` - strict mode
- `buffer_p` - buffer to save snapshot to.
- `buffer_size` - the buffer's size.
- return value
  - the size of snapshot, if it was generated succesfully (i.e. there are no syntax errors in source
    code, buffer size is sufficient, and snapshot support is enabled in current configuration through
    JERRY_ENABLE_SNAPSHOT_SAVE)
  - 0 otherwise.

**Example**

```c
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint8_t global_mode_snapshot_buffer[1024];
  const jerry_char_t *code_to_snapshot_p = "(function () { return 'string from snapshot'; }) ();";

  size_t global_mode_snapshot_size = jerry_parse_and_save_snapshot (code_to_snapshot_p,
                                                                    strlen ((const char *) code_to_snapshot_p),
                                                                    true,
                                                                    false,
                                                                    global_mode_snapshot_buffer,
                                                                    sizeof (global_mode_snapshot_buffer));

  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_exec_snapshot](#jerry_exec_snapshot)


## jerry_exec_snapshot

**Summary**

Execute snapshot from the specified buffer.

*Note*: Returned value must be freed with [jerry_release_value](#jerry_release_value) when it
is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_exec_snapshot (const void *snapshot_p,
                     size_t snapshot_size,
                     bool copy_bytecode);
```

- `snapshot_p` - pointer to snapshot
- `snapshot_size` - size of snapshot
- `copy_bytecode` - flag, indicating whether the passed snapshot buffer should be copied to the
   engine's memory. If set the engine should not reference the buffer after the function returns
   (in this case, the passed buffer could be freed after the call). Otherwise (if the flag is not
   set) - the buffer could only be freed after the engine stops (i.e. after call to jerry_cleanup).
- return value
  - result of bytecode, if run was successful
  - thrown error, otherwise

**Example**

```c
{
  jerry_value_t res;
  static uint8_t global_mode_snapshot_buffer[1024];
  const jerry_char_t *code_to_snapshot_p = "(function () { return 'string from snapshot'; }) ();";

  jerry_init (JERRY_INIT_EMPTY);
  size_t global_mode_snapshot_size = jerry_parse_and_save_snapshot (code_to_snapshot_p,
                                                                    strlen ((const char *) code_to_snapshot_p),
                                                                    true,
                                                                    global_mode_snapshot_buffer,
                                                                    sizeof (global_mode_snapshot_buffer));
  jerry_cleanup ();

  jerry_init (JERRY_INIT_EMPTY);

  res = (jerry_exec_snapshot (global_mode_snapshot_buffer,
                              global_mode_snapshot_size,
                              false);

  jerry_cleanup ();
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_parse_and_save_snapshot](#jerry_parse_and_save_snapshot)
