/* SPDX-FileCopyrightText: 2024 Google LLC */
/* SPDX-License-Identifier: Apache-2.0 */

#pragma once

#include <stddef.h>
#include <stdint.h>

//! @file applib_resource.h
//! Wrapper functions for the resource syscalls. These functions give us a nice interface to export to 3rd party apps.

//! @addtogroup Foundation
//! @{
//!   @addtogroup Resources
//! \brief Managing application resources
//!
//! Resources are data files that are bundled with your application binary and can be
//! loaded at runtime. You use resources to embed images or custom fonts in your app,
//! but also to embed any data file. Resources are always read-only.
//!
//! Resources are stored on Pebble’s flash memory and only loaded in RAM when you load
//! them. This means that you can have a large number of resources embedded inside your app,
//! even though Pebble’s RAM memory is very limited.
//!
//! See \htmlinclude UsingResources.html for information on how to embed
//! resources into your app's bundle.

//!
//!   @{

#define RESOURCE_ID_FONT_FALLBACK RESOURCE_ID_GOTHIC_14

//! Opaque reference to a resource.
//! @see \ref resource_get_handle()
typedef void * ResHandle;

//! Gets the resource handle for a file identifier.
//! @param resource_id The resource ID
//!
//! The resource IDs are auto-generated by the Pebble build process, based
//! on the `appinfo.json`. The "name" field of each resource is prefixed
//! by `RESOURCE_ID_` and made visible to the application (through the
//! `build/src/resource_ids.auto.h` header which is automatically included).
//!
//! For example, given the following fragment of `appinfo.json`:
//! \code{.json}
//!   ...
//!   "resources" : {
//!     "media": [
//!        {
//!           "name": "MY_ICON",
//!           "file": "img/icon.png",
//!           "type": "png",
//!        },
//!    ...
//! \endcode
//! The generated file identifier for this resource is `RESOURCE_ID_MY_ICON`.
//! To get a resource handle for that resource write:
//! \code{.c}
//! ResHandle rh = resource_get_handle(RESOURCE_ID_MY_ICON);
//! \endcode
ResHandle applib_resource_get_handle(uint32_t resource_id);

//! Gets the size of the resource given a resource handle.
//! @param h The handle to the resource
//! @return The size of the resource in bytes
size_t applib_resource_size(ResHandle h);

//! Copies the bytes for the resource with a given handle from flash storage into a given buffer.
//! @param h The handle to the resource
//! @param buffer The buffer to load the resource data into
//! @param max_length The maximum number of bytes to copy
//! @return The number of bytes actually copied
size_t applib_resource_load(ResHandle h, uint8_t *buffer, size_t max_length);

//! Copies a range of bytes from a resource with a given handle into a given buffer.
//! @param h The handle to the resource
//! @param start_offset The offset in bytes at which to start reading from the resource
//! @param buffer The buffer to load the resource data into
//! @param num_bytes The maximum number of bytes to copy
//! @return The number of bytes actually copied
size_t applib_resource_load_byte_range(
    ResHandle h, uint32_t start_offset, uint8_t *buffer, size_t num_bytes);

//!   @} // end addtogroup Resources
//! @} // end addtogroup Foundation

