package net.corda.libs.packaging.internal.v2

import net.corda.crypto.core.SecureHashImpl
import net.corda.crypto.core.bytes
import net.corda.libs.packaging.Cpi
import net.corda.libs.packaging.Cpk
import net.corda.libs.packaging.CpkReader
import net.corda.libs.packaging.PackagingConstants
import net.corda.libs.packaging.PackagingConstants.CPI_GROUP_POLICY_ENTRY
import net.corda.libs.packaging.core.CpiIdentifier
import net.corda.libs.packaging.core.CpiMetadata
import net.corda.libs.packaging.core.exception.PackagingException
import net.corda.libs.packaging.hash
import net.corda.libs.packaging.internal.CpiImpl
import net.corda.libs.packaging.internal.CpiLoader
import net.corda.libs.packaging.signerSummaryHash
import net.corda.utilities.time.Clock
import net.corda.utilities.time.UTCClock
import net.corda.v5.crypto.DigestAlgorithmName
import java.io.InputStream
import java.nio.file.Path
import java.nio.file.Paths
import java.util.jar.JarInputStream

class CpiLoaderV2(private val activeCordaPlatformVersion: Int, private val clock: Clock = UTCClock()) : CpiLoader {

    override fun loadCpi(
        byteArray: ByteArray,
        expansionLocation: Path,
        cpiLocation: String?,
        verifySignature: Boolean,
    ): Cpi {

        // Calculate file hash
        val hash = calculateHash(byteArray)

        // Read CPI
        JarInputStream(byteArray.inputStream()).use { jarInputStream ->

            val cpiEntries = readJar(jarInputStream)

            val groupPolicy = cpiEntries.single { it.entry.name.endsWith(CPI_GROUP_POLICY_ENTRY) }

            val cpb =
                cpiEntries
                    .filter { it.entry.name.endsWith(".cpb") }
                    .run {
                        when (this.size) {
                            0 -> null
                            1 -> this[0]
                            else -> throw PackagingException("Multiple CPBs found in CPI.")
                        }
                    }

            // Read CPB
            val cpks = if (cpb != null) {
                readCpksFromCpb(cpb.bytes.inputStream(), expansionLocation, cpiLocation).toList()
            } else {
                emptyList()
            }

            cpks.forEach {
                val minPlatformVersion = it.metadata.cordappManifest.minPlatformVersion
                if (activeCordaPlatformVersion < minPlatformVersion) {
                    throw PackagingException("Platform version of Corda is lower than minimum platform version of CPK" +
                            " ${it.metadata.cpkId.name} ($activeCordaPlatformVersion < $minPlatformVersion)")
                }
            }

            val mainAttributes = jarInputStream.manifest.mainAttributes
            return CpiImpl(
                CpiMetadata(
                    cpiId = CpiIdentifier(
                        mainAttributes.getValue(PackagingConstants.CPI_NAME_ATTRIBUTE)
                            ?: throw PackagingException("CPI name missing from manifest"),
                        mainAttributes.getValue(PackagingConstants.CPI_VERSION_ATTRIBUTE)
                            ?: throw PackagingException("CPI version missing from manifest"),
                        groupPolicy.entry.certificates.asSequence().signerSummaryHash()
                    ),
                    fileChecksum = SecureHashImpl(DigestAlgorithmName.SHA2_256.name, hash),
                    cpksMetadata = cpks.map { it.metadata },
                    groupPolicy = String(groupPolicy.bytes),
                    timestamp = clock.instant()
                ),
                cpks
            )
        }
    }

    private fun calculateHash(cpiBytes: ByteArray) = cpiBytes.hash(DigestAlgorithmName.SHA2_256).bytes

    private fun readCpksFromCpb(cpb: InputStream, expansionLocation: Path, cpiLocation: String?): List<Cpk> {
        val cpkCordappNames = hashSetOf<String>()

        return JarInputStream(cpb, false).use { cpbInputStream ->
            readJar(cpbInputStream)
                .filter { it.entry.name.endsWith(".jar") }
                .map {
                    CpkReader.readCpk(
                        it.bytes.inputStream(),
                        expansionLocation,
                        cpkLocation = cpiLocation.plus("/${it.entry.name}"),
                        verifySignature = false,
                        cpkFileName = Paths.get(it.entry.name).fileName.toString()
                    ).also {
                        val cpkCordappName = it.metadata.cpkId.name
                        if (cpkCordappNames.contains(cpkCordappName)) {
                            throw PackagingException("Multiple CPKs share the Corda-CPK-Cordapp-Name $cpkCordappName.")
                        }
                        cpkCordappNames.add(cpkCordappName)
                    }
                }
        }
    }
}
