package net.corda.flow.utils

import net.corda.data.KeyValuePair
import net.corda.data.KeyValuePairList

fun mutableKeyValuePairList() = KeyValuePairList(mutableListOf())

fun emptyKeyValuePairList() = KeyValuePairList(emptyList())

/**
 * Creates a mutable [KeyValuePairList] (this is a [KeyValuePairList] backed by a mutable list) out of another
 * [KeyValuePairList]. This method can be used to create a new [KeyValuePairList] with a new backing list which contains
 * copies of the list passed to it.
 */
fun mutableKeyValuePairListOf(initialProperties: KeyValuePairList) = mutableKeyValuePairList().apply {
    items.addAll(initialProperties.items)
}

/**
 * Creates a [KeyValueStore] from a variable number of pairs of strings.
 * @param pairs Pairs of strings, the first is considered the key, the second the value.
 * @return A [KeyValueStore] containing the keys and values from the pairs.
 */
fun keyValueStoreOf(vararg pairs: Pair<String, String>) = KeyValueStore().apply {
    pairs.forEach {
        put(it.first, it.second)
    }
}

/**
 * Creates an avro generated [KeyValuePairList] from a Kotlin Map
 * @param map The Kotlin map
 * @return An avro [KeyValuePairList]
 */
fun keyValuePairListOf(map: Map<String, String>) = mutableKeyValuePairList().apply {
    map.entries.forEach {
        items.add(KeyValuePair(it.key, it.value))
    }
}

/**
 * Transforms [KeyValuePairList] into map.
 */
fun KeyValuePairList.toMap() =
    if (items != null) items.associate { it.key to it.value } else emptyMap()

/**
 * Transforms [KeyValuePairList] into a mutable map.
 */
fun KeyValuePairList.toMutableMap() =
    if (items != null) items.associateTo(mutableMapOf()) { it.key to it.value } else mutableMapOf()

/**
 * Creates a [KeyValuePairList] from an iterable container of objects of type [T] with keys
 * generated by concatenating the [prefix] string (or the simple name of the class of [T] if [prefix] is null),
 * and the index of the element in the iterable. Note that null values are filtered out silently.
 *
 * @param prefix The prefix to use for the generated keys. If null, the simple name of the class of [T] is used.
 * @return A [KeyValuePairList] containing the generated key-value pairs.
 */
inline fun <reified T> Iterable<T>.toKeyValuePairList(prefix: String? = null): KeyValuePairList {
    val keyPrefix = prefix ?: T::class.java.simpleName
    return KeyValuePairList(
        this.filterNotNull()
            .mapIndexed { idx, value ->
                KeyValuePair(listOf(keyPrefix, idx).joinToString("."), value.toString())
            }
    )
}

/**
 * A KeyValueStore which operates much like a map from the user perspective. Internally it is backed by an Avro array
 * which means serialization and deserialization are guaranteed to produce the same object, which is not the case with
 * Avro's own built in map type.
 * This functionality cannot be applied directly to the Avro array via extension functions, because Avro reserves some
 * of the standard map named methods for its own purposes.
 *
 * @param backingList The Avro array which backs this [KeyValueStore]. The [KeyValueStore] class itself is stateless,
 * all changes to the store are reflected directly in the array.
 */
class KeyValueStore(private val backingList: KeyValuePairList = mutableKeyValuePairList()) {

    private fun KeyValuePairList.setValue(key: String, value: String) {
        for (item in items) {
            if (item.key == key) {
                item.value = value
                return
            }
        }
        items.add(KeyValuePair(key, value))
    }

    operator fun set(key: String, value: String) = backingList.setValue(key, value)
    fun put(key: String, value: String) = set(key, value)
    operator fun get(key: String) = backingList.items.find { it.key == key }?.value

    /**
     * Importantly, this property exposes the mutable Avro array directly, no conversion is carried out.
     */
    val avro
        get() = backingList
}
