package net.corda.db.persistence.testkit.components

import net.corda.db.admin.LiquibaseSchemaMigrator
import net.corda.db.admin.impl.ClassloaderChangeLog
import net.corda.db.schema.DbSchema
import net.corda.persistence.common.EntitySandboxService
import net.corda.sandboxgroupcontext.service.SandboxGroupContextComponent
import net.corda.test.util.identity.createTestHoldingIdentity
import net.corda.testing.sandboxes.VirtualNodeLoader
import net.corda.testing.sandboxes.testkit.RequireSandboxTestkit
import net.corda.virtualnode.VirtualNodeInfo
import org.osgi.service.component.annotations.Activate
import org.osgi.service.component.annotations.Component
import org.osgi.service.component.annotations.Reference
import org.slf4j.LoggerFactory
import java.util.UUID
import java.util.concurrent.atomic.AtomicInteger

@RequireSandboxTestkit
@Component(service = [ VirtualNodeService::class ])
class VirtualNodeService @Activate constructor(
    @Reference
    private val virtualNodeLoader: VirtualNodeLoader,

    @Reference
    private val dataSourceAdmin: DataSourceAdmin,

    @Reference
    private val liquibaseSchemaMigrator: LiquibaseSchemaMigrator,

    @Reference
    val entitySandboxService: EntitySandboxService,

    @Reference
    val sandboxGroupContextComponent: SandboxGroupContextComponent
) {
    private companion object {
        private const val X500_NAME = "CN=Testing, OU=Application, O=R3, L=London, C=GB"

        fun generateHoldingIdentity() = createTestHoldingIdentity(X500_NAME, UUID.randomUUID().toString())
    }
    private val logger = LoggerFactory.getLogger(this::class.java)

    private var connectionCounter = AtomicInteger(0)

    init {
        sandboxGroupContextComponent.resizeCaches(2)
    }

    fun load(resourceName: String): VirtualNodeInfo {
        val virtualNodeInfo = virtualNodeLoader.loadVirtualNode(resourceName, generateHoldingIdentity())
        val dbConnectionId = virtualNodeInfo.vaultDmlConnectionId

        // migrate DB schema
        val vaultSchema = ClassloaderChangeLog(linkedSetOf(
            ClassloaderChangeLog.ChangeLogResourceFiles(
                DbSchema::class.java.packageName,
                listOf("net/corda/db/schema/vnode-vault/db.changelog-master.xml"),
                DbSchema::class.java.classLoader
            )
        ))

        val name = "connection-${connectionCounter.incrementAndGet()}"
        val dataSource = dataSourceAdmin.getOrCreateDataSource(dbConnectionId, name)
        dataSource.connection.use { connection ->
            logger.info("Migrate DB schema for $name: ${connection.catalog}")
            liquibaseSchemaMigrator.updateDb(connection, vaultSchema, dataSourceAdmin.createSchemaName(dbConnectionId, name))
        }
        return virtualNodeInfo
    }
}
