package net.corda.processors.p2p.linkmanager.internal

import com.typesafe.config.ConfigValueFactory
import net.corda.configuration.read.ConfigurationReadService
import net.corda.cpiinfo.read.CpiInfoReadService
import net.corda.crypto.client.CryptoOpsClient
import net.corda.crypto.client.SessionEncryptionOpsClient
import net.corda.libs.configuration.SmartConfig
import net.corda.libs.configuration.merger.ConfigMerger
import net.corda.libs.statemanager.api.StateManager
import net.corda.libs.statemanager.api.StateManagerFactory
import net.corda.lifecycle.LifecycleCoordinator
import net.corda.lifecycle.LifecycleCoordinatorFactory
import net.corda.lifecycle.LifecycleCoordinatorName
import net.corda.lifecycle.LifecycleEvent
import net.corda.lifecycle.RegistrationHandle
import net.corda.lifecycle.RegistrationStatusChangeEvent
import net.corda.lifecycle.StartEvent
import net.corda.lifecycle.StopEvent
import net.corda.lifecycle.createCoordinator
import net.corda.membership.grouppolicy.GroupPolicyProvider
import net.corda.membership.persistence.client.MembershipQueryClient
import net.corda.membership.read.GroupParametersReaderService
import net.corda.membership.read.MembershipGroupReaderProvider
import net.corda.messaging.api.publisher.factory.PublisherFactory
import net.corda.messaging.api.subscription.factory.SubscriptionFactory
import net.corda.p2p.linkmanager.LinkManager
import net.corda.processors.p2p.linkmanager.LinkManagerProcessor
import net.corda.schema.configuration.BootConfig
import net.corda.schema.configuration.MessagingConfig.Subscription.POLL_TIMEOUT
import net.corda.schema.configuration.StateManagerConfig
import net.corda.schema.registry.AvroSchemaRegistry
import net.corda.utilities.debug
import net.corda.virtualnode.read.VirtualNodeInfoReadService
import org.bouncycastle.jce.provider.BouncyCastleProvider
import org.osgi.service.component.annotations.Activate
import org.osgi.service.component.annotations.Component
import org.osgi.service.component.annotations.Reference
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import java.security.Security

@Suppress("LongParameterList", "Unused")
@Component(service = [LinkManagerProcessor::class])
class LinkManagerProcessorImpl @Activate constructor(
    @Reference(service = ConfigMerger::class)
    private val configMerger: ConfigMerger,
    @Reference(service = ConfigurationReadService::class)
    private val configurationReadService: ConfigurationReadService,
    @Reference(service = LifecycleCoordinatorFactory::class)
    private val coordinatorFactory: LifecycleCoordinatorFactory,
    @Reference(service = PublisherFactory::class)
    private val publisherFactory: PublisherFactory,
    @Reference(service = SubscriptionFactory::class)
    private val subscriptionFactory: SubscriptionFactory,
    @Reference(service = GroupPolicyProvider::class)
    private val groupPolicyProvider: GroupPolicyProvider,
    @Reference(service = VirtualNodeInfoReadService::class)
    private val virtualNodeInfoReadService: VirtualNodeInfoReadService,
    @Reference(service = CpiInfoReadService::class)
    private val cpiInfoReadService: CpiInfoReadService,
    @Reference(service = CryptoOpsClient::class)
    private val cryptoOpsClient: CryptoOpsClient,
    @Reference(service = MembershipGroupReaderProvider::class)
    private val membershipGroupReaderProvider: MembershipGroupReaderProvider,
    @Reference(service = MembershipQueryClient::class)
    private val membershipQueryClient: MembershipQueryClient,
    @Reference(service = GroupParametersReaderService::class)
    private val groupParametersReaderService: GroupParametersReaderService,
    @Reference(service = AvroSchemaRegistry::class)
    private val avroSchemaRegistry: AvroSchemaRegistry,
    @Reference(service = StateManagerFactory::class)
    private val stateManagerFactory: StateManagerFactory,
    @Reference(service = SessionEncryptionOpsClient::class)
    private val sessionEncryptionOpsClient: SessionEncryptionOpsClient,
) : LinkManagerProcessor {

    private companion object {
        val log: Logger = LoggerFactory.getLogger(this::class.java.enclosingClass)
    }

    private var registration: RegistrationHandle? = null
    private var linkManager: LinkManager? = null
    private var stateManager: StateManager? = null

    private val lifecycleCoordinator = coordinatorFactory.createCoordinator<LinkManagerProcessorImpl>(::eventHandler)

    override fun start(bootConfig: SmartConfig) {
        log.info("Link manager processor starting.")
        lifecycleCoordinator.start()
        lifecycleCoordinator.postEvent(BootConfigEvent(bootConfig))
    }

    override fun stop() {
        log.info("Link manager processor stopping.")
        lifecycleCoordinator.stop()
    }

    private fun eventHandler(event: LifecycleEvent, coordinator: LifecycleCoordinator) {
        log.debug { "Link manager received event $event." }

        when (event) {
            is StartEvent -> {
                configurationReadService.start()
            }
            is RegistrationStatusChangeEvent -> {
                log.info("Link manager processor is ${event.status}")
                coordinator.updateStatus(event.status)
            }
            is BootConfigEvent -> {
                configurationReadService.bootstrapConfig(event.config)

                val localStateManager = stateManagerFactory.create(
                    event.config.getConfig(BootConfig.BOOT_STATE_MANAGER),
                    StateManagerConfig.StateType.P2P_SESSION
                ).also { it.start() }

                log.info("StateManager ${localStateManager.name} has been created and started.")

                Security.addProvider(BouncyCastleProvider())

                val linkManager = LinkManager(
                    subscriptionFactory,
                    publisherFactory,
                    coordinatorFactory,
                    configurationReadService,
                    configMerger.getMessagingConfig(event.config),
                    groupPolicyProvider,
                    virtualNodeInfoReadService,
                    cpiInfoReadService,
                    cryptoOpsClient,
                    membershipGroupReaderProvider,
                    membershipQueryClient,
                    groupParametersReaderService,
                    localStateManager,
                    sessionEncryptionOpsClient,
                    avroSchemaRegistry
                )

                stateManager = localStateManager
                this.linkManager = linkManager

                registration?.close()
                registration = lifecycleCoordinator.followStatusChangesByName(
                    setOf(
                        LifecycleCoordinatorName.forComponent<ConfigurationReadService>(),
                        linkManager.dominoTile.coordinatorName,
                    ),
                )

                linkManager.start()
            }
            is StopEvent -> {
                linkManager?.stop()
                linkManager = null
                stateManager?.stop()
                stateManager = null
                registration?.close()
                registration = null
            }
        }
    }

    private fun getMessagingConfig(bootConfig: SmartConfig): SmartConfig {
        return configMerger.getMessagingConfig(bootConfig).withValue(
            // The default value of poll timeout is quite high (6 seconds), so setting it to something lower.
            // Specifically, state & event subscriptions have an issue where they are polling with high timeout on events topic,
            // leading to slow syncing upon startup. See: https://r3-cev.atlassian.net/browse/CORE-3163
            POLL_TIMEOUT,
            ConfigValueFactory.fromAnyRef(100),
        )
    }
}

data class BootConfigEvent(val config: SmartConfig) : LifecycleEvent
