/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.mapred;

import java.io.IOException;
import java.net.InetSocketAddress;

import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.mapreduce.MRConfig;
import org.apache.hadoop.mapreduce.protocol.ClientProtocol;
import org.apache.hadoop.mapreduce.protocol.ClientProtocolProvider;
import org.apache.hadoop.mapreduce.server.jobtracker.JTConfig;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.UserGroupInformation;

@InterfaceAudience.Private
public class JobTrackerClientProtocolProvider extends ClientProtocolProvider {

  @Override
  public ClientProtocol create(Configuration conf) throws IOException {
    String framework = conf.get(MRConfig.FRAMEWORK_NAME);
    if (!MRConfig.CLASSIC_FRAMEWORK_NAME.equals(framework)) {
      return null;
    }
    String tracker = conf.get(JTConfig.JT_IPC_ADDRESS, "local");
    if (!"local".equals(tracker)) {
      return createRPCProxy(JobTracker.getAddress(conf), conf);
    } else {
      throw new IOException("Invalid \"" + JTConfig.JT_IPC_ADDRESS
          + "\" configuration value for JobTracker: \""
          + tracker + "\"");
    }
  }

  @Override
  public ClientProtocol create(InetSocketAddress addr, Configuration conf)
      throws IOException {
    return createRPCProxy(addr, conf);
  }

  private ClientProtocol createRPCProxy(InetSocketAddress addr,
      Configuration conf) throws IOException {
    return (ClientProtocol) RPC.getProxy(ClientProtocol.class,
        ClientProtocol.versionID, addr, UserGroupInformation.getCurrentUser(),
        conf, NetUtils.getSocketFactory(conf, ClientProtocol.class));
  }

  @Override
  public void close(ClientProtocol clientProtocol) throws IOException {
    RPC.stopProxy(clientProtocol);
  }

}
