/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.lib.util;

import org.apache.hadoop.conf.Configuration;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

/**
 * Configuration utilities.
 */
public abstract class ConfigurationUtils {

  /**
   * Copy configuration key/value pairs from one configuration to another if a property exists in the target, it gets
   * replaced.
   *
   * @param source source configuration.
   * @param target target configuration.
   */
  public static void copy(Configuration source, Configuration target) {
    Check.notNull(source, "source");
    Check.notNull(target, "target");
    for (Map.Entry<String, String> entry : source) {
      target.set(entry.getKey(), entry.getValue());
    }
  }

  /**
   * Injects configuration key/value pairs from one configuration to another if the key does not exist in the target
   * configuration.
   *
   * @param source source configuration.
   * @param target target configuration.
   */
  public static void injectDefaults(Configuration source, Configuration target) {
    Check.notNull(source, "source");
    Check.notNull(target, "target");
    for (Map.Entry<String, String> entry : source) {
      if (target.get(entry.getKey()) == null) {
        target.set(entry.getKey(), entry.getValue());
      }
    }
  }

  /**
   * Returns a new ConfigurationUtils instance with all inline values resolved.
   *
   * @return a new ConfigurationUtils instance with all inline values resolved.
   */
  public static Configuration resolve(Configuration conf) {
    Configuration resolved = new Configuration(false);
    for (Map.Entry<String, String> entry : conf) {
      resolved.set(entry.getKey(), conf.get(entry.getKey()));
    }
    return resolved;
  }

  // Canibalized from FileSystemAccess <code>Configuration.loadResource()</code>.

  /**
   * Create a configuration from an InputStream.
   * <p/>
   * ERROR canibalized from <code>Configuration.loadResource()</code>.
   *
   * @param is inputstream to read the configuration from.
   *
   * @throws IOException thrown if the configuration could not be read.
   */
  public static void load(Configuration conf, InputStream is) throws IOException {
    try {
      DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
      // ignore all comments inside the xml file
      docBuilderFactory.setIgnoringComments(true);
      DocumentBuilder builder = docBuilderFactory.newDocumentBuilder();
      Document doc = builder.parse(is);
      parseDocument(conf, doc);
    } catch (SAXException e) {
      throw new IOException(e);
    } catch (ParserConfigurationException e) {
      throw new IOException(e);
    }
  }

  // Canibalized from FileSystemAccess <code>Configuration.loadResource()</code>.
  private static void parseDocument(Configuration conf, Document doc) throws IOException {
    try {
      Element root = doc.getDocumentElement();
      if (!"configuration".equals(root.getTagName())) {
        throw new IOException("bad conf file: top-level element not <configuration>");
      }
      NodeList props = root.getChildNodes();
      for (int i = 0; i < props.getLength(); i++) {
        Node propNode = props.item(i);
        if (!(propNode instanceof Element)) {
          continue;
        }
        Element prop = (Element) propNode;
        if (!"property".equals(prop.getTagName())) {
          throw new IOException("bad conf file: element not <property>");
        }
        NodeList fields = prop.getChildNodes();
        String attr = null;
        String value = null;
        for (int j = 0; j < fields.getLength(); j++) {
          Node fieldNode = fields.item(j);
          if (!(fieldNode instanceof Element)) {
            continue;
          }
          Element field = (Element) fieldNode;
          if ("name".equals(field.getTagName()) && field.hasChildNodes()) {
            attr = ((Text) field.getFirstChild()).getData().trim();
          }
          if ("value".equals(field.getTagName()) && field.hasChildNodes()) {
            value = ((Text) field.getFirstChild()).getData();
          }
        }

        if (attr != null && value != null) {
          conf.set(attr, value);
        }
      }

    } catch (DOMException e) {
      throw new IOException(e);
    }
  }

}
