/*
 * Copyright (c) 2015 Cossack Labs Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef THEMISPP_MESSAGE_TEST_HPP
#define THEMISPP_MESSAGE_TEST_HPP

#include <iostream>

#include <common/sput.h>

#include <themispp/secure_keygen.hpp>
#include <themispp/secure_message.hpp>

#include "utils.hpp"

namespace themispp
{
namespace secure_message_test
{

static const std::vector<uint8_t> message1 = as_bytes("this is test message1 for Secure Message test");
static const std::vector<uint8_t> message2 = as_bytes("this is test message2 for Secure Message test");

template <asym_algs alg_t_p>
static int secure_message_test_()
{
    try {
        themispp::secure_key_pair_generator_t<alg_t_p> g;
        std::vector<uint8_t> private_key(g.get_priv());
        std::vector<uint8_t> public_key(g.get_pub());
        g.gen();
        std::vector<uint8_t> peer_private_key(g.get_priv());
        std::vector<uint8_t> peer_public_key(g.get_pub());
        g.gen();
        std::vector<uint8_t> intruder_private_key(g.get_priv());
        std::vector<uint8_t> intruder_public_key(g.get_pub());
        themispp::secure_message_t a(private_key, peer_public_key);
        themispp::secure_message_t b(peer_private_key, public_key);
        themispp::secure_message_t c(intruder_private_key, public_key);

        sput_fail_unless(private_key != peer_private_key, "generate two identical key pairs", __LINE__);
        std::vector<uint8_t> encrypted_message = a.encrypt(message1);
        std::vector<uint8_t> decrypted_message = b.decrypt(encrypted_message);
        sput_fail_unless(message1 == decrypted_message, "decryption fail", __LINE__);
        try {
            decrypted_message = c.decrypt(encrypted_message);
            sput_fail_unless(false, "decryption fail", __LINE__);
        } catch (themispp::exception_t& e) {
            sput_fail_unless(e.status() == THEMIS_FAIL, "decryption by intruder", __LINE__);
        }
    } catch (themispp::exception_t& e) {
        sput_fail_unless(false, e.what(), __LINE__);
    }
    return 0;
}

static int secure_message_test_rsa()
{
    return secure_message_test_<themispp::RSA>();
}

static int secure_message_test_ec()
{
    return secure_message_test_<themispp::EC>();
}

template <asym_algs alg_t_p>
static int secure_message_sign_verify_test()
{
    try {
        themispp::secure_key_pair_generator_t<alg_t_p> g;
        std::vector<uint8_t> private_key(g.get_priv());
        std::vector<uint8_t> public_key(g.get_pub());
        g.gen();
        std::vector<uint8_t> peer_private_key(g.get_priv());
        std::vector<uint8_t> peer_public_key(g.get_pub());
        g.gen();
        std::vector<uint8_t> intruder_private_key(g.get_priv());
        std::vector<uint8_t> intruder_public_key(g.get_pub());
        themispp::secure_message_t a(private_key, peer_public_key);
        themispp::secure_message_t b(peer_private_key, public_key);
        themispp::secure_message_t c(intruder_private_key, public_key);

        sput_fail_unless(private_key != peer_private_key, "generate two identical key pairs", __LINE__);
        std::vector<uint8_t> a_signed_message = a.sign(message1);
        std::vector<uint8_t> verified_message = b.verify(a_signed_message);
        sput_fail_unless(message1 == verified_message, "verification fail", __LINE__);
        // corrupt one byte
        a_signed_message[2]++;
        try {
            verified_message = c.verify(a_signed_message);
            sput_fail_unless(false, "verification fail", __LINE__);
        } catch (themispp::exception_t& e) {
            sput_fail_unless(e.status() == THEMIS_INVALID_PARAMETER, "verification by intruder", __LINE__);
        }
    } catch (themispp::exception_t& e) {
        sput_fail_unless(false, e.what(), __LINE__);
    }
    return 0;
}

static int secure_message_sign_verify_test_rsa()
{
    return secure_message_sign_verify_test<themispp::RSA>();
}

static int secure_message_sign_verify_test_ec()
{
    return secure_message_sign_verify_test<themispp::EC>();
}

static int secure_message_test_key_mismatch()
{
    themispp::secure_key_pair_generator_t<themispp::EC> gen_ec;
    std::vector<uint8_t> private_key(gen_ec.get_priv());
    std::vector<uint8_t> public_key(gen_ec.get_pub());
    try {
        themispp::secure_message_t a(public_key, private_key);
        sput_fail_unless(false, "mismatched keys (no failure)", __LINE__);
    } catch (const themispp::exception_t& e) {
        sput_fail_unless(e.status() == THEMIS_INVALID_PARAMETER, "mismatched keys", __LINE__);
    }
    return 0;
}

static int secure_message_verify_overlong_elliptic_signatures()
{
    std::vector<uint8_t> public_key = as_bytes((uint8_t[]){
        0x55, 0x45, 0x43, 0x32, 0x00, 0x00, 0x00, 0x2d, 0x84, 0xa1, 0x02, 0xb2, 0x03, 0x32, 0x23,
        0xfc, 0x44, 0xb0, 0x30, 0xe9, 0xed, 0x8b, 0x31, 0x79, 0xa3, 0xe6, 0x86, 0xb4, 0x6f, 0xda,
        0xde, 0xb7, 0x77, 0xf0, 0x71, 0x37, 0x1e, 0xaf, 0x58, 0xc7, 0x04, 0xba, 0x6c, 0x82, 0x10,
    });

    // NIST P-256 curve, 69..72 bytes long signatures, with and without zero padding
    std::vector<std::vector<uint8_t> > signatures;
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x46, 0x02, 0x21, 0x00, 0xd9, 0x14, 0xeb, 0xbb, 0x55, 0xc8, 0x12, 0x35, 0x6e,
        0x39, 0x56, 0x61, 0x58, 0x88, 0x94, 0xeb, 0xc8, 0x02, 0x19, 0x2d, 0x6f, 0x2c, 0x86, 0xaf,
        0x2b, 0x6f, 0xe1, 0xf1, 0x3a, 0x77, 0xe7, 0x23, 0x02, 0x21, 0x00, 0xf6, 0xb4, 0xf2, 0xd9,
        0xfb, 0xb1, 0x93, 0x3e, 0xb2, 0x20, 0x1e, 0xf1, 0x4f, 0x8b, 0xcd, 0xa2, 0x1f, 0xb8, 0x21,
        0x28, 0x44, 0xa5, 0xb1, 0xb1, 0x70, 0x35, 0x14, 0xae, 0xe1, 0x05, 0x5a, 0xa4,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x45, 0x02, 0x20, 0x6c, 0x69, 0x2e, 0x82, 0x71, 0x78, 0x4c, 0xb9, 0x53, 0x41,
        0x90, 0xc2, 0xe1, 0xf7, 0x8c, 0x21, 0xe8, 0xcc, 0xeb, 0x08, 0x67, 0xff, 0x8a, 0xfe, 0x58,
        0x01, 0xbc, 0xf0, 0xb1, 0xab, 0x14, 0x72, 0x02, 0x21, 0x00, 0xa9, 0xbf, 0x8d, 0x6b, 0x6f,
        0xc5, 0xf4, 0x5d, 0x2f, 0xb8, 0x14, 0xc6, 0xbd, 0xf2, 0x51, 0x4d, 0x3c, 0x81, 0xf7, 0x74,
        0x65, 0x08, 0x45, 0x50, 0xe8, 0xa9, 0xa0, 0xe1, 0x09, 0x72, 0xf5, 0xc3, 0x00,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x45, 0x02, 0x20, 0x6c, 0x69, 0x2e, 0x82, 0x71, 0x78, 0x4c, 0xb9, 0x53, 0x41,
        0x90, 0xc2, 0xe1, 0xf7, 0x8c, 0x21, 0xe8, 0xcc, 0xeb, 0x08, 0x67, 0xff, 0x8a, 0xfe, 0x58,
        0x01, 0xbc, 0xf0, 0xb1, 0xab, 0x14, 0x72, 0x02, 0x21, 0x00, 0xa9, 0xbf, 0x8d, 0x6b, 0x6f,
        0xc5, 0xf4, 0x5d, 0x2f, 0xb8, 0x14, 0xc6, 0xbd, 0xf2, 0x51, 0x4d, 0x3c, 0x81, 0xf7, 0x74,
        0x65, 0x08, 0x45, 0x50, 0xe8, 0xa9, 0xa0, 0xe1, 0x09, 0x72, 0xf5, 0xc3,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x44, 0x02, 0x20, 0x4c, 0x0e, 0xd6, 0x8b, 0xa2, 0x74, 0x42, 0xf7, 0x0c, 0xdc,
        0xe3, 0x28, 0xf8, 0x0b, 0xfe, 0x73, 0x1c, 0x94, 0x85, 0x97, 0x19, 0xd6, 0x27, 0x50, 0x52,
        0xdb, 0xbf, 0x56, 0xa5, 0xbb, 0x21, 0xce, 0x02, 0x20, 0x46, 0x65, 0xc4, 0xa0, 0x6c, 0x9e,
        0x7e, 0xff, 0xd6, 0xe0, 0x99, 0x79, 0x4e, 0x2b, 0x63, 0x29, 0x0a, 0xd1, 0x6e, 0x95, 0x9b,
        0xd5, 0x12, 0x3a, 0x8e, 0x09, 0xf2, 0xb6, 0x2d, 0xe4, 0x87, 0x7c, 0x00, 0x00,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x44, 0x02, 0x20, 0x4c, 0x0e, 0xd6, 0x8b, 0xa2, 0x74, 0x42, 0xf7, 0x0c, 0xdc,
        0xe3, 0x28, 0xf8, 0x0b, 0xfe, 0x73, 0x1c, 0x94, 0x85, 0x97, 0x19, 0xd6, 0x27, 0x50, 0x52,
        0xdb, 0xbf, 0x56, 0xa5, 0xbb, 0x21, 0xce, 0x02, 0x20, 0x46, 0x65, 0xc4, 0xa0, 0x6c, 0x9e,
        0x7e, 0xff, 0xd6, 0xe0, 0x99, 0x79, 0x4e, 0x2b, 0x63, 0x29, 0x0a, 0xd1, 0x6e, 0x95, 0x9b,
        0xd5, 0x12, 0x3a, 0x8e, 0x09, 0xf2, 0xb6, 0x2d, 0xe4, 0x87, 0x7c,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x43, 0x02, 0x20, 0x20, 0x33, 0xd5, 0x62, 0xd7, 0xb2, 0x59, 0x76, 0x9e, 0xc1,
        0xe5, 0x05, 0xee, 0xf0, 0x98, 0x1e, 0xde, 0xe6, 0x4d, 0xa7, 0x64, 0x72, 0xf7, 0xd8, 0x3c,
        0x17, 0xa5, 0x51, 0x26, 0xd7, 0x70, 0xe5, 0x02, 0x1f, 0x3e, 0x02, 0x53, 0x15, 0xa2, 0x09,
        0x0b, 0x3e, 0xc8, 0xbf, 0xca, 0x32, 0x97, 0x2d, 0x5e, 0x0a, 0x48, 0x50, 0x8b, 0x12, 0xc7,
        0x52, 0x8c, 0x01, 0x0b, 0x2c, 0x9f, 0xd2, 0x27, 0x45, 0x90, 0x00, 0x00, 0x00,
    }));
    signatures.push_back(as_bytes((uint8_t[]){
        0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
        0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
        0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
        0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
        0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
        0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
        0x6c, 0x30, 0x43, 0x02, 0x20, 0x20, 0x33, 0xd5, 0x62, 0xd7, 0xb2, 0x59, 0x76, 0x9e, 0xc1,
        0xe5, 0x05, 0xee, 0xf0, 0x98, 0x1e, 0xde, 0xe6, 0x4d, 0xa7, 0x64, 0x72, 0xf7, 0xd8, 0x3c,
        0x17, 0xa5, 0x51, 0x26, 0xd7, 0x70, 0xe5, 0x02, 0x1f, 0x3e, 0x02, 0x53, 0x15, 0xa2, 0x09,
        0x0b, 0x3e, 0xc8, 0xbf, 0xca, 0x32, 0x97, 0x2d, 0x5e, 0x0a, 0x48, 0x50, 0x8b, 0x12, 0xc7,
        0x52, 0x8c, 0x01, 0x0b, 0x2c, 0x9f, 0xd2, 0x27, 0x45, 0x90,
    }));

    std::vector<uint8_t> expected = as_bytes(
        "Boléro is a one-movement orchestral piece by the French composer Maurice Ravel");

    themispp::secure_message_t smessage(std::vector<uint8_t>(), public_key);
    for (size_t i = 0; i < signatures.size(); i++) {
        std::vector<uint8_t> verified;
        try {
            verified = smessage.verify(signatures[i]);
        } catch (const themispp::exception_t&) {
            sput_fail_unless(false, "verification failed", __LINE__);
            continue;
        }
        sput_fail_unless(verified == expected, "message matches", __LINE__);
    }
    return 0;
}

inline void run_secure_message_test()
{
    sput_enter_suite("ThemisPP secure message test");
    sput_run_test(secure_message_test_rsa, "secure_message_test_rsa", __FILE__);
    sput_run_test(secure_message_test_ec, "secure_message_test_ec", __FILE__);
    sput_run_test(secure_message_sign_verify_test_rsa, "secure_message_sign_verify_test_rsa", __FILE__);
    sput_run_test(secure_message_sign_verify_test_ec, "secure_message_sign_verify_test_ec", __FILE__);
    sput_run_test(secure_message_test_key_mismatch, "secure_message_test_key_mismatch", __FILE__);
    sput_run_test(secure_message_verify_overlong_elliptic_signatures,
                  "secure_message_verify_overlong_elliptic_signatures",
                  __FILE__);
}

} // namespace secure_message_test
} // namespace themispp
#endif
