/*
* Copyright (c) 2015 Cossack Labs Limited
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package com.cossacklabs.themis.test;

import java.util.Arrays;
import java.util.Random;
import java.nio.charset.StandardCharsets;

import com.cossacklabs.themis.KeyGenerationException;
import com.cossacklabs.themis.Keypair;
import com.cossacklabs.themis.KeypairGenerator;
import com.cossacklabs.themis.NullArgumentException;
import com.cossacklabs.themis.PublicKey;
import com.cossacklabs.themis.SecureMessage;
import com.cossacklabs.themis.SecureMessageWrapException;

import static org.junit.Assert.*;
import org.junit.Test;

public class SecureMessageTest {
	
	@Test
	public void runTest() {
		
		Keypair aPair = null;
		Keypair bPair = null;
		
		try {
			aPair = KeypairGenerator.generateKeypair();
			bPair = KeypairGenerator.generateKeypair();
		} catch (KeyGenerationException e) {
			fail("Failed to generate keypairs");
		}
		
		assertNotNull(aPair);
		assertNotNull(bPair);

		Random rand = new Random();
		
		int messageLength = 0;
		
		do {
			messageLength = rand.nextInt(2048);
		} while (0 == messageLength);
		
		byte[] message = new byte[messageLength];
		rand.nextBytes(message);
		
		try {
			SecureMessage aWrapper = new SecureMessage(aPair.getPrivateKey(), bPair.getPublicKey());
			SecureMessage bWrapper = new SecureMessage(bPair.getPrivateKey(), aPair.getPublicKey());
			
			byte[] wrappedMessage = aWrapper.wrap(message);
			
			assertTrue(message.length < wrappedMessage.length);
			
			byte[] unwrappedMessage = bWrapper.unwrap(wrappedMessage);
			
			assertTrue(Arrays.equals(message, unwrappedMessage));


			SecureMessage signer = new SecureMessage(aPair.getPrivateKey());
			SecureMessage verifier = new SecureMessage(aPair.getPublicKey());

			byte[] signedMessage = signer.sign(message);

			assertTrue(message.length < signedMessage.length);

			byte[] verifiedMessage = verifier.verify(signedMessage);

			assertTrue(Arrays.equals(message, verifiedMessage));

		} catch (NullArgumentException e) {
			fail(e.getClass().getCanonicalName() + ": " + e.getMessage());
		} catch (SecureMessageWrapException e) {
			fail(e.getClass().getCanonicalName());
		}
	}

	// Make sure JavaThemis can handle overlong signed messages produced by
	// old versions of GoThemis, ThemisPP, and JavaThemis itself.
	@Test
	public void verifyOverlongEllipticSignatures() {

		final PublicKey publicKey = new PublicKey(asByteArray(new int[] {
			0x55, 0x45, 0x43, 0x32, 0x00, 0x00, 0x00, 0x2d, 0x84, 0xa1, 0x02, 0xb2, 0x03, 0x32, 0x23,
			0xfc, 0x44, 0xb0, 0x30, 0xe9, 0xed, 0x8b, 0x31, 0x79, 0xa3, 0xe6, 0x86, 0xb4, 0x6f, 0xda,
			0xde, 0xb7, 0x77, 0xf0, 0x71, 0x37, 0x1e, 0xaf, 0x58, 0xc7, 0x04, 0xba, 0x6c, 0x82, 0x10,
		}));

		// NIST P-256 curve, 69..72 bytes long signatures, with and without zero padding
		final byte[][] signatures = new byte[][] {
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x46, 0x02, 0x21, 0x00, 0xd9, 0x14, 0xeb, 0xbb, 0x55, 0xc8, 0x12, 0x35, 0x6e,
				0x39, 0x56, 0x61, 0x58, 0x88, 0x94, 0xeb, 0xc8, 0x02, 0x19, 0x2d, 0x6f, 0x2c, 0x86, 0xaf,
				0x2b, 0x6f, 0xe1, 0xf1, 0x3a, 0x77, 0xe7, 0x23, 0x02, 0x21, 0x00, 0xf6, 0xb4, 0xf2, 0xd9,
				0xfb, 0xb1, 0x93, 0x3e, 0xb2, 0x20, 0x1e, 0xf1, 0x4f, 0x8b, 0xcd, 0xa2, 0x1f, 0xb8, 0x21,
				0x28, 0x44, 0xa5, 0xb1, 0xb1, 0x70, 0x35, 0x14, 0xae, 0xe1, 0x05, 0x5a, 0xa4,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x45, 0x02, 0x20, 0x6c, 0x69, 0x2e, 0x82, 0x71, 0x78, 0x4c, 0xb9, 0x53, 0x41,
				0x90, 0xc2, 0xe1, 0xf7, 0x8c, 0x21, 0xe8, 0xcc, 0xeb, 0x08, 0x67, 0xff, 0x8a, 0xfe, 0x58,
				0x01, 0xbc, 0xf0, 0xb1, 0xab, 0x14, 0x72, 0x02, 0x21, 0x00, 0xa9, 0xbf, 0x8d, 0x6b, 0x6f,
				0xc5, 0xf4, 0x5d, 0x2f, 0xb8, 0x14, 0xc6, 0xbd, 0xf2, 0x51, 0x4d, 0x3c, 0x81, 0xf7, 0x74,
				0x65, 0x08, 0x45, 0x50, 0xe8, 0xa9, 0xa0, 0xe1, 0x09, 0x72, 0xf5, 0xc3, 0x00,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x45, 0x02, 0x20, 0x6c, 0x69, 0x2e, 0x82, 0x71, 0x78, 0x4c, 0xb9, 0x53, 0x41,
				0x90, 0xc2, 0xe1, 0xf7, 0x8c, 0x21, 0xe8, 0xcc, 0xeb, 0x08, 0x67, 0xff, 0x8a, 0xfe, 0x58,
				0x01, 0xbc, 0xf0, 0xb1, 0xab, 0x14, 0x72, 0x02, 0x21, 0x00, 0xa9, 0xbf, 0x8d, 0x6b, 0x6f,
				0xc5, 0xf4, 0x5d, 0x2f, 0xb8, 0x14, 0xc6, 0xbd, 0xf2, 0x51, 0x4d, 0x3c, 0x81, 0xf7, 0x74,
				0x65, 0x08, 0x45, 0x50, 0xe8, 0xa9, 0xa0, 0xe1, 0x09, 0x72, 0xf5, 0xc3,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x44, 0x02, 0x20, 0x4c, 0x0e, 0xd6, 0x8b, 0xa2, 0x74, 0x42, 0xf7, 0x0c, 0xdc,
				0xe3, 0x28, 0xf8, 0x0b, 0xfe, 0x73, 0x1c, 0x94, 0x85, 0x97, 0x19, 0xd6, 0x27, 0x50, 0x52,
				0xdb, 0xbf, 0x56, 0xa5, 0xbb, 0x21, 0xce, 0x02, 0x20, 0x46, 0x65, 0xc4, 0xa0, 0x6c, 0x9e,
				0x7e, 0xff, 0xd6, 0xe0, 0x99, 0x79, 0x4e, 0x2b, 0x63, 0x29, 0x0a, 0xd1, 0x6e, 0x95, 0x9b,
				0xd5, 0x12, 0x3a, 0x8e, 0x09, 0xf2, 0xb6, 0x2d, 0xe4, 0x87, 0x7c, 0x00, 0x00,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x44, 0x02, 0x20, 0x4c, 0x0e, 0xd6, 0x8b, 0xa2, 0x74, 0x42, 0xf7, 0x0c, 0xdc,
				0xe3, 0x28, 0xf8, 0x0b, 0xfe, 0x73, 0x1c, 0x94, 0x85, 0x97, 0x19, 0xd6, 0x27, 0x50, 0x52,
				0xdb, 0xbf, 0x56, 0xa5, 0xbb, 0x21, 0xce, 0x02, 0x20, 0x46, 0x65, 0xc4, 0xa0, 0x6c, 0x9e,
				0x7e, 0xff, 0xd6, 0xe0, 0x99, 0x79, 0x4e, 0x2b, 0x63, 0x29, 0x0a, 0xd1, 0x6e, 0x95, 0x9b,
				0xd5, 0x12, 0x3a, 0x8e, 0x09, 0xf2, 0xb6, 0x2d, 0xe4, 0x87, 0x7c,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x43, 0x02, 0x20, 0x20, 0x33, 0xd5, 0x62, 0xd7, 0xb2, 0x59, 0x76, 0x9e, 0xc1,
				0xe5, 0x05, 0xee, 0xf0, 0x98, 0x1e, 0xde, 0xe6, 0x4d, 0xa7, 0x64, 0x72, 0xf7, 0xd8, 0x3c,
				0x17, 0xa5, 0x51, 0x26, 0xd7, 0x70, 0xe5, 0x02, 0x1f, 0x3e, 0x02, 0x53, 0x15, 0xa2, 0x09,
				0x0b, 0x3e, 0xc8, 0xbf, 0xca, 0x32, 0x97, 0x2d, 0x5e, 0x0a, 0x48, 0x50, 0x8b, 0x12, 0xc7,
				0x52, 0x8c, 0x01, 0x0b, 0x2c, 0x9f, 0xd2, 0x27, 0x45, 0x90, 0x00, 0x00, 0x00,
			}),
			asByteArray(new int[] {
				0x20, 0x26, 0x04, 0x26, 0x4f, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x42, 0x6f, 0x6c,
				0xc3, 0xa9, 0x72, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, 0x6f, 0x6e, 0x65, 0x2d, 0x6d,
				0x6f, 0x76, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x20, 0x6f, 0x72, 0x63, 0x68, 0x65, 0x73, 0x74,
				0x72, 0x61, 0x6c, 0x20, 0x70, 0x69, 0x65, 0x63, 0x65, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68,
				0x65, 0x20, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6f, 0x73,
				0x65, 0x72, 0x20, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x20, 0x52, 0x61, 0x76, 0x65,
				0x6c, 0x30, 0x43, 0x02, 0x20, 0x20, 0x33, 0xd5, 0x62, 0xd7, 0xb2, 0x59, 0x76, 0x9e, 0xc1,
				0xe5, 0x05, 0xee, 0xf0, 0x98, 0x1e, 0xde, 0xe6, 0x4d, 0xa7, 0x64, 0x72, 0xf7, 0xd8, 0x3c,
				0x17, 0xa5, 0x51, 0x26, 0xd7, 0x70, 0xe5, 0x02, 0x1f, 0x3e, 0x02, 0x53, 0x15, 0xa2, 0x09,
				0x0b, 0x3e, 0xc8, 0xbf, 0xca, 0x32, 0x97, 0x2d, 0x5e, 0x0a, 0x48, 0x50, 0x8b, 0x12, 0xc7,
				0x52, 0x8c, 0x01, 0x0b, 0x2c, 0x9f, 0xd2, 0x27, 0x45, 0x90,
			}),
		};

		final byte[] expected = "Bol\u00E9ro is a one-movement orchestral piece by the French composer Maurice Ravel".getBytes(StandardCharsets.UTF_8);

		SecureMessage smessage = new SecureMessage(publicKey);

		for (int i = 0; i < signatures.length; i++) {
			byte[] verified = null;
			try {
				verified = smessage.verify(signatures[i]);
			} catch (SecureMessageWrapException e) {
				fail("verification failed for signature #" + i + ": "
					+ e.getClass().getCanonicalName() + ": "
					+ e.getMessage());
			}
			assertArrayEquals(expected, verified);
		}
	}

	// You cannot write byte literals in Java, and even if you could, they
	// would be *signed*. Go with this roundabout way to keep blobs readable
	// (for a certain definition thereof).
	private static byte[] asByteArray(int[] array) {
		byte[] bytes = new byte[array.length];
		for (int i = 0; i < array.length; i++) {
			bytes[i] = (byte)array[i];
		}
		return bytes;
	}
}
