<?php

/*
 * This file is part of composer/satis.
 *
 * (c) Composer <https://github.com/composer>
 *
 * For the full copyright and license information, please view
 * the LICENSE file that was distributed with this source code.
 */

namespace Composer\Satis\Command;

use Composer\Command\BaseCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Composer\Json\JsonFile;

class PurgeCommand extends BaseCommand
{
    protected function configure()
    {
        $this->setName('purge')
            ->setDescription('Purge packages')
            ->setDefinition(array(
                new InputArgument('file', InputArgument::OPTIONAL, 'Json file to use', './satis.json'),
                new InputArgument('output-dir', InputArgument::OPTIONAL, 'Location where to output built files', null),
            ))
            ->setHelp(
<<<EOT
The <info>purge</info> command deletes useless archive files, depending
on given json file (satis.json is used by default) and the
lastest json file in the include directory of the given output-dir.

In your satis.json (or other name you give), you must define
"archive" argument.

EOT
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $configFile = $input->getArgument('file');
        $file = new JsonFile($configFile);
        if (!$file->exists()) {
            $output->writeln('<error>File not found: '.$configFile.'</error>');

            return 1;
        }
        $config = $file->read();

        /**
         * Check whether archive is defined
         */
        if (!isset($config['archive']) || !isset($config['archive']['directory'])) {
            $output->writeln('<error>You must define "archive" parameter in your '.$configFile.'</error>');

            return 1;
        }

        if (!$outputDir = $input->getArgument('output-dir')) {
            throw new \InvalidArgumentException('The output dir must be specified as second argument');
        }

        $files = glob($outputDir."/include/*.json");

        if (empty($files)) {
            $output->writeln('<info>No log file</info>');

            return 1;
        }

        $files = array_combine($files, array_map("filemtime", $files));
        arsort($files);

        $file = file_get_contents(key($files));
        $json = json_decode($file, true);

        $prefix = $config['archive']['directory'] . '/';
        if (isset($config['archive']['prefix-url'])) {
            $prefix = $config['archive']['prefix-url'] . '/' . $prefix;
        }
        $length = strlen($prefix);

        $needed = array();
        foreach ($json['packages'] as $package) {
            foreach ($package as $version) {
                if (!isset($version['dist']['url'])) {
                    continue;
                }
                $url = $version['dist']['url'];
                if (substr($url, 0, $length) === $prefix) {
                    $needed[] = substr($url, $length);
                }
            }
        }

        /**
         * Regular files in output-dir
         */
        $files = scandir($outputDir."/".$config['archive']['directory'], 1);

        if (empty($files)) {
            $output->writeln('<info>No archived files</info>');

            return 1;
        }

        $unreferenced = array_diff($files, $needed);

        /**
         * Removed unreferenced files
         */
        foreach ($unreferenced as $file) {
            $path = $outputDir."/".$config['archive']['directory']."/".$file;
            if (!is_file($path)) {
                continue;
            }

            $output->writeln("<info>".$file." :: deleted</info>");

            unlink($path);
        }

        $output->writeln("<info>Purge :: finished</info>");

        return 0;
    }
}
