/*
Package o11y is the primary entrypoint for wiring up a standard configuration of the o11y
observability system.
*/
package o11y

import (
	"context"
	"fmt"
	"io"
	"math"
	"os"
	"time"

	"github.com/DataDog/datadog-go/statsd"
	"github.com/cenkalti/backoff/v5"
	"github.com/rollbar/rollbar-go"
	"go.opentelemetry.io/otel/attribute"
	sdktrace "go.opentelemetry.io/otel/sdk/trace"
	semconv "go.opentelemetry.io/otel/semconv/v1.12.0"

	"github.com/circleci/ex/config/secret"
	"github.com/circleci/ex/o11y"
	"github.com/circleci/ex/o11y/otel"
)

// SampleOut is the maximum value the sample rate can be
const SampleOut = math.MaxUint32

// OtelConfig contains all the things we need to configure for otel based instrumentation.
type OtelConfig struct {
	GrpcHostAndPort string

	// HTTPTracesURL configures a host for exporting traces to http[s]://host[:port][/path]
	HTTPTracesURL string

	// HTTPAuthorization is the authorization token to send with http requests
	HTTPAuthorization secret.String

	Dataset string
	// UseEnvironments will cause spans to be sent to the new honeycomb environments
	UseEnvironments bool

	// This will tell the collectors to bypass the server side tail based sampler
	DisableTailSampling bool

	// DisableText prevents output to stdout for noisy services. Ignored if no other no hosts are supplied
	DisableText bool

	Test bool

	SampleTraces  bool
	SampleKeyFunc func(map[string]interface{}) string
	SampleRates   map[string]uint

	Statsd                  string
	StatsNamespace          string
	StatsdTelemetryDisabled bool

	RollbarToken      secret.String
	RollbarEnv        string
	RollbarServerRoot string
	RollbarDisabled   bool

	Version string
	Service string
	Mode    string

	// Metrics allows setting a custom metrics client. Typically, the default Statsd provider is preferred.
	// The provided value will be closed by the cleanup function
	Metrics o11y.ClosableMetricsProvider

	// Override the default writer for text span output
	Writer io.Writer

	// SpanExporters allows you explicitly provide a set of exporters, as an advanced use-case.
	SpanExporters []sdktrace.SpanExporter

	// ProviderFunc is used to provide a custom provider.
	ProviderFunc func(conf otel.Config) (o11y.Provider, error)
}

// Otel is the primary entrypoint to initialize the o11y system for otel.
func Otel(ctx context.Context, o OtelConfig) (context.Context, func(context.Context), error) {
	hostname, _ := os.Hostname()

	cfg := o.ToOTEL()
	cfg.SpanExporters = o.SpanExporters

	mProv, err := metricsProvider(ctx, o, hostname)
	if err != nil {
		return ctx, nil, fmt.Errorf("metrics provider failed: %w", err)
	}
	cfg.Metrics = mProv

	if o.ProviderFunc == nil {
		o.ProviderFunc = otel.New
	}
	o11yProvider, err := o.ProviderFunc(cfg)
	if err != nil {
		return ctx, nil, err
	}

	o11yProvider.AddGlobalField("service", o.Service)
	o11yProvider.AddGlobalField("version", o.Version)
	if o.Mode != "" {
		o11yProvider.AddGlobalField("mode", o.Mode)
	}

	if o.RollbarToken != "" {
		client := rollbar.NewAsync(o.RollbarToken.Raw(), o.RollbarEnv, o.Version, hostname, o.RollbarServerRoot)
		client.SetEnabled(!o.RollbarDisabled)
		client.Message(rollbar.INFO, "Deployment")
		o11yProvider = rollbarOtelProvider{
			Provider:      o11yProvider,
			rollBarClient: client,
		}
	}

	ctx = o11y.WithProvider(ctx, o11yProvider)

	return ctx, o11yProvider.Close, nil
}

func (o *OtelConfig) ToOTEL() otel.Config {
	cfg := otel.Config{
		GrpcHostAndPort:   o.GrpcHostAndPort,
		HTTPTracesURL:     o.HTTPTracesURL,
		HTTPAuthorization: o.HTTPAuthorization,
		Dataset:           o.Dataset,
		ResourceAttributes: []attribute.KeyValue{
			semconv.ServiceNameKey.String(o.Service),
			semconv.ServiceVersionKey.String(o.Version),
			// Other Config specific fields
			attribute.String("service.mode", o.Mode),

			// HC Backwards compatible fields - can remove once boards are updated
			attribute.String("service", o.Service),
			attribute.String("mode", o.Mode),
			attribute.String("version", o.Version),
		},

		DisableText: o.DisableText,

		SampleTraces:  o.SampleTraces,
		SampleKeyFunc: o.SampleKeyFunc,
		SampleRates:   o.SampleRates,

		Test: o.Test,

		Writer: o.Writer,
	}
	if o.UseEnvironments {
		cfg.ResourceAttributes = append(cfg.ResourceAttributes, attribute.Bool("meta.environments", true))
	}
	if o.DisableTailSampling {
		cfg.ResourceAttributes = append(cfg.ResourceAttributes, attribute.Bool("meta.sampling.disabled", true))
	}
	return cfg
}

// N.B this copies the block from Setup, but don't factor that out since the HC stuff will be removed soon
// TODO - delete this comment after HC cleanup
func metricsProvider(ctx context.Context, o OtelConfig, hostname string) (o11y.ClosableMetricsProvider, error) {
	if o.Metrics != nil {
		return o.Metrics, nil
	}
	if o.Statsd == "" {
		return &statsd.NoOpClient{}, nil
	}

	tags := []string{
		"service:" + o.Service,
		"version:" + o.Version,
		"hostname:" + hostname,
	}
	if o.Mode != "" {
		tags = append(tags, "mode:"+o.Mode)
	}

	statsdOpts := []statsd.Option{
		statsd.WithNamespace(o.StatsNamespace),
		statsd.WithTags(tags),
	}
	if o.StatsdTelemetryDisabled {
		statsdOpts = append(statsdOpts, statsd.WithoutTelemetry())
	}

	stats, err := backoff.Retry(ctx, func() (*statsd.Client, error) {
		return statsd.New(o.Statsd, statsdOpts...)
	}, backoff.WithBackOff(backoff.NewConstantBackOff(time.Second)), backoff.WithMaxTries(31))
	if err != nil {
		return nil, err
	}
	return stats, nil
}

type rollbarOtelProvider struct {
	o11y.Provider
	rollBarClient *rollbar.Client
}

func (p rollbarOtelProvider) Close(ctx context.Context) {
	p.Provider.Close(ctx)
	_ = p.rollBarClient.Close()
}

func (p rollbarOtelProvider) RollBarClient() *rollbar.Client {
	return p.rollBarClient
}

func (p rollbarOtelProvider) RawProvider() *otel.Provider {
	return p.Provider.(*otel.Provider)
}
