package io.circe.argus.macros

import java.io.File
import java.time.ZonedDateTime
import java.util.UUID

import io.circe.argus.HasSchemaSource
import io.circe.argus.json.JsonDiff
import io.circe.argus.schema.Schema
import cats.syntax.either._
import io.circe._
import io.circe.syntax._
import org.scalatest.flatspec.AnyFlatSpec
import org.scalatest.matchers.should.Matchers

import scala.io.Source

class FromSchemaSpec extends AnyFlatSpec with Matchers with JsonMatchers {

  "Making schemas" should "build case classes" in {
    @fromSchemaJson("""
      {
        "definitions": {
          "Address": {
            "type": "object",
            "properties": {
              "number": { "type": "integer" },
              "street": { "type": "string" }
            }
          },
          "SSN": { "type": "string" }
        },
        "type": "object",
        "properties": {
          "name": { "type": "string" },
          "address": { "$ref": "#/definitions/Address" },
          "ssn": { "$ref": "#/definitions/SSN" }
        },
        "required" : ["name"]
      }
    """)
    object Foo
    import Foo._

    val address = new Address(number=Some(101), street=Some("Main St"))
    val root = new Root("Fred", Some(address), Some("107-245"))
    root.name should === ("Fred")
    root.address should === (Some(Address(Some(101), Some("Main St"))))
    root.ssn should === (Some("107-245"))
  }

  it should "build nested schemas (and name them after the field name)" in {
    @fromSchemaJson("""
      {
        "type": "object",
        "properties": {
          "a": {
            "type": "object",
            "properties": {
              "b" : { "type": "string" }
            }
          }
        }
      }
    """)
    object Foo
    import Foo._

    val a = Root.A(Some("bar"))
    Root(Some(a)).a.get.b.get should === ("bar")
  }

  it should "build enum types" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "country": { "enum": ["UK", "USA", "NZ"] }
      }
    }
    """)
    object Foo
    import Foo._

    val root = Root(Some(Root.CountryEnums.NZ))
    root.country should === (Some(Root.CountryEnums.NZ))
  }

  it should "build single-value enum types" in {
    @fromSchemaJson("""
    {
      "type" : "object",
      "required" : ["pet"],
      "properties": {
        "pet": {
          "oneOf": [
            { "$ref": "#/definitions/Cat" },
            { "$ref": "#/definitions/Dog" }
          ]
        }
      },
      "definitions": {
        "Cat": {
          "type": "object",
          "properties" : { "species" : { "enum" : ["cat"] } },
          "required" : ["species"]
        },
        "Dog": {
          "type": "object",
          "properties" : { "species" : { "enum" : ["dog"] } },
          "required" : ["species"]
        }
      }
    }
    """)
    object Foo
    import Foo._

    // it would be nice to also generate these implicits when each
    // case has a distinct value
    implicit def cat2PetUnion(c: Cat): Root.PetUnion = Root.PetCat(c)
    implicit def dog2PetUnion(d: Dog): Root.PetUnion = Root.PetDog(d)

    var owner = Root(pet = Cat(species = Cat.SpeciesEnums.Cat))

    owner = owner.copy(pet = Dog(species = Dog.SpeciesEnums.Dog))

    // we can also create Cat with the default single-enum constructor
    owner = owner.copy(pet = Cat())
    owner = owner.copy(pet = Dog())
  }

  it should "build union types" in {
    @fromSchemaJson("""
    {
      "type" : "object",
      "definitions": {
        "Street": {
          "type": "object",
          "properties" : { "street" : { "type" : "string" } },
          "required" : ["street"]
        }
      },
      "properties": {
        "address": {
          "oneOf": [
            { "type": "string" },
            { "$ref": "#/definitions/Street" }
          ]
        }
      }
    }
    """)
    object Foo
    import Foo._

    val string = Root.AddressString("Main St")
    val street = Root.AddressStreet(Street("1010 Main St"))
    val stringAsUnion = string: Root.AddressUnion
    val streetAsUnion = street: Root.AddressUnion
    Root(Some(street)).address.get match {
      case Root.AddressStreet(st: Street) => st === (Street("1010 Main St"));
      case _ => fail("Didn't match type")
    }
  }

  it should "build union types without Union suffix when configured" in {
    @fromSchemaJson("""
    {
      "type" : "object",
      "definitions": {
        "Street": {
          "type": "object",
          "properties" : { "street" : { "type" : "string" } },
          "required" : ["street"]
        }
      },
      "properties": {
        "address": {
          "oneOf": [
            { "type": "string" },
            { "$ref": "#/definitions/Street" }
          ]
        }
      }
    }
    """, unionSuffix = false)
    object Foo
    import Foo._

    val string = Root.AddressString("Main St")
    val street = Root.AddressStreet(Street("1010 Main St"))
    val stringAsUnion = string: Root.Address
    val streetAsUnion = street: Root.Address
    Root(Some(street)).address.get match {
      case Root.AddressStreet(st: Street) => st === (Street("1010 Main St"));
      case _ => fail("Didn't match type")
    }
  }

  it should "build array types" in {
    @fromSchemaJson("""
    {
      "definitions" : {
        "Editor" : {
          "type" : "object",
          "properties" : {
            "name" : { "type" : "string" },
            "location" : { "type" : "string" }
          }
        }
      },
      "type" : "object",
      "properties": {
        "days" : { "type" : "array", "items": { "type" : "string" } },
        "people": {
          "type": "array",
          "items": { "$ref": "#/definitions/Editor" }
        }
      }
    }
    """)
    object Foo
    import Foo._

    val ed1 = Editor(Some("Bob"), Some("CA"))
    val ed2 = Editor(Some("Jill"), Some("NY"))
    val root = Root(people=Some(List(ed1, ed2)), days=Some(List("Mon", "Fri")))
    root.people should === (Some(List(ed1, ed2)))
    root.days should === (Some(List("Mon", "Fri")))
  }

  it should "build type alias's for simple definitions" in {
    @fromSchemaJson("""
    {
      "definitions" : {
        "SSN" : { "type": "string" },
        "Names" : { "type": "array", "items": { "type": "string" } }
      }
    }
    """)
    object Foo
    import Foo._

    implicitly[SSN =:= String]
    implicitly[Names =:= List[String]]
  }

  "Building Circe Codecs" should "encode case classes" in {
    @fromSchemaResource("/simple.json", jsonEng=Some(JsonEngs.Circe))
    object Foo
    import Foo._
    import Foo.Implicits._

    val address = Address(Some(31), Some("Main St"))
    val root = Root(Some(List("Bob", "Smith")), Some(26), Some(address), Some(123))

    root.asJson should beSameJsonAs ("""
      |{
      |  "name" : [ "Bob", "Smith" ],
      |  "age" : 26,
      |  "address" : {
      |    "number" : 31,
      |    "street" : "Main St"
      |  },
      |  "erdosNumber": 123
      |}
    """.stripMargin)
  }

  it should "decode case classes" in {
    @fromSchemaResource("/simple.json")
    object Foo
    import Foo._
    import Foo.Implicits._

    val json = """
      |{
      |  "name" : [ "Bob", "Smith" ],
      |  "age" : 26,
      |  "address" : {
      |    "number" : 31,
      |    "street" : "Main St"
      |  },
      |  "erdosNumber": 123
      |}
    """.stripMargin
    val root = parser.decode[Root](json).toOption.get

    root.name should === (Some(List("Bob", "Smith")))
    root.age should === (Some(26))
    root.address should === (Some(Address(Some(31), Some("Main St"))))
    root.erdosNumber === (Some(123))
  }

  it should "encode enum type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "country": { "enum": ["UK", "USA", "NZ"] }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._
    import io.circe.syntax._

    val root = Root(Some(Root.CountryEnums.NZ))
    root.asJson should beSameJsonAs ("""
      |{
      |  "country": "NZ"
      |}
    """.stripMargin)
  }

  it should "decode enum type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "country": { "enum": ["UK", "USA", "NZ"] }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val json =
      """
        |{
        |  "country": "NZ"
        |}
      """.stripMargin
    val root = parser.decode[Root](json).toOption.get

    root.country should === (Some(Root.CountryEnums.NZ))
  }

  it should "encode UUID type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "id": {
          "type": "string",
          "format": "uuid"
        }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._
    import io.circe.syntax._

    val uuid = "38400000-8cf0-11bd-b23e-10b96e4ef00d"
    val root = Root(Some(UUID.fromString(uuid)))
    root.asJson should beSameJsonAs (s"""
                                       |{
                                       |  "id": "$uuid"
                                       |}
                                     """.stripMargin)
  }

  it should "decode UUID type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "id": {
          "type": "string",
          "format": "uuid"
        }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val uuid = "38400000-8cf0-11bd-b23e-10b96e4ef00d"
    val json =
      s"""
        |{
        |  "id": "$uuid"
        |}
      """.stripMargin
    val root = parser.decode[Root](json).toOption.get

    root.id should === (Some(UUID.fromString(uuid)))
  }

  it should "encode ZonedDateTime type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "createdAt": {
          "type": "string",
          "format": "date-time"
        }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._
    import io.circe.syntax._

    val dateTime = "2017-01-01T10:00:00.001Z"
    val root = Root(Some(ZonedDateTime.parse(dateTime)))
    root.asJson should beSameJsonAs (s"""
                                        |{
                                        |  "createdAt": "$dateTime"
                                        |}
                                     """.stripMargin)
  }

  it should "decode ZonedDateTime type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "createdAt": {
          "type": "string",
          "format": "date-time"
        }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val dateTime = "2017-01-01T10:00:00.001Z"
    val json =
      s"""
         |{
         |  "createdAt": "$dateTime"
         |}
      """.stripMargin
    val root = parser.decode[Root](json).toOption.get

    root.createdAt should === (Some(ZonedDateTime.parse(dateTime)))
  }

  it should "return an error on decoding for datetime with wrong format" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "createdAt": {
          "type": "string",
          "format": "date-time"
        }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val dateTime = "wrongDateTime"
    val json =
      s"""
         |{
         |  "createdAt": "$dateTime"
         |}
      """.stripMargin
    val root = parser.decode[Root](json)

    root should be ('left)
    root.left.get match {
      case d: DecodingFailure => d.getMessage should === ("ZonedDateTime (Text 'wrongDateTime' could not be parsed at index 0): DownField(createdAt)")
      case e@_ => fail(s"Wrong error type: ${e.getClass.getName}")
    }
  }

  it should "return a DecodeFailure if it can't decode an enum type" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "country": { "enum": ["UK", "USA", "NZ"] }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val json =
      """
        |{
        |  "country": "oops"
        |}
      """.stripMargin
    parser.decode[Root](json).isLeft shouldBe (true)
  }

  it should "encode any wrappers" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "misc": { }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val values = Map("a" -> 1, "b" -> List(1.1, 2.2), "c" -> Map( "d" -> "bar", "e" -> 3.14 ))
    val root = Root(Some(Root.Misc(values)))

    root.asJson should beSameJsonAs("""
    {
      "misc": { "a": 1, "b": [1.1, 2.2], "c": { "d": "bar", "e": 3.14 } }
    }
    """)
  }

  it should "encode any wrappers with array types (which need special handling)" in {
      @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "misc": { }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val values = Array(Array("a", 1), Array(1), Array(2.2), Array(3L), Array(true), Array(3.toShort),
                       Array(3.0f), Array("a"), Array(2, 2.2))
    val root = Root(Some(Root.Misc(values)))

    root.asJson should beSameJsonAs("""
    {
      "misc": [ ["a", 1], [1], [2.2], [3], [true], [3], [3.0], ["a"], [2, 2.2] ]
    }
    """)
  }

  it should "decode any wrappers" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "misc": { }
      }
    }
    """)
    object Foo
    import Foo._
    import Foo.Implicits._

    val json =
      """
        |{
        |  "misc": { "a": 1, "b": [1, 2.0, "foo"], "c": { "d": "bar" } }
        |}
      """.stripMargin

    val root = parser.decode[Root](json).toOption.get
    val values = Map("a" -> 1, "b" -> List(1, 2.0, "foo"), "c" -> Map( "d" -> "bar" ))
    root.misc should === (Some(Root.Misc(values)))
  }

  it should "let you override encoders/decoders with higher priority implicits" in {
    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "name": { "type" : "string" }
      }
    }
    """)
    object Foo
    import Foo._

    object Implicits extends Foo.LowPriorityImplicits {
      implicit val betterEncoder: Encoder[Foo.Root] = Encoder.instance { r => "override".asJson }
      implicit val betterDecoder: Decoder[Foo.Root] = Decoder.instance((h: HCursor) =>
        Either.right(Root(name=Some("override")))
      )
    }
    import Implicits._

    val root = parser.decode[Root]("""{ "name": "fred" }""").toOption.get
    root.name should === (Some("override"))
    root.copy(name=Some("james")).asJson should beSameJsonAs("\"override\"")
  }

  "Params" should "support outPath and write out the generated code" in {
    @fromSchemaResource("/simple.json", outPath=Some("/tmp/Simple.scala"))
    object Simple

    val file = new File("/tmp/Simple.scala")
    file should exist

    val lines = Source.fromFile(file).getLines.toList
    lines.head should === ("object Simple {")
    lines.size should be >= 10
  }

  it should "support outPath with a package name and write out the generated code" in {
    @fromSchemaResource("/simple.json", outPath=Some("/tmp/SimplePackage.scala"), outPathPackage = Some("org.argus.simple"))
    object Simple

    val file = new File("/tmp/SimplePackage.scala")
    file should exist

    val lines = Source.fromFile(file).getLines.toList
    lines.head should === ("package org.argus.simple;")
    lines.size should be >= 10
  }

  it should "support name, and name the root element using it" in {
    @fromSchemaResource("/simple.json", name="Person")
    object Schema

    Schema.Person(age=Some(42)).age should === (Some(42))
  }

  it should "support raw schema inclusion" in {
    val expected = """
    {
      "definitions" : {
        "SSN" : { "type": "string" },
        "Names" : { "type": "array", "items": { "type": "string" } }
      }
    }
    """.filter(_ != '\n')

    @fromSchemaJson("""
    {
      "definitions" : {
        "SSN" : { "type": "string" },
        "Names" : { "type": "array", "items": { "type": "string" } }
      }
    }
    """, rawSchema = true)
    object Foo

    Foo.schemaSource should === (expected)
  }

  it should "generate HasSchemaSource instances" in {
    val expected = """
    {
      "type": "object",
      "properties": {
        "name": { "type" : "string" }
      }
    }
    """.filter(_ != '\n')

    @fromSchemaJson("""
    {
      "type": "object",
      "properties": {
        "name": { "type" : "string" }
      }
    }
    """, rawSchema = true, runtime = true)
    object Foo

    HasSchemaSource[Foo.Root].value should === (expected)
  }

  "Complex example" should "work end to end" in {
    @fromSchemaResource("/vega-lite-schema.json")
    object Vega
    import Vega._
    import Vega.Implicits._
    import io.circe.syntax._

    val json =
      """
        |{
        |  "description": "A bar chart showing the US population distribution of age groups and gender in 2000.",
        |  "data": { "values": [ {"a": 1, "b" : 2.0, "c": "NZ" }, {"a": 2, "b": 3.14, "c": "US" } ] },
        |  "transform": {
        |    "filter": "datum.year == 2000",
        |    "calculate": [{"field": "gender", "expr": "datum.sex == 2 ? \"Female\" : \"Male\""}]
        |  },
        |  "mark": "bar",
        |  "encoding": {
        |    "x": {
        |      "field": "age", "type": "ordinal",
        |      "scale": {"bandSize": 17}
        |    },
        |    "y": {
        |      "aggregate": "sum", "field": "people", "type": "quantitative",
        |      "axis": {"title": "population"}
        |    },
        |    "color": {
        |      "field": "gender", "type": "nominal",
        |      "scale": {"range": ["#e377c2","#1f77b4"]}
        |    }
        |  },
        |  "config": {
        |    "mark": {"opacity": 0.6, "stacked" : "none"}
        |  }
        |}
      """.stripMargin

    val res = parser.decode[RootUnion](json).toOption.get
    res.asJson should beSameJsonAs(json)
  }

}
