---
author: Lucas Duailibe (@duailibe)
authorURL: https://github.com/duailibe
title: "Prettier 1.17: More quotes options and support for shared configs"
---

This release brings long-requested flexibility to quotes around object properties, allows Prettier configuration to be shared in the form of packages, adds a [LWC] parser, adds support for new GraphQL syntax and fixes lots of formatting bugs.

[lwc]: https://developer.salesforce.com/docs/component-library/documentation/lwc

<!--truncate-->

## Highlights

### JavaScript

#### Add an option to modify when Prettier quotes object properties ([#5934] by [@azz])

**`--quote-props <as-needed|preserve|consistent>`**

`as-needed` **(default)** - Only add quotes around object properties where required. Current behaviour.
`preserve` - Respect the input. This is useful for users of Google's Closure Compiler in Advanced Mode, which treats quoted properties differently.
`consistent` - If _at least one_ property in an object requires quotes, quote all properties - this is like ESLint's [`consistent-as-needed`](https://eslint.org/docs/rules/quote-props) option.

<!-- prettier-ignore -->
```js
// Input
const headers = {
  accept: "application/json",
  "content-type": "application/json",
  "origin": "prettier.io"
};

// Output (Prettier 1.16 or --quote-props=as-needed)
const headers = {
  accept: "application/json",
  "content-type": "application/json",
  origin: "prettier.io"
};

// Output (--quote-props=consistent)
const headers = {
  "accept": "application/json",
  "content-type": "application/json",
  "origin": "prettier.io"
};

// Output (--quote-props=preserve)
const headers = {
  accept: "application/json",
  "content-type": "application/json",
  "origin": "prettier.io"
};
```

### Config

#### Support shared configurations ([#5963] by [@azz])

Sharing a Prettier configuration is simple: just publish a module that exports a configuration object, say `@company/prettier-config`, and reference it in your `package.json`:

```json
{
  "name": "my-cool-library",
  "version": "9000.0.1",
  "prettier": "@company/prettier-config"
}
```

If you don't want to use `package.json`, you can use any of the supported extensions to export a string, e.g. `.prettierrc.json`:

```json
"@company/prettier-config"
```

[@azz] has created an example configuration package. You can [see the source on GitHub](https://github.com/azz/prettier-config) or [install it via npm](https://www.npmjs.com/package/@azz/prettier-config).

> Note: This method does **not** offer a way to _extend_ the configuration to overwrite some properties from the shared configuration. If you need to do that, import the file in a `.prettierrc.js` file and export the modifications, e.g:
>
> ```js
> module.exports = {
>   ...require("@company/prettier-config"),
>   semi: false
> };
> ```

## General

### JavaScript

#### Respect newlines between parameters ([#5836] by [@evilebottnawi])

<!-- prettier-ignore -->
```js
// Input
function foo(
  one,

  two,
  three,
  four,


  five,
  six,
  seven,
  eight,
  nine,
  ten,

  eleven

) {}

// Output (Prettier 1.16)
function foo(
  one,
  two,
  three,
  four,
  five,
  six,
  seven,
  eight,
  nine,
  ten,
  eleven
) {}

// Output (Prettier 1.17)
function foo(
  one,

  two,
  three,
  four,

  five,
  six,
  seven,
  eight,
  nine,
  ten,

  eleven
) {}
```

#### Fix multiline dynamic import comments ([#6025] by [@noahsug])

<!-- prettier-ignore -->
```js
// Input
import(
  /* Hello */
  'something'
  /* Hello */
)
import(
  'myreallylongdynamicallyloadedmodulenamemyreallylongdynamicallyloadedmodulename'
)

// Output (Prettier stable)
import(/* Hello */
"something");
/* Hello */
import('myreallylongdynamicallyloadedmodulenamemyreallylongdynamicallyloadedmodulename');

// Output (Prettier master)
import(
  /* Hello */
  'something'
  /* Hello */
)
import(
  'myreallylongdynamicallyloadedmodulenamemyreallylongdynamicallyloadedmodulename'
);
```

#### Add parentheses for immediately-constructed functions and class ([#5996] by [@bakkot])

<!-- prettier-ignore -->
```js
// Input
new class {};
new function() {}

// Output (Prettier stable)
new class {}();
new function() {}();

// Output (Prettier master)
new (class {})();
new (function() {})();
```

#### Fix parens logic for optional chaining expressions and closure type casts ([#5843] by [@yangsu])

Logic introduced in #4542 will print parens in the wrong places and produce invalid code for optional chaining expressions (with more than 2 nodes) or closure type casts that end in function calls.

<!-- prettier-ignore -->
```js
// Input
(a?.b[c]).c();
let value = /** @type {string} */ (this.members[0]).functionCall();

// Output (Prettier stable)
a(?.b[c]).c();
let value = /** @type {string} */ this(.members[0]).functionCall();

// Output (Prettier master)
(a?.b[c]).c();
let value = /** @type {string} */ (this.members[0]).functionCall();
```

### Markdown

#### Do not align table contents if it exceeds the print width and `--prose-wrap never` is set ([#5701] by [@chenshuai2144])

The aligned table is less readable than the compact one if it's particularly long and the word wrapping is not enabled in the editor so we now print them as compact tables in these situations.

<!-- prettier-ignore -->
```md
<!-- Input -->
| Property | Description | Type | Default |
| -------- | ----------- | ---- | ------- |
| bordered | Toggles rendering of the border around the list | boolean | false |
| itemLayout | The layout of list, default is `horizontal`, If a vertical list is desired, set the itemLayout property to `vertical` | string | - |

<!-- Output (Prettier stable, --prose-wrap never) -->
| Property   | Description                                                                                                           | Type    | Default |
| ---------- | --------------------------------------------------------------------------------------------------------------------- | ------- | ------- |
| bordered   | Toggles rendering of the border around the list                                                                       | boolean | false   |
| itemLayout | The layout of list, default is `horizontal`, If a vertical list is desired, set the itemLayout property to `vertical` | string  | -       |

<!-- Output (Prettier master, --prose-wrap never) -->
| Property | Description | Type | Default |
| --- | --- | --- | --- |
| bordered | Toggles rendering of the border around the list | boolean | false |
| itemLayout | The layout of list, default is `horizontal`, If a vertical list is desired, set the itemLayout property to `vertical` | string | - |
```

### TypeScript

#### Support `readonly` operator ([#6027] by [@ikatyang])

<!-- prettier-ignore -->
```ts
// Input
declare const array: readonly number[];

// Output (Prettier stable)
// SyntaxError: ',' expected.

// Output (Prettier master)
declare const array: readonly number[];
```

### HTML

#### Add support for Lightning Web Components ([#5800] by [@ntotten])

Supports [Lightning Web Components (LWC)][lwc] template format for HTML attributes by adding a new parser called `lwc`.

<!-- prettier-ignore -->
```html
// Input
<my-element data-for={value}></my-element>

// Output (Prettier stable)
<my-element data-for="{value}"></my-element>

// Output (Prettier master)
<my-element data-for={value}></my-element>
```

#### Angular: Don't add unnecessary parentheses to pipes ([#5929] by [@voithos])

In some cases, wrapping parentheses were being added to certain pipes inside attributes, but they are no longer added when they don't affect the result of the expression.

<!-- prettier-ignore -->
```html
// Input
<div *ngIf="isRendered | async"></div>

// Output (Prettier stable)
<div *ngIf="(isRendered | async)"></div>

// Output (Prettier master)
<div *ngIf="isRendered | async"></div>
```

### GraphQL

#### Support variable directives ([#6020] by [@adek05])

Prettier now supports formatting variable directives.

<!-- prettier-ignore -->
```
// Input
query Q($variable: Int   @directive) {node}

// Output (Prettier stable)
query Q($variable: Int) {
  node
}

// Output (Prettier master)
query Q($variable: Int @directive) {
  node
}
```

#### Support GraphQL fragment variables ([#6016] by [@adek05])

Prettier now supports formatting fragment variables.

```
// Input
fragment F($var: Int) on Type { node }

// Output (Prettier stable)
// Fails to parse

// Output (Prettier master)
fragment F($var: Int) on Type {
   node
}
```

### CLI

#### Automatically discover scoped plugins ([#5945] by [@Kocal])

Prettier now supports automatically loading scoped plugins named `@scope-name/prettier-plugin-*`.

[@adek05]: https://github.com/adek05
[@azz]: https://github.com/azz
[@bakkot]: https://github.com/bakkot
[@chenshuai2144]: https://github.com/chenshuai2144
[@evilebottnawi]: https://github.com/evilebottnawi
[@ikatyang]: https://github.com/ikatyang
[@kocal]: https://github.com/Kocal
[@noahsug]: https://github.com/noahsug
[@ntotten]: https://github.com/ntotten
[@voithos]: https://github.com/voithos
[@yangsu]: https://github.com/yangsu
[#5836]: https://github.com/prettier/prettier/pull/5836
[#5701]: https://github.com/prettier/prettier/pull/5701
[#5800]: https://github.com/prettier/prettier/pull/5800
[#5843]: https://github.com/prettier/prettier/pull/5843
[#5929]: https://github.com/prettier/prettier/pull/5929
[#5934]: https://github.com/prettier/prettier/pull/5934
[#5945]: https://github.com/prettier/prettier/pull/5945
[#5963]: https://github.com/prettier/prettier/pull/5963
[#5996]: https://github.com/prettier/prettier/pull/5996
[#6016]: https://github.com/prettier/prettier/pull/6016
[#6020]: https://github.com/prettier/prettier/pull/6020
[#6025]: https://github.com/prettier/prettier/pull/6025
[#6027]: https://github.com/prettier/prettier/pull/6027
