# Modify Database and Transaction

中文版请见[这里](modify-database-and-transaction-cn.md)

In [Getting Start](getting-start.md), we have learned how to create the `Database` instance and define your database entity. Now,
we start learn how to write SQL statements with SQLlin.

## Insert

The class `Database` has overloaded function operator that type is `<T> Database.(Database.() -> T) -> T`. When you invoke
the operator function, it will produce a _DatabaseScope_. Yeah, that is an operator function's lambda parameter. Any SQL statement
must be written in _DatabaseScope_. And, the inner SQL statements will only be executed when the _DatabaseScope_ ended.

You already know, the _INSERT_, _DELETE_, _UPDATE_ and _SELECT_ SQL statements are used for table operation. So, before you write
your SQL statements, you also need to get a `Table` instance, like this:

```kotlin
private val database = Database(name = "Person.db", path = getGlobalPath(), version = 1)

fun sample() {
    database {
        PersonTable { table ->
            // Write your SQL statements...
        }
    }
}
```

The `PersonTable` is generated by _sqllin-processor_, because `Person` is annotated the `@DBRow`
annotation. Any class that be annotated the `@DBRow` will be generated a `Table` object, its name is
`class name + 'Table'`.

Now, let's do the real _INSERT_:

```kotlin
fun sample() {
    database {
        PersonTable { table ->
            table INSERT Person(age = 4, name = "Tom")
            table INSERT listOf(
                Person(age = 10, name = "Nick"),
                Person(age = 3, name = "Jerry"),
                Person(age = 8, name = "Jack"),
            )
        }
    }
}
```

The _INSERT_ statement could insert objects directly. You can insert one or multiple objects once.

## Delete

The _DELETE_ statement will be slightly more complex than _INSERT_. SQLlin doesn't delete objects like
[Jetpack Room](https://developer.android.com/training/data-storage/room), we use the _WHERE_ clause:

```kotlin
fun sample() {
    database {
        PersonTable { table ->
            table DELETE WHERE(age GTE 10 OR (name NEQ "Jerry"))
        }
    }
}
```

Let's understand the _WHERE_ clause. `WHERE` function receives a `ClauseCondiction` as a parameter. The `age` and `name` in the example is used for representing columns'
name, they are extension property with `Table`, their type are `ClauseElement`, and be generated by _sqllin-processor_(KSP).

The `ClauseElement` has a series of operators that used for representing the SQL operators like: `=`, `>`, `<`, `LIKE`, `IN`, `IS` etc.. When a `ClauseElement` invoke a
operator, we will get a `ClauseCondiction`. Multiple `ClauseCondiction`s would use the `AND` or `OR` operator link and produce a new `ClauseCondiction`.

The chart of between SQL operators and SQLlin operators like this:

|SQL|SQLlin|
|---|---|
|=|EQ|
|!= |NEQ|
|<|LT|
|<=|LTE|
|>|GT|
|>=|GTE|
|BETWEEN|BETWEEN|
|IN|IN|
|LIKE|LIKE|
|GLOB|GLOB|
|OR|OR|
|AND|AND|

Sometimes, we want to delete all data in the table. At this time, the _DELETE_ statement doesn't have _WHERE_ clause:

```SQL
DELETE FROM person
```

In SQLlin we can write this to achieve the same effect:

```kotlin
fun sample() {
    database {
        PersonTable { table ->
            table DELETE X
        }
    }
}
```
The `X` is a Kotlin `object`(singleton).

## Update

The _UPDATE_ statement is similar to _DELETE_, it also use a _WHERE_ clause to limit update conditions. But, the
difference is _UPDATE_ statement owns a particular _SET_ clause:

```kotlin
fun sample() {
    database {
        PersonTable { table ->
            table UPDATE SET { age = 5 } WHERE (name NEQ "Tom")
        }
    }
}
```

The _SET_ clause is different from other clauses, it receives a lambda as parameter, you can set the new value to column in the
lambda. The `age` in the set lambda is a writable property that also be generated by KSP, and it only available in _SET_
clause, it different with `age` readonly property in _WHERE_ clause.

You also could write the _UPDATE_ statement without _WHERE_ clause that used for update all rows, but you should use it with caution.

## Transaction

Use transaction is simple in SQLlin, you just need to use the `transaction {...}` wrap your SQL statements:

```kotlin
fun sample() {
    database {
        transaction {
            PersonTable { table ->
                table INSERT Person(age = 4, name = "Tom")
                table INSERT listOf(
                    Person(age = 10, name = "Nick"),
                    Person(age = 3, name = "Jerry"),
                    Person(age = 8, name = "Jack"),
                )
                table UPDATE SET { age = 5 } WHERE (name NEQ "Tom")
            }
        }
    }
}
```

The `transaction {...}` is a member function in `Database`, it inside or outside of `TABLE(databaseName) {...}` doesn't matter.

## Next Step

You have learned how to use _INSERT_, _DELETE_ and _UPDATE_ statements. Next step you will learn _SELECT_ statement. The
_SELECT_ statement is more complex than other statements, be ready :).

- [Query](query.md)
- [Concurrency Safety](concurrency-safety.md)
- [SQL Functions](sql-functions.md)
- [Advanced Query](advanced-query.md)