# Advanced Query

中文版请见[这里](advanced-query-cn.md)

We have learned basic query and using SQL functions in query condition. Let's learn some query’s advanced skills.

## Unions

The _UNION_ operator used for merge two _SELECT_ statements' results and these results must be of the same type.

In SQL, _UNION_ operator between with the two _SELECT_ statements, but in SQLlin, we use a higher-order function to
implement _UNION_:

```kotlin
fun sample() {
    lateinit var selectStatement: SelectStatement<Person>
    database {
        PersonTable { table ->
            selectStatement = UNION {
                table SELECT WHERE (age GTE 5)
                table SELECT WHERE (length(name) LTE 8)
            }
        }
    }
}
```

You just need to write your _SELECT_ statements in `UNION {...}` block. There must be at least two _SELECT_ statements
inside the `UNION {...}` block, if not, you will get a `IllegalStateException` when runtime.

If you want to use _UNION_ and _UNION ALL_ interchangeably, just use `UNION {...}` or `UNION_ALL {...}` block nesting:

```kotlin
fun sample() {
    lateinit var selectStatement: SelectStatement<Person>
    database {
        PersonTable { table ->
            selectStatement = UNION {
                table SELECT WHERE (age GTE 5)
                UNION_ALL {
                    table SELECT WHERE (length(name) LTE 8)
                    table SELECT WHERE (name EQ "Tom")
                }
            }
        }
    }
}
```

Above code equals the SQL:

```roomsql
SELECT * FROM person WHERE age >= 5
UNION
SELECT * FROM person WHERE length(name) <= 8
UNION ALL
SELECT * FROM person WHERE name = "Tom"
```

## Subqueries

SQLlin doesn't yet support subqueries, we will develop as soon as possible.

## Join

SQLlin supports joining tables now.

We need other two database entities:

```kotlin
@DBRow("transcript")
@Serializable
data class Transcript(
    val name: String?,
    val math: Int,
    val english: Int,
)

@Serializable
data class Student(
    val name: String?,
    val age: Int?,
    val math: Int,
    val english: Int,
)

@Serializable
data class CrossJoinStudent(
    val age: Int?,
    val math: Int,
    val english: Int,
)
```

The `Transcript` represents a other table. And the `Student` represents the join query results' type(so `Student`
doesn't need to be annotated `@DBRow`), it owns all column names that belong to `Person` and `Transcript`.

### Cross Join

```kotlin
fun joinSample() {
    db {
        PersonTable { table ->
            table SELECT CROSS_JOIN<CrossJoinStudent>(TranscriptTable)
        }
    }
}
```

The `CROSS_JOIN` function receives one or multiple `Table`s as parameters. In normal _SELECT_ statements, the statements' querying results' type is
depended on the `Table` that be generated by _sqllin-processor_, but _JOIN_ operator will change it to specific type. In above sample, `CROSS_JOIN` changes
the type to `CrossJoinStudent`.

Note, because of _CROSS JOIN_ owns feature in SQL. If the columns that be queried by _SELECT_ statement that with _CROSS JOIN_ clause include the same
name column in the two tables, this will causing the query to fail. Because of a class isn't allowed to have multiple properties those have same name, _sqllin-dsl_
doesn't support the _CROSS JOIN_ with columns of the same name.

### Inner Join

```kotlin
fun joinSample() {
    db {
        PersonTable { table ->
            table SELECT INNER_JOIN<Student>(TranscriptTable) USING name
            table SELECT NATURAL_INNER_JOIN<Student>(TranscriptTable)
            table SELECT INNER_JOIN<CrossJoinStudent>(TranscriptTable) ON (name EQ TranscriptTable.name)
        }
    }
}
```

The `INNER_JOIN` is similar to `CROSS_JOIN`, the deference is `INNER_JOIN` need to connect a `USING` or `ON` clause. If a _INNER JOIN_ statement
without the `USING` or `ON` clause, it is incomplete, but your code still be compiled and will do nothing in runtime.

The `NATURAL_INNER_JOIN` will produce a complete _SELECT_ statement(the same with `CROSS_JOIN`). So, you can't add `USING` or `ON` clause behind it, this is
guaranteed by Kotlin compiler.

Note, the behavior of `INNER_JOIN` clause with `ON` clause is same to `CROSS_JOIN`, you can't select the column that has same name in two tables.

The `INNER_JOIN` have an alias that named `JOIN`, and `NATURAL_INNER_JOIN` also have an alias that named `NATURAL_JOIN`. That's liked you can
bypass the `INNER` keyword in SQL's inner join query.


### Left Outer Join

```kotlin
fun joinSample() {
    db {
        PersonTable { table ->
            table SELECT LEFT_OUTER_JOIN<Student>(TranscriptTable) USING name
            table SELECT NATURAL_LEFT_OUTER_JOIN<Student>(TranscriptTable)
            table SELECT LEFT_OUTER_JOIN<CrossJoinStudent>(TranscriptTable) ON (name EQ TranscriptTable.name)
        }
    }
}
```

The `LEFT_OUTER_JOIN`'s usage is very similar to `INNER_JOIN`, the difference just is their API names.

## Finally

You have learned all usage with SQLlin, enjoy it and stay concerned about SQLlin's update :)