use crate::keywords::{
    A_NPLUS_BNOTATION_PSEUDO_CLASSES, A_NPLUS_BOF_SNOTATION_PSEUDO_CLASSES,
    AT_RULE_PAGE_PSEUDO_CLASSES, CSS_MODULE_PSEUDO_CLASSES, HTML_TAGS, KNOWN_CHROME_PROPERTIES,
    KNOWN_EDGE_PROPERTIES, KNOWN_EXPLORER_PROPERTIES, KNOWN_FIREFOX_PROPERTIES, KNOWN_PROPERTIES,
    KNOWN_SAFARI_PROPERTIES, KNOWN_SAMSUNG_INTERNET_PROPERTIES, KNOWN_US_BROWSER_PROPERTIES,
    LEVEL_ONE_AND_TWO_PSEUDO_ELEMENTS, LINGUISTIC_PSEUDO_CLASSES,
    LOGICAL_COMBINATIONS_PSEUDO_CLASSES, LONGHAND_SUB_PROPERTIES_OF_SHORTHAND_PROPERTIES,
    MATH_ML_TAGS, MEDIA_FEATURE_NAMES, OTHER_PSEUDO_CLASSES, OTHER_PSEUDO_ELEMENTS,
    RESET_TO_INITIAL_PROPERTIES_BY_BORDER, RESET_TO_INITIAL_PROPERTIES_BY_FONT,
    RESOURCE_STATE_PSEUDO_CLASSES, SHADOW_TREE_PSEUDO_ELEMENTS, SHORTHAND_PROPERTIES, SVG_TAGS,
    VENDOR_PREFIXES, VENDOR_SPECIFIC_PSEUDO_ELEMENTS,
};

use biome_string_case::{StrLikeExtension, StrOnlyExtension};

pub fn is_css_variable(value: &str) -> bool {
    value.to_ascii_lowercase_cow().starts_with("var(")
}

/// Check if the value is a double-dashed custom function.
pub fn is_custom_function(value: &str) -> bool {
    value.starts_with("--")
}

// Returns the vendor prefix extracted from an input string.
pub fn vender_prefix(prop: &str) -> &'static str {
    for prefix in VENDOR_PREFIXES.iter() {
        if prop.starts_with(prefix) {
            return prefix;
        }
    }
    ""
}

pub fn is_pseudo_elements(prop: &str) -> bool {
    LEVEL_ONE_AND_TWO_PSEUDO_ELEMENTS
        .binary_search(&prop)
        .is_ok()
        || VENDOR_SPECIFIC_PSEUDO_ELEMENTS.binary_search(&prop).is_ok()
        || SHADOW_TREE_PSEUDO_ELEMENTS.binary_search(&prop).is_ok()
        || OTHER_PSEUDO_ELEMENTS.binary_search(&prop).is_ok()
}

/// Check if the input string is custom selector
/// See https://drafts.csswg.org/css-extensions/#custom-selectors for more details
pub fn is_custom_selector(prop: &str) -> bool {
    prop.starts_with("--")
}

pub fn is_page_pseudo_class(prop: &str) -> bool {
    AT_RULE_PAGE_PSEUDO_CLASSES.binary_search(&prop).is_ok()
}

pub fn is_known_pseudo_class(prop: &str) -> bool {
    LEVEL_ONE_AND_TWO_PSEUDO_ELEMENTS
        .binary_search(&prop)
        .is_ok()
        || A_NPLUS_BNOTATION_PSEUDO_CLASSES
            .binary_search(&prop)
            .is_ok()
        || A_NPLUS_BOF_SNOTATION_PSEUDO_CLASSES
            .binary_search(&prop)
            .is_ok()
        || LINGUISTIC_PSEUDO_CLASSES.binary_search(&prop).is_ok()
        || LOGICAL_COMBINATIONS_PSEUDO_CLASSES
            .binary_search(&prop)
            .is_ok()
        || RESOURCE_STATE_PSEUDO_CLASSES.binary_search(&prop).is_ok()
        || OTHER_PSEUDO_CLASSES.binary_search(&prop).is_ok()
}

pub fn is_known_properties(prop: &str) -> bool {
    KNOWN_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_CHROME_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_EDGE_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_EXPLORER_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_FIREFOX_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_SAFARI_PROPERTIES.binary_search(&prop).is_ok()
        || KNOWN_SAMSUNG_INTERNET_PROPERTIES
            .binary_search(&prop)
            .is_ok()
        || KNOWN_US_BROWSER_PROPERTIES.binary_search(&prop).is_ok()
}

pub fn vendor_prefixed(props: &str) -> bool {
    props.starts_with("-webkit-")
        || props.starts_with("-moz-")
        || props.starts_with("-ms-")
        || props.starts_with("-o-")
}

/// Check if the input string is a media feature name.
pub fn is_media_feature_name(prop: &str) -> bool {
    let input = prop.to_ascii_lowercase_cow();
    let count = MEDIA_FEATURE_NAMES.binary_search(&input.as_ref());
    if count.is_ok() {
        return true;
    }
    let mut has_vendor_prefix = false;
    for prefix in VENDOR_PREFIXES.iter() {
        if input.starts_with(prefix) {
            has_vendor_prefix = true;
            break;
        }
    }
    if has_vendor_prefix {
        for feature_name in MEDIA_FEATURE_NAMES.iter() {
            if input.ends_with(feature_name) {
                return true;
            }
        }
    }
    false
}

pub fn get_longhand_sub_properties(shorthand_property: &str) -> &'static [&'static str] {
    if let Ok(index) = SHORTHAND_PROPERTIES.binary_search(&shorthand_property) {
        return LONGHAND_SUB_PROPERTIES_OF_SHORTHAND_PROPERTIES[index];
    }

    &[]
}

pub fn get_reset_to_initial_properties(shorthand_property: &str) -> &'static [&'static str] {
    match shorthand_property {
        "border" => &RESET_TO_INITIAL_PROPERTIES_BY_BORDER,
        "font" => &RESET_TO_INITIAL_PROPERTIES_BY_FONT,
        _ => &[],
    }
}

fn is_custom_element(prop: &str) -> bool {
    prop.contains('-') && prop.eq(prop.to_lowercase_cow().as_ref())
}

/// Check if the input string is a known type selector.
pub fn is_known_type_selector(prop: &str) -> bool {
    let input = prop.to_ascii_lowercase_cow();
    HTML_TAGS.binary_search(&input.as_ref()).is_ok()
        || SVG_TAGS.binary_search(&prop).is_ok()
        || MATH_ML_TAGS.binary_search(&input.as_ref()).is_ok()
        || is_custom_element(prop)
}

/// Check if the input string is a CSS module pseudo-class.
///
/// CSS modules support special pseudo-classes like `:global` and `:local` for
/// scoping control.
/// These are only valid when CSS modules are enabled.
///
/// See https://github.com/css-modules/css-modules for more details.
pub fn is_css_module_pseudo_class(prop: &str) -> bool {
    CSS_MODULE_PSEUDO_CLASSES.binary_search(&prop).is_ok()
}
