const stamps = require('../stamps');

exports.up = function (knex) {
  return Promise.resolve()
    // Create queue schema and object_queue table
    .then(() => knex.schema.raw('CREATE SCHEMA IF NOT EXISTS queue'))

    .then(() => knex.schema.withSchema('queue').createTable('object_queue', table => {
      table.specificType(
        'id',
        'integer GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY'
      );
      table.uuid('bucketId', 255);
      table.string('path', 1024).notNullable();
      table.boolean('full').notNullable().defaultTo(false);
      table.integer('retries').notNullable().defaultTo(0);
      stamps(knex, table);

      // Add two partial indexes to ensure null bucketIds are also unique constrained
      // @see {@link https://stackoverflow.com/a/8289253}
      table.unique(['bucketId', 'path'], { predicate: knex.whereNotNull('bucketId') });
      table.unique('path', { predicate: knex.whereNull('bucketId') });
    }))

    // Add index to object path column
    .then(() => knex.schema.alterTable('object', table => {
      table.string('path', 1024).index().notNullable().alter();
    }))

    // Add isLatest column to version table
    .then(() => knex.schema.alterTable('version', table => {
      table.boolean('isLatest').notNullable().defaultTo(false);
    }))

    // Add notNullable to tag columns
    .then(() => knex.schema.alterTable('tag', table => {
      table.string('key', 128).notNullable().alter();
      table.string('value', 256).notNullable().alter();
    }));
};

exports.down = function (knex) {
  return Promise.resolve()
    // Drop notNullable from tag columns
    .then(() => knex.schema.alterTable('tag', table => {
      table.string('key', 128).alter();
      table.string('value', 256).alter();
    }))

    // Drop isLatest column in version table
    .then(() => knex.schema.alterTable('version', table => {
      table.dropColumn('isLatest');
    }))

    // Drop index from object path column
    .then(() => knex.schema.alterTable('object', table => {
      table.dropIndex('path');
    }))

    // Drop queue schema and object_queue table
    .then(() => knex.schema.withSchema('queue').dropTableIfExists('object_queue'))
    .then(() => knex.schema.dropSchemaIfExists('queue'));
};
