var expect = require('chai').expect;
var vmssWindowsAntiMalwareExt = require('./vmssWindowsAntiMalwareExt');

const virtualMachineScaleSets = [
    {
        'name': 'test-vmss',
        'id': '/subscriptions/123/resourceGroups/test-rg/providers/Microsoft.Compute/virtualMachineScaleSets/test-vmss',
        'type': 'Microsoft.Compute/virtualMachineScaleSets',
        'virtualMachineProfile': {
            "storageProfile": {
                "osDisk": {
                "osType": "windows",
                },
            },
            'extensionProfile': {
                'extensions': [
                    {
                        'name': 'iaasantimalware',
                        'properties': {
                            'autoUpgradeMinorVersion': false,
                            'publisher': 'Microsoft.ManagedServices',
                            'type': 'AADSSHLoginForLinux',
                            'typeHandlerVersion': '1.0',
                        }
                    }
                ]
            }
        }
    },
    {
        'name': 'test-vmss',
        'id': '/subscriptions/123/resourceGroups/test-rg/providers/Microsoft.Compute/virtualMachineScaleSets/test-vmss',
        'type': 'Microsoft.Compute/virtualMachineScaleSets',
        'virtualMachineProfile': {
            "storageProfile": {
                "osDisk": {
                "osType": "windows",
                },
            },
            'extensionProfile': {
                'extensions': [
                    {
                        'name': 'AADLoginForWindows',
                        'properties': {
                            'autoUpgradeMinorVersion': false,
                            'publisher': 'Microsoft.ManagedServices',
                            'type': 'AADLoginForWindows',
                            'typeHandlerVersion': '1.0',
                        }
                    }
                ]
            }
        }
    },
    {
        'name': 'test-vmss',
        'id': '/subscriptions/123/resourceGroups/test-rg/providers/Microsoft.Compute/virtualMachineScaleSets/test-vmss',
        'type': 'Microsoft.Compute/virtualMachineScaleSets',
        'virtualMachineProfile': {
            "storageProfile": {
                "osDisk": {
                "osType": "windows",
                },
            },
            'extensionProfile': {
                'extensions': []
            }
        }
    }
];

const createCache = (virtualMachineScaleSets) => {
    let machine = {};
    if (virtualMachineScaleSets) {
        machine['data'] = virtualMachineScaleSets;
    }
    return {
        virtualMachineScaleSets: {
            listAll: {
                'eastus': machine
            }
        }
    };
};

describe('vmssWindowsAntiMalwareExt', function() {
    describe('run', function() {
        it('should give passing result if no virtual machine scale sets', function(done) {
            const cache = createCache([]);
            vmssWindowsAntiMalwareExt.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('No existing Virtual Machine Scale Sets found');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give unknown result if unable to query for virtual machine scale sets', function(done) {
            const cache = createCache();
            vmssWindowsAntiMalwareExt.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(3);
                expect(results[0].message).to.include('Unable to query for Virtual Machine Scale Sets');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give passing result if windows Virtual Machine Scale Set has anti malware extension installed', function(done) {
            const cache = createCache([virtualMachineScaleSets[0]]);
            vmssWindowsAntiMalwareExt.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('Windows Virtual Machine Scale Set has IaaS Antimalware extension installed');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });
        it('should give failing result if Virtual Machine Scale Set does not have anti malware extension installed', function(done) {
            const cache = createCache([virtualMachineScaleSets[1]]);
            vmssWindowsAntiMalwareExt.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('Windows Virtual Machine Scale Set does not have IaaS Antimalware extension installed');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });
        it('should give failing result if no extensions installed', function(done) {
            const cache = createCache([virtualMachineScaleSets[2]]);
            vmssWindowsAntiMalwareExt.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('No VMSS Extensions found');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });
    });
});