# Copyright (C) 2022 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import pytest
from test_util import *
from access_constraints.access_constraint_helper import *
from assert_extension import *
from assertion_harness import *
from display_times_helper import *
from interfaces.stk_objects import *
from logger import *
from math2 import *
from vehicle.vehicle_gfx import *
from vehicle.vehicle_vo import *
from ansys.stk.core.utilities.colors import *
from ansys.stk.core.stkobjects import *


class EarlyBoundTests(TestBase):
    def __init__(self, *args, **kwargs):
        super(EarlyBoundTests, self).__init__(*args, **kwargs)

    basicColorRed: Color = Colors.from_argb(255, 255, 0, 0)
    labelColorRed: Color = Colors.from_argb(255, 255, 0, 0)
    labelColorRedBlue: Color = Colors.from_argb(255, 255, 0, 128)
    labelColorGreenBlue: Color = Colors.from_argb(255, 0, 255, 128)
    centroidColorRed: Color = Colors.from_argb(255, 255, 0, 0)
    boundaryColorRedGreen: Color = Colors.from_argb(255, 255, 128, 0)
    centroidColorRedBlue: Color = Colors.from_argb(255, 255, 0, 255)

    # region OneTimeSetUp
    @staticmethod
    def setUpClass():
        TestBase.Initialize()

        TestBase.LoadTestScenario(Path.Combine("AreaTargetTests", "AreaTargetTests.sc"))
        EarlyBoundTests.AG_AT = AreaTarget(TestBase.Application.current_scenario.children["AreaTarget1"])

    # endregion

    # region OneTimeTearDown
    @staticmethod
    def tearDownClass():
        EarlyBoundTests.AG_AT = None
        TestBase.Uninitialize()

    # endregion

    # region Static DataMembers
    AG_AT: "AreaTarget" = None
    # endregion

    # region CommonTasks
    def test_CommonTasks(self):
        EarlyBoundTests.AG_AT.common_tasks.set_area_type_ellipse(1, 2, 12)
        Assert.assertEqual(AreaType.ELLIPSE, EarlyBoundTests.AG_AT.area_type)
        ellipse: "AreaTypeEllipse" = clr.CastAs(EarlyBoundTests.AG_AT.area_type_data, AreaTypeEllipse)
        Assert.assertEqual(1, ellipse.semi_major_axis)
        Assert.assertEqual(2, ellipse.semi_minor_axis)
        Assert.assertEqual(12, ellipse.bearing)

        latlons = [[0, 0], [1, 0]]
        patterns: "AreaTypePatternCollection" = EarlyBoundTests.AG_AT.common_tasks.set_area_type_pattern(latlons)
        Assert.assertEqual(2, patterns.count)
        patterns.remove_all()
        Assert.assertEqual(0, patterns.count)

    # endregion

    # region Basic
    @category("Basic Tests")
    def test_Basic(self):
        TestBase.logger.WriteLine("----- BASIC TEST ----- BEGIN -----")
        EarlyBoundTests.AG_AT.area_type = AreaType.ELLIPSE
        Assert.assertEqual(AreaType.ELLIPSE, EarlyBoundTests.AG_AT.area_type)
        ellipse: "AreaTypeEllipse" = AreaTypeEllipse(EarlyBoundTests.AG_AT.area_type_data)
        ellipse.bearing = 1
        Assert.assertEqual(1, ellipse.bearing)
        ellipse.semi_major_axis = 301
        Assert.assertEqual(301, ellipse.semi_major_axis)
        ellipse.semi_minor_axis = 151
        Assert.assertEqual(151, ellipse.semi_minor_axis)

        EarlyBoundTests.AG_AT.area_type = AreaType.PATTERN
        Assert.assertEqual(AreaType.PATTERN, EarlyBoundTests.AG_AT.area_type)
        patterns: "AreaTypePatternCollection" = AreaTypePatternCollection(EarlyBoundTests.AG_AT.area_type_data)
        Assert.assertIsNotNone(patterns)
        self.Units.set_current_unit("LongitudeUnit", "deg")
        self.Units.set_current_unit("LatitudeUnit", "deg")
        TestBase.logger.WriteLine3("The current Area Type Pattern collection contains: {0} elements", patterns.count)
        patterns.add(1, 2)
        TestBase.logger.WriteLine3("The new Area Type Pattern collection contains: {0} elements", patterns.count)
        TestBase.logger.WriteLine7(
            "Element 0: Lattitude = {0}, Longitude = {1}", patterns[0].latitude, patterns[0].longitude
        )
        idx: int = 1
        pattern: "AreaTypePattern"
        for pattern in patterns:
            Assert.assertIsNotNone(pattern)
            pattern.latitude = (idx * idx) / 100
            Assert.assertEqual(((idx * idx) / 100), pattern.latitude)
            pattern.longitude = idx
            Assert.assertEqual(idx, pattern.longitude)
            idx += 1

        size: int = patterns.count
        pattern2: "AreaTypePattern" = patterns.add(0.02, 0.02)
        Assert.assertEqual((size + 1), patterns.count)
        patterns.remove(size)
        Assert.assertEqual(size, patterns.count)
        patterns.remove_all()
        Assert.assertEqual(0, patterns.count)

        patterns.add(1, 2)
        patterns.add(2, 3)
        patterns.add(4, 5)
        patterns.add(5, 6)
        TestBase.logger.WriteLine3("The new Area Type Pattern collection contains: {0} elements", patterns.count)

        iIndex: int = 0
        while iIndex < patterns.count:
            TestBase.logger.WriteLine8(
                "\tElement {0}: Lat = {1}, Lon = {2}", iIndex, patterns[iIndex].latitude, patterns[iIndex].longitude
            )

            iIndex += 1

        pattern3: "AreaTypePattern" = patterns.insert(3, 4, 2)
        Assert.assertEqual(5, patterns.count)
        Assert.assertEqual(pattern3.latitude, 3)
        Assert.assertEqual(pattern3.longitude, 4)

        pattern4: "AreaTypePattern" = patterns[2]
        Assert.assertEqual(pattern4.latitude, 3)
        Assert.assertEqual(pattern4.longitude, 4)

        TestBase.logger.WriteLine3("The new Area Type Pattern collection contains: {0} elements", patterns.count)

        iIndex: int = 0
        while iIndex < patterns.count:
            TestBase.logger.WriteLine8(
                "\tElement {0}: Lat = {1}, Lon = {2}", iIndex, patterns[iIndex].latitude, patterns[iIndex].longitude
            )

            iIndex += 1

        latlons = patterns.to_array()

        i: int = 0
        while i <= (len(latlons) - 1):
            TestBase.logger.WriteLine6("Latitude = {0}", latlons[i][0])
            TestBase.logger.WriteLine6("Longitude = {0}", latlons[i][1])

            i += 1

        patterns.remove_all()
        Assert.assertEqual(0, patterns.count)

        EarlyBoundTests.AG_AT.automatic_computation_of_centroid = False
        Assert.assertFalse(EarlyBoundTests.AG_AT.automatic_computation_of_centroid)
        EarlyBoundTests.AG_AT.use_local_time_offset = True
        Assert.assertTrue(EarlyBoundTests.AG_AT.use_local_time_offset)
        EarlyBoundTests.AG_AT.use_terrain_data = False
        Assert.assertFalse(EarlyBoundTests.AG_AT.use_terrain_data)
        EarlyBoundTests.AG_AT.local_time_offset = 2
        Assert.assertEqual(2, EarlyBoundTests.AG_AT.local_time_offset)

        bIsAllowed: bool = EarlyBoundTests.AG_AT.allow_object_access
        EarlyBoundTests.AG_AT.allow_object_access = True
        Assert.assertTrue(EarlyBoundTests.AG_AT.allow_object_access)
        EarlyBoundTests.AG_AT.allow_object_access = False
        Assert.assertFalse(EarlyBoundTests.AG_AT.allow_object_access)
        EarlyBoundTests.AG_AT.allow_object_access = bIsAllowed
        Assert.assertEqual(bIsAllowed, EarlyBoundTests.AG_AT.allow_object_access)

        pl: "ISTKObject" = TestBase.Application.current_scenario.children["JupiterAnalytic"]
        fa: "Facility" = Facility(TestBase.Application.current_scenario.children["Facility1"])
        Assert.assertIsNotNone(pl)
        areaTargetObject: "ISTKObject" = clr.CastAs(EarlyBoundTests.AG_AT, ISTKObject)
        if areaTargetObject.is_access_supported():
            areaTargetObject.get_access_to_object(clr.CastAs(fa, ISTKObject)).compute_access()
            areaTargetObject.get_access("*/Planet/Planet1").compute_access()
            areaTargetObject.get_access(pl.path).compute_access()

            dpc: "DataProviderCollection" = areaTargetObject.get_access(pl.path).data_providers
            TestBase.logger.WriteLine3("The Data Provider collection contains: {0} elements", dpc.count)

            i: int = 0
            while i < dpc.count:
                TestBase.logger.WriteLine8("Element {0}:\tType = {1},\tName = {2}", i, dpc[i].type, dpc[i].name)

                i += 1

            if areaTargetObject.is_object_coverage_supported():
                cov: "ObjectCoverage" = areaTargetObject.object_coverage
                TestBase.logger.WriteLine(cov.data_providers[0].name)

            (ISTKObject(EarlyBoundTests.AG_AT)).get_access_to_object(clr.CastAs(fa, ISTKObject)).remove_access()
            (ISTKObject(EarlyBoundTests.AG_AT)).get_access("*/Planet/Planet1").remove_access()
            (ISTKObject(EarlyBoundTests.AG_AT)).get_access(pl.path).remove_access()

        st: "ISTKObject" = TestBase.Application.current_scenario.children["Star1"]
        Assert.assertIsNotNone(st)
        areaTargetObject.get_access(st.path)
        areaTargetObject.get_access_to_object(st)
        st.get_access(areaTargetObject.path)
        st.get_access_to_object(clr.CastAs(EarlyBoundTests.AG_AT, ISTKObject))

        with pytest.raises(Exception):
            areaTargetObject.get_access(TestBase.Application.current_scenario.path)

        TestBase.logger.WriteLine("----- BASIC TEST ----- END -----")

    # endregion

    # region Graphics
    @category("Graphics Tests")
    def test_Graphics(self):
        TestBase.logger.WriteLine("----- GRAPHICS TEST ----- BEGIN -----")
        gfx: "AreaTargetGraphics" = EarlyBoundTests.AG_AT.graphics
        gfx.show_graphics = False
        Assert.assertFalse(gfx.show_graphics)
        gfx.show_graphics = True
        Assert.assertTrue(gfx.show_graphics)
        gfx.inherit = False
        Assert.assertFalse(gfx.inherit)
        gfx.color = EarlyBoundTests.basicColorRed
        AssertEx.AreEqual(EarlyBoundTests.basicColorRed, gfx.color)
        gfx.centroid_color = EarlyBoundTests.centroidColorRed
        AssertEx.AreEqual(EarlyBoundTests.centroidColorRed, gfx.centroid_color)
        gfx.label_color = EarlyBoundTests.labelColorRed
        AssertEx.AreEqual(EarlyBoundTests.labelColorRed, gfx.label_color)
        gfx.boundary_color = EarlyBoundTests.boundaryColorRedGreen
        AssertEx.AreEqual(EarlyBoundTests.boundaryColorRedGreen, gfx.boundary_color)
        gfx.boundary_fill = True
        Assert.assertTrue(gfx.boundary_fill)
        gfx.boundary_color = EarlyBoundTests.boundaryColorRedGreen
        AssertEx.AreEqual(EarlyBoundTests.boundaryColorRedGreen, gfx.boundary_color)
        gfx.show_boundary_points = True
        Assert.assertTrue(gfx.show_boundary_points)
        gfx.boundary_style = LineStyle.DASH_DOT_DOTTED
        Assert.assertEqual(LineStyle.DASH_DOT_DOTTED, gfx.boundary_style)
        gfx.boundary_style = LineStyle.DOTTED
        Assert.assertEqual(LineStyle.DOTTED, gfx.boundary_style)
        gfx.show_boundary = True
        Assert.assertTrue(gfx.show_boundary)
        gfx.boundary_width = 2
        Assert.assertEqual(2, gfx.boundary_width)
        gfx.show_bounding_rectangle = True
        Assert.assertTrue(gfx.show_bounding_rectangle)
        gfx.show_centroid = True
        Assert.assertTrue(gfx.show_centroid)
        gfx.centroid_color = EarlyBoundTests.centroidColorRedBlue
        AssertEx.AreEqual(EarlyBoundTests.centroidColorRedBlue, gfx.centroid_color)
        gfx.show_label = True
        Assert.assertTrue(gfx.show_label)
        gfx.label_color = EarlyBoundTests.labelColorGreenBlue
        AssertEx.AreEqual(EarlyBoundTests.labelColorGreenBlue, gfx.label_color)
        gfx.marker_style = "Star"
        Assert.assertEqual("Star", gfx.marker_style)
        gfx.label_name = "My target"
        Assert.assertEqual("My target", gfx.label_name)
        gfx.use_instance_name_label = True
        Assert.assertTrue(gfx.use_instance_name_label)
        Assert.assertEqual("AreaTarget1", gfx.label_name)

        TestBase.Application.load_custom_marker(TestBase.GetScenarioFile("gp_marker.bmp"))
        gfx.marker_style = TestBase.GetScenarioFile("gp_marker.bmp")

        gfx.boundary_fill_percent_translucency = 55.0
        Assert.assertAlmostEqual(55.0, gfx.boundary_fill_percent_translucency, delta=Math2.Epsilon12)

        oHelper = GfxLabelNoteHelper(self.Units)
        oHelper.Run(gfx.label_notes)
        TestBase.logger.WriteLine("----- GRAPHICS TEST ----- END -----")

    # endregion

    # region VOBorderWall
    @category("VO Tests")
    def test_VOBorderWall(self):
        TestBase.logger.WriteLine("----- THE VO BORDER WALL TEST ----- BEGIN -----")
        oHelper = VOBorderWallHelper(self.Units)
        oHelper.Run(EarlyBoundTests.AG_AT.graphics_3d.border_wall, False)
        TestBase.logger.WriteLine("----- THE VO BORDER WALL TEST ----- END -----")

    # endregion

    # region VOVectors
    @category("VO Tests")
    def test_VOVectors(self):
        oHelper = VOVectorsHelper(self.Units, TestBase.Application)
        oHelper.Run(EarlyBoundTests.AG_AT.graphics_3d.vector, True)

    # endregion

    # region VO
    @category("VO Tests")
    def test_VO(self):
        TestBase.logger.WriteLine("----- THE VO TEST ----- BEGIN -----")
        vo: "AreaTargetGraphics3D" = EarlyBoundTests.AG_AT.graphics_3d
        Assert.assertIsNotNone(vo)
        # set DistanceUnit
        TestBase.logger.WriteLine5(
            "\tThe current DistanceUnit format is: {0}", self.Units.get_current_unit_abbrv("DistanceUnit")
        )
        self.Units.set_current_unit("DistanceUnit", "km")
        TestBase.logger.WriteLine5(
            "\tThe new DistanceUnit format is: {0}", self.Units.get_current_unit_abbrv("DistanceUnit")
        )
        Assert.assertEqual("km", self.Units.get_current_unit_abbrv("DistanceUnit"))
        # set AngleUnit
        TestBase.logger.WriteLine5(
            "\tThe current AngleUnit format is: {0}", self.Units.get_current_unit_abbrv("AngleUnit")
        )
        self.Units.set_current_unit("AngleUnit", "deg")
        TestBase.logger.WriteLine5("\tThe new AngleUnit format is: {0}", self.Units.get_current_unit_abbrv("AngleUnit"))
        Assert.assertEqual("deg", self.Units.get_current_unit_abbrv("AngleUnit"))
        # EnableLabelMaxViewingDist (false)
        TestBase.logger.WriteLine4("\tThe current EnableLabelMaxViewingDist is: {0}", vo.enable_label_max_viewing_dist)
        vo.enable_label_max_viewing_dist = False
        TestBase.logger.WriteLine4("\tThe new EnableLabelMaxViewingDist is: {0}", vo.enable_label_max_viewing_dist)
        Assert.assertFalse(vo.enable_label_max_viewing_dist)
        # LabelMaxViewingDist
        with pytest.raises(Exception):
            vo.label_maximum_viewing_dist = 1000000000000.0
        # EnableLabelMaxViewingDist (true)
        vo.enable_label_max_viewing_dist = True
        TestBase.logger.WriteLine4("\tThe new EnableLabelMaxViewingDist is: {0}", vo.enable_label_max_viewing_dist)
        Assert.assertTrue(vo.enable_label_max_viewing_dist)
        # LabelMaxViewingDist
        TestBase.logger.WriteLine6("\tThe current LabelMaxViewingDist is: {0}", vo.label_maximum_viewing_dist)
        vo.label_maximum_viewing_dist = 1000000000000.0
        TestBase.logger.WriteLine6("\tThe new LabelMaxViewingDist is: {0}", vo.label_maximum_viewing_dist)
        Assert.assertEqual(1000000000000.0, vo.label_maximum_viewing_dist)
        # FillInterior (false)
        TestBase.logger.WriteLine4("\tThe current FillInterior is: {0}", vo.fill_interior)
        vo.fill_interior = False
        TestBase.logger.WriteLine4("\tThe new FillInterior is: {0}", vo.fill_interior)
        Assert.assertFalse(vo.fill_interior)
        # LabelMaxViewingDist
        with pytest.raises(Exception):
            vo.percent_translucency_interior = 34
        # FillGranularity
        with pytest.raises(Exception):
            vo.fill_granularity = 44
        # FillInterior (true)
        vo.fill_interior = True
        TestBase.logger.WriteLine4("\tThe new FillInterior is: {0}", vo.fill_interior)
        Assert.assertTrue(vo.fill_interior)
        # PercentTranslucencyInterior
        TestBase.logger.WriteLine6(
            "\tThe current PercentTranslucencyInterior is: {0}", vo.percent_translucency_interior
        )
        vo.percent_translucency_interior = 12
        TestBase.logger.WriteLine6("\tThe new PercentTranslucencyInterior is: {0}", vo.percent_translucency_interior)
        Assert.assertEqual(12, vo.percent_translucency_interior)
        # FillGranularity
        TestBase.logger.WriteLine6("\tThe current FillGranularity is: {0}", vo.fill_granularity)
        vo.fill_granularity = 0.345
        TestBase.logger.WriteLine6("\tThe new FillGranularity is: {0}", vo.fill_granularity)
        Assert.assertEqual(0.345, vo.fill_granularity)
        with pytest.raises(Exception):
            vo.fill_granularity = 0.001
        with pytest.raises(Exception):
            vo.fill_granularity = 6.1
        TestBase.logger.WriteLine("----- THE VO TEST ----- END -----")

    # endregion

    # region DisplayTimes
    @category("Graphics Tests")
    def test_DisplayTimes(self):
        oHelper = DisplayTimesHelper(TestBase.Application)
        oHelper.Run(IDisplayTime(EarlyBoundTests.AG_AT))

    # endregion

    # region AccessConstraints
    @category("AccessConstraints Tests")
    def test_AccessConstraints(self):
        oHelper = AccessConstraintHelper(self.Units)
        oHelper.DoTest(
            EarlyBoundTests.AG_AT.access_constraints, ISTKObject(EarlyBoundTests.AG_AT), TestBase.TemporaryDirectory
        )

    # endregion

    # region STKObject
    @category("Basic Tests")
    def test_STKObject(self):
        oHelper = STKObjectHelper()
        areaTargetObject: "ISTKObject" = clr.CastAs(EarlyBoundTests.AG_AT, ISTKObject)
        oHelper.Run(areaTargetObject)
        oHelper.TestObjectFilesArray(areaTargetObject.object_files)

    # endregion

    # region AccessDataDisplay
    @category("Graphics Tests")
    def test_AccessDataDisplay(self):
        # test Access VO DataDisplays
        oSatellite: "Satellite" = Satellite(TestBase.Application.current_scenario.children["Satellite1"])
        Assert.assertNotEqual(None, oSatellite)
        oSatellite.set_propagator_type(PropagatorType.TWO_BODY)
        Assert.assertEqual(PropagatorType.TWO_BODY, oSatellite.propagator_type)
        oPropagator: "PropagatorTwoBody" = PropagatorTwoBody(oSatellite.propagator)
        Assert.assertNotEqual(None, oPropagator)
        oPropagator.propagate()

        # get access to satellite
        oAccess: "Access" = (ISTKObject(EarlyBoundTests.AG_AT)).get_access_to_object(clr.CastAs(oSatellite, ISTKObject))
        Assert.assertNotEqual(None, oAccess)
        oAccess.compute_access()
        helper = VODataDisplayHelper(TestBase.Application)
        helper.Run(oAccess.data_displays, True, False)
        oAccess.remove_access()

    # endregion
