# Copyright (C) 2022 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""
Access and manipulate visual elements in STK.

These include STK globe terrain
and imagery, camera control, 3D models, triangle meshes, surface polygons and
polylines, text batches, screen overlays, scene lighting, and raster and
projection streaming. STK Graphics is available in STK, using UI plugins, as
well as in STK Engine custom applications.
"""

__all__ = ["AGICustomTerrainOverlay", "AGICustomTerrainOverlayFactory", "AGIProcessedImageGlobeOverlay",
"AGIProcessedImageGlobeOverlayFactory", "AGIProcessedTerrainOverlay", "AGIProcessedTerrainOverlayFactory",
"AGIRoamImageGlobeOverlay", "AGIRoamImageGlobeOverlayFactory", "AlphaFromLuminanceFilter",
"AlphaFromLuminanceFilterFactory", "AlphaFromPixelFilter", "AlphaFromPixelFilterFactory", "AlphaFromRasterFilter",
"AlphaFromRasterFilterFactory", "AltitudeDisplayCondition", "AltitudeDisplayConditionFactory", "AntiAliasingMethod",
"AxesPrimitive", "AxesPrimitiveFactory", "BandExtractFilter", "BandExtractFilterFactory", "BandOrderFilter",
"BandOrderFilterFactory", "BatchPrimitiveIndex", "BinaryLogicOperation", "BlurFilter", "BlurFilterFactory",
"BlurMethod", "BoundingSphere", "BoundingSphereFactory", "BoxTriangulatorInitializer", "BrightnessFilter",
"BrightnessFilterFactory", "Camera", "CameraSnapshot", "CameraVideoRecording", "CentralBodyGraphics",
"CentralBodyGraphicsIndexer", "Clouds", "ColorToLuminanceFilter", "ColorToLuminanceFilterFactory",
"CompositeDisplayCondition", "CompositeDisplayConditionFactory", "CompositePrimitive", "CompositePrimitiveFactory",
"ConstantDisplayCondition", "ConstantDisplayConditionFactory", "ConstrainedUpAxis", "ContrastFilter",
"ContrastFilterFactory", "ConvolutionFilter", "ConvolutionFilterFactory", "CustomImageGlobeOverlay",
"CustomImageGlobeOverlayPluginActivator", "CustomImageGlobeOverlayPluginActivatorFactory",
"CustomImageGlobeOverlayPluginProxy", "CylinderFillOptions", "CylinderTriangulatorInitializer", "DisplayCondition",
"DistanceDisplayCondition", "DistanceDisplayConditionFactory", "DistanceToGlobeOverlayDisplayCondition",
"DistanceToGlobeOverlayDisplayConditionFactory", "DistanceToPositionDisplayCondition",
"DistanceToPositionDisplayConditionFactory", "DistanceToPrimitiveDisplayCondition",
"DistanceToPrimitiveDisplayConditionFactory", "DurationPathPrimitiveUpdatePolicy",
"DurationPathPrimitiveUpdatePolicyFactory", "EdgeDetectFilter", "EdgeDetectFilterFactory", "EdgeDetectMethod",
"EllipsoidTriangulatorInitializer", "ExtrudedPolylineTriangulatorInitializer", "ExtrudedPolylineTriangulatorResult",
"FaceCullingMode", "FactoryAndInitializers", "FilteringRasterStream", "FilteringRasterStreamFactory", "FlipFilter",
"FlipFilterFactory", "FontStyle", "FrameRate", "GammaCorrectionFilter", "GammaCorrectionFilterFactory",
"GaussianBlurFilter", "GaussianBlurFilterFactory", "GeospatialImageGlobeOverlay", "GeospatialImageGlobeOverlayFactory",
"GlobeImageOverlay", "GlobeImageOverlayAddCompleteEventArgs", "GlobeImageOverlayInitializer", "GlobeOverlay",
"GlobeOverlaySettings", "GradientDetectFilter", "GradientDetectFilterFactory", "GradientDetectMethod", "GraphicsFont",
"GraphicsFontFactory", "GreatArcInterpolator", "GreatArcInterpolatorFactory", "IConvolutionFilter", "IDisplayCondition",
"IGlobeImageOverlay", "IGlobeOverlay", "IKmlContainer", "IKmlFeature", "IOverlay", "IPathPrimitiveUpdatePolicy",
"IPositionInterpolator", "IPrimitive", "IProjection", "IRaster", "IRasterFilter", "IRasterStream", "IScreenOverlay",
"IScreenOverlayCollectionBase", "IScreenOverlayContainer", "ITerrainOverlay", "ITriangulatorResult", "ImageCollection",
"Jpeg2000CompressionProfile", "Jpeg2000WriterInitializer", "KmlContainer", "KmlDocument", "KmlDocumentCollection",
"KmlDocumentLoadedEventArgs", "KmlFeature", "KmlFeatureCollection", "KmlFolder", "KmlGraphics", "KmlNetworkLink",
"KmlNetworkLinkRefreshMode", "KmlNetworkLinkViewRefreshMode", "LevelsFilter", "LevelsFilterFactory", "Lighting",
"MagnificationFilter", "MapProjection", "MarkerBatchPrimitive", "MarkerBatchPrimitiveFactory",
"MarkerBatchPrimitiveOptionalParameters", "MarkerBatchPrimitiveOptionalParametersFactory", "MarkerBatchRenderPass",
"MarkerBatchRenderingMethod", "MarkerBatchSizeSource", "MarkerBatchSizeUnit", "MarkerBatchSortOrder",
"MaximumCountPathPrimitiveUpdatePolicy", "MaximumCountPathPrimitiveUpdatePolicyFactory", "MinificationFilter",
"ModelArticulation", "ModelArticulationCollection", "ModelPrimitive", "ModelPrimitiveFactory", "ModelTransformation",
"ModelTransformationType", "ModelUpAxis", "ObjectCollection", "Origin", "OutlineAppearance", "Overlay",
"OverlayAspectRatioMode", "OverlayRole", "PathPoint", "PathPointCollection", "PathPointFactory", "PathPrimitive",
"PathPrimitiveFactory", "PathPrimitiveRemoveLocation", "PathPrimitiveUpdatePolicy", "PickResult",
"PickResultCollection", "PixelSizeDisplayCondition", "PixelSizeDisplayConditionFactory", "PointBatchPrimitive",
"PointBatchPrimitiveFactory", "PointBatchPrimitiveOptionalParameters", "PointBatchPrimitiveOptionalParametersFactory",
"PolylinePrimitive", "PolylinePrimitiveFactory", "PolylinePrimitiveOptionalParameters",
"PolylinePrimitiveOptionalParametersFactory", "PolylineType", "PositionInterpolator", "Primitive",
"PrimitiveIndicesOrderHint", "PrimitiveManager", "PrimitivesSortOrder", "ProjectedRasterOverlay",
"ProjectedRasterOverlayFactory", "Projection", "ProjectionFactory", "ProjectionRasterStreamPluginActivator",
"ProjectionRasterStreamPluginActivatorFactory", "ProjectionRasterStreamPluginProxy", "ProjectionStream", "Raster",
"RasterAttributes", "RasterAttributesFactory", "RasterBand", "RasterFactory", "RasterFilter", "RasterFlipAxis",
"RasterFormat", "RasterImageGlobeOverlay", "RasterImageGlobeOverlayFactory", "RasterOrientation", "RasterSharpenMethod",
"RasterStream", "RasterType", "RefreshRate", "RenderPass", "RenderPassHint", "RendererShadingModel",
"RendererTexture2D", "RendererTextureTemplate2D", "RenderingEventArgs", "RhumbLineInterpolator",
"RhumbLineInterpolatorFactory", "RotateFilter", "RotateFilterFactory", "Scene", "SceneCollection",
"SceneDisplayCondition", "SceneDisplayConditionFactory", "SceneGlobeOverlaySettings", "SceneManager",
"SceneManagerInitializer", "ScreenOverlay", "ScreenOverlayCollection", "ScreenOverlayCollectionBase",
"ScreenOverlayFactory", "ScreenOverlayManager", "ScreenOverlayOrigin", "ScreenOverlayPickResult",
"ScreenOverlayPickResultCollection", "ScreenOverlayPinningOrigin", "ScreenOverlayUnit", "SequenceFilter",
"SequenceFilterFactory", "SetHint", "SharpenFilter", "SharpenFilterFactory", "SnapshotFileFormat", "SolidPrimitive",
"SolidPrimitiveFactory", "SolidTriangulatorResult", "StereoProjectionMode", "Stereoscopic", "StereoscopicDisplayMode",
"SurfaceExtentTriangulatorInitializer", "SurfaceMeshPrimitive", "SurfaceMeshPrimitiveFactory",
"SurfaceMeshRenderingMethod", "SurfacePolygonTriangulatorInitializer", "SurfaceShapesInitializer",
"SurfaceShapesResult", "SurfaceTriangulatorResult", "TerrainOverlay", "TerrainOverlayAddCompleteEventArgs",
"TerrainOverlayCollection", "TerrainOverlayInitializer", "TextBatchPrimitive", "TextBatchPrimitiveFactory",
"TextBatchPrimitiveOptionalParameters", "TextBatchPrimitiveOptionalParametersFactory", "TextOverlay",
"TextOverlayFactory", "Texture2DFactory", "TextureFilter2D", "TextureFilter2DFactory", "TextureFormat", "TextureMatrix",
"TextureMatrixFactory", "TextureScreenOverlay", "TextureScreenOverlayFactory", "TextureWrap",
"TimeIntervalDisplayCondition", "TimeIntervalDisplayConditionFactory", "TriangleMeshPrimitive",
"TriangleMeshPrimitiveFactory", "TriangleMeshPrimitiveOptionalParameters",
"TriangleMeshPrimitiveOptionalParametersFactory", "TriangulatorResult", "VectorPrimitive", "VectorPrimitiveFactory",
"VideoFormat", "VideoPlayback", "VideoStream", "VideoStreamFactory", "Visibility", "VisualEffects", "WindingOrder"]

from ctypes import POINTER
from enum import IntEnum, IntFlag
import typing

from .analysis_workbench import (
    IVectorGeometryToolAxes,
    IVectorGeometryToolPoint,
    IVectorGeometryToolSystem,
    IVectorGeometryToolVector,
)
from .internal import coclassutil as agcls, comutil as agcom, marshall as agmarshall
from .internal.apiutil import (
    EnumeratorProxy,
    InterfaceProxy,
    OutArg,
    SupportsDeleteCallback,
    get_interface_property,
    initialize_from_source_object,
    set_class_attribute,
    set_interface_attribute,
)
from .internal.comutil import IPictureDisp, IUnknown
from .internal.eventutil import (
    IImageCollectionEventHandler,
    IKmlGraphicsEventHandler,
    ISceneEventHandler,
    ITerrainOverlayCollectionEventHandler,
)
from .stkutil import Date, IOrientation
from .utilities import colors as agcolor
from .utilities.exceptions import STKRuntimeError


def _raise_uninitialized_error(*args):
    raise STKRuntimeError("Valid STK object model classes are returned from STK methods and should not be created independently.")

class CylinderFillOptions(IntFlag):
    """
    Cylinder faces that can be filled.

    Examples
    --------
    Combine enumerations with the logical OR operator:
    >>> from ansys.stk.core.graphics import CylinderFillOptions
    >>>
    >>> # CylinderFillOptions inherits from enum.IntFlag and may be combined
    >>> # using the `|` operator
    >>> cyl_fill = CylinderFillOptions.BOTTOM_CAP | CylinderFillOptions.TOP_CAP
    """

    WALL = 1
    """Fill the cylinder's wall."""
    BOTTOM_CAP = 2
    """Fill the cylinder's bottom cap."""
    TOP_CAP = 4
    """Fill the cylinder's top cap."""
    ALL = 7
    """Completely fill the cylinder, including its wall, bottom, and top cap."""

CylinderFillOptions.WALL.__doc__ = "Fill the cylinder's wall."
CylinderFillOptions.BOTTOM_CAP.__doc__ = "Fill the cylinder's bottom cap."
CylinderFillOptions.TOP_CAP.__doc__ = "Fill the cylinder's top cap."
CylinderFillOptions.ALL.__doc__ = "Completely fill the cylinder, including its wall, bottom, and top cap."

agcls.AgTypeNameMap["CylinderFillOptions"] = CylinderFillOptions

class WindingOrder(IntEnum):
    """Specify the order for positions or front facing triangles. Winding order is important for triangulation and backface culling."""

    COUNTER_CLOCKWISE = 0
    """Positions or triangles are defined in counter-clockwise order."""
    CLOCKWISE = 1
    """Positions or triangles are defined in clockwise order."""
    COMPUTE = 2
    """The winding order is unknown and should be computed. For best performance, only use this value if you do not know the actual winding order."""

WindingOrder.COUNTER_CLOCKWISE.__doc__ = "Positions or triangles are defined in counter-clockwise order."
WindingOrder.CLOCKWISE.__doc__ = "Positions or triangles are defined in clockwise order."
WindingOrder.COMPUTE.__doc__ = "The winding order is unknown and should be computed. For best performance, only use this value if you do not know the actual winding order."

agcls.AgTypeNameMap["WindingOrder"] = WindingOrder

class SnapshotFileFormat(IntEnum):
    """When using camera snapshot or camera video recording to save a snapshot to a file, this specifies the file format."""

    BMP = 0
    """BMP file format."""
    TIFF = 2
    """TIFF file format."""
    JPEG = 3
    """JPEG file format."""
    PNG = 4
    """PNG file format."""

SnapshotFileFormat.BMP.__doc__ = "BMP file format."
SnapshotFileFormat.TIFF.__doc__ = "TIFF file format."
SnapshotFileFormat.JPEG.__doc__ = "JPEG file format."
SnapshotFileFormat.PNG.__doc__ = "PNG file format."

agcls.AgTypeNameMap["SnapshotFileFormat"] = SnapshotFileFormat

class VideoFormat(IntEnum):
    """When using camera video recording to record a video, this specifies the file format."""

    H264 = 0
    """H.264 file format."""
    WMV = 1
    """Windows Media Video (WMV) file format."""

VideoFormat.H264.__doc__ = "H.264 file format."
VideoFormat.WMV.__doc__ = "Windows Media Video (WMV) file format."

agcls.AgTypeNameMap["VideoFormat"] = VideoFormat

class ConstrainedUpAxis(IntEnum):
    """When setting the camera'saxes, this defines which axis of the axes is up in screen space, where up is from the bottom to the top of the screen."""

    X = 0
    """The X axis is constrained."""
    Y = 1
    """The Y axis is constrained."""
    Z = 2
    """The Z axis is constrained."""
    NEGATIVE_X = 3
    """The negative X axis is constrained."""
    NEGATIVE_Y = 4
    """The negative Y axis is constrained."""
    NEGATIVE_Z = 5
    """The negative Z axis is constrained."""
    NONE = 6
    """No axis is contained."""

ConstrainedUpAxis.X.__doc__ = "The X axis is constrained."
ConstrainedUpAxis.Y.__doc__ = "The Y axis is constrained."
ConstrainedUpAxis.Z.__doc__ = "The Z axis is constrained."
ConstrainedUpAxis.NEGATIVE_X.__doc__ = "The negative X axis is constrained."
ConstrainedUpAxis.NEGATIVE_Y.__doc__ = "The negative Y axis is constrained."
ConstrainedUpAxis.NEGATIVE_Z.__doc__ = "The negative Z axis is constrained."
ConstrainedUpAxis.NONE.__doc__ = "No axis is contained."

agcls.AgTypeNameMap["ConstrainedUpAxis"] = ConstrainedUpAxis

class OverlayRole(IntEnum):
    """The role of a globe overlay."""

    BASE = 0
    """The globe overlay is a base image."""
    NIGHT = 1
    """The globe overlay is a base image that is only displayed on the area of the central body that is not lit by the sun."""
    SPECULAR = 2
    """The globe overlay is a base image that shows the glint of the sun on the central body."""
    NORMAL = 4
    """The globe overlay is a normal overlay without a special role."""
    NONE = 5
    """The role of the globe overlay isn't set."""

OverlayRole.BASE.__doc__ = "The globe overlay is a base image."
OverlayRole.NIGHT.__doc__ = "The globe overlay is a base image that is only displayed on the area of the central body that is not lit by the sun."
OverlayRole.SPECULAR.__doc__ = "The globe overlay is a base image that shows the glint of the sun on the central body."
OverlayRole.NORMAL.__doc__ = "The globe overlay is a normal overlay without a special role."
OverlayRole.NONE.__doc__ = "The role of the globe overlay isn't set."

agcls.AgTypeNameMap["OverlayRole"] = OverlayRole

class PrimitiveIndicesOrderHint(IntEnum):
    """An optimization hint optionally provided to a primitive'sSetPartial method to enhance performance."""

    NOT_SORTED = 0
    """The indices passed to SetPartial are not sorted. Therefore, the primitive may sort them to improve performance of writing its geometry to video memory."""
    SORTED_ASCENDING = 1
    """The indices passed to SetPartial are sorted in ascending order. Therefore, the primitive does not need to sort them. It is recommended to only use SortedAscending if it is easy and efficient for you to provide the indices in ascending order..."""

PrimitiveIndicesOrderHint.NOT_SORTED.__doc__ = "The indices passed to SetPartial are not sorted. Therefore, the primitive may sort them to improve performance of writing its geometry to video memory."
PrimitiveIndicesOrderHint.SORTED_ASCENDING.__doc__ = "The indices passed to SetPartial are sorted in ascending order. Therefore, the primitive does not need to sort them. It is recommended to only use SortedAscending if it is easy and efficient for you to provide the indices in ascending order..."

agcls.AgTypeNameMap["PrimitiveIndicesOrderHint"] = PrimitiveIndicesOrderHint

class OverlayAspectRatioMode(IntEnum):
    """Specify whether the aspect ratio of a texture will be maintained during sizing of a screen overlay."""

    NONE = 0
    """The aspect ratio of the texture is not maintained during sizing of the screen overlay."""
    WIDTH = 1
    """The aspect ratio of the texture is maintained based on the width property of the screen overlay. When used, the height property is ignored and the height is automatically calculated based on the aspect ratio of the texture and the overlay's width property."""
    HEIGHT = 2
    """The aspect ratio of the texture is maintained based on the height property of the screen overlay. When used, the width property is ignored and the width is automatically calculated based on the aspect ratio of the texture and the overlay's height property."""

OverlayAspectRatioMode.NONE.__doc__ = "The aspect ratio of the texture is not maintained during sizing of the screen overlay."
OverlayAspectRatioMode.WIDTH.__doc__ = "The aspect ratio of the texture is maintained based on the width property of the screen overlay. When used, the height property is ignored and the height is automatically calculated based on the aspect ratio of the texture and the overlay's width property."
OverlayAspectRatioMode.HEIGHT.__doc__ = "The aspect ratio of the texture is maintained based on the height property of the screen overlay. When used, the width property is ignored and the width is automatically calculated based on the aspect ratio of the texture and the overlay's height property."

agcls.AgTypeNameMap["OverlayAspectRatioMode"] = OverlayAspectRatioMode

class MapProjection(IntEnum):
    """The projection of the pixel data returned from a custom image globe overlay."""

    MERCATOR = 0
    """Mercator projection."""
    EQUIDISTANT_CYLINDRICAL = 1
    """Equidistant Cylindrical projection."""

MapProjection.MERCATOR.__doc__ = "Mercator projection."
MapProjection.EQUIDISTANT_CYLINDRICAL.__doc__ = "Equidistant Cylindrical projection."

agcls.AgTypeNameMap["MapProjection"] = MapProjection

class MarkerBatchRenderingMethod(IntEnum):
    """Rendering methods available for use by the marker batch primitive. Different methods may have different performance characteristics and require different video card support. When in doubt, use Automatic."""

    GEOMETRY_SHADER = 0
    """Render the marker batch using a geometry shader."""
    VERTEX_SHADER = 1
    """Render the marker batch using a vertex shader."""
    AUTOMATIC = 2
    """Render the marker batch using an automatically selected method based on the capabilities of the video card."""
    FIXED_FUNCTION = 3
    """Render the marker batch using the fixed function pipeline. Generally, this is the slowest method but it supports all video cards."""

MarkerBatchRenderingMethod.GEOMETRY_SHADER.__doc__ = "Render the marker batch using a geometry shader."
MarkerBatchRenderingMethod.VERTEX_SHADER.__doc__ = "Render the marker batch using a vertex shader."
MarkerBatchRenderingMethod.AUTOMATIC.__doc__ = "Render the marker batch using an automatically selected method based on the capabilities of the video card."
MarkerBatchRenderingMethod.FIXED_FUNCTION.__doc__ = "Render the marker batch using the fixed function pipeline. Generally, this is the slowest method but it supports all video cards."

agcls.AgTypeNameMap["MarkerBatchRenderingMethod"] = MarkerBatchRenderingMethod

class MarkerBatchRenderPass(IntEnum):
    """The pass during which the marker batch is rendered."""

    OPAQUE = 0
    """The marker batch contains all opaque textures and therefore should be rendered using the opaque pass."""
    TRANSLUCENT = 1
    """The marker batch contains textures with translucency and therefore should be rendered using the translucent pass. For correct blending of overlapping textures, also consider using back to front."""
    BASED_ON_TRANSLUCENCY = 2
    """The marker batch render pass should be determined based on the marker batch's translucency. This includes the translucency set per marker when SetColors is used."""

MarkerBatchRenderPass.OPAQUE.__doc__ = "The marker batch contains all opaque textures and therefore should be rendered using the opaque pass."
MarkerBatchRenderPass.TRANSLUCENT.__doc__ = "The marker batch contains textures with translucency and therefore should be rendered using the translucent pass. For correct blending of overlapping textures, also consider using back to front."
MarkerBatchRenderPass.BASED_ON_TRANSLUCENCY.__doc__ = "The marker batch render pass should be determined based on the marker batch's translucency. This includes the translucency set per marker when SetColors is used."

agcls.AgTypeNameMap["MarkerBatchRenderPass"] = MarkerBatchRenderPass

class MarkerBatchSizeSource(IntEnum):
    """Determine which marker batch property is used to size each marker in a marker batch."""

    FROM_TEXTURE = 0
    """The size of each marker is the same as the size of its texture. If the marker is not textured, the user defined size is used instead."""
    USER_DEFINED = 1
    """The size of each marker in the marker batch is user defined. Either all markers have the same size (size) or each marker has a user defined size (SetSizes)."""

MarkerBatchSizeSource.FROM_TEXTURE.__doc__ = "The size of each marker is the same as the size of its texture. If the marker is not textured, the user defined size is used instead."
MarkerBatchSizeSource.USER_DEFINED.__doc__ = "The size of each marker in the marker batch is user defined. Either all markers have the same size (size) or each marker has a user defined size (SetSizes)."

agcls.AgTypeNameMap["MarkerBatchSizeSource"] = MarkerBatchSizeSource

class MarkerBatchSortOrder(IntEnum):
    """The order in which markers in a marker batch are sorted before rendering."""

    BACK_TO_FRONT = 0
    """The markers are sorted in back to front order before rendering. For overlapping translucent markers, this enables correct blending results..."""
    FRONT_TO_BACK = 1
    """The markers are sorted in front to back order before rendering. For overlapping opaque markers, this can enable the GPU to quickly eliminate markers that are hidden behind other markers..."""
    BY_TEXTURE = 2
    """The markers are sorted by texture. This minimizes costly texture changes during rendering and does not require resorting when the camera moves or a marker changes position."""

MarkerBatchSortOrder.BACK_TO_FRONT.__doc__ = "The markers are sorted in back to front order before rendering. For overlapping translucent markers, this enables correct blending results..."
MarkerBatchSortOrder.FRONT_TO_BACK.__doc__ = "The markers are sorted in front to back order before rendering. For overlapping opaque markers, this can enable the GPU to quickly eliminate markers that are hidden behind other markers..."
MarkerBatchSortOrder.BY_TEXTURE.__doc__ = "The markers are sorted by texture. This minimizes costly texture changes during rendering and does not require resorting when the camera moves or a marker changes position."

agcls.AgTypeNameMap["MarkerBatchSortOrder"] = MarkerBatchSortOrder

class MarkerBatchSizeUnit(IntEnum):
    """The unit for marker sizes in a marker batch."""

    PIXELS = 0
    """The size of a marker is defined in pixels. As the camera zooms in and out the size of the marker doesn't change."""
    METERS = 1
    """The size of a marker is defined in meters."""

MarkerBatchSizeUnit.PIXELS.__doc__ = "The size of a marker is defined in pixels. As the camera zooms in and out the size of the marker doesn't change."
MarkerBatchSizeUnit.METERS.__doc__ = "The size of a marker is defined in meters."

agcls.AgTypeNameMap["MarkerBatchSizeUnit"] = MarkerBatchSizeUnit

class ModelTransformationType(IntEnum):
    """Transformation types that define the way a model transformation changes the geometry of the model articulation it is associated with."""

    TRANSLATE_X = 0
    """Translate the geometry along the X axis."""
    TRANSLATE_Y = 1
    """Translate the geometry along the Y axis."""
    TRANSLATE_Z = 2
    """Translate the geometry along the Z axis."""
    ROTATE_X = 3
    """Rotate the geometry about the X axis."""
    ROTATE_Y = 4
    """Rotate the geometry about the Y axis."""
    ROTATE_Z = 5
    """Rotate the geometry about the Z axis."""
    SCALE_X = 6
    """Scale the geometry along the X axis."""
    SCALE_Y = 7
    """Scale the geometry along the Y axis."""
    SCALE_Z = 8
    """Scale the geometry along the Z axis."""
    SCALE_UNIFORM = 9
    """Scale the geometry uniformly along the X, Y, and Z axes."""
    TEXTURE_TRANSLATE_X = 10
    """Translate the texture associated with the geometry along the X axis."""
    TEXTURE_TRANSLATE_Y = 11
    """Translate the texture associated with the geometry along the Y axis."""
    TEXTURE_TRANSLATE_Z = 12
    """Translate the texture associated with the geometry along the Z axis."""
    TEXTURE_ROTATE_X = 13
    """Rotate the texture associated with the geometry about the X axis."""
    TEXTURE_ROTATE_Y = 14
    """Rotate the texture associated with the geometry about the Y axis."""
    TEXTURE_ROTATE_Z = 15
    """Rotate the texture associated with the geometry about the Z axis."""
    TEXTURE_SCALE_X = 16
    """Scale the texture associated with the geometry along the X axis."""
    TEXTURE_SCALE_Y = 17
    """Scale the texture associated with the geometry along the Y axis."""
    TEXTURE_SCALE_Z = 18
    """Scale the texture associated with the geometry along the Z axis."""
    TEXTURE_SCALE_UNIFORM = 19
    """Scale the texture associated with the geometry uniformly along the X, Y, and Z axes."""
    TRANSLATE_RED = 20
    """Translate the red band of color associated with the geometry."""
    TRANSLATE_GREEN = 21
    """Translate the green band of color associated with the geometry."""
    TRANSLATE_BLUE = 22
    """Translate the blue band of color associated with the geometry."""

ModelTransformationType.TRANSLATE_X.__doc__ = "Translate the geometry along the X axis."
ModelTransformationType.TRANSLATE_Y.__doc__ = "Translate the geometry along the Y axis."
ModelTransformationType.TRANSLATE_Z.__doc__ = "Translate the geometry along the Z axis."
ModelTransformationType.ROTATE_X.__doc__ = "Rotate the geometry about the X axis."
ModelTransformationType.ROTATE_Y.__doc__ = "Rotate the geometry about the Y axis."
ModelTransformationType.ROTATE_Z.__doc__ = "Rotate the geometry about the Z axis."
ModelTransformationType.SCALE_X.__doc__ = "Scale the geometry along the X axis."
ModelTransformationType.SCALE_Y.__doc__ = "Scale the geometry along the Y axis."
ModelTransformationType.SCALE_Z.__doc__ = "Scale the geometry along the Z axis."
ModelTransformationType.SCALE_UNIFORM.__doc__ = "Scale the geometry uniformly along the X, Y, and Z axes."
ModelTransformationType.TEXTURE_TRANSLATE_X.__doc__ = "Translate the texture associated with the geometry along the X axis."
ModelTransformationType.TEXTURE_TRANSLATE_Y.__doc__ = "Translate the texture associated with the geometry along the Y axis."
ModelTransformationType.TEXTURE_TRANSLATE_Z.__doc__ = "Translate the texture associated with the geometry along the Z axis."
ModelTransformationType.TEXTURE_ROTATE_X.__doc__ = "Rotate the texture associated with the geometry about the X axis."
ModelTransformationType.TEXTURE_ROTATE_Y.__doc__ = "Rotate the texture associated with the geometry about the Y axis."
ModelTransformationType.TEXTURE_ROTATE_Z.__doc__ = "Rotate the texture associated with the geometry about the Z axis."
ModelTransformationType.TEXTURE_SCALE_X.__doc__ = "Scale the texture associated with the geometry along the X axis."
ModelTransformationType.TEXTURE_SCALE_Y.__doc__ = "Scale the texture associated with the geometry along the Y axis."
ModelTransformationType.TEXTURE_SCALE_Z.__doc__ = "Scale the texture associated with the geometry along the Z axis."
ModelTransformationType.TEXTURE_SCALE_UNIFORM.__doc__ = "Scale the texture associated with the geometry uniformly along the X, Y, and Z axes."
ModelTransformationType.TRANSLATE_RED.__doc__ = "Translate the red band of color associated with the geometry."
ModelTransformationType.TRANSLATE_GREEN.__doc__ = "Translate the green band of color associated with the geometry."
ModelTransformationType.TRANSLATE_BLUE.__doc__ = "Translate the blue band of color associated with the geometry."

agcls.AgTypeNameMap["ModelTransformationType"] = ModelTransformationType

class Origin(IntEnum):
    """Vertical and horizontal origin."""

    BOTTOM_LEFT = 0
    """The object's origin is its bottom, left corner."""
    BOTTOM_CENTER = 1
    """The object's origin is its bottom, center point."""
    BOTTOM_RIGHT = 2
    """The object's origin is its bottom, right corner."""
    CENTER_LEFT = 3
    """The object's origin is its center, left point."""
    CENTER = 4
    """The object's origin is its center."""
    CENTER_RIGHT = 5
    """The object's origin is its center, right point."""
    TOP_LEFT = 6
    """The object's origin is its top, left corner."""
    TOP_CENTER = 7
    """The object's origin is its top, center point."""
    TOP_RIGHT = 8
    """The object's origin is its top, right corner."""

Origin.BOTTOM_LEFT.__doc__ = "The object's origin is its bottom, left corner."
Origin.BOTTOM_CENTER.__doc__ = "The object's origin is its bottom, center point."
Origin.BOTTOM_RIGHT.__doc__ = "The object's origin is its bottom, right corner."
Origin.CENTER_LEFT.__doc__ = "The object's origin is its center, left point."
Origin.CENTER.__doc__ = "The object's origin is its center."
Origin.CENTER_RIGHT.__doc__ = "The object's origin is its center, right point."
Origin.TOP_LEFT.__doc__ = "The object's origin is its top, left corner."
Origin.TOP_CENTER.__doc__ = "The object's origin is its top, center point."
Origin.TOP_RIGHT.__doc__ = "The object's origin is its top, right corner."

agcls.AgTypeNameMap["Origin"] = Origin

class PathPrimitiveRemoveLocation(IntEnum):
    """Represents the location of a point to be removed."""

    FRONT = 0
    """Remove from the front of the line."""
    BACK = 1
    """Remove from the back of the line."""

PathPrimitiveRemoveLocation.FRONT.__doc__ = "Remove from the front of the line."
PathPrimitiveRemoveLocation.BACK.__doc__ = "Remove from the back of the line."

agcls.AgTypeNameMap["PathPrimitiveRemoveLocation"] = PathPrimitiveRemoveLocation

class PrimitivesSortOrder(IntEnum):
    """The order in which primitives are sorted before rendering."""

    BY_STATE = 0
    """Primitives are sorted by their internal state before rendering. This provides good performance but can lead to blending artifacts with translucent primitives along the same line of sight."""
    BACK_TO_FRONT = 1
    """Primitives are sorted in back to front order before rendering. For translucent primitives, this enables correct blending results. This may not perform as well as PrimitivesSortOrderByState since the CPU has to sort the primitives before rendering."""

PrimitivesSortOrder.BY_STATE.__doc__ = "Primitives are sorted by their internal state before rendering. This provides good performance but can lead to blending artifacts with translucent primitives along the same line of sight."
PrimitivesSortOrder.BACK_TO_FRONT.__doc__ = "Primitives are sorted in back to front order before rendering. For translucent primitives, this enables correct blending results. This may not perform as well as PrimitivesSortOrderByState since the CPU has to sort the primitives before rendering."

agcls.AgTypeNameMap["PrimitivesSortOrder"] = PrimitivesSortOrder

class RefreshRate(IntEnum):
    """The rate at which animation frames will occur."""

    FASTEST = 0
    """The animation will occur as fast as possible. The maximum frame rate is the refresh rate of the display or is the maximum that video card is capable of if the video card's vertical sync is off."""
    TARGETED_FRAMES_PER_SECOND = 1
    """The animation will target a specified frame rate."""

RefreshRate.FASTEST.__doc__ = "The animation will occur as fast as possible. The maximum frame rate is the refresh rate of the display or is the maximum that video card is capable of if the video card's vertical sync is off."
RefreshRate.TARGETED_FRAMES_PER_SECOND.__doc__ = "The animation will target a specified frame rate."

agcls.AgTypeNameMap["RefreshRate"] = RefreshRate

class RenderPass(IntFlag):
    """Describes when a primitive will be rendered. Some primitives need to be rendered during or at a certain time. For example, translucent primitives need to be rendered after opaque primitives to allow proper blending..."""

    OPAQUE = 1
    """Render during the opaque rendering pass."""
    TRANSLUCENT = 2
    """Render during the translucent rendering pass."""
    CENTRAL_BODY_CLIPPED = 4
    """Render before the central body is rendered."""
    ORDERED_COMPOSITE_CENTRAL_BODY_CLIPPED = 8
    """Rendered in an ordered composite before all other primitives and before the central body is rendered."""
    ORDERED_COMPOSITE = 16
    """Rendered in an ordered composite before all primitives but after the central body is rendered."""
    TERRAIN = 32
    """Render after the terrain is rendered."""

RenderPass.OPAQUE.__doc__ = "Render during the opaque rendering pass."
RenderPass.TRANSLUCENT.__doc__ = "Render during the translucent rendering pass."
RenderPass.CENTRAL_BODY_CLIPPED.__doc__ = "Render before the central body is rendered."
RenderPass.ORDERED_COMPOSITE_CENTRAL_BODY_CLIPPED.__doc__ = "Rendered in an ordered composite before all other primitives and before the central body is rendered."
RenderPass.ORDERED_COMPOSITE.__doc__ = "Rendered in an ordered composite before all primitives but after the central body is rendered."
RenderPass.TERRAIN.__doc__ = "Render after the terrain is rendered."

agcls.AgTypeNameMap["RenderPass"] = RenderPass

class RenderPassHint(IntEnum):
    """An optimization hint optionally provided to a primitive'sSet method to enhance performance when per-position colors are used."""

    OPAQUE = 0
    """The collection of colors contains only opaque colors. This implies that each color's alpha component is 255."""
    TRANSLUCENT = 1
    """The collection of colors contains translucent colors. This implies that at least one color has an alpha component that is not 255."""
    UNKNOWN = 2
    """It is unknown if the collection of colors contains opaque or translucent colors."""

RenderPassHint.OPAQUE.__doc__ = "The collection of colors contains only opaque colors. This implies that each color's alpha component is 255."
RenderPassHint.TRANSLUCENT.__doc__ = "The collection of colors contains translucent colors. This implies that at least one color has an alpha component that is not 255."
RenderPassHint.UNKNOWN.__doc__ = "It is unknown if the collection of colors contains opaque or translucent colors."

agcls.AgTypeNameMap["RenderPassHint"] = RenderPassHint

class ScreenOverlayOrigin(IntEnum):
    """Specify the origin of a screen overlay, as well as the direction of the horizontal and vertical axes. The origin specifies both the origin in the parent overlay's coordinate system and the origin within the overlay itself that is positioned."""

    BOTTOM_LEFT = 0
    """When the X and Y position are both set to 0, this value places the bottom, left corner of the overlay in its parent's bottom, left corner. Increasing X values move the overlay to the right and increasing Y values move it up."""
    BOTTOM_CENTER = 1
    """When the X and Y position are both set to 0, this value places the bottom edge of the overlay at its parent's bottom edge and the center of the overlay is horizontally centered within its parent..."""
    BOTTOM_RIGHT = 2
    """When the X and Y position are both set to 0, this value places the bottom, right corner of the overlay in its parent's bottom, right corner. Increasing X values move the overlay to the left and increasing Y values move it up."""
    CENTER_LEFT = 3
    """When the X and Y position are both set to 0, this value places the left edge of the overlay at its parent's left edge and the center of the overlay is vertically centered within its parent..."""
    CENTER = 4
    """When the X and Y position are both set to 0, this value places the center of the overlay at its parent's center. Increasing X values move the overlay to the right and increasing Y values move it up."""
    CENTER_RIGHT = 5
    """When the X and Y position are both set to 0, this value places the right edge of the overlay at its parent's right edge and the center of the overlay is vertically centered within its parent..."""
    TOP_LEFT = 6
    """When the X and Y position are both set to 0, this value places the top, left corner of the overlay in its parent's top, left corner. Increasing X values move the overlay to the right and increasing Y values move it down."""
    TOP_CENTER = 7
    """When the X and Y position are both set to 0, this value places the top edge of the overlay at its parent's top edge and the center of the overlay is horizontally centered within its parent..."""
    TOP_RIGHT = 8
    """When the X and Y position are both set to 0, this value places the top, right corner of the overlay in its parent's top, right corner. Increasing X values move the overlay to the left and increasing Y values move it down."""

ScreenOverlayOrigin.BOTTOM_LEFT.__doc__ = "When the X and Y position are both set to 0, this value places the bottom, left corner of the overlay in its parent's bottom, left corner. Increasing X values move the overlay to the right and increasing Y values move it up."
ScreenOverlayOrigin.BOTTOM_CENTER.__doc__ = "When the X and Y position are both set to 0, this value places the bottom edge of the overlay at its parent's bottom edge and the center of the overlay is horizontally centered within its parent..."
ScreenOverlayOrigin.BOTTOM_RIGHT.__doc__ = "When the X and Y position are both set to 0, this value places the bottom, right corner of the overlay in its parent's bottom, right corner. Increasing X values move the overlay to the left and increasing Y values move it up."
ScreenOverlayOrigin.CENTER_LEFT.__doc__ = "When the X and Y position are both set to 0, this value places the left edge of the overlay at its parent's left edge and the center of the overlay is vertically centered within its parent..."
ScreenOverlayOrigin.CENTER.__doc__ = "When the X and Y position are both set to 0, this value places the center of the overlay at its parent's center. Increasing X values move the overlay to the right and increasing Y values move it up."
ScreenOverlayOrigin.CENTER_RIGHT.__doc__ = "When the X and Y position are both set to 0, this value places the right edge of the overlay at its parent's right edge and the center of the overlay is vertically centered within its parent..."
ScreenOverlayOrigin.TOP_LEFT.__doc__ = "When the X and Y position are both set to 0, this value places the top, left corner of the overlay in its parent's top, left corner. Increasing X values move the overlay to the right and increasing Y values move it down."
ScreenOverlayOrigin.TOP_CENTER.__doc__ = "When the X and Y position are both set to 0, this value places the top edge of the overlay at its parent's top edge and the center of the overlay is horizontally centered within its parent..."
ScreenOverlayOrigin.TOP_RIGHT.__doc__ = "When the X and Y position are both set to 0, this value places the top, right corner of the overlay in its parent's top, right corner. Increasing X values move the overlay to the left and increasing Y values move it down."

agcls.AgTypeNameMap["ScreenOverlayOrigin"] = ScreenOverlayOrigin

class ScreenOverlayPinningOrigin(IntEnum):
    """Specify the origin of the pinning position of the screen overlay, as well as the direction of the horizontal and vertical axes for that pinning position. The pinning origin specifies the origin of the pinning position in the overlay's coordinate system."""

    BOTTOM_LEFT = 0
    """When the X and Y pinning position are both set to 0, this value places the pinning position in the overlay's bottom, left corner. Increasing X values move the pinning position to the right and increasing Y values move it up."""
    BOTTOM_CENTER = 1
    """When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's bottom edge and the pinning position is horizontally centered within its overlay..."""
    BOTTOM_RIGHT = 2
    """When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's bottom, right corner. Increasing X values move the pinning position to the left and increasing Y values move it up."""
    CENTER_LEFT = 3
    """When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's left edge and the pinning position is vertically centered within its overlay..."""
    CENTER = 4
    """When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's center. Increasing X values move the pinning position to the right and increasing Y values move it up."""
    CENTER_RIGHT = 5
    """When the X and Y pinning pinning position are both set to 0, this value places the pinning position at its overlay's right edge and the pinning position is vertically centered within its overlay..."""
    TOP_LEFT = 6
    """When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's top, left corner. Increasing X values move the pinning position to the right and increasing Y values move it down."""
    TOP_CENTER = 7
    """When the X and Y pinning position are both set to 0, this value places the pinning position at its overlays's top edge and the pinning position is horizontally centered within its overlay..."""
    TOP_RIGHT = 8
    """When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's top, right corner. Increasing X values move the pinning position to the left and increasing Y values move it down."""
    AUTOMATIC = 9
    """The pinning origin is automatically set to the origin of the overlay. For instance, if the origin of the overlay is ScreenOverlayOrigin.BottomLeft, the pinning origin will also be equivalent to ScreenOverlayPinningOrigin.BottomLeft."""

ScreenOverlayPinningOrigin.BOTTOM_LEFT.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position in the overlay's bottom, left corner. Increasing X values move the pinning position to the right and increasing Y values move it up."
ScreenOverlayPinningOrigin.BOTTOM_CENTER.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's bottom edge and the pinning position is horizontally centered within its overlay..."
ScreenOverlayPinningOrigin.BOTTOM_RIGHT.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's bottom, right corner. Increasing X values move the pinning position to the left and increasing Y values move it up."
ScreenOverlayPinningOrigin.CENTER_LEFT.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's left edge and the pinning position is vertically centered within its overlay..."
ScreenOverlayPinningOrigin.CENTER.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position at its overlay's center. Increasing X values move the pinning position to the right and increasing Y values move it up."
ScreenOverlayPinningOrigin.CENTER_RIGHT.__doc__ = "When the X and Y pinning pinning position are both set to 0, this value places the pinning position at its overlay's right edge and the pinning position is vertically centered within its overlay..."
ScreenOverlayPinningOrigin.TOP_LEFT.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's top, left corner. Increasing X values move the pinning position to the right and increasing Y values move it down."
ScreenOverlayPinningOrigin.TOP_CENTER.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position at its overlays's top edge and the pinning position is horizontally centered within its overlay..."
ScreenOverlayPinningOrigin.TOP_RIGHT.__doc__ = "When the X and Y pinning position are both set to 0, this value places the pinning position in its overlay's top, right corner. Increasing X values move the pinning position to the left and increasing Y values move it down."
ScreenOverlayPinningOrigin.AUTOMATIC.__doc__ = "The pinning origin is automatically set to the origin of the overlay. For instance, if the origin of the overlay is ScreenOverlayOrigin.BottomLeft, the pinning origin will also be equivalent to ScreenOverlayPinningOrigin.BottomLeft."

agcls.AgTypeNameMap["ScreenOverlayPinningOrigin"] = ScreenOverlayPinningOrigin

class ScreenOverlayUnit(IntEnum):
    """A unit specifying how a screen overlay is sized and positioned relative to its parent."""

    PIXEL = 0
    """The value is specified in pixels."""
    PERCENT = 1
    """The value is specified as a fraction of the overlay's parent, where 0.0 is 0% and 1.0 is 100%."""

ScreenOverlayUnit.PIXEL.__doc__ = "The value is specified in pixels."
ScreenOverlayUnit.PERCENT.__doc__ = "The value is specified as a fraction of the overlay's parent, where 0.0 is 0% and 1.0 is 100%."

agcls.AgTypeNameMap["ScreenOverlayUnit"] = ScreenOverlayUnit

class SurfaceMeshRenderingMethod(IntEnum):
    """Rendering methods available for use by the surface mesh primitive. Different methods may have different performance characteristics and require different video card support. When in doubt, use Automatic."""

    GEOMETRY_SHADER = 0
    """Render the surface mesh using a geometry shader."""
    VERTEX_SHADER = 1
    """Render the surface mesh using a vertex shader."""
    AUTOMATIC = 2
    """Render the surface mesh using an automatically selected method based on the capabilities of the video card."""

SurfaceMeshRenderingMethod.GEOMETRY_SHADER.__doc__ = "Render the surface mesh using a geometry shader."
SurfaceMeshRenderingMethod.VERTEX_SHADER.__doc__ = "Render the surface mesh using a vertex shader."
SurfaceMeshRenderingMethod.AUTOMATIC.__doc__ = "Render the surface mesh using an automatically selected method based on the capabilities of the video card."

agcls.AgTypeNameMap["SurfaceMeshRenderingMethod"] = SurfaceMeshRenderingMethod

class Visibility(IntEnum):
    """Result of a visibility test, such as testing if a sphere intersects a frustum."""

    NONE = 0
    """The object is not visible."""
    PARTIAL = 1
    """The object is partially visible."""
    ALL = 2
    """The object is completely visible."""

Visibility.NONE.__doc__ = "The object is not visible."
Visibility.PARTIAL.__doc__ = "The object is partially visible."
Visibility.ALL.__doc__ = "The object is completely visible."

agcls.AgTypeNameMap["Visibility"] = Visibility

class AntiAliasingMethod(IntEnum):
    """The multisample anti-aliasing (MSAA) options for Scenes. As the level of anti-aliasing increases, performance will generally decrease, but the quality of the anti-aliasing will improve."""

    OFF = 0
    """No anti-aliasing."""
    FXAA = 1
    """Fast Approximate Anti aliasing."""
    TWO_X = 2
    """Anti-aliasing at 2x."""
    FOUR_X = 4
    """Anti-aliasing at 4x."""
    EIGHT_X = 8
    """Anti-aliasing at 8x."""
    SIXTEEN_X = 16
    """Anti-aliasing at 16x."""
    THIRTY_TWO_X = 32
    """Anti-aliasing at 32x."""
    SIXTY_FOUR_X = 64
    """Anti-aliasing at 64x."""

AntiAliasingMethod.OFF.__doc__ = "No anti-aliasing."
AntiAliasingMethod.FXAA.__doc__ = "Fast Approximate Anti aliasing."
AntiAliasingMethod.TWO_X.__doc__ = "Anti-aliasing at 2x."
AntiAliasingMethod.FOUR_X.__doc__ = "Anti-aliasing at 4x."
AntiAliasingMethod.EIGHT_X.__doc__ = "Anti-aliasing at 8x."
AntiAliasingMethod.SIXTEEN_X.__doc__ = "Anti-aliasing at 16x."
AntiAliasingMethod.THIRTY_TWO_X.__doc__ = "Anti-aliasing at 32x."
AntiAliasingMethod.SIXTY_FOUR_X.__doc__ = "Anti-aliasing at 64x."

agcls.AgTypeNameMap["AntiAliasingMethod"] = AntiAliasingMethod

class BinaryLogicOperation(IntEnum):
    """Binary logic operations that can be used by composite display condition."""

    AND = 0
    """Logically and together display conditions in the composite."""
    OR = 1
    """Logically or together display conditions in the composite."""

BinaryLogicOperation.AND.__doc__ = "Logically and together display conditions in the composite."
BinaryLogicOperation.OR.__doc__ = "Logically or together display conditions in the composite."

agcls.AgTypeNameMap["BinaryLogicOperation"] = BinaryLogicOperation

class BlurMethod(IntEnum):
    """The method used to blur or smooth a raster."""

    MEAN = 0
    """Provide smoothing and noise reduction through mean filtering."""
    BASIC = 1
    """Provide basic blurring."""

BlurMethod.MEAN.__doc__ = "Provide smoothing and noise reduction through mean filtering."
BlurMethod.BASIC.__doc__ = "Provide basic blurring."

agcls.AgTypeNameMap["BlurMethod"] = BlurMethod

class EdgeDetectMethod(IntEnum):
    """The method used to detect edges in a raster."""

    VERTICAL = 0
    """Detects vertical edges."""
    HORIZONTAL = 1
    """Detects horizontal edges."""
    LEFT_DIAGONAL = 2
    """Detects left diagonal edges."""
    RIGHT_DIAGONAL = 3
    """Detects right diagonal edges."""
    LAPLACIAN = 4
    """Detects edges using the Laplacian method."""
    PREWITT_LAPLACIAN = 5
    """Detects edges using the Prewitt-Laplacian method."""
    SOBEL_VERTICAL = 6
    """Detects vertical edges using the Sobel method."""
    SOBEL_HORIZONTAL = 7
    """Detects horizontal edges using the Sobel method."""

EdgeDetectMethod.VERTICAL.__doc__ = "Detects vertical edges."
EdgeDetectMethod.HORIZONTAL.__doc__ = "Detects horizontal edges."
EdgeDetectMethod.LEFT_DIAGONAL.__doc__ = "Detects left diagonal edges."
EdgeDetectMethod.RIGHT_DIAGONAL.__doc__ = "Detects right diagonal edges."
EdgeDetectMethod.LAPLACIAN.__doc__ = "Detects edges using the Laplacian method."
EdgeDetectMethod.PREWITT_LAPLACIAN.__doc__ = "Detects edges using the Prewitt-Laplacian method."
EdgeDetectMethod.SOBEL_VERTICAL.__doc__ = "Detects vertical edges using the Sobel method."
EdgeDetectMethod.SOBEL_HORIZONTAL.__doc__ = "Detects horizontal edges using the Sobel method."

agcls.AgTypeNameMap["EdgeDetectMethod"] = EdgeDetectMethod

class RasterFlipAxis(IntEnum):
    """The axis on which a raster will be flipped."""

    HORIZONTAL = 0
    """The horizontal axis."""
    VERTICAL = 1
    """The vertical axis."""

RasterFlipAxis.HORIZONTAL.__doc__ = "The horizontal axis."
RasterFlipAxis.VERTICAL.__doc__ = "The vertical axis."

agcls.AgTypeNameMap["RasterFlipAxis"] = RasterFlipAxis

class GradientDetectMethod(IntEnum):
    """The method used to detect gradients in a raster. Gradient detection is commonly referred to as embossing."""

    EAST = 0
    """Detects east gradients."""
    NORTH = 1
    """Detects north gradients."""
    WEST = 2
    """Detects west gradients."""
    SOUTH = 3
    """Detects south gradients."""
    NORTH_EAST = 4
    """Detects north east gradients."""
    NORTH_WEST = 5
    """Detects north west gradients."""
    SOUTH_EAST = 6
    """Detects south east gradients."""
    SOUTH_WEST = 7
    """Detects south west gradients."""

GradientDetectMethod.EAST.__doc__ = "Detects east gradients."
GradientDetectMethod.NORTH.__doc__ = "Detects north gradients."
GradientDetectMethod.WEST.__doc__ = "Detects west gradients."
GradientDetectMethod.SOUTH.__doc__ = "Detects south gradients."
GradientDetectMethod.NORTH_EAST.__doc__ = "Detects north east gradients."
GradientDetectMethod.NORTH_WEST.__doc__ = "Detects north west gradients."
GradientDetectMethod.SOUTH_EAST.__doc__ = "Detects south east gradients."
GradientDetectMethod.SOUTH_WEST.__doc__ = "Detects south west gradients."

agcls.AgTypeNameMap["GradientDetectMethod"] = GradientDetectMethod

class Jpeg2000CompressionProfile(IntEnum):
    """Define the profile used when encoding a JPEG 2000 file."""

    DEFAULT = 0
    """This is the default profile, which is recommended for those unfamiliar with the others."""
    NITF_BIIF_NPJE = 1
    """This profile is designed for U.S. and NATO military applications."""
    NITF_BIIF_EPJE = 2
    """This profile is designed for U.S. and NATO military applications. Based on NPJE, the profile is used for image exploitation, and improves image read times for lower resolutions in large images as compared to NPJE."""

Jpeg2000CompressionProfile.DEFAULT.__doc__ = "This is the default profile, which is recommended for those unfamiliar with the others."
Jpeg2000CompressionProfile.NITF_BIIF_NPJE.__doc__ = "This profile is designed for U.S. and NATO military applications."
Jpeg2000CompressionProfile.NITF_BIIF_EPJE.__doc__ = "This profile is designed for U.S. and NATO military applications. Based on NPJE, the profile is used for image exploitation, and improves image read times for lower resolutions in large images as compared to NPJE."

agcls.AgTypeNameMap["Jpeg2000CompressionProfile"] = Jpeg2000CompressionProfile

class RasterBand(IntEnum):
    """Common band types that may be contained within a raster dataset. Each band can be thought of as a set of values, which are most commonly associated with colors when the raster represents an image..."""

    RED = 0
    """A band containing red values."""
    GREEN = 1
    """A band containing green values."""
    BLUE = 2
    """A band containing blue values."""
    ALPHA = 3
    """A band containing alpha values."""
    LUMINANCE = 4
    """A band containing luminance values."""

RasterBand.RED.__doc__ = "A band containing red values."
RasterBand.GREEN.__doc__ = "A band containing green values."
RasterBand.BLUE.__doc__ = "A band containing blue values."
RasterBand.ALPHA.__doc__ = "A band containing alpha values."
RasterBand.LUMINANCE.__doc__ = "A band containing luminance values."

agcls.AgTypeNameMap["RasterBand"] = RasterBand

class RasterFormat(IntEnum):
    """Common raster band layouts that may be contained within a raster dataset. Each pixel of the raster will contain the bands defined by the layout in the specified order. A typical color raster image will have an rgbraster format."""

    RED = 0
    """The format of the raster is a single band consisting of red values."""
    GREEN = 1
    """The format of the raster is a single band consisting of green values."""
    BLUE = 2
    """The format of the raster is a single band consisting of blue values."""
    ALPHA = 3
    """The format of the raster is a single band consisting of alpha values."""
    RGB = 4
    """The format of the raster is three bands consisting of red, green, and blue values."""
    BGR = 5
    """The format of the raster is three bands consisting of blue, green, and red values."""
    RGBA = 6
    """The format of the raster is four bands consisting of red, green, blue, and alpha values."""
    BGRA = 7
    """The format of the raster is four bands consisting of blue, green, red, and alpha values."""
    LUMINANCE = 8
    """The format of the raster is a single band consisting of luminance values."""
    LUMINANCE_ALPHA = 9
    """The format of the raster is two bands consisting of luminance and alpha values."""

RasterFormat.RED.__doc__ = "The format of the raster is a single band consisting of red values."
RasterFormat.GREEN.__doc__ = "The format of the raster is a single band consisting of green values."
RasterFormat.BLUE.__doc__ = "The format of the raster is a single band consisting of blue values."
RasterFormat.ALPHA.__doc__ = "The format of the raster is a single band consisting of alpha values."
RasterFormat.RGB.__doc__ = "The format of the raster is three bands consisting of red, green, and blue values."
RasterFormat.BGR.__doc__ = "The format of the raster is three bands consisting of blue, green, and red values."
RasterFormat.RGBA.__doc__ = "The format of the raster is four bands consisting of red, green, blue, and alpha values."
RasterFormat.BGRA.__doc__ = "The format of the raster is four bands consisting of blue, green, red, and alpha values."
RasterFormat.LUMINANCE.__doc__ = "The format of the raster is a single band consisting of luminance values."
RasterFormat.LUMINANCE_ALPHA.__doc__ = "The format of the raster is two bands consisting of luminance and alpha values."

agcls.AgTypeNameMap["RasterFormat"] = RasterFormat

class RasterOrientation(IntEnum):
    """The vertical orientation of the raster."""

    TOP_TO_BOTTOM = 0
    """The raster has a top to bottom orientation."""
    BOTTOM_TO_TOP = 1
    """The raster has a bottom to top orientation."""

RasterOrientation.TOP_TO_BOTTOM.__doc__ = "The raster has a top to bottom orientation."
RasterOrientation.BOTTOM_TO_TOP.__doc__ = "The raster has a bottom to top orientation."

agcls.AgTypeNameMap["RasterOrientation"] = RasterOrientation

class RasterType(IntEnum):
    """The type of data contained within each band of a raster dataset."""

    UNSIGNED_BYTE = 0
    """Each value contained within a band can be represented with an unsigned byte."""
    BYTE = 1
    """Each value contained within a band can be represented with a byte."""
    UNSIGNED_SHORT = 2
    """Each value contained within a band can be represented with an unsigned short."""
    SHORT = 3
    """Each value contained within a band can be represented with a short."""
    UNSIGNED_INT = 4
    """Each value contained within a band can be represented with an unsigned int."""
    INT = 5
    """Each value contained within a band can be represented with an int."""
    FLOAT = 6
    """Each value contained within a band can be represented with a float."""
    DOUBLE = 7
    """Each value contained within a band can be represented with a double."""

RasterType.UNSIGNED_BYTE.__doc__ = "Each value contained within a band can be represented with an unsigned byte."
RasterType.BYTE.__doc__ = "Each value contained within a band can be represented with a byte."
RasterType.UNSIGNED_SHORT.__doc__ = "Each value contained within a band can be represented with an unsigned short."
RasterType.SHORT.__doc__ = "Each value contained within a band can be represented with a short."
RasterType.UNSIGNED_INT.__doc__ = "Each value contained within a band can be represented with an unsigned int."
RasterType.INT.__doc__ = "Each value contained within a band can be represented with an int."
RasterType.FLOAT.__doc__ = "Each value contained within a band can be represented with a float."
RasterType.DOUBLE.__doc__ = "Each value contained within a band can be represented with a double."

agcls.AgTypeNameMap["RasterType"] = RasterType

class RasterSharpenMethod(IntEnum):
    """The method used to sharpen a raster."""

    MEAN_REMOVAL = 0
    """Provide sharpening through mean removal."""
    BASIC = 1
    """Provide basic sharpening."""

RasterSharpenMethod.MEAN_REMOVAL.__doc__ = "Provide sharpening through mean removal."
RasterSharpenMethod.BASIC.__doc__ = "Provide basic sharpening."

agcls.AgTypeNameMap["RasterSharpenMethod"] = RasterSharpenMethod

class VideoPlayback(IntEnum):
    """Specify how the video stream will playback. When the playback is set to real time, the video will playback in real time..."""

    REAL_TIME = 0
    """The video will playback in realtime."""
    MAPPED = 1
    """The video will playback in coincidence with the scene manager time within the interval defined by interval start time and interval end time."""

VideoPlayback.REAL_TIME.__doc__ = "The video will playback in realtime."
VideoPlayback.MAPPED.__doc__ = "The video will playback in coincidence with the scene manager time within the interval defined by interval start time and interval end time."

agcls.AgTypeNameMap["VideoPlayback"] = VideoPlayback

class KmlNetworkLinkRefreshMode(IntEnum):
    """Define the options available for a KmlNetworkLink's RefreshMode property."""

    ON_CHANGE = 0
    """Refresh when the document is loaded and whenever the Link parameters change (the default)."""
    ON_INTERVAL = 1
    """Refresh the network link at the duration specified by refresh interval."""
    ON_EXPIRE = 2
    """Refresh the network link when the expiration time is reached."""

KmlNetworkLinkRefreshMode.ON_CHANGE.__doc__ = "Refresh when the document is loaded and whenever the Link parameters change (the default)."
KmlNetworkLinkRefreshMode.ON_INTERVAL.__doc__ = "Refresh the network link at the duration specified by refresh interval."
KmlNetworkLinkRefreshMode.ON_EXPIRE.__doc__ = "Refresh the network link when the expiration time is reached."

agcls.AgTypeNameMap["KmlNetworkLinkRefreshMode"] = KmlNetworkLinkRefreshMode

class KmlNetworkLinkViewRefreshMode(IntEnum):
    """Define the options available for a KmlNetworkLink's ViewRefreshMode property."""

    NEVER = 0
    """Do not refresh the network link when the camera's view changes."""
    ON_REQUEST = 1
    """Only refresh the network link when it is explicitly refreshed by calling the refresh method."""
    ON_STOP = 2
    """Refresh the network link after camera view movement stops for the duration specified by view refresh time."""
    ON_REGION = 3
    """Refresh the network link when the region associated with this link becomes active."""

KmlNetworkLinkViewRefreshMode.NEVER.__doc__ = "Do not refresh the network link when the camera's view changes."
KmlNetworkLinkViewRefreshMode.ON_REQUEST.__doc__ = "Only refresh the network link when it is explicitly refreshed by calling the refresh method."
KmlNetworkLinkViewRefreshMode.ON_STOP.__doc__ = "Refresh the network link after camera view movement stops for the duration specified by view refresh time."
KmlNetworkLinkViewRefreshMode.ON_REGION.__doc__ = "Refresh the network link when the region associated with this link becomes active."

agcls.AgTypeNameMap["KmlNetworkLinkViewRefreshMode"] = KmlNetworkLinkViewRefreshMode

class ModelUpAxis(IntEnum):
    """When setting the camera'saxes, this defines which axis of the axes is up in screen space, where up is from the bottom to the top of the screen."""

    X = 0
    """The positive X axis."""
    Y = 1
    """The positive Y axis."""
    Z = 2
    """The positive Z axis."""
    NEGATIVE_X = 3
    """The negative X axis."""
    NEGATIVE_Y = 4
    """The negative Y axis."""
    NEGATIVE_Z = 5
    """The negative Z axis."""

ModelUpAxis.X.__doc__ = "The positive X axis."
ModelUpAxis.Y.__doc__ = "The positive Y axis."
ModelUpAxis.Z.__doc__ = "The positive Z axis."
ModelUpAxis.NEGATIVE_X.__doc__ = "The negative X axis."
ModelUpAxis.NEGATIVE_Y.__doc__ = "The negative Y axis."
ModelUpAxis.NEGATIVE_Z.__doc__ = "The negative Z axis."

agcls.AgTypeNameMap["ModelUpAxis"] = ModelUpAxis

class OutlineAppearance(IntEnum):
    """Possible appearances of an outline. Front lines are lines on front facing geometry and back lines are lines on back facing geometry."""

    FRONT_AND_BACK_LINES = 0
    """Both the front and back lines are displayed."""
    FRONT_LINES_ONLY = 1
    """Only the front lines are displayed. This can be used to declutter the outline."""
    STYLIZE_BACK_LINES = 2
    """Both the front and back lines are displayed. The back lines are displayed using a different color, translucency, and width. This is used to declutter the outline but still provide a visual cue for the back facing geometry."""

OutlineAppearance.FRONT_AND_BACK_LINES.__doc__ = "Both the front and back lines are displayed."
OutlineAppearance.FRONT_LINES_ONLY.__doc__ = "Only the front lines are displayed. This can be used to declutter the outline."
OutlineAppearance.STYLIZE_BACK_LINES.__doc__ = "Both the front and back lines are displayed. The back lines are displayed using a different color, translucency, and width. This is used to declutter the outline but still provide a visual cue for the back facing geometry."

agcls.AgTypeNameMap["OutlineAppearance"] = OutlineAppearance

class PolylineType(IntEnum):
    """Describes how to interpret positions defining a polyline."""

    LINES = 0
    """Every two positions define a line segment. Line segments are not required to be connected to each other."""
    LINE_STRIP = 1
    """After the first position, each additional position defines a line segment from the previous position to the current position."""
    POINTS = 2
    """Lines are drawn as points."""

PolylineType.LINES.__doc__ = "Every two positions define a line segment. Line segments are not required to be connected to each other."
PolylineType.LINE_STRIP.__doc__ = "After the first position, each additional position defines a line segment from the previous position to the current position."
PolylineType.POINTS.__doc__ = "Lines are drawn as points."

agcls.AgTypeNameMap["PolylineType"] = PolylineType

class FaceCullingMode(IntEnum):
    """Identifies whether front- and/or back-facing triangles are culled."""

    CULL_FACE_FRONT = 0
    """Front-facing triangles are culled."""
    CULL_FACE_BACK = 1
    """Back-facing triangles are culled."""
    CULL_FACE_FRONT_AND_BACK = 2
    """Both front- and back-facing triangles are culled. Therefore, no triangles are rendered."""
    CULL_FACE_NEITHER = 3
    """No triangles are culled."""

FaceCullingMode.CULL_FACE_FRONT.__doc__ = "Front-facing triangles are culled."
FaceCullingMode.CULL_FACE_BACK.__doc__ = "Back-facing triangles are culled."
FaceCullingMode.CULL_FACE_FRONT_AND_BACK.__doc__ = "Both front- and back-facing triangles are culled. Therefore, no triangles are rendered."
FaceCullingMode.CULL_FACE_NEITHER.__doc__ = "No triangles are culled."

agcls.AgTypeNameMap["FaceCullingMode"] = FaceCullingMode

class TextureFormat(IntEnum):
    """The format of individual texels in a texture."""

    ALPHA4 = 0
    """Each texel is an 4-bit alpha component."""
    ALPHA8 = 1
    """Each texel is an 8-bit alpha component."""
    ALPHA12 = 2
    """Each texel is an 12-bit alpha component."""
    ALPHA16 = 3
    """Each texel is an 16-bit alpha component."""
    R3G3B2 = 4
    """Each texel contains 3-bit red and green components and 2-bit blue component."""
    RGB4 = 5
    """Each texel contains 4-bit red, green, and blue components."""
    RGB5 = 6
    """Each texel contains 5-bit red, green, and blue components."""
    RGB8 = 7
    """Each texel contains 8-bit red, green, and blue components."""
    RGB10 = 8
    """Each texel contains 10-bit red, green, and blue components."""
    RGB12 = 9
    """Each texel contains 12-bit red, green, and blue components."""
    RGB16 = 10
    """Each texel contains 16-bit red, green, and blue components."""
    RGB16_F = 11
    """Each texel contains 16-bit red, green, and blue floating point components."""
    RGB32_F = 12
    """Each texel contains 32-bit red, green, and blue floating point components."""
    RGBA2 = 13
    """Each texel contains 2-bit red, green, blue, and alpha components."""
    RGBA4 = 14
    """Each texel contains 4-bit red, green, blue, and alpha components."""
    RGB5_A1 = 15
    """Each texel contains 5-bit red, green, blue components and 1-bit alpha component."""
    RGBA8 = 16
    """Each texel contains 8-bit red, green, blue, and alpha components."""
    RGB10_A2 = 17
    """Each texel contains 10-bit red, green, blue components and 2-bit alpha component."""
    RGBA12 = 18
    """Each texel contains 12-bit red, green, blue, and alpha components."""
    RGBA16 = 19
    """Each texel contains 16-bit red, green, blue, and alpha components."""
    RGBA16_F = 20
    """Each texel contains 16-bit red, green, blue, and alpha floating point components."""
    RGBA32_F = 21
    """Each texel contains 32-bit red, green, blue, and alpha floating point components."""
    LUMINANCE4 = 22
    """Each texel is an 4-bit luminance, e.g. intensity, component."""
    LUMINANCE8 = 23
    """Each texel is an 8-bit luminance, e.g. intensity, component."""
    LUMINANCE12 = 24
    """Each texel is an 12-bit luminance, e.g. intensity, component."""
    LUMINANCE16 = 25
    """Each texel is an 16-bit luminance, e.g. intensity, component."""
    LUMINANCE16_F = 26
    """Each texel is an 16-bit luminance, e.g. intensity, floating point component."""
    LUMINANCE32_F = 27
    """Each texel is an 32-bit luminance, e.g. intensity, floating point component."""
    LUMINANCE4_ALPHA4 = 28
    """Each texel contains an 4-bit luminance and 4-bit alpha component."""
    LUMINANCE6_ALPHA2 = 29
    """Each texel contains an 6-bit luminance and 2-bit alpha component."""
    LUMINANCE8_ALPHA8 = 30
    """Each texel contains an 8-bit luminance and 8-bit alpha component."""
    LUMINANCE12_ALPHA4 = 31
    """Each texel contains an 12-bit luminance and 4-bit alpha component."""
    LUMINANCE12_ALPHA12 = 32
    """Each texel contains an 12-bit luminance and 12-bit alpha component."""
    LUMINANCE16_ALPHA16 = 33
    """Each texel contains an 16-bit luminance and 16-bit alpha component."""
    LUMINANCE16_ALPHA16_F = 34
    """Each texel contains an 16-bit luminance and 16-bit alpha floating point component."""
    LUMINANCE32_ALPHA32_F = 35
    """Each texel contains an 32-bit luminance and 32-bit alpha floating point component."""

TextureFormat.ALPHA4.__doc__ = "Each texel is an 4-bit alpha component."
TextureFormat.ALPHA8.__doc__ = "Each texel is an 8-bit alpha component."
TextureFormat.ALPHA12.__doc__ = "Each texel is an 12-bit alpha component."
TextureFormat.ALPHA16.__doc__ = "Each texel is an 16-bit alpha component."
TextureFormat.R3G3B2.__doc__ = "Each texel contains 3-bit red and green components and 2-bit blue component."
TextureFormat.RGB4.__doc__ = "Each texel contains 4-bit red, green, and blue components."
TextureFormat.RGB5.__doc__ = "Each texel contains 5-bit red, green, and blue components."
TextureFormat.RGB8.__doc__ = "Each texel contains 8-bit red, green, and blue components."
TextureFormat.RGB10.__doc__ = "Each texel contains 10-bit red, green, and blue components."
TextureFormat.RGB12.__doc__ = "Each texel contains 12-bit red, green, and blue components."
TextureFormat.RGB16.__doc__ = "Each texel contains 16-bit red, green, and blue components."
TextureFormat.RGB16_F.__doc__ = "Each texel contains 16-bit red, green, and blue floating point components."
TextureFormat.RGB32_F.__doc__ = "Each texel contains 32-bit red, green, and blue floating point components."
TextureFormat.RGBA2.__doc__ = "Each texel contains 2-bit red, green, blue, and alpha components."
TextureFormat.RGBA4.__doc__ = "Each texel contains 4-bit red, green, blue, and alpha components."
TextureFormat.RGB5_A1.__doc__ = "Each texel contains 5-bit red, green, blue components and 1-bit alpha component."
TextureFormat.RGBA8.__doc__ = "Each texel contains 8-bit red, green, blue, and alpha components."
TextureFormat.RGB10_A2.__doc__ = "Each texel contains 10-bit red, green, blue components and 2-bit alpha component."
TextureFormat.RGBA12.__doc__ = "Each texel contains 12-bit red, green, blue, and alpha components."
TextureFormat.RGBA16.__doc__ = "Each texel contains 16-bit red, green, blue, and alpha components."
TextureFormat.RGBA16_F.__doc__ = "Each texel contains 16-bit red, green, blue, and alpha floating point components."
TextureFormat.RGBA32_F.__doc__ = "Each texel contains 32-bit red, green, blue, and alpha floating point components."
TextureFormat.LUMINANCE4.__doc__ = "Each texel is an 4-bit luminance, e.g. intensity, component."
TextureFormat.LUMINANCE8.__doc__ = "Each texel is an 8-bit luminance, e.g. intensity, component."
TextureFormat.LUMINANCE12.__doc__ = "Each texel is an 12-bit luminance, e.g. intensity, component."
TextureFormat.LUMINANCE16.__doc__ = "Each texel is an 16-bit luminance, e.g. intensity, component."
TextureFormat.LUMINANCE16_F.__doc__ = "Each texel is an 16-bit luminance, e.g. intensity, floating point component."
TextureFormat.LUMINANCE32_F.__doc__ = "Each texel is an 32-bit luminance, e.g. intensity, floating point component."
TextureFormat.LUMINANCE4_ALPHA4.__doc__ = "Each texel contains an 4-bit luminance and 4-bit alpha component."
TextureFormat.LUMINANCE6_ALPHA2.__doc__ = "Each texel contains an 6-bit luminance and 2-bit alpha component."
TextureFormat.LUMINANCE8_ALPHA8.__doc__ = "Each texel contains an 8-bit luminance and 8-bit alpha component."
TextureFormat.LUMINANCE12_ALPHA4.__doc__ = "Each texel contains an 12-bit luminance and 4-bit alpha component."
TextureFormat.LUMINANCE12_ALPHA12.__doc__ = "Each texel contains an 12-bit luminance and 12-bit alpha component."
TextureFormat.LUMINANCE16_ALPHA16.__doc__ = "Each texel contains an 16-bit luminance and 16-bit alpha component."
TextureFormat.LUMINANCE16_ALPHA16_F.__doc__ = "Each texel contains an 16-bit luminance and 16-bit alpha floating point component."
TextureFormat.LUMINANCE32_ALPHA32_F.__doc__ = "Each texel contains an 32-bit luminance and 32-bit alpha floating point component."

agcls.AgTypeNameMap["TextureFormat"] = TextureFormat

class MagnificationFilter(IntEnum):
    """The filter used when the pixel being textured maps to an area less than or equal to one texel."""

    NEAREST = 0
    """Use the texel that is closest to the center of the pixel being textured. This usually faster than Linear but can produce images with sharper edges."""
    LINEAR = 1
    """Use the weighted average of the four texels that are closest to the center of the pixel being textured."""

MagnificationFilter.NEAREST.__doc__ = "Use the texel that is closest to the center of the pixel being textured. This usually faster than Linear but can produce images with sharper edges."
MagnificationFilter.LINEAR.__doc__ = "Use the weighted average of the four texels that are closest to the center of the pixel being textured."

agcls.AgTypeNameMap["MagnificationFilter"] = MagnificationFilter

class MinificationFilter(IntEnum):
    """The filter used when the pixel being textured maps to an area greater than one texel."""

    NEAREST = 0
    """Use the texel that is closest to the center of the pixel being textured."""
    LINEAR = 1
    """Use the weighted average of the four (for 2D textures, two for 1D textures) texels that are closest to the center of the pixel being textured."""
    NEAREST_MIP_MAP_NEAREST = 2
    """Use the mipmap that most closely matches the size of the pixel being textured. Then use the texel from that mipmap that is closest to the center of the pixel being textured."""
    LINEAR_MIP_MAP_NEAREST = 3
    """Use the mipmap that most closely matches the size of the pixel being textured. Then use the weighted average of the four (for 2D textures, two for 1D textures) texels from that mipmap that are closest to the center of the pixel being textured."""
    NEAREST_MIP_MAP_LINEAR = 4
    """Use the two mipmaps that most closely match the size of the pixel being textured. Determine the texel that is closest to the center of the pixel being textured in each mipmap. The final texture value is a weighted average of these two texels."""
    LINEAR_MIP_MAP_LINEAR = 5
    """Use the two mipmaps that most closely match the size of the pixel being textured. Determine the weighted average of the four (for 2D textures, two for 1D textures) texels that are closest to the center of the pixel being textured in each mipmap..."""

MinificationFilter.NEAREST.__doc__ = "Use the texel that is closest to the center of the pixel being textured."
MinificationFilter.LINEAR.__doc__ = "Use the weighted average of the four (for 2D textures, two for 1D textures) texels that are closest to the center of the pixel being textured."
MinificationFilter.NEAREST_MIP_MAP_NEAREST.__doc__ = "Use the mipmap that most closely matches the size of the pixel being textured. Then use the texel from that mipmap that is closest to the center of the pixel being textured."
MinificationFilter.LINEAR_MIP_MAP_NEAREST.__doc__ = "Use the mipmap that most closely matches the size of the pixel being textured. Then use the weighted average of the four (for 2D textures, two for 1D textures) texels from that mipmap that are closest to the center of the pixel being textured."
MinificationFilter.NEAREST_MIP_MAP_LINEAR.__doc__ = "Use the two mipmaps that most closely match the size of the pixel being textured. Determine the texel that is closest to the center of the pixel being textured in each mipmap. The final texture value is a weighted average of these two texels."
MinificationFilter.LINEAR_MIP_MAP_LINEAR.__doc__ = "Use the two mipmaps that most closely match the size of the pixel being textured. Determine the weighted average of the four (for 2D textures, two for 1D textures) texels that are closest to the center of the pixel being textured in each mipmap..."

agcls.AgTypeNameMap["MinificationFilter"] = MinificationFilter

class RendererShadingModel(IntEnum):
    """Identifies which shade model to use. The primitive can be drawn with a single color or multiple colors."""

    FLAT = 0
    """Flat shading renders the primitive with one color across all of the vertices."""
    GOURAUD = 1
    """Gouraud shading renders the primitive by interpolating the color between the vertices."""

RendererShadingModel.FLAT.__doc__ = "Flat shading renders the primitive with one color across all of the vertices."
RendererShadingModel.GOURAUD.__doc__ = "Gouraud shading renders the primitive by interpolating the color between the vertices."

agcls.AgTypeNameMap["RendererShadingModel"] = RendererShadingModel

class TextureWrap(IntEnum):
    """Determine how to handle textures coordinates that fall outside of the range [0, 1]."""

    CLAMP = 0
    """Clamp the texture coordinate to the range [0, 1]."""
    CLAMP_TO_BORDER = 1
    """Clamp the texture coordinate to the range [-1/2N, 1 + 1/2N], where N is the size the texture in the direction of clamping."""
    CLAMP_TO_EDGE = 2
    """Clamp the texture coordinate to the range [1/2N, 1 - 1/2N], where N is the size the texture in the direction of clamping."""
    MIRRORED_REPEAT = 3
    """If the integer part of the texture coordinate is even, use the fractional part of the texture coordinate. Otherwise, use one minus the fractional part of the texture coordinate."""
    REPEAT = 4
    """Ignore the integer part of the texture coordinate."""

TextureWrap.CLAMP.__doc__ = "Clamp the texture coordinate to the range [0, 1]."
TextureWrap.CLAMP_TO_BORDER.__doc__ = "Clamp the texture coordinate to the range [-1/2N, 1 + 1/2N], where N is the size the texture in the direction of clamping."
TextureWrap.CLAMP_TO_EDGE.__doc__ = "Clamp the texture coordinate to the range [1/2N, 1 - 1/2N], where N is the size the texture in the direction of clamping."
TextureWrap.MIRRORED_REPEAT.__doc__ = "If the integer part of the texture coordinate is even, use the fractional part of the texture coordinate. Otherwise, use one minus the fractional part of the texture coordinate."
TextureWrap.REPEAT.__doc__ = "Ignore the integer part of the texture coordinate."

agcls.AgTypeNameMap["TextureWrap"] = TextureWrap

class SetHint(IntEnum):
    """An optimization hint optionally provided to primitives to enhance performance for static or dynamic primitives. See the Set Hint Performance Overview for selecting an appropriate value."""

    INFREQUENT = 0
    """Rendering is optimized for static geometry. The primitive's vertices are not going to be updated with Set() or SetPartial() calls. Calls to SetPartial() will fail. Calls to Set() are allowed but may not be as efficient as SetHintFrequent."""
    PARTIAL = 1
    """Rendering is optimized for dynamic geometry. The primitive's vertices are expected to be updated with SetPartial() - some or all of the vertices will change but the number of vertices will not."""
    FREQUENT = 2
    """Rendering is optimized for streaming geometry. The primitive's vertices are expected to be updated with Set() - all the vertices will change and/or the number of vertices will change. Calls to SetPartial() will fail."""

SetHint.INFREQUENT.__doc__ = "Rendering is optimized for static geometry. The primitive's vertices are not going to be updated with Set() or SetPartial() calls. Calls to SetPartial() will fail. Calls to Set() are allowed but may not be as efficient as SetHintFrequent."
SetHint.PARTIAL.__doc__ = "Rendering is optimized for dynamic geometry. The primitive's vertices are expected to be updated with SetPartial() - some or all of the vertices will change but the number of vertices will not."
SetHint.FREQUENT.__doc__ = "Rendering is optimized for streaming geometry. The primitive's vertices are expected to be updated with Set() - all the vertices will change and/or the number of vertices will change. Calls to SetPartial() will fail."

agcls.AgTypeNameMap["SetHint"] = SetHint

class StereoProjectionMode(IntEnum):
    """The stereoscopic projection mode used for the left and right eye scenes."""

    PARALLEL = 0
    """Parallel projection. Parallel projection will ignore any projection distance that has been set and instead use a parallel projection for each eye. In parallel mode all objects will appear to go into the screen..."""
    FIXED_DISTANCE = 1
    """Fixed distance projection. Objects at the fixed distance will appear to have no depth. Objects further than the distance will appear to go into the screen. Objects nearer than the distance will appear to pop out of the screen."""
    AUTOMATIC = 2
    """Automatic distance projection. Automatic distance projection will ignore any projection distance that has been set and instead automatically calculates the projection distance based on the distance between the camera and the center of the scene."""

StereoProjectionMode.PARALLEL.__doc__ = "Parallel projection. Parallel projection will ignore any projection distance that has been set and instead use a parallel projection for each eye. In parallel mode all objects will appear to go into the screen..."
StereoProjectionMode.FIXED_DISTANCE.__doc__ = "Fixed distance projection. Objects at the fixed distance will appear to have no depth. Objects further than the distance will appear to go into the screen. Objects nearer than the distance will appear to pop out of the screen."
StereoProjectionMode.AUTOMATIC.__doc__ = "Automatic distance projection. Automatic distance projection will ignore any projection distance that has been set and instead automatically calculates the projection distance based on the distance between the camera and the center of the scene."

agcls.AgTypeNameMap["StereoProjectionMode"] = StereoProjectionMode

class StereoscopicDisplayMode(IntEnum):
    """The stereoscopic display mode. To use a particular stereoscopic display mode, ensure that your system supports the feature and that it is enabled."""

    OFF = 0
    """No stereoscopic."""
    QUAD_BUFFER = 1
    """OpenGL Quad-buffer stereoscopic."""
    ANAGLYPH = 2
    """Anaglyph or two-color stereoscopic."""
    LEFT_EYE = 3
    """Left eye view of the stereoscopic scene."""
    RIGHT_EYE = 4
    """Right eye view of the stereoscopic scene."""
    SIDE_BY_SIDE = 5
    """Side-by-side stereoscopic. Left and right eye views are rendered next to each other in the same window."""

StereoscopicDisplayMode.OFF.__doc__ = "No stereoscopic."
StereoscopicDisplayMode.QUAD_BUFFER.__doc__ = "OpenGL Quad-buffer stereoscopic."
StereoscopicDisplayMode.ANAGLYPH.__doc__ = "Anaglyph or two-color stereoscopic."
StereoscopicDisplayMode.LEFT_EYE.__doc__ = "Left eye view of the stereoscopic scene."
StereoscopicDisplayMode.RIGHT_EYE.__doc__ = "Right eye view of the stereoscopic scene."
StereoscopicDisplayMode.SIDE_BY_SIDE.__doc__ = "Side-by-side stereoscopic. Left and right eye views are rendered next to each other in the same window."

agcls.AgTypeNameMap["StereoscopicDisplayMode"] = StereoscopicDisplayMode

class FontStyle(IntEnum):
    """Font styles."""

    REGULAR = 0x0
    """Regular font."""
    BOLD = 0x1
    """Bold font."""
    ITALIC = 0x2
    """Italic font."""
    UNDERLINE = 0x4
    """Underlined font."""
    STRIKEOUT = 0x8
    """Strike-out style."""

FontStyle.REGULAR.__doc__ = "Regular font."
FontStyle.BOLD.__doc__ = "Bold font."
FontStyle.ITALIC.__doc__ = "Italic font."
FontStyle.UNDERLINE.__doc__ = "Underlined font."
FontStyle.STRIKEOUT.__doc__ = "Strike-out style."

agcls.AgTypeNameMap["FontStyle"] = FontStyle


class IScreenOverlayContainer(object):
    """The interface for screen overlays that contain a collection of other overlays. This interface is implemented by ScreenOverlayManager and ScreenOverlay."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_overlays_method_offset = 1
    _get_padding_method_offset = 2
    _set_padding_method_offset = 3
    _get_display_method_offset = 4
    _set_display_method_offset = 5
    _metadata = {
        "iid_data" : (5104812672920574391, 4741981752173751691),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IScreenOverlayContainer."""
        initialize_from_source_object(self, source_object, IScreenOverlayContainer)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IScreenOverlayContainer)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IScreenOverlayContainer, None)

    _get_overlays_metadata = { "offset" : _get_overlays_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlays(self) -> "ScreenOverlayCollection":
        """Get the collection of overlays that are contained within this overlay."""
        return self._intf.get_property(IScreenOverlayContainer._metadata, IScreenOverlayContainer._get_overlays_metadata)

    _get_padding_metadata = { "offset" : _get_padding_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def padding(self) -> list:
        """Get or set the padding surrounding the overlays that are contained within this overlay. The array contains the components of the padding arranged in the order left, top, right, bottom."""
        return self._intf.get_property(IScreenOverlayContainer._metadata, IScreenOverlayContainer._get_padding_metadata)

    _set_padding_metadata = { "offset" : _set_padding_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @padding.setter
    def padding(self, padding:list) -> None:
        return self._intf.set_property(IScreenOverlayContainer._metadata, IScreenOverlayContainer._set_padding_metadata, padding)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set if this overlay and the collection of overlays that are contained within this overlay should be rendered."""
        return self._intf.get_property(IScreenOverlayContainer._metadata, IScreenOverlayContainer._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(IScreenOverlayContainer._metadata, IScreenOverlayContainer._set_display_metadata, display)

    _property_names[overlays] = "overlays"
    _property_names[padding] = "padding"
    _property_names[display] = "display"


agcls.AgClassCatalog.add_catalog_entry((5104812672920574391, 4741981752173751691), IScreenOverlayContainer)
agcls.AgTypeNameMap["IScreenOverlayContainer"] = IScreenOverlayContainer

class ITriangulatorResult(object):
    """The result from triangulation: a triangle mesh defined using an indexed triangle list. This is commonly visualized with the triangle mesh primitive or surface mesh primitive."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_positions_method_offset = 1
    _get_normals_method_offset = 2
    _get_indices_method_offset = 3
    _get_triangle_winding_order_method_offset = 4
    _get_bounding_sphere_method_offset = 5
    _metadata = {
        "iid_data" : (5076915160592738393, 3604822095127444111),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITriangulatorResult."""
        initialize_from_source_object(self, source_object, ITriangulatorResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITriangulatorResult)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITriangulatorResult, None)

    _get_positions_metadata = { "offset" : _get_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def positions(self) -> list:
        """Get the positions of the mesh. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(ITriangulatorResult._metadata, ITriangulatorResult._get_positions_metadata)

    _get_normals_metadata = { "offset" : _get_normals_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def normals(self) -> list:
        """Get the normals of the mesh. Every position in positions has corresponding normal. Normals are commonly used for lighting. Three array elements (in the order x, y, z) constitute one normal."""
        return self._intf.get_property(ITriangulatorResult._metadata, ITriangulatorResult._get_normals_metadata)

    _get_indices_metadata = { "offset" : _get_indices_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def indices(self) -> list:
        """Get indices into positions and normals. Every 3 indices represent 1 triangle. The indices returned consider the three components of a position or normal (x, y, and z) as a single array element..."""
        return self._intf.get_property(ITriangulatorResult._metadata, ITriangulatorResult._get_indices_metadata)

    _get_triangle_winding_order_metadata = { "offset" : _get_triangle_winding_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(WindingOrder),) }
    @property
    def triangle_winding_order(self) -> "WindingOrder":
        """Get the orientation of front-facing triangles in the mesh."""
        return self._intf.get_property(ITriangulatorResult._metadata, ITriangulatorResult._get_triangle_winding_order_metadata)

    _get_bounding_sphere_metadata = { "offset" : _get_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def bounding_sphere(self) -> "BoundingSphere":
        """Get the bounding sphere that encompasses the mesh."""
        return self._intf.get_property(ITriangulatorResult._metadata, ITriangulatorResult._get_bounding_sphere_metadata)

    _property_names[positions] = "positions"
    _property_names[normals] = "normals"
    _property_names[indices] = "indices"
    _property_names[triangle_winding_order] = "triangle_winding_order"
    _property_names[bounding_sphere] = "bounding_sphere"


agcls.AgClassCatalog.add_catalog_entry((5076915160592738393, 3604822095127444111), ITriangulatorResult)
agcls.AgTypeNameMap["ITriangulatorResult"] = ITriangulatorResult

class IGlobeOverlay(object):
    """The base class of all terrain overlay and globe image overlay objects."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_method_offset = 1
    _get_extent_method_offset = 2
    _get_role_method_offset = 3
    _get_uri_as_string_method_offset = 4
    _get_is_valid_method_offset = 5
    _get_display_condition_method_offset = 6
    _set_display_condition_method_offset = 7
    _metadata = {
        "iid_data" : (5111137678954940866, 12503701247266746518),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IGlobeOverlay."""
        initialize_from_source_object(self, source_object, IGlobeOverlay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IGlobeOverlay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IGlobeOverlay, None)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get the central body that the globe overlay is displayed on. It will return <see langword='null' /> if the globe overlay hasn't been added to a central body."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_central_body_metadata)

    _get_extent_metadata = { "offset" : _get_extent_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def extent(self) -> list:
        """Get the cartographic extent that represents the area covered by the globe overlay. The array elements are arranged in the order west longitude, south latitude, east longitude, north latitude."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_extent_metadata)

    _get_role_metadata = { "offset" : _get_role_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(OverlayRole),) }
    @property
    def role(self) -> "OverlayRole":
        """Get the globe overlay role of the globe overlay."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_role_metadata)

    _get_uri_as_string_metadata = { "offset" : _get_uri_as_string_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def uri_as_string(self) -> str:
        """Get the absolute URI specifying the location of the globe overlay."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_uri_as_string_metadata)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Get whether or not the overlay is valid. It can be invalid because of a missing file, corrupt file, unlicensed file, or a file on the incorrect central body."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_is_valid_metadata)

    _get_display_condition_metadata = { "offset" : _get_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def display_condition(self) -> "IDisplayCondition":
        """Get or set the display condition that controls whether or not the globe overlay is displayed."""
        return self._intf.get_property(IGlobeOverlay._metadata, IGlobeOverlay._get_display_condition_metadata)

    _set_display_condition_metadata = { "offset" : _set_display_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"),) }
    @display_condition.setter
    def display_condition(self, display_condition:"IDisplayCondition") -> None:
        return self._intf.set_property(IGlobeOverlay._metadata, IGlobeOverlay._set_display_condition_metadata, display_condition)

    _property_names[central_body] = "central_body"
    _property_names[extent] = "extent"
    _property_names[role] = "role"
    _property_names[uri_as_string] = "uri_as_string"
    _property_names[is_valid] = "is_valid"
    _property_names[display_condition] = "display_condition"


agcls.AgClassCatalog.add_catalog_entry((5111137678954940866, 12503701247266746518), IGlobeOverlay)
agcls.AgTypeNameMap["IGlobeOverlay"] = IGlobeOverlay

class IPathPrimitiveUpdatePolicy(object):
    """A class that encapsulates the update logic for a path primitive. Derived classes must implement the Update method."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _update_method_offset = 1
    _metadata = {
        "iid_data" : (4972495962199984526, 7764744302225031324),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IPathPrimitiveUpdatePolicy."""
        initialize_from_source_object(self, source_object, IPathPrimitiveUpdatePolicy)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IPathPrimitiveUpdatePolicy)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IPathPrimitiveUpdatePolicy, None)

    _update_metadata = { "offset" : _update_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("PathPrimitive"), agmarshall.InterfaceInArg("Date"),) }
    def update(self, path_primitive:"PathPrimitive", date:"Date") -> None:
        """Update the pathPrimitive at the specified date."""
        return self._intf.invoke(IPathPrimitiveUpdatePolicy._metadata, IPathPrimitiveUpdatePolicy._update_metadata, path_primitive, date)



agcls.AgClassCatalog.add_catalog_entry((4972495962199984526, 7764744302225031324), IPathPrimitiveUpdatePolicy)
agcls.AgTypeNameMap["IPathPrimitiveUpdatePolicy"] = IPathPrimitiveUpdatePolicy

class IProjection(object):
    """A projection represents a simplified camera with a position, orientation, and field of view horizontal and field of view vertical..."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_position_method_offset = 1
    _set_position_method_offset = 2
    _get_orientation_method_offset = 3
    _set_orientation_method_offset = 4
    _get_field_of_view_horizontal_method_offset = 5
    _set_field_of_view_horizontal_method_offset = 6
    _get_field_of_view_vertical_method_offset = 7
    _set_field_of_view_vertical_method_offset = 8
    _get_near_plane_method_offset = 9
    _set_near_plane_method_offset = 10
    _get_far_plane_method_offset = 11
    _set_far_plane_method_offset = 12
    _metadata = {
        "iid_data" : (5604768975439149570, 12374794011928356263),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IProjection."""
        initialize_from_source_object(self, source_object, IProjection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IProjection)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IProjection, None)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the cartesian defining the position of the projection in the central body's fixed reference frame. The array contains the components of the position arranged in the order x, y, z."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_position_metadata, position)

    _get_orientation_metadata = { "offset" : _get_orientation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def orientation(self) -> "IOrientation":
        """Get or set the unit quaternion defining the orientation of the projection in the central body's fixed reference frame."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_orientation_metadata)

    _set_orientation_metadata = { "offset" : _set_orientation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IOrientation"),) }
    @orientation.setter
    def orientation(self, orientation:"IOrientation") -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_orientation_metadata, orientation)

    _get_field_of_view_horizontal_metadata = { "offset" : _get_field_of_view_horizontal_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def field_of_view_horizontal(self) -> float:
        """Get or set the horizontal field of view associated with the projection."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_field_of_view_horizontal_metadata)

    _set_field_of_view_horizontal_metadata = { "offset" : _set_field_of_view_horizontal_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @field_of_view_horizontal.setter
    def field_of_view_horizontal(self, field_of_view_horizontal:float) -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_field_of_view_horizontal_metadata, field_of_view_horizontal)

    _get_field_of_view_vertical_metadata = { "offset" : _get_field_of_view_vertical_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def field_of_view_vertical(self) -> float:
        """Get or set the vertical field of view associated with the projection."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_field_of_view_vertical_metadata)

    _set_field_of_view_vertical_metadata = { "offset" : _set_field_of_view_vertical_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @field_of_view_vertical.setter
    def field_of_view_vertical(self, field_of_view_vertical:float) -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_field_of_view_vertical_metadata, field_of_view_vertical)

    _get_near_plane_metadata = { "offset" : _get_near_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def near_plane(self) -> float:
        """Get or set the near plane associated with the projection."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_near_plane_metadata)

    _set_near_plane_metadata = { "offset" : _set_near_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @near_plane.setter
    def near_plane(self, near_plane:float) -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_near_plane_metadata, near_plane)

    _get_far_plane_metadata = { "offset" : _get_far_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def far_plane(self) -> float:
        """Get or set the far plane associated with the projection."""
        return self._intf.get_property(IProjection._metadata, IProjection._get_far_plane_metadata)

    _set_far_plane_metadata = { "offset" : _set_far_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @far_plane.setter
    def far_plane(self, far_plane:float) -> None:
        return self._intf.set_property(IProjection._metadata, IProjection._set_far_plane_metadata, far_plane)

    _property_names[position] = "position"
    _property_names[orientation] = "orientation"
    _property_names[field_of_view_horizontal] = "field_of_view_horizontal"
    _property_names[field_of_view_vertical] = "field_of_view_vertical"
    _property_names[near_plane] = "near_plane"
    _property_names[far_plane] = "far_plane"


agcls.AgClassCatalog.add_catalog_entry((5604768975439149570, 12374794011928356263), IProjection)
agcls.AgTypeNameMap["IProjection"] = IProjection

class IScreenOverlayCollectionBase(object):
    """The common base class for collections of overlays held by screen overlay and by screen overlay manager."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_is_read_only_method_offset = 2
    _item_method_offset = 3
    _get__new_enum_method_offset = 4
    _contains_method_offset = 5
    _remove_method_offset = 6
    _clear_method_offset = 7
    _add_method_offset = 8
    _metadata = {
        "iid_data" : (5748013200866698085, 10184546868944523949),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IScreenOverlayCollectionBase."""
        initialize_from_source_object(self, source_object, IScreenOverlayCollectionBase)
        self.__dict__["_enumerator"] = None
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IScreenOverlayCollectionBase)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IScreenOverlayCollectionBase, None)
    def __iter__(self):
        """Create an iterator for the IScreenOverlayCollectionBase object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IScreenOverlay":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of screen overlays in the collection."""
        return self._intf.get_property(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._get_count_metadata)

    _get_is_read_only_metadata = { "offset" : _get_is_read_only_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_read_only(self) -> bool:
        """Get a value indicating whether the collection is read-only."""
        return self._intf.get_property(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._get_is_read_only_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "IScreenOverlay":
        """Get the overlay at the specified index."""
        return self._intf.invoke(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._get__new_enum_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IScreenOverlay"), agmarshall.VariantBoolArg,) }
    def contains(self, item:"IScreenOverlay") -> bool:
        """Determine whether the collection contains a specific overlay."""
        return self._intf.invoke(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._contains_metadata, item, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IScreenOverlay"), agmarshall.VariantBoolArg,) }
    def remove(self, item:"IScreenOverlay") -> bool:
        """Remove the first occurrence of a specific overlay from the collection."""
        return self._intf.invoke(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._remove_metadata, item, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all overlays from the collection."""
        return self._intf.invoke(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._clear_metadata, )

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IScreenOverlay"),) }
    def add(self, item:"IScreenOverlay") -> None:
        """Add an overlay to the collection."""
        return self._intf.invoke(IScreenOverlayCollectionBase._metadata, IScreenOverlayCollectionBase._add_metadata, item)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[is_read_only] = "is_read_only"
    _property_names[_new_enum] = "_new_enum"


agcls.AgClassCatalog.add_catalog_entry((5748013200866698085, 10184546868944523949), IScreenOverlayCollectionBase)
agcls.AgTypeNameMap["IScreenOverlayCollectionBase"] = IScreenOverlayCollectionBase

class IDisplayCondition(object):
    """When assigned to objects, such as primitives or globe overlays, display conditions are evaluated to determine if the object should be rendered."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5532500929025461904, 13284471583527689404),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IDisplayCondition."""
        initialize_from_source_object(self, source_object, IDisplayCondition)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IDisplayCondition)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IDisplayCondition, None)



agcls.AgClassCatalog.add_catalog_entry((5532500929025461904, 13284471583527689404), IDisplayCondition)
agcls.AgTypeNameMap["IDisplayCondition"] = IDisplayCondition

class IGlobeImageOverlay(object):
    """A globe overlay that shows an image."""

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_translucency_method_offset = 1
    _set_translucency_method_offset = 2
    _get_use_altitude_based_translucency_method_offset = 3
    _set_use_altitude_based_translucency_method_offset = 4
    _get_altitude_based_translucency_lower_translucency_method_offset = 5
    _set_altitude_based_translucency_lower_translucency_method_offset = 6
    _get_altitude_based_translucency_upper_translucency_method_offset = 7
    _set_altitude_based_translucency_upper_translucency_method_offset = 8
    _get_altitude_based_translucency_lower_altitude_method_offset = 9
    _set_altitude_based_translucency_lower_altitude_method_offset = 10
    _get_altitude_based_translucency_upper_altitude_method_offset = 11
    _set_altitude_based_translucency_upper_altitude_method_offset = 12
    _get_more_than_one_image_globe_overlay_supported_method_offset = 13
    _metadata = {
        "iid_data" : (4736968619611124189, 4590632582554593668),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IGlobeImageOverlay."""
        initialize_from_source_object(self, source_object, IGlobeImageOverlay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IGlobeImageOverlay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IGlobeImageOverlay, None)

    _get_translucency_metadata = { "offset" : _get_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def translucency(self) -> float:
        """Get or set the translucency value for the image. The translucency is between 0 and 1, where 0 is fully opaque and 1 is invisible."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_translucency_metadata)

    _set_translucency_metadata = { "offset" : _set_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @translucency.setter
    def translucency(self, translucency:float) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_translucency_metadata, translucency)

    _get_use_altitude_based_translucency_metadata = { "offset" : _get_use_altitude_based_translucency_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_altitude_based_translucency(self) -> bool:
        """Get or set whether to use altitude to determine the translucency value for the image or not. If <see langword='false' />, the translucency value is used..."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_use_altitude_based_translucency_metadata)

    _set_use_altitude_based_translucency_metadata = { "offset" : _set_use_altitude_based_translucency_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_altitude_based_translucency.setter
    def use_altitude_based_translucency(self, use_altitude_based_translucency:bool) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_use_altitude_based_translucency_metadata, use_altitude_based_translucency)

    _get_altitude_based_translucency_lower_translucency_metadata = { "offset" : _get_altitude_based_translucency_lower_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def altitude_based_translucency_lower_translucency(self) -> float:
        """Get or set the lower translucency value for the image when use altitude based translucency is set to <see langword='true' />..."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_altitude_based_translucency_lower_translucency_metadata)

    _set_altitude_based_translucency_lower_translucency_metadata = { "offset" : _set_altitude_based_translucency_lower_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @altitude_based_translucency_lower_translucency.setter
    def altitude_based_translucency_lower_translucency(self, altitude_based_translucency_lower_translucency:float) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_altitude_based_translucency_lower_translucency_metadata, altitude_based_translucency_lower_translucency)

    _get_altitude_based_translucency_upper_translucency_metadata = { "offset" : _get_altitude_based_translucency_upper_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def altitude_based_translucency_upper_translucency(self) -> float:
        """Get or set the upper translucency value for the image when use altitude based translucency is set to <see langword='true' />..."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_altitude_based_translucency_upper_translucency_metadata)

    _set_altitude_based_translucency_upper_translucency_metadata = { "offset" : _set_altitude_based_translucency_upper_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @altitude_based_translucency_upper_translucency.setter
    def altitude_based_translucency_upper_translucency(self, altitude_based_translucency_upper_translucency:float) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_altitude_based_translucency_upper_translucency_metadata, altitude_based_translucency_upper_translucency)

    _get_altitude_based_translucency_lower_altitude_metadata = { "offset" : _get_altitude_based_translucency_lower_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_based_translucency_lower_altitude(self) -> float:
        """Get or set the lower altitude bound used to calculate translucency for the image when use altitude based translucency is set to <see langword='true' />..."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_altitude_based_translucency_lower_altitude_metadata)

    _set_altitude_based_translucency_lower_altitude_metadata = { "offset" : _set_altitude_based_translucency_lower_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_based_translucency_lower_altitude.setter
    def altitude_based_translucency_lower_altitude(self, altitude_based_translucency_lower_altitude:float) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_altitude_based_translucency_lower_altitude_metadata, altitude_based_translucency_lower_altitude)

    _get_altitude_based_translucency_upper_altitude_metadata = { "offset" : _get_altitude_based_translucency_upper_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_based_translucency_upper_altitude(self) -> float:
        """Get or set the upper altitude bound used to calculate translucency for the image when use altitude based translucency is set to <see langword='true' />..."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_altitude_based_translucency_upper_altitude_metadata)

    _set_altitude_based_translucency_upper_altitude_metadata = { "offset" : _set_altitude_based_translucency_upper_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_based_translucency_upper_altitude.setter
    def altitude_based_translucency_upper_altitude(self, altitude_based_translucency_upper_altitude:float) -> None:
        return self._intf.set_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._set_altitude_based_translucency_upper_altitude_metadata, altitude_based_translucency_upper_altitude)

    _get_more_than_one_image_globe_overlay_supported_metadata = { "offset" : _get_more_than_one_image_globe_overlay_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def more_than_one_image_globe_overlay_supported(self) -> bool:
        """Get whether or not the video card allows for more than one image globe overlay globe image overlay to be added."""
        return self._intf.get_property(IGlobeImageOverlay._metadata, IGlobeImageOverlay._get_more_than_one_image_globe_overlay_supported_metadata)

    _property_names[translucency] = "translucency"
    _property_names[use_altitude_based_translucency] = "use_altitude_based_translucency"
    _property_names[altitude_based_translucency_lower_translucency] = "altitude_based_translucency_lower_translucency"
    _property_names[altitude_based_translucency_upper_translucency] = "altitude_based_translucency_upper_translucency"
    _property_names[altitude_based_translucency_lower_altitude] = "altitude_based_translucency_lower_altitude"
    _property_names[altitude_based_translucency_upper_altitude] = "altitude_based_translucency_upper_altitude"
    _property_names[more_than_one_image_globe_overlay_supported] = "more_than_one_image_globe_overlay_supported"


agcls.AgClassCatalog.add_catalog_entry((4736968619611124189, 4590632582554593668), IGlobeImageOverlay)
agcls.AgTypeNameMap["IGlobeImageOverlay"] = IGlobeImageOverlay

class IConvolutionFilter(object):
    """Apply convolution to the source raster. Convolution is the modification of a pixel's value based on the values of its surrounding pixels. The kernel is the numerical matrix that is applied to each pixel in this process..."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_divisor_method_offset = 1
    _set_divisor_method_offset = 2
    _get_offset_method_offset = 3
    _set_offset_method_offset = 4
    _get_kernel_method_offset = 5
    _set_kernel_method_offset = 6
    _metadata = {
        "iid_data" : (4661190070888522861, 10534747934616289203),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IConvolutionFilter."""
        initialize_from_source_object(self, source_object, IConvolutionFilter)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IConvolutionFilter)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IConvolutionFilter, None)

    _get_divisor_metadata = { "offset" : _get_divisor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def divisor(self) -> float:
        """Get or set the divisor for the result of the convolution kernel operation."""
        return self._intf.get_property(IConvolutionFilter._metadata, IConvolutionFilter._get_divisor_metadata)

    _set_divisor_metadata = { "offset" : _set_divisor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @divisor.setter
    def divisor(self, divisor:float) -> None:
        return self._intf.set_property(IConvolutionFilter._metadata, IConvolutionFilter._set_divisor_metadata, divisor)

    _get_offset_metadata = { "offset" : _get_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def offset(self) -> float:
        """Get or set the offset for the result of the convolution kernel operation. The value is added to the result of the operation."""
        return self._intf.get_property(IConvolutionFilter._metadata, IConvolutionFilter._get_offset_metadata)

    _set_offset_metadata = { "offset" : _set_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @offset.setter
    def offset(self, offset:float) -> None:
        return self._intf.set_property(IConvolutionFilter._metadata, IConvolutionFilter._set_offset_metadata, offset)

    _get_kernel_metadata = { "offset" : _get_kernel_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def kernel(self) -> list:
        """Get or set the convolution kernel of the filter. The array contains the 9 elements of the kernel of the convolution matrix..."""
        return self._intf.get_property(IConvolutionFilter._metadata, IConvolutionFilter._get_kernel_metadata)

    _set_kernel_metadata = { "offset" : _set_kernel_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @kernel.setter
    def kernel(self, kernel:list) -> None:
        return self._intf.set_property(IConvolutionFilter._metadata, IConvolutionFilter._set_kernel_metadata, kernel)

    _property_names[divisor] = "divisor"
    _property_names[offset] = "offset"
    _property_names[kernel] = "kernel"


agcls.AgClassCatalog.add_catalog_entry((4661190070888522861, 10534747934616289203), IConvolutionFilter)
agcls.AgTypeNameMap["IConvolutionFilter"] = IConvolutionFilter

class IRaster(object):
    """A raster dataset. A raster consists of one or more bands, or sets of values, which are most commonly associated with colors when the raster represents an image..."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_attributes_method_offset = 1
    _get_width_method_offset = 2
    _get_height_method_offset = 3
    _flip_method_offset = 4
    _rotate_method_offset = 5
    _apply_method_offset = 6
    _apply_in_place_method_offset = 7
    _extract_band_method_offset = 8
    _extract_band_from_raster_format_method_offset = 9
    _copy_from_raster_method_offset = 10
    _metadata = {
        "iid_data" : (4904839826723909985, 5664351486392510908),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IRaster."""
        initialize_from_source_object(self, source_object, IRaster)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IRaster)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IRaster, None)

    _get_attributes_metadata = { "offset" : _get_attributes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def attributes(self) -> "RasterAttributes":
        """Get the raster attributes that define the raster data."""
        return self._intf.get_property(IRaster._metadata, IRaster._get_attributes_metadata)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def width(self) -> int:
        """Get the width of the raster in pixels."""
        return self._intf.get_property(IRaster._metadata, IRaster._get_width_metadata)

    _get_height_metadata = { "offset" : _get_height_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def height(self) -> int:
        """Get the height of the raster in pixels."""
        return self._intf.get_property(IRaster._metadata, IRaster._get_height_metadata)

    _flip_metadata = { "offset" : _flip_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RasterFlipAxis),) }
    def flip(self, axis:"RasterFlipAxis") -> None:
        """Flips the raster along the given axis."""
        return self._intf.invoke(IRaster._metadata, IRaster._flip_metadata, axis)

    _rotate_metadata = { "offset" : _rotate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    def rotate(self, angle:float) -> None:
        """Rotate the raster by the given angle."""
        return self._intf.invoke(IRaster._metadata, IRaster._rotate_metadata, angle)

    _apply_metadata = { "offset" : _apply_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IRasterFilter"), agmarshall.InterfaceOutArg,) }
    def apply(self, filter:"IRasterFilter") -> "IRaster":
        """Apply a raster filter to the raster and returns a new raster with the results of the filtering. The current raster is not modified."""
        return self._intf.invoke(IRaster._metadata, IRaster._apply_metadata, filter, OutArg())

    _apply_in_place_metadata = { "offset" : _apply_in_place_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRasterFilter"),) }
    def apply_in_place(self, filter:"IRasterFilter") -> None:
        """Apply a raster filter to the raster. The current raster will contain the results of the filtering."""
        return self._intf.invoke(IRaster._metadata, IRaster._apply_in_place_metadata, filter)

    _extract_band_metadata = { "offset" : _extract_band_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(RasterBand), agmarshall.InterfaceOutArg,) }
    def extract_band(self, band:"RasterBand") -> "IRaster":
        """Extract the band of raster data associated with the given raster band."""
        return self._intf.invoke(IRaster._metadata, IRaster._extract_band_metadata, band, OutArg())

    _extract_band_from_raster_format_metadata = { "offset" : _extract_band_from_raster_format_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(RasterFormat), agmarshall.InterfaceOutArg,) }
    def extract_band_from_raster_format(self, format:"RasterFormat") -> "IRaster":
        """Extract the bands of raster data associated with the given raster format."""
        return self._intf.invoke(IRaster._metadata, IRaster._extract_band_from_raster_format_metadata, format, OutArg())

    _copy_from_raster_metadata = { "offset" : _copy_from_raster_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRaster"),) }
    def copy_from_raster(self, raster:"IRaster") -> None:
        """Copy the data associated with the given raster into this raster."""
        return self._intf.invoke(IRaster._metadata, IRaster._copy_from_raster_metadata, raster)

    _property_names[attributes] = "attributes"
    _property_names[width] = "width"
    _property_names[height] = "height"


agcls.AgClassCatalog.add_catalog_entry((4904839826723909985, 5664351486392510908), IRaster)
agcls.AgTypeNameMap["IRaster"] = IRaster

class IRasterFilter(object):
    """A filter for processing raster datasets. RasterFilter is the base class for all raster filters..."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5173582657377882717, 9127510129600190340),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IRasterFilter."""
        initialize_from_source_object(self, source_object, IRasterFilter)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IRasterFilter)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IRasterFilter, None)



agcls.AgClassCatalog.add_catalog_entry((5173582657377882717, 9127510129600190340), IRasterFilter)
agcls.AgTypeNameMap["IRasterFilter"] = IRasterFilter

class IRasterStream(object):
    """A raster, the data of which, is updated dynamically at the specified update delta. The class can be used to stream video and other dynamic raster data to textures and other raster clients..."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_update_delta_method_offset = 1
    _set_update_delta_method_offset = 2
    _update_method_offset = 3
    _metadata = {
        "iid_data" : (5206566972312003786, 8988437618500445837),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IRasterStream."""
        initialize_from_source_object(self, source_object, IRasterStream)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IRasterStream)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IRasterStream, None)

    _get_update_delta_metadata = { "offset" : _get_update_delta_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def update_delta(self) -> float:
        """Get or set the update delta of the raster stream in seconds. The update delta defines the interval at which the Update method will be called. The default update delta is 0, which will call the Update method every time the scene manager time changes..."""
        return self._intf.get_property(IRasterStream._metadata, IRasterStream._get_update_delta_metadata)

    _set_update_delta_metadata = { "offset" : _set_update_delta_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @update_delta.setter
    def update_delta(self, update_delta:float) -> None:
        return self._intf.set_property(IRasterStream._metadata, IRasterStream._set_update_delta_metadata, update_delta)

    _update_metadata = { "offset" : _update_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.InterfaceInArg("Date"), agmarshall.VariantBoolArg,) }
    def update(self, time:"Date", next_time:"Date") -> bool:
        """When overridden in a derived class, updates the raster data associated with the raster stream at the specified time..."""
        return self._intf.invoke(IRasterStream._metadata, IRasterStream._update_metadata, time, next_time, OutArg())

    _property_names[update_delta] = "update_delta"


agcls.AgClassCatalog.add_catalog_entry((5206566972312003786, 8988437618500445837), IRasterStream)
agcls.AgTypeNameMap["IRasterStream"] = IRasterStream

class IKmlContainer(object):
    """A KmlContainer contains a collection of children kml features."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_children_method_offset = 1
    _metadata = {
        "iid_data" : (4787473808012639137, 8060652129531487635),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IKmlContainer."""
        initialize_from_source_object(self, source_object, IKmlContainer)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IKmlContainer)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IKmlContainer, None)

    _get_children_metadata = { "offset" : _get_children_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def children(self) -> "KmlFeatureCollection":
        """The collection of kml features that are children of this container."""
        return self._intf.get_property(IKmlContainer._metadata, IKmlContainer._get_children_metadata)

    _property_names[children] = "children"


agcls.AgClassCatalog.add_catalog_entry((4787473808012639137, 8060652129531487635), IKmlContainer)
agcls.AgTypeNameMap["IKmlContainer"] = IKmlContainer

class IKmlFeature(object):
    """A KML feature."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_loaded_method_offset = 1
    _get_display_method_offset = 2
    _set_display_method_offset = 3
    _get_content_method_offset = 4
    _get_name_method_offset = 5
    _get_description_method_offset = 6
    _get_snippet_method_offset = 7
    _get_bounding_sphere_method_offset = 8
    _fly_to_method_offset = 9
    _metadata = {
        "iid_data" : (5360086555408352970, 584045744064033934),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IKmlFeature."""
        initialize_from_source_object(self, source_object, IKmlFeature)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IKmlFeature)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IKmlFeature, None)

    _get_is_loaded_metadata = { "offset" : _get_is_loaded_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_loaded(self) -> bool:
        """Get whether the document associated with this feature is loaded."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_is_loaded_metadata)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set whether this feature will be displayed in the Scene."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(IKmlFeature._metadata, IKmlFeature._set_display_metadata, display)

    _get_content_metadata = { "offset" : _get_content_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def content(self) -> str:
        """Get the content associated with this feature."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_content_metadata)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of this feature."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_name_metadata)

    _get_description_metadata = { "offset" : _get_description_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def description(self) -> str:
        """Get the description associated with this feature."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_description_metadata)

    _get_snippet_metadata = { "offset" : _get_snippet_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def snippet(self) -> str:
        """Get the snippet associated with this feature."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_snippet_metadata)

    _get_bounding_sphere_metadata = { "offset" : _get_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def bounding_sphere(self) -> "BoundingSphere":
        """The bounding sphere encompassing the area associated with this feature."""
        return self._intf.get_property(IKmlFeature._metadata, IKmlFeature._get_bounding_sphere_metadata)

    _fly_to_metadata = { "offset" : _fly_to_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def fly_to(self) -> None:
        """Move the camera to the area encompassing this feature."""
        return self._intf.invoke(IKmlFeature._metadata, IKmlFeature._fly_to_metadata, )

    _property_names[is_loaded] = "is_loaded"
    _property_names[display] = "display"
    _property_names[content] = "content"
    _property_names[name] = "name"
    _property_names[description] = "description"
    _property_names[snippet] = "snippet"
    _property_names[bounding_sphere] = "bounding_sphere"


agcls.AgClassCatalog.add_catalog_entry((5360086555408352970, 584045744064033934), IKmlFeature)
agcls.AgTypeNameMap["IKmlFeature"] = IKmlFeature

class IOverlay(object):
    """A visible element drawn in screen space. Overlays are useful for floating logos, heads up displays, and integrating user interfaces into the 3D window."""

    _num_methods = 76
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_position_method_offset = 1
    _set_position_method_offset = 2
    _get_pinning_position_method_offset = 3
    _set_pinning_position_method_offset = 4
    _get_x_method_offset = 5
    _set_x_method_offset = 6
    _get_x_unit_method_offset = 7
    _set_x_unit_method_offset = 8
    _get_y_method_offset = 9
    _set_y_method_offset = 10
    _get_y_unit_method_offset = 11
    _set_y_unit_method_offset = 12
    _get_size_method_offset = 13
    _set_size_method_offset = 14
    _get_width_method_offset = 15
    _set_width_method_offset = 16
    _get_width_unit_method_offset = 17
    _set_width_unit_method_offset = 18
    _get_height_method_offset = 19
    _set_height_method_offset = 20
    _get_height_unit_method_offset = 21
    _set_height_unit_method_offset = 22
    _get_minimum_size_method_offset = 23
    _set_minimum_size_method_offset = 24
    _get_maximum_size_method_offset = 25
    _set_maximum_size_method_offset = 26
    _get_bounds_method_offset = 27
    _get_border_color_method_offset = 28
    _set_border_color_method_offset = 29
    _get_border_size_method_offset = 30
    _set_border_size_method_offset = 31
    _get_border_translucency_method_offset = 32
    _set_border_translucency_method_offset = 33
    _get_translation_x_method_offset = 34
    _set_translation_x_method_offset = 35
    _get_translation_y_method_offset = 36
    _set_translation_y_method_offset = 37
    _get_rotation_angle_method_offset = 38
    _set_rotation_angle_method_offset = 39
    _get_rotation_point_method_offset = 40
    _set_rotation_point_method_offset = 41
    _get_scale_method_offset = 42
    _set_scale_method_offset = 43
    _get_flip_x_method_offset = 44
    _set_flip_x_method_offset = 45
    _get_flip_y_method_offset = 46
    _set_flip_y_method_offset = 47
    _get_origin_method_offset = 48
    _set_origin_method_offset = 49
    _get_pinning_origin_method_offset = 50
    _set_pinning_origin_method_offset = 51
    _get_parent_method_offset = 52
    _get_translucency_method_offset = 53
    _set_translucency_method_offset = 54
    _get_color_method_offset = 55
    _set_color_method_offset = 56
    _get_picking_enabled_method_offset = 57
    _set_picking_enabled_method_offset = 58
    _get_clip_to_parent_method_offset = 59
    _set_clip_to_parent_method_offset = 60
    _get_display_method_offset = 61
    _set_display_method_offset = 62
    _get_control_position_method_offset = 63
    _get_control_size_method_offset = 64
    _get_control_bounds_method_offset = 65
    _get_display_condition_method_offset = 66
    _set_display_condition_method_offset = 67
    _get_overlays_method_offset = 68
    _get_padding_method_offset = 69
    _set_padding_method_offset = 70
    _bring_to_front_method_offset = 71
    _send_to_back_method_offset = 72
    _overlay_to_control_method_offset = 73
    _control_to_overlay_method_offset = 74
    _get_tag_method_offset = 75
    _set_tag_method_offset = 76
    _metadata = {
        "iid_data" : (4975169524664922677, 13740715011966152579),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IOverlay."""
        initialize_from_source_object(self, source_object, IOverlay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IOverlay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IOverlay, None)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the position of the overlay relative to its parent. The array represents the position of the overlay and has a size of 4. The elements are in the order x position, y position, x screen overlay unit, y screen overlay unit."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_position_metadata, position)

    _get_pinning_position_metadata = { "offset" : _get_pinning_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def pinning_position(self) -> list:
        """Get or set the pinning position of the overlay, relative to the overlay, which determines the point on the overlay that corresponds to the position property. The array represents the pinning position of the overlay and has a size of 4..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_pinning_position_metadata)

    _set_pinning_position_metadata = { "offset" : _set_pinning_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @pinning_position.setter
    def pinning_position(self, pinning_position:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_pinning_position_metadata, pinning_position)

    _get_x_metadata = { "offset" : _get_x_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def x(self) -> float:
        """Get or set the X position of the overlay relative to its parent. The unit in which the position is defined is specified by the x unit property. The position is measured horizontally from the origin, which is, by default, the lower-left corner."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_x_metadata)

    _set_x_metadata = { "offset" : _set_x_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @x.setter
    def x(self, x:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_x_metadata, x)

    _get_x_unit_metadata = { "offset" : _get_x_unit_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @property
    def x_unit(self) -> "ScreenOverlayUnit":
        """Get or set the unit of the x property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_x_unit_metadata)

    _set_x_unit_metadata = { "offset" : _set_x_unit_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @x_unit.setter
    def x_unit(self, x_unit:"ScreenOverlayUnit") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_x_unit_metadata, x_unit)

    _get_y_metadata = { "offset" : _get_y_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def y(self) -> float:
        """Get or set the Y position of the overlay relative to its parent. The unit in which the position is defined is specified by the y unit property. The position is measured vertically from the origin, which is, by default, the lower-left corner."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_y_metadata)

    _set_y_metadata = { "offset" : _set_y_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @y.setter
    def y(self, y:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_y_metadata, y)

    _get_y_unit_metadata = { "offset" : _get_y_unit_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @property
    def y_unit(self) -> "ScreenOverlayUnit":
        """Get or set the unit of the y property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_y_unit_metadata)

    _set_y_unit_metadata = { "offset" : _set_y_unit_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @y_unit.setter
    def y_unit(self, y_unit:"ScreenOverlayUnit") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_y_unit_metadata, y_unit)

    _get_size_metadata = { "offset" : _get_size_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def size(self) -> list:
        """Get or set the size of the overlay. The array elements represent the size of the overlay in the order width, height, width screen overlay unit, height screen overlay unit."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_size_metadata)

    _set_size_metadata = { "offset" : _set_size_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @size.setter
    def size(self, size:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_size_metadata, size)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def width(self) -> float:
        """Get or set the width of the overlay. The unit in which the width is defined is specified by the width unit property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_width_metadata)

    _set_width_metadata = { "offset" : _set_width_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @width.setter
    def width(self, width:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_width_metadata, width)

    _get_width_unit_metadata = { "offset" : _get_width_unit_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @property
    def width_unit(self) -> "ScreenOverlayUnit":
        """Get or set the unit of the width property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_width_unit_metadata)

    _set_width_unit_metadata = { "offset" : _set_width_unit_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @width_unit.setter
    def width_unit(self, width_unit:"ScreenOverlayUnit") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_width_unit_metadata, width_unit)

    _get_height_metadata = { "offset" : _get_height_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def height(self) -> float:
        """Get or set the height of the overlay. The unit in which the height is defined is specified by the height unit property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_height_metadata)

    _set_height_metadata = { "offset" : _set_height_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @height.setter
    def height(self, height:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_height_metadata, height)

    _get_height_unit_metadata = { "offset" : _get_height_unit_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @property
    def height_unit(self) -> "ScreenOverlayUnit":
        """Get or set the unit of the height property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_height_unit_metadata)

    _set_height_unit_metadata = { "offset" : _set_height_unit_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayUnit),) }
    @height_unit.setter
    def height_unit(self, height_unit:"ScreenOverlayUnit") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_height_unit_metadata, height_unit)

    _get_minimum_size_metadata = { "offset" : _get_minimum_size_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def minimum_size(self) -> list:
        """Get or set the minimum size of the overlay. The overlay will never be smaller than this size, even if the overlay's size is specified as a percentage of its parent and its parent is very small..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_minimum_size_metadata)

    _set_minimum_size_metadata = { "offset" : _set_minimum_size_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @minimum_size.setter
    def minimum_size(self, minimum_size:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_minimum_size_metadata, minimum_size)

    _get_maximum_size_metadata = { "offset" : _get_maximum_size_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def maximum_size(self) -> list:
        """Get or set the maximum size of the overlay. The overlay will never be larger than this size, even if the overlay's size is specified as a percentage of its parent and its parent is very large..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_maximum_size_metadata)

    _set_maximum_size_metadata = { "offset" : _set_maximum_size_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @maximum_size.setter
    def maximum_size(self, maximum_size:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_maximum_size_metadata, maximum_size)

    _get_bounds_metadata = { "offset" : _get_bounds_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def bounds(self) -> list:
        """Get the bounds of the overlay relative to its parent. The array contains the properties defining the bounds in the order left x location, top y location, width, height."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_bounds_metadata)

    _get_border_color_metadata = { "offset" : _get_border_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def border_color(self) -> agcolor.Color:
        """Get or set the overlay's border color. By default, the border color is white. However, also by default, the overlay has a border size of 0.0 so the border is not displayed."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_border_color_metadata)

    _set_border_color_metadata = { "offset" : _set_border_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @border_color.setter
    def border_color(self, border_color:agcolor.Color) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_border_color_metadata, border_color)

    _get_border_size_metadata = { "offset" : _get_border_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def border_size(self) -> int:
        """Get or set the size of the overlay's border. By default, this is 0.0 so the border is not displayed."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_border_size_metadata)

    _set_border_size_metadata = { "offset" : _set_border_size_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @border_size.setter
    def border_size(self, border_size:int) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_border_size_metadata, border_size)

    _get_border_translucency_metadata = { "offset" : _get_border_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def border_translucency(self) -> float:
        """Get or set the translucency of the overlay border. Translucency is a value between 0.0 and 1.0, where 0.0 is completely opaque and 1.0 is completely transparent."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_border_translucency_metadata)

    _set_border_translucency_metadata = { "offset" : _set_border_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @border_translucency.setter
    def border_translucency(self, border_translucency:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_border_translucency_metadata, border_translucency)

    _get_translation_x_metadata = { "offset" : _get_translation_x_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def translation_x(self) -> float:
        """Get or set the value with which the overlay will be translated from the X value of the position property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_translation_x_metadata)

    _set_translation_x_metadata = { "offset" : _set_translation_x_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @translation_x.setter
    def translation_x(self, translation_x:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_translation_x_metadata, translation_x)

    _get_translation_y_metadata = { "offset" : _get_translation_y_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def translation_y(self) -> float:
        """Get or set the value with which the overlay will be translated from the Y value of the position property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_translation_y_metadata)

    _set_translation_y_metadata = { "offset" : _set_translation_y_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @translation_y.setter
    def translation_y(self, translation_y:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_translation_y_metadata, translation_y)

    _get_rotation_angle_metadata = { "offset" : _get_rotation_angle_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def rotation_angle(self) -> float:
        """Get or set the counter-clockwise rotation of the overlay. The overlay is rotated around the point specified by the rotation point property."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_rotation_angle_metadata)

    _set_rotation_angle_metadata = { "offset" : _set_rotation_angle_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @rotation_angle.setter
    def rotation_angle(self, rotation_angle:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_rotation_angle_metadata, rotation_angle)

    _get_rotation_point_metadata = { "offset" : _get_rotation_point_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def rotation_point(self) -> list:
        """Get or set the point that the overlay is rotated around when the rotation angle property has a value other than 0.0. The array contains seven elements defining the properties of the rotation point..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_rotation_point_metadata)

    _set_rotation_point_metadata = { "offset" : _set_rotation_point_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @rotation_point.setter
    def rotation_point(self, rotation_point:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_rotation_point_metadata, rotation_point)

    _get_scale_metadata = { "offset" : _get_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def scale(self) -> float:
        """Get or set the fractional value used to scale the overlay's size property. A value greater than 1.0 will make the overlay larger while a value less than 1.0 will make it smaller."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_scale_metadata)

    _set_scale_metadata = { "offset" : _set_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @scale.setter
    def scale(self, scale:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_scale_metadata, scale)

    _get_flip_x_metadata = { "offset" : _get_flip_x_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def flip_x(self) -> bool:
        """Get or set whether the overlay will be flipped along its X axis."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_flip_x_metadata)

    _set_flip_x_metadata = { "offset" : _set_flip_x_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @flip_x.setter
    def flip_x(self, flip_x:bool) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_flip_x_metadata, flip_x)

    _get_flip_y_metadata = { "offset" : _get_flip_y_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def flip_y(self) -> bool:
        """Get or set whether the overlay will be flipped along its Y axis."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_flip_y_metadata)

    _set_flip_y_metadata = { "offset" : _set_flip_y_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @flip_y.setter
    def flip_y(self, flip_y:bool) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_flip_y_metadata, flip_y)

    _get_origin_metadata = { "offset" : _get_origin_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayOrigin),) }
    @property
    def origin(self) -> "ScreenOverlayOrigin":
        """Get or set the origin from which the overlay's position is defined. By default, the value of this property is bottom left..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_origin_metadata)

    _set_origin_metadata = { "offset" : _set_origin_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayOrigin),) }
    @origin.setter
    def origin(self, origin:"ScreenOverlayOrigin") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_origin_metadata, origin)

    _get_pinning_origin_metadata = { "offset" : _get_pinning_origin_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayPinningOrigin),) }
    @property
    def pinning_origin(self) -> "ScreenOverlayPinningOrigin":
        """Get or set the origin of the pinning position property, relative to the overlay..."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_pinning_origin_metadata)

    _set_pinning_origin_metadata = { "offset" : _set_pinning_origin_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ScreenOverlayPinningOrigin),) }
    @pinning_origin.setter
    def pinning_origin(self, pinning_origin:"ScreenOverlayPinningOrigin") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_pinning_origin_metadata, pinning_origin)

    _get_parent_metadata = { "offset" : _get_parent_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def parent(self) -> "IScreenOverlayContainer":
        """Get the overlay's parent. This may be another overlay if this overlay was added to that overlay's overlays collection. Or, it may be the screen overlay manager if this overlay was added to the scene manager'sscreen overlays collection."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_parent_metadata)

    _get_translucency_metadata = { "offset" : _get_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def translucency(self) -> float:
        """Get or set the overlay's translucency. Translucency is a value between 0.0 and 1.0, where 0.0 is completely opaque and 1.0 is completely transparent."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_translucency_metadata)

    _set_translucency_metadata = { "offset" : _set_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @translucency.setter
    def translucency(self, translucency:float) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_translucency_metadata, translucency)

    _get_color_metadata = { "offset" : _get_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def color(self) -> agcolor.Color:
        """Get or set the overlay's color. By default, the overlay is white."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_color_metadata)

    _set_color_metadata = { "offset" : _set_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @color.setter
    def color(self, color:agcolor.Color) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_color_metadata, color)

    _get_picking_enabled_metadata = { "offset" : _get_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def picking_enabled(self) -> bool:
        """Get or set a value indicating whether or not picking on the overlay is enabled. If picking is disabled, this overlay will never show up in the result of PickScreenOverlay, even if it occupies the specified pick position."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_picking_enabled_metadata)

    _set_picking_enabled_metadata = { "offset" : _set_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @picking_enabled.setter
    def picking_enabled(self, picking_enabled:bool) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_picking_enabled_metadata, picking_enabled)

    _get_clip_to_parent_metadata = { "offset" : _get_clip_to_parent_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def clip_to_parent(self) -> bool:
        """Get or set a value indicating whether or not the overlay will be clipped by the bounds of its parent. If this property is <see langword='false' />, part of this overlay may be visible outside of its parent's bounds."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_clip_to_parent_metadata)

    _set_clip_to_parent_metadata = { "offset" : _set_clip_to_parent_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @clip_to_parent.setter
    def clip_to_parent(self, clip_to_parent:bool) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_clip_to_parent_metadata, clip_to_parent)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set if this overlay and the collection of overlays that are contained within this overlay should be rendered."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_display_metadata, display)

    _get_control_position_metadata = { "offset" : _get_control_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def control_position(self) -> list:
        """Get the position of the overlay in coordinates relative to the overall globe control. The array represents the position of the overlay and has a size of 4. The elements are in the order x position, y position, x screen overlay unit, y screen overlay unit."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_control_position_metadata)

    _get_control_size_metadata = { "offset" : _get_control_size_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def control_size(self) -> list:
        """Get the size of the overlay in coordinates relative to the overall globe control. The elements are in the order width, height, width screen overlay unit, height screen overlay unit."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_control_size_metadata)

    _get_control_bounds_metadata = { "offset" : _get_control_bounds_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def control_bounds(self) -> list:
        """Get the bounds of the overlay in coordinates relative to the overall globe control. The array contains the properties defining the bounds in the order left x location, top y location, width, height."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_control_bounds_metadata)

    _get_display_condition_metadata = { "offset" : _get_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def display_condition(self) -> "IDisplayCondition":
        """Get or set the display condition that determines if the overlay should be rendered. Both this and display must evaluate to true for the overlay to be rendered."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_display_condition_metadata)

    _set_display_condition_metadata = { "offset" : _set_display_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"),) }
    @display_condition.setter
    def display_condition(self, display_condition:"IDisplayCondition") -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_display_condition_metadata, display_condition)

    _get_overlays_metadata = { "offset" : _get_overlays_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlays(self) -> "ScreenOverlayCollection":
        """Get the collection of overlays that are contained within this overlay."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_overlays_metadata)

    _get_padding_metadata = { "offset" : _get_padding_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def padding(self) -> list:
        """Get or set the padding surrounding the overlays that are contained within this overlay. The array contains the components of the padding arranged in the order left, top, right, bottom."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_padding_metadata)

    _set_padding_metadata = { "offset" : _set_padding_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @padding.setter
    def padding(self, padding:list) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_padding_metadata, padding)

    _bring_to_front_metadata = { "offset" : _bring_to_front_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def bring_to_front(self) -> None:
        """Brings the overlay to the front of the z-order, so it is on top of all other overlays with the same parent."""
        return self._intf.invoke(IOverlay._metadata, IOverlay._bring_to_front_metadata, )

    _send_to_back_metadata = { "offset" : _send_to_back_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def send_to_back(self) -> None:
        """Send the overlay to the back of the z-order, so it is underneath all other overlays with the same parent."""
        return self._intf.invoke(IOverlay._metadata, IOverlay._send_to_back_metadata, )

    _overlay_to_control_metadata = { "offset" : _overlay_to_control_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.LPSafearrayArg,) }
    def overlay_to_control(self, x:float, y:float) -> list:
        """Transform a given position, specified relative to the overlay, into coordinates relative to the overall globe control..."""
        return self._intf.invoke(IOverlay._metadata, IOverlay._overlay_to_control_metadata, x, y, OutArg())

    _control_to_overlay_metadata = { "offset" : _control_to_overlay_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.LPSafearrayArg,) }
    def control_to_overlay(self, x:float, y:float) -> list:
        """Transform a given position, specified relative to the overall globe control, into coordinates relative to this overlay..."""
        return self._intf.invoke(IOverlay._metadata, IOverlay._control_to_overlay_metadata, x, y, OutArg())

    _get_tag_metadata = { "offset" : _get_tag_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def tag(self) -> typing.Any:
        """Get or set custom value associated with this primitive."""
        return self._intf.get_property(IOverlay._metadata, IOverlay._get_tag_metadata)

    _set_tag_metadata = { "offset" : _set_tag_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @tag.setter
    def tag(self, tag:typing.Any) -> None:
        return self._intf.set_property(IOverlay._metadata, IOverlay._set_tag_metadata, tag)

    _property_names[position] = "position"
    _property_names[pinning_position] = "pinning_position"
    _property_names[x] = "x"
    _property_names[x_unit] = "x_unit"
    _property_names[y] = "y"
    _property_names[y_unit] = "y_unit"
    _property_names[size] = "size"
    _property_names[width] = "width"
    _property_names[width_unit] = "width_unit"
    _property_names[height] = "height"
    _property_names[height_unit] = "height_unit"
    _property_names[minimum_size] = "minimum_size"
    _property_names[maximum_size] = "maximum_size"
    _property_names[bounds] = "bounds"
    _property_names[border_color] = "border_color"
    _property_names[border_size] = "border_size"
    _property_names[border_translucency] = "border_translucency"
    _property_names[translation_x] = "translation_x"
    _property_names[translation_y] = "translation_y"
    _property_names[rotation_angle] = "rotation_angle"
    _property_names[rotation_point] = "rotation_point"
    _property_names[scale] = "scale"
    _property_names[flip_x] = "flip_x"
    _property_names[flip_y] = "flip_y"
    _property_names[origin] = "origin"
    _property_names[pinning_origin] = "pinning_origin"
    _property_names[parent] = "parent"
    _property_names[translucency] = "translucency"
    _property_names[color] = "color"
    _property_names[picking_enabled] = "picking_enabled"
    _property_names[clip_to_parent] = "clip_to_parent"
    _property_names[display] = "display"
    _property_names[control_position] = "control_position"
    _property_names[control_size] = "control_size"
    _property_names[control_bounds] = "control_bounds"
    _property_names[display_condition] = "display_condition"
    _property_names[overlays] = "overlays"
    _property_names[padding] = "padding"
    _property_names[tag] = "tag"


agcls.AgClassCatalog.add_catalog_entry((4975169524664922677, 13740715011966152579), IOverlay)
agcls.AgTypeNameMap["IOverlay"] = IOverlay

class IPositionInterpolator(object):
    """Position interpolators compute positions based on a collection of input positions. Position interpolators are used in conjunction with the polyline primitive to render things such as great arcs and rhumb lines."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_polyline_type_method_offset = 1
    _interpolate_method_offset = 2
    _metadata = {
        "iid_data" : (4813879190160674146, 10385975851796783768),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IPositionInterpolator."""
        initialize_from_source_object(self, source_object, IPositionInterpolator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IPositionInterpolator)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IPositionInterpolator, None)

    _get_polyline_type_metadata = { "offset" : _get_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def polyline_type(self) -> "PolylineType":
        """Get the polyline type of positions returned from interpolate."""
        return self._intf.get_property(IPositionInterpolator._metadata, IPositionInterpolator._get_polyline_type_metadata)

    _interpolate_metadata = { "offset" : _interpolate_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def interpolate(self, positions:list) -> list:
        """Compute interpolated positions based on the input positions. Returns an array of positions in the order x, y, z."""
        return self._intf.invoke(IPositionInterpolator._metadata, IPositionInterpolator._interpolate_metadata, positions, OutArg())

    _property_names[polyline_type] = "polyline_type"


agcls.AgClassCatalog.add_catalog_entry((4813879190160674146, 10385975851796783768), IPositionInterpolator)
agcls.AgTypeNameMap["IPositionInterpolator"] = IPositionInterpolator

class IPrimitive(object):
    """Primitives represent objects rendered in the 3D scene."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_frame_method_offset = 1
    _set_reference_frame_method_offset = 2
    _get_bounding_sphere_method_offset = 3
    _set_bounding_sphere_method_offset = 4
    _get_automatically_compute_bounding_sphere_method_offset = 5
    _set_automatically_compute_bounding_sphere_method_offset = 6
    _get_display_condition_method_offset = 7
    _set_display_condition_method_offset = 8
    _get_display_method_offset = 9
    _set_display_method_offset = 10
    _get_color_method_offset = 11
    _set_color_method_offset = 12
    _get_translucency_method_offset = 13
    _set_translucency_method_offset = 14
    _get_tag_method_offset = 15
    _set_tag_method_offset = 16
    _metadata = {
        "iid_data" : (5491912536264793211, 14009178631662877855),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IPrimitive."""
        initialize_from_source_object(self, source_object, IPrimitive)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IPrimitive)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IPrimitive, None)

    _get_reference_frame_metadata = { "offset" : _get_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_frame(self) -> "IVectorGeometryToolSystem":
        """Get or set the reference frame this primitive is defined and rendered in."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_reference_frame_metadata)

    _set_reference_frame_metadata = { "offset" : _set_reference_frame_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"),) }
    @reference_frame.setter
    def reference_frame(self, reference_frame:"IVectorGeometryToolSystem") -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_reference_frame_metadata, reference_frame)

    _get_bounding_sphere_metadata = { "offset" : _get_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def bounding_sphere(self) -> "BoundingSphere":
        """Get or set the bounding sphere that encompasses the primitive. The center is defined in the primitive's reference frame."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_bounding_sphere_metadata)

    _set_bounding_sphere_metadata = { "offset" : _set_bounding_sphere_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("BoundingSphere"),) }
    @bounding_sphere.setter
    def bounding_sphere(self, bounding_sphere:"BoundingSphere") -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_bounding_sphere_metadata, bounding_sphere)

    _get_automatically_compute_bounding_sphere_metadata = { "offset" : _get_automatically_compute_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def automatically_compute_bounding_sphere(self) -> bool:
        """Get or set if the primitive's bounding sphere is automatically computed."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_automatically_compute_bounding_sphere_metadata)

    _set_automatically_compute_bounding_sphere_metadata = { "offset" : _set_automatically_compute_bounding_sphere_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @automatically_compute_bounding_sphere.setter
    def automatically_compute_bounding_sphere(self, automatically_compute_bounding_sphere:bool) -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_automatically_compute_bounding_sphere_metadata, automatically_compute_bounding_sphere)

    _get_display_condition_metadata = { "offset" : _get_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def display_condition(self) -> "IDisplayCondition":
        """Get or set the display condition that determines if the primitive should be rendered. Both this and display must evaluate to true for the primitive to be rendered."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_display_condition_metadata)

    _set_display_condition_metadata = { "offset" : _set_display_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"),) }
    @display_condition.setter
    def display_condition(self, display_condition:"IDisplayCondition") -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_display_condition_metadata, display_condition)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set if the primitive should be rendered. Both this and display condition must evaluate to true for the primitive to be rendered."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_display_metadata, display)

    _get_color_metadata = { "offset" : _get_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def color(self) -> agcolor.Color:
        """Get or set the primitive's color."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_color_metadata)

    _set_color_metadata = { "offset" : _set_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @color.setter
    def color(self, color:agcolor.Color) -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_color_metadata, color)

    _get_translucency_metadata = { "offset" : _get_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def translucency(self) -> float:
        """Get or set the primitive's translucency. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_translucency_metadata)

    _set_translucency_metadata = { "offset" : _set_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @translucency.setter
    def translucency(self, translucency:float) -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_translucency_metadata, translucency)

    _get_tag_metadata = { "offset" : _get_tag_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def tag(self) -> typing.Any:
        """Get or set custom value associated with this primitive."""
        return self._intf.get_property(IPrimitive._metadata, IPrimitive._get_tag_metadata)

    _set_tag_metadata = { "offset" : _set_tag_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @tag.setter
    def tag(self, tag:typing.Any) -> None:
        return self._intf.set_property(IPrimitive._metadata, IPrimitive._set_tag_metadata, tag)

    _property_names[reference_frame] = "reference_frame"
    _property_names[bounding_sphere] = "bounding_sphere"
    _property_names[automatically_compute_bounding_sphere] = "automatically_compute_bounding_sphere"
    _property_names[display_condition] = "display_condition"
    _property_names[display] = "display"
    _property_names[color] = "color"
    _property_names[translucency] = "translucency"
    _property_names[tag] = "tag"


agcls.AgClassCatalog.add_catalog_entry((5491912536264793211, 14009178631662877855), IPrimitive)
agcls.AgTypeNameMap["IPrimitive"] = IPrimitive

class IScreenOverlay(object):
    """A visible element drawn in screen space. Overlays are useful for floating logos, heads up displays, and integrating user interfaces into the 3D window."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5414131607122331821, 17536076725021774741),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IScreenOverlay."""
        initialize_from_source_object(self, source_object, IScreenOverlay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IScreenOverlay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IScreenOverlay, None)



agcls.AgClassCatalog.add_catalog_entry((5414131607122331821, 17536076725021774741), IScreenOverlay)
agcls.AgTypeNameMap["IScreenOverlay"] = IScreenOverlay

class ITerrainOverlay(object):
    """A globe overlay which shows terrain."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_offset_method_offset = 1
    _set_altitude_offset_method_offset = 2
    _get_altitude_scale_method_offset = 3
    _set_altitude_scale_method_offset = 4
    _get_supported_method_offset = 5
    _metadata = {
        "iid_data" : (4855015852445367860, 8081296487407790247),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITerrainOverlay."""
        initialize_from_source_object(self, source_object, ITerrainOverlay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITerrainOverlay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITerrainOverlay, None)

    _get_altitude_offset_metadata = { "offset" : _get_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_offset(self) -> float:
        """Get or set the value from which to offset the terrain height."""
        return self._intf.get_property(ITerrainOverlay._metadata, ITerrainOverlay._get_altitude_offset_metadata)

    _set_altitude_offset_metadata = { "offset" : _set_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_offset.setter
    def altitude_offset(self, altitude_offset:float) -> None:
        return self._intf.set_property(ITerrainOverlay._metadata, ITerrainOverlay._set_altitude_offset_metadata, altitude_offset)

    _get_altitude_scale_metadata = { "offset" : _get_altitude_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_scale(self) -> float:
        """Get or set the value from which to scale the terrain height."""
        return self._intf.get_property(ITerrainOverlay._metadata, ITerrainOverlay._get_altitude_scale_metadata)

    _set_altitude_scale_metadata = { "offset" : _set_altitude_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_scale.setter
    def altitude_scale(self, altitude_scale:float) -> None:
        return self._intf.set_property(ITerrainOverlay._metadata, ITerrainOverlay._set_altitude_scale_metadata, altitude_scale)

    _get_supported_metadata = { "offset" : _get_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def supported(self) -> bool:
        """Get whether the video card supports adding terrain overlay objects. Video cards that support OpenGL 1.2 or higher support terrain overlay objects."""
        return self._intf.get_property(ITerrainOverlay._metadata, ITerrainOverlay._get_supported_metadata)

    _property_names[altitude_offset] = "altitude_offset"
    _property_names[altitude_scale] = "altitude_scale"
    _property_names[supported] = "supported"


agcls.AgClassCatalog.add_catalog_entry((4855015852445367860, 8081296487407790247), ITerrainOverlay)
agcls.AgTypeNameMap["ITerrainOverlay"] = ITerrainOverlay



class PathPoint(SupportsDeleteCallback):
    """A path point used with the Path Primitive."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_position_method_offset = 1
    _set_position_method_offset = 2
    _get_date_method_offset = 3
    _get_color_method_offset = 4
    _set_color_method_offset = 5
    _get_translucency_method_offset = 6
    _set_translucency_method_offset = 7
    _get_outline_color_method_offset = 8
    _set_outline_color_method_offset = 9
    _get_outline_translucency_method_offset = 10
    _set_outline_translucency_method_offset = 11
    _get_is_translucent_method_offset = 12
    _metadata = {
        "iid_data" : (4847933377036680354, 16398371258109956499),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PathPoint)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """A path point position as a one-dimensional array of three elements corresponding to (X,Y,Z) cartesian coordinates."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(PathPoint._metadata, PathPoint._set_position_metadata, position)

    _get_date_metadata = { "offset" : _get_date_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def date(self) -> "Date":
        """A date/time of the path point."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_date_metadata)

    _get_color_metadata = { "offset" : _get_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def color(self) -> agcolor.Color:
        """The path point color."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_color_metadata)

    _set_color_metadata = { "offset" : _set_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @color.setter
    def color(self, color:agcolor.Color) -> None:
        return self._intf.set_property(PathPoint._metadata, PathPoint._set_color_metadata, color)

    _get_translucency_metadata = { "offset" : _get_translucency_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def translucency(self) -> float:
        """The path point translucency."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_translucency_metadata)

    _set_translucency_metadata = { "offset" : _set_translucency_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @translucency.setter
    def translucency(self, translucency:float) -> None:
        return self._intf.set_property(PathPoint._metadata, PathPoint._set_translucency_metadata, translucency)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """The path point outline color."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(PathPoint._metadata, PathPoint._set_outline_color_metadata, outline_color)

    _get_outline_translucency_metadata = { "offset" : _get_outline_translucency_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def outline_translucency(self) -> float:
        """The path point outline translucency."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_outline_translucency_metadata)

    _set_outline_translucency_metadata = { "offset" : _set_outline_translucency_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @outline_translucency.setter
    def outline_translucency(self, outline_translucency:float) -> None:
        return self._intf.set_property(PathPoint._metadata, PathPoint._set_outline_translucency_metadata, outline_translucency)

    _get_is_translucent_metadata = { "offset" : _get_is_translucent_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_translucent(self) -> bool:
        """Whether the path point is translucent."""
        return self._intf.get_property(PathPoint._metadata, PathPoint._get_is_translucent_metadata)

    _property_names[position] = "position"
    _property_names[date] = "date"
    _property_names[color] = "color"
    _property_names[translucency] = "translucency"
    _property_names[outline_color] = "outline_color"
    _property_names[outline_translucency] = "outline_translucency"
    _property_names[is_translucent] = "is_translucent"

    def __init__(self, source_object=None):
        """Construct an object of type PathPoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PathPoint)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PathPoint, [PathPoint, ])

agcls.AgClassCatalog.add_catalog_entry((5556206296547912371, 8503175498596258195), PathPoint)
agcls.AgTypeNameMap["PathPoint"] = PathPoint

class PathPointFactory(SupportsDeleteCallback):
    """Create Path Primitive's path points."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _initialize_method_offset = 1
    _initialize_with_date_method_offset = 2
    _initialize_with_date_and_position_method_offset = 3
    _initialize_with_date_position_and_color_method_offset = 4
    _initialize_with_date_position_color_and_translucency_method_offset = 5
    _metadata = {
        "iid_data" : (5266970694628181700, 7367838054053532080),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PathPointFactory)

    _initialize_metadata = { "offset" : _initialize_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def initialize(self) -> "PathPoint":
        """Initialize a new path point."""
        return self._intf.invoke(PathPointFactory._metadata, PathPointFactory._initialize_metadata, OutArg())

    _initialize_with_date_metadata = { "offset" : _initialize_with_date_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.InterfaceOutArg,) }
    def initialize_with_date(self, path_point_date:"Date") -> "PathPoint":
        """Initialize a new path point with the given date."""
        return self._intf.invoke(PathPointFactory._metadata, PathPointFactory._initialize_with_date_metadata, path_point_date, OutArg())

    _initialize_with_date_and_position_metadata = { "offset" : _initialize_with_date_and_position_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.LPSafearrayArg, agmarshall.InterfaceOutArg,) }
    def initialize_with_date_and_position(self, path_point_date:"Date", position:list) -> "PathPoint":
        """Initialize a new path point with the given date and position."""
        return self._intf.invoke(PathPointFactory._metadata, PathPointFactory._initialize_with_date_and_position_metadata, path_point_date, position, OutArg())

    _initialize_with_date_position_and_color_metadata = { "offset" : _initialize_with_date_position_and_color_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.OLE_COLOR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.LPSafearrayArg, agmarshall.OLEColorArg, agmarshall.InterfaceOutArg,) }
    def initialize_with_date_position_and_color(self, path_point_date:"Date", position:list, color:agcolor.Color) -> "PathPoint":
        """Initialize a new path point with the given date, position and color."""
        return self._intf.invoke(PathPointFactory._metadata, PathPointFactory._initialize_with_date_position_and_color_metadata, path_point_date, position, color, OutArg())

    _initialize_with_date_position_color_and_translucency_metadata = { "offset" : _initialize_with_date_position_color_and_translucency_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.OLE_COLOR, agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.LPSafearrayArg, agmarshall.OLEColorArg, agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def initialize_with_date_position_color_and_translucency(self, path_point_date:"Date", position:list, color:agcolor.Color, translucency:float) -> "PathPoint":
        """Initialize a new path point with the given date, position, color and translucency."""
        return self._intf.invoke(PathPointFactory._metadata, PathPointFactory._initialize_with_date_position_color_and_translucency_metadata, path_point_date, position, color, translucency, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type PathPointFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PathPointFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PathPointFactory, [PathPointFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5489537290029782725, 4580717587449338530), PathPointFactory)
agcls.AgTypeNameMap["PathPointFactory"] = PathPointFactory

class BoundingSphere(SupportsDeleteCallback):
    """
    A sphere that encapsulates an object.

    Examples
    --------
    Create a Bounding Sphere:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> sphere = manager.initializers.bounding_sphere.initialize([[-1061.22], [-5773.98], [4456.04]], 100)
    """

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_center_method_offset = 1
    _get_radius_method_offset = 2
    _metadata = {
        "iid_data" : (4943385150456231790, 7841321641557182081),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BoundingSphere)

    _get_center_metadata = { "offset" : _get_center_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def center(self) -> list:
        """A center of the bounding sphere. The center point is specified as one-dimensional array with three elements corresponding to (X,Y,Z) cartesian coordinates."""
        return self._intf.get_property(BoundingSphere._metadata, BoundingSphere._get_center_metadata)

    _get_radius_metadata = { "offset" : _get_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius(self) -> float:
        """A radius of the bounding sphere."""
        return self._intf.get_property(BoundingSphere._metadata, BoundingSphere._get_radius_metadata)

    _property_names[center] = "center"
    _property_names[radius] = "radius"

    def __init__(self, source_object=None):
        """Construct an object of type BoundingSphere."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BoundingSphere)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BoundingSphere, [BoundingSphere, ])

agcls.AgClassCatalog.add_catalog_entry((5560913133829781863, 6998370718496213136), BoundingSphere)
agcls.AgTypeNameMap["BoundingSphere"] = BoundingSphere

class BoundingSphereFactory(SupportsDeleteCallback):
    """Create instances of the bounding sphere type."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _initialize_method_offset = 1
    _get_maximum_radius_bounding_sphere_method_offset = 2
    _metadata = {
        "iid_data" : (5476713812153226512, 18394739804475761564),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BoundingSphereFactory)

    _initialize_metadata = { "offset" : _initialize_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def initialize(self, center:list, radius:float) -> "BoundingSphere":
        """Create instances of BoundingSphere."""
        return self._intf.invoke(BoundingSphereFactory._metadata, BoundingSphereFactory._initialize_metadata, center, radius, OutArg())

    _get_maximum_radius_bounding_sphere_metadata = { "offset" : _get_maximum_radius_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maximum_radius_bounding_sphere(self) -> "BoundingSphere":
        """Get the bounding sphere of maximum possible radius."""
        return self._intf.get_property(BoundingSphereFactory._metadata, BoundingSphereFactory._get_maximum_radius_bounding_sphere_metadata)

    _property_names[maximum_radius_bounding_sphere] = "maximum_radius_bounding_sphere"

    def __init__(self, source_object=None):
        """Construct an object of type BoundingSphereFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BoundingSphereFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BoundingSphereFactory, [BoundingSphereFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5271520243375596029, 5819040470255887235), BoundingSphereFactory)
agcls.AgTypeNameMap["BoundingSphereFactory"] = BoundingSphereFactory

class TextureFilter2D(SupportsDeleteCallback):
    """A texture filter."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minification_filter_method_offset = 1
    _get_magnification_filter_method_offset = 2
    _get_wrap_s_method_offset = 3
    _get_wrap_t_method_offset = 4
    _get_nearest_clamp_to_edge_method_offset = 5
    _get_nearest_repeat_method_offset = 6
    _get_linear_clamp_to_edge_method_offset = 7
    _get_linear_repeat_method_offset = 8
    _metadata = {
        "iid_data" : (5541023743832152076, 10611954755374831295),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextureFilter2D)

    _get_minification_filter_metadata = { "offset" : _get_minification_filter_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MinificationFilter),) }
    @property
    def minification_filter(self) -> "MinificationFilter":
        """Get the minification filter used when the pixel being textured maps to an area less than or equal to one texel."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_minification_filter_metadata)

    _get_magnification_filter_metadata = { "offset" : _get_magnification_filter_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MagnificationFilter),) }
    @property
    def magnification_filter(self) -> "MagnificationFilter":
        """Get the magnification filter used when the pixel being textured maps to an area greater than one texel."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_magnification_filter_metadata)

    _get_wrap_s_metadata = { "offset" : _get_wrap_s_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TextureWrap),) }
    @property
    def wrap_s(self) -> "TextureWrap":
        """Get the texture wrap for the s direction."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_wrap_s_metadata)

    _get_wrap_t_metadata = { "offset" : _get_wrap_t_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TextureWrap),) }
    @property
    def wrap_t(self) -> "TextureWrap":
        """Get the texture wrap for the t direction."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_wrap_t_metadata)

    _get_nearest_clamp_to_edge_metadata = { "offset" : _get_nearest_clamp_to_edge_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def nearest_clamp_to_edge(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Nearest, MagnificationFilter: Nearest, WrapS: ClampToEdge, WrapT: ClampToEdge."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_nearest_clamp_to_edge_metadata)

    _get_nearest_repeat_metadata = { "offset" : _get_nearest_repeat_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def nearest_repeat(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Nearest, MagnificationFilter: Nearest, WrapS: Repeat, WrapT: Repeat."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_nearest_repeat_metadata)

    _get_linear_clamp_to_edge_metadata = { "offset" : _get_linear_clamp_to_edge_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def linear_clamp_to_edge(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Linear, MagnificationFilter: Linear, WrapS: ClampToEdge, WrapT: ClampToEdge."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_linear_clamp_to_edge_metadata)

    _get_linear_repeat_metadata = { "offset" : _get_linear_repeat_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def linear_repeat(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Linear, MagnificationFilter: Linear, WrapS: Repeat, WrapT: Repeat."""
        return self._intf.get_property(TextureFilter2D._metadata, TextureFilter2D._get_linear_repeat_metadata)

    _property_names[minification_filter] = "minification_filter"
    _property_names[magnification_filter] = "magnification_filter"
    _property_names[wrap_s] = "wrap_s"
    _property_names[wrap_t] = "wrap_t"
    _property_names[nearest_clamp_to_edge] = "nearest_clamp_to_edge"
    _property_names[nearest_repeat] = "nearest_repeat"
    _property_names[linear_clamp_to_edge] = "linear_clamp_to_edge"
    _property_names[linear_repeat] = "linear_repeat"

    def __init__(self, source_object=None):
        """Construct an object of type TextureFilter2D."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TextureFilter2D)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TextureFilter2D, [TextureFilter2D, ])

agcls.AgClassCatalog.add_catalog_entry((5701511467830951335, 18354553182752180127), TextureFilter2D)
agcls.AgTypeNameMap["TextureFilter2D"] = TextureFilter2D

class TextureFilter2DFactory(SupportsDeleteCallback):
    """Create texture filters."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_nearest_clamp_to_edge_method_offset = 1
    _get_nearest_repeat_method_offset = 2
    _get_linear_clamp_to_edge_method_offset = 3
    _get_linear_repeat_method_offset = 4
    _initialize_method_offset = 5
    _initialize_with_texture_wrap_method_offset = 6
    _initialize_with_minification_and_magnification_method_offset = 7
    _metadata = {
        "iid_data" : (5619732544747564303, 16515141689419169207),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextureFilter2DFactory)

    _get_nearest_clamp_to_edge_metadata = { "offset" : _get_nearest_clamp_to_edge_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def nearest_clamp_to_edge(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Nearest, MagnificationFilter: Nearest, WrapS: ClampToEdge, WrapT: ClampToEdge."""
        return self._intf.get_property(TextureFilter2DFactory._metadata, TextureFilter2DFactory._get_nearest_clamp_to_edge_metadata)

    _get_nearest_repeat_metadata = { "offset" : _get_nearest_repeat_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def nearest_repeat(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Nearest, MagnificationFilter: Nearest, WrapS: Repeat, WrapT: Repeat."""
        return self._intf.get_property(TextureFilter2DFactory._metadata, TextureFilter2DFactory._get_nearest_repeat_metadata)

    _get_linear_clamp_to_edge_metadata = { "offset" : _get_linear_clamp_to_edge_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def linear_clamp_to_edge(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Linear, MagnificationFilter: Linear, WrapS: ClampToEdge, WrapT: ClampToEdge."""
        return self._intf.get_property(TextureFilter2DFactory._metadata, TextureFilter2DFactory._get_linear_clamp_to_edge_metadata)

    _get_linear_repeat_metadata = { "offset" : _get_linear_repeat_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def linear_repeat(self) -> "TextureFilter2D":
        """Get a texture filter with the following properties: MinificationFilter: Linear, MagnificationFilter: Linear, WrapS: Repeat, WrapT: Repeat."""
        return self._intf.get_property(TextureFilter2DFactory._metadata, TextureFilter2DFactory._get_linear_repeat_metadata)

    _initialize_metadata = { "offset" : _initialize_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, agcom.LONG, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(MinificationFilter), agmarshall.EnumArg(MagnificationFilter), agmarshall.EnumArg(TextureWrap), agmarshall.EnumArg(TextureWrap), agmarshall.InterfaceOutArg,) }
    def initialize(self, minification_filter:"MinificationFilter", magnification_filter:"MagnificationFilter", wrap_s:"TextureWrap", wrap_t:"TextureWrap") -> "TextureFilter2D":
        """Create a texture filter using the specified minification/magnification options and texture wrap."""
        return self._intf.invoke(TextureFilter2DFactory._metadata, TextureFilter2DFactory._initialize_metadata, minification_filter, magnification_filter, wrap_s, wrap_t, OutArg())

    _initialize_with_texture_wrap_metadata = { "offset" : _initialize_with_texture_wrap_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(TextureWrap), agmarshall.EnumArg(TextureWrap), agmarshall.InterfaceOutArg,) }
    def initialize_with_texture_wrap(self, wrap_s:"TextureWrap", wrap_t:"TextureWrap") -> "TextureFilter2D":
        """Create a texture filter using the specified texture wrap."""
        return self._intf.invoke(TextureFilter2DFactory._metadata, TextureFilter2DFactory._initialize_with_texture_wrap_metadata, wrap_s, wrap_t, OutArg())

    _initialize_with_minification_and_magnification_metadata = { "offset" : _initialize_with_minification_and_magnification_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(MinificationFilter), agmarshall.EnumArg(MagnificationFilter), agmarshall.InterfaceOutArg,) }
    def initialize_with_minification_and_magnification(self, minification_filter:"MinificationFilter", magnification_filter:"MagnificationFilter") -> "TextureFilter2D":
        """Create a texture filter using the specified minification/magnification options."""
        return self._intf.invoke(TextureFilter2DFactory._metadata, TextureFilter2DFactory._initialize_with_minification_and_magnification_metadata, minification_filter, magnification_filter, OutArg())

    _property_names[nearest_clamp_to_edge] = "nearest_clamp_to_edge"
    _property_names[nearest_repeat] = "nearest_repeat"
    _property_names[linear_clamp_to_edge] = "linear_clamp_to_edge"
    _property_names[linear_repeat] = "linear_repeat"

    def __init__(self, source_object=None):
        """Construct an object of type TextureFilter2DFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TextureFilter2DFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TextureFilter2DFactory, [TextureFilter2DFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5071727819948482052, 5422654417578581655), TextureFilter2DFactory)
agcls.AgTypeNameMap["TextureFilter2DFactory"] = TextureFilter2DFactory

class RendererTexture2D(SupportsDeleteCallback):
    """A 2D Texture. A texture represents an image that is ready for use by objects such as primitives and overlays. Textures typically reside in video memory."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_template_method_offset = 1
    _metadata = {
        "iid_data" : (5417815738939846526, 12898891467340117641),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RendererTexture2D)

    _get_template_metadata = { "offset" : _get_template_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def template(self) -> "RendererTextureTemplate2D":
        """Get a template from which the texture was created."""
        return self._intf.get_property(RendererTexture2D._metadata, RendererTexture2D._get_template_metadata)

    _property_names[template] = "template"

    def __init__(self, source_object=None):
        """Construct an object of type RendererTexture2D."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RendererTexture2D)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RendererTexture2D, [RendererTexture2D, ])

agcls.AgClassCatalog.add_catalog_entry((5381155970277415318, 3622939501334827702), RendererTexture2D)
agcls.AgTypeNameMap["RendererTexture2D"] = RendererTexture2D

class RendererTextureTemplate2D(SupportsDeleteCallback):
    """Template object containing attributes required to create a 2D texture."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_internal_format_method_offset = 1
    _get_width_method_offset = 2
    _get_height_method_offset = 3
    _metadata = {
        "iid_data" : (5316263756575800396, 1184309665638889406),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RendererTextureTemplate2D)

    _get_internal_format_metadata = { "offset" : _get_internal_format_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TextureFormat),) }
    @property
    def internal_format(self) -> "TextureFormat":
        """Get an internal format of the texture."""
        return self._intf.get_property(RendererTextureTemplate2D._metadata, RendererTextureTemplate2D._get_internal_format_metadata)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def width(self) -> int:
        """Get a width of the texture in texels."""
        return self._intf.get_property(RendererTextureTemplate2D._metadata, RendererTextureTemplate2D._get_width_metadata)

    _get_height_metadata = { "offset" : _get_height_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def height(self) -> int:
        """Get a height of the texture in texels."""
        return self._intf.get_property(RendererTextureTemplate2D._metadata, RendererTextureTemplate2D._get_height_metadata)

    _property_names[internal_format] = "internal_format"
    _property_names[width] = "width"
    _property_names[height] = "height"

    def __init__(self, source_object=None):
        """Construct an object of type RendererTextureTemplate2D."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RendererTextureTemplate2D)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RendererTextureTemplate2D, [RendererTextureTemplate2D, ])

agcls.AgClassCatalog.add_catalog_entry((4903592959625323336, 10408150276974255496), RendererTextureTemplate2D)
agcls.AgTypeNameMap["RendererTextureTemplate2D"] = RendererTextureTemplate2D

class PathPointCollection(SupportsDeleteCallback):
    """A collection of path points."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5688438264765407641, 15401492857366987175),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PathPointCollection)
    def __iter__(self):
        """Create an iterator for the PathPointCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "PathPoint":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(PathPointCollection._metadata, PathPointCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "PathPoint":
        """Return a path point at the specified position in the collection."""
        return self._intf.invoke(PathPointCollection._metadata, PathPointCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(PathPointCollection._metadata, PathPointCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type PathPointCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PathPointCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PathPointCollection, [PathPointCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4718379932532326826, 11444711569154001853), PathPointCollection)
agcls.AgTypeNameMap["PathPointCollection"] = PathPointCollection

class ObjectCollection(SupportsDeleteCallback):
    """A collection of objects."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (4775746742546311368, 7552453374623214760),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ObjectCollection)
    def __iter__(self):
        """Create an iterator for the ObjectCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> typing.Any:
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(ObjectCollection._metadata, ObjectCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.LongArg, agmarshall.VariantArg,) }
    def item(self, index:int) -> typing.Any:
        """Return an item in the collection at the specified index."""
        return self._intf.invoke(ObjectCollection._metadata, ObjectCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(ObjectCollection._metadata, ObjectCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ObjectCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ObjectCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ObjectCollection, [ObjectCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5689274977493123915, 2183068153647623591), ObjectCollection)
agcls.AgTypeNameMap["ObjectCollection"] = ObjectCollection

class SceneCollection(SupportsDeleteCallback):
    """A collection of scenes."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5372104015339339678, 805902352111732909),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SceneCollection)
    def __iter__(self):
        """Create an iterator for the SceneCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "Scene":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Total number of scenes in the collection."""
        return self._intf.get_property(SceneCollection._metadata, SceneCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "Scene":
        """Return a scene in the collection at a specified index."""
        return self._intf.invoke(SceneCollection._metadata, SceneCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(SceneCollection._metadata, SceneCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type SceneCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SceneCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SceneCollection, [SceneCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5301811212438284748, 14423411918377718926), SceneCollection)
agcls.AgTypeNameMap["SceneCollection"] = SceneCollection

class ScreenOverlayPickResultCollection(SupportsDeleteCallback):
    """A collection of pick results."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5200561598852205868, 13511979759557490105),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ScreenOverlayPickResultCollection)
    def __iter__(self):
        """Create an iterator for the ScreenOverlayPickResultCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ScreenOverlayPickResult":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(ScreenOverlayPickResultCollection._metadata, ScreenOverlayPickResultCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "ScreenOverlayPickResult":
        """Get an element at the specified position in the collection."""
        return self._intf.invoke(ScreenOverlayPickResultCollection._metadata, ScreenOverlayPickResultCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(ScreenOverlayPickResultCollection._metadata, ScreenOverlayPickResultCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlayPickResultCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ScreenOverlayPickResultCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlayPickResultCollection, [ScreenOverlayPickResultCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4677343897895766869, 11872430607576403131), ScreenOverlayPickResultCollection)
agcls.AgTypeNameMap["ScreenOverlayPickResultCollection"] = ScreenOverlayPickResultCollection

class GlobeImageOverlayAddCompleteEventArgs(SupportsDeleteCallback):
    """The event is raised when the globe image overlay is displayed for the first time after being added using AddAsync."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_overlay_method_offset = 1
    _metadata = {
        "iid_data" : (4932803116935452617, 13441102795444558522),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GlobeImageOverlayAddCompleteEventArgs)

    _get_overlay_metadata = { "offset" : _get_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlay(self) -> "IGlobeImageOverlay":
        """The overlay object that was added earlier using AddAsync."""
        return self._intf.get_property(GlobeImageOverlayAddCompleteEventArgs._metadata, GlobeImageOverlayAddCompleteEventArgs._get_overlay_metadata)

    _property_names[overlay] = "overlay"

    def __init__(self, source_object=None):
        """Construct an object of type GlobeImageOverlayAddCompleteEventArgs."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GlobeImageOverlayAddCompleteEventArgs)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GlobeImageOverlayAddCompleteEventArgs, [GlobeImageOverlayAddCompleteEventArgs, ])

agcls.AgClassCatalog.add_catalog_entry((5231802800479662057, 2104184534495932053), GlobeImageOverlayAddCompleteEventArgs)
agcls.AgTypeNameMap["GlobeImageOverlayAddCompleteEventArgs"] = GlobeImageOverlayAddCompleteEventArgs

class TerrainOverlayAddCompleteEventArgs(SupportsDeleteCallback):
    """The event is raised when the terrain overlay is displayed for the first time after having been added using AddAsync."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_overlay_method_offset = 1
    _metadata = {
        "iid_data" : (4791585591808952706, 1636302179759337885),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TerrainOverlayAddCompleteEventArgs)

    _get_overlay_metadata = { "offset" : _get_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlay(self) -> "ITerrainOverlay":
        """The terrain overlay being displayed for the first time."""
        return self._intf.get_property(TerrainOverlayAddCompleteEventArgs._metadata, TerrainOverlayAddCompleteEventArgs._get_overlay_metadata)

    _property_names[overlay] = "overlay"

    def __init__(self, source_object=None):
        """Construct an object of type TerrainOverlayAddCompleteEventArgs."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TerrainOverlayAddCompleteEventArgs)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TerrainOverlayAddCompleteEventArgs, [TerrainOverlayAddCompleteEventArgs, ])

agcls.AgClassCatalog.add_catalog_entry((4866096509316980678, 14412758423447289486), TerrainOverlayAddCompleteEventArgs)
agcls.AgTypeNameMap["TerrainOverlayAddCompleteEventArgs"] = TerrainOverlayAddCompleteEventArgs

class PickResultCollection(SupportsDeleteCallback):
    """A collection of picked objects."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5309094562372375985, 3068566077694695567),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PickResultCollection)
    def __iter__(self):
        """Create an iterator for the PickResultCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "PickResult":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(PickResultCollection._metadata, PickResultCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "PickResult":
        """Return a picked object at the specified position in the collection."""
        return self._intf.invoke(PickResultCollection._metadata, PickResultCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(PickResultCollection._metadata, PickResultCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type PickResultCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PickResultCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PickResultCollection, [PickResultCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5712441455425732440, 17907447014582202282), PickResultCollection)
agcls.AgTypeNameMap["PickResultCollection"] = PickResultCollection

class RenderingEventArgs(SupportsDeleteCallback):
    """The event is raised when the scene is rendered."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_time_method_offset = 1
    _get_time_in_ep_secs_method_offset = 2
    _metadata = {
        "iid_data" : (4863600425244224034, 13619959685560500882),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RenderingEventArgs)

    _get_time_metadata = { "offset" : _get_time_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def time(self) -> "Date":
        """The time of the rendering event."""
        return self._intf.get_property(RenderingEventArgs._metadata, RenderingEventArgs._get_time_metadata)

    _get_time_in_ep_secs_metadata = { "offset" : _get_time_in_ep_secs_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_in_ep_secs(self) -> float:
        """The time of the rendering event (in STK's scenario epoch units)."""
        return self._intf.get_property(RenderingEventArgs._metadata, RenderingEventArgs._get_time_in_ep_secs_metadata)

    _property_names[time] = "time"
    _property_names[time_in_ep_secs] = "time_in_ep_secs"

    def __init__(self, source_object=None):
        """Construct an object of type RenderingEventArgs."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RenderingEventArgs)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RenderingEventArgs, [RenderingEventArgs, ])

agcls.AgClassCatalog.add_catalog_entry((5392329581593296225, 914494406188763269), RenderingEventArgs)
agcls.AgTypeNameMap["RenderingEventArgs"] = RenderingEventArgs

class BatchPrimitiveIndex(SupportsDeleteCallback):
    """Represents an individual item index that is associated with a batch primitive. Provides the Index of the individual item and the Primitive that contains that index..."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_index_method_offset = 1
    _get_primitive_method_offset = 2
    _metadata = {
        "iid_data" : (5055628909548856713, 16099616916943409061),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BatchPrimitiveIndex)

    _get_index_metadata = { "offset" : _get_index_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def index(self) -> int:
        """The index of the item contained by the Primitive."""
        return self._intf.get_property(BatchPrimitiveIndex._metadata, BatchPrimitiveIndex._get_index_metadata)

    _get_primitive_metadata = { "offset" : _get_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def primitive(self) -> "IPrimitive":
        """The Primitive that contains the item at the Index specified."""
        return self._intf.get_property(BatchPrimitiveIndex._metadata, BatchPrimitiveIndex._get_primitive_metadata)

    _property_names[index] = "index"
    _property_names[primitive] = "primitive"

    def __init__(self, source_object=None):
        """Construct an object of type BatchPrimitiveIndex."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BatchPrimitiveIndex)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BatchPrimitiveIndex, [BatchPrimitiveIndex, ])

agcls.AgClassCatalog.add_catalog_entry((5200164995912440169, 5658385359959925641), BatchPrimitiveIndex)
agcls.AgTypeNameMap["BatchPrimitiveIndex"] = BatchPrimitiveIndex

class KmlDocumentCollection(SupportsDeleteCallback):
    """A collection of KML documents."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5732338628688371025, 7645622644953202819),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlDocumentCollection)
    def __iter__(self):
        """Create an iterator for the KmlDocumentCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "KmlDocument":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(KmlDocumentCollection._metadata, KmlDocumentCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "KmlDocument":
        """Get an element at the specified position in the collection."""
        return self._intf.invoke(KmlDocumentCollection._metadata, KmlDocumentCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(KmlDocumentCollection._metadata, KmlDocumentCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type KmlDocumentCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlDocumentCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlDocumentCollection, [KmlDocumentCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5215825726924847476, 8941538243989311415), KmlDocumentCollection)
agcls.AgTypeNameMap["KmlDocumentCollection"] = KmlDocumentCollection

class KmlFeatureCollection(SupportsDeleteCallback):
    """A collection of KML features."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5283280579351142389, 14460084154557776063),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlFeatureCollection)
    def __iter__(self):
        """Create an iterator for the KmlFeatureCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IKmlFeature":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """A total number of elements in the collection."""
        return self._intf.get_property(KmlFeatureCollection._metadata, KmlFeatureCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "IKmlFeature":
        """Get an element at the specified position in the collection."""
        return self._intf.invoke(KmlFeatureCollection._metadata, KmlFeatureCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(KmlFeatureCollection._metadata, KmlFeatureCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type KmlFeatureCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlFeatureCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlFeatureCollection, [KmlFeatureCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5727078008905494174, 12095817447017663925), KmlFeatureCollection)
agcls.AgTypeNameMap["KmlFeatureCollection"] = KmlFeatureCollection

class KmlDocumentLoadedEventArgs(SupportsDeleteCallback):
    """The event is raised when a KML document has been loaded."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_document_method_offset = 1
    _get_exception_method_offset = 2
    _metadata = {
        "iid_data" : (5040252077483722883, 4444885371105009835),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlDocumentLoadedEventArgs)

    _get_document_metadata = { "offset" : _get_document_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def document(self) -> "KmlDocument":
        """Get the KML document associated with the load event."""
        return self._intf.get_property(KmlDocumentLoadedEventArgs._metadata, KmlDocumentLoadedEventArgs._get_document_metadata)

    _get_exception_metadata = { "offset" : _get_exception_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def exception(self) -> str:
        """Return an error message, if an error occurred while loading the KML; otherwise an empty string."""
        return self._intf.get_property(KmlDocumentLoadedEventArgs._metadata, KmlDocumentLoadedEventArgs._get_exception_metadata)

    _property_names[document] = "document"
    _property_names[exception] = "exception"

    def __init__(self, source_object=None):
        """Construct an object of type KmlDocumentLoadedEventArgs."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlDocumentLoadedEventArgs)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlDocumentLoadedEventArgs, [KmlDocumentLoadedEventArgs, ])

agcls.AgClassCatalog.add_catalog_entry((5646942824342644840, 16385106561904486318), KmlDocumentLoadedEventArgs)
agcls.AgTypeNameMap["KmlDocumentLoadedEventArgs"] = KmlDocumentLoadedEventArgs

class FactoryAndInitializers(SupportsDeleteCallback):
    """Methods and properties are used to initialize new primitives, display conditions, screen overlays, textures and many other types; compute and retrieve triangulator results and access global properties (what's known as static properties, static methods a..."""

    _num_methods = 83
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_box_triangulator_method_offset = 1
    _get_cylinder_triangulator_method_offset = 2
    _get_ellipsoid_triangulator_method_offset = 3
    _get_extruded_polyline_triangulator_method_offset = 4
    _get_surface_extent_triangulator_method_offset = 5
    _get_surface_polygon_triangulator_method_offset = 6
    _get_surface_shapes_method_offset = 7
    _get_agi_processed_image_globe_overlay_method_offset = 8
    _get_agi_processed_terrain_overlay_method_offset = 9
    _get_agi_roam_image_globe_overlay_method_offset = 10
    _get_custom_image_globe_overlay_plugin_activator_method_offset = 11
    _get_geospatial_image_globe_overlay_method_offset = 12
    _get_projected_raster_overlay_method_offset = 13
    _get_projection_method_offset = 14
    _get_altitude_display_condition_method_offset = 15
    _get_composite_display_condition_method_offset = 16
    _get_composite_primitive_method_offset = 17
    _get_constant_display_condition_method_offset = 18
    _get_distance_display_condition_method_offset = 19
    _get_distance_to_globe_overlay_display_condition_method_offset = 20
    _get_distance_to_position_display_condition_method_offset = 21
    _get_distance_to_primitive_display_condition_method_offset = 22
    _get_duration_path_primitive_update_policy_method_offset = 23
    _get_globe_image_overlay_method_offset = 24
    _get_graphics_font_method_offset = 25
    _get_great_arc_interpolator_method_offset = 26
    _get_alpha_from_luminance_filter_method_offset = 27
    _get_alpha_from_pixel_filter_method_offset = 28
    _get_alpha_from_raster_filter_method_offset = 29
    _get_band_extract_filter_method_offset = 30
    _get_band_order_filter_method_offset = 31
    _get_blur_filter_method_offset = 32
    _get_brightness_filter_method_offset = 33
    _get_color_to_luminance_filter_method_offset = 34
    _get_contrast_filter_method_offset = 35
    _get_convolution_filter_method_offset = 36
    _get_edge_detect_filter_method_offset = 37
    _get_filtering_raster_stream_method_offset = 38
    _get_flip_filter_method_offset = 39
    _get_gamma_correction_filter_method_offset = 40
    _get_gaussian_blur_filter_method_offset = 41
    _get_gradient_detect_filter_method_offset = 42
    _get_jpeg2000_writer_method_offset = 43
    _get_levels_filter_method_offset = 44
    _get_projection_raster_stream_plugin_activator_method_offset = 45
    _get_raster_method_offset = 46
    _get_raster_attributes_method_offset = 47
    _get_rotate_filter_method_offset = 48
    _get_sequence_filter_method_offset = 49
    _get_sharpen_filter_method_offset = 50
    _get_video_stream_method_offset = 51
    _get_marker_batch_primitive_method_offset = 52
    _get_marker_batch_primitive_optional_parameters_method_offset = 53
    _get_maximum_count_path_primitive_update_policy_method_offset = 54
    _get_model_primitive_method_offset = 55
    _get_path_primitive_method_offset = 56
    _get_pixel_size_display_condition_method_offset = 57
    _get_point_batch_primitive_method_offset = 58
    _get_polyline_primitive_method_offset = 59
    _get_raster_image_globe_overlay_method_offset = 60
    _get_rhumb_line_interpolator_method_offset = 61
    _get_scene_display_condition_method_offset = 62
    _get_scene_manager_method_offset = 63
    _get_screen_overlay_method_offset = 64
    _get_solid_primitive_method_offset = 65
    _get_surface_mesh_primitive_method_offset = 66
    _get_terrain_overlay_method_offset = 67
    _get_text_batch_primitive_method_offset = 68
    _get_text_batch_primitive_optional_parameters_method_offset = 69
    _get_texture_matrix_method_offset = 70
    _get_texture_screen_overlay_method_offset = 71
    _get_time_interval_display_condition_method_offset = 72
    _get_triangle_mesh_primitive_method_offset = 73
    _get_triangle_mesh_primitive_optional_parameters_method_offset = 74
    _get_texture_filter_2d_method_offset = 75
    _get_bounding_sphere_method_offset = 76
    _get_path_point_method_offset = 77
    _get_text_overlay_method_offset = 78
    _get_agi_custom_terrain_overlay_method_offset = 79
    _get_axes_primitive_method_offset = 80
    _get_vector_primitive_method_offset = 81
    _get_polyline_primitive_optional_parameters_method_offset = 82
    _get_point_batch_primitive_optional_parameters_method_offset = 83
    _metadata = {
        "iid_data" : (5413969555641802483, 7605805363040079775),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, FactoryAndInitializers)

    _get_box_triangulator_metadata = { "offset" : _get_box_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def box_triangulator(self) -> "BoxTriangulatorInitializer":
        """Access global methods and properties of BoxTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_box_triangulator_metadata)

    _get_cylinder_triangulator_metadata = { "offset" : _get_cylinder_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def cylinder_triangulator(self) -> "CylinderTriangulatorInitializer":
        """Access global methods and properties of CylinderTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_cylinder_triangulator_metadata)

    _get_ellipsoid_triangulator_metadata = { "offset" : _get_ellipsoid_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def ellipsoid_triangulator(self) -> "EllipsoidTriangulatorInitializer":
        """Access global methods and properties of EllipsoidTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_ellipsoid_triangulator_metadata)

    _get_extruded_polyline_triangulator_metadata = { "offset" : _get_extruded_polyline_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def extruded_polyline_triangulator(self) -> "ExtrudedPolylineTriangulatorInitializer":
        """Access global methods and properties of ExtrudedPolylineTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_extruded_polyline_triangulator_metadata)

    _get_surface_extent_triangulator_metadata = { "offset" : _get_surface_extent_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def surface_extent_triangulator(self) -> "SurfaceExtentTriangulatorInitializer":
        """Access global methods and properties of SurfaceExtentTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_surface_extent_triangulator_metadata)

    _get_surface_polygon_triangulator_metadata = { "offset" : _get_surface_polygon_triangulator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def surface_polygon_triangulator(self) -> "SurfacePolygonTriangulatorInitializer":
        """Access global methods and properties of SurfacePolygonTriangulator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_surface_polygon_triangulator_metadata)

    _get_surface_shapes_metadata = { "offset" : _get_surface_shapes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def surface_shapes(self) -> "SurfaceShapesInitializer":
        """Access global methods and properties of SurfaceShapes (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_surface_shapes_metadata)

    _get_agi_processed_image_globe_overlay_metadata = { "offset" : _get_agi_processed_image_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def agi_processed_image_globe_overlay(self) -> "AGIProcessedImageGlobeOverlayFactory":
        """Access global methods and properties of AGIProcessedImageGlobeOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_agi_processed_image_globe_overlay_metadata)

    _get_agi_processed_terrain_overlay_metadata = { "offset" : _get_agi_processed_terrain_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def agi_processed_terrain_overlay(self) -> "AGIProcessedTerrainOverlayFactory":
        """Access global methods and properties of AGIProcessedTerrainOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_agi_processed_terrain_overlay_metadata)

    _get_agi_roam_image_globe_overlay_metadata = { "offset" : _get_agi_roam_image_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def agi_roam_image_globe_overlay(self) -> "AGIRoamImageGlobeOverlayFactory":
        """Access global methods and properties of AGIRoamImageGlobeOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_agi_roam_image_globe_overlay_metadata)

    _get_custom_image_globe_overlay_plugin_activator_metadata = { "offset" : _get_custom_image_globe_overlay_plugin_activator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_image_globe_overlay_plugin_activator(self) -> "CustomImageGlobeOverlayPluginActivatorFactory":
        """Access global methods and properties of CustomImageGlobeOverlayPluginActivator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_custom_image_globe_overlay_plugin_activator_metadata)

    _get_geospatial_image_globe_overlay_metadata = { "offset" : _get_geospatial_image_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def geospatial_image_globe_overlay(self) -> "GeospatialImageGlobeOverlayFactory":
        """Access global methods and properties of GeospatialImageGlobeOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_geospatial_image_globe_overlay_metadata)

    _get_projected_raster_overlay_metadata = { "offset" : _get_projected_raster_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def projected_raster_overlay(self) -> "ProjectedRasterOverlayFactory":
        """Access global methods and properties of ProjectedRasterOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_projected_raster_overlay_metadata)

    _get_projection_metadata = { "offset" : _get_projection_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def projection(self) -> "ProjectionFactory":
        """Access global methods and properties of Projection (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_projection_metadata)

    _get_altitude_display_condition_metadata = { "offset" : _get_altitude_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_display_condition(self) -> "AltitudeDisplayConditionFactory":
        """Access global methods and properties of AltitudeDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_altitude_display_condition_metadata)

    _get_composite_display_condition_metadata = { "offset" : _get_composite_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def composite_display_condition(self) -> "CompositeDisplayConditionFactory":
        """Access global methods and properties of CompositeDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_composite_display_condition_metadata)

    _get_composite_primitive_metadata = { "offset" : _get_composite_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def composite_primitive(self) -> "CompositePrimitiveFactory":
        """Access global methods and properties of CompositePrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_composite_primitive_metadata)

    _get_constant_display_condition_metadata = { "offset" : _get_constant_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def constant_display_condition(self) -> "ConstantDisplayConditionFactory":
        """Access global methods and properties of ConstantDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_constant_display_condition_metadata)

    _get_distance_display_condition_metadata = { "offset" : _get_distance_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_display_condition(self) -> "DistanceDisplayConditionFactory":
        """Access global methods and properties of DistanceDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_distance_display_condition_metadata)

    _get_distance_to_globe_overlay_display_condition_metadata = { "offset" : _get_distance_to_globe_overlay_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_to_globe_overlay_display_condition(self) -> "DistanceToGlobeOverlayDisplayConditionFactory":
        """Access global methods and properties of DistanceToGlobeOverlayDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_distance_to_globe_overlay_display_condition_metadata)

    _get_distance_to_position_display_condition_metadata = { "offset" : _get_distance_to_position_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_to_position_display_condition(self) -> "DistanceToPositionDisplayConditionFactory":
        """Access global methods and properties of DistanceToPositionDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_distance_to_position_display_condition_metadata)

    _get_distance_to_primitive_display_condition_metadata = { "offset" : _get_distance_to_primitive_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_to_primitive_display_condition(self) -> "DistanceToPrimitiveDisplayConditionFactory":
        """Access global methods and properties of DistanceToPrimitiveDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_distance_to_primitive_display_condition_metadata)

    _get_duration_path_primitive_update_policy_metadata = { "offset" : _get_duration_path_primitive_update_policy_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def duration_path_primitive_update_policy(self) -> "DurationPathPrimitiveUpdatePolicyFactory":
        """Access global methods and properties of DurationPathPrimitiveUpdatePolicy (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_duration_path_primitive_update_policy_metadata)

    _get_globe_image_overlay_metadata = { "offset" : _get_globe_image_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def globe_image_overlay(self) -> "GlobeImageOverlayInitializer":
        """Access global methods and properties of GlobeImageOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_globe_image_overlay_metadata)

    _get_graphics_font_metadata = { "offset" : _get_graphics_font_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def graphics_font(self) -> "GraphicsFontFactory":
        """Access global methods and properties of GraphicsFont (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_graphics_font_metadata)

    _get_great_arc_interpolator_metadata = { "offset" : _get_great_arc_interpolator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def great_arc_interpolator(self) -> "GreatArcInterpolatorFactory":
        """Access global methods and properties of GreatArcInterpolator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_great_arc_interpolator_metadata)

    _get_alpha_from_luminance_filter_metadata = { "offset" : _get_alpha_from_luminance_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def alpha_from_luminance_filter(self) -> "AlphaFromLuminanceFilterFactory":
        """Access global methods and properties of AlphaFromLuminanceFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_alpha_from_luminance_filter_metadata)

    _get_alpha_from_pixel_filter_metadata = { "offset" : _get_alpha_from_pixel_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def alpha_from_pixel_filter(self) -> "AlphaFromPixelFilterFactory":
        """Access global methods and properties of AlphaFromPixelFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_alpha_from_pixel_filter_metadata)

    _get_alpha_from_raster_filter_metadata = { "offset" : _get_alpha_from_raster_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def alpha_from_raster_filter(self) -> "AlphaFromRasterFilterFactory":
        """Access global methods and properties of AlphaFromRasterFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_alpha_from_raster_filter_metadata)

    _get_band_extract_filter_metadata = { "offset" : _get_band_extract_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def band_extract_filter(self) -> "BandExtractFilterFactory":
        """Access global methods and properties of BandExtractFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_band_extract_filter_metadata)

    _get_band_order_filter_metadata = { "offset" : _get_band_order_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def band_order_filter(self) -> "BandOrderFilterFactory":
        """Access global methods and properties of BandOrderFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_band_order_filter_metadata)

    _get_blur_filter_metadata = { "offset" : _get_blur_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def blur_filter(self) -> "BlurFilterFactory":
        """Access global methods and properties of BlurFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_blur_filter_metadata)

    _get_brightness_filter_metadata = { "offset" : _get_brightness_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def brightness_filter(self) -> "BrightnessFilterFactory":
        """Access global methods and properties of BrightnessFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_brightness_filter_metadata)

    _get_color_to_luminance_filter_metadata = { "offset" : _get_color_to_luminance_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def color_to_luminance_filter(self) -> "ColorToLuminanceFilterFactory":
        """Access global methods and properties of ColorToLuminanceFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_color_to_luminance_filter_metadata)

    _get_contrast_filter_metadata = { "offset" : _get_contrast_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def contrast_filter(self) -> "ContrastFilterFactory":
        """Access global methods and properties of ContrastFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_contrast_filter_metadata)

    _get_convolution_filter_metadata = { "offset" : _get_convolution_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convolution_filter(self) -> "ConvolutionFilterFactory":
        """Access global methods and properties of ConvolutionFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_convolution_filter_metadata)

    _get_edge_detect_filter_metadata = { "offset" : _get_edge_detect_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def edge_detect_filter(self) -> "EdgeDetectFilterFactory":
        """Access global methods and properties of EdgeDetectFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_edge_detect_filter_metadata)

    _get_filtering_raster_stream_metadata = { "offset" : _get_filtering_raster_stream_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def filtering_raster_stream(self) -> "FilteringRasterStreamFactory":
        """Access global methods and properties of FilteringRasterStream (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_filtering_raster_stream_metadata)

    _get_flip_filter_metadata = { "offset" : _get_flip_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def flip_filter(self) -> "FlipFilterFactory":
        """Access global methods and properties of FlipFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_flip_filter_metadata)

    _get_gamma_correction_filter_metadata = { "offset" : _get_gamma_correction_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def gamma_correction_filter(self) -> "GammaCorrectionFilterFactory":
        """Access global methods and properties of GammaCorrectionFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_gamma_correction_filter_metadata)

    _get_gaussian_blur_filter_metadata = { "offset" : _get_gaussian_blur_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def gaussian_blur_filter(self) -> "GaussianBlurFilterFactory":
        """Access global methods and properties of GaussianBlurFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_gaussian_blur_filter_metadata)

    _get_gradient_detect_filter_metadata = { "offset" : _get_gradient_detect_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def gradient_detect_filter(self) -> "GradientDetectFilterFactory":
        """Access global methods and properties of GradientDetectFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_gradient_detect_filter_metadata)

    _get_jpeg2000_writer_metadata = { "offset" : _get_jpeg2000_writer_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def jpeg2000_writer(self) -> "Jpeg2000WriterInitializer":
        """Access global methods and properties of Jpeg2000Writer (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_jpeg2000_writer_metadata)

    _get_levels_filter_metadata = { "offset" : _get_levels_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def levels_filter(self) -> "LevelsFilterFactory":
        """Access global methods and properties of LevelsFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_levels_filter_metadata)

    _get_projection_raster_stream_plugin_activator_metadata = { "offset" : _get_projection_raster_stream_plugin_activator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def projection_raster_stream_plugin_activator(self) -> "ProjectionRasterStreamPluginActivatorFactory":
        """Access global methods and properties of ProjectionRasterStreamPluginActivator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_projection_raster_stream_plugin_activator_metadata)

    _get_raster_metadata = { "offset" : _get_raster_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster(self) -> "RasterFactory":
        """Access global methods and properties of Raster (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_raster_metadata)

    _get_raster_attributes_metadata = { "offset" : _get_raster_attributes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster_attributes(self) -> "RasterAttributesFactory":
        """Access global methods and properties of RasterAttributes (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_raster_attributes_metadata)

    _get_rotate_filter_metadata = { "offset" : _get_rotate_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def rotate_filter(self) -> "RotateFilterFactory":
        """Access global methods and properties of RotateFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_rotate_filter_metadata)

    _get_sequence_filter_metadata = { "offset" : _get_sequence_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sequence_filter(self) -> "SequenceFilterFactory":
        """Access global methods and properties of SequenceFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_sequence_filter_metadata)

    _get_sharpen_filter_metadata = { "offset" : _get_sharpen_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sharpen_filter(self) -> "SharpenFilterFactory":
        """Access global methods and properties of SharpenFilter (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_sharpen_filter_metadata)

    _get_video_stream_metadata = { "offset" : _get_video_stream_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def video_stream(self) -> "VideoStreamFactory":
        """Access global methods and properties of VideoStream (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_video_stream_metadata)

    _get_marker_batch_primitive_metadata = { "offset" : _get_marker_batch_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def marker_batch_primitive(self) -> "MarkerBatchPrimitiveFactory":
        """Access global methods and properties of MarkerBatchPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_marker_batch_primitive_metadata)

    _get_marker_batch_primitive_optional_parameters_metadata = { "offset" : _get_marker_batch_primitive_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def marker_batch_primitive_optional_parameters(self) -> "MarkerBatchPrimitiveOptionalParametersFactory":
        """Access global methods and properties of MarkerBatchPrimitiveOptionalParameters (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_marker_batch_primitive_optional_parameters_metadata)

    _get_maximum_count_path_primitive_update_policy_metadata = { "offset" : _get_maximum_count_path_primitive_update_policy_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maximum_count_path_primitive_update_policy(self) -> "MaximumCountPathPrimitiveUpdatePolicyFactory":
        """Access global methods and properties of MaximumCountPathPrimitiveUpdatePolicy (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_maximum_count_path_primitive_update_policy_metadata)

    _get_model_primitive_metadata = { "offset" : _get_model_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def model_primitive(self) -> "ModelPrimitiveFactory":
        """Access global methods and properties of ModelPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_model_primitive_metadata)

    _get_path_primitive_metadata = { "offset" : _get_path_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def path_primitive(self) -> "PathPrimitiveFactory":
        """Access global methods and properties of PathPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_path_primitive_metadata)

    _get_pixel_size_display_condition_metadata = { "offset" : _get_pixel_size_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def pixel_size_display_condition(self) -> "PixelSizeDisplayConditionFactory":
        """Access global methods and properties of PixelSizeDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_pixel_size_display_condition_metadata)

    _get_point_batch_primitive_metadata = { "offset" : _get_point_batch_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point_batch_primitive(self) -> "PointBatchPrimitiveFactory":
        """Access global methods and properties of PointBatchPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_point_batch_primitive_metadata)

    _get_polyline_primitive_metadata = { "offset" : _get_polyline_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def polyline_primitive(self) -> "PolylinePrimitiveFactory":
        """Access global methods and properties of PolylinePrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_polyline_primitive_metadata)

    _get_raster_image_globe_overlay_metadata = { "offset" : _get_raster_image_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster_image_globe_overlay(self) -> "RasterImageGlobeOverlayFactory":
        """Access global methods and properties of RasterImageGlobeOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_raster_image_globe_overlay_metadata)

    _get_rhumb_line_interpolator_metadata = { "offset" : _get_rhumb_line_interpolator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def rhumb_line_interpolator(self) -> "RhumbLineInterpolatorFactory":
        """Access global methods and properties of RhumbLineInterpolator (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_rhumb_line_interpolator_metadata)

    _get_scene_display_condition_metadata = { "offset" : _get_scene_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scene_display_condition(self) -> "SceneDisplayConditionFactory":
        """Access global methods and properties of SceneDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_scene_display_condition_metadata)

    _get_scene_manager_metadata = { "offset" : _get_scene_manager_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scene_manager(self) -> "SceneManagerInitializer":
        """Access global methods and properties of SceneManager (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_scene_manager_metadata)

    _get_screen_overlay_metadata = { "offset" : _get_screen_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def screen_overlay(self) -> "ScreenOverlayFactory":
        """Access global methods and properties of ScreenOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_screen_overlay_metadata)

    _get_solid_primitive_metadata = { "offset" : _get_solid_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def solid_primitive(self) -> "SolidPrimitiveFactory":
        """Access global methods and properties of SolidPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_solid_primitive_metadata)

    _get_surface_mesh_primitive_metadata = { "offset" : _get_surface_mesh_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def surface_mesh_primitive(self) -> "SurfaceMeshPrimitiveFactory":
        """Access global methods and properties of SurfaceMeshPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_surface_mesh_primitive_metadata)

    _get_terrain_overlay_metadata = { "offset" : _get_terrain_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def terrain_overlay(self) -> "TerrainOverlayInitializer":
        """Access global methods and properties of TerrainOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_terrain_overlay_metadata)

    _get_text_batch_primitive_metadata = { "offset" : _get_text_batch_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def text_batch_primitive(self) -> "TextBatchPrimitiveFactory":
        """Access global methods and properties of TextBatchPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_text_batch_primitive_metadata)

    _get_text_batch_primitive_optional_parameters_metadata = { "offset" : _get_text_batch_primitive_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def text_batch_primitive_optional_parameters(self) -> "TextBatchPrimitiveOptionalParametersFactory":
        """Access global methods and properties of TextBatchPrimitiveOptionalParameters (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_text_batch_primitive_optional_parameters_metadata)

    _get_texture_matrix_metadata = { "offset" : _get_texture_matrix_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_matrix(self) -> "TextureMatrixFactory":
        """Access global methods and properties of TextureMatrix (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_texture_matrix_metadata)

    _get_texture_screen_overlay_metadata = { "offset" : _get_texture_screen_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_screen_overlay(self) -> "TextureScreenOverlayFactory":
        """Access global methods and properties of TextureScreenOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_texture_screen_overlay_metadata)

    _get_time_interval_display_condition_metadata = { "offset" : _get_time_interval_display_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def time_interval_display_condition(self) -> "TimeIntervalDisplayConditionFactory":
        """Access global methods and properties of TimeIntervalDisplayCondition (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_time_interval_display_condition_metadata)

    _get_triangle_mesh_primitive_metadata = { "offset" : _get_triangle_mesh_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def triangle_mesh_primitive(self) -> "TriangleMeshPrimitiveFactory":
        """Access global methods and properties of TriangleMeshPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_triangle_mesh_primitive_metadata)

    _get_triangle_mesh_primitive_optional_parameters_metadata = { "offset" : _get_triangle_mesh_primitive_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def triangle_mesh_primitive_optional_parameters(self) -> "TriangleMeshPrimitiveOptionalParametersFactory":
        """Access global methods and properties of TriangleMeshPrimitiveOptionalParameters (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_triangle_mesh_primitive_optional_parameters_metadata)

    _get_texture_filter_2d_metadata = { "offset" : _get_texture_filter_2d_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_filter_2d(self) -> "TextureFilter2DFactory":
        """Factory creates texture filters."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_texture_filter_2d_metadata)

    _get_bounding_sphere_metadata = { "offset" : _get_bounding_sphere_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def bounding_sphere(self) -> "BoundingSphereFactory":
        """Factory creates bounding spheres."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_bounding_sphere_metadata)

    _get_path_point_metadata = { "offset" : _get_path_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def path_point(self) -> "PathPointFactory":
        """Factory creates path points."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_path_point_metadata)

    _get_text_overlay_metadata = { "offset" : _get_text_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def text_overlay(self) -> "TextOverlayFactory":
        """Access global methods and properties of TextOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_text_overlay_metadata)

    _get_agi_custom_terrain_overlay_metadata = { "offset" : _get_agi_custom_terrain_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def agi_custom_terrain_overlay(self) -> "AGICustomTerrainOverlayFactory":
        """Access global methods and properties of AGICustomTerrainOverlay (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_agi_custom_terrain_overlay_metadata)

    _get_axes_primitive_metadata = { "offset" : _get_axes_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def axes_primitive(self) -> "AxesPrimitiveFactory":
        """Access global methods and properties of AxesPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_axes_primitive_metadata)

    _get_vector_primitive_metadata = { "offset" : _get_vector_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vector_primitive(self) -> "VectorPrimitiveFactory":
        """Access global methods and properties of VectorPrimitive (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_vector_primitive_metadata)

    _get_polyline_primitive_optional_parameters_metadata = { "offset" : _get_polyline_primitive_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def polyline_primitive_optional_parameters(self) -> "PolylinePrimitiveOptionalParametersFactory":
        """Access global methods and properties of PolylinePrimitiveOptionalParameters (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_polyline_primitive_optional_parameters_metadata)

    _get_point_batch_primitive_optional_parameters_metadata = { "offset" : _get_point_batch_primitive_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point_batch_primitive_optional_parameters(self) -> "PointBatchPrimitiveOptionalParametersFactory":
        """Access global methods and properties of PointBatchPrimitiveOptionalParameters (what's known as static properties, static methods and constructors in languages such as C++, C#, etc.)."""
        return self._intf.get_property(FactoryAndInitializers._metadata, FactoryAndInitializers._get_point_batch_primitive_optional_parameters_metadata)

    _property_names[box_triangulator] = "box_triangulator"
    _property_names[cylinder_triangulator] = "cylinder_triangulator"
    _property_names[ellipsoid_triangulator] = "ellipsoid_triangulator"
    _property_names[extruded_polyline_triangulator] = "extruded_polyline_triangulator"
    _property_names[surface_extent_triangulator] = "surface_extent_triangulator"
    _property_names[surface_polygon_triangulator] = "surface_polygon_triangulator"
    _property_names[surface_shapes] = "surface_shapes"
    _property_names[agi_processed_image_globe_overlay] = "agi_processed_image_globe_overlay"
    _property_names[agi_processed_terrain_overlay] = "agi_processed_terrain_overlay"
    _property_names[agi_roam_image_globe_overlay] = "agi_roam_image_globe_overlay"
    _property_names[custom_image_globe_overlay_plugin_activator] = "custom_image_globe_overlay_plugin_activator"
    _property_names[geospatial_image_globe_overlay] = "geospatial_image_globe_overlay"
    _property_names[projected_raster_overlay] = "projected_raster_overlay"
    _property_names[projection] = "projection"
    _property_names[altitude_display_condition] = "altitude_display_condition"
    _property_names[composite_display_condition] = "composite_display_condition"
    _property_names[composite_primitive] = "composite_primitive"
    _property_names[constant_display_condition] = "constant_display_condition"
    _property_names[distance_display_condition] = "distance_display_condition"
    _property_names[distance_to_globe_overlay_display_condition] = "distance_to_globe_overlay_display_condition"
    _property_names[distance_to_position_display_condition] = "distance_to_position_display_condition"
    _property_names[distance_to_primitive_display_condition] = "distance_to_primitive_display_condition"
    _property_names[duration_path_primitive_update_policy] = "duration_path_primitive_update_policy"
    _property_names[globe_image_overlay] = "globe_image_overlay"
    _property_names[graphics_font] = "graphics_font"
    _property_names[great_arc_interpolator] = "great_arc_interpolator"
    _property_names[alpha_from_luminance_filter] = "alpha_from_luminance_filter"
    _property_names[alpha_from_pixel_filter] = "alpha_from_pixel_filter"
    _property_names[alpha_from_raster_filter] = "alpha_from_raster_filter"
    _property_names[band_extract_filter] = "band_extract_filter"
    _property_names[band_order_filter] = "band_order_filter"
    _property_names[blur_filter] = "blur_filter"
    _property_names[brightness_filter] = "brightness_filter"
    _property_names[color_to_luminance_filter] = "color_to_luminance_filter"
    _property_names[contrast_filter] = "contrast_filter"
    _property_names[convolution_filter] = "convolution_filter"
    _property_names[edge_detect_filter] = "edge_detect_filter"
    _property_names[filtering_raster_stream] = "filtering_raster_stream"
    _property_names[flip_filter] = "flip_filter"
    _property_names[gamma_correction_filter] = "gamma_correction_filter"
    _property_names[gaussian_blur_filter] = "gaussian_blur_filter"
    _property_names[gradient_detect_filter] = "gradient_detect_filter"
    _property_names[jpeg2000_writer] = "jpeg2000_writer"
    _property_names[levels_filter] = "levels_filter"
    _property_names[projection_raster_stream_plugin_activator] = "projection_raster_stream_plugin_activator"
    _property_names[raster] = "raster"
    _property_names[raster_attributes] = "raster_attributes"
    _property_names[rotate_filter] = "rotate_filter"
    _property_names[sequence_filter] = "sequence_filter"
    _property_names[sharpen_filter] = "sharpen_filter"
    _property_names[video_stream] = "video_stream"
    _property_names[marker_batch_primitive] = "marker_batch_primitive"
    _property_names[marker_batch_primitive_optional_parameters] = "marker_batch_primitive_optional_parameters"
    _property_names[maximum_count_path_primitive_update_policy] = "maximum_count_path_primitive_update_policy"
    _property_names[model_primitive] = "model_primitive"
    _property_names[path_primitive] = "path_primitive"
    _property_names[pixel_size_display_condition] = "pixel_size_display_condition"
    _property_names[point_batch_primitive] = "point_batch_primitive"
    _property_names[polyline_primitive] = "polyline_primitive"
    _property_names[raster_image_globe_overlay] = "raster_image_globe_overlay"
    _property_names[rhumb_line_interpolator] = "rhumb_line_interpolator"
    _property_names[scene_display_condition] = "scene_display_condition"
    _property_names[scene_manager] = "scene_manager"
    _property_names[screen_overlay] = "screen_overlay"
    _property_names[solid_primitive] = "solid_primitive"
    _property_names[surface_mesh_primitive] = "surface_mesh_primitive"
    _property_names[terrain_overlay] = "terrain_overlay"
    _property_names[text_batch_primitive] = "text_batch_primitive"
    _property_names[text_batch_primitive_optional_parameters] = "text_batch_primitive_optional_parameters"
    _property_names[texture_matrix] = "texture_matrix"
    _property_names[texture_screen_overlay] = "texture_screen_overlay"
    _property_names[time_interval_display_condition] = "time_interval_display_condition"
    _property_names[triangle_mesh_primitive] = "triangle_mesh_primitive"
    _property_names[triangle_mesh_primitive_optional_parameters] = "triangle_mesh_primitive_optional_parameters"
    _property_names[texture_filter_2d] = "texture_filter_2d"
    _property_names[bounding_sphere] = "bounding_sphere"
    _property_names[path_point] = "path_point"
    _property_names[text_overlay] = "text_overlay"
    _property_names[agi_custom_terrain_overlay] = "agi_custom_terrain_overlay"
    _property_names[axes_primitive] = "axes_primitive"
    _property_names[vector_primitive] = "vector_primitive"
    _property_names[polyline_primitive_optional_parameters] = "polyline_primitive_optional_parameters"
    _property_names[point_batch_primitive_optional_parameters] = "point_batch_primitive_optional_parameters"

    def __init__(self, source_object=None):
        """Construct an object of type FactoryAndInitializers."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, FactoryAndInitializers)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, FactoryAndInitializers, [FactoryAndInitializers, ])

agcls.AgClassCatalog.add_catalog_entry((5029710387976957897, 13954049756575440778), FactoryAndInitializers)
agcls.AgTypeNameMap["FactoryAndInitializers"] = FactoryAndInitializers

class ExtrudedPolylineTriangulatorResult(ITriangulatorResult, SupportsDeleteCallback):
    """The result from extruded polyline triangulation: a triangle mesh defined using an indexed triangle list with top and bottom boundary positions. The mesh is commonly visualized with the triangle mesh primitive or surface mesh primitive..."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_top_boundary_positions_method_offset = 1
    _get_bottom_boundary_positions_method_offset = 2
    _get_boundary_positions_winding_order_method_offset = 3
    _metadata = {
        "iid_data" : (5585319328923046007, 9526893615741571244),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ExtrudedPolylineTriangulatorResult)

    _get_top_boundary_positions_metadata = { "offset" : _get_top_boundary_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def top_boundary_positions(self) -> list:
        """Get the boundary positions along the top of the extrusion. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(ExtrudedPolylineTriangulatorResult._metadata, ExtrudedPolylineTriangulatorResult._get_top_boundary_positions_metadata)

    _get_bottom_boundary_positions_metadata = { "offset" : _get_bottom_boundary_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def bottom_boundary_positions(self) -> list:
        """Get the boundary positions along the bottom of the extrusion. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(ExtrudedPolylineTriangulatorResult._metadata, ExtrudedPolylineTriangulatorResult._get_bottom_boundary_positions_metadata)

    _get_boundary_positions_winding_order_metadata = { "offset" : _get_boundary_positions_winding_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(WindingOrder),) }
    @property
    def boundary_positions_winding_order(self) -> "WindingOrder":
        """Get the winding order of top boundary positions and bottom boundary positions."""
        return self._intf.get_property(ExtrudedPolylineTriangulatorResult._metadata, ExtrudedPolylineTriangulatorResult._get_boundary_positions_winding_order_metadata)

    _property_names[top_boundary_positions] = "top_boundary_positions"
    _property_names[bottom_boundary_positions] = "bottom_boundary_positions"
    _property_names[boundary_positions_winding_order] = "boundary_positions_winding_order"

    def __init__(self, source_object=None):
        """Construct an object of type ExtrudedPolylineTriangulatorResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ExtrudedPolylineTriangulatorResult)
        ITriangulatorResult.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITriangulatorResult._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ExtrudedPolylineTriangulatorResult, [ExtrudedPolylineTriangulatorResult, ITriangulatorResult])

agcls.AgClassCatalog.add_catalog_entry((4874148548191491803, 705205959735039921), ExtrudedPolylineTriangulatorResult)
agcls.AgTypeNameMap["ExtrudedPolylineTriangulatorResult"] = ExtrudedPolylineTriangulatorResult

class SolidTriangulatorResult(ITriangulatorResult, SupportsDeleteCallback):
    """The result from a triangulation of a solid: a triangle mesh defined using an indexed triangle list and positions outlining the solid. It is recommended to visualize the solid using a solid primitive..."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_outline_indices_method_offset = 1
    _get_outline_positions_method_offset = 2
    _get_outline_polyline_type_method_offset = 3
    _get_closed_method_offset = 4
    _metadata = {
        "iid_data" : (5604440859576239731, 8920246755834744996),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SolidTriangulatorResult)

    _get_outline_indices_metadata = { "offset" : _get_outline_indices_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def outline_indices(self) -> list:
        """Get indices into positions that define the positions outlining the solid. The indices returned consider the three components of a position (x, y, and z) as a single array element..."""
        return self._intf.get_property(SolidTriangulatorResult._metadata, SolidTriangulatorResult._get_outline_indices_metadata)

    _get_outline_positions_metadata = { "offset" : _get_outline_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def outline_positions(self) -> list:
        """Get the positions outlining the solid. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(SolidTriangulatorResult._metadata, SolidTriangulatorResult._get_outline_positions_metadata)

    _get_outline_polyline_type_metadata = { "offset" : _get_outline_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def outline_polyline_type(self) -> "PolylineType":
        """Get the polyline type of outline indices and outline positions."""
        return self._intf.get_property(SolidTriangulatorResult._metadata, SolidTriangulatorResult._get_outline_polyline_type_metadata)

    _get_closed_metadata = { "offset" : _get_closed_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def closed(self) -> bool:
        """Get whether the solid is closed. For example, a box with six faces is closed. If one face is removed, the box is open."""
        return self._intf.get_property(SolidTriangulatorResult._metadata, SolidTriangulatorResult._get_closed_metadata)

    _property_names[outline_indices] = "outline_indices"
    _property_names[outline_positions] = "outline_positions"
    _property_names[outline_polyline_type] = "outline_polyline_type"
    _property_names[closed] = "closed"

    def __init__(self, source_object=None):
        """Construct an object of type SolidTriangulatorResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SolidTriangulatorResult)
        ITriangulatorResult.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITriangulatorResult._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SolidTriangulatorResult, [SolidTriangulatorResult, ITriangulatorResult])

agcls.AgClassCatalog.add_catalog_entry((5292312989871528695, 2932714005618817949), SolidTriangulatorResult)
agcls.AgTypeNameMap["SolidTriangulatorResult"] = SolidTriangulatorResult

class SurfaceShapesResult(SupportsDeleteCallback):
    """Represents the boundary positions of a shape on the surface computed from by a surface shapes method."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_positions_method_offset = 1
    _get_positions_winding_order_method_offset = 2
    _get_polyline_type_method_offset = 3
    _metadata = {
        "iid_data" : (5629457714368026075, 2389901432840870802),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SurfaceShapesResult)

    _get_positions_metadata = { "offset" : _get_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def positions(self) -> list:
        """Get the positions of the computed shape. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(SurfaceShapesResult._metadata, SurfaceShapesResult._get_positions_metadata)

    _get_positions_winding_order_metadata = { "offset" : _get_positions_winding_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(WindingOrder),) }
    @property
    def positions_winding_order(self) -> "WindingOrder":
        """Get the winding order of positions."""
        return self._intf.get_property(SurfaceShapesResult._metadata, SurfaceShapesResult._get_positions_winding_order_metadata)

    _get_polyline_type_metadata = { "offset" : _get_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def polyline_type(self) -> "PolylineType":
        """Get the polyline type of positions."""
        return self._intf.get_property(SurfaceShapesResult._metadata, SurfaceShapesResult._get_polyline_type_metadata)

    _property_names[positions] = "positions"
    _property_names[positions_winding_order] = "positions_winding_order"
    _property_names[polyline_type] = "polyline_type"

    def __init__(self, source_object=None):
        """Construct an object of type SurfaceShapesResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SurfaceShapesResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SurfaceShapesResult, [SurfaceShapesResult, ])

agcls.AgClassCatalog.add_catalog_entry((5562832838748717934, 17103991771881976193), SurfaceShapesResult)
agcls.AgTypeNameMap["SurfaceShapesResult"] = SurfaceShapesResult

class SurfaceTriangulatorResult(ITriangulatorResult, SupportsDeleteCallback):
    """The result from a triangulation on the surface of a central body: a triangle mesh defined using an indexed triangle list and boundary positions surrounding the mesh..."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_granularity_method_offset = 1
    _get_boundary_indices_method_offset = 2
    _get_boundary_positions_method_offset = 3
    _get_boundary_positions_winding_order_method_offset = 4
    _get_boundary_polyline_type_method_offset = 5
    _metadata = {
        "iid_data" : (5555338356055792958, 12277620616557637510),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SurfaceTriangulatorResult)

    _get_granularity_metadata = { "offset" : _get_granularity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def granularity(self) -> float:
        """Get the granularity used when the triangulation was computed. Lower granularities are more precise but create more triangles."""
        return self._intf.get_property(SurfaceTriangulatorResult._metadata, SurfaceTriangulatorResult._get_granularity_metadata)

    _get_boundary_indices_metadata = { "offset" : _get_boundary_indices_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def boundary_indices(self) -> list:
        """Get indices into positions that define the boundary positions that surround the mesh. The indices returned consider the three components of a position (x, y, and z) as a single array element..."""
        return self._intf.get_property(SurfaceTriangulatorResult._metadata, SurfaceTriangulatorResult._get_boundary_indices_metadata)

    _get_boundary_positions_metadata = { "offset" : _get_boundary_positions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def boundary_positions(self) -> list:
        """Get the boundary positions that surround the mesh. Three array elements (in the order x, y, z) constitute one position."""
        return self._intf.get_property(SurfaceTriangulatorResult._metadata, SurfaceTriangulatorResult._get_boundary_positions_metadata)

    _get_boundary_positions_winding_order_metadata = { "offset" : _get_boundary_positions_winding_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(WindingOrder),) }
    @property
    def boundary_positions_winding_order(self) -> "WindingOrder":
        """Get the winding order of boundary positions."""
        return self._intf.get_property(SurfaceTriangulatorResult._metadata, SurfaceTriangulatorResult._get_boundary_positions_winding_order_metadata)

    _get_boundary_polyline_type_metadata = { "offset" : _get_boundary_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def boundary_polyline_type(self) -> "PolylineType":
        """Get the polyline type of boundary positions."""
        return self._intf.get_property(SurfaceTriangulatorResult._metadata, SurfaceTriangulatorResult._get_boundary_polyline_type_metadata)

    _property_names[granularity] = "granularity"
    _property_names[boundary_indices] = "boundary_indices"
    _property_names[boundary_positions] = "boundary_positions"
    _property_names[boundary_positions_winding_order] = "boundary_positions_winding_order"
    _property_names[boundary_polyline_type] = "boundary_polyline_type"

    def __init__(self, source_object=None):
        """Construct an object of type SurfaceTriangulatorResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SurfaceTriangulatorResult)
        ITriangulatorResult.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITriangulatorResult._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SurfaceTriangulatorResult, [SurfaceTriangulatorResult, ITriangulatorResult])

agcls.AgClassCatalog.add_catalog_entry((5317895839875531743, 55317102968090019), SurfaceTriangulatorResult)
agcls.AgTypeNameMap["SurfaceTriangulatorResult"] = SurfaceTriangulatorResult

class TriangulatorResult(ITriangulatorResult, SupportsDeleteCallback):
    """The result from triangulation: a triangle mesh defined using an indexed triangle list. This is commonly visualized with the triangle mesh primitive or surface mesh primitive."""
    def __init__(self, source_object=None):
        """Construct an object of type TriangulatorResult."""
        SupportsDeleteCallback.__init__(self)
        ITriangulatorResult.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITriangulatorResult._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TriangulatorResult, [ITriangulatorResult])

agcls.AgClassCatalog.add_catalog_entry((5139444277268572249, 3108066989369069979), TriangulatorResult)
agcls.AgTypeNameMap["TriangulatorResult"] = TriangulatorResult

class AGICustomTerrainOverlay(ITerrainOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A terrain overlay for handling AGI Cesium Terrain."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5647706262866986038, 13474225891505749390),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AGICustomTerrainOverlay)


    def __init__(self, source_object=None):
        """Construct an object of type AGICustomTerrainOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AGICustomTerrainOverlay)
        ITerrainOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITerrainOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AGICustomTerrainOverlay, [AGICustomTerrainOverlay, ITerrainOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5233896634715199280, 3319534929678221749), AGICustomTerrainOverlay)
agcls.AgTypeNameMap["AGICustomTerrainOverlay"] = AGICustomTerrainOverlay

class AGIProcessedImageGlobeOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay for handling AGI Processed Image (PDTTX) files."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5415390494021483722, 18356321176920839827),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AGIProcessedImageGlobeOverlay)


    def __init__(self, source_object=None):
        """Construct an object of type AGIProcessedImageGlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AGIProcessedImageGlobeOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AGIProcessedImageGlobeOverlay, [AGIProcessedImageGlobeOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((4698652200510509190, 4410810686274608779), AGIProcessedImageGlobeOverlay)
agcls.AgTypeNameMap["AGIProcessedImageGlobeOverlay"] = AGIProcessedImageGlobeOverlay

class AGIProcessedTerrainOverlay(ITerrainOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A terrain overlay for handling AGI Processed Terrain (PDTT) files."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4623415458303466258, 7863817902119241133),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AGIProcessedTerrainOverlay)


    def __init__(self, source_object=None):
        """Construct an object of type AGIProcessedTerrainOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AGIProcessedTerrainOverlay)
        ITerrainOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITerrainOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AGIProcessedTerrainOverlay, [AGIProcessedTerrainOverlay, ITerrainOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5555210745660264736, 9436563425815460013), AGIProcessedTerrainOverlay)
agcls.AgTypeNameMap["AGIProcessedTerrainOverlay"] = AGIProcessedTerrainOverlay

class AGIRoamImageGlobeOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay for handling ROAM (TXM/TXB) files."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4984774588206252727, 3347210740923457974),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AGIRoamImageGlobeOverlay)


    def __init__(self, source_object=None):
        """Construct an object of type AGIRoamImageGlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AGIRoamImageGlobeOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AGIRoamImageGlobeOverlay, [AGIRoamImageGlobeOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5242539083339407783, 14850989973976265913), AGIRoamImageGlobeOverlay)
agcls.AgTypeNameMap["AGIRoamImageGlobeOverlay"] = AGIRoamImageGlobeOverlay

class CameraSnapshot(SupportsDeleteCallback):
    """Takes snapshots of the 3D window."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _save_to_file_method_offset = 1
    _save_to_file_with_width_and_dpi_method_offset = 2
    _save_to_clipboard_method_offset = 3
    _save_to_raster_method_offset = 4
    _save_to_texture_method_offset = 5
    _metadata = {
        "iid_data" : (5310587618602945936, 15099585874018153631),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CameraSnapshot)

    _save_to_file_metadata = { "offset" : _save_to_file_method_offset,
            "arg_types" : (agcom.BSTR, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.EnumArg(SnapshotFileFormat),) }
    def save_to_file(self, filename:str, camera_snapshot_format:"SnapshotFileFormat") -> None:
        """Save a snapshot of the 3D window to the filename with the specified format."""
        return self._intf.invoke(CameraSnapshot._metadata, CameraSnapshot._save_to_file_metadata, filename, camera_snapshot_format)

    _save_to_file_with_width_and_dpi_metadata = { "offset" : _save_to_file_with_width_and_dpi_method_offset,
            "arg_types" : (agcom.BSTR, agcom.LONG, agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.EnumArg(SnapshotFileFormat), agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def save_to_file_with_width_and_dpi(self, filename:str, camera_snapshot_format:"SnapshotFileFormat", width_in_inches:float, dots_per_inch:float) -> None:
        """Save a snapshot of the 3D window to the filename with the specified format at high resolution..."""
        return self._intf.invoke(CameraSnapshot._metadata, CameraSnapshot._save_to_file_with_width_and_dpi_metadata, filename, camera_snapshot_format, width_in_inches, dots_per_inch)

    _save_to_clipboard_metadata = { "offset" : _save_to_clipboard_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def save_to_clipboard(self) -> None:
        """Save a single frame of the 3D window to the clipboard."""
        return self._intf.invoke(CameraSnapshot._metadata, CameraSnapshot._save_to_clipboard_metadata, )

    _save_to_raster_metadata = { "offset" : _save_to_raster_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def save_to_raster(self) -> "IRaster":
        """Save a snapshot of the 3D window to a raster."""
        return self._intf.invoke(CameraSnapshot._metadata, CameraSnapshot._save_to_raster_metadata, OutArg())

    _save_to_texture_metadata = { "offset" : _save_to_texture_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def save_to_texture(self) -> "RendererTexture2D":
        """Save a snapshot of the 3D window to a texture 2d."""
        return self._intf.invoke(CameraSnapshot._metadata, CameraSnapshot._save_to_texture_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CameraSnapshot."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CameraSnapshot)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CameraSnapshot, [CameraSnapshot, ])

agcls.AgClassCatalog.add_catalog_entry((4857812757827340764, 12820205993576870016), CameraSnapshot)
agcls.AgTypeNameMap["CameraSnapshot"] = CameraSnapshot

class CameraVideoRecording(SupportsDeleteCallback):
    """Records the 3D window to either a movie file or to consecutively ordered image files each time the scene is rendered."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_recording_method_offset = 1
    _start_recording_method_offset = 2
    _start_recording_frame_stack_method_offset = 3
    _stop_recording_method_offset = 4
    _start_recording_video_method_offset = 5
    _metadata = {
        "iid_data" : (4979543382962073210, 764756984121433475),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CameraVideoRecording)

    _get_is_recording_metadata = { "offset" : _get_is_recording_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_recording(self) -> bool:
        """Get if recording is occurring or not."""
        return self._intf.get_property(CameraVideoRecording._metadata, CameraVideoRecording._get_is_recording_metadata)

    _start_recording_metadata = { "offset" : _start_recording_method_offset,
            "arg_types" : (agcom.BSTR, agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.IntArg, agmarshall.IntArg,) }
    def start_recording(self, wmv_filename:str, video_bit_rate:int, video_frame_rate:int) -> None:
        """Do not use this method, as it is deprecated. Use the overload taking a video format instead. Starts recording a file in the WMV format at the specified bit and frame rate."""
        return self._intf.invoke(CameraVideoRecording._metadata, CameraVideoRecording._start_recording_metadata, wmv_filename, video_bit_rate, video_frame_rate)

    _start_recording_frame_stack_metadata = { "offset" : _start_recording_frame_stack_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(SnapshotFileFormat), agmarshall.IntArg, agmarshall.IntArg,) }
    def start_recording_frame_stack(self, file_directory:str, file_prefix:str, camera_snapshot_file_format:"SnapshotFileFormat", starting_frame_number:int, number_of_frame_digits:int) -> None:
        """Start recording a frame stack. Each frame is saved as a separate image file. The filename of each frame is defined by a prefix followed by a frame number."""
        return self._intf.invoke(CameraVideoRecording._metadata, CameraVideoRecording._start_recording_frame_stack_metadata, file_directory, file_prefix, camera_snapshot_file_format, starting_frame_number, number_of_frame_digits)

    _stop_recording_metadata = { "offset" : _stop_recording_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def stop_recording(self) -> None:
        """Stop recording."""
        return self._intf.invoke(CameraVideoRecording._metadata, CameraVideoRecording._stop_recording_metadata, )

    _start_recording_video_metadata = { "offset" : _start_recording_video_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(VideoFormat), agmarshall.IntArg, agmarshall.IntArg,) }
    def start_recording_video(self, file_directory:str, file_prefix:str, video_format:"VideoFormat", video_bit_rate:int, video_frame_rate:int) -> None:
        """Start recording a video file at the specified bit and frame rate."""
        return self._intf.invoke(CameraVideoRecording._metadata, CameraVideoRecording._start_recording_video_metadata, file_directory, file_prefix, video_format, video_bit_rate, video_frame_rate)

    _property_names[is_recording] = "is_recording"

    def __init__(self, source_object=None):
        """Construct an object of type CameraVideoRecording."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CameraVideoRecording)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CameraVideoRecording, [CameraVideoRecording, ])

agcls.AgClassCatalog.add_catalog_entry((4949773353057298900, 8240883984009465791), CameraVideoRecording)
agcls.AgTypeNameMap["CameraVideoRecording"] = CameraVideoRecording

class CentralBodyGraphicsIndexer(SupportsDeleteCallback):
    """An indexer into the central body graphics for a particular central body, which provides graphical properties such as showing or hiding the central body in the scene, and working with terrain and imagery for the specified central body."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_earth_method_offset = 1
    _get_moon_method_offset = 2
    _get_sun_method_offset = 3
    _item_method_offset = 4
    _get_by_name_method_offset = 5
    _metadata = {
        "iid_data" : (5339898603870531521, 4215796469053669526),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CentralBodyGraphicsIndexer)

    _get_earth_metadata = { "offset" : _get_earth_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def earth(self) -> "CentralBodyGraphics":
        """Get the central body graphics for the planet Earth. This is equivalent to passing a central body equal to an instance of earth central body to the indexer."""
        return self._intf.get_property(CentralBodyGraphicsIndexer._metadata, CentralBodyGraphicsIndexer._get_earth_metadata)

    _get_moon_metadata = { "offset" : _get_moon_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def moon(self) -> "CentralBodyGraphics":
        """Get the central body graphics for the Moon."""
        return self._intf.get_property(CentralBodyGraphicsIndexer._metadata, CentralBodyGraphicsIndexer._get_moon_metadata)

    _get_sun_metadata = { "offset" : _get_sun_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sun(self) -> "CentralBodyGraphics":
        """Get the central body graphics for the Sun."""
        return self._intf.get_property(CentralBodyGraphicsIndexer._metadata, CentralBodyGraphicsIndexer._get_sun_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def item(self, central_body:str) -> "CentralBodyGraphics":
        """Get the central body graphics for the specified central body."""
        return self._intf.invoke(CentralBodyGraphicsIndexer._metadata, CentralBodyGraphicsIndexer._item_metadata, central_body, OutArg())

    _get_by_name_metadata = { "offset" : _get_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_by_name(self, name:str) -> "CentralBodyGraphics":
        """Return the central body graphics for the central body with the given name."""
        return self._intf.invoke(CentralBodyGraphicsIndexer._metadata, CentralBodyGraphicsIndexer._get_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[earth] = "earth"
    _property_names[moon] = "moon"
    _property_names[sun] = "sun"

    def __init__(self, source_object=None):
        """Construct an object of type CentralBodyGraphicsIndexer."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CentralBodyGraphicsIndexer)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CentralBodyGraphicsIndexer, [CentralBodyGraphicsIndexer, ])

agcls.AgClassCatalog.add_catalog_entry((4664517785508533741, 18264689342556895135), CentralBodyGraphicsIndexer)
agcls.AgTypeNameMap["CentralBodyGraphicsIndexer"] = CentralBodyGraphicsIndexer

class CustomImageGlobeOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay that allows for a user defined image to be specified."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_translucent_method_offset = 1
    _get_maximum_meters_per_pixel_method_offset = 2
    _get_projection_method_offset = 3
    _start_up_method_offset = 4
    _shut_down_method_offset = 5
    _clear_cache_method_offset = 6
    _reload_method_offset = 7
    _read_method_offset = 8
    _metadata = {
        "iid_data" : (5579198083994513106, 9725554734426398627),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CustomImageGlobeOverlay)

    _get_is_translucent_metadata = { "offset" : _get_is_translucent_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_translucent(self) -> bool:
        """Get whether the overlay contains translucent imagery."""
        return self._intf.get_property(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._get_is_translucent_metadata)

    _get_maximum_meters_per_pixel_metadata = { "offset" : _get_maximum_meters_per_pixel_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_meters_per_pixel(self) -> float:
        """Get the maximum resolution of the inlay in meters per pixel."""
        return self._intf.get_property(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._get_maximum_meters_per_pixel_metadata)

    _get_projection_metadata = { "offset" : _get_projection_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MapProjection),) }
    @property
    def projection(self) -> "MapProjection":
        """Get the map projection. Valid values are mercator and equidistant cylindrical."""
        return self._intf.get_property(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._get_projection_metadata)

    _start_up_metadata = { "offset" : _start_up_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Scene"),) }
    def start_up(self, scene:"Scene") -> None:
        """Initiate start-up when imagery is being added to the globe."""
        return self._intf.invoke(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._start_up_metadata, scene)

    _shut_down_metadata = { "offset" : _shut_down_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Scene"),) }
    def shut_down(self, scene:"Scene") -> None:
        """Initiate shutdown when imagery is being removed from the globe."""
        return self._intf.invoke(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._shut_down_metadata, scene)

    _clear_cache_metadata = { "offset" : _clear_cache_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear_cache(self) -> None:
        """Clear the image data cache associated with this instance. This is equivalent to deleting and re-adding the overlay."""
        return self._intf.invoke(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._clear_cache_metadata, )

    _reload_metadata = { "offset" : _reload_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reload(self) -> None:
        """Reload the image data associated with this instance. Preserves the current image data until new image data replaces it."""
        return self._intf.invoke(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._reload_metadata, )

    _read_metadata = { "offset" : _read_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.Variant, POINTER(agcom.PVOID), POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.VariantArg, agmarshall.IPictureDispArg, agmarshall.VariantBoolArg,) }
    def read(self, extent:list, user_tile_data:typing.Any, image:IPictureDisp) -> bool:
        """Read a tile from the specified extent, scales it to and stores the result in image."""
        return self._intf.invoke(CustomImageGlobeOverlay._metadata, CustomImageGlobeOverlay._read_metadata, extent, user_tile_data, image, OutArg())

    _property_names[is_translucent] = "is_translucent"
    _property_names[maximum_meters_per_pixel] = "maximum_meters_per_pixel"
    _property_names[projection] = "projection"

    def __init__(self, source_object=None):
        """Construct an object of type CustomImageGlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CustomImageGlobeOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CustomImageGlobeOverlay, [CustomImageGlobeOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5121324544174012502, 11507939295407134083), CustomImageGlobeOverlay)
agcls.AgTypeNameMap["CustomImageGlobeOverlay"] = CustomImageGlobeOverlay

class CustomImageGlobeOverlayPluginActivator(SupportsDeleteCallback):
    """The Activator class provides methods to load COM plugins that implement custom image globe overlays. For more information about custom image globe overlays, see the STK Programming Interface."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_from_display_name_method_offset = 1
    _get_available_display_names_method_offset = 2
    _metadata = {
        "iid_data" : (4630720187928517408, 4059776964853808547),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CustomImageGlobeOverlayPluginActivator)

    _create_from_display_name_metadata = { "offset" : _create_from_display_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_display_name(self, display_name:str) -> "CustomImageGlobeOverlayPluginProxy":
        """Load a custom image globe overlay COM plugin associated with the specified display name and returns a proxy object that allows accessing the custom image globe overlays implemented by the plugin."""
        return self._intf.invoke(CustomImageGlobeOverlayPluginActivator._metadata, CustomImageGlobeOverlayPluginActivator._create_from_display_name_metadata, display_name, OutArg())

    _get_available_display_names_metadata = { "offset" : _get_available_display_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_available_display_names(self) -> list:
        """Get a list of available custom image globe overlay Display Names (Programmatic Identifiers)."""
        return self._intf.invoke(CustomImageGlobeOverlayPluginActivator._metadata, CustomImageGlobeOverlayPluginActivator._get_available_display_names_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CustomImageGlobeOverlayPluginActivator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CustomImageGlobeOverlayPluginActivator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CustomImageGlobeOverlayPluginActivator, [CustomImageGlobeOverlayPluginActivator, ])

agcls.AgClassCatalog.add_catalog_entry((5403836162981649170, 18332814256462960269), CustomImageGlobeOverlayPluginActivator)
agcls.AgTypeNameMap["CustomImageGlobeOverlayPluginActivator"] = CustomImageGlobeOverlayPluginActivator

class CustomImageGlobeOverlayPluginProxy(SupportsDeleteCallback):
    """A proxy class provides access to a custom image globe overlay implemented by a plugin. Proxies are instantiated using custom image globe overlay plugin activator."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_custom_image_globe_overlay_method_offset = 1
    _get_is_custom_image_globe_overlay_supported_method_offset = 2
    _get_real_plugin_object_method_offset = 3
    _metadata = {
        "iid_data" : (4961013627642428157, 3328458864433877170),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CustomImageGlobeOverlayPluginProxy)

    _get_custom_image_globe_overlay_metadata = { "offset" : _get_custom_image_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_image_globe_overlay(self) -> "CustomImageGlobeOverlay":
        """Return a custom image globe overlay."""
        return self._intf.get_property(CustomImageGlobeOverlayPluginProxy._metadata, CustomImageGlobeOverlayPluginProxy._get_custom_image_globe_overlay_metadata)

    _get_is_custom_image_globe_overlay_supported_metadata = { "offset" : _get_is_custom_image_globe_overlay_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_custom_image_globe_overlay_supported(self) -> bool:
        """Return true if custom image globe overlays are supported."""
        return self._intf.get_property(CustomImageGlobeOverlayPluginProxy._metadata, CustomImageGlobeOverlayPluginProxy._get_is_custom_image_globe_overlay_supported_metadata)

    _get_real_plugin_object_metadata = { "offset" : _get_real_plugin_object_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def real_plugin_object(self) -> typing.Any:
        """Return a pointer to plugin object's IUnknown interface."""
        return self._intf.get_property(CustomImageGlobeOverlayPluginProxy._metadata, CustomImageGlobeOverlayPluginProxy._get_real_plugin_object_metadata)

    _property_names[custom_image_globe_overlay] = "custom_image_globe_overlay"
    _property_names[is_custom_image_globe_overlay_supported] = "is_custom_image_globe_overlay_supported"
    _property_names[real_plugin_object] = "real_plugin_object"

    def __init__(self, source_object=None):
        """Construct an object of type CustomImageGlobeOverlayPluginProxy."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CustomImageGlobeOverlayPluginProxy)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CustomImageGlobeOverlayPluginProxy, [CustomImageGlobeOverlayPluginProxy, ])

agcls.AgClassCatalog.add_catalog_entry((5287787800528969618, 3978382521805652152), CustomImageGlobeOverlayPluginProxy)
agcls.AgTypeNameMap["CustomImageGlobeOverlayPluginProxy"] = CustomImageGlobeOverlayPluginProxy

class GeospatialImageGlobeOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay for handling `JPEG 2000 <https://jpeg.org/jpeg2000/>`_ (.jp2), ECW (.ecw), ECWP, and MrSid (.sid) image formats in the WGS84 geographic projection."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_transparent_color_method_offset = 1
    _set_use_transparent_color_method_offset = 2
    _get_transparent_color_method_offset = 3
    _set_transparent_color_method_offset = 4
    _metadata = {
        "iid_data" : (5501235223544899002, 14977920513653948607),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GeospatialImageGlobeOverlay)

    _get_use_transparent_color_metadata = { "offset" : _get_use_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_transparent_color(self) -> bool:
        """Get or set whether transparent color should be used."""
        return self._intf.get_property(GeospatialImageGlobeOverlay._metadata, GeospatialImageGlobeOverlay._get_use_transparent_color_metadata)

    _set_use_transparent_color_metadata = { "offset" : _set_use_transparent_color_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_transparent_color.setter
    def use_transparent_color(self, use_transparent_color:bool) -> None:
        return self._intf.set_property(GeospatialImageGlobeOverlay._metadata, GeospatialImageGlobeOverlay._set_use_transparent_color_metadata, use_transparent_color)

    _get_transparent_color_metadata = { "offset" : _get_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def transparent_color(self) -> agcolor.Color:
        """Get or set the color that will become transparent."""
        return self._intf.get_property(GeospatialImageGlobeOverlay._metadata, GeospatialImageGlobeOverlay._get_transparent_color_metadata)

    _set_transparent_color_metadata = { "offset" : _set_transparent_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @transparent_color.setter
    def transparent_color(self, transparent_color:agcolor.Color) -> None:
        return self._intf.set_property(GeospatialImageGlobeOverlay._metadata, GeospatialImageGlobeOverlay._set_transparent_color_metadata, transparent_color)

    _property_names[use_transparent_color] = "use_transparent_color"
    _property_names[transparent_color] = "transparent_color"

    def __init__(self, source_object=None):
        """Construct an object of type GeospatialImageGlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GeospatialImageGlobeOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GeospatialImageGlobeOverlay, [GeospatialImageGlobeOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5299127955564734487, 15153468011881559482), GeospatialImageGlobeOverlay)
agcls.AgTypeNameMap["GeospatialImageGlobeOverlay"] = GeospatialImageGlobeOverlay

class GlobeOverlay(IGlobeOverlay, SupportsDeleteCallback):
    """The base class of all terrain overlay and globe image overlay objects."""
    def __init__(self, source_object=None):
        """Construct an object of type GlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GlobeOverlay, [IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((4864694773533807784, 5277725294694145455), GlobeOverlay)
agcls.AgTypeNameMap["GlobeOverlay"] = GlobeOverlay

class GlobeOverlaySettings(SupportsDeleteCallback):
    """Settings used by globe overlay objects. These setting affect all scenes."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_terrain_cache_size_method_offset = 1
    _set_terrain_cache_size_method_offset = 2
    _get_imagery_cache_size_method_offset = 3
    _set_imagery_cache_size_method_offset = 4
    _get_preload_terrain_and_imagery_method_offset = 5
    _set_preload_terrain_and_imagery_method_offset = 6
    _metadata = {
        "iid_data" : (4807517707255994079, 16599142282845173912),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GlobeOverlaySettings)

    _get_terrain_cache_size_metadata = { "offset" : _get_terrain_cache_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def terrain_cache_size(self) -> int:
        """Get or set the size of the terrain cache in megabytes. It is not recommended to go above 128 megabytes. Large cache sizes can slow down rendering since so much imagery will be rendered."""
        return self._intf.get_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._get_terrain_cache_size_metadata)

    _set_terrain_cache_size_metadata = { "offset" : _set_terrain_cache_size_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @terrain_cache_size.setter
    def terrain_cache_size(self, terrain_cache_size:int) -> None:
        return self._intf.set_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._set_terrain_cache_size_metadata, terrain_cache_size)

    _get_imagery_cache_size_metadata = { "offset" : _get_imagery_cache_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def imagery_cache_size(self) -> int:
        """Get or set the size of the imagery cache in megabytes. It is not recommended to go above 128 megabytes. Large cache sizes can slow down rendering since so much imagery will be rendered."""
        return self._intf.get_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._get_imagery_cache_size_metadata)

    _set_imagery_cache_size_metadata = { "offset" : _set_imagery_cache_size_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @imagery_cache_size.setter
    def imagery_cache_size(self, imagery_cache_size:int) -> None:
        return self._intf.set_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._set_imagery_cache_size_metadata, imagery_cache_size)

    _get_preload_terrain_and_imagery_metadata = { "offset" : _get_preload_terrain_and_imagery_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def preload_terrain_and_imagery(self) -> bool:
        """Get or set whether terrain and imagery are preloaded. When set to true, terrain and imagery are preloaded to get the best visual quality; when set to false, they are not preloaded..."""
        return self._intf.get_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._get_preload_terrain_and_imagery_metadata)

    _set_preload_terrain_and_imagery_metadata = { "offset" : _set_preload_terrain_and_imagery_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @preload_terrain_and_imagery.setter
    def preload_terrain_and_imagery(self, preload_terrain_and_imagery:bool) -> None:
        return self._intf.set_property(GlobeOverlaySettings._metadata, GlobeOverlaySettings._set_preload_terrain_and_imagery_metadata, preload_terrain_and_imagery)

    _property_names[terrain_cache_size] = "terrain_cache_size"
    _property_names[imagery_cache_size] = "imagery_cache_size"
    _property_names[preload_terrain_and_imagery] = "preload_terrain_and_imagery"

    def __init__(self, source_object=None):
        """Construct an object of type GlobeOverlaySettings."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GlobeOverlaySettings)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GlobeOverlaySettings, [GlobeOverlaySettings, ])

agcls.AgClassCatalog.add_catalog_entry((5096239221085204322, 4767820054995129484), GlobeOverlaySettings)
agcls.AgTypeNameMap["GlobeOverlaySettings"] = GlobeOverlaySettings

class Lighting(SupportsDeleteCallback):
    """
    Lighting in the 3D scene.

    Examples
    --------
    Control the Lighting of the 3D scene:
    >>> # Scenario scenario: Scenario object
    >>> # Modify the lighting levels
    >>> manager = scenario.scene_manager
    >>> lighting = manager.scenes.item(0).lighting
    >>> lighting.ambient_intensity = 0.20  # Percent
    >>> lighting.diffuse_intensity = 4  # Percent
    >>> lighting.night_lights_intensity = 5  # Percent

    Set Vehicle Lighting Properties:
    >>> # Satellite satellite: Satellite object
    >>> lighting = satellite.graphics.lighting
    >>> # Settings for vehicle in sunlight
    >>> sunlight = lighting.sunlight
    >>> sunlight.visible = True
    >>> sunlight.color = Colors.Yellow
    >>> sunlight.line_width = LineWidth.WIDTH4
    >>> # Settings for vehicle in penumbra
    >>> penumbra = lighting.penumbra
    >>> penumbra.visible = True
    >>> penumbra.color = Colors.Orange
    >>> penumbra.line_width = LineWidth.WIDTH3
    >>> # Settings for vehicle in umbra
    >>> umbra = lighting.umbra
    >>> umbra.visible = True
    >>> umbra.color = Colors.Red
    >>> umbra.line_width = LineWidth.WIDTH2
    """

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_enabled_method_offset = 1
    _set_enabled_method_offset = 2
    _get_ambient_intensity_method_offset = 3
    _set_ambient_intensity_method_offset = 4
    _get_diffuse_intensity_method_offset = 5
    _set_diffuse_intensity_method_offset = 6
    _get_night_lights_intensity_method_offset = 7
    _set_night_lights_intensity_method_offset = 8
    _metadata = {
        "iid_data" : (5237498334712483872, 1351019444030006455),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Lighting)

    _get_enabled_metadata = { "offset" : _get_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enabled(self) -> bool:
        """Get or set whether or not lighting is enabled."""
        return self._intf.get_property(Lighting._metadata, Lighting._get_enabled_metadata)

    _set_enabled_metadata = { "offset" : _set_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enabled.setter
    def enabled(self, enabled:bool) -> None:
        return self._intf.set_property(Lighting._metadata, Lighting._set_enabled_metadata, enabled)

    _get_ambient_intensity_metadata = { "offset" : _get_ambient_intensity_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def ambient_intensity(self) -> float:
        """Get or set the ambient intensity throughout the scene."""
        return self._intf.get_property(Lighting._metadata, Lighting._get_ambient_intensity_metadata)

    _set_ambient_intensity_metadata = { "offset" : _set_ambient_intensity_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @ambient_intensity.setter
    def ambient_intensity(self, ambient_intensity:float) -> None:
        return self._intf.set_property(Lighting._metadata, Lighting._set_ambient_intensity_metadata, ambient_intensity)

    _get_diffuse_intensity_metadata = { "offset" : _get_diffuse_intensity_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def diffuse_intensity(self) -> float:
        """Get or set the diffuse intensity from the sun."""
        return self._intf.get_property(Lighting._metadata, Lighting._get_diffuse_intensity_metadata)

    _set_diffuse_intensity_metadata = { "offset" : _set_diffuse_intensity_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @diffuse_intensity.setter
    def diffuse_intensity(self, diffuse_intensity:float) -> None:
        return self._intf.set_property(Lighting._metadata, Lighting._set_diffuse_intensity_metadata, diffuse_intensity)

    _get_night_lights_intensity_metadata = { "offset" : _get_night_lights_intensity_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def night_lights_intensity(self) -> float:
        """Get or set the overall brightness for the night light's image overlay, night overlay."""
        return self._intf.get_property(Lighting._metadata, Lighting._get_night_lights_intensity_metadata)

    _set_night_lights_intensity_metadata = { "offset" : _set_night_lights_intensity_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @night_lights_intensity.setter
    def night_lights_intensity(self, night_lights_intensity:float) -> None:
        return self._intf.set_property(Lighting._metadata, Lighting._set_night_lights_intensity_metadata, night_lights_intensity)

    _property_names[enabled] = "enabled"
    _property_names[ambient_intensity] = "ambient_intensity"
    _property_names[diffuse_intensity] = "diffuse_intensity"
    _property_names[night_lights_intensity] = "night_lights_intensity"

    def __init__(self, source_object=None):
        """Construct an object of type Lighting."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Lighting)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Lighting, [Lighting, ])

agcls.AgClassCatalog.add_catalog_entry((5650660233190705200, 7654417412287974021), Lighting)
agcls.AgTypeNameMap["Lighting"] = Lighting

class PathPrimitiveUpdatePolicy(IPathPrimitiveUpdatePolicy, SupportsDeleteCallback):
    """A class that encapsulates the update logic for a path primitive. Derived classes must implement the Update method."""
    def __init__(self, source_object=None):
        """Construct an object of type PathPrimitiveUpdatePolicy."""
        SupportsDeleteCallback.__init__(self)
        IPathPrimitiveUpdatePolicy.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPathPrimitiveUpdatePolicy._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PathPrimitiveUpdatePolicy, [IPathPrimitiveUpdatePolicy])

agcls.AgClassCatalog.add_catalog_entry((5489232455939643999, 11995564172842310537), PathPrimitiveUpdatePolicy)
agcls.AgTypeNameMap["PathPrimitiveUpdatePolicy"] = PathPrimitiveUpdatePolicy

class ProjectedRasterOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay which projects a raster onto the terrain or surface of the central body. You can also enable projection onto models by setting projected raster model projection to true for a Scene..."""

    _num_methods = 36
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_raster_method_offset = 1
    _set_raster_method_offset = 2
    _get_projection_method_offset = 3
    _set_projection_method_offset = 4
    _get_show_shadows_method_offset = 5
    _set_show_shadows_method_offset = 6
    _get_show_frustum_method_offset = 7
    _set_show_frustum_method_offset = 8
    _get_show_far_plane_method_offset = 9
    _set_show_far_plane_method_offset = 10
    _get_color_method_offset = 11
    _set_color_method_offset = 12
    _get_frustum_color_method_offset = 13
    _set_frustum_color_method_offset = 14
    _get_far_plane_color_method_offset = 15
    _set_far_plane_color_method_offset = 16
    _get_shadow_color_method_offset = 17
    _set_shadow_color_method_offset = 18
    _get_border_color_method_offset = 19
    _set_border_color_method_offset = 20
    _get_border_width_method_offset = 21
    _set_border_width_method_offset = 22
    _get_frustum_translucency_method_offset = 23
    _set_frustum_translucency_method_offset = 24
    _get_far_plane_translucency_method_offset = 25
    _set_far_plane_translucency_method_offset = 26
    _get_shadow_translucency_method_offset = 27
    _set_shadow_translucency_method_offset = 28
    _get_border_translucency_method_offset = 29
    _set_border_translucency_method_offset = 30
    _get_use_transparent_color_method_offset = 31
    _set_use_transparent_color_method_offset = 32
    _get_transparent_color_method_offset = 33
    _set_transparent_color_method_offset = 34
    _get_directions_method_offset = 35
    _get_supported_method_offset = 36
    _metadata = {
        "iid_data" : (4883959137172591341, 4886933176258315195),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProjectedRasterOverlay)

    _get_raster_metadata = { "offset" : _get_raster_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster(self) -> "IRaster":
        """Get or set the raster that is projected."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_raster_metadata)

    _set_raster_metadata = { "offset" : _set_raster_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRaster"),) }
    @raster.setter
    def raster(self, raster:"IRaster") -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_raster_metadata, raster)

    _get_projection_metadata = { "offset" : _get_projection_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def projection(self) -> "IProjection":
        """Get or set the projection that projects the raster."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_projection_metadata)

    _set_projection_metadata = { "offset" : _set_projection_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IProjection"),) }
    @projection.setter
    def projection(self, projection:"IProjection") -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_projection_metadata, projection)

    _get_show_shadows_metadata = { "offset" : _get_show_shadows_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_shadows(self) -> bool:
        """Get or set whether to show shadows or not. When set to true, the raster will only be projected onto parts of the terrain visible from the projection's position. When false, the raster will project onto any terrain inside the projection's view frustum..."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_show_shadows_metadata)

    _set_show_shadows_metadata = { "offset" : _set_show_shadows_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_shadows.setter
    def show_shadows(self, show_shadows:bool) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_show_shadows_metadata, show_shadows)

    _get_show_frustum_metadata = { "offset" : _get_show_frustum_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_frustum(self) -> bool:
        """Get or set whether to show the frustum of the projection."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_show_frustum_metadata)

    _set_show_frustum_metadata = { "offset" : _set_show_frustum_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_frustum.setter
    def show_frustum(self, show_frustum:bool) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_show_frustum_metadata, show_frustum)

    _get_show_far_plane_metadata = { "offset" : _get_show_far_plane_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_far_plane(self) -> bool:
        """Get or set whether to show the far plane of the projection. If this is set to true, you will see the projected raster even when it does not intersect terrain."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_show_far_plane_metadata)

    _set_show_far_plane_metadata = { "offset" : _set_show_far_plane_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_far_plane.setter
    def show_far_plane(self, show_far_plane:bool) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_show_far_plane_metadata, show_far_plane)

    _get_color_metadata = { "offset" : _get_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def color(self) -> agcolor.Color:
        """Get or set the color of the projected raster."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_color_metadata)

    _set_color_metadata = { "offset" : _set_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @color.setter
    def color(self, color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_color_metadata, color)

    _get_frustum_color_metadata = { "offset" : _get_frustum_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def frustum_color(self) -> agcolor.Color:
        """Get or set the color of the projection's frustum."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_frustum_color_metadata)

    _set_frustum_color_metadata = { "offset" : _set_frustum_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @frustum_color.setter
    def frustum_color(self, frustum_color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_frustum_color_metadata, frustum_color)

    _get_far_plane_color_metadata = { "offset" : _get_far_plane_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def far_plane_color(self) -> agcolor.Color:
        """Get or set the color of the projection's far plane."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_far_plane_color_metadata)

    _set_far_plane_color_metadata = { "offset" : _set_far_plane_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @far_plane_color.setter
    def far_plane_color(self, far_plane_color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_far_plane_color_metadata, far_plane_color)

    _get_shadow_color_metadata = { "offset" : _get_shadow_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def shadow_color(self) -> agcolor.Color:
        """Get or set the color of the projection's shadow."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_shadow_color_metadata)

    _set_shadow_color_metadata = { "offset" : _set_shadow_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @shadow_color.setter
    def shadow_color(self, shadow_color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_shadow_color_metadata, shadow_color)

    _get_border_color_metadata = { "offset" : _get_border_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def border_color(self) -> agcolor.Color:
        """Get or set the color of the projection's border."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_border_color_metadata)

    _set_border_color_metadata = { "offset" : _set_border_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @border_color.setter
    def border_color(self, border_color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_border_color_metadata, border_color)

    _get_border_width_metadata = { "offset" : _get_border_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def border_width(self) -> float:
        """Get or set the width of the projection's border."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_border_width_metadata)

    _set_border_width_metadata = { "offset" : _set_border_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @border_width.setter
    def border_width(self, border_width:float) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_border_width_metadata, border_width)

    _get_frustum_translucency_metadata = { "offset" : _get_frustum_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def frustum_translucency(self) -> float:
        """Get or set the translucency of the projection's frustum."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_frustum_translucency_metadata)

    _set_frustum_translucency_metadata = { "offset" : _set_frustum_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @frustum_translucency.setter
    def frustum_translucency(self, frustum_translucency:float) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_frustum_translucency_metadata, frustum_translucency)

    _get_far_plane_translucency_metadata = { "offset" : _get_far_plane_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def far_plane_translucency(self) -> float:
        """Get or set the translucency of the projection's far plane."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_far_plane_translucency_metadata)

    _set_far_plane_translucency_metadata = { "offset" : _set_far_plane_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @far_plane_translucency.setter
    def far_plane_translucency(self, far_plane_translucency:float) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_far_plane_translucency_metadata, far_plane_translucency)

    _get_shadow_translucency_metadata = { "offset" : _get_shadow_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def shadow_translucency(self) -> float:
        """Get or set the translucency of the projection's shadow."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_shadow_translucency_metadata)

    _set_shadow_translucency_metadata = { "offset" : _set_shadow_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @shadow_translucency.setter
    def shadow_translucency(self, shadow_translucency:float) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_shadow_translucency_metadata, shadow_translucency)

    _get_border_translucency_metadata = { "offset" : _get_border_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def border_translucency(self) -> float:
        """Get or set the translucency of the projection's border."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_border_translucency_metadata)

    _set_border_translucency_metadata = { "offset" : _set_border_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @border_translucency.setter
    def border_translucency(self, border_translucency:float) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_border_translucency_metadata, border_translucency)

    _get_use_transparent_color_metadata = { "offset" : _get_use_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_transparent_color(self) -> bool:
        """Get or set whether transparent color should be used."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_use_transparent_color_metadata)

    _set_use_transparent_color_metadata = { "offset" : _set_use_transparent_color_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_transparent_color.setter
    def use_transparent_color(self, use_transparent_color:bool) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_use_transparent_color_metadata, use_transparent_color)

    _get_transparent_color_metadata = { "offset" : _get_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def transparent_color(self) -> agcolor.Color:
        """Get or set the color that will become transparent."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_transparent_color_metadata)

    _set_transparent_color_metadata = { "offset" : _set_transparent_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @transparent_color.setter
    def transparent_color(self, transparent_color:agcolor.Color) -> None:
        return self._intf.set_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._set_transparent_color_metadata, transparent_color)

    _get_directions_metadata = { "offset" : _get_directions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def directions(self) -> list:
        """Get the direction vectors in the central body's fixed reference frame that define the projection's frustum..."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_directions_metadata)

    _get_supported_metadata = { "offset" : _get_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def supported(self) -> bool:
        """Get whether or not the video card supports the projected raster overlay."""
        return self._intf.get_property(ProjectedRasterOverlay._metadata, ProjectedRasterOverlay._get_supported_metadata)

    _property_names[raster] = "raster"
    _property_names[projection] = "projection"
    _property_names[show_shadows] = "show_shadows"
    _property_names[show_frustum] = "show_frustum"
    _property_names[show_far_plane] = "show_far_plane"
    _property_names[color] = "color"
    _property_names[frustum_color] = "frustum_color"
    _property_names[far_plane_color] = "far_plane_color"
    _property_names[shadow_color] = "shadow_color"
    _property_names[border_color] = "border_color"
    _property_names[border_width] = "border_width"
    _property_names[frustum_translucency] = "frustum_translucency"
    _property_names[far_plane_translucency] = "far_plane_translucency"
    _property_names[shadow_translucency] = "shadow_translucency"
    _property_names[border_translucency] = "border_translucency"
    _property_names[use_transparent_color] = "use_transparent_color"
    _property_names[transparent_color] = "transparent_color"
    _property_names[directions] = "directions"
    _property_names[supported] = "supported"

    def __init__(self, source_object=None):
        """Construct an object of type ProjectedRasterOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProjectedRasterOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProjectedRasterOverlay, [ProjectedRasterOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((4696496787392859056, 16340637864943862462), ProjectedRasterOverlay)
agcls.AgTypeNameMap["ProjectedRasterOverlay"] = ProjectedRasterOverlay

class Projection(IProjection, SupportsDeleteCallback):
    """A projection represents a simplified camera with a position, orientation, and field of view horizontal and field of view vertical..."""
    def __init__(self, source_object=None):
        """Construct an object of type Projection."""
        SupportsDeleteCallback.__init__(self)
        IProjection.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProjection._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Projection, [IProjection])

agcls.AgClassCatalog.add_catalog_entry((4703414202910819741, 11526519806636668054), Projection)
agcls.AgTypeNameMap["Projection"] = Projection

class ProjectionStream(IProjection, SupportsDeleteCallback):
    """A projection that is updated dynamically at the specified update delta. The class can be used to stream projection data to projection clients, like projected raster overlay..."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_update_delta_method_offset = 1
    _set_update_delta_method_offset = 2
    _update_method_offset = 3
    _metadata = {
        "iid_data" : (5166404080678366159, 7416564084710466476),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProjectionStream)

    _get_update_delta_metadata = { "offset" : _get_update_delta_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def update_delta(self) -> float:
        """Get or set the update delta of the projection stream in seconds. The update delta defines the interval at which the Update method will be called..."""
        return self._intf.get_property(ProjectionStream._metadata, ProjectionStream._get_update_delta_metadata)

    _set_update_delta_metadata = { "offset" : _set_update_delta_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @update_delta.setter
    def update_delta(self, update_delta:float) -> None:
        return self._intf.set_property(ProjectionStream._metadata, ProjectionStream._set_update_delta_metadata, update_delta)

    _update_metadata = { "offset" : _update_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("Date"), agmarshall.InterfaceInArg("Date"), agmarshall.VariantBoolArg,) }
    def update(self, time:"Date", next_time:"Date") -> bool:
        """When overridden in a derived class, updates the projection data associated with the projection stream at the specified time. When the Update method is called, the projection stream contains the current projection data..."""
        return self._intf.invoke(ProjectionStream._metadata, ProjectionStream._update_metadata, time, next_time, OutArg())

    _property_names[update_delta] = "update_delta"

    def __init__(self, source_object=None):
        """Construct an object of type ProjectionStream."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProjectionStream)
        IProjection.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProjection._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProjectionStream, [ProjectionStream, IProjection])

agcls.AgClassCatalog.add_catalog_entry((5236009813300156213, 4436082934118817445), ProjectionStream)
agcls.AgTypeNameMap["ProjectionStream"] = ProjectionStream

class SceneGlobeOverlaySettings(SupportsDeleteCallback):
    """Settings used by globe overlay objects. These settings only affect the scene."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_anti_alias_imagery_method_offset = 1
    _set_anti_alias_imagery_method_offset = 2
    _get_terrain_mesh_pixel_error_method_offset = 3
    _set_terrain_mesh_pixel_error_method_offset = 4
    _get_imagery_pixel_error_method_offset = 5
    _set_imagery_pixel_error_method_offset = 6
    _get_projected_raster_model_projection_method_offset = 7
    _set_projected_raster_model_projection_method_offset = 8
    _metadata = {
        "iid_data" : (5574400525919150531, 8204789827079609220),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SceneGlobeOverlaySettings)

    _get_anti_alias_imagery_metadata = { "offset" : _get_anti_alias_imagery_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def anti_alias_imagery(self) -> bool:
        """Get or set a value indicating whether or not imagery is anti-aliased."""
        return self._intf.get_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._get_anti_alias_imagery_metadata)

    _set_anti_alias_imagery_metadata = { "offset" : _set_anti_alias_imagery_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @anti_alias_imagery.setter
    def anti_alias_imagery(self, anti_alias_imagery:bool) -> None:
        return self._intf.set_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._set_anti_alias_imagery_metadata, anti_alias_imagery)

    _get_terrain_mesh_pixel_error_metadata = { "offset" : _get_terrain_mesh_pixel_error_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def terrain_mesh_pixel_error(self) -> float:
        """Get or set the pixel error for terrain meshes. This is the number of pixels that the rendered terrain is different from the actual terrain data. The default is 2.0 pixels."""
        return self._intf.get_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._get_terrain_mesh_pixel_error_metadata)

    _set_terrain_mesh_pixel_error_metadata = { "offset" : _set_terrain_mesh_pixel_error_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @terrain_mesh_pixel_error.setter
    def terrain_mesh_pixel_error(self, terrain_mesh_pixel_error:float) -> None:
        return self._intf.set_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._set_terrain_mesh_pixel_error_metadata, terrain_mesh_pixel_error)

    _get_imagery_pixel_error_metadata = { "offset" : _get_imagery_pixel_error_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def imagery_pixel_error(self) -> float:
        """Get or set the pixel error for imagery. This is the number of pixels that the rendered imagery is different from the actual imagery data. The default is 1.0 pixel."""
        return self._intf.get_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._get_imagery_pixel_error_metadata)

    _set_imagery_pixel_error_metadata = { "offset" : _set_imagery_pixel_error_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @imagery_pixel_error.setter
    def imagery_pixel_error(self, imagery_pixel_error:float) -> None:
        return self._intf.set_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._set_imagery_pixel_error_metadata, imagery_pixel_error)

    _get_projected_raster_model_projection_metadata = { "offset" : _get_projected_raster_model_projection_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def projected_raster_model_projection(self) -> bool:
        """Get or set whether projected raster globe overlays will also project onto models."""
        return self._intf.get_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._get_projected_raster_model_projection_metadata)

    _set_projected_raster_model_projection_metadata = { "offset" : _set_projected_raster_model_projection_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @projected_raster_model_projection.setter
    def projected_raster_model_projection(self, projected_raster_model_projection:bool) -> None:
        return self._intf.set_property(SceneGlobeOverlaySettings._metadata, SceneGlobeOverlaySettings._set_projected_raster_model_projection_metadata, projected_raster_model_projection)

    _property_names[anti_alias_imagery] = "anti_alias_imagery"
    _property_names[terrain_mesh_pixel_error] = "terrain_mesh_pixel_error"
    _property_names[imagery_pixel_error] = "imagery_pixel_error"
    _property_names[projected_raster_model_projection] = "projected_raster_model_projection"

    def __init__(self, source_object=None):
        """Construct an object of type SceneGlobeOverlaySettings."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SceneGlobeOverlaySettings)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SceneGlobeOverlaySettings, [SceneGlobeOverlaySettings, ])

agcls.AgClassCatalog.add_catalog_entry((5538264776613401247, 14183016745653159312), SceneGlobeOverlaySettings)
agcls.AgTypeNameMap["SceneGlobeOverlaySettings"] = SceneGlobeOverlaySettings

class ScreenOverlayCollectionBase(IScreenOverlayCollectionBase, SupportsDeleteCallback):
    """The common base class for collections of overlays held by screen overlay and by screen overlay manager."""
    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlayCollectionBase."""
        SupportsDeleteCallback.__init__(self)
        IScreenOverlayCollectionBase.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IScreenOverlayCollectionBase._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlayCollectionBase, [IScreenOverlayCollectionBase])

agcls.AgClassCatalog.add_catalog_entry((5029264620485230918, 9940193730906732965), ScreenOverlayCollectionBase)
agcls.AgTypeNameMap["ScreenOverlayCollectionBase"] = ScreenOverlayCollectionBase

class Texture2DFactory(SupportsDeleteCallback):
    """A factory for creating texture 2d objects from various sources."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _load_from_string_uri_method_offset = 1
    _from_raster_method_offset = 2
    _metadata = {
        "iid_data" : (4925222235676858792, 13269238914304119718),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Texture2DFactory)

    _load_from_string_uri_metadata = { "offset" : _load_from_string_uri_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def load_from_string_uri(self, uri:str) -> "RendererTexture2D":
        """Create a new texture from a Uri, which can be a file, HTTP, HTTPS, or FTP source. See raster for a list of supported raster formats."""
        return self._intf.invoke(Texture2DFactory._metadata, Texture2DFactory._load_from_string_uri_metadata, uri, OutArg())

    _from_raster_metadata = { "offset" : _from_raster_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IRaster"), agmarshall.InterfaceOutArg,) }
    def from_raster(self, raster:"IRaster") -> "RendererTexture2D":
        """Create a new texture from a raster."""
        return self._intf.invoke(Texture2DFactory._metadata, Texture2DFactory._from_raster_metadata, raster, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type Texture2DFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Texture2DFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Texture2DFactory, [Texture2DFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5367179128558865548, 12869832894502700982), Texture2DFactory)
agcls.AgTypeNameMap["Texture2DFactory"] = Texture2DFactory

class VisualEffects(SupportsDeleteCallback):
    """Control various post processing effects that can be applied to the scene."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_lens_flare_enabled_method_offset = 1
    _set_lens_flare_enabled_method_offset = 2
    _get_vignette_enabled_method_offset = 3
    _set_vignette_enabled_method_offset = 4
    _get_vignette_strength_method_offset = 5
    _set_vignette_strength_method_offset = 6
    _metadata = {
        "iid_data" : (5261215102135709933, 16937211111774516119),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, VisualEffects)

    _get_lens_flare_enabled_metadata = { "offset" : _get_lens_flare_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def lens_flare_enabled(self) -> bool:
        """Get or set whether or not the lens flare effect is enabled."""
        return self._intf.get_property(VisualEffects._metadata, VisualEffects._get_lens_flare_enabled_metadata)

    _set_lens_flare_enabled_metadata = { "offset" : _set_lens_flare_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @lens_flare_enabled.setter
    def lens_flare_enabled(self, lens_flare_enabled:bool) -> None:
        return self._intf.set_property(VisualEffects._metadata, VisualEffects._set_lens_flare_enabled_metadata, lens_flare_enabled)

    _get_vignette_enabled_metadata = { "offset" : _get_vignette_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def vignette_enabled(self) -> bool:
        """Get or set whether or not the vignette effect is enabled. This simulates light being blocked by the lens hood, resulting in a slight darkening at the perimeter of the 3D Window."""
        return self._intf.get_property(VisualEffects._metadata, VisualEffects._get_vignette_enabled_metadata)

    _set_vignette_enabled_metadata = { "offset" : _set_vignette_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @vignette_enabled.setter
    def vignette_enabled(self, vignette_enabled:bool) -> None:
        return self._intf.set_property(VisualEffects._metadata, VisualEffects._set_vignette_enabled_metadata, vignette_enabled)

    _get_vignette_strength_metadata = { "offset" : _get_vignette_strength_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def vignette_strength(self) -> float:
        """Set the strength of the vignette effect, values between [0.001 and 5.0], with larger values resulting in more pronounced darkening around the perimeter of the 3D window."""
        return self._intf.get_property(VisualEffects._metadata, VisualEffects._get_vignette_strength_metadata)

    _set_vignette_strength_metadata = { "offset" : _set_vignette_strength_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @vignette_strength.setter
    def vignette_strength(self, vignette_strength:float) -> None:
        return self._intf.set_property(VisualEffects._metadata, VisualEffects._set_vignette_strength_metadata, vignette_strength)

    _property_names[lens_flare_enabled] = "lens_flare_enabled"
    _property_names[vignette_enabled] = "vignette_enabled"
    _property_names[vignette_strength] = "vignette_strength"

    def __init__(self, source_object=None):
        """Construct an object of type VisualEffects."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, VisualEffects)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, VisualEffects, [VisualEffects, ])

agcls.AgClassCatalog.add_catalog_entry((4977432288877815513, 5251131146797115811), VisualEffects)
agcls.AgTypeNameMap["VisualEffects"] = VisualEffects

class AltitudeDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive altitude interval that determines when an object is rendered based on the camera's altitude relative to a central body."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_altitude_method_offset = 1
    _set_minimum_altitude_method_offset = 2
    _get_maximum_altitude_method_offset = 3
    _set_maximum_altitude_method_offset = 4
    _get_central_body_method_offset = 5
    _set_central_body_method_offset = 6
    _metadata = {
        "iid_data" : (5393166715270093478, 17984733492443543221),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AltitudeDisplayCondition)

    _get_minimum_altitude_metadata = { "offset" : _get_minimum_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_altitude(self) -> float:
        """Get or set the minimum altitude of the inclusive altitude interval. Use Double.MinValue to ignore checking the minimum altitude."""
        return self._intf.get_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._get_minimum_altitude_metadata)

    _set_minimum_altitude_metadata = { "offset" : _set_minimum_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_altitude.setter
    def minimum_altitude(self, minimum_altitude:float) -> None:
        return self._intf.set_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._set_minimum_altitude_metadata, minimum_altitude)

    _get_maximum_altitude_metadata = { "offset" : _get_maximum_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_altitude(self) -> float:
        """Get or set the maximum altitude of the inclusive altitude interval. Use Double.MaxValue to ignore checking the maximum altitude."""
        return self._intf.get_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._get_maximum_altitude_metadata)

    _set_maximum_altitude_metadata = { "offset" : _set_maximum_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_altitude.setter
    def maximum_altitude(self, maximum_altitude:float) -> None:
        return self._intf.set_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._set_maximum_altitude_metadata, maximum_altitude)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get or set the central body to which the altitude is relative."""
        return self._intf.get_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(AltitudeDisplayCondition._metadata, AltitudeDisplayCondition._set_central_body_metadata, central_body)

    _property_names[minimum_altitude] = "minimum_altitude"
    _property_names[maximum_altitude] = "maximum_altitude"
    _property_names[central_body] = "central_body"

    def __init__(self, source_object=None):
        """Construct an object of type AltitudeDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AltitudeDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AltitudeDisplayCondition, [AltitudeDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((4955129817930386074, 3136574299604980158), AltitudeDisplayCondition)
agcls.AgTypeNameMap["AltitudeDisplayCondition"] = AltitudeDisplayCondition

class AxesPrimitive(IPrimitive, SupportsDeleteCallback):
    """Render an axes in the 3D scene."""

    _num_methods = 22
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_lighting_method_offset = 1
    _set_lighting_method_offset = 2
    _get_label_method_offset = 3
    _set_label_method_offset = 4
    _get_display_label_method_offset = 5
    _set_display_label_method_offset = 6
    _get_display_trace_method_offset = 7
    _set_display_trace_method_offset = 8
    _get_display_sweep_method_offset = 9
    _set_display_sweep_method_offset = 10
    _get_display_lines_method_offset = 11
    _set_display_lines_method_offset = 12
    _get_persistence_width_method_offset = 13
    _set_persistence_width_method_offset = 14
    _get_fade_persistence_method_offset = 15
    _set_fade_persistence_method_offset = 16
    _get_persistence_duration_method_offset = 17
    _set_persistence_duration_method_offset = 18
    _get_length_method_offset = 19
    _set_length_method_offset = 20
    _get_width_method_offset = 21
    _set_width_method_offset = 22
    _metadata = {
        "iid_data" : (5071979667285801873, 4985509215786764179),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AxesPrimitive)

    _get_lighting_metadata = { "offset" : _get_lighting_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def lighting(self) -> bool:
        """Get or set whether the primitive is lit."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_lighting_metadata)

    _set_lighting_metadata = { "offset" : _set_lighting_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @lighting.setter
    def lighting(self, lighting:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_lighting_metadata, lighting)

    _get_label_metadata = { "offset" : _get_label_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def label(self) -> str:
        """Get or set the axes label."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_label_metadata)

    _set_label_metadata = { "offset" : _set_label_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @label.setter
    def label(self, label:str) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_label_metadata, label)

    _get_display_label_metadata = { "offset" : _get_display_label_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_label(self) -> bool:
        """Get or set whether the axes' label is displayed."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_display_label_metadata)

    _set_display_label_metadata = { "offset" : _set_display_label_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_label.setter
    def display_label(self, display_label:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_display_label_metadata, display_label)

    _get_display_trace_metadata = { "offset" : _get_display_trace_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_trace(self) -> bool:
        """Get or set whether the persistence trace (points) is displayed."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_display_trace_metadata)

    _set_display_trace_metadata = { "offset" : _set_display_trace_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_trace.setter
    def display_trace(self, display_trace:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_display_trace_metadata, display_trace)

    _get_display_sweep_metadata = { "offset" : _get_display_sweep_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_sweep(self) -> bool:
        """Get or set whether the persistence sweep (triangles) is displayed."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_display_sweep_metadata)

    _set_display_sweep_metadata = { "offset" : _set_display_sweep_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_sweep.setter
    def display_sweep(self, display_sweep:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_display_sweep_metadata, display_sweep)

    _get_display_lines_metadata = { "offset" : _get_display_lines_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_lines(self) -> bool:
        """Get or set whether persistence lines are displayed."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_display_lines_metadata)

    _set_display_lines_metadata = { "offset" : _set_display_lines_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_lines.setter
    def display_lines(self, display_lines:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_display_lines_metadata, display_lines)

    _get_persistence_width_metadata = { "offset" : _get_persistence_width_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def persistence_width(self) -> float:
        """Get or set persistence point/line width."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_persistence_width_metadata)

    _set_persistence_width_metadata = { "offset" : _set_persistence_width_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @persistence_width.setter
    def persistence_width(self, persistence_width:float) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_persistence_width_metadata, persistence_width)

    _get_fade_persistence_metadata = { "offset" : _get_fade_persistence_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def fade_persistence(self) -> bool:
        """Get or set whether the persistence path should fade over time."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_fade_persistence_metadata)

    _set_fade_persistence_metadata = { "offset" : _set_fade_persistence_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @fade_persistence.setter
    def fade_persistence(self, fade_persistence:bool) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_fade_persistence_metadata, fade_persistence)

    _get_persistence_duration_metadata = { "offset" : _get_persistence_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def persistence_duration(self) -> float:
        """Get or set the maximum duration of the persistence path."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_persistence_duration_metadata)

    _set_persistence_duration_metadata = { "offset" : _set_persistence_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @persistence_duration.setter
    def persistence_duration(self, persistence_duration:float) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_persistence_duration_metadata, persistence_duration)

    _get_length_metadata = { "offset" : _get_length_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def length(self) -> float:
        """Get or set the axes' source-to-arrow-tip length."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_length_metadata)

    _set_length_metadata = { "offset" : _set_length_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @length.setter
    def length(self, length:float) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_length_metadata, length)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def width(self) -> float:
        """Get or set the width in pixels. As the camera distances changes from this primitive, the geometry will autoscale to maintain this thickness."""
        return self._intf.get_property(AxesPrimitive._metadata, AxesPrimitive._get_width_metadata)

    _set_width_metadata = { "offset" : _set_width_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @width.setter
    def width(self, width:float) -> None:
        return self._intf.set_property(AxesPrimitive._metadata, AxesPrimitive._set_width_metadata, width)

    _property_names[lighting] = "lighting"
    _property_names[label] = "label"
    _property_names[display_label] = "display_label"
    _property_names[display_trace] = "display_trace"
    _property_names[display_sweep] = "display_sweep"
    _property_names[display_lines] = "display_lines"
    _property_names[persistence_width] = "persistence_width"
    _property_names[fade_persistence] = "fade_persistence"
    _property_names[persistence_duration] = "persistence_duration"
    _property_names[length] = "length"
    _property_names[width] = "width"

    def __init__(self, source_object=None):
        """Construct an object of type AxesPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AxesPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AxesPrimitive, [AxesPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5590121640589148486, 2256538052145833109), AxesPrimitive)
agcls.AgTypeNameMap["AxesPrimitive"] = AxesPrimitive

class Camera(SupportsDeleteCallback):
    """
    Implemented by the scene camera. Contains operations to manipulate the camera position, view direction and orientation in the scene.

    Examples
    --------
    Change the camera reference frame:
    >>> # Scenario scenario: Scenario object
    >>> # STKObjectRoot root: STK Object Model Root
    >>> manager = scenario.scene_manager
    >>> manager.scenes.item(0).camera.view_central_body(
    >>>     "Earth", root.central_bodies.earth.analysis_workbench_components.axes.item("Fixed")
    >>> )
    >>> manager.render()

    Change the camera view to Imagery Extents:
    >>> # Scenario scenario: Scenario object
    >>> # AGIProcessedImageGlobeOverlay imageryTile: Image Overlay object
    >>> manager = scenario.scene_manager
    >>> extent = imageryTile.extent
    >>> # Change extent in the default 3D window
    >>> manager.scenes.item(0).camera.view_extent("Earth", extent)
    >>> manager.render()
    """

    _num_methods = 50
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_position_method_offset = 1
    _set_position_method_offset = 2
    _get_reference_point_method_offset = 3
    _set_reference_point_method_offset = 4
    _get_direction_method_offset = 5
    _set_direction_method_offset = 6
    _get_up_vector_method_offset = 7
    _set_up_vector_method_offset = 8
    _get_distance_method_offset = 9
    _set_distance_method_offset = 10
    _get_axes_method_offset = 11
    _set_axes_method_offset = 12
    _get_constrained_up_axis_method_offset = 13
    _set_constrained_up_axis_method_offset = 14
    _get_allow_rotation_over_constrained_up_axis_method_offset = 15
    _set_allow_rotation_over_constrained_up_axis_method_offset = 16
    _get_lock_view_direction_method_offset = 17
    _set_lock_view_direction_method_offset = 18
    _get_field_of_view_method_offset = 19
    _set_field_of_view_method_offset = 20
    _get_horizontal_field_of_view_method_offset = 21
    _get_vertical_field_of_view_method_offset = 22
    _get_near_plane_method_offset = 23
    _set_near_plane_method_offset = 24
    _get_far_plane_method_offset = 25
    _set_far_plane_method_offset = 26
    _get_far_near_plane_ratio_method_offset = 27
    _set_far_near_plane_ratio_method_offset = 28
    _get_distance_per_radius_method_offset = 29
    _get_snapshot_method_offset = 30
    _get_video_recording_method_offset = 31
    _get_pixel_size_per_distance_method_offset = 32
    _get_position_reference_frame_method_offset = 33
    _get_reference_point_reference_frame_method_offset = 34
    _visibility_test_method_offset = 35
    _cartographic_to_window_method_offset = 36
    _try_cartographic_to_window_method_offset = 37
    _window_to_cartographic_method_offset = 38
    _try_window_to_cartographic_method_offset = 39
    _view_central_body_method_offset = 40
    _view_extent_method_offset = 41
    _view_rectangular_extent_method_offset = 42
    _view_with_up_axis_method_offset = 43
    _view_method_offset = 44
    _view_direction_with_up_axis_method_offset = 45
    _view_direction_method_offset = 46
    _view_offset_with_up_axis_method_offset = 47
    _view_offset_method_offset = 48
    _view_offset_direction_with_up_axis_method_offset = 49
    _view_offset_direction_method_offset = 50
    _metadata = {
        "iid_data" : (5415728535553925060, 1568220948406149766),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Camera)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the position of the camera. The array contains the components of the position arranged in the order x, y, z."""
        return self._intf.get_property(Camera._metadata, Camera._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_position_metadata, position)

    _get_reference_point_metadata = { "offset" : _get_reference_point_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def reference_point(self) -> list:
        """Get or set the reference point of the camera. The array contains the components of the reference point arranged in the order x, y, z."""
        return self._intf.get_property(Camera._metadata, Camera._get_reference_point_metadata)

    _set_reference_point_metadata = { "offset" : _set_reference_point_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @reference_point.setter
    def reference_point(self, reference_point:list) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_reference_point_metadata, reference_point)

    _get_direction_metadata = { "offset" : _get_direction_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def direction(self) -> list:
        """Get or set the direction of the camera in axes. The array contains the components of the direction arranged in the order x, y, z."""
        return self._intf.get_property(Camera._metadata, Camera._get_direction_metadata)

    _set_direction_metadata = { "offset" : _set_direction_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @direction.setter
    def direction(self, direction:list) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_direction_metadata, direction)

    _get_up_vector_metadata = { "offset" : _get_up_vector_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def up_vector(self) -> list:
        """Get or set the up direction of the camera in axes. The array contains the components of the direction arranged in the order x, y, z."""
        return self._intf.get_property(Camera._metadata, Camera._get_up_vector_metadata)

    _set_up_vector_metadata = { "offset" : _set_up_vector_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @up_vector.setter
    def up_vector(self, up_vector:list) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_up_vector_metadata, up_vector)

    _get_distance_metadata = { "offset" : _get_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def distance(self) -> float:
        """Get or set the distance the camera position is from the reference point."""
        return self._intf.get_property(Camera._metadata, Camera._get_distance_metadata)

    _set_distance_metadata = { "offset" : _set_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @distance.setter
    def distance(self, distance:float) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_distance_metadata, distance)

    _get_axes_metadata = { "offset" : _get_axes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def axes(self) -> "IVectorGeometryToolAxes":
        """Get or set camera's axes of rotation."""
        return self._intf.get_property(Camera._metadata, Camera._get_axes_metadata)

    _set_axes_metadata = { "offset" : _set_axes_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    @axes.setter
    def axes(self, axes:"IVectorGeometryToolAxes") -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_axes_metadata, axes)

    _get_constrained_up_axis_metadata = { "offset" : _get_constrained_up_axis_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConstrainedUpAxis),) }
    @property
    def constrained_up_axis(self) -> "ConstrainedUpAxis":
        """Get or set the axis to constrain the up vector to."""
        return self._intf.get_property(Camera._metadata, Camera._get_constrained_up_axis_metadata)

    _set_constrained_up_axis_metadata = { "offset" : _set_constrained_up_axis_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ConstrainedUpAxis),) }
    @constrained_up_axis.setter
    def constrained_up_axis(self, constrained_up_axis:"ConstrainedUpAxis") -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_constrained_up_axis_metadata, constrained_up_axis)

    _get_allow_rotation_over_constrained_up_axis_metadata = { "offset" : _get_allow_rotation_over_constrained_up_axis_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def allow_rotation_over_constrained_up_axis(self) -> bool:
        """Get or set whether the camera can rotate over the constrained up axis. For example, if true the camera would be able to flip over the North Pole and view the globe upside down."""
        return self._intf.get_property(Camera._metadata, Camera._get_allow_rotation_over_constrained_up_axis_metadata)

    _set_allow_rotation_over_constrained_up_axis_metadata = { "offset" : _set_allow_rotation_over_constrained_up_axis_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @allow_rotation_over_constrained_up_axis.setter
    def allow_rotation_over_constrained_up_axis(self, allow_rotation_over_constrained_up_axis:bool) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_allow_rotation_over_constrained_up_axis_metadata, allow_rotation_over_constrained_up_axis)

    _get_lock_view_direction_metadata = { "offset" : _get_lock_view_direction_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def lock_view_direction(self) -> bool:
        """Get or set whether the camera's direction is locked."""
        return self._intf.get_property(Camera._metadata, Camera._get_lock_view_direction_metadata)

    _set_lock_view_direction_metadata = { "offset" : _set_lock_view_direction_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @lock_view_direction.setter
    def lock_view_direction(self, lock_view_direction:bool) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_lock_view_direction_metadata, lock_view_direction)

    _get_field_of_view_metadata = { "offset" : _get_field_of_view_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def field_of_view(self) -> float:
        """Get or set field of view. The field of view is applied to the larger of the window dimensions. For example, if the window width was 640 and the height was 480, the field of view applies to the horizontal..."""
        return self._intf.get_property(Camera._metadata, Camera._get_field_of_view_metadata)

    _set_field_of_view_metadata = { "offset" : _set_field_of_view_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @field_of_view.setter
    def field_of_view(self, field_of_view:float) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_field_of_view_metadata, field_of_view)

    _get_horizontal_field_of_view_metadata = { "offset" : _get_horizontal_field_of_view_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def horizontal_field_of_view(self) -> float:
        """Get the horizontal field of view."""
        return self._intf.get_property(Camera._metadata, Camera._get_horizontal_field_of_view_metadata)

    _get_vertical_field_of_view_metadata = { "offset" : _get_vertical_field_of_view_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def vertical_field_of_view(self) -> float:
        """Get the vertical field of view."""
        return self._intf.get_property(Camera._metadata, Camera._get_vertical_field_of_view_metadata)

    _get_near_plane_metadata = { "offset" : _get_near_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def near_plane(self) -> float:
        """Get or set the distance from the camera to the near plane."""
        return self._intf.get_property(Camera._metadata, Camera._get_near_plane_metadata)

    _set_near_plane_metadata = { "offset" : _set_near_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @near_plane.setter
    def near_plane(self, near_plane:float) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_near_plane_metadata, near_plane)

    _get_far_plane_metadata = { "offset" : _get_far_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def far_plane(self) -> float:
        """Get or set the distance from the camera to the far plane."""
        return self._intf.get_property(Camera._metadata, Camera._get_far_plane_metadata)

    _set_far_plane_metadata = { "offset" : _set_far_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @far_plane.setter
    def far_plane(self, far_plane:float) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_far_plane_metadata, far_plane)

    _get_far_near_plane_ratio_metadata = { "offset" : _get_far_near_plane_ratio_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def far_near_plane_ratio(self) -> float:
        """Get or set the value that is used to compute subdivisions of the viewing frustum. A large value will be faster but lose z-value precision. A small value will have better precision but perform slower..."""
        return self._intf.get_property(Camera._metadata, Camera._get_far_near_plane_ratio_metadata)

    _set_far_near_plane_ratio_metadata = { "offset" : _set_far_near_plane_ratio_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @far_near_plane_ratio.setter
    def far_near_plane_ratio(self, far_near_plane_ratio:float) -> None:
        return self._intf.set_property(Camera._metadata, Camera._set_far_near_plane_ratio_metadata, far_near_plane_ratio)

    _get_distance_per_radius_metadata = { "offset" : _get_distance_per_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def distance_per_radius(self) -> float:
        """Return the distance that the Camera's Position should be from the ReferencePoint in order to ensure that a sphere with a 1 meter radius centered at the ReferencePoint fits entirely in the view frustum."""
        return self._intf.get_property(Camera._metadata, Camera._get_distance_per_radius_metadata)

    _get_snapshot_metadata = { "offset" : _get_snapshot_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def snapshot(self) -> "CameraSnapshot":
        """Get the camera snapshot settings."""
        return self._intf.get_property(Camera._metadata, Camera._get_snapshot_metadata)

    _get_video_recording_metadata = { "offset" : _get_video_recording_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def video_recording(self) -> "CameraVideoRecording":
        """Get the camera video recorder."""
        return self._intf.get_property(Camera._metadata, Camera._get_video_recording_metadata)

    _get_pixel_size_per_distance_metadata = { "offset" : _get_pixel_size_per_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def pixel_size_per_distance(self) -> float:
        """Get the approximate number of meters covered by a pixel that is 1 meter away from the camera. This is commonly multiplied by the distance from the camera to an object to compute the approximate number of meters covered by a pixel of the object."""
        return self._intf.get_property(Camera._metadata, Camera._get_pixel_size_per_distance_metadata)

    _get_position_reference_frame_metadata = { "offset" : _get_position_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def position_reference_frame(self) -> "IVectorGeometryToolSystem":
        """Get the reference frame that the position is returned in. This reference frame is composed of the camera's from point and the axes."""
        return self._intf.get_property(Camera._metadata, Camera._get_position_reference_frame_metadata)

    _get_reference_point_reference_frame_metadata = { "offset" : _get_reference_point_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_point_reference_frame(self) -> "IVectorGeometryToolSystem":
        """Get the reference frame that the reference point is returned in. This reference frame is composed of the camera's to point and the axes."""
        return self._intf.get_property(Camera._metadata, Camera._get_reference_point_reference_frame_metadata)

    _visibility_test_metadata = { "offset" : _visibility_test_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceInArg("BoundingSphere"), agmarshall.EnumArg(Visibility),) }
    def visibility_test(self, reference_frame:"IVectorGeometryToolSystem", sphere:"BoundingSphere") -> "Visibility":
        """Get the visibility of a sphere against the view frustum and any occluding central bodies."""
        return self._intf.invoke(Camera._metadata, Camera._visibility_test_metadata, reference_frame, sphere, OutArg())

    _cartographic_to_window_metadata = { "offset" : _cartographic_to_window_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def cartographic_to_window(self, central_body:str, position:list) -> list:
        """Convert a cartographic position to a pixel coordinate relative to the globe control. This method can throw an exception. Returns an array containing the pixel coordinate (in the order x, y) of the cartographic position relative to the globe control..."""
        return self._intf.invoke(Camera._metadata, Camera._cartographic_to_window_metadata, central_body, position, OutArg())

    _try_cartographic_to_window_metadata = { "offset" : _try_cartographic_to_window_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def try_cartographic_to_window(self, central_body:str, position:list) -> list:
        """Convert a cartographic position to a pixel coordinate relative to the globe control. This method does not throw an exception."""
        return self._intf.invoke(Camera._metadata, Camera._try_cartographic_to_window_metadata, central_body, position, OutArg())

    _window_to_cartographic_metadata = { "offset" : _window_to_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def window_to_cartographic(self, central_body:str, position:list) -> list:
        """Convert a pixel coordinate relative to the globe control to a cartographic position. For speed, terrain is not considered; if the pixel coordinate does not intersect the ellipsoid, an exception is thrown. Returns the cartographic position..."""
        return self._intf.invoke(Camera._metadata, Camera._window_to_cartographic_metadata, central_body, position, OutArg())

    _try_window_to_cartographic_metadata = { "offset" : _try_window_to_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def try_window_to_cartographic(self, central_body:str, position:list) -> list:
        """Convert a pixel coordinate relative to the globe control to a cartographic position. For speed, terrain is not considered. This method does not throw an exception."""
        return self._intf.invoke(Camera._metadata, Camera._try_window_to_cartographic_metadata, central_body, position, OutArg())

    _view_central_body_metadata = { "offset" : _view_central_body_method_offset,
            "arg_types" : (agcom.BSTR, agcom.PVOID,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    def view_central_body(self, central_body:str, axes:"IVectorGeometryToolAxes") -> None:
        """Zoom to a central body and use the specified axes for rotation. The reference point is set to the center of the central body and the camera's position is set so the entire central body is visible."""
        return self._intf.invoke(Camera._metadata, Camera._view_central_body_metadata, central_body, axes)

    _view_extent_metadata = { "offset" : _view_extent_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def view_extent(self, central_body:str, extent:list) -> None:
        """Zooms to a cartographic extent on the centralBody. The camera will be looking straight down at the extent, with the up vector pointing toward local north. The axes is set to an east-north-up axes at the center of extent."""
        return self._intf.invoke(Camera._metadata, Camera._view_extent_metadata, central_body, extent)

    _view_rectangular_extent_metadata = { "offset" : _view_rectangular_extent_method_offset,
            "arg_types" : (agcom.BSTR, agcom.DOUBLE, agcom.DOUBLE, agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def view_rectangular_extent(self, central_body:str, west:float, south:float, east:float, north:float) -> None:
        """Zooms to a rectangular extent composed of west, south, east, north on the centralBody. The camera will be looking straight down at the extent, with the up vector pointing toward local north..."""
        return self._intf.invoke(Camera._metadata, Camera._view_rectangular_extent_metadata, central_body, west, south, east, north)

    _view_with_up_axis_metadata = { "offset" : _view_with_up_axis_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.LPSafearrayArg,) }
    def view_with_up_axis(self, axes:"IVectorGeometryToolAxes", camera_position:"IVectorGeometryToolPoint", reference_point:"IVectorGeometryToolPoint", up_axis:list) -> None:
        """View from a point to a point. Sets the camera's position and the reference point the camera is looking at."""
        return self._intf.invoke(Camera._metadata, Camera._view_with_up_axis_metadata, axes, camera_position, reference_point, up_axis)

    _view_metadata = { "offset" : _view_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    def view(self, axes:"IVectorGeometryToolAxes", camera_position:"IVectorGeometryToolPoint", reference_point:"IVectorGeometryToolPoint") -> None:
        """View from a point to a point. Sets the camera's position and the reference point the camera is looking at."""
        return self._intf.invoke(Camera._metadata, Camera._view_metadata, axes, camera_position, reference_point)

    _view_direction_with_up_axis_metadata = { "offset" : _view_direction_with_up_axis_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolVector"), agmarshall.LPSafearrayArg,) }
    def view_direction_with_up_axis(self, axes:"IVectorGeometryToolAxes", camera_position:"IVectorGeometryToolPoint", direction:"IVectorGeometryToolVector", up_axis:list) -> None:
        """View from a point to a direction. Sets the camera's position and the direction vector indicating where the camera is looking."""
        return self._intf.invoke(Camera._metadata, Camera._view_direction_with_up_axis_metadata, axes, camera_position, direction, up_axis)

    _view_direction_metadata = { "offset" : _view_direction_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    def view_direction(self, axes:"IVectorGeometryToolAxes", camera_position:"IVectorGeometryToolPoint", direction:"IVectorGeometryToolVector") -> None:
        """View from a point to a direction. Sets the camera's position and the direction vector indicating where the camera is looking."""
        return self._intf.invoke(Camera._metadata, Camera._view_direction_metadata, axes, camera_position, direction)

    _view_offset_with_up_axis_metadata = { "offset" : _view_offset_with_up_axis_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def view_offset_with_up_axis(self, axes:"IVectorGeometryToolAxes", reference_point:"IVectorGeometryToolPoint", offset:list, up_axis:list) -> None:
        """Set the camera's reference point - the point the camera is looking at. The camera's position is the reference point translated by the offset."""
        return self._intf.invoke(Camera._metadata, Camera._view_offset_with_up_axis_metadata, axes, reference_point, offset, up_axis)

    _view_offset_metadata = { "offset" : _view_offset_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.LPSafearrayArg,) }
    def view_offset(self, axes:"IVectorGeometryToolAxes", reference_point:"IVectorGeometryToolPoint", offset:list) -> None:
        """Set the camera's reference point - the point the camera is looking at. The camera's position is the reference point translated by the offset."""
        return self._intf.invoke(Camera._metadata, Camera._view_offset_metadata, axes, reference_point, offset)

    _view_offset_direction_with_up_axis_metadata = { "offset" : _view_offset_direction_with_up_axis_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolVector"), agmarshall.LPSafearrayArg,) }
    def view_offset_direction_with_up_axis(self, axes:"IVectorGeometryToolAxes", reference_point:"IVectorGeometryToolPoint", direction:"IVectorGeometryToolVector", up_axis:list) -> None:
        """Set the camera's reference point - the point the camera is looking at. The camera's position is the reference point translated by the direction vector."""
        return self._intf.invoke(Camera._metadata, Camera._view_offset_direction_with_up_axis_metadata, axes, reference_point, direction, up_axis)

    _view_offset_direction_metadata = { "offset" : _view_offset_direction_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("IVectorGeometryToolPoint"), agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    def view_offset_direction(self, axes:"IVectorGeometryToolAxes", reference_point:"IVectorGeometryToolPoint", direction:"IVectorGeometryToolVector") -> None:
        """Set the camera's reference point - the point the camera is looking at. The camera's position is the reference point translated by the direction vector."""
        return self._intf.invoke(Camera._metadata, Camera._view_offset_direction_metadata, axes, reference_point, direction)

    _property_names[position] = "position"
    _property_names[reference_point] = "reference_point"
    _property_names[direction] = "direction"
    _property_names[up_vector] = "up_vector"
    _property_names[distance] = "distance"
    _property_names[axes] = "axes"
    _property_names[constrained_up_axis] = "constrained_up_axis"
    _property_names[allow_rotation_over_constrained_up_axis] = "allow_rotation_over_constrained_up_axis"
    _property_names[lock_view_direction] = "lock_view_direction"
    _property_names[field_of_view] = "field_of_view"
    _property_names[horizontal_field_of_view] = "horizontal_field_of_view"
    _property_names[vertical_field_of_view] = "vertical_field_of_view"
    _property_names[near_plane] = "near_plane"
    _property_names[far_plane] = "far_plane"
    _property_names[far_near_plane_ratio] = "far_near_plane_ratio"
    _property_names[distance_per_radius] = "distance_per_radius"
    _property_names[snapshot] = "snapshot"
    _property_names[video_recording] = "video_recording"
    _property_names[pixel_size_per_distance] = "pixel_size_per_distance"
    _property_names[position_reference_frame] = "position_reference_frame"
    _property_names[reference_point_reference_frame] = "reference_point_reference_frame"

    def __init__(self, source_object=None):
        """Construct an object of type Camera."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Camera)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Camera, [Camera, ])

agcls.AgClassCatalog.add_catalog_entry((5108901659268738630, 12673905154193258132), Camera)
agcls.AgTypeNameMap["Camera"] = Camera

class CentralBodyGraphics(SupportsDeleteCallback):
    r"""
    The graphical properties associated with a particular central body. Changing the central body graphics will affect how the associated central body is rendered in a scene. For instance, to show or hide the central body, use the show property...

    Examples
    --------
    Add Imagery and Terrain to the Scene:
    >>> # Scenario scenario: Scenario object
    >>> # Retrieve the boundaries of the imported files
    >>> manager = scenario.scene_manager
    >>> # Add Terrain
    >>> installPath = r"C:\Program Files\AGI\STK 12" if os.name == "nt" else os.environ["STK_INSTALL_DIR"]
    >>> terrainTile = manager.scenes.item(0).central_bodies.earth.terrain.add_uri_string(
    >>>     os.path.join(installPath, "Data", "Resources", "stktraining", "samples", "SRTM_Skopje.pdtt")
    >>> )
    >>> extentTerrain = terrainTile.extent
    >>> print(
    >>>     "Terrain boundaries: LatMin: %s LatMax: %s LonMin: %s LonMax: %s"
    >>>     % (str(extentTerrain[0]), str(extentTerrain[2]), str(extentTerrain[1]), str(extentTerrain[3]))
    >>> )
    >>> # Add Imagery
    >>> imageryTile = manager.scenes.item(0).central_bodies.earth.imagery.add_uri_string(
    >>>     os.path.join(installPath, "Data", "Resources", "stktraining", "imagery", "NPS_OrganPipeCactus_Map.pdttx")
    >>> )
    >>> extentImagery = imageryTile.extent
    >>> print(
    >>>     "Imagery boundaries: LatMin: %s LatMax: %s LonMin: %s LonMax: %s"
    >>>     % (str(extentImagery[0]), str(extentImagery[2]), str(extentImagery[1]), str(extentImagery[3]))
    >>> )
    """

    _num_methods = 23
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_color_method_offset = 1
    _set_color_method_offset = 2
    _get_specular_color_method_offset = 3
    _set_specular_color_method_offset = 4
    _get_shininess_method_offset = 5
    _set_shininess_method_offset = 6
    _get_show_imagery_method_offset = 7
    _set_show_imagery_method_offset = 8
    _get_show_method_offset = 9
    _set_show_method_offset = 10
    _get_show_label_method_offset = 11
    _set_show_label_method_offset = 12
    _get_altitude_offset_method_offset = 13
    _set_altitude_offset_method_offset = 14
    _get_base_overlay_method_offset = 15
    _set_base_overlay_method_offset = 16
    _get_night_overlay_method_offset = 17
    _set_night_overlay_method_offset = 18
    _get_specular_overlay_method_offset = 19
    _set_specular_overlay_method_offset = 20
    _get_terrain_method_offset = 21
    _get_imagery_method_offset = 22
    _get_kml_method_offset = 23
    _metadata = {
        "iid_data" : (5748663363856356139, 14609936288003111080),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CentralBodyGraphics)

    _get_color_metadata = { "offset" : _get_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def color(self) -> agcolor.Color:
        """Get or set the color of the central body in the scene."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_color_metadata)

    _set_color_metadata = { "offset" : _set_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @color.setter
    def color(self, color:agcolor.Color) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_color_metadata, color)

    _get_specular_color_metadata = { "offset" : _get_specular_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def specular_color(self) -> agcolor.Color:
        """Get or set the specular color of the central body in the scene. The specular color is associated with the specular overlay."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_specular_color_metadata)

    _set_specular_color_metadata = { "offset" : _set_specular_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @specular_color.setter
    def specular_color(self, specular_color:agcolor.Color) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_specular_color_metadata, specular_color)

    _get_shininess_metadata = { "offset" : _get_shininess_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def shininess(self) -> float:
        """Get or set the shininess of the central body in the scene. The shininess affects the size and brightness of specular reflection associated with the specular overlay."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_shininess_metadata)

    _set_shininess_metadata = { "offset" : _set_shininess_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @shininess.setter
    def shininess(self, shininess:float) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_shininess_metadata, shininess)

    _get_show_imagery_metadata = { "offset" : _get_show_imagery_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_imagery(self) -> bool:
        """Get or set whether the imagery for central body in the scene is shown or hidden."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_show_imagery_metadata)

    _set_show_imagery_metadata = { "offset" : _set_show_imagery_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_imagery.setter
    def show_imagery(self, show_imagery:bool) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_show_imagery_metadata, show_imagery)

    _get_show_metadata = { "offset" : _get_show_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show(self) -> bool:
        """Get or set whether the central body is shown or hidden in the scene. This only affects the central body itself, not the primitives that are associated with it."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_show_metadata)

    _set_show_metadata = { "offset" : _set_show_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show.setter
    def show(self, show:bool) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_show_metadata, show)

    _get_show_label_metadata = { "offset" : _get_show_label_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_label(self) -> bool:
        """Get or set whether a label with the name of the central body should be rendered in the scene when the camera is at certain distance away from the central body."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_show_label_metadata)

    _set_show_label_metadata = { "offset" : _set_show_label_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_label.setter
    def show_label(self, show_label:bool) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_show_label_metadata, show_label)

    _get_altitude_offset_metadata = { "offset" : _get_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_offset(self) -> float:
        """Get or set the altitude that all terrain and imagery will be offset from the surface of the central body in the scene."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_altitude_offset_metadata)

    _set_altitude_offset_metadata = { "offset" : _set_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_offset.setter
    def altitude_offset(self, altitude_offset:float) -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_altitude_offset_metadata, altitude_offset)

    _get_base_overlay_metadata = { "offset" : _get_base_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_overlay(self) -> "IGlobeImageOverlay":
        """Get or set the base globe image overlay associated with the central body in the scene. The base overlay is always rendered before any other imagery..."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_base_overlay_metadata)

    _set_base_overlay_metadata = { "offset" : _set_base_overlay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    @base_overlay.setter
    def base_overlay(self, base_overlay:"IGlobeImageOverlay") -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_base_overlay_metadata, base_overlay)

    _get_night_overlay_metadata = { "offset" : _get_night_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def night_overlay(self) -> "IGlobeImageOverlay":
        """Get or set the night globe image overlay associated with the central body in the scene. The night overlay is displayed only on parts of the central body that are not in sun light..."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_night_overlay_metadata)

    _set_night_overlay_metadata = { "offset" : _set_night_overlay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    @night_overlay.setter
    def night_overlay(self, night_overlay:"IGlobeImageOverlay") -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_night_overlay_metadata, night_overlay)

    _get_specular_overlay_metadata = { "offset" : _get_specular_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def specular_overlay(self) -> "IGlobeImageOverlay":
        """Get or set the specular globe image overlay associated with the central body in the scene. The specular overlay is displayed only in the specular highlight of the central body."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_specular_overlay_metadata)

    _set_specular_overlay_metadata = { "offset" : _set_specular_overlay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    @specular_overlay.setter
    def specular_overlay(self, specular_overlay:"IGlobeImageOverlay") -> None:
        return self._intf.set_property(CentralBodyGraphics._metadata, CentralBodyGraphics._set_specular_overlay_metadata, specular_overlay)

    _get_terrain_metadata = { "offset" : _get_terrain_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def terrain(self) -> "TerrainOverlayCollection":
        """Get the collection of terrain overlay associated with the central body in the scene."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_terrain_metadata)

    _get_imagery_metadata = { "offset" : _get_imagery_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def imagery(self) -> "ImageCollection":
        """Get the collection of imagery associated with the central body in the scene."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_imagery_metadata)

    _get_kml_metadata = { "offset" : _get_kml_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def kml(self) -> "KmlGraphics":
        """Get the kml graphics associated with the central body in the scene."""
        return self._intf.get_property(CentralBodyGraphics._metadata, CentralBodyGraphics._get_kml_metadata)

    _property_names[color] = "color"
    _property_names[specular_color] = "specular_color"
    _property_names[shininess] = "shininess"
    _property_names[show_imagery] = "show_imagery"
    _property_names[show] = "show"
    _property_names[show_label] = "show_label"
    _property_names[altitude_offset] = "altitude_offset"
    _property_names[base_overlay] = "base_overlay"
    _property_names[night_overlay] = "night_overlay"
    _property_names[specular_overlay] = "specular_overlay"
    _property_names[terrain] = "terrain"
    _property_names[imagery] = "imagery"
    _property_names[kml] = "kml"

    def __init__(self, source_object=None):
        """Construct an object of type CentralBodyGraphics."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CentralBodyGraphics)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CentralBodyGraphics, [CentralBodyGraphics, ])

agcls.AgClassCatalog.add_catalog_entry((5237454306453849929, 16985193427983583917), CentralBodyGraphics)
agcls.AgTypeNameMap["CentralBodyGraphics"] = CentralBodyGraphics

class Clouds(SupportsDeleteCallback):
    """Load, show and hide clouds in the scene."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_show_method_offset = 1
    _set_show_method_offset = 2
    _get_clouds_uri_method_offset = 3
    _set_clouds_uri_method_offset = 4
    _get_roundness_method_offset = 5
    _set_roundness_method_offset = 6
    _get_altitude_method_offset = 7
    _set_altitude_method_offset = 8
    _get_is_valid_method_offset = 9
    _metadata = {
        "iid_data" : (4902781266871608760, 14063544887351084699),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Clouds)

    _get_show_metadata = { "offset" : _get_show_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show(self) -> bool:
        """Get or set whether clouds are rendered."""
        return self._intf.get_property(Clouds._metadata, Clouds._get_show_metadata)

    _set_show_metadata = { "offset" : _set_show_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show.setter
    def show(self, show:bool) -> None:
        return self._intf.set_property(Clouds._metadata, Clouds._set_show_metadata, show)

    _get_clouds_uri_metadata = { "offset" : _get_clouds_uri_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def clouds_uri(self) -> str:
        """The URI of the clouds index file. A cloud index file is an ascii file that contains a time-ordered list of images that display over the globe."""
        return self._intf.get_property(Clouds._metadata, Clouds._get_clouds_uri_metadata)

    _set_clouds_uri_metadata = { "offset" : _set_clouds_uri_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @clouds_uri.setter
    def clouds_uri(self, clouds_uri:str) -> None:
        return self._intf.set_property(Clouds._metadata, Clouds._set_clouds_uri_metadata, clouds_uri)

    _get_roundness_metadata = { "offset" : _get_roundness_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def roundness(self) -> float:
        """The roundness of the clouds."""
        return self._intf.get_property(Clouds._metadata, Clouds._get_roundness_metadata)

    _set_roundness_metadata = { "offset" : _set_roundness_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @roundness.setter
    def roundness(self, roundness:float) -> None:
        return self._intf.set_property(Clouds._metadata, Clouds._set_roundness_metadata, roundness)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """The altitude of the clouds."""
        return self._intf.get_property(Clouds._metadata, Clouds._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, altitude:float) -> None:
        return self._intf.set_property(Clouds._metadata, Clouds._set_altitude_metadata, altitude)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Return whether or not the clouds file is valid."""
        return self._intf.get_property(Clouds._metadata, Clouds._get_is_valid_metadata)

    _property_names[show] = "show"
    _property_names[clouds_uri] = "clouds_uri"
    _property_names[roundness] = "roundness"
    _property_names[altitude] = "altitude"
    _property_names[is_valid] = "is_valid"

    def __init__(self, source_object=None):
        """Construct an object of type Clouds."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Clouds)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Clouds, [Clouds, ])

agcls.AgClassCatalog.add_catalog_entry((4674132715452051907, 2514679956769629370), Clouds)
agcls.AgTypeNameMap["Clouds"] = Clouds

class CompositeDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """
    A composite of display conditions combined using a binary logic operation. For example, several time interval display condition objects can be added to a composite...

    Examples
    --------
    Display a Primitive During an Interval:
    >>> # Scenario scenario: Scenario object
    >>> # ModelPrimitive model: Graphics Primitive
    >>> manager = scenario.scene_manager
    >>> composite = manager.initializers.composite_display_condition.initialize()
    >>> root.units_preferences.item("DateFormat").set_current_unit("EpSec")
    >>> start = root.conversion_utility.new_date("EpSec", str(scenario.start_time))
    >>> stop = root.conversion_utility.new_date("EpSec", str(scenario.start_time + 600))
    >>> timeInterval = manager.initializers.time_interval_display_condition.initialize_with_times(start, stop)
    >>> composite.add(timeInterval)
    >>> model.display_condition = composite
    """

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_capacity_method_offset = 2
    _get_logic_operation_method_offset = 3
    _set_logic_operation_method_offset = 4
    _item_method_offset = 5
    _get__new_enum_method_offset = 6
    _reserve_method_offset = 7
    _add_with_negate_method_offset = 8
    _add_method_offset = 9
    _insert_with_negate_method_offset = 10
    _insert_method_offset = 11
    _remove_method_offset = 12
    _remove_at_method_offset = 13
    _clear_method_offset = 14
    _get_negate_method_offset = 15
    _set_negate_method_offset = 16
    _get_negate_at_method_offset = 17
    _set_negate_at_method_offset = 18
    _metadata = {
        "iid_data" : (5092744095359624815, 12221078393069903744),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CompositeDisplayCondition)
    def __iter__(self):
        """Create an iterator for the CompositeDisplayCondition object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IDisplayCondition":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of display conditions in the composite."""
        return self._intf.get_property(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get_count_metadata)

    _get_capacity_metadata = { "offset" : _get_capacity_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def capacity(self) -> int:
        """Get the number of display conditions for which memory has been allocated. This will always be greater or equal to count."""
        return self._intf.get_property(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get_capacity_metadata)

    _get_logic_operation_metadata = { "offset" : _get_logic_operation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BinaryLogicOperation),) }
    @property
    def logic_operation(self) -> "BinaryLogicOperation":
        """Get or set the binary logic operation applied to all display conditions in the composite when the composite is evaluated. To combine logical and and or operations in the same expression, create composites containing composites."""
        return self._intf.get_property(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get_logic_operation_metadata)

    _set_logic_operation_metadata = { "offset" : _set_logic_operation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BinaryLogicOperation),) }
    @logic_operation.setter
    def logic_operation(self, logic_operation:"BinaryLogicOperation") -> None:
        return self._intf.set_property(CompositeDisplayCondition._metadata, CompositeDisplayCondition._set_logic_operation_metadata, logic_operation)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "IDisplayCondition":
        """Return the condition at the given zero-based index."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get__new_enum_metadata)

    _reserve_metadata = { "offset" : _reserve_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def reserve(self, count:int) -> None:
        """Request enough memory for the composite to contain at least count display conditions. count will not be affected but capacity may be."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._reserve_metadata, count)

    _add_with_negate_metadata = { "offset" : _add_with_negate_method_offset,
            "arg_types" : (agcom.PVOID, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"), agmarshall.VariantBoolArg,) }
    def add_with_negate(self, display_condition:"IDisplayCondition", negate:bool) -> None:
        """Add a display condition to the end of the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._add_with_negate_metadata, display_condition, negate)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"),) }
    def add(self, display_condition:"IDisplayCondition") -> None:
        """Add a display condition to the end of the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._add_metadata, display_condition)

    _insert_with_negate_metadata = { "offset" : _insert_with_negate_method_offset,
            "arg_types" : (agcom.INT, agcom.PVOID, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceInArg("IDisplayCondition"), agmarshall.VariantBoolArg,) }
    def insert_with_negate(self, index:int, display_condition:"IDisplayCondition", negate:bool) -> None:
        """Insert a display condition at the given zero-based index, shifting existing display conditions."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._insert_with_negate_metadata, index, display_condition, negate)

    _insert_metadata = { "offset" : _insert_method_offset,
            "arg_types" : (agcom.INT, agcom.PVOID,),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceInArg("IDisplayCondition"),) }
    def insert(self, index:int, display_condition:"IDisplayCondition") -> None:
        """Insert a display condition at the given zero-based index, shifting existing display conditions."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._insert_metadata, index, display_condition)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"),) }
    def remove(self, display_condition:"IDisplayCondition") -> None:
        """Remove a display condition from the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._remove_metadata, display_condition)

    _remove_at_metadata = { "offset" : _remove_at_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def remove_at(self, index:int) -> None:
        """Remove the display condition at the given zero-based index, shifting existing display conditions."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._remove_at_metadata, index)

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all display conditions from the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._clear_metadata, )

    _get_negate_metadata = { "offset" : _get_negate_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"), agmarshall.VariantBoolArg,) }
    def get_negate(self, display_condition:"IDisplayCondition") -> bool:
        """Determine if a logical not operation is applied to a display condition in the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get_negate_metadata, display_condition, OutArg())

    _set_negate_metadata = { "offset" : _set_negate_method_offset,
            "arg_types" : (agcom.PVOID, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.InterfaceInArg("IDisplayCondition"), agmarshall.VariantBoolArg,) }
    def set_negate(self, display_condition:"IDisplayCondition", negate:bool) -> None:
        """Set if a logical not operation is applied to a display condition in the composite when the composite is evaluated."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._set_negate_metadata, display_condition, negate)

    _get_negate_at_metadata = { "offset" : _get_negate_at_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.IntArg, agmarshall.VariantBoolArg,) }
    def get_negate_at(self, index:int) -> bool:
        """Determine if a logical not operation is applied to a display condition in the composite."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._get_negate_at_metadata, index, OutArg())

    _set_negate_at_metadata = { "offset" : _set_negate_at_method_offset,
            "arg_types" : (agcom.INT, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.IntArg, agmarshall.VariantBoolArg,) }
    def set_negate_at(self, index:int, negate:bool) -> None:
        """Set if a logical not operation is applied to a display condition in the composite when the composite is evaluated."""
        return self._intf.invoke(CompositeDisplayCondition._metadata, CompositeDisplayCondition._set_negate_at_metadata, index, negate)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[capacity] = "capacity"
    _property_names[logic_operation] = "logic_operation"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CompositeDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CompositeDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CompositeDisplayCondition, [CompositeDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5056547403870315286, 10247967535919804347), CompositeDisplayCondition)
agcls.AgTypeNameMap["CompositeDisplayCondition"] = CompositeDisplayCondition

class CompositePrimitive(IPrimitive, SupportsDeleteCallback):
    """A primitive that is composed of multiple other primitives. Since composites can contain other composites, they are commonly used to build hierarchies of primitives to efficiently evaluate display conditions..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_translucent_primitives_sort_order_method_offset = 2
    _set_translucent_primitives_sort_order_method_offset = 3
    _add_method_offset = 4
    _remove_method_offset = 5
    _contains_method_offset = 6
    _clear_method_offset = 7
    _get__new_enum_method_offset = 8
    _metadata = {
        "iid_data" : (5261699596467587241, 17508946501537161350),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CompositePrimitive)
    def __iter__(self):
        """Create an iterator for the CompositePrimitive object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> typing.Any:
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of primitives in the composite."""
        return self._intf.get_property(CompositePrimitive._metadata, CompositePrimitive._get_count_metadata)

    _get_translucent_primitives_sort_order_metadata = { "offset" : _get_translucent_primitives_sort_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PrimitivesSortOrder),) }
    @property
    def translucent_primitives_sort_order(self) -> "PrimitivesSortOrder":
        """Get or set the sort order for translucent primitives in this composite. This determines a trade-off between rendering speed and quality..."""
        return self._intf.get_property(CompositePrimitive._metadata, CompositePrimitive._get_translucent_primitives_sort_order_metadata)

    _set_translucent_primitives_sort_order_metadata = { "offset" : _set_translucent_primitives_sort_order_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PrimitivesSortOrder),) }
    @translucent_primitives_sort_order.setter
    def translucent_primitives_sort_order(self, translucent_primitives_sort_order:"PrimitivesSortOrder") -> None:
        return self._intf.set_property(CompositePrimitive._metadata, CompositePrimitive._set_translucent_primitives_sort_order_metadata, translucent_primitives_sort_order)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"),) }
    def add(self, primitive:"IPrimitive") -> None:
        """Add a primitive to the composite."""
        return self._intf.invoke(CompositePrimitive._metadata, CompositePrimitive._add_metadata, primitive)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"),) }
    def remove(self, primitive:"IPrimitive") -> None:
        """Remove a primitive from the composite."""
        return self._intf.invoke(CompositePrimitive._metadata, CompositePrimitive._remove_metadata, primitive)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"), agmarshall.VariantBoolArg,) }
    def contains(self, primitive:"IPrimitive") -> bool:
        """Determine whether the composite contains a primitive."""
        return self._intf.invoke(CompositePrimitive._metadata, CompositePrimitive._contains_metadata, primitive, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all primitives from the composite."""
        return self._intf.invoke(CompositePrimitive._metadata, CompositePrimitive._clear_metadata, )

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection. The order of the primitives is not guaranteed to be the order that the primitives were added."""
        return self._intf.get_property(CompositePrimitive._metadata, CompositePrimitive._get__new_enum_metadata)

    _property_names[count] = "count"
    _property_names[translucent_primitives_sort_order] = "translucent_primitives_sort_order"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CompositePrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CompositePrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CompositePrimitive, [CompositePrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((4679944565974591928, 1696005919734520764), CompositePrimitive)
agcls.AgTypeNameMap["CompositePrimitive"] = CompositePrimitive

class ConstantDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """A display condition that evaluates to a user-defined value. This is commonly used to hide primitives by assigning to a primitive a display condition that always returns false."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_display_method_offset = 1
    _set_display_method_offset = 2
    _metadata = {
        "iid_data" : (5053686780535890043, 12360710430291808698),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ConstantDisplayCondition)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set the value the display condition evaluates to."""
        return self._intf.get_property(ConstantDisplayCondition._metadata, ConstantDisplayCondition._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(ConstantDisplayCondition._metadata, ConstantDisplayCondition._set_display_metadata, display)

    _property_names[display] = "display"

    def __init__(self, source_object=None):
        """Construct an object of type ConstantDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ConstantDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ConstantDisplayCondition, [ConstantDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((4696539255304361128, 11784871788123951251), ConstantDisplayCondition)
agcls.AgTypeNameMap["ConstantDisplayCondition"] = ConstantDisplayCondition

class DisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """When assigned to objects, such as primitives or globe overlays, display conditions are evaluated to determine if the object should be rendered."""
    def __init__(self, source_object=None):
        """Construct an object of type DisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DisplayCondition, [IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5534179541775668715, 2760636606459100605), DisplayCondition)
agcls.AgTypeNameMap["DisplayCondition"] = DisplayCondition

class DistanceDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive distance interval that determines when an object, such as a primitive, is rendered based on the distance from the camera to the object."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_distance_method_offset = 1
    _set_minimum_distance_method_offset = 2
    _get_maximum_distance_method_offset = 3
    _set_maximum_distance_method_offset = 4
    _get_minimum_distance_squared_method_offset = 5
    _get_maximum_distance_squared_method_offset = 6
    _metadata = {
        "iid_data" : (5360619169827197708, 10100252593195562917),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DistanceDisplayCondition)

    _get_minimum_distance_metadata = { "offset" : _get_minimum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance(self) -> float:
        """Get or set the minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._get_minimum_distance_metadata)

    _set_minimum_distance_metadata = { "offset" : _set_minimum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_distance.setter
    def minimum_distance(self, minimum_distance:float) -> None:
        return self._intf.set_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._set_minimum_distance_metadata, minimum_distance)

    _get_maximum_distance_metadata = { "offset" : _get_maximum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance(self) -> float:
        """Get or set the maximum distance of the inclusive distance interval. Use Double.MaxValue to ignore checking the maximum distance."""
        return self._intf.get_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._get_maximum_distance_metadata)

    _set_maximum_distance_metadata = { "offset" : _set_maximum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_distance.setter
    def maximum_distance(self, maximum_distance:float) -> None:
        return self._intf.set_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._set_maximum_distance_metadata, maximum_distance)

    _get_minimum_distance_squared_metadata = { "offset" : _get_minimum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance_squared(self) -> float:
        """Get the squared minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._get_minimum_distance_squared_metadata)

    _get_maximum_distance_squared_metadata = { "offset" : _get_maximum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance_squared(self) -> float:
        """Get the squared maximum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceDisplayCondition._metadata, DistanceDisplayCondition._get_maximum_distance_squared_metadata)

    _property_names[minimum_distance] = "minimum_distance"
    _property_names[maximum_distance] = "maximum_distance"
    _property_names[minimum_distance_squared] = "minimum_distance_squared"
    _property_names[maximum_distance_squared] = "maximum_distance_squared"

    def __init__(self, source_object=None):
        """Construct an object of type DistanceDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DistanceDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DistanceDisplayCondition, [DistanceDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((4724647536857811400, 16315685751390164157), DistanceDisplayCondition)
agcls.AgTypeNameMap["DistanceDisplayCondition"] = DistanceDisplayCondition

class DistanceToGlobeOverlayDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive distance interval that determines when an object, such as a primitive, is rendered based on the distance from the camera to the globe overlay..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_globe_overlay_method_offset = 1
    _set_globe_overlay_method_offset = 2
    _get_minimum_distance_method_offset = 3
    _set_minimum_distance_method_offset = 4
    _get_minimum_distance_squared_method_offset = 5
    _get_maximum_distance_method_offset = 6
    _set_maximum_distance_method_offset = 7
    _get_maximum_distance_squared_method_offset = 8
    _metadata = {
        "iid_data" : (5700430378497147116, 15867981441722401695),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DistanceToGlobeOverlayDisplayCondition)

    _get_globe_overlay_metadata = { "offset" : _get_globe_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def globe_overlay(self) -> "IGlobeOverlay":
        """Get or set the globe overlay associated with this instance."""
        return self._intf.get_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._get_globe_overlay_metadata)

    _set_globe_overlay_metadata = { "offset" : _set_globe_overlay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeOverlay"),) }
    @globe_overlay.setter
    def globe_overlay(self, globe_overlay:"IGlobeOverlay") -> None:
        return self._intf.set_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._set_globe_overlay_metadata, globe_overlay)

    _get_minimum_distance_metadata = { "offset" : _get_minimum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance(self) -> float:
        """Get or set the minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._get_minimum_distance_metadata)

    _set_minimum_distance_metadata = { "offset" : _set_minimum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_distance.setter
    def minimum_distance(self, minimum_distance:float) -> None:
        return self._intf.set_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._set_minimum_distance_metadata, minimum_distance)

    _get_minimum_distance_squared_metadata = { "offset" : _get_minimum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance_squared(self) -> float:
        """Get the squared minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._get_minimum_distance_squared_metadata)

    _get_maximum_distance_metadata = { "offset" : _get_maximum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance(self) -> float:
        """Get or set the maximum distance of the inclusive distance interval. Use Double.MaxValue to ignore checking the maximum distance."""
        return self._intf.get_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._get_maximum_distance_metadata)

    _set_maximum_distance_metadata = { "offset" : _set_maximum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_distance.setter
    def maximum_distance(self, maximum_distance:float) -> None:
        return self._intf.set_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._set_maximum_distance_metadata, maximum_distance)

    _get_maximum_distance_squared_metadata = { "offset" : _get_maximum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance_squared(self) -> float:
        """Get the squared maximum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToGlobeOverlayDisplayCondition._metadata, DistanceToGlobeOverlayDisplayCondition._get_maximum_distance_squared_metadata)

    _property_names[globe_overlay] = "globe_overlay"
    _property_names[minimum_distance] = "minimum_distance"
    _property_names[minimum_distance_squared] = "minimum_distance_squared"
    _property_names[maximum_distance] = "maximum_distance"
    _property_names[maximum_distance_squared] = "maximum_distance_squared"

    def __init__(self, source_object=None):
        """Construct an object of type DistanceToGlobeOverlayDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DistanceToGlobeOverlayDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DistanceToGlobeOverlayDisplayCondition, [DistanceToGlobeOverlayDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5635816365580515097, 2186556296694528415), DistanceToGlobeOverlayDisplayCondition)
agcls.AgTypeNameMap["DistanceToGlobeOverlayDisplayCondition"] = DistanceToGlobeOverlayDisplayCondition

class DistanceToPositionDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive distance interval that determines when an object, such as a primitive, is rendered based on the distance from the camera to a position defined in the given reference frame."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_distance_method_offset = 1
    _set_minimum_distance_method_offset = 2
    _get_minimum_distance_squared_method_offset = 3
    _get_maximum_distance_method_offset = 4
    _set_maximum_distance_method_offset = 5
    _get_maximum_distance_squared_method_offset = 6
    _get_position_method_offset = 7
    _set_position_method_offset = 8
    _get_reference_frame_method_offset = 9
    _set_reference_frame_method_offset = 10
    _metadata = {
        "iid_data" : (5189507309013883764, 15013292416941157531),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DistanceToPositionDisplayCondition)

    _get_minimum_distance_metadata = { "offset" : _get_minimum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance(self) -> float:
        """Get or set the minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_minimum_distance_metadata)

    _set_minimum_distance_metadata = { "offset" : _set_minimum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_distance.setter
    def minimum_distance(self, minimum_distance:float) -> None:
        return self._intf.set_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._set_minimum_distance_metadata, minimum_distance)

    _get_minimum_distance_squared_metadata = { "offset" : _get_minimum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance_squared(self) -> float:
        """Get the squared minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_minimum_distance_squared_metadata)

    _get_maximum_distance_metadata = { "offset" : _get_maximum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance(self) -> float:
        """Get or set the maximum distance of the inclusive distance interval. Use Double.MaxValue to ignore checking the maximum distance."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_maximum_distance_metadata)

    _set_maximum_distance_metadata = { "offset" : _set_maximum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_distance.setter
    def maximum_distance(self, maximum_distance:float) -> None:
        return self._intf.set_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._set_maximum_distance_metadata, maximum_distance)

    _get_maximum_distance_squared_metadata = { "offset" : _get_maximum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance_squared(self) -> float:
        """Get the squared maximum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_maximum_distance_squared_metadata)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the position used to compute the distance from the camera. The array contains the components of the position arranged in the order x, y, z."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._set_position_metadata, position)

    _get_reference_frame_metadata = { "offset" : _get_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_frame(self) -> "IVectorGeometryToolSystem":
        """Get or set the reference frame that position is defined in."""
        return self._intf.get_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._get_reference_frame_metadata)

    _set_reference_frame_metadata = { "offset" : _set_reference_frame_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"),) }
    @reference_frame.setter
    def reference_frame(self, reference_frame:"IVectorGeometryToolSystem") -> None:
        return self._intf.set_property(DistanceToPositionDisplayCondition._metadata, DistanceToPositionDisplayCondition._set_reference_frame_metadata, reference_frame)

    _property_names[minimum_distance] = "minimum_distance"
    _property_names[minimum_distance_squared] = "minimum_distance_squared"
    _property_names[maximum_distance] = "maximum_distance"
    _property_names[maximum_distance_squared] = "maximum_distance_squared"
    _property_names[position] = "position"
    _property_names[reference_frame] = "reference_frame"

    def __init__(self, source_object=None):
        """Construct an object of type DistanceToPositionDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DistanceToPositionDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DistanceToPositionDisplayCondition, [DistanceToPositionDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5302119056589105291, 9372472331107240618), DistanceToPositionDisplayCondition)
agcls.AgTypeNameMap["DistanceToPositionDisplayCondition"] = DistanceToPositionDisplayCondition

class DistanceToPrimitiveDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive distance interval that determines when an object, such as a screen overlay, is rendered based on the distance from the camera to the primitive..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_primitive_method_offset = 1
    _set_primitive_method_offset = 2
    _get_minimum_distance_method_offset = 3
    _set_minimum_distance_method_offset = 4
    _get_minimum_distance_squared_method_offset = 5
    _get_maximum_distance_method_offset = 6
    _set_maximum_distance_method_offset = 7
    _get_maximum_distance_squared_method_offset = 8
    _metadata = {
        "iid_data" : (5123944932789609666, 3575937356471366048),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DistanceToPrimitiveDisplayCondition)

    _get_primitive_metadata = { "offset" : _get_primitive_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def primitive(self) -> "IPrimitive":
        """Get or set the primitive associated with this instance."""
        return self._intf.get_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._get_primitive_metadata)

    _set_primitive_metadata = { "offset" : _set_primitive_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"),) }
    @primitive.setter
    def primitive(self, primitive:"IPrimitive") -> None:
        return self._intf.set_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._set_primitive_metadata, primitive)

    _get_minimum_distance_metadata = { "offset" : _get_minimum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance(self) -> float:
        """Get or set the minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._get_minimum_distance_metadata)

    _set_minimum_distance_metadata = { "offset" : _set_minimum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_distance.setter
    def minimum_distance(self, minimum_distance:float) -> None:
        return self._intf.set_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._set_minimum_distance_metadata, minimum_distance)

    _get_minimum_distance_squared_metadata = { "offset" : _get_minimum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_distance_squared(self) -> float:
        """Get the squared minimum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._get_minimum_distance_squared_metadata)

    _get_maximum_distance_metadata = { "offset" : _get_maximum_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance(self) -> float:
        """Get or set the maximum distance of the inclusive distance interval. Use Double.MaxValue to ignore checking the maximum distance."""
        return self._intf.get_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._get_maximum_distance_metadata)

    _set_maximum_distance_metadata = { "offset" : _set_maximum_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_distance.setter
    def maximum_distance(self, maximum_distance:float) -> None:
        return self._intf.set_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._set_maximum_distance_metadata, maximum_distance)

    _get_maximum_distance_squared_metadata = { "offset" : _get_maximum_distance_squared_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_distance_squared(self) -> float:
        """Get the squared maximum distance of the inclusive distance interval."""
        return self._intf.get_property(DistanceToPrimitiveDisplayCondition._metadata, DistanceToPrimitiveDisplayCondition._get_maximum_distance_squared_metadata)

    _property_names[primitive] = "primitive"
    _property_names[minimum_distance] = "minimum_distance"
    _property_names[minimum_distance_squared] = "minimum_distance_squared"
    _property_names[maximum_distance] = "maximum_distance"
    _property_names[maximum_distance_squared] = "maximum_distance_squared"

    def __init__(self, source_object=None):
        """Construct an object of type DistanceToPrimitiveDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DistanceToPrimitiveDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DistanceToPrimitiveDisplayCondition, [DistanceToPrimitiveDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((4865980936982214573, 11744636288922726806), DistanceToPrimitiveDisplayCondition)
agcls.AgTypeNameMap["DistanceToPrimitiveDisplayCondition"] = DistanceToPrimitiveDisplayCondition

class DurationPathPrimitiveUpdatePolicy(IPathPrimitiveUpdatePolicy, SupportsDeleteCallback):
    """path primitive update policy that removes points from remove location after a given duration."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_duration_method_offset = 1
    _set_duration_method_offset = 2
    _get_remove_location_method_offset = 3
    _set_remove_location_method_offset = 4
    _metadata = {
        "iid_data" : (5357790703849294276, 3889378876460683145),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DurationPathPrimitiveUpdatePolicy)

    _get_duration_metadata = { "offset" : _get_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def duration(self) -> float:
        """Get or set the maximum duration that a point will lie on the line."""
        return self._intf.get_property(DurationPathPrimitiveUpdatePolicy._metadata, DurationPathPrimitiveUpdatePolicy._get_duration_metadata)

    _set_duration_metadata = { "offset" : _set_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @duration.setter
    def duration(self, duration:float) -> None:
        return self._intf.set_property(DurationPathPrimitiveUpdatePolicy._metadata, DurationPathPrimitiveUpdatePolicy._set_duration_metadata, duration)

    _get_remove_location_metadata = { "offset" : _get_remove_location_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PathPrimitiveRemoveLocation),) }
    @property
    def remove_location(self) -> "PathPrimitiveRemoveLocation":
        """Get or set where the positions on the path are removed from."""
        return self._intf.get_property(DurationPathPrimitiveUpdatePolicy._metadata, DurationPathPrimitiveUpdatePolicy._get_remove_location_metadata)

    _set_remove_location_metadata = { "offset" : _set_remove_location_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PathPrimitiveRemoveLocation),) }
    @remove_location.setter
    def remove_location(self, remove_location:"PathPrimitiveRemoveLocation") -> None:
        return self._intf.set_property(DurationPathPrimitiveUpdatePolicy._metadata, DurationPathPrimitiveUpdatePolicy._set_remove_location_metadata, remove_location)

    _property_names[duration] = "duration"
    _property_names[remove_location] = "remove_location"

    def __init__(self, source_object=None):
        """Construct an object of type DurationPathPrimitiveUpdatePolicy."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DurationPathPrimitiveUpdatePolicy)
        IPathPrimitiveUpdatePolicy.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPathPrimitiveUpdatePolicy._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DurationPathPrimitiveUpdatePolicy, [DurationPathPrimitiveUpdatePolicy, IPathPrimitiveUpdatePolicy])

agcls.AgClassCatalog.add_catalog_entry((4983364944216509688, 6094502157445632944), DurationPathPrimitiveUpdatePolicy)
agcls.AgTypeNameMap["DurationPathPrimitiveUpdatePolicy"] = DurationPathPrimitiveUpdatePolicy

class FrameRate(SupportsDeleteCallback):
    """Keeps track of how many times the scenes are rendered per second."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_frames_per_second_method_offset = 1
    _get_maximum_number_of_frames_method_offset = 2
    _set_maximum_number_of_frames_method_offset = 3
    _reset_method_offset = 4
    _metadata = {
        "iid_data" : (5581082211584446583, 9818657894712909486),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, FrameRate)

    _get_frames_per_second_metadata = { "offset" : _get_frames_per_second_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def frames_per_second(self) -> float:
        """Get the current frame rate in frames per second."""
        return self._intf.get_property(FrameRate._metadata, FrameRate._get_frames_per_second_metadata)

    _get_maximum_number_of_frames_metadata = { "offset" : _get_maximum_number_of_frames_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_number_of_frames(self) -> int:
        """Get or set the maximum number of frames used to determine frame rate. More frames means that the frame rate is averaged over a longer period of time and will fluctuate less than fewer frames would. The minimum value is 2."""
        return self._intf.get_property(FrameRate._metadata, FrameRate._get_maximum_number_of_frames_metadata)

    _set_maximum_number_of_frames_metadata = { "offset" : _set_maximum_number_of_frames_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_number_of_frames.setter
    def maximum_number_of_frames(self, maximum_number_of_frames:int) -> None:
        return self._intf.set_property(FrameRate._metadata, FrameRate._set_maximum_number_of_frames_metadata, maximum_number_of_frames)

    _reset_metadata = { "offset" : _reset_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reset(self) -> None:
        """Reset the frame rate counter back to zero. The frame rate computation begins anew."""
        return self._intf.invoke(FrameRate._metadata, FrameRate._reset_metadata, )

    _property_names[frames_per_second] = "frames_per_second"
    _property_names[maximum_number_of_frames] = "maximum_number_of_frames"

    def __init__(self, source_object=None):
        """Construct an object of type FrameRate."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, FrameRate)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, FrameRate, [FrameRate, ])

agcls.AgClassCatalog.add_catalog_entry((5661427371560056203, 8596066061557213330), FrameRate)
agcls.AgTypeNameMap["FrameRate"] = FrameRate

class GlobeImageOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe overlay that shows an image."""
    def __init__(self, source_object=None):
        """Construct an object of type GlobeImageOverlay."""
        SupportsDeleteCallback.__init__(self)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GlobeImageOverlay, [IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5054499037824607794, 413485819247292294), GlobeImageOverlay)
agcls.AgTypeNameMap["GlobeImageOverlay"] = GlobeImageOverlay

class GraphicsFont(SupportsDeleteCallback):
    """A font that is suitable for use with the text batch primitive. For best performance, avoid creating duplicate font objects. Instead assign the same font object to several text batch primitives."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _get_size_method_offset = 2
    _get_bold_method_offset = 3
    _get_italic_method_offset = 4
    _get_underline_method_offset = 5
    _get_strikeout_method_offset = 6
    _get_outline_method_offset = 7
    _get_style_method_offset = 8
    _get_antialias_method_offset = 9
    _metadata = {
        "iid_data" : (4853542375175650656, 5379498553107732626),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GraphicsFont)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the typeface name of the font."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_name_metadata)

    _get_size_metadata = { "offset" : _get_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def size(self) -> int:
        """Get the size of the font."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_size_metadata)

    _get_bold_metadata = { "offset" : _get_bold_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def bold(self) -> bool:
        """Get a value that indicates whether this font is bold."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_bold_metadata)

    _get_italic_metadata = { "offset" : _get_italic_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def italic(self) -> bool:
        """Get a value that indicates whether this font is italic."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_italic_metadata)

    _get_underline_metadata = { "offset" : _get_underline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def underline(self) -> bool:
        """Get a value that indicates whether this font is underlined."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_underline_metadata)

    _get_strikeout_metadata = { "offset" : _get_strikeout_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def strikeout(self) -> bool:
        """Get a value that indicates whether this font has a horizontal line through its characters."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_strikeout_metadata)

    _get_outline_metadata = { "offset" : _get_outline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def outline(self) -> bool:
        """Get a value that indicates whether this font has an outline around its characters."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_outline_metadata)

    _get_style_metadata = { "offset" : _get_style_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FontStyle),) }
    @property
    def style(self) -> "FontStyle":
        """Get the font style enumeration that contains the style for this font. This does not include the outline property."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_style_metadata)

    _get_antialias_metadata = { "offset" : _get_antialias_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def antialias(self) -> bool:
        """Get a value that indicates whether this font is antialiased."""
        return self._intf.get_property(GraphicsFont._metadata, GraphicsFont._get_antialias_metadata)

    _property_names[name] = "name"
    _property_names[size] = "size"
    _property_names[bold] = "bold"
    _property_names[italic] = "italic"
    _property_names[underline] = "underline"
    _property_names[strikeout] = "strikeout"
    _property_names[outline] = "outline"
    _property_names[style] = "style"
    _property_names[antialias] = "antialias"

    def __init__(self, source_object=None):
        """Construct an object of type GraphicsFont."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GraphicsFont)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GraphicsFont, [GraphicsFont, ])

agcls.AgClassCatalog.add_catalog_entry((5356821517110532063, 5967199020230626694), GraphicsFont)
agcls.AgTypeNameMap["GraphicsFont"] = GraphicsFont

class GreatArcInterpolator(IPositionInterpolator, SupportsDeleteCallback):
    """The great arc interpolator computes interpolated positions along a great arc. A great arc is the shortest path between two positions on an ellipsoid."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_method_offset = 1
    _set_central_body_method_offset = 2
    _get_granularity_method_offset = 3
    _set_granularity_method_offset = 4
    _metadata = {
        "iid_data" : (5255382523329933505, 2913107983155980950),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GreatArcInterpolator)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get or set the central body used when interpolating with interpolate."""
        return self._intf.get_property(GreatArcInterpolator._metadata, GreatArcInterpolator._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(GreatArcInterpolator._metadata, GreatArcInterpolator._set_central_body_metadata, central_body)

    _get_granularity_metadata = { "offset" : _get_granularity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def granularity(self) -> float:
        """Get or set the granularity used when interpolating with interpolate. Lower granularities are more precise but create more positions."""
        return self._intf.get_property(GreatArcInterpolator._metadata, GreatArcInterpolator._get_granularity_metadata)

    _set_granularity_metadata = { "offset" : _set_granularity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @granularity.setter
    def granularity(self, granularity:float) -> None:
        return self._intf.set_property(GreatArcInterpolator._metadata, GreatArcInterpolator._set_granularity_metadata, granularity)

    _property_names[central_body] = "central_body"
    _property_names[granularity] = "granularity"

    def __init__(self, source_object=None):
        """Construct an object of type GreatArcInterpolator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GreatArcInterpolator)
        IPositionInterpolator.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPositionInterpolator._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GreatArcInterpolator, [GreatArcInterpolator, IPositionInterpolator])

agcls.AgClassCatalog.add_catalog_entry((4684714647278503789, 7032991952565857945), GreatArcInterpolator)
agcls.AgTypeNameMap["GreatArcInterpolator"] = GreatArcInterpolator

class ImageCollection(SupportsDeleteCallback):
    """A collection of globe image overlay objects."""

    _num_methods = 20
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_is_read_only_method_offset = 2
    _item_method_offset = 3
    _get__new_enum_method_offset = 4
    _contains_method_offset = 5
    _contains_uri_string_method_offset = 6
    _remove_method_offset = 7
    _clear_method_offset = 8
    _add_method_offset = 9
    _add_async_method_offset = 10
    _index_of_method_offset = 11
    _index_of_uri_string_method_offset = 12
    _add_uri_string_method_offset = 13
    _add_async_uri_string_method_offset = 14
    _swap_method_offset = 15
    _swap_by_index_method_offset = 16
    _move_method_offset = 17
    _move_by_index_method_offset = 18
    _bring_to_front_method_offset = 19
    _send_to_back_method_offset = 20
    _metadata = {
        "iid_data" : (5161567200153924179, 13845863922263759796),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ImageCollection)
    def __iter__(self):
        """Create an iterator for the ImageCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IGlobeImageOverlay":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval
    def subscribe(self) -> IImageCollectionEventHandler:
        """Return an IImageCollectionEventHandler that is subscribed to handle events associated with this instance of ImageCollection."""
        return IImageCollectionEventHandler(self._intf)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of globe overlay objects in the collection."""
        return self._intf.get_property(ImageCollection._metadata, ImageCollection._get_count_metadata)

    _get_is_read_only_metadata = { "offset" : _get_is_read_only_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_read_only(self) -> bool:
        """Get whether or not the collection is read only."""
        return self._intf.get_property(ImageCollection._metadata, ImageCollection._get_is_read_only_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "IGlobeImageOverlay":
        """Get the globe image overlay at the specified index."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Construct an iterator that can be used to iterate the collection."""
        return self._intf.get_property(ImageCollection._metadata, ImageCollection._get__new_enum_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"), agmarshall.VariantBoolArg,) }
    def contains(self, imagery_overlay:"IGlobeImageOverlay") -> bool:
        """Check the presence of a globe image overlay in the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._contains_metadata, imagery_overlay, OutArg())

    _contains_uri_string_metadata = { "offset" : _contains_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains_uri_string(self, string_uri:str) -> bool:
        """Check the presence of a globe image overlay with the specified Uri in the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._contains_uri_string_metadata, string_uri, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"), agmarshall.VariantBoolArg,) }
    def remove(self, globe_overlay:"IGlobeImageOverlay") -> bool:
        """Remove a globe image overlay from the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._remove_metadata, globe_overlay, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all globe image overlay objects from the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._clear_metadata, )

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    def add(self, imagery_overlay:"IGlobeImageOverlay") -> None:
        """Add imageryOverlay to the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._add_metadata, imagery_overlay)

    _add_async_metadata = { "offset" : _add_async_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    def add_async(self, imagery_overlay:"IGlobeImageOverlay") -> None:
        """Add imageryOverlay to the collection asynchronously."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._add_async_metadata, imagery_overlay)

    _index_of_metadata = { "offset" : _index_of_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.INT),),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"), agmarshall.IntArg,) }
    def index_of(self, imagery_overlay:"IGlobeImageOverlay") -> int:
        """Get the index of the specified globe image overlay."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._index_of_metadata, imagery_overlay, OutArg())

    _index_of_uri_string_metadata = { "offset" : _index_of_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.INT),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.IntArg,) }
    def index_of_uri_string(self, string_uri:str) -> int:
        """Get the index of the globe image overlay with the specified Uri."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._index_of_uri_string_metadata, string_uri, OutArg())

    _add_uri_string_metadata = { "offset" : _add_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_uri_string(self, uri:str) -> "IGlobeImageOverlay":
        """Create a globe overlay from the uri, which represents a uri, and adds it to the collection."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._add_uri_string_metadata, uri, OutArg())

    _add_async_uri_string_metadata = { "offset" : _add_async_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_async_uri_string(self, uri:str) -> "IGlobeImageOverlay":
        """Create a globe image overlay from the uri, which represents a uri, and adds it to the collection asynchronously."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._add_async_uri_string_metadata, uri, OutArg())

    _swap_metadata = { "offset" : _swap_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"), agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    def swap(self, imagery_overlay1:"IGlobeImageOverlay", imagery_overlay2:"IGlobeImageOverlay") -> None:
        """Swap the position of two globe image overlay objects."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._swap_metadata, imagery_overlay1, imagery_overlay2)

    _swap_by_index_metadata = { "offset" : _swap_by_index_method_offset,
            "arg_types" : (agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg,) }
    def swap_by_index(self, index1:int, index2:int) -> None:
        """Swap the position of two globe image overlay objects at the specified indices."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._swap_by_index_metadata, index1, index2)

    _move_metadata = { "offset" : _move_method_offset,
            "arg_types" : (agcom.PVOID, agcom.INT,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"), agmarshall.IntArg,) }
    def move(self, imagery_overlay:"IGlobeImageOverlay", new_position:int) -> None:
        """Move the globe image overlay to the specified position."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._move_metadata, imagery_overlay, new_position)

    _move_by_index_metadata = { "offset" : _move_by_index_method_offset,
            "arg_types" : (agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg,) }
    def move_by_index(self, index:int, new_position:int) -> None:
        """Move the globe image overlay at the specified index to the specified position."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._move_by_index_metadata, index, new_position)

    _bring_to_front_metadata = { "offset" : _bring_to_front_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    def bring_to_front(self, imagery_overlay:"IGlobeImageOverlay") -> None:
        """Brings the globe image overlay to the front of the collection so it is rendered first or on the bottom."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._bring_to_front_metadata, imagery_overlay)

    _send_to_back_metadata = { "offset" : _send_to_back_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IGlobeImageOverlay"),) }
    def send_to_back(self, imagery_overlay:"IGlobeImageOverlay") -> None:
        """Send the globe image overlay to the back of the collection so it is rendered last or on the top."""
        return self._intf.invoke(ImageCollection._metadata, ImageCollection._send_to_back_metadata, imagery_overlay)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[is_read_only] = "is_read_only"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ImageCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ImageCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ImageCollection, [ImageCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4995629646927179478, 4428875884696510368), ImageCollection)
agcls.AgTypeNameMap["ImageCollection"] = ImageCollection

class AlphaFromLuminanceFilter(IRasterFilter, SupportsDeleteCallback):
    """Add an alpha band to the source raster derived from the luminance of the raster's color bands."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5749565162954789694, 15745680047360476803),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AlphaFromLuminanceFilter)


    def __init__(self, source_object=None):
        """Construct an object of type AlphaFromLuminanceFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AlphaFromLuminanceFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AlphaFromLuminanceFilter, [AlphaFromLuminanceFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4722571570623942599, 5628231633018734501), AlphaFromLuminanceFilter)
agcls.AgTypeNameMap["AlphaFromLuminanceFilter"] = AlphaFromLuminanceFilter

class AlphaFromPixelFilter(IRasterFilter, SupportsDeleteCallback):
    """Add an alpha band to the source raster based on the value of its first pixel. All pixels in the source raster that are the same color as the first pixel will be made transparent."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5487909280219947171, 14441549139800145842),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AlphaFromPixelFilter)


    def __init__(self, source_object=None):
        """Construct an object of type AlphaFromPixelFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AlphaFromPixelFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AlphaFromPixelFilter, [AlphaFromPixelFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5429264897698608035, 15843209762042389916), AlphaFromPixelFilter)
agcls.AgTypeNameMap["AlphaFromPixelFilter"] = AlphaFromPixelFilter

class AlphaFromRasterFilter(IRasterFilter, SupportsDeleteCallback):
    """Add an alpha band to the source raster derived from the color bands or alpha of another raster. This filter can be used to apply an alpha mask to the source raster."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_raster_method_offset = 1
    _set_raster_method_offset = 2
    _metadata = {
        "iid_data" : (5071268630863441758, 8072945114305197197),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AlphaFromRasterFilter)

    _get_raster_metadata = { "offset" : _get_raster_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster(self) -> "IRaster":
        """Get or set the raster that the source raster will use to derive an alpha band."""
        return self._intf.get_property(AlphaFromRasterFilter._metadata, AlphaFromRasterFilter._get_raster_metadata)

    _set_raster_metadata = { "offset" : _set_raster_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRaster"),) }
    @raster.setter
    def raster(self, raster:"IRaster") -> None:
        return self._intf.set_property(AlphaFromRasterFilter._metadata, AlphaFromRasterFilter._set_raster_metadata, raster)

    _property_names[raster] = "raster"

    def __init__(self, source_object=None):
        """Construct an object of type AlphaFromRasterFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AlphaFromRasterFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AlphaFromRasterFilter, [AlphaFromRasterFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5547320112593478317, 12913140405057436290), AlphaFromRasterFilter)
agcls.AgTypeNameMap["AlphaFromRasterFilter"] = AlphaFromRasterFilter

class BandExtractFilter(IRasterFilter, SupportsDeleteCallback):
    """Extract a band or set of bands from the source raster. The extract format property specifies the bands and the order of the bands that will be extracted."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_extract_format_method_offset = 1
    _set_extract_format_method_offset = 2
    _metadata = {
        "iid_data" : (5553327131329707707, 6692894464708926139),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BandExtractFilter)

    _get_extract_format_metadata = { "offset" : _get_extract_format_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterFormat),) }
    @property
    def extract_format(self) -> "RasterFormat":
        """Get or set the raster format containing the bands and the order of the bands that will be extracted."""
        return self._intf.get_property(BandExtractFilter._metadata, BandExtractFilter._get_extract_format_metadata)

    _set_extract_format_metadata = { "offset" : _set_extract_format_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RasterFormat),) }
    @extract_format.setter
    def extract_format(self, extract_format:"RasterFormat") -> None:
        return self._intf.set_property(BandExtractFilter._metadata, BandExtractFilter._set_extract_format_metadata, extract_format)

    _property_names[extract_format] = "extract_format"

    def __init__(self, source_object=None):
        """Construct an object of type BandExtractFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BandExtractFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BandExtractFilter, [BandExtractFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4670582174218548926, 16305220027283324064), BandExtractFilter)
agcls.AgTypeNameMap["BandExtractFilter"] = BandExtractFilter

class BandOrderFilter(IRasterFilter, SupportsDeleteCallback):
    """Reorders or swizzles the bands of the source raster to match the band order of the raster format specified by the band order property. When maintain raster format is true, the source raster's format is maintained after swizzling."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_band_order_method_offset = 1
    _set_band_order_method_offset = 2
    _get_maintain_raster_format_method_offset = 3
    _set_maintain_raster_format_method_offset = 4
    _metadata = {
        "iid_data" : (5455804067318556677, 15241427824966143111),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BandOrderFilter)

    _get_band_order_metadata = { "offset" : _get_band_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterFormat),) }
    @property
    def band_order(self) -> "RasterFormat":
        """Get or set the raster format indicating the desired order of the bands in the source raster."""
        return self._intf.get_property(BandOrderFilter._metadata, BandOrderFilter._get_band_order_metadata)

    _set_band_order_metadata = { "offset" : _set_band_order_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RasterFormat),) }
    @band_order.setter
    def band_order(self, band_order:"RasterFormat") -> None:
        return self._intf.set_property(BandOrderFilter._metadata, BandOrderFilter._set_band_order_metadata, band_order)

    _get_maintain_raster_format_metadata = { "offset" : _get_maintain_raster_format_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def maintain_raster_format(self) -> bool:
        """Get or set a value indicating whether to maintain the source raster's format after the filter is applied. When true, the source raster's format is maintained after swizzling."""
        return self._intf.get_property(BandOrderFilter._metadata, BandOrderFilter._get_maintain_raster_format_metadata)

    _set_maintain_raster_format_metadata = { "offset" : _set_maintain_raster_format_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @maintain_raster_format.setter
    def maintain_raster_format(self, maintain_raster_format:bool) -> None:
        return self._intf.set_property(BandOrderFilter._metadata, BandOrderFilter._set_maintain_raster_format_metadata, maintain_raster_format)

    _property_names[band_order] = "band_order"
    _property_names[maintain_raster_format] = "maintain_raster_format"

    def __init__(self, source_object=None):
        """Construct an object of type BandOrderFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BandOrderFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BandOrderFilter, [BandOrderFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5508330069622562004, 4666322603866800813), BandOrderFilter)
agcls.AgTypeNameMap["BandOrderFilter"] = BandOrderFilter

class BlurFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply a convolution filter to blur or smooth the source raster. Can be used to reduce noise in the raster."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_method_offset = 1
    _set_method_method_offset = 2
    _metadata = {
        "iid_data" : (5156019592751358897, 14730170447071393964),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BlurFilter)

    _get_method_metadata = { "offset" : _get_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BlurMethod),) }
    @property
    def method(self) -> "BlurMethod":
        """Get or set the method used to blur the source raster."""
        return self._intf.get_property(BlurFilter._metadata, BlurFilter._get_method_metadata)

    _set_method_metadata = { "offset" : _set_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BlurMethod),) }
    @method.setter
    def method(self, method:"BlurMethod") -> None:
        return self._intf.set_property(BlurFilter._metadata, BlurFilter._set_method_metadata, method)

    _property_names[method] = "method"

    def __init__(self, source_object=None):
        """Construct an object of type BlurFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BlurFilter)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BlurFilter, [BlurFilter, IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5356994569384825487, 17441113390663035052), BlurFilter)
agcls.AgTypeNameMap["BlurFilter"] = BlurFilter

class BrightnessFilter(IRasterFilter, SupportsDeleteCallback):
    """Adjusts the brightness of the source raster's color bands. The adjustment to brightness is a value between -1 and 1, corresponding to least bright to most bright."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_adjustment_method_offset = 1
    _set_adjustment_method_offset = 2
    _metadata = {
        "iid_data" : (4802486208183870698, 10278011080970254984),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BrightnessFilter)

    _get_adjustment_metadata = { "offset" : _get_adjustment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def adjustment(self) -> float:
        """Get or set the brightness adjustment value for the filter. The value must be between -1 and 1, corresponding to least bright to most bright."""
        return self._intf.get_property(BrightnessFilter._metadata, BrightnessFilter._get_adjustment_metadata)

    _set_adjustment_metadata = { "offset" : _set_adjustment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @adjustment.setter
    def adjustment(self, adjustment:float) -> None:
        return self._intf.set_property(BrightnessFilter._metadata, BrightnessFilter._set_adjustment_metadata, adjustment)

    _property_names[adjustment] = "adjustment"

    def __init__(self, source_object=None):
        """Construct an object of type BrightnessFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BrightnessFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BrightnessFilter, [BrightnessFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4740712509231548782, 14535850336571086782), BrightnessFilter)
agcls.AgTypeNameMap["BrightnessFilter"] = BrightnessFilter

class ColorToLuminanceFilter(IRasterFilter, SupportsDeleteCallback):
    """Extract a luminance band derived from the color bands of the source raster."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4984200951915944335, 10697548902017290638),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ColorToLuminanceFilter)


    def __init__(self, source_object=None):
        """Construct an object of type ColorToLuminanceFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ColorToLuminanceFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ColorToLuminanceFilter, [ColorToLuminanceFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5263196667381229446, 16063846929670924173), ColorToLuminanceFilter)
agcls.AgTypeNameMap["ColorToLuminanceFilter"] = ColorToLuminanceFilter

class ContrastFilter(IRasterFilter, SupportsDeleteCallback):
    """Adjusts the contrast of the source raster. The adjustment to contrast is a value between -1 and 1, corresponding to least contrast to most contrast."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_adjustment_method_offset = 1
    _set_adjustment_method_offset = 2
    _metadata = {
        "iid_data" : (5275228079972303811, 11026900883018049925),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ContrastFilter)

    _get_adjustment_metadata = { "offset" : _get_adjustment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def adjustment(self) -> float:
        """Get or set the contrast adjustment value for the filter. The value must be between -1 and 1, corresponding to least contrast to most contrast."""
        return self._intf.get_property(ContrastFilter._metadata, ContrastFilter._get_adjustment_metadata)

    _set_adjustment_metadata = { "offset" : _set_adjustment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @adjustment.setter
    def adjustment(self, adjustment:float) -> None:
        return self._intf.set_property(ContrastFilter._metadata, ContrastFilter._set_adjustment_metadata, adjustment)

    _property_names[adjustment] = "adjustment"

    def __init__(self, source_object=None):
        """Construct an object of type ContrastFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ContrastFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ContrastFilter, [ContrastFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5610975873394682488, 7350282519019509636), ContrastFilter)
agcls.AgTypeNameMap["ContrastFilter"] = ContrastFilter

class ConvolutionFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply convolution to the source raster. Convolution is the modification of a pixel's value based on the values of its surrounding pixels. The kernel is the numerical matrix that is applied to each pixel in this process..."""
    def __init__(self, source_object=None):
        """Construct an object of type ConvolutionFilter."""
        SupportsDeleteCallback.__init__(self)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ConvolutionFilter, [IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5313620495361217438, 6605686187729587107), ConvolutionFilter)
agcls.AgTypeNameMap["ConvolutionFilter"] = ConvolutionFilter

class EdgeDetectFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply a convolution filter to detect edges in the source raster."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_method_offset = 1
    _set_method_method_offset = 2
    _metadata = {
        "iid_data" : (4622826918932764086, 170869495788315295),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, EdgeDetectFilter)

    _get_method_metadata = { "offset" : _get_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EdgeDetectMethod),) }
    @property
    def method(self) -> "EdgeDetectMethod":
        """Get or set the method used to detect edges in the source raster."""
        return self._intf.get_property(EdgeDetectFilter._metadata, EdgeDetectFilter._get_method_metadata)

    _set_method_metadata = { "offset" : _set_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EdgeDetectMethod),) }
    @method.setter
    def method(self, method:"EdgeDetectMethod") -> None:
        return self._intf.set_property(EdgeDetectFilter._metadata, EdgeDetectFilter._set_method_metadata, method)

    _property_names[method] = "method"

    def __init__(self, source_object=None):
        """Construct an object of type EdgeDetectFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, EdgeDetectFilter)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, EdgeDetectFilter, [EdgeDetectFilter, IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5684449418894848279, 6096710836454509957), EdgeDetectFilter)
agcls.AgTypeNameMap["EdgeDetectFilter"] = EdgeDetectFilter

class FilteringRasterStream(IRasterStream, IRaster, SupportsDeleteCallback):
    """A class decorator for applying a raster filter to each update of a raster stream. Can be used to apply filters to videos and other raster streams as they are updated."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_filter_method_offset = 1
    _get_stream_method_offset = 2
    _metadata = {
        "iid_data" : (4850591175353137771, 7754674969793305223),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, FilteringRasterStream)

    _get_filter_metadata = { "offset" : _get_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def filter(self) -> "IRasterFilter":
        """Get the raster filter that will be applied to the raster stream on each update."""
        return self._intf.get_property(FilteringRasterStream._metadata, FilteringRasterStream._get_filter_metadata)

    _get_stream_metadata = { "offset" : _get_stream_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def stream(self) -> "IRasterStream":
        """Get the raster stream that will have the raster filter applied on each update."""
        return self._intf.get_property(FilteringRasterStream._metadata, FilteringRasterStream._get_stream_metadata)

    _property_names[filter] = "filter"
    _property_names[stream] = "stream"

    def __init__(self, source_object=None):
        """Construct an object of type FilteringRasterStream."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, FilteringRasterStream)
        IRasterStream.__init__(self, source_object)
        IRaster.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterStream._private_init(self, intf)
        IRaster._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, FilteringRasterStream, [FilteringRasterStream, IRasterStream, IRaster])

agcls.AgClassCatalog.add_catalog_entry((5586045703451962665, 1497506555285606334), FilteringRasterStream)
agcls.AgTypeNameMap["FilteringRasterStream"] = FilteringRasterStream

class FlipFilter(IRasterFilter, SupportsDeleteCallback):
    """Flips the source raster along the given flip axis."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_flip_axis_method_offset = 1
    _set_flip_axis_method_offset = 2
    _metadata = {
        "iid_data" : (4807072714807237384, 3168786305469941428),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, FlipFilter)

    _get_flip_axis_metadata = { "offset" : _get_flip_axis_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterFlipAxis),) }
    @property
    def flip_axis(self) -> "RasterFlipAxis":
        """Get or set the flip axis along which the source raster will be flipped."""
        return self._intf.get_property(FlipFilter._metadata, FlipFilter._get_flip_axis_metadata)

    _set_flip_axis_metadata = { "offset" : _set_flip_axis_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RasterFlipAxis),) }
    @flip_axis.setter
    def flip_axis(self, flip_axis:"RasterFlipAxis") -> None:
        return self._intf.set_property(FlipFilter._metadata, FlipFilter._set_flip_axis_metadata, flip_axis)

    _property_names[flip_axis] = "flip_axis"

    def __init__(self, source_object=None):
        """Construct an object of type FlipFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, FlipFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, FlipFilter, [FlipFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5437148251484636253, 14073922228357934739), FlipFilter)
agcls.AgTypeNameMap["FlipFilter"] = FlipFilter

class GammaCorrectionFilter(IRasterFilter, SupportsDeleteCallback):
    """Apply gamma correction to the source raster. The gamma is a value between .2 and 5. The default gamma value is 2.2."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_gamma_method_offset = 1
    _set_gamma_method_offset = 2
    _metadata = {
        "iid_data" : (5172314188843020349, 6527923600942516619),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GammaCorrectionFilter)

    _get_gamma_metadata = { "offset" : _get_gamma_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def gamma(self) -> float:
        """Get or set the gamma value for the filter. The value must be between .2 and 5. The default gamma value is 2.2."""
        return self._intf.get_property(GammaCorrectionFilter._metadata, GammaCorrectionFilter._get_gamma_metadata)

    _set_gamma_metadata = { "offset" : _set_gamma_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @gamma.setter
    def gamma(self, gamma:float) -> None:
        return self._intf.set_property(GammaCorrectionFilter._metadata, GammaCorrectionFilter._set_gamma_metadata, gamma)

    _property_names[gamma] = "gamma"

    def __init__(self, source_object=None):
        """Construct an object of type GammaCorrectionFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GammaCorrectionFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GammaCorrectionFilter, [GammaCorrectionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4971206815743257794, 3457815322738687141), GammaCorrectionFilter)
agcls.AgTypeNameMap["GammaCorrectionFilter"] = GammaCorrectionFilter

class GaussianBlurFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply a convolution filter to blur the source raster using the Gaussian function."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4989874083915146590, 17595274729084466359),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GaussianBlurFilter)


    def __init__(self, source_object=None):
        """Construct an object of type GaussianBlurFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GaussianBlurFilter)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GaussianBlurFilter, [GaussianBlurFilter, IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5243991897188228892, 16020720209640064388), GaussianBlurFilter)
agcls.AgTypeNameMap["GaussianBlurFilter"] = GaussianBlurFilter

class GradientDetectFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply a convolution filter to detect gradients in the source raster."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_method_offset = 1
    _set_method_method_offset = 2
    _metadata = {
        "iid_data" : (5172471266902150476, 1346924725858697615),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, GradientDetectFilter)

    _get_method_metadata = { "offset" : _get_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(GradientDetectMethod),) }
    @property
    def method(self) -> "GradientDetectMethod":
        """Get or set the method used to detect gradients in the source raster."""
        return self._intf.get_property(GradientDetectFilter._metadata, GradientDetectFilter._get_method_metadata)

    _set_method_metadata = { "offset" : _set_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(GradientDetectMethod),) }
    @method.setter
    def method(self, method:"GradientDetectMethod") -> None:
        return self._intf.set_property(GradientDetectFilter._metadata, GradientDetectFilter._set_method_metadata, method)

    _property_names[method] = "method"

    def __init__(self, source_object=None):
        """Construct an object of type GradientDetectFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, GradientDetectFilter)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, GradientDetectFilter, [GradientDetectFilter, IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5702333424249933103, 4668764555581523625), GradientDetectFilter)
agcls.AgTypeNameMap["GradientDetectFilter"] = GradientDetectFilter

class LevelsFilter(IRasterFilter, SupportsDeleteCallback):
    """Adjusts the band levels of the source raster linearly."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_level_adjustment_method_offset = 1
    _clear_adjustments_method_offset = 2
    _metadata = {
        "iid_data" : (5146153998781700821, 7514838199004102037),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LevelsFilter)

    _set_level_adjustment_metadata = { "offset" : _set_level_adjustment_method_offset,
            "arg_types" : (agcom.LONG, agcom.INT,),
            "marshallers" : (agmarshall.EnumArg(RasterBand), agmarshall.IntArg,) }
    def set_level_adjustment(self, band:"RasterBand", adjustment:int) -> None:
        """Set the linear level adjustment value for the given raster band. Both negative and positive values are accepted."""
        return self._intf.invoke(LevelsFilter._metadata, LevelsFilter._set_level_adjustment_metadata, band, adjustment)

    _clear_adjustments_metadata = { "offset" : _clear_adjustments_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear_adjustments(self) -> None:
        """Clear the level adjustment values for all bands."""
        return self._intf.invoke(LevelsFilter._metadata, LevelsFilter._clear_adjustments_metadata, )


    def __init__(self, source_object=None):
        """Construct an object of type LevelsFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LevelsFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LevelsFilter, [LevelsFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5172734455644479740, 6694396930833260958), LevelsFilter)
agcls.AgTypeNameMap["LevelsFilter"] = LevelsFilter

class ProjectionRasterStreamPluginActivator(SupportsDeleteCallback):
    """The Activator class provides methods to load COM plugins that implement projection and raster streaming. For more information about the projection and raster plugins, see the STK Programming Interface."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_from_display_name_method_offset = 1
    _get_available_display_names_method_offset = 2
    _metadata = {
        "iid_data" : (4988091618102776374, 3631186009741722512),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProjectionRasterStreamPluginActivator)

    _create_from_display_name_metadata = { "offset" : _create_from_display_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_display_name(self, display_name:str) -> "ProjectionRasterStreamPluginProxy":
        """Load a projection/raster COM plugin associated with the specified display name and returns a proxy object that allows accessing the raster and projection streams implemented by the plugin."""
        return self._intf.invoke(ProjectionRasterStreamPluginActivator._metadata, ProjectionRasterStreamPluginActivator._create_from_display_name_metadata, display_name, OutArg())

    _get_available_display_names_metadata = { "offset" : _get_available_display_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_available_display_names(self) -> list:
        """Get a list of available projection/raster plugins' Display Names (Programmatic Identifiers)."""
        return self._intf.invoke(ProjectionRasterStreamPluginActivator._metadata, ProjectionRasterStreamPluginActivator._get_available_display_names_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type ProjectionRasterStreamPluginActivator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProjectionRasterStreamPluginActivator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProjectionRasterStreamPluginActivator, [ProjectionRasterStreamPluginActivator, ])

agcls.AgClassCatalog.add_catalog_entry((5592773854379758628, 9250413242335431090), ProjectionRasterStreamPluginActivator)
agcls.AgTypeNameMap["ProjectionRasterStreamPluginActivator"] = ProjectionRasterStreamPluginActivator

class ProjectionRasterStreamPluginProxy(SupportsDeleteCallback):
    """A proxy class provides access to the raster and projection streams implemented by a plugin. Proxies are instantiated using projection raster stream plugin activator."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_raster_stream_method_offset = 1
    _get_projection_stream_method_offset = 2
    _get_is_raster_stream_supported_method_offset = 3
    _get_is_projection_stream_supported_method_offset = 4
    _get_real_plugin_object_method_offset = 5
    _metadata = {
        "iid_data" : (5215328971719760162, 15539262404705496705),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProjectionRasterStreamPluginProxy)

    _get_raster_stream_metadata = { "offset" : _get_raster_stream_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def raster_stream(self) -> "IRasterStream":
        """Return a raster stream."""
        return self._intf.get_property(ProjectionRasterStreamPluginProxy._metadata, ProjectionRasterStreamPluginProxy._get_raster_stream_metadata)

    _get_projection_stream_metadata = { "offset" : _get_projection_stream_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def projection_stream(self) -> "ProjectionStream":
        """Return a projection stream."""
        return self._intf.get_property(ProjectionRasterStreamPluginProxy._metadata, ProjectionRasterStreamPluginProxy._get_projection_stream_metadata)

    _get_is_raster_stream_supported_metadata = { "offset" : _get_is_raster_stream_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_raster_stream_supported(self) -> bool:
        """Return true if the raster streaming is supported."""
        return self._intf.get_property(ProjectionRasterStreamPluginProxy._metadata, ProjectionRasterStreamPluginProxy._get_is_raster_stream_supported_metadata)

    _get_is_projection_stream_supported_metadata = { "offset" : _get_is_projection_stream_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_projection_stream_supported(self) -> bool:
        """Return true if the projection streaming is supported."""
        return self._intf.get_property(ProjectionRasterStreamPluginProxy._metadata, ProjectionRasterStreamPluginProxy._get_is_projection_stream_supported_metadata)

    _get_real_plugin_object_metadata = { "offset" : _get_real_plugin_object_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def real_plugin_object(self) -> typing.Any:
        """Return a pointer to plugin object's IUnknown interface."""
        return self._intf.get_property(ProjectionRasterStreamPluginProxy._metadata, ProjectionRasterStreamPluginProxy._get_real_plugin_object_metadata)

    _property_names[raster_stream] = "raster_stream"
    _property_names[projection_stream] = "projection_stream"
    _property_names[is_raster_stream_supported] = "is_raster_stream_supported"
    _property_names[is_projection_stream_supported] = "is_projection_stream_supported"
    _property_names[real_plugin_object] = "real_plugin_object"

    def __init__(self, source_object=None):
        """Construct an object of type ProjectionRasterStreamPluginProxy."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProjectionRasterStreamPluginProxy)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProjectionRasterStreamPluginProxy, [ProjectionRasterStreamPluginProxy, ])

agcls.AgClassCatalog.add_catalog_entry((4666815726913761780, 2357661832715587970), ProjectionRasterStreamPluginProxy)
agcls.AgTypeNameMap["ProjectionRasterStreamPluginProxy"] = ProjectionRasterStreamPluginProxy

class Raster(IRaster, SupportsDeleteCallback):
    """A raster dataset. A raster consists of one or more bands, or sets of values, which are most commonly associated with colors when the raster represents an image..."""
    def __init__(self, source_object=None):
        """Construct an object of type Raster."""
        SupportsDeleteCallback.__init__(self)
        IRaster.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRaster._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Raster, [IRaster])

agcls.AgClassCatalog.add_catalog_entry((5051161684986260026, 10586440812956719515), Raster)
agcls.AgTypeNameMap["Raster"] = Raster

class RasterAttributes(SupportsDeleteCallback):
    """The attributes describing a raster dataset. raster attributes define the memory layout of a raster, and includes properties defining the order of each raster band that the raster contains, as specified by the raster format..."""

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_format_method_offset = 1
    _get_type_method_offset = 2
    _get_orientation_method_offset = 3
    _get_byte_length_method_offset = 4
    _get_width_method_offset = 5
    _get_height_method_offset = 6
    _get_pixel_aspect_ratio_method_offset = 7
    _get_row_alignment_method_offset = 8
    _get_number_of_bands_method_offset = 9
    _get_row_stride_method_offset = 10
    _has_band_method_offset = 11
    _metadata = {
        "iid_data" : (4735466585165008087, 2729013748843522195),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RasterAttributes)

    _get_format_metadata = { "offset" : _get_format_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterFormat),) }
    @property
    def format(self) -> "RasterFormat":
        """Get the raster format associated with the attributes."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_format_metadata)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterType),) }
    @property
    def type(self) -> "RasterType":
        """Get the raster type associated with the attributes."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_type_metadata)

    _get_orientation_metadata = { "offset" : _get_orientation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterOrientation),) }
    @property
    def orientation(self) -> "RasterOrientation":
        """Get the raster orientation associated with the attributes."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_orientation_metadata)

    _get_byte_length_metadata = { "offset" : _get_byte_length_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def byte_length(self) -> int:
        """Get the size of the raster data in bytes."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_byte_length_metadata)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def width(self) -> int:
        """Get the width of the raster in pixels."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_width_metadata)

    _get_height_metadata = { "offset" : _get_height_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def height(self) -> int:
        """Get the height of the raster in pixels."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_height_metadata)

    _get_pixel_aspect_ratio_metadata = { "offset" : _get_pixel_aspect_ratio_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def pixel_aspect_ratio(self) -> float:
        """Get the pixel aspect ratio of the raster."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_pixel_aspect_ratio_metadata)

    _get_row_alignment_metadata = { "offset" : _get_row_alignment_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def row_alignment(self) -> int:
        """Get the row alignment of the raster data in bytes."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_row_alignment_metadata)

    _get_number_of_bands_metadata = { "offset" : _get_number_of_bands_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def number_of_bands(self) -> int:
        """Get the number of bands comprising the raster."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_number_of_bands_metadata)

    _get_row_stride_metadata = { "offset" : _get_row_stride_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def row_stride(self) -> int:
        """Get the stride or scan/row width in bytes of the raster data."""
        return self._intf.get_property(RasterAttributes._metadata, RasterAttributes._get_row_stride_metadata)

    _has_band_metadata = { "offset" : _has_band_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(RasterBand), agmarshall.VariantBoolArg,) }
    def has_band(self, band:"RasterBand") -> bool:
        """Get whether the raster contains the given band."""
        return self._intf.invoke(RasterAttributes._metadata, RasterAttributes._has_band_metadata, band, OutArg())

    _property_names[format] = "format"
    _property_names[type] = "type"
    _property_names[orientation] = "orientation"
    _property_names[byte_length] = "byte_length"
    _property_names[width] = "width"
    _property_names[height] = "height"
    _property_names[pixel_aspect_ratio] = "pixel_aspect_ratio"
    _property_names[row_alignment] = "row_alignment"
    _property_names[number_of_bands] = "number_of_bands"
    _property_names[row_stride] = "row_stride"

    def __init__(self, source_object=None):
        """Construct an object of type RasterAttributes."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RasterAttributes)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RasterAttributes, [RasterAttributes, ])

agcls.AgClassCatalog.add_catalog_entry((4877134759318171914, 3568483647780173482), RasterAttributes)
agcls.AgTypeNameMap["RasterAttributes"] = RasterAttributes

class RasterFilter(IRasterFilter, SupportsDeleteCallback):
    """A filter for processing raster datasets. RasterFilter is the base class for all raster filters..."""
    def __init__(self, source_object=None):
        """Construct an object of type RasterFilter."""
        SupportsDeleteCallback.__init__(self)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RasterFilter, [IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4815410842910981470, 4827660083801860526), RasterFilter)
agcls.AgTypeNameMap["RasterFilter"] = RasterFilter

class RasterStream(IRasterStream, IRaster, SupportsDeleteCallback):
    """A raster, the data of which, is updated dynamically at the specified update delta. The class can be used to stream video and other dynamic raster data to textures and other raster clients..."""
    def __init__(self, source_object=None):
        """Construct an object of type RasterStream."""
        SupportsDeleteCallback.__init__(self)
        IRasterStream.__init__(self, source_object)
        IRaster.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterStream._private_init(self, intf)
        IRaster._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RasterStream, [IRasterStream, IRaster])

agcls.AgClassCatalog.add_catalog_entry((5092607167961947230, 5072153283411839421), RasterStream)
agcls.AgTypeNameMap["RasterStream"] = RasterStream

class RotateFilter(IRasterFilter, SupportsDeleteCallback):
    """Rotate the source raster clockwise by the specified angle."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_angle_method_offset = 1
    _set_angle_method_offset = 2
    _metadata = {
        "iid_data" : (5284416578445493792, 4664926411814426046),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RotateFilter)

    _get_angle_metadata = { "offset" : _get_angle_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def angle(self) -> float:
        """Get or set the counterclockwise rotation angle."""
        return self._intf.get_property(RotateFilter._metadata, RotateFilter._get_angle_metadata)

    _set_angle_metadata = { "offset" : _set_angle_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @angle.setter
    def angle(self, angle:float) -> None:
        return self._intf.set_property(RotateFilter._metadata, RotateFilter._set_angle_metadata, angle)

    _property_names[angle] = "angle"

    def __init__(self, source_object=None):
        """Construct an object of type RotateFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RotateFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RotateFilter, [RotateFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5555725880336034116, 6980653760117414296), RotateFilter)
agcls.AgTypeNameMap["RotateFilter"] = RotateFilter

class SequenceFilter(IRasterFilter, SupportsDeleteCallback):
    """Apply a sequence of filters to the source raster in the order in which they were added. When continue on failure is set to true, subsequent filters will still be applied to the source raster even if one or more filters in the sequence cannot be applied."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_continue_on_failure_method_offset = 1
    _set_continue_on_failure_method_offset = 2
    _get_count_method_offset = 3
    _add_method_offset = 4
    _remove_method_offset = 5
    _clear_method_offset = 6
    _contains_method_offset = 7
    _metadata = {
        "iid_data" : (4723761315043259595, 8495024962827569328),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SequenceFilter)

    _get_continue_on_failure_metadata = { "offset" : _get_continue_on_failure_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def continue_on_failure(self) -> bool:
        """Get or set whether to continue applying filters in the sequence regardless of individual filter failures. When set to true, subsequent filters will still be applied to the source raster even if one or more filters in the sequence cannot be applied."""
        return self._intf.get_property(SequenceFilter._metadata, SequenceFilter._get_continue_on_failure_metadata)

    _set_continue_on_failure_metadata = { "offset" : _set_continue_on_failure_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @continue_on_failure.setter
    def continue_on_failure(self, continue_on_failure:bool) -> None:
        return self._intf.set_property(SequenceFilter._metadata, SequenceFilter._set_continue_on_failure_metadata, continue_on_failure)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of filters in the collection."""
        return self._intf.get_property(SequenceFilter._metadata, SequenceFilter._get_count_metadata)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRasterFilter"),) }
    def add(self, filter:"IRasterFilter") -> None:
        """Add a filter to the sequence."""
        return self._intf.invoke(SequenceFilter._metadata, SequenceFilter._add_metadata, filter)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IRasterFilter"),) }
    def remove(self, filter:"IRasterFilter") -> None:
        """Remove a filter from the sequence."""
        return self._intf.invoke(SequenceFilter._metadata, SequenceFilter._remove_metadata, filter)

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Clear all filters from the sequence."""
        return self._intf.invoke(SequenceFilter._metadata, SequenceFilter._clear_metadata, )

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IRasterFilter"), agmarshall.VariantBoolArg,) }
    def contains(self, filter:"IRasterFilter") -> bool:
        """Return true if the sequence contains the filter."""
        return self._intf.invoke(SequenceFilter._metadata, SequenceFilter._contains_metadata, filter, OutArg())

    _property_names[continue_on_failure] = "continue_on_failure"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type SequenceFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SequenceFilter)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SequenceFilter, [SequenceFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((5178454568068120612, 18196126380549294229), SequenceFilter)
agcls.AgTypeNameMap["SequenceFilter"] = SequenceFilter

class SharpenFilter(IConvolutionFilter, IRasterFilter, SupportsDeleteCallback):
    """Apply a convolution filter to increase the sharpness of the source raster."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_method_offset = 1
    _set_method_method_offset = 2
    _metadata = {
        "iid_data" : (4866048002927502413, 16947294708104642190),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SharpenFilter)

    _get_method_metadata = { "offset" : _get_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RasterSharpenMethod),) }
    @property
    def method(self) -> "RasterSharpenMethod":
        """Get or set the method used to sharpen the source raster."""
        return self._intf.get_property(SharpenFilter._metadata, SharpenFilter._get_method_metadata)

    _set_method_metadata = { "offset" : _set_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RasterSharpenMethod),) }
    @method.setter
    def method(self, method:"RasterSharpenMethod") -> None:
        return self._intf.set_property(SharpenFilter._metadata, SharpenFilter._set_method_metadata, method)

    _property_names[method] = "method"

    def __init__(self, source_object=None):
        """Construct an object of type SharpenFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SharpenFilter)
        IConvolutionFilter.__init__(self, source_object)
        IRasterFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IConvolutionFilter._private_init(self, intf)
        IRasterFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SharpenFilter, [SharpenFilter, IConvolutionFilter, IRasterFilter])

agcls.AgClassCatalog.add_catalog_entry((4732670840778244873, 7942996488696084135), SharpenFilter)
agcls.AgTypeNameMap["SharpenFilter"] = SharpenFilter

class VideoStream(IRasterStream, IRaster, SupportsDeleteCallback):
    """A raster stream that streams from a video. The video can be read from a file, or streamed from an HTTP, RTP, UDP, or TCP source. See the Video Streams Overview for a list of supported video formats and Uri usage."""

    _num_methods = 35
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_uri_method_offset = 1
    _get_playback_method_offset = 2
    _set_playback_method_offset = 3
    _get_frame_rate_method_offset = 4
    _set_frame_rate_method_offset = 5
    _get_interval_start_time_method_offset = 6
    _set_interval_start_time_method_offset = 7
    _get_interval_end_time_method_offset = 8
    _set_interval_end_time_method_offset = 9
    _get_start_time_method_offset = 10
    _set_start_time_method_offset = 11
    _get_end_time_method_offset = 12
    _set_end_time_method_offset = 13
    _get_start_frame_method_offset = 14
    _set_start_frame_method_offset = 15
    _get_end_frame_method_offset = 16
    _set_end_frame_method_offset = 17
    _get_loop_method_offset = 18
    _set_loop_method_offset = 19
    _get_is_playing_method_offset = 20
    _get_packet_acquirement_yield_time_method_offset = 21
    _set_packet_acquirement_yield_time_method_offset = 22
    _get_packet_buffer_limit_method_offset = 23
    _set_packet_buffer_limit_method_offset = 24
    _get_allow_frame_drop_method_offset = 25
    _set_allow_frame_drop_method_offset = 26
    _get_enable_audio_method_offset = 27
    _set_enable_audio_method_offset = 28
    _reinitialize_with_string_uri_method_offset = 29
    _play_method_offset = 30
    _pause_method_offset = 31
    _stop_method_offset = 32
    _reset_method_offset = 33
    _close_method_offset = 34
    _get_audio_uri_method_offset = 35
    _metadata = {
        "iid_data" : (5243502072694622494, 5354655161679499171),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, VideoStream)

    _get_uri_metadata = { "offset" : _get_uri_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def uri(self) -> str:
        """Get the uri of the video."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_uri_metadata)

    _get_playback_metadata = { "offset" : _get_playback_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VideoPlayback),) }
    @property
    def playback(self) -> "VideoPlayback":
        """Get or set the video playback mode of the video."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_playback_metadata)

    _set_playback_metadata = { "offset" : _set_playback_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VideoPlayback),) }
    @playback.setter
    def playback(self, playback:"VideoPlayback") -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_playback_metadata, playback)

    _get_frame_rate_metadata = { "offset" : _get_frame_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def frame_rate(self) -> float:
        """Get or set the frame rate of the video when the playback property is set to real time. If this property is not set, the internal framerate of the video is used."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_frame_rate_metadata)

    _set_frame_rate_metadata = { "offset" : _set_frame_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @frame_rate.setter
    def frame_rate(self, frame_rate:float) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_frame_rate_metadata, frame_rate)

    _get_interval_start_time_metadata = { "offset" : _get_interval_start_time_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval_start_time(self) -> "Date":
        """Get or set the scene manager time at which the video will begin playing when the playback property is set to time interval."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_interval_start_time_metadata)

    _set_interval_start_time_metadata = { "offset" : _set_interval_start_time_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Date"),) }
    @interval_start_time.setter
    def interval_start_time(self, interval_start_time:"Date") -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_interval_start_time_metadata, interval_start_time)

    _get_interval_end_time_metadata = { "offset" : _get_interval_end_time_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval_end_time(self) -> "Date":
        """Get or set the scene manager time at which the video will stop playing when the playback property is set to time interval."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_interval_end_time_metadata)

    _set_interval_end_time_metadata = { "offset" : _set_interval_end_time_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Date"),) }
    @interval_end_time.setter
    def interval_end_time(self, interval_end_time:"Date") -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_interval_end_time_metadata, interval_end_time)

    _get_start_time_metadata = { "offset" : _get_start_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_time(self) -> float:
        """Get or set the start time of the video in seconds. Changing the start time property also changes the start frame property to the frame in the video that corresponds to the specified time."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_start_time_metadata)

    _set_start_time_metadata = { "offset" : _set_start_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_time.setter
    def start_time(self, start_time:float) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_start_time_metadata, start_time)

    _get_end_time_metadata = { "offset" : _get_end_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def end_time(self) -> float:
        """Get or set the end time of the video in seconds. Changing the end time property also changes the end frame property to the frame in the video that corresponds to the specified time."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_end_time_metadata)

    _set_end_time_metadata = { "offset" : _set_end_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @end_time.setter
    def end_time(self, end_time:float) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_end_time_metadata, end_time)

    _get_start_frame_metadata = { "offset" : _get_start_frame_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def start_frame(self) -> int:
        """Get or set the start frame of the video. Changing the start frame property also changes the start time property to the time in the video that corresponds to the specified frame."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_start_frame_metadata)

    _set_start_frame_metadata = { "offset" : _set_start_frame_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @start_frame.setter
    def start_frame(self, start_frame:int) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_start_frame_metadata, start_frame)

    _get_end_frame_metadata = { "offset" : _get_end_frame_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def end_frame(self) -> int:
        """Get or set the end frame of the video. Changing the end frame property also changes the end time property to the time in the video that corresponds to the specified frame."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_end_frame_metadata)

    _set_end_frame_metadata = { "offset" : _set_end_frame_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @end_frame.setter
    def end_frame(self, end_frame:int) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_end_frame_metadata, end_frame)

    _get_loop_metadata = { "offset" : _get_loop_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def loop(self) -> bool:
        """Get or set whether the video will loop when it reaches its last frame when the playback property is set to real time."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_loop_metadata)

    _set_loop_metadata = { "offset" : _set_loop_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @loop.setter
    def loop(self, loop:bool) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_loop_metadata, loop)

    _get_is_playing_metadata = { "offset" : _get_is_playing_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_playing(self) -> bool:
        """Get whether or not the video is playing. Use the play, pause, stop, and Reset methods to control the playback of the video when the playback property is set to real time."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_is_playing_metadata)

    _get_packet_acquirement_yield_time_metadata = { "offset" : _get_packet_acquirement_yield_time_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def packet_acquirement_yield_time(self) -> int:
        """Get or set the thread processing yield time for asynchronous streaming of video over common protocols like udp. Setting a high value may increase performance, but may cause frames or packets to drop, effecting visual quality..."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_packet_acquirement_yield_time_metadata)

    _set_packet_acquirement_yield_time_metadata = { "offset" : _set_packet_acquirement_yield_time_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @packet_acquirement_yield_time.setter
    def packet_acquirement_yield_time(self, packet_acquirement_yield_time:int) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_packet_acquirement_yield_time_metadata, packet_acquirement_yield_time)

    _get_packet_buffer_limit_metadata = { "offset" : _get_packet_buffer_limit_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def packet_buffer_limit(self) -> int:
        """Get or set a value indicating the buffering limit for packets when processing a video stream..."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_packet_buffer_limit_metadata)

    _set_packet_buffer_limit_metadata = { "offset" : _set_packet_buffer_limit_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @packet_buffer_limit.setter
    def packet_buffer_limit(self, packet_buffer_limit:int) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_packet_buffer_limit_metadata, packet_buffer_limit)

    _get_allow_frame_drop_metadata = { "offset" : _get_allow_frame_drop_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def allow_frame_drop(self) -> bool:
        """Get or set a value indicating if frames should be dropped if video processing can not keep up with a video stream. If this is set to false, the video quality may degrade when the processing load is too high..."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_allow_frame_drop_metadata)

    _set_allow_frame_drop_metadata = { "offset" : _set_allow_frame_drop_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @allow_frame_drop.setter
    def allow_frame_drop(self, allow_frame_drop:bool) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_allow_frame_drop_metadata, allow_frame_drop)

    _get_enable_audio_metadata = { "offset" : _get_enable_audio_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_audio(self) -> bool:
        """Get or set a value indicating if the encoded audio stream should be synchronized to video playback If this is set to false, the audio stream will be disabled..."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_enable_audio_metadata)

    _set_enable_audio_metadata = { "offset" : _set_enable_audio_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_audio.setter
    def enable_audio(self, enable_audio:bool) -> None:
        return self._intf.set_property(VideoStream._metadata, VideoStream._set_enable_audio_metadata, enable_audio)

    _reinitialize_with_string_uri_metadata = { "offset" : _reinitialize_with_string_uri_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def reinitialize_with_string_uri(self, uri:str) -> None:
        """Reinitializes the video stream from a Uri, which can be a file, HTTP, RTP, UDP, or TCP source. See the Video Streams Overview for a list of supported video formats and Uri usage."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._reinitialize_with_string_uri_metadata, uri)

    _play_metadata = { "offset" : _play_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def play(self) -> None:
        """Begins playing the video when the playback property is set to real time."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._play_metadata, )

    _pause_metadata = { "offset" : _pause_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def pause(self) -> None:
        """Pauses the video when the playback property is set to real time."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._pause_metadata, )

    _stop_metadata = { "offset" : _stop_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def stop(self) -> None:
        """Stop the video when the playback property is set to real time. Stopping the video will seek to the first frame and pause playback. Use the Play method to begin playing the video again."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._stop_metadata, )

    _reset_metadata = { "offset" : _reset_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reset(self) -> None:
        """Seeks the video to its first frame and begins playing the video when the playback property is set to real time."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._reset_metadata, )

    _close_metadata = { "offset" : _close_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def close(self) -> None:
        """Close the video stream and any associated resources."""
        return self._intf.invoke(VideoStream._metadata, VideoStream._close_metadata, )

    _get_audio_uri_metadata = { "offset" : _get_audio_uri_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def audio_uri(self) -> str:
        """Get the uri of the audio."""
        return self._intf.get_property(VideoStream._metadata, VideoStream._get_audio_uri_metadata)

    _property_names[uri] = "uri"
    _property_names[playback] = "playback"
    _property_names[frame_rate] = "frame_rate"
    _property_names[interval_start_time] = "interval_start_time"
    _property_names[interval_end_time] = "interval_end_time"
    _property_names[start_time] = "start_time"
    _property_names[end_time] = "end_time"
    _property_names[start_frame] = "start_frame"
    _property_names[end_frame] = "end_frame"
    _property_names[loop] = "loop"
    _property_names[is_playing] = "is_playing"
    _property_names[packet_acquirement_yield_time] = "packet_acquirement_yield_time"
    _property_names[packet_buffer_limit] = "packet_buffer_limit"
    _property_names[allow_frame_drop] = "allow_frame_drop"
    _property_names[enable_audio] = "enable_audio"
    _property_names[audio_uri] = "audio_uri"

    def __init__(self, source_object=None):
        """Construct an object of type VideoStream."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, VideoStream)
        IRasterStream.__init__(self, source_object)
        IRaster.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRasterStream._private_init(self, intf)
        IRaster._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, VideoStream, [VideoStream, IRasterStream, IRaster])

agcls.AgClassCatalog.add_catalog_entry((5264945989350337942, 5699888751394667448), VideoStream)
agcls.AgTypeNameMap["VideoStream"] = VideoStream

class KmlContainer(IKmlContainer, IKmlFeature, SupportsDeleteCallback):
    """A KmlContainer contains a collection of children kml features."""
    def __init__(self, source_object=None):
        """Construct an object of type KmlContainer."""
        SupportsDeleteCallback.__init__(self)
        IKmlContainer.__init__(self, source_object)
        IKmlFeature.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IKmlContainer._private_init(self, intf)
        IKmlFeature._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlContainer, [IKmlContainer, IKmlFeature])

agcls.AgClassCatalog.add_catalog_entry((5207640554160657301, 13820029748964872873), KmlContainer)
agcls.AgTypeNameMap["KmlContainer"] = KmlContainer

class KmlDocument(IKmlContainer, IKmlFeature, SupportsDeleteCallback):
    """A KML document."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_uri_method_offset = 1
    _metadata = {
        "iid_data" : (5741031632277098238, 15148679251139210129),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlDocument)

    _get_uri_metadata = { "offset" : _get_uri_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def uri(self) -> str:
        """Get the uri of the document."""
        return self._intf.get_property(KmlDocument._metadata, KmlDocument._get_uri_metadata)

    _property_names[uri] = "uri"

    def __init__(self, source_object=None):
        """Construct an object of type KmlDocument."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlDocument)
        IKmlContainer.__init__(self, source_object)
        IKmlFeature.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IKmlContainer._private_init(self, intf)
        IKmlFeature._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlDocument, [KmlDocument, IKmlContainer, IKmlFeature])

agcls.AgClassCatalog.add_catalog_entry((5491088363673258432, 7394732558227195802), KmlDocument)
agcls.AgTypeNameMap["KmlDocument"] = KmlDocument

class KmlFeature(IKmlFeature, SupportsDeleteCallback):
    """A KML feature."""
    def __init__(self, source_object=None):
        """Construct an object of type KmlFeature."""
        SupportsDeleteCallback.__init__(self)
        IKmlFeature.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IKmlFeature._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlFeature, [IKmlFeature])

agcls.AgClassCatalog.add_catalog_entry((5287385400858546424, 10878200550364879277), KmlFeature)
agcls.AgTypeNameMap["KmlFeature"] = KmlFeature

class KmlFolder(IKmlContainer, IKmlFeature, SupportsDeleteCallback):
    """A KML folder."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5086621492657184689, 2300093376995260832),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlFolder)


    def __init__(self, source_object=None):
        """Construct an object of type KmlFolder."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlFolder)
        IKmlContainer.__init__(self, source_object)
        IKmlFeature.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IKmlContainer._private_init(self, intf)
        IKmlFeature._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlFolder, [KmlFolder, IKmlContainer, IKmlFeature])

agcls.AgClassCatalog.add_catalog_entry((4798630207118252685, 15038644232865021358), KmlFolder)
agcls.AgTypeNameMap["KmlFolder"] = KmlFolder

class KmlGraphics(SupportsDeleteCallback):
    """Provide loading and unloading of kml documents for a particular central body."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_documents_method_offset = 1
    _load_document_method_offset = 2
    _load_document_string_method_offset = 3
    _load_method_offset = 4
    _load_document_async_method_offset = 5
    _load_document_async_string_method_offset = 6
    _load_async_method_offset = 7
    _unload_method_offset = 8
    _unload_all_method_offset = 9
    _metadata = {
        "iid_data" : (5576095172211422984, 18055224525665917108),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlGraphics)
    def subscribe(self) -> IKmlGraphicsEventHandler:
        """Return an IKmlGraphicsEventHandler that is subscribed to handle events associated with this instance of KmlGraphics."""
        return IKmlGraphicsEventHandler(self._intf)

    _get_documents_metadata = { "offset" : _get_documents_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def documents(self) -> "KmlDocumentCollection":
        """The collection of kml documents that are currently loaded."""
        return self._intf.get_property(KmlGraphics._metadata, KmlGraphics._get_documents_metadata)

    _load_document_metadata = { "offset" : _load_document_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def load_document(self, kml_uri:str) -> "KmlDocument":
        """Load a kml document from a uri."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_document_metadata, kml_uri, OutArg())

    _load_document_string_metadata = { "offset" : _load_document_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def load_document_string(self, kml_uri:str) -> "KmlDocument":
        """Load a kml document from a Uri."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_document_string_metadata, kml_uri, OutArg())

    _load_metadata = { "offset" : _load_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def load(self, kml_document:str) -> "KmlDocument":
        """Load a kml document from a string containing the document."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_metadata, kml_document, OutArg())

    _load_document_async_metadata = { "offset" : _load_document_async_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def load_document_async(self, kml_uri:str) -> None:
        """Load a kml document asynchronously from a uri. The document loaded event is raised when the document is loaded."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_document_async_metadata, kml_uri)

    _load_document_async_string_metadata = { "offset" : _load_document_async_string_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def load_document_async_string(self, kml_uri:str) -> None:
        """Load a kml document asynchronously from a Uri. The document loaded event is raised when the document is loaded."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_document_async_string_metadata, kml_uri)

    _load_async_metadata = { "offset" : _load_async_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def load_async(self, kml_document:str) -> None:
        """Load a kml document asynchronously from a string containing the document. The document loaded event is raised when the document is loaded."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._load_async_metadata, kml_document)

    _unload_metadata = { "offset" : _unload_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("KmlDocument"),) }
    def unload(self, kml_document:"KmlDocument") -> None:
        """Unloads a kml document. All associated visual features will be removed from the Scene. Once a KmlDocument is unloaded, it is no longer valid and will throw when accessing properties or methods."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._unload_metadata, kml_document)

    _unload_all_metadata = { "offset" : _unload_all_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def unload_all(self) -> None:
        """Unloads all kml documents associated with this central body. All associated visual features will be removed from the Scene. Once a KmlDocument is unloaded, it is no longer valid and will throw when accessing properties or methods."""
        return self._intf.invoke(KmlGraphics._metadata, KmlGraphics._unload_all_metadata, )

    _property_names[documents] = "documents"

    def __init__(self, source_object=None):
        """Construct an object of type KmlGraphics."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlGraphics)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlGraphics, [KmlGraphics, ])

agcls.AgClassCatalog.add_catalog_entry((4645898364801840606, 1292503210015328640), KmlGraphics)
agcls.AgTypeNameMap["KmlGraphics"] = KmlGraphics

class KmlNetworkLink(IKmlFeature, SupportsDeleteCallback):
    """A KML network link."""

    _num_methods = 17
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_uri_method_offset = 1
    _get_refresh_mode_method_offset = 2
    _set_refresh_mode_method_offset = 3
    _get_refresh_interval_method_offset = 4
    _set_refresh_interval_method_offset = 5
    _get_view_refresh_mode_method_offset = 6
    _set_view_refresh_mode_method_offset = 7
    _get_view_refresh_time_method_offset = 8
    _set_view_refresh_time_method_offset = 9
    _get_view_bound_scale_method_offset = 10
    _set_view_bound_scale_method_offset = 11
    _get_minimum_refresh_period_method_offset = 12
    _get_cookie_method_offset = 13
    _get_message_method_offset = 14
    _get_link_snippet_method_offset = 15
    _get_expires_method_offset = 16
    _refresh_method_offset = 17
    _metadata = {
        "iid_data" : (4922395965834848898, 9735513844274272176),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, KmlNetworkLink)

    _get_uri_metadata = { "offset" : _get_uri_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def uri(self) -> str:
        """Get the uri of the network link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_uri_metadata)

    _get_refresh_mode_metadata = { "offset" : _get_refresh_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(KmlNetworkLinkRefreshMode),) }
    @property
    def refresh_mode(self) -> "KmlNetworkLinkRefreshMode":
        """Get or set the refresh mode of the network link. The refresh mode specifies a time-based refresh policy for this link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_refresh_mode_metadata)

    _set_refresh_mode_metadata = { "offset" : _set_refresh_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(KmlNetworkLinkRefreshMode),) }
    @refresh_mode.setter
    def refresh_mode(self, refresh_mode:"KmlNetworkLinkRefreshMode") -> None:
        return self._intf.set_property(KmlNetworkLink._metadata, KmlNetworkLink._set_refresh_mode_metadata, refresh_mode)

    _get_refresh_interval_metadata = { "offset" : _get_refresh_interval_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def refresh_interval(self) -> float:
        """Get or set the interval duration at which this network link will refresh, when refresh mode is set to on interval."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_refresh_interval_metadata)

    _set_refresh_interval_metadata = { "offset" : _set_refresh_interval_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @refresh_interval.setter
    def refresh_interval(self, refresh_interval:float) -> None:
        return self._intf.set_property(KmlNetworkLink._metadata, KmlNetworkLink._set_refresh_interval_metadata, refresh_interval)

    _get_view_refresh_mode_metadata = { "offset" : _get_view_refresh_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(KmlNetworkLinkViewRefreshMode),) }
    @property
    def view_refresh_mode(self) -> "KmlNetworkLinkViewRefreshMode":
        """Get or set the view refresh mode of the network link. The view refresh mode specifies the refresh policy for the when the camera's view changes."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_view_refresh_mode_metadata)

    _set_view_refresh_mode_metadata = { "offset" : _set_view_refresh_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(KmlNetworkLinkViewRefreshMode),) }
    @view_refresh_mode.setter
    def view_refresh_mode(self, view_refresh_mode:"KmlNetworkLinkViewRefreshMode") -> None:
        return self._intf.set_property(KmlNetworkLink._metadata, KmlNetworkLink._set_view_refresh_mode_metadata, view_refresh_mode)

    _get_view_refresh_time_metadata = { "offset" : _get_view_refresh_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def view_refresh_time(self) -> float:
        """Get or set the duration after camera view movement stops that this network link will refresh, when view refresh mode is set to on stop."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_view_refresh_time_metadata)

    _set_view_refresh_time_metadata = { "offset" : _set_view_refresh_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @view_refresh_time.setter
    def view_refresh_time(self, view_refresh_time:float) -> None:
        return self._intf.set_property(KmlNetworkLink._metadata, KmlNetworkLink._set_view_refresh_time_metadata, view_refresh_time)

    _get_view_bound_scale_metadata = { "offset" : _get_view_bound_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def view_bound_scale(self) -> float:
        """Get or set the value that scales the bounding box defining the view associated with this network link. A value less than 1.0 specifies to use less than the full view (screen). A value greater than 1..."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_view_bound_scale_metadata)

    _set_view_bound_scale_metadata = { "offset" : _set_view_bound_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @view_bound_scale.setter
    def view_bound_scale(self, view_bound_scale:float) -> None:
        return self._intf.set_property(KmlNetworkLink._metadata, KmlNetworkLink._set_view_bound_scale_metadata, view_bound_scale)

    _get_minimum_refresh_period_metadata = { "offset" : _get_minimum_refresh_period_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_refresh_period(self) -> float:
        """Get the duration that is the minimum allowed time between refreshes of this network link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_minimum_refresh_period_metadata)

    _get_cookie_metadata = { "offset" : _get_cookie_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def cookie(self) -> str:
        """Get the cookie string associated with this network link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_cookie_metadata)

    _get_message_metadata = { "offset" : _get_message_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def message(self) -> str:
        """Get the message string associated with this network link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_message_metadata)

    _get_link_snippet_metadata = { "offset" : _get_link_snippet_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def link_snippet(self) -> str:
        """Get the link snippet associated with this network link."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_link_snippet_metadata)

    _get_expires_metadata = { "offset" : _get_expires_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def expires(self) -> str:
        """Get the string specifying the date/time this network should expire and be refreshed."""
        return self._intf.get_property(KmlNetworkLink._metadata, KmlNetworkLink._get_expires_metadata)

    _refresh_metadata = { "offset" : _refresh_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def refresh(self) -> None:
        """Refresh the network link."""
        return self._intf.invoke(KmlNetworkLink._metadata, KmlNetworkLink._refresh_metadata, )

    _property_names[uri] = "uri"
    _property_names[refresh_mode] = "refresh_mode"
    _property_names[refresh_interval] = "refresh_interval"
    _property_names[view_refresh_mode] = "view_refresh_mode"
    _property_names[view_refresh_time] = "view_refresh_time"
    _property_names[view_bound_scale] = "view_bound_scale"
    _property_names[minimum_refresh_period] = "minimum_refresh_period"
    _property_names[cookie] = "cookie"
    _property_names[message] = "message"
    _property_names[link_snippet] = "link_snippet"
    _property_names[expires] = "expires"

    def __init__(self, source_object=None):
        """Construct an object of type KmlNetworkLink."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, KmlNetworkLink)
        IKmlFeature.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IKmlFeature._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, KmlNetworkLink, [KmlNetworkLink, IKmlFeature])

agcls.AgClassCatalog.add_catalog_entry((5504684742633507453, 16929891538998501770), KmlNetworkLink)
agcls.AgTypeNameMap["KmlNetworkLink"] = KmlNetworkLink

class MarkerBatchPrimitive(IPrimitive, SupportsDeleteCallback):
    """Render one or more markers in the 3D scene. Markers are 2D images that always face the viewer which can be sized in pixels or meters. Markers are also referred to as sprites or billboards..."""

    _num_methods = 54
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_size_source_method_offset = 1
    _get_sort_order_method_offset = 2
    _get_set_hint_method_offset = 3
    _get_rendering_method_method_offset = 4
    _get_render_pass_method_offset = 5
    _set_render_pass_method_offset = 6
    _get_bounding_sphere_scale_method_offset = 7
    _set_bounding_sphere_scale_method_offset = 8
    _get_distance_display_condition_per_marker_method_offset = 9
    _set_distance_display_condition_per_marker_method_offset = 10
    _get_texture_method_offset = 11
    _set_texture_method_offset = 12
    _get_size_unit_method_offset = 13
    _set_size_unit_method_offset = 14
    _get_size_method_offset = 15
    _set_size_method_offset = 16
    _get_origin_method_offset = 17
    _set_origin_method_offset = 18
    _get_pixel_offset_method_offset = 19
    _set_pixel_offset_method_offset = 20
    _get_eye_offset_method_offset = 21
    _set_eye_offset_method_offset = 22
    _get_rotation_method_offset = 23
    _set_rotation_method_offset = 24
    _get_texture_coordinate_method_offset = 25
    _set_texture_coordinate_method_offset = 26
    _get_wireframe_method_offset = 27
    _set_wireframe_method_offset = 28
    _get_per_item_picking_enabled_method_offset = 29
    _set_per_item_picking_enabled_method_offset = 30
    _get_texture_filter_method_offset = 31
    _set_texture_filter_method_offset = 32
    _set_method_offset = 33
    _set_with_optional_parameters_method_offset = 34
    _set_with_optional_parameters_and_render_pass_hint_method_offset = 35
    _set_cartographic_method_offset = 36
    _set_cartographic_with_optional_parameters_method_offset = 37
    _set_cartographic_with_optional_parameters_and_render_pass_hint_method_offset = 38
    _set_partial_method_offset = 39
    _set_partial_with_indices_order_method_offset = 40
    _set_partial_with_optional_parameters_method_offset = 41
    _set_partial_with_optional_parameters_indices_order_and_render_pass_method_offset = 42
    _set_partial_cartographic_method_offset = 43
    _set_partial_cartographic_with_indices_order_method_offset = 44
    _set_partial_cartographic_with_optional_parameters_method_offset = 45
    _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_method_offset = 46
    _supported_method_offset = 47
    _get_clamp_to_pixel_method_offset = 48
    _set_clamp_to_pixel_method_offset = 49
    _get_central_body_clipped_method_offset = 50
    _set_central_body_clipped_method_offset = 51
    _align_to_screen_method_offset = 52
    _align_to_north_method_offset = 53
    _align_to_axis_method_offset = 54
    _metadata = {
        "iid_data" : (5541225076266399937, 16789973218996649861),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MarkerBatchPrimitive)

    _get_size_source_metadata = { "offset" : _get_size_source_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchSizeSource),) }
    @property
    def size_source(self) -> "MarkerBatchSizeSource":
        """Get the source used for the size of markers in the batch."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_size_source_metadata)

    _get_sort_order_metadata = { "offset" : _get_sort_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchSortOrder),) }
    @property
    def sort_order(self) -> "MarkerBatchSortOrder":
        """Get the order in which markers in the marker batch are sorted before rendering."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_sort_order_metadata)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_set_hint_metadata)

    _get_rendering_method_metadata = { "offset" : _get_rendering_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchRenderingMethod),) }
    @property
    def rendering_method(self) -> "MarkerBatchRenderingMethod":
        """Get the rendering method used to render the marker batch."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_rendering_method_metadata)

    _get_render_pass_metadata = { "offset" : _get_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchRenderPass),) }
    @property
    def render_pass(self) -> "MarkerBatchRenderPass":
        """Get or set the pass during which the marker batch is rendered."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_render_pass_metadata)

    _set_render_pass_metadata = { "offset" : _set_render_pass_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchRenderPass),) }
    @render_pass.setter
    def render_pass(self, render_pass:"MarkerBatchRenderPass") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_render_pass_metadata, render_pass)

    _get_bounding_sphere_scale_metadata = { "offset" : _get_bounding_sphere_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def bounding_sphere_scale(self) -> float:
        """Get or set the scale applied to the radius of this primitive's bounding sphere."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_bounding_sphere_scale_metadata)

    _set_bounding_sphere_scale_metadata = { "offset" : _set_bounding_sphere_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @bounding_sphere_scale.setter
    def bounding_sphere_scale(self, bounding_sphere_scale:float) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_bounding_sphere_scale_metadata, bounding_sphere_scale)

    _get_distance_display_condition_per_marker_metadata = { "offset" : _get_distance_display_condition_per_marker_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_display_condition_per_marker(self) -> "DistanceDisplayCondition":
        """Get or set a distance display condition that is evaluated per marker in the marker batch during rendering. This is different than display condition, which is evaluated once for the entire marker batch..."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_distance_display_condition_per_marker_metadata)

    _set_distance_display_condition_per_marker_metadata = { "offset" : _set_distance_display_condition_per_marker_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("DistanceDisplayCondition"),) }
    @distance_display_condition_per_marker.setter
    def distance_display_condition_per_marker(self, distance_display_condition_per_marker:"DistanceDisplayCondition") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_distance_display_condition_per_marker_metadata, distance_display_condition_per_marker)

    _get_texture_metadata = { "offset" : _get_texture_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture(self) -> "RendererTexture2D":
        """Get or set the per-batch texture, which is applied to each marker in the batch."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_texture_metadata)

    _set_texture_metadata = { "offset" : _set_texture_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("RendererTexture2D"),) }
    @texture.setter
    def texture(self, texture:"RendererTexture2D") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_texture_metadata, texture)

    _get_size_unit_metadata = { "offset" : _get_size_unit_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchSizeUnit),) }
    @property
    def size_unit(self) -> "MarkerBatchSizeUnit":
        """Get or set the unit that each marker's size is defined in."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_size_unit_metadata)

    _set_size_unit_metadata = { "offset" : _set_size_unit_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchSizeUnit),) }
    @size_unit.setter
    def size_unit(self, size_unit:"MarkerBatchSizeUnit") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_size_unit_metadata, size_unit)

    _get_size_metadata = { "offset" : _get_size_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def size(self) -> list:
        """Get or set the per-batch size, which is applied to each marker in the batch. The array contains one width followed by one height."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_size_metadata)

    _set_size_metadata = { "offset" : _set_size_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @size.setter
    def size(self, size:list) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_size_metadata, size)

    _get_origin_metadata = { "offset" : _get_origin_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Origin),) }
    @property
    def origin(self) -> "Origin":
        """Get or set the per-batch origin, which is applied to each marker in the batch."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_origin_metadata)

    _set_origin_metadata = { "offset" : _set_origin_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(Origin),) }
    @origin.setter
    def origin(self, origin:"Origin") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_origin_metadata, origin)

    _get_pixel_offset_metadata = { "offset" : _get_pixel_offset_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def pixel_offset(self) -> list:
        """Get or set the per-batch pixel offset, which is applied to each marker in the batch. The array contains one x pixel offset followed by one y pixel offset."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_pixel_offset_metadata)

    _set_pixel_offset_metadata = { "offset" : _set_pixel_offset_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @pixel_offset.setter
    def pixel_offset(self, pixel_offset:list) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_pixel_offset_metadata, pixel_offset)

    _get_eye_offset_metadata = { "offset" : _get_eye_offset_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def eye_offset(self) -> list:
        """Get or set the per-batch eye offset, which is applied to each marker in the batch. The array contains the components of the eye offset in the order x, y, z."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_eye_offset_metadata)

    _set_eye_offset_metadata = { "offset" : _set_eye_offset_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @eye_offset.setter
    def eye_offset(self, eye_offset:list) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_eye_offset_metadata, eye_offset)

    _get_rotation_metadata = { "offset" : _get_rotation_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def rotation(self) -> float:
        """Get or set the per-batch rotation angle which is applied to each marker in the batch."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_rotation_metadata)

    _set_rotation_metadata = { "offset" : _set_rotation_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @rotation.setter
    def rotation(self, rotation:float) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_rotation_metadata, rotation)

    _get_texture_coordinate_metadata = { "offset" : _get_texture_coordinate_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def texture_coordinate(self) -> list:
        """Get or set the per-batch texture coordinate, which is applied to each marker in the batch. The array contains the texture coordinates arranged in the order s, t, p, q."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_texture_coordinate_metadata)

    _set_texture_coordinate_metadata = { "offset" : _set_texture_coordinate_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @texture_coordinate.setter
    def texture_coordinate(self, texture_coordinate:list) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_texture_coordinate_metadata, texture_coordinate)

    _get_wireframe_metadata = { "offset" : _get_wireframe_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def wireframe(self) -> bool:
        """Get or set whether the primitive is rendered in wireframe. This is useful for debugging."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_wireframe_metadata)

    _set_wireframe_metadata = { "offset" : _set_wireframe_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @wireframe.setter
    def wireframe(self, wireframe:bool) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_wireframe_metadata, wireframe)

    _get_per_item_picking_enabled_metadata = { "offset" : _get_per_item_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def per_item_picking_enabled(self) -> bool:
        """Get or set whether individual marker indices will be included in the pick results returned from the scene's Pick method. Each marker index that is picked will be returned as a batch primitive index."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_per_item_picking_enabled_metadata)

    _set_per_item_picking_enabled_metadata = { "offset" : _set_per_item_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @per_item_picking_enabled.setter
    def per_item_picking_enabled(self, per_item_picking_enabled:bool) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_per_item_picking_enabled_metadata, per_item_picking_enabled)

    _get_texture_filter_metadata = { "offset" : _get_texture_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_filter(self) -> "TextureFilter2D":
        """Get or set the filter used for per-marker or per-batch textures."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_texture_filter_metadata)

    _set_texture_filter_metadata = { "offset" : _set_texture_filter_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TextureFilter2D"),) }
    @texture_filter.setter
    def texture_filter(self, texture_filter:"TextureFilter2D") -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_texture_filter_metadata, texture_filter)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set(self, positions:list) -> None:
        """Define the positions of markers in a marker batch. The markers are rendered in the primitive's reference frame."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_metadata, positions)

    _set_with_optional_parameters_metadata = { "offset" : _set_with_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"),) }
    def set_with_optional_parameters(self, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters") -> None:
        """Define the positions and optional per-marker parameters of markers in a marker batch. The markers are rendered in the primitive's reference frame."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_with_optional_parameters_metadata, positions, optional_parameters)

    _set_with_optional_parameters_and_render_pass_hint_metadata = { "offset" : _set_with_optional_parameters_and_render_pass_hint_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.EnumArg(RenderPassHint),) }
    def set_with_optional_parameters_and_render_pass_hint(self, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", render_pass_hint:"RenderPassHint") -> None:
        """Define the positions and optional per-marker parameters of markers in a marker batch. The markers are rendered in the primitive's reference frame. renderPassHint is provided for efficiency."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_with_optional_parameters_and_render_pass_hint_metadata, positions, optional_parameters, render_pass_hint)

    _set_cartographic_metadata = { "offset" : _set_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic(self, central_body:str, positions:list) -> None:
        """For convenience. Defines the positions of markers in a marker batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_cartographic_metadata, central_body, positions)

    _set_cartographic_with_optional_parameters_metadata = { "offset" : _set_cartographic_with_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"),) }
    def set_cartographic_with_optional_parameters(self, central_body:str, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters") -> None:
        """For convenience. Defines the positions and optional per-marker parameters of markers in a marker batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_cartographic_with_optional_parameters_metadata, central_body, positions, optional_parameters)

    _set_cartographic_with_optional_parameters_and_render_pass_hint_metadata = { "offset" : _set_cartographic_with_optional_parameters_and_render_pass_hint_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.EnumArg(RenderPassHint),) }
    def set_cartographic_with_optional_parameters_and_render_pass_hint(self, central_body:str, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Defines the positions and optional per-marker parameters of markers in a marker batch using cartographic positions. renderPassHint is provided for efficiency..."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_cartographic_with_optional_parameters_and_render_pass_hint_metadata, central_body, positions, optional_parameters, render_pass_hint)

    _set_partial_metadata = { "offset" : _set_partial_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial(self, positions:list, indices:list) -> None:
        """Update a subset of marker positions in a marker batch."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_metadata, positions, indices)

    _set_partial_with_indices_order_metadata = { "offset" : _set_partial_with_indices_order_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_with_indices_order(self, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """Update a subset of marker positions in a marker batch."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_with_indices_order_metadata, positions, indices, indices_order_hint)

    _set_partial_with_optional_parameters_metadata = { "offset" : _set_partial_with_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_with_optional_parameters(self, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", indices:list) -> None:
        """Update a subset of marker positions and/or per-marker parameters in a marker batch."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_with_optional_parameters_metadata, positions, optional_parameters, indices)

    _set_partial_with_optional_parameters_indices_order_and_render_pass_metadata = { "offset" : _set_partial_with_optional_parameters_indices_order_and_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_with_optional_parameters_indices_order_and_render_pass(self, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """Update a subset of marker positions and/or per-marker parameters in a marker batch."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_with_optional_parameters_indices_order_and_render_pass_metadata, positions, optional_parameters, indices, indices_order_hint, render_pass_hint)

    _set_partial_cartographic_metadata = { "offset" : _set_partial_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic(self, central_body:str, positions:list, indices:list) -> None:
        """For convenience. Updates a subset of positions in a marker batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_cartographic_metadata, central_body, positions, indices)

    _set_partial_cartographic_with_indices_order_metadata = { "offset" : _set_partial_cartographic_with_indices_order_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_cartographic_with_indices_order(self, central_body:str, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """For convenience. Updates a subset of positions in a marker batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_cartographic_with_indices_order_metadata, central_body, positions, indices, indices_order_hint)

    _set_partial_cartographic_with_optional_parameters_metadata = { "offset" : _set_partial_cartographic_with_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic_with_optional_parameters(self, central_body:str, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", indices:list) -> None:
        """For convenience. Updates a subset of positions and/or optional per-marker parameters of markers in a marker batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_cartographic_with_optional_parameters_metadata, central_body, positions, optional_parameters, indices)

    _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_metadata = { "offset" : _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("MarkerBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass(self, central_body:str, positions:list, optional_parameters:"MarkerBatchPrimitiveOptionalParameters", indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Updates a subset of positions and/or optional per-marker parameters of markers in a marker batch using cartographic positions. renderPassHint is provided for efficiency..."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_metadata, central_body, positions, optional_parameters, indices, indices_order_hint, render_pass_hint)

    _supported_metadata = { "offset" : _supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(MarkerBatchRenderingMethod), agmarshall.VariantBoolArg,) }
    def supported(self, rendering_method:"MarkerBatchRenderingMethod") -> bool:
        """Determine whether or not the video card supports the marker batch primitive with the given renderingMethod."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._supported_metadata, rendering_method, OutArg())

    _get_clamp_to_pixel_metadata = { "offset" : _get_clamp_to_pixel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def clamp_to_pixel(self) -> bool:
        """Get or set whether the screen space position of each marker is clamped to a pixel."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_clamp_to_pixel_metadata)

    _set_clamp_to_pixel_metadata = { "offset" : _set_clamp_to_pixel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @clamp_to_pixel.setter
    def clamp_to_pixel(self, clamp_to_pixel:bool) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_clamp_to_pixel_metadata, clamp_to_pixel)

    _get_central_body_clipped_metadata = { "offset" : _get_central_body_clipped_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def central_body_clipped(self) -> bool:
        """Get or set whether the markers are clipped by the central body."""
        return self._intf.get_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._get_central_body_clipped_metadata)

    _set_central_body_clipped_metadata = { "offset" : _set_central_body_clipped_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @central_body_clipped.setter
    def central_body_clipped(self, central_body_clipped:bool) -> None:
        return self._intf.set_property(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._set_central_body_clipped_metadata, central_body_clipped)

    _align_to_screen_metadata = { "offset" : _align_to_screen_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def align_to_screen(self) -> None:
        """Set the up vector of the markers to always be aligned to the up vector of the camera. This is the default alignment."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._align_to_screen_metadata, )

    _align_to_north_metadata = { "offset" : _align_to_north_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def align_to_north(self, central_body:str) -> None:
        """Set the up vector of the markers to point towards the north axis of centralBody. It will be aligned with the tangent vector of the surface that points north."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._align_to_north_metadata, central_body)

    _align_to_axis_metadata = { "offset" : _align_to_axis_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def align_to_axis(self, central_body:str, axis:list) -> None:
        """Set the up vector of the markers to point towards the axis of centralBody. It will be aligned with the tangent vector of the surface that points towards the axis..."""
        return self._intf.invoke(MarkerBatchPrimitive._metadata, MarkerBatchPrimitive._align_to_axis_metadata, central_body, axis)

    _property_names[size_source] = "size_source"
    _property_names[sort_order] = "sort_order"
    _property_names[set_hint] = "set_hint"
    _property_names[rendering_method] = "rendering_method"
    _property_names[render_pass] = "render_pass"
    _property_names[bounding_sphere_scale] = "bounding_sphere_scale"
    _property_names[distance_display_condition_per_marker] = "distance_display_condition_per_marker"
    _property_names[texture] = "texture"
    _property_names[size_unit] = "size_unit"
    _property_names[size] = "size"
    _property_names[origin] = "origin"
    _property_names[pixel_offset] = "pixel_offset"
    _property_names[eye_offset] = "eye_offset"
    _property_names[rotation] = "rotation"
    _property_names[texture_coordinate] = "texture_coordinate"
    _property_names[wireframe] = "wireframe"
    _property_names[per_item_picking_enabled] = "per_item_picking_enabled"
    _property_names[texture_filter] = "texture_filter"
    _property_names[clamp_to_pixel] = "clamp_to_pixel"
    _property_names[central_body_clipped] = "central_body_clipped"

    def __init__(self, source_object=None):
        """Construct an object of type MarkerBatchPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MarkerBatchPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MarkerBatchPrimitive, [MarkerBatchPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5505857127152130349, 2653133157377855163), MarkerBatchPrimitive)
agcls.AgTypeNameMap["MarkerBatchPrimitive"] = MarkerBatchPrimitive

class MarkerBatchPrimitiveOptionalParameters(SupportsDeleteCallback):
    """Optional per-marker parameters for marker batch primitive that overrides the marker batch's per-batch parameters..."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_textures_method_offset = 1
    _set_sizes_method_offset = 2
    _set_colors_method_offset = 3
    _set_origins_method_offset = 4
    _set_pixel_offsets_method_offset = 5
    _set_eye_offsets_method_offset = 6
    _set_rotations_method_offset = 7
    _set_texture_coordinates_method_offset = 8
    _set_time_interval_display_conditions_method_offset = 9
    _set_displays_method_offset = 10
    _metadata = {
        "iid_data" : (4926552851670611518, 9718265972343758992),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MarkerBatchPrimitiveOptionalParameters)

    _set_textures_metadata = { "offset" : _set_textures_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_textures(self, textures:list) -> None:
        """Define a collection of textures, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_textures_metadata, textures)

    _set_sizes_metadata = { "offset" : _set_sizes_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_sizes(self, sizes:list) -> None:
        """Define a collection of sizes, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_sizes_metadata, sizes)

    _set_colors_metadata = { "offset" : _set_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_colors(self, colors:list) -> None:
        """Define a collection of colors, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_colors_metadata, colors)

    _set_origins_metadata = { "offset" : _set_origins_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_origins(self, origins:list) -> None:
        """Define a collection of origins, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_origins_metadata, origins)

    _set_pixel_offsets_metadata = { "offset" : _set_pixel_offsets_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_pixel_offsets(self, pixel_offsets:list) -> None:
        """Define a collection of pixel offsets, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_pixel_offsets_metadata, pixel_offsets)

    _set_eye_offsets_metadata = { "offset" : _set_eye_offsets_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_eye_offsets(self, eye_offsets:list) -> None:
        """Define a collection of eye offsets, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_eye_offsets_metadata, eye_offsets)

    _set_rotations_metadata = { "offset" : _set_rotations_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_rotations(self, rotation_angles:list) -> None:
        """Define a collection of rotation angles, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_rotations_metadata, rotation_angles)

    _set_texture_coordinates_metadata = { "offset" : _set_texture_coordinates_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_texture_coordinates(self, texture_coordinates:list) -> None:
        """Define a collection of texture coordinates, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_texture_coordinates_metadata, texture_coordinates)

    _set_time_interval_display_conditions_metadata = { "offset" : _set_time_interval_display_conditions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_time_interval_display_conditions(self, time_intervals:list) -> None:
        """Define a collection of time interval display conditions, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_time_interval_display_conditions_metadata, time_intervals)

    _set_displays_metadata = { "offset" : _set_displays_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_displays(self, displays:list) -> None:
        """Define a collection of boolean display flags, one for each marker in the batch."""
        return self._intf.invoke(MarkerBatchPrimitiveOptionalParameters._metadata, MarkerBatchPrimitiveOptionalParameters._set_displays_metadata, displays)


    def __init__(self, source_object=None):
        """Construct an object of type MarkerBatchPrimitiveOptionalParameters."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MarkerBatchPrimitiveOptionalParameters)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MarkerBatchPrimitiveOptionalParameters, [MarkerBatchPrimitiveOptionalParameters, ])

agcls.AgClassCatalog.add_catalog_entry((4765125297646667603, 6704393496158662787), MarkerBatchPrimitiveOptionalParameters)
agcls.AgTypeNameMap["MarkerBatchPrimitiveOptionalParameters"] = MarkerBatchPrimitiveOptionalParameters

class MaximumCountPathPrimitiveUpdatePolicy(IPathPrimitiveUpdatePolicy, SupportsDeleteCallback):
    """path primitive update policy that removes points from remove location when the number of points in the path exceeds maximum count."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_maximum_count_method_offset = 1
    _set_maximum_count_method_offset = 2
    _get_remove_location_method_offset = 3
    _set_remove_location_method_offset = 4
    _metadata = {
        "iid_data" : (5099271029598780192, 3154004167287703454),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MaximumCountPathPrimitiveUpdatePolicy)

    _get_maximum_count_metadata = { "offset" : _get_maximum_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_count(self) -> int:
        """Get or set the maximum number of points in the path."""
        return self._intf.get_property(MaximumCountPathPrimitiveUpdatePolicy._metadata, MaximumCountPathPrimitiveUpdatePolicy._get_maximum_count_metadata)

    _set_maximum_count_metadata = { "offset" : _set_maximum_count_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_count.setter
    def maximum_count(self, maximum_count:int) -> None:
        return self._intf.set_property(MaximumCountPathPrimitiveUpdatePolicy._metadata, MaximumCountPathPrimitiveUpdatePolicy._set_maximum_count_metadata, maximum_count)

    _get_remove_location_metadata = { "offset" : _get_remove_location_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PathPrimitiveRemoveLocation),) }
    @property
    def remove_location(self) -> "PathPrimitiveRemoveLocation":
        """Get or set where the positions on the path are removed from."""
        return self._intf.get_property(MaximumCountPathPrimitiveUpdatePolicy._metadata, MaximumCountPathPrimitiveUpdatePolicy._get_remove_location_metadata)

    _set_remove_location_metadata = { "offset" : _set_remove_location_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PathPrimitiveRemoveLocation),) }
    @remove_location.setter
    def remove_location(self, remove_location:"PathPrimitiveRemoveLocation") -> None:
        return self._intf.set_property(MaximumCountPathPrimitiveUpdatePolicy._metadata, MaximumCountPathPrimitiveUpdatePolicy._set_remove_location_metadata, remove_location)

    _property_names[maximum_count] = "maximum_count"
    _property_names[remove_location] = "remove_location"

    def __init__(self, source_object=None):
        """Construct an object of type MaximumCountPathPrimitiveUpdatePolicy."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MaximumCountPathPrimitiveUpdatePolicy)
        IPathPrimitiveUpdatePolicy.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPathPrimitiveUpdatePolicy._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MaximumCountPathPrimitiveUpdatePolicy, [MaximumCountPathPrimitiveUpdatePolicy, IPathPrimitiveUpdatePolicy])

agcls.AgClassCatalog.add_catalog_entry((5521743977193259552, 5353922303322892948), MaximumCountPathPrimitiveUpdatePolicy)
agcls.AgTypeNameMap["MaximumCountPathPrimitiveUpdatePolicy"] = MaximumCountPathPrimitiveUpdatePolicy

class ModelArticulation(SupportsDeleteCallback):
    """A model articulation identifies geometry on the model and is a collection of transformations that can be applied to that geometry."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _get_count_method_offset = 2
    _item_method_offset = 3
    _get_item_by_string_method_offset = 4
    _get_by_name_method_offset = 5
    _contains_method_offset = 6
    _get__new_enum_method_offset = 7
    _metadata = {
        "iid_data" : (5348326078254719861, 10849601624272152503),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ModelArticulation)
    def __iter__(self):
        """Create an iterator for the ModelArticulation object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ModelTransformation":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the articulation."""
        return self._intf.get_property(ModelArticulation._metadata, ModelArticulation._get_name_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of transformations in the collection."""
        return self._intf.get_property(ModelArticulation._metadata, ModelArticulation._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "ModelTransformation":
        """Get the transformation at the given index. The index is zero-based."""
        return self._intf.invoke(ModelArticulation._metadata, ModelArticulation._item_metadata, index, OutArg())

    _get_item_by_string_metadata = { "offset" : _get_item_by_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_string(self, name:str) -> "ModelTransformation":
        """Get a transformation by name."""
        return self._intf.invoke(ModelArticulation._metadata, ModelArticulation._get_item_by_string_metadata, name, OutArg())

    _get_by_name_metadata = { "offset" : _get_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_by_name(self, name:str) -> "ModelTransformation":
        """Get a transformation by name."""
        return self._intf.invoke(ModelArticulation._metadata, ModelArticulation._get_by_name_metadata, name, OutArg())

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Return true if the collection contains the transformation."""
        return self._intf.invoke(ModelArticulation._metadata, ModelArticulation._contains_metadata, name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return the enumerator for this collection."""
        return self._intf.get_property(ModelArticulation._metadata, ModelArticulation._get__new_enum_metadata)

    __getitem__ = item


    _property_names[name] = "name"
    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ModelArticulation."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ModelArticulation)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ModelArticulation, [ModelArticulation, ])

agcls.AgClassCatalog.add_catalog_entry((5583363848773512818, 10659230879766474660), ModelArticulation)
agcls.AgTypeNameMap["ModelArticulation"] = ModelArticulation

class ModelArticulationCollection(SupportsDeleteCallback):
    """A collection containing a model primitive's available articulations. A model articulation identifies geometry on the model and is a collection of transformations that can be applied to that geometry."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get_item_by_string_method_offset = 3
    _get_by_name_method_offset = 4
    _contains_method_offset = 5
    _get__new_enum_method_offset = 6
    _metadata = {
        "iid_data" : (4825692293668155476, 4454785697263668652),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ModelArticulationCollection)
    def __iter__(self):
        """Create an iterator for the ModelArticulationCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ModelArticulation":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of articulations in the collection."""
        return self._intf.get_property(ModelArticulationCollection._metadata, ModelArticulationCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "ModelArticulation":
        """Get the articulation at the given index. The index is zero-based."""
        return self._intf.invoke(ModelArticulationCollection._metadata, ModelArticulationCollection._item_metadata, index, OutArg())

    _get_item_by_string_metadata = { "offset" : _get_item_by_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_string(self, name:str) -> "ModelArticulation":
        """Get an articulation by name."""
        return self._intf.invoke(ModelArticulationCollection._metadata, ModelArticulationCollection._get_item_by_string_metadata, name, OutArg())

    _get_by_name_metadata = { "offset" : _get_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_by_name(self, name:str) -> "ModelArticulation":
        """Get an articulation by name."""
        return self._intf.invoke(ModelArticulationCollection._metadata, ModelArticulationCollection._get_by_name_metadata, name, OutArg())

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Return true if the collection contains the articulation."""
        return self._intf.invoke(ModelArticulationCollection._metadata, ModelArticulationCollection._contains_metadata, name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return the enumerator for this collection."""
        return self._intf.get_property(ModelArticulationCollection._metadata, ModelArticulationCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ModelArticulationCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ModelArticulationCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ModelArticulationCollection, [ModelArticulationCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4860630207067099658, 12153871741288218515), ModelArticulationCollection)
agcls.AgTypeNameMap["ModelArticulationCollection"] = ModelArticulationCollection

class ModelPrimitive(IPrimitive, SupportsDeleteCallback):
    """The model primitive loads and renders `COLLADA <https://www.khronos.org/collada/>`_ (DAE) and AGI `MDL <https://support.agi.com/3d-models>`_ (MDL) models."""

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_uri_as_string_method_offset = 1
    _get_scale_method_offset = 2
    _set_scale_method_offset = 3
    _get_position_method_offset = 4
    _set_position_method_offset = 5
    _get_orientation_method_offset = 6
    _set_orientation_method_offset = 7
    _get_articulations_method_offset = 8
    _load_with_string_uri_method_offset = 9
    _load_with_string_uri_and_up_axis_method_offset = 10
    _set_position_cartographic_method_offset = 11
    _metadata = {
        "iid_data" : (5605268163691457879, 14557257887553554869),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ModelPrimitive)

    _get_uri_as_string_metadata = { "offset" : _get_uri_as_string_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def uri_as_string(self) -> str:
        """Get the URI of the file used to load the file."""
        return self._intf.get_property(ModelPrimitive._metadata, ModelPrimitive._get_uri_as_string_metadata)

    _get_scale_metadata = { "offset" : _get_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def scale(self) -> float:
        """Get or set the linear scale used to increase or decrease the size of the rendered model."""
        return self._intf.get_property(ModelPrimitive._metadata, ModelPrimitive._get_scale_metadata)

    _set_scale_metadata = { "offset" : _set_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @scale.setter
    def scale(self, scale:float) -> None:
        return self._intf.set_property(ModelPrimitive._metadata, ModelPrimitive._set_scale_metadata, scale)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the position of the model. The position is defined in the model's reference frame. The array contains the components of the position in the order x, y, z."""
        return self._intf.get_property(ModelPrimitive._metadata, ModelPrimitive._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(ModelPrimitive._metadata, ModelPrimitive._set_position_metadata, position)

    _get_orientation_metadata = { "offset" : _get_orientation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def orientation(self) -> "IOrientation":
        """Get or set the model's orientation. The quaternion is a rotation from the model's local axes to the axes of the model's reference frame."""
        return self._intf.get_property(ModelPrimitive._metadata, ModelPrimitive._get_orientation_metadata)

    _set_orientation_metadata = { "offset" : _set_orientation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IOrientation"),) }
    @orientation.setter
    def orientation(self, orientation:"IOrientation") -> None:
        return self._intf.set_property(ModelPrimitive._metadata, ModelPrimitive._set_orientation_metadata, orientation)

    _get_articulations_metadata = { "offset" : _get_articulations_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def articulations(self) -> "ModelArticulationCollection":
        """Get the model's articulations. Articulations identify geometry and contain transformations for manipulating that geometry."""
        return self._intf.get_property(ModelPrimitive._metadata, ModelPrimitive._get_articulations_metadata)

    _load_with_string_uri_metadata = { "offset" : _load_with_string_uri_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def load_with_string_uri(self, uri:str) -> None:
        """For convenience. Loads a `COLLADA <https://www.khronos.org/collada/>`_ (DAE) or AGI `MDL <https://support.agi.com/3d-models>`_ (MDL) model using a file path."""
        return self._intf.invoke(ModelPrimitive._metadata, ModelPrimitive._load_with_string_uri_metadata, uri)

    _load_with_string_uri_and_up_axis_metadata = { "offset" : _load_with_string_uri_and_up_axis_method_offset,
            "arg_types" : (agcom.BSTR, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.EnumArg(ModelUpAxis),) }
    def load_with_string_uri_and_up_axis(self, uri:str, up_axis:"ModelUpAxis") -> None:
        """For convenience. Loads a `COLLADA <https://www.khronos.org/collada/>`_ (DAE) or AGI `MDL <https://support.agi.com/3d-models>`_ (MDL) model using a file path."""
        return self._intf.invoke(ModelPrimitive._metadata, ModelPrimitive._load_with_string_uri_and_up_axis_metadata, uri, up_axis)

    _set_position_cartographic_metadata = { "offset" : _set_position_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def set_position_cartographic(self, central_body:str, position:list) -> None:
        """For convenience. Sets the cartographic position of the model. This also sets position."""
        return self._intf.invoke(ModelPrimitive._metadata, ModelPrimitive._set_position_cartographic_metadata, central_body, position)

    _property_names[uri_as_string] = "uri_as_string"
    _property_names[scale] = "scale"
    _property_names[position] = "position"
    _property_names[orientation] = "orientation"
    _property_names[articulations] = "articulations"

    def __init__(self, source_object=None):
        """Construct an object of type ModelPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ModelPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ModelPrimitive, [ModelPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5272009851139634904, 10934403071105800840), ModelPrimitive)
agcls.AgTypeNameMap["ModelPrimitive"] = ModelPrimitive

class ModelTransformation(SupportsDeleteCallback):
    """A model transformation defines a transformation that is applied to geometry on a model primitive. That geometry is identified by the model articulation which contains the transformation..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_current_value_method_offset = 1
    _set_current_value_method_offset = 2
    _get_minimum_value_method_offset = 3
    _get_maximum_value_method_offset = 4
    _get_default_value_method_offset = 5
    _get_range_method_offset = 6
    _get_name_method_offset = 7
    _get_type_method_offset = 8
    _metadata = {
        "iid_data" : (5281930043538243524, 11725494751790391716),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ModelTransformation)

    _get_current_value_metadata = { "offset" : _get_current_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def current_value(self) -> float:
        """Get or set the current value of the transformation. The current value of the transformation will be reflected in the geometry of the model primitive that it is associated with."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_current_value_metadata)

    _set_current_value_metadata = { "offset" : _set_current_value_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @current_value.setter
    def current_value(self, current_value:float) -> None:
        return self._intf.set_property(ModelTransformation._metadata, ModelTransformation._set_current_value_metadata, current_value)

    _get_minimum_value_metadata = { "offset" : _get_minimum_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_value(self) -> float:
        """Get the minimum value of the transformation."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_minimum_value_metadata)

    _get_maximum_value_metadata = { "offset" : _get_maximum_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_value(self) -> float:
        """Get the maximum value of the transformation."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_maximum_value_metadata)

    _get_default_value_metadata = { "offset" : _get_default_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def default_value(self) -> float:
        """Get the default value of the transformation. The current value property of the transformation will have this value when the model primitive is initialized."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_default_value_metadata)

    _get_range_metadata = { "offset" : _get_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range(self) -> float:
        """Get the value range of the transformation. Equivalent to the difference of the maximum value and minimum value properties."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_range_metadata)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the transformation."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_name_metadata)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ModelTransformationType),) }
    @property
    def type(self) -> "ModelTransformationType":
        """Get the model transformation type associated with the transformation."""
        return self._intf.get_property(ModelTransformation._metadata, ModelTransformation._get_type_metadata)

    _property_names[current_value] = "current_value"
    _property_names[minimum_value] = "minimum_value"
    _property_names[maximum_value] = "maximum_value"
    _property_names[default_value] = "default_value"
    _property_names[range] = "range"
    _property_names[name] = "name"
    _property_names[type] = "type"

    def __init__(self, source_object=None):
        """Construct an object of type ModelTransformation."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ModelTransformation)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ModelTransformation, [ModelTransformation, ])

agcls.AgClassCatalog.add_catalog_entry((5494058340694028521, 8976331231816559258), ModelTransformation)
agcls.AgTypeNameMap["ModelTransformation"] = ModelTransformation

class Overlay(IOverlay, IScreenOverlayContainer, SupportsDeleteCallback):
    """A visible element drawn in screen space. Overlays are useful for floating logos, heads up displays, and integrating user interfaces into the 3D window."""
    def __init__(self, source_object=None):
        """Construct an object of type Overlay."""
        SupportsDeleteCallback.__init__(self)
        IOverlay.__init__(self, source_object)
        IScreenOverlayContainer.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IOverlay._private_init(self, intf)
        IScreenOverlayContainer._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Overlay, [IOverlay, IScreenOverlayContainer])

agcls.AgClassCatalog.add_catalog_entry((4796068382168091195, 15893869600297649596), Overlay)
agcls.AgTypeNameMap["Overlay"] = Overlay

class PathPrimitive(IPrimitive, SupportsDeleteCallback):
    """Render a line to the 3D scene. Similar to the polyline primitive; however, the PathPrimitive was designed for the efficient addition/removal of points to/from the front or back of the line."""

    _num_methods = 31
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_capacity_method_offset = 2
    _get_update_policy_method_offset = 3
    _set_update_policy_method_offset = 4
    _get_polyline_type_method_offset = 5
    _set_polyline_type_method_offset = 6
    _get_width_method_offset = 7
    _set_width_method_offset = 8
    _get_minimum_width_supported_method_offset = 9
    _get_maximum_width_supported_method_offset = 10
    _get_display_outline_method_offset = 11
    _set_display_outline_method_offset = 12
    _get_outline_width_method_offset = 13
    _set_outline_width_method_offset = 14
    _get_per_item_picking_enabled_method_offset = 15
    _set_per_item_picking_enabled_method_offset = 16
    _item_method_offset = 17
    _get__new_enum_method_offset = 18
    _add_front_method_offset = 19
    _add_range_to_front_method_offset = 20
    _add_back_method_offset = 21
    _add_range_to_back_method_offset = 22
    _remove_front_method_offset = 23
    _remove_all_before_method_offset = 24
    _remove_back_method_offset = 25
    _remove_all_after_method_offset = 26
    _front_method_offset = 27
    _back_method_offset = 28
    _clear_method_offset = 29
    _get_central_body_clipped_method_offset = 30
    _set_central_body_clipped_method_offset = 31
    _metadata = {
        "iid_data" : (4696601357917811424, 5934065904138946973),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PathPrimitive)
    def __iter__(self):
        """Create an iterator for the PathPrimitive object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "PathPoint":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Return the number of points."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_count_metadata)

    _get_capacity_metadata = { "offset" : _get_capacity_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def capacity(self) -> int:
        """Return the capacity that was set during object construction. The capacity is the amount of memory reserved for storing the points on the path. This will be automatically updated when adding/removing points."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_capacity_metadata)

    _get_update_policy_metadata = { "offset" : _get_update_policy_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def update_policy(self) -> "IPathPrimitiveUpdatePolicy":
        """Get or set how the primitive will be updated based on the current animation time."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_update_policy_metadata)

    _set_update_policy_metadata = { "offset" : _set_update_policy_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPathPrimitiveUpdatePolicy"),) }
    @update_policy.setter
    def update_policy(self, update_policy:"IPathPrimitiveUpdatePolicy") -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_update_policy_metadata, update_policy)

    _get_polyline_type_metadata = { "offset" : _get_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def polyline_type(self) -> "PolylineType":
        """Get or set how the primitive interprets the positions."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_polyline_type_metadata)

    _set_polyline_type_metadata = { "offset" : _set_polyline_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @polyline_type.setter
    def polyline_type(self, polyline_type:"PolylineType") -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_polyline_type_metadata, polyline_type)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def width(self) -> float:
        """Get or set the line width, in pixels."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_width_metadata)

    _set_width_metadata = { "offset" : _set_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @width.setter
    def width(self, width:float) -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_width_metadata, width)

    _get_minimum_width_supported_metadata = { "offset" : _get_minimum_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def minimum_width_supported(self) -> float:
        """Get the minimum width, in pixels, supported by the video card."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_minimum_width_supported_metadata)

    _get_maximum_width_supported_metadata = { "offset" : _get_maximum_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def maximum_width_supported(self) -> float:
        """Get the maximum width, in pixels, supported by the video card."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_maximum_width_supported_metadata)

    _get_display_outline_metadata = { "offset" : _get_display_outline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_outline(self) -> bool:
        """Get or set whether an outline is rendered around the line."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_display_outline_metadata)

    _set_display_outline_metadata = { "offset" : _set_display_outline_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_outline.setter
    def display_outline(self, display_outline:bool) -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_display_outline_metadata, display_outline)

    _get_outline_width_metadata = { "offset" : _get_outline_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_width(self) -> float:
        """Get or set the width, in pixels, of the outline around the line."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_outline_width_metadata)

    _set_outline_width_metadata = { "offset" : _set_outline_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_width.setter
    def outline_width(self, outline_width:float) -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_outline_width_metadata, outline_width)

    _get_per_item_picking_enabled_metadata = { "offset" : _get_per_item_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def per_item_picking_enabled(self) -> bool:
        """Get or set whether individual line indices will be included in the pick results returned from the scene's Pick method. Each line index that is picked will be returned as a batch primitive index."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_per_item_picking_enabled_metadata)

    _set_per_item_picking_enabled_metadata = { "offset" : _set_per_item_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @per_item_picking_enabled.setter
    def per_item_picking_enabled(self, per_item_picking_enabled:bool) -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_per_item_picking_enabled_metadata, per_item_picking_enabled)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "PathPoint":
        """Return the point at the given zero-based index."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get__new_enum_metadata)

    _add_front_metadata = { "offset" : _add_front_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("PathPoint"),) }
    def add_front(self, path_point:"PathPoint") -> None:
        """Add a path point to the front of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._add_front_metadata, path_point)

    _add_range_to_front_metadata = { "offset" : _add_range_to_front_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def add_range_to_front(self, positions:list) -> None:
        """Add the range of path points to the front of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._add_range_to_front_metadata, positions)

    _add_back_metadata = { "offset" : _add_back_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("PathPoint"),) }
    def add_back(self, path_point:"PathPoint") -> None:
        """Add a path point to the back of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._add_back_metadata, path_point)

    _add_range_to_back_metadata = { "offset" : _add_range_to_back_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def add_range_to_back(self, positions:list) -> None:
        """Add the range of path points to the back of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._add_range_to_back_metadata, positions)

    _remove_front_metadata = { "offset" : _remove_front_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove_front(self) -> None:
        """Remove a path point to the front of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._remove_front_metadata, )

    _remove_all_before_metadata = { "offset" : _remove_all_before_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def remove_all_before(self, index:int) -> None:
        """Remove all points before index."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._remove_all_before_metadata, index)

    _remove_back_metadata = { "offset" : _remove_back_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove_back(self) -> None:
        """Remove a path point to the back of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._remove_back_metadata, )

    _remove_all_after_metadata = { "offset" : _remove_all_after_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def remove_all_after(self, index:int) -> None:
        """Remove all points after index."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._remove_all_after_metadata, index)

    _front_metadata = { "offset" : _front_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def front(self) -> "PathPoint":
        """Access the path point at the front of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._front_metadata, OutArg())

    _back_metadata = { "offset" : _back_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def back(self) -> "PathPoint":
        """Access the path point at the back of the line."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._back_metadata, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all of the points."""
        return self._intf.invoke(PathPrimitive._metadata, PathPrimitive._clear_metadata, )

    _get_central_body_clipped_metadata = { "offset" : _get_central_body_clipped_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def central_body_clipped(self) -> bool:
        """Get or set whether the polyline will be clipped by the central body."""
        return self._intf.get_property(PathPrimitive._metadata, PathPrimitive._get_central_body_clipped_metadata)

    _set_central_body_clipped_metadata = { "offset" : _set_central_body_clipped_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @central_body_clipped.setter
    def central_body_clipped(self, central_body_clipped:bool) -> None:
        return self._intf.set_property(PathPrimitive._metadata, PathPrimitive._set_central_body_clipped_metadata, central_body_clipped)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[capacity] = "capacity"
    _property_names[update_policy] = "update_policy"
    _property_names[polyline_type] = "polyline_type"
    _property_names[width] = "width"
    _property_names[minimum_width_supported] = "minimum_width_supported"
    _property_names[maximum_width_supported] = "maximum_width_supported"
    _property_names[display_outline] = "display_outline"
    _property_names[outline_width] = "outline_width"
    _property_names[per_item_picking_enabled] = "per_item_picking_enabled"
    _property_names[_new_enum] = "_new_enum"
    _property_names[central_body_clipped] = "central_body_clipped"

    def __init__(self, source_object=None):
        """Construct an object of type PathPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PathPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PathPrimitive, [PathPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5627175414395611667, 1200547442396153242), PathPrimitive)
agcls.AgTypeNameMap["PathPrimitive"] = PathPrimitive

class PickResult(SupportsDeleteCallback):
    """A single result from Pick."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_objects_method_offset = 1
    _get_depth_method_offset = 2
    _get_position_method_offset = 3
    _metadata = {
        "iid_data" : (5469424382237706852, 5401777624284253099),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PickResult)

    _get_objects_metadata = { "offset" : _get_objects_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def objects(self) -> "ObjectCollection":
        """Get a collection of objects that were on the pick stack for the picked object."""
        return self._intf.get_property(PickResult._metadata, PickResult._get_objects_metadata)

    _get_depth_metadata = { "offset" : _get_depth_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def depth(self) -> float:
        """Get the depth of the picked location in the 3D scene."""
        return self._intf.get_property(PickResult._metadata, PickResult._get_depth_metadata)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get the position of the picked location in the central body's fixed reference frame. The array contains the components of the position arranged in the order x, y, z."""
        return self._intf.get_property(PickResult._metadata, PickResult._get_position_metadata)

    _property_names[objects] = "objects"
    _property_names[depth] = "depth"
    _property_names[position] = "position"

    def __init__(self, source_object=None):
        """Construct an object of type PickResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PickResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PickResult, [PickResult, ])

agcls.AgClassCatalog.add_catalog_entry((4613396686185153582, 5313864718031071897), PickResult)
agcls.AgTypeNameMap["PickResult"] = PickResult

class PixelSizeDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """Define an inclusive interval, in pixels, that determines when an object, such as a primitive, is rendered based on the number of pixels the object's bounding sphere (or in the case of screen overlays, bounding rectangle) covers on the screen..."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_pixel_size_method_offset = 1
    _set_minimum_pixel_size_method_offset = 2
    _get_maximum_pixel_size_method_offset = 3
    _set_maximum_pixel_size_method_offset = 4
    _metadata = {
        "iid_data" : (5696107621766714896, 13788591222461518237),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PixelSizeDisplayCondition)

    _get_minimum_pixel_size_metadata = { "offset" : _get_minimum_pixel_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def minimum_pixel_size(self) -> int:
        """Get or set the minimum pixel size of the inclusive distance interval."""
        return self._intf.get_property(PixelSizeDisplayCondition._metadata, PixelSizeDisplayCondition._get_minimum_pixel_size_metadata)

    _set_minimum_pixel_size_metadata = { "offset" : _set_minimum_pixel_size_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @minimum_pixel_size.setter
    def minimum_pixel_size(self, minimum_pixel_size:int) -> None:
        return self._intf.set_property(PixelSizeDisplayCondition._metadata, PixelSizeDisplayCondition._set_minimum_pixel_size_metadata, minimum_pixel_size)

    _get_maximum_pixel_size_metadata = { "offset" : _get_maximum_pixel_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_pixel_size(self) -> int:
        """Get or set the maximum pixel size of the inclusive distance interval. Use Int32.MaxValue to ignore checking the maximum distance."""
        return self._intf.get_property(PixelSizeDisplayCondition._metadata, PixelSizeDisplayCondition._get_maximum_pixel_size_metadata)

    _set_maximum_pixel_size_metadata = { "offset" : _set_maximum_pixel_size_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_pixel_size.setter
    def maximum_pixel_size(self, maximum_pixel_size:int) -> None:
        return self._intf.set_property(PixelSizeDisplayCondition._metadata, PixelSizeDisplayCondition._set_maximum_pixel_size_metadata, maximum_pixel_size)

    _property_names[minimum_pixel_size] = "minimum_pixel_size"
    _property_names[maximum_pixel_size] = "maximum_pixel_size"

    def __init__(self, source_object=None):
        """Construct an object of type PixelSizeDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PixelSizeDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PixelSizeDisplayCondition, [PixelSizeDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5639872464997702223, 13268018770298113449), PixelSizeDisplayCondition)
agcls.AgTypeNameMap["PixelSizeDisplayCondition"] = PixelSizeDisplayCondition

class PointBatchPrimitive(IPrimitive, SupportsDeleteCallback):
    """
    Render one or more points in the 3D scene. Each point in the batch has a unique position and an optional color. All points in the batch share the same pixel size. For best performance, avoid creating lots of batches with only a few points each...

    Examples
    --------
    Draw a Point Primitive and set properties:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> point = manager.initializers.point_batch_primitive.initialize()
    >>> ptPosition = [[0], [-1], [0]]  # Lat, Lon, Alt
    >>>
    >>> point.set_cartographic("Earth", ptPosition)
    >>> point.pixel_size = 15
    >>> point.color = Colors.Lime
    >>> point.display_outline = True
    >>> point.outline_width = 5
    >>> point.outline_color = Colors.Red
    >>>
    >>> manager.primitives.add(point)
    >>> # Render the Scene
    >>> manager.render()
    """

    _num_methods = 34
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_display_outline_method_offset = 1
    _set_display_outline_method_offset = 2
    _get_outline_color_method_offset = 3
    _set_outline_color_method_offset = 4
    _get_outline_translucency_method_offset = 5
    _set_outline_translucency_method_offset = 6
    _get_outline_width_method_offset = 7
    _set_outline_width_method_offset = 8
    _get_pixel_size_method_offset = 9
    _set_pixel_size_method_offset = 10
    _get_minimum_pixel_size_supported_method_offset = 11
    _get_maximum_pixel_size_supported_method_offset = 12
    _get_distance_display_condition_per_point_method_offset = 13
    _set_distance_display_condition_per_point_method_offset = 14
    _get_set_hint_method_offset = 15
    _get_per_item_picking_enabled_method_offset = 16
    _set_per_item_picking_enabled_method_offset = 17
    _set_method_offset = 18
    _set_with_colors_method_offset = 19
    _set_with_colors_and_render_pass_method_offset = 20
    _set_cartographic_method_offset = 21
    _set_cartographic_with_colors_method_offset = 22
    _set_cartographic_with_colors_and_render_pass_method_offset = 23
    _set_partial_method_offset = 24
    _set_partial_with_indices_order_method_offset = 25
    _set_partial_with_colors_method_offset = 26
    _set_partial_with_colors_indices_order_and_render_pass_method_offset = 27
    _set_partial_cartographic_method_offset = 28
    _set_partial_cartographic_with_indices_order_method_offset = 29
    _set_partial_cartographic_with_colors_method_offset = 30
    _set_partial_cartographic_with_colors_indices_order_and_render_pass_method_offset = 31
    _get_central_body_clipped_method_offset = 32
    _set_central_body_clipped_method_offset = 33
    _set_with_optional_parameters_method_offset = 34
    _metadata = {
        "iid_data" : (4907018134519311818, 5049913687853084042),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PointBatchPrimitive)

    _get_display_outline_metadata = { "offset" : _get_display_outline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_outline(self) -> bool:
        """Get or set whether an outline is rendered around each point in the batch."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_display_outline_metadata)

    _set_display_outline_metadata = { "offset" : _set_display_outline_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_outline.setter
    def display_outline(self, display_outline:bool) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_display_outline_metadata, display_outline)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """Get or set the outline's color."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_outline_color_metadata, outline_color)

    _get_outline_translucency_metadata = { "offset" : _get_outline_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_translucency(self) -> float:
        """Get or set the translucency of the outline. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_outline_translucency_metadata)

    _set_outline_translucency_metadata = { "offset" : _set_outline_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_translucency.setter
    def outline_translucency(self, outline_translucency:float) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_outline_translucency_metadata, outline_translucency)

    _get_outline_width_metadata = { "offset" : _get_outline_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_width(self) -> float:
        """Get or set the size, in pixels, of the outline around each point in the batch."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_outline_width_metadata)

    _set_outline_width_metadata = { "offset" : _set_outline_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_width.setter
    def outline_width(self, outline_width:float) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_outline_width_metadata, outline_width)

    _get_pixel_size_metadata = { "offset" : _get_pixel_size_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def pixel_size(self) -> float:
        """Get or set the size, in pixels, of each point in the point batch."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_pixel_size_metadata)

    _set_pixel_size_metadata = { "offset" : _set_pixel_size_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @pixel_size.setter
    def pixel_size(self, pixel_size:float) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_pixel_size_metadata, pixel_size)

    _get_minimum_pixel_size_supported_metadata = { "offset" : _get_minimum_pixel_size_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def minimum_pixel_size_supported(self) -> float:
        """Get the minimum pixel size supported by the video card."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_minimum_pixel_size_supported_metadata)

    _get_maximum_pixel_size_supported_metadata = { "offset" : _get_maximum_pixel_size_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def maximum_pixel_size_supported(self) -> float:
        """Get the maximum pixel size supported by the video card."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_maximum_pixel_size_supported_metadata)

    _get_distance_display_condition_per_point_metadata = { "offset" : _get_distance_display_condition_per_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_display_condition_per_point(self) -> "DistanceDisplayCondition":
        """Get or set a distance display condition that is evaluated per point in the point batch during rendering. This is different than display condition, which is evaluated once for the entire point batch..."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_distance_display_condition_per_point_metadata)

    _set_distance_display_condition_per_point_metadata = { "offset" : _set_distance_display_condition_per_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("DistanceDisplayCondition"),) }
    @distance_display_condition_per_point.setter
    def distance_display_condition_per_point(self, distance_display_condition_per_point:"DistanceDisplayCondition") -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_distance_display_condition_per_point_metadata, distance_display_condition_per_point)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_set_hint_metadata)

    _get_per_item_picking_enabled_metadata = { "offset" : _get_per_item_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def per_item_picking_enabled(self) -> bool:
        """Get or set whether individual point indices will be included in the pick results returned from the scene's Pick method. Each point index that is picked will be returned as a batch primitive index."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_per_item_picking_enabled_metadata)

    _set_per_item_picking_enabled_metadata = { "offset" : _set_per_item_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @per_item_picking_enabled.setter
    def per_item_picking_enabled(self, per_item_picking_enabled:bool) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_per_item_picking_enabled_metadata, per_item_picking_enabled)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set(self, positions:list) -> None:
        """Define the positions of points in a point batch. The points are rendered in the primitive's reference frame."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_metadata, positions)

    _set_with_colors_metadata = { "offset" : _set_with_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_with_colors(self, positions:list, colors:list) -> None:
        """Define the positions and colors of points in a point batch. The points are rendered in the primitive's reference frame."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_with_colors_metadata, positions, colors)

    _set_with_colors_and_render_pass_metadata = { "offset" : _set_with_colors_and_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(RenderPassHint),) }
    def set_with_colors_and_render_pass(self, positions:list, colors:list, render_pass_hint:"RenderPassHint") -> None:
        """Define the positions and colors of points in a point batch. The points are rendered in the primitive's reference frame. renderPassHint is provided for efficiency."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_with_colors_and_render_pass_metadata, positions, colors, render_pass_hint)

    _set_cartographic_metadata = { "offset" : _set_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic(self, central_body:str, positions:list) -> None:
        """For convenience. Defines the positions of points in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_cartographic_metadata, central_body, positions)

    _set_cartographic_with_colors_metadata = { "offset" : _set_cartographic_with_colors_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic_with_colors(self, central_body:str, positions:list, colors:list) -> None:
        """For convenience. Defines the positions and colors of points in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_cartographic_with_colors_metadata, central_body, positions, colors)

    _set_cartographic_with_colors_and_render_pass_metadata = { "offset" : _set_cartographic_with_colors_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(RenderPassHint),) }
    def set_cartographic_with_colors_and_render_pass(self, central_body:str, positions:list, colors:list, render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Defines the positions and colors of points in a point batch using cartographic positions. renderPassHint is provided for efficiency. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_cartographic_with_colors_and_render_pass_metadata, central_body, positions, colors, render_pass_hint)

    _set_partial_metadata = { "offset" : _set_partial_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial(self, positions:list, indices:list) -> None:
        """Update a subset of positions in a point batch."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_metadata, positions, indices)

    _set_partial_with_indices_order_metadata = { "offset" : _set_partial_with_indices_order_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_with_indices_order(self, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """Update a subset of positions in a point batch."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_with_indices_order_metadata, positions, indices, indices_order_hint)

    _set_partial_with_colors_metadata = { "offset" : _set_partial_with_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_with_colors(self, positions:list, colors:list, indices:list) -> None:
        """Update a subset of positions and/or colors in a point batch."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_with_colors_metadata, positions, colors, indices)

    _set_partial_with_colors_indices_order_and_render_pass_metadata = { "offset" : _set_partial_with_colors_indices_order_and_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_with_colors_indices_order_and_render_pass(self, positions:list, colors:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """Update a subset of positions and/or colors in a point batch."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_with_colors_indices_order_and_render_pass_metadata, positions, colors, indices, indices_order_hint, render_pass_hint)

    _set_partial_cartographic_metadata = { "offset" : _set_partial_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic(self, central_body:str, positions:list, indices:list) -> None:
        """For convenience. Updates a subset of positions in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_cartographic_metadata, central_body, positions, indices)

    _set_partial_cartographic_with_indices_order_metadata = { "offset" : _set_partial_cartographic_with_indices_order_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_cartographic_with_indices_order(self, central_body:str, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """For convenience. Updates a subset of positions in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_cartographic_with_indices_order_metadata, central_body, positions, indices, indices_order_hint)

    _set_partial_cartographic_with_colors_metadata = { "offset" : _set_partial_cartographic_with_colors_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic_with_colors(self, central_body:str, positions:list, colors:list, indices:list) -> None:
        """For convenience. Updates a subset of positions and/or colors in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_cartographic_with_colors_metadata, central_body, positions, colors, indices)

    _set_partial_cartographic_with_colors_indices_order_and_render_pass_metadata = { "offset" : _set_partial_cartographic_with_colors_indices_order_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_cartographic_with_colors_indices_order_and_render_pass(self, central_body:str, positions:list, colors:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Updates a subset of positions and/or colors in a point batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_partial_cartographic_with_colors_indices_order_and_render_pass_metadata, central_body, positions, colors, indices, indices_order_hint, render_pass_hint)

    _get_central_body_clipped_metadata = { "offset" : _get_central_body_clipped_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def central_body_clipped(self) -> bool:
        """Get or set whether individual points will be clipped by the central body."""
        return self._intf.get_property(PointBatchPrimitive._metadata, PointBatchPrimitive._get_central_body_clipped_metadata)

    _set_central_body_clipped_metadata = { "offset" : _set_central_body_clipped_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @central_body_clipped.setter
    def central_body_clipped(self, central_body_clipped:bool) -> None:
        return self._intf.set_property(PointBatchPrimitive._metadata, PointBatchPrimitive._set_central_body_clipped_metadata, central_body_clipped)

    _set_with_optional_parameters_metadata = { "offset" : _set_with_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("PointBatchPrimitiveOptionalParameters"), agmarshall.EnumArg(RenderPassHint),) }
    def set_with_optional_parameters(self, positions:list, colors:list, optional_parameters:"PointBatchPrimitiveOptionalParameters", render_pass_hint:"RenderPassHint") -> None:
        """Define the positions, colors, and optional parameters of points in a point batch. The points are rendered in the primitive's reference frame. renderPassHint is provided for efficiency."""
        return self._intf.invoke(PointBatchPrimitive._metadata, PointBatchPrimitive._set_with_optional_parameters_metadata, positions, colors, optional_parameters, render_pass_hint)

    _property_names[display_outline] = "display_outline"
    _property_names[outline_color] = "outline_color"
    _property_names[outline_translucency] = "outline_translucency"
    _property_names[outline_width] = "outline_width"
    _property_names[pixel_size] = "pixel_size"
    _property_names[minimum_pixel_size_supported] = "minimum_pixel_size_supported"
    _property_names[maximum_pixel_size_supported] = "maximum_pixel_size_supported"
    _property_names[distance_display_condition_per_point] = "distance_display_condition_per_point"
    _property_names[set_hint] = "set_hint"
    _property_names[per_item_picking_enabled] = "per_item_picking_enabled"
    _property_names[central_body_clipped] = "central_body_clipped"

    def __init__(self, source_object=None):
        """Construct an object of type PointBatchPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PointBatchPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PointBatchPrimitive, [PointBatchPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((4771838424736961315, 17789363099158793403), PointBatchPrimitive)
agcls.AgTypeNameMap["PointBatchPrimitive"] = PointBatchPrimitive

class PointBatchPrimitiveOptionalParameters(SupportsDeleteCallback):
    """Optional per-point parameters for point batch primitive that overrides the point batch primitive's global parameters..."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_pixel_sizes_method_offset = 1
    _metadata = {
        "iid_data" : (5205056675059894066, 5948443580933544383),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PointBatchPrimitiveOptionalParameters)

    _set_pixel_sizes_metadata = { "offset" : _set_pixel_sizes_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_pixel_sizes(self, pixel_sizes:list) -> None:
        """Define a collection of pixel sizes, one for each point in the point batch."""
        return self._intf.invoke(PointBatchPrimitiveOptionalParameters._metadata, PointBatchPrimitiveOptionalParameters._set_pixel_sizes_metadata, pixel_sizes)


    def __init__(self, source_object=None):
        """Construct an object of type PointBatchPrimitiveOptionalParameters."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PointBatchPrimitiveOptionalParameters)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PointBatchPrimitiveOptionalParameters, [PointBatchPrimitiveOptionalParameters, ])

agcls.AgClassCatalog.add_catalog_entry((5230597701691714270, 598952128910254753), PointBatchPrimitiveOptionalParameters)
agcls.AgTypeNameMap["PointBatchPrimitiveOptionalParameters"] = PointBatchPrimitiveOptionalParameters

class PolylinePrimitive(IPrimitive, SupportsDeleteCallback):
    """Render a polyline in the 3D scene. Each line segment may have a different color. A polyline can be constructed with a position interpolator to render great arcs or rhumb lines."""

    _num_methods = 42
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_width_method_offset = 1
    _set_width_method_offset = 2
    _get_minimum_width_supported_method_offset = 3
    _get_maximum_width_supported_method_offset = 4
    _get_position_interpolator_method_offset = 5
    _get_polyline_type_method_offset = 6
    _get_set_hint_method_offset = 7
    _get_display_outline_method_offset = 8
    _set_display_outline_method_offset = 9
    _get_outline_color_method_offset = 10
    _set_outline_color_method_offset = 11
    _get_outline_translucency_method_offset = 12
    _set_outline_translucency_method_offset = 13
    _get_outline_width_method_offset = 14
    _set_outline_width_method_offset = 15
    _get_per_item_picking_enabled_method_offset = 16
    _set_per_item_picking_enabled_method_offset = 17
    _set_method_offset = 18
    _set_with_colors_method_offset = 19
    _set_with_colors_and_hint_method_offset = 20
    _set_with_surface_shapes_result_method_offset = 21
    _set_with_surface_triangulator_result_method_offset = 22
    _set_with_solid_triangulator_result_method_offset = 23
    _set_cartographic_method_offset = 24
    _set_cartographic_with_colors_method_offset = 25
    _set_cartographic_with_colors_and_hint_method_offset = 26
    _set_subset_method_offset = 27
    _set_subset_cartographic_method_offset = 28
    _set_partial_method_offset = 29
    _set_partial_with_indices_order_method_offset = 30
    _set_partial_with_colors_method_offset = 31
    _set_partial_with_colors_indices_order_and_render_pass_hint_method_offset = 32
    _set_partial_cartographic_method_offset = 33
    _set_partial_cartographic_with_indices_order_method_offset = 34
    _set_partial_cartographic_with_colors_method_offset = 35
    _set_partial_cartographic_with_colors_indices_order_and_render_pass_method_offset = 36
    _get_central_body_clipped_method_offset = 37
    _set_central_body_clipped_method_offset = 38
    _set_with_colors_and_optional_parameters_method_offset = 39
    _set_cartographic_with_colors_and_optional_parameters_method_offset = 40
    _set_partial_with_colors_and_optional_parameters_method_offset = 41
    _set_partial_cartographic_with_optional_parameters_method_offset = 42
    _metadata = {
        "iid_data" : (4917159150693309483, 4735648436991941304),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PolylinePrimitive)

    _get_width_metadata = { "offset" : _get_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def width(self) -> float:
        """Get or set the line width, in pixels."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_width_metadata)

    _set_width_metadata = { "offset" : _set_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @width.setter
    def width(self, width:float) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_width_metadata, width)

    _get_minimum_width_supported_metadata = { "offset" : _get_minimum_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def minimum_width_supported(self) -> float:
        """Get the minimum width, in pixels, supported by the video card."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_minimum_width_supported_metadata)

    _get_maximum_width_supported_metadata = { "offset" : _get_maximum_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def maximum_width_supported(self) -> float:
        """Get the maximum width, in pixels, supported by the video card."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_maximum_width_supported_metadata)

    _get_position_interpolator_metadata = { "offset" : _get_position_interpolator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def position_interpolator(self) -> "IPositionInterpolator":
        """Get the position interpolator applied to positions passed to Set, SetCartographic, SetSubset, and SetSubsetCartographic methods. When this property is null, linear interpolation is used."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_position_interpolator_metadata)

    _get_polyline_type_metadata = { "offset" : _get_polyline_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PolylineType),) }
    @property
    def polyline_type(self) -> "PolylineType":
        """Get how the polyline interprets the positions passed to Set methods."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_polyline_type_metadata)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_set_hint_metadata)

    _get_display_outline_metadata = { "offset" : _get_display_outline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_outline(self) -> bool:
        """Get or set whether an outline is rendered around the polyline."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_display_outline_metadata)

    _set_display_outline_metadata = { "offset" : _set_display_outline_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_outline.setter
    def display_outline(self, display_outline:bool) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_display_outline_metadata, display_outline)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """Get or set the outline's color."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_outline_color_metadata, outline_color)

    _get_outline_translucency_metadata = { "offset" : _get_outline_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_translucency(self) -> float:
        """Get or set the translucency of the outline. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_outline_translucency_metadata)

    _set_outline_translucency_metadata = { "offset" : _set_outline_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_translucency.setter
    def outline_translucency(self, outline_translucency:float) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_outline_translucency_metadata, outline_translucency)

    _get_outline_width_metadata = { "offset" : _get_outline_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_width(self) -> float:
        """Get or set the width, in pixels, of the outline around the polyline."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_outline_width_metadata)

    _set_outline_width_metadata = { "offset" : _set_outline_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_width.setter
    def outline_width(self, outline_width:float) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_outline_width_metadata, outline_width)

    _get_per_item_picking_enabled_metadata = { "offset" : _get_per_item_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def per_item_picking_enabled(self) -> bool:
        """Get or set whether individual line indices will be included in the pick results returned from the scene's Pick method. Each line index that is picked will be returned as a batch primitive index."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_per_item_picking_enabled_metadata)

    _set_per_item_picking_enabled_metadata = { "offset" : _set_per_item_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @per_item_picking_enabled.setter
    def per_item_picking_enabled(self, per_item_picking_enabled:bool) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_per_item_picking_enabled_metadata, per_item_picking_enabled)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set(self, positions:list) -> None:
        """Define the positions for a polyline primitive. The polyline is rendered in its reference frame."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_metadata, positions)

    _set_with_colors_metadata = { "offset" : _set_with_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_with_colors(self, positions:list, colors:list) -> None:
        """Define the positions and colors of a polyline. The polyline is rendered in its reference frame."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_colors_metadata, positions, colors)

    _set_with_colors_and_hint_metadata = { "offset" : _set_with_colors_and_hint_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(RenderPassHint),) }
    def set_with_colors_and_hint(self, positions:list, colors:list, render_pass_hint:"RenderPassHint") -> None:
        """Define the positions and colors of a polyline. The polyline is rendered in its reference frame. renderPassHint is provided for efficiency."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_colors_and_hint_metadata, positions, colors, render_pass_hint)

    _set_with_surface_shapes_result_metadata = { "offset" : _set_with_surface_shapes_result_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SurfaceShapesResult"),) }
    def set_with_surface_shapes_result(self, surface_shapes_result:"SurfaceShapesResult") -> None:
        """Define the positions of a polyline using the positions of the specified surfaceShapesResult."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_surface_shapes_result_metadata, surface_shapes_result)

    _set_with_surface_triangulator_result_metadata = { "offset" : _set_with_surface_triangulator_result_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SurfaceTriangulatorResult"),) }
    def set_with_surface_triangulator_result(self, surface_triangulator_result:"SurfaceTriangulatorResult") -> None:
        """Define the positions of a polyline using the boundary positions of the specified surfaceTriangulatorResult."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_surface_triangulator_result_metadata, surface_triangulator_result)

    _set_with_solid_triangulator_result_metadata = { "offset" : _set_with_solid_triangulator_result_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SolidTriangulatorResult"),) }
    def set_with_solid_triangulator_result(self, solid_triangulator_result:"SolidTriangulatorResult") -> None:
        """Define the positions of a polyline using the outline positions of the specified solidTriangulatorResult."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_solid_triangulator_result_metadata, solid_triangulator_result)

    _set_cartographic_metadata = { "offset" : _set_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic(self, central_body:str, positions:list) -> None:
        """For convenience. Defines the positions of a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_cartographic_metadata, central_body, positions)

    _set_cartographic_with_colors_metadata = { "offset" : _set_cartographic_with_colors_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic_with_colors(self, central_body:str, positions:list, colors:list) -> None:
        """For convenience. Defines the positions and colors of a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_cartographic_with_colors_metadata, central_body, positions, colors)

    _set_cartographic_with_colors_and_hint_metadata = { "offset" : _set_cartographic_with_colors_and_hint_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(RenderPassHint),) }
    def set_cartographic_with_colors_and_hint(self, central_body:str, positions:list, colors:list, render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Defines the positions and colors of a polyline using cartographic positions. renderPassHint is provided for efficiency. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_cartographic_with_colors_and_hint_metadata, central_body, positions, colors, render_pass_hint)

    _set_subset_metadata = { "offset" : _set_subset_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.IntArg, agmarshall.IntArg,) }
    def set_subset(self, positions:list, index:int, count:int) -> None:
        """Define the positions of a polyline using a subset of input positions."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_subset_metadata, positions, index, count)

    _set_subset_cartographic_metadata = { "offset" : _set_subset_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.IntArg, agmarshall.IntArg,) }
    def set_subset_cartographic(self, central_body:str, positions:list, index:int, count:int) -> None:
        """For convenience. Defines the positions of a polyline using a subset of input cartographic positions. This is equivalent to converting the subset of positions to cartesian and calling SetSubset."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_subset_cartographic_metadata, central_body, positions, index, count)

    _set_partial_metadata = { "offset" : _set_partial_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial(self, positions:list, indices:list) -> None:
        """Update a subset of positions in a polyline."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_metadata, positions, indices)

    _set_partial_with_indices_order_metadata = { "offset" : _set_partial_with_indices_order_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_with_indices_order(self, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """Update a subset of positions in a polyline."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_with_indices_order_metadata, positions, indices, indices_order_hint)

    _set_partial_with_colors_metadata = { "offset" : _set_partial_with_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_with_colors(self, positions:list, colors:list, indices:list) -> None:
        """Update a subset of positions and/or colors in a polyline."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_with_colors_metadata, positions, colors, indices)

    _set_partial_with_colors_indices_order_and_render_pass_hint_metadata = { "offset" : _set_partial_with_colors_indices_order_and_render_pass_hint_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_with_colors_indices_order_and_render_pass_hint(self, positions:list, colors:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """Update a subset of positions and/or colors in a polyline."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_with_colors_indices_order_and_render_pass_hint_metadata, positions, colors, indices, indices_order_hint, render_pass_hint)

    _set_partial_cartographic_metadata = { "offset" : _set_partial_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic(self, central_body:str, positions:list, indices:list) -> None:
        """For convenience. Updates a subset of positions in a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_cartographic_metadata, central_body, positions, indices)

    _set_partial_cartographic_with_indices_order_metadata = { "offset" : _set_partial_cartographic_with_indices_order_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_cartographic_with_indices_order(self, central_body:str, positions:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """For convenience. Updates a subset of positions in a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_cartographic_with_indices_order_metadata, central_body, positions, indices, indices_order_hint)

    _set_partial_cartographic_with_colors_metadata = { "offset" : _set_partial_cartographic_with_colors_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic_with_colors(self, central_body:str, positions:list, colors:list, indices:list) -> None:
        """For convenience. Updates a subset of positions and/or colors in a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_cartographic_with_colors_metadata, central_body, positions, colors, indices)

    _set_partial_cartographic_with_colors_indices_order_and_render_pass_metadata = { "offset" : _set_partial_cartographic_with_colors_indices_order_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_cartographic_with_colors_indices_order_and_render_pass(self, central_body:str, positions:list, colors:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Updates a subset of positions and/or colors in a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_cartographic_with_colors_indices_order_and_render_pass_metadata, central_body, positions, colors, indices, indices_order_hint, render_pass_hint)

    _get_central_body_clipped_metadata = { "offset" : _get_central_body_clipped_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def central_body_clipped(self) -> bool:
        """Get or set whether the polyline will be clipped by the central body."""
        return self._intf.get_property(PolylinePrimitive._metadata, PolylinePrimitive._get_central_body_clipped_metadata)

    _set_central_body_clipped_metadata = { "offset" : _set_central_body_clipped_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @central_body_clipped.setter
    def central_body_clipped(self, central_body_clipped:bool) -> None:
        return self._intf.set_property(PolylinePrimitive._metadata, PolylinePrimitive._set_central_body_clipped_metadata, central_body_clipped)

    _set_with_colors_and_optional_parameters_metadata = { "offset" : _set_with_colors_and_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("PolylinePrimitiveOptionalParameters"),) }
    def set_with_colors_and_optional_parameters(self, positions:list, colors:list, optional_parameters:"PolylinePrimitiveOptionalParameters") -> None:
        """Define the positions, colors, and/or optional point properties of a polyline. The polyline is rendered in its reference frame."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_with_colors_and_optional_parameters_metadata, positions, colors, optional_parameters)

    _set_cartographic_with_colors_and_optional_parameters_metadata = { "offset" : _set_cartographic_with_colors_and_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("PolylinePrimitiveOptionalParameters"),) }
    def set_cartographic_with_colors_and_optional_parameters(self, central_body:str, positions:list, colors:list, optional_parameters:"PolylinePrimitiveOptionalParameters") -> None:
        """For convenience. Defines the positions, colors, and/or optional point properties of a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_cartographic_with_colors_and_optional_parameters_metadata, central_body, positions, colors, optional_parameters)

    _set_partial_with_colors_and_optional_parameters_metadata = { "offset" : _set_partial_with_colors_and_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("PolylinePrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_with_colors_and_optional_parameters(self, positions:list, colors:list, optional_parameters:"PolylinePrimitiveOptionalParameters", indices:list) -> None:
        """Update a subset of positions, colors, and/or optional point properties in a polyline."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_with_colors_and_optional_parameters_metadata, positions, colors, optional_parameters, indices)

    _set_partial_cartographic_with_optional_parameters_metadata = { "offset" : _set_partial_cartographic_with_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("PolylinePrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic_with_optional_parameters(self, central_body:str, positions:list, colors:list, optional_parameters:"PolylinePrimitiveOptionalParameters", indices:list) -> None:
        """For convenience. Updates a subset of positions, colors, and/or optional point properties in a polyline using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(PolylinePrimitive._metadata, PolylinePrimitive._set_partial_cartographic_with_optional_parameters_metadata, central_body, positions, colors, optional_parameters, indices)

    _property_names[width] = "width"
    _property_names[minimum_width_supported] = "minimum_width_supported"
    _property_names[maximum_width_supported] = "maximum_width_supported"
    _property_names[position_interpolator] = "position_interpolator"
    _property_names[polyline_type] = "polyline_type"
    _property_names[set_hint] = "set_hint"
    _property_names[display_outline] = "display_outline"
    _property_names[outline_color] = "outline_color"
    _property_names[outline_translucency] = "outline_translucency"
    _property_names[outline_width] = "outline_width"
    _property_names[per_item_picking_enabled] = "per_item_picking_enabled"
    _property_names[central_body_clipped] = "central_body_clipped"

    def __init__(self, source_object=None):
        """Construct an object of type PolylinePrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PolylinePrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PolylinePrimitive, [PolylinePrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5123752712456450241, 10739590127227499430), PolylinePrimitive)
agcls.AgTypeNameMap["PolylinePrimitive"] = PolylinePrimitive

class PolylinePrimitiveOptionalParameters(SupportsDeleteCallback):
    """Optional per-point or per-segment parameters for polyline primitive that overrides the polyline primitive's global parameters..."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_time_intervals_method_offset = 1
    _metadata = {
        "iid_data" : (4899695596699319324, 7572311148448238225),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PolylinePrimitiveOptionalParameters)

    _set_time_intervals_metadata = { "offset" : _set_time_intervals_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_time_intervals(self, time_intervals:list) -> None:
        """Define a collection of TimeIntervals defined by MinimumTime and MaximumTime in Epoch Seconds, one for each point in the Polyline."""
        return self._intf.invoke(PolylinePrimitiveOptionalParameters._metadata, PolylinePrimitiveOptionalParameters._set_time_intervals_metadata, time_intervals)


    def __init__(self, source_object=None):
        """Construct an object of type PolylinePrimitiveOptionalParameters."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PolylinePrimitiveOptionalParameters)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PolylinePrimitiveOptionalParameters, [PolylinePrimitiveOptionalParameters, ])

agcls.AgClassCatalog.add_catalog_entry((5054854826939456422, 15110470621112478879), PolylinePrimitiveOptionalParameters)
agcls.AgTypeNameMap["PolylinePrimitiveOptionalParameters"] = PolylinePrimitiveOptionalParameters

class PositionInterpolator(IPositionInterpolator, SupportsDeleteCallback):
    """Position interpolators compute positions based on a collection of input positions. Position interpolators are used in conjunction with the polyline primitive to render things such as great arcs and rhumb lines."""
    def __init__(self, source_object=None):
        """Construct an object of type PositionInterpolator."""
        SupportsDeleteCallback.__init__(self)
        IPositionInterpolator.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPositionInterpolator._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PositionInterpolator, [IPositionInterpolator])

agcls.AgClassCatalog.add_catalog_entry((5719643167839805508, 13043993113695684767), PositionInterpolator)
agcls.AgTypeNameMap["PositionInterpolator"] = PositionInterpolator

class Primitive(IPrimitive, SupportsDeleteCallback):
    """Primitives represent objects rendered in the 3D scene."""
    def __init__(self, source_object=None):
        """Construct an object of type Primitive."""
        SupportsDeleteCallback.__init__(self)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Primitive, [IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((4738072676921658624, 14848128681026293391), Primitive)
agcls.AgTypeNameMap["Primitive"] = Primitive

class PrimitiveManager(SupportsDeleteCallback):
    """The primitive manager contains spatial data structures used to efficiently render primitives. Once a primitive is constructed, it must be added to the primitive manager before it will be rendered."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_precision_exponent_method_offset = 2
    _set_precision_exponent_method_offset = 3
    _get_translucent_primitives_sort_order_method_offset = 4
    _set_translucent_primitives_sort_order_method_offset = 5
    _add_method_offset = 6
    _remove_method_offset = 7
    _contains_method_offset = 8
    _clear_method_offset = 9
    _get__new_enum_method_offset = 10
    _metadata = {
        "iid_data" : (5712467222312543641, 18164203771391217825),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PrimitiveManager)
    def __iter__(self):
        """Create an iterator for the PrimitiveManager object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> typing.Any:
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of primitives in the manager."""
        return self._intf.get_property(PrimitiveManager._metadata, PrimitiveManager._get_count_metadata)

    _get_precision_exponent_metadata = { "offset" : _get_precision_exponent_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def precision_exponent(self) -> int:
        """Do not use this property, as it is deprecated. This property is no longer in use Gets or sets the exponent used to compute the maximum precision for primitive rendering. For example, a value of -3 indicates the maximum precision of 2^-3, 0.125 m along the x, y, or z axis..."""
        return self._intf.get_property(PrimitiveManager._metadata, PrimitiveManager._get_precision_exponent_metadata)

    _set_precision_exponent_metadata = { "offset" : _set_precision_exponent_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @precision_exponent.setter
    def precision_exponent(self, precision_exponent:int) -> None:
        return self._intf.set_property(PrimitiveManager._metadata, PrimitiveManager._set_precision_exponent_metadata, precision_exponent)

    _get_translucent_primitives_sort_order_metadata = { "offset" : _get_translucent_primitives_sort_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PrimitivesSortOrder),) }
    @property
    def translucent_primitives_sort_order(self) -> "PrimitivesSortOrder":
        """Get or set the sort order for translucent primitives in the primitive manager. This determines a trade-off between rendering speed and quality..."""
        return self._intf.get_property(PrimitiveManager._metadata, PrimitiveManager._get_translucent_primitives_sort_order_metadata)

    _set_translucent_primitives_sort_order_metadata = { "offset" : _set_translucent_primitives_sort_order_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PrimitivesSortOrder),) }
    @translucent_primitives_sort_order.setter
    def translucent_primitives_sort_order(self, translucent_primitives_sort_order:"PrimitivesSortOrder") -> None:
        return self._intf.set_property(PrimitiveManager._metadata, PrimitiveManager._set_translucent_primitives_sort_order_metadata, translucent_primitives_sort_order)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"),) }
    def add(self, primitive:"IPrimitive") -> None:
        """Add a primitive to the manager. Primitives must be added to the manager to be rendered."""
        return self._intf.invoke(PrimitiveManager._metadata, PrimitiveManager._add_metadata, primitive)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"),) }
    def remove(self, primitive:"IPrimitive") -> None:
        """Remove a primitive from the manager. The primitive is no longer rendered unless it is added back into the manager."""
        return self._intf.invoke(PrimitiveManager._metadata, PrimitiveManager._remove_metadata, primitive)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IPrimitive"), agmarshall.VariantBoolArg,) }
    def contains(self, primitive:"IPrimitive") -> bool:
        """Determine whether the manager contains a primitive."""
        return self._intf.invoke(PrimitiveManager._metadata, PrimitiveManager._contains_metadata, primitive, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all primitives from the manager."""
        return self._intf.invoke(PrimitiveManager._metadata, PrimitiveManager._clear_metadata, )

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that iterates through the collection. The order of the primitives is not guaranteed to be the order that the primitives were added."""
        return self._intf.get_property(PrimitiveManager._metadata, PrimitiveManager._get__new_enum_metadata)

    _property_names[count] = "count"
    _property_names[precision_exponent] = "precision_exponent"
    _property_names[translucent_primitives_sort_order] = "translucent_primitives_sort_order"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type PrimitiveManager."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PrimitiveManager)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PrimitiveManager, [PrimitiveManager, ])

agcls.AgClassCatalog.add_catalog_entry((5759528432845737503, 11218549662454612891), PrimitiveManager)
agcls.AgTypeNameMap["PrimitiveManager"] = PrimitiveManager

class RasterImageGlobeOverlay(IGlobeImageOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe image overlay for handling rasters."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_transparent_color_method_offset = 1
    _set_use_transparent_color_method_offset = 2
    _get_transparent_color_method_offset = 3
    _set_transparent_color_method_offset = 4
    _metadata = {
        "iid_data" : (4649284088551794065, 16792077003963540662),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RasterImageGlobeOverlay)

    _get_use_transparent_color_metadata = { "offset" : _get_use_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_transparent_color(self) -> bool:
        """Get or set whether transparent color should be used."""
        return self._intf.get_property(RasterImageGlobeOverlay._metadata, RasterImageGlobeOverlay._get_use_transparent_color_metadata)

    _set_use_transparent_color_metadata = { "offset" : _set_use_transparent_color_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_transparent_color.setter
    def use_transparent_color(self, use_transparent_color:bool) -> None:
        return self._intf.set_property(RasterImageGlobeOverlay._metadata, RasterImageGlobeOverlay._set_use_transparent_color_metadata, use_transparent_color)

    _get_transparent_color_metadata = { "offset" : _get_transparent_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def transparent_color(self) -> agcolor.Color:
        """Get or set the color that will become transparent."""
        return self._intf.get_property(RasterImageGlobeOverlay._metadata, RasterImageGlobeOverlay._get_transparent_color_metadata)

    _set_transparent_color_metadata = { "offset" : _set_transparent_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @transparent_color.setter
    def transparent_color(self, transparent_color:agcolor.Color) -> None:
        return self._intf.set_property(RasterImageGlobeOverlay._metadata, RasterImageGlobeOverlay._set_transparent_color_metadata, transparent_color)

    _property_names[use_transparent_color] = "use_transparent_color"
    _property_names[transparent_color] = "transparent_color"

    def __init__(self, source_object=None):
        """Construct an object of type RasterImageGlobeOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RasterImageGlobeOverlay)
        IGlobeImageOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IGlobeImageOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RasterImageGlobeOverlay, [RasterImageGlobeOverlay, IGlobeImageOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5590537877066628758, 18201123914066150057), RasterImageGlobeOverlay)
agcls.AgTypeNameMap["RasterImageGlobeOverlay"] = RasterImageGlobeOverlay

class RhumbLineInterpolator(IPositionInterpolator, SupportsDeleteCallback):
    """The rhumb line interpolator computes interpolated positions along a rhumb line. Rhumb lines are lines of constant bearing. They appear as straight lines on a Mercator 2D map projection and are well suited to navigation."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_method_offset = 1
    _set_central_body_method_offset = 2
    _get_granularity_method_offset = 3
    _set_granularity_method_offset = 4
    _metadata = {
        "iid_data" : (5614337469871736325, 9221184782651004840),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RhumbLineInterpolator)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get or set the central body used when interpolating with interpolate."""
        return self._intf.get_property(RhumbLineInterpolator._metadata, RhumbLineInterpolator._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(RhumbLineInterpolator._metadata, RhumbLineInterpolator._set_central_body_metadata, central_body)

    _get_granularity_metadata = { "offset" : _get_granularity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def granularity(self) -> float:
        """Get or set the granularity used when interpolating with interpolate. Lower granularities are more precise but create more positions."""
        return self._intf.get_property(RhumbLineInterpolator._metadata, RhumbLineInterpolator._get_granularity_metadata)

    _set_granularity_metadata = { "offset" : _set_granularity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @granularity.setter
    def granularity(self, granularity:float) -> None:
        return self._intf.set_property(RhumbLineInterpolator._metadata, RhumbLineInterpolator._set_granularity_metadata, granularity)

    _property_names[central_body] = "central_body"
    _property_names[granularity] = "granularity"

    def __init__(self, source_object=None):
        """Construct an object of type RhumbLineInterpolator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RhumbLineInterpolator)
        IPositionInterpolator.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPositionInterpolator._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RhumbLineInterpolator, [RhumbLineInterpolator, IPositionInterpolator])

agcls.AgClassCatalog.add_catalog_entry((4905411456932095490, 5486952346846465436), RhumbLineInterpolator)
agcls.AgTypeNameMap["RhumbLineInterpolator"] = RhumbLineInterpolator

class Scene(SupportsDeleteCallback):
    """
    A scene provides properties and functionality that are reflected in the rendering of the globe control that it is associated with. An globe control's scene is available from the scene property...

    Examples
    --------
    Control the Display of Stars and Water Texture:
    >>> # Scenario scenario: Scenario object
    >>> # Turn off the stars and water texture
    >>> manager = scenario.scene_manager
    >>> manager.scenes.item(0).show_stars = False
    >>> manager.scenes.item(0).show_water_surface = False
    """

    _num_methods = 25
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_camera_method_offset = 1
    _get_lighting_method_offset = 2
    _get_show_sunshine_method_offset = 3
    _set_show_sunshine_method_offset = 4
    _get_central_bodies_method_offset = 5
    _get_background_color_method_offset = 6
    _set_background_color_method_offset = 7
    _get_shade_sky_based_on_altitude_method_offset = 8
    _set_shade_sky_based_on_altitude_method_offset = 9
    _get_show_stars_method_offset = 10
    _set_show_stars_method_offset = 11
    _get_globe_overlay_settings_method_offset = 12
    _render_method_offset = 13
    _pick_method_offset = 14
    _pick_rectangular_method_offset = 15
    _pick_screen_overlays_method_offset = 16
    _get_scene_id_method_offset = 17
    _get_show_water_surface_method_offset = 18
    _set_show_water_surface_method_offset = 19
    _get_anti_aliasing_method_offset = 20
    _set_anti_aliasing_method_offset = 21
    _get_visual_effects_method_offset = 22
    _get_clouds_method_offset = 23
    _get_show_star_labels_method_offset = 24
    _set_show_star_labels_method_offset = 25
    _metadata = {
        "iid_data" : (5305222447490254466, 9009928902241702280),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Scene)
    def subscribe(self) -> ISceneEventHandler:
        """Return an ISceneEventHandler that is subscribed to handle events associated with this instance of Scene."""
        return ISceneEventHandler(self._intf)

    _get_camera_metadata = { "offset" : _get_camera_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def camera(self) -> "Camera":
        """Get the camera associated with the scene, which affects the view that is rendered by the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_camera_metadata)

    _get_lighting_metadata = { "offset" : _get_lighting_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def lighting(self) -> "Lighting":
        """Get the lighting associated with the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_lighting_metadata)

    _get_show_sunshine_metadata = { "offset" : _get_show_sunshine_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_sunshine(self) -> bool:
        """Get or set whether sunshine is rendered by the Sun central body. Sunshine renders a halo effect around the sun when it is viewed in the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_show_sunshine_metadata)

    _set_show_sunshine_metadata = { "offset" : _set_show_sunshine_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_sunshine.setter
    def show_sunshine(self, show_sunshine:bool) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_show_sunshine_metadata, show_sunshine)

    _get_central_bodies_metadata = { "offset" : _get_central_bodies_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def central_bodies(self) -> "CentralBodyGraphicsIndexer":
        """Get the central body graphics for a specified central body."""
        return self._intf.get_property(Scene._metadata, Scene._get_central_bodies_metadata)

    _get_background_color_metadata = { "offset" : _get_background_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def background_color(self) -> agcolor.Color:
        """Get or set the background color of the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_background_color_metadata)

    _set_background_color_metadata = { "offset" : _set_background_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @background_color.setter
    def background_color(self, background_color:agcolor.Color) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_background_color_metadata, background_color)

    _get_shade_sky_based_on_altitude_metadata = { "offset" : _get_shade_sky_based_on_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def shade_sky_based_on_altitude(self) -> bool:
        """Get or set whether the sky will be shaded based on camera altitude. When shade sky based on altitude is set to true, the sky will become more blue as the Camera gets closer to the surface of the central body."""
        return self._intf.get_property(Scene._metadata, Scene._get_shade_sky_based_on_altitude_metadata)

    _set_shade_sky_based_on_altitude_metadata = { "offset" : _set_shade_sky_based_on_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @shade_sky_based_on_altitude.setter
    def shade_sky_based_on_altitude(self, shade_sky_based_on_altitude:bool) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_shade_sky_based_on_altitude_metadata, shade_sky_based_on_altitude)

    _get_show_stars_metadata = { "offset" : _get_show_stars_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_stars(self) -> bool:
        """Get or set whether stars are shown or hidden in the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_show_stars_metadata)

    _set_show_stars_metadata = { "offset" : _set_show_stars_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_stars.setter
    def show_stars(self, show_stars:bool) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_show_stars_metadata, show_stars)

    _get_globe_overlay_settings_metadata = { "offset" : _get_globe_overlay_settings_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def globe_overlay_settings(self) -> "SceneGlobeOverlaySettings":
        """Get the scene globe overlay settings for the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_globe_overlay_settings_metadata)

    _render_metadata = { "offset" : _render_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def render(self) -> None:
        """Render the scene. To render all the scenes within an application, use the Render method."""
        return self._intf.invoke(Scene._metadata, Scene._render_metadata, )

    _pick_metadata = { "offset" : _pick_method_offset,
            "arg_types" : (agcom.INT, agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def pick(self, x:int, y:int) -> "PickResultCollection":
        """Execute a pick at the given x, y and returns a depth sorted collection of picked objects. The coordinate origin is top, left. To pick screen overlays, use the PickScreenOverlays method."""
        return self._intf.invoke(Scene._metadata, Scene._pick_metadata, x, y, OutArg())

    _pick_rectangular_metadata = { "offset" : _pick_rectangular_method_offset,
            "arg_types" : (agcom.INT, agcom.INT, agcom.INT, agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg, agmarshall.IntArg, agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def pick_rectangular(self, left:int, bottom:int, right:int, top:int) -> "PickResultCollection":
        """Execute a pick in the given rectangular region and returns a depth sorted collection of picked objects. The coordinate origin is top, left. To pick screen overlays, use the PickScreenOverlays method."""
        return self._intf.invoke(Scene._metadata, Scene._pick_rectangular_metadata, left, bottom, right, top, OutArg())

    _pick_screen_overlays_metadata = { "offset" : _pick_screen_overlays_method_offset,
            "arg_types" : (agcom.INT, agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def pick_screen_overlays(self, x:int, y:int) -> "ScreenOverlayPickResultCollection":
        """Execute a pick on screen overlays at the given x, y and returns a front to back sorted collection of picked overlays. The coordinate origin is top, left. To pick other objects in the scene, use the Pick method."""
        return self._intf.invoke(Scene._metadata, Scene._pick_screen_overlays_metadata, x, y, OutArg())

    _get_scene_id_metadata = { "offset" : _get_scene_id_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def scene_id(self) -> int:
        """Return the scene identifier."""
        return self._intf.get_property(Scene._metadata, Scene._get_scene_id_metadata)

    _get_show_water_surface_metadata = { "offset" : _get_show_water_surface_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_water_surface(self) -> bool:
        """Get or set whether water surface on earth is shown or hidden in the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_show_water_surface_metadata)

    _set_show_water_surface_metadata = { "offset" : _set_show_water_surface_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_water_surface.setter
    def show_water_surface(self, show_water_surface:bool) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_show_water_surface_metadata, show_water_surface)

    _get_anti_aliasing_metadata = { "offset" : _get_anti_aliasing_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AntiAliasingMethod),) }
    @property
    def anti_aliasing(self) -> "AntiAliasingMethod":
        """Get or set the multisample anti-aliasing (MSAA) option for this scene. As the level of anti-aliasing increases, performance will generally decrease, but the quality of the anti-aliasing will improve."""
        return self._intf.get_property(Scene._metadata, Scene._get_anti_aliasing_metadata)

    _set_anti_aliasing_metadata = { "offset" : _set_anti_aliasing_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AntiAliasingMethod),) }
    @anti_aliasing.setter
    def anti_aliasing(self, anti_aliasing:"AntiAliasingMethod") -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_anti_aliasing_metadata, anti_aliasing)

    _get_visual_effects_metadata = { "offset" : _get_visual_effects_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def visual_effects(self) -> "VisualEffects":
        """Get the visual  effects associated with the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_visual_effects_metadata)

    _get_clouds_metadata = { "offset" : _get_clouds_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def clouds(self) -> "Clouds":
        """Get the clouds for the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_clouds_metadata)

    _get_show_star_labels_metadata = { "offset" : _get_show_star_labels_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_star_labels(self) -> bool:
        """Get or set whether stars labels are shown or hidden in the scene."""
        return self._intf.get_property(Scene._metadata, Scene._get_show_star_labels_metadata)

    _set_show_star_labels_metadata = { "offset" : _set_show_star_labels_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_star_labels.setter
    def show_star_labels(self, show_star_labels:bool) -> None:
        return self._intf.set_property(Scene._metadata, Scene._set_show_star_labels_metadata, show_star_labels)

    _property_names[camera] = "camera"
    _property_names[lighting] = "lighting"
    _property_names[show_sunshine] = "show_sunshine"
    _property_names[central_bodies] = "central_bodies"
    _property_names[background_color] = "background_color"
    _property_names[shade_sky_based_on_altitude] = "shade_sky_based_on_altitude"
    _property_names[show_stars] = "show_stars"
    _property_names[globe_overlay_settings] = "globe_overlay_settings"
    _property_names[scene_id] = "scene_id"
    _property_names[show_water_surface] = "show_water_surface"
    _property_names[anti_aliasing] = "anti_aliasing"
    _property_names[visual_effects] = "visual_effects"
    _property_names[clouds] = "clouds"
    _property_names[show_star_labels] = "show_star_labels"

    def __init__(self, source_object=None):
        """Construct an object of type Scene."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Scene)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Scene, [Scene, ])

agcls.AgClassCatalog.add_catalog_entry((5425863975955114481, 4372717846621692570), Scene)
agcls.AgTypeNameMap["Scene"] = Scene

class SceneDisplayCondition(IDisplayCondition, SupportsDeleteCallback):
    """A display condition used to control what scene or scenes an object, such as a primitive, is rendered in. This is used to show an object in some scenes and hide it in others."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_display_in_scene_method_offset = 1
    _get_display_in_scene_method_offset = 2
    _display_only_in_scene_method_offset = 3
    _metadata = {
        "iid_data" : (5047996913438946447, 14608630073827982726),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SceneDisplayCondition)

    _set_display_in_scene_metadata = { "offset" : _set_display_in_scene_method_offset,
            "arg_types" : (agcom.PVOID, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.InterfaceInArg("Scene"), agmarshall.VariantBoolArg,) }
    def set_display_in_scene(self, scene:"Scene", on:bool) -> None:
        """Allow or disallows rendering for in a particular scene for."""
        return self._intf.invoke(SceneDisplayCondition._metadata, SceneDisplayCondition._set_display_in_scene_metadata, scene, on)

    _get_display_in_scene_metadata = { "offset" : _get_display_in_scene_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("Scene"), agmarshall.VariantBoolArg,) }
    def get_display_in_scene(self, scene:"Scene") -> bool:
        """Determine whether the display condition allows rendering rendering in the given scene."""
        return self._intf.invoke(SceneDisplayCondition._metadata, SceneDisplayCondition._get_display_in_scene_metadata, scene, OutArg())

    _display_only_in_scene_metadata = { "offset" : _display_only_in_scene_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Scene"),) }
    def display_only_in_scene(self, scene:"Scene") -> None:
        """Allow rendering only in the given scene. The display condition will not allow rendering in other scenes, including newly created ones."""
        return self._intf.invoke(SceneDisplayCondition._metadata, SceneDisplayCondition._display_only_in_scene_metadata, scene)


    def __init__(self, source_object=None):
        """Construct an object of type SceneDisplayCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SceneDisplayCondition)
        IDisplayCondition.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplayCondition._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SceneDisplayCondition, [SceneDisplayCondition, IDisplayCondition])

agcls.AgClassCatalog.add_catalog_entry((5504831685330777187, 14908496243111103623), SceneDisplayCondition)
agcls.AgTypeNameMap["SceneDisplayCondition"] = SceneDisplayCondition

class SceneManager(SupportsDeleteCallback):
    """The static scene manager class provides global properties and functionality that apply to all scenes and thus affect the rendering of every globe control..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_primitives_method_offset = 1
    _get_screen_overlays_method_offset = 2
    _get_textures_method_offset = 3
    _get_globe_overlay_settings_method_offset = 4
    _get_scenes_method_offset = 5
    _render_method_offset = 6
    _get_initializers_method_offset = 7
    _get_frame_rate_method_offset = 8
    _metadata = {
        "iid_data" : (5338880585643038703, 1390793286196929705),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SceneManager)

    _get_primitives_metadata = { "offset" : _get_primitives_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def primitives(self) -> "PrimitiveManager":
        """Get the primitive manager, which is used to add primitives to your scenes."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_primitives_metadata)

    _get_screen_overlays_metadata = { "offset" : _get_screen_overlays_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def screen_overlays(self) -> "ScreenOverlayManager":
        """Get the screen overlay manager, which is used to add screen overlays to your scenes."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_screen_overlays_metadata)

    _get_textures_metadata = { "offset" : _get_textures_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def textures(self) -> "Texture2DFactory":
        """Get the texture 2d factory, which can be used to create textures from various sources."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_textures_metadata)

    _get_globe_overlay_settings_metadata = { "offset" : _get_globe_overlay_settings_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def globe_overlay_settings(self) -> "GlobeOverlaySettings":
        """Get the globe overlay settings, which are used to set global settings for all globe overlays."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_globe_overlay_settings_metadata)

    _get_scenes_metadata = { "offset" : _get_scenes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scenes(self) -> "SceneCollection":
        """Get a read-only collection of scenes that are associated with the scene manager."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_scenes_metadata)

    _render_metadata = { "offset" : _render_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def render(self) -> None:
        """Render all scenes within an application. To render a specific scene, use the Render method."""
        return self._intf.invoke(SceneManager._metadata, SceneManager._render_metadata, )

    _get_initializers_metadata = { "offset" : _get_initializers_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def initializers(self) -> "FactoryAndInitializers":
        """Allow the user to create or initialize primitives, display conditions, tringulators and other types of objects."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_initializers_metadata)

    _get_frame_rate_metadata = { "offset" : _get_frame_rate_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def frame_rate(self) -> "FrameRate":
        """Get the frame rate class, which can be used to keep track of how fast scenes are being <see ref='Render'>rendered</see>."""
        return self._intf.get_property(SceneManager._metadata, SceneManager._get_frame_rate_metadata)

    _property_names[primitives] = "primitives"
    _property_names[screen_overlays] = "screen_overlays"
    _property_names[textures] = "textures"
    _property_names[globe_overlay_settings] = "globe_overlay_settings"
    _property_names[scenes] = "scenes"
    _property_names[initializers] = "initializers"
    _property_names[frame_rate] = "frame_rate"

    def __init__(self, source_object=None):
        """Construct an object of type SceneManager."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SceneManager)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SceneManager, [SceneManager, ])

agcls.AgClassCatalog.add_catalog_entry((5055513188644242239, 14100567721448590776), SceneManager)
agcls.AgTypeNameMap["SceneManager"] = SceneManager

class ScreenOverlay(IScreenOverlay, IOverlay, IScreenOverlayContainer, SupportsDeleteCallback):
    """A visible element drawn in screen space. Overlays are useful for floating logos, heads up displays, and integrating user interfaces into the 3D window."""
    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlay."""
        SupportsDeleteCallback.__init__(self)
        IScreenOverlay.__init__(self, source_object)
        IOverlay.__init__(self, source_object)
        IScreenOverlayContainer.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IScreenOverlay._private_init(self, intf)
        IOverlay._private_init(self, intf)
        IScreenOverlayContainer._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlay, [IScreenOverlay, IOverlay, IScreenOverlayContainer])

agcls.AgClassCatalog.add_catalog_entry((5135883245183464378, 7950147496367229875), ScreenOverlay)
agcls.AgTypeNameMap["ScreenOverlay"] = ScreenOverlay

class ScreenOverlayCollection(IScreenOverlayCollectionBase, SupportsDeleteCallback):
    """A collection of screen overlays."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5262217257495667236, 17532824272504716714),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ScreenOverlayCollection)


    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlayCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ScreenOverlayCollection)
        IScreenOverlayCollectionBase.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IScreenOverlayCollectionBase._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlayCollection, [ScreenOverlayCollection, IScreenOverlayCollectionBase])

agcls.AgClassCatalog.add_catalog_entry((5428143835074754195, 7795253318375075755), ScreenOverlayCollection)
agcls.AgTypeNameMap["ScreenOverlayCollection"] = ScreenOverlayCollection

class ScreenOverlayManager(IScreenOverlayCollectionBase, IScreenOverlayContainer, SupportsDeleteCallback):
    """The top-level container for screen overlays. All child screen overlays that are added to this container are specified relative to the overall globe control."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_bounds_method_offset = 1
    _get_overlays_method_offset = 2
    _get_padding_method_offset = 3
    _set_padding_method_offset = 4
    _get_display_method_offset = 5
    _set_display_method_offset = 6
    _metadata = {
        "iid_data" : (5252209601608109701, 10720637566953264540),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ScreenOverlayManager)

    _get_bounds_metadata = { "offset" : _get_bounds_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def bounds(self) -> list:
        """Get the overall bounds of the globe control. The array contains the properties defining the bounds in the order left x location, top y location, width, height."""
        return self._intf.get_property(ScreenOverlayManager._metadata, ScreenOverlayManager._get_bounds_metadata)

    _get_overlays_metadata = { "offset" : _get_overlays_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlays(self) -> "ScreenOverlayCollection":
        """Get the collection of overlays that are contained within this manager."""
        return self._intf.get_property(ScreenOverlayManager._metadata, ScreenOverlayManager._get_overlays_metadata)

    _get_padding_metadata = { "offset" : _get_padding_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def padding(self) -> list:
        """Get or set the padding surrounding the overlays that are contained within this manager. The array contains the components of the padding arranged in the order left, top, right, bottom."""
        return self._intf.get_property(ScreenOverlayManager._metadata, ScreenOverlayManager._get_padding_metadata)

    _set_padding_metadata = { "offset" : _set_padding_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @padding.setter
    def padding(self, padding:list) -> None:
        return self._intf.set_property(ScreenOverlayManager._metadata, ScreenOverlayManager._set_padding_metadata, padding)

    _get_display_metadata = { "offset" : _get_display_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display(self) -> bool:
        """Get or set if the collection of overlays that are contained within this manager should be rendered."""
        return self._intf.get_property(ScreenOverlayManager._metadata, ScreenOverlayManager._get_display_metadata)

    _set_display_metadata = { "offset" : _set_display_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display.setter
    def display(self, display:bool) -> None:
        return self._intf.set_property(ScreenOverlayManager._metadata, ScreenOverlayManager._set_display_metadata, display)

    _property_names[bounds] = "bounds"
    _property_names[overlays] = "overlays"
    _property_names[padding] = "padding"
    _property_names[display] = "display"

    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlayManager."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ScreenOverlayManager)
        IScreenOverlayCollectionBase.__init__(self, source_object)
        IScreenOverlayContainer.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IScreenOverlayCollectionBase._private_init(self, intf)
        IScreenOverlayContainer._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlayManager, [ScreenOverlayManager, IScreenOverlayCollectionBase, IScreenOverlayContainer])

agcls.AgClassCatalog.add_catalog_entry((4941666073412103198, 3199753952958351539), ScreenOverlayManager)
agcls.AgTypeNameMap["ScreenOverlayManager"] = ScreenOverlayManager

class ScreenOverlayPickResult(SupportsDeleteCallback):
    """Describes a picked screen overlay as a result of a call to pick screen overlays."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_position_method_offset = 1
    _get_control_position_method_offset = 2
    _get_overlay_method_offset = 3
    _metadata = {
        "iid_data" : (5121641088858368209, 13022394605157415062),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ScreenOverlayPickResult)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get the position that was picked within the picked overlay. The array represents the picked position and has a size of 4. The elements are in the order x position, y position, x screen overlay unit, y screen overlay unit."""
        return self._intf.get_property(ScreenOverlayPickResult._metadata, ScreenOverlayPickResult._get_position_metadata)

    _get_control_position_metadata = { "offset" : _get_control_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def control_position(self) -> list:
        """Get the position that was picked within the overall globe control. This is essentially the same position that was passed to pick screen overlays. The array represents the picked position and has a size of 4..."""
        return self._intf.get_property(ScreenOverlayPickResult._metadata, ScreenOverlayPickResult._get_control_position_metadata)

    _get_overlay_metadata = { "offset" : _get_overlay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def overlay(self) -> "IScreenOverlay":
        """Get the screen overlay that was picked."""
        return self._intf.get_property(ScreenOverlayPickResult._metadata, ScreenOverlayPickResult._get_overlay_metadata)

    _property_names[position] = "position"
    _property_names[control_position] = "control_position"
    _property_names[overlay] = "overlay"

    def __init__(self, source_object=None):
        """Construct an object of type ScreenOverlayPickResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ScreenOverlayPickResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ScreenOverlayPickResult, [ScreenOverlayPickResult, ])

agcls.AgClassCatalog.add_catalog_entry((5411628992035762574, 18017194983832957605), ScreenOverlayPickResult)
agcls.AgTypeNameMap["ScreenOverlayPickResult"] = ScreenOverlayPickResult

class SolidPrimitive(IPrimitive, SupportsDeleteCallback):
    """
    Render filled solid objects and their outlines. Example solids include boxes and ellipsoids. Various effects are supported, such as displaying the solid's silhouette, and hiding the outline of the backside of the solid...

    Examples
    --------
    Draw a Solid Cylinder Primitive and set properties:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> originCylinder = root.conversion_utility.new_position_on_earth()
    >>> originCylinder.assign_geodetic(0, 7, 100)
    >>>
    >>> orientCylinder = root.conversion_utility.new_orientation()
    >>> orientCylinder.assign_az_el(0, 0, AzElAboutBoresight.ROTATE)
    >>>
    >>> cylinder = manager.initializers.cylinder_triangulator.create_simple(200, 100)
    >>> solidCylinder = manager.initializers.solid_primitive.initialize()
    >>> solidCylinder.reference_frame = root.central_bodies.earth.analysis_workbench_components.systems.item("Fixed")
    >>> solidCylinder.position = originCylinder.query_cartesian_array()
    >>> solidCylinder.set_with_result(cylinder)
    >>> solidCylinder.color = Colors.Lime
    >>> solidCylinder.outline_color = Colors.Blue
    >>> solidCylinder.outline_width = 3
    >>> solidCylinder.translucency = 0.75
    >>> solidCylinder.rotation = orientCylinder
    >>> manager.primitives.add(solidCylinder)
    >>> manager.render()

    Draw a Solid Ellipsoid Primitive and set properties:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> originEllipsoid = root.conversion_utility.new_position_on_earth()
    >>> originEllipsoid.assign_geodetic(0, 5, 100)
    >>>
    >>> orientEllipsoid = root.conversion_utility.new_orientation()
    >>> orientEllipsoid.assign_az_el(0, 0, AzElAboutBoresight.ROTATE)
    >>>
    >>> radii = [[200], [100], [100]]
    >>> ellipsoid = manager.initializers.ellipsoid_triangulator.compute_simple(radii)
    >>> solidEllipsoid = manager.initializers.solid_primitive.initialize()
    >>> solidEllipsoid.reference_frame = root.central_bodies.earth.analysis_workbench_components.systems.item(
    >>>     "Fixed"
    >>> )  # vgtSat.Systems.item('Body')
    >>> solidEllipsoid.position = originEllipsoid.query_cartesian_array()
    >>> solidEllipsoid.set_with_result(ellipsoid)
    >>> solidEllipsoid.color = Colors.White
    >>> solidEllipsoid.outline_color = Colors.DeepPink
    >>> solidEllipsoid.translucency = 0.75
    >>> solidEllipsoid.rotation = orientEllipsoid
    >>> manager.primitives.add(solidEllipsoid)
    >>> manager.render()

    Draw a Solid Box Primitive and set properties:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> originBox = root.conversion_utility.new_position_on_earth()
    >>> originBox.assign_geodetic(0, 3, 100)
    >>>
    >>> orientBox = root.conversion_utility.new_orientation()
    >>> orientBox.assign_az_el(0, 0, AzElAboutBoresight.ROTATE)
    >>>
    >>> size = [[100], [100], [200]]
    >>> result = manager.initializers.box_triangulator.compute(size)
    >>> solidBox = manager.initializers.solid_primitive.initialize()
    >>> solidBox.reference_frame = root.central_bodies.earth.analysis_workbench_components.systems.item("Fixed")
    >>> solidBox.position = originBox.query_cartesian_array()
    >>> solidBox.set_with_result(result)
    >>> solidBox.color = Colors.Red
    >>> solidBox.outline_color = Colors.Cyan
    >>> solidBox.translucency = 0.75
    >>> solidBox.rotation = orientBox
    >>> manager.primitives.add(solidBox)
    >>> manager.render()
    """

    _num_methods = 39
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_affected_by_lighting_method_offset = 1
    _set_affected_by_lighting_method_offset = 2
    _get_display_fill_method_offset = 3
    _set_display_fill_method_offset = 4
    _get_display_silhouette_method_offset = 5
    _set_display_silhouette_method_offset = 6
    _get_silhouette_color_method_offset = 7
    _set_silhouette_color_method_offset = 8
    _get_silhouette_translucency_method_offset = 9
    _set_silhouette_translucency_method_offset = 10
    _get_silhouette_width_method_offset = 11
    _set_silhouette_width_method_offset = 12
    _get_minimum_silhouette_width_supported_method_offset = 13
    _get_maximum_silhouette_width_supported_method_offset = 14
    _get_display_outline_method_offset = 15
    _set_display_outline_method_offset = 16
    _get_outline_color_method_offset = 17
    _set_outline_color_method_offset = 18
    _get_outline_translucency_method_offset = 19
    _set_outline_translucency_method_offset = 20
    _get_outline_width_method_offset = 21
    _set_outline_width_method_offset = 22
    _get_outline_appearance_method_offset = 23
    _set_outline_appearance_method_offset = 24
    _get_back_line_color_method_offset = 25
    _set_back_line_color_method_offset = 26
    _get_back_line_translucency_method_offset = 27
    _set_back_line_translucency_method_offset = 28
    _get_position_method_offset = 29
    _set_position_method_offset = 30
    _get_rotation_method_offset = 31
    _set_rotation_method_offset = 32
    _get_scale_method_offset = 33
    _set_scale_method_offset = 34
    _get_back_line_width_method_offset = 35
    _set_back_line_width_method_offset = 36
    _get_set_hint_method_offset = 37
    _set_with_result_method_offset = 38
    _set_method_offset = 39
    _metadata = {
        "iid_data" : (5543346789797488819, 2660259113497939099),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SolidPrimitive)

    _get_affected_by_lighting_metadata = { "offset" : _get_affected_by_lighting_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def affected_by_lighting(self) -> bool:
        """Get or set whether the primitive is affected by lighting."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_affected_by_lighting_metadata)

    _set_affected_by_lighting_metadata = { "offset" : _set_affected_by_lighting_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @affected_by_lighting.setter
    def affected_by_lighting(self, affected_by_lighting:bool) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_affected_by_lighting_metadata, affected_by_lighting)

    _get_display_fill_metadata = { "offset" : _get_display_fill_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_fill(self) -> bool:
        """Get or set whether the solid's fill is displayed."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_display_fill_metadata)

    _set_display_fill_metadata = { "offset" : _set_display_fill_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_fill.setter
    def display_fill(self, display_fill:bool) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_display_fill_metadata, display_fill)

    _get_display_silhouette_metadata = { "offset" : _get_display_silhouette_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_silhouette(self) -> bool:
        """Get or set whether the solid's silhouette is displayed."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_display_silhouette_metadata)

    _set_display_silhouette_metadata = { "offset" : _set_display_silhouette_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_silhouette.setter
    def display_silhouette(self, display_silhouette:bool) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_display_silhouette_metadata, display_silhouette)

    _get_silhouette_color_metadata = { "offset" : _get_silhouette_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def silhouette_color(self) -> agcolor.Color:
        """Get or set the silhouette's color."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_silhouette_color_metadata)

    _set_silhouette_color_metadata = { "offset" : _set_silhouette_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @silhouette_color.setter
    def silhouette_color(self, silhouette_color:agcolor.Color) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_silhouette_color_metadata, silhouette_color)

    _get_silhouette_translucency_metadata = { "offset" : _get_silhouette_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def silhouette_translucency(self) -> float:
        """Get or set the silhouette's translucency. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_silhouette_translucency_metadata)

    _set_silhouette_translucency_metadata = { "offset" : _set_silhouette_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @silhouette_translucency.setter
    def silhouette_translucency(self, silhouette_translucency:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_silhouette_translucency_metadata, silhouette_translucency)

    _get_silhouette_width_metadata = { "offset" : _get_silhouette_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def silhouette_width(self) -> float:
        """Get or set the silhouette' width, in pixels."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_silhouette_width_metadata)

    _set_silhouette_width_metadata = { "offset" : _set_silhouette_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @silhouette_width.setter
    def silhouette_width(self, silhouette_width:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_silhouette_width_metadata, silhouette_width)

    _get_minimum_silhouette_width_supported_metadata = { "offset" : _get_minimum_silhouette_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def minimum_silhouette_width_supported(self) -> float:
        """Get the minimum silhouette width, in pixels, supported by the video card."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_minimum_silhouette_width_supported_metadata)

    _get_maximum_silhouette_width_supported_metadata = { "offset" : _get_maximum_silhouette_width_supported_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def maximum_silhouette_width_supported(self) -> float:
        """Get the maximum silhouette width, in pixels, supported by the video card."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_maximum_silhouette_width_supported_metadata)

    _get_display_outline_metadata = { "offset" : _get_display_outline_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def display_outline(self) -> bool:
        """Get or set whether the solid's outline is displayed."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_display_outline_metadata)

    _set_display_outline_metadata = { "offset" : _set_display_outline_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @display_outline.setter
    def display_outline(self, display_outline:bool) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_display_outline_metadata, display_outline)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """Get or set the outline's color."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_outline_color_metadata, outline_color)

    _get_outline_translucency_metadata = { "offset" : _get_outline_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_translucency(self) -> float:
        """Get or set the outline's translucency. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_outline_translucency_metadata)

    _set_outline_translucency_metadata = { "offset" : _set_outline_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_translucency.setter
    def outline_translucency(self, outline_translucency:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_outline_translucency_metadata, outline_translucency)

    _get_outline_width_metadata = { "offset" : _get_outline_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_width(self) -> float:
        """Get or set the outline's width, in pixels."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_outline_width_metadata)

    _set_outline_width_metadata = { "offset" : _set_outline_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_width.setter
    def outline_width(self, outline_width:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_outline_width_metadata, outline_width)

    _get_outline_appearance_metadata = { "offset" : _get_outline_appearance_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(OutlineAppearance),) }
    @property
    def outline_appearance(self) -> "OutlineAppearance":
        """Get or set the outline's appearance."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_outline_appearance_metadata)

    _set_outline_appearance_metadata = { "offset" : _set_outline_appearance_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(OutlineAppearance),) }
    @outline_appearance.setter
    def outline_appearance(self, outline_appearance:"OutlineAppearance") -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_outline_appearance_metadata, outline_appearance)

    _get_back_line_color_metadata = { "offset" : _get_back_line_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def back_line_color(self) -> agcolor.Color:
        """Get or set the back line's color."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_back_line_color_metadata)

    _set_back_line_color_metadata = { "offset" : _set_back_line_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @back_line_color.setter
    def back_line_color(self, back_line_color:agcolor.Color) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_back_line_color_metadata, back_line_color)

    _get_back_line_translucency_metadata = { "offset" : _get_back_line_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def back_line_translucency(self) -> float:
        """Get or set the back line's translucency. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_back_line_translucency_metadata)

    _set_back_line_translucency_metadata = { "offset" : _set_back_line_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @back_line_translucency.setter
    def back_line_translucency(self, back_line_translucency:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_back_line_translucency_metadata, back_line_translucency)

    _get_position_metadata = { "offset" : _get_position_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def position(self) -> list:
        """Get or set the solid's position. The position is defined in the solid's reference frame. The array contains the components of the position in the order x, y, z."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_position_metadata)

    _set_position_metadata = { "offset" : _set_position_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @position.setter
    def position(self, position:list) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_position_metadata, position)

    _get_rotation_metadata = { "offset" : _get_rotation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def rotation(self) -> "IOrientation":
        """Get or set the rotation applied to the solid before rendering."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_rotation_metadata)

    _set_rotation_metadata = { "offset" : _set_rotation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IOrientation"),) }
    @rotation.setter
    def rotation(self, rotation:"IOrientation") -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_rotation_metadata, rotation)

    _get_scale_metadata = { "offset" : _get_scale_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def scale(self) -> list:
        """Get or set a non-uniform scale that is applied to the solid to increase or decrease its rendered size. The array contains the scale for each component of the size in the order x scale, y scale, z scale."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_scale_metadata)

    _set_scale_metadata = { "offset" : _set_scale_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @scale.setter
    def scale(self, scale:list) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_scale_metadata, scale)

    _get_back_line_width_metadata = { "offset" : _get_back_line_width_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def back_line_width(self) -> float:
        """Get or set the back line's width, in pixels."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_back_line_width_metadata)

    _set_back_line_width_metadata = { "offset" : _set_back_line_width_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @back_line_width.setter
    def back_line_width(self, back_line_width:float) -> None:
        return self._intf.set_property(SolidPrimitive._metadata, SolidPrimitive._set_back_line_width_metadata, back_line_width)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(SolidPrimitive._metadata, SolidPrimitive._get_set_hint_metadata)

    _set_with_result_metadata = { "offset" : _set_with_result_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SolidTriangulatorResult"),) }
    def set_with_result(self, solid_triangulator_result:"SolidTriangulatorResult") -> None:
        """Define the solid using the specified solidTriangulatorResult. The solid is rendered in the primitive's reference frame."""
        return self._intf.invoke(SolidPrimitive._metadata, SolidPrimitive._set_with_result_metadata, solid_triangulator_result)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.PVOID, agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(WindingOrder), agmarshall.InterfaceInArg("BoundingSphere"), agmarshall.VariantBoolArg,) }
    def set(self, positions:list, normals:list, indices:list, outline_indices:list, winding_order:"WindingOrder", bounding_sphere:"BoundingSphere", closed:bool) -> None:
        """Define the solid using the specified parameters. The solid is rendered in the primitive's reference frame."""
        return self._intf.invoke(SolidPrimitive._metadata, SolidPrimitive._set_metadata, positions, normals, indices, outline_indices, winding_order, bounding_sphere, closed)

    _property_names[affected_by_lighting] = "affected_by_lighting"
    _property_names[display_fill] = "display_fill"
    _property_names[display_silhouette] = "display_silhouette"
    _property_names[silhouette_color] = "silhouette_color"
    _property_names[silhouette_translucency] = "silhouette_translucency"
    _property_names[silhouette_width] = "silhouette_width"
    _property_names[minimum_silhouette_width_supported] = "minimum_silhouette_width_supported"
    _property_names[maximum_silhouette_width_supported] = "maximum_silhouette_width_supported"
    _property_names[display_outline] = "display_outline"
    _property_names[outline_color] = "outline_color"
    _property_names[outline_translucency] = "outline_translucency"
    _property_names[outline_width] = "outline_width"
    _property_names[outline_appearance] = "outline_appearance"
    _property_names[back_line_color] = "back_line_color"
    _property_names[back_line_translucency] = "back_line_translucency"
    _property_names[position] = "position"
    _property_names[rotation] = "rotation"
    _property_names[scale] = "scale"
    _property_names[back_line_width] = "back_line_width"
    _property_names[set_hint] = "set_hint"

    def __init__(self, source_object=None):
        """Construct an object of type SolidPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SolidPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SolidPrimitive, [SolidPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5239544537293274560, 13352578250398471069), SolidPrimitive)
agcls.AgTypeNameMap["SolidPrimitive"] = SolidPrimitive

class Stereoscopic(SupportsDeleteCallback):
    """Get the stereoscopic options for all Scenes. To use a particular stereoscopic display mode, ensure that your system supports the feature and that it is enabled."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_display_mode_method_offset = 1
    _set_display_mode_method_offset = 2
    _get_projection_mode_method_offset = 3
    _set_projection_mode_method_offset = 4
    _get_projection_distance_method_offset = 5
    _set_projection_distance_method_offset = 6
    _get_eye_separation_factor_method_offset = 7
    _set_eye_separation_factor_method_offset = 8
    _metadata = {
        "iid_data" : (4779955330586042651, 5663574790190885304),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Stereoscopic)

    _get_display_mode_metadata = { "offset" : _get_display_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StereoscopicDisplayMode),) }
    @property
    def display_mode(self) -> "StereoscopicDisplayMode":
        """Get or set the stereoscopic display mode for all Scenes. To use a particular stereoscopic display mode, ensure that your system supports the feature and that it is enabled."""
        return self._intf.get_property(Stereoscopic._metadata, Stereoscopic._get_display_mode_metadata)

    _set_display_mode_metadata = { "offset" : _set_display_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StereoscopicDisplayMode),) }
    @display_mode.setter
    def display_mode(self, display_mode:"StereoscopicDisplayMode") -> None:
        return self._intf.set_property(Stereoscopic._metadata, Stereoscopic._set_display_mode_metadata, display_mode)

    _get_projection_mode_metadata = { "offset" : _get_projection_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StereoProjectionMode),) }
    @property
    def projection_mode(self) -> "StereoProjectionMode":
        """Get or set whether the type of stereo projection that will be used."""
        return self._intf.get_property(Stereoscopic._metadata, Stereoscopic._get_projection_mode_metadata)

    _set_projection_mode_metadata = { "offset" : _set_projection_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StereoProjectionMode),) }
    @projection_mode.setter
    def projection_mode(self, projection_mode:"StereoProjectionMode") -> None:
        return self._intf.set_property(Stereoscopic._metadata, Stereoscopic._set_projection_mode_metadata, projection_mode)

    _get_projection_distance_metadata = { "offset" : _get_projection_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def projection_distance(self) -> float:
        """Get or set the projection distance. If projection mode is set to eStkGraphicsStereoProjectionAutomatic, the value of this property will be ignored."""
        return self._intf.get_property(Stereoscopic._metadata, Stereoscopic._get_projection_distance_metadata)

    _set_projection_distance_metadata = { "offset" : _set_projection_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @projection_distance.setter
    def projection_distance(self, projection_distance:float) -> None:
        return self._intf.set_property(Stereoscopic._metadata, Stereoscopic._set_projection_distance_metadata, projection_distance)

    _get_eye_separation_factor_metadata = { "offset" : _get_eye_separation_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def eye_separation_factor(self) -> float:
        """Get or set the eye separation factor."""
        return self._intf.get_property(Stereoscopic._metadata, Stereoscopic._get_eye_separation_factor_metadata)

    _set_eye_separation_factor_metadata = { "offset" : _set_eye_separation_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @eye_separation_factor.setter
    def eye_separation_factor(self, eye_separation_factor:float) -> None:
        return self._intf.set_property(Stereoscopic._metadata, Stereoscopic._set_eye_separation_factor_metadata, eye_separation_factor)

    _property_names[display_mode] = "display_mode"
    _property_names[projection_mode] = "projection_mode"
    _property_names[projection_distance] = "projection_distance"
    _property_names[eye_separation_factor] = "eye_separation_factor"

    def __init__(self, source_object=None):
        """Construct an object of type Stereoscopic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Stereoscopic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Stereoscopic, [Stereoscopic, ])

agcls.AgClassCatalog.add_catalog_entry((5004629829167520124, 7449422875768445375), Stereoscopic)
agcls.AgTypeNameMap["Stereoscopic"] = Stereoscopic

class SurfaceMeshPrimitive(IPrimitive, SupportsDeleteCallback):
    """
    A triangle mesh primitive for meshes on the surface that need to conform to terrain.

    Examples
    --------
    Draw a new Surface Mesh:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> cartesianPts = [
    >>>     [6030.721052],
    >>>     [1956.627139],
    >>>     [-692.397578],
    >>>     [5568.375825],
    >>>     [2993.600713],
    >>>     [-841.076362],
    >>>     [5680.743568],
    >>>     [2490.379622],
    >>>     [-1480.882721],
    >>> ]  # X, Y, Z (km)
    >>>
    >>> triangles = manager.initializers.surface_polygon_triangulator.compute("Earth", cartesianPts)
    >>> surfaceMesh = manager.initializers.surface_mesh_primitive.initialize()
    >>> surfaceMesh.color = Colors.Red
    >>> surfaceMesh.set(triangles)
    >>> manager.primitives.add(surfaceMesh)
    >>> manager.render()
    """

    _num_methods = 17
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_texture_method_offset = 1
    _set_texture_method_offset = 2
    _get_wireframe_method_offset = 3
    _set_wireframe_method_offset = 4
    _get_triangle_winding_order_method_offset = 5
    _get_set_hint_method_offset = 6
    _get_rendering_method_method_offset = 7
    _get_texture_filter_method_offset = 8
    _set_texture_filter_method_offset = 9
    _get_texture_matrix_method_offset = 10
    _set_texture_matrix_method_offset = 11
    _get_transparent_texture_border_method_offset = 12
    _set_transparent_texture_border_method_offset = 13
    _set_method_offset = 14
    _set_without_texturing_method_offset = 15
    _supported_method_offset = 16
    _supported_with_default_rendering_method_method_offset = 17
    _metadata = {
        "iid_data" : (4690670744547618907, 12009834772259629965),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SurfaceMeshPrimitive)

    _get_texture_metadata = { "offset" : _get_texture_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture(self) -> "RendererTexture2D":
        """Get or set the texture applied to this primitive when rendering."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_texture_metadata)

    _set_texture_metadata = { "offset" : _set_texture_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("RendererTexture2D"),) }
    @texture.setter
    def texture(self, texture:"RendererTexture2D") -> None:
        return self._intf.set_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_texture_metadata, texture)

    _get_wireframe_metadata = { "offset" : _get_wireframe_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def wireframe(self) -> bool:
        """Get or set whether the primitive is rendered in wireframe. This is useful for debugging."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_wireframe_metadata)

    _set_wireframe_metadata = { "offset" : _set_wireframe_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @wireframe.setter
    def wireframe(self, wireframe:bool) -> None:
        return self._intf.set_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_wireframe_metadata, wireframe)

    _get_triangle_winding_order_metadata = { "offset" : _get_triangle_winding_order_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(WindingOrder),) }
    @property
    def triangle_winding_order(self) -> "WindingOrder":
        """Get the orientation of front-facing triangles in the mesh."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_triangle_winding_order_metadata)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_set_hint_metadata)

    _get_rendering_method_metadata = { "offset" : _get_rendering_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SurfaceMeshRenderingMethod),) }
    @property
    def rendering_method(self) -> "SurfaceMeshRenderingMethod":
        """Get the rendering method used to render the mesh."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_rendering_method_metadata)

    _get_texture_filter_metadata = { "offset" : _get_texture_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_filter(self) -> "TextureFilter2D":
        """Get or set the filter used when a texture is applied to this primitive."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_texture_filter_metadata)

    _set_texture_filter_metadata = { "offset" : _set_texture_filter_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TextureFilter2D"),) }
    @texture_filter.setter
    def texture_filter(self, texture_filter:"TextureFilter2D") -> None:
        return self._intf.set_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_texture_filter_metadata, texture_filter)

    _get_texture_matrix_metadata = { "offset" : _get_texture_matrix_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_matrix(self) -> "TextureMatrix":
        """Get or set the matrix used to transform texture coordinates when a texture is applied to this primitive."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_texture_matrix_metadata)

    _set_texture_matrix_metadata = { "offset" : _set_texture_matrix_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TextureMatrix"),) }
    @texture_matrix.setter
    def texture_matrix(self, texture_matrix:"TextureMatrix") -> None:
        return self._intf.set_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_texture_matrix_metadata, texture_matrix)

    _get_transparent_texture_border_metadata = { "offset" : _get_transparent_texture_border_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def transparent_texture_border(self) -> bool:
        """Get or set the boolean that defines if the color obtained from texture coordinates beyond the texture border should be considered transparent or not. This is typically used in conjunction with the a texture matrix."""
        return self._intf.get_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._get_transparent_texture_border_metadata)

    _set_transparent_texture_border_metadata = { "offset" : _set_transparent_texture_border_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @transparent_texture_border.setter
    def transparent_texture_border(self, transparent_texture_border:bool) -> None:
        return self._intf.set_property(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_transparent_texture_border_metadata, transparent_texture_border)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SurfaceTriangulatorResult"),) }
    def set(self, surface_triangulator:"SurfaceTriangulatorResult") -> None:
        """Define the surface mesh using the specified surfaceTriangulator. The mesh is rendered in the primitive's reference frame."""
        return self._intf.invoke(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_metadata, surface_triangulator)

    _set_without_texturing_metadata = { "offset" : _set_without_texturing_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("SurfaceTriangulatorResult"),) }
    def set_without_texturing(self, surface_triangulator:"SurfaceTriangulatorResult") -> None:
        """Define the surface mesh using the specified surfaceTriangulator. The mesh is rendered in the primitive's reference frame."""
        return self._intf.invoke(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._set_without_texturing_metadata, surface_triangulator)

    _supported_metadata = { "offset" : _supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(SurfaceMeshRenderingMethod), agmarshall.VariantBoolArg,) }
    def supported(self, rendering_method:"SurfaceMeshRenderingMethod") -> bool:
        """Determine whether or not the video card supports the surface mesh primitive with the given renderingMethod."""
        return self._intf.invoke(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._supported_metadata, rendering_method, OutArg())

    _supported_with_default_rendering_method_metadata = { "offset" : _supported_with_default_rendering_method_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    def supported_with_default_rendering_method(self) -> bool:
        """Determine whether or not the video card supports the surface mesh primitive. This is equivalent to calling Supported with automatic."""
        return self._intf.invoke(SurfaceMeshPrimitive._metadata, SurfaceMeshPrimitive._supported_with_default_rendering_method_metadata, OutArg())

    _property_names[texture] = "texture"
    _property_names[wireframe] = "wireframe"
    _property_names[triangle_winding_order] = "triangle_winding_order"
    _property_names[set_hint] = "set_hint"
    _property_names[rendering_method] = "rendering_method"
    _property_names[texture_filter] = "texture_filter"
    _property_names[texture_matrix] = "texture_matrix"
    _property_names[transparent_texture_border] = "transparent_texture_border"

    def __init__(self, source_object=None):
        """Construct an object of type SurfaceMeshPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SurfaceMeshPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SurfaceMeshPrimitive, [SurfaceMeshPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5411334062372739956, 6728209386876429451), SurfaceMeshPrimitive)
agcls.AgTypeNameMap["SurfaceMeshPrimitive"] = SurfaceMeshPrimitive

class TerrainOverlayCollection(SupportsDeleteCallback):
    """A collection of terrain overlay objects."""

    _num_methods = 20
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_count_method_offset = 1
    _get_is_read_only_method_offset = 2
    _item_method_offset = 3
    _get__new_enum_method_offset = 4
    _contains_method_offset = 5
    _contains_uri_string_method_offset = 6
    _remove_method_offset = 7
    _clear_method_offset = 8
    _add_method_offset = 9
    _add_async_method_offset = 10
    _index_of_method_offset = 11
    _index_of_uri_string_method_offset = 12
    _add_uri_string_method_offset = 13
    _add_async_uri_string_method_offset = 14
    _swap_method_offset = 15
    _swap_by_index_method_offset = 16
    _move_method_offset = 17
    _move_by_index_method_offset = 18
    _bring_to_front_method_offset = 19
    _send_to_back_method_offset = 20
    _metadata = {
        "iid_data" : (4834722276085425062, 14155078082643029135),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TerrainOverlayCollection)
    def __iter__(self):
        """Create an iterator for the TerrainOverlayCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITerrainOverlay":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval
    def subscribe(self) -> ITerrainOverlayCollectionEventHandler:
        """Return an ITerrainOverlayCollectionEventHandler that is subscribed to handle events associated with this instance of TerrainOverlayCollection."""
        return ITerrainOverlayCollectionEventHandler(self._intf)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of terrain overlay objects in the collection."""
        return self._intf.get_property(TerrainOverlayCollection._metadata, TerrainOverlayCollection._get_count_metadata)

    _get_is_read_only_metadata = { "offset" : _get_is_read_only_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_read_only(self) -> bool:
        """Get whether or not the collection is read only."""
        return self._intf.get_property(TerrainOverlayCollection._metadata, TerrainOverlayCollection._get_is_read_only_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "ITerrainOverlay":
        """Get the terrain overlay at the specified index."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Construct an iterator that can be used to iterate the collection."""
        return self._intf.get_property(TerrainOverlayCollection._metadata, TerrainOverlayCollection._get__new_enum_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"), agmarshall.VariantBoolArg,) }
    def contains(self, terrain_overlay:"ITerrainOverlay") -> bool:
        """Check the presence of a terrain overlay in the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._contains_metadata, terrain_overlay, OutArg())

    _contains_uri_string_metadata = { "offset" : _contains_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains_uri_string(self, string_uri:str) -> bool:
        """Check the presence of a terrain overlay with the specified Uri in the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._contains_uri_string_metadata, string_uri, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"), agmarshall.VariantBoolArg,) }
    def remove(self, terrain_overlay:"ITerrainOverlay") -> bool:
        """Remove a terrain overlay from the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._remove_metadata, terrain_overlay, OutArg())

    _clear_metadata = { "offset" : _clear_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear(self) -> None:
        """Remove all terrain overlay objects from the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._clear_metadata, )

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"),) }
    def add(self, terrain_overlay:"ITerrainOverlay") -> None:
        """Add terrainOverlay to the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._add_metadata, terrain_overlay)

    _add_async_metadata = { "offset" : _add_async_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"),) }
    def add_async(self, terrain_overlay:"ITerrainOverlay") -> None:
        """Add terrainOverlay to the collection asynchronously."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._add_async_metadata, terrain_overlay)

    _index_of_metadata = { "offset" : _index_of_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.INT),),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"), agmarshall.IntArg,) }
    def index_of(self, terrain_overlay:"ITerrainOverlay") -> int:
        """Get the index of the specified terrain overlay."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._index_of_metadata, terrain_overlay, OutArg())

    _index_of_uri_string_metadata = { "offset" : _index_of_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.INT),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.IntArg,) }
    def index_of_uri_string(self, string_uri:str) -> int:
        """Get the index of the terrain overlay with the specified Uri."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._index_of_uri_string_metadata, string_uri, OutArg())

    _add_uri_string_metadata = { "offset" : _add_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_uri_string(self, uri:str) -> "ITerrainOverlay":
        """Create a terrain overlay from the uri, which represents a uri, and adds it to the collection."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._add_uri_string_metadata, uri, OutArg())

    _add_async_uri_string_metadata = { "offset" : _add_async_uri_string_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_async_uri_string(self, uri:str) -> "ITerrainOverlay":
        """Create a terrain overlay from the uri, which represents a uri, and adds it to the collection asynchronously."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._add_async_uri_string_metadata, uri, OutArg())

    _swap_metadata = { "offset" : _swap_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"), agmarshall.InterfaceInArg("ITerrainOverlay"),) }
    def swap(self, terrain_overlay1:"ITerrainOverlay", terrain_overlay2:"ITerrainOverlay") -> None:
        """Swap the position of two terrain overlay objects."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._swap_metadata, terrain_overlay1, terrain_overlay2)

    _swap_by_index_metadata = { "offset" : _swap_by_index_method_offset,
            "arg_types" : (agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg,) }
    def swap_by_index(self, index1:int, index2:int) -> None:
        """Swap the position of two terrain overlay objects at the specified indices."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._swap_by_index_metadata, index1, index2)

    _move_metadata = { "offset" : _move_method_offset,
            "arg_types" : (agcom.PVOID, agcom.INT,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"), agmarshall.IntArg,) }
    def move(self, terrain_overlay:"ITerrainOverlay", new_position:int) -> None:
        """Move the terrain overlay to the specified position."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._move_metadata, terrain_overlay, new_position)

    _move_by_index_metadata = { "offset" : _move_by_index_method_offset,
            "arg_types" : (agcom.INT, agcom.INT,),
            "marshallers" : (agmarshall.IntArg, agmarshall.IntArg,) }
    def move_by_index(self, index:int, new_position:int) -> None:
        """Move the terrain overlay at the specified index to the specified position."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._move_by_index_metadata, index, new_position)

    _bring_to_front_metadata = { "offset" : _bring_to_front_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"),) }
    def bring_to_front(self, terrain_overlay:"ITerrainOverlay") -> None:
        """Brings the terrain overlay to the front of the collection so it is rendered first or on the bottom."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._bring_to_front_metadata, terrain_overlay)

    _send_to_back_metadata = { "offset" : _send_to_back_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITerrainOverlay"),) }
    def send_to_back(self, terrain_overlay:"ITerrainOverlay") -> None:
        """Send the terrain overlay to the back of the collection so it is rendered last or on the top."""
        return self._intf.invoke(TerrainOverlayCollection._metadata, TerrainOverlayCollection._send_to_back_metadata, terrain_overlay)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[is_read_only] = "is_read_only"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TerrainOverlayCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TerrainOverlayCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TerrainOverlayCollection, [TerrainOverlayCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5026986988139254343, 12046810184580739503), TerrainOverlayCollection)
agcls.AgTypeNameMap["TerrainOverlayCollection"] = TerrainOverlayCollection

class TerrainOverlay(ITerrainOverlay, IGlobeOverlay, SupportsDeleteCallback):
    """A globe overlay which shows terrain."""
    def __init__(self, source_object=None):
        """Construct an object of type TerrainOverlay."""
        SupportsDeleteCallback.__init__(self)
        ITerrainOverlay.__init__(self, source_object)
        IGlobeOverlay.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITerrainOverlay._private_init(self, intf)
        IGlobeOverlay._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TerrainOverlay, [ITerrainOverlay, IGlobeOverlay])

agcls.AgClassCatalog.add_catalog_entry((5009137188527673329, 1512801370180160660), TerrainOverlay)
agcls.AgTypeNameMap["TerrainOverlay"] = TerrainOverlay

class TextBatchPrimitive(IPrimitive, SupportsDeleteCallback):
    """
    Render one or more strings in the 3D scene. For best performance, avoid creating lots of batches with only a few strings each. See the Batching Performance Overview.

    Examples
    --------
    Draw a new Text Primitive:
    >>> # Scenario scenario: Scenario object
    >>> manager = scenario.scene_manager
    >>> font = manager.initializers.graphics_font.initialize_with_name_size_font_style_outline(
    >>>     "MS Sans Serif", 24, FontStyle.BOLD, True
    >>> )
    >>> textBatch = manager.initializers.text_batch_primitive.initialize_with_graphics_font(font)
    >>> textBatch.set_cartographic("Earth", [[0], [0], [0]], ["Example Text"])  # Lat, Lon, Alt
    >>> manager.primitives.add(textBatch)
    """

    _num_methods = 32
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_set_hint_method_offset = 1
    _get_bounding_sphere_scale_method_offset = 2
    _set_bounding_sphere_scale_method_offset = 3
    _get_font_method_offset = 4
    _get_outline_color_method_offset = 5
    _set_outline_color_method_offset = 6
    _get_outline_translucency_method_offset = 7
    _set_outline_translucency_method_offset = 8
    _get_align_to_pixel_method_offset = 9
    _set_align_to_pixel_method_offset = 10
    _get_distance_display_condition_per_string_method_offset = 11
    _set_distance_display_condition_per_string_method_offset = 12
    _get_per_item_picking_enabled_method_offset = 13
    _set_per_item_picking_enabled_method_offset = 14
    _get_texture_filter_method_offset = 15
    _set_texture_filter_method_offset = 16
    _set_method_offset = 17
    _set_with_optional_parameters_method_offset = 18
    _set_with_optional_parameters_and_render_pass_method_offset = 19
    _set_cartographic_method_offset = 20
    _set_cartographic_with_optional_parameters_method_offset = 21
    _set_cartographic_with_optional_parameters_and_render_pass_method_offset = 22
    _set_partial_method_offset = 23
    _set_partial_with_indices_order_method_offset = 24
    _set_partial_with_optional_parameters_method_offset = 25
    _set_partial_with_optional_parameters_indices_order_and_render_pass_method_offset = 26
    _set_partial_cartographic_method_offset = 27
    _set_partial_cartographic_with_indices_order_method_offset = 28
    _set_partial_cartographic_with_optional_parameters_method_offset = 29
    _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_method_offset = 30
    _get_render_in_screen_space_method_offset = 31
    _set_render_in_screen_space_method_offset = 32
    _metadata = {
        "iid_data" : (4634979056355844859, 3124701436410088883),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextBatchPrimitive)

    _get_set_hint_metadata = { "offset" : _get_set_hint_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SetHint),) }
    @property
    def set_hint(self) -> "SetHint":
        """Get the primitive's set hint. See the Set Hint Performance Overview for selecting an appropriate value to construct the primitive with."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_set_hint_metadata)

    _get_bounding_sphere_scale_metadata = { "offset" : _get_bounding_sphere_scale_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def bounding_sphere_scale(self) -> float:
        """Get or set the scale applied to the radius of this primitive's bounding sphere."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_bounding_sphere_scale_metadata)

    _set_bounding_sphere_scale_metadata = { "offset" : _set_bounding_sphere_scale_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @bounding_sphere_scale.setter
    def bounding_sphere_scale(self, bounding_sphere_scale:float) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_bounding_sphere_scale_metadata, bounding_sphere_scale)

    _get_font_metadata = { "offset" : _get_font_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def font(self) -> "GraphicsFont":
        """Get the font used to render the text batch."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_font_metadata)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """Get or set the text's outline color."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_outline_color_metadata, outline_color)

    _get_outline_translucency_metadata = { "offset" : _get_outline_translucency_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def outline_translucency(self) -> float:
        """Get or set the text's outline translucency. Translucency is between 0 and 1, where 0 is opaque and 1 is transparent."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_outline_translucency_metadata)

    _set_outline_translucency_metadata = { "offset" : _set_outline_translucency_method_offset,
            "arg_types" : (agcom.FLOAT,),
            "marshallers" : (agmarshall.FloatArg,) }
    @outline_translucency.setter
    def outline_translucency(self, outline_translucency:float) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_outline_translucency_metadata, outline_translucency)

    _get_align_to_pixel_metadata = { "offset" : _get_align_to_pixel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def align_to_pixel(self) -> bool:
        """Get or set whether the screen space position of each string is aligned to a pixel."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_align_to_pixel_metadata)

    _set_align_to_pixel_metadata = { "offset" : _set_align_to_pixel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @align_to_pixel.setter
    def align_to_pixel(self, align_to_pixel:bool) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_align_to_pixel_metadata, align_to_pixel)

    _get_distance_display_condition_per_string_metadata = { "offset" : _get_distance_display_condition_per_string_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def distance_display_condition_per_string(self) -> "DistanceDisplayCondition":
        """Get or set a distance display condition that is evaluated per string in the text batch during rendering. This is different than display condition, which is evaluated once for the entire text batch..."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_distance_display_condition_per_string_metadata)

    _set_distance_display_condition_per_string_metadata = { "offset" : _set_distance_display_condition_per_string_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("DistanceDisplayCondition"),) }
    @distance_display_condition_per_string.setter
    def distance_display_condition_per_string(self, distance_display_condition_per_string:"DistanceDisplayCondition") -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_distance_display_condition_per_string_metadata, distance_display_condition_per_string)

    _get_per_item_picking_enabled_metadata = { "offset" : _get_per_item_picking_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def per_item_picking_enabled(self) -> bool:
        """Get or set whether individual text indices will be included in the pick results returned from the scene's Pick method. Each text index that is picked will be returned as a batch primitive index."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_per_item_picking_enabled_metadata)

    _set_per_item_picking_enabled_metadata = { "offset" : _set_per_item_picking_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @per_item_picking_enabled.setter
    def per_item_picking_enabled(self, per_item_picking_enabled:bool) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_per_item_picking_enabled_metadata, per_item_picking_enabled)

    _get_texture_filter_metadata = { "offset" : _get_texture_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def texture_filter(self) -> "TextureFilter2D":
        """Get or set the filter used to filter the texture-based font."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_texture_filter_metadata)

    _set_texture_filter_metadata = { "offset" : _set_texture_filter_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TextureFilter2D"),) }
    @texture_filter.setter
    def texture_filter(self, texture_filter:"TextureFilter2D") -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_texture_filter_metadata, texture_filter)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set(self, positions:list, text:list) -> None:
        """Define the positions and text of strings in a text batch. The strings are rendered in the primitive's reference frame."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_metadata, positions, text)

    _set_with_optional_parameters_metadata = { "offset" : _set_with_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"),) }
    def set_with_optional_parameters(self, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters") -> None:
        """Define the positions, text, and optional parameters of strings in a text batch. The strings are rendered in the primitive's reference frame."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_with_optional_parameters_metadata, positions, text, optional_parameters)

    _set_with_optional_parameters_and_render_pass_metadata = { "offset" : _set_with_optional_parameters_and_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.EnumArg(RenderPassHint),) }
    def set_with_optional_parameters_and_render_pass(self, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", render_pass_hint:"RenderPassHint") -> None:
        """Define the positions, text, and optional parameters of strings in a text batch. The strings are rendered in the primitive's reference frame. renderPassHint is provided for efficiency."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_with_optional_parameters_and_render_pass_metadata, positions, text, optional_parameters, render_pass_hint)

    _set_cartographic_metadata = { "offset" : _set_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_cartographic(self, central_body:str, positions:list, text:list) -> None:
        """For convenience. Defines the positions and text of strings in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_cartographic_metadata, central_body, positions, text)

    _set_cartographic_with_optional_parameters_metadata = { "offset" : _set_cartographic_with_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"),) }
    def set_cartographic_with_optional_parameters(self, central_body:str, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters") -> None:
        """For convenience. Defines the positions, text, and optional parameters of strings in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_cartographic_with_optional_parameters_metadata, central_body, positions, text, optional_parameters)

    _set_cartographic_with_optional_parameters_and_render_pass_metadata = { "offset" : _set_cartographic_with_optional_parameters_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.EnumArg(RenderPassHint),) }
    def set_cartographic_with_optional_parameters_and_render_pass(self, central_body:str, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Defines the positions, text, and optional parameters of strings in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling Set."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_cartographic_with_optional_parameters_and_render_pass_metadata, central_body, positions, text, optional_parameters, render_pass_hint)

    _set_partial_metadata = { "offset" : _set_partial_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial(self, positions:list, text:list, indices:list) -> None:
        """Update a subset of positions and/or text in a text batch."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_metadata, positions, text, indices)

    _set_partial_with_indices_order_metadata = { "offset" : _set_partial_with_indices_order_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_with_indices_order(self, positions:list, text:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """Update a subset of positions and/or text in a text batch."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_with_indices_order_metadata, positions, text, indices, indices_order_hint)

    _set_partial_with_optional_parameters_metadata = { "offset" : _set_partial_with_optional_parameters_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_with_optional_parameters(self, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", indices:list) -> None:
        """Update a subset of positions, text, and/or optional per-string parameters in a text batch."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_with_optional_parameters_metadata, positions, text, optional_parameters, indices)

    _set_partial_with_optional_parameters_indices_order_and_render_pass_metadata = { "offset" : _set_partial_with_optional_parameters_indices_order_and_render_pass_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_with_optional_parameters_indices_order_and_render_pass(self, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """Update a subset of positions, text, and/or optional per-string parameters in a text batch."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_with_optional_parameters_indices_order_and_render_pass_metadata, positions, text, optional_parameters, indices, indices_order_hint, render_pass_hint)

    _set_partial_cartographic_metadata = { "offset" : _set_partial_cartographic_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic(self, central_body:str, positions:list, text:list, indices:list) -> None:
        """For convenience. Updates a subset of positions and/or text in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_cartographic_metadata, central_body, positions, text, indices)

    _set_partial_cartographic_with_indices_order_metadata = { "offset" : _set_partial_cartographic_with_indices_order_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint),) }
    def set_partial_cartographic_with_indices_order(self, central_body:str, positions:list, text:list, indices:list, indices_order_hint:"PrimitiveIndicesOrderHint") -> None:
        """For convenience. Updates a subset of positions and/or text in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_cartographic_with_indices_order_metadata, central_body, positions, text, indices, indices_order_hint)

    _set_partial_cartographic_with_optional_parameters_metadata = { "offset" : _set_partial_cartographic_with_optional_parameters_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg,) }
    def set_partial_cartographic_with_optional_parameters(self, central_body:str, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", indices:list) -> None:
        """For convenience. Updates a subset of positions, text, and/or per-string parameters in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_cartographic_with_optional_parameters_metadata, central_body, positions, text, optional_parameters, indices)

    _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_metadata = { "offset" : _set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY), agcom.PVOID, POINTER(agcom.LPSAFEARRAY), agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg, agmarshall.InterfaceInArg("TextBatchPrimitiveOptionalParameters"), agmarshall.LPSafearrayArg, agmarshall.EnumArg(PrimitiveIndicesOrderHint), agmarshall.EnumArg(RenderPassHint),) }
    def set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass(self, central_body:str, positions:list, text:list, optional_parameters:"TextBatchPrimitiveOptionalParameters", indices:list, indices_order_hint:"PrimitiveIndicesOrderHint", render_pass_hint:"RenderPassHint") -> None:
        """For convenience. Updates a subset of positions, text, and/or per-string parameters in a text batch using cartographic positions. This is equivalent to converting each position in positions to cartesian and calling SetPartial."""
        return self._intf.invoke(TextBatchPrimitive._metadata, TextBatchPrimitive._set_partial_cartographic_with_optional_parameters_indices_order_and_render_pass_metadata, central_body, positions, text, optional_parameters, indices, indices_order_hint, render_pass_hint)

    _get_render_in_screen_space_metadata = { "offset" : _get_render_in_screen_space_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def render_in_screen_space(self) -> bool:
        """Get or set whether the primitive is positioned and rendered in screen space coordinates."""
        return self._intf.get_property(TextBatchPrimitive._metadata, TextBatchPrimitive._get_render_in_screen_space_metadata)

    _set_render_in_screen_space_metadata = { "offset" : _set_render_in_screen_space_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @render_in_screen_space.setter
    def render_in_screen_space(self, render_in_screen_space:bool) -> None:
        return self._intf.set_property(TextBatchPrimitive._metadata, TextBatchPrimitive._set_render_in_screen_space_metadata, render_in_screen_space)

    _property_names[set_hint] = "set_hint"
    _property_names[bounding_sphere_scale] = "bounding_sphere_scale"
    _property_names[font] = "font"
    _property_names[outline_color] = "outline_color"
    _property_names[outline_translucency] = "outline_translucency"
    _property_names[align_to_pixel] = "align_to_pixel"
    _property_names[distance_display_condition_per_string] = "distance_display_condition_per_string"
    _property_names[per_item_picking_enabled] = "per_item_picking_enabled"
    _property_names[texture_filter] = "texture_filter"
    _property_names[render_in_screen_space] = "render_in_screen_space"

    def __init__(self, source_object=None):
        """Construct an object of type TextBatchPrimitive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TextBatchPrimitive)
        IPrimitive.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPrimitive._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TextBatchPrimitive, [TextBatchPrimitive, IPrimitive])

agcls.AgClassCatalog.add_catalog_entry((5192917499857114645, 15090987484832364472), TextBatchPrimitive)
agcls.AgTypeNameMap["TextBatchPrimitive"] = TextBatchPrimitive

class TextBatchPrimitiveOptionalParameters(SupportsDeleteCallback):
    """Optional per-string and per-batch parameters for text batch primitive..."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_origin_method_offset = 1
    _set_origin_method_offset = 2
    _get_eye_offset_method_offset = 3
    _set_eye_offset_method_offset = 4
    _get_pixel_offset_method_offset = 5
    _set_pixel_offset_method_offset = 6
    _get_maximum_string_length_method_offset = 7
    _set_maximum_string_length_method_offset = 8
    _set_origins_method_offset = 9
    _set_eye_offsets_method_offset = 10
    _set_pixel_offsets_method_offset = 11
    _set_colors_method_offset = 12
    _get_screen_space_rendering_method_offset = 13
    _set_screen_space_rendering_method_offset = 14
    _metadata = {
        "iid_data" : (5706768872985264431, 9209360119003680904),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextBatchPrimitiveOptionalParameters)

    _get_origin_metadata = { "offset" : _get_origin_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Origin),) }
    @property
    def origin(self) -> "Origin":
        """Get or set the per-batch origin, which is applied to each string in the text batch."""
        return self._intf.get_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._get_origin_metadata)

    _set_origin_metadata = { "offset" : _set_origin_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(Origin),) }
    @origin.setter
    def origin(self, origin:"Origin") -> None:
        return self._intf.set_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_origin_metadata, origin)

    _get_eye_offset_metadata = { "offset" : _get_eye_offset_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def eye_offset(self) -> list:
        """Get or set the per-batch eye offset, which is applied to each string in the text batch. The array contains the components of the offset arranged in the order x, y, z."""
        return self._intf.get_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._get_eye_offset_metadata)

    _set_eye_offset_metadata = { "offset" : _set_eye_offset_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @eye_offset.setter
    def eye_offset(self, eye_offset:list) -> None:
        return self._intf.set_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_eye_offset_metadata, eye_offset)

    _get_pixel_offset_metadata = { "offset" : _get_pixel_offset_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def pixel_offset(self) -> list:
        """Get or set the per-batch pixel offset, which is applied to each string in the text batch. The array contains one x pixel offset followed by one y pixel offset."""
        return self._intf.get_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._get_pixel_offset_metadata)

    _set_pixel_offset_metadata = { "offset" : _set_pixel_offset_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @pixel_offset.setter
    def pixel_offset(self, pixel_offset:list) -> None:
        return self._intf.set_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_pixel_offset_metadata, pixel_offset)

    _get_maximum_string_length_metadata = { "offset" : _get_maximum_string_length_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_string_length(self) -> int:
        """Get or set the maximum length of each string in the text batch."""
        return self._intf.get_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._get_maximum_string_length_metadata)

    _set_maximum_string_length_metadata = { "offset" : _set_maximum_string_length_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_string_length.setter
    def maximum_string_length(self, maximum_string_length:int) -> None:
        return self._intf.set_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_maximum_string_length_metadata, maximum_string_length)

    _set_origins_metadata = { "offset" : _set_origins_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_origins(self, origins:list) -> None:
        """Define a collection of origins, one for each string in the batch."""
        return self._intf.invoke(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_origins_metadata, origins)

    _set_eye_offsets_metadata = { "offset" : _set_eye_offsets_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_eye_offsets(self, eye_offsets:list) -> None:
        """Define a collection of eye offsets, one for each string in the batch."""
        return self._intf.invoke(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_eye_offsets_metadata, eye_offsets)

    _set_pixel_offsets_metadata = { "offset" : _set_pixel_offsets_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_pixel_offsets(self, pixel_offsets:list) -> None:
        """Define a collection of pixel offsets, one for each string in the batch."""
        return self._intf.invoke(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_pixel_offsets_metadata, pixel_offsets)

    _set_colors_metadata = { "offset" : _set_colors_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_colors(self, colors:list) -> None:
        """Define a collection of colors, one for each string in the batch."""
        return self._intf.invoke(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_colors_metadata, colors)

    _get_screen_space_rendering_metadata = { "offset" : _get_screen_space_rendering_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def screen_space_rendering(self) -> bool:
        """Get or set a flag that informs the CTextBatchPrimitive to use rendering optimized for screen space text."""
        return self._intf.get_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._get_screen_space_rendering_metadata)

    _set_screen_space_rendering_metadata = { "offset" : _set_screen_space_rendering_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @screen_space_rendering.setter
    def screen_space_rendering(self, screen_space_rendering:bool) -> None:
        return self._intf.set_property(TextBatchPrimitiveOptionalParameters._metadata, TextBatchPrimitiveOptionalParameters._set_screen_space_rendering_metadata, screen_space_rendering)

    _property_names[origin] = "origin"
    _property_names[eye_offset] = "eye_offset"
    _property_names[pixel_offset] = "pixel_offset"
    _property_names[maximum_string_length] = "maximum_string_length"
    _property_names[screen_space_rendering] = "screen_space_rendering"

    def __init__(self, source_object=None):
        """Construct an object of type TextBatchPrimitiveOptionalParameters."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TextBatchPrimitiveOptionalParameters)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TextBatchPrimitiveOptionalParameters, [TextBatchPrimitiveOptionalParameters, ])

agcls.AgClassCatalog.add_catalog_entry((5647850369699816160, 11320403267934911642), TextBatchPrimitiveOptionalParameters)
agcls.AgTypeNameMap["TextBatchPrimitiveOptionalParameters"] = TextBatchPrimitiveOptionalParameters

class TextOverlay(IScreenOverlay, IOverlay, IScreenOverlayContainer, SupportsDeleteCallback):
    """A rectangular overlay that contains text."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_text_method_offset = 1
    _set_text_method_offset = 2
    _get_outline_color_method_offset = 3
    _set_outline_color_method_offset = 4
    _get_font_method_offset = 5
    _set_font_method_offset = 6
    _metadata = {
        "iid_data" : (5705180996127980072, 10503608201518791570),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextOverlay)

    _get_text_metadata = { "offset" : _get_text_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def text(self) -> str:
        r"""Set the Text to be rendered to an overlay. Newline characters ('\n') will mark the start of the next line in the text."""
        return self._intf.get_property(TextOverlay._metadata, TextOverlay._get_text_metadata)

    _set_text_metadata = { "offset" : _set_text_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @text.setter
    def text(self, text:str) -> None:
        return self._intf.set_property(TextOverlay._metadata, TextOverlay._set_text_metadata, text)

    _get_outline_color_metadata = { "offset" : _get_outline_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def outline_color(self) -> agcolor.Color:
        """Get or set the text's outline color."""
        return self._intf.get_property(TextOverlay._metadata, TextOverlay._get_outline_color_metadata)

    _set_outline_color_metadata = { "offset" : _set_outline_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @outline_color.setter
    def outline_color(self, outline_color:agcolor.Color) -> None:
        return self._intf.set_property(TextOverlay._metadata, TextOverlay._set_outline_color_metadata, outline_color)

    _get_font_metadata = { "offset" : _get_font_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def font(self) -> "GraphicsFont":
        """Get the graphics font used to style the text."""
        return self._intf.get_property(TextOverlay._metadata, TextOverlay._get_font_metadata)

    _set_font_metadata = { "offset" : _set_font_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("GraphicsFont"),) }
    @font.setter
    def font(self, font:"GraphicsFont") -> None:
        return self._intf.set_property(TextOverlay._metadata, TextOverlay._set_font_metadata, font)

    _property_names[text] = "text"
    _property_names[outline_color] = "outline_color"
    _property_names[font] = "font"

    def __init__(self, source_object=None):
        """Construct an object of type TextOverlay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TextOverlay)
        IScreenOverlay.__init__(self, source_object)
        IOverlay.__init__(self, source_object)
        IScreenOverlayContainer.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IScreenOverlay._private_init(self, intf)
        IOverlay._private_init(self, intf)
        IScreenOverlayContainer._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TextOverlay, [TextOverlay, IScreenOverlay, IOverlay, IScreenOverlayContainer])

agcls.AgClassCatalog.add_catalog_entry((5427148005289374195, 14840843638839670698), TextOverlay)
agcls.AgTypeNameMap["TextOverlay"] = TextOverlay

class TextureMatrix(SupportsDeleteCallback):
    """A 4 by 4 matrix applied to a texture coordinate."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_m11_method_offset = 1
    _get_m12_method_offset = 2
    _get_m13_method_offset = 3
    _get_m14_method_offset = 4
    _get_m21_method_offset = 5
    _get_m22_method_offset = 6
    _get_m23_method_offset = 7
    _get_m24_method_offset = 8
    _get_m31_method_offset = 9
    _get_m32_method_offset = 10
    _get_m33_method_offset = 11
    _get_m34_method_offset = 12
    _get_m41_method_offset = 13
    _get_m42_method_offset = 14
    _get_m43_method_offset = 15
    _get_m44_method_offset = 16
    _metadata = {
        "iid_data" : (5075276615207636111, 4269654310333162905),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TextureMatrix)

    _get_m11_metadata = { "offset" : _get_m11_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m11(self) -> float:
        """Get the element in the first row, first column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m11_metadata)

    _get_m12_metadata = { "offset" : _get_m12_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m12(self) -> float:
        """Get the element in the first row, second column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m12_metadata)

    _get_m13_metadata = { "offset" : _get_m13_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m13(self) -> float:
        """Get the element in the first row, third column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m13_metadata)

    _get_m14_metadata = { "offset" : _get_m14_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m14(self) -> float:
        """Get the element in the first row, fourth column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m14_metadata)

    _get_m21_metadata = { "offset" : _get_m21_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m21(self) -> float:
        """Get the element in the second row, first column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m21_metadata)

    _get_m22_metadata = { "offset" : _get_m22_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m22(self) -> float:
        """Get the element in the second row, second column."""
        return self._intf.get_property(TextureMatrix._metadata, TextureMatrix._get_m22_metadata)

    _get_m23_metadata = { "offset" : _get_m23_method_offset,
            "arg_types" : (POINTER(agcom.FLOAT),),
            "marshallers" : (agmarshall.FloatArg,) }
    @property
    def m23(self)