# Copyright (C) 2022 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""Object Model components specifically designed to support STK Aviator."""

__all__ = ["ADDSForecastType", "ADDSMessage", "ADDSMessageCollection", "ADDSMessageExtrapolationType",
"ADDSMessageInterpolationType", "ADDSMissingMessageType", "AFPROPFuelType", "AGLMSL", "ARINC424Airport",
"ARINC424Helipad", "ARINC424Navaid", "ARINC424Runway", "ARINC424Source", "ARINC424Waypoint",
"AccelerationAdvancedAccelerationMode", "AccelerationManeuverAerodynamicPropulsionMode", "AccelerationManeuverMode",
"AccelerationMode", "AccelerationPerformanceModelOverride", "AdvancedFixedWingAerodynamicStrategy",
"AdvancedFixedWingElectricPowerplant", "AdvancedFixedWingEmpiricalJetEngine", "AdvancedFixedWingExternalAerodynamic",
"AdvancedFixedWingExternalPropulsion", "AdvancedFixedWingFourPointAerodynamic", "AdvancedFixedWingGeometry",
"AdvancedFixedWingGeometryBasic", "AdvancedFixedWingGeometryVariable", "AdvancedFixedWingPistonPowerplant",
"AdvancedFixedWingPowerplantStrategy", "AdvancedFixedWingRamjetBasic", "AdvancedFixedWingScramjetBasic",
"AdvancedFixedWingSubSuperHypersonicAerodynamic", "AdvancedFixedWingSubSuperHypersonicPropulsion",
"AdvancedFixedWingSubsonicAerodynamic", "AdvancedFixedWingSupersonicAerodynamic", "AdvancedFixedWingTool",
"AdvancedFixedWingTurbofanBasicABPowerplant", "AdvancedFixedWingTurbofanBasicABPropulsion",
"AdvancedFixedWingTurbojetBasicABPowerplant", "AdvancedFixedWingTurbojetBasicABPropulsion",
"AdvancedFixedWingTurbopropPowerplant", "AerodynamicPropulsionFlightMode", "AerodynamicPropulsionManeuverModeHelper",
"AerodynamicPropulsionSimpleMode", "AileronRollFlightPath", "AileronRollMode", "AircraftAcceleration",
"AircraftAccelerationMode", "AircraftAdvancedAccelerationModel", "AircraftAdvancedClimbModel",
"AircraftAdvancedCruiseModel", "AircraftAdvancedDescentModel", "AircraftAdvancedLandingModel",
"AircraftAdvancedTakeoffModel", "AircraftAerodynamic", "AircraftAerodynamicStrategy", "AircraftBasicAccelerationModel",
"AircraftBasicClimbModel", "AircraftBasicCruiseModel", "AircraftBasicDescentModel", "AircraftBasicFixedWingAerodynamic",
"AircraftBasicFixedWingPropulsion", "AircraftBasicLandingModel", "AircraftBasicTakeoffModel", "AircraftCategory",
"AircraftClimb", "AircraftCruise", "AircraftDescent", "AircraftExternalAerodynamic", "AircraftExternalPropulsion",
"AircraftLanding", "AircraftModel", "AircraftModels", "AircraftPropulsion", "AircraftPropulsionStrategy",
"AircraftSimpleAerodynamic", "AircraftSimplePropulsion", "AircraftTakeoff", "AircraftTerrainFollow",
"AircraftTerrainFollowModel", "AircraftVTOL", "AircraftVTOLModel", "AirportCategory", "AirspeedType",
"AltitudeConstraintManeuverMode", "AltitudeMSLAndLevelOffOptions", "AltitudeMSLOptions", "AltitudeOptions",
"AltitudeReference", "AndOr", "AngleMode", "ApproachMode", "ArcAltitudeAndDelayOptions", "ArcAltitudeOptions",
"ArcOptions", "ArcVerticalPlaneOptions", "AtmosphereModel", "AtmosphereModelBasic", "AtmosphereModelType",
"AttitudeTransitions", "AutopilotAltitudeControlMode", "AutopilotAltitudeMode", "AutopilotHorizontalPlaneMode",
"AviatorNumericalIntegrator", "AviatorPropagator", "Ballistic3DControlMode", "BasicFixedWingForwardFlightLiftHelper",
"BasicFixedWingPropulsionMode", "BasicManeuverAirspeedMode", "BasicManeuverAirspeedOptions",
"BasicManeuverAltitudeLimit", "BasicManeuverFuelFlowType", "BasicManeuverGlideSpeedControlMode",
"BasicManeuverReferenceFrame", "BasicManeuverStrategy", "BasicManeuverStrategyAileronRoll",
"BasicManeuverStrategyAirspeedPerformanceLimits", "BasicManeuverStrategyAutopilotNavigation",
"BasicManeuverStrategyAutopilotProf", "BasicManeuverStrategyBallistic3D", "BasicManeuverStrategyBarrelRoll",
"BasicManeuverStrategyBezier", "BasicManeuverStrategyCruiseProfile", "BasicManeuverStrategyFlyAOA",
"BasicManeuverStrategyGlideProfile", "BasicManeuverStrategyIntercept", "BasicManeuverStrategyLTAHover",
"BasicManeuverStrategyLoop", "BasicManeuverStrategyNavigationControlLimit", "BasicManeuverStrategyPitch3D",
"BasicManeuverStrategyPoweredCruiseMode", "BasicManeuverStrategyPull", "BasicManeuverStrategyPushPull",
"BasicManeuverStrategyRelativeBearing", "BasicManeuverStrategyRelativeCourse",
"BasicManeuverStrategyRelativeFlightPathAngle", "BasicManeuverStrategyRelativeSpeedAltitude",
"BasicManeuverStrategyRendezvous", "BasicManeuverStrategyRollingPull", "BasicManeuverStrategySimpleTurn",
"BasicManeuverStrategySmoothAcceleration", "BasicManeuverStrategySmoothTurn", "BasicManeuverStrategyStationkeeping",
"BasicManeuverStrategyStraightAhead", "BasicManeuverStrategyWeave", "BasicManeuverTargetPositionVelocity",
"BasicManeuverTargetPositionVelocityNoisyBearingRange", "BasicManeuverTargetPositionVelocityNoisySurfTarget",
"CEAFuelType", "CalculationOptions", "Catalog", "ClimbAndDescentTransitions", "ClimbSpeedType", "ClosureMode",
"ClosureValue", "Configuration", "CruiseAirspeedAndProfileOptions", "CruiseAirspeedOptions", "CruiseAirspeedProfile",
"CruiseMaxPerformanceSpeedType", "CruiseSpeed", "DAFIFHelipad", "DAFIFRunway", "DAFIFSource", "DAFIFWaypoint",
"DelayAltitudeMode", "DelayTurnDirection", "DepartureSpeedMode", "DescentSpeedType", "EnrouteAndDelayOptions",
"EnrouteOptions", "EnrouteTurnDirectionOptions", "EphemShiftRotateAltitudeMode", "EphemShiftRotateCourseMode",
"ErrorCodes", "ExtEphemFlightMode", "FlightLineProcedureType", "FlyAOALeftRight", "FlyToFlightPathAngleMode",
"FormationFlyerStopCondition", "FourPointAerodynamic", "FuelFlowType", "FuelModelKeroseneAFPROP",
"FuelModelKeroseneCEA", "FuelTankExternal", "FuelTankInternal", "HoldRefuelDumpMode", "HoldingDirection",
"HoldingEntryManeuver", "HoldingProfileMode", "HoverAltitudeMode", "HoverAltitudeOptions", "HoverHeadingMode",
"HoverMode", "IARINC424Airport", "IARINC424Item", "IAdvancedFixedWingGeometry", "IAdvancedFixedWingPowerplant",
"IAutomationStrategyFactory", "IAviatorVehicle", "IBasicFixedWingLiftHelper", "IBasicManeuverStrategy",
"ICatalogAirport", "ICatalogItem", "ICatalogNavaid", "ICatalogRunway", "ICatalogSource", "ICatalogVTOLPoint",
"ICatalogWaypoint", "IConnect", "ICruiseAirspeedAndProfileOptions", "IDAFIFItem", "IEnrouteAndDelayOptions",
"IPerformanceModel", "IProcedure", "ISite", "ISiteUnknown", "IStation", "IVerticalPlaneOptions", "InterceptMode",
"JetEngineExhaustNozzleType", "JetEngineIntakeType", "JetEngineTechnologyLevel", "JetEngineTurbineType", "JetFuelType",
"JoinExitArcMethod", "LandingApproachFixRangeMode", "LandingCruiseAirspeedAndProfileOptions", "LandingEnrouteOptions",
"LandingEnterDownwindPattern", "LandingInterceptGlideslope", "LandingStandardInstrumentApproach",
"LandingVerticalPlaneOptions", "LaunchAttitudeMode", "LaunchDynamicStateBearingReference",
"LaunchDynamicStateCoordFrame", "LevelTurns", "LineOrientation", "MinimizeSiteProcedureTimeDiff",
"MissileAdvancedAerodynamic", "MissileAerodynamic", "MissileAerodynamicStrategy", "MissileExternalAerodynamic",
"MissileExternalPropulsion", "MissileFourPointAerodynamic", "MissileModel", "MissileModels", "MissilePropulsion",
"MissilePropulsionStrategy", "MissileRamjetPropulsion", "MissileRocketPropulsion", "MissileSimpleAerodynamic",
"MissileSimplePropulsion", "MissileTurbojetPropulsion", "Mission", "NavaidCategory", "NavigationOptions",
"NavigatorTurnDirection", "PayloadStation", "PerformanceModel", "PerformanceModelOptions", "PerformanceModelOverride",
"Phase", "PhaseCollection", "PhaseOfFlight", "Pitch3DControlMode", "PointToPointMode", "Procedure", "ProcedureAirway",
"ProcedureAirwayRouter", "ProcedureArcEnroute", "ProcedureArcPointToPoint", "ProcedureAreaTargetSearch",
"ProcedureBasicManeuver", "ProcedureBasicPointToPoint", "ProcedureCollection", "ProcedureDelay", "ProcedureEnroute",
"ProcedureExtEphem", "ProcedureFastTimeOptions", "ProcedureFlightLine", "ProcedureFormationFlyer",
"ProcedureFormationRecover", "ProcedureHoldingCircular", "ProcedureHoldingFigure8", "ProcedureHoldingRacetrack",
"ProcedureHover", "ProcedureHoverTranslate", "ProcedureInFormation", "ProcedureLanding", "ProcedureLaunch",
"ProcedureLaunchDynamicState", "ProcedureLaunchWaypoint", "ProcedureParallelFlightLine", "ProcedureReferenceState",
"ProcedureSuperProcedure", "ProcedureTakeoff", "ProcedureTerrainFollow", "ProcedureTimeOptions",
"ProcedureTransitionToForwardFlight", "ProcedureTransitionToHover", "ProcedureType", "ProcedureVGTPoint",
"ProcedureVerticalLanding", "ProcedureVerticalTakeoff", "ProfileControlLimit", "PropulsionEfficiencies",
"PropulsionThrust", "PullMode", "PushPull", "RamjetMode", "ReferenceStateAttitudeMode",
"ReferenceStateForwardFlightOptions", "ReferenceStateHoverOptions", "ReferenceStateLateralAccelerationMode",
"ReferenceStateLongitudinalAccelerationMode", "ReferenceStatePerformanceMode", "ReferenceStateTakeoffLandingOptions",
"ReferenceStateWeightOnWheelsOptions", "RefuelDumpMode", "RefuelDumpProperties", "RelativeAbsoluteBearing",
"RelativeAltitudeMode", "RelativeSpeedAltitudeStopCondition", "RendezvousStopCondition", "RollLeftRight",
"RollUprightInverted", "RollingPullMode", "RotorcraftAerodynamic", "RotorcraftModel", "RotorcraftModels",
"RotorcraftPowerplantType", "RotorcraftPropulsion", "RunwayCategory", "RunwayHeadingOptions", "RunwayHighLowEnd",
"STKObjectWaypointOffsetMode", "ScramjetMode", "SearchPatternCourseMode", "Site", "SiteAirportFromCatalog",
"SiteDynamicState", "SiteEndOfPrevProcedure", "SiteNavaidFromCatalog", "SiteReferenceState",
"SiteRelativeToPrevProcedure", "SiteRelativeToSTKObject", "SiteRunway", "SiteRunwayFromCatalog", "SiteSTKAreaTarget",
"SiteSTKObjectWaypoint", "SiteSTKStaticObject", "SiteSTKVehicle", "SiteSuperProcedure", "SiteType", "SiteVTOLPoint",
"SiteVTOLPointFromCatalog", "SiteWaypoint", "SiteWaypointFromCatalog", "SmoothAccelerationLeftRight",
"SmoothAccelerationStopConditions", "SmoothTurnFlightPathAngleMode", "SmoothTurnMode", "StationCollection",
"StationkeepingStopCondition", "StraightAheadReferenceFrame", "TakeoffDeparturePoint", "TakeoffLandingSpeedMode",
"TakeoffLowTransition", "TakeoffMode", "TakeoffNormal", "TargetPositionVelocityType", "TrajectoryBlendMode",
"TransitionToHoverMode", "TurbineMode", "TurnDirection", "TurnMode", "UserRunway", "UserRunwaySource", "UserVTOLPoint",
"UserVTOLPointSource", "UserWaypoint", "UserWaypointSource", "VTOLFinalHeadingMode", "VTOLHeadingMode",
"VTOLPointCategory", "VTOLRateMode", "VTOLTransitionMode", "VTOLTranslationFinalCourseMode", "VTOLTranslationMode",
"VertLandingMode", "VerticalPlaneAndFlightPathOptions", "VerticalPlaneOptions", "WaypointCategory",
"WindAtmosphereModelSource", "WindModel", "WindModelADDS", "WindModelConstant", "WindModelType"]

from ctypes import POINTER
from enum import IntEnum
import typing

from ...internal import coclassutil as agcls, comutil as agcom, marshall as agmarshall
from ...internal.apiutil import (
    EnumeratorProxy,
    InterfaceProxy,
    OutArg,
    SupportsDeleteCallback,
    get_interface_property,
    initialize_from_source_object,
    set_class_attribute,
    set_interface_attribute,
)
from ...internal.comutil import IDispatch, IUnknown
from ...utilities.exceptions import STKRuntimeError


def _raise_uninitialized_error(*args):
    raise STKRuntimeError("Valid STK object model classes are returned from STK methods and should not be created independently.")

class ErrorCodes(IntEnum):
    """Error Codes."""

    OBJECT_NOT_FOUND = (((1 << 31) | (4 << 16)) | (0x200 + 1))
    """Object not found."""
    INDEX_OUT_OF_RANGE = (((1 << 31) | (4 << 16)) | (0x200 + 2))
    """Index out of range."""
    INVALID_ATTRIBUTE = (((1 << 31) | (4 << 16)) | (0x200 + 3))
    """The attribute is invalid."""
    COMMAND_FAILED = (((1 << 31) | (4 << 16)) | (0x200 + 4))
    """The command failed."""
    ERROR_INVALID_ARG = (((1 << 31) | (4 << 16)) | (0x200 + 5))
    """Invalid argument."""
    EMPTY_ARG = (((1 << 31) | (4 << 16)) | (0x200 + 6))
    """Empty argument."""
    OBJECT_NOT_REMOVED = (((1 << 31) | (4 << 16)) | (0x200 + 7))
    """Object was not removed."""
    FAILED_TO_RENAME_OBJECT = (((1 << 31) | (4 << 16)) | (0x200 + 8))
    """Error renaming object."""
    UNKNOWN_CLASS_TYPE = (((1 << 31) | (4 << 16)) | (0x200 + 9))
    """Unknown class type."""
    FAILED_TO_CREATE_OBJECT = (((1 << 31) | (4 << 16)) | (0x200 + 10))
    """Failed to create the object."""
    OBJECT_LINK_INVALID_CHOICE = (((1 << 31) | (4 << 16)) | (0x200 + 11))
    """Invalid choice for object link."""
    OBJECT_LINK_NO_CHOICES = (((1 << 31) | (4 << 16)) | (0x200 + 12))
    """No choices available."""
    READ_ONLY_ATTRIBUTE = (((1 << 31) | (4 << 16)) | (0x200 + 13))
    """Read only attribute."""
    INVALID_CSTR_LIST = (((1 << 31) | (4 << 16)) | (0x200 + 14))
    """Invalid constraint list."""
    INVALID_CONSTRAINT = (((1 << 31) | (4 << 16)) | (0x200 + 15))
    """InvalidConstraint."""
    LIST_READ_ONLY = (((1 << 31) | (4 << 16)) | (0x200 + 16))
    """List if read only."""
    LIST_INSERT_FAILED = (((1 << 31) | (4 << 16)) | (0x200 + 17))
    """Failed to insert the item into the list."""
    INVALID_LENGTH = (((1 << 31) | (4 << 16)) | (0x200 + 18))
    """Length is invalid."""
    FAILED_TO_LOAD_FILE = (((1 << 31) | (4 << 16)) | (0x200 + 19))
    """Error loading a file."""
    INVALID_OPERATION = (((1 << 31) | (4 << 16)) | (0x200 + 20))
    """Invalid operation."""
    METHOD_INVOKE_FAILED = (((1 << 31) | (4 << 16)) | (0x200 + 21))
    """Method failed."""
    DEPRECATED = (((1 << 31) | (4 << 16)) | (0x200 + 22))
    """Deprecated property or method."""

ErrorCodes.OBJECT_NOT_FOUND.__doc__ = "Object not found."
ErrorCodes.INDEX_OUT_OF_RANGE.__doc__ = "Index out of range."
ErrorCodes.INVALID_ATTRIBUTE.__doc__ = "The attribute is invalid."
ErrorCodes.COMMAND_FAILED.__doc__ = "The command failed."
ErrorCodes.ERROR_INVALID_ARG.__doc__ = "Invalid argument."
ErrorCodes.EMPTY_ARG.__doc__ = "Empty argument."
ErrorCodes.OBJECT_NOT_REMOVED.__doc__ = "Object was not removed."
ErrorCodes.FAILED_TO_RENAME_OBJECT.__doc__ = "Error renaming object."
ErrorCodes.UNKNOWN_CLASS_TYPE.__doc__ = "Unknown class type."
ErrorCodes.FAILED_TO_CREATE_OBJECT.__doc__ = "Failed to create the object."
ErrorCodes.OBJECT_LINK_INVALID_CHOICE.__doc__ = "Invalid choice for object link."
ErrorCodes.OBJECT_LINK_NO_CHOICES.__doc__ = "No choices available."
ErrorCodes.READ_ONLY_ATTRIBUTE.__doc__ = "Read only attribute."
ErrorCodes.INVALID_CSTR_LIST.__doc__ = "Invalid constraint list."
ErrorCodes.INVALID_CONSTRAINT.__doc__ = "InvalidConstraint."
ErrorCodes.LIST_READ_ONLY.__doc__ = "List if read only."
ErrorCodes.LIST_INSERT_FAILED.__doc__ = "Failed to insert the item into the list."
ErrorCodes.INVALID_LENGTH.__doc__ = "Length is invalid."
ErrorCodes.FAILED_TO_LOAD_FILE.__doc__ = "Error loading a file."
ErrorCodes.INVALID_OPERATION.__doc__ = "Invalid operation."
ErrorCodes.METHOD_INVOKE_FAILED.__doc__ = "Method failed."
ErrorCodes.DEPRECATED.__doc__ = "Deprecated property or method."

agcls.AgTypeNameMap["ErrorCodes"] = ErrorCodes

class ClosureValue(IntEnum):
    """The closure value."""

    CLOSURE_MODE = 0
    """The closure mode."""
    MAX_ANGLE = 1
    """The HOBS max angle offset."""
    ANGLE_TOL = 2
    """The HOBS angle tolerance."""

ClosureValue.CLOSURE_MODE.__doc__ = "The closure mode."
ClosureValue.MAX_ANGLE.__doc__ = "The HOBS max angle offset."
ClosureValue.ANGLE_TOL.__doc__ = "The HOBS angle tolerance."

agcls.AgTypeNameMap["ClosureValue"] = ClosureValue

class ProcedureType(IntEnum):
    """Aviator procedure types."""

    PROCEDURE_AIRWAY = 0
    """Airway procedure."""
    PROCEDURE_AIRWAY_ROUTER = 1
    """Airway Router procedure."""
    PROCEDURE_ARC_ENROUTE = 2
    """ArcEnroute procedure."""
    PROCEDURE_ARC_POINT_TO_POINT = 3
    """ArcPointToPoint procedure."""
    PROCEDURE_AREA_TARGET_SEARCH = 4
    """Area Target Search procedure."""
    PROCEDURE_BASIC_MANEUVER = 5
    """Basic Maneuver procedure."""
    PROCEDURE_BASIC_POINT_TO_POINT = 6
    """Basic Point to Point procedure."""
    PROCEDURE_DELAY = 7
    """Delay procedure."""
    PROCEDURE_ENROUTE = 8
    """Enroute procedure."""
    PROCEDURE_FLIGHT_LINE = 9
    """Flight Line procedure."""
    PROCEDURE_FORMATION_RECOVER = 10
    """Formation Recover procedure."""
    PROCEDURE_HOLDING_CIRCULAR = 11
    """Holding Circular procedure."""
    PROCEDURE_HOLDING_FIGURE8 = 12
    """Holding Figure 8 procedure."""
    PROCEDURE_HOLDING_RACETRACK = 13
    """Holding Racetrack procedure."""
    PROCEDURE_HOVER = 14
    """Hover procedure."""
    PROCEDURE_HOVER_TRANSLATE = 15
    """Hover Translate procedure."""
    PROCEDURE_IN_FORMATION = 16
    """In Formation procedure."""
    PROCEDURE_LANDING = 17
    """Landing procedure."""
    PROCEDURE_LAUNCH = 18
    """Launch procedure."""
    PROCEDURE_PARALLEL_FLIGHT_LINE = 19
    """Parallel Flight Line procedure."""
    PROCEDURE_REFERENCE_STATE = 20
    """Reference State procedure."""
    PROCEDURE_SUPER_PROCEDURE = 21
    """Super Procedure procedure."""
    PROCEDURE_TAKEOFF = 22
    """Takeoff procedure."""
    PROCEDURE_TERRAIN_FOLLOWING = 23
    """Terrain Following procedure."""
    PROCEDURE_TRANSITION_TO_FORWARD_FLIGHT = 24
    """Transition to Forward Flight procedure."""
    PROCEDURE_TRANSITION_TO_HOVER = 25
    """Transition To Hover procedure."""
    PROCEDURE_VERTICAL_LANDING = 26
    """Vertical Landing procedure."""
    PROCEDURE_VERTICAL_TAKEOFF = 27
    """Vertical Takeoff procedure."""
    PROCEDURE_VGT_POINT = 28
    """VGT Point procedure."""
    PROCEDURE_LAUNCH_DYNAMIC_STATE = 29
    """LaunchDynState procedure."""
    PROCEDURE_LAUNCH_WAYPOINT = 30
    """LaunchWaypoint procedure."""
    PROCEDURE_FORMATION_FLYER = 31
    """FormationFlyer procedure."""
    PROCEDURE_EXT_EPHEM = 32
    """ExtEphem procedure."""

ProcedureType.PROCEDURE_AIRWAY.__doc__ = "Airway procedure."
ProcedureType.PROCEDURE_AIRWAY_ROUTER.__doc__ = "Airway Router procedure."
ProcedureType.PROCEDURE_ARC_ENROUTE.__doc__ = "ArcEnroute procedure."
ProcedureType.PROCEDURE_ARC_POINT_TO_POINT.__doc__ = "ArcPointToPoint procedure."
ProcedureType.PROCEDURE_AREA_TARGET_SEARCH.__doc__ = "Area Target Search procedure."
ProcedureType.PROCEDURE_BASIC_MANEUVER.__doc__ = "Basic Maneuver procedure."
ProcedureType.PROCEDURE_BASIC_POINT_TO_POINT.__doc__ = "Basic Point to Point procedure."
ProcedureType.PROCEDURE_DELAY.__doc__ = "Delay procedure."
ProcedureType.PROCEDURE_ENROUTE.__doc__ = "Enroute procedure."
ProcedureType.PROCEDURE_FLIGHT_LINE.__doc__ = "Flight Line procedure."
ProcedureType.PROCEDURE_FORMATION_RECOVER.__doc__ = "Formation Recover procedure."
ProcedureType.PROCEDURE_HOLDING_CIRCULAR.__doc__ = "Holding Circular procedure."
ProcedureType.PROCEDURE_HOLDING_FIGURE8.__doc__ = "Holding Figure 8 procedure."
ProcedureType.PROCEDURE_HOLDING_RACETRACK.__doc__ = "Holding Racetrack procedure."
ProcedureType.PROCEDURE_HOVER.__doc__ = "Hover procedure."
ProcedureType.PROCEDURE_HOVER_TRANSLATE.__doc__ = "Hover Translate procedure."
ProcedureType.PROCEDURE_IN_FORMATION.__doc__ = "In Formation procedure."
ProcedureType.PROCEDURE_LANDING.__doc__ = "Landing procedure."
ProcedureType.PROCEDURE_LAUNCH.__doc__ = "Launch procedure."
ProcedureType.PROCEDURE_PARALLEL_FLIGHT_LINE.__doc__ = "Parallel Flight Line procedure."
ProcedureType.PROCEDURE_REFERENCE_STATE.__doc__ = "Reference State procedure."
ProcedureType.PROCEDURE_SUPER_PROCEDURE.__doc__ = "Super Procedure procedure."
ProcedureType.PROCEDURE_TAKEOFF.__doc__ = "Takeoff procedure."
ProcedureType.PROCEDURE_TERRAIN_FOLLOWING.__doc__ = "Terrain Following procedure."
ProcedureType.PROCEDURE_TRANSITION_TO_FORWARD_FLIGHT.__doc__ = "Transition to Forward Flight procedure."
ProcedureType.PROCEDURE_TRANSITION_TO_HOVER.__doc__ = "Transition To Hover procedure."
ProcedureType.PROCEDURE_VERTICAL_LANDING.__doc__ = "Vertical Landing procedure."
ProcedureType.PROCEDURE_VERTICAL_TAKEOFF.__doc__ = "Vertical Takeoff procedure."
ProcedureType.PROCEDURE_VGT_POINT.__doc__ = "VGT Point procedure."
ProcedureType.PROCEDURE_LAUNCH_DYNAMIC_STATE.__doc__ = "LaunchDynState procedure."
ProcedureType.PROCEDURE_LAUNCH_WAYPOINT.__doc__ = "LaunchWaypoint procedure."
ProcedureType.PROCEDURE_FORMATION_FLYER.__doc__ = "FormationFlyer procedure."
ProcedureType.PROCEDURE_EXT_EPHEM.__doc__ = "ExtEphem procedure."

agcls.AgTypeNameMap["ProcedureType"] = ProcedureType

class SiteType(IntEnum):
    """Aviator site types."""

    SITE_AIRPORT_FROM_CATALOG = 0
    """Airport from Catalog site."""
    SITE_END_OF_PREV_PROCEDURE = 1
    """End of Previous Procedure site."""
    SITE_NAVAID_FROM_CATALOG = 2
    """Navaid from Catalog site."""
    SITE_REFERENCE_STATE = 3
    """Reference State site."""
    SITE_RELATIVE_TO_PREV_PROCEDURE = 4
    """Relative to Previous Procedure site."""
    SITE_RELATIVE_TO_STATIONARY_STK_OBJECT = 5
    """Relative to Stationary STK Object site."""
    SITE_RUNWAY = 6
    """Runway site."""
    SITE_RUNWAY_FROM_CATALOG = 7
    """Runway from Catalog site."""
    SITE_STK_AREA_TARGET = 8
    """STK Area Target site."""
    SITE_STK_OBJECT_WAYPOINT = 9
    """STK Object Waypoint site."""
    SITE_STK_STATIC_OBJECT = 10
    """STK STatic Object site."""
    SITE_STK_VEHICLE = 11
    """STK Vehicle site."""
    SITE_SUPER_PROCEDURE = 12
    """Super Procedure site."""
    SITE_VTOL_POINT = 13
    """VTOL Point site."""
    SITE_VTOL_POINT_FROM_CATALOG = 14
    """VTOL Point from Catalog site."""
    SITE_WAYPOINT = 15
    """Waypoint site."""
    SITE_WAYPOINT_FROM_CATALOG = 16
    """Waypoint from Catalog site."""
    SITE_DYNAMIC_STATE = 17
    """DynState site."""

SiteType.SITE_AIRPORT_FROM_CATALOG.__doc__ = "Airport from Catalog site."
SiteType.SITE_END_OF_PREV_PROCEDURE.__doc__ = "End of Previous Procedure site."
SiteType.SITE_NAVAID_FROM_CATALOG.__doc__ = "Navaid from Catalog site."
SiteType.SITE_REFERENCE_STATE.__doc__ = "Reference State site."
SiteType.SITE_RELATIVE_TO_PREV_PROCEDURE.__doc__ = "Relative to Previous Procedure site."
SiteType.SITE_RELATIVE_TO_STATIONARY_STK_OBJECT.__doc__ = "Relative to Stationary STK Object site."
SiteType.SITE_RUNWAY.__doc__ = "Runway site."
SiteType.SITE_RUNWAY_FROM_CATALOG.__doc__ = "Runway from Catalog site."
SiteType.SITE_STK_AREA_TARGET.__doc__ = "STK Area Target site."
SiteType.SITE_STK_OBJECT_WAYPOINT.__doc__ = "STK Object Waypoint site."
SiteType.SITE_STK_STATIC_OBJECT.__doc__ = "STK STatic Object site."
SiteType.SITE_STK_VEHICLE.__doc__ = "STK Vehicle site."
SiteType.SITE_SUPER_PROCEDURE.__doc__ = "Super Procedure site."
SiteType.SITE_VTOL_POINT.__doc__ = "VTOL Point site."
SiteType.SITE_VTOL_POINT_FROM_CATALOG.__doc__ = "VTOL Point from Catalog site."
SiteType.SITE_WAYPOINT.__doc__ = "Waypoint site."
SiteType.SITE_WAYPOINT_FROM_CATALOG.__doc__ = "Waypoint from Catalog site."
SiteType.SITE_DYNAMIC_STATE.__doc__ = "DynState site."

agcls.AgTypeNameMap["SiteType"] = SiteType

class BasicManeuverStrategy(IntEnum):
    """Basic maneuver strategy types."""

    STRAIGHT_AHEAD = 0
    """Straight Ahead strategy."""
    WEAVE = 1
    """Weave strategy."""

BasicManeuverStrategy.STRAIGHT_AHEAD.__doc__ = "Straight Ahead strategy."
BasicManeuverStrategy.WEAVE.__doc__ = "Weave strategy."

agcls.AgTypeNameMap["BasicManeuverStrategy"] = BasicManeuverStrategy

class StraightAheadReferenceFrame(IntEnum):
    """Straight Ahead basic maneuver Reference Frame."""

    MAINTAIN_COURSE = 0
    """Maintain Course."""
    MAINTAIN_HEADING = 1
    """Maintain Heading."""
    NO_LATERAL_ACC = 2
    """No Lateral Accelerations."""
    COMPENSATE_CORIOLIS = 3
    """Compensate For Coriolis."""

StraightAheadReferenceFrame.MAINTAIN_COURSE.__doc__ = "Maintain Course."
StraightAheadReferenceFrame.MAINTAIN_HEADING.__doc__ = "Maintain Heading."
StraightAheadReferenceFrame.NO_LATERAL_ACC.__doc__ = "No Lateral Accelerations."
StraightAheadReferenceFrame.COMPENSATE_CORIOLIS.__doc__ = "Compensate For Coriolis."

agcls.AgTypeNameMap["StraightAheadReferenceFrame"] = StraightAheadReferenceFrame

class AirspeedType(IntEnum):
    """Airspeed types."""

    MACH = 0
    """Mach number."""
    EAS = 1
    """Equivalent airspeed."""
    CAS = 2
    """Calibrated airspeed."""
    TAS = 3
    """True airspeed."""

AirspeedType.MACH.__doc__ = "Mach number."
AirspeedType.EAS.__doc__ = "Equivalent airspeed."
AirspeedType.CAS.__doc__ = "Calibrated airspeed."
AirspeedType.TAS.__doc__ = "True airspeed."

agcls.AgTypeNameMap["AirspeedType"] = AirspeedType

class AerodynamicPropulsionSimpleMode(IntEnum):
    """Aircraft operating mode for basic acceleration models with aerodynamics set to Simple."""

    FIXED_WING = 0
    """Fixed wing operatiog mode."""
    HELICOPTER = 1
    """Helicopter operating mode."""

AerodynamicPropulsionSimpleMode.FIXED_WING.__doc__ = "Fixed wing operatiog mode."
AerodynamicPropulsionSimpleMode.HELICOPTER.__doc__ = "Helicopter operating mode."

agcls.AgTypeNameMap["AerodynamicPropulsionSimpleMode"] = AerodynamicPropulsionSimpleMode

class AerodynamicPropulsionFlightMode(IntEnum):
    """Flight mode for the Aero/Prop maneuver mode helper in aircraft acceleration models."""

    FLIGHT_PERFORMANCE_FORWARD_FLIGHT = 0
    """Forward Flight mode."""
    FLIGHT_PERFORMANCE_HOVER = 1
    """Hover mode."""
    FLIGHT_PERFORMANCE_TAKEOFF = 2
    """Takeoff mode."""
    FLIGHT_PERFORMANCE_LANDING = 3
    """Landing mode."""
    FLIGHT_PERFORMANCE_WEIGHT_ON_WHEELS = 4
    """Weight on wheels mode."""

AerodynamicPropulsionFlightMode.FLIGHT_PERFORMANCE_FORWARD_FLIGHT.__doc__ = "Forward Flight mode."
AerodynamicPropulsionFlightMode.FLIGHT_PERFORMANCE_HOVER.__doc__ = "Hover mode."
AerodynamicPropulsionFlightMode.FLIGHT_PERFORMANCE_TAKEOFF.__doc__ = "Takeoff mode."
AerodynamicPropulsionFlightMode.FLIGHT_PERFORMANCE_LANDING.__doc__ = "Landing mode."
AerodynamicPropulsionFlightMode.FLIGHT_PERFORMANCE_WEIGHT_ON_WHEELS.__doc__ = "Weight on wheels mode."

agcls.AgTypeNameMap["AerodynamicPropulsionFlightMode"] = AerodynamicPropulsionFlightMode

class PhaseOfFlight(IntEnum):
    """Flight mode for basic maneuver procedures."""

    FLIGHT_PHASE_TAKEOFF = 1
    """Takeoff flight mode."""
    FLIGHT_PHASE_CLIMB = 2
    """Forward flight climb flight mode."""
    FLIGHT_PHASE_CRUISE = 3
    """Forward flight - cruise flight mode."""
    FLIGHT_PHASE_DESCEND = 4
    """Forward flight descend flight mode."""
    FLIGHT_PHASE_LANDING = 5
    """Landing flight mode."""
    FLIGHT_PHASE_VTOL = 6
    """Rotary wing / hover flight mode."""

PhaseOfFlight.FLIGHT_PHASE_TAKEOFF.__doc__ = "Takeoff flight mode."
PhaseOfFlight.FLIGHT_PHASE_CLIMB.__doc__ = "Forward flight climb flight mode."
PhaseOfFlight.FLIGHT_PHASE_CRUISE.__doc__ = "Forward flight - cruise flight mode."
PhaseOfFlight.FLIGHT_PHASE_DESCEND.__doc__ = "Forward flight descend flight mode."
PhaseOfFlight.FLIGHT_PHASE_LANDING.__doc__ = "Landing flight mode."
PhaseOfFlight.FLIGHT_PHASE_VTOL.__doc__ = "Rotary wing / hover flight mode."

agcls.AgTypeNameMap["PhaseOfFlight"] = PhaseOfFlight

class CruiseSpeed(IntEnum):
    """Cruise airspeed type for the procedure."""

    MIN_AIRSPEED = 0
    """Minimum airspeed."""
    MAX_ENDURANCE_AIRSPEED = 1
    """Maximum endurance airspeed."""
    MAX_RANGE_AIRSPEED = 2
    """Maximum range airspeed."""
    OTHER_AIRSPEED = 3
    """Other airspeed."""
    MAX_AIRSPEED = 4
    """Maximum airspeed."""
    MAX_PERFORMANCE_AIRSPEED = 6
    """Maximum performance airspeed."""

CruiseSpeed.MIN_AIRSPEED.__doc__ = "Minimum airspeed."
CruiseSpeed.MAX_ENDURANCE_AIRSPEED.__doc__ = "Maximum endurance airspeed."
CruiseSpeed.MAX_RANGE_AIRSPEED.__doc__ = "Maximum range airspeed."
CruiseSpeed.OTHER_AIRSPEED.__doc__ = "Other airspeed."
CruiseSpeed.MAX_AIRSPEED.__doc__ = "Maximum airspeed."
CruiseSpeed.MAX_PERFORMANCE_AIRSPEED.__doc__ = "Maximum performance airspeed."

agcls.AgTypeNameMap["CruiseSpeed"] = CruiseSpeed

class TakeoffMode(IntEnum):
    """Takeoff procedure mode."""

    TAKEOFF_NORMAL = 0
    """Normal takeoff mode."""
    TAKEOFF_FLY_TO_DEPARTURE_POINT = 1
    """Fly to departure point takeoff mode."""
    TAKEOFF_LOW_TRANSITION = 2
    """Low transition takeoff."""

TakeoffMode.TAKEOFF_NORMAL.__doc__ = "Normal takeoff mode."
TakeoffMode.TAKEOFF_FLY_TO_DEPARTURE_POINT.__doc__ = "Fly to departure point takeoff mode."
TakeoffMode.TAKEOFF_LOW_TRANSITION.__doc__ = "Low transition takeoff."

agcls.AgTypeNameMap["TakeoffMode"] = TakeoffMode

class ApproachMode(IntEnum):
    """Landing procedure approach mode."""

    STANDARD_INSTRUMENT_APPROACH = 0
    """Standard instrument approach mode."""
    INTERCEPT_GLIDESLOPE = 1
    """Intercept Glideslope approach mode."""
    ENTER_DOWNWIND_PATTERN = 2
    """Enter downwind pattern approach mode."""

ApproachMode.STANDARD_INSTRUMENT_APPROACH.__doc__ = "Standard instrument approach mode."
ApproachMode.INTERCEPT_GLIDESLOPE.__doc__ = "Intercept Glideslope approach mode."
ApproachMode.ENTER_DOWNWIND_PATTERN.__doc__ = "Enter downwind pattern approach mode."

agcls.AgTypeNameMap["ApproachMode"] = ApproachMode

class NavigatorTurnDirection(IntEnum):
    """Turn mode for procedures with Enroute Turn Direction options."""

    NAVIGATOR_TURN_AUTO = 0
    """Automatic turn. Aviator will determine the direction of the turn."""
    NAVIGATOR_TURN_LEFT = 1
    """Left turn."""
    NAVIGATOR_TURN_RIGHT = 2
    """Right turn."""

NavigatorTurnDirection.NAVIGATOR_TURN_AUTO.__doc__ = "Automatic turn. Aviator will determine the direction of the turn."
NavigatorTurnDirection.NAVIGATOR_TURN_LEFT.__doc__ = "Left turn."
NavigatorTurnDirection.NAVIGATOR_TURN_RIGHT.__doc__ = "Right turn."

agcls.AgTypeNameMap["NavigatorTurnDirection"] = NavigatorTurnDirection

class BasicManeuverFuelFlowType(IntEnum):
    """Fuel flow type for basic maneuver procedures."""

    BASIC_MANEUVER_FUEL_FLOW_TAKEOFF = 0
    """Fuel flow defined for the current Takeoff performance model."""
    BASIC_MANEUVER_FUEL_FLOW_CRUISE = 1
    """Fuel flow defined for the current Cruise performance model."""
    BASIC_MANEUVER_FUEL_FLOW_LANDING = 2
    """Fuel flow defined for the current Landing performance model."""
    BASIC_MANEUVER_FUEL_FLOW_VTOL = 3
    """Fuel flow defined for the current VTOL performance model."""
    BASIC_MANEUVER_FUEL_FLOW_AERODYNAMIC_PROPULSION = 4
    """Fuel flow defined for the Aerodynamics and Propulsion Analysis component of the current basic acceleration performance model."""
    BASIC_MANEUVER_FUEL_FLOW_OVERRIDE = 5
    """Fuel flow defined manually."""
    BASIC_MANEUVER_FUEL_FLOW_THRUST_MODEL = 6
    """Fuel flow defined using the thrust model for this maneuver."""

BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_TAKEOFF.__doc__ = "Fuel flow defined for the current Takeoff performance model."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_CRUISE.__doc__ = "Fuel flow defined for the current Cruise performance model."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_LANDING.__doc__ = "Fuel flow defined for the current Landing performance model."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_VTOL.__doc__ = "Fuel flow defined for the current VTOL performance model."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_AERODYNAMIC_PROPULSION.__doc__ = "Fuel flow defined for the Aerodynamics and Propulsion Analysis component of the current basic acceleration performance model."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_OVERRIDE.__doc__ = "Fuel flow defined manually."
BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_THRUST_MODEL.__doc__ = "Fuel flow defined using the thrust model for this maneuver."

agcls.AgTypeNameMap["BasicManeuverFuelFlowType"] = BasicManeuverFuelFlowType

class BasicManeuverAltitudeLimit(IntEnum):
    """The type of response Aviator will have if the maneuver attempts to exceed the altitude limit."""

    BASIC_MANEUVER_ALTITUDE_LIMIT_ERROR = 0
    """Error when altitude limit exceeded."""
    BASIC_MANEUVER_ALTITUDE_LIMIT_STOP = 1
    """Stop when altitude limit exceeded."""
    BASIC_MANEUVER_ALTITUDE_LIMIT_CONTINUE = 2
    """Continue when altitude limit exceeded."""

BasicManeuverAltitudeLimit.BASIC_MANEUVER_ALTITUDE_LIMIT_ERROR.__doc__ = "Error when altitude limit exceeded."
BasicManeuverAltitudeLimit.BASIC_MANEUVER_ALTITUDE_LIMIT_STOP.__doc__ = "Stop when altitude limit exceeded."
BasicManeuverAltitudeLimit.BASIC_MANEUVER_ALTITUDE_LIMIT_CONTINUE.__doc__ = "Continue when altitude limit exceeded."

agcls.AgTypeNameMap["BasicManeuverAltitudeLimit"] = BasicManeuverAltitudeLimit

class RunwayHighLowEnd(IntEnum):
    """Runway heading that the aircraft will use."""

    HIGH_END = 0
    """High end."""
    LOW_END = 1
    """Low end."""
    HEADWIND = 2
    """Headwind."""

RunwayHighLowEnd.HIGH_END.__doc__ = "High end."
RunwayHighLowEnd.LOW_END.__doc__ = "Low end."
RunwayHighLowEnd.HEADWIND.__doc__ = "Headwind."

agcls.AgTypeNameMap["RunwayHighLowEnd"] = RunwayHighLowEnd

class BasicManeuverReferenceFrame(IntEnum):
    """Reference frame for the basic maneuver strategy."""

    EARTH_FRAME = 0
    """Earth frame."""
    WIND_FRAME = 1
    """Wind frame."""

BasicManeuverReferenceFrame.EARTH_FRAME.__doc__ = "Earth frame."
BasicManeuverReferenceFrame.WIND_FRAME.__doc__ = "Wind frame."

agcls.AgTypeNameMap["BasicManeuverReferenceFrame"] = BasicManeuverReferenceFrame

class BasicManeuverStrategyNavigationControlLimit(IntEnum):
    """Define the control limits for the aircraft during the maneuver."""

    NAVIGATION_USE_ACCELERATION_PERFORMANCE_MODEL = 0
    """Use Accel Perf Model."""
    NAVIGATION_MIN_TURN_RADIUS = 1
    """Specify min turn radius."""
    NAVIGATION_MAX_TURN_RATE = 2
    """Specify max turn rate."""
    NAVIGATION_MAX_HORIZONTAL_ACCELERATION = 3
    """Specify max horiz accel."""

BasicManeuverStrategyNavigationControlLimit.NAVIGATION_USE_ACCELERATION_PERFORMANCE_MODEL.__doc__ = "Use Accel Perf Model."
BasicManeuverStrategyNavigationControlLimit.NAVIGATION_MIN_TURN_RADIUS.__doc__ = "Specify min turn radius."
BasicManeuverStrategyNavigationControlLimit.NAVIGATION_MAX_TURN_RATE.__doc__ = "Specify max turn rate."
BasicManeuverStrategyNavigationControlLimit.NAVIGATION_MAX_HORIZONTAL_ACCELERATION.__doc__ = "Specify max horiz accel."

agcls.AgTypeNameMap["BasicManeuverStrategyNavigationControlLimit"] = BasicManeuverStrategyNavigationControlLimit

class AccelerationManeuverMode(IntEnum):
    """The mode that the aircraft will adhere to the specified acceleration parameters."""

    ACCELERATION_MANEUVER_MODE_NORMAL = 0
    """Constant value (default)."""
    ACCELERATION_MANEUVER_MODE_DENSITY_SCALE = 1
    """Scale by amtmospheric density. The aircraft will consider dynamic pressure when calculating turn radius."""
    ACCELERATION_MANEUVER_MODE_AERODYNAMIC_PROPULSION = 2
    """Aero/Prop maneuver mode."""

AccelerationManeuverMode.ACCELERATION_MANEUVER_MODE_NORMAL.__doc__ = "Constant value (default)."
AccelerationManeuverMode.ACCELERATION_MANEUVER_MODE_DENSITY_SCALE.__doc__ = "Scale by amtmospheric density. The aircraft will consider dynamic pressure when calculating turn radius."
AccelerationManeuverMode.ACCELERATION_MANEUVER_MODE_AERODYNAMIC_PROPULSION.__doc__ = "Aero/Prop maneuver mode."

agcls.AgTypeNameMap["AccelerationManeuverMode"] = AccelerationManeuverMode

class AircraftAerodynamicStrategy(IntEnum):
    """The aerodynamic strategy used to compute lift, drag, angle of attack, sideslip and intermediate / derived values."""

    AIRCRAFT_AERODYNAMIC_SIMPLE = 0
    """Simple aerodynamics."""
    AIRCRAFT_AERODYNAMIC_EXTERNAL_FILE = 1
    """External file aerodynamics."""
    AIRCRAFT_AERODYNAMIC_BASIC_FIXED_WING = 2
    """Basic fixed wing aerodynamics."""
    AIRCRAFT_AERODYNAMIC_ADVANCED_MISSILE = 3
    """Advanced missile aerodynamics."""
    AIRCRAFT_AERODYNAMIC_FOUR_POINT = 4
    """Four Point aerodynamics."""

AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE.__doc__ = "Simple aerodynamics."
AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_EXTERNAL_FILE.__doc__ = "External file aerodynamics."
AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_BASIC_FIXED_WING.__doc__ = "Basic fixed wing aerodynamics."
AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_ADVANCED_MISSILE.__doc__ = "Advanced missile aerodynamics."
AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_FOUR_POINT.__doc__ = "Four Point aerodynamics."

agcls.AgTypeNameMap["AircraftAerodynamicStrategy"] = AircraftAerodynamicStrategy

class AircraftPropulsionStrategy(IntEnum):
    """The propulsion strategy used to compute thrust and throttle setting."""

    AIRCRAFT_PROPULSION_SIMPLE = 0
    """Simple propulsion."""
    AIRCRAFT_PROPULSION_EXTERNAL_FILE = 1
    """External file propulsion."""
    AIRCRAFT_PROPULSION_BASIC_FIXED_WING = 2
    """Basic fixed wing propulsion."""
    AIRCRAFT_PROPULSION_MISSILE_RAMJET = 3
    """Missile - Ramjet propulsion."""
    AIRCRAFT_PROPULSION_MISSILE_ROCKET = 4
    """Missile - Rocket propulsion."""
    AIRCRAFT_PROPULSION_MISSILE_TURBOJET = 5
    """Missile - Turbojet propulsion."""

AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE.__doc__ = "Simple propulsion."
AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_EXTERNAL_FILE.__doc__ = "External file propulsion."
AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_BASIC_FIXED_WING.__doc__ = "Basic fixed wing propulsion."
AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_MISSILE_RAMJET.__doc__ = "Missile - Ramjet propulsion."
AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_MISSILE_ROCKET.__doc__ = "Missile - Rocket propulsion."
AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_MISSILE_TURBOJET.__doc__ = "Missile - Turbojet propulsion."

agcls.AgTypeNameMap["AircraftPropulsionStrategy"] = AircraftPropulsionStrategy

class AGLMSL(IntEnum):
    """The altitude mode."""

    ALTITUDE_AGL = 0
    """AGl altitude. ALtitude above local terrain."""
    ALTITUDE_MSL = 1
    """MSL altitude. Altitude above sea level."""

AGLMSL.ALTITUDE_AGL.__doc__ = "AGl altitude. ALtitude above local terrain."
AGLMSL.ALTITUDE_MSL.__doc__ = "MSL altitude. Altitude above sea level."

agcls.AgTypeNameMap["AGLMSL"] = AGLMSL

class LandingApproachFixRangeMode(IntEnum):
    """The reference point on the runway for the Approach Fix Range."""

    RELATIVE_TO_RUNWAY_CENTER = 0
    """Runway center."""
    RELATIVE_TO_RUNWAY_END = 1
    """Runway end."""

LandingApproachFixRangeMode.RELATIVE_TO_RUNWAY_CENTER.__doc__ = "Runway center."
LandingApproachFixRangeMode.RELATIVE_TO_RUNWAY_END.__doc__ = "Runway end."

agcls.AgTypeNameMap["LandingApproachFixRangeMode"] = LandingApproachFixRangeMode

class AccelerationAdvancedAccelerationMode(IntEnum):
    """Acceleration mode for aircraft advanced acceleration models."""

    ACCELERATION_MODE_MAX_ACCELERATION = 0
    """Max acceleration."""
    ACCELERATION_MODE_OVERRIDE_ACCELERATION = 1
    """Manually override the acceleration."""

AccelerationAdvancedAccelerationMode.ACCELERATION_MODE_MAX_ACCELERATION.__doc__ = "Max acceleration."
AccelerationAdvancedAccelerationMode.ACCELERATION_MODE_OVERRIDE_ACCELERATION.__doc__ = "Manually override the acceleration."

agcls.AgTypeNameMap["AccelerationAdvancedAccelerationMode"] = AccelerationAdvancedAccelerationMode

class AccelerationManeuverAerodynamicPropulsionMode(IntEnum):
    """The mode used for the Aero/Prop maneuver mode helper for aircraft basic acceleration models."""

    USE_THRUST_AND_LIFT_COEFFICIENT = 0
    """Use Thrust and Lift Coefficient."""
    USE_LIFT_COEFFICIENT_ONLY = 1
    """Use Lift Coefficient only."""

AccelerationManeuverAerodynamicPropulsionMode.USE_THRUST_AND_LIFT_COEFFICIENT.__doc__ = "Use Thrust and Lift Coefficient."
AccelerationManeuverAerodynamicPropulsionMode.USE_LIFT_COEFFICIENT_ONLY.__doc__ = "Use Lift Coefficient only."

agcls.AgTypeNameMap["AccelerationManeuverAerodynamicPropulsionMode"] = AccelerationManeuverAerodynamicPropulsionMode

class BasicManeuverStrategyAirspeedPerformanceLimits(IntEnum):
    """The type of response Aviator will have if the basic maneuver attempts to exceed the airspeed limit."""

    CONSTRAIN_IF_VIOLATED = 0
    """Constrain the aircraft to not exceed the airspeed limit."""
    STOP_IF_VIOLATED = 1
    """Stop when airspeed limit exceeded.."""
    ERROR_IF_VIOLATED = 2
    """Error when airspeed limit exceeded."""
    IGNORE_IF_VIOLATED = 3
    """Ignore when airspeed limit exceeded."""

BasicManeuverStrategyAirspeedPerformanceLimits.CONSTRAIN_IF_VIOLATED.__doc__ = "Constrain the aircraft to not exceed the airspeed limit."
BasicManeuverStrategyAirspeedPerformanceLimits.STOP_IF_VIOLATED.__doc__ = "Stop when airspeed limit exceeded.."
BasicManeuverStrategyAirspeedPerformanceLimits.ERROR_IF_VIOLATED.__doc__ = "Error when airspeed limit exceeded."
BasicManeuverStrategyAirspeedPerformanceLimits.IGNORE_IF_VIOLATED.__doc__ = "Ignore when airspeed limit exceeded."

agcls.AgTypeNameMap["BasicManeuverStrategyAirspeedPerformanceLimits"] = BasicManeuverStrategyAirspeedPerformanceLimits

class BasicManeuverStrategyPoweredCruiseMode(IntEnum):
    """Powered Cruise Options."""

    GLIDE_SPECIFY_UN_POWERED_CRUISE = 0
    """The mode selected is unpowered options."""
    GLIDE_SPECIFY_THROTTLE = 1
    """The mode selected is specify throttle."""
    GLIDE_SPECIFY_THRUST_MODEL = 2
    """The mode selected is thrust model."""

BasicManeuverStrategyPoweredCruiseMode.GLIDE_SPECIFY_UN_POWERED_CRUISE.__doc__ = "The mode selected is unpowered options."
BasicManeuverStrategyPoweredCruiseMode.GLIDE_SPECIFY_THROTTLE.__doc__ = "The mode selected is specify throttle."
BasicManeuverStrategyPoweredCruiseMode.GLIDE_SPECIFY_THRUST_MODEL.__doc__ = "The mode selected is thrust model."

agcls.AgTypeNameMap["BasicManeuverStrategyPoweredCruiseMode"] = BasicManeuverStrategyPoweredCruiseMode

class TurnMode(IntEnum):
    """The mode to specify an aircraft's level turn performance for acceleration performance models."""

    TURN_MODE_TURN_G = 0
    """Turn G. The standard G force of the aircraft in a turn."""
    TURN_MODE_BANK_ANGLE = 1
    """Bank angle."""
    TURN_MODE_ACCELERATION = 2
    """Turn acceleration."""
    TURN_MODE_RADIUS = 3
    """Turn radius."""
    TURN_MODE_RATE = 4
    """Turn rate."""

TurnMode.TURN_MODE_TURN_G.__doc__ = "Turn G. The standard G force of the aircraft in a turn."
TurnMode.TURN_MODE_BANK_ANGLE.__doc__ = "Bank angle."
TurnMode.TURN_MODE_ACCELERATION.__doc__ = "Turn acceleration."
TurnMode.TURN_MODE_RADIUS.__doc__ = "Turn radius."
TurnMode.TURN_MODE_RATE.__doc__ = "Turn rate."

agcls.AgTypeNameMap["TurnMode"] = TurnMode

class PointToPointMode(IntEnum):
    """The heading or course of the aircraft at the beginning of the procedure."""

    FLY_DIRECT = 0
    """Fly direct."""
    ARRIVE_ON_COURSE_FOR_NEXT = 1
    """Arrive on course for next procedure."""
    ARRIVE_ON_COURSE = 2
    """Arrive on course."""
    INSCRIBED_TURN = 3
    """Inscribed turn."""
    ARRIVE_ON_HDG_INTO_WIND = 4
    """Arrive on heading into wind."""

PointToPointMode.FLY_DIRECT.__doc__ = "Fly direct."
PointToPointMode.ARRIVE_ON_COURSE_FOR_NEXT.__doc__ = "Arrive on course for next procedure."
PointToPointMode.ARRIVE_ON_COURSE.__doc__ = "Arrive on course."
PointToPointMode.INSCRIBED_TURN.__doc__ = "Inscribed turn."
PointToPointMode.ARRIVE_ON_HDG_INTO_WIND.__doc__ = "Arrive on heading into wind."

agcls.AgTypeNameMap["PointToPointMode"] = PointToPointMode

class AltitudeConstraintManeuverMode(IntEnum):
    """Turn mode for procedures that may require a level off maneuver."""

    LEVEL_OFF_AUTOMATIC_MANEUVER = 0
    """Automatic turn.Aviator will determine the direction of the turn."""
    LEVEL_OFF_LEFT_TURN_MANEUVER = 1
    """Left turn."""
    LEVEL_OFF_RIGHT_TURN_MANEUVER = 2
    """Right turn."""
    LEVEL_OFF_NO_MANEUVER = 3
    """No turn. A level off maneuver will not be performed."""

AltitudeConstraintManeuverMode.LEVEL_OFF_AUTOMATIC_MANEUVER.__doc__ = "Automatic turn.Aviator will determine the direction of the turn."
AltitudeConstraintManeuverMode.LEVEL_OFF_LEFT_TURN_MANEUVER.__doc__ = "Left turn."
AltitudeConstraintManeuverMode.LEVEL_OFF_RIGHT_TURN_MANEUVER.__doc__ = "Right turn."
AltitudeConstraintManeuverMode.LEVEL_OFF_NO_MANEUVER.__doc__ = "No turn. A level off maneuver will not be performed."

agcls.AgTypeNameMap["AltitudeConstraintManeuverMode"] = AltitudeConstraintManeuverMode

class WindModelType(IntEnum):
    """The wind model type."""

    CONSTANT_WIND = 0
    """Constant Wind/Speed."""
    ADDS = 1
    """NOAA ADDS Service."""
    DISABLED = 2
    """Disabled wind model."""

WindModelType.CONSTANT_WIND.__doc__ = "Constant Wind/Speed."
WindModelType.ADDS.__doc__ = "NOAA ADDS Service."
WindModelType.DISABLED.__doc__ = "Disabled wind model."

agcls.AgTypeNameMap["WindModelType"] = WindModelType

class WindAtmosphereModelSource(IntEnum):
    """The source for the wind or atmosphere model."""

    SCENARIO_MODEL = 0
    """Scenario Model."""
    MISSION_MODEL = 1
    """Mission Model."""
    PROCEDURE_MODEL = 2
    """Procedure Model."""

WindAtmosphereModelSource.SCENARIO_MODEL.__doc__ = "Scenario Model."
WindAtmosphereModelSource.MISSION_MODEL.__doc__ = "Mission Model."
WindAtmosphereModelSource.PROCEDURE_MODEL.__doc__ = "Procedure Model."

agcls.AgTypeNameMap["WindAtmosphereModelSource"] = WindAtmosphereModelSource

class ADDSMessageInterpolationType(IntEnum):
    """The interpolation method for the wind conditions."""

    INTERPOLATION_ONE_POINT = 0
    """One Point Interpolation."""
    INTERPOLATION_TWO_POINT = 1
    """Two Point Interpolation."""

ADDSMessageInterpolationType.INTERPOLATION_ONE_POINT.__doc__ = "One Point Interpolation."
ADDSMessageInterpolationType.INTERPOLATION_TWO_POINT.__doc__ = "Two Point Interpolation."

agcls.AgTypeNameMap["ADDSMessageInterpolationType"] = ADDSMessageInterpolationType

class ADDSMissingMessageType(IntEnum):
    """The wind effect to apply if there is an interval gap between messages."""

    MISSING_MESSAGE_ZERO_WIND = 0
    """Zero Wind."""
    MISSING_MESSAGE_INTERPOLATION_END_POINTS = 1
    """Interpolate End Points."""

ADDSMissingMessageType.MISSING_MESSAGE_ZERO_WIND.__doc__ = "Zero Wind."
ADDSMissingMessageType.MISSING_MESSAGE_INTERPOLATION_END_POINTS.__doc__ = "Interpolate End Points."

agcls.AgTypeNameMap["ADDSMissingMessageType"] = ADDSMissingMessageType

class ADDSMessageExtrapolationType(IntEnum):
    """The wind effect to apply if the procedure(s) extend beyond the intervals of any available messages."""

    EXTRAPOLATION_MESSAGE_ZERO_WIND = 0
    """Zero Wind."""
    EXTRAPOLATION_MESSAGE_HOLD_END_POINTS = 1
    """Hold End Point Wind."""

ADDSMessageExtrapolationType.EXTRAPOLATION_MESSAGE_ZERO_WIND.__doc__ = "Zero Wind."
ADDSMessageExtrapolationType.EXTRAPOLATION_MESSAGE_HOLD_END_POINTS.__doc__ = "Hold End Point Wind."

agcls.AgTypeNameMap["ADDSMessageExtrapolationType"] = ADDSMessageExtrapolationType

class ADDSForecastType(IntEnum):
    """The forecast type for the NOAA ADDS message."""

    HOUR_6 = 0
    """6 hour forecast."""
    HOUR_12 = 1
    """12 hour forecast."""
    HOUR_24 = 2
    """24 hour forecast."""

ADDSForecastType.HOUR_6.__doc__ = "6 hour forecast."
ADDSForecastType.HOUR_12.__doc__ = "12 hour forecast."
ADDSForecastType.HOUR_24.__doc__ = "24 hour forecast."

agcls.AgTypeNameMap["ADDSForecastType"] = ADDSForecastType

class AtmosphereModelType(IntEnum):
    """The basic atmosphere model type."""

    STANDARD1976 = 0
    """1976 U.S. Standard Atmosphere."""
    MIL_HOT = 1
    """U.S. MIL HDBK 310 - Hot."""
    MIL_COLD = 2
    """U.S. MIL HDBK 310 - Cold."""
    MIL_LOW_DENSITY = 3
    """U.S. MIL HDBK 310 - Low Density."""
    MIL_HIGH_DENSITY = 4
    """U.S. MIL HDBK 310 - High Density."""
    MIL_INTERPOLATE = 5
    """Interpolate MIL HDBK 310 Data."""

AtmosphereModelType.STANDARD1976.__doc__ = "1976 U.S. Standard Atmosphere."
AtmosphereModelType.MIL_HOT.__doc__ = "U.S. MIL HDBK 310 - Hot."
AtmosphereModelType.MIL_COLD.__doc__ = "U.S. MIL HDBK 310 - Cold."
AtmosphereModelType.MIL_LOW_DENSITY.__doc__ = "U.S. MIL HDBK 310 - Low Density."
AtmosphereModelType.MIL_HIGH_DENSITY.__doc__ = "U.S. MIL HDBK 310 - High Density."
AtmosphereModelType.MIL_INTERPOLATE.__doc__ = "Interpolate MIL HDBK 310 Data."

agcls.AgTypeNameMap["AtmosphereModelType"] = AtmosphereModelType

class SmoothTurnMode(IntEnum):
    """The basic maneuver smooth turn mode."""

    SMOOTH_TURN_LOAD_FACTOR = 0
    """Specify the load factor of the smooth turn."""
    SMOOTH_TURN_ROLL_ANGLE = 1
    """Specify the roll angle of the smooth turn."""

SmoothTurnMode.SMOOTH_TURN_LOAD_FACTOR.__doc__ = "Specify the load factor of the smooth turn."
SmoothTurnMode.SMOOTH_TURN_ROLL_ANGLE.__doc__ = "Specify the roll angle of the smooth turn."

agcls.AgTypeNameMap["SmoothTurnMode"] = SmoothTurnMode

class PerformanceModelOverride(IntEnum):
    """The performance model override mode."""

    PERFORMANCE_MODEL_VALUE = 0
    """Use the performance model value."""
    OVERRIDE = 1
    """Override the performance model value."""

PerformanceModelOverride.PERFORMANCE_MODEL_VALUE.__doc__ = "Use the performance model value."
PerformanceModelOverride.OVERRIDE.__doc__ = "Override the performance model value."

agcls.AgTypeNameMap["PerformanceModelOverride"] = PerformanceModelOverride

class BasicManeuverAirspeedMode(IntEnum):
    """The basic maneuver airspeed mode."""

    MAINTAIN_CURRENT_AIRSPEED = 0
    """Maintain the current airspeed."""
    MAINTAIN_SPECIFIED_AIRSPEED = 1
    """Maintain the specified airspeed."""
    MAINTAIN_MIN_AIRSPEED = 2
    """Maintain the minimum airspeed for the aircraft."""
    MAINTAIN_MAX_ENDURANCE_AIRSPEED = 3
    """Maintain the maximum endurance airspeed for the aircraft."""
    MAINTAIN_MAX_RANGE_AIRSPEED = 4
    """Maintain the maximum range airspeed for the aircraft."""
    MAINTAIN_MAX_AIRSPEED = 5
    """Maintain the maximum airspeed for the aircraft."""
    MAINTAIN_MAX_PERFORMANCE_AIRSPEED = 6
    """Maintain the maximum performance airspeed for the aircraft."""
    ACCELERATION_AT_G = 7
    """Accelerate at the specified rate."""
    DECELERATION_AT_G = 8
    """Decelerate at the specified rate."""
    ACCELERATION_DECELERATION_UNDER_GRAVITY = 9
    """Accel/Decel at the force of gravity (no drag, no thrust)."""
    ACCELERATION_DECELERATION_AERODYNAMIC_PROPULSION = 10
    """Accel/Decel using Aero/Propulsion with throttle setting."""
    THRUST = 11
    """Specify thrust (using drag from Aerodynamics model)."""
    INTERPOLATE_ACCELERATION_DECELERATION = 12
    """Interpolate Accelerate/Decelerate over interval."""

BasicManeuverAirspeedMode.MAINTAIN_CURRENT_AIRSPEED.__doc__ = "Maintain the current airspeed."
BasicManeuverAirspeedMode.MAINTAIN_SPECIFIED_AIRSPEED.__doc__ = "Maintain the specified airspeed."
BasicManeuverAirspeedMode.MAINTAIN_MIN_AIRSPEED.__doc__ = "Maintain the minimum airspeed for the aircraft."
BasicManeuverAirspeedMode.MAINTAIN_MAX_ENDURANCE_AIRSPEED.__doc__ = "Maintain the maximum endurance airspeed for the aircraft."
BasicManeuverAirspeedMode.MAINTAIN_MAX_RANGE_AIRSPEED.__doc__ = "Maintain the maximum range airspeed for the aircraft."
BasicManeuverAirspeedMode.MAINTAIN_MAX_AIRSPEED.__doc__ = "Maintain the maximum airspeed for the aircraft."
BasicManeuverAirspeedMode.MAINTAIN_MAX_PERFORMANCE_AIRSPEED.__doc__ = "Maintain the maximum performance airspeed for the aircraft."
BasicManeuverAirspeedMode.ACCELERATION_AT_G.__doc__ = "Accelerate at the specified rate."
BasicManeuverAirspeedMode.DECELERATION_AT_G.__doc__ = "Decelerate at the specified rate."
BasicManeuverAirspeedMode.ACCELERATION_DECELERATION_UNDER_GRAVITY.__doc__ = "Accel/Decel at the force of gravity (no drag, no thrust)."
BasicManeuverAirspeedMode.ACCELERATION_DECELERATION_AERODYNAMIC_PROPULSION.__doc__ = "Accel/Decel using Aero/Propulsion with throttle setting."
BasicManeuverAirspeedMode.THRUST.__doc__ = "Specify thrust (using drag from Aerodynamics model)."
BasicManeuverAirspeedMode.INTERPOLATE_ACCELERATION_DECELERATION.__doc__ = "Interpolate Accelerate/Decelerate over interval."

agcls.AgTypeNameMap["BasicManeuverAirspeedMode"] = BasicManeuverAirspeedMode

class AileronRollFlightPath(IntEnum):
    """The flight path option for an aileron roll strategy for a basic maneuver procedure."""

    STRAIGHT_LINE_FLIGHT_PATH = 0
    """Fly a straight line flight path."""
    ZERO_G_FLIGHT_PATH = 1
    """Fly a zero G flight path."""

AileronRollFlightPath.STRAIGHT_LINE_FLIGHT_PATH.__doc__ = "Fly a straight line flight path."
AileronRollFlightPath.ZERO_G_FLIGHT_PATH.__doc__ = "Fly a zero G flight path."

agcls.AgTypeNameMap["AileronRollFlightPath"] = AileronRollFlightPath

class RollLeftRight(IntEnum):
    """The roll direction for an aileron roll strategy for a basic maneuver procedure."""

    ROLL_LEFT = 0
    """Roll left."""
    ROLL_RIGHT = 1
    """Roll right."""

RollLeftRight.ROLL_LEFT.__doc__ = "Roll left."
RollLeftRight.ROLL_RIGHT.__doc__ = "Roll right."

agcls.AgTypeNameMap["RollLeftRight"] = RollLeftRight

class RollUprightInverted(IntEnum):
    """The orientation for an aileron roll strategy for a basic maneuver procedure."""

    ROLL_UPRIGHT = 0
    """Upright roll."""
    ROLL_INVERTED = 1
    """Inverted roll."""

RollUprightInverted.ROLL_UPRIGHT.__doc__ = "Upright roll."
RollUprightInverted.ROLL_INVERTED.__doc__ = "Inverted roll."

agcls.AgTypeNameMap["RollUprightInverted"] = RollUprightInverted

class AileronRollMode(IntEnum):
    """The roll mode aileron roll strategy for a basic maneuver procedure."""

    ROLL_TO_ANGLE = 0
    """Specify the angle to roll."""
    ROLL_TO_ORIENTATION = 1
    """Specify the orientation to roll to."""

AileronRollMode.ROLL_TO_ANGLE.__doc__ = "Specify the angle to roll."
AileronRollMode.ROLL_TO_ORIENTATION.__doc__ = "Specify the orientation to roll to."

agcls.AgTypeNameMap["AileronRollMode"] = AileronRollMode

class FlyAOALeftRight(IntEnum):
    """The roll direction for a Fly AOA strategy for a basic maneuver procedure."""

    FLY_AOA_LEFT = 0
    """Roll left."""
    FLY_AOA_RIGHT = 1
    """Roll right."""
    FLY_AOA_NO_ROLL = 2
    """No roll."""

FlyAOALeftRight.FLY_AOA_LEFT.__doc__ = "Roll left."
FlyAOALeftRight.FLY_AOA_RIGHT.__doc__ = "Roll right."
FlyAOALeftRight.FLY_AOA_NO_ROLL.__doc__ = "No roll."

agcls.AgTypeNameMap["FlyAOALeftRight"] = FlyAOALeftRight

class SmoothAccelerationLeftRight(IntEnum):
    """The roll direction for a smooth acceleration strategy for a basic maneuver procedure."""

    SMOOTH_ACCELERATION_LEFT = 0
    """Roll left."""
    SMOOTH_ACCELERATION_RIGHT = 1
    """Roll right."""
    SMOOTH_ACCELERATION_NO_ROLL = 2
    """No roll."""

SmoothAccelerationLeftRight.SMOOTH_ACCELERATION_LEFT.__doc__ = "Roll left."
SmoothAccelerationLeftRight.SMOOTH_ACCELERATION_RIGHT.__doc__ = "Roll right."
SmoothAccelerationLeftRight.SMOOTH_ACCELERATION_NO_ROLL.__doc__ = "No roll."

agcls.AgTypeNameMap["SmoothAccelerationLeftRight"] = SmoothAccelerationLeftRight

class PullMode(IntEnum):
    """The pull mode for a pull strategy of a basic maneuver procedure."""

    PULL_TO_ANGLE = 0
    """Pull to the specified angle."""
    PULL_TO_HORIZON = 1
    """Pull to the horizon plus the additional angle."""

PullMode.PULL_TO_ANGLE.__doc__ = "Pull to the specified angle."
PullMode.PULL_TO_HORIZON.__doc__ = "Pull to the horizon plus the additional angle."

agcls.AgTypeNameMap["PullMode"] = PullMode

class RollingPullMode(IntEnum):
    """The rolling pull mode for a rolling pull strategy of a basic maneuver procedure."""

    ROLL_TO_ANGLE_MODE = 0
    """Roll to the specified angle."""
    ROLL_TO_ORIENTATION_MODE = 1
    """Roll to the specified orientaiton plus additional angle."""
    PULL_TO_ANGLE_MODE = 2
    """Pull to the specified angle."""
    PULL_TO_HORIZON_MODE = 3
    """Pull to the horizon plus the additional angle."""

RollingPullMode.ROLL_TO_ANGLE_MODE.__doc__ = "Roll to the specified angle."
RollingPullMode.ROLL_TO_ORIENTATION_MODE.__doc__ = "Roll to the specified orientaiton plus additional angle."
RollingPullMode.PULL_TO_ANGLE_MODE.__doc__ = "Pull to the specified angle."
RollingPullMode.PULL_TO_HORIZON_MODE.__doc__ = "Pull to the horizon plus the additional angle."

agcls.AgTypeNameMap["RollingPullMode"] = RollingPullMode

class SmoothAccelerationStopConditions(IntEnum):
    """The rolling pull mode for a rolling pull strategy of a basic maneuver procedure."""

    ROLL_RATE_AND_LOAD_FACTOR = 0
    """Roll rate and load factor rate achieved."""
    ROLL_RATE_OR_LOAD_FACTOR = 1
    """Roll rate or load factor rate achieved."""
    SMOOTH_ACCELERATION_NORMAL_STOP_CONDITIONS = 2
    """Basic stop conditions."""

SmoothAccelerationStopConditions.ROLL_RATE_AND_LOAD_FACTOR.__doc__ = "Roll rate and load factor rate achieved."
SmoothAccelerationStopConditions.ROLL_RATE_OR_LOAD_FACTOR.__doc__ = "Roll rate or load factor rate achieved."
SmoothAccelerationStopConditions.SMOOTH_ACCELERATION_NORMAL_STOP_CONDITIONS.__doc__ = "Basic stop conditions."

agcls.AgTypeNameMap["SmoothAccelerationStopConditions"] = SmoothAccelerationStopConditions

class AutopilotHorizontalPlaneMode(IntEnum):
    """The autopilot mode for an autopilot - horizontal plane strategy of a basic maneuver procedure."""

    AUTOPILOT_ABSOLUTE_HEADING = 0
    """The absolute heading mode."""
    AUTOPILOT_ABSOLUTE_COURSE = 1
    """The absolute course mode."""
    AUTOPILOT_RELATIVE_HEADING = 2
    """The relative heading change mode."""
    AUTOPILOT_RELATIVE_COURSE = 3
    """The relative course change mode."""
    AUTOPILOT_HEADING_RATE = 4
    """The set heading rate mode."""
    AUTOPILOT_COURSE_RATE = 5
    """The set course rate  mode."""

AutopilotHorizontalPlaneMode.AUTOPILOT_ABSOLUTE_HEADING.__doc__ = "The absolute heading mode."
AutopilotHorizontalPlaneMode.AUTOPILOT_ABSOLUTE_COURSE.__doc__ = "The absolute course mode."
AutopilotHorizontalPlaneMode.AUTOPILOT_RELATIVE_HEADING.__doc__ = "The relative heading change mode."
AutopilotHorizontalPlaneMode.AUTOPILOT_RELATIVE_COURSE.__doc__ = "The relative course change mode."
AutopilotHorizontalPlaneMode.AUTOPILOT_HEADING_RATE.__doc__ = "The set heading rate mode."
AutopilotHorizontalPlaneMode.AUTOPILOT_COURSE_RATE.__doc__ = "The set course rate  mode."

agcls.AgTypeNameMap["AutopilotHorizontalPlaneMode"] = AutopilotHorizontalPlaneMode

class AngleMode(IntEnum):
    """The angle mode for a barrel roll strategy of a basic maneuver procedure."""

    RELATIVE_ANGLE = 0
    """The absolute angle option."""
    ABSOLUTE_ANGLE = 1
    """The relative angle option."""

AngleMode.RELATIVE_ANGLE.__doc__ = "The absolute angle option."
AngleMode.ABSOLUTE_ANGLE.__doc__ = "The relative angle option."

agcls.AgTypeNameMap["AngleMode"] = AngleMode

class HoverAltitudeMode(IntEnum):
    """The altitude mode for the lighter than air hover strategy of a basic maneuver procedure."""

    HOVER_HOLD_INIT_ALTITUDE = 0
    """The hold initial altitude mode."""
    HOVER_SPECIFY_ALTITUDE = 1
    """The specify altitude mode."""
    HOVER_SPECIFY_ALTITUDE_CHANGE = 2
    """The specify altitude change mode."""
    HOVER_SPECIFY_ALTITUDE_RATE = 3
    """The specify altitude rate mode."""
    HOVER_HOLD_INIT_ALTITUDE_RATE = 4
    """The hold initial altitude rate mode."""
    HOVER_PARACHUTE = 5
    """The parachute mode."""

HoverAltitudeMode.HOVER_HOLD_INIT_ALTITUDE.__doc__ = "The hold initial altitude mode."
HoverAltitudeMode.HOVER_SPECIFY_ALTITUDE.__doc__ = "The specify altitude mode."
HoverAltitudeMode.HOVER_SPECIFY_ALTITUDE_CHANGE.__doc__ = "The specify altitude change mode."
HoverAltitudeMode.HOVER_SPECIFY_ALTITUDE_RATE.__doc__ = "The specify altitude rate mode."
HoverAltitudeMode.HOVER_HOLD_INIT_ALTITUDE_RATE.__doc__ = "The hold initial altitude rate mode."
HoverAltitudeMode.HOVER_PARACHUTE.__doc__ = "The parachute mode."

agcls.AgTypeNameMap["HoverAltitudeMode"] = HoverAltitudeMode

class HoverHeadingMode(IntEnum):
    """The heading mode for the lighter than air hover strategy of a basic maneuver procedure."""

    HOVER_RELATIVE = 0
    """The relative to start heading mode."""
    HOVER_ABSOLUTE = 1
    """The absolute heading mode."""
    HOVER_INTO_WIND = 2
    """The align into wind heading mode."""
    HOVER_OPPOSITE_WIND = 3
    """The align opposite wind heading mode."""

HoverHeadingMode.HOVER_RELATIVE.__doc__ = "The relative to start heading mode."
HoverHeadingMode.HOVER_ABSOLUTE.__doc__ = "The absolute heading mode."
HoverHeadingMode.HOVER_INTO_WIND.__doc__ = "The align into wind heading mode."
HoverHeadingMode.HOVER_OPPOSITE_WIND.__doc__ = "The align opposite wind heading mode."

agcls.AgTypeNameMap["HoverHeadingMode"] = HoverHeadingMode

class AutopilotAltitudeMode(IntEnum):
    """The altitude mode for the autopilot - vertical plane strategy of a basic maneuver procedure."""

    AUTOPILOT_HOLD_INIT_ALTITUDE = 0
    """The hold initial altitude mode."""
    AUTOPILOT_SPECIFY_ALTITUDE = 1
    """The specify altitude mode."""
    AUTOPILOT_SPECIFY_ALTITUDE_CHANGE = 2
    """The specify altitude change mode."""
    AUTOPILOT_SPECIFY_ALTITUDE_RATE = 3
    """The specify altitude rate mode."""
    AUTOPILOT_HOLD_INIT_ALTITUDE_RATE = 4
    """The hold initial altitude rate mode."""
    AUTOPILOT_SPECIFY_FLIGHT_PATH_ANGLE = 5
    """The specify wind frame flight path angle mode."""
    AUTOPILOT_HOLD_INIT_FLIGHT_PATH_ANGLE = 6
    """The hold initial wind frame flight path angle mode."""
    AUTOPILOT_BALLISTIC = 7
    """The ballistic flight path mode."""

AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_ALTITUDE.__doc__ = "The hold initial altitude mode."
AutopilotAltitudeMode.AUTOPILOT_SPECIFY_ALTITUDE.__doc__ = "The specify altitude mode."
AutopilotAltitudeMode.AUTOPILOT_SPECIFY_ALTITUDE_CHANGE.__doc__ = "The specify altitude change mode."
AutopilotAltitudeMode.AUTOPILOT_SPECIFY_ALTITUDE_RATE.__doc__ = "The specify altitude rate mode."
AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_ALTITUDE_RATE.__doc__ = "The hold initial altitude rate mode."
AutopilotAltitudeMode.AUTOPILOT_SPECIFY_FLIGHT_PATH_ANGLE.__doc__ = "The specify wind frame flight path angle mode."
AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_FLIGHT_PATH_ANGLE.__doc__ = "The hold initial wind frame flight path angle mode."
AutopilotAltitudeMode.AUTOPILOT_BALLISTIC.__doc__ = "The ballistic flight path mode."

agcls.AgTypeNameMap["AutopilotAltitudeMode"] = AutopilotAltitudeMode

class AutopilotAltitudeControlMode(IntEnum):
    """The altitude control mode for the autopilot - vertical plane strategy of a basic maneuver procedure."""

    AUTOPILOT_ALTITUDE_RATE = 0
    """The control altitude rate mode."""
    AUTOPILOT_FLIGHT_PATH_ANGLE = 1
    """The control flight path angle mode."""
    AUTOPILOT_PERFORMANCE_MODELS = 2
    """The climb/descent performance models mode."""

AutopilotAltitudeControlMode.AUTOPILOT_ALTITUDE_RATE.__doc__ = "The control altitude rate mode."
AutopilotAltitudeControlMode.AUTOPILOT_FLIGHT_PATH_ANGLE.__doc__ = "The control flight path angle mode."
AutopilotAltitudeControlMode.AUTOPILOT_PERFORMANCE_MODELS.__doc__ = "The climb/descent performance models mode."

agcls.AgTypeNameMap["AutopilotAltitudeControlMode"] = AutopilotAltitudeControlMode

class ClosureMode(IntEnum):
    """The closure mode for guidance strategies of the basic maneuver procedure."""

    CLOSURE_NOT_SET = 0
    """The closure is not set. The maneuver will continue whether or not the aircraft is closing with the target."""
    CLOSURE_REQUIRED = 1
    """The closure is required."""
    HOBS = 2
    """The high off boresight mode."""

ClosureMode.CLOSURE_NOT_SET.__doc__ = "The closure is not set. The maneuver will continue whether or not the aircraft is closing with the target."
ClosureMode.CLOSURE_REQUIRED.__doc__ = "The closure is required."
ClosureMode.HOBS.__doc__ = "The high off boresight mode."

agcls.AgTypeNameMap["ClosureMode"] = ClosureMode

class InterceptMode(IntEnum):
    """The intercept mode for the intercept strategy of the basic maneuver procedure."""

    TARGET_ASPECT = 0
    """The target aspect mode. The aircraft will maintain an approach angle with the target."""
    LATERAL_SEPARATION = 1
    """The lateral separation mode. The aircraft will guide to a specific distance from the target."""

InterceptMode.TARGET_ASPECT.__doc__ = "The target aspect mode. The aircraft will maintain an approach angle with the target."
InterceptMode.LATERAL_SEPARATION.__doc__ = "The lateral separation mode. The aircraft will guide to a specific distance from the target."

agcls.AgTypeNameMap["InterceptMode"] = InterceptMode

class RendezvousStopCondition(IntEnum):
    """The stop condition options for a rendezvous formation strategy of the basic maneuver procedure."""

    STOP_NORMAL = 0
    """The basic stopping conditions will be used."""
    STOP_AFTER_TARGET_CURRENT_PROCEDURE = 1
    """Stop after the target completes the current procedure."""
    STOP_AFTER_TARGET_CURRENT_PHASE = 2
    """Stop after the target completes the current phase."""
    STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES = 3
    """Stop when the target enters a new mode of flight."""
    STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES = 4
    """Stop when the target enters a new performance phase."""

RendezvousStopCondition.STOP_NORMAL.__doc__ = "The basic stopping conditions will be used."
RendezvousStopCondition.STOP_AFTER_TARGET_CURRENT_PROCEDURE.__doc__ = "Stop after the target completes the current procedure."
RendezvousStopCondition.STOP_AFTER_TARGET_CURRENT_PHASE.__doc__ = "Stop after the target completes the current phase."
RendezvousStopCondition.STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES.__doc__ = "Stop when the target enters a new mode of flight."
RendezvousStopCondition.STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES.__doc__ = "Stop when the target enters a new performance phase."

agcls.AgTypeNameMap["RendezvousStopCondition"] = RendezvousStopCondition

class FormationFlyerStopCondition(IntEnum):
    """The stop condition options for a Formation Flyer procedure."""

    FORMATION_FLYER_STOP_AFTER_FULL_MISSION = 0
    """Stop After FullMission."""
    FORMATION_FLYER_STOP_AFTER_TIME = 1
    """Stop After Time."""
    FORMATION_FLYER_STOP_AFTER_FUEL_STATE = 2
    """Stop After FuelState."""
    FORMATION_FLYER_STOP_AFTER_DOWN_RANGE = 3
    """Stop After DownRange."""
    FORMATION_FLYER_STOP_WHEN_TARGET_PROCEDURE_CHANGES = 4
    """Stop When TargetProcedure Changes."""
    FORMATION_FLYER_STOP_WHEN_TARGET_MISSION_CHANGES = 5
    """Stop When Target Mission Changes."""
    FORMATION_FLYER_STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES = 6
    """Stop When Target PhaseOfFlight Changes."""
    FORMATION_FLYER_STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES = 7
    """Stop When Target PerfMode Changes."""

FormationFlyerStopCondition.FORMATION_FLYER_STOP_AFTER_FULL_MISSION.__doc__ = "Stop After FullMission."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_AFTER_TIME.__doc__ = "Stop After Time."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_AFTER_FUEL_STATE.__doc__ = "Stop After FuelState."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_AFTER_DOWN_RANGE.__doc__ = "Stop After DownRange."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_WHEN_TARGET_PROCEDURE_CHANGES.__doc__ = "Stop When TargetProcedure Changes."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_WHEN_TARGET_MISSION_CHANGES.__doc__ = "Stop When Target Mission Changes."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES.__doc__ = "Stop When Target PhaseOfFlight Changes."
FormationFlyerStopCondition.FORMATION_FLYER_STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES.__doc__ = "Stop When Target PerfMode Changes."

agcls.AgTypeNameMap["FormationFlyerStopCondition"] = FormationFlyerStopCondition

class ExtEphemFlightMode(IntEnum):
    """Flight mode enums for ExtEphem."""

    EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_CLIMB = 0
    """ForwardFlightClimb."""
    EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_CRUISE = 1
    """ForwardFlightCruise."""
    EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_DESCEND = 2
    """ForwardFlightDescend."""
    EXT_EPHEM_FLIGHT_MODE_LANDING = 3
    """Landing."""
    EXT_EPHEM_FLIGHT_MODE_LANDING_WOW = 4
    """LandingWOW."""
    EXT_EPHEM_FLIGHT_MODE_TAKEOFF = 5
    """Takeoff."""
    EXT_EPHEM_FLIGHT_MODE_TAKEOFF_WOW = 6
    """TakeoffWOW."""
    EXT_EPHEM_FLIGHT_MODE_VTOL_HOVER = 7
    """VTOLHover ."""

ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_CLIMB.__doc__ = "ForwardFlightClimb."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_CRUISE.__doc__ = "ForwardFlightCruise."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_FORWARD_FLIGHT_DESCEND.__doc__ = "ForwardFlightDescend."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_LANDING.__doc__ = "Landing."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_LANDING_WOW.__doc__ = "LandingWOW."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_TAKEOFF.__doc__ = "Takeoff."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_TAKEOFF_WOW.__doc__ = "TakeoffWOW."
ExtEphemFlightMode.EXT_EPHEM_FLIGHT_MODE_VTOL_HOVER.__doc__ = "VTOLHover ."

agcls.AgTypeNameMap["ExtEphemFlightMode"] = ExtEphemFlightMode

class AccelerationPerformanceModelOverride(IntEnum):
    """The acceleration performance model override mode."""

    ACCELERATION_PERFORMANCE_MODEL_VALUE = 0
    """Use the acceleration performance model value."""
    ACCELERATION_OVERRIDE = 1
    """Override the performance model value."""
    ACCELERATION_NO_LIMIT = 2
    """Set no limit on the acceleration."""

AccelerationPerformanceModelOverride.ACCELERATION_PERFORMANCE_MODEL_VALUE.__doc__ = "Use the acceleration performance model value."
AccelerationPerformanceModelOverride.ACCELERATION_OVERRIDE.__doc__ = "Override the performance model value."
AccelerationPerformanceModelOverride.ACCELERATION_NO_LIMIT.__doc__ = "Set no limit on the acceleration."

agcls.AgTypeNameMap["AccelerationPerformanceModelOverride"] = AccelerationPerformanceModelOverride

class StationkeepingStopCondition(IntEnum):
    """The stop condition options for a stationkeeping strategy."""

    STOP_CONDITION_NOT_SET = 0
    """The basic stopping conditions will be used."""
    STOP_AFTER_TURN_COUNT = 1
    """Stop after a specified number of turns."""
    STOP_AFTER_DURATION = 2
    """Stop after a specified duration."""
    STOP_AFTER_TIME = 3
    """Stop at the specified time."""

StationkeepingStopCondition.STOP_CONDITION_NOT_SET.__doc__ = "The basic stopping conditions will be used."
StationkeepingStopCondition.STOP_AFTER_TURN_COUNT.__doc__ = "Stop after a specified number of turns."
StationkeepingStopCondition.STOP_AFTER_DURATION.__doc__ = "Stop after a specified duration."
StationkeepingStopCondition.STOP_AFTER_TIME.__doc__ = "Stop at the specified time."

agcls.AgTypeNameMap["StationkeepingStopCondition"] = StationkeepingStopCondition

class TurnDirection(IntEnum):
    """The roll direction for an aileron roll strategy for a basic maneuver procedure."""

    TURN_LEFT = 0
    """Turn left."""
    TURN_RIGHT = 2
    """Turn right."""

TurnDirection.TURN_LEFT.__doc__ = "Turn left."
TurnDirection.TURN_RIGHT.__doc__ = "Turn right."

agcls.AgTypeNameMap["TurnDirection"] = TurnDirection

class ProfileControlLimit(IntEnum):
    """Define the control limits for a profile strategy of a basic maneuver procedure."""

    PROFILE_ACCELERATION_PERFORMANCE_MODEL = 0
    """Use Accel Perf Model."""
    PROFILE_PITCH_RATE = 1
    """Specify the pitch rate."""

ProfileControlLimit.PROFILE_ACCELERATION_PERFORMANCE_MODEL.__doc__ = "Use Accel Perf Model."
ProfileControlLimit.PROFILE_PITCH_RATE.__doc__ = "Specify the pitch rate."

agcls.AgTypeNameMap["ProfileControlLimit"] = ProfileControlLimit

class RelativeSpeedAltitudeStopCondition(IntEnum):
    """The stop condition options for a relative speed/altitude strategy."""

    RELATIVE_SPEED_ALTITUDE_STOP_NORMAL = 0
    """The basic stopping conditions will be used."""
    RELATIVE_SPEED_ALTITUDE_STOP_MIN_RANGE_FOR_EQUAL_SPEED = 1
    """Stop when the aircraft achieves the range for equal speed."""
    RELATIVE_SPEED_ALTITUDE_STOP_TRANSITION_SPEED_RANGE = 2
    """Stop when the aircraft achieves the range to transition speed."""
    RELATIVE_SPEED_ALTITUDE_STOP_AFTER_TARGET_CURRENT_PROCEDURE = 3
    """Stop after the target completes the current procedure."""
    RELATIVE_SPEED_ALTITUDE_STOP_AFTER_TARGET_CURRENT_PHASE = 4
    """Stop after the target completes the current phase."""
    RELATIVE_SPEED_ALTITUDE_STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES = 5
    """Stop when the target enters a new mode of flight."""
    RELATIVE_SPEED_ALTITUDE_STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES = 6
    """Stop when the target enters a new performance phase."""

RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_NORMAL.__doc__ = "The basic stopping conditions will be used."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_MIN_RANGE_FOR_EQUAL_SPEED.__doc__ = "Stop when the aircraft achieves the range for equal speed."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_TRANSITION_SPEED_RANGE.__doc__ = "Stop when the aircraft achieves the range to transition speed."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_AFTER_TARGET_CURRENT_PROCEDURE.__doc__ = "Stop after the target completes the current procedure."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_AFTER_TARGET_CURRENT_PHASE.__doc__ = "Stop after the target completes the current phase."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_WHEN_TARGET_PERFORMANCE_MODE_CHANGES.__doc__ = "Stop when the target enters a new mode of flight."
RelativeSpeedAltitudeStopCondition.RELATIVE_SPEED_ALTITUDE_STOP_WHEN_TARGET_PHASE_OF_FLIGHT_CHANGES.__doc__ = "Stop when the target enters a new performance phase."

agcls.AgTypeNameMap["RelativeSpeedAltitudeStopCondition"] = RelativeSpeedAltitudeStopCondition

class RelativeAltitudeMode(IntEnum):
    """The relative altitude mode for a relative speed/altitude strategy."""

    HOLD_OFFSET_ALTITUDE = 0
    """Maintain the specified altitude offset from the target."""
    HOLD_INIT_ALTITUDE_OFFSET = 1
    """Maintain the altitude offset at the beginning of the maneuver."""
    HOLD_ELEVATION_ANGLE = 2
    """Maintain the specified elevation angle from the target."""
    HOLD_INIT_ELEVATION_ANGLE = 3
    """Maintain the elevation angle at the beginning of the maneuver."""

RelativeAltitudeMode.HOLD_OFFSET_ALTITUDE.__doc__ = "Maintain the specified altitude offset from the target."
RelativeAltitudeMode.HOLD_INIT_ALTITUDE_OFFSET.__doc__ = "Maintain the altitude offset at the beginning of the maneuver."
RelativeAltitudeMode.HOLD_ELEVATION_ANGLE.__doc__ = "Maintain the specified elevation angle from the target."
RelativeAltitudeMode.HOLD_INIT_ELEVATION_ANGLE.__doc__ = "Maintain the elevation angle at the beginning of the maneuver."

agcls.AgTypeNameMap["RelativeAltitudeMode"] = RelativeAltitudeMode

class FlyToFlightPathAngleMode(IntEnum):
    """The flight path angle mode mode for a bezier profile strategy."""

    FLY_TO_ALTITUDE_RATE = 0
    """Fly to a specified altitude rate."""
    FLY_TO_FLIGHT_PATH_ANGLE = 1
    """Fly to a specified flight path angle."""

FlyToFlightPathAngleMode.FLY_TO_ALTITUDE_RATE.__doc__ = "Fly to a specified altitude rate."
FlyToFlightPathAngleMode.FLY_TO_FLIGHT_PATH_ANGLE.__doc__ = "Fly to a specified flight path angle."

agcls.AgTypeNameMap["FlyToFlightPathAngleMode"] = FlyToFlightPathAngleMode

class PushPull(IntEnum):
    """The option to pull up or push over for a push/pull profile strategy."""

    PULL_UP = 0
    """Pull up."""
    PUSH_OVER = 1
    """Push over."""

PushPull.PULL_UP.__doc__ = "Pull up."
PushPull.PUSH_OVER.__doc__ = "Push over."

agcls.AgTypeNameMap["PushPull"] = PushPull

class AccelerationMode(IntEnum):
    """The acceleration/decelation option for a push/pull profile strategy."""

    ACCELERATION = 0
    """Accelerate at specified G."""
    DECELERATION = 1
    """Decelerate at specified G."""
    MAINTAIN_SPEED = 2
    """Maintain the initial airspeed."""

AccelerationMode.ACCELERATION.__doc__ = "Accelerate at specified G."
AccelerationMode.DECELERATION.__doc__ = "Decelerate at specified G."
AccelerationMode.MAINTAIN_SPEED.__doc__ = "Maintain the initial airspeed."

agcls.AgTypeNameMap["AccelerationMode"] = AccelerationMode

class DelayAltitudeMode(IntEnum):
    """The altitude options for a delay procedure."""

    DELAY_LEVEL_OFF = 0
    """Use a level off maneuver to fly to the default altitude."""
    DELAY_DEFAULT_CRUISE_ALTITUDE = 1
    """The default cruise altitude."""
    DELAY_OVERRIDE = 2
    """Override the default altitude."""

DelayAltitudeMode.DELAY_LEVEL_OFF.__doc__ = "Use a level off maneuver to fly to the default altitude."
DelayAltitudeMode.DELAY_DEFAULT_CRUISE_ALTITUDE.__doc__ = "The default cruise altitude."
DelayAltitudeMode.DELAY_OVERRIDE.__doc__ = "Override the default altitude."

agcls.AgTypeNameMap["DelayAltitudeMode"] = DelayAltitudeMode

class JoinExitArcMethod(IntEnum):
    """The options to join or exit an arc."""

    JOIN_EXIT_OUTBOUND = 0
    """The aircraft begins/ends the procedure on a course away from the site."""
    JOIN_EXIT_ON = 1
    """The aircraft begins/ends the procedure at the start bearing."""
    JOIN_EXIT_INBOUND = 2
    """The aircraft begins/ends on a course towards the site."""

JoinExitArcMethod.JOIN_EXIT_OUTBOUND.__doc__ = "The aircraft begins/ends the procedure on a course away from the site."
JoinExitArcMethod.JOIN_EXIT_ON.__doc__ = "The aircraft begins/ends the procedure at the start bearing."
JoinExitArcMethod.JOIN_EXIT_INBOUND.__doc__ = "The aircraft begins/ends on a course towards the site."

agcls.AgTypeNameMap["JoinExitArcMethod"] = JoinExitArcMethod

class FlightLineProcedureType(IntEnum):
    """The procedure methodology used to calculate the flight line."""

    PROCEDURE_TYPE_ENROUTE = 0
    """An enroute procedure type."""
    PROCEDURE_TYPE_BASIC_POINT_TO_POINT = 1
    """A basic point to point procedure type."""
    PROCEDURE_TYPE_TERRAIN_FOLLOW = 2
    """A terrain follow procedure type."""

FlightLineProcedureType.PROCEDURE_TYPE_ENROUTE.__doc__ = "An enroute procedure type."
FlightLineProcedureType.PROCEDURE_TYPE_BASIC_POINT_TO_POINT.__doc__ = "A basic point to point procedure type."
FlightLineProcedureType.PROCEDURE_TYPE_TERRAIN_FOLLOW.__doc__ = "A terrain follow procedure type."

agcls.AgTypeNameMap["FlightLineProcedureType"] = FlightLineProcedureType

class TransitionToHoverMode(IntEnum):
    """The type of hover to transition to."""

    FULL_STOP = 0
    """Transition to a full stop hover."""
    TRANSLATION_ONLY = 1
    """Transition to a hover with a constant translation rate."""
    TRANSLATION_AND_ALTITUDE = 2
    """Transition to a hover with a constant translation and altitude rate."""

TransitionToHoverMode.FULL_STOP.__doc__ = "Transition to a full stop hover."
TransitionToHoverMode.TRANSLATION_ONLY.__doc__ = "Transition to a hover with a constant translation rate."
TransitionToHoverMode.TRANSLATION_AND_ALTITUDE.__doc__ = "Transition to a hover with a constant translation and altitude rate."

agcls.AgTypeNameMap["TransitionToHoverMode"] = TransitionToHoverMode

class VTOLRateMode(IntEnum):
    """The rate mode for the VTOL procedure."""

    HALT_AUTOMATIC = 0
    """The rate during the procedure will be maintained."""
    ALWAYS_STOP = 1
    """The rate will be set to zero."""

VTOLRateMode.HALT_AUTOMATIC.__doc__ = "The rate during the procedure will be maintained."
VTOLRateMode.ALWAYS_STOP.__doc__ = "The rate will be set to zero."

agcls.AgTypeNameMap["VTOLRateMode"] = VTOLRateMode

class HoldingProfileMode(IntEnum):
    """How the aircraft will perform during the holding pattern with respect to airspeed and altitude."""

    STK8_COMPATIBLE = 0
    """The aircraft will fly the pattern at the speed at which it arrived at the entry point."""
    LEVEL_OFF_CRUISE_SPEED = 1
    """The aircraft will level off and fly at the cruise speed."""
    CLIMB_DESCENT_ON_STATION = 2
    """The aircraft will climb or descend to the specified altitude."""

HoldingProfileMode.STK8_COMPATIBLE.__doc__ = "The aircraft will fly the pattern at the speed at which it arrived at the entry point."
HoldingProfileMode.LEVEL_OFF_CRUISE_SPEED.__doc__ = "The aircraft will level off and fly at the cruise speed."
HoldingProfileMode.CLIMB_DESCENT_ON_STATION.__doc__ = "The aircraft will climb or descend to the specified altitude."

agcls.AgTypeNameMap["HoldingProfileMode"] = HoldingProfileMode

class HoldingDirection(IntEnum):
    """The turn direction for the aircraft to enter the holding pattern."""

    INBOUND_LEFT_TURN = 0
    """Turn left on the way inbound to the site."""
    INBOUND_RIGHT_TURN = 1
    """Turn right on the way inbound to the site."""
    OUTBOUND_LEFT_TURN = 2
    """Turn left on the way outbound from the site."""
    OUTBOUND_RIGHT_TURN = 3
    """Turn right on the way outbound from the site."""

HoldingDirection.INBOUND_LEFT_TURN.__doc__ = "Turn left on the way inbound to the site."
HoldingDirection.INBOUND_RIGHT_TURN.__doc__ = "Turn right on the way inbound to the site."
HoldingDirection.OUTBOUND_LEFT_TURN.__doc__ = "Turn left on the way outbound from the site."
HoldingDirection.OUTBOUND_RIGHT_TURN.__doc__ = "Turn right on the way outbound from the site."

agcls.AgTypeNameMap["HoldingDirection"] = HoldingDirection

class HoldRefuelDumpMode(IntEnum):
    """Define when the aircraft will leave the holding pattern after it has completed refueling or dumping fuel."""

    FULL_NUMER_OF_TURNS = 0
    """Turn left on the way inbound to the site."""
    EXIT_AT_END_OF_TURN = 1
    """Turn right on the way inbound to the site."""
    IMMEDIATE_EXIT = 2
    """Turn left on the way outbound from the site."""

HoldRefuelDumpMode.FULL_NUMER_OF_TURNS.__doc__ = "Turn left on the way inbound to the site."
HoldRefuelDumpMode.EXIT_AT_END_OF_TURN.__doc__ = "Turn right on the way inbound to the site."
HoldRefuelDumpMode.IMMEDIATE_EXIT.__doc__ = "Turn left on the way outbound from the site."

agcls.AgTypeNameMap["HoldRefuelDumpMode"] = HoldRefuelDumpMode

class HoldingEntryManeuver(IntEnum):
    """Define how the aircraft will enter the holding pattern."""

    HOLD_ENTRY_NO_MANEUVER = 0
    """The aircraft will enter the holding pattern at the normal holding point."""
    USE_STANDARD_ENTRY_TURNS = 1
    """The aircraft will enter the holding pattern using a standard entry maneuvers defined in the FAA Instrument Flying Handbook."""
    USE_ALTERNATE_ENTRY_POINTS = 2
    """The aircraft will enter the holding pattern at an alternate entry point."""

HoldingEntryManeuver.HOLD_ENTRY_NO_MANEUVER.__doc__ = "The aircraft will enter the holding pattern at the normal holding point."
HoldingEntryManeuver.USE_STANDARD_ENTRY_TURNS.__doc__ = "The aircraft will enter the holding pattern using a standard entry maneuvers defined in the FAA Instrument Flying Handbook."
HoldingEntryManeuver.USE_ALTERNATE_ENTRY_POINTS.__doc__ = "The aircraft will enter the holding pattern at an alternate entry point."

agcls.AgTypeNameMap["HoldingEntryManeuver"] = HoldingEntryManeuver

class VTOLTransitionMode(IntEnum):
    """The mode to specify the course of the transition maneuver."""

    TRANSITION_RELATIVE_HDG = 0
    """Specify a heading relative to the previous procedure."""
    TRANSITION_ABSOLUTE_HDG = 1
    """Specify an absolute heading."""
    TRANSITION_INTO_WIND = 2
    """Transition into the wind direction."""

VTOLTransitionMode.TRANSITION_RELATIVE_HDG.__doc__ = "Specify a heading relative to the previous procedure."
VTOLTransitionMode.TRANSITION_ABSOLUTE_HDG.__doc__ = "Specify an absolute heading."
VTOLTransitionMode.TRANSITION_INTO_WIND.__doc__ = "Transition into the wind direction."

agcls.AgTypeNameMap["VTOLTransitionMode"] = VTOLTransitionMode

class VTOLFinalHeadingMode(IntEnum):
    """The mode to specify the heading at the end of the maneuver."""

    FINAL_HEADING_RELATIVE = 0
    """Specify a heading relative to the previous procedure."""
    FINAL_HEADING_ABSOLUTE = 1
    """Specify an absolute heading."""
    FINAL_HEADING_TRANSLATION_COURSE = 2
    """The aircraft have the same heading as the translation bearing."""

VTOLFinalHeadingMode.FINAL_HEADING_RELATIVE.__doc__ = "Specify a heading relative to the previous procedure."
VTOLFinalHeadingMode.FINAL_HEADING_ABSOLUTE.__doc__ = "Specify an absolute heading."
VTOLFinalHeadingMode.FINAL_HEADING_TRANSLATION_COURSE.__doc__ = "The aircraft have the same heading as the translation bearing."

agcls.AgTypeNameMap["VTOLFinalHeadingMode"] = VTOLFinalHeadingMode

class VTOLTranslationMode(IntEnum):
    """The mode to specify the translation of the VTOL maneuver."""

    SET_BEARING_AND_RANGE = 0
    """The aircraft will translate on a specific bearing and range."""
    COME_TO_STOP = 1
    """The aircraft will come to a complete stop."""
    MAINTAIN_RATE = 2
    """The aircraft will continue to translate with the same rate and direction as it began the procedure."""

VTOLTranslationMode.SET_BEARING_AND_RANGE.__doc__ = "The aircraft will translate on a specific bearing and range."
VTOLTranslationMode.COME_TO_STOP.__doc__ = "The aircraft will come to a complete stop."
VTOLTranslationMode.MAINTAIN_RATE.__doc__ = "The aircraft will continue to translate with the same rate and direction as it began the procedure."

agcls.AgTypeNameMap["VTOLTranslationMode"] = VTOLTranslationMode

class VTOLTranslationFinalCourseMode(IntEnum):
    """The mode to specify the final course of the VTOL maneuver."""

    TRANSLATE_DIRECT = 0
    """The aircraft will translate directly along the specified bearing and range."""
    BISECT_INBOUND_OUTBOUND = 1
    """The aircraft will translate along a bisecting line between the inbound and outbound course."""
    ANTICIPATE_NEXT_TRANSLATION = 2
    """The aircraft will evaluate the procedure ahead to determine the translation bearing and rate."""

VTOLTranslationFinalCourseMode.TRANSLATE_DIRECT.__doc__ = "The aircraft will translate directly along the specified bearing and range."
VTOLTranslationFinalCourseMode.BISECT_INBOUND_OUTBOUND.__doc__ = "The aircraft will translate along a bisecting line between the inbound and outbound course."
VTOLTranslationFinalCourseMode.ANTICIPATE_NEXT_TRANSLATION.__doc__ = "The aircraft will evaluate the procedure ahead to determine the translation bearing and rate."

agcls.AgTypeNameMap["VTOLTranslationFinalCourseMode"] = VTOLTranslationFinalCourseMode

class HoverMode(IntEnum):
    """The hover mode."""

    HOVER_MODE_FIXED_TIME = 0
    """Hover in place for a fixed time."""
    HOVER_MODE_MANEUVER = 1
    """Perform a hovering maneuver."""

HoverMode.HOVER_MODE_FIXED_TIME.__doc__ = "Hover in place for a fixed time."
HoverMode.HOVER_MODE_MANEUVER.__doc__ = "Perform a hovering maneuver."

agcls.AgTypeNameMap["HoverMode"] = HoverMode

class VTOLHeadingMode(IntEnum):
    """The heading mode for the hover maneuver."""

    HEADING_INDEPENDENT = 0
    """The aircraft's heading is independent of its translation."""
    HEADING_ALIGN_TRANSLATION_COURSE = 1
    """The aircraft will align its heading with the translation course."""
    HEADING_INTO_WIND = 2
    """The aircraft will set its heading into the wind."""

VTOLHeadingMode.HEADING_INDEPENDENT.__doc__ = "The aircraft's heading is independent of its translation."
VTOLHeadingMode.HEADING_ALIGN_TRANSLATION_COURSE.__doc__ = "The aircraft will align its heading with the translation course."
VTOLHeadingMode.HEADING_INTO_WIND.__doc__ = "The aircraft will set its heading into the wind."

agcls.AgTypeNameMap["VTOLHeadingMode"] = VTOLHeadingMode

class VertLandingMode(IntEnum):
    """The heading mode for a vertical landing maneuver."""

    VERT_LANDING_INDEPENDENT = 0
    """The aircraft's heading is independent of its translation."""
    VERT_LANDING_ALIGN_TRANSLATION_COURSE = 1
    """The aircraft will align its heading with the translation course."""
    VERT_LANDING_INTO_WIND = 2
    """The aircraft will set its heading into the wind."""
    VERT_LANDING_ALIGN_TRANSLATION_COURSE_OVERRIDE = 3
    """The aircraft will align its heading with the translation course  will achieve the specified heading upon arriving."""

VertLandingMode.VERT_LANDING_INDEPENDENT.__doc__ = "The aircraft's heading is independent of its translation."
VertLandingMode.VERT_LANDING_ALIGN_TRANSLATION_COURSE.__doc__ = "The aircraft will align its heading with the translation course."
VertLandingMode.VERT_LANDING_INTO_WIND.__doc__ = "The aircraft will set its heading into the wind."
VertLandingMode.VERT_LANDING_ALIGN_TRANSLATION_COURSE_OVERRIDE.__doc__ = "The aircraft will align its heading with the translation course  will achieve the specified heading upon arriving."

agcls.AgTypeNameMap["VertLandingMode"] = VertLandingMode

class LaunchAttitudeMode(IntEnum):
    """The attitude mode for the launch procedure."""

    LAUNCH_ALIGN_DIRECTION_VECTOR = 0
    """The aircraft's attitude is aligned with the specified direction vector."""
    LAUNCH_HOLD_PARENT_ATTITUDE = 1
    """The aircraft will maintain the parent vehicle's attiude at the time of launch."""
    LAUNCH_VTOL = 2
    """The aircraft launches in level flight."""

LaunchAttitudeMode.LAUNCH_ALIGN_DIRECTION_VECTOR.__doc__ = "The aircraft's attitude is aligned with the specified direction vector."
LaunchAttitudeMode.LAUNCH_HOLD_PARENT_ATTITUDE.__doc__ = "The aircraft will maintain the parent vehicle's attiude at the time of launch."
LaunchAttitudeMode.LAUNCH_VTOL.__doc__ = "The aircraft launches in level flight."

agcls.AgTypeNameMap["LaunchAttitudeMode"] = LaunchAttitudeMode

class FuelFlowType(IntEnum):
    """The fuel flow type to use for the procedure."""

    FUEL_FLOW_TAKEOFF = 0
    """Fuel flow defined for the current Takeoff performance model."""
    FUEL_FLOW_CRUISE = 1
    """Fuel flow defined for the current Cruise performance model."""
    FUEL_FLOW_LANDING = 2
    """Fuel flow defined for the current Landing performance model."""
    FUEL_FLOW_VTOL = 3
    """Fuel flow defined for the current VTOL performance model."""
    FUEL_FLOW_AERODYNAMIC_PROPULSION = 4
    """Fuel flow defined for the Aerodynamics and Propulsion Analysis component of the current basic acceleration performance model."""
    FUEL_FLOW_OVERRIDE = 5
    """Fuel flow defined manually."""

FuelFlowType.FUEL_FLOW_TAKEOFF.__doc__ = "Fuel flow defined for the current Takeoff performance model."
FuelFlowType.FUEL_FLOW_CRUISE.__doc__ = "Fuel flow defined for the current Cruise performance model."
FuelFlowType.FUEL_FLOW_LANDING.__doc__ = "Fuel flow defined for the current Landing performance model."
FuelFlowType.FUEL_FLOW_VTOL.__doc__ = "Fuel flow defined for the current VTOL performance model."
FuelFlowType.FUEL_FLOW_AERODYNAMIC_PROPULSION.__doc__ = "Fuel flow defined for the Aerodynamics and Propulsion Analysis component of the current basic acceleration performance model."
FuelFlowType.FUEL_FLOW_OVERRIDE.__doc__ = "Fuel flow defined manually."

agcls.AgTypeNameMap["FuelFlowType"] = FuelFlowType

class LineOrientation(IntEnum):
    """The orientation for a parallel flight line procedure."""

    FLIGHT_LINE_TO_LEFT = 0
    """Left of the previous flight line."""
    FLIGHT_LINE_TO_RIGHT = 1
    """Right of the previous flight line."""

LineOrientation.FLIGHT_LINE_TO_LEFT.__doc__ = "Left of the previous flight line."
LineOrientation.FLIGHT_LINE_TO_RIGHT.__doc__ = "Right of the previous flight line."

agcls.AgTypeNameMap["LineOrientation"] = LineOrientation

class RelativeAbsoluteBearing(IntEnum):
    """The options for a bearing that can be relative or absolute."""

    RELATIVE_BEARING = 0
    """A bearing relative to the previous course."""
    TRUE_BEARING = 1
    """A bearing using true north."""
    MAGNETIC_BEARING = 2
    """A bearing using magnetic north."""

RelativeAbsoluteBearing.RELATIVE_BEARING.__doc__ = "A bearing relative to the previous course."
RelativeAbsoluteBearing.TRUE_BEARING.__doc__ = "A bearing using true north."
RelativeAbsoluteBearing.MAGNETIC_BEARING.__doc__ = "A bearing using magnetic north."

agcls.AgTypeNameMap["RelativeAbsoluteBearing"] = RelativeAbsoluteBearing

class BasicFixedWingPropulsionMode(IntEnum):
    """The option to specify the thrust (jet engines) or power (propellers)."""

    SPECIFY_THRUST = 0
    """Jet - Specify net thrust."""
    SPECIFY_POWER = 1
    """Propeller - Specify net power."""

BasicFixedWingPropulsionMode.SPECIFY_THRUST.__doc__ = "Jet - Specify net thrust."
BasicFixedWingPropulsionMode.SPECIFY_POWER.__doc__ = "Propeller - Specify net power."

agcls.AgTypeNameMap["BasicFixedWingPropulsionMode"] = BasicFixedWingPropulsionMode

class ClimbSpeedType(IntEnum):
    """The mode to calculate the aircraft's airspeed while climbing for an advanced climb performance model."""

    CLIMB_SPEED_BEST_RATE = 0
    """The speed at which the aircraft's rate of climb is maximized."""
    CLIMB_SPEED_BEST_ANGLE = 1
    """The speed at which the aircraft will gain the most altitude over downrange distance."""
    CLIMB_SPEED_MIN_FUEL = 2
    """The speed at which the aircraft's fuel consumption is minimized."""
    CLIMB_SPEED_OVERRIDE = 3
    """Manually specify the speed."""

ClimbSpeedType.CLIMB_SPEED_BEST_RATE.__doc__ = "The speed at which the aircraft's rate of climb is maximized."
ClimbSpeedType.CLIMB_SPEED_BEST_ANGLE.__doc__ = "The speed at which the aircraft will gain the most altitude over downrange distance."
ClimbSpeedType.CLIMB_SPEED_MIN_FUEL.__doc__ = "The speed at which the aircraft's fuel consumption is minimized."
ClimbSpeedType.CLIMB_SPEED_OVERRIDE.__doc__ = "Manually specify the speed."

agcls.AgTypeNameMap["ClimbSpeedType"] = ClimbSpeedType

class CruiseMaxPerformanceSpeedType(IntEnum):
    """The method for defining the maximum performance airspeed of the aircraft for an advanced cruise model."""

    CORNER_SPEED = 0
    """The lowest speed at which the aircraft can generate lift while pulling maximum Gs."""
    MAX_PS_DRY_THRUST = 1
    """The maximum specific excess power without afterburning."""
    MAX_PS_AFTERBURNER = 2
    """The maximum specific excess power while afterburning."""
    MAX_SPEED_DRY_THRUST = 3
    """The maximum speed without afterburning."""
    MAX_RANGE_AFTERBURNER = 4
    """The speed which maximizes the aircraft range while afterburning."""

CruiseMaxPerformanceSpeedType.CORNER_SPEED.__doc__ = "The lowest speed at which the aircraft can generate lift while pulling maximum Gs."
CruiseMaxPerformanceSpeedType.MAX_PS_DRY_THRUST.__doc__ = "The maximum specific excess power without afterburning."
CruiseMaxPerformanceSpeedType.MAX_PS_AFTERBURNER.__doc__ = "The maximum specific excess power while afterburning."
CruiseMaxPerformanceSpeedType.MAX_SPEED_DRY_THRUST.__doc__ = "The maximum speed without afterburning."
CruiseMaxPerformanceSpeedType.MAX_RANGE_AFTERBURNER.__doc__ = "The speed which maximizes the aircraft range while afterburning."

agcls.AgTypeNameMap["CruiseMaxPerformanceSpeedType"] = CruiseMaxPerformanceSpeedType

class DescentSpeedType(IntEnum):
    """The method for calculating the aircraft's airspeed while descending."""

    DESCENT_MAX_RANGE_CRUISE = 0
    """A variable airspeed that maximizes the distance that the aircraft can fly."""
    DESCENT_MAX_GLIDE_RATIO = 1
    """The speed at which the aircraft's glide ratio is maximized."""
    DESCENT_MIN_SINK_RATE = 2
    """The speed at which the aircraft's rate of descent is minimized."""
    DESCENT_STALL_SPEED_RATIO = 3
    """A speed relative to the aircraft's stall speed.."""
    DESCENT_SPEED_OVERRIDE = 4
    """Manually define the airspeed."""

DescentSpeedType.DESCENT_MAX_RANGE_CRUISE.__doc__ = "A variable airspeed that maximizes the distance that the aircraft can fly."
DescentSpeedType.DESCENT_MAX_GLIDE_RATIO.__doc__ = "The speed at which the aircraft's glide ratio is maximized."
DescentSpeedType.DESCENT_MIN_SINK_RATE.__doc__ = "The speed at which the aircraft's rate of descent is minimized."
DescentSpeedType.DESCENT_STALL_SPEED_RATIO.__doc__ = "A speed relative to the aircraft's stall speed.."
DescentSpeedType.DESCENT_SPEED_OVERRIDE.__doc__ = "Manually define the airspeed."

agcls.AgTypeNameMap["DescentSpeedType"] = DescentSpeedType

class TakeoffLandingSpeedMode(IntEnum):
    """The method for calculating the aircraft's speed upon leaving the ground or at wheels down."""

    TAKEOFF_LANDING_STALL_SPEED_RATIO = 0
    """A multiple of the aircraft's stall speed."""
    TAKEOFF_LANDING_ANGLE_OF_ATTACK = 1
    """The speed needed to achieve lift at the specified Angle of Attack."""

TakeoffLandingSpeedMode.TAKEOFF_LANDING_STALL_SPEED_RATIO.__doc__ = "A multiple of the aircraft's stall speed."
TakeoffLandingSpeedMode.TAKEOFF_LANDING_ANGLE_OF_ATTACK.__doc__ = "The speed needed to achieve lift at the specified Angle of Attack."

agcls.AgTypeNameMap["TakeoffLandingSpeedMode"] = TakeoffLandingSpeedMode

class DepartureSpeedMode(IntEnum):
    """The method for calculating the aircraft's airspeed upon leaving the ground."""

    MAX_CLIMB_ANGLE = 0
    """The aircraft will fly at the max climb angle."""
    MAX_CLIMB_RATE = 1
    """The aircraft will fly at the max climb rate."""
    USE_CLIMB_MODEL = 2
    """The aircraft will fly at the airspeed defined in the Climb performance model."""

DepartureSpeedMode.MAX_CLIMB_ANGLE.__doc__ = "The aircraft will fly at the max climb angle."
DepartureSpeedMode.MAX_CLIMB_RATE.__doc__ = "The aircraft will fly at the max climb rate."
DepartureSpeedMode.USE_CLIMB_MODEL.__doc__ = "The aircraft will fly at the airspeed defined in the Climb performance model."

agcls.AgTypeNameMap["DepartureSpeedMode"] = DepartureSpeedMode

class AdvancedFixedWingAerodynamicStrategy(IntEnum):
    """The aerodynamic strategy for the Advanced Fixed Wing Tool."""

    EXTERNAL_AERODYNAMIC_FILE = 0
    """Define the aerodynamics using an external .aero file."""
    SUB_SUPER_HYPER_AERODYNAMIC = 1
    """Define the aerodynamics using a model derived from first principles that is valid for the full speed range of high speed aircraft."""
    SUBSONIC_AERODYNAMIC = 2
    """Define the aerodynamics for an aircraft that generally travels at subsonic speeds."""
    SUPERSONIC_AERODYNAMIC = 3
    """Define the aerodynamics for an aircraft that generally travels at supersonic speeds."""
    FOUR_POINT_AERODYNAMIC = 4
    """Define the aerodynamics for an aircraft defined at 4 different points in the flight envelope."""

AdvancedFixedWingAerodynamicStrategy.EXTERNAL_AERODYNAMIC_FILE.__doc__ = "Define the aerodynamics using an external .aero file."
AdvancedFixedWingAerodynamicStrategy.SUB_SUPER_HYPER_AERODYNAMIC.__doc__ = "Define the aerodynamics using a model derived from first principles that is valid for the full speed range of high speed aircraft."
AdvancedFixedWingAerodynamicStrategy.SUBSONIC_AERODYNAMIC.__doc__ = "Define the aerodynamics for an aircraft that generally travels at subsonic speeds."
AdvancedFixedWingAerodynamicStrategy.SUPERSONIC_AERODYNAMIC.__doc__ = "Define the aerodynamics for an aircraft that generally travels at supersonic speeds."
AdvancedFixedWingAerodynamicStrategy.FOUR_POINT_AERODYNAMIC.__doc__ = "Define the aerodynamics for an aircraft defined at 4 different points in the flight envelope."

agcls.AgTypeNameMap["AdvancedFixedWingAerodynamicStrategy"] = AdvancedFixedWingAerodynamicStrategy

class AdvancedFixedWingGeometry(IntEnum):
    """The method to define the wing geometry of an aircraft in the Advanced Fixed Wing Tool."""

    BASIC_GEOMETRY = 0
    """Define the static properties of an immovable wing."""
    VARIABLE_GEOMETRY = 1
    """Define the minimum and maximum properties of an adjustable wing."""

AdvancedFixedWingGeometry.BASIC_GEOMETRY.__doc__ = "Define the static properties of an immovable wing."
AdvancedFixedWingGeometry.VARIABLE_GEOMETRY.__doc__ = "Define the minimum and maximum properties of an adjustable wing."

agcls.AgTypeNameMap["AdvancedFixedWingGeometry"] = AdvancedFixedWingGeometry

class AdvancedFixedWingPowerplantStrategy(IntEnum):
    """The powerplant strategy for the Advanced Fixed Wing Tool."""

    ELECTRIC_POWERPLANT = 0
    """An electric engine."""
    EXTERNAL_PROPULSION_FILE = 1
    """Define the powerplant using an external .prop file."""
    PISTON_POWERPLANT = 2
    """A piston, or reciprocating, engine."""
    SUB_SUPER_HYPER_POWERPLANT = 3
    """A thermodynamic model that includes turbine, ramjet, and scramjet performance modes."""
    TURBOFAN_BASIC_AB = 4
    """A thermodynamic model of a dual-spool turbofan engine that has an afterburner."""
    TURBOFAN_HIGH_BYPASS = 5
    """An empirical model for a turbofan engine that produces significantly more fan thrust than jet thrust."""
    TURBOFAN_LOW_BYPASS = 6
    """An empirical model for a turbofan engine that produces significantly more jet thrust than fan thrust."""
    TURBOFAN_LOW_BYPASS_AFTERBURNING = 7
    """An empirical model for a turbofan engine that has an afterburner and produces significantly more jet thrust than fan thrust."""
    TURBOJET_AFTERBURNING = 8
    """An empirical model that models a turbojet engine that has an afterburner."""
    TURBOJET_BASIC_AB = 9
    """An empirical model that models a turbojet engine."""
    TURBOJET = 10
    """An empirical model that models a turbojet engine."""
    TURBOPROP = 11
    """An empirical model that models a turboprop engine."""

AdvancedFixedWingPowerplantStrategy.ELECTRIC_POWERPLANT.__doc__ = "An electric engine."
AdvancedFixedWingPowerplantStrategy.EXTERNAL_PROPULSION_FILE.__doc__ = "Define the powerplant using an external .prop file."
AdvancedFixedWingPowerplantStrategy.PISTON_POWERPLANT.__doc__ = "A piston, or reciprocating, engine."
AdvancedFixedWingPowerplantStrategy.SUB_SUPER_HYPER_POWERPLANT.__doc__ = "A thermodynamic model that includes turbine, ramjet, and scramjet performance modes."
AdvancedFixedWingPowerplantStrategy.TURBOFAN_BASIC_AB.__doc__ = "A thermodynamic model of a dual-spool turbofan engine that has an afterburner."
AdvancedFixedWingPowerplantStrategy.TURBOFAN_HIGH_BYPASS.__doc__ = "An empirical model for a turbofan engine that produces significantly more fan thrust than jet thrust."
AdvancedFixedWingPowerplantStrategy.TURBOFAN_LOW_BYPASS.__doc__ = "An empirical model for a turbofan engine that produces significantly more jet thrust than fan thrust."
AdvancedFixedWingPowerplantStrategy.TURBOFAN_LOW_BYPASS_AFTERBURNING.__doc__ = "An empirical model for a turbofan engine that has an afterburner and produces significantly more jet thrust than fan thrust."
AdvancedFixedWingPowerplantStrategy.TURBOJET_AFTERBURNING.__doc__ = "An empirical model that models a turbojet engine that has an afterburner."
AdvancedFixedWingPowerplantStrategy.TURBOJET_BASIC_AB.__doc__ = "An empirical model that models a turbojet engine."
AdvancedFixedWingPowerplantStrategy.TURBOJET.__doc__ = "An empirical model that models a turbojet engine."
AdvancedFixedWingPowerplantStrategy.TURBOPROP.__doc__ = "An empirical model that models a turboprop engine."

agcls.AgTypeNameMap["AdvancedFixedWingPowerplantStrategy"] = AdvancedFixedWingPowerplantStrategy

class MissileAerodynamicStrategy(IntEnum):
    """The aerodynamic strategy used to compute lift, drag, angle of attack, sideslip and intermediate / derived values."""

    MISSILE_AERODYNAMIC_SIMPLE = 0
    """Simple aerodynamics."""
    MISSILE_AERODYNAMIC_EXTERNAL_FILE = 1
    """External file aerodynamics."""
    MISSILE_AERODYNAMIC_ADVANCED = 2
    """Advanced missile aerodynamics."""
    MISSILE_AERODYNAMIC_FOUR_POINT = 3
    """FourPoint missile aerodynamics."""

MissileAerodynamicStrategy.MISSILE_AERODYNAMIC_SIMPLE.__doc__ = "Simple aerodynamics."
MissileAerodynamicStrategy.MISSILE_AERODYNAMIC_EXTERNAL_FILE.__doc__ = "External file aerodynamics."
MissileAerodynamicStrategy.MISSILE_AERODYNAMIC_ADVANCED.__doc__ = "Advanced missile aerodynamics."
MissileAerodynamicStrategy.MISSILE_AERODYNAMIC_FOUR_POINT.__doc__ = "FourPoint missile aerodynamics."

agcls.AgTypeNameMap["MissileAerodynamicStrategy"] = MissileAerodynamicStrategy

class MissilePropulsionStrategy(IntEnum):
    """The propulsion strategy used to compute thrust and throttle setting."""

    MISSILE_PROPULSION_SIMPLE = 0
    """Simple propulsion."""
    MISSILE_PROPULSION_EXTERNAL_FILE = 1
    """External file propulsion."""
    MISSILE_PROPULSION_RAMJET = 2
    """Ramjet propulsion."""
    MISSILE_PROPULSION_ROCKET = 3
    """Rocket propulsion."""
    MISSILE_PROPULSION_TURBOJET = 4
    """Turbojet propulsion."""

MissilePropulsionStrategy.MISSILE_PROPULSION_SIMPLE.__doc__ = "Simple propulsion."
MissilePropulsionStrategy.MISSILE_PROPULSION_EXTERNAL_FILE.__doc__ = "External file propulsion."
MissilePropulsionStrategy.MISSILE_PROPULSION_RAMJET.__doc__ = "Ramjet propulsion."
MissilePropulsionStrategy.MISSILE_PROPULSION_ROCKET.__doc__ = "Rocket propulsion."
MissilePropulsionStrategy.MISSILE_PROPULSION_TURBOJET.__doc__ = "Turbojet propulsion."

agcls.AgTypeNameMap["MissilePropulsionStrategy"] = MissilePropulsionStrategy

class RotorcraftPowerplantType(IntEnum):
    """The powerplant type for a rotorcraft."""

    ROTORCRAFT_ELECTRIC = 0
    """Electric propulsion."""
    ROTORCRAFT_TURBOSHAFT = 1
    """Turboshaft propulsion."""
    ROTORCRAFT_PISTON = 2
    """Piston propulsion."""

RotorcraftPowerplantType.ROTORCRAFT_ELECTRIC.__doc__ = "Electric propulsion."
RotorcraftPowerplantType.ROTORCRAFT_TURBOSHAFT.__doc__ = "Turboshaft propulsion."
RotorcraftPowerplantType.ROTORCRAFT_PISTON.__doc__ = "Piston propulsion."

agcls.AgTypeNameMap["RotorcraftPowerplantType"] = RotorcraftPowerplantType

class MinimizeSiteProcedureTimeDiff(IntEnum):
    """Options for minimizing the time difference between the procedure and site times."""

    MINIMIZE_TIME_DIFFERENCE_OFF = 0
    """Do not converge the procedure and site times."""
    MINIMIZE_TIME_DIFFERENCE_ALWAYS = 1
    """Always recalculate."""
    MINIMIZE_TIME_DIFFERENCE_NEXT_UPDATE = 2
    """Only recalculate when updated."""

MinimizeSiteProcedureTimeDiff.MINIMIZE_TIME_DIFFERENCE_OFF.__doc__ = "Do not converge the procedure and site times."
MinimizeSiteProcedureTimeDiff.MINIMIZE_TIME_DIFFERENCE_ALWAYS.__doc__ = "Always recalculate."
MinimizeSiteProcedureTimeDiff.MINIMIZE_TIME_DIFFERENCE_NEXT_UPDATE.__doc__ = "Only recalculate when updated."

agcls.AgTypeNameMap["MinimizeSiteProcedureTimeDiff"] = MinimizeSiteProcedureTimeDiff

class STKObjectWaypointOffsetMode(IntEnum):
    """The options to offset the site location relative to the STK Object."""

    OFFSET_NONE = 0
    """No offset."""
    OFFSET_BEARING_RANGE = 1
    """Bearing/Range (relative to North)."""
    OFFSET_VGT_POINT = 2
    """VGT Point."""
    OFFSET_RELATIVE_BEARING_RANGE = 3
    """Bearing/Reange relative to Vehicle course."""

STKObjectWaypointOffsetMode.OFFSET_NONE.__doc__ = "No offset."
STKObjectWaypointOffsetMode.OFFSET_BEARING_RANGE.__doc__ = "Bearing/Range (relative to North)."
STKObjectWaypointOffsetMode.OFFSET_VGT_POINT.__doc__ = "VGT Point."
STKObjectWaypointOffsetMode.OFFSET_RELATIVE_BEARING_RANGE.__doc__ = "Bearing/Reange relative to Vehicle course."

agcls.AgTypeNameMap["STKObjectWaypointOffsetMode"] = STKObjectWaypointOffsetMode

class SearchPatternCourseMode(IntEnum):
    """The mode to determine the course of the search pattern."""

    COURSE_MODE_LOW = 0
    """Automatically begin the search pattern within 0 - 180 degrees."""
    COURSE_MODE_HIGH = 1
    """Automatically begin the search pattern within 180 - 360 degrees."""
    COURSE_MODE_OVERRIDE = 2
    """Manually define the course of the search pattern."""

SearchPatternCourseMode.COURSE_MODE_LOW.__doc__ = "Automatically begin the search pattern within 0 - 180 degrees."
SearchPatternCourseMode.COURSE_MODE_HIGH.__doc__ = "Automatically begin the search pattern within 180 - 360 degrees."
SearchPatternCourseMode.COURSE_MODE_OVERRIDE.__doc__ = "Manually define the course of the search pattern."

agcls.AgTypeNameMap["SearchPatternCourseMode"] = SearchPatternCourseMode

class DelayTurnDirection(IntEnum):
    """Turn mode for procedures with Delay options."""

    DELAY_TURN_AUTO = 0
    """Automatic turn. Aviator will determine the direction of the turn."""
    DELAY_TURN_LEFT = 1
    """Left turn."""
    DELAY_TURN_RIGHT = 2
    """Right turn."""

DelayTurnDirection.DELAY_TURN_AUTO.__doc__ = "Automatic turn. Aviator will determine the direction of the turn."
DelayTurnDirection.DELAY_TURN_LEFT.__doc__ = "Left turn."
DelayTurnDirection.DELAY_TURN_RIGHT.__doc__ = "Right turn."

agcls.AgTypeNameMap["DelayTurnDirection"] = DelayTurnDirection

class TrajectoryBlendMode(IntEnum):
    """The interpolation mode to determine the aircraft's position and velocity."""

    BLEND_BODY_QUADRATIC = 0
    """Quadratic interpolation using the body frame."""
    BLEND_BODY_CUBIC = 1
    """Cubic interpolation using the body frame."""
    BLEND_LH_QUADRATIC = 2
    """Quadratic interpolation using the local horizontal frame."""
    BLEND_LH_CUBIC = 3
    """Cubic interpolation using the local horizontal frame."""
    BLEND_ECF_QUADRATIC = 4
    """Quadratic interpolation using the Earth-Centered Fixed frame."""
    BLEND_ECF_CUBIC = 5
    """Cubic interpolation using the Earth-Centered Fixed frame."""

TrajectoryBlendMode.BLEND_BODY_QUADRATIC.__doc__ = "Quadratic interpolation using the body frame."
TrajectoryBlendMode.BLEND_BODY_CUBIC.__doc__ = "Cubic interpolation using the body frame."
TrajectoryBlendMode.BLEND_LH_QUADRATIC.__doc__ = "Quadratic interpolation using the local horizontal frame."
TrajectoryBlendMode.BLEND_LH_CUBIC.__doc__ = "Cubic interpolation using the local horizontal frame."
TrajectoryBlendMode.BLEND_ECF_QUADRATIC.__doc__ = "Quadratic interpolation using the Earth-Centered Fixed frame."
TrajectoryBlendMode.BLEND_ECF_CUBIC.__doc__ = "Cubic interpolation using the Earth-Centered Fixed frame."

agcls.AgTypeNameMap["TrajectoryBlendMode"] = TrajectoryBlendMode

class ReferenceStatePerformanceMode(IntEnum):
    """The type of motion the aircraft is engaged in at the reference state."""

    REFERENCE_STATE_CLIMB = 0
    """Forward Flight - Climb performance mode."""
    REFERENCE_STATE_CRUISE = 1
    """Forward Flight - Cruise performance mode."""
    REFERENCE_STATE_DESCEND = 2
    """Forward Flight - Descend performance mode."""
    REFERENCE_STATE_HOVER = 3
    """Hover performance mode."""
    REFERENCE_STATE_LANDING = 4
    """Landing performance mode."""
    REFERENCE_STATE_TAKEOFF = 5
    """Takeoff performance mode."""
    REFERENCE_STATE_LANDING_ROLLOUT = 6
    """Weight On Wheels - Landing Rollout performance mode."""
    REFERENCE_STATE_TAKEOFF_RUN = 7
    """Weight On Wheels - Takeoff Run performance mode."""

ReferenceStatePerformanceMode.REFERENCE_STATE_CLIMB.__doc__ = "Forward Flight - Climb performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_CRUISE.__doc__ = "Forward Flight - Cruise performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_DESCEND.__doc__ = "Forward Flight - Descend performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_HOVER.__doc__ = "Hover performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_LANDING.__doc__ = "Landing performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_TAKEOFF.__doc__ = "Takeoff performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_LANDING_ROLLOUT.__doc__ = "Weight On Wheels - Landing Rollout performance mode."
ReferenceStatePerformanceMode.REFERENCE_STATE_TAKEOFF_RUN.__doc__ = "Weight On Wheels - Takeoff Run performance mode."

agcls.AgTypeNameMap["ReferenceStatePerformanceMode"] = ReferenceStatePerformanceMode

class ReferenceStateLongitudinalAccelerationMode(IntEnum):
    """The mode to specify the longitudinal acceleration of the aircraft."""

    SPECIFY_TAS_DOT = 0
    """Specify the true airspeed acceleration."""
    SPECIFY_GROUND_SPEED_DOT = 1
    """Specify the groundspeed acceleration."""

ReferenceStateLongitudinalAccelerationMode.SPECIFY_TAS_DOT.__doc__ = "Specify the true airspeed acceleration."
ReferenceStateLongitudinalAccelerationMode.SPECIFY_GROUND_SPEED_DOT.__doc__ = "Specify the groundspeed acceleration."

agcls.AgTypeNameMap["ReferenceStateLongitudinalAccelerationMode"] = ReferenceStateLongitudinalAccelerationMode

class ReferenceStateLateralAccelerationMode(IntEnum):
    """The mode to specify the lateral acceleration of the aircraft."""

    SPECIFY_HEADING_DOT = 0
    """Specify the heading rate of change."""
    SPECIFY_COURSE_DOT = 1
    """Specify the course rate of change."""

ReferenceStateLateralAccelerationMode.SPECIFY_HEADING_DOT.__doc__ = "Specify the heading rate of change."
ReferenceStateLateralAccelerationMode.SPECIFY_COURSE_DOT.__doc__ = "Specify the course rate of change."

agcls.AgTypeNameMap["ReferenceStateLateralAccelerationMode"] = ReferenceStateLateralAccelerationMode

class ReferenceStateAttitudeMode(IntEnum):
    """The mode to specify the attitude rate of change."""

    SPECIFY_PUSH_PULL_G = 0
    """Specify the push/pull G."""
    SPECIFY_PITCH_RATE = 1
    """Specify the pitch rate of change."""

ReferenceStateAttitudeMode.SPECIFY_PUSH_PULL_G.__doc__ = "Specify the push/pull G."
ReferenceStateAttitudeMode.SPECIFY_PITCH_RATE.__doc__ = "Specify the pitch rate of change."

agcls.AgTypeNameMap["ReferenceStateAttitudeMode"] = ReferenceStateAttitudeMode

class AndOr(IntEnum):
    """The option to specify AND or OR."""

    AND = 0
    """And option."""
    OR = 1
    """Or option."""

AndOr.AND.__doc__ = "And option."
AndOr.OR.__doc__ = "Or option."

agcls.AgTypeNameMap["AndOr"] = AndOr

class JetEngineTechnologyLevel(IntEnum):
    """The technology level of the jet engine."""

    IDEAL = 0
    """Ideal (perfect efficiencies)."""
    LEVEL1 = 1
    """Level 1 (1945-1965)."""
    LEVEL2 = 2
    """Level 2 (1965-1985)."""
    LEVEL3 = 3
    """Level 3 (1985-2005)."""
    LEVEL4 = 4
    """Level 4 (2005-2025)."""
    LEVEL5 = 5
    """Level 5 (2025-2045)."""

JetEngineTechnologyLevel.IDEAL.__doc__ = "Ideal (perfect efficiencies)."
JetEngineTechnologyLevel.LEVEL1.__doc__ = "Level 1 (1945-1965)."
JetEngineTechnologyLevel.LEVEL2.__doc__ = "Level 2 (1965-1985)."
JetEngineTechnologyLevel.LEVEL3.__doc__ = "Level 3 (1985-2005)."
JetEngineTechnologyLevel.LEVEL4.__doc__ = "Level 4 (2005-2025)."
JetEngineTechnologyLevel.LEVEL5.__doc__ = "Level 5 (2025-2045)."

agcls.AgTypeNameMap["JetEngineTechnologyLevel"] = JetEngineTechnologyLevel

class JetEngineIntakeType(IntEnum):
    """The intake type of the jet engine."""

    SUBSONIC_NACELLES = 0
    """Subsonic nacelles."""
    SUBSONIC_EMBEDDED = 1
    """Subsonic embedded."""
    SUPERSONIC_EMBEDDED = 2
    """Supersonic embedded."""

JetEngineIntakeType.SUBSONIC_NACELLES.__doc__ = "Subsonic nacelles."
JetEngineIntakeType.SUBSONIC_EMBEDDED.__doc__ = "Subsonic embedded."
JetEngineIntakeType.SUPERSONIC_EMBEDDED.__doc__ = "Supersonic embedded."

agcls.AgTypeNameMap["JetEngineIntakeType"] = JetEngineIntakeType

class JetEngineTurbineType(IntEnum):
    """The turbine type of the jet engine."""

    UNCOOLED = 0
    """Uncooled turbine."""
    COOLED = 1
    """Cooled turbine."""

JetEngineTurbineType.UNCOOLED.__doc__ = "Uncooled turbine."
JetEngineTurbineType.COOLED.__doc__ = "Cooled turbine."

agcls.AgTypeNameMap["JetEngineTurbineType"] = JetEngineTurbineType

class JetEngineExhaustNozzleType(IntEnum):
    """The exhaust nozzle type of the jet engine."""

    FIXED_AREA_CONVERGENT = 0
    """Fixed Area Convergent."""
    VARIABLE_AREA_CONVERGENT = 1
    """Variable Area Convergent."""
    VARIABLE_AREA_CONVERGENT_DIVERGENT = 2
    """Variable Area Convergent - Divergent."""

JetEngineExhaustNozzleType.FIXED_AREA_CONVERGENT.__doc__ = "Fixed Area Convergent."
JetEngineExhaustNozzleType.VARIABLE_AREA_CONVERGENT.__doc__ = "Variable Area Convergent."
JetEngineExhaustNozzleType.VARIABLE_AREA_CONVERGENT_DIVERGENT.__doc__ = "Variable Area Convergent - Divergent."

agcls.AgTypeNameMap["JetEngineExhaustNozzleType"] = JetEngineExhaustNozzleType

class JetFuelType(IntEnum):
    """The jet fuel type."""

    KEROSENE_AFPROP = 0
    """A kerosene fuel model based on AFPROP."""
    KEROSENE_CEA = 1
    """A kerosene fuel model based on NASA CEA."""
    HYDROGEN = 2
    """A hydrogen fuel model based on NASA CEA."""

JetFuelType.KEROSENE_AFPROP.__doc__ = "A kerosene fuel model based on AFPROP."
JetFuelType.KEROSENE_CEA.__doc__ = "A kerosene fuel model based on NASA CEA."
JetFuelType.HYDROGEN.__doc__ = "A hydrogen fuel model based on NASA CEA."

agcls.AgTypeNameMap["JetFuelType"] = JetFuelType

class AFPROPFuelType(IntEnum):
    """The AFPROP fuel type."""

    AFPROP_OVERRIDE = 0
    """Override the specific energy."""
    AFPROP_JET_A = 1
    """Jet A fuel model based on AFPROP."""
    AFPROP_JET_A1 = 2
    """Jet A-1 fuel model based on AFPROP."""
    AFPROPJP5 = 3
    """JP-5 fuel model based on AFPROP."""
    AFPROPJP7 = 4
    """JP-7 fuel model based on AFPROP."""

AFPROPFuelType.AFPROP_OVERRIDE.__doc__ = "Override the specific energy."
AFPROPFuelType.AFPROP_JET_A.__doc__ = "Jet A fuel model based on AFPROP."
AFPROPFuelType.AFPROP_JET_A1.__doc__ = "Jet A-1 fuel model based on AFPROP."
AFPROPFuelType.AFPROPJP5.__doc__ = "JP-5 fuel model based on AFPROP."
AFPROPFuelType.AFPROPJP7.__doc__ = "JP-7 fuel model based on AFPROP."

agcls.AgTypeNameMap["AFPROPFuelType"] = AFPROPFuelType

class CEAFuelType(IntEnum):
    """The CEA fuel type."""

    CEA_OVERRIDE = 0
    """Override the specific energy."""
    CEA_JET_A = 1
    """Jet A fuel model based on NASA CEA."""
    CEA_JET_A1 = 2
    """Jet A-1 fuel model based on NASA CEA."""
    CEAJP5 = 3
    """JP-5 fuel model based on NASA CEA."""
    CEAJP7 = 4
    """JP-7 fuel model based on NASA CEA."""

CEAFuelType.CEA_OVERRIDE.__doc__ = "Override the specific energy."
CEAFuelType.CEA_JET_A.__doc__ = "Jet A fuel model based on NASA CEA."
CEAFuelType.CEA_JET_A1.__doc__ = "Jet A-1 fuel model based on NASA CEA."
CEAFuelType.CEAJP5.__doc__ = "JP-5 fuel model based on NASA CEA."
CEAFuelType.CEAJP7.__doc__ = "JP-7 fuel model based on NASA CEA."

agcls.AgTypeNameMap["CEAFuelType"] = CEAFuelType

class TurbineMode(IntEnum):
    """The turbine mode for a Sub/Super/Hypersonic powerplant."""

    TURBINE_MODE_DISABLED = 0
    """Disabled turbine mode."""
    TURBINE_MODE_TURBOJET_BASIC_AB = 1
    """Turbojet - Basic w/AB (Thermodynamic model)."""
    TURBINE_MODE_TURBOFAN_BASIC_AB = 2
    """Turbofan - Basic w/AB (Thermodynamic model)."""

TurbineMode.TURBINE_MODE_DISABLED.__doc__ = "Disabled turbine mode."
TurbineMode.TURBINE_MODE_TURBOJET_BASIC_AB.__doc__ = "Turbojet - Basic w/AB (Thermodynamic model)."
TurbineMode.TURBINE_MODE_TURBOFAN_BASIC_AB.__doc__ = "Turbofan - Basic w/AB (Thermodynamic model)."

agcls.AgTypeNameMap["TurbineMode"] = TurbineMode

class RamjetMode(IntEnum):
    """The ramjet mode for a Sub/Super/Hypersonic powerplant."""

    RAMJET_MODE_DISABLED = 0
    """Disabled ramjet mode."""
    RAMJET_MODE_BASIC = 1
    """Ramjet - Basic mode."""

RamjetMode.RAMJET_MODE_DISABLED.__doc__ = "Disabled ramjet mode."
RamjetMode.RAMJET_MODE_BASIC.__doc__ = "Ramjet - Basic mode."

agcls.AgTypeNameMap["RamjetMode"] = RamjetMode

class ScramjetMode(IntEnum):
    """The scramjet mode for a Sub/Super/Hypersonic powerplant."""

    SCRAMJET_MODE_DISABLED = 0
    """Disabled scramjet mode."""
    SCRAMJET_MODE_BASIC = 1
    """Scramjet - Basic mode."""

ScramjetMode.SCRAMJET_MODE_DISABLED.__doc__ = "Disabled scramjet mode."
ScramjetMode.SCRAMJET_MODE_BASIC.__doc__ = "Scramjet - Basic mode."

agcls.AgTypeNameMap["ScramjetMode"] = ScramjetMode

class AviatorNumericalIntegrator(IntEnum):
    """The numerical integrator to be used for the procedure."""

    RUNGE_KUTTA4 = 0
    """Runge-Kutta 4th Order Fixed Time Step Integrator."""
    RUNGE_KUTTA45 = 1
    """Runge-Kutta 4th Order Adaptive Time Step Integrator."""

AviatorNumericalIntegrator.RUNGE_KUTTA4.__doc__ = "Runge-Kutta 4th Order Fixed Time Step Integrator."
AviatorNumericalIntegrator.RUNGE_KUTTA45.__doc__ = "Runge-Kutta 4th Order Adaptive Time Step Integrator."

agcls.AgTypeNameMap["AviatorNumericalIntegrator"] = AviatorNumericalIntegrator

class Ballistic3DControlMode(IntEnum):
    """The control mode used to define the ballistic 3D strategy of the basic maneuver procedure."""

    BALLISTIC_3D_COMPENSATE_FOR_WIND = 0
    """The vehicle will compensate for the wind along its ballistic trajectory."""
    BALLISTIC_3D_WIND_PUSHES_VEHICLE = 1
    """Allow the wind to affect the ballistic trajectory."""
    BALLISTIC_3D_PARACHUTE_MODE = 2
    """Model the vehicle as a parachute."""

Ballistic3DControlMode.BALLISTIC_3D_COMPENSATE_FOR_WIND.__doc__ = "The vehicle will compensate for the wind along its ballistic trajectory."
Ballistic3DControlMode.BALLISTIC_3D_WIND_PUSHES_VEHICLE.__doc__ = "Allow the wind to affect the ballistic trajectory."
Ballistic3DControlMode.BALLISTIC_3D_PARACHUTE_MODE.__doc__ = "Model the vehicle as a parachute."

agcls.AgTypeNameMap["Ballistic3DControlMode"] = Ballistic3DControlMode

class LaunchDynamicStateCoordFrame(IntEnum):
    """The coordinate frame used for a LaunchDynState procedure."""

    LAUNCH_DYNAMIC_STATE_COORD_FRAME_BODY = 0
    """The object's body frame."""
    LAUNCH_DYNAMIC_STATE_COORD_FRAME_LOCAL_HORIZONTAL = 1
    """The object's local horizontal frame."""

LaunchDynamicStateCoordFrame.LAUNCH_DYNAMIC_STATE_COORD_FRAME_BODY.__doc__ = "The object's body frame."
LaunchDynamicStateCoordFrame.LAUNCH_DYNAMIC_STATE_COORD_FRAME_LOCAL_HORIZONTAL.__doc__ = "The object's local horizontal frame."

agcls.AgTypeNameMap["LaunchDynamicStateCoordFrame"] = LaunchDynamicStateCoordFrame

class LaunchDynamicStateBearingReference(IntEnum):
    """The vector used as a bearing reference for a LaunchDynState procedure."""

    LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_VELOCITY = 0
    """The object's velocity vector."""
    LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_COORD_FRAME_X = 1
    """The X vector of the object's coordinate frame."""
    LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_NORTH = 2
    """The object's vector to North."""

LaunchDynamicStateBearingReference.LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_VELOCITY.__doc__ = "The object's velocity vector."
LaunchDynamicStateBearingReference.LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_COORD_FRAME_X.__doc__ = "The X vector of the object's coordinate frame."
LaunchDynamicStateBearingReference.LAUNCH_DYNAMIC_STATE_BEARING_REFERENCE_NORTH.__doc__ = "The object's vector to North."

agcls.AgTypeNameMap["LaunchDynamicStateBearingReference"] = LaunchDynamicStateBearingReference

class AltitudeReference(IntEnum):
    """The altitude reference."""

    ALTITUDE_REFERENCE_WGS84 = 0
    """Altitude above WGS84 ground reference."""
    ALTITUDE_REFERENCE_MSL = 1
    """Altitude above Mean Sea Level (MSL)."""
    ALTITUDE_REFERENCE_TERRAIN = 2
    """ALtitude above terrain."""

AltitudeReference.ALTITUDE_REFERENCE_WGS84.__doc__ = "Altitude above WGS84 ground reference."
AltitudeReference.ALTITUDE_REFERENCE_MSL.__doc__ = "Altitude above Mean Sea Level (MSL)."
AltitudeReference.ALTITUDE_REFERENCE_TERRAIN.__doc__ = "ALtitude above terrain."

agcls.AgTypeNameMap["AltitudeReference"] = AltitudeReference

class SmoothTurnFlightPathAngleMode(IntEnum):
    """The flight path angle mode for the Smooth Turn strategy of the Basic Maneuver procedure."""

    SMOOTH_TURN_FLIGHT_PATH_ANGLE_HOLD_INITIAL = 0
    """Hold the initial flight path angle."""
    SMOOTH_TURN_FLIGHT_PATH_ANGLE_LEVEL_OFF = 1
    """Levell off the flight path angle."""

SmoothTurnFlightPathAngleMode.SMOOTH_TURN_FLIGHT_PATH_ANGLE_HOLD_INITIAL.__doc__ = "Hold the initial flight path angle."
SmoothTurnFlightPathAngleMode.SMOOTH_TURN_FLIGHT_PATH_ANGLE_LEVEL_OFF.__doc__ = "Levell off the flight path angle."

agcls.AgTypeNameMap["SmoothTurnFlightPathAngleMode"] = SmoothTurnFlightPathAngleMode

class Pitch3DControlMode(IntEnum):
    """The control mode used to define the pitch 3D strategy of the basic maneuver procedure."""

    PITCH_3D_COMPENSATE_FOR_WIND = 0
    """The vehicle will compensate for the wind along its trajectory."""
    PITCH_3D_WIND_PUSHES_VEHICLE = 1
    """Allow the wind to affect the trajectory."""

Pitch3DControlMode.PITCH_3D_COMPENSATE_FOR_WIND.__doc__ = "The vehicle will compensate for the wind along its trajectory."
Pitch3DControlMode.PITCH_3D_WIND_PUSHES_VEHICLE.__doc__ = "Allow the wind to affect the trajectory."

agcls.AgTypeNameMap["Pitch3DControlMode"] = Pitch3DControlMode

class RefuelDumpMode(IntEnum):
    """The modes used to define procedure refuel/dump modes."""

    REFUEL_DUMP_DISABLED = 0
    """RefuelDump Disabled."""
    REFUEL_TOP_OFF = 1
    """Refuel Top Off."""
    REFUEL_TO_FUEL_STATE = 2
    """Refuel to specified fuel state."""
    REFUEL_TO_WEIGHT = 3
    """Refuel to specified weight."""
    REFUEL_QUANTITY = 4
    """Refuel specified quantity."""
    DUMP_TO_FUEL_STATE = 5
    """Dump to specified fuel state."""
    DUMP_TO_WEIGHT = 6
    """Dump to specified weight."""
    DUMP_QUANTITY = 7
    """Dump specified quantity."""

RefuelDumpMode.REFUEL_DUMP_DISABLED.__doc__ = "RefuelDump Disabled."
RefuelDumpMode.REFUEL_TOP_OFF.__doc__ = "Refuel Top Off."
RefuelDumpMode.REFUEL_TO_FUEL_STATE.__doc__ = "Refuel to specified fuel state."
RefuelDumpMode.REFUEL_TO_WEIGHT.__doc__ = "Refuel to specified weight."
RefuelDumpMode.REFUEL_QUANTITY.__doc__ = "Refuel specified quantity."
RefuelDumpMode.DUMP_TO_FUEL_STATE.__doc__ = "Dump to specified fuel state."
RefuelDumpMode.DUMP_TO_WEIGHT.__doc__ = "Dump to specified weight."
RefuelDumpMode.DUMP_QUANTITY.__doc__ = "Dump specified quantity."

agcls.AgTypeNameMap["RefuelDumpMode"] = RefuelDumpMode

class BasicManeuverGlideSpeedControlMode(IntEnum):
    """The modes used to define basic maneuver glide speed control modes."""

    GLIDE_SPEED_IMMEDIATE_CHANGE = 0
    """Immediate speed change."""
    GLIDE_SPEED_AT_ALTITUDE = 1
    """Achieve speed at altitude."""

BasicManeuverGlideSpeedControlMode.GLIDE_SPEED_IMMEDIATE_CHANGE.__doc__ = "Immediate speed change."
BasicManeuverGlideSpeedControlMode.GLIDE_SPEED_AT_ALTITUDE.__doc__ = "Achieve speed at altitude."

agcls.AgTypeNameMap["BasicManeuverGlideSpeedControlMode"] = BasicManeuverGlideSpeedControlMode

class TargetPositionVelocityType(IntEnum):
    """The target pos/vel type."""

    SURFACE_TARGET_POSITION_VELOCITY = 0
    """Noisy Surface Target PosVel."""
    BEARING_RANGE_TARGET_POSITION_VELOCITY = 1
    """Noisy Bearing Range Target PosVel."""
    DISABLED_POSITION_VELOCITY = 2
    """Disabled Target PosVel."""

TargetPositionVelocityType.SURFACE_TARGET_POSITION_VELOCITY.__doc__ = "Noisy Surface Target PosVel."
TargetPositionVelocityType.BEARING_RANGE_TARGET_POSITION_VELOCITY.__doc__ = "Noisy Bearing Range Target PosVel."
TargetPositionVelocityType.DISABLED_POSITION_VELOCITY.__doc__ = "Disabled Target PosVel."

agcls.AgTypeNameMap["TargetPositionVelocityType"] = TargetPositionVelocityType

class EphemShiftRotateAltitudeMode(IntEnum):
    """Ephem alt mode."""

    ALTITUDE_MODE_MSL = 0
    """Enumeration EphemShiftRotateAltitudeMode:eAltModeMSL."""
    ALTITUDE_MODE_WGS = 1
    """Enumeration EphemShiftRotateAltitudeMode:eAltModeWGS."""
    ALTITUDE_MODE_RELATIVE = 2
    """Enumeration EphemShiftRotateAltitudeMode:eAltModeRel."""

EphemShiftRotateAltitudeMode.ALTITUDE_MODE_MSL.__doc__ = "Enumeration EphemShiftRotateAltitudeMode:eAltModeMSL."
EphemShiftRotateAltitudeMode.ALTITUDE_MODE_WGS.__doc__ = "Enumeration EphemShiftRotateAltitudeMode:eAltModeWGS."
EphemShiftRotateAltitudeMode.ALTITUDE_MODE_RELATIVE.__doc__ = "Enumeration EphemShiftRotateAltitudeMode:eAltModeRel."

agcls.AgTypeNameMap["EphemShiftRotateAltitudeMode"] = EphemShiftRotateAltitudeMode

class EphemShiftRotateCourseMode(IntEnum):
    """Ephem course mode."""

    COURSE_MODE_TRUE = 0
    """Enumeration EphemShiftRotateCourseMode:eCourseModeTrue."""
    COURSE_MODE_MAGNITUDE = 1
    """Enumeration EphemShiftRotateCourseMode:eCourseModeMag."""
    COURSE_MODE_RELATIVE = 2
    """Enumeration EphemShiftRotateCourseMode:eCourseModeRel."""

EphemShiftRotateCourseMode.COURSE_MODE_TRUE.__doc__ = "Enumeration EphemShiftRotateCourseMode:eCourseModeTrue."
EphemShiftRotateCourseMode.COURSE_MODE_MAGNITUDE.__doc__ = "Enumeration EphemShiftRotateCourseMode:eCourseModeMag."
EphemShiftRotateCourseMode.COURSE_MODE_RELATIVE.__doc__ = "Enumeration EphemShiftRotateCourseMode:eCourseModeRel."

agcls.AgTypeNameMap["EphemShiftRotateCourseMode"] = EphemShiftRotateCourseMode


class ISite(object):
    """
    Interface to access Site options.

    Examples
    --------
    Rename a procedure and its site:
    >>> # IProcedure procedure: Procedure object
    >>> # Rename the procedure
    >>> procedure.name = "New Procedure"
    >>> # Get the site corresponding to the procedure
    >>> site = procedure.site
    >>> # Rename the site
    >>> site.name = "New Site"
    """

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _set_name_method_offset = 2
    _metadata = {
        "iid_data" : (5465348812426829484, 1980086686715111829),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISite."""
        initialize_from_source_object(self, source_object, ISite)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISite)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISite, None)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the site."""
        return self._intf.get_property(ISite._metadata, ISite._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        """Get or set the name of the site."""
        return self._intf.set_property(ISite._metadata, ISite._set_name_metadata, value)

    _property_names[name] = "name"


agcls.AgClassCatalog.add_catalog_entry((5465348812426829484, 1980086686715111829), ISite)
agcls.AgTypeNameMap["ISite"] = ISite

class ICatalogItem(object):
    """
    Interface used to access the options for a Catalog Item in the Aviator Catalog. Use this interface to Create, Remove, Duplicate, or Rename items in the catalog.

    Examples
    --------
    Create a new performance model for an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the names of the current acceleration models
    >>> modelNames = acceleration.child_names
    >>> # Check how many models there are
    >>> modelCount = len(modelNames)
    >>> # Get the child types (for example AGI Basic Acceleration Model, Advanced Acceleration Model)
    >>> modelTypes = acceleration.child_types
    >>> # Create a new performance model of type "Advanced Acceleration Model"
    >>> newPerformanceModel = acceleration.add_child_of_type("Advanced Acceleration Model", "Model Name")
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _set_name_method_offset = 2
    _get_description_method_offset = 3
    _duplicate_method_offset = 4
    _remove_method_offset = 5
    _save_method_offset = 6
    _get_is_read_only_method_offset = 7
    _get_child_names_method_offset = 8
    _get_child_item_by_name_method_offset = 9
    _get_child_types_method_offset = 10
    _add_default_child_method_offset = 11
    _add_child_of_type_method_offset = 12
    _contains_child_item_method_offset = 13
    _metadata = {
        "iid_data" : (5458227820228317150, 3292997058202486194),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogItem."""
        initialize_from_source_object(self, source_object, ICatalogItem)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogItem)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogItem, None)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the catalog item."""
        return self._intf.get_property(ICatalogItem._metadata, ICatalogItem._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        """Get or set the name of the catalog item."""
        return self._intf.set_property(ICatalogItem._metadata, ICatalogItem._set_name_metadata, value)

    _get_description_metadata = { "offset" : _get_description_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def description(self) -> str:
        """Get the description of the catalog item."""
        return self._intf.get_property(ICatalogItem._metadata, ICatalogItem._get_description_metadata)

    _duplicate_metadata = { "offset" : _duplicate_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def duplicate(self) -> "ICatalogItem":
        """Duplicates the catalog item."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._duplicate_metadata, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove(self) -> None:
        """Remove the catalog item."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._remove_metadata, )

    _save_metadata = { "offset" : _save_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def save(self) -> None:
        """Save the catalog item."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._save_metadata, )

    _get_is_read_only_metadata = { "offset" : _get_is_read_only_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_read_only(self) -> bool:
        """Get whether the catalog item is read only."""
        return self._intf.get_property(ICatalogItem._metadata, ICatalogItem._get_is_read_only_metadata)

    _get_child_names_metadata = { "offset" : _get_child_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def child_names(self) -> list:
        """Get the child names of the catalog item."""
        return self._intf.get_property(ICatalogItem._metadata, ICatalogItem._get_child_names_metadata)

    _get_child_item_by_name_metadata = { "offset" : _get_child_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_child_item_by_name(self, child_name:str) -> "ICatalogItem":
        """Get the child of the catalog item with the given name."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._get_child_item_by_name_metadata, child_name, OutArg())

    _get_child_types_metadata = { "offset" : _get_child_types_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def child_types(self) -> list:
        """Get the child types."""
        return self._intf.get_property(ICatalogItem._metadata, ICatalogItem._get_child_types_metadata)

    _add_default_child_metadata = { "offset" : _add_default_child_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_default_child(self, child_name:str) -> "ICatalogItem":
        """Create a new child with the given name and default type."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._add_default_child_metadata, child_name, OutArg())

    _add_child_of_type_metadata = { "offset" : _add_child_of_type_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_child_of_type(self, child_type:str, child_name:str) -> "ICatalogItem":
        """Create a new child with the given name and specified type."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._add_child_of_type_metadata, child_type, child_name, OutArg())

    _contains_child_item_metadata = { "offset" : _contains_child_item_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains_child_item(self, child_item:str) -> bool:
        """Get whether the catalog item is contains the given child item."""
        return self._intf.invoke(ICatalogItem._metadata, ICatalogItem._contains_child_item_metadata, child_item, OutArg())

    _property_names[name] = "name"
    _property_names[description] = "description"
    _property_names[is_read_only] = "is_read_only"
    _property_names[child_names] = "child_names"
    _property_names[child_types] = "child_types"


agcls.AgClassCatalog.add_catalog_entry((5458227820228317150, 3292997058202486194), ICatalogItem)
agcls.AgTypeNameMap["ICatalogItem"] = ICatalogItem

class IBasicManeuverStrategy(object):
    """Interface used to access options for a Basic Maneuver Strategy."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4744335311711001158, 10362982119561193112),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IBasicManeuverStrategy."""
        initialize_from_source_object(self, source_object, IBasicManeuverStrategy)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IBasicManeuverStrategy)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IBasicManeuverStrategy, None)



agcls.AgClassCatalog.add_catalog_entry((4744335311711001158, 10362982119561193112), IBasicManeuverStrategy)
agcls.AgTypeNameMap["IBasicManeuverStrategy"] = IBasicManeuverStrategy

class IAviatorVehicle(object):
    """Interface for a vehicle in Aviator."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_catalog_item_method_offset = 1
    _metadata = {
        "iid_data" : (5343444076281023740, 16945725846674352552),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAviatorVehicle."""
        initialize_from_source_object(self, source_object, IAviatorVehicle)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAviatorVehicle)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAviatorVehicle, None)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(IAviatorVehicle._metadata, IAviatorVehicle._get_as_catalog_item_metadata, OutArg())



agcls.AgClassCatalog.add_catalog_entry((5343444076281023740, 16945725846674352552), IAviatorVehicle)
agcls.AgTypeNameMap["IAviatorVehicle"] = IAviatorVehicle

class IARINC424Item(object):
    """Interface used to access the options for an ARINC424 Item found in the Aviator catalog."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_catalog_item_method_offset = 1
    _get_value_method_offset = 2
    _get_all_fields_method_offset = 3
    _get_all_fields_and_values_method_offset = 4
    _copy_site_method_offset = 5
    _metadata = {
        "iid_data" : (5321250993774161200, 8939444136967404965),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IARINC424Item."""
        initialize_from_source_object(self, source_object, IARINC424Item)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IARINC424Item)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IARINC424Item, None)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(IARINC424Item._metadata, IARINC424Item._get_as_catalog_item_metadata, OutArg())

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg,) }
    def get_value(self, field_name:str) -> typing.Any:
        """Get the value of the field with the given name."""
        return self._intf.invoke(IARINC424Item._metadata, IARINC424Item._get_value_metadata, field_name, OutArg())

    _get_all_fields_metadata = { "offset" : _get_all_fields_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_fields(self) -> list:
        """Get all the field names."""
        return self._intf.invoke(IARINC424Item._metadata, IARINC424Item._get_all_fields_metadata, OutArg())

    _get_all_fields_and_values_metadata = { "offset" : _get_all_fields_and_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_fields_and_values(self) -> list:
        """Get all the field names along with the corresponding value."""
        return self._intf.invoke(IARINC424Item._metadata, IARINC424Item._get_all_fields_and_values_metadata, OutArg())

    _copy_site_metadata = { "offset" : _copy_site_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_site(self) -> None:
        """Copy the site to the clipboard."""
        return self._intf.invoke(IARINC424Item._metadata, IARINC424Item._copy_site_metadata, )



agcls.AgClassCatalog.add_catalog_entry((5321250993774161200, 8939444136967404965), IARINC424Item)
agcls.AgTypeNameMap["IARINC424Item"] = IARINC424Item

class IStation(object):
    """Interface used to access a station for an Aviator aircraft."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5722385785492826861, 3941921955448583610),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IStation."""
        initialize_from_source_object(self, source_object, IStation)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IStation)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IStation, None)



agcls.AgClassCatalog.add_catalog_entry((5722385785492826861, 3941921955448583610), IStation)
agcls.AgTypeNameMap["IStation"] = IStation

class ICatalogSource(object):
    """Interface used to access options for a source in the Aviator Catalog. Examples of sources include User Aircraft Models, ARINC424runways, User Runways, etc."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_child_names_method_offset = 1
    _contains_method_offset = 2
    _remove_child_method_offset = 3
    _save_method_offset = 4
    _metadata = {
        "iid_data" : (5049762571904518511, 12921546796455048600),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogSource."""
        initialize_from_source_object(self, source_object, ICatalogSource)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogSource)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogSource, None)

    _get_child_names_metadata = { "offset" : _get_child_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def child_names(self) -> list:
        """Return the child names."""
        return self._intf.get_property(ICatalogSource._metadata, ICatalogSource._get_child_names_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, aircraft_name:str) -> bool:
        """Check whether the catalog source contains the catalog item with the given name."""
        return self._intf.invoke(ICatalogSource._metadata, ICatalogSource._contains_metadata, aircraft_name, OutArg())

    _remove_child_metadata = { "offset" : _remove_child_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove_child(self, child_name:str) -> None:
        """Remove the child with the given name."""
        return self._intf.invoke(ICatalogSource._metadata, ICatalogSource._remove_child_metadata, child_name)

    _save_metadata = { "offset" : _save_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def save(self) -> None:
        """Save the catalog item."""
        return self._intf.invoke(ICatalogSource._metadata, ICatalogSource._save_metadata, )

    _property_names[child_names] = "child_names"


agcls.AgClassCatalog.add_catalog_entry((5049762571904518511, 12921546796455048600), ICatalogSource)
agcls.AgTypeNameMap["ICatalogSource"] = ICatalogSource

class IBasicFixedWingLiftHelper(object):
    """Interface used to access Lift Coefficient Helper in the Basic Fixed Wing Aerodynamics interface for the Basic Acceleration Model of an aircraft."""

    _num_methods = 0
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _metadata = {
        "iid_data" : (4688474907311176393, 2468997903478744199),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IBasicFixedWingLiftHelper."""
        initialize_from_source_object(self, source_object, IBasicFixedWingLiftHelper)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IBasicFixedWingLiftHelper)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IBasicFixedWingLiftHelper, None)



agcls.AgClassCatalog.add_catalog_entry((4688474907311176393, 2468997903478744199), IBasicFixedWingLiftHelper)
agcls.AgTypeNameMap["IBasicFixedWingLiftHelper"] = IBasicFixedWingLiftHelper

class ICatalogRunway(object):
    """Interface used to access a runway in the Aviator catalog."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5582374247106176888, 13141122444054183846),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogRunway."""
        initialize_from_source_object(self, source_object, ICatalogRunway)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogRunway)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogRunway, None)



agcls.AgClassCatalog.add_catalog_entry((5582374247106176888, 13141122444054183846), ICatalogRunway)
agcls.AgTypeNameMap["ICatalogRunway"] = ICatalogRunway

class ICatalogAirport(object):
    """Interface used to access a airport in the Aviator catalog."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5517232244535054220, 17256594794181634488),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogAirport."""
        initialize_from_source_object(self, source_object, ICatalogAirport)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogAirport)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogAirport, None)



agcls.AgClassCatalog.add_catalog_entry((5517232244535054220, 17256594794181634488), ICatalogAirport)
agcls.AgTypeNameMap["ICatalogAirport"] = ICatalogAirport

class ICatalogNavaid(object):
    """Interface used to access a navaid in the Aviator catalog."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5313582233894378368, 56929069166080155),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogNavaid."""
        initialize_from_source_object(self, source_object, ICatalogNavaid)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogNavaid)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogNavaid, None)



agcls.AgClassCatalog.add_catalog_entry((5313582233894378368, 56929069166080155), ICatalogNavaid)
agcls.AgTypeNameMap["ICatalogNavaid"] = ICatalogNavaid

class ICatalogVTOLPoint(object):
    """Interface used to access a VTOL Point in the Aviator catalog."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5738919117285590368, 14261233550328315012),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogVTOLPoint."""
        initialize_from_source_object(self, source_object, ICatalogVTOLPoint)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogVTOLPoint)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogVTOLPoint, None)



agcls.AgClassCatalog.add_catalog_entry((5738919117285590368, 14261233550328315012), ICatalogVTOLPoint)
agcls.AgTypeNameMap["ICatalogVTOLPoint"] = ICatalogVTOLPoint

class ICatalogWaypoint(object):
    """Interface used to access a waypoint in the Aviator catalog."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5532823826035277770, 17986873366649893022),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICatalogWaypoint."""
        initialize_from_source_object(self, source_object, ICatalogWaypoint)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICatalogWaypoint)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICatalogWaypoint, None)



agcls.AgClassCatalog.add_catalog_entry((5532823826035277770, 17986873366649893022), ICatalogWaypoint)
agcls.AgTypeNameMap["ICatalogWaypoint"] = ICatalogWaypoint

class IARINC424Airport(object):
    """Do not use this interface, as it is deprecated. Use IARINC424Item instead."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_catalog_item_method_offset = 1
    _metadata = {
        "iid_data" : (4806586137712321246, 15546437888766560677),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IARINC424Airport."""
        initialize_from_source_object(self, source_object, IARINC424Airport)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IARINC424Airport)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IARINC424Airport, None)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(IARINC424Airport._metadata, IARINC424Airport._get_as_catalog_item_metadata, OutArg())



agcls.AgClassCatalog.add_catalog_entry((4806586137712321246, 15546437888766560677), IARINC424Airport)
agcls.AgTypeNameMap["IARINC424Airport"] = IARINC424Airport

class IDAFIFItem(object):
    """Interface used to access the options for an DAFIF Item found in the Aviator catalog."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_value_method_offset = 1
    _get_all_fields_method_offset = 2
    _get_all_fields_and_values_method_offset = 3
    _copy_site_method_offset = 4
    _get_as_catalog_item_method_offset = 5
    _metadata = {
        "iid_data" : (5172829076274399625, 12291090328348071297),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IDAFIFItem."""
        initialize_from_source_object(self, source_object, IDAFIFItem)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IDAFIFItem)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IDAFIFItem, None)

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg,) }
    def get_value(self, field_name:str) -> typing.Any:
        """Get the value of the field with the given name."""
        return self._intf.invoke(IDAFIFItem._metadata, IDAFIFItem._get_value_metadata, field_name, OutArg())

    _get_all_fields_metadata = { "offset" : _get_all_fields_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_fields(self) -> list:
        """Get all the field names."""
        return self._intf.invoke(IDAFIFItem._metadata, IDAFIFItem._get_all_fields_metadata, OutArg())

    _get_all_fields_and_values_metadata = { "offset" : _get_all_fields_and_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_fields_and_values(self) -> list:
        """Get all the field names along with the corresponding value."""
        return self._intf.invoke(IDAFIFItem._metadata, IDAFIFItem._get_all_fields_and_values_metadata, OutArg())

    _copy_site_metadata = { "offset" : _copy_site_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_site(self) -> None:
        """Copy the site to the clipboard."""
        return self._intf.invoke(IDAFIFItem._metadata, IDAFIFItem._copy_site_metadata, )

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(IDAFIFItem._metadata, IDAFIFItem._get_as_catalog_item_metadata, OutArg())



agcls.AgClassCatalog.add_catalog_entry((5172829076274399625, 12291090328348071297), IDAFIFItem)
agcls.AgTypeNameMap["IDAFIFItem"] = IDAFIFItem

class IVerticalPlaneOptions(object):
    """Interface used to access the Vertical Plane options for an Aviator procedure."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_min_enroute_flight_path_angle_method_offset = 1
    _set_min_enroute_flight_path_angle_method_offset = 2
    _get_max_enroute_flight_path_angle_method_offset = 3
    _set_max_enroute_flight_path_angle_method_offset = 4
    _get_max_vert_plane_radius_factor_method_offset = 5
    _set_max_vert_plane_radius_factor_method_offset = 6
    _metadata = {
        "iid_data" : (5520552088906406195, 394820396869463446),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVerticalPlaneOptions."""
        initialize_from_source_object(self, source_object, IVerticalPlaneOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVerticalPlaneOptions)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVerticalPlaneOptions, None)

    _get_min_enroute_flight_path_angle_metadata = { "offset" : _get_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def min_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._get_min_enroute_flight_path_angle_metadata)

    _set_min_enroute_flight_path_angle_metadata = { "offset" : _set_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @min_enroute_flight_path_angle.setter
    def min_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._set_min_enroute_flight_path_angle_metadata, value)

    _get_max_enroute_flight_path_angle_metadata = { "offset" : _get_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._get_max_enroute_flight_path_angle_metadata)

    _set_max_enroute_flight_path_angle_metadata = { "offset" : _set_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_enroute_flight_path_angle.setter
    def max_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._set_max_enroute_flight_path_angle_metadata, value)

    _get_max_vert_plane_radius_factor_metadata = { "offset" : _get_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_vert_plane_radius_factor(self) -> float:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.get_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._get_max_vert_plane_radius_factor_metadata)

    _set_max_vert_plane_radius_factor_metadata = { "offset" : _set_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_vert_plane_radius_factor.setter
    def max_vert_plane_radius_factor(self, value:float) -> None:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.set_property(IVerticalPlaneOptions._metadata, IVerticalPlaneOptions._set_max_vert_plane_radius_factor_metadata, value)

    _property_names[min_enroute_flight_path_angle] = "min_enroute_flight_path_angle"
    _property_names[max_enroute_flight_path_angle] = "max_enroute_flight_path_angle"
    _property_names[max_vert_plane_radius_factor] = "max_vert_plane_radius_factor"


agcls.AgClassCatalog.add_catalog_entry((5520552088906406195, 394820396869463446), IVerticalPlaneOptions)
agcls.AgTypeNameMap["IVerticalPlaneOptions"] = IVerticalPlaneOptions

class IEnrouteAndDelayOptions(object):
    """Interface used to access the Enroute options for an Aviator procedure."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_delay_enroute_climb_descents_method_offset = 1
    _set_delay_enroute_climb_descents_method_offset = 2
    _get_use_max_speed_turns_method_offset = 3
    _set_use_max_speed_turns_method_offset = 4
    _get_max_turn_radius_factor_method_offset = 5
    _set_max_turn_radius_factor_method_offset = 6
    _metadata = {
        "iid_data" : (5068556317467918075, 1237280450976607145),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IEnrouteAndDelayOptions."""
        initialize_from_source_object(self, source_object, IEnrouteAndDelayOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IEnrouteAndDelayOptions)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IEnrouteAndDelayOptions, None)

    _get_delay_enroute_climb_descents_metadata = { "offset" : _get_delay_enroute_climb_descents_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def delay_enroute_climb_descents(self) -> bool:
        """Opt whether to delay the enroute climb or descent."""
        return self._intf.get_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._get_delay_enroute_climb_descents_metadata)

    _set_delay_enroute_climb_descents_metadata = { "offset" : _set_delay_enroute_climb_descents_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @delay_enroute_climb_descents.setter
    def delay_enroute_climb_descents(self, value:bool) -> None:
        """Opt whether to delay the enroute climb or descent."""
        return self._intf.set_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._set_delay_enroute_climb_descents_metadata, value)

    _get_use_max_speed_turns_metadata = { "offset" : _get_use_max_speed_turns_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_max_speed_turns(self) -> bool:
        """Opt whether to use the max speed turns."""
        return self._intf.get_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._get_use_max_speed_turns_metadata)

    _set_use_max_speed_turns_metadata = { "offset" : _set_use_max_speed_turns_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_max_speed_turns.setter
    def use_max_speed_turns(self, value:bool) -> None:
        """Opt whether to use the max speed turns."""
        return self._intf.set_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._set_use_max_speed_turns_metadata, value)

    _get_max_turn_radius_factor_metadata = { "offset" : _get_max_turn_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_turn_radius_factor(self) -> float:
        """Get or set the maximum turn radius factor."""
        return self._intf.get_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._get_max_turn_radius_factor_metadata)

    _set_max_turn_radius_factor_metadata = { "offset" : _set_max_turn_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_turn_radius_factor.setter
    def max_turn_radius_factor(self, value:float) -> None:
        """Get or set the maximum turn radius factor."""
        return self._intf.set_property(IEnrouteAndDelayOptions._metadata, IEnrouteAndDelayOptions._set_max_turn_radius_factor_metadata, value)

    _property_names[delay_enroute_climb_descents] = "delay_enroute_climb_descents"
    _property_names[use_max_speed_turns] = "use_max_speed_turns"
    _property_names[max_turn_radius_factor] = "max_turn_radius_factor"


agcls.AgClassCatalog.add_catalog_entry((5068556317467918075, 1237280450976607145), IEnrouteAndDelayOptions)
agcls.AgTypeNameMap["IEnrouteAndDelayOptions"] = IEnrouteAndDelayOptions

class ICruiseAirspeedAndProfileOptions(object):
    """Interface used to access the cruise airspeed options that also include a profile field."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_cruise_speed_type_method_offset = 1
    _set_cruise_speed_type_method_offset = 2
    _get_other_airspeed_type_method_offset = 3
    _get_other_airspeed_method_offset = 4
    _set_other_airspeed_method_offset = 5
    _get_fly_cruise_airspeed_profile_method_offset = 6
    _set_fly_cruise_airspeed_profile_method_offset = 7
    _metadata = {
        "iid_data" : (5050149451706347699, 5240254480125122223),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICruiseAirspeedAndProfileOptions."""
        initialize_from_source_object(self, source_object, ICruiseAirspeedAndProfileOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICruiseAirspeedAndProfileOptions)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICruiseAirspeedAndProfileOptions, None)

    _get_cruise_speed_type_metadata = { "offset" : _get_cruise_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(CruiseSpeed),) }
    @property
    def cruise_speed_type(self) -> "CruiseSpeed":
        """Get or set the method for determining the aircraft's airspeed."""
        return self._intf.get_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._get_cruise_speed_type_metadata)

    _set_cruise_speed_type_metadata = { "offset" : _set_cruise_speed_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(CruiseSpeed),) }
    @cruise_speed_type.setter
    def cruise_speed_type(self, value:"CruiseSpeed") -> None:
        """Get or set the method for determining the aircraft's airspeed."""
        return self._intf.set_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._set_cruise_speed_type_metadata, value)

    _get_other_airspeed_type_metadata = { "offset" : _get_other_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def other_airspeed_type(self) -> "AirspeedType":
        """Get the airspeed type for the other airspeed option."""
        return self._intf.get_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._get_other_airspeed_type_metadata)

    _get_other_airspeed_metadata = { "offset" : _get_other_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def other_airspeed(self) -> float:
        """Get the airspeed for the other airspeed option."""
        return self._intf.get_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._get_other_airspeed_metadata)

    _set_other_airspeed_metadata = { "offset" : _set_other_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_other_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the cruise airspeed. This option is only enabled if the cruise speed type is set to other."""
        return self._intf.invoke(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._set_other_airspeed_metadata, airspeed_type, airspeed)

    _get_fly_cruise_airspeed_profile_metadata = { "offset" : _get_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def fly_cruise_airspeed_profile(self) -> bool:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.get_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._get_fly_cruise_airspeed_profile_metadata)

    _set_fly_cruise_airspeed_profile_metadata = { "offset" : _set_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @fly_cruise_airspeed_profile.setter
    def fly_cruise_airspeed_profile(self, value:bool) -> None:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.set_property(ICruiseAirspeedAndProfileOptions._metadata, ICruiseAirspeedAndProfileOptions._set_fly_cruise_airspeed_profile_metadata, value)

    _property_names[cruise_speed_type] = "cruise_speed_type"
    _property_names[other_airspeed_type] = "other_airspeed_type"
    _property_names[other_airspeed] = "other_airspeed"
    _property_names[fly_cruise_airspeed_profile] = "fly_cruise_airspeed_profile"


agcls.AgClassCatalog.add_catalog_entry((5050149451706347699, 5240254480125122223), ICruiseAirspeedAndProfileOptions)
agcls.AgTypeNameMap["ICruiseAirspeedAndProfileOptions"] = ICruiseAirspeedAndProfileOptions

class IAutomationStrategyFactory(object):
    """Interface used to send connect commands to Aviator objects."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _construct_strategy_method_offset = 1
    _metadata = {
        "iid_data" : (5748143157570064853, 17141228144581304450),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAutomationStrategyFactory."""
        initialize_from_source_object(self, source_object, IAutomationStrategyFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAutomationStrategyFactory)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAutomationStrategyFactory, None)

    _construct_strategy_metadata = { "offset" : _construct_strategy_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IUnknown"), agmarshall.InterfaceOutArg,) }
    def construct_strategy(self, obj:"IUnknown") -> "IBasicManeuverStrategy":
        """Construct the strategy."""
        return self._intf.invoke(IAutomationStrategyFactory._metadata, IAutomationStrategyFactory._construct_strategy_metadata, obj, OutArg())



agcls.AgClassCatalog.add_catalog_entry((5748143157570064853, 17141228144581304450), IAutomationStrategyFactory)
agcls.AgTypeNameMap["IAutomationStrategyFactory"] = IAutomationStrategyFactory

class IConnect(object):
    """Interface used to send connect commands to Aviator objects."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _execute_command_method_offset = 1
    _metadata = {
        "iid_data" : (5321476311283717631, 17088119313231705791),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IConnect."""
        initialize_from_source_object(self, source_object, IConnect)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IConnect)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IConnect, None)

    _execute_command_metadata = { "offset" : _execute_command_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def execute_command(self, command:str) -> typing.Any:
        """Send a connect command to an Aviator object."""
        return self._intf.invoke(IConnect._metadata, IConnect._execute_command_metadata, command, OutArg())



agcls.AgClassCatalog.add_catalog_entry((5321476311283717631, 17088119313231705791), IConnect)
agcls.AgTypeNameMap["IConnect"] = IConnect

class IProcedure(object):
    """
    Interface used to access the options for a procedure. Use this interface to get the Site and Get the time options for the current procedure.

    Examples
    --------
    Configure the wind and atmosphere for a procedure:
    >>> # IProcedure procedure: Procedure object
    >>> # Get the wind model for the procedure
    >>> windModel = procedure.wind_model
    >>> # Use the procedure model
    >>> windModel.wind_model_source = WindAtmosphereModelSource.PROCEDURE_MODEL
    >>> # Let's use constant wind
    >>> windModel.wind_model_type = WindModelType.CONSTANT_WIND
    >>> # Get the constant wind model options
    >>> constantWind = windModel.mode_as_constant
    >>> # Set the wind bearing
    >>> constantWind.wind_bearing = 30
    >>> # Set the wind speed
    >>> constantWind.wind_speed = 5
    >>>
    >>> # Get the atmosphere model used for the procedure
    >>> atmosphere = procedure.atmosphere_model
    >>> # Let's use the procedure model
    >>> atmosphere.atmosphere_model_source = WindAtmosphereModelSource.PROCEDURE_MODEL
    >>> # Get the basic atmosphere options
    >>> basicAtmosphere = atmosphere.mode_as_basic
    >>> # Use standard 1976 atmosphere
    >>> basicAtmosphere.basic_model_type = AtmosphereModelType.STANDARD1976

    Configure a procedure's time options:
    >>> # IProcedure procedure: Procedure object
    >>> # Get the time in epoch seconds
    >>> root.units_preferences.set_current_unit("DateFormat", "EpSec")
    >>> # Get the time options
    >>> timeOptions = procedure.time_options
    >>> # Get the start time
    >>> startTime = timeOptions.start_time
    >>> # Set the procedure to interrupt after 15 seconds
    >>> timeOptions.set_interrupt_time(15)

    Rename a procedure and its site:
    >>> # IProcedure procedure: Procedure object
    >>> # Rename the procedure
    >>> procedure.name = "New Procedure"
    >>> # Get the site corresponding to the procedure
    >>> site = procedure.site
    >>> # Rename the site
    >>> site.name = "New Site"
    """

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _set_name_method_offset = 2
    _get_site_method_offset = 3
    _get_time_options_method_offset = 4
    _get_wind_model_method_offset = 5
    _get_atmosphere_model_method_offset = 6
    _get_calculation_options_method_offset = 7
    _get_refuel_dump_is_supported_method_offset = 8
    _get_refuel_dump_properties_method_offset = 9
    _get_fast_time_options_method_offset = 10
    _metadata = {
        "iid_data" : (5354902126736355035, 15840099138053069727),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IProcedure."""
        initialize_from_source_object(self, source_object, IProcedure)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IProcedure)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IProcedure, None)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        """Get or set the name of the procedure."""
        return self._intf.set_property(IProcedure._metadata, IProcedure._set_name_metadata, value)

    _get_site_metadata = { "offset" : _get_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def site(self) -> "ISite":
        """Get the site of the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_site_metadata)

    _get_time_options_metadata = { "offset" : _get_time_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def time_options(self) -> "ProcedureTimeOptions":
        """Get the time options for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_time_options_metadata)

    _get_wind_model_metadata = { "offset" : _get_wind_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def wind_model(self) -> "WindModel":
        """Get the wind model for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_wind_model_metadata)

    _get_atmosphere_model_metadata = { "offset" : _get_atmosphere_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def atmosphere_model(self) -> "AtmosphereModel":
        """Get the mission atmosphere model."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_atmosphere_model_metadata)

    _get_calculation_options_metadata = { "offset" : _get_calculation_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def calculation_options(self) -> "CalculationOptions":
        """Get the calculation options for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_calculation_options_metadata)

    _get_refuel_dump_is_supported_metadata = { "offset" : _get_refuel_dump_is_supported_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def refuel_dump_is_supported(self) -> bool:
        """Refuel/dump is supported for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_refuel_dump_is_supported_metadata)

    _get_refuel_dump_properties_metadata = { "offset" : _get_refuel_dump_properties_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def refuel_dump_properties(self) -> "RefuelDumpProperties":
        """Get the refuel/dump properties for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_refuel_dump_properties_metadata)

    _get_fast_time_options_metadata = { "offset" : _get_fast_time_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def fast_time_options(self) -> "ProcedureFastTimeOptions":
        """Get the fast time options (without validation or constraints) for the current procedure."""
        return self._intf.get_property(IProcedure._metadata, IProcedure._get_fast_time_options_metadata)

    _property_names[name] = "name"
    _property_names[site] = "site"
    _property_names[time_options] = "time_options"
    _property_names[wind_model] = "wind_model"
    _property_names[atmosphere_model] = "atmosphere_model"
    _property_names[calculation_options] = "calculation_options"
    _property_names[refuel_dump_is_supported] = "refuel_dump_is_supported"
    _property_names[refuel_dump_properties] = "refuel_dump_properties"
    _property_names[fast_time_options] = "fast_time_options"


agcls.AgClassCatalog.add_catalog_entry((5354902126736355035, 15840099138053069727), IProcedure)
agcls.AgTypeNameMap["IProcedure"] = IProcedure

class IPerformanceModel(object):
    """
    Interface for a performance model of an Aviator vehicle.

    Examples
    --------
    Create a new performance model for an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the names of the current acceleration models
    >>> modelNames = acceleration.child_names
    >>> # Check how many models there are
    >>> modelCount = len(modelNames)
    >>> # Get the child types (for example AGI Basic Acceleration Model, Advanced Acceleration Model)
    >>> modelTypes = acceleration.child_types
    >>> # Create a new performance model of type "Advanced Acceleration Model"
    >>> newPerformanceModel = acceleration.add_child_of_type("Advanced Acceleration Model", "Model Name")
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4722239720283002114, 10874464253498804653),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IPerformanceModel."""
        initialize_from_source_object(self, source_object, IPerformanceModel)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IPerformanceModel)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IPerformanceModel, None)



agcls.AgClassCatalog.add_catalog_entry((4722239720283002114, 10874464253498804653), IPerformanceModel)
agcls.AgTypeNameMap["IPerformanceModel"] = IPerformanceModel

class IAdvancedFixedWingGeometry(object):
    """Interface used to access the options for the wing geometry in the advanced fixed wing tool."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4940865896231186286, 11578475062663105194),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAdvancedFixedWingGeometry."""
        initialize_from_source_object(self, source_object, IAdvancedFixedWingGeometry)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAdvancedFixedWingGeometry)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAdvancedFixedWingGeometry, None)



agcls.AgClassCatalog.add_catalog_entry((4940865896231186286, 11578475062663105194), IAdvancedFixedWingGeometry)
agcls.AgTypeNameMap["IAdvancedFixedWingGeometry"] = IAdvancedFixedWingGeometry

class IAdvancedFixedWingPowerplant(object):
    """Interface for a powerplant strategy in the advanced fixed wing tool."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5713896957878092641, 17244840932805749889),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAdvancedFixedWingPowerplant."""
        initialize_from_source_object(self, source_object, IAdvancedFixedWingPowerplant)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAdvancedFixedWingPowerplant)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAdvancedFixedWingPowerplant, None)



agcls.AgClassCatalog.add_catalog_entry((5713896957878092641, 17244840932805749889), IAdvancedFixedWingPowerplant)
agcls.AgTypeNameMap["IAdvancedFixedWingPowerplant"] = IAdvancedFixedWingPowerplant

class ISiteUnknown(object):
    """Interface of an unknown site."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4927199749278391215, 245580461777180035),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISiteUnknown."""
        initialize_from_source_object(self, source_object, ISiteUnknown)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISiteUnknown)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISiteUnknown, None)



agcls.AgClassCatalog.add_catalog_entry((4927199749278391215, 245580461777180035), ISiteUnknown)
agcls.AgTypeNameMap["ISiteUnknown"] = ISiteUnknown



class SiteWaypoint(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a waypoint site."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_as_site_method_offset = 5
    _metadata = {
        "iid_data" : (5613574524731284566, 17207683338010306734),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteWaypoint)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the waypoint latitude."""
        return self._intf.get_property(SiteWaypoint._metadata, SiteWaypoint._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        """Get or set the waypoint latitude."""
        return self._intf.set_property(SiteWaypoint._metadata, SiteWaypoint._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the waypoint longitude."""
        return self._intf.get_property(SiteWaypoint._metadata, SiteWaypoint._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        """Get or set the waypoint longitude."""
        return self._intf.set_property(SiteWaypoint._metadata, SiteWaypoint._set_longitude_metadata, value)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteWaypoint._metadata, SiteWaypoint._get_as_site_metadata, OutArg())

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"

    def __init__(self, source_object=None):
        """Construct an object of type SiteWaypoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteWaypoint)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteWaypoint, [SiteWaypoint, ISite])

agcls.AgClassCatalog.add_catalog_entry((5108603041944758707, 10578588674234114712), SiteWaypoint)
agcls.AgTypeNameMap["SiteWaypoint"] = SiteWaypoint

class SiteEndOfPrevProcedure(ISite, SupportsDeleteCallback):
    """Interface used to access the options for an End of Previous Procedure site type."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_site_method_offset = 1
    _metadata = {
        "iid_data" : (5004093185145679190, 7313314850458100887),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteEndOfPrevProcedure)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteEndOfPrevProcedure._metadata, SiteEndOfPrevProcedure._get_as_site_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SiteEndOfPrevProcedure."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteEndOfPrevProcedure)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteEndOfPrevProcedure, [SiteEndOfPrevProcedure, ISite])

agcls.AgClassCatalog.add_catalog_entry((5621720374100682259, 1489086470013473170), SiteEndOfPrevProcedure)
agcls.AgTypeNameMap["SiteEndOfPrevProcedure"] = SiteEndOfPrevProcedure

class SiteVTOLPoint(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a VTOL Point site."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _get_altitude_reference_method_offset = 7
    _set_altitude_reference_method_offset = 8
    _get_as_site_method_offset = 9
    _metadata = {
        "iid_data" : (5600278003038443155, 12434436858381676162),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteVTOLPoint)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the VTOL Point latitude."""
        return self._intf.get_property(SiteVTOLPoint._metadata, SiteVTOLPoint._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        """Get or set the VTOL Point latitude."""
        return self._intf.set_property(SiteVTOLPoint._metadata, SiteVTOLPoint._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the VTOL Point longitude."""
        return self._intf.get_property(SiteVTOLPoint._metadata, SiteVTOLPoint._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        """Get or set the VTOL Point longitude."""
        return self._intf.set_property(SiteVTOLPoint._metadata, SiteVTOLPoint._set_longitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the altitude for the site."""
        return self._intf.get_property(SiteVTOLPoint._metadata, SiteVTOLPoint._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Get or set the altitude for the site."""
        return self._intf.set_property(SiteVTOLPoint._metadata, SiteVTOLPoint._set_altitude_metadata, value)

    _get_altitude_reference_metadata = { "offset" : _get_altitude_reference_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @property
    def altitude_reference(self) -> "AGLMSL":
        """Get or set the altitude reference."""
        return self._intf.get_property(SiteVTOLPoint._metadata, SiteVTOLPoint._get_altitude_reference_metadata)

    _set_altitude_reference_metadata = { "offset" : _set_altitude_reference_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @altitude_reference.setter
    def altitude_reference(self, value:"AGLMSL") -> None:
        """Get or set the altitude reference."""
        return self._intf.set_property(SiteVTOLPoint._metadata, SiteVTOLPoint._set_altitude_reference_metadata, value)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteVTOLPoint._metadata, SiteVTOLPoint._get_as_site_metadata, OutArg())

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[altitude] = "altitude"
    _property_names[altitude_reference] = "altitude_reference"

    def __init__(self, source_object=None):
        """Construct an object of type SiteVTOLPoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteVTOLPoint)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteVTOLPoint, [SiteVTOLPoint, ISite])

agcls.AgClassCatalog.add_catalog_entry((4843052394715389444, 16917492219472377009), SiteVTOLPoint)
agcls.AgTypeNameMap["SiteVTOLPoint"] = SiteVTOLPoint

class SiteReferenceState(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a Reference State site."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_site_method_offset = 1
    _metadata = {
        "iid_data" : (5183490168050866731, 113889888384022661),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteReferenceState)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteReferenceState._metadata, SiteReferenceState._get_as_site_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SiteReferenceState."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteReferenceState)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteReferenceState, [SiteReferenceState, ISite])

agcls.AgClassCatalog.add_catalog_entry((5669257357524304127, 16224037961252013460), SiteReferenceState)
agcls.AgTypeNameMap["SiteReferenceState"] = SiteReferenceState

class SiteSTKVehicle(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a STK Vehicle site."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_object_name_method_offset = 1
    _set_object_name_method_offset = 2
    _get_valid_object_names_method_offset = 3
    _get_as_site_method_offset = 4
    _metadata = {
        "iid_data" : (5253646862927661627, 14251339590362269831),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteSTKVehicle)

    _get_object_name_metadata = { "offset" : _get_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def object_name(self) -> str:
        """Get or set the object name to link to."""
        return self._intf.get_property(SiteSTKVehicle._metadata, SiteSTKVehicle._get_object_name_metadata)

    _set_object_name_metadata = { "offset" : _set_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @object_name.setter
    def object_name(self, value:str) -> None:
        """Get or set the object name to link to."""
        return self._intf.set_property(SiteSTKVehicle._metadata, SiteSTKVehicle._set_object_name_metadata, value)

    _get_valid_object_names_metadata = { "offset" : _get_valid_object_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def valid_object_names(self) -> list:
        """Return the valid object names."""
        return self._intf.get_property(SiteSTKVehicle._metadata, SiteSTKVehicle._get_valid_object_names_metadata)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteSTKVehicle._metadata, SiteSTKVehicle._get_as_site_metadata, OutArg())

    _property_names[object_name] = "object_name"
    _property_names[valid_object_names] = "valid_object_names"

    def __init__(self, source_object=None):
        """Construct an object of type SiteSTKVehicle."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteSTKVehicle)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteSTKVehicle, [SiteSTKVehicle, ISite])

agcls.AgClassCatalog.add_catalog_entry((5597757840433509036, 13479290176329506690), SiteSTKVehicle)
agcls.AgTypeNameMap["SiteSTKVehicle"] = SiteSTKVehicle

class SiteSuperProcedure(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a Super Procedure site."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_site_method_offset = 1
    _metadata = {
        "iid_data" : (5571997691972875262, 2757918785063417476),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteSuperProcedure)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteSuperProcedure._metadata, SiteSuperProcedure._get_as_site_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SiteSuperProcedure."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteSuperProcedure)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteSuperProcedure, [SiteSuperProcedure, ISite])

agcls.AgClassCatalog.add_catalog_entry((5371976027007327351, 12208188571124011188), SiteSuperProcedure)
agcls.AgTypeNameMap["SiteSuperProcedure"] = SiteSuperProcedure

class SiteRelativeToPrevProcedure(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a Relative to Previous Procedure site."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_bearing_mode_method_offset = 1
    _set_bearing_mode_method_offset = 2
    _get_bearing_method_offset = 3
    _set_bearing_method_offset = 4
    _get_range_method_offset = 5
    _set_range_method_offset = 6
    _get_as_site_method_offset = 7
    _metadata = {
        "iid_data" : (5759285823815417596, 3125305812269783207),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteRelativeToPrevProcedure)

    _get_bearing_mode_metadata = { "offset" : _get_bearing_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RelativeAbsoluteBearing),) }
    @property
    def bearing_mode(self) -> "RelativeAbsoluteBearing":
        """Get or set the bearing reference."""
        return self._intf.get_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._get_bearing_mode_metadata)

    _set_bearing_mode_metadata = { "offset" : _set_bearing_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RelativeAbsoluteBearing),) }
    @bearing_mode.setter
    def bearing_mode(self, value:"RelativeAbsoluteBearing") -> None:
        """Get or set the bearing reference."""
        return self._intf.set_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._set_bearing_mode_metadata, value)

    _get_bearing_metadata = { "offset" : _get_bearing_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def bearing(self) -> typing.Any:
        """Get or set the bearing to define the site."""
        return self._intf.get_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._get_bearing_metadata)

    _set_bearing_metadata = { "offset" : _set_bearing_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @bearing.setter
    def bearing(self, value:typing.Any) -> None:
        """Get or set the bearing to define the site."""
        return self._intf.set_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._set_bearing_metadata, value)

    _get_range_metadata = { "offset" : _get_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range(self) -> float:
        """Get or set the range from the previous procedure."""
        return self._intf.get_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._get_range_metadata)

    _set_range_metadata = { "offset" : _set_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @range.setter
    def range(self, value:float) -> None:
        """Get or set the range from the previous procedure."""
        return self._intf.set_property(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._set_range_metadata, value)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteRelativeToPrevProcedure._metadata, SiteRelativeToPrevProcedure._get_as_site_metadata, OutArg())

    _property_names[bearing_mode] = "bearing_mode"
    _property_names[bearing] = "bearing"
    _property_names[range] = "range"

    def __init__(self, source_object=None):
        """Construct an object of type SiteRelativeToPrevProcedure."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteRelativeToPrevProcedure)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteRelativeToPrevProcedure, [SiteRelativeToPrevProcedure, ISite])

agcls.AgClassCatalog.add_catalog_entry((4958528407379756360, 17186724269580697779), SiteRelativeToPrevProcedure)
agcls.AgTypeNameMap["SiteRelativeToPrevProcedure"] = SiteRelativeToPrevProcedure

class SiteSTKObjectWaypoint(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a STK Object Waypoint site."""

    _num_methods = 20
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_object_name_method_offset = 1
    _set_object_name_method_offset = 2
    _get_valid_object_names_method_offset = 3
    _get_min_time_method_offset = 4
    _get_waypoint_time_method_offset = 5
    _set_waypoint_time_method_offset = 6
    _get_minimize_site_procedure_time_diff_method_offset = 7
    _set_minimize_site_procedure_time_diff_method_offset = 8
    _get_max_time_method_offset = 9
    _get_offset_mode_method_offset = 10
    _set_offset_mode_method_offset = 11
    _get_bearing_method_offset = 12
    _set_bearing_method_offset = 13
    _get_use_magnetic_bearing_method_offset = 14
    _set_use_magnetic_bearing_method_offset = 15
    _get_range_method_offset = 16
    _set_range_method_offset = 17
    _get_vgt_point_method_offset = 18
    _set_vgt_point_method_offset = 19
    _get_as_site_method_offset = 20
    _metadata = {
        "iid_data" : (4734056756504829268, 17555142127045754021),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteSTKObjectWaypoint)

    _get_object_name_metadata = { "offset" : _get_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def object_name(self) -> str:
        """Get or set the object name to link to."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_object_name_metadata)

    _set_object_name_metadata = { "offset" : _set_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @object_name.setter
    def object_name(self, value:str) -> None:
        """Get or set the object name to link to."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_object_name_metadata, value)

    _get_valid_object_names_metadata = { "offset" : _get_valid_object_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def valid_object_names(self) -> list:
        """Return the valid object names."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_valid_object_names_metadata)

    _get_min_time_metadata = { "offset" : _get_min_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def min_time(self) -> typing.Any:
        """Get the earliest time that the object is available as a site."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_min_time_metadata)

    _get_waypoint_time_metadata = { "offset" : _get_waypoint_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def waypoint_time(self) -> typing.Any:
        """Get or set the time at which the object's position will be used as a waypoint."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_waypoint_time_metadata)

    _set_waypoint_time_metadata = { "offset" : _set_waypoint_time_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @waypoint_time.setter
    def waypoint_time(self, value:typing.Any) -> None:
        """Get or set the time at which the object's position will be used as a waypoint."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_waypoint_time_metadata, value)

    _get_minimize_site_procedure_time_diff_metadata = { "offset" : _get_minimize_site_procedure_time_diff_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(MinimizeSiteProcedureTimeDiff),) }
    @property
    def minimize_site_procedure_time_diff(self) -> "MinimizeSiteProcedureTimeDiff":
        """Get or set the mode to minimize the time difference between the procedure and site times."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_minimize_site_procedure_time_diff_metadata)

    _set_minimize_site_procedure_time_diff_metadata = { "offset" : _set_minimize_site_procedure_time_diff_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(MinimizeSiteProcedureTimeDiff),) }
    @minimize_site_procedure_time_diff.setter
    def minimize_site_procedure_time_diff(self, value:"MinimizeSiteProcedureTimeDiff") -> None:
        """Get or set the mode to minimize the time difference between the procedure and site times."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_minimize_site_procedure_time_diff_metadata, value)

    _get_max_time_metadata = { "offset" : _get_max_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_time(self) -> typing.Any:
        """Get the latest time that the object is available as a site."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_max_time_metadata)

    _get_offset_mode_metadata = { "offset" : _get_offset_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STKObjectWaypointOffsetMode),) }
    @property
    def offset_mode(self) -> "STKObjectWaypointOffsetMode":
        """Get or set the mode to offset the site location relative from the STK Object."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_offset_mode_metadata)

    _set_offset_mode_metadata = { "offset" : _set_offset_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STKObjectWaypointOffsetMode),) }
    @offset_mode.setter
    def offset_mode(self, value:"STKObjectWaypointOffsetMode") -> None:
        """Get or set the mode to offset the site location relative from the STK Object."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_offset_mode_metadata, value)

    _get_bearing_metadata = { "offset" : _get_bearing_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def bearing(self) -> typing.Any:
        """Get or set the bearing offset of the site location relative to the object's position."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_bearing_metadata)

    _set_bearing_metadata = { "offset" : _set_bearing_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @bearing.setter
    def bearing(self, value:typing.Any) -> None:
        """Get or set the bearing offset of the site location relative to the object's position."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_bearing_metadata, value)

    _get_use_magnetic_bearing_metadata = { "offset" : _get_use_magnetic_bearing_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_magnetic_bearing(self) -> bool:
        """Get or set the option to use a magnetic heading for the bearing."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_use_magnetic_bearing_metadata)

    _set_use_magnetic_bearing_metadata = { "offset" : _set_use_magnetic_bearing_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_magnetic_bearing.setter
    def use_magnetic_bearing(self, value:bool) -> None:
        """Get or set the option to use a magnetic heading for the bearing."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_use_magnetic_bearing_metadata, value)

    _get_range_metadata = { "offset" : _get_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range(self) -> float:
        """Get or set the range offset of the site location relative to the object's position."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_range_metadata)

    _set_range_metadata = { "offset" : _set_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @range.setter
    def range(self, value:float) -> None:
        """Get or set the range offset of the site location relative to the object's position."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_range_metadata, value)

    _get_vgt_point_metadata = { "offset" : _get_vgt_point_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def vgt_point(self) -> str:
        """Get or set the reference VGT Point."""
        return self._intf.get_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_vgt_point_metadata)

    _set_vgt_point_metadata = { "offset" : _set_vgt_point_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @vgt_point.setter
    def vgt_point(self, value:str) -> None:
        """Get or set the reference VGT Point."""
        return self._intf.set_property(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._set_vgt_point_metadata, value)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteSTKObjectWaypoint._metadata, SiteSTKObjectWaypoint._get_as_site_metadata, OutArg())

    _property_names[object_name] = "object_name"
    _property_names[valid_object_names] = "valid_object_names"
    _property_names[min_time] = "min_time"
    _property_names[waypoint_time] = "waypoint_time"
    _property_names[minimize_site_procedure_time_diff] = "minimize_site_procedure_time_diff"
    _property_names[max_time] = "max_time"
    _property_names[offset_mode] = "offset_mode"
    _property_names[bearing] = "bearing"
    _property_names[use_magnetic_bearing] = "use_magnetic_bearing"
    _property_names[range] = "range"
    _property_names[vgt_point] = "vgt_point"

    def __init__(self, source_object=None):
        """Construct an object of type SiteSTKObjectWaypoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteSTKObjectWaypoint)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteSTKObjectWaypoint, [SiteSTKObjectWaypoint, ISite])

agcls.AgClassCatalog.add_catalog_entry((4708768213832241837, 15242180819678770347), SiteSTKObjectWaypoint)
agcls.AgTypeNameMap["SiteSTKObjectWaypoint"] = SiteSTKObjectWaypoint

class SiteSTKStaticObject(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a STK Static Object site."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_object_name_method_offset = 1
    _set_object_name_method_offset = 2
    _get_valid_object_names_method_offset = 3
    _get_as_site_method_offset = 4
    _metadata = {
        "iid_data" : (5314851912555770583, 18348917179993581711),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteSTKStaticObject)

    _get_object_name_metadata = { "offset" : _get_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def object_name(self) -> str:
        """Get or set the object name to link to."""
        return self._intf.get_property(SiteSTKStaticObject._metadata, SiteSTKStaticObject._get_object_name_metadata)

    _set_object_name_metadata = { "offset" : _set_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @object_name.setter
    def object_name(self, value:str) -> None:
        """Get or set the object name to link to."""
        return self._intf.set_property(SiteSTKStaticObject._metadata, SiteSTKStaticObject._set_object_name_metadata, value)

    _get_valid_object_names_metadata = { "offset" : _get_valid_object_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def valid_object_names(self) -> list:
        """Return the valid object names."""
        return self._intf.get_property(SiteSTKStaticObject._metadata, SiteSTKStaticObject._get_valid_object_names_metadata)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteSTKStaticObject._metadata, SiteSTKStaticObject._get_as_site_metadata, OutArg())

    _property_names[object_name] = "object_name"
    _property_names[valid_object_names] = "valid_object_names"

    def __init__(self, source_object=None):
        """Construct an object of type SiteSTKStaticObject."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteSTKStaticObject)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteSTKStaticObject, [SiteSTKStaticObject, ISite])

agcls.AgClassCatalog.add_catalog_entry((5632434088179407466, 7029154613056126903), SiteSTKStaticObject)
agcls.AgTypeNameMap["SiteSTKStaticObject"] = SiteSTKStaticObject

class SiteRelativeToSTKObject(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a Relative to Stationary STK Object site."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_object_name_method_offset = 1
    _set_object_name_method_offset = 2
    _get_valid_object_names_method_offset = 3
    _get_bearing_method_offset = 4
    _set_bearing_method_offset = 5
    _get_use_magnetic_bearing_method_offset = 6
    _set_use_magnetic_bearing_method_offset = 7
    _get_range_method_offset = 8
    _set_range_method_offset = 9
    _get_as_site_method_offset = 10
    _metadata = {
        "iid_data" : (5151549297227299121, 9177416622143414688),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteRelativeToSTKObject)

    _get_object_name_metadata = { "offset" : _get_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def object_name(self) -> str:
        """Get or set the object name to link to."""
        return self._intf.get_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_object_name_metadata)

    _set_object_name_metadata = { "offset" : _set_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @object_name.setter
    def object_name(self, value:str) -> None:
        """Get or set the object name to link to."""
        return self._intf.set_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._set_object_name_metadata, value)

    _get_valid_object_names_metadata = { "offset" : _get_valid_object_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def valid_object_names(self) -> list:
        """Return the valid object names."""
        return self._intf.get_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_valid_object_names_metadata)

    _get_bearing_metadata = { "offset" : _get_bearing_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def bearing(self) -> typing.Any:
        """Get or set the bearing from the STK object."""
        return self._intf.get_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_bearing_metadata)

    _set_bearing_metadata = { "offset" : _set_bearing_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @bearing.setter
    def bearing(self, value:typing.Any) -> None:
        """Get or set the bearing from the STK object."""
        return self._intf.set_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._set_bearing_metadata, value)

    _get_use_magnetic_bearing_metadata = { "offset" : _get_use_magnetic_bearing_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_magnetic_bearing(self) -> bool:
        """Get or set the option to use a magnetic bearing."""
        return self._intf.get_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_use_magnetic_bearing_metadata)

    _set_use_magnetic_bearing_metadata = { "offset" : _set_use_magnetic_bearing_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_magnetic_bearing.setter
    def use_magnetic_bearing(self, value:bool) -> None:
        """Get or set the option to use a magnetic bearing."""
        return self._intf.set_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._set_use_magnetic_bearing_metadata, value)

    _get_range_metadata = { "offset" : _get_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range(self) -> float:
        """Get or set the range from the STK object."""
        return self._intf.get_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_range_metadata)

    _set_range_metadata = { "offset" : _set_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @range.setter
    def range(self, value:float) -> None:
        """Get or set the range from the STK object."""
        return self._intf.set_property(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._set_range_metadata, value)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteRelativeToSTKObject._metadata, SiteRelativeToSTKObject._get_as_site_metadata, OutArg())

    _property_names[object_name] = "object_name"
    _property_names[valid_object_names] = "valid_object_names"
    _property_names[bearing] = "bearing"
    _property_names[use_magnetic_bearing] = "use_magnetic_bearing"
    _property_names[range] = "range"

    def __init__(self, source_object=None):
        """Construct an object of type SiteRelativeToSTKObject."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteRelativeToSTKObject)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteRelativeToSTKObject, [SiteRelativeToSTKObject, ISite])

agcls.AgClassCatalog.add_catalog_entry((5732220974452340660, 18011894982964506785), SiteRelativeToSTKObject)
agcls.AgTypeNameMap["SiteRelativeToSTKObject"] = SiteRelativeToSTKObject

class SiteSTKAreaTarget(ISite, SupportsDeleteCallback):
    """Interface used to access the options for a STK Area Target site."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_object_name_method_offset = 1
    _set_object_name_method_offset = 2
    _get_valid_object_names_method_offset = 3
    _get_as_site_method_offset = 4
    _metadata = {
        "iid_data" : (4695342907041492328, 2553066883356031928),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteSTKAreaTarget)

    _get_object_name_metadata = { "offset" : _get_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def object_name(self) -> str:
        """Get or set the object name to link to."""
        return self._intf.get_property(SiteSTKAreaTarget._metadata, SiteSTKAreaTarget._get_object_name_metadata)

    _set_object_name_metadata = { "offset" : _set_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @object_name.setter
    def object_name(self, value:str) -> None:
        """Get or set the object name to link to."""
        return self._intf.set_property(SiteSTKAreaTarget._metadata, SiteSTKAreaTarget._set_object_name_metadata, value)

    _get_valid_object_names_metadata = { "offset" : _get_valid_object_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def valid_object_names(self) -> list:
        """Return the valid object names."""
        return self._intf.get_property(SiteSTKAreaTarget._metadata, SiteSTKAreaTarget._get_valid_object_names_metadata)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteSTKAreaTarget._metadata, SiteSTKAreaTarget._get_as_site_metadata, OutArg())

    _property_names[object_name] = "object_name"
    _property_names[valid_object_names] = "valid_object_names"

    def __init__(self, source_object=None):
        """Construct an object of type SiteSTKAreaTarget."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteSTKAreaTarget)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteSTKAreaTarget, [SiteSTKAreaTarget, ISite])

agcls.AgClassCatalog.add_catalog_entry((4643775838845076022, 17375698151001436338), SiteSTKAreaTarget)
agcls.AgTypeNameMap["SiteSTKAreaTarget"] = SiteSTKAreaTarget

class SiteRunway(ISite, SupportsDeleteCallback):
    """
    Interface used to access the options for a Runway site type.

    Examples
    --------
    Configure a runway site:
    >>> # SiteRunway runway: Runway object
    >>> # Set the latitude, longitude, and altitude
    >>> runway.latitude = 41
    >>> runway.longitude = 77
    >>> runway.altitude = 5
    >>>
    >>> # Set the altitude reference
    >>> runway.altitude_reference = AGLMSL.ALTITUDE_MSL
    >>>
    >>> # Set the heading
    >>> runway.high_end_heading = 195
    >>> # Opt to use true heading
    >>> runway.is_magnetic = False
    >>>
    >>> # Set the length of the runway
    >>> runway.length = 5
    >>>
    >>> # Rename the runway
    >>> runway.name = "New User Runway"
    >>> # Add the runway to the catalog to use it for next time
    >>> runway.add_to_catalog(1)

    Configure a runway site from a runway in the Aviator catalog:
    >>> # SiteRunway runway: Runway object
    >>> # Catalog catalog: Aviator catalog object
    >>> # Get the source of user runways
    >>> userRunways = catalog.runway_category.user_runways
    >>> # Check that the runway exists in the catalog
    >>> if userRunways.contains("New User Runway") is True:
    >>>     # If so, get the user runway with the given name
    >>>     runwayFromCatalog = userRunways.get_user_runway("New User Runway")
    >>>     # Copy the parameters of that runway
    >>>     runway.copy_from_catalog(runwayFromCatalog)
    """

    _num_methods = 19
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_method_offset = 1
    _set_altitude_method_offset = 2
    _get_latitude_method_offset = 3
    _set_latitude_method_offset = 4
    _get_longitude_method_offset = 5
    _set_longitude_method_offset = 6
    _get_length_method_offset = 7
    _set_length_method_offset = 8
    _get_altitude_reference_method_offset = 9
    _set_altitude_reference_method_offset = 10
    _get_low_end_heading_method_offset = 11
    _set_low_end_heading_method_offset = 12
    _get_high_end_heading_method_offset = 13
    _set_high_end_heading_method_offset = 14
    _get_is_magnetic_method_offset = 15
    _set_is_magnetic_method_offset = 16
    _add_to_catalog_method_offset = 17
    _copy_from_catalog_method_offset = 18
    _get_as_site_method_offset = 19
    _metadata = {
        "iid_data" : (5308502759604380100, 9664305760779056786),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SiteRunway)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the runway altitude."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Get or set the runway altitude."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_altitude_metadata, value)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the runway latitude."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        """Get or set the runway latitude."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the runway longitude."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        """Get or set the runway longitude."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_longitude_metadata, value)

    _get_length_metadata = { "offset" : _get_length_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def length(self) -> float:
        """Get or set the length of the runway."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_length_metadata)

    _set_length_metadata = { "offset" : _set_length_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @length.setter
    def length(self, value:float) -> None:
        """Get or set the length of the runway."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_length_metadata, value)

    _get_altitude_reference_metadata = { "offset" : _get_altitude_reference_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @property
    def altitude_reference(self) -> "AGLMSL":
        """Get or set the altitude reference for the runway."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_altitude_reference_metadata)

    _set_altitude_reference_metadata = { "offset" : _set_altitude_reference_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @altitude_reference.setter
    def altitude_reference(self, value:"AGLMSL") -> None:
        """Get or set the altitude reference for the runway."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_altitude_reference_metadata, value)

    _get_low_end_heading_metadata = { "offset" : _get_low_end_heading_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def low_end_heading(self) -> typing.Any:
        """Get or set the low end heading of the runway."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_low_end_heading_metadata)

    _set_low_end_heading_metadata = { "offset" : _set_low_end_heading_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @low_end_heading.setter
    def low_end_heading(self, value:typing.Any) -> None:
        """Get or set the low end heading of the runway."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_low_end_heading_metadata, value)

    _get_high_end_heading_metadata = { "offset" : _get_high_end_heading_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def high_end_heading(self) -> typing.Any:
        """Get or set the high end heading of the runway."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_high_end_heading_metadata)

    _set_high_end_heading_metadata = { "offset" : _set_high_end_heading_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @high_end_heading.setter
    def high_end_heading(self, value:typing.Any) -> None:
        """Get or set the high end heading of the runway."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_high_end_heading_metadata, value)

    _get_is_magnetic_metadata = { "offset" : _get_is_magnetic_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_magnetic(self) -> bool:
        """Opt whether to use a magnetic heading for the runway heading."""
        return self._intf.get_property(SiteRunway._metadata, SiteRunway._get_is_magnetic_metadata)

    _set_is_magnetic_metadata = { "offset" : _set_is_magnetic_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @is_magnetic.setter
    def is_magnetic(self, value:bool) -> None:
        """Opt whether to use a magnetic heading for the runway heading."""
        return self._intf.set_property(SiteRunway._metadata, SiteRunway._set_is_magnetic_metadata, value)

    _add_to_catalog_metadata = { "offset" : _add_to_catalog_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    def add_to_catalog(self, overwrite:bool) -> None:
        """Add the runway to the catalog."""
        return self._intf.invoke(SiteRunway._metadata, SiteRunway._add_to_catalog_metadata, overwrite)

    _copy_from_catalog_metadata = { "offset" : _copy_from_catalog_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICatalogRunway"),) }
    def copy_from_catalog(self, runway:"ICatalogRunway") -> None:
        """Copy the information from the runway stored in the catalog."""
        return self._intf.invoke(SiteRunway._metadata, SiteRunway._copy_from_catalog_metadata, runway)

    _get_as_site_metadata = { "offset" : _get_as_site_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_site(self) -> "ISite":
        """Get the site interface."""
        return self._intf.invoke(SiteRunway._metadata, SiteRunway._get_as_site_metadata, OutArg())

    _property_names[altitude] = "altitude"
    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[length] = "length"
    _property_names[altitude_reference] = "altitude_reference"
    _property_names[low_end_heading] = "low_end_heading"
    _property_names[high_end_heading] = "high_end_heading"
    _property_names[is_magnetic] = "is_magnetic"

    def __init__(self, source_object=None):
        """Construct an object of type SiteRunway."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SiteRunway)
        ISite.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SiteRunway, [SiteRunway, ISite])

agcls.AgClassCatalog.add_catalog_entry((5138752020114409457, 17128394562080548993), SiteRunway)
agcls.AgTypeNameMap["SiteRunway"] = SiteRunway

class Site(ISite, ISiteUnknown, SupportsDeleteCallback):
    """Class defining an unknown site type."""
    def __init__(self, source_object=None):
        """Construct an object of type Site."""
        SupportsDeleteCallback.__init__(self)
        ISite.__init__(self, source_object)
        ISiteUnknown.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISite._private_init(self, intf)
        ISiteUnknown._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Site, [ISite, ISiteUnknown])

agcls.AgClassCatalog.add_catalog_entry((4930703979729104237, 439084692498343612), Site)
agcls.AgTypeNameMap["Site"] = Site

class ProcedureLanding(IProcedure, SupportsDeleteCallback):
    """
    Interface used to access the options for a landing procedure.

    Examples
    --------
    Add and configure a landing procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a landing procedure
    >>> landing = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_LANDING)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = landing.runway_heading_options
    >>> # Land from the low end
    >>> headingOptions.runway_mode = RunwayHighLowEnd.LOW_END
    >>>
    >>> # Use a standard instrument approach
    >>> landing.approach_mode = ApproachMode.STANDARD_INSTRUMENT_APPROACH
    >>> # Get the options for a standard instrument approach
    >>> sia = landing.mode_as_standard_instrument_approach
    >>> # Change the approach altitude
    >>> sia.approach_altitude = 1000
    >>> # Change the glideslope
    >>> sia.glideslope = 4
    >>> # Offset the runway altitude
    >>> sia.runway_altitude_offset = 10
    >>> # Use the terrain as an altitude reference for the runway
    >>> sia.use_runway_terrain = True
    """

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_mode_as_standard_instrument_approach_method_offset = 1
    _get_mode_as_intercept_glideslope_method_offset = 2
    _get_mode_as_enter_downwind_pattern_method_offset = 3
    _get_runway_heading_options_method_offset = 4
    _get_enroute_cruise_airspeed_options_method_offset = 5
    _get_enroute_turn_direction_options_method_offset = 6
    _get_enroute_options_method_offset = 7
    _get_vertical_plane_options_method_offset = 8
    _get_approach_mode_method_offset = 9
    _set_approach_mode_method_offset = 10
    _get_as_procedure_method_offset = 11
    _metadata = {
        "iid_data" : (5173821936820820516, 1444667070054508934),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureLanding)

    _get_mode_as_standard_instrument_approach_metadata = { "offset" : _get_mode_as_standard_instrument_approach_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_standard_instrument_approach(self) -> "LandingStandardInstrumentApproach":
        """Get the interface for a standard instrument approach landing."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_mode_as_standard_instrument_approach_metadata)

    _get_mode_as_intercept_glideslope_metadata = { "offset" : _get_mode_as_intercept_glideslope_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_intercept_glideslope(self) -> "LandingInterceptGlideslope":
        """Get the interface for an intercept glideslope landing."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_mode_as_intercept_glideslope_metadata)

    _get_mode_as_enter_downwind_pattern_metadata = { "offset" : _get_mode_as_enter_downwind_pattern_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_enter_downwind_pattern(self) -> "LandingEnterDownwindPattern":
        """Get the interface for a downwind pattern landing."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_mode_as_enter_downwind_pattern_metadata)

    _get_runway_heading_options_metadata = { "offset" : _get_runway_heading_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def runway_heading_options(self) -> "RunwayHeadingOptions":
        """Get the runway heading options."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_runway_heading_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "ICruiseAirspeedAndProfileOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_enroute_cruise_airspeed_options_metadata)

    _get_enroute_turn_direction_options_metadata = { "offset" : _get_enroute_turn_direction_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_turn_direction_options(self) -> "EnrouteTurnDirectionOptions":
        """Get the enroute turn direction options."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_enroute_turn_direction_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "IEnrouteAndDelayOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_enroute_options_metadata)

    _get_vertical_plane_options_metadata = { "offset" : _get_vertical_plane_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vertical_plane_options(self) -> "IVerticalPlaneOptions":
        """Get the vertical plane options."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_vertical_plane_options_metadata)

    _get_approach_mode_metadata = { "offset" : _get_approach_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ApproachMode),) }
    @property
    def approach_mode(self) -> "ApproachMode":
        """Get or set the type of landing the aircraft will perform."""
        return self._intf.get_property(ProcedureLanding._metadata, ProcedureLanding._get_approach_mode_metadata)

    _set_approach_mode_metadata = { "offset" : _set_approach_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ApproachMode),) }
    @approach_mode.setter
    def approach_mode(self, value:"ApproachMode") -> None:
        """Get or set the type of landing the aircraft will perform."""
        return self._intf.set_property(ProcedureLanding._metadata, ProcedureLanding._set_approach_mode_metadata, value)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureLanding._metadata, ProcedureLanding._get_as_procedure_metadata, OutArg())

    _property_names[mode_as_standard_instrument_approach] = "mode_as_standard_instrument_approach"
    _property_names[mode_as_intercept_glideslope] = "mode_as_intercept_glideslope"
    _property_names[mode_as_enter_downwind_pattern] = "mode_as_enter_downwind_pattern"
    _property_names[runway_heading_options] = "runway_heading_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"
    _property_names[enroute_turn_direction_options] = "enroute_turn_direction_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[vertical_plane_options] = "vertical_plane_options"
    _property_names[approach_mode] = "approach_mode"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureLanding."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureLanding)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureLanding, [ProcedureLanding, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5217645111817093152, 16954556210728333749), ProcedureLanding)
agcls.AgTypeNameMap["ProcedureLanding"] = ProcedureLanding

class ProcedureEnroute(IProcedure, SupportsDeleteCallback):
    """
    Interface used to access the options for an enroute procedure.

    Examples
    --------
    Add and configure an en-route procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add an enroute procedure with a site type of End of Previous Procedure
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>> # Get the altitude options
    >>> altitudeOptions = enroute.altitude_msl_options
    >>> # To specify an altitude, turn off the option to use the default cruise altitude
    >>> altitudeOptions.use_default_cruise_altitude = False
    >>> # Set the altitude
    >>> altitudeOptions.msl_altitude = 10000
    >>>
    >>> # Get the navigation options
    >>> navigationOptions = enroute.navigation_options
    >>> # Set the route to arrive on a specified course
    >>> navigationOptions.navigation_mode = PointToPointMode.ARRIVE_ON_COURSE
    >>> # Set the course
    >>> navigationOptions.arrive_on_course = 30
    >>> # Use a magnetic heading
    >>> navigationOptions.use_magnetic_heading = True
    >>>
    >>> # Get the navigation options
    >>> airspeedOptions = enroute.enroute_cruise_airspeed_options
    >>> # Fly at max speed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.MAX_AIRSPEED
    >>> # To specify an airspeed to fly at, set the speed type to other airspeed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.OTHER_AIRSPEED
    >>> # Then set the airspeed and airspeed type
    >>> airspeedOptions.set_other_airspeed(AirspeedType.TAS, 200)
    """

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_msl_options_method_offset = 1
    _get_navigation_options_method_offset = 2
    _get_enroute_options_method_offset = 3
    _get_enroute_cruise_airspeed_options_method_offset = 4
    _get_as_procedure_method_offset = 5
    _metadata = {
        "iid_data" : (5434799576980412844, 10892687876198075050),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureEnroute)

    _get_altitude_msl_options_metadata = { "offset" : _get_altitude_msl_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_msl_options(self) -> "AltitudeMSLAndLevelOffOptions":
        """Get the altitude MSL options."""
        return self._intf.get_property(ProcedureEnroute._metadata, ProcedureEnroute._get_altitude_msl_options_metadata)

    _get_navigation_options_metadata = { "offset" : _get_navigation_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def navigation_options(self) -> "NavigationOptions":
        """Get the navigation options."""
        return self._intf.get_property(ProcedureEnroute._metadata, ProcedureEnroute._get_navigation_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "IEnrouteAndDelayOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureEnroute._metadata, ProcedureEnroute._get_enroute_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureEnroute._metadata, ProcedureEnroute._get_enroute_cruise_airspeed_options_metadata)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureEnroute._metadata, ProcedureEnroute._get_as_procedure_metadata, OutArg())

    _property_names[altitude_msl_options] = "altitude_msl_options"
    _property_names[navigation_options] = "navigation_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureEnroute."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureEnroute)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureEnroute, [ProcedureEnroute, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5519935132979626266, 10381875481096635325), ProcedureEnroute)
agcls.AgTypeNameMap["ProcedureEnroute"] = ProcedureEnroute

class ProcedureExtEphem(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for an ExtEphem procedure."""

    _num_methods = 29
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_ephemeris_file_method_offset = 1
    _get_ephemeris_file_method_offset = 2
    _get_ephemeris_file_duration_method_offset = 3
    _set_use_start_duration_method_offset = 4
    _get_use_start_duration_method_offset = 5
    _set_start_time_method_offset = 6
    _get_start_time_method_offset = 7
    _set_duration_method_offset = 8
    _get_duration_method_offset = 9
    _get_flight_mode_method_offset = 10
    _set_flight_mode_method_offset = 11
    _get_as_procedure_method_offset = 12
    _set_use_shift_rotate_method_offset = 13
    _get_use_shift_rotate_method_offset = 14
    _get_shift_time_method_offset = 15
    _set_shift_time_method_offset = 16
    _get_latitude_method_offset = 17
    _set_latitude_method_offset = 18
    _get_longitude_method_offset = 19
    _set_longitude_method_offset = 20
    _get_altitude_method_offset = 21
    _set_altitude_method_offset = 22
    _get_course_method_offset = 23
    _set_course_method_offset = 24
    _get_course_mode_method_offset = 25
    _set_course_mode_method_offset = 26
    _get_altitude_mode_method_offset = 27
    _set_altitude_mode_method_offset = 28
    _shift_rotate_set_method_offset = 29
    _metadata = {
        "iid_data" : (4706318164488420241, 10859321374984907949),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureExtEphem)

    _get_ephemeris_file_metadata = { "offset" : _get_ephemeris_file_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def ephemeris_file(self) -> str:
        """Get the ephemeris filename."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_ephemeris_file_metadata)

    _set_ephemeris_file_metadata = { "offset" : _set_ephemeris_file_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @ephemeris_file.setter
    def ephemeris_file(self, value:str) -> None:
        """Set the ephemeris filename."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_ephemeris_file_metadata, value)

    _get_ephemeris_file_duration_metadata = { "offset" : _get_ephemeris_file_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def ephemeris_file_duration(self) -> float:
        """Get the ephemeris file duration."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_ephemeris_file_duration_metadata)

    _get_use_start_duration_metadata = { "offset" : _get_use_start_duration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_start_duration(self) -> bool:
        """Get whether to use Start and Duration."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_use_start_duration_metadata)

    _set_use_start_duration_metadata = { "offset" : _set_use_start_duration_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_start_duration.setter
    def use_start_duration(self, use_sd:bool) -> None:
        """Set whether to use Start and Duration."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_use_start_duration_metadata, use_sd)

    _get_start_time_metadata = { "offset" : _get_start_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_time(self) -> float:
        """Get the start time."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_start_time_metadata)

    _set_start_time_metadata = { "offset" : _set_start_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_time.setter
    def start_time(self, start:float) -> None:
        """Set the start time."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_start_time_metadata, start)

    _get_duration_metadata = { "offset" : _get_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def duration(self) -> float:
        """Get the duration."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_duration_metadata)

    _set_duration_metadata = { "offset" : _set_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @duration.setter
    def duration(self, start:float) -> None:
        """Set the duration."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_duration_metadata, start)

    _get_flight_mode_metadata = { "offset" : _get_flight_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ExtEphemFlightMode),) }
    @property
    def flight_mode(self) -> "ExtEphemFlightMode":
        """Get the flight mode."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_flight_mode_metadata)

    _set_flight_mode_metadata = { "offset" : _set_flight_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ExtEphemFlightMode),) }
    @flight_mode.setter
    def flight_mode(self, mode:"ExtEphemFlightMode") -> None:
        """Set the flight mode."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_flight_mode_metadata, mode)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureExtEphem._metadata, ProcedureExtEphem._get_as_procedure_metadata, OutArg())

    _get_use_shift_rotate_metadata = { "offset" : _get_use_shift_rotate_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_shift_rotate(self) -> bool:
        """Get whether to use Shift/Rotate."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_use_shift_rotate_metadata)

    _set_use_shift_rotate_metadata = { "offset" : _set_use_shift_rotate_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_shift_rotate.setter
    def use_shift_rotate(self, use_sd:bool) -> None:
        """Set whether to use Shift/Rotate."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_use_shift_rotate_metadata, use_sd)

    _get_shift_time_metadata = { "offset" : _get_shift_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def shift_time(self) -> float:
        """Get the shift time."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_shift_time_metadata)

    _set_shift_time_metadata = { "offset" : _set_shift_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @shift_time.setter
    def shift_time(self, value:float) -> None:
        """Set the shift time."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_shift_time_metadata, value)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def latitude(self) -> float:
        """Get the waypoint latitude."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @latitude.setter
    def latitude(self, value:float) -> None:
        """Set the waypoint latitude."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def longitude(self) -> float:
        """Get the waypoint longitude."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @longitude.setter
    def longitude(self, value:float) -> None:
        """Set the waypoint longitude."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_longitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get the altitude."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Set the altitude."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_altitude_metadata, value)

    _get_course_metadata = { "offset" : _get_course_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def course(self) -> float:
        """Get the course."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_course_metadata)

    _set_course_metadata = { "offset" : _set_course_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @course.setter
    def course(self, value:float) -> None:
        """Set the course."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_course_metadata, value)

    _get_course_mode_metadata = { "offset" : _get_course_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EphemShiftRotateCourseMode),) }
    @property
    def course_mode(self) -> "EphemShiftRotateCourseMode":
        """Get the course mode."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_course_mode_metadata)

    _set_course_mode_metadata = { "offset" : _set_course_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EphemShiftRotateCourseMode),) }
    @course_mode.setter
    def course_mode(self, altmode:"EphemShiftRotateCourseMode") -> None:
        """Set the course mode."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_course_mode_metadata, altmode)

    _get_altitude_mode_metadata = { "offset" : _get_altitude_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EphemShiftRotateAltitudeMode),) }
    @property
    def altitude_mode(self) -> "EphemShiftRotateAltitudeMode":
        """Get the alt mode."""
        return self._intf.get_property(ProcedureExtEphem._metadata, ProcedureExtEphem._get_altitude_mode_metadata)

    _set_altitude_mode_metadata = { "offset" : _set_altitude_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EphemShiftRotateAltitudeMode),) }
    @altitude_mode.setter
    def altitude_mode(self, altmode:"EphemShiftRotateAltitudeMode") -> None:
        """Set the alt mode."""
        return self._intf.set_property(ProcedureExtEphem._metadata, ProcedureExtEphem._set_altitude_mode_metadata, altmode)

    _shift_rotate_set_metadata = { "offset" : _shift_rotate_set_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def shift_rotate_set(self) -> None:
        """Shift rotate set values."""
        return self._intf.invoke(ProcedureExtEphem._metadata, ProcedureExtEphem._shift_rotate_set_metadata, )

    _property_names[ephemeris_file] = "ephemeris_file"
    _property_names[ephemeris_file_duration] = "ephemeris_file_duration"
    _property_names[use_start_duration] = "use_start_duration"
    _property_names[start_time] = "start_time"
    _property_names[duration] = "duration"
    _property_names[flight_mode] = "flight_mode"
    _property_names[use_shift_rotate] = "use_shift_rotate"
    _property_names[shift_time] = "shift_time"
    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[altitude] = "altitude"
    _property_names[course] = "course"
    _property_names[course_mode] = "course_mode"
    _property_names[altitude_mode] = "altitude_mode"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureExtEphem."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureExtEphem)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureExtEphem, [ProcedureExtEphem, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5229605635028001773, 1223198393308242336), ProcedureExtEphem)
agcls.AgTypeNameMap["ProcedureExtEphem"] = ProcedureExtEphem

class ProcedureFormationFlyer(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for an enroute procedure."""

    _num_methods = 19
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_min_time_step_method_offset = 1
    _set_min_time_step_method_offset = 2
    _get_max_time_step_method_offset = 3
    _set_max_time_step_method_offset = 4
    _get_cross_range_close_rate_method_offset = 5
    _set_cross_range_close_rate_method_offset = 6
    _get_initial_close_max_speed_advantage_method_offset = 7
    _set_initial_close_max_speed_advantage_method_offset = 8
    _get_stop_condition_method_offset = 9
    _set_stop_condition_method_offset = 10
    _get_stop_time_method_offset = 11
    _set_stop_time_method_offset = 12
    _get_stop_down_range_method_offset = 13
    _set_stop_down_range_method_offset = 14
    _get_stop_fuel_state_method_offset = 15
    _set_stop_fuel_state_method_offset = 16
    _get_as_procedure_method_offset = 17
    _set_stop_on_hover_method_offset = 18
    _get_stop_on_hover_method_offset = 19
    _metadata = {
        "iid_data" : (4637041047160961161, 14166705763825796783),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureFormationFlyer)

    _get_min_time_step_metadata = { "offset" : _get_min_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_time_step(self) -> float:
        """Get Min time Step."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_min_time_step_metadata)

    _set_min_time_step_metadata = { "offset" : _set_min_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_time_step.setter
    def min_time_step(self, value:float) -> None:
        """Set Min time Step."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_min_time_step_metadata, value)

    _get_max_time_step_metadata = { "offset" : _get_max_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_time_step(self) -> float:
        """Get Max time Step."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_max_time_step_metadata)

    _set_max_time_step_metadata = { "offset" : _set_max_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_time_step.setter
    def max_time_step(self, value:float) -> None:
        """Set Max time Step."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_max_time_step_metadata, value)

    _get_cross_range_close_rate_metadata = { "offset" : _get_cross_range_close_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cross_range_close_rate(self) -> float:
        """Get Cross Range close rate."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_cross_range_close_rate_metadata)

    _set_cross_range_close_rate_metadata = { "offset" : _set_cross_range_close_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cross_range_close_rate.setter
    def cross_range_close_rate(self, value:float) -> None:
        """Set Cross Range close rate."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_cross_range_close_rate_metadata, value)

    _get_initial_close_max_speed_advantage_metadata = { "offset" : _get_initial_close_max_speed_advantage_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def initial_close_max_speed_advantage(self) -> float:
        """Get Initial Close Max Speed Advantage."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_initial_close_max_speed_advantage_metadata)

    _set_initial_close_max_speed_advantage_metadata = { "offset" : _set_initial_close_max_speed_advantage_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @initial_close_max_speed_advantage.setter
    def initial_close_max_speed_advantage(self, value:float) -> None:
        """Set Initial Close Max Speed Advantage."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_initial_close_max_speed_advantage_metadata, value)

    _get_stop_condition_metadata = { "offset" : _get_stop_condition_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FormationFlyerStopCondition),) }
    @property
    def stop_condition(self) -> "FormationFlyerStopCondition":
        """Get Stop condition."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_stop_condition_metadata)

    _set_stop_condition_metadata = { "offset" : _set_stop_condition_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FormationFlyerStopCondition),) }
    @stop_condition.setter
    def stop_condition(self, value:"FormationFlyerStopCondition") -> None:
        """Set Stop condition."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_stop_condition_metadata, value)

    _get_stop_time_metadata = { "offset" : _get_stop_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_time(self) -> float:
        """Get stop time."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_stop_time_metadata)

    _set_stop_time_metadata = { "offset" : _set_stop_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_time.setter
    def stop_time(self, value:float) -> None:
        """Set stop time."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_stop_time_metadata, value)

    _get_stop_down_range_metadata = { "offset" : _get_stop_down_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_down_range(self) -> float:
        """Get stop downrange."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_stop_down_range_metadata)

    _set_stop_down_range_metadata = { "offset" : _set_stop_down_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_down_range.setter
    def stop_down_range(self, value:float) -> None:
        """Set stop downrange."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_stop_down_range_metadata, value)

    _get_stop_fuel_state_metadata = { "offset" : _get_stop_fuel_state_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_fuel_state(self) -> float:
        """Get stop fuel state."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_stop_fuel_state_metadata)

    _set_stop_fuel_state_metadata = { "offset" : _set_stop_fuel_state_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_fuel_state.setter
    def stop_fuel_state(self, value:float) -> None:
        """Set stop fuel state."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_stop_fuel_state_metadata, value)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_as_procedure_metadata, OutArg())

    _get_stop_on_hover_metadata = { "offset" : _get_stop_on_hover_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def stop_on_hover(self) -> bool:
        """Get Stop on hover mode."""
        return self._intf.get_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._get_stop_on_hover_metadata)

    _set_stop_on_hover_metadata = { "offset" : _set_stop_on_hover_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @stop_on_hover.setter
    def stop_on_hover(self, stop_on_hover:bool) -> None:
        """Set Stop on hover mode."""
        return self._intf.set_property(ProcedureFormationFlyer._metadata, ProcedureFormationFlyer._set_stop_on_hover_metadata, stop_on_hover)

    _property_names[min_time_step] = "min_time_step"
    _property_names[max_time_step] = "max_time_step"
    _property_names[cross_range_close_rate] = "cross_range_close_rate"
    _property_names[initial_close_max_speed_advantage] = "initial_close_max_speed_advantage"
    _property_names[stop_condition] = "stop_condition"
    _property_names[stop_time] = "stop_time"
    _property_names[stop_down_range] = "stop_down_range"
    _property_names[stop_fuel_state] = "stop_fuel_state"
    _property_names[stop_on_hover] = "stop_on_hover"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureFormationFlyer."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureFormationFlyer)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureFormationFlyer, [ProcedureFormationFlyer, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5626350792346175468, 8231896308722198688), ProcedureFormationFlyer)
agcls.AgTypeNameMap["ProcedureFormationFlyer"] = ProcedureFormationFlyer

class ProcedureBasicPointToPoint(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for a basic point to point procedure."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_options_method_offset = 1
    _get_navigation_options_method_offset = 2
    _get_enroute_options_method_offset = 3
    _get_enroute_cruise_airspeed_options_method_offset = 4
    _get_vertical_plane_options_method_offset = 5
    _get_as_procedure_method_offset = 6
    _metadata = {
        "iid_data" : (5190483266878387264, 613211873364248510),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureBasicPointToPoint)

    _get_altitude_options_metadata = { "offset" : _get_altitude_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_options(self) -> "AltitudeOptions":
        """Get the altitude options."""
        return self._intf.get_property(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_altitude_options_metadata)

    _get_navigation_options_metadata = { "offset" : _get_navigation_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def navigation_options(self) -> "NavigationOptions":
        """Get the navigation options."""
        return self._intf.get_property(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_navigation_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "EnrouteOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_enroute_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "ICruiseAirspeedAndProfileOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_enroute_cruise_airspeed_options_metadata)

    _get_vertical_plane_options_metadata = { "offset" : _get_vertical_plane_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vertical_plane_options(self) -> "VerticalPlaneAndFlightPathOptions":
        """Get the vertical plane options."""
        return self._intf.get_property(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_vertical_plane_options_metadata)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureBasicPointToPoint._metadata, ProcedureBasicPointToPoint._get_as_procedure_metadata, OutArg())

    _property_names[altitude_options] = "altitude_options"
    _property_names[navigation_options] = "navigation_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"
    _property_names[vertical_plane_options] = "vertical_plane_options"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureBasicPointToPoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureBasicPointToPoint)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureBasicPointToPoint, [ProcedureBasicPointToPoint, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((4658651044347023593, 13646940327219296687), ProcedureBasicPointToPoint)
agcls.AgTypeNameMap["ProcedureBasicPointToPoint"] = ProcedureBasicPointToPoint

class ProcedureArcEnroute(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for an arc enroute procedure."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_options_method_offset = 1
    _get_arc_options_method_offset = 2
    _get_arc_cruise_airspeed_options_method_offset = 3
    _get_enroute_options_method_offset = 4
    _get_enroute_cruise_airspeed_options_method_offset = 5
    _get_enroute_turn_direction_options_method_offset = 6
    _get_as_procedure_method_offset = 7
    _metadata = {
        "iid_data" : (5178802862750644614, 11742865424791326105),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureArcEnroute)

    _get_altitude_options_metadata = { "offset" : _get_altitude_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_options(self) -> "ArcAltitudeAndDelayOptions":
        """Get the altitude options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_altitude_options_metadata)

    _get_arc_options_metadata = { "offset" : _get_arc_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arc_options(self) -> "ArcOptions":
        """Get the arc options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_arc_options_metadata)

    _get_arc_cruise_airspeed_options_metadata = { "offset" : _get_arc_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arc_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the arc cruise airspeed options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_arc_cruise_airspeed_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "IEnrouteAndDelayOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_enroute_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_enroute_cruise_airspeed_options_metadata)

    _get_enroute_turn_direction_options_metadata = { "offset" : _get_enroute_turn_direction_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_turn_direction_options(self) -> "EnrouteTurnDirectionOptions":
        """Get the enroute turn direction options."""
        return self._intf.get_property(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_enroute_turn_direction_options_metadata)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureArcEnroute._metadata, ProcedureArcEnroute._get_as_procedure_metadata, OutArg())

    _property_names[altitude_options] = "altitude_options"
    _property_names[arc_options] = "arc_options"
    _property_names[arc_cruise_airspeed_options] = "arc_cruise_airspeed_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"
    _property_names[enroute_turn_direction_options] = "enroute_turn_direction_options"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureArcEnroute."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureArcEnroute)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureArcEnroute, [ProcedureArcEnroute, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5255200399109307333, 13934917253357422254), ProcedureArcEnroute)
agcls.AgTypeNameMap["ProcedureArcEnroute"] = ProcedureArcEnroute

class ProcedureArcPointToPoint(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for an arc point to point procedure."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_options_method_offset = 1
    _get_arc_options_method_offset = 2
    _get_arc_cruise_airspeed_options_method_offset = 3
    _get_enroute_options_method_offset = 4
    _get_enroute_cruise_airspeed_options_method_offset = 5
    _get_enroute_turn_direction_options_method_offset = 6
    _get_fly_cruise_airspeed_profile_method_offset = 7
    _set_fly_cruise_airspeed_profile_method_offset = 8
    _get_vertical_plane_options_method_offset = 9
    _get_as_procedure_method_offset = 10
    _metadata = {
        "iid_data" : (4627887812079693651, 17276351109473528244),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureArcPointToPoint)

    _get_altitude_options_metadata = { "offset" : _get_altitude_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_options(self) -> "ArcAltitudeOptions":
        """Get the altitude options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_altitude_options_metadata)

    _get_arc_options_metadata = { "offset" : _get_arc_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arc_options(self) -> "ArcOptions":
        """Get the arc options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_arc_options_metadata)

    _get_arc_cruise_airspeed_options_metadata = { "offset" : _get_arc_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arc_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the arc cruise airspeed options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_arc_cruise_airspeed_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "EnrouteOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_enroute_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_enroute_cruise_airspeed_options_metadata)

    _get_enroute_turn_direction_options_metadata = { "offset" : _get_enroute_turn_direction_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_turn_direction_options(self) -> "EnrouteTurnDirectionOptions":
        """Get the enroute turn direction options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_enroute_turn_direction_options_metadata)

    _get_fly_cruise_airspeed_profile_metadata = { "offset" : _get_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def fly_cruise_airspeed_profile(self) -> bool:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_fly_cruise_airspeed_profile_metadata)

    _set_fly_cruise_airspeed_profile_metadata = { "offset" : _set_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @fly_cruise_airspeed_profile.setter
    def fly_cruise_airspeed_profile(self, value:bool) -> None:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.set_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._set_fly_cruise_airspeed_profile_metadata, value)

    _get_vertical_plane_options_metadata = { "offset" : _get_vertical_plane_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vertical_plane_options(self) -> "ArcVerticalPlaneOptions":
        """Get the vertical plane options."""
        return self._intf.get_property(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_vertical_plane_options_metadata)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureArcPointToPoint._metadata, ProcedureArcPointToPoint._get_as_procedure_metadata, OutArg())

    _property_names[altitude_options] = "altitude_options"
    _property_names[arc_options] = "arc_options"
    _property_names[arc_cruise_airspeed_options] = "arc_cruise_airspeed_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"
    _property_names[enroute_turn_direction_options] = "enroute_turn_direction_options"
    _property_names[fly_cruise_airspeed_profile] = "fly_cruise_airspeed_profile"
    _property_names[vertical_plane_options] = "vertical_plane_options"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureArcPointToPoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureArcPointToPoint)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureArcPointToPoint, [ProcedureArcPointToPoint, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((4700081710069607543, 13277755077444961177), ProcedureArcPointToPoint)
agcls.AgTypeNameMap["ProcedureArcPointToPoint"] = ProcedureArcPointToPoint

class ProcedureFlightLine(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for a flight line procedure."""

    _num_methods = 20
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_options_method_offset = 1
    _get_fly_cruise_airspeed_profile_method_offset = 2
    _set_fly_cruise_airspeed_profile_method_offset = 3
    _get_flight_line_airspeed_options_method_offset = 4
    _get_enroute_options_method_offset = 5
    _get_enroute_turn_direction_options_method_offset = 6
    _get_enroute_cruise_airspeed_options_method_offset = 7
    _get_procedure_type_method_offset = 8
    _set_procedure_type_method_offset = 9
    _get_outbound_course_method_offset = 10
    _set_outbound_course_method_offset = 11
    _get_use_magnetic_heading_method_offset = 12
    _set_use_magnetic_heading_method_offset = 13
    _get_leg_length_method_offset = 14
    _set_leg_length_method_offset = 15
    _get_must_level_off_method_offset = 16
    _set_must_level_off_method_offset = 17
    _get_level_off_mode_method_offset = 18
    _set_level_off_mode_method_offset = 19
    _get_as_procedure_method_offset = 20
    _metadata = {
        "iid_data" : (5641659199997018566, 6013725867356615561),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureFlightLine)

    _get_altitude_options_metadata = { "offset" : _get_altitude_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def altitude_options(self) -> "AltitudeOptions":
        """Get the altitude options."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_altitude_options_metadata)

    _get_fly_cruise_airspeed_profile_metadata = { "offset" : _get_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def fly_cruise_airspeed_profile(self) -> bool:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_fly_cruise_airspeed_profile_metadata)

    _set_fly_cruise_airspeed_profile_metadata = { "offset" : _set_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @fly_cruise_airspeed_profile.setter
    def fly_cruise_airspeed_profile(self, value:bool) -> None:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_fly_cruise_airspeed_profile_metadata, value)

    _get_flight_line_airspeed_options_metadata = { "offset" : _get_flight_line_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def flight_line_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the flight line airspeed options."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_flight_line_airspeed_options_metadata)

    _get_enroute_options_metadata = { "offset" : _get_enroute_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_options(self) -> "EnrouteOptions":
        """Get the enroute options."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_enroute_options_metadata)

    _get_enroute_turn_direction_options_metadata = { "offset" : _get_enroute_turn_direction_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_turn_direction_options(self) -> "EnrouteTurnDirectionOptions":
        """Get the enroute turn direction options."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_enroute_turn_direction_options_metadata)

    _get_enroute_cruise_airspeed_options_metadata = { "offset" : _get_enroute_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def enroute_cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the enroute cruise airspeed options."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_enroute_cruise_airspeed_options_metadata)

    _get_procedure_type_metadata = { "offset" : _get_procedure_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FlightLineProcedureType),) }
    @property
    def procedure_type(self) -> "FlightLineProcedureType":
        """Get or set the procedure methodology used to calculate the flight line."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_procedure_type_metadata)

    _set_procedure_type_metadata = { "offset" : _set_procedure_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FlightLineProcedureType),) }
    @procedure_type.setter
    def procedure_type(self, value:"FlightLineProcedureType") -> None:
        """Get or set the procedure methodology used to calculate the flight line."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_procedure_type_metadata, value)

    _get_outbound_course_metadata = { "offset" : _get_outbound_course_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def outbound_course(self) -> typing.Any:
        """Get or set the outbound course."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_outbound_course_metadata)

    _set_outbound_course_metadata = { "offset" : _set_outbound_course_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @outbound_course.setter
    def outbound_course(self, value:typing.Any) -> None:
        """Get or set the outbound course."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_outbound_course_metadata, value)

    _get_use_magnetic_heading_metadata = { "offset" : _get_use_magnetic_heading_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_magnetic_heading(self) -> bool:
        """Get or set the option to use a magnetic heading."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_use_magnetic_heading_metadata)

    _set_use_magnetic_heading_metadata = { "offset" : _set_use_magnetic_heading_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_magnetic_heading.setter
    def use_magnetic_heading(self, value:bool) -> None:
        """Get or set the option to use a magnetic heading."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_use_magnetic_heading_metadata, value)

    _get_leg_length_metadata = { "offset" : _get_leg_length_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def leg_length(self) -> float:
        """Get or set the length of the flight line."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_leg_length_metadata)

    _set_leg_length_metadata = { "offset" : _set_leg_length_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @leg_length.setter
    def leg_length(self, value:float) -> None:
        """Get or set the length of the flight line."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_leg_length_metadata, value)

    _get_must_level_off_metadata = { "offset" : _get_must_level_off_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def must_level_off(self) -> bool:
        """Opt whether the procedure must level off."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_must_level_off_metadata)

    _set_must_level_off_metadata = { "offset" : _set_must_level_off_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @must_level_off.setter
    def must_level_off(self, value:bool) -> None:
        """Opt whether the procedure must level off."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_must_level_off_metadata, value)

    _get_level_off_mode_metadata = { "offset" : _get_level_off_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @property
    def level_off_mode(self) -> "AltitudeConstraintManeuverMode":
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.get_property(ProcedureFlightLine._metadata, ProcedureFlightLine._get_level_off_mode_metadata)

    _set_level_off_mode_metadata = { "offset" : _set_level_off_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @level_off_mode.setter
    def level_off_mode(self, value:"AltitudeConstraintManeuverMode") -> None:
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.set_property(ProcedureFlightLine._metadata, ProcedureFlightLine._set_level_off_mode_metadata, value)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureFlightLine._metadata, ProcedureFlightLine._get_as_procedure_metadata, OutArg())

    _property_names[altitude_options] = "altitude_options"
    _property_names[fly_cruise_airspeed_profile] = "fly_cruise_airspeed_profile"
    _property_names[flight_line_airspeed_options] = "flight_line_airspeed_options"
    _property_names[enroute_options] = "enroute_options"
    _property_names[enroute_turn_direction_options] = "enroute_turn_direction_options"
    _property_names[enroute_cruise_airspeed_options] = "enroute_cruise_airspeed_options"
    _property_names[procedure_type] = "procedure_type"
    _property_names[outbound_course] = "outbound_course"
    _property_names[use_magnetic_heading] = "use_magnetic_heading"
    _property_names[leg_length] = "leg_length"
    _property_names[must_level_off] = "must_level_off"
    _property_names[level_off_mode] = "level_off_mode"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureFlightLine."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureFlightLine)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureFlightLine, [ProcedureFlightLine, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5232752425516520970, 13442831997956569777), ProcedureFlightLine)
agcls.AgTypeNameMap["ProcedureFlightLine"] = ProcedureFlightLine

class ProcedureDelay(IProcedure, SupportsDeleteCallback):
    """Interface used to access the options for a delay procedure."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_altitude_mode_method_offset = 1
    _set_altitude_mode_method_offset = 2
    _get_altitude_method_offset = 3
    _set_altitude_method_offset = 4
    _get_cruise_airspeed_options_method_offset = 5
    _get_turn_direction_method_offset = 6
    _set_turn_direction_method_offset = 7
    _get_turn_radius_factor_method_offset = 8
    _set_turn_radius_factor_method_offset = 9
    _metadata = {
        "iid_data" : (5716195463449211488, 12706362011475005615),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureDelay)

    _get_altitude_mode_metadata = { "offset" : _get_altitude_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DelayAltitudeMode),) }
    @property
    def altitude_mode(self) -> "DelayAltitudeMode":
        """Get or set the mode for handling the altitude of the aircraft."""
        return self._intf.get_property(ProcedureDelay._metadata, ProcedureDelay._get_altitude_mode_metadata)

    _set_altitude_mode_metadata = { "offset" : _set_altitude_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DelayAltitudeMode),) }
    @altitude_mode.setter
    def altitude_mode(self, value:"DelayAltitudeMode") -> None:
        """Get or set the mode for handling the altitude of the aircraft."""
        return self._intf.set_property(ProcedureDelay._metadata, ProcedureDelay._set_altitude_mode_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the requested altitude of the procedure."""
        return self._intf.get_property(ProcedureDelay._metadata, ProcedureDelay._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Get or set the requested altitude of the procedure."""
        return self._intf.set_property(ProcedureDelay._metadata, ProcedureDelay._set_altitude_metadata, value)

    _get_cruise_airspeed_options_metadata = { "offset" : _get_cruise_airspeed_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def cruise_airspeed_options(self) -> "CruiseAirspeedOptions":
        """Get the interface for the cruise airspeed options."""
        return self._intf.get_property(ProcedureDelay._metadata, ProcedureDelay._get_cruise_airspeed_options_metadata)

    _get_turn_direction_metadata = { "offset" : _get_turn_direction_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def turn_direction(self) -> "NavigatorTurnDirection":
        """Get or set the turn direction of the procedure."""
        return self._intf.get_property(ProcedureDelay._metadata, ProcedureDelay._get_turn_direction_metadata)

    _set_turn_direction_metadata = { "offset" : _set_turn_direction_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @turn_direction.setter
    def turn_direction(self, value:"NavigatorTurnDirection") -> None:
        """Get or set the turn direction of the procedure."""
        return self._intf.set_property(ProcedureDelay._metadata, ProcedureDelay._set_turn_direction_metadata, value)

    _get_turn_radius_factor_metadata = { "offset" : _get_turn_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def turn_radius_factor(self) -> float:
        """Get or set the turn radius factor."""
        return self._intf.get_property(ProcedureDelay._metadata, ProcedureDelay._get_turn_radius_factor_metadata)

    _set_turn_radius_factor_metadata = { "offset" : _set_turn_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @turn_radius_factor.setter
    def turn_radius_factor(self, value:float) -> None:
        """Get or set the turn radius factor."""
        return self._intf.set_property(ProcedureDelay._metadata, ProcedureDelay._set_turn_radius_factor_metadata, value)

    _property_names[altitude_mode] = "altitude_mode"
    _property_names[altitude] = "altitude"
    _property_names[cruise_airspeed_options] = "cruise_airspeed_options"
    _property_names[turn_direction] = "turn_direction"
    _property_names[turn_radius_factor] = "turn_radius_factor"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureDelay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureDelay)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureDelay, [ProcedureDelay, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5704318683765486662, 710367158900072076), ProcedureDelay)
agcls.AgTypeNameMap["ProcedureDelay"] = ProcedureDelay

class ProcedureTakeoff(IProcedure, SupportsDeleteCallback):
    """
    Interface used to access the options for a takeoff procedure.

    Examples
    --------
    Add a takeoff procedure from a runway:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a takeoff procedure with a runway as a site
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = takeoff.runway_heading_options
    >>> # Opt to use the headwind runway
    >>> headingOptions.runway_mode = RunwayHighLowEnd.HEADWIND
    >>>
    >>> # Set the takeoff mode and get that interface
    >>> takeoff.takeoff_mode = TakeoffMode.TAKEOFF_NORMAL
    >>> takeoffNormal = takeoff.mode_as_normal
    >>>
    >>> # Set the takeoff climb angle
    >>> takeoffNormal.takeoff_climb_angle = 5
    >>> # Set the departure altitude above the runway
    >>> takeoffNormal.departure_altitude = 600
    >>> # Set the altitude offset for the runway
    >>> takeoffNormal.runway_altitude_offset = 10
    >>> # Use terrain for the runway's altitude
    >>> takeoffNormal.use_runway_terrain = True
    """

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_runway_heading_options_method_offset = 1
    _get_mode_as_normal_method_offset = 2
    _get_mode_as_departure_point_method_offset = 3
    _get_mode_as_low_transition_method_offset = 4
    _get_takeoff_mode_method_offset = 5
    _set_takeoff_mode_method_offset = 6
    _get_as_procedure_method_offset = 7
    _metadata = {
        "iid_data" : (4746749933092015974, 13227298329395480749),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureTakeoff)

    _get_runway_heading_options_metadata = { "offset" : _get_runway_heading_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def runway_heading_options(self) -> "RunwayHeadingOptions":
        """Get the runway heading options."""
        return self._intf.get_property(ProcedureTakeoff._metadata, ProcedureTakeoff._get_runway_heading_options_metadata)

    _get_mode_as_normal_metadata = { "offset" : _get_mode_as_normal_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_normal(self) -> "TakeoffNormal":
        """Get the interface for a normal takeoff."""
        return self._intf.get_property(ProcedureTakeoff._metadata, ProcedureTakeoff._get_mode_as_normal_metadata)

    _get_mode_as_departure_point_metadata = { "offset" : _get_mode_as_departure_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_departure_point(self) -> "TakeoffDeparturePoint":
        """Get the interface for a departure point takeoff."""
        return self._intf.get_property(ProcedureTakeoff._metadata, ProcedureTakeoff._get_mode_as_departure_point_metadata)

    _get_mode_as_low_transition_metadata = { "offset" : _get_mode_as_low_transition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_low_transition(self) -> "TakeoffLowTransition":
        """Get the interface for a low transition takeoff."""
        return self._intf.get_property(ProcedureTakeoff._metadata, ProcedureTakeoff._get_mode_as_low_transition_metadata)

    _get_takeoff_mode_metadata = { "offset" : _get_takeoff_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TakeoffMode),) }
    @property
    def takeoff_mode(self) -> "TakeoffMode":
        """Get or set the type of takeoff the aircraft will perform."""
        return self._intf.get_property(ProcedureTakeoff._metadata, ProcedureTakeoff._get_takeoff_mode_metadata)

    _set_takeoff_mode_metadata = { "offset" : _set_takeoff_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TakeoffMode),) }
    @takeoff_mode.setter
    def takeoff_mode(self, value:"TakeoffMode") -> None:
        """Get or set the type of takeoff the aircraft will perform."""
        return self._intf.set_property(ProcedureTakeoff._metadata, ProcedureTakeoff._set_takeoff_mode_metadata, value)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureTakeoff._metadata, ProcedureTakeoff._get_as_procedure_metadata, OutArg())

    _property_names[runway_heading_options] = "runway_heading_options"
    _property_names[mode_as_normal] = "mode_as_normal"
    _property_names[mode_as_departure_point] = "mode_as_departure_point"
    _property_names[mode_as_low_transition] = "mode_as_low_transition"
    _property_names[takeoff_mode] = "takeoff_mode"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureTakeoff."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureTakeoff)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureTakeoff, [ProcedureTakeoff, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((4622436812587213547, 17163227853363219857), ProcedureTakeoff)
agcls.AgTypeNameMap["ProcedureTakeoff"] = ProcedureTakeoff

class ProcedureCollection(SupportsDeleteCallback):
    """
    Interface used to access the collection of procedures for a given phase in a mission. Use this interface to Get, Add, or Remove a procedure.

    Examples
    --------
    Add a takeoff procedure from a runway:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a takeoff procedure with a runway as a site
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = takeoff.runway_heading_options
    >>> # Opt to use the headwind runway
    >>> headingOptions.runway_mode = RunwayHighLowEnd.HEADWIND
    >>>
    >>> # Set the takeoff mode and get that interface
    >>> takeoff.takeoff_mode = TakeoffMode.TAKEOFF_NORMAL
    >>> takeoffNormal = takeoff.mode_as_normal
    >>>
    >>> # Set the takeoff climb angle
    >>> takeoffNormal.takeoff_climb_angle = 5
    >>> # Set the departure altitude above the runway
    >>> takeoffNormal.departure_altitude = 600
    >>> # Set the altitude offset for the runway
    >>> takeoffNormal.runway_altitude_offset = 10
    >>> # Use terrain for the runway's altitude
    >>> takeoffNormal.use_runway_terrain = True

    Add and configure a landing procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a landing procedure
    >>> landing = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_LANDING)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = landing.runway_heading_options
    >>> # Land from the low end
    >>> headingOptions.runway_mode = RunwayHighLowEnd.LOW_END
    >>>
    >>> # Use a standard instrument approach
    >>> landing.approach_mode = ApproachMode.STANDARD_INSTRUMENT_APPROACH
    >>> # Get the options for a standard instrument approach
    >>> sia = landing.mode_as_standard_instrument_approach
    >>> # Change the approach altitude
    >>> sia.approach_altitude = 1000
    >>> # Change the glideslope
    >>> sia.glideslope = 4
    >>> # Offset the runway altitude
    >>> sia.runway_altitude_offset = 10
    >>> # Use the terrain as an altitude reference for the runway
    >>> sia.use_runway_terrain = True

    Add and configure an en-route procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add an enroute procedure with a site type of End of Previous Procedure
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>> # Get the altitude options
    >>> altitudeOptions = enroute.altitude_msl_options
    >>> # To specify an altitude, turn off the option to use the default cruise altitude
    >>> altitudeOptions.use_default_cruise_altitude = False
    >>> # Set the altitude
    >>> altitudeOptions.msl_altitude = 10000
    >>>
    >>> # Get the navigation options
    >>> navigationOptions = enroute.navigation_options
    >>> # Set the route to arrive on a specified course
    >>> navigationOptions.navigation_mode = PointToPointMode.ARRIVE_ON_COURSE
    >>> # Set the course
    >>> navigationOptions.arrive_on_course = 30
    >>> # Use a magnetic heading
    >>> navigationOptions.use_magnetic_heading = True
    >>>
    >>> # Get the navigation options
    >>> airspeedOptions = enroute.enroute_cruise_airspeed_options
    >>> # Fly at max speed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.MAX_AIRSPEED
    >>> # To specify an airspeed to fly at, set the speed type to other airspeed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.OTHER_AIRSPEED
    >>> # Then set the airspeed and airspeed type
    >>> airspeedOptions.set_other_airspeed(AirspeedType.TAS, 200)

    Add and configure a basic maneuver procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a basic maneuver procedure
    >>> basicManeuver = procedures.add(SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_BASIC_MANEUVER)
    >>>
    >>> # Set the navigation to use a Straight Ahead strategy
    >>> basicManeuver.navigation_strategy_type = "Straight Ahead"
    >>> # Get the options for the straight ahead strategy
    >>> straightAhead = basicManeuver.navigation
    >>> # Opt to maintain course (as opposed to maintain heading)
    >>> straightAhead.reference_frame = StraightAheadReferenceFrame.MAINTAIN_COURSE
    >>>
    >>> # Set the profile to use a Autopilot - Vertical Plane strategy
    >>> basicManeuver.profile_strategy_type = "Autopilot - Vertical Plane"
    >>> # Get the options for the profile strategy
    >>> autopilot = basicManeuver.profile
    >>> # Opt to maintain the initial altitude
    >>> autopilot.altitude_mode = AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_ALTITUDE
    >>> airspeedOptions = autopilot.airspeed_options
    >>> # Opt to maintain a specified airspeed
    >>> airspeedOptions.airspeed_mode = BasicManeuverAirspeedMode.MAINTAIN_SPECIFIED_AIRSPEED
    >>> # Specify the airspeed
    >>> airspeedOptions.specified_airspeed = 250
    >>>
    >>> # Configure the options on the Attitude / Performance / Fuel page
    >>> basicManeuver.flight_mode = PhaseOfFlight.FLIGHT_PHASE_CRUISE
    >>> # Override the fuel flow
    >>> basicManeuver.fuel_flow_type = BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_OVERRIDE
    >>> basicManeuver.override_fuel_flow_value = 1000
    >>>
    >>> # Set the basic stopping conditions
    >>> basicManeuver.use_max_downrange = True
    >>> basicManeuver.max_downrange = 10
    >>> basicManeuver.use_stop_fuel_state = False
    >>> basicManeuver.use_max_time_of_flight = False

    Add and remove procedures:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Add a takeoff procedure with a runway as a site. This will add the procedure
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>> # Add a procedure at a given index (starting from 0)
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>>
    >>> # Make sure to propagate the mission to calculate the route
    >>> propagator.propagate()
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Check to see if the mission is valid (must first be propagated)
    >>> isValid = mission.is_valid
    >>>
    >>> # Get the number of procedures
    >>> procedureCount = procedures.count
    >>> # Remove the procedure at the given index
    >>> procedures.remove_at_index(1)
    >>> # Remove the given procedure
    >>> procedures.remove(takeoff)
    >>>
    >>> # Propagate the mission
    >>> propagator.propagate()
    """

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _add_method_offset = 4
    _add_at_index_method_offset = 5
    _remove_method_offset = 6
    _remove_at_index_method_offset = 7
    _enable_auto_propagate_method_offset = 8
    _disable_auto_propagate_method_offset = 9
    _metadata = {
        "iid_data" : (4672125039402126824, 4866911027730476462),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureCollection)
    def __iter__(self):
        """Create an iterator for the ProcedureCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IProcedure":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the number of elements in a collection."""
        return self._intf.get_property(ProcedureCollection._metadata, ProcedureCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "IProcedure":
        """Given an index, returns an element in the collection."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that can iterate through the collection."""
        return self._intf.get_property(ProcedureCollection._metadata, ProcedureCollection._get__new_enum_metadata)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(SiteType), agmarshall.EnumArg(ProcedureType), agmarshall.InterfaceOutArg,) }
    def add(self, site_type:"SiteType", procedure_type:"ProcedureType") -> "IProcedure":
        """Add a procedure with the specified site at the end of the current phase."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._add_metadata, site_type, procedure_type, OutArg())

    _add_at_index_metadata = { "offset" : _add_at_index_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.EnumArg(SiteType), agmarshall.EnumArg(ProcedureType), agmarshall.InterfaceOutArg,) }
    def add_at_index(self, index:int, site_type:"SiteType", procedure_type:"ProcedureType") -> "IProcedure":
        """Add a procedure with the specified site at the given index."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._add_at_index_metadata, index, site_type, procedure_type, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IProcedure"),) }
    def remove(self, procedure:"IProcedure") -> None:
        """Remove given procedure."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._remove_metadata, procedure)

    _remove_at_index_metadata = { "offset" : _remove_at_index_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    def remove_at_index(self, index:int) -> None:
        """Remove procedure at the given index."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._remove_at_index_metadata, index)

    _enable_auto_propagate_metadata = { "offset" : _enable_auto_propagate_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def enable_auto_propagate(self) -> None:
        """Enable automatically propagating the mission. Aviator will automatically propagate before adding a procedure, ensuring a valid initial state for the new procedure."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._enable_auto_propagate_metadata, )

    _disable_auto_propagate_metadata = { "offset" : _disable_auto_propagate_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def disable_auto_propagate(self) -> None:
        """Disable automatically propagating the mission. Use with caution. Aviator will not automatically propagate before adding new procedures."""
        return self._intf.invoke(ProcedureCollection._metadata, ProcedureCollection._disable_auto_propagate_metadata, )

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureCollection, [ProcedureCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5616253919542791990, 16838747785619043462), ProcedureCollection)
agcls.AgTypeNameMap["ProcedureCollection"] = ProcedureCollection

class Phase(SupportsDeleteCallback):
    """
    Interface used to access the phase options for a mission.

    Examples
    --------
    Configure the performance models to be used in the phase:
    >>> # Phase phase: Phase object
    >>> # Get the acceleration performance model used for the current phase
    >>> acceleration = phase.get_performance_model_by_type("Acceleration")
    >>> # Check if it is linked to the catalog
    >>> isLinkedToCatalog = acceleration.is_linked_to_catalog
    >>> # Use the performance model in the catalog named "Built-In Model"
    >>> acceleration.link_to_catalog("Built-In Model")
    >>>
    >>> # Get the VTOL performance model
    >>> vtol = phase.get_performance_model_by_type("VTOL")
    >>> # Create a new vtol model of type AGI VTOL Model. Note that this new model does not exist in the catalog and only exists in the phase.
    >>> vtol.create_new("AGI VTOL Model")
    >>> # Rename the performance model
    >>> vtol.rename("Temporary VTOL Model")

    Add a new phase and use the same performance models as the first phase:
    >>> # PhaseCollection phases: Phase Collection object
    >>> # Add a new phase at the end of the mission
    >>> newPhase = phases.add()
    >>> # Rename the phase
    >>> newPhase.name = "New Phase"
    >>> # Copy the performance models from the first phase and paste it to the new phase
    >>> phases[0].copy_performance_models()
    >>> newPhase.paste_performance_models()
    """

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_procedures_method_offset = 1
    _get_name_method_offset = 2
    _set_name_method_offset = 3
    _get_performance_model_by_type_method_offset = 4
    _set_default_performance_models_method_offset = 5
    _copy_performance_models_method_offset = 6
    _paste_performance_models_method_offset = 7
    _metadata = {
        "iid_data" : (5065421894605992373, 7712798079751778479),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Phase)

    _get_procedures_metadata = { "offset" : _get_procedures_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def procedures(self) -> "ProcedureCollection":
        """Return the procedure collection."""
        return self._intf.get_property(Phase._metadata, Phase._get_procedures_metadata)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the phase."""
        return self._intf.get_property(Phase._metadata, Phase._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        """Get or set the name of the phase."""
        return self._intf.set_property(Phase._metadata, Phase._set_name_metadata, value)

    _get_performance_model_by_type_metadata = { "offset" : _get_performance_model_by_type_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_performance_model_by_type(self, type:str) -> "PerformanceModelOptions":
        """Get the active performance model for the given category type."""
        return self._intf.invoke(Phase._metadata, Phase._get_performance_model_by_type_metadata, type, OutArg())

    _set_default_performance_models_metadata = { "offset" : _set_default_performance_models_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def set_default_performance_models(self) -> None:
        """Set the phase to use the default performance models of the aircraft."""
        return self._intf.invoke(Phase._metadata, Phase._set_default_performance_models_metadata, )

    _copy_performance_models_metadata = { "offset" : _copy_performance_models_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_performance_models(self) -> None:
        """Create a copy of the active performance models for the current phase."""
        return self._intf.invoke(Phase._metadata, Phase._copy_performance_models_metadata, )

    _paste_performance_models_metadata = { "offset" : _paste_performance_models_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def paste_performance_models(self) -> None:
        """Paste the performance models."""
        return self._intf.invoke(Phase._metadata, Phase._paste_performance_models_metadata, )

    _property_names[procedures] = "procedures"
    _property_names[name] = "name"

    def __init__(self, source_object=None):
        """Construct an object of type Phase."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Phase)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Phase, [Phase, ])

agcls.AgClassCatalog.add_catalog_entry((5327259017682921546, 13780760074728122774), Phase)
agcls.AgTypeNameMap["Phase"] = Phase

class PhaseCollection(SupportsDeleteCallback):
    """
    Interface used to access the collection of phases for a mission.

    Examples
    --------
    Add a new phase and use the same performance models as the first phase:
    >>> # PhaseCollection phases: Phase Collection object
    >>> # Add a new phase at the end of the mission
    >>> newPhase = phases.add()
    >>> # Rename the phase
    >>> newPhase.name = "New Phase"
    >>> # Copy the performance models from the first phase and paste it to the new phase
    >>> phases[0].copy_performance_models()
    >>> newPhase.paste_performance_models()
    """

    _num_methods = 7
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _add_method_offset = 4
    _add_at_index_method_offset = 5
    _remove_method_offset = 6
    _remove_at_index_method_offset = 7
    _metadata = {
        "iid_data" : (5253053724572556766, 10071354963761500314),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PhaseCollection)
    def __iter__(self):
        """Create an iterator for the PhaseCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "Phase":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the number of elements in a collection."""
        return self._intf.get_property(PhaseCollection._metadata, PhaseCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "Phase":
        """Given an index, returns an element in the collection."""
        return self._intf.invoke(PhaseCollection._metadata, PhaseCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return an enumerator that can iterate through the collection."""
        return self._intf.get_property(PhaseCollection._metadata, PhaseCollection._get__new_enum_metadata)

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def add(self) -> "Phase":
        """Add a phase at the end of the mission."""
        return self._intf.invoke(PhaseCollection._metadata, PhaseCollection._add_metadata, OutArg())

    _add_at_index_metadata = { "offset" : _add_at_index_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def add_at_index(self, index:int) -> "Phase":
        """Add a phase at the given index."""
        return self._intf.invoke(PhaseCollection._metadata, PhaseCollection._add_at_index_metadata, index, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Phase"),) }
    def remove(self, phase:"Phase") -> None:
        """Remove given phase."""
        return self._intf.invoke(PhaseCollection._metadata, PhaseCollection._remove_metadata, phase)

    _remove_at_index_metadata = { "offset" : _remove_at_index_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    def remove_at_index(self, index:int) -> None:
        """Remove phase at the given index."""
        return self._intf.invoke(PhaseCollection._metadata, PhaseCollection._remove_at_index_metadata, index)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type PhaseCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PhaseCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PhaseCollection, [PhaseCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5426301579413729339, 2385420409047666822), PhaseCollection)
agcls.AgTypeNameMap["PhaseCollection"] = PhaseCollection

class Mission(SupportsDeleteCallback):
    """
    Interface for the mission of an aircraft using the Aviator propagator.

    Examples
    --------
    Set the Configuration used for the Mission:
    >>> # Mission mission: Aviator Mission object
    >>> # Get the configuration used for the mission
    >>> configuration = mission.configuration
    >>> # Set the max landing weight
    >>> configuration.max_landing_weight = 300000
    >>> # Set the empty weight
    >>> configuration.empty_weight = 210000
    >>> # Update the center of gravity of the aircraft when empty
    >>> configuration.set_empty_cg(2, 0, 1)
    >>>
    >>> # Get the stations
    >>> stations = configuration.get_stations()
    >>> # Check if there is an internal fuel station
    >>> if stations.contains_station("Internal Fuel") is True:
    >>>     # Get the fuel tank
    >>>     fuelTank = stations.get_internal_fuel_tank_by_name("Internal Fuel")
    >>>     # Set the capacity of the fuel tank
    >>>     fuelTank.capacity = 175000
    >>>     # Set the initial state of the fuel tank
    >>>     fuelTank.initial_fuel_state = 125000
    >>>
    >>> # Add a new payload station
    >>> newPayload = stations.add_payload_station()
    >>> # Set the position of the payload station
    >>> newPayload.set_position(0, 2, 0)
    >>> # Add an external fuel tank
    >>> externalTank = newPayload.add_external_fuel_tank()
    >>> # Set the empty weight of the tank
    >>> externalTank.empty_weight = 2000

    Set the aircraft used for the mission to an aircraft found in the Aviator catalog:
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Get the Aviator catalog
    >>> catalog = propagator.aviator_catalog
    >>> # Get the aircraft category
    >>> category = catalog.aircraft_category
    >>> # Get the user aircraft models
    >>> aircraftModels = category.aircraft_models
    >>> # Get the basic fighter
    >>> fighter = aircraftModels.get_aircraft("Basic Fighter")
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Set the vehicle used for the mission
    >>> mission.vehicle = fighter

    Configure the weather and atmosphere of the Mission:
    >>> # Mission mission: Aviator Mission object
    >>> # Get the wind model used for the mission
    >>> windModel = mission.wind_model
    >>> # Let's use the mission model
    >>> windModel.wind_model_source = WindAtmosphereModelSource.MISSION_MODEL
    >>> # Let's use constant wind
    >>> windModel.wind_model_type = WindModelType.CONSTANT_WIND
    >>> # Get the constant wind model options
    >>> constantWind = windModel.mode_as_constant
    >>> # Set the wind bearing
    >>> constantWind.wind_bearing = 30
    >>> # Set the wind speed
    >>> constantWind.wind_speed = 5
    >>>
    >>> # Get the atmosphere model used for the mission
    >>> atmosphere = mission.atmosphere_model
    >>> # Let's use the mission model
    >>> atmosphere.atmosphere_model_source = WindAtmosphereModelSource.MISSION_MODEL
    >>> # Get the basic atmosphere options
    >>> basicAtmosphere = atmosphere.mode_as_basic
    >>> # Use standard 1976 atmosphere
    >>> basicAtmosphere.basic_model_type = AtmosphereModelType.STANDARD1976
    >>> # Opt to override the values
    >>> basicAtmosphere.use_non_standard_atmosphere = True
    >>> # Override the temperature
    >>> basicAtmosphere.temperature = 290

    Add and remove procedures:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Add a takeoff procedure with a runway as a site. This will add the procedure
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>> # Add a procedure at a given index (starting from 0)
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>>
    >>> # Make sure to propagate the mission to calculate the route
    >>> propagator.propagate()
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Check to see if the mission is valid (must first be propagated)
    >>> isValid = mission.is_valid
    >>>
    >>> # Get the number of procedures
    >>> procedureCount = procedures.count
    >>> # Remove the procedure at the given index
    >>> procedures.remove_at_index(1)
    >>> # Remove the given procedure
    >>> procedures.remove(takeoff)
    >>>
    >>> # Propagate the mission
    >>> propagator.propagate()
    """

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_phases_method_offset = 1
    _get_vehicle_method_offset = 2
    _set_vehicle_method_offset = 3
    _get_configuration_method_offset = 4
    _get_wind_model_method_offset = 5
    _get_atmosphere_model_method_offset = 6
    _get_is_valid_method_offset = 7
    _get_first_invalid_procedure_method_offset = 8
    _metadata = {
        "iid_data" : (5166308313264454994, 7736963905921891476),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Mission)

    _get_phases_metadata = { "offset" : _get_phases_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def phases(self) -> "PhaseCollection":
        """Get the mission phases."""
        return self._intf.get_property(Mission._metadata, Mission._get_phases_metadata)

    _get_vehicle_metadata = { "offset" : _get_vehicle_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vehicle(self) -> "IAviatorVehicle":
        """Get or set the vehicle used in the mission."""
        return self._intf.get_property(Mission._metadata, Mission._get_vehicle_metadata)

    _set_vehicle_metadata = { "offset" : _set_vehicle_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAviatorVehicle"),) }
    @vehicle.setter
    def vehicle(self, aviator_vehicle:"IAviatorVehicle") -> None:
        """Get or set the vehicle used in the mission."""
        return self._intf.set_property(Mission._metadata, Mission._set_vehicle_metadata, aviator_vehicle)

    _get_configuration_metadata = { "offset" : _get_configuration_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def configuration(self) -> "Configuration":
        """Get the aircraft's configuration for the mission."""
        return self._intf.get_property(Mission._metadata, Mission._get_configuration_metadata)

    _get_wind_model_metadata = { "offset" : _get_wind_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def wind_model(self) -> "WindModel":
        """Get the mission wind model."""
        return self._intf.get_property(Mission._metadata, Mission._get_wind_model_metadata)

    _get_atmosphere_model_metadata = { "offset" : _get_atmosphere_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def atmosphere_model(self) -> "AtmosphereModel":
        """Get the mission atmosphere model."""
        return self._intf.get_property(Mission._metadata, Mission._get_atmosphere_model_metadata)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Check whether the mission is valid. Calling this property will propagate the mission."""
        return self._intf.get_property(Mission._metadata, Mission._get_is_valid_metadata)

    _get_first_invalid_procedure_metadata = { "offset" : _get_first_invalid_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_first_invalid_procedure(self) -> "IProcedure":
        """Get the first invalid procedure in the mission. Calling this method will propagate the mission."""
        return self._intf.invoke(Mission._metadata, Mission._get_first_invalid_procedure_metadata, OutArg())

    _property_names[phases] = "phases"
    _property_names[vehicle] = "vehicle"
    _property_names[configuration] = "configuration"
    _property_names[wind_model] = "wind_model"
    _property_names[atmosphere_model] = "atmosphere_model"
    _property_names[is_valid] = "is_valid"

    def __init__(self, source_object=None):
        """Construct an object of type Mission."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Mission)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Mission, [Mission, ])

agcls.AgClassCatalog.add_catalog_entry((5084660225869613847, 3815698403988913814), Mission)
agcls.AgTypeNameMap["Mission"] = Mission

class AviatorPropagator(SupportsDeleteCallback):
    """
    Interface used to access the Aviator interface for an aircraft. Use this interface to get the mission or Aviator catalog.

    Examples
    --------
    Configure the Aviator propagator:
    >>> # Aircraft aircraft: Aircraft object
    >>> # Set to Propagator to Aviator
    >>> aircraft.set_route_type(PropagatorType.AVIATOR)
    >>> # Get the aircraft's route
    >>> aircraftRoute = aircraft.route
    >>> # Get the Aviator propagator
    >>> propagator = aircraftRoute.aviator_propagator
    >>> # Get the Aviator mission
    >>> mission = propagator.aviator_mission
    >>> # Get the list of phases from the mission
    >>> phases = mission.phases
    >>> # Get the list of procedures from the first phase
    >>> procedures = phases[0].procedures
    >>> # Propagate the route
    >>> propagator.propagate()
    """

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_aviator_mission_method_offset = 1
    _propagate_method_offset = 2
    _get_auto_recalculate_method_offset = 3
    _set_auto_recalculate_method_offset = 4
    _get_aviator_catalog_method_offset = 5
    _metadata = {
        "iid_data" : (5661397388601039162, 10681986619092325766),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AviatorPropagator)

    _get_aviator_mission_metadata = { "offset" : _get_aviator_mission_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aviator_mission(self) -> "Mission":
        """Get the Aviator mission."""
        return self._intf.get_property(AviatorPropagator._metadata, AviatorPropagator._get_aviator_mission_metadata)

    _propagate_metadata = { "offset" : _propagate_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def propagate(self) -> None:
        """Apply All Change."""
        return self._intf.invoke(AviatorPropagator._metadata, AviatorPropagator._propagate_metadata, )

    _get_auto_recalculate_metadata = { "offset" : _get_auto_recalculate_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def auto_recalculate(self) -> bool:
        """Opt whether to have the propagator auto recalculate."""
        return self._intf.get_property(AviatorPropagator._metadata, AviatorPropagator._get_auto_recalculate_metadata)

    _set_auto_recalculate_metadata = { "offset" : _set_auto_recalculate_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @auto_recalculate.setter
    def auto_recalculate(self, auto_recalc:bool) -> None:
        """Opt whether to have the propagator auto recalculate."""
        return self._intf.set_property(AviatorPropagator._metadata, AviatorPropagator._set_auto_recalculate_metadata, auto_recalc)

    _get_aviator_catalog_metadata = { "offset" : _get_aviator_catalog_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aviator_catalog(self) -> "Catalog":
        """Get the Aviator catalog."""
        return self._intf.get_property(AviatorPropagator._metadata, AviatorPropagator._get_aviator_catalog_metadata)

    _property_names[aviator_mission] = "aviator_mission"
    _property_names[auto_recalculate] = "auto_recalculate"
    _property_names[aviator_catalog] = "aviator_catalog"

    def __init__(self, source_object=None):
        """Construct an object of type AviatorPropagator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AviatorPropagator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AviatorPropagator, [AviatorPropagator, ])

agcls.AgClassCatalog.add_catalog_entry((4966342315327056185, 8021477894575781281), AviatorPropagator)
agcls.AgTypeNameMap["AviatorPropagator"] = AviatorPropagator

class ProcedureBasicManeuver(IProcedure, SupportsDeleteCallback):
    """
    Interface used to access the options for a Basic Maneuver procedure.

    Examples
    --------
    Add and configure a basic maneuver procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a basic maneuver procedure
    >>> basicManeuver = procedures.add(SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_BASIC_MANEUVER)
    >>>
    >>> # Set the navigation to use a Straight Ahead strategy
    >>> basicManeuver.navigation_strategy_type = "Straight Ahead"
    >>> # Get the options for the straight ahead strategy
    >>> straightAhead = basicManeuver.navigation
    >>> # Opt to maintain course (as opposed to maintain heading)
    >>> straightAhead.reference_frame = StraightAheadReferenceFrame.MAINTAIN_COURSE
    >>>
    >>> # Set the profile to use a Autopilot - Vertical Plane strategy
    >>> basicManeuver.profile_strategy_type = "Autopilot - Vertical Plane"
    >>> # Get the options for the profile strategy
    >>> autopilot = basicManeuver.profile
    >>> # Opt to maintain the initial altitude
    >>> autopilot.altitude_mode = AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_ALTITUDE
    >>> airspeedOptions = autopilot.airspeed_options
    >>> # Opt to maintain a specified airspeed
    >>> airspeedOptions.airspeed_mode = BasicManeuverAirspeedMode.MAINTAIN_SPECIFIED_AIRSPEED
    >>> # Specify the airspeed
    >>> airspeedOptions.specified_airspeed = 250
    >>>
    >>> # Configure the options on the Attitude / Performance / Fuel page
    >>> basicManeuver.flight_mode = PhaseOfFlight.FLIGHT_PHASE_CRUISE
    >>> # Override the fuel flow
    >>> basicManeuver.fuel_flow_type = BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_OVERRIDE
    >>> basicManeuver.override_fuel_flow_value = 1000
    >>>
    >>> # Set the basic stopping conditions
    >>> basicManeuver.use_max_downrange = True
    >>> basicManeuver.max_downrange = 10
    >>> basicManeuver.use_stop_fuel_state = False
    >>> basicManeuver.use_max_time_of_flight = False
    """

    _num_methods = 37
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_time_of_flight_method_offset = 1
    _set_max_time_of_flight_method_offset = 2
    _get_use_max_time_of_flight_method_offset = 3
    _set_use_max_time_of_flight_method_offset = 4
    _get_stop_fuel_state_method_offset = 5
    _set_stop_fuel_state_method_offset = 6
    _get_use_stop_fuel_state_method_offset = 7
    _set_use_stop_fuel_state_method_offset = 8
    _get_max_downrange_method_offset = 9
    _set_max_downrange_method_offset = 10
    _get_use_max_downrange_method_offset = 11
    _set_use_max_downrange_method_offset = 12
    _get_altitude_limit_mode_method_offset = 13
    _set_altitude_limit_mode_method_offset = 14
    _get_terrain_impact_mode_method_offset = 15
    _set_terrain_impact_mode_method_offset = 16
    _get_terrain_impact_time_offset_method_offset = 17
    _set_terrain_impact_time_offset_method_offset = 18
    _get_navigation_strategy_type_method_offset = 19
    _set_navigation_strategy_type_method_offset = 20
    _get_navigation_method_offset = 21
    _get_profile_strategy_type_method_offset = 22
    _set_profile_strategy_type_method_offset = 23
    _get_profile_method_offset = 24
    _get_flight_mode_method_offset = 25
    _set_flight_mode_method_offset = 26
    _get_fuel_flow_type_method_offset = 27
    _set_fuel_flow_type_method_offset = 28
    _get_override_fuel_flow_value_method_offset = 29
    _set_override_fuel_flow_value_method_offset = 30
    _get_scale_fuel_flow_method_offset = 31
    _set_scale_fuel_flow_method_offset = 32
    _get_attitude_blend_time_method_offset = 33
    _set_attitude_blend_time_method_offset = 34
    _get_control_time_constant_method_offset = 35
    _set_control_time_constant_method_offset = 36
    _get_as_procedure_method_offset = 37
    _metadata = {
        "iid_data" : (4946730629668897604, 11088550870477238442),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureBasicManeuver)

    _get_max_time_of_flight_metadata = { "offset" : _get_max_time_of_flight_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_time_of_flight(self) -> typing.Any:
        """Get the max time of flight."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_max_time_of_flight_metadata)

    _set_max_time_of_flight_metadata = { "offset" : _set_max_time_of_flight_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_time_of_flight.setter
    def max_time_of_flight(self, value:typing.Any) -> None:
        """Set the max time of flight."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_max_time_of_flight_metadata, value)

    _get_use_max_time_of_flight_metadata = { "offset" : _get_use_max_time_of_flight_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_max_time_of_flight(self) -> bool:
        """Get whether to use max time of flight."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_use_max_time_of_flight_metadata)

    _set_use_max_time_of_flight_metadata = { "offset" : _set_use_max_time_of_flight_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_max_time_of_flight.setter
    def use_max_time_of_flight(self, value:bool) -> None:
        """Set whether to use max time of flight."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_use_max_time_of_flight_metadata, value)

    _get_stop_fuel_state_metadata = { "offset" : _get_stop_fuel_state_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_fuel_state(self) -> float:
        """Get the stop fuel state value."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_stop_fuel_state_metadata)

    _set_stop_fuel_state_metadata = { "offset" : _set_stop_fuel_state_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_fuel_state.setter
    def stop_fuel_state(self, value:float) -> None:
        """Set the stop fuel state value."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_stop_fuel_state_metadata, value)

    _get_use_stop_fuel_state_metadata = { "offset" : _get_use_stop_fuel_state_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_stop_fuel_state(self) -> bool:
        """Get whether to use stop fuel state."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_use_stop_fuel_state_metadata)

    _set_use_stop_fuel_state_metadata = { "offset" : _set_use_stop_fuel_state_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_stop_fuel_state.setter
    def use_stop_fuel_state(self, value:bool) -> None:
        """Set whether to use stop fuel state."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_use_stop_fuel_state_metadata, value)

    _get_max_downrange_metadata = { "offset" : _get_max_downrange_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_downrange(self) -> float:
        """Get the max down range."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_max_downrange_metadata)

    _set_max_downrange_metadata = { "offset" : _set_max_downrange_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_downrange.setter
    def max_downrange(self, value:float) -> None:
        """Set the max down range."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_max_downrange_metadata, value)

    _get_use_max_downrange_metadata = { "offset" : _get_use_max_downrange_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_max_downrange(self) -> bool:
        """Get whether to use max down range."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_use_max_downrange_metadata)

    _set_use_max_downrange_metadata = { "offset" : _set_use_max_downrange_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_max_downrange.setter
    def use_max_downrange(self, value:bool) -> None:
        """Set whether to use max down range."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_use_max_downrange_metadata, value)

    _get_altitude_limit_mode_metadata = { "offset" : _get_altitude_limit_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverAltitudeLimit),) }
    @property
    def altitude_limit_mode(self) -> "BasicManeuverAltitudeLimit":
        """Get the altitude limit mode."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_altitude_limit_mode_metadata)

    _set_altitude_limit_mode_metadata = { "offset" : _set_altitude_limit_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverAltitudeLimit),) }
    @altitude_limit_mode.setter
    def altitude_limit_mode(self, value:"BasicManeuverAltitudeLimit") -> None:
        """Set the altitude limit mode."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_altitude_limit_mode_metadata, value)

    _get_terrain_impact_mode_metadata = { "offset" : _get_terrain_impact_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverAltitudeLimit),) }
    @property
    def terrain_impact_mode(self) -> "BasicManeuverAltitudeLimit":
        """Get the terrain impact mode."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_terrain_impact_mode_metadata)

    _set_terrain_impact_mode_metadata = { "offset" : _set_terrain_impact_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverAltitudeLimit),) }
    @terrain_impact_mode.setter
    def terrain_impact_mode(self, value:"BasicManeuverAltitudeLimit") -> None:
        """Set the terrain impact mode."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_terrain_impact_mode_metadata, value)

    _get_terrain_impact_time_offset_metadata = { "offset" : _get_terrain_impact_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def terrain_impact_time_offset(self) -> float:
        """Get the terrain impact time offset."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_terrain_impact_time_offset_metadata)

    _set_terrain_impact_time_offset_metadata = { "offset" : _set_terrain_impact_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @terrain_impact_time_offset.setter
    def terrain_impact_time_offset(self, value:float) -> None:
        """Set the terrain impact time offset."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_terrain_impact_time_offset_metadata, value)

    _get_navigation_strategy_type_metadata = { "offset" : _get_navigation_strategy_type_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def navigation_strategy_type(self) -> str:
        """Get the navigation strategy type."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_navigation_strategy_type_metadata)

    _set_navigation_strategy_type_metadata = { "offset" : _set_navigation_strategy_type_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @navigation_strategy_type.setter
    def navigation_strategy_type(self, value:str) -> None:
        """Set the navigation strategy type."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_navigation_strategy_type_metadata, value)

    _get_navigation_metadata = { "offset" : _get_navigation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def navigation(self) -> "IBasicManeuverStrategy":
        """Get the interface for the navigation strategy."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_navigation_metadata)

    _get_profile_strategy_type_metadata = { "offset" : _get_profile_strategy_type_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def profile_strategy_type(self) -> str:
        """Get the profile strategy type."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_profile_strategy_type_metadata)

    _set_profile_strategy_type_metadata = { "offset" : _set_profile_strategy_type_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @profile_strategy_type.setter
    def profile_strategy_type(self, value:str) -> None:
        """Set the profile strategy type."""
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_profile_strategy_type_metadata, value)

    _get_profile_metadata = { "offset" : _get_profile_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def profile(self) -> "IBasicManeuverStrategy":
        """Get the interface for the profile strategy."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_profile_metadata)

    _get_flight_mode_metadata = { "offset" : _get_flight_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PhaseOfFlight),) }
    @property
    def flight_mode(self) -> "PhaseOfFlight":
        """Get or set the type of performance model  that the aircraft will use to fly the maneuver."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_flight_mode_metadata)

    _set_flight_mode_metadata = { "offset" : _set_flight_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PhaseOfFlight),) }
    @flight_mode.setter
    def flight_mode(self, value:"PhaseOfFlight") -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_flight_mode_metadata, value)

    _get_fuel_flow_type_metadata = { "offset" : _get_fuel_flow_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverFuelFlowType),) }
    @property
    def fuel_flow_type(self) -> "BasicManeuverFuelFlowType":
        """Get or set the source used to calculate the fuel flow for the maneuver."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_fuel_flow_type_metadata)

    _set_fuel_flow_type_metadata = { "offset" : _set_fuel_flow_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverFuelFlowType),) }
    @fuel_flow_type.setter
    def fuel_flow_type(self, value:"BasicManeuverFuelFlowType") -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_fuel_flow_type_metadata, value)

    _get_override_fuel_flow_value_metadata = { "offset" : _get_override_fuel_flow_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def override_fuel_flow_value(self) -> float:
        """Get or set the value used for the Override Fuel Flow type. The fuel flow type must be set to Override to access this value."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_override_fuel_flow_value_metadata)

    _set_override_fuel_flow_value_metadata = { "offset" : _set_override_fuel_flow_value_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @override_fuel_flow_value.setter
    def override_fuel_flow_value(self, value:float) -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_override_fuel_flow_value_metadata, value)

    _get_scale_fuel_flow_metadata = { "offset" : _get_scale_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow(self) -> bool:
        """Opt whether to scale the fuel flow based on the aircraft's actual attitude."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_scale_fuel_flow_metadata)

    _set_scale_fuel_flow_metadata = { "offset" : _set_scale_fuel_flow_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow.setter
    def scale_fuel_flow(self, value:bool) -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_scale_fuel_flow_metadata, value)

    _get_attitude_blend_time_metadata = { "offset" : _get_attitude_blend_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def attitude_blend_time(self) -> float:
        """Get or set the amount of time that the aircraft will spend transitioning from the attitude of the previous maneuver to the attitude at the beginning of the current maneuver."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_attitude_blend_time_metadata)

    _set_attitude_blend_time_metadata = { "offset" : _set_attitude_blend_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @attitude_blend_time.setter
    def attitude_blend_time(self, value:float) -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_attitude_blend_time_metadata, value)

    _get_control_time_constant_metadata = { "offset" : _get_control_time_constant_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def control_time_constant(self) -> float:
        """A smoothing constant for the performance of control surfaces."""
        return self._intf.get_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_control_time_constant_metadata)

    _set_control_time_constant_metadata = { "offset" : _set_control_time_constant_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @control_time_constant.setter
    def control_time_constant(self, value:float) -> None:
        return self._intf.set_property(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._set_control_time_constant_metadata, value)

    _get_as_procedure_metadata = { "offset" : _get_as_procedure_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_procedure(self) -> "IProcedure":
        """Get the procedure interface."""
        return self._intf.invoke(ProcedureBasicManeuver._metadata, ProcedureBasicManeuver._get_as_procedure_metadata, OutArg())

    _property_names[max_time_of_flight] = "max_time_of_flight"
    _property_names[use_max_time_of_flight] = "use_max_time_of_flight"
    _property_names[stop_fuel_state] = "stop_fuel_state"
    _property_names[use_stop_fuel_state] = "use_stop_fuel_state"
    _property_names[max_downrange] = "max_downrange"
    _property_names[use_max_downrange] = "use_max_downrange"
    _property_names[altitude_limit_mode] = "altitude_limit_mode"
    _property_names[terrain_impact_mode] = "terrain_impact_mode"
    _property_names[terrain_impact_time_offset] = "terrain_impact_time_offset"
    _property_names[navigation_strategy_type] = "navigation_strategy_type"
    _property_names[navigation] = "navigation"
    _property_names[profile_strategy_type] = "profile_strategy_type"
    _property_names[profile] = "profile"
    _property_names[flight_mode] = "flight_mode"
    _property_names[fuel_flow_type] = "fuel_flow_type"
    _property_names[override_fuel_flow_value] = "override_fuel_flow_value"
    _property_names[scale_fuel_flow] = "scale_fuel_flow"
    _property_names[attitude_blend_time] = "attitude_blend_time"
    _property_names[control_time_constant] = "control_time_constant"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureBasicManeuver."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureBasicManeuver)
        IProcedure.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProcedure._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureBasicManeuver, [ProcedureBasicManeuver, IProcedure])

agcls.AgClassCatalog.add_catalog_entry((5438306318378493224, 5970325316227680169), ProcedureBasicManeuver)
agcls.AgTypeNameMap["ProcedureBasicManeuver"] = ProcedureBasicManeuver

class BasicManeuverStrategyWeave(IBasicManeuverStrategy, SupportsDeleteCallback):
    """Interface used to access options for a weave strategy of a basic maneuver procedure."""

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_heading_change_method_offset = 1
    _set_heading_change_method_offset = 2
    _get_max_num_cycles_method_offset = 3
    _set_max_num_cycles_method_offset = 4
    _get_max_distance_method_offset = 5
    _set_max_distance_method_offset = 6
    _get_control_limit_mode_method_offset = 7
    _get_control_limit_turn_radius_method_offset = 8
    _get_control_limit_turn_rate_method_offset = 9
    _get_control_limit_horizontal_acceleration_method_offset = 10
    _set_control_limit_method_offset = 11
    _get_compensate_for_coriolis_acceleration_method_offset = 12
    _set_compensate_for_coriolis_acceleration_method_offset = 13
    _metadata = {
        "iid_data" : (5498161033082324708, 15050332780891237523),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BasicManeuverStrategyWeave)

    _get_heading_change_metadata = { "offset" : _get_heading_change_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def heading_change(self) -> typing.Any:
        """Get or set the direction in which the aircraft will begin the weave pattern."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_heading_change_metadata)

    _set_heading_change_metadata = { "offset" : _set_heading_change_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @heading_change.setter
    def heading_change(self, value:typing.Any) -> None:
        return self._intf.set_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._set_heading_change_metadata, value)

    _get_max_num_cycles_metadata = { "offset" : _get_max_num_cycles_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_num_cycles(self) -> float:
        """Get or set the number of times the aircraft will fly the pattern."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_max_num_cycles_metadata)

    _set_max_num_cycles_metadata = { "offset" : _set_max_num_cycles_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_num_cycles.setter
    def max_num_cycles(self, value:float) -> None:
        return self._intf.set_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._set_max_num_cycles_metadata, value)

    _get_max_distance_metadata = { "offset" : _get_max_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_distance(self) -> float:
        """Get or set the maximum ground distance the aircraft will travel while performing the weave cyces."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_max_distance_metadata)

    _set_max_distance_metadata = { "offset" : _set_max_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_distance.setter
    def max_distance(self, value:float) -> None:
        return self._intf.set_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._set_max_distance_metadata, value)

    _get_control_limit_mode_metadata = { "offset" : _get_control_limit_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverStrategyNavigationControlLimit),) }
    @property
    def control_limit_mode(self) -> "BasicManeuverStrategyNavigationControlLimit":
        """Get the method to define the control limits of the aircraft during the maneuver."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_control_limit_mode_metadata)

    _get_control_limit_turn_radius_metadata = { "offset" : _get_control_limit_turn_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def control_limit_turn_radius(self) -> float:
        """Get the specified turn radius for a control limit mode of specify min turn radius."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_control_limit_turn_radius_metadata)

    _get_control_limit_turn_rate_metadata = { "offset" : _get_control_limit_turn_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def control_limit_turn_rate(self) -> typing.Any:
        """Get the specified turn rate for a control limit mode of specify max turn rate."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_control_limit_turn_rate_metadata)

    _get_control_limit_horizontal_acceleration_metadata = { "offset" : _get_control_limit_horizontal_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def control_limit_horizontal_acceleration(self) -> float:
        """Get the specified horizontal acceleration for a control limit mode of specify max horiz accel."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_control_limit_horizontal_acceleration_metadata)

    _set_control_limit_metadata = { "offset" : _set_control_limit_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverStrategyNavigationControlLimit), agmarshall.DoubleArg,) }
    def set_control_limit(self, control_limit_mode:"BasicManeuverStrategyNavigationControlLimit", control_limit_value:float) -> None:
        """Set the method and corresponding value to define the control limits of the aircraft during the maneuver."""
        return self._intf.invoke(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._set_control_limit_metadata, control_limit_mode, control_limit_value)

    _get_compensate_for_coriolis_acceleration_metadata = { "offset" : _get_compensate_for_coriolis_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def compensate_for_coriolis_acceleration(self) -> bool:
        """Get or set the option to compensate for the acceleration due to the Coriolis effect."""
        return self._intf.get_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._get_compensate_for_coriolis_acceleration_metadata)

    _set_compensate_for_coriolis_acceleration_metadata = { "offset" : _set_compensate_for_coriolis_acceleration_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @compensate_for_coriolis_acceleration.setter
    def compensate_for_coriolis_acceleration(self, value:bool) -> None:
        return self._intf.set_property(BasicManeuverStrategyWeave._metadata, BasicManeuverStrategyWeave._set_compensate_for_coriolis_acceleration_metadata, value)

    _property_names[heading_change] = "heading_change"
    _property_names[max_num_cycles] = "max_num_cycles"
    _property_names[max_distance] = "max_distance"
    _property_names[control_limit_mode] = "control_limit_mode"
    _property_names[control_limit_turn_radius] = "control_limit_turn_radius"
    _property_names[control_limit_turn_rate] = "control_limit_turn_rate"
    _property_names[control_limit_horizontal_acceleration] = "control_limit_horizontal_acceleration"
    _property_names[compensate_for_coriolis_acceleration] = "compensate_for_coriolis_acceleration"

    def __init__(self, source_object=None):
        """Construct an object of type BasicManeuverStrategyWeave."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BasicManeuverStrategyWeave)
        IBasicManeuverStrategy.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBasicManeuverStrategy._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BasicManeuverStrategyWeave, [BasicManeuverStrategyWeave, IBasicManeuverStrategy])

agcls.AgClassCatalog.add_catalog_entry((4741432811209582740, 17876430357664068016), BasicManeuverStrategyWeave)
agcls.AgTypeNameMap["BasicManeuverStrategyWeave"] = BasicManeuverStrategyWeave

class ProcedureTimeOptions(SupportsDeleteCallback):
    """
    Interface used to access the time options for the current procedure. Use this interface to set an Interrupt Time or Fixed Duration for a procedure.

    Examples
    --------
    Configure a procedure's time options:
    >>> # IProcedure procedure: Procedure object
    >>> # Get the time in epoch seconds
    >>> root.units_preferences.set_current_unit("DateFormat", "EpSec")
    >>> # Get the time options
    >>> timeOptions = procedure.time_options
    >>> # Get the start time
    >>> startTime = timeOptions.start_time
    >>> # Set the procedure to interrupt after 15 seconds
    >>> timeOptions.set_interrupt_time(15)
    """

    _num_methods = 15
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_start_time_enabled_method_offset = 1
    _get_use_start_time_method_offset = 2
    _set_use_start_time_method_offset = 3
    _get_start_time_method_offset = 4
    _set_start_time_method_offset = 5
    _get_interrupt_time_enabled_method_offset = 6
    _get_use_interrupt_time_method_offset = 7
    _set_use_interrupt_time_method_offset = 8
    _get_interrupt_time_method_offset = 9
    _set_interrupt_time_method_offset = 10
    _get_stop_time_enabled_method_offset = 11
    _get_use_stop_time_method_offset = 12
    _set_use_stop_time_method_offset = 13
    _get_stop_time_method_offset = 14
    _set_stop_time_method_offset = 15
    _metadata = {
        "iid_data" : (5431356329456530164, 11282614112339302277),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProcedureTimeOptions)

    _get_start_time_enabled_metadata = { "offset" : _get_start_time_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def start_time_enabled(self) -> bool:
        """Check to see if the start time is enabled for this procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_start_time_enabled_metadata)

    _get_use_start_time_metadata = { "offset" : _get_use_start_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_start_time(self) -> bool:
        """Opt whether to set a start time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_use_start_time_metadata)

    _set_use_start_time_metadata = { "offset" : _set_use_start_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_start_time.setter
    def use_start_time(self, value:bool) -> None:
        """Opt whether to set a start time for the procedure."""
        return self._intf.set_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_use_start_time_metadata, value)

    _get_start_time_metadata = { "offset" : _get_start_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def start_time(self) -> typing.Any:
        """Start time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_start_time_metadata)

    _set_start_time_metadata = { "offset" : _set_start_time_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_start_time(self, time:typing.Any) -> None:
        """Set the start time for the procedure."""
        return self._intf.invoke(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_start_time_metadata, time)

    _get_interrupt_time_enabled_metadata = { "offset" : _get_interrupt_time_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def interrupt_time_enabled(self) -> bool:
        """Check to see if the interrupt time is enabled for this procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_interrupt_time_enabled_metadata)

    _get_use_interrupt_time_metadata = { "offset" : _get_use_interrupt_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_interrupt_time(self) -> bool:
        """Opt whether to set an interrupt time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_use_interrupt_time_metadata)

    _set_use_interrupt_time_metadata = { "offset" : _set_use_interrupt_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_interrupt_time.setter
    def use_interrupt_time(self, value:bool) -> None:
        """Opt whether to set an interrupt time for the procedure."""
        return self._intf.set_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_use_interrupt_time_metadata, value)

    _get_interrupt_time_metadata = { "offset" : _get_interrupt_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def interrupt_time(self) -> typing.Any:
        """Interrupt time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_interrupt_time_metadata)

    _set_interrupt_time_metadata = { "offset" : _set_interrupt_time_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_interrupt_time(self, time:typing.Any) -> None:
        """Set the interrupt time for the procedure."""
        return self._intf.invoke(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_interrupt_time_metadata, time)

    _get_stop_time_enabled_metadata = { "offset" : _get_stop_time_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def stop_time_enabled(self) -> bool:
        """Check to see if the stop time is enabled for this procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_stop_time_enabled_metadata)

    _get_use_stop_time_metadata = { "offset" : _get_use_stop_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_stop_time(self) -> bool:
        """Opt whether to set a stop  time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_use_stop_time_metadata)

    _set_use_stop_time_metadata = { "offset" : _set_use_stop_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_stop_time.setter
    def use_stop_time(self, value:bool) -> None:
        """Opt whether to set a stop time for the procedure."""
        return self._intf.set_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_use_stop_time_metadata, value)

    _get_stop_time_metadata = { "offset" : _get_stop_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def stop_time(self) -> typing.Any:
        """Stop time for the procedure."""
        return self._intf.get_property(ProcedureTimeOptions._metadata, ProcedureTimeOptions._get_stop_time_metadata)

    _set_stop_time_metadata = { "offset" : _set_stop_time_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_stop_time(self, time:typing.Any) -> None:
        """Set the stop time for the procedure."""
        return self._intf.invoke(ProcedureTimeOptions._metadata, ProcedureTimeOptions._set_stop_time_metadata, time)

    _property_names[start_time_enabled] = "start_time_enabled"
    _property_names[use_start_time] = "use_start_time"
    _property_names[start_time] = "start_time"
    _property_names[interrupt_time_enabled] = "interrupt_time_enabled"
    _property_names[use_interrupt_time] = "use_interrupt_time"
    _property_names[interrupt_time] = "interrupt_time"
    _property_names[stop_time_enabled] = "stop_time_enabled"
    _property_names[use_stop_time] = "use_stop_time"
    _property_names[stop_time] = "stop_time"

    def __init__(self, source_object=None):
        """Construct an object of type ProcedureTimeOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProcedureTimeOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProcedureTimeOptions, [ProcedureTimeOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4806824256453033598, 10159111947367507632), ProcedureTimeOptions)
agcls.AgTypeNameMap["ProcedureTimeOptions"] = ProcedureTimeOptions

class CalculationOptions(SupportsDeleteCallback):
    """Interface used to access the calculation options for a procedure or phase."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_relative_motion_factor_method_offset = 1
    _set_max_relative_motion_factor_method_offset = 2
    _get_state_cache_time_interval_method_offset = 3
    _set_state_cache_time_interval_method_offset = 4
    _get_time_resolution_method_offset = 5
    _set_time_resolution_method_offset = 6
    _get_max_iterations_method_offset = 7
    _set_max_iterations_method_offset = 8
    _get_max_bad_steps_method_offset = 9
    _set_max_bad_steps_method_offset = 10
    _get_integrator_type_method_offset = 11
    _set_integrator_type_method_offset = 12
    _get_integrator_type_string_method_offset = 13
    _set_integrator_type_string_method_offset = 14
    _metadata = {
        "iid_data" : (4930309390064999262, 5777468632786382513),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationOptions)

    _get_max_relative_motion_factor_metadata = { "offset" : _get_max_relative_motion_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_relative_motion_factor(self) -> float:
        """Get or set the maximum motion allowed between sampling points."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_max_relative_motion_factor_metadata)

    _set_max_relative_motion_factor_metadata = { "offset" : _set_max_relative_motion_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_relative_motion_factor.setter
    def max_relative_motion_factor(self, value:float) -> None:
        """Get or set the maximum motion allowed between sampling points."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_max_relative_motion_factor_metadata, value)

    _get_state_cache_time_interval_metadata = { "offset" : _get_state_cache_time_interval_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def state_cache_time_interval(self) -> float:
        """Get or set the time interval used to store the state information."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_state_cache_time_interval_metadata)

    _set_state_cache_time_interval_metadata = { "offset" : _set_state_cache_time_interval_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @state_cache_time_interval.setter
    def state_cache_time_interval(self, value:float) -> None:
        """Get or set the time interval used to store the state information."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_state_cache_time_interval_metadata, value)

    _get_time_resolution_metadata = { "offset" : _get_time_resolution_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_resolution(self) -> float:
        """Get or set the tolerance for resolving time calculations."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_time_resolution_metadata)

    _set_time_resolution_metadata = { "offset" : _set_time_resolution_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_resolution.setter
    def time_resolution(self, value:float) -> None:
        """Get or set the tolerance for resolving time calculations."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_time_resolution_metadata, value)

    _get_max_iterations_metadata = { "offset" : _get_max_iterations_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def max_iterations(self) -> int:
        """Get or set the maximum number of iterations per time step Aviator will attempt."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_max_iterations_metadata)

    _set_max_iterations_metadata = { "offset" : _set_max_iterations_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @max_iterations.setter
    def max_iterations(self, value:int) -> None:
        """Get or set the maximum number of iterations per time step Aviator will attempt."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_max_iterations_metadata, value)

    _get_max_bad_steps_metadata = { "offset" : _get_max_bad_steps_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def max_bad_steps(self) -> int:
        """Get or set the maximum number of bad steps Aviator will allow before ceasing calculation."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_max_bad_steps_metadata)

    _set_max_bad_steps_metadata = { "offset" : _set_max_bad_steps_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @max_bad_steps.setter
    def max_bad_steps(self, value:int) -> None:
        """Get or set the maximum number of bad steps Aviator will allow before ceasing calculation."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_max_bad_steps_metadata, value)

    _get_integrator_type_metadata = { "offset" : _get_integrator_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AviatorNumericalIntegrator),) }
    @property
    def integrator_type(self) -> "AviatorNumericalIntegrator":
        """Get or set the integrator type."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_integrator_type_metadata)

    _set_integrator_type_metadata = { "offset" : _set_integrator_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AviatorNumericalIntegrator),) }
    @integrator_type.setter
    def integrator_type(self, value:"AviatorNumericalIntegrator") -> None:
        """Get or set the integrator type."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_integrator_type_metadata, value)

    _get_integrator_type_string_metadata = { "offset" : _get_integrator_type_string_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def integrator_type_string(self) -> str:
        """Get or set the integrator type as a string value. Use this for custom integrators."""
        return self._intf.get_property(CalculationOptions._metadata, CalculationOptions._get_integrator_type_string_metadata)

    _set_integrator_type_string_metadata = { "offset" : _set_integrator_type_string_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @integrator_type_string.setter
    def integrator_type_string(self, value:str) -> None:
        """Get or set the integrator type as a string value. Use this for custom integrators."""
        return self._intf.set_property(CalculationOptions._metadata, CalculationOptions._set_integrator_type_string_metadata, value)

    _property_names[max_relative_motion_factor] = "max_relative_motion_factor"
    _property_names[state_cache_time_interval] = "state_cache_time_interval"
    _property_names[time_resolution] = "time_resolution"
    _property_names[max_iterations] = "max_iterations"
    _property_names[max_bad_steps] = "max_bad_steps"
    _property_names[integrator_type] = "integrator_type"
    _property_names[integrator_type_string] = "integrator_type_string"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationOptions, [CalculationOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5747373335486459111, 3684632747864185757), CalculationOptions)
agcls.AgTypeNameMap["CalculationOptions"] = CalculationOptions

class AircraftCategory(SupportsDeleteCallback):
    """
    Interface used to access the Aircraft Category in the Aviator Catalog.

    Examples
    --------
    Set the aircraft used for the mission to an aircraft found in the Aviator catalog:
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Get the Aviator catalog
    >>> catalog = propagator.aviator_catalog
    >>> # Get the aircraft category
    >>> category = catalog.aircraft_category
    >>> # Get the user aircraft models
    >>> aircraftModels = category.aircraft_models
    >>> # Get the basic fighter
    >>> fighter = aircraftModels.get_aircraft("Basic Fighter")
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Set the vehicle used for the mission
    >>> mission.vehicle = fighter
    """

    _num_methods = 3
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _get_aircraft_models_method_offset = 1
    _get_missile_models_method_offset = 2
    _get_rotorcraft_models_method_offset = 3
    _metadata = {
        "iid_data" : (5393766329145449059, 6703502100005194163),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftCategory)

    _get_aircraft_models_metadata = { "offset" : _get_aircraft_models_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aircraft_models(self) -> "AircraftModels":
        """Get the user aircraft models."""
        return self._intf.get_property(AircraftCategory._metadata, AircraftCategory._get_aircraft_models_metadata)

    _get_missile_models_metadata = { "offset" : _get_missile_models_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def missile_models(self) -> "MissileModels":
        """Get the user missile models."""
        return self._intf.get_property(AircraftCategory._metadata, AircraftCategory._get_missile_models_metadata)

    _get_rotorcraft_models_metadata = { "offset" : _get_rotorcraft_models_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def rotorcraft_models(self) -> "RotorcraftModels":
        """Get the user rotorcraft models."""
        return self._intf.get_property(AircraftCategory._metadata, AircraftCategory._get_rotorcraft_models_metadata)

    _property_names[aircraft_models] = "aircraft_models"
    _property_names[missile_models] = "missile_models"
    _property_names[rotorcraft_models] = "rotorcraft_models"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftCategory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftCategory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftCategory, [AircraftCategory, ])

agcls.AgClassCatalog.add_catalog_entry((5038526480345777203, 17565896557184368030), AircraftCategory)
agcls.AgTypeNameMap["AircraftCategory"] = AircraftCategory

class Catalog(SupportsDeleteCallback):
    """
    Interface used to access the Aviator catalog.

    Examples
    --------
    Set the aircraft used for the mission to an aircraft found in the Aviator catalog:
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Get the Aviator catalog
    >>> catalog = propagator.aviator_catalog
    >>> # Get the aircraft category
    >>> category = catalog.aircraft_category
    >>> # Get the user aircraft models
    >>> aircraftModels = category.aircraft_models
    >>> # Get the basic fighter
    >>> fighter = aircraftModels.get_aircraft("Basic Fighter")
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Set the vehicle used for the mission
    >>> mission.vehicle = fighter

    Configure a runway site from a runway in the Aviator catalog:
    >>> # SiteRunway runway: Runway object
    >>> # Catalog catalog: Aviator catalog object
    >>> # Get the source of user runways
    >>> userRunways = catalog.runway_category.user_runways
    >>> # Check that the runway exists in the catalog
    >>> if userRunways.contains("New User Runway") is True:
    >>>     # If so, get the user runway with the given name
    >>>     runwayFromCatalog = userRunways.get_user_runway("New User Runway")
    >>>     # Copy the parameters of that runway
    >>>     runway.copy_from_catalog(runwayFromCatalog)
    """

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_aircraft_category_method_offset = 1
    _get_runway_category_method_offset = 2
    _get_airport_category_method_offset = 3
    _get_navaid_category_method_offset = 4
    _get_vtol_point_category_method_offset = 5
    _get_waypoint_category_method_offset = 6
    _metadata = {
        "iid_data" : (4867560862593147186, 13685186326758395068),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, Catalog)

    _get_aircraft_category_metadata = { "offset" : _get_aircraft_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aircraft_category(self) -> "AircraftCategory":
        """Get the aircraft category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_aircraft_category_metadata)

    _get_runway_category_metadata = { "offset" : _get_runway_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def runway_category(self) -> "RunwayCategory":
        """Get the runway category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_runway_category_metadata)

    _get_airport_category_metadata = { "offset" : _get_airport_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def airport_category(self) -> "AirportCategory":
        """Get the airport category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_airport_category_metadata)

    _get_navaid_category_metadata = { "offset" : _get_navaid_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def navaid_category(self) -> "NavaidCategory":
        """Get the navaid category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_navaid_category_metadata)

    _get_vtol_point_category_metadata = { "offset" : _get_vtol_point_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vtol_point_category(self) -> "VTOLPointCategory":
        """Get the vtol point category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_vtol_point_category_metadata)

    _get_waypoint_category_metadata = { "offset" : _get_waypoint_category_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def waypoint_category(self) -> "WaypointCategory":
        """Get the waypoint category."""
        return self._intf.get_property(Catalog._metadata, Catalog._get_waypoint_category_metadata)

    _property_names[aircraft_category] = "aircraft_category"
    _property_names[runway_category] = "runway_category"
    _property_names[airport_category] = "airport_category"
    _property_names[navaid_category] = "navaid_category"
    _property_names[vtol_point_category] = "vtol_point_category"
    _property_names[waypoint_category] = "waypoint_category"

    def __init__(self, source_object=None):
        """Construct an object of type Catalog."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, Catalog)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, Catalog, [Catalog, ])

agcls.AgClassCatalog.add_catalog_entry((4907045706268839733, 1198668682488985004), Catalog)
agcls.AgTypeNameMap["Catalog"] = Catalog

class AircraftModel(IAviatorVehicle, ICatalogItem, SupportsDeleteCallback):
    """
    Interface used to access the aircraft options in the Aviator catalog.

    Examples
    --------
    Configure the Advanced Fixed Wing Tool and set the aircraft to use the resulting performance models:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the advanced fixed wing tool
    >>> advFixedWingTool = aviatorAircraft.advanced_fixed_wing_tool
    >>> # Set the basic geometry
    >>> advFixedWingTool.wing_area = 300
    >>> advFixedWingTool.flaps_area = 50
    >>> advFixedWingTool.speedbrakes_area = 10
    >>> # Set the structural and human factor limits
    >>> advFixedWingTool.max_altitude = 65000
    >>> advFixedWingTool.max_mach = 0.98
    >>> advFixedWingTool.max_eas = 460
    >>> advFixedWingTool.min_load_factor = -2.5
    >>> advFixedWingTool.max_load_factor = 4.5
    >>>
    >>> # Opt to enforce the max temperature limit
    >>> advFixedWingTool.use_max_temperature_limit = True
    >>> advFixedWingTool.max_temperature = 900
    >>>
    >>> # Use a subsonic aerodynamic strategy
    >>> advFixedWingTool.aerodynamic_strategy = AdvancedFixedWingAerodynamicStrategy.SUBSONIC_AERODYNAMIC
    >>> # Cache the aerodynamic data to improve calculation speed
    >>> advFixedWingTool.cache_aerodynamic_data = True
    >>> # Use a high bypass turbofan
    >>> advFixedWingTool.powerplant_strategy = AdvancedFixedWingPowerplantStrategy.TURBOFAN_HIGH_BYPASS
    >>> # Cache the fuel flow data to improve calculation speed
    >>> advFixedWingTool.cache_fuel_flow = True
    >>>
    >>> # Create the corresponding performance models that reference the advanced fixed wing tool
    >>> # Specify the name, whether to override any existing models with the same name, and whether to set the new models as the default performance models
    >>> advFixedWingTool.create_all_performance_models("AdvancedModels", True, True)
    >>>
    >>> # Save the changes in the catalog
    >>> aviatorAircraft.save()

    Set the aircraft used for the mission to an aircraft found in the Aviator catalog:
    >>> # AviatorPropagator propagator: Aviator Propagator object
    >>> # Get the Aviator catalog
    >>> catalog = propagator.aviator_catalog
    >>> # Get the aircraft category
    >>> category = catalog.aircraft_category
    >>> # Get the user aircraft models
    >>> aircraftModels = category.aircraft_models
    >>> # Get the basic fighter
    >>> fighter = aircraftModels.get_aircraft("Basic Fighter")
    >>> # Get the mission
    >>> mission = propagator.aviator_mission
    >>> # Set the vehicle used for the mission
    >>> mission.vehicle = fighter

    Create a new performance model for an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the names of the current acceleration models
    >>> modelNames = acceleration.child_names
    >>> # Check how many models there are
    >>> modelCount = len(modelNames)
    >>> # Get the child types (for example AGI Basic Acceleration Model, Advanced Acceleration Model)
    >>> modelTypes = acceleration.child_types
    >>> # Create a new performance model of type "Advanced Acceleration Model"
    >>> newPerformanceModel = acceleration.add_child_of_type("Advanced Acceleration Model", "Model Name")
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()

    Configure the basic cruise performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the cruise type
    >>> cruise = aviatorAircraft.cruise
    >>> # Get the build in performance model
    >>> basicCruiseModel = cruise.get_built_in_model()
    >>>
    >>> # Set the ceiling altitude
    >>> basicCruiseModel.ceiling_altitude = 50000
    >>> # Set the default cruise altitude
    >>> basicCruiseModel.default_cruise_altitude = 10000
    >>> # Set the airspeed type
    >>> basicCruiseModel.airspeed_type = AirspeedType.TAS
    >>> # Opt to not use the fuel flow calculated by the aero/prop model and instead specify the values
    >>> basicCruiseModel.use_aerodynamic_propulsion_fuel = False
    >>>
    >>> # Set the various airspeeds and fuel flows
    >>> basicCruiseModel.min_airspeed = 110
    >>> basicCruiseModel.min_airspeed_fuel_flow = 10000
    >>>
    >>> basicCruiseModel.max_endurance_airspeed = 135
    >>> basicCruiseModel.max_endurance_fuel_flow = 8000
    >>>
    >>> basicCruiseModel.max_airspeed = 570
    >>> basicCruiseModel.max_airspeed_fuel_flow = 30000
    >>>
    >>> basicCruiseModel.max_range_airspeed = 140
    >>> basicCruiseModel.max_range_fuel_flow = 9000
    >>>
    >>> basicCruiseModel.max_performance_airspeed = 150
    >>> basicCruiseModel.max_performance_airspeed_fuel_flow = 12000
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()

    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_performance_model_types_method_offset = 1
    _get_acceleration_method_offset = 2
    _get_climb_method_offset = 3
    _get_cruise_method_offset = 4
    _get_descent_method_offset = 5
    _get_landing_method_offset = 6
    _get_takeoff_method_offset = 7
    _get_default_configuration_method_offset = 8
    _get_advanced_fixed_wing_tool_method_offset = 9
    _get_as_catalog_item_method_offset = 10
    _get_vtol_method_offset = 11
    _get_terrain_follow_method_offset = 12
    _metadata = {
        "iid_data" : (4972221985443648774, 11607092758271299260),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftModel)

    _get_performance_model_types_metadata = { "offset" : _get_performance_model_types_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def performance_model_types(self) -> list:
        """Get the types of performance models."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_performance_model_types_metadata)

    _get_acceleration_metadata = { "offset" : _get_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def acceleration(self) -> "AircraftAcceleration":
        """Get the acceleration interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_acceleration_metadata)

    _get_climb_metadata = { "offset" : _get_climb_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def climb(self) -> "AircraftClimb":
        """Get the climb interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_climb_metadata)

    _get_cruise_metadata = { "offset" : _get_cruise_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def cruise(self) -> "AircraftCruise":
        """Get the cruise interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_cruise_metadata)

    _get_descent_metadata = { "offset" : _get_descent_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def descent(self) -> "AircraftDescent":
        """Get the descent interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_descent_metadata)

    _get_landing_metadata = { "offset" : _get_landing_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def landing(self) -> "AircraftLanding":
        """Get the landing interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_landing_metadata)

    _get_takeoff_metadata = { "offset" : _get_takeoff_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def takeoff(self) -> "AircraftTakeoff":
        """Get the takeoff interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_takeoff_metadata)

    _get_default_configuration_metadata = { "offset" : _get_default_configuration_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def default_configuration(self) -> "Configuration":
        """Get the aircraft's default configuration as saved in the catalog."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_default_configuration_metadata)

    _get_advanced_fixed_wing_tool_metadata = { "offset" : _get_advanced_fixed_wing_tool_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def advanced_fixed_wing_tool(self) -> "AdvancedFixedWingTool":
        """Get the Advanced Fixed Wing Tool for the aircraft."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_advanced_fixed_wing_tool_metadata)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftModel._metadata, AircraftModel._get_as_catalog_item_metadata, OutArg())

    _get_vtol_metadata = { "offset" : _get_vtol_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vtol(self) -> "AircraftVTOL":
        """Get the VTOL interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_vtol_metadata)

    _get_terrain_follow_metadata = { "offset" : _get_terrain_follow_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def terrain_follow(self) -> "AircraftTerrainFollow":
        """Get the TerrainFollow interface."""
        return self._intf.get_property(AircraftModel._metadata, AircraftModel._get_terrain_follow_metadata)

    _property_names[performance_model_types] = "performance_model_types"
    _property_names[acceleration] = "acceleration"
    _property_names[climb] = "climb"
    _property_names[cruise] = "cruise"
    _property_names[descent] = "descent"
    _property_names[landing] = "landing"
    _property_names[takeoff] = "takeoff"
    _property_names[default_configuration] = "default_configuration"
    _property_names[advanced_fixed_wing_tool] = "advanced_fixed_wing_tool"
    _property_names[vtol] = "vtol"
    _property_names[terrain_follow] = "terrain_follow"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftModel)
        IAviatorVehicle.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAviatorVehicle._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftModel, [AircraftModel, IAviatorVehicle, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5272649341851046442, 10204808036967213963), AircraftModel)
agcls.AgTypeNameMap["AircraftModel"] = AircraftModel

class MissileModel(IAviatorVehicle, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the missile options in the Aviator catalog."""

    _num_methods = 47
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_load_factor_method_offset = 1
    _set_max_load_factor_method_offset = 2
    _get_maneuver_mode_method_offset = 3
    _set_maneuver_mode_method_offset = 4
    _get_maneuver_mode_helper_method_offset = 5
    _get_attitude_transitions_method_offset = 6
    _get_ignore_flight_path_angle_for_climb_descent_transitions_method_offset = 7
    _set_ignore_flight_path_angle_for_climb_descent_transitions_method_offset = 8
    _get_climb_airspeed_method_offset = 9
    _get_climb_airspeed_type_method_offset = 10
    _set_climb_airspeed_method_offset = 11
    _get_climb_max_flight_path_angle_method_offset = 12
    _set_climb_max_flight_path_angle_method_offset = 13
    _get_climb_min_flight_path_angle_method_offset = 14
    _set_climb_min_flight_path_angle_method_offset = 15
    _get_climb_fail_on_insufficient_performance_method_offset = 16
    _set_climb_fail_on_insufficient_performance_method_offset = 17
    _get_cruise_max_airspeed_method_offset = 18
    _get_cruise_max_airspeed_type_method_offset = 19
    _set_cruise_max_airspeed_method_offset = 20
    _get_cruise_default_altitude_method_offset = 21
    _set_cruise_default_altitude_method_offset = 22
    _get_descent_airspeed_method_offset = 23
    _get_descent_airspeed_type_method_offset = 24
    _set_descent_airspeed_method_offset = 25
    _get_descent_max_flight_path_angle_method_offset = 26
    _set_descent_max_flight_path_angle_method_offset = 27
    _get_descent_min_flight_path_angle_method_offset = 28
    _set_descent_min_flight_path_angle_method_offset = 29
    _get_descent_fail_on_insufficient_performance_method_offset = 30
    _set_descent_fail_on_insufficient_performance_method_offset = 31
    _get_use_total_temp_limit_method_offset = 32
    _set_use_total_temp_limit_method_offset = 33
    _get_total_temp_limit_method_offset = 34
    _set_total_temp_limit_method_offset = 35
    _get_use_mach_limit_method_offset = 36
    _set_use_mach_limit_method_offset = 37
    _get_mach_limit_method_offset = 38
    _set_mach_limit_method_offset = 39
    _get_use_eas_limit_method_offset = 40
    _set_use_eas_limit_method_offset = 41
    _get_eas_limit_method_offset = 42
    _set_eas_limit_method_offset = 43
    _get_default_configuration_method_offset = 44
    _get_aerodynamics_method_offset = 45
    _get_propulsion_method_offset = 46
    _get_as_catalog_item_method_offset = 47
    _metadata = {
        "iid_data" : (5036037641348746326, 5450613477693494170),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MissileModel)

    _get_max_load_factor_metadata = { "offset" : _get_max_load_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_load_factor(self) -> float:
        """Get or set the maximum load factor that the missile can withstand while maneuvering."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_max_load_factor_metadata)

    _set_max_load_factor_metadata = { "offset" : _set_max_load_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_load_factor.setter
    def max_load_factor(self, value:float) -> None:
        """Get or set the maximum load factor that the missile can withstand while maneuvering."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_max_load_factor_metadata, value)

    _get_maneuver_mode_metadata = { "offset" : _get_maneuver_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @property
    def maneuver_mode(self) -> "AccelerationManeuverMode":
        """Get or set the mode that the missile will adhere to the specified load factor. Scale by atmospheric density will cause the missile to consider dynamic pressure when calculating turn radius."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_maneuver_mode_metadata)

    _set_maneuver_mode_metadata = { "offset" : _set_maneuver_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @maneuver_mode.setter
    def maneuver_mode(self, value:"AccelerationManeuverMode") -> None:
        """Get or set the mode that the missile will adhere to the specified load factor. Scale by atmospheric density will cause the missile to consider dynamic pressure when calculating turn radius."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_maneuver_mode_metadata, value)

    _get_maneuver_mode_helper_metadata = { "offset" : _get_maneuver_mode_helper_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maneuver_mode_helper(self) -> "AerodynamicPropulsionManeuverModeHelper":
        """Get the interface for the Aero/Prop Maneuver Mode helper. The maneuver mode must be set to Aero/Prop to access this interface."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_maneuver_mode_helper_metadata)

    _get_attitude_transitions_metadata = { "offset" : _get_attitude_transitions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def attitude_transitions(self) -> "AttitudeTransitions":
        """Get the attitude transitions interface."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_attitude_transitions_metadata)

    _get_ignore_flight_path_angle_for_climb_descent_transitions_metadata = { "offset" : _get_ignore_flight_path_angle_for_climb_descent_transitions_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def ignore_flight_path_angle_for_climb_descent_transitions(self) -> bool:
        """Opt whether to ignore the flight path angle limits for climb and descent transitions."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_ignore_flight_path_angle_for_climb_descent_transitions_metadata)

    _set_ignore_flight_path_angle_for_climb_descent_transitions_metadata = { "offset" : _set_ignore_flight_path_angle_for_climb_descent_transitions_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @ignore_flight_path_angle_for_climb_descent_transitions.setter
    def ignore_flight_path_angle_for_climb_descent_transitions(self, value:bool) -> None:
        """Opt whether to ignore the flight path angle limits for climb and descent transitions."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_ignore_flight_path_angle_for_climb_descent_transitions_metadata, value)

    _get_climb_airspeed_metadata = { "offset" : _get_climb_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def climb_airspeed(self) -> float:
        """Get the standard airspeed of the missile while climbing."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_climb_airspeed_metadata)

    _get_climb_airspeed_type_metadata = { "offset" : _get_climb_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def climb_airspeed_type(self) -> "AirspeedType":
        """Get the climb airspeed type."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_climb_airspeed_type_metadata)

    _set_climb_airspeed_metadata = { "offset" : _set_climb_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_climb_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the missile's climb airspeed and airspeed type."""
        return self._intf.invoke(MissileModel._metadata, MissileModel._set_climb_airspeed_metadata, airspeed_type, airspeed)

    _get_climb_max_flight_path_angle_metadata = { "offset" : _get_climb_max_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def climb_max_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum flight path angle of the missile's flight path while climbing."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_climb_max_flight_path_angle_metadata)

    _set_climb_max_flight_path_angle_metadata = { "offset" : _set_climb_max_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @climb_max_flight_path_angle.setter
    def climb_max_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum flight path angle of the missile's flight path while climbing."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_climb_max_flight_path_angle_metadata, value)

    _get_climb_min_flight_path_angle_metadata = { "offset" : _get_climb_min_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def climb_min_flight_path_angle(self) -> typing.Any:
        """Get or set the minimum flight path angle of the missile's flight path while climbing."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_climb_min_flight_path_angle_metadata)

    _set_climb_min_flight_path_angle_metadata = { "offset" : _set_climb_min_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @climb_min_flight_path_angle.setter
    def climb_min_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the minimum flight path angle of the missile's flight path while climbing."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_climb_min_flight_path_angle_metadata, value)

    _get_climb_fail_on_insufficient_performance_metadata = { "offset" : _get_climb_fail_on_insufficient_performance_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def climb_fail_on_insufficient_performance(self) -> bool:
        """Opt whether to fail while climbing if there is insufficient specific excess power."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_climb_fail_on_insufficient_performance_metadata)

    _set_climb_fail_on_insufficient_performance_metadata = { "offset" : _set_climb_fail_on_insufficient_performance_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @climb_fail_on_insufficient_performance.setter
    def climb_fail_on_insufficient_performance(self, value:bool) -> None:
        """Opt whether to fail while climbing if there is insufficient specific excess power."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_climb_fail_on_insufficient_performance_metadata, value)

    _get_cruise_max_airspeed_metadata = { "offset" : _get_cruise_max_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cruise_max_airspeed(self) -> float:
        """Get the maximum airspeed of the missile while cruising."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_cruise_max_airspeed_metadata)

    _get_cruise_max_airspeed_type_metadata = { "offset" : _get_cruise_max_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def cruise_max_airspeed_type(self) -> "AirspeedType":
        """Get the cruise airspeed type."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_cruise_max_airspeed_type_metadata)

    _set_cruise_max_airspeed_metadata = { "offset" : _set_cruise_max_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_cruise_max_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the missile's max cruise airspeed and airspeed type."""
        return self._intf.invoke(MissileModel._metadata, MissileModel._set_cruise_max_airspeed_metadata, airspeed_type, airspeed)

    _get_cruise_default_altitude_metadata = { "offset" : _get_cruise_default_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cruise_default_altitude(self) -> float:
        """Get or set the missile's default cruising altitude."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_cruise_default_altitude_metadata)

    _set_cruise_default_altitude_metadata = { "offset" : _set_cruise_default_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cruise_default_altitude.setter
    def cruise_default_altitude(self, value:float) -> None:
        """Get or set the missile's default cruising altitude."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_cruise_default_altitude_metadata, value)

    _get_descent_airspeed_metadata = { "offset" : _get_descent_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def descent_airspeed(self) -> float:
        """Get the standard airspeed of the missile while descending."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_descent_airspeed_metadata)

    _get_descent_airspeed_type_metadata = { "offset" : _get_descent_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def descent_airspeed_type(self) -> "AirspeedType":
        """Get the descent airspeed type."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_descent_airspeed_type_metadata)

    _set_descent_airspeed_metadata = { "offset" : _set_descent_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_descent_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the missile's descent airspeed and airspeed type."""
        return self._intf.invoke(MissileModel._metadata, MissileModel._set_descent_airspeed_metadata, airspeed_type, airspeed)

    _get_descent_max_flight_path_angle_metadata = { "offset" : _get_descent_max_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def descent_max_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum flight path angle of the missile's flight path while descending."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_descent_max_flight_path_angle_metadata)

    _set_descent_max_flight_path_angle_metadata = { "offset" : _set_descent_max_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @descent_max_flight_path_angle.setter
    def descent_max_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum flight path angle of the missile's flight path while descending."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_descent_max_flight_path_angle_metadata, value)

    _get_descent_min_flight_path_angle_metadata = { "offset" : _get_descent_min_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def descent_min_flight_path_angle(self) -> typing.Any:
        """Get or set the minimum flight path angle of the missile's flight path while descending."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_descent_min_flight_path_angle_metadata)

    _set_descent_min_flight_path_angle_metadata = { "offset" : _set_descent_min_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @descent_min_flight_path_angle.setter
    def descent_min_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the minimum flight path angle of the missile's flight path while descending."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_descent_min_flight_path_angle_metadata, value)

    _get_descent_fail_on_insufficient_performance_metadata = { "offset" : _get_descent_fail_on_insufficient_performance_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def descent_fail_on_insufficient_performance(self) -> bool:
        """Opt whether to fail while descending if there is insufficient specific excess power."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_descent_fail_on_insufficient_performance_metadata)

    _set_descent_fail_on_insufficient_performance_metadata = { "offset" : _set_descent_fail_on_insufficient_performance_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @descent_fail_on_insufficient_performance.setter
    def descent_fail_on_insufficient_performance(self, value:bool) -> None:
        """Opt whether to fail while descending if there is insufficient specific excess power."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_descent_fail_on_insufficient_performance_metadata, value)

    _get_use_total_temp_limit_metadata = { "offset" : _get_use_total_temp_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_total_temp_limit(self) -> bool:
        """Opt whether to limit the speed of the missile so the specified temperature is not exceeded."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_use_total_temp_limit_metadata)

    _set_use_total_temp_limit_metadata = { "offset" : _set_use_total_temp_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_total_temp_limit.setter
    def use_total_temp_limit(self, value:bool) -> None:
        """Opt whether to limit the speed of the missile so the specified temperature is not exceeded."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_use_total_temp_limit_metadata, value)

    _get_total_temp_limit_metadata = { "offset" : _get_total_temp_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def total_temp_limit(self) -> float:
        """Get or set the maximum total temperature limit of the missile."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_total_temp_limit_metadata)

    _set_total_temp_limit_metadata = { "offset" : _set_total_temp_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @total_temp_limit.setter
    def total_temp_limit(self, value:float) -> None:
        """Get or set the maximum total temperature limit of the missile."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_total_temp_limit_metadata, value)

    _get_use_mach_limit_metadata = { "offset" : _get_use_mach_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_mach_limit(self) -> bool:
        """Opt whether to limit the speed of the missile so the specified mach number is not exceeded."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_use_mach_limit_metadata)

    _set_use_mach_limit_metadata = { "offset" : _set_use_mach_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_mach_limit.setter
    def use_mach_limit(self, value:bool) -> None:
        """Opt whether to limit the speed of the missile so the specified mach number is not exceeded."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_use_mach_limit_metadata, value)

    _get_mach_limit_metadata = { "offset" : _get_mach_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def mach_limit(self) -> float:
        """Get or set the maximum allowable mach number."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_mach_limit_metadata)

    _set_mach_limit_metadata = { "offset" : _set_mach_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @mach_limit.setter
    def mach_limit(self, value:float) -> None:
        """Get or set the maximum allowable mach number."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_mach_limit_metadata, value)

    _get_use_eas_limit_metadata = { "offset" : _get_use_eas_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_eas_limit(self) -> bool:
        """Opt whether to limit the speed of the missile so the specified Equivalent Airspeed is not exceeded."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_use_eas_limit_metadata)

    _set_use_eas_limit_metadata = { "offset" : _set_use_eas_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_eas_limit.setter
    def use_eas_limit(self, value:bool) -> None:
        """Opt whether to limit the speed of the missile so the specified Equivalent Airspeed is not exceeded."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_use_eas_limit_metadata, value)

    _get_eas_limit_metadata = { "offset" : _get_eas_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def eas_limit(self) -> float:
        """Get or set the maximum allowable Equivalent Airspeed."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_eas_limit_metadata)

    _set_eas_limit_metadata = { "offset" : _set_eas_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @eas_limit.setter
    def eas_limit(self, value:float) -> None:
        """Get or set the maximum allowable Equivalent Airspeed."""
        return self._intf.set_property(MissileModel._metadata, MissileModel._set_eas_limit_metadata, value)

    _get_default_configuration_metadata = { "offset" : _get_default_configuration_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def default_configuration(self) -> "Configuration":
        """Get the aircraft's default configuration as saved in the catalog."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_default_configuration_metadata)

    _get_aerodynamics_metadata = { "offset" : _get_aerodynamics_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aerodynamics(self) -> "MissileAerodynamic":
        """Get the aerodynamics interface."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_aerodynamics_metadata)

    _get_propulsion_metadata = { "offset" : _get_propulsion_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def propulsion(self) -> "MissilePropulsion":
        """Get the propulsion interface."""
        return self._intf.get_property(MissileModel._metadata, MissileModel._get_propulsion_metadata)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(MissileModel._metadata, MissileModel._get_as_catalog_item_metadata, OutArg())

    _property_names[max_load_factor] = "max_load_factor"
    _property_names[maneuver_mode] = "maneuver_mode"
    _property_names[maneuver_mode_helper] = "maneuver_mode_helper"
    _property_names[attitude_transitions] = "attitude_transitions"
    _property_names[ignore_flight_path_angle_for_climb_descent_transitions] = "ignore_flight_path_angle_for_climb_descent_transitions"
    _property_names[climb_airspeed] = "climb_airspeed"
    _property_names[climb_airspeed_type] = "climb_airspeed_type"
    _property_names[climb_max_flight_path_angle] = "climb_max_flight_path_angle"
    _property_names[climb_min_flight_path_angle] = "climb_min_flight_path_angle"
    _property_names[climb_fail_on_insufficient_performance] = "climb_fail_on_insufficient_performance"
    _property_names[cruise_max_airspeed] = "cruise_max_airspeed"
    _property_names[cruise_max_airspeed_type] = "cruise_max_airspeed_type"
    _property_names[cruise_default_altitude] = "cruise_default_altitude"
    _property_names[descent_airspeed] = "descent_airspeed"
    _property_names[descent_airspeed_type] = "descent_airspeed_type"
    _property_names[descent_max_flight_path_angle] = "descent_max_flight_path_angle"
    _property_names[descent_min_flight_path_angle] = "descent_min_flight_path_angle"
    _property_names[descent_fail_on_insufficient_performance] = "descent_fail_on_insufficient_performance"
    _property_names[use_total_temp_limit] = "use_total_temp_limit"
    _property_names[total_temp_limit] = "total_temp_limit"
    _property_names[use_mach_limit] = "use_mach_limit"
    _property_names[mach_limit] = "mach_limit"
    _property_names[use_eas_limit] = "use_eas_limit"
    _property_names[eas_limit] = "eas_limit"
    _property_names[default_configuration] = "default_configuration"
    _property_names[aerodynamics] = "aerodynamics"
    _property_names[propulsion] = "propulsion"

    def __init__(self, source_object=None):
        """Construct an object of type MissileModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MissileModel)
        IAviatorVehicle.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAviatorVehicle._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MissileModel, [MissileModel, IAviatorVehicle, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5537525572496185256, 6677635249753434025), MissileModel)
agcls.AgTypeNameMap["MissileModel"] = MissileModel

class RotorcraftModel(IAviatorVehicle, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the rotorcraft options in the Aviator catalog."""

    _num_methods = 44
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_altitude_method_offset = 1
    _set_max_altitude_method_offset = 2
    _get_default_cruise_altitude_method_offset = 3
    _set_default_cruise_altitude_method_offset = 4
    _get_descent_rate_factor_method_offset = 5
    _set_descent_rate_factor_method_offset = 6
    _get_max_climb_angle_method_offset = 7
    _set_max_climb_angle_method_offset = 8
    _get_climb_at_cruise_airspeed_method_offset = 9
    _set_climb_at_cruise_airspeed_method_offset = 10
    _get_max_descent_angle_method_offset = 11
    _set_max_descent_angle_method_offset = 12
    _get_min_descent_rate_method_offset = 13
    _set_min_descent_rate_method_offset = 14
    _get_max_load_factor_method_offset = 15
    _set_max_load_factor_method_offset = 16
    _get_roll_rate_method_offset = 17
    _set_roll_rate_method_offset = 18
    _get_pitch_rate_method_offset = 19
    _set_pitch_rate_method_offset = 20
    _get_yaw_rate_method_offset = 21
    _set_yaw_rate_method_offset = 22
    _get_yaw_rate_dot_method_offset = 23
    _set_yaw_rate_dot_method_offset = 24
    _get_max_transition_pitch_angle_method_offset = 25
    _set_max_transition_pitch_angle_method_offset = 26
    _get_tf_max_flight_path_angle_method_offset = 27
    _set_tf_max_flight_path_angle_method_offset = 28
    _get_tf_terrain_window_method_offset = 29
    _set_tf_terrain_window_method_offset = 30
    _get_compute_delta_altitude_method_offset = 31
    _set_compute_delta_altitude_method_offset = 32
    _get_max_safe_airspeed_method_offset = 33
    _get_max_safe_airspeed_type_method_offset = 34
    _set_max_safe_airspeed_method_offset = 35
    _get_max_safe_translation_speed_method_offset = 36
    _get_max_safe_translation_speed_type_method_offset = 37
    _set_max_safe_translation_speed_method_offset = 38
    _get_ignore_flight_path_angle_for_climb_descent_transitions_method_offset = 39
    _set_ignore_flight_path_angle_for_climb_descent_transitions_method_offset = 40
    _get_default_configuration_method_offset = 41
    _get_aerodynamics_method_offset = 42
    _get_propulsion_method_offset = 43
    _get_as_catalog_item_method_offset = 44
    _metadata = {
        "iid_data" : (5082120676369899385, 4325334539633823629),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RotorcraftModel)

    _get_max_altitude_metadata = { "offset" : _get_max_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_altitude(self) -> float:
        """Get or set the maximum altitude at which the rotorcraft is capable of operating."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_altitude_metadata)

    _set_max_altitude_metadata = { "offset" : _set_max_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_altitude.setter
    def max_altitude(self, value:float) -> None:
        """Get or set the maximum altitude at which the rotorcraft is capable of operating."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_max_altitude_metadata, value)

    _get_default_cruise_altitude_metadata = { "offset" : _get_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def default_cruise_altitude(self) -> float:
        """Get or set the rotorcraft's default cruising altitude."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_default_cruise_altitude_metadata)

    _set_default_cruise_altitude_metadata = { "offset" : _set_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @default_cruise_altitude.setter
    def default_cruise_altitude(self, value:float) -> None:
        """Get or set the rotorcraft's default cruising altitude."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_default_cruise_altitude_metadata, value)

    _get_descent_rate_factor_metadata = { "offset" : _get_descent_rate_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def descent_rate_factor(self) -> float:
        """Get or set the descent rate of the rotorcraft as a factor multiplied by the altitude change rate calculated at zero throttle."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_descent_rate_factor_metadata)

    _set_descent_rate_factor_metadata = { "offset" : _set_descent_rate_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @descent_rate_factor.setter
    def descent_rate_factor(self, value:float) -> None:
        """Get or set the descent rate of the rotorcraft as a factor multiplied by the altitude change rate calculated at zero throttle."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_descent_rate_factor_metadata, value)

    _get_max_climb_angle_metadata = { "offset" : _get_max_climb_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_climb_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the rotorcraft's flight path while climbing."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_climb_angle_metadata)

    _set_max_climb_angle_metadata = { "offset" : _set_max_climb_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_climb_angle.setter
    def max_climb_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the rotorcraft's flight path while climbing."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_max_climb_angle_metadata, value)

    _get_climb_at_cruise_airspeed_metadata = { "offset" : _get_climb_at_cruise_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def climb_at_cruise_airspeed(self) -> bool:
        """Select to define the climbing airspeed of the rotorcraft using the cruise airspeed of the current procedure."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_climb_at_cruise_airspeed_metadata)

    _set_climb_at_cruise_airspeed_metadata = { "offset" : _set_climb_at_cruise_airspeed_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @climb_at_cruise_airspeed.setter
    def climb_at_cruise_airspeed(self, value:bool) -> None:
        """Select to define the climbing airspeed of the rotorcraft using the cruise airspeed of the current procedure."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_climb_at_cruise_airspeed_metadata, value)

    _get_max_descent_angle_metadata = { "offset" : _get_max_descent_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_descent_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the rotorcraft's flight path while descending."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_descent_angle_metadata)

    _set_max_descent_angle_metadata = { "offset" : _set_max_descent_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_descent_angle.setter
    def max_descent_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the rotorcraft's flight path while descending."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_max_descent_angle_metadata, value)

    _get_min_descent_rate_metadata = { "offset" : _get_min_descent_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_descent_rate(self) -> float:
        """Get or set the minimum rate at which the aircraft will descend once established in a steady descent."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_min_descent_rate_metadata)

    _set_min_descent_rate_metadata = { "offset" : _set_min_descent_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_descent_rate.setter
    def min_descent_rate(self, value:float) -> None:
        """Get or set the minimum rate at which the aircraft will descend once established in a steady descent."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_min_descent_rate_metadata, value)

    _get_max_load_factor_metadata = { "offset" : _get_max_load_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_load_factor(self) -> float:
        """Get or set the maximum load factor that the aircraft can bear while maneuvering in formation."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_load_factor_metadata)

    _set_max_load_factor_metadata = { "offset" : _set_max_load_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_load_factor.setter
    def max_load_factor(self, value:float) -> None:
        """Get or set the maximum load factor that the aircraft can bear while maneuvering in formation."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_max_load_factor_metadata, value)

    _get_roll_rate_metadata = { "offset" : _get_roll_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def roll_rate(self) -> typing.Any:
        """Get or set the standard roll rate of the rotorcraft in a turn."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_roll_rate_metadata)

    _set_roll_rate_metadata = { "offset" : _set_roll_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @roll_rate.setter
    def roll_rate(self, value:typing.Any) -> None:
        """Get or set the standard roll rate of the rotorcraft in a turn."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_roll_rate_metadata, value)

    _get_pitch_rate_metadata = { "offset" : _get_pitch_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def pitch_rate(self) -> typing.Any:
        """Get or set the pitch rate when transitioning between attitude modes or procedures."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_pitch_rate_metadata)

    _set_pitch_rate_metadata = { "offset" : _set_pitch_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @pitch_rate.setter
    def pitch_rate(self, value:typing.Any) -> None:
        """Get or set the pitch rate when transitioning between attitude modes or procedures."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_pitch_rate_metadata, value)

    _get_yaw_rate_metadata = { "offset" : _get_yaw_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def yaw_rate(self) -> typing.Any:
        """Get or set the yaw rate when transitioning between attitude modes."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_yaw_rate_metadata)

    _set_yaw_rate_metadata = { "offset" : _set_yaw_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @yaw_rate.setter
    def yaw_rate(self, value:typing.Any) -> None:
        """Get or set the yaw rate when transitioning between attitude modes."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_yaw_rate_metadata, value)

    _get_yaw_rate_dot_metadata = { "offset" : _get_yaw_rate_dot_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def yaw_rate_dot(self) -> typing.Any:
        """Get or set the rate of change of the yaw rate."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_yaw_rate_dot_metadata)

    _set_yaw_rate_dot_metadata = { "offset" : _set_yaw_rate_dot_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @yaw_rate_dot.setter
    def yaw_rate_dot(self, value:typing.Any) -> None:
        """Get or set the rate of change of the yaw rate."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_yaw_rate_dot_metadata, value)

    _get_max_transition_pitch_angle_metadata = { "offset" : _get_max_transition_pitch_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_transition_pitch_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the flight path when transitioning between forward flight and hovering."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_transition_pitch_angle_metadata)

    _set_max_transition_pitch_angle_metadata = { "offset" : _set_max_transition_pitch_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_transition_pitch_angle.setter
    def max_transition_pitch_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the flight path when transitioning between forward flight and hovering."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_max_transition_pitch_angle_metadata, value)

    _get_tf_max_flight_path_angle_metadata = { "offset" : _get_tf_max_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def tf_max_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the flight path when the rotorcraft is engaged in terrain following flight."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_tf_max_flight_path_angle_metadata)

    _set_tf_max_flight_path_angle_metadata = { "offset" : _set_tf_max_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @tf_max_flight_path_angle.setter
    def tf_max_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the flight path when the rotorcraft is engaged in terrain following flight."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_tf_max_flight_path_angle_metadata, value)

    _get_tf_terrain_window_metadata = { "offset" : _get_tf_terrain_window_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tf_terrain_window(self) -> float:
        """Get or set the time interval over which terrain points are sampled when the rotorcraft is engaged in terrain following flight."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_tf_terrain_window_metadata)

    _set_tf_terrain_window_metadata = { "offset" : _set_tf_terrain_window_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tf_terrain_window.setter
    def tf_terrain_window(self, value:float) -> None:
        """Get or set the time interval over which terrain points are sampled when the rotorcraft is engaged in terrain following flight."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_tf_terrain_window_metadata, value)

    _get_compute_delta_altitude_metadata = { "offset" : _get_compute_delta_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def compute_delta_altitude(self) -> float:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_compute_delta_altitude_metadata)

    _set_compute_delta_altitude_metadata = { "offset" : _set_compute_delta_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @compute_delta_altitude.setter
    def compute_delta_altitude(self, value:float) -> None:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_compute_delta_altitude_metadata, value)

    _get_max_safe_airspeed_metadata = { "offset" : _get_max_safe_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_safe_airspeed(self) -> float:
        """Get the maximum cruising airspeed of the rotorcraft."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_safe_airspeed_metadata)

    _get_max_safe_airspeed_type_metadata = { "offset" : _get_max_safe_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def max_safe_airspeed_type(self) -> "AirspeedType":
        """Get the maximum safe airspeed type."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_safe_airspeed_type_metadata)

    _set_max_safe_airspeed_metadata = { "offset" : _set_max_safe_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_max_safe_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the maximum safe airspeed and airspeed type."""
        return self._intf.invoke(RotorcraftModel._metadata, RotorcraftModel._set_max_safe_airspeed_metadata, airspeed_type, airspeed)

    _get_max_safe_translation_speed_metadata = { "offset" : _get_max_safe_translation_speed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_safe_translation_speed(self) -> float:
        """Get the maximum translation speed of the rotorcraft."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_safe_translation_speed_metadata)

    _get_max_safe_translation_speed_type_metadata = { "offset" : _get_max_safe_translation_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def max_safe_translation_speed_type(self) -> "AirspeedType":
        """Get the maximum safe translation speed type."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_max_safe_translation_speed_type_metadata)

    _set_max_safe_translation_speed_metadata = { "offset" : _set_max_safe_translation_speed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_max_safe_translation_speed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the maximum safe translation airspeed and airspeed type."""
        return self._intf.invoke(RotorcraftModel._metadata, RotorcraftModel._set_max_safe_translation_speed_metadata, airspeed_type, airspeed)

    _get_ignore_flight_path_angle_for_climb_descent_transitions_metadata = { "offset" : _get_ignore_flight_path_angle_for_climb_descent_transitions_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def ignore_flight_path_angle_for_climb_descent_transitions(self) -> bool:
        """Opt to ignore load factor limits when pushing over or pulling up."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_ignore_flight_path_angle_for_climb_descent_transitions_metadata)

    _set_ignore_flight_path_angle_for_climb_descent_transitions_metadata = { "offset" : _set_ignore_flight_path_angle_for_climb_descent_transitions_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @ignore_flight_path_angle_for_climb_descent_transitions.setter
    def ignore_flight_path_angle_for_climb_descent_transitions(self, value:bool) -> None:
        """Opt to ignore load factor limits when pushing over or pulling up."""
        return self._intf.set_property(RotorcraftModel._metadata, RotorcraftModel._set_ignore_flight_path_angle_for_climb_descent_transitions_metadata, value)

    _get_default_configuration_metadata = { "offset" : _get_default_configuration_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def default_configuration(self) -> "Configuration":
        """Get the aircraft's default configuration as saved in the catalog."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_default_configuration_metadata)

    _get_aerodynamics_metadata = { "offset" : _get_aerodynamics_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aerodynamics(self) -> "RotorcraftAerodynamic":
        """Get the aerodynamics interface."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_aerodynamics_metadata)

    _get_propulsion_metadata = { "offset" : _get_propulsion_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def propulsion(self) -> "RotorcraftPropulsion":
        """Get the propulsion interface."""
        return self._intf.get_property(RotorcraftModel._metadata, RotorcraftModel._get_propulsion_metadata)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(RotorcraftModel._metadata, RotorcraftModel._get_as_catalog_item_metadata, OutArg())

    _property_names[max_altitude] = "max_altitude"
    _property_names[default_cruise_altitude] = "default_cruise_altitude"
    _property_names[descent_rate_factor] = "descent_rate_factor"
    _property_names[max_climb_angle] = "max_climb_angle"
    _property_names[climb_at_cruise_airspeed] = "climb_at_cruise_airspeed"
    _property_names[max_descent_angle] = "max_descent_angle"
    _property_names[min_descent_rate] = "min_descent_rate"
    _property_names[max_load_factor] = "max_load_factor"
    _property_names[roll_rate] = "roll_rate"
    _property_names[pitch_rate] = "pitch_rate"
    _property_names[yaw_rate] = "yaw_rate"
    _property_names[yaw_rate_dot] = "yaw_rate_dot"
    _property_names[max_transition_pitch_angle] = "max_transition_pitch_angle"
    _property_names[tf_max_flight_path_angle] = "tf_max_flight_path_angle"
    _property_names[tf_terrain_window] = "tf_terrain_window"
    _property_names[compute_delta_altitude] = "compute_delta_altitude"
    _property_names[max_safe_airspeed] = "max_safe_airspeed"
    _property_names[max_safe_airspeed_type] = "max_safe_airspeed_type"
    _property_names[max_safe_translation_speed] = "max_safe_translation_speed"
    _property_names[max_safe_translation_speed_type] = "max_safe_translation_speed_type"
    _property_names[ignore_flight_path_angle_for_climb_descent_transitions] = "ignore_flight_path_angle_for_climb_descent_transitions"
    _property_names[default_configuration] = "default_configuration"
    _property_names[aerodynamics] = "aerodynamics"
    _property_names[propulsion] = "propulsion"

    def __init__(self, source_object=None):
        """Construct an object of type RotorcraftModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RotorcraftModel)
        IAviatorVehicle.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAviatorVehicle._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RotorcraftModel, [RotorcraftModel, IAviatorVehicle, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5132696765661705826, 14502770979691808656), RotorcraftModel)
agcls.AgTypeNameMap["RotorcraftModel"] = RotorcraftModel

class RotorcraftAerodynamic(SupportsDeleteCallback):
    """Interface used to access the aerodynamics options for a rotorcraft."""

    _num_methods = 22
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_rotor_count_method_offset = 1
    _set_rotor_count_method_offset = 2
    _get_rotor_diameter_method_offset = 3
    _set_rotor_diameter_method_offset = 4
    _get_blades_per_rotor_method_offset = 5
    _set_blades_per_rotor_method_offset = 6
    _get_blade_chord_method_offset = 7
    _set_blade_chord_method_offset = 8
    _get_rotor_tip_mach_method_offset = 9
    _set_rotor_tip_mach_method_offset = 10
    _get_fuselage_flat_plate_area_method_offset = 11
    _set_fuselage_flat_plate_area_method_offset = 12
    _get_tail_rotor_offset_method_offset = 13
    _set_tail_rotor_offset_method_offset = 14
    _get_tail_rotor_diameter_method_offset = 15
    _set_tail_rotor_diameter_method_offset = 16
    _get_blade_profile_drag_cd0_method_offset = 17
    _set_blade_profile_drag_cd0_method_offset = 18
    _get_blade_profile_drag_k_method_offset = 19
    _set_blade_profile_drag_k_method_offset = 20
    _get_induced_power_correction_factor_method_offset = 21
    _set_induced_power_correction_factor_method_offset = 22
    _metadata = {
        "iid_data" : (5318240503192117773, 12951249891304096688),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RotorcraftAerodynamic)

    _get_rotor_count_metadata = { "offset" : _get_rotor_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def rotor_count(self) -> int:
        """Get or set the number of rotors."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_rotor_count_metadata)

    _set_rotor_count_metadata = { "offset" : _set_rotor_count_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @rotor_count.setter
    def rotor_count(self, value:int) -> None:
        """Get or set the number of rotors."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_rotor_count_metadata, value)

    _get_rotor_diameter_metadata = { "offset" : _get_rotor_diameter_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def rotor_diameter(self) -> float:
        """Get or set the diameter of the rotor."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_rotor_diameter_metadata)

    _set_rotor_diameter_metadata = { "offset" : _set_rotor_diameter_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @rotor_diameter.setter
    def rotor_diameter(self, value:float) -> None:
        """Get or set the diameter of the rotor."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_rotor_diameter_metadata, value)

    _get_blades_per_rotor_metadata = { "offset" : _get_blades_per_rotor_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def blades_per_rotor(self) -> int:
        """Get or set the number of blades on each rotor."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_blades_per_rotor_metadata)

    _set_blades_per_rotor_metadata = { "offset" : _set_blades_per_rotor_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @blades_per_rotor.setter
    def blades_per_rotor(self, value:int) -> None:
        """Get or set the number of blades on each rotor."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_blades_per_rotor_metadata, value)

    _get_blade_chord_metadata = { "offset" : _get_blade_chord_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def blade_chord(self) -> float:
        """Get or set the chord length of the blade."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_blade_chord_metadata)

    _set_blade_chord_metadata = { "offset" : _set_blade_chord_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @blade_chord.setter
    def blade_chord(self, value:float) -> None:
        """Get or set the chord length of the blade."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_blade_chord_metadata, value)

    _get_rotor_tip_mach_metadata = { "offset" : _get_rotor_tip_mach_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def rotor_tip_mach(self) -> float:
        """Get or set the Mach number of the advancing blade tip."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_rotor_tip_mach_metadata)

    _set_rotor_tip_mach_metadata = { "offset" : _set_rotor_tip_mach_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @rotor_tip_mach.setter
    def rotor_tip_mach(self, value:float) -> None:
        """Get or set the Mach number of the advancing blade tip."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_rotor_tip_mach_metadata, value)

    _get_fuselage_flat_plate_area_metadata = { "offset" : _get_fuselage_flat_plate_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuselage_flat_plate_area(self) -> float:
        """Get or set the flat plate area for the fuselage."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_fuselage_flat_plate_area_metadata)

    _set_fuselage_flat_plate_area_metadata = { "offset" : _set_fuselage_flat_plate_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuselage_flat_plate_area.setter
    def fuselage_flat_plate_area(self, value:float) -> None:
        """Get or set the flat plate area for the fuselage."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_fuselage_flat_plate_area_metadata, value)

    _get_tail_rotor_offset_metadata = { "offset" : _get_tail_rotor_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tail_rotor_offset(self) -> float:
        """Get or set the offset of the tail rotor."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_tail_rotor_offset_metadata)

    _set_tail_rotor_offset_metadata = { "offset" : _set_tail_rotor_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tail_rotor_offset.setter
    def tail_rotor_offset(self, value:float) -> None:
        """Get or set the offset of the tail rotor."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_tail_rotor_offset_metadata, value)

    _get_tail_rotor_diameter_metadata = { "offset" : _get_tail_rotor_diameter_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tail_rotor_diameter(self) -> float:
        """Get or set the diameter of the tail rotor."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_tail_rotor_diameter_metadata)

    _set_tail_rotor_diameter_metadata = { "offset" : _set_tail_rotor_diameter_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tail_rotor_diameter.setter
    def tail_rotor_diameter(self, value:float) -> None:
        """Get or set the diameter of the tail rotor."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_tail_rotor_diameter_metadata, value)

    _get_blade_profile_drag_cd0_metadata = { "offset" : _get_blade_profile_drag_cd0_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def blade_profile_drag_cd0(self) -> float:
        """Get or set the drag coefficient when the rotor disc does not generate any lift."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_blade_profile_drag_cd0_metadata)

    _set_blade_profile_drag_cd0_metadata = { "offset" : _set_blade_profile_drag_cd0_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @blade_profile_drag_cd0.setter
    def blade_profile_drag_cd0(self, value:float) -> None:
        """Get or set the drag coefficient when the rotor disc does not generate any lift."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_blade_profile_drag_cd0_metadata, value)

    _get_blade_profile_drag_k_metadata = { "offset" : _get_blade_profile_drag_k_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def blade_profile_drag_k(self) -> float:
        """Get or set the induced drag coefficient, which accounts for how lift generation impacts drag."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_blade_profile_drag_k_metadata)

    _set_blade_profile_drag_k_metadata = { "offset" : _set_blade_profile_drag_k_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @blade_profile_drag_k.setter
    def blade_profile_drag_k(self, value:float) -> None:
        """Get or set the induced drag coefficient, which accounts for how lift generation impacts drag."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_blade_profile_drag_k_metadata, value)

    _get_induced_power_correction_factor_metadata = { "offset" : _get_induced_power_correction_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def induced_power_correction_factor(self) -> float:
        """Get or set the slop factor that accounts for losses."""
        return self._intf.get_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._get_induced_power_correction_factor_metadata)

    _set_induced_power_correction_factor_metadata = { "offset" : _set_induced_power_correction_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @induced_power_correction_factor.setter
    def induced_power_correction_factor(self, value:float) -> None:
        """Get or set the slop factor that accounts for losses."""
        return self._intf.set_property(RotorcraftAerodynamic._metadata, RotorcraftAerodynamic._set_induced_power_correction_factor_metadata, value)

    _property_names[rotor_count] = "rotor_count"
    _property_names[rotor_diameter] = "rotor_diameter"
    _property_names[blades_per_rotor] = "blades_per_rotor"
    _property_names[blade_chord] = "blade_chord"
    _property_names[rotor_tip_mach] = "rotor_tip_mach"
    _property_names[fuselage_flat_plate_area] = "fuselage_flat_plate_area"
    _property_names[tail_rotor_offset] = "tail_rotor_offset"
    _property_names[tail_rotor_diameter] = "tail_rotor_diameter"
    _property_names[blade_profile_drag_cd0] = "blade_profile_drag_cd0"
    _property_names[blade_profile_drag_k] = "blade_profile_drag_k"
    _property_names[induced_power_correction_factor] = "induced_power_correction_factor"

    def __init__(self, source_object=None):
        """Construct an object of type RotorcraftAerodynamic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RotorcraftAerodynamic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RotorcraftAerodynamic, [RotorcraftAerodynamic, ])

agcls.AgClassCatalog.add_catalog_entry((5254408426110746023, 3824899600395506365), RotorcraftAerodynamic)
agcls.AgTypeNameMap["RotorcraftAerodynamic"] = RotorcraftAerodynamic

class RotorcraftPropulsion(SupportsDeleteCallback):
    """Interface used to access the Propulsion options for a rotorcraft."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_powerplant_type_method_offset = 1
    _set_powerplant_type_method_offset = 2
    _get_max_sl_power_method_offset = 3
    _set_max_sl_power_method_offset = 4
    _get_max_sl_fuel_flow_method_offset = 5
    _set_max_sl_fuel_flow_method_offset = 6
    _metadata = {
        "iid_data" : (5259623472655829368, 12428325399528728501),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RotorcraftPropulsion)

    _get_powerplant_type_metadata = { "offset" : _get_powerplant_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RotorcraftPowerplantType),) }
    @property
    def powerplant_type(self) -> "RotorcraftPowerplantType":
        """Get or set the rotorcraft's powerplant type."""
        return self._intf.get_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._get_powerplant_type_metadata)

    _set_powerplant_type_metadata = { "offset" : _set_powerplant_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RotorcraftPowerplantType),) }
    @powerplant_type.setter
    def powerplant_type(self, value:"RotorcraftPowerplantType") -> None:
        """Get or set the rotorcraft's powerplant type."""
        return self._intf.set_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._set_powerplant_type_metadata, value)

    _get_max_sl_power_metadata = { "offset" : _get_max_sl_power_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_sl_power(self) -> float:
        """Get or set the maximum power at sea level."""
        return self._intf.get_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._get_max_sl_power_metadata)

    _set_max_sl_power_metadata = { "offset" : _set_max_sl_power_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_sl_power.setter
    def max_sl_power(self, value:float) -> None:
        """Get or set the maximum power at sea level."""
        return self._intf.set_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._set_max_sl_power_metadata, value)

    _get_max_sl_fuel_flow_metadata = { "offset" : _get_max_sl_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_sl_fuel_flow(self) -> float:
        """Get or set the maximum fuel flow at sea level."""
        return self._intf.get_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._get_max_sl_fuel_flow_metadata)

    _set_max_sl_fuel_flow_metadata = { "offset" : _set_max_sl_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_sl_fuel_flow.setter
    def max_sl_fuel_flow(self, value:float) -> None:
        """Get or set the maximum fuel flow at sea level."""
        return self._intf.set_property(RotorcraftPropulsion._metadata, RotorcraftPropulsion._set_max_sl_fuel_flow_metadata, value)

    _property_names[powerplant_type] = "powerplant_type"
    _property_names[max_sl_power] = "max_sl_power"
    _property_names[max_sl_fuel_flow] = "max_sl_fuel_flow"

    def __init__(self, source_object=None):
        """Construct an object of type RotorcraftPropulsion."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RotorcraftPropulsion)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RotorcraftPropulsion, [RotorcraftPropulsion, ])

agcls.AgClassCatalog.add_catalog_entry((5760757719442343448, 132377700848243866), RotorcraftPropulsion)
agcls.AgTypeNameMap["RotorcraftPropulsion"] = RotorcraftPropulsion

class AircraftAcceleration(ICatalogItem, SupportsDeleteCallback):
    """
    Interface used to access the acceleration options for an aircraft in the Aviator catalog.

    Examples
    --------
    Create a new performance model for an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the names of the current acceleration models
    >>> modelNames = acceleration.child_names
    >>> # Check how many models there are
    >>> modelCount = len(modelNames)
    >>> # Get the child types (for example AGI Basic Acceleration Model, Advanced Acceleration Model)
    >>> modelTypes = acceleration.child_types
    >>> # Create a new performance model of type "Advanced Acceleration Model"
    >>> newPerformanceModel = acceleration.add_child_of_type("Advanced Acceleration Model", "Model Name")
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()

    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_acceleration_by_name_method_offset = 2
    _get_advanced_acceleration_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (5381104812027548008, 3758078246891655319),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAcceleration)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicAccelerationModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftAcceleration._metadata, AircraftAcceleration._get_built_in_model_metadata, OutArg())

    _get_basic_acceleration_by_name_metadata = { "offset" : _get_basic_acceleration_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_acceleration_by_name(self, name:str) -> "AircraftBasicAccelerationModel":
        """Get the basic acceleration model with the given name."""
        return self._intf.invoke(AircraftAcceleration._metadata, AircraftAcceleration._get_basic_acceleration_by_name_metadata, name, OutArg())

    _get_advanced_acceleration_by_name_metadata = { "offset" : _get_advanced_acceleration_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_acceleration_by_name(self, name:str) -> "AircraftAdvancedAccelerationModel":
        """Get the advanced acceleration model with the given name."""
        return self._intf.invoke(AircraftAcceleration._metadata, AircraftAcceleration._get_advanced_acceleration_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAcceleration._metadata, AircraftAcceleration._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftAcceleration."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAcceleration)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAcceleration, [AircraftAcceleration, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5513893073497569268, 18062891268736021946), AircraftAcceleration)
agcls.AgTypeNameMap["AircraftAcceleration"] = AircraftAcceleration

class AircraftBasicAccelerationModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """
    Interface used to access the basic acceleration model options for an acceleration model of an aircraft in the Aviator catalog.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_level_turns_method_offset = 1
    _get_attitude_transitions_method_offset = 2
    _get_climb_and_descent_transitions_method_offset = 3
    _get_aerodynamics_method_offset = 4
    _get_propulsion_method_offset = 5
    _get_as_catalog_item_method_offset = 6
    _metadata = {
        "iid_data" : (4918589609424149010, 10824534501852204216),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicAccelerationModel)

    _get_level_turns_metadata = { "offset" : _get_level_turns_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def level_turns(self) -> "LevelTurns":
        """Get the level turns interface."""
        return self._intf.get_property(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_level_turns_metadata)

    _get_attitude_transitions_metadata = { "offset" : _get_attitude_transitions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def attitude_transitions(self) -> "AttitudeTransitions":
        """Get the attitude transitions interface."""
        return self._intf.get_property(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_attitude_transitions_metadata)

    _get_climb_and_descent_transitions_metadata = { "offset" : _get_climb_and_descent_transitions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def climb_and_descent_transitions(self) -> "ClimbAndDescentTransitions":
        """Get the climb and descent transitions interface."""
        return self._intf.get_property(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_climb_and_descent_transitions_metadata)

    _get_aerodynamics_metadata = { "offset" : _get_aerodynamics_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def aerodynamics(self) -> "AircraftAerodynamic":
        """Get the aerodynamics interface."""
        return self._intf.get_property(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_aerodynamics_metadata)

    _get_propulsion_metadata = { "offset" : _get_propulsion_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def propulsion(self) -> "AircraftPropulsion":
        """Get the propulsion interface."""
        return self._intf.get_property(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_propulsion_metadata)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicAccelerationModel._metadata, AircraftBasicAccelerationModel._get_as_catalog_item_metadata, OutArg())

    _property_names[level_turns] = "level_turns"
    _property_names[attitude_transitions] = "attitude_transitions"
    _property_names[climb_and_descent_transitions] = "climb_and_descent_transitions"
    _property_names[aerodynamics] = "aerodynamics"
    _property_names[propulsion] = "propulsion"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicAccelerationModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicAccelerationModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicAccelerationModel, [AircraftBasicAccelerationModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5233981690568010157, 10284429996961284255), AircraftBasicAccelerationModel)
agcls.AgTypeNameMap["AircraftBasicAccelerationModel"] = AircraftBasicAccelerationModel

class AircraftClimb(ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the climb options for an aircraft in the Aviator catalog."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_climb_by_name_method_offset = 2
    _get_advanced_climb_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (5263267494218211610, 12329219289477949313),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftClimb)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicClimbModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftClimb._metadata, AircraftClimb._get_built_in_model_metadata, OutArg())

    _get_basic_climb_by_name_metadata = { "offset" : _get_basic_climb_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_climb_by_name(self, name:str) -> "AircraftBasicClimbModel":
        """Get the basic climb model with the given name."""
        return self._intf.invoke(AircraftClimb._metadata, AircraftClimb._get_basic_climb_by_name_metadata, name, OutArg())

    _get_advanced_climb_by_name_metadata = { "offset" : _get_advanced_climb_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_climb_by_name(self, name:str) -> "AircraftAdvancedClimbModel":
        """Get the advanced climb model with the given name."""
        return self._intf.invoke(AircraftClimb._metadata, AircraftClimb._get_advanced_climb_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftClimb._metadata, AircraftClimb._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftClimb."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftClimb)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftClimb, [AircraftClimb, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5617790646698935986, 14996628473608278423), AircraftClimb)
agcls.AgTypeNameMap["AircraftClimb"] = AircraftClimb

class AircraftCruise(ICatalogItem, SupportsDeleteCallback):
    """
    Interface used to access the cruise options for an aircraft in the Aviator catalog.

    Examples
    --------
    Configure the basic cruise performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the cruise type
    >>> cruise = aviatorAircraft.cruise
    >>> # Get the build in performance model
    >>> basicCruiseModel = cruise.get_built_in_model()
    >>>
    >>> # Set the ceiling altitude
    >>> basicCruiseModel.ceiling_altitude = 50000
    >>> # Set the default cruise altitude
    >>> basicCruiseModel.default_cruise_altitude = 10000
    >>> # Set the airspeed type
    >>> basicCruiseModel.airspeed_type = AirspeedType.TAS
    >>> # Opt to not use the fuel flow calculated by the aero/prop model and instead specify the values
    >>> basicCruiseModel.use_aerodynamic_propulsion_fuel = False
    >>>
    >>> # Set the various airspeeds and fuel flows
    >>> basicCruiseModel.min_airspeed = 110
    >>> basicCruiseModel.min_airspeed_fuel_flow = 10000
    >>>
    >>> basicCruiseModel.max_endurance_airspeed = 135
    >>> basicCruiseModel.max_endurance_fuel_flow = 8000
    >>>
    >>> basicCruiseModel.max_airspeed = 570
    >>> basicCruiseModel.max_airspeed_fuel_flow = 30000
    >>>
    >>> basicCruiseModel.max_range_airspeed = 140
    >>> basicCruiseModel.max_range_fuel_flow = 9000
    >>>
    >>> basicCruiseModel.max_performance_airspeed = 150
    >>> basicCruiseModel.max_performance_airspeed_fuel_flow = 12000
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_cruise_by_name_method_offset = 2
    _get_advanced_cruise_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (5128623341727715783, 9002706155677001097),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftCruise)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicCruiseModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftCruise._metadata, AircraftCruise._get_built_in_model_metadata, OutArg())

    _get_basic_cruise_by_name_metadata = { "offset" : _get_basic_cruise_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_cruise_by_name(self, name:str) -> "AircraftBasicCruiseModel":
        """Get the basic cruise model with the given name."""
        return self._intf.invoke(AircraftCruise._metadata, AircraftCruise._get_basic_cruise_by_name_metadata, name, OutArg())

    _get_advanced_cruise_by_name_metadata = { "offset" : _get_advanced_cruise_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_cruise_by_name(self, name:str) -> "AircraftAdvancedCruiseModel":
        """Get the advanced cruise model with the given name."""
        return self._intf.invoke(AircraftCruise._metadata, AircraftCruise._get_advanced_cruise_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftCruise._metadata, AircraftCruise._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftCruise."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftCruise)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftCruise, [AircraftCruise, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4952713164558545716, 391472350963315864), AircraftCruise)
agcls.AgTypeNameMap["AircraftCruise"] = AircraftCruise

class AircraftDescent(ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the descent options for an aircraft in the Aviator catalog."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_descent_by_name_method_offset = 2
    _get_advanced_descent_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (5297052565690495579, 8482664394541642129),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftDescent)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicDescentModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftDescent._metadata, AircraftDescent._get_built_in_model_metadata, OutArg())

    _get_basic_descent_by_name_metadata = { "offset" : _get_basic_descent_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_descent_by_name(self, name:str) -> "AircraftBasicDescentModel":
        """Get the basic descent model with the given name."""
        return self._intf.invoke(AircraftDescent._metadata, AircraftDescent._get_basic_descent_by_name_metadata, name, OutArg())

    _get_advanced_descent_by_name_metadata = { "offset" : _get_advanced_descent_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_descent_by_name(self, name:str) -> "AircraftAdvancedDescentModel":
        """Get the advanced descent model with the given name."""
        return self._intf.invoke(AircraftDescent._metadata, AircraftDescent._get_advanced_descent_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftDescent._metadata, AircraftDescent._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftDescent."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftDescent)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftDescent, [AircraftDescent, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5218114732559411329, 18271609367029705879), AircraftDescent)
agcls.AgTypeNameMap["AircraftDescent"] = AircraftDescent

class AircraftLanding(ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the landing options for an aircraft in the Aviator catalog."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_landing_by_name_method_offset = 2
    _get_advanced_landing_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (4810761550589613477, 16758738712672605105),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftLanding)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicLandingModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftLanding._metadata, AircraftLanding._get_built_in_model_metadata, OutArg())

    _get_basic_landing_by_name_metadata = { "offset" : _get_basic_landing_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_landing_by_name(self, name:str) -> "AircraftBasicLandingModel":
        """Get the basic Landing model with the given name."""
        return self._intf.invoke(AircraftLanding._metadata, AircraftLanding._get_basic_landing_by_name_metadata, name, OutArg())

    _get_advanced_landing_by_name_metadata = { "offset" : _get_advanced_landing_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_landing_by_name(self, name:str) -> "AircraftAdvancedLandingModel":
        """Get the advanced Landing model with the given name."""
        return self._intf.invoke(AircraftLanding._metadata, AircraftLanding._get_advanced_landing_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftLanding._metadata, AircraftLanding._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftLanding."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftLanding)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftLanding, [AircraftLanding, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4740693708489443339, 736458658073889207), AircraftLanding)
agcls.AgTypeNameMap["AircraftLanding"] = AircraftLanding

class AircraftTakeoff(ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the takeoff options for an aircraft in the Aviator catalog."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_built_in_model_method_offset = 1
    _get_basic_takeoff_by_name_method_offset = 2
    _get_advanced_takeoff_by_name_method_offset = 3
    _get_as_catalog_item_method_offset = 4
    _metadata = {
        "iid_data" : (5694947358439793555, 17623821908663075205),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftTakeoff)

    _get_built_in_model_metadata = { "offset" : _get_built_in_model_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_built_in_model(self) -> "AircraftBasicTakeoffModel":
        """Get the built-in model."""
        return self._intf.invoke(AircraftTakeoff._metadata, AircraftTakeoff._get_built_in_model_metadata, OutArg())

    _get_basic_takeoff_by_name_metadata = { "offset" : _get_basic_takeoff_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_basic_takeoff_by_name(self, name:str) -> "AircraftBasicTakeoffModel":
        """Get the basic Takeoff model with the given name."""
        return self._intf.invoke(AircraftTakeoff._metadata, AircraftTakeoff._get_basic_takeoff_by_name_metadata, name, OutArg())

    _get_advanced_takeoff_by_name_metadata = { "offset" : _get_advanced_takeoff_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_advanced_takeoff_by_name(self, name:str) -> "AircraftAdvancedTakeoffModel":
        """Get the advanced Takeoff model with the given name."""
        return self._intf.invoke(AircraftTakeoff._metadata, AircraftTakeoff._get_advanced_takeoff_by_name_metadata, name, OutArg())

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftTakeoff._metadata, AircraftTakeoff._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type AircraftTakeoff."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftTakeoff)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftTakeoff, [AircraftTakeoff, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5587969153310880849, 14792461406398889898), AircraftTakeoff)
agcls.AgTypeNameMap["AircraftTakeoff"] = AircraftTakeoff

class AircraftBasicClimbModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the basic climb model options for a climb model of an aircraft in the Aviator catalog."""

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_ceiling_altitude_method_offset = 1
    _set_ceiling_altitude_method_offset = 2
    _get_airspeed_method_offset = 3
    _get_airspeed_type_method_offset = 4
    _set_airspeed_method_offset = 5
    _get_altitude_rate_method_offset = 6
    _set_altitude_rate_method_offset = 7
    _get_use_aerodynamic_propulsion_fuel_method_offset = 8
    _set_use_aerodynamic_propulsion_fuel_method_offset = 9
    _get_scale_fuel_flow_by_non_std_density_method_offset = 10
    _set_scale_fuel_flow_by_non_std_density_method_offset = 11
    _get_fuel_flow_method_offset = 12
    _set_fuel_flow_method_offset = 13
    _get_enable_relative_airspeed_tolerance_method_offset = 14
    _set_enable_relative_airspeed_tolerance_method_offset = 15
    _get_relative_airspeed_tolerance_method_offset = 16
    _set_relative_airspeed_tolerance_method_offset = 17
    _get_as_catalog_item_method_offset = 18
    _metadata = {
        "iid_data" : (4965986558778284076, 16667235744495728830),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicClimbModel)

    _get_ceiling_altitude_metadata = { "offset" : _get_ceiling_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def ceiling_altitude(self) -> float:
        """Get or set the ceiling altitude."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_ceiling_altitude_metadata)

    _set_ceiling_altitude_metadata = { "offset" : _set_ceiling_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @ceiling_altitude.setter
    def ceiling_altitude(self, value:float) -> None:
        """Get or set the ceiling altitude."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_ceiling_altitude_metadata, value)

    _get_airspeed_metadata = { "offset" : _get_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def airspeed(self) -> float:
        """Get the airsepeed."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_airspeed_metadata)

    _get_airspeed_type_metadata = { "offset" : _get_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_type(self) -> "AirspeedType":
        """Get the airspeed type."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_airspeed_type_metadata)

    _set_airspeed_metadata = { "offset" : _set_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the airspeed type and value."""
        return self._intf.invoke(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_airspeed_metadata, airspeed_type, airspeed)

    _get_altitude_rate_metadata = { "offset" : _get_altitude_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_rate(self) -> float:
        """Get or set the altitude rate."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_altitude_rate_metadata)

    _set_altitude_rate_metadata = { "offset" : _set_altitude_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_rate.setter
    def altitude_rate(self, value:float) -> None:
        """Get or set the altitude rate."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_altitude_rate_metadata, value)

    _get_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _get_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_aerodynamic_propulsion_fuel(self) -> bool:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_use_aerodynamic_propulsion_fuel_metadata)

    _set_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _set_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_aerodynamic_propulsion_fuel.setter
    def use_aerodynamic_propulsion_fuel(self, value:bool) -> None:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_use_aerodynamic_propulsion_fuel_metadata, value)

    _get_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _get_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow_by_non_std_density(self) -> bool:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_scale_fuel_flow_by_non_std_density_metadata)

    _set_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _set_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow_by_non_std_density.setter
    def scale_fuel_flow_by_non_std_density(self, value:bool) -> None:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_scale_fuel_flow_by_non_std_density_metadata, value)

    _get_fuel_flow_metadata = { "offset" : _get_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_flow(self) -> float:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_fuel_flow_metadata)

    _set_fuel_flow_metadata = { "offset" : _set_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_flow.setter
    def fuel_flow(self, value:float) -> None:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_fuel_flow_metadata, value)

    _get_enable_relative_airspeed_tolerance_metadata = { "offset" : _get_enable_relative_airspeed_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_relative_airspeed_tolerance(self) -> bool:
        """Get or set whether to enable relative airspeed tolerance."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_enable_relative_airspeed_tolerance_metadata)

    _set_enable_relative_airspeed_tolerance_metadata = { "offset" : _set_enable_relative_airspeed_tolerance_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_relative_airspeed_tolerance.setter
    def enable_relative_airspeed_tolerance(self, value:bool) -> None:
        """Get or set whether to enable relative airspeed tolerance."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_enable_relative_airspeed_tolerance_metadata, value)

    _get_relative_airspeed_tolerance_metadata = { "offset" : _get_relative_airspeed_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_airspeed_tolerance(self) -> float:
        """Get or set the relative airspeed tolerance."""
        return self._intf.get_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_relative_airspeed_tolerance_metadata)

    _set_relative_airspeed_tolerance_metadata = { "offset" : _set_relative_airspeed_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_airspeed_tolerance.setter
    def relative_airspeed_tolerance(self, value:float) -> None:
        """Get or set the relative airspeed tolerance."""
        return self._intf.set_property(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._set_relative_airspeed_tolerance_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicClimbModel._metadata, AircraftBasicClimbModel._get_as_catalog_item_metadata, OutArg())

    _property_names[ceiling_altitude] = "ceiling_altitude"
    _property_names[airspeed] = "airspeed"
    _property_names[airspeed_type] = "airspeed_type"
    _property_names[altitude_rate] = "altitude_rate"
    _property_names[use_aerodynamic_propulsion_fuel] = "use_aerodynamic_propulsion_fuel"
    _property_names[scale_fuel_flow_by_non_std_density] = "scale_fuel_flow_by_non_std_density"
    _property_names[fuel_flow] = "fuel_flow"
    _property_names[enable_relative_airspeed_tolerance] = "enable_relative_airspeed_tolerance"
    _property_names[relative_airspeed_tolerance] = "relative_airspeed_tolerance"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicClimbModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicClimbModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicClimbModel, [AircraftBasicClimbModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5488216727438286377, 10220011439221355917), AircraftBasicClimbModel)
agcls.AgTypeNameMap["AircraftBasicClimbModel"] = AircraftBasicClimbModel

class AircraftAdvancedClimbModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the advanced climb model options for a climb model of an aircraft in the Aviator catalog."""

    _num_methods = 21
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_climb_speed_type_method_offset = 1
    _set_climb_speed_type_method_offset = 2
    _get_climb_override_airspeed_type_method_offset = 3
    _get_climb_override_airspeed_method_offset = 4
    _set_climb_override_airspeed_method_offset = 5
    _get_use_afterburner_method_offset = 6
    _set_use_afterburner_method_offset = 7
    _get_use_airspeed_limit_method_offset = 8
    _set_use_airspeed_limit_method_offset = 9
    _get_altitude_limit_method_offset = 10
    _set_altitude_limit_method_offset = 11
    _get_airspeed_limit_type_method_offset = 12
    _get_airspeed_limit_method_offset = 13
    _set_airspeed_limit_method_offset = 14
    _get_use_flight_path_angle_limit_method_offset = 15
    _set_use_flight_path_angle_limit_method_offset = 16
    _get_flight_path_angle_method_offset = 17
    _set_flight_path_angle_method_offset = 18
    _get_compute_delta_altitude_method_offset = 19
    _set_compute_delta_altitude_method_offset = 20
    _get_as_catalog_item_method_offset = 21
    _metadata = {
        "iid_data" : (5543563085120883607, 6217179742438742698),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedClimbModel)

    _get_climb_speed_type_metadata = { "offset" : _get_climb_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ClimbSpeedType),) }
    @property
    def climb_speed_type(self) -> "ClimbSpeedType":
        """Get or set the mode to calculate the aircraft's airspeed while climbing."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_climb_speed_type_metadata)

    _set_climb_speed_type_metadata = { "offset" : _set_climb_speed_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ClimbSpeedType),) }
    @climb_speed_type.setter
    def climb_speed_type(self, value:"ClimbSpeedType") -> None:
        """Get or set the mode to calculate the aircraft's airspeed while climbing."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_climb_speed_type_metadata, value)

    _get_climb_override_airspeed_type_metadata = { "offset" : _get_climb_override_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def climb_override_airspeed_type(self) -> "AirspeedType":
        """Get the override airspeed type."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_climb_override_airspeed_type_metadata)

    _get_climb_override_airspeed_metadata = { "offset" : _get_climb_override_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def climb_override_airspeed(self) -> float:
        """Get the override airsepeed."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_climb_override_airspeed_metadata)

    _set_climb_override_airspeed_metadata = { "offset" : _set_climb_override_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_climb_override_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the override airspeed and airspeed type."""
        return self._intf.invoke(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_climb_override_airspeed_metadata, airspeed_type, airspeed)

    _get_use_afterburner_metadata = { "offset" : _get_use_afterburner_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_afterburner(self) -> bool:
        """Opt to use the engine's afterburner when climbing if available."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_use_afterburner_metadata)

    _set_use_afterburner_metadata = { "offset" : _set_use_afterburner_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_afterburner.setter
    def use_afterburner(self, value:bool) -> None:
        """Opt to use the engine's afterburner when climbing if available."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_use_afterburner_metadata, value)

    _get_use_airspeed_limit_metadata = { "offset" : _get_use_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_airspeed_limit(self) -> bool:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_use_airspeed_limit_metadata)

    _set_use_airspeed_limit_metadata = { "offset" : _set_use_airspeed_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_airspeed_limit.setter
    def use_airspeed_limit(self, value:bool) -> None:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_use_airspeed_limit_metadata, value)

    _get_altitude_limit_metadata = { "offset" : _get_altitude_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_limit(self) -> float:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_altitude_limit_metadata)

    _set_altitude_limit_metadata = { "offset" : _set_altitude_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_limit.setter
    def altitude_limit(self, value:float) -> None:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_altitude_limit_metadata, value)

    _get_airspeed_limit_type_metadata = { "offset" : _get_airspeed_limit_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_limit_type(self) -> "AirspeedType":
        """Get the airspeed limit type."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_airspeed_limit_type_metadata)

    _get_airspeed_limit_metadata = { "offset" : _get_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def airspeed_limit(self) -> float:
        """Get the airsepeed limit below the altitude threshold."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_airspeed_limit_metadata)

    _set_airspeed_limit_metadata = { "offset" : _set_airspeed_limit_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_airspeed_limit(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the airspeed limit and airspeed type below the altitude threshold."""
        return self._intf.invoke(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_airspeed_limit_metadata, airspeed_type, airspeed)

    _get_use_flight_path_angle_limit_metadata = { "offset" : _get_use_flight_path_angle_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_flight_path_angle_limit(self) -> bool:
        """Opt to limit the flight path angle."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_use_flight_path_angle_limit_metadata)

    _set_use_flight_path_angle_limit_metadata = { "offset" : _set_use_flight_path_angle_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_flight_path_angle_limit.setter
    def use_flight_path_angle_limit(self, value:bool) -> None:
        """Opt to limit the flight path angle."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_use_flight_path_angle_limit_metadata, value)

    _get_flight_path_angle_metadata = { "offset" : _get_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def flight_path_angle(self) -> typing.Any:
        """Get the flight path angle limit."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_flight_path_angle_metadata)

    _set_flight_path_angle_metadata = { "offset" : _set_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_flight_path_angle(self, angle:typing.Any) -> None:
        """Enable the flight path angle limit an set the flight path angle value."""
        return self._intf.invoke(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_flight_path_angle_metadata, angle)

    _get_compute_delta_altitude_metadata = { "offset" : _get_compute_delta_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def compute_delta_altitude(self) -> float:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.get_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_compute_delta_altitude_metadata)

    _set_compute_delta_altitude_metadata = { "offset" : _set_compute_delta_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @compute_delta_altitude.setter
    def compute_delta_altitude(self, value:float) -> None:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.set_property(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._set_compute_delta_altitude_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedClimbModel._metadata, AircraftAdvancedClimbModel._get_as_catalog_item_metadata, OutArg())

    _property_names[climb_speed_type] = "climb_speed_type"
    _property_names[climb_override_airspeed_type] = "climb_override_airspeed_type"
    _property_names[climb_override_airspeed] = "climb_override_airspeed"
    _property_names[use_afterburner] = "use_afterburner"
    _property_names[use_airspeed_limit] = "use_airspeed_limit"
    _property_names[altitude_limit] = "altitude_limit"
    _property_names[airspeed_limit_type] = "airspeed_limit_type"
    _property_names[airspeed_limit] = "airspeed_limit"
    _property_names[use_flight_path_angle_limit] = "use_flight_path_angle_limit"
    _property_names[flight_path_angle] = "flight_path_angle"
    _property_names[compute_delta_altitude] = "compute_delta_altitude"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedClimbModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedClimbModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedClimbModel, [AircraftAdvancedClimbModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4736708800223409411, 13979770057904007563), AircraftAdvancedClimbModel)
agcls.AgTypeNameMap["AircraftAdvancedClimbModel"] = AircraftAdvancedClimbModel

class AircraftBasicCruiseModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """
    Interface used to access the basic cruise model options for a cruise model of an aircraft in the Aviator catalog.

    Examples
    --------
    Configure the basic cruise performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the cruise type
    >>> cruise = aviatorAircraft.cruise
    >>> # Get the build in performance model
    >>> basicCruiseModel = cruise.get_built_in_model()
    >>>
    >>> # Set the ceiling altitude
    >>> basicCruiseModel.ceiling_altitude = 50000
    >>> # Set the default cruise altitude
    >>> basicCruiseModel.default_cruise_altitude = 10000
    >>> # Set the airspeed type
    >>> basicCruiseModel.airspeed_type = AirspeedType.TAS
    >>> # Opt to not use the fuel flow calculated by the aero/prop model and instead specify the values
    >>> basicCruiseModel.use_aerodynamic_propulsion_fuel = False
    >>>
    >>> # Set the various airspeeds and fuel flows
    >>> basicCruiseModel.min_airspeed = 110
    >>> basicCruiseModel.min_airspeed_fuel_flow = 10000
    >>>
    >>> basicCruiseModel.max_endurance_airspeed = 135
    >>> basicCruiseModel.max_endurance_fuel_flow = 8000
    >>>
    >>> basicCruiseModel.max_airspeed = 570
    >>> basicCruiseModel.max_airspeed_fuel_flow = 30000
    >>>
    >>> basicCruiseModel.max_range_airspeed = 140
    >>> basicCruiseModel.max_range_fuel_flow = 9000
    >>>
    >>> basicCruiseModel.max_performance_airspeed = 150
    >>> basicCruiseModel.max_performance_airspeed_fuel_flow = 12000
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 31
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_ceiling_altitude_method_offset = 1
    _set_ceiling_altitude_method_offset = 2
    _get_default_cruise_altitude_method_offset = 3
    _set_default_cruise_altitude_method_offset = 4
    _get_use_aerodynamic_propulsion_fuel_method_offset = 5
    _set_use_aerodynamic_propulsion_fuel_method_offset = 6
    _get_scale_fuel_flow_by_non_std_density_method_offset = 7
    _set_scale_fuel_flow_by_non_std_density_method_offset = 8
    _get_airspeed_type_method_offset = 9
    _set_airspeed_type_method_offset = 10
    _get_min_airspeed_method_offset = 11
    _set_min_airspeed_method_offset = 12
    _get_max_endurance_airspeed_method_offset = 13
    _set_max_endurance_airspeed_method_offset = 14
    _get_max_range_airspeed_method_offset = 15
    _set_max_range_airspeed_method_offset = 16
    _get_max_airspeed_method_offset = 17
    _set_max_airspeed_method_offset = 18
    _get_max_performance_airspeed_method_offset = 19
    _set_max_performance_airspeed_method_offset = 20
    _get_min_airspeed_fuel_flow_method_offset = 21
    _set_min_airspeed_fuel_flow_method_offset = 22
    _get_max_endurance_fuel_flow_method_offset = 23
    _set_max_endurance_fuel_flow_method_offset = 24
    _get_max_range_fuel_flow_method_offset = 25
    _set_max_range_fuel_flow_method_offset = 26
    _get_max_airspeed_fuel_flow_method_offset = 27
    _set_max_airspeed_fuel_flow_method_offset = 28
    _get_max_performance_airspeed_fuel_flow_method_offset = 29
    _set_max_performance_airspeed_fuel_flow_method_offset = 30
    _get_as_catalog_item_method_offset = 31
    _metadata = {
        "iid_data" : (5314586510594823485, 6471808013847381131),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicCruiseModel)

    _get_ceiling_altitude_metadata = { "offset" : _get_ceiling_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def ceiling_altitude(self) -> float:
        """Get or set the maximum altitude above mean sea level."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_ceiling_altitude_metadata)

    _set_ceiling_altitude_metadata = { "offset" : _set_ceiling_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @ceiling_altitude.setter
    def ceiling_altitude(self, value:float) -> None:
        """Get or set the maximum altitude above mean sea level."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_ceiling_altitude_metadata, value)

    _get_default_cruise_altitude_metadata = { "offset" : _get_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def default_cruise_altitude(self) -> float:
        """Get or set the aircraft's default cruise altitude."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_default_cruise_altitude_metadata)

    _set_default_cruise_altitude_metadata = { "offset" : _set_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @default_cruise_altitude.setter
    def default_cruise_altitude(self, value:float) -> None:
        """Get or set the aircraft's default cruise altitude."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_default_cruise_altitude_metadata, value)

    _get_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _get_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_aerodynamic_propulsion_fuel(self) -> bool:
        """Opt to use the fuel flow calculated by the acceleration performance model."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_use_aerodynamic_propulsion_fuel_metadata)

    _set_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _set_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_aerodynamic_propulsion_fuel.setter
    def use_aerodynamic_propulsion_fuel(self, value:bool) -> None:
        """Opt to use the fuel flow calculated by the acceleration performance model."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_use_aerodynamic_propulsion_fuel_metadata, value)

    _get_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _get_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow_by_non_std_density(self) -> bool:
        """Opt to scale the fuel flow by the aircraft's actual altitude."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_scale_fuel_flow_by_non_std_density_metadata)

    _set_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _set_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow_by_non_std_density.setter
    def scale_fuel_flow_by_non_std_density(self, value:bool) -> None:
        """Opt to scale the fuel flow by the aircraft's actual altitude."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_scale_fuel_flow_by_non_std_density_metadata, value)

    _get_airspeed_type_metadata = { "offset" : _get_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_type(self) -> "AirspeedType":
        """Get or set the airspeed type."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_airspeed_type_metadata)

    _set_airspeed_type_metadata = { "offset" : _set_airspeed_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @airspeed_type.setter
    def airspeed_type(self, value:"AirspeedType") -> None:
        """Get or set the airspeed type."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_airspeed_type_metadata, value)

    _get_min_airspeed_metadata = { "offset" : _get_min_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_airspeed(self) -> float:
        """Get or set the minimum cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_min_airspeed_metadata)

    _set_min_airspeed_metadata = { "offset" : _set_min_airspeed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_airspeed.setter
    def min_airspeed(self, value:float) -> None:
        """Get or set the minimum cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_min_airspeed_metadata, value)

    _get_max_endurance_airspeed_metadata = { "offset" : _get_max_endurance_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_endurance_airspeed(self) -> float:
        """Get or set the cruising airspeed that will provide the maximum flying time for the aircraft."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_endurance_airspeed_metadata)

    _set_max_endurance_airspeed_metadata = { "offset" : _set_max_endurance_airspeed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_endurance_airspeed.setter
    def max_endurance_airspeed(self, value:float) -> None:
        """Get or set the cruising airspeed that will provide the maximum flying time for the aircraft."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_endurance_airspeed_metadata, value)

    _get_max_range_airspeed_metadata = { "offset" : _get_max_range_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_range_airspeed(self) -> float:
        """Get or set the maximum range cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_range_airspeed_metadata)

    _set_max_range_airspeed_metadata = { "offset" : _set_max_range_airspeed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_range_airspeed.setter
    def max_range_airspeed(self, value:float) -> None:
        """Get or set the maximum range cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_range_airspeed_metadata, value)

    _get_max_airspeed_metadata = { "offset" : _get_max_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_airspeed(self) -> float:
        """Get or set the maximum cruisng airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_airspeed_metadata)

    _set_max_airspeed_metadata = { "offset" : _set_max_airspeed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_airspeed.setter
    def max_airspeed(self, value:float) -> None:
        """Get or set the maximum cruisng airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_airspeed_metadata, value)

    _get_max_performance_airspeed_metadata = { "offset" : _get_max_performance_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_performance_airspeed(self) -> float:
        """Get or set the custom performance airspeed that can be used to model specific flight conditions."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_performance_airspeed_metadata)

    _set_max_performance_airspeed_metadata = { "offset" : _set_max_performance_airspeed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_performance_airspeed.setter
    def max_performance_airspeed(self, value:float) -> None:
        """Get or set the custom performance airspeed that can be used to model specific flight conditions."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_performance_airspeed_metadata, value)

    _get_min_airspeed_fuel_flow_metadata = { "offset" : _get_min_airspeed_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_airspeed_fuel_flow(self) -> float:
        """Get or set the fuel flow for the minimum cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_min_airspeed_fuel_flow_metadata)

    _set_min_airspeed_fuel_flow_metadata = { "offset" : _set_min_airspeed_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_airspeed_fuel_flow.setter
    def min_airspeed_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the minimum cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_min_airspeed_fuel_flow_metadata, value)

    _get_max_endurance_fuel_flow_metadata = { "offset" : _get_max_endurance_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_endurance_fuel_flow(self) -> float:
        """Get or set the fuel flow for the maximum endurance cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_endurance_fuel_flow_metadata)

    _set_max_endurance_fuel_flow_metadata = { "offset" : _set_max_endurance_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_endurance_fuel_flow.setter
    def max_endurance_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the maximumendurance cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_endurance_fuel_flow_metadata, value)

    _get_max_range_fuel_flow_metadata = { "offset" : _get_max_range_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_range_fuel_flow(self) -> float:
        """Get or set the fuel flow for the maximum range cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_range_fuel_flow_metadata)

    _set_max_range_fuel_flow_metadata = { "offset" : _set_max_range_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_range_fuel_flow.setter
    def max_range_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the maximum range cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_range_fuel_flow_metadata, value)

    _get_max_airspeed_fuel_flow_metadata = { "offset" : _get_max_airspeed_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_airspeed_fuel_flow(self) -> float:
        """Get or set the fuel flow for the maximum cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_airspeed_fuel_flow_metadata)

    _set_max_airspeed_fuel_flow_metadata = { "offset" : _set_max_airspeed_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_airspeed_fuel_flow.setter
    def max_airspeed_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the maximum cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_airspeed_fuel_flow_metadata, value)

    _get_max_performance_airspeed_fuel_flow_metadata = { "offset" : _get_max_performance_airspeed_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_performance_airspeed_fuel_flow(self) -> float:
        """Get or set the fuel flow for the maximum performance cruising airspeed."""
        return self._intf.get_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_max_performance_airspeed_fuel_flow_metadata)

    _set_max_performance_airspeed_fuel_flow_metadata = { "offset" : _set_max_performance_airspeed_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_performance_airspeed_fuel_flow.setter
    def max_performance_airspeed_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the maximum performance cruising airspeed."""
        return self._intf.set_property(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._set_max_performance_airspeed_fuel_flow_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicCruiseModel._metadata, AircraftBasicCruiseModel._get_as_catalog_item_metadata, OutArg())

    _property_names[ceiling_altitude] = "ceiling_altitude"
    _property_names[default_cruise_altitude] = "default_cruise_altitude"
    _property_names[use_aerodynamic_propulsion_fuel] = "use_aerodynamic_propulsion_fuel"
    _property_names[scale_fuel_flow_by_non_std_density] = "scale_fuel_flow_by_non_std_density"
    _property_names[airspeed_type] = "airspeed_type"
    _property_names[min_airspeed] = "min_airspeed"
    _property_names[max_endurance_airspeed] = "max_endurance_airspeed"
    _property_names[max_range_airspeed] = "max_range_airspeed"
    _property_names[max_airspeed] = "max_airspeed"
    _property_names[max_performance_airspeed] = "max_performance_airspeed"
    _property_names[min_airspeed_fuel_flow] = "min_airspeed_fuel_flow"
    _property_names[max_endurance_fuel_flow] = "max_endurance_fuel_flow"
    _property_names[max_range_fuel_flow] = "max_range_fuel_flow"
    _property_names[max_airspeed_fuel_flow] = "max_airspeed_fuel_flow"
    _property_names[max_performance_airspeed_fuel_flow] = "max_performance_airspeed_fuel_flow"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicCruiseModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicCruiseModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicCruiseModel, [AircraftBasicCruiseModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5471844237425838798, 12111710160149214125), AircraftBasicCruiseModel)
agcls.AgTypeNameMap["AircraftBasicCruiseModel"] = AircraftBasicCruiseModel

class AircraftAdvancedCruiseModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the advanced cruise model options for a cruise model of an aircraft in the Aviator catalog."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_default_cruise_altitude_method_offset = 1
    _set_default_cruise_altitude_method_offset = 2
    _get_max_performance_airspeed_method_offset = 3
    _set_max_performance_airspeed_method_offset = 4
    _get_use_airspeed_limit_method_offset = 5
    _set_use_airspeed_limit_method_offset = 6
    _get_altitude_limit_method_offset = 7
    _set_altitude_limit_method_offset = 8
    _get_airspeed_limit_type_method_offset = 9
    _get_airspeed_limit_method_offset = 10
    _set_airspeed_limit_method_offset = 11
    _get_compute_delta_downrange_method_offset = 12
    _set_compute_delta_downrange_method_offset = 13
    _get_as_catalog_item_method_offset = 14
    _metadata = {
        "iid_data" : (4951863565701278341, 9812484038019628430),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedCruiseModel)

    _get_default_cruise_altitude_metadata = { "offset" : _get_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def default_cruise_altitude(self) -> float:
        """Get or set the aircraft's default cruising altitude."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_default_cruise_altitude_metadata)

    _set_default_cruise_altitude_metadata = { "offset" : _set_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @default_cruise_altitude.setter
    def default_cruise_altitude(self, value:float) -> None:
        """Get or set the aircraft's default cruising altitude."""
        return self._intf.set_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_default_cruise_altitude_metadata, value)

    _get_max_performance_airspeed_metadata = { "offset" : _get_max_performance_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(CruiseMaxPerformanceSpeedType),) }
    @property
    def max_performance_airspeed(self) -> "CruiseMaxPerformanceSpeedType":
        """Get or set the method for defining the maximum performance airspeed of the aircraft with respect to its altitude."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_max_performance_airspeed_metadata)

    _set_max_performance_airspeed_metadata = { "offset" : _set_max_performance_airspeed_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(CruiseMaxPerformanceSpeedType),) }
    @max_performance_airspeed.setter
    def max_performance_airspeed(self, value:"CruiseMaxPerformanceSpeedType") -> None:
        """Get or set the method for defining the maximum performance airspeed of the aircraft with respect to its altitude."""
        return self._intf.set_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_max_performance_airspeed_metadata, value)

    _get_use_airspeed_limit_metadata = { "offset" : _get_use_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_airspeed_limit(self) -> bool:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_use_airspeed_limit_metadata)

    _set_use_airspeed_limit_metadata = { "offset" : _set_use_airspeed_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_airspeed_limit.setter
    def use_airspeed_limit(self, value:bool) -> None:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.set_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_use_airspeed_limit_metadata, value)

    _get_altitude_limit_metadata = { "offset" : _get_altitude_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_limit(self) -> float:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_altitude_limit_metadata)

    _set_altitude_limit_metadata = { "offset" : _set_altitude_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_limit.setter
    def altitude_limit(self, value:float) -> None:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.set_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_altitude_limit_metadata, value)

    _get_airspeed_limit_type_metadata = { "offset" : _get_airspeed_limit_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_limit_type(self) -> "AirspeedType":
        """Get the airspeed limit type."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_airspeed_limit_type_metadata)

    _get_airspeed_limit_metadata = { "offset" : _get_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def airspeed_limit(self) -> float:
        """Get the airsepeed limit below the altitude threshold."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_airspeed_limit_metadata)

    _set_airspeed_limit_metadata = { "offset" : _set_airspeed_limit_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_airspeed_limit(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the airspeed limit and airspeed type below the altitude threshold."""
        return self._intf.invoke(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_airspeed_limit_metadata, airspeed_type, airspeed)

    _get_compute_delta_downrange_metadata = { "offset" : _get_compute_delta_downrange_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def compute_delta_downrange(self) -> float:
        """Get or set the maximum change in downrange distance in a computed segment before the data is sampled again."""
        return self._intf.get_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_compute_delta_downrange_metadata)

    _set_compute_delta_downrange_metadata = { "offset" : _set_compute_delta_downrange_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @compute_delta_downrange.setter
    def compute_delta_downrange(self, value:float) -> None:
        """Get or set the maximum change in downrange distance in a computed segment before the data is sampled again."""
        return self._intf.set_property(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._set_compute_delta_downrange_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedCruiseModel._metadata, AircraftAdvancedCruiseModel._get_as_catalog_item_metadata, OutArg())

    _property_names[default_cruise_altitude] = "default_cruise_altitude"
    _property_names[max_performance_airspeed] = "max_performance_airspeed"
    _property_names[use_airspeed_limit] = "use_airspeed_limit"
    _property_names[altitude_limit] = "altitude_limit"
    _property_names[airspeed_limit_type] = "airspeed_limit_type"
    _property_names[airspeed_limit] = "airspeed_limit"
    _property_names[compute_delta_downrange] = "compute_delta_downrange"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedCruiseModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedCruiseModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedCruiseModel, [AircraftAdvancedCruiseModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5455870576064233751, 10403403645709593512), AircraftAdvancedCruiseModel)
agcls.AgTypeNameMap["AircraftAdvancedCruiseModel"] = AircraftAdvancedCruiseModel

class AircraftBasicDescentModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the basic descent model options for a descent model of an aircraft in the Aviator catalog."""

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_ceiling_altitude_method_offset = 1
    _set_ceiling_altitude_method_offset = 2
    _get_airspeed_method_offset = 3
    _get_airspeed_type_method_offset = 4
    _set_airspeed_method_offset = 5
    _get_altitude_rate_method_offset = 6
    _set_altitude_rate_method_offset = 7
    _get_use_aerodynamic_propulsion_fuel_method_offset = 8
    _set_use_aerodynamic_propulsion_fuel_method_offset = 9
    _get_scale_fuel_flow_by_non_std_density_method_offset = 10
    _set_scale_fuel_flow_by_non_std_density_method_offset = 11
    _get_fuel_flow_method_offset = 12
    _set_fuel_flow_method_offset = 13
    _get_enable_relative_airspeed_tolerance_method_offset = 14
    _set_enable_relative_airspeed_tolerance_method_offset = 15
    _get_relative_airspeed_tolerance_method_offset = 16
    _set_relative_airspeed_tolerance_method_offset = 17
    _get_as_catalog_item_method_offset = 18
    _metadata = {
        "iid_data" : (4619424590425145819, 1963764848038236049),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicDescentModel)

    _get_ceiling_altitude_metadata = { "offset" : _get_ceiling_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def ceiling_altitude(self) -> float:
        """Get or set the ceiling altitude."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_ceiling_altitude_metadata)

    _set_ceiling_altitude_metadata = { "offset" : _set_ceiling_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @ceiling_altitude.setter
    def ceiling_altitude(self, value:float) -> None:
        """Get or set the ceiling altitude."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_ceiling_altitude_metadata, value)

    _get_airspeed_metadata = { "offset" : _get_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def airspeed(self) -> float:
        """Get the airsepeed."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_airspeed_metadata)

    _get_airspeed_type_metadata = { "offset" : _get_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_type(self) -> "AirspeedType":
        """Get the airspeed type."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_airspeed_type_metadata)

    _set_airspeed_metadata = { "offset" : _set_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the airspeed type and value."""
        return self._intf.invoke(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_airspeed_metadata, airspeed_type, airspeed)

    _get_altitude_rate_metadata = { "offset" : _get_altitude_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_rate(self) -> float:
        """Get or set the altitude rate."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_altitude_rate_metadata)

    _set_altitude_rate_metadata = { "offset" : _set_altitude_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_rate.setter
    def altitude_rate(self, value:float) -> None:
        """Get or set the altitude rate."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_altitude_rate_metadata, value)

    _get_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _get_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_aerodynamic_propulsion_fuel(self) -> bool:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_use_aerodynamic_propulsion_fuel_metadata)

    _set_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _set_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_aerodynamic_propulsion_fuel.setter
    def use_aerodynamic_propulsion_fuel(self, value:bool) -> None:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_use_aerodynamic_propulsion_fuel_metadata, value)

    _get_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _get_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow_by_non_std_density(self) -> bool:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_scale_fuel_flow_by_non_std_density_metadata)

    _set_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _set_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow_by_non_std_density.setter
    def scale_fuel_flow_by_non_std_density(self, value:bool) -> None:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_scale_fuel_flow_by_non_std_density_metadata, value)

    _get_fuel_flow_metadata = { "offset" : _get_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_flow(self) -> float:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_fuel_flow_metadata)

    _set_fuel_flow_metadata = { "offset" : _set_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_flow.setter
    def fuel_flow(self, value:float) -> None:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_fuel_flow_metadata, value)

    _get_enable_relative_airspeed_tolerance_metadata = { "offset" : _get_enable_relative_airspeed_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_relative_airspeed_tolerance(self) -> bool:
        """Get or set whether to enable relative airspeed tolerance."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_enable_relative_airspeed_tolerance_metadata)

    _set_enable_relative_airspeed_tolerance_metadata = { "offset" : _set_enable_relative_airspeed_tolerance_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_relative_airspeed_tolerance.setter
    def enable_relative_airspeed_tolerance(self, value:bool) -> None:
        """Get or set whether to enable relative airspeed tolerance."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_enable_relative_airspeed_tolerance_metadata, value)

    _get_relative_airspeed_tolerance_metadata = { "offset" : _get_relative_airspeed_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_airspeed_tolerance(self) -> float:
        """Get or set the relative airspeed tolerance."""
        return self._intf.get_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_relative_airspeed_tolerance_metadata)

    _set_relative_airspeed_tolerance_metadata = { "offset" : _set_relative_airspeed_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_airspeed_tolerance.setter
    def relative_airspeed_tolerance(self, value:float) -> None:
        """Get or set the relative airspeed tolerance."""
        return self._intf.set_property(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._set_relative_airspeed_tolerance_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicDescentModel._metadata, AircraftBasicDescentModel._get_as_catalog_item_metadata, OutArg())

    _property_names[ceiling_altitude] = "ceiling_altitude"
    _property_names[airspeed] = "airspeed"
    _property_names[airspeed_type] = "airspeed_type"
    _property_names[altitude_rate] = "altitude_rate"
    _property_names[use_aerodynamic_propulsion_fuel] = "use_aerodynamic_propulsion_fuel"
    _property_names[scale_fuel_flow_by_non_std_density] = "scale_fuel_flow_by_non_std_density"
    _property_names[fuel_flow] = "fuel_flow"
    _property_names[enable_relative_airspeed_tolerance] = "enable_relative_airspeed_tolerance"
    _property_names[relative_airspeed_tolerance] = "relative_airspeed_tolerance"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicDescentModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicDescentModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicDescentModel, [AircraftBasicDescentModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4788927177679408495, 6388236481941240196), AircraftBasicDescentModel)
agcls.AgTypeNameMap["AircraftBasicDescentModel"] = AircraftBasicDescentModel

class AircraftAdvancedDescentModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the advanced descent model options for a descent model of an aircraft in the Aviator catalog."""

    _num_methods = 19
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_descent_speed_type_method_offset = 1
    _set_descent_speed_type_method_offset = 2
    _get_descent_stall_speed_ratio_method_offset = 3
    _set_descent_stall_speed_ratio_method_offset = 4
    _get_descent_override_airspeed_type_method_offset = 5
    _get_descent_override_airspeed_method_offset = 6
    _set_descent_override_airspeed_method_offset = 7
    _get_speedbrakes_method_offset = 8
    _set_speedbrakes_method_offset = 9
    _get_use_airspeed_limit_method_offset = 10
    _set_use_airspeed_limit_method_offset = 11
    _get_altitude_limit_method_offset = 12
    _set_altitude_limit_method_offset = 13
    _get_airspeed_limit_type_method_offset = 14
    _get_airspeed_limit_method_offset = 15
    _set_airspeed_limit_method_offset = 16
    _get_compute_delta_altitude_method_offset = 17
    _set_compute_delta_altitude_method_offset = 18
    _get_as_catalog_item_method_offset = 19
    _metadata = {
        "iid_data" : (5138930714286807665, 1693484297723898786),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedDescentModel)

    _get_descent_speed_type_metadata = { "offset" : _get_descent_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DescentSpeedType),) }
    @property
    def descent_speed_type(self) -> "DescentSpeedType":
        """Get or set the mode to calculate the aircraft's airspeed while descending ."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_descent_speed_type_metadata)

    _set_descent_speed_type_metadata = { "offset" : _set_descent_speed_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DescentSpeedType),) }
    @descent_speed_type.setter
    def descent_speed_type(self, value:"DescentSpeedType") -> None:
        """Get or set the mode to calculate the aircraft's airspeed while descending ."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_descent_speed_type_metadata, value)

    _get_descent_stall_speed_ratio_metadata = { "offset" : _get_descent_stall_speed_ratio_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def descent_stall_speed_ratio(self) -> float:
        """Get or set the ratio of the airspeed upon leaving the ground to the stall speed."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_descent_stall_speed_ratio_metadata)

    _set_descent_stall_speed_ratio_metadata = { "offset" : _set_descent_stall_speed_ratio_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @descent_stall_speed_ratio.setter
    def descent_stall_speed_ratio(self, value:float) -> None:
        """Get or set the ratio of the airspeed upon leaving the ground to the stall speed."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_descent_stall_speed_ratio_metadata, value)

    _get_descent_override_airspeed_type_metadata = { "offset" : _get_descent_override_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def descent_override_airspeed_type(self) -> "AirspeedType":
        """Get the override airspeed type."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_descent_override_airspeed_type_metadata)

    _get_descent_override_airspeed_metadata = { "offset" : _get_descent_override_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def descent_override_airspeed(self) -> float:
        """Get the override airsepeed."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_descent_override_airspeed_metadata)

    _set_descent_override_airspeed_metadata = { "offset" : _set_descent_override_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_descent_override_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the override airspeed and airspeed type."""
        return self._intf.invoke(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_descent_override_airspeed_metadata, airspeed_type, airspeed)

    _get_speedbrakes_metadata = { "offset" : _get_speedbrakes_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def speedbrakes(self) -> float:
        """Get or set the extension of the speedbrakes during the landing."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_speedbrakes_metadata)

    _set_speedbrakes_metadata = { "offset" : _set_speedbrakes_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @speedbrakes.setter
    def speedbrakes(self, value:float) -> None:
        """Get or set the extension of the speedbrakes during the landing."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_speedbrakes_metadata, value)

    _get_use_airspeed_limit_metadata = { "offset" : _get_use_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_airspeed_limit(self) -> bool:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_use_airspeed_limit_metadata)

    _set_use_airspeed_limit_metadata = { "offset" : _set_use_airspeed_limit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_airspeed_limit.setter
    def use_airspeed_limit(self, value:bool) -> None:
        """Opt to limit the airspeed below a specified altitude."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_use_airspeed_limit_metadata, value)

    _get_altitude_limit_metadata = { "offset" : _get_altitude_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_limit(self) -> float:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_altitude_limit_metadata)

    _set_altitude_limit_metadata = { "offset" : _set_altitude_limit_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_limit.setter
    def altitude_limit(self, value:float) -> None:
        """Get or set the altitude threshold, below which the airspeed limit will be applied."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_altitude_limit_metadata, value)

    _get_airspeed_limit_type_metadata = { "offset" : _get_airspeed_limit_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def airspeed_limit_type(self) -> "AirspeedType":
        """Get the airspeed limit type."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_airspeed_limit_type_metadata)

    _get_airspeed_limit_metadata = { "offset" : _get_airspeed_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def airspeed_limit(self) -> float:
        """Get the airsepeed limit below the altitude threshold."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_airspeed_limit_metadata)

    _set_airspeed_limit_metadata = { "offset" : _set_airspeed_limit_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_airspeed_limit(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the airspeed limit and airspeed type below the altitude threshold."""
        return self._intf.invoke(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_airspeed_limit_metadata, airspeed_type, airspeed)

    _get_compute_delta_altitude_metadata = { "offset" : _get_compute_delta_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def compute_delta_altitude(self) -> float:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.get_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_compute_delta_altitude_metadata)

    _set_compute_delta_altitude_metadata = { "offset" : _set_compute_delta_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @compute_delta_altitude.setter
    def compute_delta_altitude(self, value:float) -> None:
        """Get or set the maximum change in altitude in a computed segment before the data is sampled again."""
        return self._intf.set_property(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._set_compute_delta_altitude_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedDescentModel._metadata, AircraftAdvancedDescentModel._get_as_catalog_item_metadata, OutArg())

    _property_names[descent_speed_type] = "descent_speed_type"
    _property_names[descent_stall_speed_ratio] = "descent_stall_speed_ratio"
    _property_names[descent_override_airspeed_type] = "descent_override_airspeed_type"
    _property_names[descent_override_airspeed] = "descent_override_airspeed"
    _property_names[speedbrakes] = "speedbrakes"
    _property_names[use_airspeed_limit] = "use_airspeed_limit"
    _property_names[altitude_limit] = "altitude_limit"
    _property_names[airspeed_limit_type] = "airspeed_limit_type"
    _property_names[airspeed_limit] = "airspeed_limit"
    _property_names[compute_delta_altitude] = "compute_delta_altitude"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedDescentModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedDescentModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedDescentModel, [AircraftAdvancedDescentModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5324825486015993906, 2788835445536991165), AircraftAdvancedDescentModel)
agcls.AgTypeNameMap["AircraftAdvancedDescentModel"] = AircraftAdvancedDescentModel

class AircraftBasicTakeoffModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the basic takeoff model options for a takeoff model of an aircraft in the Aviator catalog."""

    _num_methods = 17
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_takeoff_speed_method_offset = 1
    _get_takeoff_speed_type_method_offset = 2
    _set_takeoff_speed_method_offset = 3
    _get_sea_level_ground_roll_method_offset = 4
    _set_sea_level_ground_roll_method_offset = 5
    _get_departure_speed_method_offset = 6
    _get_departure_speed_type_method_offset = 7
    _set_departure_speed_method_offset = 8
    _get_use_aerodynamic_propulsion_fuel_method_offset = 9
    _set_use_aerodynamic_propulsion_fuel_method_offset = 10
    _get_scale_fuel_flow_by_non_std_density_method_offset = 11
    _set_scale_fuel_flow_by_non_std_density_method_offset = 12
    _get_acceleration_fuel_flow_method_offset = 13
    _set_acceleration_fuel_flow_method_offset = 14
    _get_departure_fuel_flow_method_offset = 15
    _set_departure_fuel_flow_method_offset = 16
    _get_as_catalog_item_method_offset = 17
    _metadata = {
        "iid_data" : (5032744767634970533, 755518156549444771),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicTakeoffModel)

    _get_takeoff_speed_metadata = { "offset" : _get_takeoff_speed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_speed(self) -> float:
        """Get the speed to which the aircraft accelerates on its ground roll for takeoff."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_takeoff_speed_metadata)

    _get_takeoff_speed_type_metadata = { "offset" : _get_takeoff_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def takeoff_speed_type(self) -> "AirspeedType":
        """Get the takeoff speed type."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_takeoff_speed_type_metadata)

    _set_takeoff_speed_metadata = { "offset" : _set_takeoff_speed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_takeoff_speed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the takeoff speed of the aircraft."""
        return self._intf.invoke(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_takeoff_speed_metadata, airspeed_type, airspeed)

    _get_sea_level_ground_roll_metadata = { "offset" : _get_sea_level_ground_roll_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def sea_level_ground_roll(self) -> float:
        """Get or set the distance the aircraft travels along the ground while accelerationg to takeoff at sea level."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_sea_level_ground_roll_metadata)

    _set_sea_level_ground_roll_metadata = { "offset" : _set_sea_level_ground_roll_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @sea_level_ground_roll.setter
    def sea_level_ground_roll(self, value:float) -> None:
        """Get or set the distance the aircraft travels along the ground while accelerationg to takeoff at sea level."""
        return self._intf.set_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_sea_level_ground_roll_metadata, value)

    _get_departure_speed_metadata = { "offset" : _get_departure_speed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_speed(self) -> float:
        """Get the aircraft's speed upon leaving the ground."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_departure_speed_metadata)

    _get_departure_speed_type_metadata = { "offset" : _get_departure_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def departure_speed_type(self) -> "AirspeedType":
        """Get the departure speed type."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_departure_speed_type_metadata)

    _set_departure_speed_metadata = { "offset" : _set_departure_speed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_departure_speed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the departure speed of the aircraft."""
        return self._intf.invoke(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_departure_speed_metadata, airspeed_type, airspeed)

    _get_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _get_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_aerodynamic_propulsion_fuel(self) -> bool:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_use_aerodynamic_propulsion_fuel_metadata)

    _set_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _set_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_aerodynamic_propulsion_fuel.setter
    def use_aerodynamic_propulsion_fuel(self, value:bool) -> None:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.set_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_use_aerodynamic_propulsion_fuel_metadata, value)

    _get_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _get_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow_by_non_std_density(self) -> bool:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_scale_fuel_flow_by_non_std_density_metadata)

    _set_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _set_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow_by_non_std_density.setter
    def scale_fuel_flow_by_non_std_density(self, value:bool) -> None:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.set_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_scale_fuel_flow_by_non_std_density_metadata, value)

    _get_acceleration_fuel_flow_metadata = { "offset" : _get_acceleration_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def acceleration_fuel_flow(self) -> float:
        """Get or set the aircraft's fuel flow rate while accelerating during takeoff."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_acceleration_fuel_flow_metadata)

    _set_acceleration_fuel_flow_metadata = { "offset" : _set_acceleration_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @acceleration_fuel_flow.setter
    def acceleration_fuel_flow(self, value:float) -> None:
        """Get or set the aircraft's fuel flow rate while accelerating during takeoff."""
        return self._intf.set_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_acceleration_fuel_flow_metadata, value)

    _get_departure_fuel_flow_metadata = { "offset" : _get_departure_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_fuel_flow(self) -> float:
        """Get or set the aircraft's fuel flow rate at departure speed."""
        return self._intf.get_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_departure_fuel_flow_metadata)

    _set_departure_fuel_flow_metadata = { "offset" : _set_departure_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @departure_fuel_flow.setter
    def departure_fuel_flow(self, value:float) -> None:
        """Get or set the aircraft's fuel flow rate at departure speed."""
        return self._intf.set_property(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._set_departure_fuel_flow_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicTakeoffModel._metadata, AircraftBasicTakeoffModel._get_as_catalog_item_metadata, OutArg())

    _property_names[takeoff_speed] = "takeoff_speed"
    _property_names[takeoff_speed_type] = "takeoff_speed_type"
    _property_names[sea_level_ground_roll] = "sea_level_ground_roll"
    _property_names[departure_speed] = "departure_speed"
    _property_names[departure_speed_type] = "departure_speed_type"
    _property_names[use_aerodynamic_propulsion_fuel] = "use_aerodynamic_propulsion_fuel"
    _property_names[scale_fuel_flow_by_non_std_density] = "scale_fuel_flow_by_non_std_density"
    _property_names[acceleration_fuel_flow] = "acceleration_fuel_flow"
    _property_names[departure_fuel_flow] = "departure_fuel_flow"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicTakeoffModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicTakeoffModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicTakeoffModel, [AircraftBasicTakeoffModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5606729336629723337, 16741769561151290246), AircraftBasicTakeoffModel)
agcls.AgTypeNameMap["AircraftBasicTakeoffModel"] = AircraftBasicTakeoffModel

class AircraftAdvancedTakeoffModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the advanced takeoff model options for a takeoff model of an aircraft in the Aviator catalog."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_takeoff_speed_mode_method_offset = 1
    _set_takeoff_speed_mode_method_offset = 2
    _get_stall_speed_ratio_method_offset = 3
    _set_stall_speed_ratio_method_offset = 4
    _get_angle_of_attack_method_offset = 5
    _set_angle_of_attack_method_offset = 6
    _get_flaps_method_offset = 7
    _set_flaps_method_offset = 8
    _get_departure_speed_mode_method_offset = 9
    _set_departure_speed_mode_method_offset = 10
    _get_departure_speed_limit_method_offset = 11
    _get_departure_speed_limit_type_method_offset = 12
    _set_departure_speed_limit_method_offset = 13
    _get_use_afterburner_method_offset = 14
    _set_use_afterburner_method_offset = 15
    _get_as_catalog_item_method_offset = 16
    _metadata = {
        "iid_data" : (5320830156872402224, 10166003988108511629),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedTakeoffModel)

    _get_takeoff_speed_mode_metadata = { "offset" : _get_takeoff_speed_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TakeoffLandingSpeedMode),) }
    @property
    def takeoff_speed_mode(self) -> "TakeoffLandingSpeedMode":
        """Get or set the mode to calculate the aircraft's airspeed upon leaving the ground."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_takeoff_speed_mode_metadata)

    _set_takeoff_speed_mode_metadata = { "offset" : _set_takeoff_speed_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TakeoffLandingSpeedMode),) }
    @takeoff_speed_mode.setter
    def takeoff_speed_mode(self, value:"TakeoffLandingSpeedMode") -> None:
        """Get or set the mode to calculate the aircraft's airspeed upon leaving the ground."""
        return self._intf.set_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_takeoff_speed_mode_metadata, value)

    _get_stall_speed_ratio_metadata = { "offset" : _get_stall_speed_ratio_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stall_speed_ratio(self) -> float:
        """Get the ratio of the airspeed to the stall speed upon leaving the ground."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_stall_speed_ratio_metadata)

    _set_stall_speed_ratio_metadata = { "offset" : _set_stall_speed_ratio_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    def set_stall_speed_ratio(self, stall_speed_ratio:float) -> None:
        """Set the takeoff speed mode to StallSpeedRatio and specify the stall speed ratio."""
        return self._intf.invoke(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_stall_speed_ratio_metadata, stall_speed_ratio)

    _get_angle_of_attack_metadata = { "offset" : _get_angle_of_attack_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def angle_of_attack(self) -> typing.Any:
        """Get the Angle of Attack upon leaving the ground."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_angle_of_attack_metadata)

    _set_angle_of_attack_metadata = { "offset" : _set_angle_of_attack_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_angle_of_attack(self, angle_of_attack:typing.Any) -> None:
        """Set the takeoff speed mode to AngleOfAttack and specify the angle of attack."""
        return self._intf.invoke(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_angle_of_attack_metadata, angle_of_attack)

    _get_flaps_metadata = { "offset" : _get_flaps_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def flaps(self) -> float:
        """Get or set the extension of the flaps during takeoff."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_flaps_metadata)

    _set_flaps_metadata = { "offset" : _set_flaps_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @flaps.setter
    def flaps(self, value:float) -> None:
        """Get or set the extension of the flaps during takeoff."""
        return self._intf.set_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_flaps_metadata, value)

    _get_departure_speed_mode_metadata = { "offset" : _get_departure_speed_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DepartureSpeedMode),) }
    @property
    def departure_speed_mode(self) -> "DepartureSpeedMode":
        """Get or set the mode to calculate the airspeed upon leaving the ground."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_departure_speed_mode_metadata)

    _set_departure_speed_mode_metadata = { "offset" : _set_departure_speed_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DepartureSpeedMode),) }
    @departure_speed_mode.setter
    def departure_speed_mode(self, value:"DepartureSpeedMode") -> None:
        """Get or set the mode to calculate the airspeed upon leaving the ground."""
        return self._intf.set_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_departure_speed_mode_metadata, value)

    _get_departure_speed_limit_metadata = { "offset" : _get_departure_speed_limit_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_speed_limit(self) -> float:
        """Get the aircraft's maximum airspeed upon leaving the ground."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_departure_speed_limit_metadata)

    _get_departure_speed_limit_type_metadata = { "offset" : _get_departure_speed_limit_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def departure_speed_limit_type(self) -> "AirspeedType":
        """Get the departure speed limim type."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_departure_speed_limit_type_metadata)

    _set_departure_speed_limit_metadata = { "offset" : _set_departure_speed_limit_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_departure_speed_limit(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the departure speed limit of the aircraft."""
        return self._intf.invoke(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_departure_speed_limit_metadata, airspeed_type, airspeed)

    _get_use_afterburner_metadata = { "offset" : _get_use_afterburner_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_afterburner(self) -> bool:
        """Opt whether to use the afterburner if it is possible."""
        return self._intf.get_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_use_afterburner_metadata)

    _set_use_afterburner_metadata = { "offset" : _set_use_afterburner_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_afterburner.setter
    def use_afterburner(self, value:bool) -> None:
        """Opt whether to use the afterburner if it is possible."""
        return self._intf.set_property(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._set_use_afterburner_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedTakeoffModel._metadata, AircraftAdvancedTakeoffModel._get_as_catalog_item_metadata, OutArg())

    _property_names[takeoff_speed_mode] = "takeoff_speed_mode"
    _property_names[stall_speed_ratio] = "stall_speed_ratio"
    _property_names[angle_of_attack] = "angle_of_attack"
    _property_names[flaps] = "flaps"
    _property_names[departure_speed_mode] = "departure_speed_mode"
    _property_names[departure_speed_limit] = "departure_speed_limit"
    _property_names[departure_speed_limit_type] = "departure_speed_limit_type"
    _property_names[use_afterburner] = "use_afterburner"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedTakeoffModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedTakeoffModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedTakeoffModel, [AircraftAdvancedTakeoffModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5096944394266841251, 9249196675554252163), AircraftAdvancedTakeoffModel)
agcls.AgTypeNameMap["AircraftAdvancedTakeoffModel"] = AircraftAdvancedTakeoffModel

class AircraftBasicLandingModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the basic landing model options for a landing model of an aircraft in the Aviator catalog."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_landing_speed_method_offset = 1
    _get_landing_speed_type_method_offset = 2
    _set_landing_speed_method_offset = 3
    _get_sea_level_ground_roll_method_offset = 4
    _set_sea_level_ground_roll_method_offset = 5
    _get_use_aerodynamic_propulsion_fuel_method_offset = 6
    _set_use_aerodynamic_propulsion_fuel_method_offset = 7
    _get_scale_fuel_flow_by_non_std_density_method_offset = 8
    _set_scale_fuel_flow_by_non_std_density_method_offset = 9
    _get_fuel_flow_method_offset = 10
    _set_fuel_flow_method_offset = 11
    _get_as_catalog_item_method_offset = 12
    _metadata = {
        "iid_data" : (4905149018706593974, 16513096893308921742),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicLandingModel)

    _get_landing_speed_metadata = { "offset" : _get_landing_speed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def landing_speed(self) -> float:
        """Get the landing speed of the aircraft."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_landing_speed_metadata)

    _get_landing_speed_type_metadata = { "offset" : _get_landing_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def landing_speed_type(self) -> "AirspeedType":
        """Get the landing speed type."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_landing_speed_type_metadata)

    _set_landing_speed_metadata = { "offset" : _set_landing_speed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_landing_speed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the landing speed of the aircraft."""
        return self._intf.invoke(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._set_landing_speed_metadata, airspeed_type, airspeed)

    _get_sea_level_ground_roll_metadata = { "offset" : _get_sea_level_ground_roll_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def sea_level_ground_roll(self) -> float:
        """Get or set the distance the aircraft travels along the ground while decelerating to a stop at sea level."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_sea_level_ground_roll_metadata)

    _set_sea_level_ground_roll_metadata = { "offset" : _set_sea_level_ground_roll_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @sea_level_ground_roll.setter
    def sea_level_ground_roll(self, value:float) -> None:
        """Get or set the distance the aircraft travels along the ground while decelerating to a stop at sea level."""
        return self._intf.set_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._set_sea_level_ground_roll_metadata, value)

    _get_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _get_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_aerodynamic_propulsion_fuel(self) -> bool:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_use_aerodynamic_propulsion_fuel_metadata)

    _set_use_aerodynamic_propulsion_fuel_metadata = { "offset" : _set_use_aerodynamic_propulsion_fuel_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_aerodynamic_propulsion_fuel.setter
    def use_aerodynamic_propulsion_fuel(self, value:bool) -> None:
        """Get or set whether to use Aero/Propulsion fuel flow."""
        return self._intf.set_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._set_use_aerodynamic_propulsion_fuel_metadata, value)

    _get_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _get_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def scale_fuel_flow_by_non_std_density(self) -> bool:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_scale_fuel_flow_by_non_std_density_metadata)

    _set_scale_fuel_flow_by_non_std_density_metadata = { "offset" : _set_scale_fuel_flow_by_non_std_density_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @scale_fuel_flow_by_non_std_density.setter
    def scale_fuel_flow_by_non_std_density(self, value:bool) -> None:
        """Get or set whether to scale fuel flow by non std density."""
        return self._intf.set_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._set_scale_fuel_flow_by_non_std_density_metadata, value)

    _get_fuel_flow_metadata = { "offset" : _get_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_flow(self) -> float:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.get_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_fuel_flow_metadata)

    _set_fuel_flow_metadata = { "offset" : _set_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_flow.setter
    def fuel_flow(self, value:float) -> None:
        """Get or set the Sea Level Fuel Flow."""
        return self._intf.set_property(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._set_fuel_flow_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftBasicLandingModel._metadata, AircraftBasicLandingModel._get_as_catalog_item_metadata, OutArg())

    _property_names[landing_speed] = "landing_speed"
    _property_names[landing_speed_type] = "landing_speed_type"
    _property_names[sea_level_ground_roll] = "sea_level_ground_roll"
    _property_names[use_aerodynamic_propulsion_fuel] = "use_aerodynamic_propulsion_fuel"
    _property_names[scale_fuel_flow_by_non_std_density] = "scale_fuel_flow_by_non_std_density"
    _property_names[fuel_flow] = "fuel_flow"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicLandingModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicLandingModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicLandingModel, [AircraftBasicLandingModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4668682694342627584, 2103924401706785409), AircraftBasicLandingModel)
agcls.AgTypeNameMap["AircraftBasicLandingModel"] = AircraftBasicLandingModel

class AircraftAdvancedLandingModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the advanced landing model options for a landing model of an aircraft in the Aviator catalog."""

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_landing_speed_mode_method_offset = 1
    _set_landing_speed_mode_method_offset = 2
    _get_stall_speed_ratio_method_offset = 3
    _set_stall_speed_ratio_method_offset = 4
    _get_angle_of_attack_method_offset = 5
    _set_angle_of_attack_method_offset = 6
    _get_flaps_method_offset = 7
    _set_flaps_method_offset = 8
    _get_speedbrakes_method_offset = 9
    _set_speedbrakes_method_offset = 10
    _get_braking_deceleration_g_method_offset = 11
    _set_braking_deceleration_g_method_offset = 12
    _get_as_catalog_item_method_offset = 13
    _metadata = {
        "iid_data" : (5277953693984597077, 10394557647785555631),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedLandingModel)

    _get_landing_speed_mode_metadata = { "offset" : _get_landing_speed_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TakeoffLandingSpeedMode),) }
    @property
    def landing_speed_mode(self) -> "TakeoffLandingSpeedMode":
        """Get or set the mode to calculate the aircraft's speed at wheels down."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_landing_speed_mode_metadata)

    _set_landing_speed_mode_metadata = { "offset" : _set_landing_speed_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TakeoffLandingSpeedMode),) }
    @landing_speed_mode.setter
    def landing_speed_mode(self, value:"TakeoffLandingSpeedMode") -> None:
        """Get or set the mode to calculate the aircraft's speed at wheels down."""
        return self._intf.set_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_landing_speed_mode_metadata, value)

    _get_stall_speed_ratio_metadata = { "offset" : _get_stall_speed_ratio_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stall_speed_ratio(self) -> float:
        """Get the ratio of the airspeed to the stall speed at wheels down."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_stall_speed_ratio_metadata)

    _set_stall_speed_ratio_metadata = { "offset" : _set_stall_speed_ratio_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    def set_stall_speed_ratio(self, stall_speed_ratio:float) -> None:
        """Set the landing speed mode to StallSpeedRatio and specify the stall speed ratio."""
        return self._intf.invoke(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_stall_speed_ratio_metadata, stall_speed_ratio)

    _get_angle_of_attack_metadata = { "offset" : _get_angle_of_attack_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def angle_of_attack(self) -> typing.Any:
        """Get the Angle of Attack at wheels down."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_angle_of_attack_metadata)

    _set_angle_of_attack_metadata = { "offset" : _set_angle_of_attack_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_angle_of_attack(self, angle_of_attack:typing.Any) -> None:
        """Set the landing speed mode to AngleOfAttack and specify the angle of attack."""
        return self._intf.invoke(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_angle_of_attack_metadata, angle_of_attack)

    _get_flaps_metadata = { "offset" : _get_flaps_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def flaps(self) -> float:
        """Get or set the extension of the flaps during the landing."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_flaps_metadata)

    _set_flaps_metadata = { "offset" : _set_flaps_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @flaps.setter
    def flaps(self, value:float) -> None:
        """Get or set the extension of the flaps during the landing."""
        return self._intf.set_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_flaps_metadata, value)

    _get_speedbrakes_metadata = { "offset" : _get_speedbrakes_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def speedbrakes(self) -> float:
        """Get or set the extension of the speedbrakes during the landing."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_speedbrakes_metadata)

    _set_speedbrakes_metadata = { "offset" : _set_speedbrakes_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @speedbrakes.setter
    def speedbrakes(self, value:float) -> None:
        """Get or set the extension of the speedbrakes during the landing."""
        return self._intf.set_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_speedbrakes_metadata, value)

    _get_braking_deceleration_g_metadata = { "offset" : _get_braking_deceleration_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def braking_deceleration_g(self) -> float:
        """Get or set the deceleration rate, in G, when braking."""
        return self._intf.get_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_braking_deceleration_g_metadata)

    _set_braking_deceleration_g_metadata = { "offset" : _set_braking_deceleration_g_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @braking_deceleration_g.setter
    def braking_deceleration_g(self, value:float) -> None:
        """Get or set the deceleration rate, in G, when braking."""
        return self._intf.set_property(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._set_braking_deceleration_g_metadata, value)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedLandingModel._metadata, AircraftAdvancedLandingModel._get_as_catalog_item_metadata, OutArg())

    _property_names[landing_speed_mode] = "landing_speed_mode"
    _property_names[stall_speed_ratio] = "stall_speed_ratio"
    _property_names[angle_of_attack] = "angle_of_attack"
    _property_names[flaps] = "flaps"
    _property_names[speedbrakes] = "speedbrakes"
    _property_names[braking_deceleration_g] = "braking_deceleration_g"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedLandingModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedLandingModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedLandingModel, [AircraftAdvancedLandingModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4698133591076713687, 3121932985451287191), AircraftAdvancedLandingModel)
agcls.AgTypeNameMap["AircraftAdvancedLandingModel"] = AircraftAdvancedLandingModel

class AirportCategory(SupportsDeleteCallback):
    """Interface used to access the airports in the Aviator catalog."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_arinc424_airports_method_offset = 1
    _metadata = {
        "iid_data" : (4716319275932126710, 2976160558904714927),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AirportCategory)

    _get_arinc424_airports_metadata = { "offset" : _get_arinc424_airports_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arinc424_airports(self) -> "ARINC424Source":
        """Get the ARINC-424 airports."""
        return self._intf.get_property(AirportCategory._metadata, AirportCategory._get_arinc424_airports_metadata)

    _property_names[arinc424_airports] = "arinc424_airports"

    def __init__(self, source_object=None):
        """Construct an object of type AirportCategory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AirportCategory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AirportCategory, [AirportCategory, ])

agcls.AgClassCatalog.add_catalog_entry((4948677995080953579, 11068242944747772575), AirportCategory)
agcls.AgTypeNameMap["AirportCategory"] = AirportCategory

class ARINC424Airport(IARINC424Item, ICatalogAirport, ICatalogWaypoint, ICatalogItem, SupportsDeleteCallback):
    """Class defining an ARINC424 Airport."""
    def __init__(self, source_object=None):
        """Construct an object of type ARINC424Airport."""
        SupportsDeleteCallback.__init__(self)
        IARINC424Item.__init__(self, source_object)
        ICatalogAirport.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IARINC424Item._private_init(self, intf)
        ICatalogAirport._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ARINC424Airport, [IARINC424Item, ICatalogAirport, ICatalogWaypoint, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5153344260385549656, 2532207200311691662), ARINC424Airport)
agcls.AgTypeNameMap["ARINC424Airport"] = ARINC424Airport

class ARINC424Runway(IARINC424Item, ICatalogRunway, ICatalogWaypoint, ICatalogItem, SupportsDeleteCallback):
    """Do not use this interface, as it is deprecated. Use IARINC424Item instead."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_catalog_item_method_offset = 1
    _metadata = {
        "iid_data" : (5300123620389747771, 18053247007351242896),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ARINC424Runway)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(ARINC424Runway._metadata, ARINC424Runway._get_as_catalog_item_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type ARINC424Runway."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ARINC424Runway)
        IARINC424Item.__init__(self, source_object)
        ICatalogRunway.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IARINC424Item._private_init(self, intf)
        ICatalogRunway._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ARINC424Runway, [ARINC424Runway, IARINC424Item, ICatalogRunway, ICatalogWaypoint, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4920080154187198702, 7391099673982507661), ARINC424Runway)
agcls.AgTypeNameMap["ARINC424Runway"] = ARINC424Runway

class DAFIFRunway(IDAFIFItem, ICatalogRunway, ICatalogWaypoint, ICatalogItem, SupportsDeleteCallback):
    """Class defining an DAFIF Runway."""
    def __init__(self, source_object=None):
        """Construct an object of type DAFIFRunway."""
        SupportsDeleteCallback.__init__(self)
        IDAFIFItem.__init__(self, source_object)
        ICatalogRunway.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDAFIFItem._private_init(self, intf)
        ICatalogRunway._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DAFIFRunway, [IDAFIFItem, ICatalogRunway, ICatalogWaypoint, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4742072681057648609, 7698952542910555306), DAFIFRunway)
agcls.AgTypeNameMap["DAFIFRunway"] = DAFIFRunway

class DAFIFHelipad(IDAFIFItem, ICatalogVTOLPoint, ICatalogWaypoint, ICatalogItem, SupportsDeleteCallback):
    """Class defining an DAFIF Helipad."""
    def __init__(self, source_object=None):
        """Construct an object of type DAFIFHelipad."""
        SupportsDeleteCallback.__init__(self)
        IDAFIFItem.__init__(self, source_object)
        ICatalogVTOLPoint.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDAFIFItem._private_init(self, intf)
        ICatalogVTOLPoint._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DAFIFHelipad, [IDAFIFItem, ICatalogVTOLPoint, ICatalogWaypoint, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5242889605316044604, 11897854504103653769), DAFIFHelipad)
agcls.AgTypeNameMap["DAFIFHelipad"] = DAFIFHelipad

class DAFIFWaypoint(IDAFIFItem, ICatalogWaypoint, ICatalogItem, SupportsDeleteCallback):
    """Class defining an DAFIF Waypoint."""
    def __init__(self, source_object=None):
        """Construct an object of type DAFIFWaypoint."""
        SupportsDeleteCallback.__init__(self)
        IDAFIFItem.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDAFIFItem._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DAFIFWaypoint, [IDAFIFItem, ICatalogWaypoint, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((4681954691512884825, 10308348976302292395), DAFIFWaypoint)
agcls.AgTypeNameMap["DAFIFWaypoint"] = DAFIFWaypoint

class RunwayCategory(SupportsDeleteCallback):
    """Interface used to access runways in the Aviator catalog."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_user_runways_method_offset = 1
    _get_arinc424_runways_method_offset = 2
    _get_dafif_runways_method_offset = 3
    _metadata = {
        "iid_data" : (5167318891465223627, 14287069589925218227),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RunwayCategory)

    _get_user_runways_metadata = { "offset" : _get_user_runways_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_runways(self) -> "UserRunwaySource":
        """Get the user runways."""
        return self._intf.get_property(RunwayCategory._metadata, RunwayCategory._get_user_runways_metadata)

    _get_arinc424_runways_metadata = { "offset" : _get_arinc424_runways_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def arinc424_runways(self) -> "ARINC424Source":
        """Get the ARINC-424 runways."""
        return self._intf.get_property(RunwayCategory._metadata, RunwayCategory._get_arinc424_runways_metadata)

    _get_dafif_runways_metadata = { "offset" : _get_dafif_runways_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def dafif_runways(self) -> "DAFIFSource":
        """Get the DAFIF runways."""
        return self._intf.get_property(RunwayCategory._metadata, RunwayCategory._get_dafif_runways_metadata)

    _property_names[user_runways] = "user_runways"
    _property_names[arinc424_runways] = "arinc424_runways"
    _property_names[dafif_runways] = "dafif_runways"

    def __init__(self, source_object=None):
        """Construct an object of type RunwayCategory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RunwayCategory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RunwayCategory, [RunwayCategory, ])

agcls.AgClassCatalog.add_catalog_entry((5469050913701344868, 8227015865766767039), RunwayCategory)
agcls.AgTypeNameMap["RunwayCategory"] = RunwayCategory

class UserRunwaySource(ICatalogSource, SupportsDeleteCallback):
    """
    Interface used to access the user runways in the Aviator catalog.

    Examples
    --------
    Configure a runway site from a runway in the Aviator catalog:
    >>> # SiteRunway runway: Runway object
    >>> # Catalog catalog: Aviator catalog object
    >>> # Get the source of user runways
    >>> userRunways = catalog.runway_category.user_runways
    >>> # Check that the runway exists in the catalog
    >>> if userRunways.contains("New User Runway") is True:
    >>>     # If so, get the user runway with the given name
    >>>     runwayFromCatalog = userRunways.get_user_runway("New User Runway")
    >>>     # Copy the parameters of that runway
    >>>     runway.copy_from_catalog(runwayFromCatalog)
    """

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_user_runway_method_offset = 1
    _add_user_runway_method_offset = 2
    _get_as_catalog_source_method_offset = 3
    _metadata = {
        "iid_data" : (4676451489891113138, 8977042287783605675),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, UserRunwaySource)

    _get_user_runway_metadata = { "offset" : _get_user_runway_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_user_runway(self, name:str) -> "UserRunway":
        """Get the user runway with the given name."""
        return self._intf.invoke(UserRunwaySource._metadata, UserRunwaySource._get_user_runway_metadata, name, OutArg())

    _add_user_runway_metadata = { "offset" : _add_user_runway_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add_user_runway(self, name:str) -> "UserRunway":
        """Create a new user runway with the given name."""
        return self._intf.invoke(UserRunwaySource._metadata, UserRunwaySource._add_user_runway_metadata, name, OutArg())

    _get_as_catalog_source_metadata = { "offset" : _get_as_catalog_source_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_source(self) -> "ICatalogSource":
        """Get the catalog source interface for this object."""
        return self._intf.invoke(UserRunwaySource._metadata, UserRunwaySource._get_as_catalog_source_metadata, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type UserRunwaySource."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, UserRunwaySource)
        ICatalogSource.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogSource._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, UserRunwaySource, [UserRunwaySource, ICatalogSource])

agcls.AgClassCatalog.add_catalog_entry((5635637392444233572, 17245173151766416020), UserRunwaySource)
agcls.AgTypeNameMap["UserRunwaySource"] = UserRunwaySource

class UserRunway(ICatalogRunway, ICatalogItem, ICatalogWaypoint, SupportsDeleteCallback):
    """Interface used to access a user runway in the Aviator catalog."""

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_as_catalog_item_method_offset = 1
    _get_altitude_method_offset = 2
    _set_altitude_method_offset = 3
    _get_terrain_altitude_method_offset = 4
    _get_latitude_method_offset = 5
    _set_latitude_method_offset = 6
    _get_longitude_method_offset = 7
    _set_longitude_method_offset = 8
    _get_length_method_offset = 9
    _set_length_method_offset = 10
    _get_low_end_heading_method_offset = 11
    _set_low_end_heading_method_offset = 12
    _get_high_end_heading_method_offset = 13
    _set_high_end_heading_method_offset = 14
    _get_is_magnetic_method_offset = 15
    _set_is_magnetic_method_offset = 16
    _copy_site_method_offset = 17
    _paste_site_method_offset = 18
    _metadata = {
        "iid_data" : (4820628736157176607, 17908593095754821544),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, UserRunway)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(UserRunway._metadata, UserRunway._get_as_catalog_item_metadata, OutArg())

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the runway altitude."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Get or set the runway altitude."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_altitude_metadata, value)

    _get_terrain_altitude_metadata = { "offset" : _get_terrain_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    def get_terrain_altitude(self) -> float:
        """Set the runway altitude to the terrain altitude."""
        return self._intf.invoke(UserRunway._metadata, UserRunway._get_terrain_altitude_metadata, OutArg())

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the runway latitude."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        """Get or set the runway latitude."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the runway longitude."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        """Get or set the runway longitude."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_longitude_metadata, value)

    _get_length_metadata = { "offset" : _get_length_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def length(self) -> float:
        """Get or set the length of the runway."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_length_metadata)

    _set_length_metadata = { "offset" : _set_length_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @length.setter
    def length(self, value:float) -> None:
        """Get or set the length of the runway."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_length_metadata, value)

    _get_low_end_heading_metadata = { "offset" : _get_low_end_heading_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def low_end_heading(self) -> typing.Any:
        """Get or set the low end heading of the runway."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_low_end_heading_metadata)

    _set_low_end_heading_metadata = { "offset" : _set_low_end_heading_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @low_end_heading.setter
    def low_end_heading(self, value:typing.Any) -> None:
        """Get or set the low end heading of the runway."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_low_end_heading_metadata, value)

    _get_high_end_heading_metadata = { "offset" : _get_high_end_heading_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def high_end_heading(self) -> typing.Any:
        """Get or set the high end heading of the runway."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_high_end_heading_metadata)

    _set_high_end_heading_metadata = { "offset" : _set_high_end_heading_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @high_end_heading.setter
    def high_end_heading(self, value:typing.Any) -> None:
        """Get or set the high end heading of the runway."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_high_end_heading_metadata, value)

    _get_is_magnetic_metadata = { "offset" : _get_is_magnetic_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_magnetic(self) -> bool:
        """Opt whether to use a magnetic heading for the runway heading."""
        return self._intf.get_property(UserRunway._metadata, UserRunway._get_is_magnetic_metadata)

    _set_is_magnetic_metadata = { "offset" : _set_is_magnetic_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @is_magnetic.setter
    def is_magnetic(self, value:bool) -> None:
        """Opt whether to use a magnetic heading for the runway heading."""
        return self._intf.set_property(UserRunway._metadata, UserRunway._set_is_magnetic_metadata, value)

    _copy_site_metadata = { "offset" : _copy_site_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_site(self) -> None:
        """Copy the site to the clipboard."""
        return self._intf.invoke(UserRunway._metadata, UserRunway._copy_site_metadata, )

    _paste_site_metadata = { "offset" : _paste_site_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def paste_site(self) -> None:
        """Paste the site to the clipboard."""
        return self._intf.invoke(UserRunway._metadata, UserRunway._paste_site_metadata, )

    _property_names[altitude] = "altitude"
    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[length] = "length"
    _property_names[low_end_heading] = "low_end_heading"
    _property_names[high_end_heading] = "high_end_heading"
    _property_names[is_magnetic] = "is_magnetic"

    def __init__(self, source_object=None):
        """Construct an object of type UserRunway."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, UserRunway)
        ICatalogRunway.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
        ICatalogWaypoint.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogRunway._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
        ICatalogWaypoint._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, UserRunway, [UserRunway, ICatalogRunway, ICatalogItem, ICatalogWaypoint])

agcls.AgClassCatalog.add_catalog_entry((5691789674305805005, 304905329320076219), UserRunway)
agcls.AgTypeNameMap["UserRunway"] = UserRunway

class AltitudeMSLOptions(SupportsDeleteCallback):
    """Interface used to access the altitude MSL options for an Aviator procedure."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_default_cruise_altitude_method_offset = 1
    _set_use_default_cruise_altitude_method_offset = 2
    _get_msl_altitude_method_offset = 3
    _set_msl_altitude_method_offset = 4
    _metadata = {
        "iid_data" : (5354678070619953400, 16650566902719874200),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AltitudeMSLOptions)

    _get_use_default_cruise_altitude_metadata = { "offset" : _get_use_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_default_cruise_altitude(self) -> bool:
        """Opt whether to use the default cruise altitude."""
        return self._intf.get_property(AltitudeMSLOptions._metadata, AltitudeMSLOptions._get_use_default_cruise_altitude_metadata)

    _set_use_default_cruise_altitude_metadata = { "offset" : _set_use_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_default_cruise_altitude.setter
    def use_default_cruise_altitude(self, value:bool) -> None:
        """Opt whether to use the default cruise altitude."""
        return self._intf.set_property(AltitudeMSLOptions._metadata, AltitudeMSLOptions._set_use_default_cruise_altitude_metadata, value)

    _get_msl_altitude_metadata = { "offset" : _get_msl_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def msl_altitude(self) -> float:
        """Get or set the MSL altitude. Can only be used when the option to use the default cruise altitude is off."""
        return self._intf.get_property(AltitudeMSLOptions._metadata, AltitudeMSLOptions._get_msl_altitude_metadata)

    _set_msl_altitude_metadata = { "offset" : _set_msl_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @msl_altitude.setter
    def msl_altitude(self, value:float) -> None:
        """Get or set the MSL altitude. Can only be used when the option to use the default cruise altitude is off."""
        return self._intf.set_property(AltitudeMSLOptions._metadata, AltitudeMSLOptions._set_msl_altitude_metadata, value)

    _property_names[use_default_cruise_altitude] = "use_default_cruise_altitude"
    _property_names[msl_altitude] = "msl_altitude"

    def __init__(self, source_object=None):
        """Construct an object of type AltitudeMSLOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AltitudeMSLOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AltitudeMSLOptions, [AltitudeMSLOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4711346236767512285, 12895539298887267007), AltitudeMSLOptions)
agcls.AgTypeNameMap["AltitudeMSLOptions"] = AltitudeMSLOptions

class AltitudeOptions(SupportsDeleteCallback):
    """Interface used to access the altitude options for an Aviator procedure."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_default_cruise_altitude_method_offset = 1
    _set_use_default_cruise_altitude_method_offset = 2
    _get_altitude_reference_method_offset = 3
    _set_altitude_reference_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _metadata = {
        "iid_data" : (4916664038627200200, 6937122557711862173),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AltitudeOptions)

    _get_use_default_cruise_altitude_metadata = { "offset" : _get_use_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_default_cruise_altitude(self) -> bool:
        """Opt whether to use the default cruise altitude."""
        return self._intf.get_property(AltitudeOptions._metadata, AltitudeOptions._get_use_default_cruise_altitude_metadata)

    _set_use_default_cruise_altitude_metadata = { "offset" : _set_use_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_default_cruise_altitude.setter
    def use_default_cruise_altitude(self, value:bool) -> None:
        """Opt whether to use the default cruise altitude."""
        return self._intf.set_property(AltitudeOptions._metadata, AltitudeOptions._set_use_default_cruise_altitude_metadata, value)

    _get_altitude_reference_metadata = { "offset" : _get_altitude_reference_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @property
    def altitude_reference(self) -> "AGLMSL":
        """Get or set the altitude reference."""
        return self._intf.get_property(AltitudeOptions._metadata, AltitudeOptions._get_altitude_reference_metadata)

    _set_altitude_reference_metadata = { "offset" : _set_altitude_reference_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AGLMSL),) }
    @altitude_reference.setter
    def altitude_reference(self, value:"AGLMSL") -> None:
        """Get or set the altitude reference."""
        return self._intf.set_property(AltitudeOptions._metadata, AltitudeOptions._set_altitude_reference_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the altitude for the procedure."""
        return self._intf.get_property(AltitudeOptions._metadata, AltitudeOptions._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        """Get or set the altitude for the procedure."""
        return self._intf.set_property(AltitudeOptions._metadata, AltitudeOptions._set_altitude_metadata, value)

    _property_names[use_default_cruise_altitude] = "use_default_cruise_altitude"
    _property_names[altitude_reference] = "altitude_reference"
    _property_names[altitude] = "altitude"

    def __init__(self, source_object=None):
        """Construct an object of type AltitudeOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AltitudeOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AltitudeOptions, [AltitudeOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5605824035098034871, 2592958110125322650), AltitudeOptions)
agcls.AgTypeNameMap["AltitudeOptions"] = AltitudeOptions

class ArcAltitudeOptions(SupportsDeleteCallback):
    """Interface used to access the altitude options for an Arc procedure."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_default_cruise_altitude_method_offset = 1
    _set_use_default_cruise_altitude_method_offset = 2
    _get_start_arc_altitude_method_offset = 3
    _set_start_arc_altitude_method_offset = 4
    _get_stop_arc_altitude_method_offset = 5
    _set_stop_arc_altitude_method_offset = 6
    _metadata = {
        "iid_data" : (5311428139126596917, 7870750672865060761),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ArcAltitudeOptions)

    _get_use_default_cruise_altitude_metadata = { "offset" : _get_use_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_default_cruise_altitude(self) -> bool:
        """Opt whether to use the default cruise altitude."""
        return self._intf.get_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._get_use_default_cruise_altitude_metadata)

    _set_use_default_cruise_altitude_metadata = { "offset" : _set_use_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_default_cruise_altitude.setter
    def use_default_cruise_altitude(self, value:bool) -> None:
        """Opt whether to use the default cruise altitude."""
        return self._intf.set_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._set_use_default_cruise_altitude_metadata, value)

    _get_start_arc_altitude_metadata = { "offset" : _get_start_arc_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_arc_altitude(self) -> float:
        """Get or set the altitude at the beginning of the arc."""
        return self._intf.get_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._get_start_arc_altitude_metadata)

    _set_start_arc_altitude_metadata = { "offset" : _set_start_arc_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_arc_altitude.setter
    def start_arc_altitude(self, value:float) -> None:
        """Get or set the altitude at the beginning of the arc."""
        return self._intf.set_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._set_start_arc_altitude_metadata, value)

    _get_stop_arc_altitude_metadata = { "offset" : _get_stop_arc_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_arc_altitude(self) -> float:
        """Get or set the altitude at the end of the arc."""
        return self._intf.get_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._get_stop_arc_altitude_metadata)

    _set_stop_arc_altitude_metadata = { "offset" : _set_stop_arc_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_arc_altitude.setter
    def stop_arc_altitude(self, value:float) -> None:
        """Get or set the altitude at the end of the arc."""
        return self._intf.set_property(ArcAltitudeOptions._metadata, ArcAltitudeOptions._set_stop_arc_altitude_metadata, value)

    _property_names[use_default_cruise_altitude] = "use_default_cruise_altitude"
    _property_names[start_arc_altitude] = "start_arc_altitude"
    _property_names[stop_arc_altitude] = "stop_arc_altitude"

    def __init__(self, source_object=None):
        """Construct an object of type ArcAltitudeOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ArcAltitudeOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ArcAltitudeOptions, [ArcAltitudeOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5643814476002063505, 18340846669876596410), ArcAltitudeOptions)
agcls.AgTypeNameMap["ArcAltitudeOptions"] = ArcAltitudeOptions

class ArcAltitudeAndDelayOptions(SupportsDeleteCallback):
    """Interface used to access the altitude options for an Arc procedure."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_default_cruise_altitude_method_offset = 1
    _set_use_default_cruise_altitude_method_offset = 2
    _get_delay_arc_climb_descents_method_offset = 3
    _set_delay_arc_climb_descents_method_offset = 4
    _get_start_arc_altitude_method_offset = 5
    _set_start_arc_altitude_method_offset = 6
    _get_stop_arc_altitude_method_offset = 7
    _set_stop_arc_altitude_method_offset = 8
    _metadata = {
        "iid_data" : (5091741236913358761, 4527708200198965125),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ArcAltitudeAndDelayOptions)

    _get_use_default_cruise_altitude_metadata = { "offset" : _get_use_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_default_cruise_altitude(self) -> bool:
        """Opt whether to use the default cruise altitude."""
        return self._intf.get_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._get_use_default_cruise_altitude_metadata)

    _set_use_default_cruise_altitude_metadata = { "offset" : _set_use_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_default_cruise_altitude.setter
    def use_default_cruise_altitude(self, value:bool) -> None:
        """Opt whether to use the default cruise altitude."""
        return self._intf.set_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._set_use_default_cruise_altitude_metadata, value)

    _get_delay_arc_climb_descents_metadata = { "offset" : _get_delay_arc_climb_descents_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def delay_arc_climb_descents(self) -> bool:
        """Delay the climb/descend such that the stop arc altitude will be achieved by the end of the procedure."""
        return self._intf.get_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._get_delay_arc_climb_descents_metadata)

    _set_delay_arc_climb_descents_metadata = { "offset" : _set_delay_arc_climb_descents_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @delay_arc_climb_descents.setter
    def delay_arc_climb_descents(self, value:bool) -> None:
        """Delay the climb/descend such that the stop arc altitude will be achieved by the end of the procedure."""
        return self._intf.set_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._set_delay_arc_climb_descents_metadata, value)

    _get_start_arc_altitude_metadata = { "offset" : _get_start_arc_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_arc_altitude(self) -> float:
        """Get or set the altitude at the beginning of the arc."""
        return self._intf.get_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._get_start_arc_altitude_metadata)

    _set_start_arc_altitude_metadata = { "offset" : _set_start_arc_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_arc_altitude.setter
    def start_arc_altitude(self, value:float) -> None:
        """Get or set the altitude at the beginning of the arc."""
        return self._intf.set_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._set_start_arc_altitude_metadata, value)

    _get_stop_arc_altitude_metadata = { "offset" : _get_stop_arc_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_arc_altitude(self) -> float:
        """Get or set the altitude at the end of the arc."""
        return self._intf.get_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._get_stop_arc_altitude_metadata)

    _set_stop_arc_altitude_metadata = { "offset" : _set_stop_arc_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_arc_altitude.setter
    def stop_arc_altitude(self, value:float) -> None:
        """Get or set the altitude at the end of the arc."""
        return self._intf.set_property(ArcAltitudeAndDelayOptions._metadata, ArcAltitudeAndDelayOptions._set_stop_arc_altitude_metadata, value)

    _property_names[use_default_cruise_altitude] = "use_default_cruise_altitude"
    _property_names[delay_arc_climb_descents] = "delay_arc_climb_descents"
    _property_names[start_arc_altitude] = "start_arc_altitude"
    _property_names[stop_arc_altitude] = "stop_arc_altitude"

    def __init__(self, source_object=None):
        """Construct an object of type ArcAltitudeAndDelayOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ArcAltitudeAndDelayOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ArcAltitudeAndDelayOptions, [ArcAltitudeAndDelayOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5626426987589423015, 10342354197599988640), ArcAltitudeAndDelayOptions)
agcls.AgTypeNameMap["ArcAltitudeAndDelayOptions"] = ArcAltitudeAndDelayOptions

class ArcOptions(SupportsDeleteCallback):
    """Interface used to access the arc options for a procedure."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_turn_direction_method_offset = 1
    _set_turn_direction_method_offset = 2
    _get_start_bearing_method_offset = 3
    _set_start_bearing_method_offset = 4
    _get_use_magnetic_heading_method_offset = 5
    _set_use_magnetic_heading_method_offset = 6
    _get_radius_method_offset = 7
    _set_radius_method_offset = 8
    _get_turn_angle_method_offset = 9
    _set_turn_angle_method_offset = 10
    _get_join_arc_method_offset = 11
    _set_join_arc_method_offset = 12
    _get_exit_arc_method_offset = 13
    _set_exit_arc_method_offset = 14
    _metadata = {
        "iid_data" : (4881719014414109863, 13875083150837249449),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ArcOptions)

    _get_turn_direction_metadata = { "offset" : _get_turn_direction_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TurnDirection),) }
    @property
    def turn_direction(self) -> "TurnDirection":
        """Get or set the turn direction to the arc."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_turn_direction_metadata)

    _set_turn_direction_metadata = { "offset" : _set_turn_direction_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TurnDirection),) }
    @turn_direction.setter
    def turn_direction(self, value:"TurnDirection") -> None:
        """Get or set the turn direction to the arc."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_turn_direction_metadata, value)

    _get_start_bearing_metadata = { "offset" : _get_start_bearing_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def start_bearing(self) -> typing.Any:
        """Get or set the bearing from the site to the arc."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_start_bearing_metadata)

    _set_start_bearing_metadata = { "offset" : _set_start_bearing_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @start_bearing.setter
    def start_bearing(self, value:typing.Any) -> None:
        """Get or set the bearing from the site to the arc."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_start_bearing_metadata, value)

    _get_use_magnetic_heading_metadata = { "offset" : _get_use_magnetic_heading_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_magnetic_heading(self) -> bool:
        """Get or set the option to use a magnetic heading for the start bearing."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_use_magnetic_heading_metadata)

    _set_use_magnetic_heading_metadata = { "offset" : _set_use_magnetic_heading_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_magnetic_heading.setter
    def use_magnetic_heading(self, value:bool) -> None:
        """Get or set the option to use a magnetic heading for the start bearing."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_use_magnetic_heading_metadata, value)

    _get_radius_metadata = { "offset" : _get_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius(self) -> float:
        """Get or set the radius of the arc."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_radius_metadata)

    _set_radius_metadata = { "offset" : _set_radius_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius.setter
    def radius(self, value:float) -> None:
        """Get or set the radius of the arc."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_radius_metadata, value)

    _get_turn_angle_metadata = { "offset" : _get_turn_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def turn_angle(self) -> typing.Any:
        """Get or set the length of the arc the aircraft will fly."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_turn_angle_metadata)

    _set_turn_angle_metadata = { "offset" : _set_turn_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @turn_angle.setter
    def turn_angle(self, value:typing.Any) -> None:
        """Get or set the length of the arc the aircraft will fly."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_turn_angle_metadata, value)

    _get_join_arc_metadata = { "offset" : _get_join_arc_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(JoinExitArcMethod),) }
    @property
    def join_arc(self) -> "JoinExitArcMethod":
        """Get or set the method to join the arc."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_join_arc_metadata)

    _set_join_arc_metadata = { "offset" : _set_join_arc_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(JoinExitArcMethod),) }
    @join_arc.setter
    def join_arc(self, value:"JoinExitArcMethod") -> None:
        """Get or set the method to join the arc."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_join_arc_metadata, value)

    _get_exit_arc_metadata = { "offset" : _get_exit_arc_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(JoinExitArcMethod),) }
    @property
    def exit_arc(self) -> "JoinExitArcMethod":
        """Get or set the method to exit the arc."""
        return self._intf.get_property(ArcOptions._metadata, ArcOptions._get_exit_arc_metadata)

    _set_exit_arc_metadata = { "offset" : _set_exit_arc_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(JoinExitArcMethod),) }
    @exit_arc.setter
    def exit_arc(self, value:"JoinExitArcMethod") -> None:
        """Get or set the method to exit the arc."""
        return self._intf.set_property(ArcOptions._metadata, ArcOptions._set_exit_arc_metadata, value)

    _property_names[turn_direction] = "turn_direction"
    _property_names[start_bearing] = "start_bearing"
    _property_names[use_magnetic_heading] = "use_magnetic_heading"
    _property_names[radius] = "radius"
    _property_names[turn_angle] = "turn_angle"
    _property_names[join_arc] = "join_arc"
    _property_names[exit_arc] = "exit_arc"

    def __init__(self, source_object=None):
        """Construct an object of type ArcOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ArcOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ArcOptions, [ArcOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4985001686384732151, 3276579562627893893), ArcOptions)
agcls.AgTypeNameMap["ArcOptions"] = ArcOptions

class AltitudeMSLAndLevelOffOptions(SupportsDeleteCallback):
    """
    Interface used to access the altitude MSL and Level off options for an Aviator procedure.

    Examples
    --------
    Add and configure an en-route procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add an enroute procedure with a site type of End of Previous Procedure
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>> # Get the altitude options
    >>> altitudeOptions = enroute.altitude_msl_options
    >>> # To specify an altitude, turn off the option to use the default cruise altitude
    >>> altitudeOptions.use_default_cruise_altitude = False
    >>> # Set the altitude
    >>> altitudeOptions.msl_altitude = 10000
    >>>
    >>> # Get the navigation options
    >>> navigationOptions = enroute.navigation_options
    >>> # Set the route to arrive on a specified course
    >>> navigationOptions.navigation_mode = PointToPointMode.ARRIVE_ON_COURSE
    >>> # Set the course
    >>> navigationOptions.arrive_on_course = 30
    >>> # Use a magnetic heading
    >>> navigationOptions.use_magnetic_heading = True
    >>>
    >>> # Get the navigation options
    >>> airspeedOptions = enroute.enroute_cruise_airspeed_options
    >>> # Fly at max speed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.MAX_AIRSPEED
    >>> # To specify an airspeed to fly at, set the speed type to other airspeed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.OTHER_AIRSPEED
    >>> # Then set the airspeed and airspeed type
    >>> airspeedOptions.set_other_airspeed(AirspeedType.TAS, 200)
    """

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_default_cruise_altitude_method_offset = 1
    _set_use_default_cruise_altitude_method_offset = 2
    _get_msl_altitude_method_offset = 3
    _set_msl_altitude_method_offset = 4
    _get_must_level_off_method_offset = 5
    _set_must_level_off_method_offset = 6
    _get_level_off_mode_method_offset = 7
    _set_level_off_mode_method_offset = 8
    _metadata = {
        "iid_data" : (4970555218238608998, 11906085318817631130),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AltitudeMSLAndLevelOffOptions)

    _get_use_default_cruise_altitude_metadata = { "offset" : _get_use_default_cruise_altitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_default_cruise_altitude(self) -> bool:
        """Opt whether to use the default cruise altitude."""
        return self._intf.get_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._get_use_default_cruise_altitude_metadata)

    _set_use_default_cruise_altitude_metadata = { "offset" : _set_use_default_cruise_altitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_default_cruise_altitude.setter
    def use_default_cruise_altitude(self, value:bool) -> None:
        """Opt whether to use the default cruise altitude."""
        return self._intf.set_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._set_use_default_cruise_altitude_metadata, value)

    _get_msl_altitude_metadata = { "offset" : _get_msl_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def msl_altitude(self) -> float:
        """Get the MSL altitude. Can only be used when the option to use the default cruise altitude is off."""
        return self._intf.get_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._get_msl_altitude_metadata)

    _set_msl_altitude_metadata = { "offset" : _set_msl_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @msl_altitude.setter
    def msl_altitude(self, value:float) -> None:
        """Set the MSL altitude. Can only be used when the option to use the default cruise altitude is off."""
        return self._intf.set_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._set_msl_altitude_metadata, value)

    _get_must_level_off_metadata = { "offset" : _get_must_level_off_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def must_level_off(self) -> bool:
        """Opt whether the procedure must level off."""
        return self._intf.get_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._get_must_level_off_metadata)

    _set_must_level_off_metadata = { "offset" : _set_must_level_off_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @must_level_off.setter
    def must_level_off(self, value:bool) -> None:
        """Opt whether the procedure must level off."""
        return self._intf.set_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._set_must_level_off_metadata, value)

    _get_level_off_mode_metadata = { "offset" : _get_level_off_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @property
    def level_off_mode(self) -> "AltitudeConstraintManeuverMode":
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.get_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._get_level_off_mode_metadata)

    _set_level_off_mode_metadata = { "offset" : _set_level_off_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @level_off_mode.setter
    def level_off_mode(self, value:"AltitudeConstraintManeuverMode") -> None:
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.set_property(AltitudeMSLAndLevelOffOptions._metadata, AltitudeMSLAndLevelOffOptions._set_level_off_mode_metadata, value)

    _property_names[use_default_cruise_altitude] = "use_default_cruise_altitude"
    _property_names[msl_altitude] = "msl_altitude"
    _property_names[must_level_off] = "must_level_off"
    _property_names[level_off_mode] = "level_off_mode"

    def __init__(self, source_object=None):
        """Construct an object of type AltitudeMSLAndLevelOffOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AltitudeMSLAndLevelOffOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AltitudeMSLAndLevelOffOptions, [AltitudeMSLAndLevelOffOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4623922148496958771, 14914904139938911918), AltitudeMSLAndLevelOffOptions)
agcls.AgTypeNameMap["AltitudeMSLAndLevelOffOptions"] = AltitudeMSLAndLevelOffOptions

class CruiseAirspeedOptions(SupportsDeleteCallback):
    """
    Interface used to access the Cruise Airspeed options for an Aviator procedure.

    Examples
    --------
    Add and configure an en-route procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add an enroute procedure with a site type of End of Previous Procedure
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>> # Get the altitude options
    >>> altitudeOptions = enroute.altitude_msl_options
    >>> # To specify an altitude, turn off the option to use the default cruise altitude
    >>> altitudeOptions.use_default_cruise_altitude = False
    >>> # Set the altitude
    >>> altitudeOptions.msl_altitude = 10000
    >>>
    >>> # Get the navigation options
    >>> navigationOptions = enroute.navigation_options
    >>> # Set the route to arrive on a specified course
    >>> navigationOptions.navigation_mode = PointToPointMode.ARRIVE_ON_COURSE
    >>> # Set the course
    >>> navigationOptions.arrive_on_course = 30
    >>> # Use a magnetic heading
    >>> navigationOptions.use_magnetic_heading = True
    >>>
    >>> # Get the navigation options
    >>> airspeedOptions = enroute.enroute_cruise_airspeed_options
    >>> # Fly at max speed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.MAX_AIRSPEED
    >>> # To specify an airspeed to fly at, set the speed type to other airspeed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.OTHER_AIRSPEED
    >>> # Then set the airspeed and airspeed type
    >>> airspeedOptions.set_other_airspeed(AirspeedType.TAS, 200)
    """

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_cruise_speed_type_method_offset = 1
    _set_cruise_speed_type_method_offset = 2
    _get_other_airspeed_type_method_offset = 3
    _get_other_airspeed_method_offset = 4
    _set_other_airspeed_method_offset = 5
    _metadata = {
        "iid_data" : (4775037409602090834, 14321643648366655389),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CruiseAirspeedOptions)

    _get_cruise_speed_type_metadata = { "offset" : _get_cruise_speed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(CruiseSpeed),) }
    @property
    def cruise_speed_type(self) -> "CruiseSpeed":
        """Get or set the method for determining the aircraft's airspeed."""
        return self._intf.get_property(CruiseAirspeedOptions._metadata, CruiseAirspeedOptions._get_cruise_speed_type_metadata)

    _set_cruise_speed_type_metadata = { "offset" : _set_cruise_speed_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(CruiseSpeed),) }
    @cruise_speed_type.setter
    def cruise_speed_type(self, value:"CruiseSpeed") -> None:
        """Get or set the method for determining the aircraft's airspeed."""
        return self._intf.set_property(CruiseAirspeedOptions._metadata, CruiseAirspeedOptions._set_cruise_speed_type_metadata, value)

    _get_other_airspeed_type_metadata = { "offset" : _get_other_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def other_airspeed_type(self) -> "AirspeedType":
        """Get the airspeed type for the other airspeed option."""
        return self._intf.get_property(CruiseAirspeedOptions._metadata, CruiseAirspeedOptions._get_other_airspeed_type_metadata)

    _get_other_airspeed_metadata = { "offset" : _get_other_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def other_airspeed(self) -> float:
        """Get the airspeed for the other airspeed option."""
        return self._intf.get_property(CruiseAirspeedOptions._metadata, CruiseAirspeedOptions._get_other_airspeed_metadata)

    _set_other_airspeed_metadata = { "offset" : _set_other_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_other_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the cruise airspeed. This option is only enabled if the cruise speed type is set to other."""
        return self._intf.invoke(CruiseAirspeedOptions._metadata, CruiseAirspeedOptions._set_other_airspeed_metadata, airspeed_type, airspeed)

    _property_names[cruise_speed_type] = "cruise_speed_type"
    _property_names[other_airspeed_type] = "other_airspeed_type"
    _property_names[other_airspeed] = "other_airspeed"

    def __init__(self, source_object=None):
        """Construct an object of type CruiseAirspeedOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CruiseAirspeedOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CruiseAirspeedOptions, [CruiseAirspeedOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5589499567568918408, 8793492467480374660), CruiseAirspeedOptions)
agcls.AgTypeNameMap["CruiseAirspeedOptions"] = CruiseAirspeedOptions

class CruiseAirspeedProfile(SupportsDeleteCallback):
    """Interface used to access the Cruise Profile options for an Aviator procedure."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_fly_cruise_airspeed_profile_method_offset = 1
    _set_fly_cruise_airspeed_profile_method_offset = 2
    _metadata = {
        "iid_data" : (4932691083798754313, 2757480245686328988),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CruiseAirspeedProfile)

    _get_fly_cruise_airspeed_profile_metadata = { "offset" : _get_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def fly_cruise_airspeed_profile(self) -> bool:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.get_property(CruiseAirspeedProfile._metadata, CruiseAirspeedProfile._get_fly_cruise_airspeed_profile_metadata)

    _set_fly_cruise_airspeed_profile_metadata = { "offset" : _set_fly_cruise_airspeed_profile_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @fly_cruise_airspeed_profile.setter
    def fly_cruise_airspeed_profile(self, value:bool) -> None:
        """Opt whether the aircraft immediately adopts the selected cruise airspeed or gradually begins accelerating/decelerating in the previous procedure."""
        return self._intf.set_property(CruiseAirspeedProfile._metadata, CruiseAirspeedProfile._set_fly_cruise_airspeed_profile_metadata, value)

    _property_names[fly_cruise_airspeed_profile] = "fly_cruise_airspeed_profile"

    def __init__(self, source_object=None):
        """Construct an object of type CruiseAirspeedProfile."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CruiseAirspeedProfile)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CruiseAirspeedProfile, [CruiseAirspeedProfile, ])

agcls.AgClassCatalog.add_catalog_entry((5042125039608830422, 186775197327029686), CruiseAirspeedProfile)
agcls.AgTypeNameMap["CruiseAirspeedProfile"] = CruiseAirspeedProfile

class CruiseAirspeedAndProfileOptions(ICruiseAirspeedAndProfileOptions, SupportsDeleteCallback):
    """Class defining the cruise airspeed and profile options in a procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type CruiseAirspeedAndProfileOptions."""
        SupportsDeleteCallback.__init__(self)
        ICruiseAirspeedAndProfileOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICruiseAirspeedAndProfileOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CruiseAirspeedAndProfileOptions, [ICruiseAirspeedAndProfileOptions])

agcls.AgClassCatalog.add_catalog_entry((5023146659635499349, 10240031022027307191), CruiseAirspeedAndProfileOptions)
agcls.AgTypeNameMap["CruiseAirspeedAndProfileOptions"] = CruiseAirspeedAndProfileOptions

class LandingCruiseAirspeedAndProfileOptions(ICruiseAirspeedAndProfileOptions, SupportsDeleteCallback):
    """Class defining the cruise airspeed and profile options for a landing procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type LandingCruiseAirspeedAndProfileOptions."""
        SupportsDeleteCallback.__init__(self)
        ICruiseAirspeedAndProfileOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICruiseAirspeedAndProfileOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingCruiseAirspeedAndProfileOptions, [ICruiseAirspeedAndProfileOptions])

agcls.AgClassCatalog.add_catalog_entry((5282497060921620892, 7823813257265650837), LandingCruiseAirspeedAndProfileOptions)
agcls.AgTypeNameMap["LandingCruiseAirspeedAndProfileOptions"] = LandingCruiseAirspeedAndProfileOptions

class EnrouteOptions(SupportsDeleteCallback):
    """Interface used to access the Enroute options for an Aviator procedure."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_max_speed_turns_method_offset = 1
    _set_use_max_speed_turns_method_offset = 2
    _get_max_turn_radius_factor_method_offset = 3
    _set_max_turn_radius_factor_method_offset = 4
    _metadata = {
        "iid_data" : (5388097155578164474, 7061394601143800450),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, EnrouteOptions)

    _get_use_max_speed_turns_metadata = { "offset" : _get_use_max_speed_turns_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_max_speed_turns(self) -> bool:
        """Opt whether to use the max speed turns."""
        return self._intf.get_property(EnrouteOptions._metadata, EnrouteOptions._get_use_max_speed_turns_metadata)

    _set_use_max_speed_turns_metadata = { "offset" : _set_use_max_speed_turns_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_max_speed_turns.setter
    def use_max_speed_turns(self, value:bool) -> None:
        """Opt whether to use the max speed turns."""
        return self._intf.set_property(EnrouteOptions._metadata, EnrouteOptions._set_use_max_speed_turns_metadata, value)

    _get_max_turn_radius_factor_metadata = { "offset" : _get_max_turn_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_turn_radius_factor(self) -> float:
        """Get or set the maximum turn radius factor."""
        return self._intf.get_property(EnrouteOptions._metadata, EnrouteOptions._get_max_turn_radius_factor_metadata)

    _set_max_turn_radius_factor_metadata = { "offset" : _set_max_turn_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_turn_radius_factor.setter
    def max_turn_radius_factor(self, value:float) -> None:
        """Get or set the maximum turn radius factor."""
        return self._intf.set_property(EnrouteOptions._metadata, EnrouteOptions._set_max_turn_radius_factor_metadata, value)

    _property_names[use_max_speed_turns] = "use_max_speed_turns"
    _property_names[max_turn_radius_factor] = "max_turn_radius_factor"

    def __init__(self, source_object=None):
        """Construct an object of type EnrouteOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, EnrouteOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, EnrouteOptions, [EnrouteOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5601440232813659681, 2626155638734127273), EnrouteOptions)
agcls.AgTypeNameMap["EnrouteOptions"] = EnrouteOptions

class EnrouteAndDelayOptions(IEnrouteAndDelayOptions, SupportsDeleteCallback):
    """Class defining the enroute and delay options in a procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type EnrouteAndDelayOptions."""
        SupportsDeleteCallback.__init__(self)
        IEnrouteAndDelayOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IEnrouteAndDelayOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, EnrouteAndDelayOptions, [IEnrouteAndDelayOptions])

agcls.AgClassCatalog.add_catalog_entry((5633379882483759369, 15566420089287312029), EnrouteAndDelayOptions)
agcls.AgTypeNameMap["EnrouteAndDelayOptions"] = EnrouteAndDelayOptions

class LandingEnrouteOptions(IEnrouteAndDelayOptions, SupportsDeleteCallback):
    """Class defining the enroute options in a landing procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type LandingEnrouteOptions."""
        SupportsDeleteCallback.__init__(self)
        IEnrouteAndDelayOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IEnrouteAndDelayOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingEnrouteOptions, [IEnrouteAndDelayOptions])

agcls.AgClassCatalog.add_catalog_entry((5505692743636881819, 1797000207370880952), LandingEnrouteOptions)
agcls.AgTypeNameMap["LandingEnrouteOptions"] = LandingEnrouteOptions

class EnrouteTurnDirectionOptions(SupportsDeleteCallback):
    """Interface used to access the Enroute Turn Direction options for an Aviator procedure."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_enroute_first_turn_method_offset = 1
    _set_enroute_first_turn_method_offset = 2
    _get_enroute_second_turn_method_offset = 3
    _set_enroute_second_turn_method_offset = 4
    _metadata = {
        "iid_data" : (4825843845888691284, 5666336038605995175),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, EnrouteTurnDirectionOptions)

    _get_enroute_first_turn_metadata = { "offset" : _get_enroute_first_turn_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def enroute_first_turn(self) -> "NavigatorTurnDirection":
        """Option for the first turn."""
        return self._intf.get_property(EnrouteTurnDirectionOptions._metadata, EnrouteTurnDirectionOptions._get_enroute_first_turn_metadata)

    _set_enroute_first_turn_metadata = { "offset" : _set_enroute_first_turn_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @enroute_first_turn.setter
    def enroute_first_turn(self, value:"NavigatorTurnDirection") -> None:
        """Option for the first turn."""
        return self._intf.set_property(EnrouteTurnDirectionOptions._metadata, EnrouteTurnDirectionOptions._set_enroute_first_turn_metadata, value)

    _get_enroute_second_turn_metadata = { "offset" : _get_enroute_second_turn_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def enroute_second_turn(self) -> "NavigatorTurnDirection":
        """Option for the second turn."""
        return self._intf.get_property(EnrouteTurnDirectionOptions._metadata, EnrouteTurnDirectionOptions._get_enroute_second_turn_metadata)

    _set_enroute_second_turn_metadata = { "offset" : _set_enroute_second_turn_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @enroute_second_turn.setter
    def enroute_second_turn(self, value:"NavigatorTurnDirection") -> None:
        """Option for the second turn."""
        return self._intf.set_property(EnrouteTurnDirectionOptions._metadata, EnrouteTurnDirectionOptions._set_enroute_second_turn_metadata, value)

    _property_names[enroute_first_turn] = "enroute_first_turn"
    _property_names[enroute_second_turn] = "enroute_second_turn"

    def __init__(self, source_object=None):
        """Construct an object of type EnrouteTurnDirectionOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, EnrouteTurnDirectionOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, EnrouteTurnDirectionOptions, [EnrouteTurnDirectionOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5525734658029426163, 7829029184116166294), EnrouteTurnDirectionOptions)
agcls.AgTypeNameMap["EnrouteTurnDirectionOptions"] = EnrouteTurnDirectionOptions

class NavigationOptions(SupportsDeleteCallback):
    """
    Interface used to access the navigation options for an Aviator procedure.

    Examples
    --------
    Add and configure an en-route procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add an enroute procedure with a site type of End of Previous Procedure
    >>> enroute = procedures.add_at_index(1, SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_ENROUTE)
    >>> # Get the altitude options
    >>> altitudeOptions = enroute.altitude_msl_options
    >>> # To specify an altitude, turn off the option to use the default cruise altitude
    >>> altitudeOptions.use_default_cruise_altitude = False
    >>> # Set the altitude
    >>> altitudeOptions.msl_altitude = 10000
    >>>
    >>> # Get the navigation options
    >>> navigationOptions = enroute.navigation_options
    >>> # Set the route to arrive on a specified course
    >>> navigationOptions.navigation_mode = PointToPointMode.ARRIVE_ON_COURSE
    >>> # Set the course
    >>> navigationOptions.arrive_on_course = 30
    >>> # Use a magnetic heading
    >>> navigationOptions.use_magnetic_heading = True
    >>>
    >>> # Get the navigation options
    >>> airspeedOptions = enroute.enroute_cruise_airspeed_options
    >>> # Fly at max speed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.MAX_AIRSPEED
    >>> # To specify an airspeed to fly at, set the speed type to other airspeed
    >>> airspeedOptions.cruise_speed_type = CruiseSpeed.OTHER_AIRSPEED
    >>> # Then set the airspeed and airspeed type
    >>> airspeedOptions.set_other_airspeed(AirspeedType.TAS, 200)
    """

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_navigation_mode_method_offset = 1
    _set_navigation_mode_method_offset = 2
    _get_arrive_on_course_method_offset = 3
    _set_arrive_on_course_method_offset = 4
    _get_use_magnetic_heading_method_offset = 5
    _set_use_magnetic_heading_method_offset = 6
    _get_enroute_first_turn_method_offset = 7
    _set_enroute_first_turn_method_offset = 8
    _get_enroute_second_turn_method_offset = 9
    _set_enroute_second_turn_method_offset = 10
    _metadata = {
        "iid_data" : (4993099939815057319, 3478925526245202352),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, NavigationOptions)

    _get_navigation_mode_metadata = { "offset" : _get_navigation_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PointToPointMode),) }
    @property
    def navigation_mode(self) -> "PointToPointMode":
        """Get or set the navigation mode."""
        return self._intf.get_property(NavigationOptions._metadata, NavigationOptions._get_navigation_mode_metadata)

    _set_navigation_mode_metadata = { "offset" : _set_navigation_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PointToPointMode),) }
    @navigation_mode.setter
    def navigation_mode(self, value:"PointToPointMode") -> None:
        """Get or set the navigation mode."""
        return self._intf.set_property(NavigationOptions._metadata, NavigationOptions._set_navigation_mode_metadata, value)

    _get_arrive_on_course_metadata = { "offset" : _get_arrive_on_course_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def arrive_on_course(self) -> typing.Any:
        """Get or set the aircraft will start or arrive at the procedure site with the specified course. The nav mode must be set to Arrive on Course to set this value."""
        return self._intf.get_property(NavigationOptions._metadata, NavigationOptions._get_arrive_on_course_metadata)

    _set_arrive_on_course_metadata = { "offset" : _set_arrive_on_course_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @arrive_on_course.setter
    def arrive_on_course(self, value:typing.Any) -> None:
        """Get or set the aircraft will start or arrive at the procedure site with the specified course. The nav mode must be set to Arrive on Course to set this value."""
        return self._intf.set_property(NavigationOptions._metadata, NavigationOptions._set_arrive_on_course_metadata, value)

    _get_use_magnetic_heading_metadata = { "offset" : _get_use_magnetic_heading_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_magnetic_heading(self) -> bool:
        """Opt whether to use a magnetic heading to arrive on course. The nav mode must be set to Arrive on Course to set this value."""
        return self._intf.get_property(NavigationOptions._metadata, NavigationOptions._get_use_magnetic_heading_metadata)

    _set_use_magnetic_heading_metadata = { "offset" : _set_use_magnetic_heading_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_magnetic_heading.setter
    def use_magnetic_heading(self, value:bool) -> None:
        """Opt whether to use a magnetic heading to arrive on course. The nav mode must be set to Arrive on Course to set this value."""
        return self._intf.set_property(NavigationOptions._metadata, NavigationOptions._set_use_magnetic_heading_metadata, value)

    _get_enroute_first_turn_metadata = { "offset" : _get_enroute_first_turn_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def enroute_first_turn(self) -> "NavigatorTurnDirection":
        """Option for the first turn."""
        return self._intf.get_property(NavigationOptions._metadata, NavigationOptions._get_enroute_first_turn_metadata)

    _set_enroute_first_turn_metadata = { "offset" : _set_enroute_first_turn_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @enroute_first_turn.setter
    def enroute_first_turn(self, value:"NavigatorTurnDirection") -> None:
        """Option for the first turn."""
        return self._intf.set_property(NavigationOptions._metadata, NavigationOptions._set_enroute_first_turn_metadata, value)

    _get_enroute_second_turn_metadata = { "offset" : _get_enroute_second_turn_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def enroute_second_turn(self) -> "NavigatorTurnDirection":
        """Option for the second turn."""
        return self._intf.get_property(NavigationOptions._metadata, NavigationOptions._get_enroute_second_turn_metadata)

    _set_enroute_second_turn_metadata = { "offset" : _set_enroute_second_turn_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @enroute_second_turn.setter
    def enroute_second_turn(self, value:"NavigatorTurnDirection") -> None:
        """Option for the second turn."""
        return self._intf.set_property(NavigationOptions._metadata, NavigationOptions._set_enroute_second_turn_metadata, value)

    _property_names[navigation_mode] = "navigation_mode"
    _property_names[arrive_on_course] = "arrive_on_course"
    _property_names[use_magnetic_heading] = "use_magnetic_heading"
    _property_names[enroute_first_turn] = "enroute_first_turn"
    _property_names[enroute_second_turn] = "enroute_second_turn"

    def __init__(self, source_object=None):
        """Construct an object of type NavigationOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, NavigationOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, NavigationOptions, [NavigationOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5297554030905568728, 15312492039192405176), NavigationOptions)
agcls.AgTypeNameMap["NavigationOptions"] = NavigationOptions

class VerticalPlaneOptions(IVerticalPlaneOptions, SupportsDeleteCallback):
    """Class defining the vertical plane options in a procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type VerticalPlaneOptions."""
        SupportsDeleteCallback.__init__(self)
        IVerticalPlaneOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IVerticalPlaneOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, VerticalPlaneOptions, [IVerticalPlaneOptions])

agcls.AgClassCatalog.add_catalog_entry((4791831385042236424, 17790584228054945718), VerticalPlaneOptions)
agcls.AgTypeNameMap["VerticalPlaneOptions"] = VerticalPlaneOptions

class ArcVerticalPlaneOptions(SupportsDeleteCallback):
    """Interface used to access the Vertical Plane options for an arc procedure."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_start_arc_flight_path_angle_method_offset = 1
    _set_start_arc_flight_path_angle_method_offset = 2
    _get_stop_arc_flight_path_angle_method_offset = 3
    _set_stop_arc_flight_path_angle_method_offset = 4
    _get_min_enroute_flight_path_angle_method_offset = 5
    _set_min_enroute_flight_path_angle_method_offset = 6
    _get_max_enroute_flight_path_angle_method_offset = 7
    _set_max_enroute_flight_path_angle_method_offset = 8
    _get_max_vert_plane_radius_factor_method_offset = 9
    _set_max_vert_plane_radius_factor_method_offset = 10
    _metadata = {
        "iid_data" : (5401521584591596237, 12745820344260646540),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ArcVerticalPlaneOptions)

    _get_start_arc_flight_path_angle_metadata = { "offset" : _get_start_arc_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def start_arc_flight_path_angle(self) -> typing.Any:
        """Get or set the pitch angle of the flight path at the start of the arc."""
        return self._intf.get_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._get_start_arc_flight_path_angle_metadata)

    _set_start_arc_flight_path_angle_metadata = { "offset" : _set_start_arc_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @start_arc_flight_path_angle.setter
    def start_arc_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the pitch angle of the flight path at the start of the arc."""
        return self._intf.set_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._set_start_arc_flight_path_angle_metadata, value)

    _get_stop_arc_flight_path_angle_metadata = { "offset" : _get_stop_arc_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def stop_arc_flight_path_angle(self) -> typing.Any:
        """Get or set the pitch angle of the flight path at the end of the arc."""
        return self._intf.get_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._get_stop_arc_flight_path_angle_metadata)

    _set_stop_arc_flight_path_angle_metadata = { "offset" : _set_stop_arc_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @stop_arc_flight_path_angle.setter
    def stop_arc_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the pitch angle of the flight path at the end of the arc."""
        return self._intf.set_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._set_stop_arc_flight_path_angle_metadata, value)

    _get_min_enroute_flight_path_angle_metadata = { "offset" : _get_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def min_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._get_min_enroute_flight_path_angle_metadata)

    _set_min_enroute_flight_path_angle_metadata = { "offset" : _set_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @min_enroute_flight_path_angle.setter
    def min_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._set_min_enroute_flight_path_angle_metadata, value)

    _get_max_enroute_flight_path_angle_metadata = { "offset" : _get_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._get_max_enroute_flight_path_angle_metadata)

    _set_max_enroute_flight_path_angle_metadata = { "offset" : _set_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_enroute_flight_path_angle.setter
    def max_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._set_max_enroute_flight_path_angle_metadata, value)

    _get_max_vert_plane_radius_factor_metadata = { "offset" : _get_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_vert_plane_radius_factor(self) -> float:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.get_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._get_max_vert_plane_radius_factor_metadata)

    _set_max_vert_plane_radius_factor_metadata = { "offset" : _set_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_vert_plane_radius_factor.setter
    def max_vert_plane_radius_factor(self, value:float) -> None:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.set_property(ArcVerticalPlaneOptions._metadata, ArcVerticalPlaneOptions._set_max_vert_plane_radius_factor_metadata, value)

    _property_names[start_arc_flight_path_angle] = "start_arc_flight_path_angle"
    _property_names[stop_arc_flight_path_angle] = "stop_arc_flight_path_angle"
    _property_names[min_enroute_flight_path_angle] = "min_enroute_flight_path_angle"
    _property_names[max_enroute_flight_path_angle] = "max_enroute_flight_path_angle"
    _property_names[max_vert_plane_radius_factor] = "max_vert_plane_radius_factor"

    def __init__(self, source_object=None):
        """Construct an object of type ArcVerticalPlaneOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ArcVerticalPlaneOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ArcVerticalPlaneOptions, [ArcVerticalPlaneOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4652016757415061557, 6686714748074581137), ArcVerticalPlaneOptions)
agcls.AgTypeNameMap["ArcVerticalPlaneOptions"] = ArcVerticalPlaneOptions

class VerticalPlaneAndFlightPathOptions(SupportsDeleteCallback):
    """Interface used to access the Vertical Plane and Final Flight Path Angle options for an Aviator procedure."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_final_flight_path_angle_method_offset = 1
    _set_final_flight_path_angle_method_offset = 2
    _get_min_enroute_flight_path_angle_method_offset = 3
    _set_min_enroute_flight_path_angle_method_offset = 4
    _get_max_enroute_flight_path_angle_method_offset = 5
    _set_max_enroute_flight_path_angle_method_offset = 6
    _get_max_vert_plane_radius_factor_method_offset = 7
    _set_max_vert_plane_radius_factor_method_offset = 8
    _metadata = {
        "iid_data" : (4866221299348241801, 14897448743116069797),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, VerticalPlaneAndFlightPathOptions)

    _get_final_flight_path_angle_metadata = { "offset" : _get_final_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def final_flight_path_angle(self) -> typing.Any:
        """Get or set the pitch angle of the flight path at the end of the procedure."""
        return self._intf.get_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._get_final_flight_path_angle_metadata)

    _set_final_flight_path_angle_metadata = { "offset" : _set_final_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @final_flight_path_angle.setter
    def final_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the pitch angle of the flight path at the end of the procedure."""
        return self._intf.set_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._set_final_flight_path_angle_metadata, value)

    _get_min_enroute_flight_path_angle_metadata = { "offset" : _get_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def min_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._get_min_enroute_flight_path_angle_metadata)

    _set_min_enroute_flight_path_angle_metadata = { "offset" : _set_min_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @min_enroute_flight_path_angle.setter
    def min_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the minimum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._set_min_enroute_flight_path_angle_metadata, value)

    _get_max_enroute_flight_path_angle_metadata = { "offset" : _get_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_enroute_flight_path_angle(self) -> typing.Any:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.get_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._get_max_enroute_flight_path_angle_metadata)

    _set_max_enroute_flight_path_angle_metadata = { "offset" : _set_max_enroute_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_enroute_flight_path_angle.setter
    def max_enroute_flight_path_angle(self, value:typing.Any) -> None:
        """Get or set the maximum pitch angle of the flight path during enroute segments of the procedure."""
        return self._intf.set_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._set_max_enroute_flight_path_angle_metadata, value)

    _get_max_vert_plane_radius_factor_metadata = { "offset" : _get_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_vert_plane_radius_factor(self) -> float:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.get_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._get_max_vert_plane_radius_factor_metadata)

    _set_max_vert_plane_radius_factor_metadata = { "offset" : _set_max_vert_plane_radius_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_vert_plane_radius_factor.setter
    def max_vert_plane_radius_factor(self, value:float) -> None:
        """Get or set the maximum amount the radius of vertical curve will be increased to minimize the flight path angle required to complete it."""
        return self._intf.set_property(VerticalPlaneAndFlightPathOptions._metadata, VerticalPlaneAndFlightPathOptions._set_max_vert_plane_radius_factor_metadata, value)

    _property_names[final_flight_path_angle] = "final_flight_path_angle"
    _property_names[min_enroute_flight_path_angle] = "min_enroute_flight_path_angle"
    _property_names[max_enroute_flight_path_angle] = "max_enroute_flight_path_angle"
    _property_names[max_vert_plane_radius_factor] = "max_vert_plane_radius_factor"

    def __init__(self, source_object=None):
        """Construct an object of type VerticalPlaneAndFlightPathOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, VerticalPlaneAndFlightPathOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, VerticalPlaneAndFlightPathOptions, [VerticalPlaneAndFlightPathOptions, ])

agcls.AgClassCatalog.add_catalog_entry((4880113513667107889, 8420899053888797363), VerticalPlaneAndFlightPathOptions)
agcls.AgTypeNameMap["VerticalPlaneAndFlightPathOptions"] = VerticalPlaneAndFlightPathOptions

class LandingVerticalPlaneOptions(IVerticalPlaneOptions, SupportsDeleteCallback):
    """Class defining the vertical plane options in a landing procedure."""
    def __init__(self, source_object=None):
        """Construct an object of type LandingVerticalPlaneOptions."""
        SupportsDeleteCallback.__init__(self)
        IVerticalPlaneOptions.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IVerticalPlaneOptions._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingVerticalPlaneOptions, [IVerticalPlaneOptions])

agcls.AgClassCatalog.add_catalog_entry((4626924203837884328, 7628565386577197988), LandingVerticalPlaneOptions)
agcls.AgTypeNameMap["LandingVerticalPlaneOptions"] = LandingVerticalPlaneOptions

class RunwayHeadingOptions(SupportsDeleteCallback):
    """
    Interface for the Runway Heading Options found in a Takeoff or Landing procedure.

    Examples
    --------
    Add a takeoff procedure from a runway:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a takeoff procedure with a runway as a site
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = takeoff.runway_heading_options
    >>> # Opt to use the headwind runway
    >>> headingOptions.runway_mode = RunwayHighLowEnd.HEADWIND
    >>>
    >>> # Set the takeoff mode and get that interface
    >>> takeoff.takeoff_mode = TakeoffMode.TAKEOFF_NORMAL
    >>> takeoffNormal = takeoff.mode_as_normal
    >>>
    >>> # Set the takeoff climb angle
    >>> takeoffNormal.takeoff_climb_angle = 5
    >>> # Set the departure altitude above the runway
    >>> takeoffNormal.departure_altitude = 600
    >>> # Set the altitude offset for the runway
    >>> takeoffNormal.runway_altitude_offset = 10
    >>> # Use terrain for the runway's altitude
    >>> takeoffNormal.use_runway_terrain = True

    Add and configure a landing procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a landing procedure
    >>> landing = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_LANDING)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = landing.runway_heading_options
    >>> # Land from the low end
    >>> headingOptions.runway_mode = RunwayHighLowEnd.LOW_END
    >>>
    >>> # Use a standard instrument approach
    >>> landing.approach_mode = ApproachMode.STANDARD_INSTRUMENT_APPROACH
    >>> # Get the options for a standard instrument approach
    >>> sia = landing.mode_as_standard_instrument_approach
    >>> # Change the approach altitude
    >>> sia.approach_altitude = 1000
    >>> # Change the glideslope
    >>> sia.glideslope = 4
    >>> # Offset the runway altitude
    >>> sia.runway_altitude_offset = 10
    >>> # Use the terrain as an altitude reference for the runway
    >>> sia.use_runway_terrain = True
    """

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_runway_mode_method_offset = 1
    _set_runway_mode_method_offset = 2
    _metadata = {
        "iid_data" : (4772948616711334669, 8458670227314885548),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, RunwayHeadingOptions)

    _get_runway_mode_metadata = { "offset" : _get_runway_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RunwayHighLowEnd),) }
    @property
    def runway_mode(self) -> "RunwayHighLowEnd":
        """Get or set the runway heading that the aircraft will use."""
        return self._intf.get_property(RunwayHeadingOptions._metadata, RunwayHeadingOptions._get_runway_mode_metadata)

    _set_runway_mode_metadata = { "offset" : _set_runway_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RunwayHighLowEnd),) }
    @runway_mode.setter
    def runway_mode(self, value:"RunwayHighLowEnd") -> None:
        """Get or set the runway heading that the aircraft will use."""
        return self._intf.set_property(RunwayHeadingOptions._metadata, RunwayHeadingOptions._set_runway_mode_metadata, value)

    _property_names[runway_mode] = "runway_mode"

    def __init__(self, source_object=None):
        """Construct an object of type RunwayHeadingOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, RunwayHeadingOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, RunwayHeadingOptions, [RunwayHeadingOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5151356003940587454, 7154037557713333679), RunwayHeadingOptions)
agcls.AgTypeNameMap["RunwayHeadingOptions"] = RunwayHeadingOptions

class LandingEnterDownwindPattern(SupportsDeleteCallback):
    """The interface used to access the options for a Downwind Pattern approach mode for a landing procedure. The approach mode must be set to Downwind Pattern to access this interface."""

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_approach_fix_range_method_offset = 1
    _set_approach_fix_range_method_offset = 2
    _get_approach_fix_range_mode_method_offset = 3
    _set_approach_fix_range_mode_method_offset = 4
    _get_abeam_distance_method_offset = 5
    _set_abeam_distance_method_offset = 6
    _get_abeam_altitude_method_offset = 7
    _set_abeam_altitude_method_offset = 8
    _get_final_turn_method_offset = 9
    _set_final_turn_method_offset = 10
    _get_glideslope_method_offset = 11
    _set_glideslope_method_offset = 12
    _get_runway_altitude_offset_method_offset = 13
    _set_runway_altitude_offset_method_offset = 14
    _get_use_runway_terrain_method_offset = 15
    _set_use_runway_terrain_method_offset = 16
    _get_touch_and_go_method_offset = 17
    _set_touch_and_go_method_offset = 18
    _metadata = {
        "iid_data" : (4734513232810016191, 3083046802263741063),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LandingEnterDownwindPattern)

    _get_approach_fix_range_metadata = { "offset" : _get_approach_fix_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def approach_fix_range(self) -> float:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_approach_fix_range_metadata)

    _set_approach_fix_range_metadata = { "offset" : _set_approach_fix_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @approach_fix_range.setter
    def approach_fix_range(self, value:float) -> None:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_approach_fix_range_metadata, value)

    _get_approach_fix_range_mode_metadata = { "offset" : _get_approach_fix_range_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @property
    def approach_fix_range_mode(self) -> "LandingApproachFixRangeMode":
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_approach_fix_range_mode_metadata)

    _set_approach_fix_range_mode_metadata = { "offset" : _set_approach_fix_range_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @approach_fix_range_mode.setter
    def approach_fix_range_mode(self, value:"LandingApproachFixRangeMode") -> None:
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_approach_fix_range_mode_metadata, value)

    _get_abeam_distance_metadata = { "offset" : _get_abeam_distance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def abeam_distance(self) -> float:
        """Get or set the distance from the runway that the aircraft will fly the parallel leg of the landing pattern."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_abeam_distance_metadata)

    _set_abeam_distance_metadata = { "offset" : _set_abeam_distance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @abeam_distance.setter
    def abeam_distance(self, value:float) -> None:
        """Get or set the distance from the runway that the aircraft will fly the parallel leg of the landing pattern."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_abeam_distance_metadata, value)

    _get_abeam_altitude_metadata = { "offset" : _get_abeam_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def abeam_altitude(self) -> float:
        """Get or set the altitude at which the aircraft will fly the parallel leg of the landing pattern."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_abeam_altitude_metadata)

    _set_abeam_altitude_metadata = { "offset" : _set_abeam_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @abeam_altitude.setter
    def abeam_altitude(self, value:float) -> None:
        """Get or set the altitude at which the aircraft will fly the parallel leg of the landing pattern."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_abeam_altitude_metadata, value)

    _get_final_turn_metadata = { "offset" : _get_final_turn_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @property
    def final_turn(self) -> "NavigatorTurnDirection":
        """Get or set the direction of the turn that the aircraft will make when it lines up over the runway to land."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_final_turn_metadata)

    _set_final_turn_metadata = { "offset" : _set_final_turn_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(NavigatorTurnDirection),) }
    @final_turn.setter
    def final_turn(self, value:"NavigatorTurnDirection") -> None:
        """Get or set the direction of the turn that the aircraft will make when it lines up over the runway to land."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_final_turn_metadata, value)

    _get_glideslope_metadata = { "offset" : _get_glideslope_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def glideslope(self) -> typing.Any:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_glideslope_metadata)

    _set_glideslope_metadata = { "offset" : _set_glideslope_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @glideslope.setter
    def glideslope(self, value:typing.Any) -> None:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_glideslope_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_runway_altitude_offset_metadata, value)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_use_runway_terrain_metadata, value)

    _get_touch_and_go_metadata = { "offset" : _get_touch_and_go_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def touch_and_go(self) -> bool:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.get_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._get_touch_and_go_metadata)

    _set_touch_and_go_metadata = { "offset" : _set_touch_and_go_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @touch_and_go.setter
    def touch_and_go(self, value:bool) -> None:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.set_property(LandingEnterDownwindPattern._metadata, LandingEnterDownwindPattern._set_touch_and_go_metadata, value)

    _property_names[approach_fix_range] = "approach_fix_range"
    _property_names[approach_fix_range_mode] = "approach_fix_range_mode"
    _property_names[abeam_distance] = "abeam_distance"
    _property_names[abeam_altitude] = "abeam_altitude"
    _property_names[final_turn] = "final_turn"
    _property_names[glideslope] = "glideslope"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[touch_and_go] = "touch_and_go"

    def __init__(self, source_object=None):
        """Construct an object of type LandingEnterDownwindPattern."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LandingEnterDownwindPattern)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingEnterDownwindPattern, [LandingEnterDownwindPattern, ])

agcls.AgClassCatalog.add_catalog_entry((5264644757506879488, 10964076385913793182), LandingEnterDownwindPattern)
agcls.AgTypeNameMap["LandingEnterDownwindPattern"] = LandingEnterDownwindPattern

class LandingInterceptGlideslope(SupportsDeleteCallback):
    """The interface used to access the options for an Intercept Glideslope approach mode for a landing procedure. The approach mode must be set to Intercept Glideslope to access this interface."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_approach_fix_range_method_offset = 1
    _set_approach_fix_range_method_offset = 2
    _get_approach_fix_range_mode_method_offset = 3
    _set_approach_fix_range_mode_method_offset = 4
    _get_glideslope_method_offset = 5
    _set_glideslope_method_offset = 6
    _get_runway_altitude_offset_method_offset = 7
    _set_runway_altitude_offset_method_offset = 8
    _get_use_runway_terrain_method_offset = 9
    _set_use_runway_terrain_method_offset = 10
    _get_touch_and_go_method_offset = 11
    _set_touch_and_go_method_offset = 12
    _metadata = {
        "iid_data" : (4722193432406915614, 3919694554656369318),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LandingInterceptGlideslope)

    _get_approach_fix_range_metadata = { "offset" : _get_approach_fix_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def approach_fix_range(self) -> float:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_approach_fix_range_metadata)

    _set_approach_fix_range_metadata = { "offset" : _set_approach_fix_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @approach_fix_range.setter
    def approach_fix_range(self, value:float) -> None:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_approach_fix_range_metadata, value)

    _get_approach_fix_range_mode_metadata = { "offset" : _get_approach_fix_range_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @property
    def approach_fix_range_mode(self) -> "LandingApproachFixRangeMode":
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_approach_fix_range_mode_metadata)

    _set_approach_fix_range_mode_metadata = { "offset" : _set_approach_fix_range_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @approach_fix_range_mode.setter
    def approach_fix_range_mode(self, value:"LandingApproachFixRangeMode") -> None:
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_approach_fix_range_mode_metadata, value)

    _get_glideslope_metadata = { "offset" : _get_glideslope_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def glideslope(self) -> typing.Any:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_glideslope_metadata)

    _set_glideslope_metadata = { "offset" : _set_glideslope_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @glideslope.setter
    def glideslope(self, value:typing.Any) -> None:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_glideslope_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_runway_altitude_offset_metadata, value)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_use_runway_terrain_metadata, value)

    _get_touch_and_go_metadata = { "offset" : _get_touch_and_go_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def touch_and_go(self) -> bool:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.get_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._get_touch_and_go_metadata)

    _set_touch_and_go_metadata = { "offset" : _set_touch_and_go_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @touch_and_go.setter
    def touch_and_go(self, value:bool) -> None:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.set_property(LandingInterceptGlideslope._metadata, LandingInterceptGlideslope._set_touch_and_go_metadata, value)

    _property_names[approach_fix_range] = "approach_fix_range"
    _property_names[approach_fix_range_mode] = "approach_fix_range_mode"
    _property_names[glideslope] = "glideslope"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[touch_and_go] = "touch_and_go"

    def __init__(self, source_object=None):
        """Construct an object of type LandingInterceptGlideslope."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LandingInterceptGlideslope)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingInterceptGlideslope, [LandingInterceptGlideslope, ])

agcls.AgClassCatalog.add_catalog_entry((5366344207817725495, 12409533042992159899), LandingInterceptGlideslope)
agcls.AgTypeNameMap["LandingInterceptGlideslope"] = LandingInterceptGlideslope

class LandingStandardInstrumentApproach(SupportsDeleteCallback):
    """
    The interface used to access the options for a Standard Instrument Approach mode for a landing procedure. The approach mode must be set to Standard Instrument Approach to access this interface.

    Examples
    --------
    Add and configure a landing procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a landing procedure
    >>> landing = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_LANDING)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = landing.runway_heading_options
    >>> # Land from the low end
    >>> headingOptions.runway_mode = RunwayHighLowEnd.LOW_END
    >>>
    >>> # Use a standard instrument approach
    >>> landing.approach_mode = ApproachMode.STANDARD_INSTRUMENT_APPROACH
    >>> # Get the options for a standard instrument approach
    >>> sia = landing.mode_as_standard_instrument_approach
    >>> # Change the approach altitude
    >>> sia.approach_altitude = 1000
    >>> # Change the glideslope
    >>> sia.glideslope = 4
    >>> # Offset the runway altitude
    >>> sia.runway_altitude_offset = 10
    >>> # Use the terrain as an altitude reference for the runway
    >>> sia.use_runway_terrain = True
    """

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_approach_altitude_method_offset = 1
    _set_approach_altitude_method_offset = 2
    _get_level_off_mode_method_offset = 3
    _set_level_off_mode_method_offset = 4
    _get_approach_fix_range_method_offset = 5
    _set_approach_fix_range_method_offset = 6
    _get_approach_fix_range_mode_method_offset = 7
    _set_approach_fix_range_mode_method_offset = 8
    _get_glideslope_method_offset = 9
    _set_glideslope_method_offset = 10
    _get_runway_altitude_offset_method_offset = 11
    _set_runway_altitude_offset_method_offset = 12
    _get_use_runway_terrain_method_offset = 13
    _set_use_runway_terrain_method_offset = 14
    _get_touch_and_go_method_offset = 15
    _set_touch_and_go_method_offset = 16
    _metadata = {
        "iid_data" : (5318231538284992196, 1134507381071316131),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LandingStandardInstrumentApproach)

    _get_approach_altitude_metadata = { "offset" : _get_approach_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def approach_altitude(self) -> float:
        """Get or set the aircraft's altitude at the Initial Approach Fix Range."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_approach_altitude_metadata)

    _set_approach_altitude_metadata = { "offset" : _set_approach_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @approach_altitude.setter
    def approach_altitude(self, value:float) -> None:
        """Get or set the aircraft's altitude at the Initial Approach Fix Range."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_approach_altitude_metadata, value)

    _get_level_off_mode_metadata = { "offset" : _get_level_off_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @property
    def level_off_mode(self) -> "AltitudeConstraintManeuverMode":
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_level_off_mode_metadata)

    _set_level_off_mode_metadata = { "offset" : _set_level_off_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AltitudeConstraintManeuverMode),) }
    @level_off_mode.setter
    def level_off_mode(self, value:"AltitudeConstraintManeuverMode") -> None:
        """Get or set the level off mode. This is only used when the must level off option is on."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_level_off_mode_metadata, value)

    _get_approach_fix_range_metadata = { "offset" : _get_approach_fix_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def approach_fix_range(self) -> float:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_approach_fix_range_metadata)

    _set_approach_fix_range_metadata = { "offset" : _set_approach_fix_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @approach_fix_range.setter
    def approach_fix_range(self, value:float) -> None:
        """Get or set the range from the reference point of the runway at which the aircraft begins its landing approach."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_approach_fix_range_metadata, value)

    _get_approach_fix_range_mode_metadata = { "offset" : _get_approach_fix_range_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @property
    def approach_fix_range_mode(self) -> "LandingApproachFixRangeMode":
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_approach_fix_range_mode_metadata)

    _set_approach_fix_range_mode_metadata = { "offset" : _set_approach_fix_range_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(LandingApproachFixRangeMode),) }
    @approach_fix_range_mode.setter
    def approach_fix_range_mode(self, value:"LandingApproachFixRangeMode") -> None:
        """Get or set the reference point on the runway for the Approach Fix Range."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_approach_fix_range_mode_metadata, value)

    _get_glideslope_metadata = { "offset" : _get_glideslope_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def glideslope(self) -> typing.Any:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_glideslope_metadata)

    _set_glideslope_metadata = { "offset" : _set_glideslope_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @glideslope.setter
    def glideslope(self, value:typing.Any) -> None:
        """Get or set the angle from the horizontal on which the aircraft descends to touchdown."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_glideslope_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_runway_altitude_offset_metadata, value)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_use_runway_terrain_metadata, value)

    _get_touch_and_go_metadata = { "offset" : _get_touch_and_go_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def touch_and_go(self) -> bool:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.get_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._get_touch_and_go_metadata)

    _set_touch_and_go_metadata = { "offset" : _set_touch_and_go_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @touch_and_go.setter
    def touch_and_go(self, value:bool) -> None:
        """Opt whether to perform a Touch and Go landing. The procedure will stop at wheels down and can be immediately followed by a takeoff procedure."""
        return self._intf.set_property(LandingStandardInstrumentApproach._metadata, LandingStandardInstrumentApproach._set_touch_and_go_metadata, value)

    _property_names[approach_altitude] = "approach_altitude"
    _property_names[level_off_mode] = "level_off_mode"
    _property_names[approach_fix_range] = "approach_fix_range"
    _property_names[approach_fix_range_mode] = "approach_fix_range_mode"
    _property_names[glideslope] = "glideslope"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[touch_and_go] = "touch_and_go"

    def __init__(self, source_object=None):
        """Construct an object of type LandingStandardInstrumentApproach."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LandingStandardInstrumentApproach)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LandingStandardInstrumentApproach, [LandingStandardInstrumentApproach, ])

agcls.AgClassCatalog.add_catalog_entry((4840677424178112842, 5244220578861766065), LandingStandardInstrumentApproach)
agcls.AgTypeNameMap["LandingStandardInstrumentApproach"] = LandingStandardInstrumentApproach

class TakeoffDeparturePoint(SupportsDeleteCallback):
    """The interface used to access the options for a Departure Point takeoff mode. The mode must be set to Departure Point to access this interface."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_takeoff_climb_angle_method_offset = 1
    _set_takeoff_climb_angle_method_offset = 2
    _get_departure_altitude_method_offset = 3
    _set_departure_altitude_method_offset = 4
    _get_departure_point_range_method_offset = 5
    _set_departure_point_range_method_offset = 6
    _get_use_runway_terrain_method_offset = 7
    _set_use_runway_terrain_method_offset = 8
    _get_runway_altitude_offset_method_offset = 9
    _set_runway_altitude_offset_method_offset = 10
    _get_hold_on_deck_method_offset = 11
    _set_hold_on_deck_method_offset = 12
    _metadata = {
        "iid_data" : (5384974105934402717, 12902722603171464861),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TakeoffDeparturePoint)

    _get_takeoff_climb_angle_metadata = { "offset" : _get_takeoff_climb_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def takeoff_climb_angle(self) -> typing.Any:
        """Get or set the angle at which the aircraft will climb from the procedure site to the departure point."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_takeoff_climb_angle_metadata)

    _set_takeoff_climb_angle_metadata = { "offset" : _set_takeoff_climb_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @takeoff_climb_angle.setter
    def takeoff_climb_angle(self, value:typing.Any) -> None:
        """Get or set the angle at which the aircraft will climb from the procedure site to the departure point."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_takeoff_climb_angle_metadata, value)

    _get_departure_altitude_metadata = { "offset" : _get_departure_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_altitude(self) -> float:
        """Get or set the aircraft's altitude when it departs the runway."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_departure_altitude_metadata)

    _set_departure_altitude_metadata = { "offset" : _set_departure_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @departure_altitude.setter
    def departure_altitude(self, value:float) -> None:
        """Get or set the aircraft's altitude when it departs the runway."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_departure_altitude_metadata, value)

    _get_departure_point_range_metadata = { "offset" : _get_departure_point_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_point_range(self) -> float:
        """Get or set the downrange distance the aircraft will travel when departing the runway."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_departure_point_range_metadata)

    _set_departure_point_range_metadata = { "offset" : _set_departure_point_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @departure_point_range.setter
    def departure_point_range(self, value:float) -> None:
        """Get or set the downrange distance the aircraft will travel when departing the runway."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_departure_point_range_metadata, value)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_use_runway_terrain_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_runway_altitude_offset_metadata, value)

    _get_hold_on_deck_metadata = { "offset" : _get_hold_on_deck_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def hold_on_deck(self) -> typing.Any:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.get_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._get_hold_on_deck_metadata)

    _set_hold_on_deck_metadata = { "offset" : _set_hold_on_deck_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @hold_on_deck.setter
    def hold_on_deck(self, value:typing.Any) -> None:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.set_property(TakeoffDeparturePoint._metadata, TakeoffDeparturePoint._set_hold_on_deck_metadata, value)

    _property_names[takeoff_climb_angle] = "takeoff_climb_angle"
    _property_names[departure_altitude] = "departure_altitude"
    _property_names[departure_point_range] = "departure_point_range"
    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[hold_on_deck] = "hold_on_deck"

    def __init__(self, source_object=None):
        """Construct an object of type TakeoffDeparturePoint."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TakeoffDeparturePoint)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TakeoffDeparturePoint, [TakeoffDeparturePoint, ])

agcls.AgClassCatalog.add_catalog_entry((5170363125142911430, 3874157076900173501), TakeoffDeparturePoint)
agcls.AgTypeNameMap["TakeoffDeparturePoint"] = TakeoffDeparturePoint

class TakeoffLowTransition(SupportsDeleteCallback):
    """The interface used to access the options for a Low Transition takeoff mode. The mode must be set to Low Transition to access this interface."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_runway_terrain_method_offset = 1
    _set_use_runway_terrain_method_offset = 2
    _get_runway_altitude_offset_method_offset = 3
    _set_runway_altitude_offset_method_offset = 4
    _get_hold_on_deck_method_offset = 5
    _set_hold_on_deck_method_offset = 6
    _metadata = {
        "iid_data" : (5665343166367508418, 13893054072396564138),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TakeoffLowTransition)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(TakeoffLowTransition._metadata, TakeoffLowTransition._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(TakeoffLowTransition._metadata, TakeoffLowTransition._set_use_runway_terrain_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(TakeoffLowTransition._metadata, TakeoffLowTransition._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(TakeoffLowTransition._metadata, TakeoffLowTransition._set_runway_altitude_offset_metadata, value)

    _get_hold_on_deck_metadata = { "offset" : _get_hold_on_deck_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def hold_on_deck(self) -> typing.Any:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.get_property(TakeoffLowTransition._metadata, TakeoffLowTransition._get_hold_on_deck_metadata)

    _set_hold_on_deck_metadata = { "offset" : _set_hold_on_deck_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @hold_on_deck.setter
    def hold_on_deck(self, value:typing.Any) -> None:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.set_property(TakeoffLowTransition._metadata, TakeoffLowTransition._set_hold_on_deck_metadata, value)

    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[hold_on_deck] = "hold_on_deck"

    def __init__(self, source_object=None):
        """Construct an object of type TakeoffLowTransition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TakeoffLowTransition)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TakeoffLowTransition, [TakeoffLowTransition, ])

agcls.AgClassCatalog.add_catalog_entry((5668791560561038875, 16923029776143935925), TakeoffLowTransition)
agcls.AgTypeNameMap["TakeoffLowTransition"] = TakeoffLowTransition

class TakeoffNormal(SupportsDeleteCallback):
    """
    The interface used to access the options for a Normal takeoff mode. The mode must be set to Normal to access this interface.

    Examples
    --------
    Add a takeoff procedure from a runway:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a takeoff procedure with a runway as a site
    >>> takeoff = procedures.add(SiteType.SITE_RUNWAY, ProcedureType.PROCEDURE_TAKEOFF)
    >>>
    >>> # Get the runway heading options
    >>> headingOptions = takeoff.runway_heading_options
    >>> # Opt to use the headwind runway
    >>> headingOptions.runway_mode = RunwayHighLowEnd.HEADWIND
    >>>
    >>> # Set the takeoff mode and get that interface
    >>> takeoff.takeoff_mode = TakeoffMode.TAKEOFF_NORMAL
    >>> takeoffNormal = takeoff.mode_as_normal
    >>>
    >>> # Set the takeoff climb angle
    >>> takeoffNormal.takeoff_climb_angle = 5
    >>> # Set the departure altitude above the runway
    >>> takeoffNormal.departure_altitude = 600
    >>> # Set the altitude offset for the runway
    >>> takeoffNormal.runway_altitude_offset = 10
    >>> # Use terrain for the runway's altitude
    >>> takeoffNormal.use_runway_terrain = True
    """

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_takeoff_climb_angle_method_offset = 1
    _set_takeoff_climb_angle_method_offset = 2
    _get_departure_altitude_method_offset = 3
    _set_departure_altitude_method_offset = 4
    _get_use_runway_terrain_method_offset = 5
    _set_use_runway_terrain_method_offset = 6
    _get_runway_altitude_offset_method_offset = 7
    _set_runway_altitude_offset_method_offset = 8
    _get_hold_on_deck_method_offset = 9
    _set_hold_on_deck_method_offset = 10
    _metadata = {
        "iid_data" : (5498845233375819305, 14637385257810436005),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TakeoffNormal)

    _get_takeoff_climb_angle_metadata = { "offset" : _get_takeoff_climb_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def takeoff_climb_angle(self) -> typing.Any:
        """Get or set the angle at which the aircraft will climb from the procedure site to the departure point."""
        return self._intf.get_property(TakeoffNormal._metadata, TakeoffNormal._get_takeoff_climb_angle_metadata)

    _set_takeoff_climb_angle_metadata = { "offset" : _set_takeoff_climb_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @takeoff_climb_angle.setter
    def takeoff_climb_angle(self, value:typing.Any) -> None:
        """Get or set the angle at which the aircraft will climb from the procedure site to the departure point."""
        return self._intf.set_property(TakeoffNormal._metadata, TakeoffNormal._set_takeoff_climb_angle_metadata, value)

    _get_departure_altitude_metadata = { "offset" : _get_departure_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def departure_altitude(self) -> float:
        """Get or set the aircraft's altitude when it departs the runway."""
        return self._intf.get_property(TakeoffNormal._metadata, TakeoffNormal._get_departure_altitude_metadata)

    _set_departure_altitude_metadata = { "offset" : _set_departure_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @departure_altitude.setter
    def departure_altitude(self, value:float) -> None:
        """Get or set the aircraft's altitude when it departs the runway."""
        return self._intf.set_property(TakeoffNormal._metadata, TakeoffNormal._set_departure_altitude_metadata, value)

    _get_use_runway_terrain_metadata = { "offset" : _get_use_runway_terrain_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_runway_terrain(self) -> bool:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.get_property(TakeoffNormal._metadata, TakeoffNormal._get_use_runway_terrain_metadata)

    _set_use_runway_terrain_metadata = { "offset" : _set_use_runway_terrain_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_runway_terrain.setter
    def use_runway_terrain(self, value:bool) -> None:
        """Opt whether to use terrain data to define the runway's ground level attitude."""
        return self._intf.set_property(TakeoffNormal._metadata, TakeoffNormal._set_use_runway_terrain_metadata, value)

    _get_runway_altitude_offset_metadata = { "offset" : _get_runway_altitude_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def runway_altitude_offset(self) -> float:
        """Get or set the altitude offset above the ground level."""
        return self._intf.get_property(TakeoffNormal._metadata, TakeoffNormal._get_runway_altitude_offset_metadata)

    _set_runway_altitude_offset_metadata = { "offset" : _set_runway_altitude_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @runway_altitude_offset.setter
    def runway_altitude_offset(self, value:float) -> None:
        """Get or set the altitude offset above the ground level."""
        return self._intf.set_property(TakeoffNormal._metadata, TakeoffNormal._set_runway_altitude_offset_metadata, value)

    _get_hold_on_deck_metadata = { "offset" : _get_hold_on_deck_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def hold_on_deck(self) -> typing.Any:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.get_property(TakeoffNormal._metadata, TakeoffNormal._get_hold_on_deck_metadata)

    _set_hold_on_deck_metadata = { "offset" : _set_hold_on_deck_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @hold_on_deck.setter
    def hold_on_deck(self, value:typing.Any) -> None:
        """Get or set the duration the aircraft will wait before beginning the takeoff."""
        return self._intf.set_property(TakeoffNormal._metadata, TakeoffNormal._set_hold_on_deck_metadata, value)

    _property_names[takeoff_climb_angle] = "takeoff_climb_angle"
    _property_names[departure_altitude] = "departure_altitude"
    _property_names[use_runway_terrain] = "use_runway_terrain"
    _property_names[runway_altitude_offset] = "runway_altitude_offset"
    _property_names[hold_on_deck] = "hold_on_deck"

    def __init__(self, source_object=None):
        """Construct an object of type TakeoffNormal."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TakeoffNormal)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TakeoffNormal, [TakeoffNormal, ])

agcls.AgClassCatalog.add_catalog_entry((4982205817421927695, 3283745831502333602), TakeoffNormal)
agcls.AgTypeNameMap["TakeoffNormal"] = TakeoffNormal

class LevelTurns(SupportsDeleteCallback):
    """
    Interface used to access the Level Turns Transitions options found in the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_turn_mode_method_offset = 1
    _get_turn_g_method_offset = 2
    _get_bank_angle_method_offset = 3
    _get_turn_acceleration_method_offset = 4
    _get_turn_radius_method_offset = 5
    _get_turn_rate_method_offset = 6
    _set_level_turn_method_offset = 7
    _get_maneuver_mode_method_offset = 8
    _set_maneuver_mode_method_offset = 9
    _get_maneuver_mode_helper_method_offset = 10
    _metadata = {
        "iid_data" : (5723269250337285450, 7849803881290906029),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LevelTurns)

    _get_turn_mode_metadata = { "offset" : _get_turn_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TurnMode),) }
    @property
    def turn_mode(self) -> "TurnMode":
        """Get the turn mode."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_turn_mode_metadata)

    _get_turn_g_metadata = { "offset" : _get_turn_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def turn_g(self) -> float:
        """Get the TurnG."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_turn_g_metadata)

    _get_bank_angle_metadata = { "offset" : _get_bank_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def bank_angle(self) -> typing.Any:
        """Get the bank angle."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_bank_angle_metadata)

    _get_turn_acceleration_metadata = { "offset" : _get_turn_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def turn_acceleration(self) -> float:
        """Get the turn acceleration."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_turn_acceleration_metadata)

    _get_turn_radius_metadata = { "offset" : _get_turn_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def turn_radius(self) -> float:
        """Get the turn radius."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_turn_radius_metadata)

    _get_turn_rate_metadata = { "offset" : _get_turn_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def turn_rate(self) -> float:
        """Get the turn rate."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_turn_rate_metadata)

    _set_level_turn_metadata = { "offset" : _set_level_turn_method_offset,
            "arg_types" : (agcom.LONG, agcom.Variant,),
            "marshallers" : (agmarshall.EnumArg(TurnMode), agmarshall.VariantArg,) }
    def set_level_turn(self, turn_mode:"TurnMode", turn_value:typing.Any) -> None:
        """Set the level turn mode and corresponding value."""
        return self._intf.invoke(LevelTurns._metadata, LevelTurns._set_level_turn_metadata, turn_mode, turn_value)

    _get_maneuver_mode_metadata = { "offset" : _get_maneuver_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @property
    def maneuver_mode(self) -> "AccelerationManeuverMode":
        """Get or set the mode that the aircraft will adhere to the specified acceleration parameters. Scale by atmospheric density will cause the aircraft to consider dynamic pressure when calculating turn radius."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_maneuver_mode_metadata)

    _set_maneuver_mode_metadata = { "offset" : _set_maneuver_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @maneuver_mode.setter
    def maneuver_mode(self, value:"AccelerationManeuverMode") -> None:
        """Get or set the mode that the aircraft will adhere to the specified acceleration parameters. Scale by atmospheric density will cause the aircraft to consider dynamic pressure when calculating turn radius."""
        return self._intf.set_property(LevelTurns._metadata, LevelTurns._set_maneuver_mode_metadata, value)

    _get_maneuver_mode_helper_metadata = { "offset" : _get_maneuver_mode_helper_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maneuver_mode_helper(self) -> "AerodynamicPropulsionManeuverModeHelper":
        """Get the interface for the Aero/Prop Maneuver Mode helper. The maneuver mode must be set to Aero/Prop to access this interface."""
        return self._intf.get_property(LevelTurns._metadata, LevelTurns._get_maneuver_mode_helper_metadata)

    _property_names[turn_mode] = "turn_mode"
    _property_names[turn_g] = "turn_g"
    _property_names[bank_angle] = "bank_angle"
    _property_names[turn_acceleration] = "turn_acceleration"
    _property_names[turn_radius] = "turn_radius"
    _property_names[turn_rate] = "turn_rate"
    _property_names[maneuver_mode] = "maneuver_mode"
    _property_names[maneuver_mode_helper] = "maneuver_mode_helper"

    def __init__(self, source_object=None):
        """Construct an object of type LevelTurns."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LevelTurns)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LevelTurns, [LevelTurns, ])

agcls.AgClassCatalog.add_catalog_entry((4842031065400171495, 15335149086412210363), LevelTurns)
agcls.AgTypeNameMap["LevelTurns"] = LevelTurns

class AttitudeTransitions(SupportsDeleteCallback):
    """
    Interface used to access the Attitude Transitions options found in the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_roll_rate_method_offset = 1
    _set_roll_rate_method_offset = 2
    _get_pitch_rate_method_offset = 3
    _set_pitch_rate_method_offset = 4
    _get_yaw_rate_method_offset = 5
    _set_yaw_rate_method_offset = 6
    _metadata = {
        "iid_data" : (4809317354746996248, 3931367782668783024),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeTransitions)

    _get_roll_rate_metadata = { "offset" : _get_roll_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def roll_rate(self) -> typing.Any:
        """Get or set the roll rate when the aircraft in a turn."""
        return self._intf.get_property(AttitudeTransitions._metadata, AttitudeTransitions._get_roll_rate_metadata)

    _set_roll_rate_metadata = { "offset" : _set_roll_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @roll_rate.setter
    def roll_rate(self, value:typing.Any) -> None:
        """Get or set the roll rate when the aircraft in a turn."""
        return self._intf.set_property(AttitudeTransitions._metadata, AttitudeTransitions._set_roll_rate_metadata, value)

    _get_pitch_rate_metadata = { "offset" : _get_pitch_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def pitch_rate(self) -> typing.Any:
        """Get or set the pitch rate when transitioning between attitude modes."""
        return self._intf.get_property(AttitudeTransitions._metadata, AttitudeTransitions._get_pitch_rate_metadata)

    _set_pitch_rate_metadata = { "offset" : _set_pitch_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @pitch_rate.setter
    def pitch_rate(self, value:typing.Any) -> None:
        """Get or set the pitch rate when transitioning between attitude modes."""
        return self._intf.set_property(AttitudeTransitions._metadata, AttitudeTransitions._set_pitch_rate_metadata, value)

    _get_yaw_rate_metadata = { "offset" : _get_yaw_rate_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def yaw_rate(self) -> typing.Any:
        """Get or set the yaw rate when transitioning between attitude modes."""
        return self._intf.get_property(AttitudeTransitions._metadata, AttitudeTransitions._get_yaw_rate_metadata)

    _set_yaw_rate_metadata = { "offset" : _set_yaw_rate_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @yaw_rate.setter
    def yaw_rate(self, value:typing.Any) -> None:
        """Get or set the yaw rate when transitioning between attitude modes."""
        return self._intf.set_property(AttitudeTransitions._metadata, AttitudeTransitions._set_yaw_rate_metadata, value)

    _property_names[roll_rate] = "roll_rate"
    _property_names[pitch_rate] = "pitch_rate"
    _property_names[yaw_rate] = "yaw_rate"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeTransitions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeTransitions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeTransitions, [AttitudeTransitions, ])

agcls.AgClassCatalog.add_catalog_entry((5584586404333393428, 12387107037886590612), AttitudeTransitions)
agcls.AgTypeNameMap["AttitudeTransitions"] = AttitudeTransitions

class ClimbAndDescentTransitions(SupportsDeleteCallback):
    """
    Interface used to access the Climb and Descent Transitions options found in the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_pull_up_g_method_offset = 1
    _set_max_pull_up_g_method_offset = 2
    _get_max_push_over_g_method_offset = 3
    _set_max_push_over_g_method_offset = 4
    _get_maneuver_mode_method_offset = 5
    _set_maneuver_mode_method_offset = 6
    _get_ignore_flight_path_angle_method_offset = 7
    _set_ignore_flight_path_angle_method_offset = 8
    _get_maneuver_mode_helper_method_offset = 9
    _metadata = {
        "iid_data" : (4787497693753872718, 5667952407318575531),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ClimbAndDescentTransitions)

    _get_max_pull_up_g_metadata = { "offset" : _get_max_pull_up_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_pull_up_g(self) -> float:
        """Get or set the force normal to the velocity vector used to transition into a climb or to a transition out of a dive into the next flight segment."""
        return self._intf.get_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._get_max_pull_up_g_metadata)

    _set_max_pull_up_g_metadata = { "offset" : _set_max_pull_up_g_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_pull_up_g.setter
    def max_pull_up_g(self, value:float) -> None:
        """Get or set the force normal to the velocity vector used to transition into a climb or to a transition out of a dive into the next flight segment."""
        return self._intf.set_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._set_max_pull_up_g_metadata, value)

    _get_max_push_over_g_metadata = { "offset" : _get_max_push_over_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_push_over_g(self) -> float:
        """Get or set the force normal to the velocity vector used to transition into a descent or to a transition from a climb into the next flight segment."""
        return self._intf.get_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._get_max_push_over_g_metadata)

    _set_max_push_over_g_metadata = { "offset" : _set_max_push_over_g_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_push_over_g.setter
    def max_push_over_g(self, value:float) -> None:
        """Get or set the force normal to the velocity vector used to transition into a descent or to a transition from a climb into the next flight segment."""
        return self._intf.set_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._set_max_push_over_g_metadata, value)

    _get_maneuver_mode_metadata = { "offset" : _get_maneuver_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @property
    def maneuver_mode(self) -> "AccelerationManeuverMode":
        """Get or set the mode that the aircraft will adhere to the specified acceleration parameters. Scale by atmospheric density will cause the aircraft to consider dynamic pressure when calculating turn radius."""
        return self._intf.get_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._get_maneuver_mode_metadata)

    _set_maneuver_mode_metadata = { "offset" : _set_maneuver_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverMode),) }
    @maneuver_mode.setter
    def maneuver_mode(self, value:"AccelerationManeuverMode") -> None:
        """Get or set the mode that the aircraft will adhere to the specified acceleration parameters. Scale by atmospheric density will cause the aircraft to consider dynamic pressure when calculating turn radius."""
        return self._intf.set_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._set_maneuver_mode_metadata, value)

    _get_ignore_flight_path_angle_metadata = { "offset" : _get_ignore_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def ignore_flight_path_angle(self) -> bool:
        """Opt whether to ignore the flight path angle."""
        return self._intf.get_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._get_ignore_flight_path_angle_metadata)

    _set_ignore_flight_path_angle_metadata = { "offset" : _set_ignore_flight_path_angle_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @ignore_flight_path_angle.setter
    def ignore_flight_path_angle(self, value:bool) -> None:
        """Opt whether to ignore the flight path angle."""
        return self._intf.set_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._set_ignore_flight_path_angle_metadata, value)

    _get_maneuver_mode_helper_metadata = { "offset" : _get_maneuver_mode_helper_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maneuver_mode_helper(self) -> "AerodynamicPropulsionManeuverModeHelper":
        """Get the interface for the Aero/Prop Maneuver Mode helper. The maneuver mode must be set to Aero/Prop to access this interface."""
        return self._intf.get_property(ClimbAndDescentTransitions._metadata, ClimbAndDescentTransitions._get_maneuver_mode_helper_metadata)

    _property_names[max_pull_up_g] = "max_pull_up_g"
    _property_names[max_push_over_g] = "max_push_over_g"
    _property_names[maneuver_mode] = "maneuver_mode"
    _property_names[ignore_flight_path_angle] = "ignore_flight_path_angle"
    _property_names[maneuver_mode_helper] = "maneuver_mode_helper"

    def __init__(self, source_object=None):
        """Construct an object of type ClimbAndDescentTransitions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ClimbAndDescentTransitions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ClimbAndDescentTransitions, [ClimbAndDescentTransitions, ])

agcls.AgClassCatalog.add_catalog_entry((5374744943989914698, 12300077720877790391), ClimbAndDescentTransitions)
agcls.AgTypeNameMap["ClimbAndDescentTransitions"] = ClimbAndDescentTransitions

class AerodynamicPropulsionManeuverModeHelper(SupportsDeleteCallback):
    """Interface used to access the The calculation mode for the Aero/Prop maneuver mode helper. Helper found in the Basic Acceleration Model of an aircraft."""

    _num_methods = 19
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_mode_method_offset = 1
    _set_mode_method_offset = 2
    _get_flight_mode_method_offset = 3
    _set_flight_mode_method_offset = 4
    _get_use_afterburner_method_offset = 5
    _set_use_afterburner_method_offset = 6
    _get_reference_weight_method_offset = 7
    _set_reference_weight_method_offset = 8
    _get_reference_altitude_method_offset = 9
    _set_reference_altitude_method_offset = 10
    _get_reference_airspeed_method_offset = 11
    _get_reference_airspeed_type_method_offset = 12
    _set_reference_airspeed_method_offset = 13
    _get_reference_load_factor_method_offset = 14
    _set_reference_load_factor_method_offset = 15
    _get_estimated_ps_method_offset = 16
    _get_control_authority_method_offset = 17
    _set_control_authority_method_offset = 18
    _get_status_message_method_offset = 19
    _metadata = {
        "iid_data" : (5328095794674712717, 8005940665812164797),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AerodynamicPropulsionManeuverModeHelper)

    _get_mode_metadata = { "offset" : _get_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverAerodynamicPropulsionMode),) }
    @property
    def mode(self) -> "AccelerationManeuverAerodynamicPropulsionMode":
        """Get or set the calculation mode for the Aero/Prop maneuver mode helper."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_mode_metadata)

    _set_mode_metadata = { "offset" : _set_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccelerationManeuverAerodynamicPropulsionMode),) }
    @mode.setter
    def mode(self, value:"AccelerationManeuverAerodynamicPropulsionMode") -> None:
        """Get or set the calculation mode for the Aero/Prop maneuver mode helper."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_mode_metadata, value)

    _get_flight_mode_metadata = { "offset" : _get_flight_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AerodynamicPropulsionFlightMode),) }
    @property
    def flight_mode(self) -> "AerodynamicPropulsionFlightMode":
        """Get or set the performance flight mode."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_flight_mode_metadata)

    _set_flight_mode_metadata = { "offset" : _set_flight_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AerodynamicPropulsionFlightMode),) }
    @flight_mode.setter
    def flight_mode(self, value:"AerodynamicPropulsionFlightMode") -> None:
        """Get or set the performance flight mode."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_flight_mode_metadata, value)

    _get_use_afterburner_metadata = { "offset" : _get_use_afterburner_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_afterburner(self) -> bool:
        """Opt whether to use the afterburner if it is possible."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_use_afterburner_metadata)

    _set_use_afterburner_metadata = { "offset" : _set_use_afterburner_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_afterburner.setter
    def use_afterburner(self, value:bool) -> None:
        """Opt whether to use the afterburner if it is possible."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_use_afterburner_metadata, value)

    _get_reference_weight_metadata = { "offset" : _get_reference_weight_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def reference_weight(self) -> float:
        """Get or set the reference weight."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_reference_weight_metadata)

    _set_reference_weight_metadata = { "offset" : _set_reference_weight_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @reference_weight.setter
    def reference_weight(self, value:float) -> None:
        """Get or set the reference weight."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_reference_weight_metadata, value)

    _get_reference_altitude_metadata = { "offset" : _get_reference_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def reference_altitude(self) -> float:
        """Get or set the reference altitude."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_reference_altitude_metadata)

    _set_reference_altitude_metadata = { "offset" : _set_reference_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @reference_altitude.setter
    def reference_altitude(self, value:float) -> None:
        """Get or set the reference altitude."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_reference_altitude_metadata, value)

    _get_reference_airspeed_metadata = { "offset" : _get_reference_airspeed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def reference_airspeed(self) -> float:
        """Get the reference airspeed."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_reference_airspeed_metadata)

    _get_reference_airspeed_type_metadata = { "offset" : _get_reference_airspeed_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AirspeedType),) }
    @property
    def reference_airspeed_type(self) -> "AirspeedType":
        """Get the reference airspeed type."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_reference_airspeed_type_metadata)

    _set_reference_airspeed_metadata = { "offset" : _set_reference_airspeed_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(AirspeedType), agmarshall.DoubleArg,) }
    def set_reference_airspeed(self, airspeed_type:"AirspeedType", airspeed:float) -> None:
        """Set the reference airspeed and reference airspeed type."""
        return self._intf.invoke(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_reference_airspeed_metadata, airspeed_type, airspeed)

    _get_reference_load_factor_metadata = { "offset" : _get_reference_load_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def reference_load_factor(self) -> float:
        """Get or set the reference load factor."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_reference_load_factor_metadata)

    _set_reference_load_factor_metadata = { "offset" : _set_reference_load_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @reference_load_factor.setter
    def reference_load_factor(self, value:float) -> None:
        """Get or set the reference load factor."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_reference_load_factor_metadata, value)

    _get_estimated_ps_metadata = { "offset" : _get_estimated_ps_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def estimated_ps(self) -> float:
        """Get the estimated specific excess power."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_estimated_ps_metadata)

    _get_control_authority_metadata = { "offset" : _get_control_authority_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def control_authority(self) -> float:
        """Get or set the control authority of how much to factor a turn over push/pull."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_control_authority_metadata)

    _set_control_authority_metadata = { "offset" : _set_control_authority_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @control_authority.setter
    def control_authority(self, value:float) -> None:
        """Get or set the control authority of how much to factor a turn over push/pull."""
        return self._intf.set_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._set_control_authority_metadata, value)

    _get_status_message_metadata = { "offset" : _get_status_message_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def status_message(self) -> str:
        """Get the status message in the message window."""
        return self._intf.get_property(AerodynamicPropulsionManeuverModeHelper._metadata, AerodynamicPropulsionManeuverModeHelper._get_status_message_metadata)

    _property_names[mode] = "mode"
    _property_names[flight_mode] = "flight_mode"
    _property_names[use_afterburner] = "use_afterburner"
    _property_names[reference_weight] = "reference_weight"
    _property_names[reference_altitude] = "reference_altitude"
    _property_names[reference_airspeed] = "reference_airspeed"
    _property_names[reference_airspeed_type] = "reference_airspeed_type"
    _property_names[reference_load_factor] = "reference_load_factor"
    _property_names[estimated_ps] = "estimated_ps"
    _property_names[control_authority] = "control_authority"
    _property_names[status_message] = "status_message"

    def __init__(self, source_object=None):
        """Construct an object of type AerodynamicPropulsionManeuverModeHelper."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AerodynamicPropulsionManeuverModeHelper)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AerodynamicPropulsionManeuverModeHelper, [AerodynamicPropulsionManeuverModeHelper, ])

agcls.AgClassCatalog.add_catalog_entry((5554483294060101200, 8600938976610717093), AerodynamicPropulsionManeuverModeHelper)
agcls.AgTypeNameMap["AerodynamicPropulsionManeuverModeHelper"] = AerodynamicPropulsionManeuverModeHelper

class AircraftAdvancedAccelerationModel(IPerformanceModel, ICatalogItem, SupportsDeleteCallback):
    """Interface used to access the Advanced Acceleration Model options of an aircraft."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_level_turns_method_offset = 1
    _get_attitude_transitions_method_offset = 2
    _get_climb_and_descent_transitions_method_offset = 3
    _get_acceleration_mode_method_offset = 4
    _get_as_catalog_item_method_offset = 5
    _metadata = {
        "iid_data" : (5313840907341593770, 18068732340055302845),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAdvancedAccelerationModel)

    _get_level_turns_metadata = { "offset" : _get_level_turns_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def level_turns(self) -> "LevelTurns":
        """Get the level turns interface."""
        return self._intf.get_property(AircraftAdvancedAccelerationModel._metadata, AircraftAdvancedAccelerationModel._get_level_turns_metadata)

    _get_attitude_transitions_metadata = { "offset" : _get_attitude_transitions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def attitude_transitions(self) -> "AttitudeTransitions":
        """Get the attitude transitions interface."""
        return self._intf.get_property(AircraftAdvancedAccelerationModel._metadata, AircraftAdvancedAccelerationModel._get_attitude_transitions_metadata)

    _get_climb_and_descent_transitions_metadata = { "offset" : _get_climb_and_descent_transitions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def climb_and_descent_transitions(self) -> "ClimbAndDescentTransitions":
        """Get the climb and descent transitions interface."""
        return self._intf.get_property(AircraftAdvancedAccelerationModel._metadata, AircraftAdvancedAccelerationModel._get_climb_and_descent_transitions_metadata)

    _get_acceleration_mode_metadata = { "offset" : _get_acceleration_mode_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def acceleration_mode(self) -> "AircraftAccelerationMode":
        """Get the acceleration mode interface."""
        return self._intf.get_property(AircraftAdvancedAccelerationModel._metadata, AircraftAdvancedAccelerationModel._get_acceleration_mode_metadata)

    _get_as_catalog_item_metadata = { "offset" : _get_as_catalog_item_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_item(self) -> "ICatalogItem":
        """Get the catalog item interface for this object."""
        return self._intf.invoke(AircraftAdvancedAccelerationModel._metadata, AircraftAdvancedAccelerationModel._get_as_catalog_item_metadata, OutArg())

    _property_names[level_turns] = "level_turns"
    _property_names[attitude_transitions] = "attitude_transitions"
    _property_names[climb_and_descent_transitions] = "climb_and_descent_transitions"
    _property_names[acceleration_mode] = "acceleration_mode"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAdvancedAccelerationModel."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAdvancedAccelerationModel)
        IPerformanceModel.__init__(self, source_object)
        ICatalogItem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPerformanceModel._private_init(self, intf)
        ICatalogItem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAdvancedAccelerationModel, [AircraftAdvancedAccelerationModel, IPerformanceModel, ICatalogItem])

agcls.AgClassCatalog.add_catalog_entry((5358492534817326369, 10261995905994973359), AircraftAdvancedAccelerationModel)
agcls.AgTypeNameMap["AircraftAdvancedAccelerationModel"] = AircraftAdvancedAccelerationModel

class AircraftAccelerationMode(SupportsDeleteCallback):
    """Interface used to set the Acceleration Mode for the Advanced Acceleration Model of an aircraft."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_acceleration_mode_method_offset = 1
    _set_acceleration_mode_method_offset = 2
    _get_acceleration_g_method_offset = 3
    _set_acceleration_g_method_offset = 4
    _metadata = {
        "iid_data" : (5657217427469576289, 12897697484585064127),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAccelerationMode)

    _get_acceleration_mode_metadata = { "offset" : _get_acceleration_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccelerationAdvancedAccelerationMode),) }
    @property
    def acceleration_mode(self) -> "AccelerationAdvancedAccelerationMode":
        """Opt whether to override the acceleration or deceleration of the aircraft."""
        return self._intf.get_property(AircraftAccelerationMode._metadata, AircraftAccelerationMode._get_acceleration_mode_metadata)

    _set_acceleration_mode_metadata = { "offset" : _set_acceleration_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccelerationAdvancedAccelerationMode),) }
    @acceleration_mode.setter
    def acceleration_mode(self, value:"AccelerationAdvancedAccelerationMode") -> None:
        """Opt whether to override the acceleration or deceleration of the aircraft."""
        return self._intf.set_property(AircraftAccelerationMode._metadata, AircraftAccelerationMode._set_acceleration_mode_metadata, value)

    _get_acceleration_g_metadata = { "offset" : _get_acceleration_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def acceleration_g(self) -> float:
        """Get or set the rate of acceleration or deceleration of the aircraft if the accleeration mode is set to override."""
        return self._intf.get_property(AircraftAccelerationMode._metadata, AircraftAccelerationMode._get_acceleration_g_metadata)

    _set_acceleration_g_metadata = { "offset" : _set_acceleration_g_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @acceleration_g.setter
    def acceleration_g(self, value:float) -> None:
        """Get or set the rate of acceleration or deceleration of the aircraft if the accleeration mode is set to override."""
        return self._intf.set_property(AircraftAccelerationMode._metadata, AircraftAccelerationMode._set_acceleration_g_metadata, value)

    _property_names[acceleration_mode] = "acceleration_mode"
    _property_names[acceleration_g] = "acceleration_g"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAccelerationMode."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAccelerationMode)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAccelerationMode, [AircraftAccelerationMode, ])

agcls.AgClassCatalog.add_catalog_entry((5216102563964361198, 17187029792935920802), AircraftAccelerationMode)
agcls.AgTypeNameMap["AircraftAccelerationMode"] = AircraftAccelerationMode

class AircraftSimpleAerodynamic(SupportsDeleteCallback):
    """
    Interface used to access the Simple Aerodynamics options for the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_operating_mode_method_offset = 1
    _set_operating_mode_method_offset = 2
    _get_s_reference_method_offset = 3
    _set_s_reference_method_offset = 4
    _get_cl_max_method_offset = 5
    _set_cl_max_method_offset = 6
    _get_cd_method_offset = 7
    _set_cd_method_offset = 8
    _metadata = {
        "iid_data" : (4795364640067016059, 13915158584522730163),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftSimpleAerodynamic)

    _get_operating_mode_metadata = { "offset" : _get_operating_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AerodynamicPropulsionSimpleMode),) }
    @property
    def operating_mode(self) -> "AerodynamicPropulsionSimpleMode":
        """Get or set the mode option to treat the aircraft as a helicopter or a fixed wing aircraft when calculating the aircraft's attitude."""
        return self._intf.get_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._get_operating_mode_metadata)

    _set_operating_mode_metadata = { "offset" : _set_operating_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AerodynamicPropulsionSimpleMode),) }
    @operating_mode.setter
    def operating_mode(self, value:"AerodynamicPropulsionSimpleMode") -> None:
        """Get or set the mode option to treat the aircraft as a helicopter or a fixed wing aircraft when calculating the aircraft's attitude."""
        return self._intf.set_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._set_operating_mode_metadata, value)

    _get_s_reference_metadata = { "offset" : _get_s_reference_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def s_reference(self) -> float:
        """Get or set the reference surface area of the aircraft."""
        return self._intf.get_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._get_s_reference_metadata)

    _set_s_reference_metadata = { "offset" : _set_s_reference_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @s_reference.setter
    def s_reference(self, value:float) -> None:
        """Get or set the reference surface area of the aircraft."""
        return self._intf.set_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._set_s_reference_metadata, value)

    _get_cl_max_metadata = { "offset" : _get_cl_max_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cl_max(self) -> float:
        """Get or set the max coefficient of lift."""
        return self._intf.get_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._get_cl_max_metadata)

    _set_cl_max_metadata = { "offset" : _set_cl_max_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cl_max.setter
    def cl_max(self, value:float) -> None:
        """Get or set the max coefficient of lift."""
        return self._intf.set_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._set_cl_max_metadata, value)

    _get_cd_metadata = { "offset" : _get_cd_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cd(self) -> float:
        """Get or set the coefficient of drag."""
        return self._intf.get_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._get_cd_metadata)

    _set_cd_metadata = { "offset" : _set_cd_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cd.setter
    def cd(self, value:float) -> None:
        """Get or set the coefficient of drag."""
        return self._intf.set_property(AircraftSimpleAerodynamic._metadata, AircraftSimpleAerodynamic._set_cd_metadata, value)

    _property_names[operating_mode] = "operating_mode"
    _property_names[s_reference] = "s_reference"
    _property_names[cl_max] = "cl_max"
    _property_names[cd] = "cd"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftSimpleAerodynamic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftSimpleAerodynamic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftSimpleAerodynamic, [AircraftSimpleAerodynamic, ])

agcls.AgClassCatalog.add_catalog_entry((4949274632612681163, 16986168694659548566), AircraftSimpleAerodynamic)
agcls.AgTypeNameMap["AircraftSimpleAerodynamic"] = AircraftSimpleAerodynamic

class AircraftExternalAerodynamic(SupportsDeleteCallback):
    """Interface used to access the External File Aerodynamics options for the Basic Acceleration Model of an aircraft."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_forward_flight_filepath_method_offset = 1
    _set_forward_flight_filepath_method_offset = 2
    _reload_forward_flight_file_method_offset = 3
    _get_forward_flight_reference_area_method_offset = 4
    _set_forward_flight_reference_area_method_offset = 5
    _get_can_set_forward_flight_reference_area_method_offset = 6
    _get_is_forward_flight_valid_method_offset = 7
    _get_takeoff_landing_filepath_method_offset = 8
    _set_takeoff_landing_filepath_method_offset = 9
    _reload_takeoff_landing_file_method_offset = 10
    _get_takeoff_landing_reference_area_method_offset = 11
    _set_takeoff_landing_reference_area_method_offset = 12
    _get_can_set_takeoff_landing_reference_area_method_offset = 13
    _get_is_takeoff_landing_valid_method_offset = 14
    _metadata = {
        "iid_data" : (4995292197392587210, 1860972800833287599),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftExternalAerodynamic)

    _get_forward_flight_filepath_metadata = { "offset" : _get_forward_flight_filepath_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def forward_flight_filepath(self) -> str:
        """Get the filepath for the forward flight aero file."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_forward_flight_filepath_metadata)

    _set_forward_flight_filepath_metadata = { "offset" : _set_forward_flight_filepath_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set_forward_flight_filepath(self, filepath:str) -> str:
        """Set the filepath for the forward flight aero file."""
        return self._intf.invoke(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._set_forward_flight_filepath_metadata, filepath, OutArg())

    _reload_forward_flight_file_metadata = { "offset" : _reload_forward_flight_file_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    def reload_forward_flight_file(self) -> str:
        """Reload the forward flight aero file."""
        return self._intf.invoke(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._reload_forward_flight_file_metadata, OutArg())

    _get_forward_flight_reference_area_metadata = { "offset" : _get_forward_flight_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_reference_area(self) -> float:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_forward_flight_reference_area_metadata)

    _set_forward_flight_reference_area_metadata = { "offset" : _set_forward_flight_reference_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_reference_area.setter
    def forward_flight_reference_area(self, value:float) -> None:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.set_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._set_forward_flight_reference_area_metadata, value)

    _get_can_set_forward_flight_reference_area_metadata = { "offset" : _get_can_set_forward_flight_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def can_set_forward_flight_reference_area(self) -> bool:
        """Check whether you can set the reference area or whether it is specified in the file."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_can_set_forward_flight_reference_area_metadata)

    _get_is_forward_flight_valid_metadata = { "offset" : _get_is_forward_flight_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_forward_flight_valid(self) -> bool:
        """Check whether the forward flight file is valid."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_is_forward_flight_valid_metadata)

    _get_takeoff_landing_filepath_metadata = { "offset" : _get_takeoff_landing_filepath_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def takeoff_landing_filepath(self) -> str:
        """Get the filepath for the takeoff and landing aero file."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_takeoff_landing_filepath_metadata)

    _set_takeoff_landing_filepath_metadata = { "offset" : _set_takeoff_landing_filepath_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set_takeoff_landing_filepath(self, filepath:str) -> str:
        """Set the filepath for the takeoff and landing aero file."""
        return self._intf.invoke(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._set_takeoff_landing_filepath_metadata, filepath, OutArg())

    _reload_takeoff_landing_file_metadata = { "offset" : _reload_takeoff_landing_file_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    def reload_takeoff_landing_file(self) -> str:
        """Reload the takeoff and landing aero file."""
        return self._intf.invoke(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._reload_takeoff_landing_file_metadata, OutArg())

    _get_takeoff_landing_reference_area_metadata = { "offset" : _get_takeoff_landing_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_reference_area(self) -> float:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_takeoff_landing_reference_area_metadata)

    _set_takeoff_landing_reference_area_metadata = { "offset" : _set_takeoff_landing_reference_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_reference_area.setter
    def takeoff_landing_reference_area(self, value:float) -> None:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.set_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._set_takeoff_landing_reference_area_metadata, value)

    _get_can_set_takeoff_landing_reference_area_metadata = { "offset" : _get_can_set_takeoff_landing_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def can_set_takeoff_landing_reference_area(self) -> bool:
        """Check whether you can set the reference area or whether it is specified in the file."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_can_set_takeoff_landing_reference_area_metadata)

    _get_is_takeoff_landing_valid_metadata = { "offset" : _get_is_takeoff_landing_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_takeoff_landing_valid(self) -> bool:
        """Check whether the takeoff and landing file is valid."""
        return self._intf.get_property(AircraftExternalAerodynamic._metadata, AircraftExternalAerodynamic._get_is_takeoff_landing_valid_metadata)

    _property_names[forward_flight_filepath] = "forward_flight_filepath"
    _property_names[forward_flight_reference_area] = "forward_flight_reference_area"
    _property_names[can_set_forward_flight_reference_area] = "can_set_forward_flight_reference_area"
    _property_names[is_forward_flight_valid] = "is_forward_flight_valid"
    _property_names[takeoff_landing_filepath] = "takeoff_landing_filepath"
    _property_names[takeoff_landing_reference_area] = "takeoff_landing_reference_area"
    _property_names[can_set_takeoff_landing_reference_area] = "can_set_takeoff_landing_reference_area"
    _property_names[is_takeoff_landing_valid] = "is_takeoff_landing_valid"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftExternalAerodynamic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftExternalAerodynamic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftExternalAerodynamic, [AircraftExternalAerodynamic, ])

agcls.AgClassCatalog.add_catalog_entry((5605284744680812914, 13571106235899430788), AircraftExternalAerodynamic)
agcls.AgTypeNameMap["AircraftExternalAerodynamic"] = AircraftExternalAerodynamic

class AircraftAerodynamic(SupportsDeleteCallback):
    """
    Interface used to access the Aerodynamics options for the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_aerodynamic_strategy_method_offset = 1
    _set_aerodynamic_strategy_method_offset = 2
    _get_mode_as_simple_method_offset = 3
    _get_mode_as_basic_fixed_wing_method_offset = 4
    _get_mode_as_external_method_offset = 5
    _get_mode_as_advanced_missile_method_offset = 6
    _get_lift_factor_method_offset = 7
    _set_lift_factor_method_offset = 8
    _get_drag_factor_method_offset = 9
    _set_drag_factor_method_offset = 10
    _get_mode_as_four_point_method_offset = 11
    _metadata = {
        "iid_data" : (5120934689185363706, 8628502199224761519),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftAerodynamic)

    _get_aerodynamic_strategy_metadata = { "offset" : _get_aerodynamic_strategy_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AircraftAerodynamicStrategy),) }
    @property
    def aerodynamic_strategy(self) -> "AircraftAerodynamicStrategy":
        """Get or set the aerodynamic strategy type."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_aerodynamic_strategy_metadata)

    _set_aerodynamic_strategy_metadata = { "offset" : _set_aerodynamic_strategy_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AircraftAerodynamicStrategy),) }
    @aerodynamic_strategy.setter
    def aerodynamic_strategy(self, value:"AircraftAerodynamicStrategy") -> None:
        """Get or set the aerodynamic strategy type."""
        return self._intf.set_property(AircraftAerodynamic._metadata, AircraftAerodynamic._set_aerodynamic_strategy_metadata, value)

    _get_mode_as_simple_metadata = { "offset" : _get_mode_as_simple_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_simple(self) -> "AircraftSimpleAerodynamic":
        """Get the interface for a simple aerodynamics strategy."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_mode_as_simple_metadata)

    _get_mode_as_basic_fixed_wing_metadata = { "offset" : _get_mode_as_basic_fixed_wing_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_basic_fixed_wing(self) -> "AircraftBasicFixedWingAerodynamic":
        """Get the interface for a basic fixed wing aerodynamics strategy."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_mode_as_basic_fixed_wing_metadata)

    _get_mode_as_external_metadata = { "offset" : _get_mode_as_external_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_external(self) -> "AircraftExternalAerodynamic":
        """Get the interface for an external file aerodynamics strategy."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_mode_as_external_metadata)

    _get_mode_as_advanced_missile_metadata = { "offset" : _get_mode_as_advanced_missile_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_advanced_missile(self) -> "MissileAdvancedAerodynamic":
        """Get the interface for an advanced missile aerodynamics strategy."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_mode_as_advanced_missile_metadata)

    _get_lift_factor_metadata = { "offset" : _get_lift_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def lift_factor(self) -> float:
        """Get or set the scalar value applied to the lift for parametric analysis."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_lift_factor_metadata)

    _set_lift_factor_metadata = { "offset" : _set_lift_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @lift_factor.setter
    def lift_factor(self, value:float) -> None:
        """Get or set the scalar value applied to the lift for parametric analysis."""
        return self._intf.set_property(AircraftAerodynamic._metadata, AircraftAerodynamic._set_lift_factor_metadata, value)

    _get_drag_factor_metadata = { "offset" : _get_drag_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def drag_factor(self) -> float:
        """Get or set the scalar value applied to the drag for parametric analysis."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_drag_factor_metadata)

    _set_drag_factor_metadata = { "offset" : _set_drag_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @drag_factor.setter
    def drag_factor(self, value:float) -> None:
        """Get or set the scalar value applied to the drag for parametric analysis."""
        return self._intf.set_property(AircraftAerodynamic._metadata, AircraftAerodynamic._set_drag_factor_metadata, value)

    _get_mode_as_four_point_metadata = { "offset" : _get_mode_as_four_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_four_point(self) -> "FourPointAerodynamic":
        """Get the interface for a four point aerodynamics strategy."""
        return self._intf.get_property(AircraftAerodynamic._metadata, AircraftAerodynamic._get_mode_as_four_point_metadata)

    _property_names[aerodynamic_strategy] = "aerodynamic_strategy"
    _property_names[mode_as_simple] = "mode_as_simple"
    _property_names[mode_as_basic_fixed_wing] = "mode_as_basic_fixed_wing"
    _property_names[mode_as_external] = "mode_as_external"
    _property_names[mode_as_advanced_missile] = "mode_as_advanced_missile"
    _property_names[lift_factor] = "lift_factor"
    _property_names[drag_factor] = "drag_factor"
    _property_names[mode_as_four_point] = "mode_as_four_point"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftAerodynamic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftAerodynamic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftAerodynamic, [AircraftAerodynamic, ])

agcls.AgClassCatalog.add_catalog_entry((5082480556701513469, 17564547437981405094), AircraftAerodynamic)
agcls.AgTypeNameMap["AircraftAerodynamic"] = AircraftAerodynamic

class AircraftBasicFixedWingAerodynamic(SupportsDeleteCallback):
    """Interface used to access Basic Fixed Wing Aerodynamics interface for the Basic Acceleration Model of an aircraft."""

    _num_methods = 32
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_forward_flight_reference_area_method_offset = 1
    _set_forward_flight_reference_area_method_offset = 2
    _get_forward_flight_use_compressible_flow_method_offset = 3
    _set_forward_flight_use_compressible_flow_method_offset = 4
    _get_forward_flight_cl0_method_offset = 5
    _set_forward_flight_cl0_method_offset = 6
    _get_forward_flight_cl_alpha_method_offset = 7
    _set_forward_flight_cl_alpha_method_offset = 8
    _get_forward_flight_min_aoa_method_offset = 9
    _set_forward_flight_min_aoa_method_offset = 10
    _get_forward_flight_max_aoa_method_offset = 11
    _set_forward_flight_max_aoa_method_offset = 12
    _get_forward_flight_cd0_method_offset = 13
    _set_forward_flight_cd0_method_offset = 14
    _get_forward_flight_k_method_offset = 15
    _set_forward_flight_k_method_offset = 16
    _get_takeoff_landing_reference_area_method_offset = 17
    _set_takeoff_landing_reference_area_method_offset = 18
    _get_takeoff_landing_use_compressible_flow_method_offset = 19
    _set_takeoff_landing_use_compressible_flow_method_offset = 20
    _get_takeoff_landing_cl0_method_offset = 21
    _set_takeoff_landing_cl0_method_offset = 22
    _get_takeoff_landing_cl_alpha_method_offset = 23
    _set_takeoff_landing_cl_alpha_method_offset = 24
    _get_takeoff_landing_min_aoa_method_offset = 25
    _set_takeoff_landing_min_aoa_method_offset = 26
    _get_takeoff_landing_max_aoa_method_offset = 27
    _set_takeoff_landing_max_aoa_method_offset = 28
    _get_takeoff_landing_cd0_method_offset = 29
    _set_takeoff_landing_cd0_method_offset = 30
    _get_takeoff_landing_k_method_offset = 31
    _set_takeoff_landing_k_method_offset = 32
    _metadata = {
        "iid_data" : (5194727978343587927, 16517105265377487783),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicFixedWingAerodynamic)

    _get_forward_flight_reference_area_metadata = { "offset" : _get_forward_flight_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_reference_area(self) -> float:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_reference_area_metadata)

    _set_forward_flight_reference_area_metadata = { "offset" : _set_forward_flight_reference_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_reference_area.setter
    def forward_flight_reference_area(self, value:float) -> None:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_reference_area_metadata, value)

    _get_forward_flight_use_compressible_flow_metadata = { "offset" : _get_forward_flight_use_compressible_flow_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def forward_flight_use_compressible_flow(self) -> bool:
        """Opt to define the aerodynamic parameters for forward flight with respect to supersonic flight conditions."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_use_compressible_flow_metadata)

    _set_forward_flight_use_compressible_flow_metadata = { "offset" : _set_forward_flight_use_compressible_flow_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @forward_flight_use_compressible_flow.setter
    def forward_flight_use_compressible_flow(self, value:bool) -> None:
        """Opt to define the aerodynamic parameters for forward flight with respect to supersonic flight conditions."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_use_compressible_flow_metadata, value)

    _get_forward_flight_cl0_metadata = { "offset" : _get_forward_flight_cl0_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_cl0(self) -> float:
        """Get or set the coefficient of lift at zero angle of attack."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_cl0_metadata)

    _set_forward_flight_cl0_metadata = { "offset" : _set_forward_flight_cl0_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_cl0.setter
    def forward_flight_cl0(self, value:float) -> None:
        """Get or set the coefficient of lift at zero angle of attack."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_cl0_metadata, value)

    _get_forward_flight_cl_alpha_metadata = { "offset" : _get_forward_flight_cl_alpha_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_cl_alpha(self) -> float:
        """Get or set the slope of the coefficient of lift curve."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_cl_alpha_metadata)

    _set_forward_flight_cl_alpha_metadata = { "offset" : _set_forward_flight_cl_alpha_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_cl_alpha.setter
    def forward_flight_cl_alpha(self, value:float) -> None:
        """Get or set the slope of the coefficient of lift curve."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_cl_alpha_metadata, value)

    _get_forward_flight_min_aoa_metadata = { "offset" : _get_forward_flight_min_aoa_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def forward_flight_min_aoa(self) -> typing.Any:
        """Get or set the minimum angle of attack possible."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_min_aoa_metadata)

    _set_forward_flight_min_aoa_metadata = { "offset" : _set_forward_flight_min_aoa_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @forward_flight_min_aoa.setter
    def forward_flight_min_aoa(self, value:typing.Any) -> None:
        """Get or set the minimum angle of attack possible."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_min_aoa_metadata, value)

    _get_forward_flight_max_aoa_metadata = { "offset" : _get_forward_flight_max_aoa_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def forward_flight_max_aoa(self) -> typing.Any:
        """Get or set the maximum angle of attack possible."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_max_aoa_metadata)

    _set_forward_flight_max_aoa_metadata = { "offset" : _set_forward_flight_max_aoa_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @forward_flight_max_aoa.setter
    def forward_flight_max_aoa(self, value:typing.Any) -> None:
        """Get or set the maximum angle of attack possible."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_max_aoa_metadata, value)

    _get_forward_flight_cd0_metadata = { "offset" : _get_forward_flight_cd0_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_cd0(self) -> float:
        """Get or set the coefficient of drag of the lifting surface at zero angle of attack."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_cd0_metadata)

    _set_forward_flight_cd0_metadata = { "offset" : _set_forward_flight_cd0_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_cd0.setter
    def forward_flight_cd0(self, value:float) -> None:
        """Get or set the coefficient of drag of the lifting surface at zero angle of attack."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_cd0_metadata, value)

    _get_forward_flight_k_metadata = { "offset" : _get_forward_flight_k_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def forward_flight_k(self) -> float:
        """Get or set the coefficient of induced drag."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_forward_flight_k_metadata)

    _set_forward_flight_k_metadata = { "offset" : _set_forward_flight_k_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @forward_flight_k.setter
    def forward_flight_k(self, value:float) -> None:
        """Get or set the coefficient of induced drag."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_forward_flight_k_metadata, value)

    _get_takeoff_landing_reference_area_metadata = { "offset" : _get_takeoff_landing_reference_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_reference_area(self) -> float:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_reference_area_metadata)

    _set_takeoff_landing_reference_area_metadata = { "offset" : _set_takeoff_landing_reference_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_reference_area.setter
    def takeoff_landing_reference_area(self, value:float) -> None:
        """Get or set the area of the lifting surface of the aircraft."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_reference_area_metadata, value)

    _get_takeoff_landing_use_compressible_flow_metadata = { "offset" : _get_takeoff_landing_use_compressible_flow_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def takeoff_landing_use_compressible_flow(self) -> bool:
        """Opt to define the aerodynamic parameters for takeoff and landing with respect to supersonic flight conditions."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_use_compressible_flow_metadata)

    _set_takeoff_landing_use_compressible_flow_metadata = { "offset" : _set_takeoff_landing_use_compressible_flow_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @takeoff_landing_use_compressible_flow.setter
    def takeoff_landing_use_compressible_flow(self, value:bool) -> None:
        """Opt to define the aerodynamic parameters for takeoff and landing with respect to supersonic flight conditions."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_use_compressible_flow_metadata, value)

    _get_takeoff_landing_cl0_metadata = { "offset" : _get_takeoff_landing_cl0_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_cl0(self) -> float:
        """Get or set the coefficient of lift at zero angle of attack."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_cl0_metadata)

    _set_takeoff_landing_cl0_metadata = { "offset" : _set_takeoff_landing_cl0_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_cl0.setter
    def takeoff_landing_cl0(self, value:float) -> None:
        """Get or set the coefficient of lift at zero angle of attack."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_cl0_metadata, value)

    _get_takeoff_landing_cl_alpha_metadata = { "offset" : _get_takeoff_landing_cl_alpha_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_cl_alpha(self) -> float:
        """Get or set the slope of the coefficient of lift curve."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_cl_alpha_metadata)

    _set_takeoff_landing_cl_alpha_metadata = { "offset" : _set_takeoff_landing_cl_alpha_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_cl_alpha.setter
    def takeoff_landing_cl_alpha(self, value:float) -> None:
        """Get or set the slope of the coefficient of lift curve."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_cl_alpha_metadata, value)

    _get_takeoff_landing_min_aoa_metadata = { "offset" : _get_takeoff_landing_min_aoa_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def takeoff_landing_min_aoa(self) -> typing.Any:
        """Get or set the minimum angle of attack possible."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_min_aoa_metadata)

    _set_takeoff_landing_min_aoa_metadata = { "offset" : _set_takeoff_landing_min_aoa_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @takeoff_landing_min_aoa.setter
    def takeoff_landing_min_aoa(self, value:typing.Any) -> None:
        """Get or set the minimum angle of attack possible."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_min_aoa_metadata, value)

    _get_takeoff_landing_max_aoa_metadata = { "offset" : _get_takeoff_landing_max_aoa_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def takeoff_landing_max_aoa(self) -> typing.Any:
        """Get or set the maximum angle of attack possible."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_max_aoa_metadata)

    _set_takeoff_landing_max_aoa_metadata = { "offset" : _set_takeoff_landing_max_aoa_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @takeoff_landing_max_aoa.setter
    def takeoff_landing_max_aoa(self, value:typing.Any) -> None:
        """Get or set the maximum angle of attack possible."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_max_aoa_metadata, value)

    _get_takeoff_landing_cd0_metadata = { "offset" : _get_takeoff_landing_cd0_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_cd0(self) -> float:
        """Get or set the coefficient of drag of the lifting surface at zero angle of attack."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_cd0_metadata)

    _set_takeoff_landing_cd0_metadata = { "offset" : _set_takeoff_landing_cd0_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_cd0.setter
    def takeoff_landing_cd0(self, value:float) -> None:
        """Get or set the coefficient of drag of the lifting surface at zero angle of attack."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_cd0_metadata, value)

    _get_takeoff_landing_k_metadata = { "offset" : _get_takeoff_landing_k_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def takeoff_landing_k(self) -> float:
        """Get or set the coefficient of induced drag."""
        return self._intf.get_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._get_takeoff_landing_k_metadata)

    _set_takeoff_landing_k_metadata = { "offset" : _set_takeoff_landing_k_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @takeoff_landing_k.setter
    def takeoff_landing_k(self, value:float) -> None:
        """Get or set the coefficient of induced drag."""
        return self._intf.set_property(AircraftBasicFixedWingAerodynamic._metadata, AircraftBasicFixedWingAerodynamic._set_takeoff_landing_k_metadata, value)

    _property_names[forward_flight_reference_area] = "forward_flight_reference_area"
    _property_names[forward_flight_use_compressible_flow] = "forward_flight_use_compressible_flow"
    _property_names[forward_flight_cl0] = "forward_flight_cl0"
    _property_names[forward_flight_cl_alpha] = "forward_flight_cl_alpha"
    _property_names[forward_flight_min_aoa] = "forward_flight_min_aoa"
    _property_names[forward_flight_max_aoa] = "forward_flight_max_aoa"
    _property_names[forward_flight_cd0] = "forward_flight_cd0"
    _property_names[forward_flight_k] = "forward_flight_k"
    _property_names[takeoff_landing_reference_area] = "takeoff_landing_reference_area"
    _property_names[takeoff_landing_use_compressible_flow] = "takeoff_landing_use_compressible_flow"
    _property_names[takeoff_landing_cl0] = "takeoff_landing_cl0"
    _property_names[takeoff_landing_cl_alpha] = "takeoff_landing_cl_alpha"
    _property_names[takeoff_landing_min_aoa] = "takeoff_landing_min_aoa"
    _property_names[takeoff_landing_max_aoa] = "takeoff_landing_max_aoa"
    _property_names[takeoff_landing_cd0] = "takeoff_landing_cd0"
    _property_names[takeoff_landing_k] = "takeoff_landing_k"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicFixedWingAerodynamic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicFixedWingAerodynamic)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicFixedWingAerodynamic, [AircraftBasicFixedWingAerodynamic, ])

agcls.AgClassCatalog.add_catalog_entry((4615386614829789996, 13505783508724948142), AircraftBasicFixedWingAerodynamic)
agcls.AgTypeNameMap["AircraftBasicFixedWingAerodynamic"] = AircraftBasicFixedWingAerodynamic

class AircraftPropulsion(SupportsDeleteCallback):
    """
    Interface used to access the propulsion options for the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_propulsion_strategy_method_offset = 1
    _set_propulsion_strategy_method_offset = 2
    _get_mode_as_simple_method_offset = 3
    _get_mode_as_basic_fixed_wing_method_offset = 4
    _get_mode_as_external_method_offset = 5
    _get_lift_factor_method_offset = 6
    _set_lift_factor_method_offset = 7
    _get_drag_factor_method_offset = 8
    _set_drag_factor_method_offset = 9
    _get_mode_as_ramjet_method_offset = 10
    _get_mode_as_turbojet_method_offset = 11
    _get_mode_as_rocket_method_offset = 12
    _metadata = {
        "iid_data" : (4760545169458717566, 10855594368442160788),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftPropulsion)

    _get_propulsion_strategy_metadata = { "offset" : _get_propulsion_strategy_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AircraftPropulsionStrategy),) }
    @property
    def propulsion_strategy(self) -> "AircraftPropulsionStrategy":
        """Get or set the propulsion strategy type."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_propulsion_strategy_metadata)

    _set_propulsion_strategy_metadata = { "offset" : _set_propulsion_strategy_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AircraftPropulsionStrategy),) }
    @propulsion_strategy.setter
    def propulsion_strategy(self, value:"AircraftPropulsionStrategy") -> None:
        """Get or set the propulsion strategy type."""
        return self._intf.set_property(AircraftPropulsion._metadata, AircraftPropulsion._set_propulsion_strategy_metadata, value)

    _get_mode_as_simple_metadata = { "offset" : _get_mode_as_simple_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_simple(self) -> "AircraftSimplePropulsion":
        """Get the interface for a simple propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_simple_metadata)

    _get_mode_as_basic_fixed_wing_metadata = { "offset" : _get_mode_as_basic_fixed_wing_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_basic_fixed_wing(self) -> "AircraftBasicFixedWingPropulsion":
        """Get the interface for a basic fixed wing propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_basic_fixed_wing_metadata)

    _get_mode_as_external_metadata = { "offset" : _get_mode_as_external_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_external(self) -> "AircraftExternalPropulsion":
        """Get the interface for an external file propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_external_metadata)

    _get_lift_factor_metadata = { "offset" : _get_lift_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def lift_factor(self) -> float:
        """Get or set the scalar value applied to the lift for parametric analysis."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_lift_factor_metadata)

    _set_lift_factor_metadata = { "offset" : _set_lift_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @lift_factor.setter
    def lift_factor(self, value:float) -> None:
        """Get or set the scalar value applied to the lift for parametric analysis."""
        return self._intf.set_property(AircraftPropulsion._metadata, AircraftPropulsion._set_lift_factor_metadata, value)

    _get_drag_factor_metadata = { "offset" : _get_drag_factor_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def drag_factor(self) -> float:
        """Get or set the scalar value applied to the drag for parametric analysis."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_drag_factor_metadata)

    _set_drag_factor_metadata = { "offset" : _set_drag_factor_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @drag_factor.setter
    def drag_factor(self, value:float) -> None:
        """Get or set the scalar value applied to the drag for parametric analysis."""
        return self._intf.set_property(AircraftPropulsion._metadata, AircraftPropulsion._set_drag_factor_metadata, value)

    _get_mode_as_ramjet_metadata = { "offset" : _get_mode_as_ramjet_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_ramjet(self) -> "MissileRamjetPropulsion":
        """Get the interface for a Ramjet propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_ramjet_metadata)

    _get_mode_as_turbojet_metadata = { "offset" : _get_mode_as_turbojet_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_turbojet(self) -> "MissileTurbojetPropulsion":
        """Get the interface for a Turbojet propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_turbojet_metadata)

    _get_mode_as_rocket_metadata = { "offset" : _get_mode_as_rocket_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def mode_as_rocket(self) -> "MissileRocketPropulsion":
        """Get the interface for a Rocket propulsion strategy."""
        return self._intf.get_property(AircraftPropulsion._metadata, AircraftPropulsion._get_mode_as_rocket_metadata)

    _property_names[propulsion_strategy] = "propulsion_strategy"
    _property_names[mode_as_simple] = "mode_as_simple"
    _property_names[mode_as_basic_fixed_wing] = "mode_as_basic_fixed_wing"
    _property_names[mode_as_external] = "mode_as_external"
    _property_names[lift_factor] = "lift_factor"
    _property_names[drag_factor] = "drag_factor"
    _property_names[mode_as_ramjet] = "mode_as_ramjet"
    _property_names[mode_as_turbojet] = "mode_as_turbojet"
    _property_names[mode_as_rocket] = "mode_as_rocket"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftPropulsion."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftPropulsion)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftPropulsion, [AircraftPropulsion, ])

agcls.AgClassCatalog.add_catalog_entry((5577330394102227800, 15792983096793871031), AircraftPropulsion)
agcls.AgTypeNameMap["AircraftPropulsion"] = AircraftPropulsion

class AircraftSimplePropulsion(SupportsDeleteCallback):
    """
    Interface used to access the Simple Propulsion options for the Basic Acceleration Model of an aircraft.

    Examples
    --------
    Configure the basic acceleration performance model of an aircraft:
    >>> # AircraftModel aviatorAircraft: Aviator Aircraft object
    >>> # Get the acceleration type
    >>> acceleration = aviatorAircraft.acceleration
    >>> # Get the build in performance model
    >>> basicAccModel = acceleration.get_built_in_model()
    >>>
    >>> # Get the level turns options
    >>> levelTurns = basicAccModel.level_turns
    >>> # Set a max bank angle of 25
    >>> levelTurns.set_level_turn(TurnMode.TURN_MODE_BANK_ANGLE, 25)
    >>> # Get the climb and descent transition options
    >>> climbAndDescent = basicAccModel.climb_and_descent_transitions
    >>> # Set the max pull up G to 1
    >>> climbAndDescent.max_pull_up_g = 1.2
    >>> # Get the attitude transition options
    >>> attitudeTransitions = basicAccModel.attitude_transitions
    >>> # Set the max roll rate to 25
    >>> attitudeTransitions.roll_rate = 25
    >>>
    >>> # Get the aerodynamics
    >>> aero = basicAccModel.aerodynamics
    >>> # Use simple aerodynamics
    >>> aero.aerodynamic_strategy = AircraftAerodynamicStrategy.AIRCRAFT_AERODYNAMIC_SIMPLE
    >>> # Get the options for the simple aerodynamics and set some parameters
    >>> simpleAero = aero.mode_as_simple
    >>> simpleAero.s_reference = 5
    >>> simpleAero.cl_max = 3.1
    >>> simpleAero.cd = 0.05
    >>>
    >>> # Get the propulsion
    >>> prop = basicAccModel.propulsion
    >>> # Use simple propulsion
    >>> prop.propulsion_strategy = AircraftPropulsionStrategy.AIRCRAFT_PROPULSION_SIMPLE
    >>> # Get the simple propulsion options and set some parameters
    >>> simpleProp = prop.mode_as_simple
    >>> simpleProp.max_thrust_acceleration = 0.6
    >>> simpleProp.min_thrust_deceleration = 0.4
    >>> simpleProp.set_density_scaling(True, 0.02)
    >>>
    >>> # Save the changes to the catalog
    >>> aviatorAircraft.save()
    """

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_max_thrust_acceleration_method_offset = 1
    _set_max_thrust_acceleration_method_offset = 2
    _get_min_thrust_deceleration_method_offset = 3
    _set_min_thrust_deceleration_method_offset = 4
    _get_use_density_scaling_method_offset = 5
    _get_density_ratio_exponent_method_offset = 6
    _set_density_scaling_method_offset = 7
    _metadata = {
        "iid_data" : (5408389755499298173, 7791957774431135407),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftSimplePropulsion)

    _get_max_thrust_acceleration_metadata = { "offset" : _get_max_thrust_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_thrust_acceleration(self) -> float:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.get_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._get_max_thrust_acceleration_metadata)

    _set_max_thrust_acceleration_metadata = { "offset" : _set_max_thrust_acceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_thrust_acceleration.setter
    def max_thrust_acceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.set_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._set_max_thrust_acceleration_metadata, value)

    _get_min_thrust_deceleration_metadata = { "offset" : _get_min_thrust_deceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_thrust_deceleration(self) -> float:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.get_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._get_min_thrust_deceleration_metadata)

    _set_min_thrust_deceleration_metadata = { "offset" : _set_min_thrust_deceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_thrust_deceleration.setter
    def min_thrust_deceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.set_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._set_min_thrust_deceleration_metadata, value)

    _get_use_density_scaling_metadata = { "offset" : _get_use_density_scaling_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_density_scaling(self) -> bool:
        """Opt whether to scale the accel/decel performance by the density ratio."""
        return self._intf.get_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._get_use_density_scaling_metadata)

    _get_density_ratio_exponent_metadata = { "offset" : _get_density_ratio_exponent_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def density_ratio_exponent(self) -> float:
        """Get the relative impace of atmospheric density on the aircraft's performance."""
        return self._intf.get_property(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._get_density_ratio_exponent_metadata)

    _set_density_scaling_metadata = { "offset" : _set_density_scaling_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL, agcom.DOUBLE,),
            "marshallers" : (agmarshall.VariantBoolArg, agmarshall.DoubleArg,) }
    def set_density_scaling(self, use_scaling:bool, exponent:float) -> None:
        """Set the option to use density scaling and set the density ratio exponent."""
        return self._intf.invoke(AircraftSimplePropulsion._metadata, AircraftSimplePropulsion._set_density_scaling_metadata, use_scaling, exponent)

    _property_names[max_thrust_acceleration] = "max_thrust_acceleration"
    _property_names[min_thrust_deceleration] = "min_thrust_deceleration"
    _property_names[use_density_scaling] = "use_density_scaling"
    _property_names[density_ratio_exponent] = "density_ratio_exponent"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftSimplePropulsion."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftSimplePropulsion)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftSimplePropulsion, [AircraftSimplePropulsion, ])

agcls.AgClassCatalog.add_catalog_entry((5707285421522482526, 1559585226005544595), AircraftSimplePropulsion)
agcls.AgTypeNameMap["AircraftSimplePropulsion"] = AircraftSimplePropulsion

class AircraftExternalPropulsion(SupportsDeleteCallback):
    """Interface used to access the External File Propulsion options for the Basic Acceleration Model of an aircraft."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_propulsion_filepath_method_offset = 1
    _set_propulsion_filepath_method_offset = 2
    _reload_propulsion_file_method_offset = 3
    _get_is_valid_method_offset = 4
    _get_can_set_acceleration_deceleration_method_offset = 5
    _get_max_thrust_acceleration_method_offset = 6
    _set_max_thrust_acceleration_method_offset = 7
    _get_min_thrust_deceleration_method_offset = 8
    _set_min_thrust_deceleration_method_offset = 9
    _get_use_density_scaling_method_offset = 10
    _get_density_ratio_exponent_method_offset = 11
    _set_density_scaling_method_offset = 12
    _metadata = {
        "iid_data" : (4857533303974722191, 17014078191631793548),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftExternalPropulsion)

    _get_propulsion_filepath_metadata = { "offset" : _get_propulsion_filepath_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def propulsion_filepath(self) -> str:
        """Get the filepath for the prop file."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_propulsion_filepath_metadata)

    _set_propulsion_filepath_metadata = { "offset" : _set_propulsion_filepath_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set_propulsion_filepath(self, filepath:str) -> str:
        """Set the filepath for the prop file."""
        return self._intf.invoke(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._set_propulsion_filepath_metadata, filepath, OutArg())

    _reload_propulsion_file_metadata = { "offset" : _reload_propulsion_file_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    def reload_propulsion_file(self) -> str:
        """Reload the prop file."""
        return self._intf.invoke(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._reload_propulsion_file_metadata, OutArg())

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Check if the prop file is valid."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_is_valid_metadata)

    _get_can_set_acceleration_deceleration_metadata = { "offset" : _get_can_set_acceleration_deceleration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def can_set_acceleration_deceleration(self) -> bool:
        """Check whether you can set the acceleration and deceleration values or whether they are specified in the file."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_can_set_acceleration_deceleration_metadata)

    _get_max_thrust_acceleration_metadata = { "offset" : _get_max_thrust_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_thrust_acceleration(self) -> float:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_max_thrust_acceleration_metadata)

    _set_max_thrust_acceleration_metadata = { "offset" : _set_max_thrust_acceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_thrust_acceleration.setter
    def max_thrust_acceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.set_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._set_max_thrust_acceleration_metadata, value)

    _get_min_thrust_deceleration_metadata = { "offset" : _get_min_thrust_deceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_thrust_deceleration(self) -> float:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_min_thrust_deceleration_metadata)

    _set_min_thrust_deceleration_metadata = { "offset" : _set_min_thrust_deceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_thrust_deceleration.setter
    def min_thrust_deceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.set_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._set_min_thrust_deceleration_metadata, value)

    _get_use_density_scaling_metadata = { "offset" : _get_use_density_scaling_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_density_scaling(self) -> bool:
        """Opt whether to scale the accel/decel performance by the density ratio."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_use_density_scaling_metadata)

    _get_density_ratio_exponent_metadata = { "offset" : _get_density_ratio_exponent_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def density_ratio_exponent(self) -> float:
        """Get the relative impace of atmospheric density on the aircraft's performance."""
        return self._intf.get_property(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._get_density_ratio_exponent_metadata)

    _set_density_scaling_metadata = { "offset" : _set_density_scaling_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL, agcom.DOUBLE,),
            "marshallers" : (agmarshall.VariantBoolArg, agmarshall.DoubleArg,) }
    def set_density_scaling(self, use_scaling:bool, exponent:float) -> None:
        """Set the option to use density scaling and set the density ratio exponent."""
        return self._intf.invoke(AircraftExternalPropulsion._metadata, AircraftExternalPropulsion._set_density_scaling_metadata, use_scaling, exponent)

    _property_names[propulsion_filepath] = "propulsion_filepath"
    _property_names[is_valid] = "is_valid"
    _property_names[can_set_acceleration_deceleration] = "can_set_acceleration_deceleration"
    _property_names[max_thrust_acceleration] = "max_thrust_acceleration"
    _property_names[min_thrust_deceleration] = "min_thrust_deceleration"
    _property_names[use_density_scaling] = "use_density_scaling"
    _property_names[density_ratio_exponent] = "density_ratio_exponent"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftExternalPropulsion."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftExternalPropulsion)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftExternalPropulsion, [AircraftExternalPropulsion, ])

agcls.AgClassCatalog.add_catalog_entry((4931266991254779167, 11152857308621733023), AircraftExternalPropulsion)
agcls.AgTypeNameMap["AircraftExternalPropulsion"] = AircraftExternalPropulsion

class AircraftBasicFixedWingPropulsion(SupportsDeleteCallback):
    """Interface used to access the Basic Fixed Wing Propulsion options for the Basic Acceleration Model of an aircraft."""

    _num_methods = 23
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_propulsion_mode_method_offset = 1
    _set_propulsion_mode_method_offset = 2
    _get_propeller_count_method_offset = 3
    _set_propeller_count_method_offset = 4
    _get_propeller_diameter_method_offset = 5
    _set_propeller_diameter_method_offset = 6
    _get_propeller_rpm_method_offset = 7
    _set_propeller_rpm_method_offset = 8
    _get_min_power_thrust_method_offset = 9
    _set_min_power_thrust_method_offset = 10
    _get_min_fuel_flow_method_offset = 11
    _set_min_fuel_flow_method_offset = 12
    _get_max_power_thrust_method_offset = 13
    _set_max_power_thrust_method_offset = 14
    _get_max_fuel_flow_method_offset = 15
    _set_max_fuel_flow_method_offset = 16
    _get_max_thrust_acceleration_method_offset = 17
    _set_max_thrust_acceleration_method_offset = 18
    _get_min_thrust_deceleration_method_offset = 19
    _set_min_thrust_deceleration_method_offset = 20
    _get_use_density_scaling_method_offset = 21
    _get_density_ratio_exponent_method_offset = 22
    _set_density_scaling_method_offset = 23
    _metadata = {
        "iid_data" : (5283772667304000625, 2502042123150193025),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AircraftBasicFixedWingPropulsion)

    _get_propulsion_mode_metadata = { "offset" : _get_propulsion_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicFixedWingPropulsionMode),) }
    @property
    def propulsion_mode(self) -> "BasicFixedWingPropulsionMode":
        """Get or set the option of whether to specify net thrust or net power."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_propulsion_mode_metadata)

    _set_propulsion_mode_metadata = { "offset" : _set_propulsion_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BasicFixedWingPropulsionMode),) }
    @propulsion_mode.setter
    def propulsion_mode(self, value:"BasicFixedWingPropulsionMode") -> None:
        """Get or set the option of whether to specify net thrust or net power."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_propulsion_mode_metadata, value)

    _get_propeller_count_metadata = { "offset" : _get_propeller_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def propeller_count(self) -> int:
        """Get or set the number of propellers."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_propeller_count_metadata)

    _set_propeller_count_metadata = { "offset" : _set_propeller_count_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @propeller_count.setter
    def propeller_count(self, value:int) -> None:
        """Get or set the number of propellers."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_propeller_count_metadata, value)

    _get_propeller_diameter_metadata = { "offset" : _get_propeller_diameter_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def propeller_diameter(self) -> float:
        """Get or set the propeller diameter."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_propeller_diameter_metadata)

    _set_propeller_diameter_metadata = { "offset" : _set_propeller_diameter_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @propeller_diameter.setter
    def propeller_diameter(self, value:float) -> None:
        """Get or set the propeller diameter."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_propeller_diameter_metadata, value)

    _get_propeller_rpm_metadata = { "offset" : _get_propeller_rpm_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def propeller_rpm(self) -> float:
        """Get or set the propeller RPM."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_propeller_rpm_metadata)

    _set_propeller_rpm_metadata = { "offset" : _set_propeller_rpm_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @propeller_rpm.setter
    def propeller_rpm(self, value:float) -> None:
        """Get or set the propeller RPM."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_propeller_rpm_metadata, value)

    _get_min_power_thrust_metadata = { "offset" : _get_min_power_thrust_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_power_thrust(self) -> float:
        """Get or set the minimum power/thrust depending on the propulsion mode."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_min_power_thrust_metadata)

    _set_min_power_thrust_metadata = { "offset" : _set_min_power_thrust_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_power_thrust.setter
    def min_power_thrust(self, value:float) -> None:
        """Get or set the minimum power/thrust depending on the propulsion mode."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_min_power_thrust_metadata, value)

    _get_min_fuel_flow_metadata = { "offset" : _get_min_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_fuel_flow(self) -> float:
        """Get or set the fuel flow for the minimum thrust/power setting."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_min_fuel_flow_metadata)

    _set_min_fuel_flow_metadata = { "offset" : _set_min_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_fuel_flow.setter
    def min_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the minimum thrust/power setting."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_min_fuel_flow_metadata, value)

    _get_max_power_thrust_metadata = { "offset" : _get_max_power_thrust_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_power_thrust(self) -> float:
        """Get or set the maximum power/thrust depending on the propulsion mode."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_max_power_thrust_metadata)

    _set_max_power_thrust_metadata = { "offset" : _set_max_power_thrust_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_power_thrust.setter
    def max_power_thrust(self, value:float) -> None:
        """Get or set the maximum power/thrust depending on the propulsion mode."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_max_power_thrust_metadata, value)

    _get_max_fuel_flow_metadata = { "offset" : _get_max_fuel_flow_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_fuel_flow(self) -> float:
        """Get or set the fuel flow for the maximum thrust/power setting."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_max_fuel_flow_metadata)

    _set_max_fuel_flow_metadata = { "offset" : _set_max_fuel_flow_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_fuel_flow.setter
    def max_fuel_flow(self, value:float) -> None:
        """Get or set the fuel flow for the maximum thrust/power setting."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_max_fuel_flow_metadata, value)

    _get_max_thrust_acceleration_metadata = { "offset" : _get_max_thrust_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_thrust_acceleration(self) -> float:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_max_thrust_acceleration_metadata)

    _set_max_thrust_acceleration_metadata = { "offset" : _set_max_thrust_acceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_thrust_acceleration.setter
    def max_thrust_acceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft speeds up at max throttle."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_max_thrust_acceleration_metadata, value)

    _get_min_thrust_deceleration_metadata = { "offset" : _get_min_thrust_deceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_thrust_deceleration(self) -> float:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_min_thrust_deceleration_metadata)

    _set_min_thrust_deceleration_metadata = { "offset" : _set_min_thrust_deceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_thrust_deceleration.setter
    def min_thrust_deceleration(self, value:float) -> None:
        """Get or set the rate at which the aircraft slows down at minimum throttle setting."""
        return self._intf.set_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_min_thrust_deceleration_metadata, value)

    _get_use_density_scaling_metadata = { "offset" : _get_use_density_scaling_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_density_scaling(self) -> bool:
        """Opt whether to scale the accel/decel performance by the density ratio."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_use_density_scaling_metadata)

    _get_density_ratio_exponent_metadata = { "offset" : _get_density_ratio_exponent_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def density_ratio_exponent(self) -> float:
        """Get the relative impace of atmospheric density on the aircraft's performance."""
        return self._intf.get_property(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._get_density_ratio_exponent_metadata)

    _set_density_scaling_metadata = { "offset" : _set_density_scaling_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL, agcom.DOUBLE,),
            "marshallers" : (agmarshall.VariantBoolArg, agmarshall.DoubleArg,) }
    def set_density_scaling(self, use_scaling:bool, exponent:float) -> None:
        """Set the option to use density scaling and set the density ratio exponent."""
        return self._intf.invoke(AircraftBasicFixedWingPropulsion._metadata, AircraftBasicFixedWingPropulsion._set_density_scaling_metadata, use_scaling, exponent)

    _property_names[propulsion_mode] = "propulsion_mode"
    _property_names[propeller_count] = "propeller_count"
    _property_names[propeller_diameter] = "propeller_diameter"
    _property_names[propeller_rpm] = "propeller_rpm"
    _property_names[min_power_thrust] = "min_power_thrust"
    _property_names[min_fuel_flow] = "min_fuel_flow"
    _property_names[max_power_thrust] = "max_power_thrust"
    _property_names[max_fuel_flow] = "max_fuel_flow"
    _property_names[max_thrust_acceleration] = "max_thrust_acceleration"
    _property_names[min_thrust_deceleration] = "min_thrust_deceleration"
    _property_names[use_density_scaling] = "use_density_scaling"
    _property_names[density_ratio_exponent] = "density_ratio_exponent"

    def __init__(self, source_object=None):
        """Construct an object of type AircraftBasicFixedWingPropulsion."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AircraftBasicFixedWingPropulsion)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AircraftBasicFixedWingPropulsion, [AircraftBasicFixedWingPropulsion, ])

agcls.AgClassCatalog.add_catalog_entry((5671810767600845155, 12908412790033902229), AircraftBasicFixedWingPropulsion)
agcls.AgTypeNameMap["AircraftBasicFixedWingPropulsion"] = AircraftBasicFixedWingPropulsion

class ARINC424Source(ICatalogSource, SupportsDeleteCallback):
    """Interface used to access the options for any ARINC424 source in the Aviator catalog."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_arinc424_item_method_offset = 1
    _get_use_master_data_file_method_offset = 2
    _set_use_master_data_file_method_offset = 3
    _get_master_data_filepath_method_offset = 4
    _set_master_data_filepath_method_offset = 5
    _get_override_data_filepath_method_offset = 6
    _set_override_data_filepath_method_offset = 7
    _get_as_catalog_source_method_offset = 8
    _metadata = {
        "iid_data" : (5252096220626246087, 6835548189431916434),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ARINC424Source)

    _get_arinc424_item_metadata = { "offset" : _get_arinc424_item_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_arinc424_item(self, name:str) -> "IARINC424Item":
        """Get the ARINC-424 item with the given name."""
        return self._intf.invoke(ARINC424Source._metadata, ARINC424Source._get_arinc424_item_metadata, name, OutArg())

    _get_use_master_data_file_metadata = { "offset" : _get_use_master_data_file_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_master_data_file(self) -> bool:
        """Opt whether to use the master data file."""
        return self._intf.get_property(ARINC424Source._metadata, ARINC424Source._get_use_master_data_file_metadata)

    _set_use_master_data_file_metadata = { "offset" : _set_use_master_data_file_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_master_data_file.setter
    def use_master_data_file(self, value:bool) -> None:
        """Opt whether to use the master data file."""
        return self._intf.set_property(ARINC424Source._metadata, ARINC424Source._set_use_master_data_file_metadata, value)

    _get_master_data_filepath_metadata = { "offset" : _get_master_data_filepath_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def master_data_filepath(self) -> str:
        """Get or set the master data file path."""
        return self._intf.get_property(ARINC424Source._metadata, ARINC424Source._get_master_data_filepath_metadata)

    _set_master_data_filepath_metadata = { "offset" : _set_master_data_filepath_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @master_data_filepath.setter
    def master_data_filepath(self, value:str) -> None:
        """Get or set the master data file path."""
        return self._intf.set_property(ARINC424Source._metadata, ARINC424Source._set_master_data_filepath_metadata, value)

    _get_override_data_filepath_metadata = { "offset" : _get_override_data_filepath_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def override_data_filepath(self) -> str:
        """Get or set the file path to the data overriding the mader data file."""
        return self._intf.get_property(ARINC424Source._metadata, ARINC424Source._get_override_data_filepath_metadata)

    _set_override_data_filepath_metadata = { "offset" : _set_override_data_filepath_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @override_data_filepath.setter
    def override_data_filepath(self, value:str) -> None:
        """Get or set the file path to the data overriding the mader data file."""
        return self._intf.set_property(ARINC424Source._metadata, ARINC424Source._set_override_data_filepath_metadata, value)

    _get_as_catalog_source_metadata = { "offset" : _get_as_catalog_source_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_source(self) -> "ICatalogSource":
        """Get the catalog source interface for this object."""
        return self._intf.invoke(ARINC424Source._metadata, ARINC424Source._get_as_catalog_source_metadata, OutArg())

    _property_names[use_master_data_file] = "use_master_data_file"
    _property_names[master_data_filepath] = "master_data_filepath"
    _property_names[override_data_filepath] = "override_data_filepath"

    def __init__(self, source_object=None):
        """Construct an object of type ARINC424Source."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ARINC424Source)
        ICatalogSource.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogSource._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ARINC424Source, [ARINC424Source, ICatalogSource])

agcls.AgClassCatalog.add_catalog_entry((4990508709075858532, 869911397036924554), ARINC424Source)
agcls.AgTypeNameMap["ARINC424Source"] = ARINC424Source

class DAFIFSource(ICatalogSource, SupportsDeleteCallback):
    """Interface used to access the options for any DAFIF source in the Aviator catalog."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_dafif_item_method_offset = 1
    _get_data_path_method_offset = 2
    _set_data_path_method_offset = 3
    _get_effective_date_method_offset = 4
    _get_expiration_date_method_offset = 5
    _get_spec_revision_method_offset = 6
    _get_as_catalog_source_method_offset = 7
    _metadata = {
        "iid_data" : (4761423141073607641, 6693699497707172788),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DAFIFSource)

    _get_dafif_item_metadata = { "offset" : _get_dafif_item_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_dafif_item(self, name:str) -> "IDAFIFItem":
        """Get the DAFIF item with the given name."""
        return self._intf.invoke(DAFIFSource._metadata, DAFIFSource._get_dafif_item_metadata, name, OutArg())

    _get_data_path_metadata = { "offset" : _get_data_path_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def data_path(self) -> str:
        """Get or set the DAFIF data path."""
        return self._intf.get_property(DAFIFSource._metadata, DAFIFSource._get_data_path_metadata)

    _set_data_path_metadata = { "offset" : _set_data_path_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @data_path.setter
    def data_path(self, value:str) -> None:
        """Get or set the DAFIF data path."""
        return self._intf.set_property(DAFIFSource._metadata, DAFIFSource._set_data_path_metadata, value)

    _get_effective_date_metadata = { "offset" : _get_effective_date_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def effective_date(self) -> str:
        """Get the effective date of the DAFIF catalog."""
        return self._intf.get_property(DAFIFSource._metadata, DAFIFSource._get_effective_date_metadata)

    _get_expiration_date_metadata = { "offset" : _get_expiration_date_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def expiration_date(self) -> str:
        """Get the expiration date of the DAFIF catalog."""
        return self._intf.get_property(DAFIFSource._metadata, DAFIFSource._get_expiration_date_metadata)

    _get_spec_revision_metadata = { "offset" : _get_spec_revision_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def spec_revision(self) -> str:
        """Get the DAFIF edition."""
        return self._intf.get_property(DAFIFSource._metadata, DAFIFSource._get_spec_revision_metadata)

    _get_as_catalog_source_metadata = { "offset" : _get_as_catalog_source_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_as_catalog_source(self) -> "ICatalogSource":
        """Get the catalog source interface for this object."""
        return self._intf.invoke(DAFIFSource._metadata, DAFIFSource._get_as_catalog_source_metadata, OutArg())

    _property_names[data_path] = "data_path"
    _property_names[effective_date] = "effective_date"
    _property_names[expiration_date] = "expiration_date"
    _property_names[spec_revision] = "spec_revision"

    def __init__(self, source_object=None):
        """Construct an object of type DAFIFSource."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DAFIFSource)
        ICatalogSource.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICatalogSource._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DAFIFSource, [DAFIFSource, ICatalogSource])

agcls.AgClassCatalog.add_catalog_entry((4669407446549507698, 4351897649948657831), DAFIFSource)
agcls.AgTypeNameMap["DAFIFSource"] = DAFIFSource

class BasicFixedWingForwardFlightLiftHelper(IBasicFixedWingLiftHelper, SupportsDeleteCallback):
    """Class defining the Lift Coefficient Helper for Forward Flight in the Basic Fixed Wing Aerodynamics interface for the Basic Acceleration Model of an aircraft."""
    def __init__(self, source_object=None):
        """Construct an object of type BasicFixedWingForwardFlightLiftHelper."""
        SupportsDeleteCallback.__init__(self)
        IBasicFixedWingLiftHelper.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBasicFixedWingLiftHelper._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BasicFixedWingForwardFlightLiftHelper, [IBasicFixedWingLiftHelper])

agcls.AgClassCatalog.add_catalog_entry((4835967416678721147, 2787928793404869025), BasicFixedWingForwardFlightLiftHelper)
agcls.AgTypeNameMap["BasicFixedWingForwardFlightLiftHelper"] = BasicFixedWingForwardFlightLiftHelper

class BasicManeuverStrategyStraightAhead(IBasicManeuverStrategy, SupportsDeleteCallback):
    """
    Interface used to access options for a Straight Ahead Strategy of a Basic Maneuver Procedure.

    Examples
    --------
    Add and configure a basic maneuver procedure:
    >>> # IProcedureCollection procedures: Procedure Collection object
    >>> # Add a basic maneuver procedure
    >>> basicManeuver = procedures.add(SiteType.SITE_END_OF_PREV_PROCEDURE, ProcedureType.PROCEDURE_BASIC_MANEUVER)
    >>>
    >>> # Set the navigation to use a Straight Ahead strategy
    >>> basicManeuver.navigation_strategy_type = "Straight Ahead"
    >>> # Get the options for the straight ahead strategy
    >>> straightAhead = basicManeuver.navigation
    >>> # Opt to maintain course (as opposed to maintain heading)
    >>> straightAhead.reference_frame = StraightAheadReferenceFrame.MAINTAIN_COURSE
    >>>
    >>> # Set the profile to use a Autopilot - Vertical Plane strategy
    >>> basicManeuver.profile_strategy_type = "Autopilot - Vertical Plane"
    >>> # Get the options for the profile strategy
    >>> autopilot = basicManeuver.profile
    >>> # Opt to maintain the initial altitude
    >>> autopilot.altitude_mode = AutopilotAltitudeMode.AUTOPILOT_HOLD_INIT_ALTITUDE
    >>> airspeedOptions = autopilot.airspeed_options
    >>> # Opt to maintain a specified airspeed
    >>> airspeedOptions.airspeed_mode = BasicManeuverAirspeedMode.MAINTAIN_SPECIFIED_AIRSPEED
    >>> # Specify the airspeed
    >>> airspeedOptions.specified_airspeed = 250
    >>>
    >>> # Configure the options on the Attitude / Performance / Fuel page
    >>> basicManeuver.flight_mode = PhaseOfFlight.FLIGHT_PHASE_CRUISE
    >>> # Override the fuel flow
    >>> basicManeuver.fuel_flow_type = BasicManeuverFuelFlowType.BASIC_MANEUVER_FUEL_FLOW_OVERRIDE
    >>> basicManeuver.override_fuel_flow_value = 1000
    >>>
    >>> # Set the basic stopping conditions
    >>> basicManeuver.use_max_downrange = True
    >>> basicManeuver.max_downrange = 10
    >>> basicManeuver.use_stop_fuel_state = False
    >>> basicManeuver.use_max_time_of_flight = False
    """

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_frame_method_offset = 1
    _set_reference_frame_method_offset = 2
    _get_compensate_for_coriolis_acceleration_method_offset = 3
    _set_compensate_for_coriolis_acceleration_method_offset = 4
    _metadata = {
        "iid_data" : (5334953136761860956, 12274532353622486171),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BasicManeuverStrategyStraightAhead)

    _get_reference_frame_metadata = { "offset" : _get_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StraightAheadReferenceFrame),) }
    @property
    def reference_frame(self) -> "StraightAheadReferenceFrame":
        """Get or set the reference frame the aircraft will use to fly straight ahead."""
        return self._intf.get_property(BasicManeuverStrategyStraightAhead._metadata, BasicManeuverStrategyStraightAhead._get_reference_frame_metadata)

    _set_reference_frame_metadata = { "offset" : _set_reference_frame_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StraightAheadReferenceFrame),) }
    @reference_frame.setter
    def reference_frame(self, value:"StraightAheadReferenceFrame") -> None:
        return self._intf.set_property(BasicManeuverStrategyStraightAhead._metadata, BasicManeuverStrategyStraightAhead._set_reference_frame_metadata, value)

    _get_compensate_for_coriolis_acceleration_metadata = { "offset" : _get_compensate_for_coriolis_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def compensate_for_coriolis_acceleration(self) -> bool:
        """Get or set the option to compensate for the acceleration due to the Coriolis effect."""
        return self._intf.get_property(BasicManeuverStrategyStraightAhead._metadata, BasicManeuverStrategyStraightAhead._get_compensate_for_coriolis_acceleration_metadata)

    _set_compensate_for_coriolis_acceleration_metadata = { "offset" : _set_compensate_for_coriolis_acceleration_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @compensate_for_coriolis_acceleration.setter
    def compensate_for_coriolis_acceleration(self, value:bool) -> None:
        return self._intf.set_property(BasicManeuverStrategyStraightAhead._metadata, BasicManeuverStrategyStraightAhead._set_compensate_for_coriolis_acceleration_metadata, value)

    _property_names[reference_frame] = "reference_frame"
    _property_names[compensate_for_coriolis_acceleration] = "compensate_for_coriolis_acceleration"

    def __init__(self, source_object=None):
        """Construct an object of type BasicManeuverStrategyStraightAhead."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BasicManeuverStrategyStraightAhead)
        IBasicManeuverStrategy.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBasicManeuverStrategy._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BasicManeuverStrategyStraightAhead, [BasicManeuverStrategyStraightAhead, IBasicManeuverStrategy])

agcls.AgClassCatalog.add_catalog_entry((5762754887795241020, 9551154128583839911), BasicManeuverStrategyStraightAhead)
agcls.AgTypeNameMap["BasicManeuverStrategyStraightAhead"] = BasicManeuverStrategyStraightAhead

class BasicManeuverStrategyCruiseProfile(IBasicManeuverStrategy, SupportsDeleteCallback):
    """Interface used to access options for a Cruise Profile Strategy of a Basic Maneuver Procedure."""

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_frame_method_offset = 1
    _set_reference_frame_method_offset = 2
    _get_use_default_cruise_altitude_method_offset = 3
    _set_use_default_cruise_altitude_method_offset = 4
    _get_level_off_method_offset = 5
    _set_level_off_method_offset = 6
    _get_requested_altitude_method_offset = 7
    _set_requested_altitude_method_offset = 8
    _get_cruise_airspeed_options_method_offset = 9
    _get_stop_after_level_off_method_offset = 10
    _set_stop_after_level_off_method_offset = 11
    _get_compensate_for_coriolis_acceleration_method_offset = 12
    _set_compensate_for_coriolis_acceleration_method_offset = 13
    _metadata = {
        "iid_data" : (5262849801508960856, 3879152463803584443),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BasicManeuverStrategyCruiseProfile)

    _get_reference_frame_metadata = { "offset" : _get_reference_frame_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BasicManeuverReferenceFrame),) }
    @property
    def reference_frame(self) -> "BasicManeuverReferenceFrame":
        """Get or set the reference frame the aircraft will use. Earth Frame will force the airc