# Copyright (C) 2022 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""Object Model components specifically designed to support STK Astrogator."""

__all__ = ["AccessCriterion", "AccessStoppingCondition", "AsTriggerCondition", "AscentType", "AtmosphereDataSource",
"AttitudeControl", "AttitudeControlFiniteAntiVelocityVector", "AttitudeControlFiniteAttitude",
"AttitudeControlFiniteFile", "AttitudeControlFinitePlugin", "AttitudeControlFiniteThrustVector",
"AttitudeControlFiniteTimeVarying", "AttitudeControlFiniteVelocityVector", "AttitudeControlImpulsiveAntiVelocityVector",
"AttitudeControlImpulsiveAttitude", "AttitudeControlImpulsiveFile", "AttitudeControlImpulsiveThrustVector",
"AttitudeControlImpulsiveVelocityVector", "AttitudeControlOptimalFiniteLagrange", "AttitudeUpdate", "AutomaticSequence",
"AutomaticSequenceCollection", "BDotRCalc", "BDotTCalc", "BMagnitudeCalc", "BPlaneCollection", "BThetaCalc",
"BaseSelection", "BisectionControl", "BisectionControlCollection", "BisectionResult", "BisectionResultCollection",
"BlendedDensity", "BodyAxis", "BulirschStoerIntegrator", "BurnoutCBFCartesian", "BurnoutGeocentric", "BurnoutGeodetic",
"BurnoutLaunchAzAltitude", "BurnoutLaunchAzRadius", "BurnoutOptions", "BurnoutType", "BurnoutVelocity", "CR3BPFunction",
"CalculationGraphCollection", "CalculationObjectAngleSign", "CalculationObjectCentralBodyReference",
"CalculationObjectCollection", "CalculationObjectDirection", "CalculationObjectElement",
"CalculationObjectLinkEmbedControlCollection", "CalculationObjectLocationSource", "CalculationObjectOrbitPlaneSource",
"CalculationObjectReference", "CalculationObjectReferenceDirection", "CalculationObjectReferenceEllipse",
"CalculationObjectRelativePosition", "CalculationObjectSunPosition", "CentralBodyAttitude", "CentralBodyComponent",
"CentralBodyComponentAttitudeIAU1994", "CentralBodyComponentAttitudeRotationCoefficientsFile",
"CentralBodyComponentCollection", "CentralBodyComponentEphemerisAnalyticOrbit", "CentralBodyComponentEphemerisFile",
"CentralBodyComponentEphemerisJPLDesignExplorerOptimizer", "CentralBodyComponentEphemerisJPLSpice",
"CentralBodyComponentEphemerisPlanetary", "CentralBodyComponentGravityModel", "CentralBodyComponentShapeOblateSpheroid",
"CentralBodyComponentShapeSphere", "CentralBodyComponentShapeTriaxialEllipsoid", "CentralBodyEphemeris",
"CentralBodyGravityModel", "CentralBodyShape", "Cira72Function", "ClearEphemerisDirection", "CoeffRungeKuttaV8th9th",
"ComplexNumber", "ConstraintCollection", "ConstraintSign", "ControlAdvanced", "ControlEngineConstant",
"ControlEngineConstantAcceleration", "ControlEngineCustom", "ControlEngineIon", "ControlEngineModelPolynomial",
"ControlEngineThrottleTable", "ControlFollow", "ControlInitState", "ControlLaunch", "ControlManeuver",
"ControlOrbitStateValue", "ControlPowerInternal", "ControlPowerProcessed", "ControlPowerSolarArray",
"ControlRepeatingGroundTrackErr", "ControlStoppingCondition", "ControlThrusters", "ControlUpdate",
"ConvergenceCriteria", "Criteria", "Criterion", "CustomFunction", "CustomFunctionScriptEngine", "DTM2012", "DTM2020",
"DensityModelPlugin", "DerivativeCalculationMethod", "DesignCR3BPObject", "DesignCR3BPObjectCollection",
"DesignCR3BPSetup", "DesignER3BPObject", "DesignER3BPObjectCollection", "DesignER3BPSetup", "DifferenceOrder",
"DifferentialCorrectorControl", "DifferentialCorrectorControlCollection", "DifferentialCorrectorResult",
"DifferentialCorrectorResultCollection", "DifferentialCorrectorScalingMethod", "DisplaySystemGeocentric",
"DisplaySystemGeodetic", "DragModelPlugin", "DragModelType", "DrawPerturbation", "EOMFuncPluginFunction", "ER3BPFunc",
"EclipsingBodiesSource", "ElementBPlane", "ElementCartesian", "ElementDelaunay", "ElementEquinoctial",
"ElementGeodetic", "ElementKeplerian", "ElementMixedSpherical", "ElementSetType", "ElementSpherical",
"ElementSphericalRangeRate", "ElementTargetVectorIncomingAsymptote", "ElementTargetVectorOutgoingAsymptote",
"ElementType", "EngineConstant", "EngineConstantAcceleration", "EngineCustom", "EngineDefinition", "EngineIon",
"EngineModelFunction", "EngineModelIspCoefficients", "EngineModelPolynomial", "EngineModelThrustCoefficients",
"EnginePlugin", "EngineThrottleTable", "EphemerisSource", "ErrorControl", "Exponential", "FollowJoin",
"FollowSeparation", "FollowSpacecraftAndFuelTank", "Formulation", "FuelTank", "GaussJacksonIntegrator",
"GeneralRelativityFunction", "GeoMagneticFluxSource", "GeoMagneticFluxUpdateRate", "GeoStationaryDriftRateModel",
"GeoStationaryInclinationMagnitude", "GoldenSectionControl", "GoldenSectionControlCollection",
"GoldenSectionDesiredOperation", "GoldenSectionResult", "GoldenSectionResultCollection", "GraphOption",
"GravParamSource", "GravitationalParameterSource", "GravityCoefficientNormalizationType", "GravityCoefficientType",
"GravityFieldFunction", "GridSearchControl", "GridSearchControlCollection", "GridSearchDesiredOperation",
"GridSearchResult", "GridSearchResultCollection", "HPOPPluginFunction", "HarrisPriester", "IAttitudeControl",
"IAttitudeControlFinite", "IAttitudeControlImpulsive", "IAttitudeControlOptimalFinite", "IBurnout",
"ICentralBodyComponentAttitude", "ICentralBodyComponentEphemeris",
"ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides", "ICentralBodyComponentShape", "IDisplaySystem", "IElement",
"IMCSSegment", "IMCSSequence", "IManeuver", "INumericalIntegrator", "IPOPTControl", "IPOPTControlCollection",
"IPOPTGoal", "IPOPTResult", "IPOPTResultCollection", "IProfile", "IStoppingConditionComponent", "IdealOrbitRadius",
"ImpulsiveDeltaVRepresentation", "Jacchia1960", "Jacchia1970", "Jacchia1971", "JacchiaBowman2008", "JacchiaRoberts",
"LambertDirectionOfMotionType", "LambertOrbitalEnergyType", "LambertSolutionOptionType", "LambertTargetCoordinateType",
"Language", "LaunchDisplaySystem", "LightingCondition", "LightingStoppingCondition", "MCSBackwardSequence", "MCSDriver",
"MCSEnd", "MCSFollow", "MCSHold", "MCSInitialState", "MCSLaunch", "MCSManeuver", "MCSOptions", "MCSPropagate",
"MCSReturn", "MCSSegmentCollection", "MCSSegmentProperties", "MCSSequence", "MCSStop", "MCSTargetSequence", "MCSUpdate",
"MSIS1986", "MSISE1990", "ManeuverFinite", "ManeuverFinitePropagator", "ManeuverImpulsive", "ManeuverOptimalFinite",
"ManeuverOptimalFiniteBounds", "ManeuverOptimalFiniteFinalBoundaryConditions",
"ManeuverOptimalFiniteInitialBoundaryConditions", "ManeuverOptimalFinitePathBoundaryConditions",
"ManeuverOptimalFiniteSNOPTOptimizer", "ManeuverOptimalFiniteSteeringNodeCollection",
"ManeuverOptimalFiniteSteeringNodeElement", "ManeuverType", "MarsGRAM2000", "MarsGRAM2001", "MarsGRAM2005",
"MarsGRAM2010", "MarsGRAM37", "MarsGRAMDensityType", "NRLMSISE2000", "NumericalIntegrator",
"NumericalPropagatorWrapper", "NumericalPropagatorWrapperCR3BP", "OptimalFiniteDiscretizationStrategy",
"OptimalFiniteExportNodesFormat", "OptimalFiniteGuessMethod", "OptimalFiniteRunMode", "OptimalFiniteSNOPTObjective",
"OptimalFiniteSNOPTScaling", "OptimalFiniteScalingOptions", "OptimalFiniteSeedMethod", "OptimalFiniteWorkingVariables",
"PluginProperties", "PointMassFunction", "PowerInternal", "PowerProcessed", "PowerSolarArray", "PredictorCorrector",
"PressureMode", "Profile", "ProfileBisection", "ProfileChangeManeuverType", "ProfileChangePropagator",
"ProfileChangeReturnSegment", "ProfileChangeStopSegment", "ProfileChangeStoppingConditionState", "ProfileCollection",
"ProfileDifferentialCorrector", "ProfileGoldenSection", "ProfileGridSearch", "ProfileIPOPTOptimizer",
"ProfileInsertDirection", "ProfileLambertProfile", "ProfileLambertSearchProfile", "ProfileMode", "ProfileRunOnce",
"ProfileSNOPTOptimizer", "ProfileScriptingTool", "ProfileSearchPlugin", "ProfileSeedFiniteManeuver", "ProfilesFinish",
"PropagatorFunctionCollection", "PropulsionMethod", "RadiationPressureFunction", "ReferenceRadiusSource",
"ReturnControl", "RootFindingAlgorithm", "RotatingCoordinateSystem", "RunCode", "RungeKutta2nd3rd", "RungeKutta4th",
"RungeKutta4th5th", "RungeKutta4thAdapt", "RungeKuttaF7th8th", "RungeKuttaV8th9th", "SNOPTControl",
"SNOPTControlCollection", "SNOPTGoal", "SNOPTResult", "SNOPTResultCollection", "SRPAerospaceT20", "SRPAerospaceT30",
"SRPGSPM04aIIA", "SRPGSPM04aIIR", "SRPGSPM04aeIIA", "SRPGSPM04aeIIR", "SRPNPlate", "SRPReflectionPlugin",
"SRPSpherical", "SRPTabulatedAreaVector", "SRPVariableArea", "STMEigenNumber", "STMPerturbationVariables",
"ScriptingCalculationObject", "ScriptingCalculationObjectCollection", "ScriptingParameter",
"ScriptingParameterCollection", "ScriptingParameterEnumerationChoice", "ScriptingParameterEnumerationChoiceCollection",
"ScriptingParameterType", "ScriptingSegment", "ScriptingSegmentCollection", "ScriptingTool", "SearchPluginControl",
"SearchPluginControlCollection", "SearchPluginResult", "SearchPluginResultCollection", "SegmentDifferenceOrder",
"SegmentState", "SegmentType", "SequenceStateToPass", "ShadowModel", "SmartRunMode", "SolarForceMethod",
"SpacecraftParameters", "SquaredType", "State", "StateCalcAbsoluteValue", "StateCalcAltitudeOfApoapsis",
"StateCalcAltitudeOfPeriapsis", "StateCalcAngle", "StateCalcApparentSolarTime", "StateCalcArgumentOfLatitude",
"StateCalcArgumentOfPeriapsis", "StateCalcBetaAngle", "StateCalcC3Energy", "StateCalcCartSTMElem",
"StateCalcCartesianElem", "StateCalcCd", "StateCalcCloseApproachBearing", "StateCalcCloseApproachCosBearing",
"StateCalcCloseApproachMagnitude", "StateCalcCloseApproachTheta", "StateCalcCloseApproachX", "StateCalcCloseApproachY",
"StateCalcCosOfVerticalFlightPathAngle", "StateCalcCr", "StateCalcCurvilinearRelativeMotion", "StateCalcCustomFunction",
"StateCalcDamageFlux", "StateCalcDamageMassFlux", "StateCalcDec", "StateCalcDecRate", "StateCalcDeltaDec",
"StateCalcDeltaFromMaster", "StateCalcDeltaRA", "StateCalcDeltaV", "StateCalcDeltaVSquared", "StateCalcDensity",
"StateCalcDifference", "StateCalcDifferenceAcrossSegmentsOtherSatellite", "StateCalcDifferenceOtherSegment",
"StateCalcDotProduct", "StateCalcDragArea", "StateCalcDriftRateFactor", "StateCalcDuration",
"StateCalcEarthMeanLocalTimeOfAscendingNode", "StateCalcEarthMeanSolarTime", "StateCalcEccentricity",
"StateCalcEccentricityAnomaly", "StateCalcEccentricityX", "StateCalcEccentricityY", "StateCalcEnvironment",
"StateCalcEpoch", "StateCalcEquinoctialElem", "StateCalcFlightPathAngle", "StateCalcFuelMass", "StateCalcGeodeticElem",
"StateCalcGeodeticMeanRightAscension", "StateCalcGeodeticTrueLongitude",
"StateCalcGeodeticTrueLongitudeAtTimeOfPerigee", "StateCalcGravCoeff", "StateCalcGravitationalParameter",
"StateCalcHeightAboveTerrain", "StateCalcImpactFlux", "StateCalcImpactMassFlux", "StateCalcInAsympDec",
"StateCalcInAsympRA", "StateCalcInVelocityAzAtPeriapsis", "StateCalcInclination", "StateCalcInclinationX",
"StateCalcInclinationY", "StateCalcInertialDeltaVMagnitude", "StateCalcInertialDeltaVx", "StateCalcInertialDeltaVy",
"StateCalcInertialDeltaVz", "StateCalcJacobiConstant", "StateCalcJacobiOsculating", "StateCalcLocalApparentSolarLon",
"StateCalcLonDriftRate", "StateCalcLonOfAscNode", "StateCalcLonOfPeriapsis", "StateCalcMCSDeltaV",
"StateCalcMCSDeltaVSquared", "StateCalcMagneticFieldDipoleL", "StateCalcManeuverSpecificImpulse",
"StateCalcManeuverTotalMassFlowRate", "StateCalcMass", "StateCalcMaxValue", "StateCalcMeanAnomaly",
"StateCalcMeanEarthLon", "StateCalcMeanMotion", "StateCalcMeanRightAscension", "StateCalcMeanValue",
"StateCalcMedianValue", "StateCalcMinValue", "StateCalcNegative", "StateCalcNumRevs", "StateCalcOnePointAccess",
"StateCalcOrbitDelaunayG", "StateCalcOrbitDelaunayH", "StateCalcOrbitDelaunayL", "StateCalcOrbitPeriod",
"StateCalcOrbitSemilatusRectum", "StateCalcOrbitStateValue", "StateCalcOutAsympDec", "StateCalcOutAsympRA",
"StateCalcOutVelocityAzAtPeriapsis", "StateCalcPi", "StateCalcPositionDifferenceOtherSegment",
"StateCalcPositionVelocityDifferenceOtherSegment", "StateCalcPower", "StateCalcPressure", "StateCalcRA",
"StateCalcRAAN", "StateCalcRARate", "StateCalcRMagnitude", "StateCalcRadOfApoapsis", "StateCalcRadOfPeriapsis",
"StateCalcRadiationPressureArea", "StateCalcRadiationPressureCoefficient", "StateCalcRangeRate",
"StateCalcRectifiedLon", "StateCalcReferenceRadius", "StateCalcRelativeAtAOLMaster",
"StateCalcRelativeGroundTrackError", "StateCalcRelativeInclination", "StateCalcRelativeMotion",
"StateCalcRelativePositionDecAngle", "StateCalcRelativePositionInPlaneAngle", "StateCalcRepeatingGroundTrackErr",
"StateCalcSEETMagneticFieldLineSeparationAngle", "StateCalcSEETSAAFlux", "StateCalcSEETVehTemp", "StateCalcSRPArea",
"StateCalcSTMEigenval", "StateCalcSTMEigenvecElem", "StateCalcScalar", "StateCalcScript", "StateCalcSemimajorAxis",
"StateCalcSequenceDeltaV", "StateCalcSequenceDeltaVSquared", "StateCalcSignedEccentricity", "StateCalcSolarBetaAngle",
"StateCalcSolarInPlaneAngle", "StateCalcSpeedOfLight", "StateCalcStandardDeviation", "StateCalcTemperature",
"StateCalcTimePastAscNode", "StateCalcTimePastPeriapsis", "StateCalcTrueAnomaly", "StateCalcTrueLon",
"StateCalcTrueLongitude", "StateCalcTwoBodyDriftRate", "StateCalcUnitAngularMomentumX", "StateCalcUnitAngularMomentumY",
"StateCalcUnitAngularMomentumZ", "StateCalcUserValue", "StateCalcVMagnitude", "StateCalcValueAtSegment",
"StateCalcValueAtSegmentOtherSat", "StateCalcVectorDec", "StateCalcVectorGeometryToolAngle", "StateCalcVectorMagnitude",
"StateCalcVectorRA", "StateCalcVectorX", "StateCalcVectorY", "StateCalcVectorZ", "StateCalcVelocityAz",
"StateCalcVelocityDifferenceOtherSegment", "StateTransformationFunction", "StateType", "StoppingCondition",
"StoppingConditionCollection", "StoppingConditionElement", "StoppingConditionType", "SunPosition",
"TabVecInterpolationMethod", "TargetSequenceAction", "TargeterGraph", "TargeterGraphActiveControl",
"TargeterGraphActiveControlCollection", "TargeterGraphCollection", "TargeterGraphResult",
"TargeterGraphResultCollection", "ThirdBodyFunction", "ThirdBodyMode", "ThrottleTableOperationMode", "ThrustType",
"Thruster", "ThrusterDirection", "ThrusterSet", "ThrusterSetCollection", "TwoBodyFunction", "USStandardAtmosphere",
"UpdateAction", "UpdateParam", "UserVariable", "UserVariableCollection", "UserVariableDefinition",
"UserVariableDefinitionCollection", "UserVariableUpdate", "UserVariableUpdateCollection", "VenusGRAM2005",
"VenusGRAMDensityType", "YarkovskyFunc"]

from ctypes import POINTER
from enum import IntEnum
import typing

from ..internal import coclassutil as agcls, comutil as agcom, marshall as agmarshall
from ..internal.apiutil import (
    EnumeratorProxy,
    InterfaceProxy,
    OutArg,
    SupportsDeleteCallback,
    get_interface_property,
    initialize_from_source_object,
    set_class_attribute,
    set_interface_attribute,
)
from ..internal.comutil import IDispatch, IUnknown
from ..stkobjects import (
    AberrationType,
    ComponentLinkEmbedControlReferenceType,
    ICloneable,
    IComponentInfo,
    IComponentLinkEmbedControl,
    IPropagator,
    IvClockHost,
    IvTimeSense,
    SolidTide,
)
from ..stkutil import IDirection, IOrientation, IRuntimeTypeInfoProvider, RuntimeTypeInfo
from ..utilities import colors as agcolor
from ..utilities.exceptions import STKRuntimeError


def _raise_uninitialized_error(*args):
    raise STKRuntimeError("Valid STK object model classes are returned from STK methods and should not be created independently.")

class GraphOption(IntEnum):
    """Mode that the mcs will run in."""

    NO_GRAPH = 0
    """Don't graph."""
    GRAPH_DIFFERENCE = 1
    """Graph the difference."""
    GRAPH_VALUE = 2
    """Graph the value."""

GraphOption.NO_GRAPH.__doc__ = "Don't graph."
GraphOption.GRAPH_DIFFERENCE.__doc__ = "Graph the difference."
GraphOption.GRAPH_VALUE.__doc__ = "Graph the value."

agcls.AgTypeNameMap["GraphOption"] = GraphOption

class SmartRunMode(IntEnum):
    """Mode that the mcs will run in."""

    ENTIRE_MCS = 0
    """Run Entire Mission Control Sequence."""
    ONLY_CHANGED = 1
    """Run Only Changed Segments."""

SmartRunMode.ENTIRE_MCS.__doc__ = "Run Entire Mission Control Sequence."
SmartRunMode.ONLY_CHANGED.__doc__ = "Run Only Changed Segments."

agcls.AgTypeNameMap["SmartRunMode"] = SmartRunMode

class Formulation(IntEnum):
    """Equinoctial Formulation."""

    POSIGRADE = 0
    """Posigrade."""
    RETROGRADE = 1
    """Retrograde."""

Formulation.POSIGRADE.__doc__ = "Posigrade."
Formulation.RETROGRADE.__doc__ = "Retrograde."

agcls.AgTypeNameMap["Formulation"] = Formulation

class LightingCondition(IntEnum):
    """The criteria of a Lighting stopping condition."""

    CRITERION_ENTER_DIRECT_SUN = 0
    """The Enter Direct Sun criterion."""
    CRITERION_EXIT_DIRECT_SUN = 1
    """The Exit Direct Sun criterion."""
    CRITERION_ENTER_UMBRA = 2
    """The Enter Umbra criterion."""
    CRITERION_EXIT_UMBRA = 3
    """The Exit Umbra criterion."""

LightingCondition.CRITERION_ENTER_DIRECT_SUN.__doc__ = "The Enter Direct Sun criterion."
LightingCondition.CRITERION_EXIT_DIRECT_SUN.__doc__ = "The Exit Direct Sun criterion."
LightingCondition.CRITERION_ENTER_UMBRA.__doc__ = "The Enter Umbra criterion."
LightingCondition.CRITERION_EXIT_UMBRA.__doc__ = "The Exit Umbra criterion."

agcls.AgTypeNameMap["LightingCondition"] = LightingCondition

class Profile(IntEnum):
    """Type of profile."""

    SEARCH_PLUGIN = 0
    """Plugin search profile."""
    DIFFERENTIAL_CORRECTOR = 2
    """Differential corrector profile."""
    CHANGE_MANEUVER_TYPE = 3
    """Change maneuver type profile."""
    SCRIPTING_TOOL = 4
    """Scripting tool profile."""
    CHANGE_RETURN_SEGMENT = 5
    """Change return segment profile."""
    CHANGE_PROPAGATOR = 6
    """Change propagator profile."""
    CHANGE_STOP_SEGMENT = 7
    """Change stop segment profile."""
    CHANGE_STOPPING_CONDITION_STATE = 8
    """Change stopping condition state profile."""
    SEED_FINITE_MANEUVER = 9
    """Change seed finite maneuver profile."""
    RUN_ONCE = 10
    """Run once profile."""
    SNOPT_OPTIMIZER = 11
    """SNOPT Optimizer profile."""
    IPOPT_OPTIMIZER = 12
    """IPOPT Optimizer profile."""
    LAMBERT_PROFILE = 13
    """Lambert profile."""
    LAMBERT_SEARCH_PROFILE = 14
    """Lambert Search profile."""
    GOLDEN_SECTION = 15
    """Golden Section profile."""
    GRID_SEARCH = 16
    """grid Search profile."""
    BISECTION = 17
    """Single Parameter Bisection profile."""

Profile.SEARCH_PLUGIN.__doc__ = "Plugin search profile."
Profile.DIFFERENTIAL_CORRECTOR.__doc__ = "Differential corrector profile."
Profile.CHANGE_MANEUVER_TYPE.__doc__ = "Change maneuver type profile."
Profile.SCRIPTING_TOOL.__doc__ = "Scripting tool profile."
Profile.CHANGE_RETURN_SEGMENT.__doc__ = "Change return segment profile."
Profile.CHANGE_PROPAGATOR.__doc__ = "Change propagator profile."
Profile.CHANGE_STOP_SEGMENT.__doc__ = "Change stop segment profile."
Profile.CHANGE_STOPPING_CONDITION_STATE.__doc__ = "Change stopping condition state profile."
Profile.SEED_FINITE_MANEUVER.__doc__ = "Change seed finite maneuver profile."
Profile.RUN_ONCE.__doc__ = "Run once profile."
Profile.SNOPT_OPTIMIZER.__doc__ = "SNOPT Optimizer profile."
Profile.IPOPT_OPTIMIZER.__doc__ = "IPOPT Optimizer profile."
Profile.LAMBERT_PROFILE.__doc__ = "Lambert profile."
Profile.LAMBERT_SEARCH_PROFILE.__doc__ = "Lambert Search profile."
Profile.GOLDEN_SECTION.__doc__ = "Golden Section profile."
Profile.GRID_SEARCH.__doc__ = "grid Search profile."
Profile.BISECTION.__doc__ = "Single Parameter Bisection profile."

agcls.AgTypeNameMap["Profile"] = Profile

class AccessCriterion(IntEnum):
    """The criteria of an Access stopping condition."""

    GAIN = 0
    """The Gain Access criterion."""
    LOSE = 1
    """The Lose Access criterion."""
    EITHER = 2
    """The Either criterion."""

AccessCriterion.GAIN.__doc__ = "The Gain Access criterion."
AccessCriterion.LOSE.__doc__ = "The Lose Access criterion."
AccessCriterion.EITHER.__doc__ = "The Either criterion."

agcls.AgTypeNameMap["AccessCriterion"] = AccessCriterion

class EclipsingBodiesSource(IntEnum):
    """The source types of the eclipsing bodies list."""

    PROPAGATOR_CENTRAL_BODY = 0
    """The eclipsing bodies list of the propagator's central body."""
    USER_DEFINED = 1
    """A user-defined eclipsing bodies list related to the propagator's central body."""
    VEHICLE_CENTRAL_BODY = 2
    """The eclipsing bodies list of the vehicle's central body."""
    VEHICLE_USER_DEFINED = 3
    """A list of eclipsing bodies from the vehicle's Basic Properties->Eclipse Bodies list."""

EclipsingBodiesSource.PROPAGATOR_CENTRAL_BODY.__doc__ = "The eclipsing bodies list of the propagator's central body."
EclipsingBodiesSource.USER_DEFINED.__doc__ = "A user-defined eclipsing bodies list related to the propagator's central body."
EclipsingBodiesSource.VEHICLE_CENTRAL_BODY.__doc__ = "The eclipsing bodies list of the vehicle's central body."
EclipsingBodiesSource.VEHICLE_USER_DEFINED.__doc__ = "A list of eclipsing bodies from the vehicle's Basic Properties->Eclipse Bodies list."

agcls.AgTypeNameMap["EclipsingBodiesSource"] = EclipsingBodiesSource

class Criterion(IntEnum):
    """The stopping condition criterion types."""

    CROSS_DECREASING = 0
    """The Cross Decreasing criterion - the stopping condition is satisfied when the parameter reaches a value equal to the trip value while decreasing."""
    CROSS_EITHER = 1
    """The Cross Either criterion - the stopping condition is satisfied when either of the above situations occurs."""
    CROSS_INCREASING = 2
    """The Cross Increasing criterion - the stopping condition is satisfied when the parameter reaches a value equal to the trip value while increasing."""

Criterion.CROSS_DECREASING.__doc__ = "The Cross Decreasing criterion - the stopping condition is satisfied when the parameter reaches a value equal to the trip value while decreasing."
Criterion.CROSS_EITHER.__doc__ = "The Cross Either criterion - the stopping condition is satisfied when either of the above situations occurs."
Criterion.CROSS_INCREASING.__doc__ = "The Cross Increasing criterion - the stopping condition is satisfied when the parameter reaches a value equal to the trip value while increasing."

agcls.AgTypeNameMap["Criterion"] = Criterion

class CalculationObjectReference(IntEnum):
    """The calculation object Reference Selection types."""

    BASIC = 0
    """The Use Basic Reference type - uses the current satellite as the reference vehicle."""
    SPECIFIED = 1
    """The UserSpecifiedReference type - uses a user specified vehicle object as the reference vehicle."""

CalculationObjectReference.BASIC.__doc__ = "The Use Basic Reference type - uses the current satellite as the reference vehicle."
CalculationObjectReference.SPECIFIED.__doc__ = "The UserSpecifiedReference type - uses a user specified vehicle object as the reference vehicle."

agcls.AgTypeNameMap["CalculationObjectReference"] = CalculationObjectReference

class CalculationObjectCentralBodyReference(IntEnum):
    """The calculation object Central Body Reference Selection types."""

    SPECIFIED = 0
    """The Use Basic Reference type - uses the current satellite as the reference vehicle."""
    PARENT = 1
    """The UserSpecifiedReference type - uses a user specified vehicle object as the reference vehicle."""

CalculationObjectCentralBodyReference.SPECIFIED.__doc__ = "The Use Basic Reference type - uses the current satellite as the reference vehicle."
CalculationObjectCentralBodyReference.PARENT.__doc__ = "The UserSpecifiedReference type - uses a user specified vehicle object as the reference vehicle."

agcls.AgTypeNameMap["CalculationObjectCentralBodyReference"] = CalculationObjectCentralBodyReference

class CalculationObjectElement(IntEnum):
    """The calculation object Element Types."""

    BROUWER_LYDDANE_MEAN_LONG = 0
    """Brouwer-Lyddane mean elements considering both the short and long period terms (i.e. resulting from averaging over the rotation of periapse). The perturbation terms are the J2, J3, J4 and J5 oblateness terms and it considers the term involving J2^2."""
    BROUWER_LYDDANE_MEAN_SHORT = 1
    """Brouwer-Lyddane Mean Short - Brouwer-Lyddane mean elements considering only the short period terms (i.e. those involving averaging over the period of the orbit) where the only perturbation force is the oblateness arising from the J2 gravity term."""
    KOZAI_IZSAK_MEAN = 2
    """Kozai-Izsak Mean - Kozai-Izsak mean elements for which only the short period terms (i.e. those involving averaging over the period of the orbit) are considered. The only perturbation force considered is the oblateness arising from the J2 gravity term."""
    OSCULATING = 3
    """Osculating."""

CalculationObjectElement.BROUWER_LYDDANE_MEAN_LONG.__doc__ = "Brouwer-Lyddane mean elements considering both the short and long period terms (i.e. resulting from averaging over the rotation of periapse). The perturbation terms are the J2, J3, J4 and J5 oblateness terms and it considers the term involving J2^2."
CalculationObjectElement.BROUWER_LYDDANE_MEAN_SHORT.__doc__ = "Brouwer-Lyddane Mean Short - Brouwer-Lyddane mean elements considering only the short period terms (i.e. those involving averaging over the period of the orbit) where the only perturbation force is the oblateness arising from the J2 gravity term."
CalculationObjectElement.KOZAI_IZSAK_MEAN.__doc__ = "Kozai-Izsak Mean - Kozai-Izsak mean elements for which only the short period terms (i.e. those involving averaging over the period of the orbit) are considered. The only perturbation force considered is the oblateness arising from the J2 gravity term."
CalculationObjectElement.OSCULATING.__doc__ = "Osculating."

agcls.AgTypeNameMap["CalculationObjectElement"] = CalculationObjectElement

class ProfileMode(IntEnum):
    """The Target Sequence profile modes."""

    ITERATE = 0
    """Iterate - the Target Sequence will run the profile as it is configured in an attempt to attain the desired solution."""
    NOT_ACTIVE = 1
    """Not Active - the Target Sequence will ignore the profile when running."""
    RUN_ONCE = 2
    """Run Once - the Target Sequence will run the profile once according to its current configuration."""
    ACTIVE = 3
    """Active - the Target Sequence will apply the change that the profile passes when running."""

ProfileMode.ITERATE.__doc__ = "Iterate - the Target Sequence will run the profile as it is configured in an attempt to attain the desired solution."
ProfileMode.NOT_ACTIVE.__doc__ = "Not Active - the Target Sequence will ignore the profile when running."
ProfileMode.RUN_ONCE.__doc__ = "Run Once - the Target Sequence will run the profile once according to its current configuration."
ProfileMode.ACTIVE.__doc__ = "Active - the Target Sequence will apply the change that the profile passes when running."

agcls.AgTypeNameMap["ProfileMode"] = ProfileMode

class ControlStoppingCondition(IntEnum):
    """The stopping condition control types."""

    TRIP_VALUE = 0
    """Trip - the value at which the stopping condition will be satisfied."""

ControlStoppingCondition.TRIP_VALUE.__doc__ = "Trip - the value at which the stopping condition will be satisfied."

agcls.AgTypeNameMap["ControlStoppingCondition"] = ControlStoppingCondition

class StateType(IntEnum):
    """The Stop segment state types."""

    ENABLED = 0
    """Enabled - the segment stops the run of the MCS."""
    DISABLED = 1
    """Disabled - the MCS ignores this segment and continues to run."""

StateType.ENABLED.__doc__ = "Enabled - the segment stops the run of the MCS."
StateType.DISABLED.__doc__ = "Disabled - the MCS ignores this segment and continues to run."

agcls.AgTypeNameMap["StateType"] = StateType

class ReturnControl(IntEnum):
    """The Return segment control types."""

    ENABLE = 0
    """Enable - returns control of the MCS run to the parent segment."""
    DISABLE = 1
    """Disable - the MCS ignores this segment and continues to run."""
    ENABLE_EXCEPT_PROFILES_BYPASS = 2
    """Enable (except Profiles bypass)- functions as enabled except when run from a Target Sequence profile (e.g., a differential corrector), which will ignore it."""

ReturnControl.ENABLE.__doc__ = "Enable - returns control of the MCS run to the parent segment."
ReturnControl.DISABLE.__doc__ = "Disable - the MCS ignores this segment and continues to run."
ReturnControl.ENABLE_EXCEPT_PROFILES_BYPASS.__doc__ = "Enable (except Profiles bypass)- functions as enabled except when run from a Target Sequence profile (e.g., a differential corrector), which will ignore it."

agcls.AgTypeNameMap["ReturnControl"] = ReturnControl

class DrawPerturbation(IntEnum):
    """The Draw Perturbation types for a Differential Corrector profile."""

    SEGMENT_COLOR = 0
    """Segment Color - perturbations will be displayed in the trajectory color defined for the individual segments within the target sequence."""
    DONT_DRAW = 1
    """Don't Draw - perturbations will not be drawn, which may decrease the run time of the target sequence."""
    TARGETER_COLOR = 2
    """Targeter Color - perturbations will be displayed in the trajectory color defined for the target sequence."""

DrawPerturbation.SEGMENT_COLOR.__doc__ = "Segment Color - perturbations will be displayed in the trajectory color defined for the individual segments within the target sequence."
DrawPerturbation.DONT_DRAW.__doc__ = "Don't Draw - perturbations will not be drawn, which may decrease the run time of the target sequence."
DrawPerturbation.TARGETER_COLOR.__doc__ = "Targeter Color - perturbations will be displayed in the trajectory color defined for the target sequence."

agcls.AgTypeNameMap["DrawPerturbation"] = DrawPerturbation

class DerivativeCalculationMethod(IntEnum):
    """The Derivative Calculation method types for a Differential Corrector profile."""

    FORWARD = 0
    """Forward Difference - ((f(x + delta) - f(x))/ delta)."""
    CENTRAL = 1
    """Central Difference - ((f(x + delta) - f(x - delta)) / 2delta)."""
    SIGNED = 2
    """Signed Difference - if x is positive, use the forward difference; if x is negative, use the backward difference."""

DerivativeCalculationMethod.FORWARD.__doc__ = "Forward Difference - ((f(x + delta) - f(x))/ delta)."
DerivativeCalculationMethod.CENTRAL.__doc__ = "Central Difference - ((f(x + delta) - f(x - delta)) / 2delta)."
DerivativeCalculationMethod.SIGNED.__doc__ = "Signed Difference - if x is positive, use the forward difference; if x is negative, use the backward difference."

agcls.AgTypeNameMap["DerivativeCalculationMethod"] = DerivativeCalculationMethod

class ConvergenceCriteria(IntEnum):
    """The Convergence Criteria types for a Differential Corrector profile."""

    EQUALITY_CONSTRAINT_WITHIN_TOLERANCE = 0
    """Equality Constraints Satisfied - the differences between the achieved and desired equality constraint values must be within the specified tolerances for convergence."""
    CONVERVENCE_CRITERIA_EITHER_EQUALITY_CONSTRAINTS_OR_CONTROL_PARAMS = 1
    """Equality Constraints Satisfied or Parameter Variations within Tolerance - the differences between the achieved and desired EC values must be within tolerances, or the last updates to the control parameters must be within tolerances for convergence."""

ConvergenceCriteria.EQUALITY_CONSTRAINT_WITHIN_TOLERANCE.__doc__ = "Equality Constraints Satisfied - the differences between the achieved and desired equality constraint values must be within the specified tolerances for convergence."
ConvergenceCriteria.CONVERVENCE_CRITERIA_EITHER_EQUALITY_CONSTRAINTS_OR_CONTROL_PARAMS.__doc__ = "Equality Constraints Satisfied or Parameter Variations within Tolerance - the differences between the achieved and desired EC values must be within tolerances, or the last updates to the control parameters must be within tolerances for convergence."

agcls.AgTypeNameMap["ConvergenceCriteria"] = ConvergenceCriteria

class DifferentialCorrectorScalingMethod(IntEnum):
    """The Scaling Method types for a Differential Corrector profile."""

    INITIAL_VALUE = 0
    """By initial value."""
    ONE_NO_SCALING = 1
    """By one (no scaling)."""
    SPECIFIED_VALUE = 2
    """By specified value."""
    TOLERANCE = 3
    """By tolerance."""

DifferentialCorrectorScalingMethod.INITIAL_VALUE.__doc__ = "By initial value."
DifferentialCorrectorScalingMethod.ONE_NO_SCALING.__doc__ = "By one (no scaling)."
DifferentialCorrectorScalingMethod.SPECIFIED_VALUE.__doc__ = "By specified value."
DifferentialCorrectorScalingMethod.TOLERANCE.__doc__ = "By tolerance."

agcls.AgTypeNameMap["DifferentialCorrectorScalingMethod"] = DifferentialCorrectorScalingMethod

class ControlUpdate(IntEnum):
    """Update segment properties that can be selected as control parameters for a Target Sequence."""

    CD = 100
    """Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."""
    CR = 101
    """Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    DRAG_AREA = 102
    """Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."""
    DRY_MASS = 103
    """Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."""
    FUEL_DENSITY = 104
    """Fuel Density - the density of the fuel tank. Enter a value in the selected mass unit per the selected distance unit cubed (e.g. kg/m^3)."""
    FUEL_MASS = 105
    """Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."""
    RADIATION_PRESSURE_AREA = 106
    """The cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Enter a value in the selected distance unit squared."""
    RADIATION_PRESSURE_COEFFICIENT = 107
    """Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    SRP_AREA = 108
    """Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Enter a value in the selected distance unit squared."""
    TANK_PRESSURE = 109
    """Tank Pressure - the fuel tank pressure. Enter a value in the selected pressure unit (e.g. Pa)."""
    TANK_TEMPERATURE = 110
    """Tank Temperature - the temperature of the fuel tank. Enter a value in the selected temperature unit."""

ControlUpdate.CD.__doc__ = "Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."
ControlUpdate.CR.__doc__ = "Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlUpdate.DRAG_AREA.__doc__ = "Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."
ControlUpdate.DRY_MASS.__doc__ = "Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlUpdate.FUEL_DENSITY.__doc__ = "Fuel Density - the density of the fuel tank. Enter a value in the selected mass unit per the selected distance unit cubed (e.g. kg/m^3)."
ControlUpdate.FUEL_MASS.__doc__ = "Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlUpdate.RADIATION_PRESSURE_AREA.__doc__ = "The cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Enter a value in the selected distance unit squared."
ControlUpdate.RADIATION_PRESSURE_COEFFICIENT.__doc__ = "Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlUpdate.SRP_AREA.__doc__ = "Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Enter a value in the selected distance unit squared."
ControlUpdate.TANK_PRESSURE.__doc__ = "Tank Pressure - the fuel tank pressure. Enter a value in the selected pressure unit (e.g. Pa)."
ControlUpdate.TANK_TEMPERATURE.__doc__ = "Tank Temperature - the temperature of the fuel tank. Enter a value in the selected temperature unit."

agcls.AgTypeNameMap["ControlUpdate"] = ControlUpdate

class ControlFollow(IntEnum):
    """Follow segment properties that can be selected as control parameters for a Target Sequence."""

    FUEL_MASS = 200
    """Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."""
    CD = 201
    """Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."""
    CR = 202
    """Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    DRAG_AREA = 203
    """Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."""
    DRY_MASS = 204
    """Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."""
    FUEL_DENSITY = 205
    """Fuel Density - the density of the fuel tank. Enter a value in the selected mass unit per the selected distance unit cubed (e.g. kg/m^3)."""
    K1 = 206
    """GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."""
    K2 = 207
    """GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (Y bias) value."""
    RADIATION_PRESSURE_AREA = 208
    """The cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Enter a value in the selected distance unit squared."""
    CK = 209
    """Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    SRP_AREA = 210
    """Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Enter a value in the selected distance unit squared."""
    TANK_PRESSURE = 211
    """Tank Pressure - the fuel tank pressure. Enter a value in the selected pressure unit (e.g. Pa)."""
    TANK_TEMPERATURE = 212
    """Tank Temperature - the temperature of the fuel tank. Enter a value in the selected temperature unit."""
    MAX_FUEL_MASS = 213
    """Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences. Enter a value in the selected mass unit (e.g. kg)."""
    TANK_VOLUME = 214
    """Tank Volume - the volume of the fuel tank. Enter a value in the selected distance unit cubed (e.g. m^3). The minimum value is 1e-019 m^3."""
    X_OFFSET = 215
    """X Offset - the distance that the spacecraft will be offset from the leader's body frame along the X axis."""
    Y_OFFSET = 216
    """Y Offset - the distance that the spacecraft will be offset from the leader's body frame along the Y axis."""
    Z_OFFSET = 217
    """Z Offset - the distance that the spacecraft will be offset from the leader's body frame along the Z axis."""

ControlFollow.FUEL_MASS.__doc__ = "Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlFollow.CD.__doc__ = "Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."
ControlFollow.CR.__doc__ = "Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlFollow.DRAG_AREA.__doc__ = "Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."
ControlFollow.DRY_MASS.__doc__ = "Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlFollow.FUEL_DENSITY.__doc__ = "Fuel Density - the density of the fuel tank. Enter a value in the selected mass unit per the selected distance unit cubed (e.g. kg/m^3)."
ControlFollow.K1.__doc__ = "GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."
ControlFollow.K2.__doc__ = "GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (Y bias) value."
ControlFollow.RADIATION_PRESSURE_AREA.__doc__ = "The cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Enter a value in the selected distance unit squared."
ControlFollow.CK.__doc__ = "Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlFollow.SRP_AREA.__doc__ = "Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Enter a value in the selected distance unit squared."
ControlFollow.TANK_PRESSURE.__doc__ = "Tank Pressure - the fuel tank pressure. Enter a value in the selected pressure unit (e.g. Pa)."
ControlFollow.TANK_TEMPERATURE.__doc__ = "Tank Temperature - the temperature of the fuel tank. Enter a value in the selected temperature unit."
ControlFollow.MAX_FUEL_MASS.__doc__ = "Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences. Enter a value in the selected mass unit (e.g. kg)."
ControlFollow.TANK_VOLUME.__doc__ = "Tank Volume - the volume of the fuel tank. Enter a value in the selected distance unit cubed (e.g. m^3). The minimum value is 1e-019 m^3."
ControlFollow.X_OFFSET.__doc__ = "X Offset - the distance that the spacecraft will be offset from the leader's body frame along the X axis."
ControlFollow.Y_OFFSET.__doc__ = "Y Offset - the distance that the spacecraft will be offset from the leader's body frame along the Y axis."
ControlFollow.Z_OFFSET.__doc__ = "Z Offset - the distance that the spacecraft will be offset from the leader's body frame along the Z axis."

agcls.AgTypeNameMap["ControlFollow"] = ControlFollow

class ControlInitState(IntEnum):
    """Initial State segment properties that can be selected as control parameters for a Target Sequence."""

    FUEL_MASS = 300
    """Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."""
    CARTESIAN_VX = 301
    """Vx Component - the X component of the spacecraft velocity vector."""
    CARTESIAN_VY = 302
    """Vy Component - the Y component of the spacecraft velocity vector."""
    CARTESIAN_VZ = 303
    """Vz Component - the Z component of the spacecraft velocity vector."""
    CARTESIAN_X = 304
    """X Component - the X component of the spacecraft position vector."""
    CARTESIAN_Y = 305
    """Y Component - the Y component of the spacecraft position vector."""
    CARTESIAN_Z = 306
    """Z Component - the Z component of the spacecraft position vector."""
    CD = 307
    """Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."""
    CR = 308
    """Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    DRAG_AREA = 309
    """Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."""
    DRY_MASS = 310
    """Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."""
    EPOCH = 311
    """Epoch - the orbit epoch."""
    FUEL_DENSITY = 312
    """Fuel Density - the density of the fuel tank."""
    K1 = 313
    """GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."""
    K2 = 314
    """GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (Y bias) value."""
    KEPLERIAN_ECCENTRICITY = 315
    """Eccentricity - the ratio of the distance between the foci to the major axis of the orbital ellipse. Dimensionless."""
    KEPLERIAN_INCLINATION = 316
    """Inclination - the angle from the +Z axis of the coordinate system to the angular momentum vector of the spacecraft's orbit."""
    KEPLERIAN_RAAN = 317
    """Right Ascension of Ascending Node - the angle between the X direction of the coordinate system and the point where the orbit crosses the X-Y plane in the +Z direction."""
    KEPLERIAN_SEMIMAJOR_AXIS = 318
    """Semimajor Axis - half the length of the major (longest) axis of the orbital ellipse."""
    KEPLERIAN_TRUE_ANOMALY = 319
    """True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."""
    KEPLERIAN_W = 320
    """Argument of Periapsis - the angle measured in the direction of spacecraft motion, in the orbit plane, from the ascending node to the periapsis of the orbit."""
    RADIATION_PRESSURE_AREA = 327
    """Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."""
    CK = 328
    """Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    SPHERICAL_AZIMUTH = 329
    """Velocity Azimuth - the angle in the spacecraft local horizontal plane between the projection of the velocity vector onto that plane and the local +Z direction measured as positive in the clockwise direction from north."""
    SPHERICAL_DECLINATION = 330
    """Declination - the angle from the X-Y plane of the coordinate system to the spacecraft position vector."""
    SPHERICAL_HORIZONTAL_FLIGHT_PATH_ANGLE = 331
    """Horizontal Flight Path Angle - the complement of the angle between the spacecraft velocity vector and the radius vector (90 deg minus the vertical flight path angle)."""
    SPHERICAL_RIGHT_ASCENSION = 332
    """Right Ascension - angle measured in the inertial equatorial plane from the inertial X axis in a right-handed sense about the inertial Z axis to the spacecraft position vector."""
    SPHERICAL_RADIUS_MAGNITUDE = 333
    """Radius Magnitude - the magnitude of the spacecraft position vector."""
    SPHERICAL_VELOCITY_MAGNITUDE = 334
    """Velocity Magnitude - the magnitude of the spacecraft velocity vector."""
    SRP_AREA = 335
    """Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."""
    TANK_PRESSURE = 336
    """Tank Pressure - the fuel tank pressure."""
    TANK_TEMPERATURE = 337
    """Tank Temperature - the temperature of the fuel tank."""
    TARGET_VECTOR_INCOMING_ASYMPTOTE_DECLINATION = 338
    """Declination of Incoming Asymptote - the declination of the incoming asymptote in the selected coordinate system."""
    TARGET_VECTOR_INCOMING_ASYMPTOTE_RIGHT_ASCENSION = 339
    """Right Ascension of Incoming Asymptote - the right ascension of the hyperbolic incoming asymptote in the selected coordinate system."""
    TARGET_VECTOR_INCOMING_VELOCITY_AZIMUTH_AT_PERIAPSIS = 340
    """Velocity Azimuth at Periapsis - the inertial flight path azimuth of the trajectory measured at periapsis."""
    TARGET_VECTOR_INCOMING_C3 = 341
    """C3 Energy - the energy of the orbit, computed as - mu / a, where mu is the gravity constant of the central body and a is the semimajor axis."""
    TARGET_VECTOR_INCOMING_RADIUS_OF_PERIAPSIS = 342
    """Radius of Periapsis - distance from the center of mass of the central body to the periapsis of the hyperbolic orbit."""
    TARGET_VECTOR_INCOMING_TRUE_ANOMALY = 343
    """True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."""
    TARGET_VECTOR_OUTGOING_ASYMPTOTE_DECLINATION = 344
    """Declination of Outgoing Asymptote - the declination of the outgoing asymptote in the selected coordinate system."""
    TARGET_VECTOR_OUTGOING_ASYMPTOTE_RIGHT_ASCENSION = 345
    """Right Ascension of Outgoing Asymptote - the right ascension of the hyperbolic outgoing asymptote in the selected coordinate system."""
    TARGET_VECTOR_OUTGOING_VELOCITY_AZIMUTH_AT_PERIAPSIS = 346
    """Velocity Azimuth at Periapsis - the inertial flight path azimuth of the trajectory measured at periapsis."""
    TARGET_VECTOR_OUTGOING_C3 = 347
    """C3 Energy - the energy of the orbit, computed as - mu / a, where mu is the gravity constant of the central body and a is the semimajor axis."""
    TARGET_VECTOR_OUTGOING_RADIUS_OF_PERIAPSIS = 348
    """Radius of Periapsis - distance from the center of mass of the central body to the periapsis of the hyperbolic orbit."""
    TARGET_VECTOR_OUTGOING_TRUE_ANOMALY = 349
    """True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."""
    MAX_FUEL_MASS = 350
    """Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences."""
    TANK_VOLUME = 351
    """Tank Volume - the volume of the fuel tank."""
    DELAUNAY_G = 352
    """G - Magnitude of orbital angular momentum, (G: sqrt(GM * p))."""
    DELAUNAY_H = 353
    """H - Z component of orbital angular momentum, (H: G cos(inc))."""
    DELAUNAY_INCLINATION = 354
    """Inclination - The angle between the angular momentum vector (perpendicular to the plane of the orbit) and the inertial Z axis."""
    DELAUNAY_L = 355
    """L - Related to two-body orbital energy, (L: sqrt(GM * a))."""
    DELAUNAY_MEAN_ANOMALY = 356
    """Mean Anomaly - The angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate."""
    DELAUNAY_RAAN = 357
    """Right Ascension of Ascending Node - The angle from the inertial X axis to the ascending node measured in a right-handed sense about the inertial Z axis in the equatorial plane."""
    DELAUNAY_SEMILATUS_RECTUM = 358
    """Semi-latus Rectum - Distance from focus to orbit at true anomaly of 90 degrees."""
    DELAUNAY_SEMIMAJOR_AXIS = 359
    """Semimajor Axis - Half the length of the major (longest) axis of the orbital ellipse."""
    DELAUNAY_W = 360
    """Argument of Periapsis - The angle from the ascending node to the eccentricity vector measured in the direction of the satellite's motion and in the orbit plane."""
    EQUINOCTIAL_H = 361
    """H - With K, describe the shape and position of periapsis of the orbit, (H: ecc * sin(RAAN + w))."""
    EQUINOCTIAL_K = 362
    """K - With H, describe the shape and position of periapsis of the orbit, (K: ecc * cos(RAAN + w))."""
    EQUINOCTIAL_MEAN_LONGITUDE = 363
    """Mean Longitude - (RAAN + w + M)."""
    EQUINOCTIAL_MEAN_MOTION = 364
    """Mean Motion - The number of orbits per day (86400 sec/period), based on assumed two-body motion."""
    EQUINOCTIAL_P = 365
    """P - With Q, describes the orientation of the orbit plane, (P: tan(inc/2) * sin(RAAN))."""
    EQUINOCTIAL_Q = 366
    """Q - With P, describes the orientation of the orbit plane, (Q: tan(inc/2) * cos(RAAN))."""
    EQUINOCTIAL_SEMIMAJOR_AXIS = 367
    """Semimajor Axis - Half the length of the major (longest) axis of the orbital ellipse."""
    MIXED_SPHERICAL_ALTITUDE = 368
    """Altitude - The object's position above or below the reference ellipsoid. Altitude is measured along a normal to the surface of the reference ellipsoid."""
    MIXED_SPHERICAL_AZIMUTH = 369
    """Azimuth - The angle in the satellite local horizontal plane between the projection of the inertial velocity vector onto this plane and the local north direction measured as positive in the clockwise direction."""
    MIXED_SPHERICAL_HORIZONTAL_FLIGHT_PATH_ANGLE = 370
    """Horizontal Flight Path Angle - The complement of the angle between the inertial velocity vector and the radius vector."""
    MIXED_SPHERICAL_LATITUDE = 371
    """Latitude - The geodetic latitude of a point is the angle between the normal to the reference ellipsoid that passes through the satellite position and the equatorial plane."""
    MIXED_SPHERICAL_LONGITUDE = 372
    """Longitude."""
    MIXED_SPHERICAL_VERTICAL_FLIGHT_PATH_ANGLE = 373
    """Vertical Flight Path Angle -  The angle between the inertial velocity vector and the radius vector."""
    MIXED_SPHERICAL_V_MAGNITUDE = 374
    """Velocity Magnitude - The magnitude of the inertial velocity vector."""
    SPHERICAL_VERTICAL_FLIGHT_PATH_ANGLE = 375
    """Vertical Flight Path Angle -  The angle between the inertial velocity vector and the radius vector."""
    KEPLERIAN_APOAPSIS_ALTITUDE_SHAPE = 376
    """Apoapsis Altitude - Shape Parameter - Distance from the surface of the central body to the point of maximum radius in the orbit."""
    KEPLERIAN_APOAPSIS_ALTITUDE_SIZE = 377
    """Apoapsis Altitude - Size Parameter - Distance from the surface of the central body to the point of maximum radius in the orbit."""
    KEPLERIAN_APOAPSIS_RADIUS_SHAPE = 378
    """Apoapsis Radius - Shape Parameter - Distance from the center of the central body to the point of maximum radius in the orbit."""
    KEPLERIAN_APOAPSIS_RADIUS_SIZE = 379
    """Apoapsis Radius - Size Parameter - Distance from the center of the central body to the point of maximum radius in the orbit."""
    KEPLERIAN_ARGUMENT_LATITUDE = 380
    """Argument of Latitude - The sum of the True Anomaly and the Argument of Perigee."""
    KEPLERIAN_ECCENTRIC_ANOMALY = 381
    """Eccentric Anomaly - Angle measured with an origin at the center of the ellipse from the direction of perigee to a point on a circumscribing circle from which a line perpendicular to the SMA intersects the position of the satellite on the ellipse."""
    KEPLERIAN_LONGITUDE_OF_ASCENDING_NODE = 382
    """Longitude of Ascending Node - Longitude of the Ascending Node is the Earth-fixed longitude where the satellite has crossed the inertial equator from south to north based on an assumption of two-body motion."""
    KEPLERIAN_MEAN_ANOMALY = 383
    """Mean Anomaly - The angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate."""
    KEPLERIAN_MEAN_MOTION = 384
    """Mean Motion - The number of orbits per day (86400 sec/period), based on assumed two-body motion."""
    KEPLERIAN_PERIAPSIS_ALTITUDE_SHAPE = 385
    """Periapsis Altitude - Shape Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."""
    KEPLERIAN_PERIAPSIS_ALTITUDE_SIZE = 386
    """Periapsis Altitude - Size Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."""
    KEPLERIAN_PERIAPSIS_RADIUS_SHAPE = 387
    """Periapsis Radius - Shape Parameter - Distance from the center of the central body to the point of minimum radius in the orbit."""
    KEPLERIAN_PERIAPSIS_RADIUS_SIZE = 388
    """Periapsis Radius - Size Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."""
    KEPLERIAN_PERIOD = 389
    """Period - The duration of one orbit, based on assumed two-body motion."""
    KEPLERIAN_TIME_PAST_ASCENDING_NODE = 390
    """Time Past Ascending Node - The elapsed time since the last ascending node crossing based on assumed two-body motion."""
    KEPLERIAN_TIME_PAST_PERIAPSIS = 391
    """Time Past Periapsis - The elapsed time since the last perigee passage based on assumed two-body motion."""
    SPHERICAL_RANGE_RATE_DECLINATION = 392
    """Declination - the angle from the X-Y plane of the coordinate system to the spacecraft position vector."""
    SPHERICAL_RANGE_RATE_RIGHT_ASCENSION = 393
    """Right Ascension - angle measured in the inertial equatorial plane from the inertial X axis in a right-handed sense about the inertial Z axis to the spacecraft position vector."""
    SPHERICAL_RANGE_RATE_RANGE = 394
    """Range - distance of an object from the center point of the coordinate system."""
    SPHERICAL_RANGE_RATE_DECLINATION_RATE = 395
    """Declination Rate - the change of the declination angle over time."""
    SPHERICAL_RANGE_RATE_RIGHT_ASCENSION_RATE = 396
    """Right Ascension Rate - the change of the right ascension angle over time."""
    SPHERICAL_RANGE_RATE_RANGE_RATE = 397
    """Range Rate - the change in the range over time."""

ControlInitState.FUEL_MASS.__doc__ = "Fuel Mass - the mass of the spacecraft propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlInitState.CARTESIAN_VX.__doc__ = "Vx Component - the X component of the spacecraft velocity vector."
ControlInitState.CARTESIAN_VY.__doc__ = "Vy Component - the Y component of the spacecraft velocity vector."
ControlInitState.CARTESIAN_VZ.__doc__ = "Vz Component - the Z component of the spacecraft velocity vector."
ControlInitState.CARTESIAN_X.__doc__ = "X Component - the X component of the spacecraft position vector."
ControlInitState.CARTESIAN_Y.__doc__ = "Y Component - the Y component of the spacecraft position vector."
ControlInitState.CARTESIAN_Z.__doc__ = "Z Component - the Z component of the spacecraft position vector."
ControlInitState.CD.__doc__ = "Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."
ControlInitState.CR.__doc__ = "Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlInitState.DRAG_AREA.__doc__ = "Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Enter a value in the selected distance unit squared."
ControlInitState.DRY_MASS.__doc__ = "Dry Mass - the mass of the spacecraft exclusive of propellant. Enter a value in the selected mass unit (e.g. kg)."
ControlInitState.EPOCH.__doc__ = "Epoch - the orbit epoch."
ControlInitState.FUEL_DENSITY.__doc__ = "Fuel Density - the density of the fuel tank."
ControlInitState.K1.__doc__ = "GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."
ControlInitState.K2.__doc__ = "GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (Y bias) value."
ControlInitState.KEPLERIAN_ECCENTRICITY.__doc__ = "Eccentricity - the ratio of the distance between the foci to the major axis of the orbital ellipse. Dimensionless."
ControlInitState.KEPLERIAN_INCLINATION.__doc__ = "Inclination - the angle from the +Z axis of the coordinate system to the angular momentum vector of the spacecraft's orbit."
ControlInitState.KEPLERIAN_RAAN.__doc__ = "Right Ascension of Ascending Node - the angle between the X direction of the coordinate system and the point where the orbit crosses the X-Y plane in the +Z direction."
ControlInitState.KEPLERIAN_SEMIMAJOR_AXIS.__doc__ = "Semimajor Axis - half the length of the major (longest) axis of the orbital ellipse."
ControlInitState.KEPLERIAN_TRUE_ANOMALY.__doc__ = "True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."
ControlInitState.KEPLERIAN_W.__doc__ = "Argument of Periapsis - the angle measured in the direction of spacecraft motion, in the orbit plane, from the ascending node to the periapsis of the orbit."
ControlInitState.RADIATION_PRESSURE_AREA.__doc__ = "Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."
ControlInitState.CK.__doc__ = "Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlInitState.SPHERICAL_AZIMUTH.__doc__ = "Velocity Azimuth - the angle in the spacecraft local horizontal plane between the projection of the velocity vector onto that plane and the local +Z direction measured as positive in the clockwise direction from north."
ControlInitState.SPHERICAL_DECLINATION.__doc__ = "Declination - the angle from the X-Y plane of the coordinate system to the spacecraft position vector."
ControlInitState.SPHERICAL_HORIZONTAL_FLIGHT_PATH_ANGLE.__doc__ = "Horizontal Flight Path Angle - the complement of the angle between the spacecraft velocity vector and the radius vector (90 deg minus the vertical flight path angle)."
ControlInitState.SPHERICAL_RIGHT_ASCENSION.__doc__ = "Right Ascension - angle measured in the inertial equatorial plane from the inertial X axis in a right-handed sense about the inertial Z axis to the spacecraft position vector."
ControlInitState.SPHERICAL_RADIUS_MAGNITUDE.__doc__ = "Radius Magnitude - the magnitude of the spacecraft position vector."
ControlInitState.SPHERICAL_VELOCITY_MAGNITUDE.__doc__ = "Velocity Magnitude - the magnitude of the spacecraft velocity vector."
ControlInitState.SRP_AREA.__doc__ = "Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."
ControlInitState.TANK_PRESSURE.__doc__ = "Tank Pressure - the fuel tank pressure."
ControlInitState.TANK_TEMPERATURE.__doc__ = "Tank Temperature - the temperature of the fuel tank."
ControlInitState.TARGET_VECTOR_INCOMING_ASYMPTOTE_DECLINATION.__doc__ = "Declination of Incoming Asymptote - the declination of the incoming asymptote in the selected coordinate system."
ControlInitState.TARGET_VECTOR_INCOMING_ASYMPTOTE_RIGHT_ASCENSION.__doc__ = "Right Ascension of Incoming Asymptote - the right ascension of the hyperbolic incoming asymptote in the selected coordinate system."
ControlInitState.TARGET_VECTOR_INCOMING_VELOCITY_AZIMUTH_AT_PERIAPSIS.__doc__ = "Velocity Azimuth at Periapsis - the inertial flight path azimuth of the trajectory measured at periapsis."
ControlInitState.TARGET_VECTOR_INCOMING_C3.__doc__ = "C3 Energy - the energy of the orbit, computed as - mu / a, where mu is the gravity constant of the central body and a is the semimajor axis."
ControlInitState.TARGET_VECTOR_INCOMING_RADIUS_OF_PERIAPSIS.__doc__ = "Radius of Periapsis - distance from the center of mass of the central body to the periapsis of the hyperbolic orbit."
ControlInitState.TARGET_VECTOR_INCOMING_TRUE_ANOMALY.__doc__ = "True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."
ControlInitState.TARGET_VECTOR_OUTGOING_ASYMPTOTE_DECLINATION.__doc__ = "Declination of Outgoing Asymptote - the declination of the outgoing asymptote in the selected coordinate system."
ControlInitState.TARGET_VECTOR_OUTGOING_ASYMPTOTE_RIGHT_ASCENSION.__doc__ = "Right Ascension of Outgoing Asymptote - the right ascension of the hyperbolic outgoing asymptote in the selected coordinate system."
ControlInitState.TARGET_VECTOR_OUTGOING_VELOCITY_AZIMUTH_AT_PERIAPSIS.__doc__ = "Velocity Azimuth at Periapsis - the inertial flight path azimuth of the trajectory measured at periapsis."
ControlInitState.TARGET_VECTOR_OUTGOING_C3.__doc__ = "C3 Energy - the energy of the orbit, computed as - mu / a, where mu is the gravity constant of the central body and a is the semimajor axis."
ControlInitState.TARGET_VECTOR_OUTGOING_RADIUS_OF_PERIAPSIS.__doc__ = "Radius of Periapsis - distance from the center of mass of the central body to the periapsis of the hyperbolic orbit."
ControlInitState.TARGET_VECTOR_OUTGOING_TRUE_ANOMALY.__doc__ = "True Anomaly - the angle from the periapsis of the orbit to the spacecraft's position vector, measured in the direction of spacecraft motion."
ControlInitState.MAX_FUEL_MASS.__doc__ = "Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences."
ControlInitState.TANK_VOLUME.__doc__ = "Tank Volume - the volume of the fuel tank."
ControlInitState.DELAUNAY_G.__doc__ = "G - Magnitude of orbital angular momentum, (G: sqrt(GM * p))."
ControlInitState.DELAUNAY_H.__doc__ = "H - Z component of orbital angular momentum, (H: G cos(inc))."
ControlInitState.DELAUNAY_INCLINATION.__doc__ = "Inclination - The angle between the angular momentum vector (perpendicular to the plane of the orbit) and the inertial Z axis."
ControlInitState.DELAUNAY_L.__doc__ = "L - Related to two-body orbital energy, (L: sqrt(GM * a))."
ControlInitState.DELAUNAY_MEAN_ANOMALY.__doc__ = "Mean Anomaly - The angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate."
ControlInitState.DELAUNAY_RAAN.__doc__ = "Right Ascension of Ascending Node - The angle from the inertial X axis to the ascending node measured in a right-handed sense about the inertial Z axis in the equatorial plane."
ControlInitState.DELAUNAY_SEMILATUS_RECTUM.__doc__ = "Semi-latus Rectum - Distance from focus to orbit at true anomaly of 90 degrees."
ControlInitState.DELAUNAY_SEMIMAJOR_AXIS.__doc__ = "Semimajor Axis - Half the length of the major (longest) axis of the orbital ellipse."
ControlInitState.DELAUNAY_W.__doc__ = "Argument of Periapsis - The angle from the ascending node to the eccentricity vector measured in the direction of the satellite's motion and in the orbit plane."
ControlInitState.EQUINOCTIAL_H.__doc__ = "H - With K, describe the shape and position of periapsis of the orbit, (H: ecc * sin(RAAN + w))."
ControlInitState.EQUINOCTIAL_K.__doc__ = "K - With H, describe the shape and position of periapsis of the orbit, (K: ecc * cos(RAAN + w))."
ControlInitState.EQUINOCTIAL_MEAN_LONGITUDE.__doc__ = "Mean Longitude - (RAAN + w + M)."
ControlInitState.EQUINOCTIAL_MEAN_MOTION.__doc__ = "Mean Motion - The number of orbits per day (86400 sec/period), based on assumed two-body motion."
ControlInitState.EQUINOCTIAL_P.__doc__ = "P - With Q, describes the orientation of the orbit plane, (P: tan(inc/2) * sin(RAAN))."
ControlInitState.EQUINOCTIAL_Q.__doc__ = "Q - With P, describes the orientation of the orbit plane, (Q: tan(inc/2) * cos(RAAN))."
ControlInitState.EQUINOCTIAL_SEMIMAJOR_AXIS.__doc__ = "Semimajor Axis - Half the length of the major (longest) axis of the orbital ellipse."
ControlInitState.MIXED_SPHERICAL_ALTITUDE.__doc__ = "Altitude - The object's position above or below the reference ellipsoid. Altitude is measured along a normal to the surface of the reference ellipsoid."
ControlInitState.MIXED_SPHERICAL_AZIMUTH.__doc__ = "Azimuth - The angle in the satellite local horizontal plane between the projection of the inertial velocity vector onto this plane and the local north direction measured as positive in the clockwise direction."
ControlInitState.MIXED_SPHERICAL_HORIZONTAL_FLIGHT_PATH_ANGLE.__doc__ = "Horizontal Flight Path Angle - The complement of the angle between the inertial velocity vector and the radius vector."
ControlInitState.MIXED_SPHERICAL_LATITUDE.__doc__ = "Latitude - The geodetic latitude of a point is the angle between the normal to the reference ellipsoid that passes through the satellite position and the equatorial plane."
ControlInitState.MIXED_SPHERICAL_LONGITUDE.__doc__ = "Longitude."
ControlInitState.MIXED_SPHERICAL_VERTICAL_FLIGHT_PATH_ANGLE.__doc__ = "Vertical Flight Path Angle -  The angle between the inertial velocity vector and the radius vector."
ControlInitState.MIXED_SPHERICAL_V_MAGNITUDE.__doc__ = "Velocity Magnitude - The magnitude of the inertial velocity vector."
ControlInitState.SPHERICAL_VERTICAL_FLIGHT_PATH_ANGLE.__doc__ = "Vertical Flight Path Angle -  The angle between the inertial velocity vector and the radius vector."
ControlInitState.KEPLERIAN_APOAPSIS_ALTITUDE_SHAPE.__doc__ = "Apoapsis Altitude - Shape Parameter - Distance from the surface of the central body to the point of maximum radius in the orbit."
ControlInitState.KEPLERIAN_APOAPSIS_ALTITUDE_SIZE.__doc__ = "Apoapsis Altitude - Size Parameter - Distance from the surface of the central body to the point of maximum radius in the orbit."
ControlInitState.KEPLERIAN_APOAPSIS_RADIUS_SHAPE.__doc__ = "Apoapsis Radius - Shape Parameter - Distance from the center of the central body to the point of maximum radius in the orbit."
ControlInitState.KEPLERIAN_APOAPSIS_RADIUS_SIZE.__doc__ = "Apoapsis Radius - Size Parameter - Distance from the center of the central body to the point of maximum radius in the orbit."
ControlInitState.KEPLERIAN_ARGUMENT_LATITUDE.__doc__ = "Argument of Latitude - The sum of the True Anomaly and the Argument of Perigee."
ControlInitState.KEPLERIAN_ECCENTRIC_ANOMALY.__doc__ = "Eccentric Anomaly - Angle measured with an origin at the center of the ellipse from the direction of perigee to a point on a circumscribing circle from which a line perpendicular to the SMA intersects the position of the satellite on the ellipse."
ControlInitState.KEPLERIAN_LONGITUDE_OF_ASCENDING_NODE.__doc__ = "Longitude of Ascending Node - Longitude of the Ascending Node is the Earth-fixed longitude where the satellite has crossed the inertial equator from south to north based on an assumption of two-body motion."
ControlInitState.KEPLERIAN_MEAN_ANOMALY.__doc__ = "Mean Anomaly - The angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate."
ControlInitState.KEPLERIAN_MEAN_MOTION.__doc__ = "Mean Motion - The number of orbits per day (86400 sec/period), based on assumed two-body motion."
ControlInitState.KEPLERIAN_PERIAPSIS_ALTITUDE_SHAPE.__doc__ = "Periapsis Altitude - Shape Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."
ControlInitState.KEPLERIAN_PERIAPSIS_ALTITUDE_SIZE.__doc__ = "Periapsis Altitude - Size Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."
ControlInitState.KEPLERIAN_PERIAPSIS_RADIUS_SHAPE.__doc__ = "Periapsis Radius - Shape Parameter - Distance from the center of the central body to the point of minimum radius in the orbit."
ControlInitState.KEPLERIAN_PERIAPSIS_RADIUS_SIZE.__doc__ = "Periapsis Radius - Size Parameter - Distance from the surface of the central body to the point of minimum radius in the orbit."
ControlInitState.KEPLERIAN_PERIOD.__doc__ = "Period - The duration of one orbit, based on assumed two-body motion."
ControlInitState.KEPLERIAN_TIME_PAST_ASCENDING_NODE.__doc__ = "Time Past Ascending Node - The elapsed time since the last ascending node crossing based on assumed two-body motion."
ControlInitState.KEPLERIAN_TIME_PAST_PERIAPSIS.__doc__ = "Time Past Periapsis - The elapsed time since the last perigee passage based on assumed two-body motion."
ControlInitState.SPHERICAL_RANGE_RATE_DECLINATION.__doc__ = "Declination - the angle from the X-Y plane of the coordinate system to the spacecraft position vector."
ControlInitState.SPHERICAL_RANGE_RATE_RIGHT_ASCENSION.__doc__ = "Right Ascension - angle measured in the inertial equatorial plane from the inertial X axis in a right-handed sense about the inertial Z axis to the spacecraft position vector."
ControlInitState.SPHERICAL_RANGE_RATE_RANGE.__doc__ = "Range - distance of an object from the center point of the coordinate system."
ControlInitState.SPHERICAL_RANGE_RATE_DECLINATION_RATE.__doc__ = "Declination Rate - the change of the declination angle over time."
ControlInitState.SPHERICAL_RANGE_RATE_RIGHT_ASCENSION_RATE.__doc__ = "Right Ascension Rate - the change of the right ascension angle over time."
ControlInitState.SPHERICAL_RANGE_RATE_RANGE_RATE.__doc__ = "Range Rate - the change in the range over time."

agcls.AgTypeNameMap["ControlInitState"] = ControlInitState

class ControlManeuver(IntEnum):
    """Maneuver segment properties that can be selected as control parameters for a Target Sequence."""

    FINITE_CARTESIAN_X = 400
    """The Cartesian X component of the thrust vector."""
    FINITE_CARTESIAN_Y = 401
    """The Cartesian Y component of the thrust vector."""
    FINITE_CARTESIAN_Z = 402
    """The Cartesian Z component of the thrust vector."""
    FINITE_EULER_ANGLES1 = 403
    """The Euler A (Phi) component of the rotation from the reference axes to the body frame."""
    FINITE_EULER_ANGLES2 = 404
    """The Euler B (Theta) component of the rotation from the reference axes to the body frame."""
    FINITE_EULER_ANGLES3 = 405
    """The Euler C (Psi) component of the rotation from the reference axes to the body frame."""
    FINITE_SPHERICAL_AZIMUTH = 406
    """The spherical azimuth component of the thrust vector."""
    FINITE_SPHERICAL_ELEVATION = 407
    """The spherical elevation component of the thrust vector."""
    IMPULSIVE_CARTESIAN_X = 408
    """The Cartesian X component of the thrust vector."""
    IMPULSIVE_CARTESIAN_Y = 409
    """The Cartesian Y component of the thrust vector."""
    IMPULSIVE_CARTESIAN_Z = 410
    """The Cartesian Z component of the thrust vector."""
    IMPULSIVE_EULER_ANGLES1 = 411
    """The Euler A (Phi) component of the rotation from the reference axes to the body frame."""
    IMPULSIVE_EULER_ANGLES2 = 412
    """The Euler B (Theta) component of the rotation from the reference axes to the body frame."""
    IMPULSIVE_EULER_ANGLES3 = 413
    """The Euler C (Psi) component of the rotation from the reference axes to the body frame."""
    IMPULSIVE_SPHERICAL_AZIMUTH = 414
    """The spherical azimuth component of the thrust vector."""
    IMPULSIVE_SPHERICAL_ELEVATION = 415
    """The spherical elevation component of the thrust vector."""
    IMPULSIVE_SPHERICAL_MAGNITUDE = 416
    """The spherical magnitude component of the thrust vector."""
    FINITE_BURN_CENTER_BIAS = 417
    """The burn center bias of the thrust."""
    FINITE_THRUST_EFFICIENCY = 418
    """The thrust efficiency value."""
    FINITE_AZIMUTH_CONSTANT_TERM = 419
    """The time varying azimuth polynomial constant term."""
    FINITE_AZIMUTH_LINEAR_TERM = 420
    """The time varying azimuth polynomial linear term."""
    FINITE_AZIMUTH_QUADRATIC_TERM = 421
    """The time varying azimuth polynomial quadratic term."""
    FINITE_AZIMUTH_CUBIC_TERM = 422
    """The time varying azimuth polynomial cubic term."""
    FINITE_AZIMUTH_QUARTIC_TERM = 423
    """The time varying azimuth polynomial quartic term."""
    FINITE_AZIMUTH_SINUSOIDAL_AMPLITUDE = 424
    """The time varying azimuth sine amplitude term."""
    FINITE_AZIMUTH_SINUSOIDAL_FREQUENCY = 425
    """The time varying azimuth sine frequency term."""
    FINITE_AZIMUTH_SINUSOIDAL_PHASE = 426
    """The time varying azimuth sine phase term."""
    FINITE_ELEVATION_CONSTANT_TERM = 427
    """The time varying elevation polynomial constant term."""
    FINITE_ELEVATION_LINEAR_TERM = 428
    """The time varying elevation polynomial linear term."""
    FINITE_ELEVATION_QUADRATIC_TERM = 429
    """The time varying elevation polynomial quadratic term."""
    FINITE_ELEVATION_CUBIC_TERM = 430
    """The time varying elevation polynomial cubic term."""
    FINITE_ELEVATION_QUARTIC_TERM = 431
    """The time varying elevation polynomial quartic term."""
    FINITE_ELEVATION_SINUSOIDAL_AMPLITUDE = 432
    """The time varying elevation sine amplitude term."""
    FINITE_ELEVATION_SINUSOIDAL_FREQUENCY = 433
    """The time varying elevation sine frequency term."""
    FINITE_ELEVATION_SINUSOIDAL_PHASE = 434
    """The time varying elevation sine phase term."""

ControlManeuver.FINITE_CARTESIAN_X.__doc__ = "The Cartesian X component of the thrust vector."
ControlManeuver.FINITE_CARTESIAN_Y.__doc__ = "The Cartesian Y component of the thrust vector."
ControlManeuver.FINITE_CARTESIAN_Z.__doc__ = "The Cartesian Z component of the thrust vector."
ControlManeuver.FINITE_EULER_ANGLES1.__doc__ = "The Euler A (Phi) component of the rotation from the reference axes to the body frame."
ControlManeuver.FINITE_EULER_ANGLES2.__doc__ = "The Euler B (Theta) component of the rotation from the reference axes to the body frame."
ControlManeuver.FINITE_EULER_ANGLES3.__doc__ = "The Euler C (Psi) component of the rotation from the reference axes to the body frame."
ControlManeuver.FINITE_SPHERICAL_AZIMUTH.__doc__ = "The spherical azimuth component of the thrust vector."
ControlManeuver.FINITE_SPHERICAL_ELEVATION.__doc__ = "The spherical elevation component of the thrust vector."
ControlManeuver.IMPULSIVE_CARTESIAN_X.__doc__ = "The Cartesian X component of the thrust vector."
ControlManeuver.IMPULSIVE_CARTESIAN_Y.__doc__ = "The Cartesian Y component of the thrust vector."
ControlManeuver.IMPULSIVE_CARTESIAN_Z.__doc__ = "The Cartesian Z component of the thrust vector."
ControlManeuver.IMPULSIVE_EULER_ANGLES1.__doc__ = "The Euler A (Phi) component of the rotation from the reference axes to the body frame."
ControlManeuver.IMPULSIVE_EULER_ANGLES2.__doc__ = "The Euler B (Theta) component of the rotation from the reference axes to the body frame."
ControlManeuver.IMPULSIVE_EULER_ANGLES3.__doc__ = "The Euler C (Psi) component of the rotation from the reference axes to the body frame."
ControlManeuver.IMPULSIVE_SPHERICAL_AZIMUTH.__doc__ = "The spherical azimuth component of the thrust vector."
ControlManeuver.IMPULSIVE_SPHERICAL_ELEVATION.__doc__ = "The spherical elevation component of the thrust vector."
ControlManeuver.IMPULSIVE_SPHERICAL_MAGNITUDE.__doc__ = "The spherical magnitude component of the thrust vector."
ControlManeuver.FINITE_BURN_CENTER_BIAS.__doc__ = "The burn center bias of the thrust."
ControlManeuver.FINITE_THRUST_EFFICIENCY.__doc__ = "The thrust efficiency value."
ControlManeuver.FINITE_AZIMUTH_CONSTANT_TERM.__doc__ = "The time varying azimuth polynomial constant term."
ControlManeuver.FINITE_AZIMUTH_LINEAR_TERM.__doc__ = "The time varying azimuth polynomial linear term."
ControlManeuver.FINITE_AZIMUTH_QUADRATIC_TERM.__doc__ = "The time varying azimuth polynomial quadratic term."
ControlManeuver.FINITE_AZIMUTH_CUBIC_TERM.__doc__ = "The time varying azimuth polynomial cubic term."
ControlManeuver.FINITE_AZIMUTH_QUARTIC_TERM.__doc__ = "The time varying azimuth polynomial quartic term."
ControlManeuver.FINITE_AZIMUTH_SINUSOIDAL_AMPLITUDE.__doc__ = "The time varying azimuth sine amplitude term."
ControlManeuver.FINITE_AZIMUTH_SINUSOIDAL_FREQUENCY.__doc__ = "The time varying azimuth sine frequency term."
ControlManeuver.FINITE_AZIMUTH_SINUSOIDAL_PHASE.__doc__ = "The time varying azimuth sine phase term."
ControlManeuver.FINITE_ELEVATION_CONSTANT_TERM.__doc__ = "The time varying elevation polynomial constant term."
ControlManeuver.FINITE_ELEVATION_LINEAR_TERM.__doc__ = "The time varying elevation polynomial linear term."
ControlManeuver.FINITE_ELEVATION_QUADRATIC_TERM.__doc__ = "The time varying elevation polynomial quadratic term."
ControlManeuver.FINITE_ELEVATION_CUBIC_TERM.__doc__ = "The time varying elevation polynomial cubic term."
ControlManeuver.FINITE_ELEVATION_QUARTIC_TERM.__doc__ = "The time varying elevation polynomial quartic term."
ControlManeuver.FINITE_ELEVATION_SINUSOIDAL_AMPLITUDE.__doc__ = "The time varying elevation sine amplitude term."
ControlManeuver.FINITE_ELEVATION_SINUSOIDAL_FREQUENCY.__doc__ = "The time varying elevation sine frequency term."
ControlManeuver.FINITE_ELEVATION_SINUSOIDAL_PHASE.__doc__ = "The time varying elevation sine phase term."

agcls.AgTypeNameMap["ControlManeuver"] = ControlManeuver

class ControlLaunch(IntEnum):
    """Launch segment properties that can be selected as control parameters for a Target Sequence."""

    EPOCH = 500
    """The date and time of the launch."""
    GEODETIC_LATITUDE = 501
    """The geodetic latitude of the launch location."""
    GEODETIC_LONGITUDE = 502
    """The geodetic longitude of the launch location."""
    GEODETIC_ALTITUDE = 503
    """The geodetic altitude of the launch location."""
    GEOCENTRIC_LATITUDE = 504
    """The geocentric latitude of the launch location."""
    GEOCENTRIC_LONGITUDE = 505
    """The geocentric longitude of the launch location."""
    GEOCENTRIC_RADIUS = 506
    """The geocentric radius of the launch location."""
    TIME_OF_FLIGHT = 507
    """The time of flight, from launch until burnout."""
    BURNOUT_GEOCENTRIC_LATITUDE = 508
    """The geocentric latitude of the burnout point."""
    BURNOUT_GEOCENTRIC_LONGITUDE = 509
    """The geocentric longitude of the burnout point."""
    BURNOUT_GEOCENTRIC_RADIUS = 510
    """The geocentric radius of the burnout point."""
    BURNOUT_GEODETIC_LATITUDE = 511
    """The geodetic latitude of the burnout point."""
    BURNOUT_GEODETIC_LONGITUDE = 512
    """The geodetic longitude of the burnout point."""
    BURNOUT_GEODETIC_ALTITUDE = 513
    """The geodetic altitude of the burnout point."""
    BURNOUT_AZIMUTH_ALTITUDE_AZIMUTH = 514
    """The geodetic azimuth of the launch trajectory."""
    BURNOUT_AZIMUTH_ALTITUDE_DOWNRANGE_DIST = 515
    """The geodetic downrange distance of the burnout point."""
    BURNOUT_AZIMUTH_ALTITUDE_ALTITUDE = 516
    """The geodetic altitude of the burnout point."""
    BURNOUT_AZIMUTH_RADIUS_AZIMUTH = 517
    """The geocentric azimuth of the launch trajectory."""
    BURNOUT_AZIMUTH_RADIUS_DOWNRANGE_DIST = 518
    """The geocentric downrange distance of the burnout point."""
    BURNOUT_AZIMUTH_RADIUS_RADIUS = 519
    """The geocentric radius of the burnout point."""
    BURNOUT_FIXED_VELOCITY = 520
    """The burnout velocity in the fixed frame."""
    BURNOUT_INERTIAL_VELOCITY = 521
    """The burnout velocity in the inertial frame."""
    BURNOUT_INERTIAL_VELOCITY_AZIMUTH = 522
    """Inertial Velocity Azimuth - the angle from the projection of north in the local horizontal plane to the inertial velocity vector, right handed."""
    BURNOUT_INERTIAL_HORIZONTAL_FLIGHT_PATH_ANGLE = 523
    """Inertial Horizontal FPA - the angle from the local horizontal to the inertial velocity vector, positive towards radius. It is also 90 degrees minus vertical flight path angle."""
    DRY_MASS = 524
    """Dry Mass - the mass of the spacecraft exclusive of propellant."""
    CD = 525
    """Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."""
    DRAG_AREA = 526
    """Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations."""
    CR = 527
    """Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    SRP_AREA = 528
    """Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."""
    CK = 529
    """Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    RADIATION_PRESSURE_AREA = 530
    """Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."""
    K1 = 531
    """GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."""
    K2 = 532
    """GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (scale) value."""
    TANK_PRESSURE = 533
    """The fuel tank pressure."""
    TANK_VOLUME = 534
    """The volume of the fuel tank."""
    TANK_TEMP = 535
    """The fuel tank temperature."""
    FUEL_DENSITY = 536
    """The density of the fuel tank."""
    FUEL_MASS = 537
    """The mass of the spacecraft propellant."""
    MAX_FUEL_MASS = 538
    """Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences."""

ControlLaunch.EPOCH.__doc__ = "The date and time of the launch."
ControlLaunch.GEODETIC_LATITUDE.__doc__ = "The geodetic latitude of the launch location."
ControlLaunch.GEODETIC_LONGITUDE.__doc__ = "The geodetic longitude of the launch location."
ControlLaunch.GEODETIC_ALTITUDE.__doc__ = "The geodetic altitude of the launch location."
ControlLaunch.GEOCENTRIC_LATITUDE.__doc__ = "The geocentric latitude of the launch location."
ControlLaunch.GEOCENTRIC_LONGITUDE.__doc__ = "The geocentric longitude of the launch location."
ControlLaunch.GEOCENTRIC_RADIUS.__doc__ = "The geocentric radius of the launch location."
ControlLaunch.TIME_OF_FLIGHT.__doc__ = "The time of flight, from launch until burnout."
ControlLaunch.BURNOUT_GEOCENTRIC_LATITUDE.__doc__ = "The geocentric latitude of the burnout point."
ControlLaunch.BURNOUT_GEOCENTRIC_LONGITUDE.__doc__ = "The geocentric longitude of the burnout point."
ControlLaunch.BURNOUT_GEOCENTRIC_RADIUS.__doc__ = "The geocentric radius of the burnout point."
ControlLaunch.BURNOUT_GEODETIC_LATITUDE.__doc__ = "The geodetic latitude of the burnout point."
ControlLaunch.BURNOUT_GEODETIC_LONGITUDE.__doc__ = "The geodetic longitude of the burnout point."
ControlLaunch.BURNOUT_GEODETIC_ALTITUDE.__doc__ = "The geodetic altitude of the burnout point."
ControlLaunch.BURNOUT_AZIMUTH_ALTITUDE_AZIMUTH.__doc__ = "The geodetic azimuth of the launch trajectory."
ControlLaunch.BURNOUT_AZIMUTH_ALTITUDE_DOWNRANGE_DIST.__doc__ = "The geodetic downrange distance of the burnout point."
ControlLaunch.BURNOUT_AZIMUTH_ALTITUDE_ALTITUDE.__doc__ = "The geodetic altitude of the burnout point."
ControlLaunch.BURNOUT_AZIMUTH_RADIUS_AZIMUTH.__doc__ = "The geocentric azimuth of the launch trajectory."
ControlLaunch.BURNOUT_AZIMUTH_RADIUS_DOWNRANGE_DIST.__doc__ = "The geocentric downrange distance of the burnout point."
ControlLaunch.BURNOUT_AZIMUTH_RADIUS_RADIUS.__doc__ = "The geocentric radius of the burnout point."
ControlLaunch.BURNOUT_FIXED_VELOCITY.__doc__ = "The burnout velocity in the fixed frame."
ControlLaunch.BURNOUT_INERTIAL_VELOCITY.__doc__ = "The burnout velocity in the inertial frame."
ControlLaunch.BURNOUT_INERTIAL_VELOCITY_AZIMUTH.__doc__ = "Inertial Velocity Azimuth - the angle from the projection of north in the local horizontal plane to the inertial velocity vector, right handed."
ControlLaunch.BURNOUT_INERTIAL_HORIZONTAL_FLIGHT_PATH_ANGLE.__doc__ = "Inertial Horizontal FPA - the angle from the local horizontal to the inertial velocity vector, positive towards radius. It is also 90 degrees minus vertical flight path angle."
ControlLaunch.DRY_MASS.__doc__ = "Dry Mass - the mass of the spacecraft exclusive of propellant."
ControlLaunch.CD.__doc__ = "Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."
ControlLaunch.DRAG_AREA.__doc__ = "Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations."
ControlLaunch.CR.__doc__ = "Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlLaunch.SRP_AREA.__doc__ = "Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."
ControlLaunch.CK.__doc__ = "Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
ControlLaunch.RADIATION_PRESSURE_AREA.__doc__ = "Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."
ControlLaunch.K1.__doc__ = "GPS Solar Radiation Pressure K1 - if you are using a non-spherical SRP model, this field defines the model's K1 (scale) value."
ControlLaunch.K2.__doc__ = "GPS Solar Radiation Pressure K2 - if you are using a non-spherical SRP model, this field defines the model's K2 (scale) value."
ControlLaunch.TANK_PRESSURE.__doc__ = "The fuel tank pressure."
ControlLaunch.TANK_VOLUME.__doc__ = "The volume of the fuel tank."
ControlLaunch.TANK_TEMP.__doc__ = "The fuel tank temperature."
ControlLaunch.FUEL_DENSITY.__doc__ = "The density of the fuel tank."
ControlLaunch.FUEL_MASS.__doc__ = "The mass of the spacecraft propellant."
ControlLaunch.MAX_FUEL_MASS.__doc__ = "Maximum Fuel Mass - the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences."

agcls.AgTypeNameMap["ControlLaunch"] = ControlLaunch

class ControlAdvanced(IntEnum):
    """Propagate segment properties that can be selected as control parameters for a Target Sequence."""

    PROPAGATE_MAX_PROPATION_TIME = 600
    """The maximum propagation time, after which the segment will end regardless of whether the stopping conditions have been satisfied."""
    PROPAGATE_MIN_PROPAGATION_TIME = 601
    """The minimum propagation time - minimum time that must elapse from the beginning of the segment until Astrogator will begin checking stopping conditions for satisfaction."""

ControlAdvanced.PROPAGATE_MAX_PROPATION_TIME.__doc__ = "The maximum propagation time, after which the segment will end regardless of whether the stopping conditions have been satisfied."
ControlAdvanced.PROPAGATE_MIN_PROPAGATION_TIME.__doc__ = "The minimum propagation time - minimum time that must elapse from the beginning of the segment until Astrogator will begin checking stopping conditions for satisfaction."

agcls.AgTypeNameMap["ControlAdvanced"] = ControlAdvanced

class TargetSequenceAction(IntEnum):
    """Action options for Target Sequence profiles."""

    RUN_NOMINAL_SEQUENCE = 0
    """Run Nominal Sequence - runs the mission control sequence."""
    RUN_ACTIVE_PROFILES = 1
    """Run Active Profiles - runs the mission control sequence allowing the active profiles to operate."""
    RUN_ACTIVE_PROFILES_ONCE = 2
    """Run Active Profiles Once - runs the profile with corrections applied to control parameters."""

TargetSequenceAction.RUN_NOMINAL_SEQUENCE.__doc__ = "Run Nominal Sequence - runs the mission control sequence."
TargetSequenceAction.RUN_ACTIVE_PROFILES.__doc__ = "Run Active Profiles - runs the mission control sequence allowing the active profiles to operate."
TargetSequenceAction.RUN_ACTIVE_PROFILES_ONCE.__doc__ = "Run Active Profiles Once - runs the profile with corrections applied to control parameters."

agcls.AgTypeNameMap["TargetSequenceAction"] = TargetSequenceAction

class ProfilesFinish(IntEnum):
    """Action options for Target Sequence profiles convergence."""

    RUN_TO_RETURN_AND_CONTINUE = 0
    """Run To Return And Continue - run to the first Return segment in the sequence, then pass control to the next segment after this target sequence. Often, the only Return is at the end of the target sequence."""
    RUN_TO_RETURN_AND_STOP = 1
    """Run To Return And Stop - run the target sequence to the first Return segment, and then stop running the MCS altogether."""
    STOP = 2
    """Stop - stop the MCS as soon as the target sequence has converged."""

ProfilesFinish.RUN_TO_RETURN_AND_CONTINUE.__doc__ = "Run To Return And Continue - run to the first Return segment in the sequence, then pass control to the next segment after this target sequence. Often, the only Return is at the end of the target sequence."
ProfilesFinish.RUN_TO_RETURN_AND_STOP.__doc__ = "Run To Return And Stop - run the target sequence to the first Return segment, and then stop running the MCS altogether."
ProfilesFinish.STOP.__doc__ = "Stop - stop the MCS as soon as the target sequence has converged."

agcls.AgTypeNameMap["ProfilesFinish"] = ProfilesFinish

class UpdateParam(IntEnum):
    """Spacecraft parameters that can be modified by an Update segment."""

    DRAG_AREA = 0
    """Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations."""
    SRP_AREA = 1
    """Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."""
    DRY_MASS = 2
    """Dry Mass - the mass of the spacecraft exclusive of propellant."""
    FUEL_MASS = 3
    """The mass of the spacecraft propellant."""
    FUEL_DENSITY = 4
    """The density of the fuel tank."""
    TANK_PRESSURE = 5
    """The fuel tank pressure."""
    TANK_TEMPERATURE = 6
    """The temperature of the fuel tank."""
    CR = 7
    """Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    CD = 8
    """Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."""
    CK = 9
    """Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."""
    RADIATION_PRESSURE_AREA = 10
    """Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."""

UpdateParam.DRAG_AREA.__doc__ = "Drag Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations."
UpdateParam.SRP_AREA.__doc__ = "Solar Radiation Pressure (Spherical) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations."
UpdateParam.DRY_MASS.__doc__ = "Dry Mass - the mass of the spacecraft exclusive of propellant."
UpdateParam.FUEL_MASS.__doc__ = "The mass of the spacecraft propellant."
UpdateParam.FUEL_DENSITY.__doc__ = "The density of the fuel tank."
UpdateParam.TANK_PRESSURE.__doc__ = "The fuel tank pressure."
UpdateParam.TANK_TEMPERATURE.__doc__ = "The temperature of the fuel tank."
UpdateParam.CR.__doc__ = "Solar Radiation Pressure (Spherical) Coefficient (Cr) - the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
UpdateParam.CD.__doc__ = "Drag Coefficient (Cd) - the dimensionless drag coefficient associated with the drag area."
UpdateParam.CK.__doc__ = "Radiation Pressure (Albedo/Thermal) Coefficient (Ck) - the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all."
UpdateParam.RADIATION_PRESSURE_AREA.__doc__ = "Radiation Pressure (Albedo/Thermal) Area - the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations."

agcls.AgTypeNameMap["UpdateParam"] = UpdateParam

class UpdateAction(IntEnum):
    """Actions for the Update segment."""

    NO_CHANGE = 0
    """No change in value - leave the current value for this parameter unchanged (ignoring any quantity that may appear in the Value column)."""
    ADD_VALUE = 1
    """Add value - add the quantity entered in the Value column to the current value for this parameter."""
    SUBTRACT_VALUE = 2
    """Subtract value - subtract the quantity entered in the Value column from the current value for this parameter."""
    SET_TO_NEW_VALUE = 3
    """Set to new value - replace the current value for this parameter with the quantity entered in the Value column."""

UpdateAction.NO_CHANGE.__doc__ = "No change in value - leave the current value for this parameter unchanged (ignoring any quantity that may appear in the Value column)."
UpdateAction.ADD_VALUE.__doc__ = "Add value - add the quantity entered in the Value column to the current value for this parameter."
UpdateAction.SUBTRACT_VALUE.__doc__ = "Subtract value - subtract the quantity entered in the Value column from the current value for this parameter."
UpdateAction.SET_TO_NEW_VALUE.__doc__ = "Set to new value - replace the current value for this parameter with the quantity entered in the Value column."

agcls.AgTypeNameMap["UpdateAction"] = UpdateAction

class PressureMode(IntEnum):
    """Pressure Mode options."""

    BLOW_DOWN = 0
    """Blow-Down - pressure decreases as propellant is consumed and the volume occupied by the pressurant gas consequently increases. This is based on the ideal gas law."""
    PRESSURE_REGULATED = 1
    """Pressure-Regulated - constant pressure is maintained in the fuel tank through some pressurization mechanism as the propellant mass decreases."""

PressureMode.BLOW_DOWN.__doc__ = "Blow-Down - pressure decreases as propellant is consumed and the volume occupied by the pressurant gas consequently increases. This is based on the ideal gas law."
PressureMode.PRESSURE_REGULATED.__doc__ = "Pressure-Regulated - constant pressure is maintained in the fuel tank through some pressurization mechanism as the propellant mass decreases."

agcls.AgTypeNameMap["PressureMode"] = PressureMode

class ThrustType(IntEnum):
    """Thrust options."""

    AFFECTS_ACCELERATION_AND_MASS_FLOW = 0
    """Affects Acceleration and Mass Flow Calculations - may represent an inefficiency in the propulsion tanks and feed lines."""
    AFFECTS_ACCELERATION_ONLY = 1
    """Affects Acceleration Only - an efficiency of 0.98 means that only 98% of the fuel will be spent to get 98% thrust. For example, a thrust efficiency affecting acceleration only may represent some problem in the combustion chamber."""

ThrustType.AFFECTS_ACCELERATION_AND_MASS_FLOW.__doc__ = "Affects Acceleration and Mass Flow Calculations - may represent an inefficiency in the propulsion tanks and feed lines."
ThrustType.AFFECTS_ACCELERATION_ONLY.__doc__ = "Affects Acceleration Only - an efficiency of 0.98 means that only 98% of the fuel will be spent to get 98% thrust. For example, a thrust efficiency affecting acceleration only may represent some problem in the combustion chamber."

agcls.AgTypeNameMap["ThrustType"] = ThrustType

class AttitudeUpdate(IntEnum):
    """Attitude Update."""

    DURING_BURN = 0
    """Update during burn - updated throughout the maneuver so as to maintain the required thrust direction. This forces the thrust vector to the specified direction at every instant throughout the burn."""
    INERTIAL_AT_IGNITION = 1
    """Inertial at ignition - specified by Attitude Control at ignition and remains the same throughout the maneuver. This fixes the thrust direction in the inertial direction calculated at the beginning of the burn and is used for inertially fixed spacecraft."""
    INERTIAL_AT_START = 2
    """Inertial at start - specified by Attitude Control at the beginning of the maneuver segment and remains the same throughout the maneuver."""

AttitudeUpdate.DURING_BURN.__doc__ = "Update during burn - updated throughout the maneuver so as to maintain the required thrust direction. This forces the thrust vector to the specified direction at every instant throughout the burn."
AttitudeUpdate.INERTIAL_AT_IGNITION.__doc__ = "Inertial at ignition - specified by Attitude Control at ignition and remains the same throughout the maneuver. This fixes the thrust direction in the inertial direction calculated at the beginning of the burn and is used for inertially fixed spacecraft."
AttitudeUpdate.INERTIAL_AT_START.__doc__ = "Inertial at start - specified by Attitude Control at the beginning of the maneuver segment and remains the same throughout the maneuver."

agcls.AgTypeNameMap["AttitudeUpdate"] = AttitudeUpdate

class PropulsionMethod(IntEnum):
    """Propulsion method options."""

    ENGINE_MODEL = 0
    """Engine Model - can be used to quickly model the firing of a single engine."""
    THRUSTER_SET = 1
    """Thruster Set - used to fire multiple engines simultaneously, and to simulate off-pulsing or complex engine pointing."""

PropulsionMethod.ENGINE_MODEL.__doc__ = "Engine Model - can be used to quickly model the firing of a single engine."
PropulsionMethod.THRUSTER_SET.__doc__ = "Thruster Set - used to fire multiple engines simultaneously, and to simulate off-pulsing or complex engine pointing."

agcls.AgTypeNameMap["PropulsionMethod"] = PropulsionMethod

class CustomFunction(IntEnum):
    """Attitude definition options for other STK functions."""

    ENABLE_PAGE_DEFINITION = 0
    """Use attitude page definition for other STK functions - the actual attitude during the maneuver is ignored and the satellite is considered to always be in the attitude specified by the page for all other calculations in STK."""
    ENABLE_MANEUVER_ATTITUDE = 1
    """Maneuver attitude will be used for the satellite during the time at the maneuver, and during the specified lead and trail times. This attitude will also show up in reports, graphs, sensor access calculations, and in the 3D Graphics window."""

CustomFunction.ENABLE_PAGE_DEFINITION.__doc__ = "Use attitude page definition for other STK functions - the actual attitude during the maneuver is ignored and the satellite is considered to always be in the attitude specified by the page for all other calculations in STK."
CustomFunction.ENABLE_MANEUVER_ATTITUDE.__doc__ = "Maneuver attitude will be used for the satellite during the time at the maneuver, and during the specified lead and trail times. This attitude will also show up in reports, graphs, sensor access calculations, and in the 3D Graphics window."

agcls.AgTypeNameMap["CustomFunction"] = CustomFunction

class BodyAxis(IntEnum):
    """Attitude body axis options."""

    PLUS_X = 0
    """The positive X engine acceleration direction."""
    PLUS_Y = 1
    """The positive Y engine acceleration direction."""
    PLUS_Z = 2
    """The positive Z engine acceleration direction."""
    MINUS_X = 3
    """The negative X engine acceleration direction."""
    MINUS_Y = 4
    """The negative Y engine acceleration direction."""
    MINUS_Z = 5
    """The negative Z engine acceleration direction."""

BodyAxis.PLUS_X.__doc__ = "The positive X engine acceleration direction."
BodyAxis.PLUS_Y.__doc__ = "The positive Y engine acceleration direction."
BodyAxis.PLUS_Z.__doc__ = "The positive Z engine acceleration direction."
BodyAxis.MINUS_X.__doc__ = "The negative X engine acceleration direction."
BodyAxis.MINUS_Y.__doc__ = "The negative Y engine acceleration direction."
BodyAxis.MINUS_Z.__doc__ = "The negative Z engine acceleration direction."

agcls.AgTypeNameMap["BodyAxis"] = BodyAxis

class ConstraintSign(IntEnum):
    """Constraint vector sign options."""

    PLUS = 0
    """Positive."""
    MINUS = 1
    """Negative."""

ConstraintSign.PLUS.__doc__ = "Positive."
ConstraintSign.MINUS.__doc__ = "Negative."

agcls.AgTypeNameMap["ConstraintSign"] = ConstraintSign

class AttitudeControl(IntEnum):
    """Attitude Control options."""

    VELOCITY_VECTOR = 0
    """Along Velocity Vector (impulsive/finite) - the total thrust vector is aligned with the spacecraft's velocity vector."""
    ANTI_VELOCITY_VECTOR = 1
    """Anti-Velocity Vector (impulsive/finite) - the total thrust vector is opposite to the spacecraft's velocity vector."""
    ATTITUDE = 2
    """Attitude (impulsive/finite) - the thrust vector direction is defined using Euler Angles or a Quaternion."""
    FILE = 3
    """File (impulsive/finite) - uses an attitude file to set the thrust vector direction."""
    THRUST_VECTOR = 4
    """Thrust Vector (impulsive/finite) - the total thrust vector is explicitly specified in Cartesian or spherical form with respect to the thrust axes."""
    PLUGIN = 5
    """Plugin (finite) - thrust vector direction is defined using a COM plugin."""
    TIME_VARYING = 6
    """Time Varying (finite) - polynomial and sine representations for the azimuth and elevation of the thrust vector."""
    LAGRANGE_INTERPOLATION = 7
    """Lagrange Interpolation (optimal finite, always set) - supports the 'Run current nodes' execution mode of the Optimal Finite Maneuver."""

AttitudeControl.VELOCITY_VECTOR.__doc__ = "Along Velocity Vector (impulsive/finite) - the total thrust vector is aligned with the spacecraft's velocity vector."
AttitudeControl.ANTI_VELOCITY_VECTOR.__doc__ = "Anti-Velocity Vector (impulsive/finite) - the total thrust vector is opposite to the spacecraft's velocity vector."
AttitudeControl.ATTITUDE.__doc__ = "Attitude (impulsive/finite) - the thrust vector direction is defined using Euler Angles or a Quaternion."
AttitudeControl.FILE.__doc__ = "File (impulsive/finite) - uses an attitude file to set the thrust vector direction."
AttitudeControl.THRUST_VECTOR.__doc__ = "Thrust Vector (impulsive/finite) - the total thrust vector is explicitly specified in Cartesian or spherical form with respect to the thrust axes."
AttitudeControl.PLUGIN.__doc__ = "Plugin (finite) - thrust vector direction is defined using a COM plugin."
AttitudeControl.TIME_VARYING.__doc__ = "Time Varying (finite) - polynomial and sine representations for the azimuth and elevation of the thrust vector."
AttitudeControl.LAGRANGE_INTERPOLATION.__doc__ = "Lagrange Interpolation (optimal finite, always set) - supports the 'Run current nodes' execution mode of the Optimal Finite Maneuver."

agcls.AgTypeNameMap["AttitudeControl"] = AttitudeControl

class FollowJoin(IntEnum):
    """Joining options for the Follow segment."""

    SPECIFY = 0
    """Specify Joining Conditions - specify joining conditions to define when the spacecraft will begin to follow the leader. Joining conditions will become apparent on a new tab - Joining."""
    AT_BEGINNING = 1
    """Join at Beginning of Leader's Ephemeris - the spacecraft will follow the leader from the beginning of the leader's ephemeris."""
    AT_END = 2
    """Join at End of Leader's Ephemeris - the spacecraft will use the leader's final ephemeris point as the initial and final state of the Follow segment; the separation parameter will automatically be set to 'Separate at End of Leader's Ephemeris'."""
    AT_FINAL_EPOCH_OF_PREVIOUS_SEG = 3
    """Join at Final Epoch of Previous Segment - the spacecraft will follow the leader from the final epoch at the end of the previous segment."""

FollowJoin.SPECIFY.__doc__ = "Specify Joining Conditions - specify joining conditions to define when the spacecraft will begin to follow the leader. Joining conditions will become apparent on a new tab - Joining."
FollowJoin.AT_BEGINNING.__doc__ = "Join at Beginning of Leader's Ephemeris - the spacecraft will follow the leader from the beginning of the leader's ephemeris."
FollowJoin.AT_END.__doc__ = "Join at End of Leader's Ephemeris - the spacecraft will use the leader's final ephemeris point as the initial and final state of the Follow segment; the separation parameter will automatically be set to 'Separate at End of Leader's Ephemeris'."
FollowJoin.AT_FINAL_EPOCH_OF_PREVIOUS_SEG.__doc__ = "Join at Final Epoch of Previous Segment - the spacecraft will follow the leader from the final epoch at the end of the previous segment."

agcls.AgTypeNameMap["FollowJoin"] = FollowJoin

class FollowSeparation(IntEnum):
    """Separation options for the Follow segment."""

    SPECIFY = 0
    """Specify Separation Conditions - specify separation conditions to define when the spacecraft will stop following the leader. Separation conditions will become apparent on a new tab - Separation."""
    AT_END_OF_LEADERS_EPHEMERIS = 1
    """Separate at End of Leader's Ephemeris - the spacecraft will follow the leader to the end of the leader's ephemeris and then separate."""

FollowSeparation.SPECIFY.__doc__ = "Specify Separation Conditions - specify separation conditions to define when the spacecraft will stop following the leader. Separation conditions will become apparent on a new tab - Separation."
FollowSeparation.AT_END_OF_LEADERS_EPHEMERIS.__doc__ = "Separate at End of Leader's Ephemeris - the spacecraft will follow the leader to the end of the leader's ephemeris and then separate."

agcls.AgTypeNameMap["FollowSeparation"] = FollowSeparation

class FollowSpacecraftAndFuelTank(IntEnum):
    """Spacecraft parameter options for the Follow segment."""

    SPECIFY = 0
    """Specify Spacecraft Configuration - manually define the spacecraft for this segment. Spacecraft physical parameters will become apparent on new tabs - Spacecraft Parameters and Fuel Tank."""
    INHERIT = 1
    """Inherit Spacecraft Configuration From Previous Segment - the spacecraft will be defined by its configuration at the end of the previous segment. The configuration will not be changed by the Follow segment."""
    LEADER = 2
    """Inherit Spacecraft Configuration From Leader - if leader is an Astrogator satellite, configuration will be defined by the leader's configuration."""

FollowSpacecraftAndFuelTank.SPECIFY.__doc__ = "Specify Spacecraft Configuration - manually define the spacecraft for this segment. Spacecraft physical parameters will become apparent on new tabs - Spacecraft Parameters and Fuel Tank."
FollowSpacecraftAndFuelTank.INHERIT.__doc__ = "Inherit Spacecraft Configuration From Previous Segment - the spacecraft will be defined by its configuration at the end of the previous segment. The configuration will not be changed by the Follow segment."
FollowSpacecraftAndFuelTank.LEADER.__doc__ = "Inherit Spacecraft Configuration From Leader - if leader is an Astrogator satellite, configuration will be defined by the leader's configuration."

agcls.AgTypeNameMap["FollowSpacecraftAndFuelTank"] = FollowSpacecraftAndFuelTank

class BurnoutOptions(IntEnum):
    """Burnout options for the Launch segment."""

    FIXED_VELOCITY = 0
    """Use Fixed Velocity - the inclination of the final state of the launch segment is determined by the arc between the launch and insertion positions, and the horizontal flight path angle is set to zero."""
    INERTIAL_VELOCITY = 1
    """Use Inertial Velocity - the final state of the launch segment is solely and completely determined by the burnout position and velocity."""

BurnoutOptions.FIXED_VELOCITY.__doc__ = "Use Fixed Velocity - the inclination of the final state of the launch segment is determined by the arc between the launch and insertion positions, and the horizontal flight path angle is set to zero."
BurnoutOptions.INERTIAL_VELOCITY.__doc__ = "Use Inertial Velocity - the final state of the launch segment is solely and completely determined by the burnout position and velocity."

agcls.AgTypeNameMap["BurnoutOptions"] = BurnoutOptions

class BurnoutType(IntEnum):
    """Burnout point definition types for the Launch segment."""

    GEOCENTRIC = 0
    """Geocentric (Planetocentric) - the burnout point is measured from the center of mass of the Earth or other central body."""
    GEODETIC = 1
    """Geodetic (Planetodetic) - the burnout point is measured along the normal to the surface of an ellipsoid defined with reference to the Earth (or other central body)."""
    LAUNCH_AZIMUTH_RADIUS = 2
    """Launch Az / Radius - the burnout point is defined in reference to distance downrange along an azimuth, measured from the center of mass of the Earth or other central body."""
    LAUNCH_AZIMUTH_ALTITUDE = 3
    """Launch Az / Alt - the burnout point is defined in reference to distance downrange along an azimuth, measured from the surface of the Earth or other central body."""
    CBF_CARTESIAN = 4
    """Central Body Fixed Cartesian - the burnout state is specified in the central-body-fixed Cartesian coordinate system."""

BurnoutType.GEOCENTRIC.__doc__ = "Geocentric (Planetocentric) - the burnout point is measured from the center of mass of the Earth or other central body."
BurnoutType.GEODETIC.__doc__ = "Geodetic (Planetodetic) - the burnout point is measured along the normal to the surface of an ellipsoid defined with reference to the Earth (or other central body)."
BurnoutType.LAUNCH_AZIMUTH_RADIUS.__doc__ = "Launch Az / Radius - the burnout point is defined in reference to distance downrange along an azimuth, measured from the center of mass of the Earth or other central body."
BurnoutType.LAUNCH_AZIMUTH_ALTITUDE.__doc__ = "Launch Az / Alt - the burnout point is defined in reference to distance downrange along an azimuth, measured from the surface of the Earth or other central body."
BurnoutType.CBF_CARTESIAN.__doc__ = "Central Body Fixed Cartesian - the burnout state is specified in the central-body-fixed Cartesian coordinate system."

agcls.AgTypeNameMap["BurnoutType"] = BurnoutType

class AscentType(IntEnum):
    """Ascent types for the Launch segment."""

    ELLIPSE_CUBIC_MOTION = 0
    """Ellipse: Cubic Motion - the motion is computed on the basis of given positions and velocities."""
    ELLIPSE_QUARTIC_MOTION = 1
    """Ellipse: Quartic Motion - the motion is computed on the basis of given positions and velocities and initial acceleration."""

AscentType.ELLIPSE_CUBIC_MOTION.__doc__ = "Ellipse: Cubic Motion - the motion is computed on the basis of given positions and velocities."
AscentType.ELLIPSE_QUARTIC_MOTION.__doc__ = "Ellipse: Quartic Motion - the motion is computed on the basis of given positions and velocities and initial acceleration."

agcls.AgTypeNameMap["AscentType"] = AscentType

class LaunchDisplaySystem(IntEnum):
    """Launch location coordinate types for the Launch segment."""

    DISPLAY_SYSTEM_GEODETIC = 0
    """Geocentric (Planetocentric) - the location is measured from the center of mass of the Earth or other central body."""
    DISPLAY_SYSTEM_GEOCENTRIC = 1
    """Geodetic (Planetodetic) - the location is measured along the normal to the surface of an ellipsoid defined with reference to the Earth (or other central body)."""

LaunchDisplaySystem.DISPLAY_SYSTEM_GEODETIC.__doc__ = "Geocentric (Planetocentric) - the location is measured from the center of mass of the Earth or other central body."
LaunchDisplaySystem.DISPLAY_SYSTEM_GEOCENTRIC.__doc__ = "Geodetic (Planetodetic) - the location is measured along the normal to the surface of an ellipsoid defined with reference to the Earth (or other central body)."

agcls.AgTypeNameMap["LaunchDisplaySystem"] = LaunchDisplaySystem

class RunCode(IntEnum):
    """The run code returned after the MCS is run."""

    MARCHING = 0
    """Marching - a segment has run successfully."""
    PROFILE_FAILURE = 1
    """Profile Failure - a search profile failed to converge."""
    ERROR = 2
    """Error - encountered an error."""
    STOPPED = 3
    """Stopped - encountered a stop segment."""
    RETURNED = 4
    """Returned - encountered a return segment."""
    CANCELLED = 5
    """Cancelled - cancelled by user."""
    HIT_GLOBAL_STOP = 6
    """Global Stop - hit a global stopping condition."""

RunCode.MARCHING.__doc__ = "Marching - a segment has run successfully."
RunCode.PROFILE_FAILURE.__doc__ = "Profile Failure - a search profile failed to converge."
RunCode.ERROR.__doc__ = "Error - encountered an error."
RunCode.STOPPED.__doc__ = "Stopped - encountered a stop segment."
RunCode.RETURNED.__doc__ = "Returned - encountered a return segment."
RunCode.CANCELLED.__doc__ = "Cancelled - cancelled by user."
RunCode.HIT_GLOBAL_STOP.__doc__ = "Global Stop - hit a global stopping condition."

agcls.AgTypeNameMap["RunCode"] = RunCode

class SequenceStateToPass(IntEnum):
    """State To Pass options for the Sequence segment."""

    INITIAL = 0
    """The initial state of the sequence."""
    FINAL = 1
    """The final state of the sequence."""

SequenceStateToPass.INITIAL.__doc__ = "The initial state of the sequence."
SequenceStateToPass.FINAL.__doc__ = "The final state of the sequence."

agcls.AgTypeNameMap["SequenceStateToPass"] = SequenceStateToPass

class ManeuverType(IntEnum):
    """Maneuver types for the maneuver segment."""

    IMPULSIVE = 0
    """Impulsive - calculates a state by adding the defined delta-V vector to the velocity of the final state of the previous segment, adds this new state to the ephemeris, and passes it to the next segment."""
    FINITE = 1
    """Finite - effectively a Propagate segment with thrust. Like Propagate segments, each point calculated by the propagator is added to the ephemeris, and propagation continues until a stopping condition is met."""
    OPTIMAL_FINITE = 2
    """Optimal Finite."""

ManeuverType.IMPULSIVE.__doc__ = "Impulsive - calculates a state by adding the defined delta-V vector to the velocity of the final state of the previous segment, adds this new state to the ephemeris, and passes it to the next segment."
ManeuverType.FINITE.__doc__ = "Finite - effectively a Propagate segment with thrust. Like Propagate segments, each point calculated by the propagator is added to the ephemeris, and propagation continues until a stopping condition is met."
ManeuverType.OPTIMAL_FINITE.__doc__ = "Optimal Finite."

agcls.AgTypeNameMap["ManeuverType"] = ManeuverType

class SegmentType(IntEnum):
    """Segment types."""

    INITIAL_STATE = 0
    """Initial State - can be used to define the initial conditions of your MCS, or of a subsequence within the MCS."""
    LAUNCH = 1
    """Launch - can be used to model a simple spacecraft launch from Earth or another central body."""
    MANEUVER = 2
    """Maneuver - can be used to model a spacecraft maneuver."""
    FOLLOW = 3
    """Follow - can be used to set the spacecraft to follow another vehicle (Satellite, Launch Vehicle, Missile, Aircraft, Ship, or Ground Vehicle) at a specified offset, and to separate from that vehicle upon meeting specified conditions."""
    HOLD = 4
    """Hold - can be used to model landing or rendezvous operations by setting the spacecraft to maintain a fixed position in reference to another object or body, until meeting specified conditions."""
    PROPAGATE = 5
    """Propagate - can be used to model the movement of the spacecraft along its current trajectory until meeting specified stopping conditions."""
    SEQUENCE = 6
    """Sequence - can be used to organize segments and define the nature of the results that are passed on to the next segment or sequence in the MCS."""
    RETURN = 7
    """Return - can be used to control the execution of the Mission Control Sequence by returning control to its parent segment."""
    TARGET_SEQUENCE = 8
    """Target Sequence - can be used to define maneuvers and propagations in terms of the goals they are intended to achieve."""
    STOP = 9
    """Stop - can be used to control the execution of the Mission Control Sequence by halting execution of the MCS."""
    UPDATE = 10
    """Update - can be used to modify some of the satellite properties to reflect changes that occur during the mission."""
    BACKWARD_SEQUENCE = 11
    """Backward Sequence - can be used to organize segments and define the nature of the results that are passed on to the next segment or sequence in the MCS."""
    END = 12
    """End - The End segment is a default segment of the MCS that functions similarly to a Return segment; it returns control to the beginning of the MCS. The End segment cannot be disabled or deleted, nor can any segments be inserted into the MCS after it."""

SegmentType.INITIAL_STATE.__doc__ = "Initial State - can be used to define the initial conditions of your MCS, or of a subsequence within the MCS."
SegmentType.LAUNCH.__doc__ = "Launch - can be used to model a simple spacecraft launch from Earth or another central body."
SegmentType.MANEUVER.__doc__ = "Maneuver - can be used to model a spacecraft maneuver."
SegmentType.FOLLOW.__doc__ = "Follow - can be used to set the spacecraft to follow another vehicle (Satellite, Launch Vehicle, Missile, Aircraft, Ship, or Ground Vehicle) at a specified offset, and to separate from that vehicle upon meeting specified conditions."
SegmentType.HOLD.__doc__ = "Hold - can be used to model landing or rendezvous operations by setting the spacecraft to maintain a fixed position in reference to another object or body, until meeting specified conditions."
SegmentType.PROPAGATE.__doc__ = "Propagate - can be used to model the movement of the spacecraft along its current trajectory until meeting specified stopping conditions."
SegmentType.SEQUENCE.__doc__ = "Sequence - can be used to organize segments and define the nature of the results that are passed on to the next segment or sequence in the MCS."
SegmentType.RETURN.__doc__ = "Return - can be used to control the execution of the Mission Control Sequence by returning control to its parent segment."
SegmentType.TARGET_SEQUENCE.__doc__ = "Target Sequence - can be used to define maneuvers and propagations in terms of the goals they are intended to achieve."
SegmentType.STOP.__doc__ = "Stop - can be used to control the execution of the Mission Control Sequence by halting execution of the MCS."
SegmentType.UPDATE.__doc__ = "Update - can be used to modify some of the satellite properties to reflect changes that occur during the mission."
SegmentType.BACKWARD_SEQUENCE.__doc__ = "Backward Sequence - can be used to organize segments and define the nature of the results that are passed on to the next segment or sequence in the MCS."
SegmentType.END.__doc__ = "End - The End segment is a default segment of the MCS that functions similarly to a Return segment; it returns control to the beginning of the MCS. The End segment cannot be disabled or deleted, nor can any segments be inserted into the MCS after it."

agcls.AgTypeNameMap["SegmentType"] = SegmentType

class ElementSetType(IntEnum):
    """Types of orbit element sets."""

    CARTESIAN = 0
    """Cartesian - specifying an orbit by three position elements and three velocity elements in a rectangular coordinate system."""
    KEPLERIAN = 1
    """Keplerian - the classical system, specifying an orbit by six elements describing its size, shape and three-dimensional orientation in space."""
    SPHERICAL = 3
    """Spherical - a system in which positions are specified as a radial distance from the origin and two angles relative to a fundamental plane."""
    TARGET_VECTOR_INCOMING_ASYMPTOTE = 4
    """Target Vector Incoming Asymptote - used for hyperbolic arrival trajectories."""
    TARGET_VECTOR_OUTGOING_ASYMPTOTE = 5
    """Target Vector Outgoing Asymptote - used for hyperbolic departure trajectories."""
    MIXED_SPHERICAL = 6
    """Mixed Spherical."""
    DELAUNAY = 7
    """Delaunay."""
    EQUINOCTIAL = 8
    """Equinoctial."""
    GEODETIC = 9
    """Geodetic."""
    B_PLANE = 10
    """BPlane."""
    SPHERICAL_RANGE_RATE = 11
    """Spherical Range Rate."""

ElementSetType.CARTESIAN.__doc__ = "Cartesian - specifying an orbit by three position elements and three velocity elements in a rectangular coordinate system."
ElementSetType.KEPLERIAN.__doc__ = "Keplerian - the classical system, specifying an orbit by six elements describing its size, shape and three-dimensional orientation in space."
ElementSetType.SPHERICAL.__doc__ = "Spherical - a system in which positions are specified as a radial distance from the origin and two angles relative to a fundamental plane."
ElementSetType.TARGET_VECTOR_INCOMING_ASYMPTOTE.__doc__ = "Target Vector Incoming Asymptote - used for hyperbolic arrival trajectories."
ElementSetType.TARGET_VECTOR_OUTGOING_ASYMPTOTE.__doc__ = "Target Vector Outgoing Asymptote - used for hyperbolic departure trajectories."
ElementSetType.MIXED_SPHERICAL.__doc__ = "Mixed Spherical."
ElementSetType.DELAUNAY.__doc__ = "Delaunay."
ElementSetType.EQUINOCTIAL.__doc__ = "Equinoctial."
ElementSetType.GEODETIC.__doc__ = "Geodetic."
ElementSetType.B_PLANE.__doc__ = "BPlane."
ElementSetType.SPHERICAL_RANGE_RATE.__doc__ = "Spherical Range Rate."

agcls.AgTypeNameMap["ElementSetType"] = ElementSetType

class Language(IntEnum):
    """Scripting language types for the Scripting Tool."""

    VB_SCRIPT = 0
    """VBScript."""
    J_SCRIPT = 1
    """JScript."""
    MATLAB = 2
    """MATLAB."""

Language.VB_SCRIPT.__doc__ = "VBScript."
Language.J_SCRIPT.__doc__ = "JScript."
Language.MATLAB.__doc__ = "MATLAB."

agcls.AgTypeNameMap["Language"] = Language

class StoppingConditionType(IntEnum):
    """Type of stopping condition."""

    BASIC = 0
    """Basic Stopping Condition."""
    BEFORE = 1
    """Before Stopping Condition."""
    ONE_POINT_ACCESS = 2
    """One Point Access Stopping Condition."""
    LIGHTING = 3
    """Lighting Stopping Condition."""

StoppingConditionType.BASIC.__doc__ = "Basic Stopping Condition."
StoppingConditionType.BEFORE.__doc__ = "Before Stopping Condition."
StoppingConditionType.ONE_POINT_ACCESS.__doc__ = "One Point Access Stopping Condition."
StoppingConditionType.LIGHTING.__doc__ = "Lighting Stopping Condition."

agcls.AgTypeNameMap["StoppingConditionType"] = StoppingConditionType

class ClearEphemerisDirection(IntEnum):
    """Direction in which to clear ephemeris."""

    BEFORE = -1
    """Clear ephemeris before selected time or state."""
    NO_CLEAR = 0
    """Do not clear ephemeris."""
    AFTER = 1
    """Clear ephemeris after selected time or state."""

ClearEphemerisDirection.BEFORE.__doc__ = "Clear ephemeris before selected time or state."
ClearEphemerisDirection.NO_CLEAR.__doc__ = "Do not clear ephemeris."
ClearEphemerisDirection.AFTER.__doc__ = "Clear ephemeris after selected time or state."

agcls.AgTypeNameMap["ClearEphemerisDirection"] = ClearEphemerisDirection

class ProfileInsertDirection(IntEnum):
    """Direction to insert profile."""

    BEFORE = 0
    """Insert profile before reference profile."""
    AFTER = 1
    """Insert profile after reference profile."""

ProfileInsertDirection.BEFORE.__doc__ = "Insert profile before reference profile."
ProfileInsertDirection.AFTER.__doc__ = "Insert profile after reference profile."

agcls.AgTypeNameMap["ProfileInsertDirection"] = ProfileInsertDirection

class RootFindingAlgorithm(IntEnum):
    """Root-finding algorithms."""

    SECANT_METHOD = 0
    """Secant method."""
    NEWTON_RAPHSON_METHOD = 1
    """Newton-Raphson method."""

RootFindingAlgorithm.SECANT_METHOD.__doc__ = "Secant method."
RootFindingAlgorithm.NEWTON_RAPHSON_METHOD.__doc__ = "Newton-Raphson method."

agcls.AgTypeNameMap["RootFindingAlgorithm"] = RootFindingAlgorithm

class ScriptingParameterType(IntEnum):
    """Scripting Tool parameter type."""

    DOUBLE = 0
    """Double."""
    QUANTITY = 1
    """Quantity."""
    DATE = 2
    """Date."""
    STRING = 3
    """String."""
    BOOLEAN = 4
    """Boolean."""
    INTEGER = 5
    """Integer."""
    ENUMERATION = 6
    """Enumeration."""

ScriptingParameterType.DOUBLE.__doc__ = "Double."
ScriptingParameterType.QUANTITY.__doc__ = "Quantity."
ScriptingParameterType.DATE.__doc__ = "Date."
ScriptingParameterType.STRING.__doc__ = "String."
ScriptingParameterType.BOOLEAN.__doc__ = "Boolean."
ScriptingParameterType.INTEGER.__doc__ = "Integer."
ScriptingParameterType.ENUMERATION.__doc__ = "Enumeration."

agcls.AgTypeNameMap["ScriptingParameterType"] = ScriptingParameterType

class SNOPTGoal(IntEnum):
    """The Goal types for a SNOPT profile."""

    MINIMIZE = 0
    """Minimize."""
    BOUND = 1
    """Bound."""

SNOPTGoal.MINIMIZE.__doc__ = "Minimize."
SNOPTGoal.BOUND.__doc__ = "Bound."

agcls.AgTypeNameMap["SNOPTGoal"] = SNOPTGoal

class IPOPTGoal(IntEnum):
    """The Goal types for a IPOPT profile."""

    MINIMIZE = 0
    """Minimize."""
    BOUND = 1
    """Bound."""

IPOPTGoal.MINIMIZE.__doc__ = "Minimize."
IPOPTGoal.BOUND.__doc__ = "Bound."

agcls.AgTypeNameMap["IPOPTGoal"] = IPOPTGoal

class OptimalFiniteSeedMethod(IntEnum):
    """Seed methods."""

    INITIAL_GUESS_FILE = 0
    """Seed initial guess from file."""
    FINITE_MANEUVER = 1
    """Seed initial guess from finite maneuver."""

OptimalFiniteSeedMethod.INITIAL_GUESS_FILE.__doc__ = "Seed initial guess from file."
OptimalFiniteSeedMethod.FINITE_MANEUVER.__doc__ = "Seed initial guess from finite maneuver."

agcls.AgTypeNameMap["OptimalFiniteSeedMethod"] = OptimalFiniteSeedMethod

class OptimalFiniteRunMode(IntEnum):
    """Run modes."""

    RUN_CURRENT_NODES = 0
    """Run current nodes."""
    OPTIMIZE_VIA_DIRECT_TRANSCRIPTION = 1
    """Optimize via direct transcription."""

OptimalFiniteRunMode.RUN_CURRENT_NODES.__doc__ = "Run current nodes."
OptimalFiniteRunMode.OPTIMIZE_VIA_DIRECT_TRANSCRIPTION.__doc__ = "Optimize via direct transcription."

agcls.AgTypeNameMap["OptimalFiniteRunMode"] = OptimalFiniteRunMode

class OptimalFiniteDiscretizationStrategy(IntEnum):
    """Discretization Strategy."""

    LEGENDRE_GAUSS_LOBATTO = 0
    """The Legendre-Gauss-Lobatto quadrature rule."""
    LEGENDRE_GAUSS_RADAU = 1
    """The Legendre-Gauss-Radau quadrature rule."""

OptimalFiniteDiscretizationStrategy.LEGENDRE_GAUSS_LOBATTO.__doc__ = "The Legendre-Gauss-Lobatto quadrature rule."
OptimalFiniteDiscretizationStrategy.LEGENDRE_GAUSS_RADAU.__doc__ = "The Legendre-Gauss-Radau quadrature rule."

agcls.AgTypeNameMap["OptimalFiniteDiscretizationStrategy"] = OptimalFiniteDiscretizationStrategy

class OptimalFiniteWorkingVariables(IntEnum):
    """Working Variables."""

    EQUINOCTIAL = 0
    """Standard inertial equinoctial coordinates."""
    MODIFIED_EQUINOCTIAL = 1
    """Use semi-latus rectum instead of SMA and true longitude instead of mean longitude."""

OptimalFiniteWorkingVariables.EQUINOCTIAL.__doc__ = "Standard inertial equinoctial coordinates."
OptimalFiniteWorkingVariables.MODIFIED_EQUINOCTIAL.__doc__ = "Use semi-latus rectum instead of SMA and true longitude instead of mean longitude."

agcls.AgTypeNameMap["OptimalFiniteWorkingVariables"] = OptimalFiniteWorkingVariables

class OptimalFiniteScalingOptions(IntEnum):
    """Scaling Options."""

    NO_SCALING = 0
    """No scaling."""
    CANONICAL_UNITS = 1
    """Scale based on canonical units."""
    INITIAL_STATE_BASED = 2
    """Scale based on the initial state."""

OptimalFiniteScalingOptions.NO_SCALING.__doc__ = "No scaling."
OptimalFiniteScalingOptions.CANONICAL_UNITS.__doc__ = "Scale based on canonical units."
OptimalFiniteScalingOptions.INITIAL_STATE_BASED.__doc__ = "Scale based on the initial state."

agcls.AgTypeNameMap["OptimalFiniteScalingOptions"] = OptimalFiniteScalingOptions

class OptimalFiniteSNOPTObjective(IntEnum):
    """Optimal Finite SNOPT objective."""

    MINIMIZE_TIME_OF_FLIGHT = 0
    """Minimize the total TOF along the collocation arc."""
    MAXIMIZE_FINAL_RAD = 1
    """Minimize the total calculated DeltaV along the collocation arc."""
    MINIMIZE_PROPELLANT_USE = 2
    """Maximize the final mass after the collocation arc."""

OptimalFiniteSNOPTObjective.MINIMIZE_TIME_OF_FLIGHT.__doc__ = "Minimize the total TOF along the collocation arc."
OptimalFiniteSNOPTObjective.MAXIMIZE_FINAL_RAD.__doc__ = "Minimize the total calculated DeltaV along the collocation arc."
OptimalFiniteSNOPTObjective.MINIMIZE_PROPELLANT_USE.__doc__ = "Maximize the final mass after the collocation arc."

agcls.AgTypeNameMap["OptimalFiniteSNOPTObjective"] = OptimalFiniteSNOPTObjective

class OptimalFiniteSNOPTScaling(IntEnum):
    """Optimal Finite SNOPT scaling option."""

    NONE = 0
    """No scaling."""
    LINEAR = 1
    """Scale linear constraints and variables."""
    ALL = 2
    """Scale all constraints and variables."""

OptimalFiniteSNOPTScaling.NONE.__doc__ = "No scaling."
OptimalFiniteSNOPTScaling.LINEAR.__doc__ = "Scale linear constraints and variables."
OptimalFiniteSNOPTScaling.ALL.__doc__ = "Scale all constraints and variables."

agcls.AgTypeNameMap["OptimalFiniteSNOPTScaling"] = OptimalFiniteSNOPTScaling

class OptimalFiniteExportNodesFormat(IntEnum):
    """Steering nodes export format."""

    AZIMUTH_ELEVATION = 0
    """Export controls as Az/El = Ra/Dec angles."""
    UNIT_VECTOR = 1
    """Export controls as unit vectors = direction cosines."""

OptimalFiniteExportNodesFormat.AZIMUTH_ELEVATION.__doc__ = "Export controls as Az/El = Ra/Dec angles."
OptimalFiniteExportNodesFormat.UNIT_VECTOR.__doc__ = "Export controls as unit vectors = direction cosines."

agcls.AgTypeNameMap["OptimalFiniteExportNodesFormat"] = OptimalFiniteExportNodesFormat

class OptimalFiniteGuessMethod(IntEnum):
    """Guess interpolation method."""

    LAGRANGE_POLYNOMIAL = 0
    """Lagrange polynomial method."""
    PIECEWISE_LINEAR = 1
    """Piecewise linear method."""

OptimalFiniteGuessMethod.LAGRANGE_POLYNOMIAL.__doc__ = "Lagrange polynomial method."
OptimalFiniteGuessMethod.PIECEWISE_LINEAR.__doc__ = "Piecewise linear method."

agcls.AgTypeNameMap["OptimalFiniteGuessMethod"] = OptimalFiniteGuessMethod

class ImpulsiveDeltaVRepresentation(IntEnum):
    """Vector representations for impulsive DeltaV specification."""

    CARTESIAN_IMPULSIVE_DELTA_V = 0
    """Cartesian - specify the impulsive DeltaV in Cartesian coordinates (X, Y, Z)."""
    SPHERICAL_IMPULSIVE_DELTA_V = 1
    """Spherical - specify the impulsive DeltaV in spherical coordinates (Azimuth, Elevation, Magnitude)."""

ImpulsiveDeltaVRepresentation.CARTESIAN_IMPULSIVE_DELTA_V.__doc__ = "Cartesian - specify the impulsive DeltaV in Cartesian coordinates (X, Y, Z)."
ImpulsiveDeltaVRepresentation.SPHERICAL_IMPULSIVE_DELTA_V.__doc__ = "Spherical - specify the impulsive DeltaV in spherical coordinates (Azimuth, Elevation, Magnitude)."

agcls.AgTypeNameMap["ImpulsiveDeltaVRepresentation"] = ImpulsiveDeltaVRepresentation

class LambertTargetCoordinateType(IntEnum):
    """Lambert Target CoordType."""

    CARTESIAN = 0
    """Cartesian - specifying an orbit by three position elements and three velocity elements in a rectangular coordinate system."""
    KEPLERIAN = 1
    """Keplerian - the classical system, specifying an orbit by six elements describing its size, shape and three-dimensional orientation in space."""

LambertTargetCoordinateType.CARTESIAN.__doc__ = "Cartesian - specifying an orbit by three position elements and three velocity elements in a rectangular coordinate system."
LambertTargetCoordinateType.KEPLERIAN.__doc__ = "Keplerian - the classical system, specifying an orbit by six elements describing its size, shape and three-dimensional orientation in space."

agcls.AgTypeNameMap["LambertTargetCoordinateType"] = LambertTargetCoordinateType

class LambertSolutionOptionType(IntEnum):
    """Lambert Solution Option Type."""

    FIXED_TIME = 0
    """Fixed time - The type of Lambert solution to calculate."""
    MIN_ECCENTRICITY = 1
    """Minimum Eccentricity - The type of Lambert solution to calculate."""
    MIN_ENERGY = 2
    """Minimum Energy - The type of Lambert solution to calculate."""

LambertSolutionOptionType.FIXED_TIME.__doc__ = "Fixed time - The type of Lambert solution to calculate."
LambertSolutionOptionType.MIN_ECCENTRICITY.__doc__ = "Minimum Eccentricity - The type of Lambert solution to calculate."
LambertSolutionOptionType.MIN_ENERGY.__doc__ = "Minimum Energy - The type of Lambert solution to calculate."

agcls.AgTypeNameMap["LambertSolutionOptionType"] = LambertSolutionOptionType

class LambertOrbitalEnergyType(IntEnum):
    """Lambert Orbital Energy Type."""

    LOW = 0
    """Low energy- This is the orbital energy for the Lambert solution that corresponds to the smaller semimajor axis solution."""
    HIGH = 1
    """High energy- This is the orbital energy for the Lambert solution that corresponds to the larger semimajor axis solution."""

LambertOrbitalEnergyType.LOW.__doc__ = "Low energy- This is the orbital energy for the Lambert solution that corresponds to the smaller semimajor axis solution."
LambertOrbitalEnergyType.HIGH.__doc__ = "High energy- This is the orbital energy for the Lambert solution that corresponds to the larger semimajor axis solution."

agcls.AgTypeNameMap["LambertOrbitalEnergyType"] = LambertOrbitalEnergyType

class LambertDirectionOfMotionType(IntEnum):
    """Lambert Direction Of Motion Type."""

    SHORT = 0
    """The short direction of motion for the Lambert solution."""
    LONG = 1
    """The long direction of motion for the Lambert solution."""

LambertDirectionOfMotionType.SHORT.__doc__ = "The short direction of motion for the Lambert solution."
LambertDirectionOfMotionType.LONG.__doc__ = "The long direction of motion for the Lambert solution."

agcls.AgTypeNameMap["LambertDirectionOfMotionType"] = LambertDirectionOfMotionType

class GoldenSectionDesiredOperation(IntEnum):
    """The types for Desired Operation/Objective of golden section profile."""

    MINIMIZE_VALUE = 0
    """Minimize value."""
    MAXIMIZE_VALUE = 1
    """Maximize value."""

GoldenSectionDesiredOperation.MINIMIZE_VALUE.__doc__ = "Minimize value."
GoldenSectionDesiredOperation.MAXIMIZE_VALUE.__doc__ = "Maximize value."

agcls.AgTypeNameMap["GoldenSectionDesiredOperation"] = GoldenSectionDesiredOperation

class GridSearchDesiredOperation(IntEnum):
    """The types for Desired Operation/Objective of Grid Search profile."""

    MINIMIZE_VALUE = 0
    """Minimize value."""
    MAXIMIZE_VALUE = 1
    """Maximize value."""

GridSearchDesiredOperation.MINIMIZE_VALUE.__doc__ = "Minimize value."
GridSearchDesiredOperation.MAXIMIZE_VALUE.__doc__ = "Maximize value."

agcls.AgTypeNameMap["GridSearchDesiredOperation"] = GridSearchDesiredOperation

class ElementType(IntEnum):
    """Which type of elements (osculating or mean)."""

    OSCULATING = 0
    """Osculating."""
    KOZAI_IZSAK_MEAN = 1
    """Kozai-Izsak Mean."""
    BROUWER_LYDDANE_MEAN_LONG = 2
    """Brouwer-Lyddane Mean Long."""
    BROUWER_LYDDANE_MEAN_SHORT = 3
    """Brouwer-Lyddane Mean Short."""

ElementType.OSCULATING.__doc__ = "Osculating."
ElementType.KOZAI_IZSAK_MEAN.__doc__ = "Kozai-Izsak Mean."
ElementType.BROUWER_LYDDANE_MEAN_LONG.__doc__ = "Brouwer-Lyddane Mean Long."
ElementType.BROUWER_LYDDANE_MEAN_SHORT.__doc__ = "Brouwer-Lyddane Mean Short."

agcls.AgTypeNameMap["ElementType"] = ElementType

class BaseSelection(IntEnum):
    """Access base object selections types."""

    SPECIFY = 0
    """An object in the current scenario."""
    CURRENT_SATELLITE = 1
    """The current satellite."""

BaseSelection.SPECIFY.__doc__ = "An object in the current scenario."
BaseSelection.CURRENT_SATELLITE.__doc__ = "The current satellite."

agcls.AgTypeNameMap["BaseSelection"] = BaseSelection

class ControlOrbitStateValue(IntEnum):
    """Orbit State Value properties that can be selected as control parameters for a Target Sequence."""

    VX = 0
    """The Vx velocity component of the orbit state."""
    VY = 1
    """The Vy velocity component of the orbit state."""
    VZ = 2
    """The Vz velocity component of the orbit state."""
    X = 3
    """The x position component of the orbit state."""
    Y = 4
    """The y position component of the orbit state."""
    Z = 5
    """The z position component of the orbit state."""

ControlOrbitStateValue.VX.__doc__ = "The Vx velocity component of the orbit state."
ControlOrbitStateValue.VY.__doc__ = "The Vy velocity component of the orbit state."
ControlOrbitStateValue.VZ.__doc__ = "The Vz velocity component of the orbit state."
ControlOrbitStateValue.X.__doc__ = "The x position component of the orbit state."
ControlOrbitStateValue.Y.__doc__ = "The y position component of the orbit state."
ControlOrbitStateValue.Z.__doc__ = "The z position component of the orbit state."

agcls.AgTypeNameMap["ControlOrbitStateValue"] = ControlOrbitStateValue

class SegmentState(IntEnum):
    """Segment state to use types."""

    INITIAL = 0
    """The segment initial state."""
    FINAL = 1
    """The segment final state."""

SegmentState.INITIAL.__doc__ = "The segment initial state."
SegmentState.FINAL.__doc__ = "The segment final state."

agcls.AgTypeNameMap["SegmentState"] = SegmentState

class DifferenceOrder(IntEnum):
    """The Difference order types."""

    INITIAL_MINUS_CURRENT = 0
    """The initial minus current difference."""
    CURRENT_MINUS_INITIAL = 1
    """The current minus initial difference."""

DifferenceOrder.INITIAL_MINUS_CURRENT.__doc__ = "The initial minus current difference."
DifferenceOrder.CURRENT_MINUS_INITIAL.__doc__ = "The current minus initial difference."

agcls.AgTypeNameMap["DifferenceOrder"] = DifferenceOrder

class SegmentDifferenceOrder(IntEnum):
    """The Difference Across Segments order types."""

    CURRENT_MINUS_SEGMENT = 0
    """The current minus segment difference."""
    SEGMENT_MINUS_CURRENT = 1
    """The segment minus current difference."""

SegmentDifferenceOrder.CURRENT_MINUS_SEGMENT.__doc__ = "The current minus segment difference."
SegmentDifferenceOrder.SEGMENT_MINUS_CURRENT.__doc__ = "The segment minus current difference."

agcls.AgTypeNameMap["SegmentDifferenceOrder"] = SegmentDifferenceOrder

class ControlRepeatingGroundTrackErr(IntEnum):
    """Repeating Ground Track Equator Error properties that can be selected as control parameters for a Target Sequence."""

    REFERENCE_LONGITUDE = 0
    """The reference longitude."""
    REPEAT_COUNT = 1
    """The repeat count."""

ControlRepeatingGroundTrackErr.REFERENCE_LONGITUDE.__doc__ = "The reference longitude."
ControlRepeatingGroundTrackErr.REPEAT_COUNT.__doc__ = "The repeat count."

agcls.AgTypeNameMap["ControlRepeatingGroundTrackErr"] = ControlRepeatingGroundTrackErr

class CalculationObjectDirection(IntEnum):
    """The direction to search for a desired value."""

    NEXT = 0
    """Search in the direction of the next calculation object."""
    PREVIOUS = 1
    """Search in the direction of the previous calculation object."""

CalculationObjectDirection.NEXT.__doc__ = "Search in the direction of the next calculation object."
CalculationObjectDirection.PREVIOUS.__doc__ = "Search in the direction of the previous calculation object."

agcls.AgTypeNameMap["CalculationObjectDirection"] = CalculationObjectDirection

class CalculationObjectOrbitPlaneSource(IntEnum):
    """The calculation object orbit plane source Types."""

    REFERENCE_SATELLITE = 0
    """Use the reference satellite to generate the orbit plane."""
    SATELLITE = 1
    """Use the satellite to generate the orbit plane."""

CalculationObjectOrbitPlaneSource.REFERENCE_SATELLITE.__doc__ = "Use the reference satellite to generate the orbit plane."
CalculationObjectOrbitPlaneSource.SATELLITE.__doc__ = "Use the satellite to generate the orbit plane."

agcls.AgTypeNameMap["CalculationObjectOrbitPlaneSource"] = CalculationObjectOrbitPlaneSource

class CalculationObjectSunPosition(IntEnum):
    """The calculation object sun location Types."""

    APPARENT_FROM_SATELLITE = 0
    """Use the sun location apparent from satellite."""
    APPARENT_FROM_REFERENCE_SATELLITE = 1
    """Use the sun location apparent from reference satellite."""
    TRUE_FROM_SATELLITE = 2
    """Use the sun location true from satellite."""
    TRUE_FROM_REFERENCE_SATELLITE = 3
    """Use the sun location true from reference satellite."""

CalculationObjectSunPosition.APPARENT_FROM_SATELLITE.__doc__ = "Use the sun location apparent from satellite."
CalculationObjectSunPosition.APPARENT_FROM_REFERENCE_SATELLITE.__doc__ = "Use the sun location apparent from reference satellite."
CalculationObjectSunPosition.TRUE_FROM_SATELLITE.__doc__ = "Use the sun location true from satellite."
CalculationObjectSunPosition.TRUE_FROM_REFERENCE_SATELLITE.__doc__ = "Use the sun location true from reference satellite."

agcls.AgTypeNameMap["CalculationObjectSunPosition"] = CalculationObjectSunPosition

class CalculationObjectAngleSign(IntEnum):
    """The sign of the angle when the relative position has a component along the orbit normal."""

    POSITIVE = 0
    """Use the positive sign for the angle when the relative position has a positive component along the orbit normal."""
    NEGATIVE = 1
    """Use the negative sign for the angle when the relative position does not have a positive component along the orbit normal."""

CalculationObjectAngleSign.POSITIVE.__doc__ = "Use the positive sign for the angle when the relative position has a positive component along the orbit normal."
CalculationObjectAngleSign.NEGATIVE.__doc__ = "Use the negative sign for the angle when the relative position does not have a positive component along the orbit normal."

agcls.AgTypeNameMap["CalculationObjectAngleSign"] = CalculationObjectAngleSign

class CalculationObjectReferenceDirection(IntEnum):
    """Direction that establishes the zero value when projected into the orbit plane."""

    REFERENCE_SATELLITE_POSITION = 0
    """Use the reference satellite position when projected into the orbit plane."""
    SATELLITE_POSITION = 1
    """Use the satellite position when projected into the orbit plane."""
    REFERENCE_SATELLITE_NADIR = 2
    """Use the reference satellite nadir (centric) when projected into the orbit plane."""
    SATELLITE_NADIR = 3
    """Use the satellite nadir (centric) position when projected into the orbit plane."""

CalculationObjectReferenceDirection.REFERENCE_SATELLITE_POSITION.__doc__ = "Use the reference satellite position when projected into the orbit plane."
CalculationObjectReferenceDirection.SATELLITE_POSITION.__doc__ = "Use the satellite position when projected into the orbit plane."
CalculationObjectReferenceDirection.REFERENCE_SATELLITE_NADIR.__doc__ = "Use the reference satellite nadir (centric) when projected into the orbit plane."
CalculationObjectReferenceDirection.SATELLITE_NADIR.__doc__ = "Use the satellite nadir (centric) position when projected into the orbit plane."

agcls.AgTypeNameMap["CalculationObjectReferenceDirection"] = CalculationObjectReferenceDirection

class CalculationObjectRelativePosition(IntEnum):
    """The calculation object relative position Types."""

    SATELLITE_TO_REFERENCE_SATELLITE = 0
    """Use the relative position satellite to reference satellite."""
    REFERENCE_SATELLITE_TO_SATELLITE = 1
    """Use the relative position reference satellite to satellite."""

CalculationObjectRelativePosition.SATELLITE_TO_REFERENCE_SATELLITE.__doc__ = "Use the relative position satellite to reference satellite."
CalculationObjectRelativePosition.REFERENCE_SATELLITE_TO_SATELLITE.__doc__ = "Use the relative position reference satellite to satellite."

agcls.AgTypeNameMap["CalculationObjectRelativePosition"] = CalculationObjectRelativePosition

class CalculationObjectReferenceEllipse(IntEnum):
    """The calculation object reference ellipse Types."""

    REFERENCE_SATELLITE_ORBIT = 0
    """Use the reference satellite orbit."""
    SATELLITE_ORBIT = 1
    """Use the satellite orbit."""

CalculationObjectReferenceEllipse.REFERENCE_SATELLITE_ORBIT.__doc__ = "Use the reference satellite orbit."
CalculationObjectReferenceEllipse.SATELLITE_ORBIT.__doc__ = "Use the satellite orbit."

agcls.AgTypeNameMap["CalculationObjectReferenceEllipse"] = CalculationObjectReferenceEllipse

class CalculationObjectLocationSource(IntEnum):
    """The calculation object location source Types."""

    REFERENCE_SATELLITE = 0
    """Use the reference satellite."""
    SATELLITE = 1
    """Use the satellite."""

CalculationObjectLocationSource.REFERENCE_SATELLITE.__doc__ = "Use the reference satellite."
CalculationObjectLocationSource.SATELLITE.__doc__ = "Use the satellite."

agcls.AgTypeNameMap["CalculationObjectLocationSource"] = CalculationObjectLocationSource

class GravitationalParameterSource(IntEnum):
    """The source of the gravitational parameter for a CStateCalcGravitationalParameter calculation object."""

    CENTRAL_BODY_FILE = 0
    """Cb File."""
    CENTRAL_BODY_FILE_SYSTEM = 1
    """Cb File -  System."""
    DESIGN_EXPLORER_OPTIMIZER_FILE = 2
    """DE File."""
    GRAVITY_FILE = 3
    """Gravity File."""

GravitationalParameterSource.CENTRAL_BODY_FILE.__doc__ = "Cb File."
GravitationalParameterSource.CENTRAL_BODY_FILE_SYSTEM.__doc__ = "Cb File -  System."
GravitationalParameterSource.DESIGN_EXPLORER_OPTIMIZER_FILE.__doc__ = "DE File."
GravitationalParameterSource.GRAVITY_FILE.__doc__ = "Gravity File."

agcls.AgTypeNameMap["GravitationalParameterSource"] = GravitationalParameterSource

class ReferenceRadiusSource(IntEnum):
    """The source of the reference radius for a CStateCalcReferenceRadius calculation object."""

    CENTRAL_BODY_FILE = 0
    """Cb File."""
    GRAVITY_FILE = 1
    """Gravity File."""

ReferenceRadiusSource.CENTRAL_BODY_FILE.__doc__ = "Cb File."
ReferenceRadiusSource.GRAVITY_FILE.__doc__ = "Gravity File."

agcls.AgTypeNameMap["ReferenceRadiusSource"] = ReferenceRadiusSource

class GravityCoefficientNormalizationType(IntEnum):
    """The normalization type for the CStateCalcGravCoeff calculation object."""

    NORMALIZED = 0
    """Normalized."""
    UNNORMALIZED = 1
    """Unnormalized."""

GravityCoefficientNormalizationType.NORMALIZED.__doc__ = "Normalized."
GravityCoefficientNormalizationType.UNNORMALIZED.__doc__ = "Unnormalized."

agcls.AgTypeNameMap["GravityCoefficientNormalizationType"] = GravityCoefficientNormalizationType

class GravityCoefficientType(IntEnum):
    """The coefficient type for the CStateCalcGravCoeff calculation object."""

    ZONAL = 0
    """Zonal."""
    COSINE = 1
    """Cosine."""
    SINE = 2
    """Sine."""

GravityCoefficientType.ZONAL.__doc__ = "Zonal."
GravityCoefficientType.COSINE.__doc__ = "Cosine."
GravityCoefficientType.SINE.__doc__ = "Sine."

agcls.AgTypeNameMap["GravityCoefficientType"] = GravityCoefficientType

class STMPerturbationVariables(IntEnum):
    """The initial and final Cartesian variational variables that describe an STM element."""

    POSITION_X = 0
    """Designates the initial or final variation in the 'x' component of position."""
    POSITION_Y = 1
    """Designates the initial or final variation in the 'y' component of position."""
    POSITION_Z = 2
    """Designates the initial or final variation in the 'z' component of position."""
    VELOCITY_X = 3
    """Designates the initial or final variation in the 'x' component of velocity."""
    VELOCITY_Y = 4
    """Designates the initial or final variation in the 'y' component of velocity."""
    VELOCITY_Z = 5
    """Designates the initial or final variation in the 'z' component of velocity."""

STMPerturbationVariables.POSITION_X.__doc__ = "Designates the initial or final variation in the 'x' component of position."
STMPerturbationVariables.POSITION_Y.__doc__ = "Designates the initial or final variation in the 'y' component of position."
STMPerturbationVariables.POSITION_Z.__doc__ = "Designates the initial or final variation in the 'z' component of position."
STMPerturbationVariables.VELOCITY_X.__doc__ = "Designates the initial or final variation in the 'x' component of velocity."
STMPerturbationVariables.VELOCITY_Y.__doc__ = "Designates the initial or final variation in the 'y' component of velocity."
STMPerturbationVariables.VELOCITY_Z.__doc__ = "Designates the initial or final variation in the 'z' component of velocity."

agcls.AgTypeNameMap["STMPerturbationVariables"] = STMPerturbationVariables

class STMEigenNumber(IntEnum):
    """The number that describes one of the 6 STM Eigenvalues or Eigenvectors."""

    NUMBER1 = 0
    """Designates the first Eigenvalue or Eigenvector."""
    NUMBER2 = 1
    """Designates the second Eigenvalue or Eigenvector."""
    NUMBER3 = 2
    """Designates the third Eigenvalue or Eigenvector."""
    NUMBER4 = 3
    """Designates the fourth Eigenvalue or Eigenvector."""
    NUMBER5 = 4
    """Designates the fifth Eigenvalue or Eigenvector."""
    NUMBER6 = 5
    """Designates the sixth Eigenvalue or Eigenvector."""

STMEigenNumber.NUMBER1.__doc__ = "Designates the first Eigenvalue or Eigenvector."
STMEigenNumber.NUMBER2.__doc__ = "Designates the second Eigenvalue or Eigenvector."
STMEigenNumber.NUMBER3.__doc__ = "Designates the third Eigenvalue or Eigenvector."
STMEigenNumber.NUMBER4.__doc__ = "Designates the fourth Eigenvalue or Eigenvector."
STMEigenNumber.NUMBER5.__doc__ = "Designates the fifth Eigenvalue or Eigenvector."
STMEigenNumber.NUMBER6.__doc__ = "Designates the sixth Eigenvalue or Eigenvector."

agcls.AgTypeNameMap["STMEigenNumber"] = STMEigenNumber

class ComplexNumber(IntEnum):
    """Whether a value represents the real or imaginary portion of a number."""

    REAL = 0
    """Designates the value as the real part."""
    IMAGINARY = 1
    """Designates the value as the imaginary part."""

ComplexNumber.REAL.__doc__ = "Designates the value as the real part."
ComplexNumber.IMAGINARY.__doc__ = "Designates the value as the imaginary part."

agcls.AgTypeNameMap["ComplexNumber"] = ComplexNumber

class SquaredType(IntEnum):
    """Whether to calculate the value as the square of the sum of each component or the sum of the squares."""

    SUM_OF_SQUARES = 0
    """Designates the calculation to be the sum of the squares."""
    SQUARE_OF_SUM = 1
    """Designates the calculation to be the square of the sum."""

SquaredType.SUM_OF_SQUARES.__doc__ = "Designates the calculation to be the sum of the squares."
SquaredType.SQUARE_OF_SUM.__doc__ = "Designates the calculation to be the square of the sum."

agcls.AgTypeNameMap["SquaredType"] = SquaredType

class GeoStationaryDriftRateModel(IntEnum):
    """Gravity models used to compute geostationary drift rate."""

    POINT_MASS = 0
    """Compute drift rate using two-body point mass gravity model."""
    POINT_MASS_PLUS_J2 = 1
    """Compute drift rate using gravity model that includes point mass plus secular effect of J2."""

GeoStationaryDriftRateModel.POINT_MASS.__doc__ = "Compute drift rate using two-body point mass gravity model."
GeoStationaryDriftRateModel.POINT_MASS_PLUS_J2.__doc__ = "Compute drift rate using gravity model that includes point mass plus secular effect of J2."

agcls.AgTypeNameMap["GeoStationaryDriftRateModel"] = GeoStationaryDriftRateModel

class GeoStationaryInclinationMagnitude(IntEnum):
    """Magnitude to use when computing the inclination vector."""

    INCLINATION_ANGLE = 0
    """Compute inclination vector magnitude as the inclination angle itself (in radins)."""
    SIN_INCLINATION = 1
    """Compute inclination vector magnitude as sin(inclination)."""
    SIN_HALF_INCLINATION = 2
    """Compute inclination vector magnitude as sin(0.5*inclination)."""
    TWICE_SIN_HALF_INCLINATION = 3
    """Compute inclination vector magnitude as 2*sin(0.5*inclination)."""
    TAN_HALF_INCLINATION = 4
    """Compute inclination vector magnitude as tan(0.5*inclination)."""
    TWICE_TAN_HALF_INCLINATION = 5
    """Compute inclination vector magnitude as 2*tan(0.5*inclination)."""

GeoStationaryInclinationMagnitude.INCLINATION_ANGLE.__doc__ = "Compute inclination vector magnitude as the inclination angle itself (in radins)."
GeoStationaryInclinationMagnitude.SIN_INCLINATION.__doc__ = "Compute inclination vector magnitude as sin(inclination)."
GeoStationaryInclinationMagnitude.SIN_HALF_INCLINATION.__doc__ = "Compute inclination vector magnitude as sin(0.5*inclination)."
GeoStationaryInclinationMagnitude.TWICE_SIN_HALF_INCLINATION.__doc__ = "Compute inclination vector magnitude as 2*sin(0.5*inclination)."
GeoStationaryInclinationMagnitude.TAN_HALF_INCLINATION.__doc__ = "Compute inclination vector magnitude as tan(0.5*inclination)."
GeoStationaryInclinationMagnitude.TWICE_TAN_HALF_INCLINATION.__doc__ = "Compute inclination vector magnitude as 2*tan(0.5*inclination)."

agcls.AgTypeNameMap["GeoStationaryInclinationMagnitude"] = GeoStationaryInclinationMagnitude

class CentralBodyGravityModel(IntEnum):
    """The gravity model."""

    ZONALS_TO_J4 = 0
    """ZonalsToJ4 - (various) Gravity model for all central bodies except Sun, Earth and Moon."""
    EARTH_SIMPLE = 1
    """Earth Simple gravity model."""
    WGS84 = 2
    """WGS84 - (Earth) World Geodetic System 1984; WGS 84 was created by the Defense Mapping Agency (DMA)."""
    EGM96 = 3
    """EGM96 - (Earth) Earth Gravity Model 1996, a geopotential model of the Earth consisting of spherical harmonic coefficients complete to degree and order 360. Developed jointly by NGA (formerly known as NIMA), NASA Goddard and Ohio State University."""
    GEMT1 = 4
    """GEMT1 - (Earth) Goddard Earth Model T1."""
    JGM2 = 5
    """JGM2 - (Earth) Joint Gravity Model version 2, a model that describes the Earth gravity field up to degree and order 70, developed by NASA/GSFC Space Geodesy Branch, the University of Texas Center for Space Research and CNES."""
    JGM3 = 6
    """JGM3 - (Earth) Joint Gravity Model version 3, a model that describes the Earth gravity field up to degree and order 70, developed by the University of Texas and NASA/GSFC."""
    WSG84EGM96 = 7
    """WGS84 EGM96 - (Earth) Uses the coefficients from EGM96 with the shape model of WGS84. This model is the recommended gravity model of the WGS84 definition document: NIMA TR8350.2, Third Edition, 4 July 1997."""
    WGS84_OLD = 8
    """WGS84 old - (Earth) Old version of WGS84."""
    GLGM2 = 9
    """GLGM2 - (Moon) GM = 4.9028029535968e+12, reference distance = 1,738,000 m."""
    LP165P = 10
    """LP165P - (Moon) GM = 4.902801056E+12, reference distance = 1,738,000.0 m."""
    ICARUS1987 = 11
    """Icarus1987 - (Mercury) GM = 2.203209e+013, reference distance = 2,439,000 m. Reference: Anderson, J. J., Colombo, G., Esposito, P. B., Lau E. L., and Trager, G. B. 'The Mass, Gravity Field, and Ephemeris of Mercury', Icarus 71, 337-349, 1987."""
    MGNP180U = 12
    """MGNP180U - (Venus) GM = 3.248585920790000E+14, reference distance = 6,051,000.0 m."""
    GMM1 = 13
    """GMM1 - (Mars) GM = 4.28283579647735e+13, reference distance = 3,394,200.0 m."""
    GMM2B = 14
    """GMM2B - (Mars) GM = 4.28283719012840e+13, reference distance = 3,397,000 m. Reference: These numbers came from the GMM-2B model published at `NASA <https://www.nasa.gov/>`_ and were gotten from Journal of Geophysical Research, November 2000."""
    MARS50_C = 15
    """Mars50c - (Mars) GM = 4.2828370371000e+13, reference distance = 3,394,200 m."""
    JUP230 = 16
    """JUP230 - (Jupiter) GM = 1.26686535e+017, reference distance = 71,492,000 m. Reference: Jacobson, R. A. The JUP230 orbit solution, 2003."""
    ASTRON2004 = 17
    """Astron2004 - (Saturn) GM = 3.7931284e+016, reference distance = 60,330,000 m."""
    ASTRON_ASTRO1991 = 18
    """AstronAstro1991 - (Neptune) GM = 6.835107e+015, reference distance = 25,225,000 m."""
    ICARUS2001 = 19
    """Icarus2001 - (Callisto) GM = 7.179292e+12, reference distance = 2,410,300 m."""
    SCIENCE1998 = 20
    """Science1998 - (Europa) GM =3.20272e+012, reference distance = 1,565,000 m."""
    NATURE1996 = 21
    """Nature1996 - (Ganymede) GM = 9.8866e+12, reference distance = 2,634,000 m."""
    J_GEO_RES2001 = 22
    """JGeoRes2001 - (Io) GM = 5.96e+12, reference distance = 1,821,600 m."""
    GGM01C = 23
    """GGM01C - (Earth)."""
    GGM02C = 24
    """GGM02C - (Earth)."""
    WGS72_ZONALS_TO_J4 = 25
    """WGS72 ZonalsToJ4 - (Earth)."""
    LP100J = 26
    """LP100J - (Moon)."""
    LP100K = 27
    """LP100K - (Moon)."""
    LP150Q = 28
    """LP150Q - (Moon)."""
    LP75G = 29
    """LP75G - (Moon)."""

CentralBodyGravityModel.ZONALS_TO_J4.__doc__ = "ZonalsToJ4 - (various) Gravity model for all central bodies except Sun, Earth and Moon."
CentralBodyGravityModel.EARTH_SIMPLE.__doc__ = "Earth Simple gravity model."
CentralBodyGravityModel.WGS84.__doc__ = "WGS84 - (Earth) World Geodetic System 1984; WGS 84 was created by the Defense Mapping Agency (DMA)."
CentralBodyGravityModel.EGM96.__doc__ = "EGM96 - (Earth) Earth Gravity Model 1996, a geopotential model of the Earth consisting of spherical harmonic coefficients complete to degree and order 360. Developed jointly by NGA (formerly known as NIMA), NASA Goddard and Ohio State University."
CentralBodyGravityModel.GEMT1.__doc__ = "GEMT1 - (Earth) Goddard Earth Model T1."
CentralBodyGravityModel.JGM2.__doc__ = "JGM2 - (Earth) Joint Gravity Model version 2, a model that describes the Earth gravity field up to degree and order 70, developed by NASA/GSFC Space Geodesy Branch, the University of Texas Center for Space Research and CNES."
CentralBodyGravityModel.JGM3.__doc__ = "JGM3 - (Earth) Joint Gravity Model version 3, a model that describes the Earth gravity field up to degree and order 70, developed by the University of Texas and NASA/GSFC."
CentralBodyGravityModel.WSG84EGM96.__doc__ = "WGS84 EGM96 - (Earth) Uses the coefficients from EGM96 with the shape model of WGS84. This model is the recommended gravity model of the WGS84 definition document: NIMA TR8350.2, Third Edition, 4 July 1997."
CentralBodyGravityModel.WGS84_OLD.__doc__ = "WGS84 old - (Earth) Old version of WGS84."
CentralBodyGravityModel.GLGM2.__doc__ = "GLGM2 - (Moon) GM = 4.9028029535968e+12, reference distance = 1,738,000 m."
CentralBodyGravityModel.LP165P.__doc__ = "LP165P - (Moon) GM = 4.902801056E+12, reference distance = 1,738,000.0 m."
CentralBodyGravityModel.ICARUS1987.__doc__ = "Icarus1987 - (Mercury) GM = 2.203209e+013, reference distance = 2,439,000 m. Reference: Anderson, J. J., Colombo, G., Esposito, P. B., Lau E. L., and Trager, G. B. 'The Mass, Gravity Field, and Ephemeris of Mercury', Icarus 71, 337-349, 1987."
CentralBodyGravityModel.MGNP180U.__doc__ = "MGNP180U - (Venus) GM = 3.248585920790000E+14, reference distance = 6,051,000.0 m."
CentralBodyGravityModel.GMM1.__doc__ = "GMM1 - (Mars) GM = 4.28283579647735e+13, reference distance = 3,394,200.0 m."
CentralBodyGravityModel.GMM2B.__doc__ = "GMM2B - (Mars) GM = 4.28283719012840e+13, reference distance = 3,397,000 m. Reference: These numbers came from the GMM-2B model published at `NASA <https://www.nasa.gov/>`_ and were gotten from Journal of Geophysical Research, November 2000."
CentralBodyGravityModel.MARS50_C.__doc__ = "Mars50c - (Mars) GM = 4.2828370371000e+13, reference distance = 3,394,200 m."
CentralBodyGravityModel.JUP230.__doc__ = "JUP230 - (Jupiter) GM = 1.26686535e+017, reference distance = 71,492,000 m. Reference: Jacobson, R. A. The JUP230 orbit solution, 2003."
CentralBodyGravityModel.ASTRON2004.__doc__ = "Astron2004 - (Saturn) GM = 3.7931284e+016, reference distance = 60,330,000 m."
CentralBodyGravityModel.ASTRON_ASTRO1991.__doc__ = "AstronAstro1991 - (Neptune) GM = 6.835107e+015, reference distance = 25,225,000 m."
CentralBodyGravityModel.ICARUS2001.__doc__ = "Icarus2001 - (Callisto) GM = 7.179292e+12, reference distance = 2,410,300 m."
CentralBodyGravityModel.SCIENCE1998.__doc__ = "Science1998 - (Europa) GM =3.20272e+012, reference distance = 1,565,000 m."
CentralBodyGravityModel.NATURE1996.__doc__ = "Nature1996 - (Ganymede) GM = 9.8866e+12, reference distance = 2,634,000 m."
CentralBodyGravityModel.J_GEO_RES2001.__doc__ = "JGeoRes2001 - (Io) GM = 5.96e+12, reference distance = 1,821,600 m."
CentralBodyGravityModel.GGM01C.__doc__ = "GGM01C - (Earth)."
CentralBodyGravityModel.GGM02C.__doc__ = "GGM02C - (Earth)."
CentralBodyGravityModel.WGS72_ZONALS_TO_J4.__doc__ = "WGS72 ZonalsToJ4 - (Earth)."
CentralBodyGravityModel.LP100J.__doc__ = "LP100J - (Moon)."
CentralBodyGravityModel.LP100K.__doc__ = "LP100K - (Moon)."
CentralBodyGravityModel.LP150Q.__doc__ = "LP150Q - (Moon)."
CentralBodyGravityModel.LP75G.__doc__ = "LP75G - (Moon)."

agcls.AgTypeNameMap["CentralBodyGravityModel"] = CentralBodyGravityModel

class CentralBodyShape(IntEnum):
    """The central body shape types."""

    TRIAXIAL_ELLIPSOID = 0
    """A triaxial ellipsoid defined by a semi-major axis, semi-mid axis and semi-minor axis."""
    OBLATE_SPHEROID = 1
    """An oblate spheroid defined by a minimum and maximum radii, and a derived flattening coefficient."""
    SPHERE = 2
    """A sphere defined by a radius."""

CentralBodyShape.TRIAXIAL_ELLIPSOID.__doc__ = "A triaxial ellipsoid defined by a semi-major axis, semi-mid axis and semi-minor axis."
CentralBodyShape.OBLATE_SPHEROID.__doc__ = "An oblate spheroid defined by a minimum and maximum radii, and a derived flattening coefficient."
CentralBodyShape.SPHERE.__doc__ = "A sphere defined by a radius."

agcls.AgTypeNameMap["CentralBodyShape"] = CentralBodyShape

class CentralBodyAttitude(IntEnum):
    """The central body attitude types."""

    IAU1994 = 0
    """An IAU 1994 attitude definition."""
    ROTATION_COEFFICIENTS_FILE = 1
    """A rotation coefficient file."""

CentralBodyAttitude.IAU1994.__doc__ = "An IAU 1994 attitude definition."
CentralBodyAttitude.ROTATION_COEFFICIENTS_FILE.__doc__ = "A rotation coefficient file."

agcls.AgTypeNameMap["CentralBodyAttitude"] = CentralBodyAttitude

class CentralBodyEphemeris(IntEnum):
    """The central body ephemeris types."""

    ANALYTIC_ORBIT = 0
    """Specified values and rates of change for the classical orbital elements."""
    FILE = 1
    """An external ephemeris (.e) file."""
    JPLDE = 2
    """Ephemerides from the Jet Propulsion Laboratory's JPL DE set are used."""
    JPLSPICE = 3
    """The SPICE propagator reads ephemeris from binary files that are in a standard format produced by the Jet Propulsion Laboratory for ephemeris for celestial bodies but can be used for spacecraft."""
    PLANETARY = 4
    """A planetary ephemeris (.pe) file."""

CentralBodyEphemeris.ANALYTIC_ORBIT.__doc__ = "Specified values and rates of change for the classical orbital elements."
CentralBodyEphemeris.FILE.__doc__ = "An external ephemeris (.e) file."
CentralBodyEphemeris.JPLDE.__doc__ = "Ephemerides from the Jet Propulsion Laboratory's JPL DE set are used."
CentralBodyEphemeris.JPLSPICE.__doc__ = "The SPICE propagator reads ephemeris from binary files that are in a standard format produced by the Jet Propulsion Laboratory for ephemeris for celestial bodies but can be used for spacecraft."
CentralBodyEphemeris.PLANETARY.__doc__ = "A planetary ephemeris (.pe) file."

agcls.AgTypeNameMap["CentralBodyEphemeris"] = CentralBodyEphemeris

class ControlPowerInternal(IntEnum):
    """Internal Power properties that can be selected as control parameters for a Target Sequence."""

    GENERATED_POWER = 0
    """The power generated by the spacecraft from internal sources."""
    PERCENT_DEGRADATION = 1
    """The percent degradation per year; degradation factor is (1 - x%/yr)timeSinceRefEpoch."""
    EPOCH = 2
    """The date and time used as a reference epoch for degradation."""

ControlPowerInternal.GENERATED_POWER.__doc__ = "The power generated by the spacecraft from internal sources."
ControlPowerInternal.PERCENT_DEGRADATION.__doc__ = "The percent degradation per year; degradation factor is (1 - x%/yr)timeSinceRefEpoch."
ControlPowerInternal.EPOCH.__doc__ = "The date and time used as a reference epoch for degradation."

agcls.AgTypeNameMap["ControlPowerInternal"] = ControlPowerInternal

class ControlPowerProcessed(IntEnum):
    """Processed Power properties that can be selected as control parameters for a Target Sequence."""

    EFFICIENCY = 0
    """The efficiency of the power processing unit(PPU)."""
    LOAD = 1
    """The power diverted from the power source and unavailable to the power processing unit(PPU)."""

ControlPowerProcessed.EFFICIENCY.__doc__ = "The efficiency of the power processing unit(PPU)."
ControlPowerProcessed.LOAD.__doc__ = "The power diverted from the power source and unavailable to the power processing unit(PPU)."

agcls.AgTypeNameMap["ControlPowerProcessed"] = ControlPowerProcessed

class ControlPowerSolarArray(IntEnum):
    """Solar Array Power properties that can be selected as control parameters for a Target Sequence."""

    C0 = 0
    """The ThermalModel.C0 coefficient."""
    C1 = 1
    """The ThermalModel.C1 coefficient."""
    C2 = 2
    """The ThermalModel.C2 coefficient."""
    C3 = 3
    """The ThermalModel.C3 coefficient."""
    C4 = 4
    """The ThermalModel.C4 coefficient."""
    AREA = 5
    """The solar array panel area."""
    EFFICIENCY = 6
    """The array efficiency in producing output power from a collection of cells."""
    CELL_EFFICIENCY = 7
    """The cell efficiency in producing output power from incident sunlight."""
    CONCENTRATION = 8
    """The solar array concentration factor."""
    INCLINATION_TO_SUN_LINE = 9
    """The angle from the panel normal vector to the apparent sun line."""
    PERCENT_DEGRADATION = 10
    """The percent degradation per year; degradation factor is (1 - x%/yr)timeSinceRefEpoch."""
    EPOCH = 11
    """The date and time used as a reference epoch for degradation."""

ControlPowerSolarArray.C0.__doc__ = "The ThermalModel.C0 coefficient."
ControlPowerSolarArray.C1.__doc__ = "The ThermalModel.C1 coefficient."
ControlPowerSolarArray.C2.__doc__ = "The ThermalModel.C2 coefficient."
ControlPowerSolarArray.C3.__doc__ = "The ThermalModel.C3 coefficient."
ControlPowerSolarArray.C4.__doc__ = "The ThermalModel.C4 coefficient."
ControlPowerSolarArray.AREA.__doc__ = "The solar array panel area."
ControlPowerSolarArray.EFFICIENCY.__doc__ = "The array efficiency in producing output power from a collection of cells."
ControlPowerSolarArray.CELL_EFFICIENCY.__doc__ = "The cell efficiency in producing output power from incident sunlight."
ControlPowerSolarArray.CONCENTRATION.__doc__ = "The solar array concentration factor."
ControlPowerSolarArray.INCLINATION_TO_SUN_LINE.__doc__ = "The angle from the panel normal vector to the apparent sun line."
ControlPowerSolarArray.PERCENT_DEGRADATION.__doc__ = "The percent degradation per year; degradation factor is (1 - x%/yr)timeSinceRefEpoch."
ControlPowerSolarArray.EPOCH.__doc__ = "The date and time used as a reference epoch for degradation."

agcls.AgTypeNameMap["ControlPowerSolarArray"] = ControlPowerSolarArray

class ThirdBodyMode(IntEnum):
    """The third body gravity mode."""

    GRAVITY_FIELD = 0
    """Define the gravitational effect as a full Gravitational Force model."""
    POINT_MASS = 1
    """Define the gravitational effect as a third body point mass effect."""

ThirdBodyMode.GRAVITY_FIELD.__doc__ = "Define the gravitational effect as a full Gravitational Force model."
ThirdBodyMode.POINT_MASS.__doc__ = "Define the gravitational effect as a third body point mass effect."

agcls.AgTypeNameMap["ThirdBodyMode"] = ThirdBodyMode

class GravParamSource(IntEnum):
    """The gravity parameter source."""

    CENTRAL_BODY_FILE = 0
    """The Cb file provided with STK; uses the default, body centered gravity source for the central body."""
    DESIGN_EXPLORER_OPTIMIZER_FILE = 1
    """A DE file; body centered for the inner planets and barycentered for the outer planets."""
    USER = 2
    """User defined; requires you to specify the mu value of the Gravitational Parameter."""
    CENTRAL_BODY_FILE_SYSTEM = 3
    """The Cb file provided with STK; uses the default, barycentered gravity source for the central body."""

GravParamSource.CENTRAL_BODY_FILE.__doc__ = "The Cb file provided with STK; uses the default, body centered gravity source for the central body."
GravParamSource.DESIGN_EXPLORER_OPTIMIZER_FILE.__doc__ = "A DE file; body centered for the inner planets and barycentered for the outer planets."
GravParamSource.USER.__doc__ = "User defined; requires you to specify the mu value of the Gravitational Parameter."
GravParamSource.CENTRAL_BODY_FILE_SYSTEM.__doc__ = "The Cb file provided with STK; uses the default, barycentered gravity source for the central body."

agcls.AgTypeNameMap["GravParamSource"] = GravParamSource

class EphemerisSource(IntEnum):
    """The ephemeris source type."""

    CENTRAL_BODY_FILE = 0
    """The Cb file provided with STK; uses the default ephemeris source for that central body."""
    DESIGN_EXPLORER_OPTIMIZER_FILE = 1
    """A DE file; body centered for the inner planets and barycentered for the outer planets."""
    SPICE_BARY = 2
    """A SPICE file, barycentered; uses the entire planetary system as a single effect, with the system center as the point mass."""
    SPICE_BODY = 3
    """A SPICE file, body centered; uses only the planet as the effect, with the planet's center as the point mass."""

EphemerisSource.CENTRAL_BODY_FILE.__doc__ = "The Cb file provided with STK; uses the default ephemeris source for that central body."
EphemerisSource.DESIGN_EXPLORER_OPTIMIZER_FILE.__doc__ = "A DE file; body centered for the inner planets and barycentered for the outer planets."
EphemerisSource.SPICE_BARY.__doc__ = "A SPICE file, barycentered; uses the entire planetary system as a single effect, with the system center as the point mass."
EphemerisSource.SPICE_BODY.__doc__ = "A SPICE file, body centered; uses only the planet as the effect, with the planet's center as the point mass."

agcls.AgTypeNameMap["EphemerisSource"] = EphemerisSource

class SolarForceMethod(IntEnum):
    """The solar force method type for a spherical or N-plate SRP model."""

    LUMINOSITY = 0
    """Luminosity - used as the value for solar flux at 1 AU in the Solar Radiation computation."""
    MEAN_FLUX = 1
    """Mean Flux - used as the value for solar flux at 1 AU in the Solar Radiation computation."""

SolarForceMethod.LUMINOSITY.__doc__ = "Luminosity - used as the value for solar flux at 1 AU in the Solar Radiation computation."
SolarForceMethod.MEAN_FLUX.__doc__ = "Mean Flux - used as the value for solar flux at 1 AU in the Solar Radiation computation."

agcls.AgTypeNameMap["SolarForceMethod"] = SolarForceMethod

class ShadowModel(IntEnum):
    """The shadow model type."""

    CYLINDRICAL = 0
    """Cylindrical - assumes the Sun to be at infinite distance so that all light coming from the Sun moves in a direction parallel to the Sun to satellite vector."""
    DUAL_CONE = 1
    """Dual Cone - uses the actual size and distance of the Sun to model regions of full, partial (penumbra) and zero (umbra) sunlight. The visible fraction of the solar disk is used to compute the acceleration during penumbra."""
    NONE = 2
    """None - turns off all shadowing of the satellite."""

ShadowModel.CYLINDRICAL.__doc__ = "Cylindrical - assumes the Sun to be at infinite distance so that all light coming from the Sun moves in a direction parallel to the Sun to satellite vector."
ShadowModel.DUAL_CONE.__doc__ = "Dual Cone - uses the actual size and distance of the Sun to model regions of full, partial (penumbra) and zero (umbra) sunlight. The visible fraction of the solar disk is used to compute the acceleration during penumbra."
ShadowModel.NONE.__doc__ = "None - turns off all shadowing of the satellite."

agcls.AgTypeNameMap["ShadowModel"] = ShadowModel

class SunPosition(IntEnum):
    """The sun position type."""

    APPARENT = 0
    """Apparent - takes into account the time required for light to travel from the sun to the position of the spacecraft."""
    APPARENT_TO_TRUE_CENTRAL_BODY = 1
    """Apparent Sun to True Cb - takes into account the time required for light to travel from the sun to the central body."""
    TRUE = 2
    """True - assumes that light from the sun reaches the spacecraft instantaneously."""

SunPosition.APPARENT.__doc__ = "Apparent - takes into account the time required for light to travel from the sun to the position of the spacecraft."
SunPosition.APPARENT_TO_TRUE_CENTRAL_BODY.__doc__ = "Apparent Sun to True Cb - takes into account the time required for light to travel from the sun to the central body."
SunPosition.TRUE.__doc__ = "True - assumes that light from the sun reaches the spacecraft instantaneously."

agcls.AgTypeNameMap["SunPosition"] = SunPosition

class AtmosphereDataSource(IntEnum):
    """The Atmospheric data source type."""

    CONSTANT = 0
    """Constant Atmospheric data source."""
    FILE = 1
    """Atmopsheric data source from file."""

AtmosphereDataSource.CONSTANT.__doc__ = "Constant Atmospheric data source."
AtmosphereDataSource.FILE.__doc__ = "Atmopsheric data source from file."

agcls.AgTypeNameMap["AtmosphereDataSource"] = AtmosphereDataSource

class GeoMagneticFluxSource(IntEnum):
    """Whether to use Kp or Ap data from the flux file."""

    AP = 0
    """Read Ap from file."""
    KP = 1
    """Read Kp from file."""

GeoMagneticFluxSource.AP.__doc__ = "Read Ap from file."
GeoMagneticFluxSource.KP.__doc__ = "Read Kp from file."

agcls.AgTypeNameMap["GeoMagneticFluxSource"] = GeoMagneticFluxSource

class GeoMagneticFluxUpdateRate(IntEnum):
    """Method for using geomagnetic flux values from the flux file."""

    RATE3_HOURLY = 0
    """Three hourly."""
    RATE3_HOURLY_CUBIC_SPLINE = 1
    """Three hourly cubic spline."""
    RATE3_HOURLY_INTERPOLATED = 2
    """Three hourly interpolated."""
    DAILY = 3
    """Daily."""

GeoMagneticFluxUpdateRate.RATE3_HOURLY.__doc__ = "Three hourly."
GeoMagneticFluxUpdateRate.RATE3_HOURLY_CUBIC_SPLINE.__doc__ = "Three hourly cubic spline."
GeoMagneticFluxUpdateRate.RATE3_HOURLY_INTERPOLATED.__doc__ = "Three hourly interpolated."
GeoMagneticFluxUpdateRate.DAILY.__doc__ = "Daily."

agcls.AgTypeNameMap["GeoMagneticFluxUpdateRate"] = GeoMagneticFluxUpdateRate

class DragModelType(IntEnum):
    """Type of Drag Model."""

    SPHERICAL = 0
    """Spherical."""
    PLUGIN = 1
    """Plugin."""
    VARIABLE_AREA = 2
    """Variable Area."""
    N_PLATE = 3
    """N-Plate."""

DragModelType.SPHERICAL.__doc__ = "Spherical."
DragModelType.PLUGIN.__doc__ = "Plugin."
DragModelType.VARIABLE_AREA.__doc__ = "Variable Area."
DragModelType.N_PLATE.__doc__ = "N-Plate."

agcls.AgTypeNameMap["DragModelType"] = DragModelType

class MarsGRAMDensityType(IntEnum):
    """Density Type for MarsGRAM Density Models."""

    LOW = 0
    """Low density."""
    MEAN = 1
    """Mean density."""
    HIGH = 2
    """High density."""
    RANDOMLY_PERTURBED = 3
    """Randomly perturbed density."""

MarsGRAMDensityType.LOW.__doc__ = "Low density."
MarsGRAMDensityType.MEAN.__doc__ = "Mean density."
MarsGRAMDensityType.HIGH.__doc__ = "High density."
MarsGRAMDensityType.RANDOMLY_PERTURBED.__doc__ = "Randomly perturbed density."

agcls.AgTypeNameMap["MarsGRAMDensityType"] = MarsGRAMDensityType

class VenusGRAMDensityType(IntEnum):
    """Density Type for VenusGRAM Density Models."""

    LOW = 0
    """Low density."""
    MEAN = 1
    """Mean density."""
    HIGH = 2
    """High density."""
    RANDOMLY_PERTURBED = 3
    """Randomly perturbed density."""

VenusGRAMDensityType.LOW.__doc__ = "Low density."
VenusGRAMDensityType.MEAN.__doc__ = "Mean density."
VenusGRAMDensityType.HIGH.__doc__ = "High density."
VenusGRAMDensityType.RANDOMLY_PERTURBED.__doc__ = "Randomly perturbed density."

agcls.AgTypeNameMap["VenusGRAMDensityType"] = VenusGRAMDensityType

class TabVecInterpolationMethod(IntEnum):
    """The interpolation method for tabulated area vector file."""

    CARTESIAN_INTERPOLATION = 0
    """Bilinear interpolation on the Cartesian grid."""
    MAGNITUDE_AND_DIRECTION_INTERPOLATION = 1
    """Bilinear interpolation in magnitude and direction space."""

TabVecInterpolationMethod.CARTESIAN_INTERPOLATION.__doc__ = "Bilinear interpolation on the Cartesian grid."
TabVecInterpolationMethod.MAGNITUDE_AND_DIRECTION_INTERPOLATION.__doc__ = "Bilinear interpolation in magnitude and direction space."

agcls.AgTypeNameMap["TabVecInterpolationMethod"] = TabVecInterpolationMethod

class ControlEngineConstantAcceleration(IntEnum):
    """Constant Acceleration and Isp engine model properties that can be selected as control parameters for a Target Sequence."""

    GRAV = 0
    """The gravitational acceleration constant at sea level on the Earth."""
    ACCELERATION = 1
    """The acceleration for the engine."""
    ISP = 2
    """The specific impulse for the engine."""

ControlEngineConstantAcceleration.GRAV.__doc__ = "The gravitational acceleration constant at sea level on the Earth."
ControlEngineConstantAcceleration.ACCELERATION.__doc__ = "The acceleration for the engine."
ControlEngineConstantAcceleration.ISP.__doc__ = "The specific impulse for the engine."

agcls.AgTypeNameMap["ControlEngineConstantAcceleration"] = ControlEngineConstantAcceleration

class ControlEngineConstant(IntEnum):
    """Constant Thrust and Isp engine model properties that can be selected as control parameters for a Target Sequence."""

    GRAV = 0
    """The gravitational acceleration constant at sea level on the Earth."""
    THRUST = 1
    """The thrust for the engine."""
    ISP = 2
    """The specific impulse for the engine."""

ControlEngineConstant.GRAV.__doc__ = "The gravitational acceleration constant at sea level on the Earth."
ControlEngineConstant.THRUST.__doc__ = "The thrust for the engine."
ControlEngineConstant.ISP.__doc__ = "The specific impulse for the engine."

agcls.AgTypeNameMap["ControlEngineConstant"] = ControlEngineConstant

class ControlEngineCustom(IntEnum):
    """Custom engine model properties that can be selected as control parameters for a Target Sequence."""

    GRAV = 0
    """Earth surface gravity acceleration for Isp conversions."""

ControlEngineCustom.GRAV.__doc__ = "Earth surface gravity acceleration for Isp conversions."

agcls.AgTypeNameMap["ControlEngineCustom"] = ControlEngineCustom

class ControlEngineThrottleTable(IntEnum):
    """Throttle table engine model properties that can be selected as control parameters for a Target Sequence."""

    GRAV = 0
    """Gravitational acceleration constant at sea level on the Earth."""
    PERCENT_DEGRADATION_PER_YEAR = 1
    """The degradation factor is (1 - x)n, where n is the time since epoch in years, and x is the percent degradation per year."""
    REFERENCE_EPOCH = 2
    """The date and time used as a reference epoch for degradation."""

ControlEngineThrottleTable.GRAV.__doc__ = "Gravitational acceleration constant at sea level on the Earth."
ControlEngineThrottleTable.PERCENT_DEGRADATION_PER_YEAR.__doc__ = "The degradation factor is (1 - x)n, where n is the time since epoch in years, and x is the percent degradation per year."
ControlEngineThrottleTable.REFERENCE_EPOCH.__doc__ = "The date and time used as a reference epoch for degradation."

agcls.AgTypeNameMap["ControlEngineThrottleTable"] = ControlEngineThrottleTable

class ControlEngineIon(IntEnum):
    """Ion engine model properties that can be selected as control parameters for a Target Sequence."""

    FLOW_RATE_CONSTANT_TERM = 0
    """FlowRateModel.C0 - the constant coefficient."""
    FLOW_RATE_LINEAR_TERM = 1
    """FlowRateModel.C1 - the linear coefficient."""
    FLOW_RATE_QUADRATIC_TERM = 2
    """FlowRateModel.C2 - the quadratic coefficient."""
    FLOW_RATE_CUBIC_TERM = 3
    """FlowRateModel.C3 - the cubic coefficient."""
    GRAV = 4
    """Gravitational acceleration constant at sea level on the Earth."""
    ISP_CONSTANT_TERM = 5
    """IspModel.C0 - the constant coefficient."""
    ISP_LINEAR_TERM = 6
    """IspModel.C1 - the linear coefficient."""
    ISP_QUADRATIC_TERM = 7
    """IspModel.C2 - the quadratic coefficient."""
    ISP_CUBIC_TERM = 8
    """IspModel.C3 - the cubic coefficient."""
    MASS_FLOW_EFFICIENCY_CONSTANT_TERM = 9
    """MassFlowEfficiencyModel.C0 - the constant coefficient."""
    MASS_FLOW_EFFICIENCY_LINEAR_TERM = 10
    """MassFlowEfficiencyModel.C1 - the linear coefficient."""
    MASS_FLOW_EFFICIENCY_QUADRATIC_TERM = 11
    """MassFlowEfficiencyModel.C2 - the quadratic coefficient."""
    MASS_FLOW_EFFICIENCY_CUBIC_TERM = 12
    """MassFlowEfficiencyModel.C3 - the cubic coefficient."""
    MAX_INPUT_POWER = 13
    """Minimum power required for the engine to produce thrust."""
    MIN_REQUIRED_POWER = 14
    """Maximum power that can be used by the engine to produce thrust."""
    PERCENT_DEGRADATION_PER_YEAR = 15
    """The degradation factor is (1 - x)n, where n is the time since epoch in years, and x is the percent degradation per year."""
    PERCENT_THROTTLE = 16
    """Percentage of available thrust to use (100 is full on, 0 is off)."""
    POWER_EFFICIENCY_CONSTANT_TERM = 17
    """PowerEfficiencyModel.C0 - the constant coefficient."""
    POWER_EFFICIENCY_LINEAR_TERM = 18
    """PowerEfficiencyModel.C1 - the linear coefficient."""
    POWER_EFFICIENCY_QUADRATIC_TERM = 19
    """PowerEfficiencyModel.C2 - the quadratic coefficient."""
    POWER_EFFICIENCY_CUBIC_TERM = 20
    """PowerEfficiencyModel.C3 - the cubic coefficient."""
    REFERENCE_EPOCH = 21
    """The date and time used as a reference epoch for degradation."""

ControlEngineIon.FLOW_RATE_CONSTANT_TERM.__doc__ = "FlowRateModel.C0 - the constant coefficient."
ControlEngineIon.FLOW_RATE_LINEAR_TERM.__doc__ = "FlowRateModel.C1 - the linear coefficient."
ControlEngineIon.FLOW_RATE_QUADRATIC_TERM.__doc__ = "FlowRateModel.C2 - the quadratic coefficient."
ControlEngineIon.FLOW_RATE_CUBIC_TERM.__doc__ = "FlowRateModel.C3 - the cubic coefficient."
ControlEngineIon.GRAV.__doc__ = "Gravitational acceleration constant at sea level on the Earth."
ControlEngineIon.ISP_CONSTANT_TERM.__doc__ = "IspModel.C0 - the constant coefficient."
ControlEngineIon.ISP_LINEAR_TERM.__doc__ = "IspModel.C1 - the linear coefficient."
ControlEngineIon.ISP_QUADRATIC_TERM.__doc__ = "IspModel.C2 - the quadratic coefficient."
ControlEngineIon.ISP_CUBIC_TERM.__doc__ = "IspModel.C3 - the cubic coefficient."
ControlEngineIon.MASS_FLOW_EFFICIENCY_CONSTANT_TERM.__doc__ = "MassFlowEfficiencyModel.C0 - the constant coefficient."
ControlEngineIon.MASS_FLOW_EFFICIENCY_LINEAR_TERM.__doc__ = "MassFlowEfficiencyModel.C1 - the linear coefficient."
ControlEngineIon.MASS_FLOW_EFFICIENCY_QUADRATIC_TERM.__doc__ = "MassFlowEfficiencyModel.C2 - the quadratic coefficient."
ControlEngineIon.MASS_FLOW_EFFICIENCY_CUBIC_TERM.__doc__ = "MassFlowEfficiencyModel.C3 - the cubic coefficient."
ControlEngineIon.MAX_INPUT_POWER.__doc__ = "Minimum power required for the engine to produce thrust."
ControlEngineIon.MIN_REQUIRED_POWER.__doc__ = "Maximum power that can be used by the engine to produce thrust."
ControlEngineIon.PERCENT_DEGRADATION_PER_YEAR.__doc__ = "The degradation factor is (1 - x)n, where n is the time since epoch in years, and x is the percent degradation per year."
ControlEngineIon.PERCENT_THROTTLE.__doc__ = "Percentage of available thrust to use (100 is full on, 0 is off)."
ControlEngineIon.POWER_EFFICIENCY_CONSTANT_TERM.__doc__ = "PowerEfficiencyModel.C0 - the constant coefficient."
ControlEngineIon.POWER_EFFICIENCY_LINEAR_TERM.__doc__ = "PowerEfficiencyModel.C1 - the linear coefficient."
ControlEngineIon.POWER_EFFICIENCY_QUADRATIC_TERM.__doc__ = "PowerEfficiencyModel.C2 - the quadratic coefficient."
ControlEngineIon.POWER_EFFICIENCY_CUBIC_TERM.__doc__ = "PowerEfficiencyModel.C3 - the cubic coefficient."
ControlEngineIon.REFERENCE_EPOCH.__doc__ = "The date and time used as a reference epoch for degradation."

agcls.AgTypeNameMap["ControlEngineIon"] = ControlEngineIon

class ControlEngineModelPolynomial(IntEnum):
    """Polynomial Thrust and Isp engine model properties that can be selected as control parameters for a Target Sequence."""

    THRUST_C0 = 0
    """The thrust C0 coefficient."""
    THRUST_C1 = 1
    """The thrust C1 coefficient."""
    THRUST_C2 = 2
    """The thrust C2 coefficient."""
    THRUST_C3 = 3
    """The thrust C3 coefficient."""
    THRUST_C4 = 4
    """The thrust C4 coefficient."""
    THRUST_C5 = 5
    """The thrust C5 coefficient."""
    THRUST_C6 = 6
    """The thrust C6 coefficient."""
    THRUST_C7 = 7
    """The thrust C7 coefficient."""
    THRUST_B7 = 8
    """The thrust B7 coefficient."""
    THRUST_E4 = 9
    """The thrust E4 coefficient."""
    THRUST_E5 = 10
    """The thrust E5 coefficient."""
    THRUST_E6 = 11
    """The thrust E6 coefficient."""
    THRUST_E7 = 12
    """The thrust E7 coefficient."""
    THRUST_K0 = 13
    """The thrust K0 coefficient."""
    THRUST_K1 = 14
    """The thrust K1 coefficient."""
    THRUST_REFERENCE_TEMPERATURE = 15
    """The thrust reference temperature, Tr."""
    ISP_C0 = 16
    """The specific impulse C0 coefficient."""
    ISP_C1 = 17
    """The specific impulse C1 coefficient."""
    ISP_C2 = 18
    """The specific impulse C2 coefficient."""
    ISP_C3 = 19
    """The specific impulse C3 coefficient."""
    ISP_C4 = 20
    """The specific impulse C4 coefficient."""
    ISP_C5 = 21
    """The specific impulse C5 coefficient."""
    ISP_C6 = 22
    """The specific impulse C6 coefficient."""
    ISP_C7 = 23
    """The specific impulse C7 coefficient."""
    ISP_B7 = 24
    """The specific impulse B7 coefficient."""
    ISP_E4 = 25
    """The specific impulse E4 coefficient."""
    ISP_E5 = 26
    """The specific impulse E5 coefficient."""
    ISP_E6 = 27
    """The specific impulse E6 coefficient."""
    ISP_E7 = 28
    """The specific impulse E7 coefficient."""
    ISP_K0 = 29
    """The specific impulse K0 coefficient."""
    ISP_K1 = 30
    """The specific impulse K1 coefficient."""
    ISP_REFERENCE_TEMP = 31
    """The specific impulse reference temperature, Tr."""
    GRAV = 32
    """The gravitational acceleration constant at sea level on the Earth."""

ControlEngineModelPolynomial.THRUST_C0.__doc__ = "The thrust C0 coefficient."
ControlEngineModelPolynomial.THRUST_C1.__doc__ = "The thrust C1 coefficient."
ControlEngineModelPolynomial.THRUST_C2.__doc__ = "The thrust C2 coefficient."
ControlEngineModelPolynomial.THRUST_C3.__doc__ = "The thrust C3 coefficient."
ControlEngineModelPolynomial.THRUST_C4.__doc__ = "The thrust C4 coefficient."
ControlEngineModelPolynomial.THRUST_C5.__doc__ = "The thrust C5 coefficient."
ControlEngineModelPolynomial.THRUST_C6.__doc__ = "The thrust C6 coefficient."
ControlEngineModelPolynomial.THRUST_C7.__doc__ = "The thrust C7 coefficient."
ControlEngineModelPolynomial.THRUST_B7.__doc__ = "The thrust B7 coefficient."
ControlEngineModelPolynomial.THRUST_E4.__doc__ = "The thrust E4 coefficient."
ControlEngineModelPolynomial.THRUST_E5.__doc__ = "The thrust E5 coefficient."
ControlEngineModelPolynomial.THRUST_E6.__doc__ = "The thrust E6 coefficient."
ControlEngineModelPolynomial.THRUST_E7.__doc__ = "The thrust E7 coefficient."
ControlEngineModelPolynomial.THRUST_K0.__doc__ = "The thrust K0 coefficient."
ControlEngineModelPolynomial.THRUST_K1.__doc__ = "The thrust K1 coefficient."
ControlEngineModelPolynomial.THRUST_REFERENCE_TEMPERATURE.__doc__ = "The thrust reference temperature, Tr."
ControlEngineModelPolynomial.ISP_C0.__doc__ = "The specific impulse C0 coefficient."
ControlEngineModelPolynomial.ISP_C1.__doc__ = "The specific impulse C1 coefficient."
ControlEngineModelPolynomial.ISP_C2.__doc__ = "The specific impulse C2 coefficient."
ControlEngineModelPolynomial.ISP_C3.__doc__ = "The specific impulse C3 coefficient."
ControlEngineModelPolynomial.ISP_C4.__doc__ = "The specific impulse C4 coefficient."
ControlEngineModelPolynomial.ISP_C5.__doc__ = "The specific impulse C5 coefficient."
ControlEngineModelPolynomial.ISP_C6.__doc__ = "The specific impulse C6 coefficient."
ControlEngineModelPolynomial.ISP_C7.__doc__ = "The specific impulse C7 coefficient."
ControlEngineModelPolynomial.ISP_B7.__doc__ = "The specific impulse B7 coefficient."
ControlEngineModelPolynomial.ISP_E4.__doc__ = "The specific impulse E4 coefficient."
ControlEngineModelPolynomial.ISP_E5.__doc__ = "The specific impulse E5 coefficient."
ControlEngineModelPolynomial.ISP_E6.__doc__ = "The specific impulse E6 coefficient."
ControlEngineModelPolynomial.ISP_E7.__doc__ = "The specific impulse E7 coefficient."
ControlEngineModelPolynomial.ISP_K0.__doc__ = "The specific impulse K0 coefficient."
ControlEngineModelPolynomial.ISP_K1.__doc__ = "The specific impulse K1 coefficient."
ControlEngineModelPolynomial.ISP_REFERENCE_TEMP.__doc__ = "The specific impulse reference temperature, Tr."
ControlEngineModelPolynomial.GRAV.__doc__ = "The gravitational acceleration constant at sea level on the Earth."

agcls.AgTypeNameMap["ControlEngineModelPolynomial"] = ControlEngineModelPolynomial

class EngineModelFunction(IntEnum):
    """The engine model function types."""

    ISP = 0
    """Isp as an independent variable of an equation."""
    POWER = 1
    """Power as an independent variable of an equation."""
    ISP_AND_POWER = 2
    """Isp and Power as independent variables of an equation."""

EngineModelFunction.ISP.__doc__ = "Isp as an independent variable of an equation."
EngineModelFunction.POWER.__doc__ = "Power as an independent variable of an equation."
EngineModelFunction.ISP_AND_POWER.__doc__ = "Isp and Power as independent variables of an equation."

agcls.AgTypeNameMap["EngineModelFunction"] = EngineModelFunction

class ThrottleTableOperationMode(IntEnum):
    """Engine operation mode."""

    ENGINE_OPERATION_REG_POLY = 0
    """Interpolation of engine performance data based on a regression polynomial model."""
    ENGINE_OPERATION_PIECEWISE_LINEAR = 1
    """Interpolation of engine performance data based on a piecewise linear model."""
    ENGINE_OPERATION_DISCRETE = 2
    """Discrete engine operation: piecewise constant engine performance as a function of available power."""

ThrottleTableOperationMode.ENGINE_OPERATION_REG_POLY.__doc__ = "Interpolation of engine performance data based on a regression polynomial model."
ThrottleTableOperationMode.ENGINE_OPERATION_PIECEWISE_LINEAR.__doc__ = "Interpolation of engine performance data based on a piecewise linear model."
ThrottleTableOperationMode.ENGINE_OPERATION_DISCRETE.__doc__ = "Discrete engine operation: piecewise constant engine performance as a function of available power."

agcls.AgTypeNameMap["ThrottleTableOperationMode"] = ThrottleTableOperationMode

class IdealOrbitRadius(IntEnum):
    """Ideal Orbit Radius."""

    EPOCH_CENTERED_AVG_SOURCE_RADIUS = 0
    """Epoch centered average source radius."""
    INSTANTANEOUS_CHARACTERISTIC_DISTANCE = 1
    """Instantanous characteristic distance."""

IdealOrbitRadius.EPOCH_CENTERED_AVG_SOURCE_RADIUS.__doc__ = "Epoch centered average source radius."
IdealOrbitRadius.INSTANTANEOUS_CHARACTERISTIC_DISTANCE.__doc__ = "Instantanous characteristic distance."

agcls.AgTypeNameMap["IdealOrbitRadius"] = IdealOrbitRadius

class RotatingCoordinateSystem(IntEnum):
    """Barycenter centered rotating system."""

    BARYCENTER_CENTERED = 0
    """Barycenter centered rotating system."""
    PRIMARY_CENTERED = 1
    """Primary centered rotating system."""
    SECONDARY_CENTERED = 2
    """Secondary centered rotating system."""
    L1_CENTERED = 3
    """L1 centered rotating system."""
    L2_CENTERED = 4
    """L2 centered rotating system."""
    L3_CENTERED = 5
    """L3 centered rotating system."""
    L4_CENTERED = 6
    """L4 centered rotating system."""
    L5_CENTERED = 7
    """L5 centered rotating system."""

RotatingCoordinateSystem.BARYCENTER_CENTERED.__doc__ = "Barycenter centered rotating system."
RotatingCoordinateSystem.PRIMARY_CENTERED.__doc__ = "Primary centered rotating system."
RotatingCoordinateSystem.SECONDARY_CENTERED.__doc__ = "Secondary centered rotating system."
RotatingCoordinateSystem.L1_CENTERED.__doc__ = "L1 centered rotating system."
RotatingCoordinateSystem.L2_CENTERED.__doc__ = "L2 centered rotating system."
RotatingCoordinateSystem.L3_CENTERED.__doc__ = "L3 centered rotating system."
RotatingCoordinateSystem.L4_CENTERED.__doc__ = "L4 centered rotating system."
RotatingCoordinateSystem.L5_CENTERED.__doc__ = "L5 centered rotating system."

agcls.AgTypeNameMap["RotatingCoordinateSystem"] = RotatingCoordinateSystem

class ControlThrusters(IntEnum):
    """Thruster properties that can be selected as control parameters for a Target Sequence."""

    EQUIVALENT_ON_TIME_PERCENT = 0
    """The equivalent on-time percentage is a factor multiplied by the thrust. The thrust is applied continuously throughout the maneuver and is reduced by the percentage. The mass flow rate is likewise reduced."""
    THRUST_EFFICIENCY = 1
    """The thruster efficiency."""
    SPHERICAL_AZIMUTH = 2
    """Thruster direction defined as a vector in the body frame. Spherical azimuth value."""
    SPHERICAL_ELEVATION = 3
    """Thruster direction defined as a vector in the body frame. Spherical elevation value."""
    CARTESIAN_X = 4
    """Thruster direction defined as a vector in the body frame. Cartesian X value."""
    CARTESIAN_Y = 5
    """Thruster direction defined as a vector in the body frame. Cartesian Y value."""
    CARTESIAN_Z = 6
    """Thruster direction defined as a vector in the body frame. Cartesian Z value."""

ControlThrusters.EQUIVALENT_ON_TIME_PERCENT.__doc__ = "The equivalent on-time percentage is a factor multiplied by the thrust. The thrust is applied continuously throughout the maneuver and is reduced by the percentage. The mass flow rate is likewise reduced."
ControlThrusters.THRUST_EFFICIENCY.__doc__ = "The thruster efficiency."
ControlThrusters.SPHERICAL_AZIMUTH.__doc__ = "Thruster direction defined as a vector in the body frame. Spherical azimuth value."
ControlThrusters.SPHERICAL_ELEVATION.__doc__ = "Thruster direction defined as a vector in the body frame. Spherical elevation value."
ControlThrusters.CARTESIAN_X.__doc__ = "Thruster direction defined as a vector in the body frame. Cartesian X value."
ControlThrusters.CARTESIAN_Y.__doc__ = "Thruster direction defined as a vector in the body frame. Cartesian Y value."
ControlThrusters.CARTESIAN_Z.__doc__ = "Thruster direction defined as a vector in the body frame. Cartesian Z value."

agcls.AgTypeNameMap["ControlThrusters"] = ControlThrusters

class ThrusterDirection(IntEnum):
    """The thruster direction type."""

    ACCELERATION = 0
    """The direction that is opposite the direction of the exhaust."""
    EXHAUST = 1
    """The direction of engine exhaust."""

ThrusterDirection.ACCELERATION.__doc__ = "The direction that is opposite the direction of the exhaust."
ThrusterDirection.EXHAUST.__doc__ = "The direction of engine exhaust."

agcls.AgTypeNameMap["ThrusterDirection"] = ThrusterDirection

class Criteria(IntEnum):
    """The criteria type."""

    EQUALS = 0
    """The test parameter must be equal (within the specified tolerance) to the specified value."""
    GREATER_THAN = 1
    """The test parameter must be greater than the specified value."""
    GREATER_THAN_MINIMUM = 2
    """The current value for the calculation object is greater by the specified tolerance than the minimum reached by that object during the segment."""
    LESS_THAN = 3
    """The test parameter must be less than the specified value."""
    LESS_THAN_MAXIMUM = 4
    """The current value for the calculation object is less by the specified tolerance than the maximum reached by that object during the segment."""
    NOT_EQUAL_TO = 5
    """The test parameter must be not equal (within the specified tolerance) to the specified value."""

Criteria.EQUALS.__doc__ = "The test parameter must be equal (within the specified tolerance) to the specified value."
Criteria.GREATER_THAN.__doc__ = "The test parameter must be greater than the specified value."
Criteria.GREATER_THAN_MINIMUM.__doc__ = "The current value for the calculation object is greater by the specified tolerance than the minimum reached by that object during the segment."
Criteria.LESS_THAN.__doc__ = "The test parameter must be less than the specified value."
Criteria.LESS_THAN_MAXIMUM.__doc__ = "The current value for the calculation object is less by the specified tolerance than the maximum reached by that object during the segment."
Criteria.NOT_EQUAL_TO.__doc__ = "The test parameter must be not equal (within the specified tolerance) to the specified value."

agcls.AgTypeNameMap["Criteria"] = Criteria

class ErrorControl(IntEnum):
    """Error Control for the numerical integrator."""

    ABSOLUTE = 0
    """The error estimate of each integrated component is compared to the absolute error tolerance. Error control with this method is based on absolute values, e.g. in meters for position, m/sec for velocity, etc."""
    RELATIVE_BY_COMPONENT = 1
    """The error estimate of each element being integrated relative to the element's value at the start of the step is compared to the relative error tolerance, and the absolute error estimate of each element is compared to the absolute error tolerance."""
    RELATIVE_TO_STATE = 2
    """Relative to State error control."""
    RELATIVE_TO_STEP = 3
    """Error estimate of each integrated component relative to the element's change in value over the step is compared to the relative error tolerance, and the absolute error estimate of each integrated component is compared to the absolute error tolerance."""

ErrorControl.ABSOLUTE.__doc__ = "The error estimate of each integrated component is compared to the absolute error tolerance. Error control with this method is based on absolute values, e.g. in meters for position, m/sec for velocity, etc."
ErrorControl.RELATIVE_BY_COMPONENT.__doc__ = "The error estimate of each element being integrated relative to the element's value at the start of the step is compared to the relative error tolerance, and the absolute error estimate of each element is compared to the absolute error tolerance."
ErrorControl.RELATIVE_TO_STATE.__doc__ = "Relative to State error control."
ErrorControl.RELATIVE_TO_STEP.__doc__ = "Error estimate of each integrated component relative to the element's change in value over the step is compared to the relative error tolerance, and the absolute error estimate of each integrated component is compared to the absolute error tolerance."

agcls.AgTypeNameMap["ErrorControl"] = ErrorControl

class PredictorCorrector(IntEnum):
    """Predictor corrector scheme for the numerical integrator."""

    FULL = 0
    """All force models are re-evaluated at each corrector step."""
    PSEUDO = 1
    """Only the two-body acceleration is re-evaluated at each corrector step."""

PredictorCorrector.FULL.__doc__ = "All force models are re-evaluated at each corrector step."
PredictorCorrector.PSEUDO.__doc__ = "Only the two-body acceleration is re-evaluated at each corrector step."

agcls.AgTypeNameMap["PredictorCorrector"] = PredictorCorrector

class NumericalIntegrator(IntEnum):
    """Numerical integrator type."""

    RUNGE_KUTTA_4TH_ADAPT = 0
    """A 4th order Runge-Kutta integrator, adapting step size by comparing one full step to two half steps. Although this technique can be quite slow compared to the other algorithms, it is very common and can be used for comparison."""
    RUNGE_KUTTA_FEHLBERG_7TH_8TH = 1
    """A 7th order Runge-Kutta-Fehlberg integrator with 8th order error control. This is the default integrator used in the HPOP propagator."""
    RUNGE_KUTTA_VERNER_8TH_9TH = 2
    """A 9th order Runge-Kutta-Verner integrator with 8th order error control."""
    BULIRSCH_STOER = 3
    """An integrator based on Richardson extrapolation with automatic step size control."""
    GAUSS_JACKSON = 4
    """A 12th order Gauss-Jackson integrator for second order ODEs. There is currently no error control implemented for this method, meaning that a fixed step size is used."""
    RUNGE_KUTTA_2ND_3RD = 5
    """A 2nd order Runge-Kutta integrator with 3rd order error control, using Bogacki and Shampine coefficients."""
    RUNGE_KUTTA_4TH_5TH = 6
    """A 4th order Runge-Kutta integrator with 5th order error control, using Cash-Karp coefficients."""
    RUNGE_KUTTA_4TH = 7
    """A 4th order Runge-Kutta integrator that does not employ error control."""

NumericalIntegrator.RUNGE_KUTTA_4TH_ADAPT.__doc__ = "A 4th order Runge-Kutta integrator, adapting step size by comparing one full step to two half steps. Although this technique can be quite slow compared to the other algorithms, it is very common and can be used for comparison."
NumericalIntegrator.RUNGE_KUTTA_FEHLBERG_7TH_8TH.__doc__ = "A 7th order Runge-Kutta-Fehlberg integrator with 8th order error control. This is the default integrator used in the HPOP propagator."
NumericalIntegrator.RUNGE_KUTTA_VERNER_8TH_9TH.__doc__ = "A 9th order Runge-Kutta-Verner integrator with 8th order error control."
NumericalIntegrator.BULIRSCH_STOER.__doc__ = "An integrator based on Richardson extrapolation with automatic step size control."
NumericalIntegrator.GAUSS_JACKSON.__doc__ = "A 12th order Gauss-Jackson integrator for second order ODEs. There is currently no error control implemented for this method, meaning that a fixed step size is used."
NumericalIntegrator.RUNGE_KUTTA_2ND_3RD.__doc__ = "A 2nd order Runge-Kutta integrator with 3rd order error control, using Bogacki and Shampine coefficients."
NumericalIntegrator.RUNGE_KUTTA_4TH_5TH.__doc__ = "A 4th order Runge-Kutta integrator with 5th order error control, using Cash-Karp coefficients."
NumericalIntegrator.RUNGE_KUTTA_4TH.__doc__ = "A 4th order Runge-Kutta integrator that does not employ error control."

agcls.AgTypeNameMap["NumericalIntegrator"] = NumericalIntegrator

class CoeffRungeKuttaV8th9th(IntEnum):
    """Coefficient sets for RKV8th(9th) integrator."""

    COEFFICIENT_1978 = 0
    """SIAM 1978."""
    EFFICIENT = 1
    """Efficient."""

CoeffRungeKuttaV8th9th.COEFFICIENT_1978.__doc__ = "SIAM 1978."
CoeffRungeKuttaV8th9th.EFFICIENT.__doc__ = "Efficient."

agcls.AgTypeNameMap["CoeffRungeKuttaV8th9th"] = CoeffRungeKuttaV8th9th


class IStoppingConditionComponent(object):
    """Properties for a stopping condition."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_stopping_condition_type_method_offset = 1
    _metadata = {
        "iid_data" : (5269503036530625707, 17454438145196234173),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IStoppingConditionComponent."""
        initialize_from_source_object(self, source_object, IStoppingConditionComponent)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IStoppingConditionComponent)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IStoppingConditionComponent, None)

    _get_stopping_condition_type_metadata = { "offset" : _get_stopping_condition_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StoppingConditionType),) }
    @property
    def stopping_condition_type(self) -> "StoppingConditionType":
        """Get the type of stopping condition."""
        return self._intf.get_property(IStoppingConditionComponent._metadata, IStoppingConditionComponent._get_stopping_condition_type_metadata)

    _property_names[stopping_condition_type] = "stopping_condition_type"


agcls.AgClassCatalog.add_catalog_entry((5269503036530625707, 17454438145196234173), IStoppingConditionComponent)
agcls.AgTypeNameMap["IStoppingConditionComponent"] = IStoppingConditionComponent

class IAttitudeControl(object):
    """Properties for attitude options for a maneuver segment."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_lead_duration_method_offset = 1
    _set_lead_duration_method_offset = 2
    _get_trail_duration_method_offset = 3
    _set_trail_duration_method_offset = 4
    _get_body_axis_method_offset = 5
    _set_body_axis_method_offset = 6
    _get_constraint_sign_method_offset = 7
    _set_constraint_sign_method_offset = 8
    _get_constraint_vector_name_method_offset = 9
    _set_constraint_vector_name_method_offset = 10
    _get_custom_function_method_offset = 11
    _set_custom_function_method_offset = 12
    _metadata = {
        "iid_data" : (4987937969676742434, 14787099081003210654),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAttitudeControl."""
        initialize_from_source_object(self, source_object, IAttitudeControl)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAttitudeControl)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAttitudeControl, None)

    _get_lead_duration_metadata = { "offset" : _get_lead_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def lead_duration(self) -> float:
        """How long before the maneuver starts the maneuver attitude. Uses Time Dimension."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_lead_duration_metadata)

    _set_lead_duration_metadata = { "offset" : _set_lead_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @lead_duration.setter
    def lead_duration(self, value:float) -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_lead_duration_metadata, value)

    _get_trail_duration_metadata = { "offset" : _get_trail_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def trail_duration(self) -> float:
        """How long to maintain that attitude after the maneuver. Uses Time Dimension."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_trail_duration_metadata)

    _set_trail_duration_metadata = { "offset" : _set_trail_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @trail_duration.setter
    def trail_duration(self, value:float) -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_trail_duration_metadata, value)

    _get_body_axis_metadata = { "offset" : _get_body_axis_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BodyAxis),) }
    @property
    def body_axis(self) -> "BodyAxis":
        """Whether the engine acceleration (the direction opposite the engine's exhaust) is aligned with positive or negative X, Y or Z body axis."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_body_axis_metadata)

    _set_body_axis_metadata = { "offset" : _set_body_axis_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BodyAxis),) }
    @body_axis.setter
    def body_axis(self, value:"BodyAxis") -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_body_axis_metadata, value)

    _get_constraint_sign_metadata = { "offset" : _get_constraint_sign_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConstraintSign),) }
    @property
    def constraint_sign(self) -> "ConstraintSign":
        """Whether the Constraint Vector is positive or negative."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_constraint_sign_metadata)

    _set_constraint_sign_metadata = { "offset" : _set_constraint_sign_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ConstraintSign),) }
    @constraint_sign.setter
    def constraint_sign(self, value:"ConstraintSign") -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_constraint_sign_metadata, value)

    _get_constraint_vector_name_metadata = { "offset" : _get_constraint_vector_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def constraint_vector_name(self) -> str:
        """Constraint Vector - the vector toward which this body vector is constrained."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_constraint_vector_name_metadata)

    _set_constraint_vector_name_metadata = { "offset" : _set_constraint_vector_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @constraint_vector_name.setter
    def constraint_vector_name(self, value:str) -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_constraint_vector_name_metadata, value)

    _get_custom_function_metadata = { "offset" : _get_custom_function_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(CustomFunction),) }
    @property
    def custom_function(self) -> "CustomFunction":
        """Get or set the attitude definition to use for other STK functions."""
        return self._intf.get_property(IAttitudeControl._metadata, IAttitudeControl._get_custom_function_metadata)

    _set_custom_function_metadata = { "offset" : _set_custom_function_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(CustomFunction),) }
    @custom_function.setter
    def custom_function(self, value:"CustomFunction") -> None:
        return self._intf.set_property(IAttitudeControl._metadata, IAttitudeControl._set_custom_function_metadata, value)

    _property_names[lead_duration] = "lead_duration"
    _property_names[trail_duration] = "trail_duration"
    _property_names[body_axis] = "body_axis"
    _property_names[constraint_sign] = "constraint_sign"
    _property_names[constraint_vector_name] = "constraint_vector_name"
    _property_names[custom_function] = "custom_function"


agcls.AgClassCatalog.add_catalog_entry((4987937969676742434, 14787099081003210654), IAttitudeControl)
agcls.AgTypeNameMap["IAttitudeControl"] = IAttitudeControl

class IAttitudeControlFinite(IAttitudeControl):
    """The attitude control of a finite maneuver."""

    _num_methods = 0
    _vtable_offset = IAttitudeControl._vtable_offset + IAttitudeControl._num_methods
    _metadata = {
        "iid_data" : (5056364145348131654, 13409090031304170403),
        "vtable_reference" : IAttitudeControl._vtable_offset + IAttitudeControl._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAttitudeControlFinite."""
        initialize_from_source_object(self, source_object, IAttitudeControlFinite)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControl._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAttitudeControlFinite)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAttitudeControlFinite, IAttitudeControl)



agcls.AgClassCatalog.add_catalog_entry((5056364145348131654, 13409090031304170403), IAttitudeControlFinite)
agcls.AgTypeNameMap["IAttitudeControlFinite"] = IAttitudeControlFinite

class IAttitudeControlImpulsive(IAttitudeControl):
    """The attitude control of an impulsive maneuver."""

    _num_methods = 0
    _vtable_offset = IAttitudeControl._vtable_offset + IAttitudeControl._num_methods
    _metadata = {
        "iid_data" : (4710479165606193296, 7418419355563075998),
        "vtable_reference" : IAttitudeControl._vtable_offset + IAttitudeControl._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAttitudeControlImpulsive."""
        initialize_from_source_object(self, source_object, IAttitudeControlImpulsive)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControl._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAttitudeControlImpulsive)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAttitudeControlImpulsive, IAttitudeControl)



agcls.AgClassCatalog.add_catalog_entry((4710479165606193296, 7418419355563075998), IAttitudeControlImpulsive)
agcls.AgTypeNameMap["IAttitudeControlImpulsive"] = IAttitudeControlImpulsive

class IAttitudeControlOptimalFinite(IAttitudeControl):
    """The attitude control of a optimal finite maneuver."""

    _num_methods = 0
    _vtable_offset = IAttitudeControl._vtable_offset + IAttitudeControl._num_methods
    _metadata = {
        "iid_data" : (5442963331835446219, 1789868880099233963),
        "vtable_reference" : IAttitudeControl._vtable_offset + IAttitudeControl._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAttitudeControlOptimalFinite."""
        initialize_from_source_object(self, source_object, IAttitudeControlOptimalFinite)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControl._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAttitudeControlOptimalFinite)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAttitudeControlOptimalFinite, IAttitudeControl)



agcls.AgClassCatalog.add_catalog_entry((5442963331835446219, 1789868880099233963), IAttitudeControlOptimalFinite)
agcls.AgTypeNameMap["IAttitudeControlOptimalFinite"] = IAttitudeControlOptimalFinite

class IManeuver(object):
    """Properties of an Impulsive Maneuver Segment."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_attitude_control_type_method_offset = 1
    _set_attitude_control_type_method_offset = 2
    _get_attitude_control_method_offset = 3
    _get_propulsion_method_method_offset = 4
    _set_propulsion_method_method_offset = 5
    _get_propulsion_method_value_method_offset = 6
    _metadata = {
        "iid_data" : (4677218856107458942, 12562241327114738828),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IManeuver."""
        initialize_from_source_object(self, source_object, IManeuver)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IManeuver)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IManeuver, None)

    _get_attitude_control_type_metadata = { "offset" : _get_attitude_control_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AttitudeControl),) }
    @property
    def attitude_control_type(self) -> "AttitudeControl":
        """Determine the attitude parameters available for you to specify."""
        return self._intf.get_property(IManeuver._metadata, IManeuver._get_attitude_control_type_metadata)

    _set_attitude_control_type_metadata = { "offset" : _set_attitude_control_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AttitudeControl),) }
    def set_attitude_control_type(self, attitude_control_type:"AttitudeControl") -> None:
        """Set the attitude control type."""
        return self._intf.invoke(IManeuver._metadata, IManeuver._set_attitude_control_type_metadata, attitude_control_type)

    _get_attitude_control_metadata = { "offset" : _get_attitude_control_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def attitude_control(self) -> "IAttitudeControl":
        """Get the attitude control properties collection."""
        return self._intf.get_property(IManeuver._metadata, IManeuver._get_attitude_control_metadata)

    _get_propulsion_method_metadata = { "offset" : _get_propulsion_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PropulsionMethod),) }
    @property
    def propulsion_method(self) -> "PropulsionMethod":
        """Get the propulsion type to be modeled."""
        return self._intf.get_property(IManeuver._metadata, IManeuver._get_propulsion_method_metadata)

    _set_propulsion_method_metadata = { "offset" : _set_propulsion_method_method_offset,
            "arg_types" : (agcom.LONG, agcom.BSTR,),
            "marshallers" : (agmarshall.EnumArg(PropulsionMethod), agmarshall.BStrArg,) }
    def set_propulsion_method(self, propulsion_method:"PropulsionMethod", value:str) -> None:
        """Set the propulsion type and associated engine/thruster set."""
        return self._intf.invoke(IManeuver._metadata, IManeuver._set_propulsion_method_metadata, propulsion_method, value)

    _get_propulsion_method_value_metadata = { "offset" : _get_propulsion_method_value_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def propulsion_method_value(self) -> str:
        """Get the specific engine model or thruster set to be used for the maneuver."""
        return self._intf.get_property(IManeuver._metadata, IManeuver._get_propulsion_method_value_metadata)

    _property_names[attitude_control_type] = "attitude_control_type"
    _property_names[attitude_control] = "attitude_control"
    _property_names[propulsion_method] = "propulsion_method"
    _property_names[propulsion_method_value] = "propulsion_method_value"


agcls.AgClassCatalog.add_catalog_entry((4677218856107458942, 12562241327114738828), IManeuver)
agcls.AgTypeNameMap["IManeuver"] = IManeuver

class IDisplaySystem(object):
    """The launch coordinate system."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4713357538953774204, 7691016069329601429),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IDisplaySystem."""
        initialize_from_source_object(self, source_object, IDisplaySystem)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IDisplaySystem)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IDisplaySystem, None)



agcls.AgClassCatalog.add_catalog_entry((4713357538953774204, 7691016069329601429), IDisplaySystem)
agcls.AgTypeNameMap["IDisplaySystem"] = IDisplaySystem

class IBurnout(object):
    """The burnout point reference frame."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4768490459946272698, 6896606911126661053),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IBurnout."""
        initialize_from_source_object(self, source_object, IBurnout)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IBurnout)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IBurnout, None)



agcls.AgClassCatalog.add_catalog_entry((4768490459946272698, 6896606911126661053), IBurnout)
agcls.AgTypeNameMap["IBurnout"] = IBurnout

class IElement(object):
    """The elements of the selected coordinate type."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5488204777371386664, 9539562261869440956),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IElement."""
        initialize_from_source_object(self, source_object, IElement)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IElement)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IElement, None)



agcls.AgClassCatalog.add_catalog_entry((5488204777371386664, 9539562261869440956), IElement)
agcls.AgTypeNameMap["IElement"] = IElement

class IMCSSegment(object):
    """General properties for segments."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_properties_method_offset = 2
    _get_initial_state_method_offset = 3
    _get_final_state_method_offset = 4
    _get_result_value_method_offset = 5
    _run_method_offset = 6
    _get_results_method_offset = 7
    _get_segment_summary_method_offset = 8
    _metadata = {
        "iid_data" : (5363594983371740097, 13023539001208839335),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IMCSSegment."""
        initialize_from_source_object(self, source_object, IMCSSegment)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IMCSSegment)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IMCSSegment, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SegmentType),) }
    @property
    def type(self) -> "SegmentType":
        """Return the type of segment."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_type_metadata)

    _get_properties_metadata = { "offset" : _get_properties_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def properties(self) -> "MCSSegmentProperties":
        """Return the properties of the segment."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_properties_metadata)

    _get_initial_state_metadata = { "offset" : _get_initial_state_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def initial_state(self) -> "State":
        """Get the initial state of the segment."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_initial_state_metadata)

    _get_final_state_metadata = { "offset" : _get_final_state_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def final_state(self) -> "State":
        """Get the final state of the segment."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_final_state_metadata)

    _get_result_value_metadata = { "offset" : _get_result_value_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg,) }
    def get_result_value(self, result_name:str) -> typing.Any:
        """Return a result value."""
        return self._intf.invoke(IMCSSegment._metadata, IMCSSegment._get_result_value_metadata, result_name, OutArg())

    _run_metadata = { "offset" : _run_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def run(self) -> "State":
        """Run the segment in individual segment mode.  See MCSDriver."""
        return self._intf.invoke(IMCSSegment._metadata, IMCSSegment._run_metadata, OutArg())

    _get_results_metadata = { "offset" : _get_results_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def results(self) -> "CalculationObjectCollection":
        """Get the results of the segment."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_results_metadata)

    _get_segment_summary_metadata = { "offset" : _get_segment_summary_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def segment_summary(self) -> "IDataProviderResult":
        """Get the segment summary report."""
        return self._intf.get_property(IMCSSegment._metadata, IMCSSegment._get_segment_summary_metadata)

    _property_names[type] = "type"
    _property_names[properties] = "properties"
    _property_names[initial_state] = "initial_state"
    _property_names[final_state] = "final_state"
    _property_names[results] = "results"
    _property_names[segment_summary] = "segment_summary"


agcls.AgClassCatalog.add_catalog_entry((5363594983371740097, 13023539001208839335), IMCSSegment)
agcls.AgTypeNameMap["IMCSSegment"] = IMCSSegment

class IMCSSequence(object):
    """Properties for a Sequence segment."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_repeat_count_method_offset = 1
    _set_repeat_count_method_offset = 2
    _get_generate_ephemeris_method_offset = 3
    _set_generate_ephemeris_method_offset = 4
    _get_segments_method_offset = 5
    _get_sequence_state_to_pass_method_offset = 6
    _set_sequence_state_to_pass_method_offset = 7
    _get_scripting_tool_method_offset = 8
    _apply_script_method_offset = 9
    _metadata = {
        "iid_data" : (5582283013374644007, 10842020077807962011),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IMCSSequence."""
        initialize_from_source_object(self, source_object, IMCSSequence)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IMCSSequence)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IMCSSequence, None)

    _get_repeat_count_metadata = { "offset" : _get_repeat_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def repeat_count(self) -> int:
        """Get or set the number of times that the sequence will be executed. A sequence that is repeated is executed immediately subsequent to the previous execution of the sequence. Dimensionless."""
        return self._intf.get_property(IMCSSequence._metadata, IMCSSequence._get_repeat_count_metadata)

    _set_repeat_count_metadata = { "offset" : _set_repeat_count_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @repeat_count.setter
    def repeat_count(self, value:int) -> None:
        return self._intf.set_property(IMCSSequence._metadata, IMCSSequence._set_repeat_count_metadata, value)

    _get_generate_ephemeris_metadata = { "offset" : _get_generate_ephemeris_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def generate_ephemeris(self) -> bool:
        """If true, the sequence generates ephemeris and displays it in the 2D and 3D Graphics windows."""
        return self._intf.get_property(IMCSSequence._metadata, IMCSSequence._get_generate_ephemeris_metadata)

    _set_generate_ephemeris_metadata = { "offset" : _set_generate_ephemeris_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @generate_ephemeris.setter
    def generate_ephemeris(self, value:bool) -> None:
        return self._intf.set_property(IMCSSequence._metadata, IMCSSequence._set_generate_ephemeris_metadata, value)

    _get_segments_metadata = { "offset" : _get_segments_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def segments(self) -> "MCSSegmentCollection":
        """Get the list of segments defined for the sequence."""
        return self._intf.get_property(IMCSSequence._metadata, IMCSSequence._get_segments_metadata)

    _get_sequence_state_to_pass_metadata = { "offset" : _get_sequence_state_to_pass_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SequenceStateToPass),) }
    @property
    def sequence_state_to_pass(self) -> "SequenceStateToPass":
        """State To Pass To Next Segment - the state of the sequence to pass."""
        return self._intf.get_property(IMCSSequence._metadata, IMCSSequence._get_sequence_state_to_pass_metadata)

    _set_sequence_state_to_pass_metadata = { "offset" : _set_sequence_state_to_pass_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SequenceStateToPass),) }
    @sequence_state_to_pass.setter
    def sequence_state_to_pass(self, value:"SequenceStateToPass") -> None:
        return self._intf.set_property(IMCSSequence._metadata, IMCSSequence._set_sequence_state_to_pass_metadata, value)

    _get_scripting_tool_metadata = { "offset" : _get_scripting_tool_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scripting_tool(self) -> "ScriptingTool":
        """Return the Scripting tool for the sequence."""
        return self._intf.get_property(IMCSSequence._metadata, IMCSSequence._get_scripting_tool_metadata)

    _apply_script_metadata = { "offset" : _apply_script_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def apply_script(self) -> None:
        """Apply the script."""
        return self._intf.invoke(IMCSSequence._metadata, IMCSSequence._apply_script_metadata, )

    _property_names[repeat_count] = "repeat_count"
    _property_names[generate_ephemeris] = "generate_ephemeris"
    _property_names[segments] = "segments"
    _property_names[sequence_state_to_pass] = "sequence_state_to_pass"
    _property_names[scripting_tool] = "scripting_tool"


agcls.AgClassCatalog.add_catalog_entry((5582283013374644007, 10842020077807962011), IMCSSequence)
agcls.AgTypeNameMap["IMCSSequence"] = IMCSSequence

class IProfile(object):
    """General properties for target sequence profiles."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _copy_method_offset = 1
    _get_name_method_offset = 2
    _set_name_method_offset = 3
    _get_status_method_offset = 4
    _get_user_comment_method_offset = 5
    _set_user_comment_method_offset = 6
    _get_mode_method_offset = 7
    _set_mode_method_offset = 8
    _get_type_method_offset = 9
    _metadata = {
        "iid_data" : (4686877490284796494, 11687945607961083032),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IProfile."""
        initialize_from_source_object(self, source_object, IProfile)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IProfile)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IProfile, None)

    _copy_metadata = { "offset" : _copy_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def copy(self) -> "IProfile":
        """Make a copy of the profile."""
        return self._intf.invoke(IProfile._metadata, IProfile._copy_metadata, OutArg())

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the profile."""
        return self._intf.get_property(IProfile._metadata, IProfile._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        return self._intf.set_property(IProfile._metadata, IProfile._set_name_metadata, value)

    _get_status_metadata = { "offset" : _get_status_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def status(self) -> str:
        """Get or set the status of the profile."""
        return self._intf.get_property(IProfile._metadata, IProfile._get_status_metadata)

    _get_user_comment_metadata = { "offset" : _get_user_comment_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def user_comment(self) -> str:
        """A user comment."""
        return self._intf.get_property(IProfile._metadata, IProfile._get_user_comment_metadata)

    _set_user_comment_metadata = { "offset" : _set_user_comment_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @user_comment.setter
    def user_comment(self, value:str) -> None:
        return self._intf.set_property(IProfile._metadata, IProfile._set_user_comment_metadata, value)

    _get_mode_metadata = { "offset" : _get_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ProfileMode),) }
    @property
    def mode(self) -> "ProfileMode":
        """Get or set the profile's mode."""
        return self._intf.get_property(IProfile._metadata, IProfile._get_mode_metadata)

    _set_mode_metadata = { "offset" : _set_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ProfileMode),) }
    @mode.setter
    def mode(self, value:"ProfileMode") -> None:
        return self._intf.set_property(IProfile._metadata, IProfile._set_mode_metadata, value)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Profile),) }
    @property
    def type(self) -> "Profile":
        """Return the current profile type."""
        return self._intf.get_property(IProfile._metadata, IProfile._get_type_metadata)

    _property_names[name] = "name"
    _property_names[status] = "status"
    _property_names[user_comment] = "user_comment"
    _property_names[mode] = "mode"
    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((4686877490284796494, 11687945607961083032), IProfile)
agcls.AgTypeNameMap["IProfile"] = IProfile

class ICentralBodyComponentEphemeris(object):
    """The central body ephemeris source."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5592778241070953614, 10499311837320329628),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICentralBodyComponentEphemeris."""
        initialize_from_source_object(self, source_object, ICentralBodyComponentEphemeris)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICentralBodyComponentEphemeris)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICentralBodyComponentEphemeris, None)



agcls.AgClassCatalog.add_catalog_entry((5592778241070953614, 10499311837320329628), ICentralBodyComponentEphemeris)
agcls.AgTypeNameMap["ICentralBodyComponentEphemeris"] = ICentralBodyComponentEphemeris

class ICentralBodyComponentShape(object):
    """The central body shape."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4933547587540680255, 5573209392764395165),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICentralBodyComponentShape."""
        initialize_from_source_object(self, source_object, ICentralBodyComponentShape)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICentralBodyComponentShape)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICentralBodyComponentShape, None)



agcls.AgClassCatalog.add_catalog_entry((4933547587540680255, 5573209392764395165), ICentralBodyComponentShape)
agcls.AgTypeNameMap["ICentralBodyComponentShape"] = ICentralBodyComponentShape

class ICentralBodyComponentAttitude(object):
    """The central body attitude."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5344723028474245754, 5582255056768029873),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICentralBodyComponentAttitude."""
        initialize_from_source_object(self, source_object, ICentralBodyComponentAttitude)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICentralBodyComponentAttitude)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICentralBodyComponentAttitude, None)



agcls.AgClassCatalog.add_catalog_entry((5344723028474245754, 5582255056768029873), ICentralBodyComponentAttitude)
agcls.AgTypeNameMap["ICentralBodyComponentAttitude"] = ICentralBodyComponentAttitude

class ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides(ICentralBodyComponentEphemeris):
    """Properties for the JPL DE ephemeris source."""

    _num_methods = 2
    _vtable_offset = ICentralBodyComponentEphemeris._vtable_offset + ICentralBodyComponentEphemeris._num_methods
    _get_jplde_filename_method_offset = 1
    _set_jplde_filename_method_offset = 2
    _metadata = {
        "iid_data" : (5096958885497432761, 15383033303903279551),
        "vtable_reference" : ICentralBodyComponentEphemeris._vtable_offset + ICentralBodyComponentEphemeris._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides."""
        initialize_from_source_object(self, source_object, ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICentralBodyComponentEphemeris._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides, ICentralBodyComponentEphemeris)

    _get_jplde_filename_metadata = { "offset" : _get_jplde_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def jplde_filename(self) -> str:
        """Get or set the DE file name."""
        return self._intf.get_property(ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides._metadata, ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides._get_jplde_filename_metadata)

    _set_jplde_filename_metadata = { "offset" : _set_jplde_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @jplde_filename.setter
    def jplde_filename(self, value:str) -> None:
        return self._intf.set_property(ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides._metadata, ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides._set_jplde_filename_metadata, value)

    _property_names[jplde_filename] = "jplde_filename"


agcls.AgClassCatalog.add_catalog_entry((5096958885497432761, 15383033303903279551), ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides)
agcls.AgTypeNameMap["ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides"] = ICentralBodyComponentEphemerisJPLDevelopmentalEphemerides

class INumericalIntegrator(object):
    """The type of numerical integrator to be used by the propagator."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4724514531705891614, 6171053309445421234),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type INumericalIntegrator."""
        initialize_from_source_object(self, source_object, INumericalIntegrator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, INumericalIntegrator)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, INumericalIntegrator, None)



agcls.AgClassCatalog.add_catalog_entry((4724514531705891614, 6171053309445421234), INumericalIntegrator)
agcls.AgTypeNameMap["INumericalIntegrator"] = INumericalIntegrator



class MCSDriver(IPropagator, SupportsDeleteCallback):
    """
    Properties for the Mission Control Sequence.

    Examples
    --------
    Set satellite propagator to Astrogator and clear segments:
    >>> # Satellite satellite: Satellite object
    >>> satellite.set_propagator_type(PropagatorType.ASTROGATOR)
    >>> driver = satellite.propagator
    >>> # Clear all segments from the MCS
    >>> driver.main_sequence.remove_all()

    Run the Astrogator® mission control sequence (MCS):
    >>> # MCSDriver driver: MCS driver interface
    >>> driver.run_mcs()
    """

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_main_sequence_method_offset = 1
    _get_options_method_offset = 2
    _get_auto_sequence_method_offset = 3
    _run_mcs_method_offset = 4
    _begin_run_method_offset = 5
    _end_run_method_offset = 6
    _clear_draw_while_calculating_graphics_method_offset = 7
    _reset_all_profiles_method_offset = 8
    _apply_all_profile_changes_method_offset = 9
    _append_run_method_offset = 10
    _append_run_from_time_method_offset = 11
    _append_run_from_state_method_offset = 12
    _run_mcs2_method_offset = 13
    _get_calculation_graphs_method_offset = 14
    _metadata = {
        "iid_data" : (5434593673786300345, 9884445645793902248),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSDriver)

    _get_main_sequence_metadata = { "offset" : _get_main_sequence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def main_sequence(self) -> "MCSSegmentCollection":
        """Get the Mission Control Sequence."""
        return self._intf.get_property(MCSDriver._metadata, MCSDriver._get_main_sequence_metadata)

    _get_options_metadata = { "offset" : _get_options_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def options(self) -> "MCSOptions":
        """Get the Mission Control Sequence options."""
        return self._intf.get_property(MCSDriver._metadata, MCSDriver._get_options_metadata)

    _get_auto_sequence_metadata = { "offset" : _get_auto_sequence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def auto_sequence(self) -> "AutomaticSequenceCollection":
        """Get the Automatic Sequences."""
        return self._intf.get_property(MCSDriver._metadata, MCSDriver._get_auto_sequence_metadata)

    _run_mcs_metadata = { "offset" : _run_mcs_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def run_mcs(self) -> None:
        """Run the current MCS."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._run_mcs_metadata, )

    _begin_run_metadata = { "offset" : _begin_run_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def begin_run(self) -> None:
        """Begins an individual segment mode run."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._begin_run_metadata, )

    _end_run_metadata = { "offset" : _end_run_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def end_run(self) -> None:
        """End an individual segment mode run."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._end_run_metadata, )

    _clear_draw_while_calculating_graphics_metadata = { "offset" : _clear_draw_while_calculating_graphics_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def clear_draw_while_calculating_graphics(self) -> None:
        """Clear the draw while calculating graphics."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._clear_draw_while_calculating_graphics_metadata, )

    _reset_all_profiles_metadata = { "offset" : _reset_all_profiles_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reset_all_profiles(self) -> None:
        """Reset all active profiles in all target sequences."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._reset_all_profiles_metadata, )

    _apply_all_profile_changes_metadata = { "offset" : _apply_all_profile_changes_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def apply_all_profile_changes(self) -> None:
        """Apply all active profile changes in all target sequences."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._apply_all_profile_changes_metadata, )

    _append_run_metadata = { "offset" : _append_run_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def append_run(self) -> None:
        """Append the existing ephemeris with another individual segment mode run."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._append_run_metadata, )

    _append_run_from_time_metadata = { "offset" : _append_run_from_time_method_offset,
            "arg_types" : (agcom.Variant, agcom.LONG,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.EnumArg(ClearEphemerisDirection),) }
    def append_run_from_time(self, epoch:typing.Any, clear_ephemeris_direction:"ClearEphemerisDirection") -> None:
        """Append the existing ephemeris with another individual segment mode run, starting at a specified time. Ephemeris is cleared from time based on clear direction."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._append_run_from_time_metadata, epoch, clear_ephemeris_direction)

    _append_run_from_state_metadata = { "offset" : _append_run_from_state_method_offset,
            "arg_types" : (agcom.PVOID, agcom.LONG,),
            "marshallers" : (agmarshall.InterfaceInArg("State"), agmarshall.EnumArg(ClearEphemerisDirection),) }
    def append_run_from_state(self, append_state:"State", clear_ephemeris_direction:"ClearEphemerisDirection") -> None:
        """Append the existing ephemeris with another individual segment mode run, starting at a specified state. Ephemeris is cleared from time based on clear direction."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._append_run_from_state_metadata, append_state, clear_ephemeris_direction)

    _run_mcs2_metadata = { "offset" : _run_mcs2_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RunCode),) }
    def run_mcs2(self) -> "RunCode":
        """Run the current MCS and returns an error code."""
        return self._intf.invoke(MCSDriver._metadata, MCSDriver._run_mcs2_metadata, OutArg())

    _get_calculation_graphs_metadata = { "offset" : _get_calculation_graphs_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def calculation_graphs(self) -> "CalculationGraphCollection":
        """Get the calculation graphs."""
        return self._intf.get_property(MCSDriver._metadata, MCSDriver._get_calculation_graphs_metadata)

    _property_names[main_sequence] = "main_sequence"
    _property_names[options] = "options"
    _property_names[auto_sequence] = "auto_sequence"
    _property_names[calculation_graphs] = "calculation_graphs"

    def __init__(self, source_object=None):
        """Construct an object of type MCSDriver."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSDriver)
        IPropagator.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IPropagator._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSDriver, [MCSDriver, IPropagator])

agcls.AgClassCatalog.add_catalog_entry((5389292584719371376, 5476973722651125132), MCSDriver)
agcls.AgTypeNameMap["MCSDriver"] = MCSDriver

class MCSSegmentCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a collection of segments."""

    _num_methods = 13
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _insert_method_offset = 2
    _remove_method_offset = 3
    _remove_all_method_offset = 4
    _get__new_enum_method_offset = 5
    _get_count_method_offset = 6
    _cut_method_offset = 7
    _paste_method_offset = 8
    _insert_copy_method_offset = 9
    _insert_by_name_method_offset = 10
    _get_provide_runtime_type_info_method_offset = 11
    _get_item_by_index_method_offset = 12
    _get_item_by_name_method_offset = 13
    _metadata = {
        "iid_data" : (5058380971033415851, 1014008794519657865),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSSegmentCollection)
    def __iter__(self):
        """Create an iterator for the MCSSegmentCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IMCSSegment":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "IMCSSegment":
        """Return the specified segment(using segment name or index number)."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._item_metadata, index_or_name, OutArg())

    _insert_metadata = { "offset" : _insert_method_offset,
            "arg_types" : (agcom.LONG, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(SegmentType), agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def insert(self, segment_type:"SegmentType", segment_name:str, segment_to_insert_before:str) -> "IMCSSegment":
        """Add a segment to the segment collection."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._insert_metadata, segment_type, segment_name, segment_to_insert_before, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, segment_name_to_remove:str) -> None:
        """Remove a segment; the End segment cannot be deleted."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._remove_metadata, segment_name_to_remove)

    _remove_all_metadata = { "offset" : _remove_all_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove_all(self) -> None:
        """Remove all segments; the End segment cannot be deleted."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._remove_all_metadata, )

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(MCSSegmentCollection._metadata, MCSSegmentCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(MCSSegmentCollection._metadata, MCSSegmentCollection._get_count_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def cut(self, segment_name_to_cut:str) -> None:
        """Copy the segment into the clipboard and removes the segment from the sequence."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._cut_metadata, segment_name_to_cut)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def paste(self, segment_to_paste_before:str) -> "IMCSSegment":
        """Pastes the segment from the clipboard and inserts in before the given segment name."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._paste_metadata, segment_to_paste_before, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IMCSSegment"), agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def insert_copy(self, segment:"IMCSSegment", segment_to_insert_before:str) -> "IMCSSegment":
        """Copy the segment pointer and inserts the copy before the given segment name."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._insert_copy_metadata, segment, segment_to_insert_before, OutArg())

    _insert_by_name_metadata = { "offset" : _insert_by_name_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def insert_by_name(self, segment_name:str, segment_to_insert_before:str) -> "IMCSSegment":
        """Insert a segment by name to the segment collection."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._insert_by_name_metadata, segment_name, segment_to_insert_before, OutArg())

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(MCSSegmentCollection._metadata, MCSSegmentCollection._get_provide_runtime_type_info_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "IMCSSegment":
        """Retrieve the specified segment(using segment index number)."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "IMCSSegment":
        """Retrieve the specified segment(using segment name)."""
        return self._intf.invoke(MCSSegmentCollection._metadata, MCSSegmentCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type MCSSegmentCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSSegmentCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSSegmentCollection, [MCSSegmentCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4909520030354968962, 14065348172908224678), MCSSegmentCollection)
agcls.AgTypeNameMap["MCSSegmentCollection"] = MCSSegmentCollection

class MCSEnd(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, SupportsDeleteCallback):
    """Properties for an End segment."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5377430998049428360, 10736208053127098559),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSEnd)


    def __init__(self, source_object=None):
        """Construct an object of type MCSEnd."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSEnd)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSEnd, [MCSEnd, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo])

agcls.AgClassCatalog.add_catalog_entry((4611836544222750290, 3791847305781604787), MCSEnd)
agcls.AgTypeNameMap["MCSEnd"] = MCSEnd

class MCSInitialState(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an Initial State segment."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_coord_system_name_method_offset = 1
    _set_coord_system_name_method_offset = 2
    _get_orbit_epoch_method_offset = 3
    _set_orbit_epoch_method_offset = 4
    _get_spacecraft_parameters_method_offset = 5
    _get_fuel_tank_method_offset = 6
    _get_element_type_method_offset = 7
    _set_element_type_method_offset = 8
    _get_element_method_offset = 9
    _enable_control_parameter_method_offset = 10
    _disable_control_parameter_method_offset = 11
    _is_control_parameter_enabled_method_offset = 12
    _get_control_parameters_available_method_offset = 13
    _get_user_variables_method_offset = 14
    _metadata = {
        "iid_data" : (5028451386241945120, 18244435719331815809),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSInitialState)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get or set the coordinate system."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_coord_system_name_metadata)

    _set_coord_system_name_metadata = { "offset" : _set_coord_system_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system_name.setter
    def coord_system_name(self, value:str) -> None:
        return self._intf.set_property(MCSInitialState._metadata, MCSInitialState._set_coord_system_name_metadata, value)

    _get_orbit_epoch_metadata = { "offset" : _get_orbit_epoch_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def orbit_epoch(self) -> typing.Any:
        """Get or set the orbit epoch. Uses DateFormat Dimension."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_orbit_epoch_metadata)

    _set_orbit_epoch_metadata = { "offset" : _set_orbit_epoch_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @orbit_epoch.setter
    def orbit_epoch(self, value:typing.Any) -> None:
        return self._intf.set_property(MCSInitialState._metadata, MCSInitialState._set_orbit_epoch_metadata, value)

    _get_spacecraft_parameters_metadata = { "offset" : _get_spacecraft_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def spacecraft_parameters(self) -> "SpacecraftParameters":
        """Get the spacecraft  parameters."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_spacecraft_parameters_metadata)

    _get_fuel_tank_metadata = { "offset" : _get_fuel_tank_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def fuel_tank(self) -> "FuelTank":
        """Get the fuel tank parameters."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_fuel_tank_metadata)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementSetType),) }
    @property
    def element_type(self) -> "ElementSetType":
        """Get the coordinate type."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_element_type_metadata)

    _set_element_type_metadata = { "offset" : _set_element_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ElementSetType),) }
    def set_element_type(self, element_type:"ElementSetType") -> None:
        """Select an coordinate type."""
        return self._intf.invoke(MCSInitialState._metadata, MCSInitialState._set_element_type_metadata, element_type)

    _get_element_metadata = { "offset" : _get_element_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def element(self) -> "IElement":
        """Get the elements of the selected coordinate type."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_element_metadata)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlInitState),) }
    def enable_control_parameter(self, param:"ControlInitState") -> None:
        """Enable a control parameter."""
        return self._intf.invoke(MCSInitialState._metadata, MCSInitialState._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlInitState),) }
    def disable_control_parameter(self, param:"ControlInitState") -> None:
        """Disables a control parameter."""
        return self._intf.invoke(MCSInitialState._metadata, MCSInitialState._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlInitState), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlInitState") -> bool:
        """Sees if a control is enabled."""
        return self._intf.invoke(MCSInitialState._metadata, MCSInitialState._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_control_parameters_available_metadata)

    _get_user_variables_metadata = { "offset" : _get_user_variables_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_variables(self) -> "UserVariableCollection":
        """Interface used to modify user variables for the initial state segment."""
        return self._intf.get_property(MCSInitialState._metadata, MCSInitialState._get_user_variables_metadata)

    _property_names[coord_system_name] = "coord_system_name"
    _property_names[orbit_epoch] = "orbit_epoch"
    _property_names[spacecraft_parameters] = "spacecraft_parameters"
    _property_names[fuel_tank] = "fuel_tank"
    _property_names[element_type] = "element_type"
    _property_names[element] = "element"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[user_variables] = "user_variables"

    def __init__(self, source_object=None):
        """Construct an object of type MCSInitialState."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSInitialState)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSInitialState, [MCSInitialState, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5119690725327440606, 12426330438899630259), MCSInitialState)
agcls.AgTypeNameMap["MCSInitialState"] = MCSInitialState

class SpacecraftParameters(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for spacecraft configuration."""

    _num_methods = 18
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_dry_mass_method_offset = 1
    _set_dry_mass_method_offset = 2
    _get_cd_method_offset = 3
    _set_cd_method_offset = 4
    _get_drag_area_method_offset = 5
    _set_drag_area_method_offset = 6
    _get_cr_method_offset = 7
    _set_cr_method_offset = 8
    _get_solar_radiation_pressure_area_method_offset = 9
    _set_solar_radiation_pressure_area_method_offset = 10
    _get_ck_method_offset = 11
    _set_ck_method_offset = 12
    _get_radiation_pressure_area_method_offset = 13
    _set_radiation_pressure_area_method_offset = 14
    _get_k1_method_offset = 15
    _set_k1_method_offset = 16
    _get_k2_method_offset = 17
    _set_k2_method_offset = 18
    _metadata = {
        "iid_data" : (5571064856599243066, 14979986637316024483),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpacecraftParameters)

    _get_dry_mass_metadata = { "offset" : _get_dry_mass_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def dry_mass(self) -> float:
        """Get or set the mass of the spacecraft exclusive of propellant. Uses Mass Dimension."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_dry_mass_metadata)

    _set_dry_mass_metadata = { "offset" : _set_dry_mass_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @dry_mass.setter
    def dry_mass(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_dry_mass_metadata, value)

    _get_cd_metadata = { "offset" : _get_cd_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cd(self) -> float:
        """Get or set the dimensionless drag coefficient associated with the drag area. Dimensionless."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_cd_metadata)

    _set_cd_metadata = { "offset" : _set_cd_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cd.setter
    def cd(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_cd_metadata, value)

    _get_drag_area_metadata = { "offset" : _get_drag_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def drag_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Uses SmallArea Dimension."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_drag_area_metadata)

    _set_drag_area_metadata = { "offset" : _set_drag_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @drag_area.setter
    def drag_area(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_drag_area_metadata, value)

    _get_cr_metadata = { "offset" : _get_cr_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cr(self) -> float:
        """Get or set the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all. Dimensionless."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_cr_metadata)

    _set_cr_metadata = { "offset" : _set_cr_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cr.setter
    def cr(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_cr_metadata, value)

    _get_solar_radiation_pressure_area_metadata = { "offset" : _get_solar_radiation_pressure_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def solar_radiation_pressure_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Uses SmallArea Dimension."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_solar_radiation_pressure_area_metadata)

    _set_solar_radiation_pressure_area_metadata = { "offset" : _set_solar_radiation_pressure_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @solar_radiation_pressure_area.setter
    def solar_radiation_pressure_area(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_solar_radiation_pressure_area_metadata, value)

    _get_ck_metadata = { "offset" : _get_ck_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def ck(self) -> float:
        """Get or set the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all. Dimensionless."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_ck_metadata)

    _set_ck_metadata = { "offset" : _set_ck_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @ck.setter
    def ck(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_ck_metadata, value)

    _get_radiation_pressure_area_metadata = { "offset" : _get_radiation_pressure_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radiation_pressure_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Uses SmallArea Dimension."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_radiation_pressure_area_metadata)

    _set_radiation_pressure_area_metadata = { "offset" : _set_radiation_pressure_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radiation_pressure_area.setter
    def radiation_pressure_area(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_radiation_pressure_area_metadata, value)

    _get_k1_metadata = { "offset" : _get_k1_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def k1(self) -> float:
        """If you are using a non-spherical SRP model, this field defines the model's GPS solar radiation pressure K1 (scale) value. Dimensionless."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_k1_metadata)

    _set_k1_metadata = { "offset" : _set_k1_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @k1.setter
    def k1(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_k1_metadata, value)

    _get_k2_metadata = { "offset" : _get_k2_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def k2(self) -> float:
        """If you are using a non-spherical SRP model, this field defines the model's GPS solar radiation pressure K2 (scale) value. Dimensionless."""
        return self._intf.get_property(SpacecraftParameters._metadata, SpacecraftParameters._get_k2_metadata)

    _set_k2_metadata = { "offset" : _set_k2_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @k2.setter
    def k2(self, value:float) -> None:
        return self._intf.set_property(SpacecraftParameters._metadata, SpacecraftParameters._set_k2_metadata, value)

    _property_names[dry_mass] = "dry_mass"
    _property_names[cd] = "cd"
    _property_names[drag_area] = "drag_area"
    _property_names[cr] = "cr"
    _property_names[solar_radiation_pressure_area] = "solar_radiation_pressure_area"
    _property_names[ck] = "ck"
    _property_names[radiation_pressure_area] = "radiation_pressure_area"
    _property_names[k1] = "k1"
    _property_names[k2] = "k2"

    def __init__(self, source_object=None):
        """Construct an object of type SpacecraftParameters."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpacecraftParameters)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpacecraftParameters, [SpacecraftParameters, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5397758463997765929, 8149019131139753100), SpacecraftParameters)
agcls.AgTypeNameMap["SpacecraftParameters"] = SpacecraftParameters

class FuelTank(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for fuel tank configuration."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_tank_pressure_method_offset = 1
    _set_tank_pressure_method_offset = 2
    _get_tank_volume_method_offset = 3
    _set_tank_volume_method_offset = 4
    _get_tank_temperature_method_offset = 5
    _set_tank_temperature_method_offset = 6
    _get_fuel_density_method_offset = 7
    _set_fuel_density_method_offset = 8
    _get_fuel_mass_method_offset = 9
    _set_fuel_mass_method_offset = 10
    _get_maximum_fuel_mass_method_offset = 11
    _set_maximum_fuel_mass_method_offset = 12
    _metadata = {
        "iid_data" : (5624502096774973488, 5173171893781027749),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, FuelTank)

    _get_tank_pressure_metadata = { "offset" : _get_tank_pressure_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tank_pressure(self) -> float:
        """Get or set the fuel tank pressure. Uses Pressure Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_tank_pressure_metadata)

    _set_tank_pressure_metadata = { "offset" : _set_tank_pressure_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tank_pressure.setter
    def tank_pressure(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_tank_pressure_metadata, value)

    _get_tank_volume_metadata = { "offset" : _get_tank_volume_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tank_volume(self) -> float:
        """Get or set the volume of the fuel tank. Uses SmallVolume Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_tank_volume_metadata)

    _set_tank_volume_metadata = { "offset" : _set_tank_volume_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tank_volume.setter
    def tank_volume(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_tank_volume_metadata, value)

    _get_tank_temperature_metadata = { "offset" : _get_tank_temperature_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tank_temperature(self) -> float:
        """Get or set the temperature of the fuel tank. Uses Temperature Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_tank_temperature_metadata)

    _set_tank_temperature_metadata = { "offset" : _set_tank_temperature_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tank_temperature.setter
    def tank_temperature(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_tank_temperature_metadata, value)

    _get_fuel_density_metadata = { "offset" : _get_fuel_density_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_density(self) -> float:
        """Get or set the density of the fuel. Uses SmallDensity Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_fuel_density_metadata)

    _set_fuel_density_metadata = { "offset" : _set_fuel_density_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_density.setter
    def fuel_density(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_fuel_density_metadata, value)

    _get_fuel_mass_metadata = { "offset" : _get_fuel_mass_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_mass(self) -> float:
        """Get or set the mass of the spacecraft propellant. Uses Mass Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_fuel_mass_metadata)

    _set_fuel_mass_metadata = { "offset" : _set_fuel_mass_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_mass.setter
    def fuel_mass(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_fuel_mass_metadata, value)

    _get_maximum_fuel_mass_metadata = { "offset" : _get_maximum_fuel_mass_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_fuel_mass(self) -> float:
        """Get or set the maximum fuel mass of the spacecraft; this parameter specifically applies to Finite Maneuver segments that are being executed in Backward Sequences. Uses Mass Dimension."""
        return self._intf.get_property(FuelTank._metadata, FuelTank._get_maximum_fuel_mass_metadata)

    _set_maximum_fuel_mass_metadata = { "offset" : _set_maximum_fuel_mass_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_fuel_mass.setter
    def maximum_fuel_mass(self, value:float) -> None:
        return self._intf.set_property(FuelTank._metadata, FuelTank._set_maximum_fuel_mass_metadata, value)

    _property_names[tank_pressure] = "tank_pressure"
    _property_names[tank_volume] = "tank_volume"
    _property_names[tank_temperature] = "tank_temperature"
    _property_names[fuel_density] = "fuel_density"
    _property_names[fuel_mass] = "fuel_mass"
    _property_names[maximum_fuel_mass] = "maximum_fuel_mass"

    def __init__(self, source_object=None):
        """Construct an object of type FuelTank."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, FuelTank)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, FuelTank, [FuelTank, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5728765514206679012, 4259677341011527818), FuelTank)
agcls.AgTypeNameMap["FuelTank"] = FuelTank

class ElementCartesian(IElement, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for Cartesian elements."""

    _num_methods = 12
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_x_method_offset = 1
    _set_x_method_offset = 2
    _get_y_method_offset = 3
    _set_y_method_offset = 4
    _get_z_method_offset = 5
    _set_z_method_offset = 6
    _get_vx_method_offset = 7
    _set_vx_method_offset = 8
    _get_vy_method_offset = 9
    _set_vy_method_offset = 10
    _get_vz_method_offset = 11
    _set_vz_method_offset = 12
    _metadata = {
        "iid_data" : (5300421637043411044, 13353383975691849384),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementCartesian)

    _get_x_metadata = { "offset" : _get_x_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def x(self) -> float:
        """X position component. Uses Distance Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_x_metadata)

    _set_x_metadata = { "offset" : _set_x_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @x.setter
    def x(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_x_metadata, value)

    _get_y_metadata = { "offset" : _get_y_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def y(self) -> float:
        """Y position component. Uses Distance Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_y_metadata)

    _set_y_metadata = { "offset" : _set_y_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @y.setter
    def y(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_y_metadata, value)

    _get_z_metadata = { "offset" : _get_z_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def z(self) -> float:
        """Z position component. Uses Distance Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_z_metadata)

    _set_z_metadata = { "offset" : _set_z_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @z.setter
    def z(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_z_metadata, value)

    _get_vx_metadata = { "offset" : _get_vx_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def vx(self) -> float:
        """Vx velocity component. Uses Rate Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_vx_metadata)

    _set_vx_metadata = { "offset" : _set_vx_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @vx.setter
    def vx(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_vx_metadata, value)

    _get_vy_metadata = { "offset" : _get_vy_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def vy(self) -> float:
        """Vy velocity component. Uses Rate Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_vy_metadata)

    _set_vy_metadata = { "offset" : _set_vy_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @vy.setter
    def vy(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_vy_metadata, value)

    _get_vz_metadata = { "offset" : _get_vz_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def vz(self) -> float:
        """Vz velocity component. Uses Rate Dimension."""
        return self._intf.get_property(ElementCartesian._metadata, ElementCartesian._get_vz_metadata)

    _set_vz_metadata = { "offset" : _set_vz_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @vz.setter
    def vz(self, value:float) -> None:
        return self._intf.set_property(ElementCartesian._metadata, ElementCartesian._set_vz_metadata, value)

    _property_names[x] = "x"
    _property_names[y] = "y"
    _property_names[z] = "z"
    _property_names[vx] = "vx"
    _property_names[vy] = "vy"
    _property_names[vz] = "vz"

    def __init__(self, source_object=None):
        """Construct an object of type ElementCartesian."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementCartesian)
        IElement.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementCartesian, [ElementCartesian, IElement, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5499013447957353870, 11386343310021492869), ElementCartesian)
agcls.AgTypeNameMap["ElementCartesian"] = ElementCartesian

class ElementKeplerian(IElement, SupportsDeleteCallback):
    """Properties for Keplerian elements."""

    _num_methods = 46
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_semimajor_axis_method_offset = 1
    _set_semimajor_axis_method_offset = 2
    _get_eccentricity_method_offset = 3
    _set_eccentricity_method_offset = 4
    _get_inclination_method_offset = 5
    _set_inclination_method_offset = 6
    _get_raan_method_offset = 7
    _set_raan_method_offset = 8
    _get_arg_of_periapsis_method_offset = 9
    _set_arg_of_periapsis_method_offset = 10
    _get_true_anomaly_method_offset = 11
    _set_true_anomaly_method_offset = 12
    _get_apoapsis_altitude_size_method_offset = 13
    _set_apoapsis_altitude_size_method_offset = 14
    _get_apoapsis_radius_size_method_offset = 15
    _set_apoapsis_radius_size_method_offset = 16
    _get_mean_motion_method_offset = 17
    _set_mean_motion_method_offset = 18
    _get_periapsis_altitude_size_method_offset = 19
    _set_periapsis_altitude_size_method_offset = 20
    _get_periapsis_radius_size_method_offset = 21
    _set_periapsis_radius_size_method_offset = 22
    _get_period_method_offset = 23
    _set_period_method_offset = 24
    _get_lan_method_offset = 25
    _set_lan_method_offset = 26
    _get_arg_of_latitude_method_offset = 27
    _set_arg_of_latitude_method_offset = 28
    _get_eccentric_anomaly_method_offset = 29
    _set_eccentric_anomaly_method_offset = 30
    _get_mean_anomaly_method_offset = 31
    _set_mean_anomaly_method_offset = 32
    _get_time_past_ascending_node_method_offset = 33
    _set_time_past_ascending_node_method_offset = 34
    _get_time_past_periapsis_method_offset = 35
    _set_time_past_periapsis_method_offset = 36
    _get_element_type_method_offset = 37
    _set_element_type_method_offset = 38
    _get_apoapsis_altitude_shape_method_offset = 39
    _set_apoapsis_altitude_shape_method_offset = 40
    _get_apoapsis_radius_shape_method_offset = 41
    _set_apoapsis_radius_shape_method_offset = 42
    _get_periapsis_altitude_shape_method_offset = 43
    _set_periapsis_altitude_shape_method_offset = 44
    _get_periapsis_radius_shape_method_offset = 45
    _set_periapsis_radius_shape_method_offset = 46
    _metadata = {
        "iid_data" : (4838552578512971254, 14667696790538271421),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementKeplerian)

    _get_semimajor_axis_metadata = { "offset" : _get_semimajor_axis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def semimajor_axis(self) -> float:
        """One-half the distance along the long axis of the elliptical orbit. Uses Distance Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_semimajor_axis_metadata)

    _set_semimajor_axis_metadata = { "offset" : _set_semimajor_axis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @semimajor_axis.setter
    def semimajor_axis(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_semimajor_axis_metadata, value)

    _get_eccentricity_metadata = { "offset" : _get_eccentricity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def eccentricity(self) -> float:
        """Describes the shape of the ellipse (a real number >= 0 and <1, where 0 = a circular orbit). Dimensionless."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_eccentricity_metadata)

    _set_eccentricity_metadata = { "offset" : _set_eccentricity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @eccentricity.setter
    def eccentricity(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_eccentricity_metadata, value)

    _get_inclination_metadata = { "offset" : _get_inclination_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def inclination(self) -> typing.Any:
        """Get or set the angle between the angular momentum vector (perpendicular to the plane of the orbit) and the inertial Z axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_inclination_metadata)

    _set_inclination_metadata = { "offset" : _set_inclination_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @inclination.setter
    def inclination(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_inclination_metadata, value)

    _get_raan_metadata = { "offset" : _get_raan_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def raan(self) -> typing.Any:
        """Get or set the angle from the inertial X axis to the ascending node measured in a right-handed sense about the inertial Z axis in the equatorial plane. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_raan_metadata)

    _set_raan_metadata = { "offset" : _set_raan_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @raan.setter
    def raan(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_raan_metadata, value)

    _get_arg_of_periapsis_metadata = { "offset" : _get_arg_of_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def arg_of_periapsis(self) -> typing.Any:
        """Get or set the angle from the ascending node to the eccentricity vector (lowest point of orbit) measured in the direction of the satellite's motion and in the orbit plane. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_arg_of_periapsis_metadata)

    _set_arg_of_periapsis_metadata = { "offset" : _set_arg_of_periapsis_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @arg_of_periapsis.setter
    def arg_of_periapsis(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_arg_of_periapsis_metadata, value)

    _get_true_anomaly_metadata = { "offset" : _get_true_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def true_anomaly(self) -> typing.Any:
        """Get or set the angle from the eccentricity vector (points toward perigee) to the satellite position vector, measured in the direction of satellite motion and in the orbit plane. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_true_anomaly_metadata)

    _set_true_anomaly_metadata = { "offset" : _set_true_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @true_anomaly.setter
    def true_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_true_anomaly_metadata, value)

    _get_apoapsis_altitude_size_metadata = { "offset" : _get_apoapsis_altitude_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def apoapsis_altitude_size(self) -> float:
        """Measured from the ``surface`` of the Earth to the points of maximum and minimum radius in the orbit. For these values, the surface of the Earth is modeled as a sphere whose radius equals the equatorial radius of the Earth. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_apoapsis_altitude_size_metadata)

    _set_apoapsis_altitude_size_metadata = { "offset" : _set_apoapsis_altitude_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @apoapsis_altitude_size.setter
    def apoapsis_altitude_size(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_apoapsis_altitude_size_metadata, value)

    _get_apoapsis_radius_size_metadata = { "offset" : _get_apoapsis_radius_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def apoapsis_radius_size(self) -> float:
        """Measured from the center of the Earth to the points of maximum and minimum radius in the orbit. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_apoapsis_radius_size_metadata)

    _set_apoapsis_radius_size_metadata = { "offset" : _set_apoapsis_radius_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @apoapsis_radius_size.setter
    def apoapsis_radius_size(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_apoapsis_radius_size_metadata, value)

    _get_mean_motion_metadata = { "offset" : _get_mean_motion_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def mean_motion(self) -> float:
        """Get or set the uniform rate of the satellite in a circular orbit of the same period, typically expressed as degrees or radians per second, or as revolutions per day. Uses AngleRate dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_mean_motion_metadata)

    _set_mean_motion_metadata = { "offset" : _set_mean_motion_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @mean_motion.setter
    def mean_motion(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_mean_motion_metadata, value)

    _get_periapsis_altitude_size_metadata = { "offset" : _get_periapsis_altitude_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def periapsis_altitude_size(self) -> float:
        """Measured from the ``surface`` of the Earth to the points of maximum and minimum radius in the orbit. For these values, the surface of the Earth is modeled as a sphere whose radius equals the equatorial radius of the Earth. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_periapsis_altitude_size_metadata)

    _set_periapsis_altitude_size_metadata = { "offset" : _set_periapsis_altitude_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @periapsis_altitude_size.setter
    def periapsis_altitude_size(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_periapsis_altitude_size_metadata, value)

    _get_periapsis_radius_size_metadata = { "offset" : _get_periapsis_radius_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def periapsis_radius_size(self) -> float:
        """Measured from the center of the Earth to the points of maximum and minimum radius in the orbit. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_periapsis_radius_size_metadata)

    _set_periapsis_radius_size_metadata = { "offset" : _set_periapsis_radius_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @periapsis_radius_size.setter
    def periapsis_radius_size(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_periapsis_radius_size_metadata, value)

    _get_period_metadata = { "offset" : _get_period_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def period(self) -> float:
        """Get or set the duration of one orbit, based on assumed two-body motion. Uses Time dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_period_metadata)

    _set_period_metadata = { "offset" : _set_period_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @period.setter
    def period(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_period_metadata, value)

    _get_lan_metadata = { "offset" : _get_lan_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def lan(self) -> typing.Any:
        """Get or set the Earth-fixed longitude where the satellite has crossed the inertial equator (the intersection of the ground track and the inertial equator) from south to north based on an assumption of two-body motion. Uses Angle dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_lan_metadata)

    _set_lan_metadata = { "offset" : _set_lan_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @lan.setter
    def lan(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_lan_metadata, value)

    _get_arg_of_latitude_metadata = { "offset" : _get_arg_of_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def arg_of_latitude(self) -> typing.Any:
        """Get or set the sum of the True Anomaly and the Argument of Perigee. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_arg_of_latitude_metadata)

    _set_arg_of_latitude_metadata = { "offset" : _set_arg_of_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @arg_of_latitude.setter
    def arg_of_latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_arg_of_latitude_metadata, value)

    _get_eccentric_anomaly_metadata = { "offset" : _get_eccentric_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def eccentric_anomaly(self) -> typing.Any:
        """Angle measured with origin at the center of the ellipse from the direction of perigee to a point on a circumscribing circle from which a line perpendicular to the SMA intersects the position of the satellite on the ellipse. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_eccentric_anomaly_metadata)

    _set_eccentric_anomaly_metadata = { "offset" : _set_eccentric_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @eccentric_anomaly.setter
    def eccentric_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_eccentric_anomaly_metadata, value)

    _get_mean_anomaly_metadata = { "offset" : _get_mean_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def mean_anomaly(self) -> typing.Any:
        """Get or set the angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate. Uses Angle Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_mean_anomaly_metadata)

    _set_mean_anomaly_metadata = { "offset" : _set_mean_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @mean_anomaly.setter
    def mean_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_mean_anomaly_metadata, value)

    _get_time_past_ascending_node_metadata = { "offset" : _get_time_past_ascending_node_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_past_ascending_node(self) -> float:
        """Get or set the elapsed time since the last ascending node crossing based on assumed two-body motion. Uses Time Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_time_past_ascending_node_metadata)

    _set_time_past_ascending_node_metadata = { "offset" : _set_time_past_ascending_node_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_past_ascending_node.setter
    def time_past_ascending_node(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_time_past_ascending_node_metadata, value)

    _get_time_past_periapsis_metadata = { "offset" : _get_time_past_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_past_periapsis(self) -> float:
        """Get or set the elapsed time since the last perigee passage based on assumed two-body motion. Uses Time Dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_time_past_periapsis_metadata)

    _set_time_past_periapsis_metadata = { "offset" : _set_time_past_periapsis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_past_periapsis.setter
    def time_past_periapsis(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_time_past_periapsis_metadata, value)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @property
    def element_type(self) -> "ElementType":
        """Which type of element (osculating or mean)."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_element_type_metadata)

    _set_element_type_metadata = { "offset" : _set_element_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @element_type.setter
    def element_type(self, value:"ElementType") -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_element_type_metadata, value)

    _get_apoapsis_altitude_shape_metadata = { "offset" : _get_apoapsis_altitude_shape_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def apoapsis_altitude_shape(self) -> float:
        """Describes the shape of the ellipse. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_apoapsis_altitude_shape_metadata)

    _set_apoapsis_altitude_shape_metadata = { "offset" : _set_apoapsis_altitude_shape_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @apoapsis_altitude_shape.setter
    def apoapsis_altitude_shape(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_apoapsis_altitude_shape_metadata, value)

    _get_apoapsis_radius_shape_metadata = { "offset" : _get_apoapsis_radius_shape_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def apoapsis_radius_shape(self) -> float:
        """Describes the shape of the ellipse. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_apoapsis_radius_shape_metadata)

    _set_apoapsis_radius_shape_metadata = { "offset" : _set_apoapsis_radius_shape_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @apoapsis_radius_shape.setter
    def apoapsis_radius_shape(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_apoapsis_radius_shape_metadata, value)

    _get_periapsis_altitude_shape_metadata = { "offset" : _get_periapsis_altitude_shape_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def periapsis_altitude_shape(self) -> float:
        """Describes the shape of the ellipse. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_periapsis_altitude_shape_metadata)

    _set_periapsis_altitude_shape_metadata = { "offset" : _set_periapsis_altitude_shape_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @periapsis_altitude_shape.setter
    def periapsis_altitude_shape(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_periapsis_altitude_shape_metadata, value)

    _get_periapsis_radius_shape_metadata = { "offset" : _get_periapsis_radius_shape_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def periapsis_radius_shape(self) -> float:
        """Describes the shape of the ellipse. Uses Distance dimension."""
        return self._intf.get_property(ElementKeplerian._metadata, ElementKeplerian._get_periapsis_radius_shape_metadata)

    _set_periapsis_radius_shape_metadata = { "offset" : _set_periapsis_radius_shape_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @periapsis_radius_shape.setter
    def periapsis_radius_shape(self, value:float) -> None:
        return self._intf.set_property(ElementKeplerian._metadata, ElementKeplerian._set_periapsis_radius_shape_metadata, value)

    _property_names[semimajor_axis] = "semimajor_axis"
    _property_names[eccentricity] = "eccentricity"
    _property_names[inclination] = "inclination"
    _property_names[raan] = "raan"
    _property_names[arg_of_periapsis] = "arg_of_periapsis"
    _property_names[true_anomaly] = "true_anomaly"
    _property_names[apoapsis_altitude_size] = "apoapsis_altitude_size"
    _property_names[apoapsis_radius_size] = "apoapsis_radius_size"
    _property_names[mean_motion] = "mean_motion"
    _property_names[periapsis_altitude_size] = "periapsis_altitude_size"
    _property_names[periapsis_radius_size] = "periapsis_radius_size"
    _property_names[period] = "period"
    _property_names[lan] = "lan"
    _property_names[arg_of_latitude] = "arg_of_latitude"
    _property_names[eccentric_anomaly] = "eccentric_anomaly"
    _property_names[mean_anomaly] = "mean_anomaly"
    _property_names[time_past_ascending_node] = "time_past_ascending_node"
    _property_names[time_past_periapsis] = "time_past_periapsis"
    _property_names[element_type] = "element_type"
    _property_names[apoapsis_altitude_shape] = "apoapsis_altitude_shape"
    _property_names[apoapsis_radius_shape] = "apoapsis_radius_shape"
    _property_names[periapsis_altitude_shape] = "periapsis_altitude_shape"
    _property_names[periapsis_radius_shape] = "periapsis_radius_shape"

    def __init__(self, source_object=None):
        """Construct an object of type ElementKeplerian."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementKeplerian)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementKeplerian, [ElementKeplerian, IElement])

agcls.AgClassCatalog.add_catalog_entry((4837397415885939909, 2998991674367307441), ElementKeplerian)
agcls.AgTypeNameMap["ElementKeplerian"] = ElementKeplerian

class ElementEquinoctial(IElement, SupportsDeleteCallback):
    """Properties for Equinoctial elements."""

    _num_methods = 16
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_semimajor_axis_method_offset = 1
    _set_semimajor_axis_method_offset = 2
    _get_mean_motion_method_offset = 3
    _set_mean_motion_method_offset = 4
    _get_h_method_offset = 5
    _set_h_method_offset = 6
    _get_k_method_offset = 7
    _set_k_method_offset = 8
    _get_p_method_offset = 9
    _set_p_method_offset = 10
    _get_q_method_offset = 11
    _set_q_method_offset = 12
    _get_mean_longitude_method_offset = 13
    _set_mean_longitude_method_offset = 14
    _get_formulation_method_offset = 15
    _set_formulation_method_offset = 16
    _metadata = {
        "iid_data" : (4929939837613149187, 12262545808188521122),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementEquinoctial)

    _get_semimajor_axis_metadata = { "offset" : _get_semimajor_axis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def semimajor_axis(self) -> float:
        """Half the length of the major axis of the orbital ellipse. Uses Distance Dimension."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_semimajor_axis_metadata)

    _set_semimajor_axis_metadata = { "offset" : _set_semimajor_axis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @semimajor_axis.setter
    def semimajor_axis(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_semimajor_axis_metadata, value)

    _get_mean_motion_metadata = { "offset" : _get_mean_motion_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def mean_motion(self) -> float:
        """Get or set the average angular rate of the satellite based on 2 body motion. Uses Rate Dimension."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_mean_motion_metadata)

    _set_mean_motion_metadata = { "offset" : _set_mean_motion_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @mean_motion.setter
    def mean_motion(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_mean_motion_metadata, value)

    _get_h_metadata = { "offset" : _get_h_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def h(self) -> float:
        """h/k collectively describe the shape of the satellite's orbit and the position of perigee. Dimensionless."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_h_metadata)

    _set_h_metadata = { "offset" : _set_h_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @h.setter
    def h(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_h_metadata, value)

    _get_k_metadata = { "offset" : _get_k_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def k(self) -> float:
        """h/k collectively describe the shape of the satellite's orbit and the position of perigee. Dimensionless."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_k_metadata)

    _set_k_metadata = { "offset" : _set_k_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @k.setter
    def k(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_k_metadata, value)

    _get_p_metadata = { "offset" : _get_p_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def p(self) -> float:
        """p/q collectively describe the orientation of the satellite's orbit plane. Dimensionless."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_p_metadata)

    _set_p_metadata = { "offset" : _set_p_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @p.setter
    def p(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_p_metadata, value)

    _get_q_metadata = { "offset" : _get_q_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def q(self) -> float:
        """p/q collectively describe the orientation of the satellite's orbit plane. Dimensionless."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_q_metadata)

    _set_q_metadata = { "offset" : _set_q_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @q.setter
    def q(self, value:float) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_q_metadata, value)

    _get_mean_longitude_metadata = { "offset" : _get_mean_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def mean_longitude(self) -> typing.Any:
        """Specify a satellite's position within its orbit at epoch and equals the sum of the classical Right Ascension of the Ascending Node, Argument of Perigee, and Mean Anomaly. Uses Angle Dimension."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_mean_longitude_metadata)

    _set_mean_longitude_metadata = { "offset" : _set_mean_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @mean_longitude.setter
    def mean_longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_mean_longitude_metadata, value)

    _get_formulation_metadata = { "offset" : _get_formulation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Formulation),) }
    @property
    def formulation(self) -> "Formulation":
        """Retrograde, which has its singularity at an inclination of 0 deg. Posigrade, which has its singularity at an inclination of 180 deg."""
        return self._intf.get_property(ElementEquinoctial._metadata, ElementEquinoctial._get_formulation_metadata)

    _set_formulation_metadata = { "offset" : _set_formulation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(Formulation),) }
    @formulation.setter
    def formulation(self, value:"Formulation") -> None:
        return self._intf.set_property(ElementEquinoctial._metadata, ElementEquinoctial._set_formulation_metadata, value)

    _property_names[semimajor_axis] = "semimajor_axis"
    _property_names[mean_motion] = "mean_motion"
    _property_names[h] = "h"
    _property_names[k] = "k"
    _property_names[p] = "p"
    _property_names[q] = "q"
    _property_names[mean_longitude] = "mean_longitude"
    _property_names[formulation] = "formulation"

    def __init__(self, source_object=None):
        """Construct an object of type ElementEquinoctial."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementEquinoctial)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementEquinoctial, [ElementEquinoctial, IElement])

agcls.AgClassCatalog.add_catalog_entry((4616647420053820287, 5121823230219393442), ElementEquinoctial)
agcls.AgTypeNameMap["ElementEquinoctial"] = ElementEquinoctial

class ElementDelaunay(IElement, SupportsDeleteCallback):
    """Properties for Delaunay elements."""

    _num_methods = 18
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_mean_anomaly_method_offset = 1
    _set_mean_anomaly_method_offset = 2
    _get_arg_of_periapsis_method_offset = 3
    _set_arg_of_periapsis_method_offset = 4
    _get_raan_method_offset = 5
    _set_raan_method_offset = 6
    _get_delaunay_l_method_offset = 7
    _set_delaunay_l_method_offset = 8
    _get_semimajor_axis_method_offset = 9
    _set_semimajor_axis_method_offset = 10
    _get_delaunay_g_method_offset = 11
    _set_delaunay_g_method_offset = 12
    _get_semilatus_rectum_method_offset = 13
    _set_semilatus_rectum_method_offset = 14
    _get_delaunay_h_method_offset = 15
    _set_delaunay_h_method_offset = 16
    _get_inclination_method_offset = 17
    _set_inclination_method_offset = 18
    _metadata = {
        "iid_data" : (4889723355609554180, 2305371170781207216),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementDelaunay)

    _get_mean_anomaly_metadata = { "offset" : _get_mean_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def mean_anomaly(self) -> typing.Any:
        """Get or set the angle from the eccentricity vector to a position vector where the satellite would be if it were always moving at its average angular rate (l). Uses Angle Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_mean_anomaly_metadata)

    _set_mean_anomaly_metadata = { "offset" : _set_mean_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @mean_anomaly.setter
    def mean_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_mean_anomaly_metadata, value)

    _get_arg_of_periapsis_metadata = { "offset" : _get_arg_of_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def arg_of_periapsis(self) -> typing.Any:
        """Get or set the angle from the ascending node to the eccentricity vector (lowest point of orbit) measured in the direction of the satellite's motion and in the orbit plane (g). Uses Angle Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_arg_of_periapsis_metadata)

    _set_arg_of_periapsis_metadata = { "offset" : _set_arg_of_periapsis_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @arg_of_periapsis.setter
    def arg_of_periapsis(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_arg_of_periapsis_metadata, value)

    _get_raan_metadata = { "offset" : _get_raan_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def raan(self) -> typing.Any:
        """Get or set the angle from the inertial X axis to the ascending node measured in a right-handed sense about the inertial Z axis in the equatorial plane (h). Uses Angle Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_raan_metadata)

    _set_raan_metadata = { "offset" : _set_raan_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @raan.setter
    def raan(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_raan_metadata, value)

    _get_delaunay_l_metadata = { "offset" : _get_delaunay_l_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delaunay_l(self) -> float:
        """Related to the two-body orbital energy. Defined as sqrt(GM * a).   Uses AreaRate Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_delaunay_l_metadata)

    _set_delaunay_l_metadata = { "offset" : _set_delaunay_l_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delaunay_l.setter
    def delaunay_l(self, value:float) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_delaunay_l_metadata, value)

    _get_semimajor_axis_metadata = { "offset" : _get_semimajor_axis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def semimajor_axis(self) -> float:
        """One-half the distance along the long axis of the elliptical orbit. Uses Distance Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_semimajor_axis_metadata)

    _set_semimajor_axis_metadata = { "offset" : _set_semimajor_axis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @semimajor_axis.setter
    def semimajor_axis(self, value:float) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_semimajor_axis_metadata, value)

    _get_delaunay_g_metadata = { "offset" : _get_delaunay_g_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delaunay_g(self) -> float:
        """Get or set the magnitude of the orbital angular momentum. Defined as sqrt(GM * p). Uses AreaRate Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_delaunay_g_metadata)

    _set_delaunay_g_metadata = { "offset" : _set_delaunay_g_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delaunay_g.setter
    def delaunay_g(self, value:float) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_delaunay_g_metadata, value)

    _get_semilatus_rectum_metadata = { "offset" : _get_semilatus_rectum_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def semilatus_rectum(self) -> float:
        """Semi-latus Rectum. Uses Distance Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_semilatus_rectum_metadata)

    _set_semilatus_rectum_metadata = { "offset" : _set_semilatus_rectum_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @semilatus_rectum.setter
    def semilatus_rectum(self, value:float) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_semilatus_rectum_metadata, value)

    _get_delaunay_h_metadata = { "offset" : _get_delaunay_h_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delaunay_h(self) -> float:
        """Get or set the Z component of the orbital angular momentum. Defined as G cos(inc).  Uses AreaRate Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_delaunay_h_metadata)

    _set_delaunay_h_metadata = { "offset" : _set_delaunay_h_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delaunay_h.setter
    def delaunay_h(self, value:float) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_delaunay_h_metadata, value)

    _get_inclination_metadata = { "offset" : _get_inclination_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def inclination(self) -> typing.Any:
        """Get or set the angle between the angular momentum vector (perpendicular to the plane of the orbit) and the inertial Z axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementDelaunay._metadata, ElementDelaunay._get_inclination_metadata)

    _set_inclination_metadata = { "offset" : _set_inclination_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @inclination.setter
    def inclination(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementDelaunay._metadata, ElementDelaunay._set_inclination_metadata, value)

    _property_names[mean_anomaly] = "mean_anomaly"
    _property_names[arg_of_periapsis] = "arg_of_periapsis"
    _property_names[raan] = "raan"
    _property_names[delaunay_l] = "delaunay_l"
    _property_names[semimajor_axis] = "semimajor_axis"
    _property_names[delaunay_g] = "delaunay_g"
    _property_names[semilatus_rectum] = "semilatus_rectum"
    _property_names[delaunay_h] = "delaunay_h"
    _property_names[inclination] = "inclination"

    def __init__(self, source_object=None):
        """Construct an object of type ElementDelaunay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementDelaunay)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementDelaunay, [ElementDelaunay, IElement])

agcls.AgClassCatalog.add_catalog_entry((5235483572359958962, 10845894774806134964), ElementDelaunay)
agcls.AgTypeNameMap["ElementDelaunay"] = ElementDelaunay

class ElementMixedSpherical(IElement, SupportsDeleteCallback):
    """Properties for Mixed Spherical elements."""

    _num_methods = 14
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_longitude_method_offset = 1
    _set_longitude_method_offset = 2
    _get_latitude_method_offset = 3
    _set_latitude_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _get_horizontal_flight_path_angle_method_offset = 7
    _set_horizontal_flight_path_angle_method_offset = 8
    _get_velocity_azimuth_method_offset = 9
    _set_velocity_azimuth_method_offset = 10
    _get_velocity_magnitude_method_offset = 11
    _set_velocity_magnitude_method_offset = 12
    _get_vertical_flight_path_angle_method_offset = 13
    _set_vertical_flight_path_angle_method_offset = 14
    _metadata = {
        "iid_data" : (5321102592411773457, 10534508528245466780),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementMixedSpherical)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Measured from -180.0 deg to +360.0 deg. Uses Angle Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_longitude_metadata, value)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Measured from -90.0 deg to +90.0 deg. The geodetic latitude of a point is the angle between (1) the normal to the reference ellipsoid that passes through the satellite position and (2) the equatorial plane. Uses Angle Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_latitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the object's position above or below the reference ellipsoid. Altitude is measured along a normal to the surface of the reference ellipsoid. Uses Distance Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_altitude_metadata, value)

    _get_horizontal_flight_path_angle_metadata = { "offset" : _get_horizontal_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def horizontal_flight_path_angle(self) -> typing.Any:
        """Horizontal (Hor FPA) or vertical (Ver FPA) flight path angle. The angle between the inertial velocity vector and the radius vector (vertical) or the complement of this angle (horizontal). Uses Angle Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_horizontal_flight_path_angle_metadata)

    _set_horizontal_flight_path_angle_metadata = { "offset" : _set_horizontal_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @horizontal_flight_path_angle.setter
    def horizontal_flight_path_angle(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_horizontal_flight_path_angle_metadata, value)

    _get_velocity_azimuth_metadata = { "offset" : _get_velocity_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def velocity_azimuth(self) -> typing.Any:
        """Get or set the angle in the satellite local horizontal plane between the projection of the inertial velocity vector onto this plane and the local north direction measured as positive in the clockwise direction. Uses Angle Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_velocity_azimuth_metadata)

    _set_velocity_azimuth_metadata = { "offset" : _set_velocity_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @velocity_azimuth.setter
    def velocity_azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_velocity_azimuth_metadata, value)

    _get_velocity_magnitude_metadata = { "offset" : _get_velocity_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def velocity_magnitude(self) -> float:
        """Get or set the angle in the satellite local horizontal plane between the projection of the inertial velocity vector onto this plane and the local north direction measured as positive in the clockwise direction. Uses Rate Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_velocity_magnitude_metadata)

    _set_velocity_magnitude_metadata = { "offset" : _set_velocity_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @velocity_magnitude.setter
    def velocity_magnitude(self, value:float) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_velocity_magnitude_metadata, value)

    _get_vertical_flight_path_angle_metadata = { "offset" : _get_vertical_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def vertical_flight_path_angle(self) -> typing.Any:
        """Horizontal (Hor FPA) or vertical (Ver FPA) flight path angle. The angle between the inertial velocity vector and the radius vector (vertical) or the complement of this angle (horizontal). Uses Angle Dimension."""
        return self._intf.get_property(ElementMixedSpherical._metadata, ElementMixedSpherical._get_vertical_flight_path_angle_metadata)

    _set_vertical_flight_path_angle_metadata = { "offset" : _set_vertical_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @vertical_flight_path_angle.setter
    def vertical_flight_path_angle(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementMixedSpherical._metadata, ElementMixedSpherical._set_vertical_flight_path_angle_metadata, value)

    _property_names[longitude] = "longitude"
    _property_names[latitude] = "latitude"
    _property_names[altitude] = "altitude"
    _property_names[horizontal_flight_path_angle] = "horizontal_flight_path_angle"
    _property_names[velocity_azimuth] = "velocity_azimuth"
    _property_names[velocity_magnitude] = "velocity_magnitude"
    _property_names[vertical_flight_path_angle] = "vertical_flight_path_angle"

    def __init__(self, source_object=None):
        """Construct an object of type ElementMixedSpherical."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementMixedSpherical)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementMixedSpherical, [ElementMixedSpherical, IElement])

agcls.AgClassCatalog.add_catalog_entry((5073171977022283093, 17307231198305551532), ElementMixedSpherical)
agcls.AgTypeNameMap["ElementMixedSpherical"] = ElementMixedSpherical

class ElementSpherical(IElement, SupportsDeleteCallback):
    """Properties for Spherical elements."""

    _num_methods = 14
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_right_ascension_method_offset = 1
    _set_right_ascension_method_offset = 2
    _get_declination_method_offset = 3
    _set_declination_method_offset = 4
    _get_radius_magnitude_method_offset = 5
    _set_radius_magnitude_method_offset = 6
    _get_horizontal_flight_path_angle_method_offset = 7
    _set_horizontal_flight_path_angle_method_offset = 8
    _get_velocity_azimuth_method_offset = 9
    _set_velocity_azimuth_method_offset = 10
    _get_velocity_magnitude_method_offset = 11
    _set_velocity_magnitude_method_offset = 12
    _get_vertical_flight_path_angle_method_offset = 13
    _set_vertical_flight_path_angle_method_offset = 14
    _metadata = {
        "iid_data" : (5756098792360695636, 8434394741165502088),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementSpherical)

    _get_right_ascension_metadata = { "offset" : _get_right_ascension_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def right_ascension(self) -> typing.Any:
        """Defined as the angle from the X axis to the projection of the satellite position vector in the equatorial plane measured as positive in the direction of the Y axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_right_ascension_metadata)

    _set_right_ascension_metadata = { "offset" : _set_right_ascension_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @right_ascension.setter
    def right_ascension(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_right_ascension_metadata, value)

    _get_declination_metadata = { "offset" : _get_declination_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def declination(self) -> typing.Any:
        """Defined as the angle between the satellite position vector and the inertial equatorial plane measured as positive toward the positive inertial Z axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_declination_metadata)

    _set_declination_metadata = { "offset" : _set_declination_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @declination.setter
    def declination(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_declination_metadata, value)

    _get_radius_magnitude_metadata = { "offset" : _get_radius_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius_magnitude(self) -> float:
        """Get or set the magnitude of the satellite position vector. Uses Distance Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_radius_magnitude_metadata)

    _set_radius_magnitude_metadata = { "offset" : _set_radius_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius_magnitude.setter
    def radius_magnitude(self, value:float) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_radius_magnitude_metadata, value)

    _get_horizontal_flight_path_angle_metadata = { "offset" : _get_horizontal_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def horizontal_flight_path_angle(self) -> typing.Any:
        """Horizontal (Hor FPA) or vertical (Ver FPA) flight path angle. The angle between the velocity vector and the radius vector (vertical) or the complement of this angle (horizontal). Uses Angle Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_horizontal_flight_path_angle_metadata)

    _set_horizontal_flight_path_angle_metadata = { "offset" : _set_horizontal_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @horizontal_flight_path_angle.setter
    def horizontal_flight_path_angle(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_horizontal_flight_path_angle_metadata, value)

    _get_velocity_azimuth_metadata = { "offset" : _get_velocity_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def velocity_azimuth(self) -> typing.Any:
        """Get or set the angle in the satellite local horizontal plane between the projection of the velocity vector onto this plane and the local north direction measured as positive in the clockwise direction. Uses Angle Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_velocity_azimuth_metadata)

    _set_velocity_azimuth_metadata = { "offset" : _set_velocity_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @velocity_azimuth.setter
    def velocity_azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_velocity_azimuth_metadata, value)

    _get_velocity_magnitude_metadata = { "offset" : _get_velocity_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def velocity_magnitude(self) -> float:
        """Get or set the magnitude of the velocity vector. Uses Rate Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_velocity_magnitude_metadata)

    _set_velocity_magnitude_metadata = { "offset" : _set_velocity_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @velocity_magnitude.setter
    def velocity_magnitude(self, value:float) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_velocity_magnitude_metadata, value)

    _get_vertical_flight_path_angle_metadata = { "offset" : _get_vertical_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def vertical_flight_path_angle(self) -> typing.Any:
        """Horizontal (Hor FPA) or vertical (Ver FPA) flight path angle. The angle between the velocity vector and the radius vector (vertical) or the complement of this angle (horizontal). Uses Angle Dimension."""
        return self._intf.get_property(ElementSpherical._metadata, ElementSpherical._get_vertical_flight_path_angle_metadata)

    _set_vertical_flight_path_angle_metadata = { "offset" : _set_vertical_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @vertical_flight_path_angle.setter
    def vertical_flight_path_angle(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSpherical._metadata, ElementSpherical._set_vertical_flight_path_angle_metadata, value)

    _property_names[right_ascension] = "right_ascension"
    _property_names[declination] = "declination"
    _property_names[radius_magnitude] = "radius_magnitude"
    _property_names[horizontal_flight_path_angle] = "horizontal_flight_path_angle"
    _property_names[velocity_azimuth] = "velocity_azimuth"
    _property_names[velocity_magnitude] = "velocity_magnitude"
    _property_names[vertical_flight_path_angle] = "vertical_flight_path_angle"

    def __init__(self, source_object=None):
        """Construct an object of type ElementSpherical."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementSpherical)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementSpherical, [ElementSpherical, IElement])

agcls.AgClassCatalog.add_catalog_entry((5178479078200941926, 400212392703168189), ElementSpherical)
agcls.AgTypeNameMap["ElementSpherical"] = ElementSpherical

class ElementTargetVectorIncomingAsymptote(IElement, SupportsDeleteCallback):
    """Properties for Target Vector Incoming Asymptote elements."""

    _num_methods = 12
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_radius_of_periapsis_method_offset = 1
    _set_radius_of_periapsis_method_offset = 2
    _get_c3_energy_method_offset = 3
    _set_c3_energy_method_offset = 4
    _get_right_ascension_of_incoming_asymptote_method_offset = 5
    _set_right_ascension_of_incoming_asymptote_method_offset = 6
    _get_declination_of_incoming_asymptote_method_offset = 7
    _set_declination_of_incoming_asymptote_method_offset = 8
    _get_velocity_azimuth_periapsis_method_offset = 9
    _set_velocity_azimuth_periapsis_method_offset = 10
    _get_true_anomaly_method_offset = 11
    _set_true_anomaly_method_offset = 12
    _metadata = {
        "iid_data" : (4892995040687317426, 11752684495085238196),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementTargetVectorIncomingAsymptote)

    _get_radius_of_periapsis_metadata = { "offset" : _get_radius_of_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius_of_periapsis(self) -> float:
        """Radius of Periapsis. Uses Distance Dimension."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_radius_of_periapsis_metadata)

    _set_radius_of_periapsis_metadata = { "offset" : _set_radius_of_periapsis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius_of_periapsis.setter
    def radius_of_periapsis(self, value:float) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_radius_of_periapsis_metadata, value)

    _get_c3_energy_metadata = { "offset" : _get_c3_energy_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def c3_energy(self) -> float:
        """C3Energy (Rate squared)."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_c3_energy_metadata)

    _set_c3_energy_metadata = { "offset" : _set_c3_energy_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @c3_energy.setter
    def c3_energy(self, value:float) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_c3_energy_metadata, value)

    _get_right_ascension_of_incoming_asymptote_metadata = { "offset" : _get_right_ascension_of_incoming_asymptote_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def right_ascension_of_incoming_asymptote(self) -> typing.Any:
        """Right Ascension of Incoming Asymptote. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_right_ascension_of_incoming_asymptote_metadata)

    _set_right_ascension_of_incoming_asymptote_metadata = { "offset" : _set_right_ascension_of_incoming_asymptote_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @right_ascension_of_incoming_asymptote.setter
    def right_ascension_of_incoming_asymptote(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_right_ascension_of_incoming_asymptote_metadata, value)

    _get_declination_of_incoming_asymptote_metadata = { "offset" : _get_declination_of_incoming_asymptote_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def declination_of_incoming_asymptote(self) -> typing.Any:
        """Declination of Incoming Asymptote. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_declination_of_incoming_asymptote_metadata)

    _set_declination_of_incoming_asymptote_metadata = { "offset" : _set_declination_of_incoming_asymptote_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @declination_of_incoming_asymptote.setter
    def declination_of_incoming_asymptote(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_declination_of_incoming_asymptote_metadata, value)

    _get_velocity_azimuth_periapsis_metadata = { "offset" : _get_velocity_azimuth_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def velocity_azimuth_periapsis(self) -> typing.Any:
        """Velocity Azimuth at Periapsis. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_velocity_azimuth_periapsis_metadata)

    _set_velocity_azimuth_periapsis_metadata = { "offset" : _set_velocity_azimuth_periapsis_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @velocity_azimuth_periapsis.setter
    def velocity_azimuth_periapsis(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_velocity_azimuth_periapsis_metadata, value)

    _get_true_anomaly_metadata = { "offset" : _get_true_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def true_anomaly(self) -> typing.Any:
        """True Anomaly. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._get_true_anomaly_metadata)

    _set_true_anomaly_metadata = { "offset" : _set_true_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @true_anomaly.setter
    def true_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorIncomingAsymptote._metadata, ElementTargetVectorIncomingAsymptote._set_true_anomaly_metadata, value)

    _property_names[radius_of_periapsis] = "radius_of_periapsis"
    _property_names[c3_energy] = "c3_energy"
    _property_names[right_ascension_of_incoming_asymptote] = "right_ascension_of_incoming_asymptote"
    _property_names[declination_of_incoming_asymptote] = "declination_of_incoming_asymptote"
    _property_names[velocity_azimuth_periapsis] = "velocity_azimuth_periapsis"
    _property_names[true_anomaly] = "true_anomaly"

    def __init__(self, source_object=None):
        """Construct an object of type ElementTargetVectorIncomingAsymptote."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementTargetVectorIncomingAsymptote)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementTargetVectorIncomingAsymptote, [ElementTargetVectorIncomingAsymptote, IElement])

agcls.AgClassCatalog.add_catalog_entry((4791707063847479912, 8451572865872127390), ElementTargetVectorIncomingAsymptote)
agcls.AgTypeNameMap["ElementTargetVectorIncomingAsymptote"] = ElementTargetVectorIncomingAsymptote

class ElementTargetVectorOutgoingAsymptote(IElement, SupportsDeleteCallback):
    """Properties for Target Vector Outgoing Asymptote elements."""

    _num_methods = 12
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_radius_of_periapsis_method_offset = 1
    _set_radius_of_periapsis_method_offset = 2
    _get_c3_energy_method_offset = 3
    _set_c3_energy_method_offset = 4
    _get_right_ascension_of_outgoing_asymptote_method_offset = 5
    _set_right_ascension_of_outgoing_asymptote_method_offset = 6
    _get_declination_of_outgoing_asymptote_method_offset = 7
    _set_declination_of_outgoing_asymptote_method_offset = 8
    _get_velocity_azimuth_periapsis_method_offset = 9
    _set_velocity_azimuth_periapsis_method_offset = 10
    _get_true_anomaly_method_offset = 11
    _set_true_anomaly_method_offset = 12
    _metadata = {
        "iid_data" : (5339488085564360630, 8608472543383636129),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementTargetVectorOutgoingAsymptote)

    _get_radius_of_periapsis_metadata = { "offset" : _get_radius_of_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius_of_periapsis(self) -> float:
        """Radius of Periapsis. Uses Distance Dimension."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_radius_of_periapsis_metadata)

    _set_radius_of_periapsis_metadata = { "offset" : _set_radius_of_periapsis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius_of_periapsis.setter
    def radius_of_periapsis(self, value:float) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_radius_of_periapsis_metadata, value)

    _get_c3_energy_metadata = { "offset" : _get_c3_energy_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def c3_energy(self) -> float:
        """C3Energy (Rate Squared)."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_c3_energy_metadata)

    _set_c3_energy_metadata = { "offset" : _set_c3_energy_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @c3_energy.setter
    def c3_energy(self, value:float) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_c3_energy_metadata, value)

    _get_right_ascension_of_outgoing_asymptote_metadata = { "offset" : _get_right_ascension_of_outgoing_asymptote_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def right_ascension_of_outgoing_asymptote(self) -> typing.Any:
        """Right Ascension of Incoming Asymptote. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_right_ascension_of_outgoing_asymptote_metadata)

    _set_right_ascension_of_outgoing_asymptote_metadata = { "offset" : _set_right_ascension_of_outgoing_asymptote_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @right_ascension_of_outgoing_asymptote.setter
    def right_ascension_of_outgoing_asymptote(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_right_ascension_of_outgoing_asymptote_metadata, value)

    _get_declination_of_outgoing_asymptote_metadata = { "offset" : _get_declination_of_outgoing_asymptote_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def declination_of_outgoing_asymptote(self) -> typing.Any:
        """Declination of Incoming Asymptote. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_declination_of_outgoing_asymptote_metadata)

    _set_declination_of_outgoing_asymptote_metadata = { "offset" : _set_declination_of_outgoing_asymptote_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @declination_of_outgoing_asymptote.setter
    def declination_of_outgoing_asymptote(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_declination_of_outgoing_asymptote_metadata, value)

    _get_velocity_azimuth_periapsis_metadata = { "offset" : _get_velocity_azimuth_periapsis_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def velocity_azimuth_periapsis(self) -> typing.Any:
        """Velocity Azimuth at Periapsis. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_velocity_azimuth_periapsis_metadata)

    _set_velocity_azimuth_periapsis_metadata = { "offset" : _set_velocity_azimuth_periapsis_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @velocity_azimuth_periapsis.setter
    def velocity_azimuth_periapsis(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_velocity_azimuth_periapsis_metadata, value)

    _get_true_anomaly_metadata = { "offset" : _get_true_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def true_anomaly(self) -> typing.Any:
        """True Anomaly. Uses Angle Dimension."""
        return self._intf.get_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._get_true_anomaly_metadata)

    _set_true_anomaly_metadata = { "offset" : _set_true_anomaly_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @true_anomaly.setter
    def true_anomaly(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementTargetVectorOutgoingAsymptote._metadata, ElementTargetVectorOutgoingAsymptote._set_true_anomaly_metadata, value)

    _property_names[radius_of_periapsis] = "radius_of_periapsis"
    _property_names[c3_energy] = "c3_energy"
    _property_names[right_ascension_of_outgoing_asymptote] = "right_ascension_of_outgoing_asymptote"
    _property_names[declination_of_outgoing_asymptote] = "declination_of_outgoing_asymptote"
    _property_names[velocity_azimuth_periapsis] = "velocity_azimuth_periapsis"
    _property_names[true_anomaly] = "true_anomaly"

    def __init__(self, source_object=None):
        """Construct an object of type ElementTargetVectorOutgoingAsymptote."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementTargetVectorOutgoingAsymptote)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementTargetVectorOutgoingAsymptote, [ElementTargetVectorOutgoingAsymptote, IElement])

agcls.AgClassCatalog.add_catalog_entry((5739406173576252558, 18013377322023481482), ElementTargetVectorOutgoingAsymptote)
agcls.AgTypeNameMap["ElementTargetVectorOutgoingAsymptote"] = ElementTargetVectorOutgoingAsymptote

class ElementGeodetic(IElement, SupportsDeleteCallback):
    """Properties for Geodetic elements."""

    _num_methods = 16
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _get_radius_magnitude_method_offset = 7
    _set_radius_magnitude_method_offset = 8
    _get_latitude_rate_method_offset = 9
    _set_latitude_rate_method_offset = 10
    _get_longitude_rate_method_offset = 11
    _set_longitude_rate_method_offset = 12
    _get_altitude_rate_method_offset = 13
    _set_altitude_rate_method_offset = 14
    _get_radius_rate_method_offset = 15
    _set_radius_rate_method_offset = 16
    _metadata = {
        "iid_data" : (5001084873375430510, 1131906645414183102),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementGeodetic)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Measured in degrees from -90.0 deg to +90.0 deg. The geodetic latitude of a point is the angle between the normal to the reference ellipsoid and the equatorial plane. Uses Angle Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Measured in degrees from -360.0 deg to +360.0 deg. The longitude of a point is the angle between the projection of the position vector in the equatorial plane and the prime meridian. Uses Angle Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_longitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Measured along an outward normal to the surface of the ellipsoid. Uses Distance Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_altitude_metadata, value)

    _get_radius_magnitude_metadata = { "offset" : _get_radius_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius_magnitude(self) -> float:
        """Measured from the center of the Earth. Specified as distance above or below the reference ellipsoid. Uses Distance Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_radius_magnitude_metadata)

    _set_radius_magnitude_metadata = { "offset" : _set_radius_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius_magnitude.setter
    def radius_magnitude(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_radius_magnitude_metadata, value)

    _get_latitude_rate_metadata = { "offset" : _get_latitude_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def latitude_rate(self) -> float:
        """Get or set the rate of change of the satellite's latitude. Uses Rate Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_latitude_rate_metadata)

    _set_latitude_rate_metadata = { "offset" : _set_latitude_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @latitude_rate.setter
    def latitude_rate(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_latitude_rate_metadata, value)

    _get_longitude_rate_metadata = { "offset" : _get_longitude_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def longitude_rate(self) -> float:
        """Get or set the rate of change of the satellite's longitude. Uses Rate Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_longitude_rate_metadata)

    _set_longitude_rate_metadata = { "offset" : _set_longitude_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @longitude_rate.setter
    def longitude_rate(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_longitude_rate_metadata, value)

    _get_altitude_rate_metadata = { "offset" : _get_altitude_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_rate(self) -> float:
        """Get or set the rate of change of the altitude. Uses Rate Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_altitude_rate_metadata)

    _set_altitude_rate_metadata = { "offset" : _set_altitude_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_rate.setter
    def altitude_rate(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_altitude_rate_metadata, value)

    _get_radius_rate_metadata = { "offset" : _get_radius_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius_rate(self) -> float:
        """Get or set the rate of change of the radius. Uses Rate Dimension."""
        return self._intf.get_property(ElementGeodetic._metadata, ElementGeodetic._get_radius_rate_metadata)

    _set_radius_rate_metadata = { "offset" : _set_radius_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius_rate.setter
    def radius_rate(self, value:float) -> None:
        return self._intf.set_property(ElementGeodetic._metadata, ElementGeodetic._set_radius_rate_metadata, value)

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[altitude] = "altitude"
    _property_names[radius_magnitude] = "radius_magnitude"
    _property_names[latitude_rate] = "latitude_rate"
    _property_names[longitude_rate] = "longitude_rate"
    _property_names[altitude_rate] = "altitude_rate"
    _property_names[radius_rate] = "radius_rate"

    def __init__(self, source_object=None):
        """Construct an object of type ElementGeodetic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementGeodetic)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementGeodetic, [ElementGeodetic, IElement])

agcls.AgClassCatalog.add_catalog_entry((4649398427299663878, 376586418108698241), ElementGeodetic)
agcls.AgTypeNameMap["ElementGeodetic"] = ElementGeodetic

class ElementBPlane(IElement, SupportsDeleteCallback):
    """Properties for BPlane elements."""

    _num_methods = 26
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_right_ascension_of_b_plane_method_offset = 1
    _set_right_ascension_of_b_plane_method_offset = 2
    _get_declination_of_b_plane_method_offset = 3
    _set_declination_of_b_plane_method_offset = 4
    _get_b_dot_r_first_b_vector_method_offset = 5
    _set_b_dot_r_first_b_vector_method_offset = 6
    _get_b_dot_t_second_b_vector_method_offset = 7
    _set_b_dot_t_second_b_vector_method_offset = 8
    _get_hyperbolic_turning_angle_method_offset = 9
    _set_hyperbolic_turning_angle_method_offset = 10
    _get_orbital_c3_energy_method_offset = 11
    _set_orbital_c3_energy_method_offset = 12
    _get_hyperbolic_v_infinity_method_offset = 13
    _set_hyperbolic_v_infinity_method_offset = 14
    _get_semimajor_axis_method_offset = 15
    _set_semimajor_axis_method_offset = 16
    _get_b_dot_t_first_b_vector_method_offset = 17
    _set_b_dot_t_first_b_vector_method_offset = 18
    _get_b_theta_first_b_vector_method_offset = 19
    _set_b_theta_first_b_vector_method_offset = 20
    _get_b_dot_r_second_b_vector_method_offset = 21
    _set_b_dot_r_second_b_vector_method_offset = 22
    _get_b_magnitude_second_b_vector_method_offset = 23
    _set_b_magnitude_second_b_vector_method_offset = 24
    _get_true_anomaly_method_offset = 25
    _set_true_anomaly_method_offset = 26
    _metadata = {
        "iid_data" : (5408340847027096584, 1882909674586958218),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementBPlane)

    _get_right_ascension_of_b_plane_metadata = { "offset" : _get_right_ascension_of_b_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def right_ascension_of_b_plane(self) -> float:
        """Right Ascension of the B-Plane Normal."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_right_ascension_of_b_plane_metadata)

    _set_right_ascension_of_b_plane_metadata = { "offset" : _set_right_ascension_of_b_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @right_ascension_of_b_plane.setter
    def right_ascension_of_b_plane(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_right_ascension_of_b_plane_metadata, value)

    _get_declination_of_b_plane_metadata = { "offset" : _get_declination_of_b_plane_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def declination_of_b_plane(self) -> float:
        """Declination of the B-Plane Normal."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_declination_of_b_plane_metadata)

    _set_declination_of_b_plane_metadata = { "offset" : _set_declination_of_b_plane_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @declination_of_b_plane.setter
    def declination_of_b_plane(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_declination_of_b_plane_metadata, value)

    _get_b_dot_r_first_b_vector_metadata = { "offset" : _get_b_dot_r_first_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_dot_r_first_b_vector(self) -> float:
        """B vector dotted with R vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_dot_r_first_b_vector_metadata)

    _set_b_dot_r_first_b_vector_metadata = { "offset" : _set_b_dot_r_first_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_dot_r_first_b_vector.setter
    def b_dot_r_first_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_dot_r_first_b_vector_metadata, value)

    _get_b_dot_t_second_b_vector_metadata = { "offset" : _get_b_dot_t_second_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_dot_t_second_b_vector(self) -> float:
        """B vector dotted with T vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_dot_t_second_b_vector_metadata)

    _set_b_dot_t_second_b_vector_metadata = { "offset" : _set_b_dot_t_second_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_dot_t_second_b_vector.setter
    def b_dot_t_second_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_dot_t_second_b_vector_metadata, value)

    _get_hyperbolic_turning_angle_metadata = { "offset" : _get_hyperbolic_turning_angle_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def hyperbolic_turning_angle(self) -> float:
        """Hyperbolic turning angle."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_hyperbolic_turning_angle_metadata)

    _set_hyperbolic_turning_angle_metadata = { "offset" : _set_hyperbolic_turning_angle_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @hyperbolic_turning_angle.setter
    def hyperbolic_turning_angle(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_hyperbolic_turning_angle_metadata, value)

    _get_orbital_c3_energy_metadata = { "offset" : _get_orbital_c3_energy_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def orbital_c3_energy(self) -> float:
        """Orbital C3 Energy."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_orbital_c3_energy_metadata)

    _set_orbital_c3_energy_metadata = { "offset" : _set_orbital_c3_energy_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @orbital_c3_energy.setter
    def orbital_c3_energy(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_orbital_c3_energy_metadata, value)

    _get_hyperbolic_v_infinity_metadata = { "offset" : _get_hyperbolic_v_infinity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def hyperbolic_v_infinity(self) -> float:
        """Hyperbolic V infinity."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_hyperbolic_v_infinity_metadata)

    _set_hyperbolic_v_infinity_metadata = { "offset" : _set_hyperbolic_v_infinity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @hyperbolic_v_infinity.setter
    def hyperbolic_v_infinity(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_hyperbolic_v_infinity_metadata, value)

    _get_semimajor_axis_metadata = { "offset" : _get_semimajor_axis_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def semimajor_axis(self) -> float:
        """Semi-major axis."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_semimajor_axis_metadata)

    _set_semimajor_axis_metadata = { "offset" : _set_semimajor_axis_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @semimajor_axis.setter
    def semimajor_axis(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_semimajor_axis_metadata, value)

    _get_b_dot_t_first_b_vector_metadata = { "offset" : _get_b_dot_t_first_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_dot_t_first_b_vector(self) -> float:
        """B vector dotted with T vector used as the first quantity to define the B vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_dot_t_first_b_vector_metadata)

    _set_b_dot_t_first_b_vector_metadata = { "offset" : _set_b_dot_t_first_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_dot_t_first_b_vector.setter
    def b_dot_t_first_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_dot_t_first_b_vector_metadata, value)

    _get_b_theta_first_b_vector_metadata = { "offset" : _get_b_theta_first_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_theta_first_b_vector(self) -> float:
        """Angle between the B vector and T vector used as the first quantity to define the B vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_theta_first_b_vector_metadata)

    _set_b_theta_first_b_vector_metadata = { "offset" : _set_b_theta_first_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_theta_first_b_vector.setter
    def b_theta_first_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_theta_first_b_vector_metadata, value)

    _get_b_dot_r_second_b_vector_metadata = { "offset" : _get_b_dot_r_second_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_dot_r_second_b_vector(self) -> float:
        """B vector dotted with T vector used as the second quantity to define the B vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_dot_r_second_b_vector_metadata)

    _set_b_dot_r_second_b_vector_metadata = { "offset" : _set_b_dot_r_second_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_dot_r_second_b_vector.setter
    def b_dot_r_second_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_dot_r_second_b_vector_metadata, value)

    _get_b_magnitude_second_b_vector_metadata = { "offset" : _get_b_magnitude_second_b_vector_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def b_magnitude_second_b_vector(self) -> float:
        """Magnitude of the B vector used as the second quantity to define the B vector."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_b_magnitude_second_b_vector_metadata)

    _set_b_magnitude_second_b_vector_metadata = { "offset" : _set_b_magnitude_second_b_vector_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @b_magnitude_second_b_vector.setter
    def b_magnitude_second_b_vector(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_b_magnitude_second_b_vector_metadata, value)

    _get_true_anomaly_metadata = { "offset" : _get_true_anomaly_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def true_anomaly(self) -> float:
        """True Anomaly."""
        return self._intf.get_property(ElementBPlane._metadata, ElementBPlane._get_true_anomaly_metadata)

    _set_true_anomaly_metadata = { "offset" : _set_true_anomaly_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @true_anomaly.setter
    def true_anomaly(self, value:float) -> None:
        return self._intf.set_property(ElementBPlane._metadata, ElementBPlane._set_true_anomaly_metadata, value)

    _property_names[right_ascension_of_b_plane] = "right_ascension_of_b_plane"
    _property_names[declination_of_b_plane] = "declination_of_b_plane"
    _property_names[b_dot_r_first_b_vector] = "b_dot_r_first_b_vector"
    _property_names[b_dot_t_second_b_vector] = "b_dot_t_second_b_vector"
    _property_names[hyperbolic_turning_angle] = "hyperbolic_turning_angle"
    _property_names[orbital_c3_energy] = "orbital_c3_energy"
    _property_names[hyperbolic_v_infinity] = "hyperbolic_v_infinity"
    _property_names[semimajor_axis] = "semimajor_axis"
    _property_names[b_dot_t_first_b_vector] = "b_dot_t_first_b_vector"
    _property_names[b_theta_first_b_vector] = "b_theta_first_b_vector"
    _property_names[b_dot_r_second_b_vector] = "b_dot_r_second_b_vector"
    _property_names[b_magnitude_second_b_vector] = "b_magnitude_second_b_vector"
    _property_names[true_anomaly] = "true_anomaly"

    def __init__(self, source_object=None):
        """Construct an object of type ElementBPlane."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementBPlane)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementBPlane, [ElementBPlane, IElement])

agcls.AgClassCatalog.add_catalog_entry((5159631324272900961, 8893182645510748334), ElementBPlane)
agcls.AgTypeNameMap["ElementBPlane"] = ElementBPlane

class ElementSphericalRangeRate(IElement, SupportsDeleteCallback):
    """Properties for Spherical Range Rate elements."""

    _num_methods = 12
    _vtable_offset = IElement._vtable_offset + IElement._num_methods
    _get_right_ascension_method_offset = 1
    _set_right_ascension_method_offset = 2
    _get_declination_method_offset = 3
    _set_declination_method_offset = 4
    _get_range_method_offset = 5
    _set_range_method_offset = 6
    _get_right_ascension_rate_method_offset = 7
    _set_right_ascension_rate_method_offset = 8
    _get_declination_rate_method_offset = 9
    _set_declination_rate_method_offset = 10
    _get_range_rate_method_offset = 11
    _set_range_rate_method_offset = 12
    _metadata = {
        "iid_data" : (5305548903845118989, 13104622450174055820),
        "vtable_reference" : IElement._vtable_offset + IElement._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ElementSphericalRangeRate)

    _get_right_ascension_metadata = { "offset" : _get_right_ascension_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def right_ascension(self) -> typing.Any:
        """Defined as the angle from the X axis to the projection of the satellite position vector in the equatorial plane measured as positive in the direction of the Y axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_right_ascension_metadata)

    _set_right_ascension_metadata = { "offset" : _set_right_ascension_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @right_ascension.setter
    def right_ascension(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_right_ascension_metadata, value)

    _get_declination_metadata = { "offset" : _get_declination_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def declination(self) -> typing.Any:
        """Defined as the angle between the satellite position vector and the inertial equatorial plane measured as positive toward the positive inertial Z axis. Uses Angle Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_declination_metadata)

    _set_declination_metadata = { "offset" : _set_declination_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @declination.setter
    def declination(self, value:typing.Any) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_declination_metadata, value)

    _get_range_metadata = { "offset" : _get_range_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range(self) -> float:
        """The magnitude of the satellite position vector. Uses Distance Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_range_metadata)

    _set_range_metadata = { "offset" : _set_range_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @range.setter
    def range(self, value:float) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_range_metadata, value)

    _get_right_ascension_rate_metadata = { "offset" : _get_right_ascension_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def right_ascension_rate(self) -> float:
        """The rate of change of the right ascension. Uses Angle Rate Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_right_ascension_rate_metadata)

    _set_right_ascension_rate_metadata = { "offset" : _set_right_ascension_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @right_ascension_rate.setter
    def right_ascension_rate(self, value:float) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_right_ascension_rate_metadata, value)

    _get_declination_rate_metadata = { "offset" : _get_declination_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def declination_rate(self) -> float:
        """The rate of change of the declination. Uses Angle Rate Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_declination_rate_metadata)

    _set_declination_rate_metadata = { "offset" : _set_declination_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @declination_rate.setter
    def declination_rate(self, value:float) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_declination_rate_metadata, value)

    _get_range_rate_metadata = { "offset" : _get_range_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def range_rate(self) -> float:
        """The rate of change of the range. Uses Rate Dimension."""
        return self._intf.get_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._get_range_rate_metadata)

    _set_range_rate_metadata = { "offset" : _set_range_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @range_rate.setter
    def range_rate(self, value:float) -> None:
        return self._intf.set_property(ElementSphericalRangeRate._metadata, ElementSphericalRangeRate._set_range_rate_metadata, value)

    _property_names[right_ascension] = "right_ascension"
    _property_names[declination] = "declination"
    _property_names[range] = "range"
    _property_names[right_ascension_rate] = "right_ascension_rate"
    _property_names[declination_rate] = "declination_rate"
    _property_names[range_rate] = "range_rate"

    def __init__(self, source_object=None):
        """Construct an object of type ElementSphericalRangeRate."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ElementSphericalRangeRate)
        IElement.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IElement._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ElementSphericalRangeRate, [ElementSphericalRangeRate, IElement])

agcls.AgClassCatalog.add_catalog_entry((4959039473664888368, 13118382122406988682), ElementSphericalRangeRate)
agcls.AgTypeNameMap["ElementSphericalRangeRate"] = ElementSphericalRangeRate

class MCSPropagate(IMCSSegment, IComponentInfo, ICloneable, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Propagate segment."""

    _num_methods = 21
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_propagator_name_method_offset = 1
    _set_propagator_name_method_offset = 2
    _get_stopping_conditions_method_offset = 3
    _get_min_propagation_time_method_offset = 4
    _set_min_propagation_time_method_offset = 5
    _get_max_propagation_time_method_offset = 6
    _set_max_propagation_time_method_offset = 7
    _get_enable_max_propagation_time_method_offset = 8
    _set_enable_max_propagation_time_method_offset = 9
    _get_enable_warning_message_method_offset = 10
    _set_enable_warning_message_method_offset = 11
    _enable_control_parameter_method_offset = 12
    _disable_control_parameter_method_offset = 13
    _is_control_parameter_enabled_method_offset = 14
    _get_control_parameters_available_method_offset = 15
    _get_override_max_propagation_time_method_offset = 16
    _set_override_max_propagation_time_method_offset = 17
    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 18
    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 19
    _get_should_reinitialize_stm_at_start_of_segment_propagation_method_offset = 20
    _set_should_reinitialize_stm_at_start_of_segment_propagation_method_offset = 21
    _metadata = {
        "iid_data" : (4982104245615881734, 11405336420499347081),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSPropagate)

    _get_propagator_name_metadata = { "offset" : _get_propagator_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def propagator_name(self) -> str:
        """Get or set the propagator."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_propagator_name_metadata)

    _set_propagator_name_metadata = { "offset" : _set_propagator_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @propagator_name.setter
    def propagator_name(self, value:str) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_propagator_name_metadata, value)

    _get_stopping_conditions_metadata = { "offset" : _get_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def stopping_conditions(self) -> "StoppingConditionCollection":
        """Get the list of stopping conditions defined for the segment."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_stopping_conditions_metadata)

    _get_min_propagation_time_metadata = { "offset" : _get_min_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_propagation_time(self) -> float:
        """Get or set the minimum time that must elapse from the beginning of the segment until Astrogator will begin checking stopping conditions for satisfaction. Uses Time Dimension."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_min_propagation_time_metadata)

    _set_min_propagation_time_metadata = { "offset" : _set_min_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_propagation_time.setter
    def min_propagation_time(self, value:float) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_min_propagation_time_metadata, value)

    _get_max_propagation_time_metadata = { "offset" : _get_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_propagation_time(self) -> float:
        """Get or set the maximum propagation time, after which the segment will end regardless of whether the stopping conditions have been satisfied. Uses Time Dimension."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_max_propagation_time_metadata)

    _set_max_propagation_time_metadata = { "offset" : _set_max_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_propagation_time.setter
    def max_propagation_time(self, value:float) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_max_propagation_time_metadata, value)

    _get_enable_max_propagation_time_metadata = { "offset" : _get_enable_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_max_propagation_time(self) -> bool:
        """If true, the maximum propagation time is enforced."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_enable_max_propagation_time_metadata)

    _set_enable_max_propagation_time_metadata = { "offset" : _set_enable_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_max_propagation_time.setter
    def enable_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_enable_max_propagation_time_metadata, value)

    _get_enable_warning_message_metadata = { "offset" : _get_enable_warning_message_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_warning_message(self) -> bool:
        """If true, Astrogator will issue a warning message if propagation is stopped by the Maximum Propagation Time parameter."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_enable_warning_message_metadata)

    _set_enable_warning_message_metadata = { "offset" : _set_enable_warning_message_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_warning_message.setter
    def enable_warning_message(self, value:bool) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_enable_warning_message_metadata, value)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced),) }
    def enable_control_parameter(self, param:"ControlAdvanced") -> None:
        """Enable or disables the specified control parameter."""
        return self._intf.invoke(MCSPropagate._metadata, MCSPropagate._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced),) }
    def disable_control_parameter(self, param:"ControlAdvanced") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSPropagate._metadata, MCSPropagate._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlAdvanced") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSPropagate._metadata, MCSPropagate._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_control_parameters_available_metadata)

    _get_override_max_propagation_time_metadata = { "offset" : _get_override_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def override_max_propagation_time(self) -> bool:
        """Override Maximum Propagation Time - if there is a duration or epoch stopping condition that occurs after the maximum propagation time, ignore the maximum propagation time."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_override_max_propagation_time_metadata)

    _set_override_max_propagation_time_metadata = { "offset" : _set_override_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @override_max_propagation_time.setter
    def override_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_override_max_propagation_time_metadata, value)

    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self) -> bool:
        """Stop immediately if propagation begins beyond an active epoch stopping condition."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata)

    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @should_stop_for_initially_surpassed_epoch_stopping_conditions.setter
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self, value:bool) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata, value)

    _get_should_reinitialize_stm_at_start_of_segment_propagation_metadata = { "offset" : _get_should_reinitialize_stm_at_start_of_segment_propagation_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def should_reinitialize_stm_at_start_of_segment_propagation(self) -> bool:
        """If this segment is propagating the state transition matrix, reset it to the identity matrix at the start of the segment."""
        return self._intf.get_property(MCSPropagate._metadata, MCSPropagate._get_should_reinitialize_stm_at_start_of_segment_propagation_metadata)

    _set_should_reinitialize_stm_at_start_of_segment_propagation_metadata = { "offset" : _set_should_reinitialize_stm_at_start_of_segment_propagation_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @should_reinitialize_stm_at_start_of_segment_propagation.setter
    def should_reinitialize_stm_at_start_of_segment_propagation(self, value:bool) -> None:
        return self._intf.set_property(MCSPropagate._metadata, MCSPropagate._set_should_reinitialize_stm_at_start_of_segment_propagation_metadata, value)

    _property_names[propagator_name] = "propagator_name"
    _property_names[stopping_conditions] = "stopping_conditions"
    _property_names[min_propagation_time] = "min_propagation_time"
    _property_names[max_propagation_time] = "max_propagation_time"
    _property_names[enable_max_propagation_time] = "enable_max_propagation_time"
    _property_names[enable_warning_message] = "enable_warning_message"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[override_max_propagation_time] = "override_max_propagation_time"
    _property_names[should_stop_for_initially_surpassed_epoch_stopping_conditions] = "should_stop_for_initially_surpassed_epoch_stopping_conditions"
    _property_names[should_reinitialize_stm_at_start_of_segment_propagation] = "should_reinitialize_stm_at_start_of_segment_propagation"

    def __init__(self, source_object=None):
        """Construct an object of type MCSPropagate."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSPropagate)
        IMCSSegment.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSPropagate, [MCSPropagate, IMCSSegment, IComponentInfo, ICloneable, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5194969086151067853, 10177006802175344288), MCSPropagate)
agcls.AgTypeNameMap["MCSPropagate"] = MCSPropagate

class State(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Spacecraft Parameters properties for the spacecraft configuration."""

    _num_methods = 33
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_element_type_method_offset = 1
    _set_element_type_method_offset = 2
    _get_element_method_offset = 3
    _get_epoch_method_offset = 4
    _set_epoch_method_offset = 5
    _get_coord_system_name_method_offset = 6
    _get_dry_mass_method_offset = 7
    _set_dry_mass_method_offset = 8
    _get_fuel_mass_method_offset = 9
    _set_fuel_mass_method_offset = 10
    _get_drag_area_method_offset = 11
    _set_drag_area_method_offset = 12
    _get_srp_area_method_offset = 13
    _set_srp_area_method_offset = 14
    _get_tank_pressure_method_offset = 15
    _set_tank_pressure_method_offset = 16
    _get_tank_temperature_method_offset = 17
    _set_tank_temperature_method_offset = 18
    _get_fuel_density_method_offset = 19
    _set_fuel_density_method_offset = 20
    _get_cr_method_offset = 21
    _set_cr_method_offset = 22
    _get_cd_method_offset = 23
    _set_cd_method_offset = 24
    _get_radiation_pressure_coefficient_method_offset = 25
    _set_radiation_pressure_coefficient_method_offset = 26
    _get_radiation_pressure_area_method_offset = 27
    _set_radiation_pressure_area_method_offset = 28
    _get_k1_method_offset = 29
    _set_k1_method_offset = 30
    _get_k2_method_offset = 31
    _set_k2_method_offset = 32
    _get_in_frame_name_method_offset = 33
    _metadata = {
        "iid_data" : (4804985124432915391, 9179349462890422656),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, State)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementSetType),) }
    @property
    def element_type(self) -> "ElementSetType":
        """Get the element type."""
        return self._intf.get_property(State._metadata, State._get_element_type_metadata)

    _set_element_type_metadata = { "offset" : _set_element_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ElementSetType),) }
    def set_element_type(self, element_type:"ElementSetType") -> None:
        """Set the element type."""
        return self._intf.invoke(State._metadata, State._set_element_type_metadata, element_type)

    _get_element_metadata = { "offset" : _get_element_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def element(self) -> "IElement":
        """Return the currently selected element type."""
        return self._intf.get_property(State._metadata, State._get_element_metadata)

    _get_epoch_metadata = { "offset" : _get_epoch_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def epoch(self) -> typing.Any:
        """Get or set the epoch of the Orbit State. Uses DateFormat Dimension."""
        return self._intf.get_property(State._metadata, State._get_epoch_metadata)

    _set_epoch_metadata = { "offset" : _set_epoch_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @epoch.setter
    def epoch(self, epoch:typing.Any) -> None:
        return self._intf.set_property(State._metadata, State._set_epoch_metadata, epoch)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get the coordinate system."""
        return self._intf.get_property(State._metadata, State._get_coord_system_name_metadata)

    _get_dry_mass_metadata = { "offset" : _get_dry_mass_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def dry_mass(self) -> float:
        """Get or set the mass of the spacecraft exclusive of propellant. Uses Mass Dimension."""
        return self._intf.get_property(State._metadata, State._get_dry_mass_metadata)

    _set_dry_mass_metadata = { "offset" : _set_dry_mass_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @dry_mass.setter
    def dry_mass(self, dry_mass:float) -> None:
        return self._intf.set_property(State._metadata, State._set_dry_mass_metadata, dry_mass)

    _get_fuel_mass_metadata = { "offset" : _get_fuel_mass_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_mass(self) -> float:
        """Get or set the mass of the spacecraft propellant. Uses Mass Dimension."""
        return self._intf.get_property(State._metadata, State._get_fuel_mass_metadata)

    _set_fuel_mass_metadata = { "offset" : _set_fuel_mass_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_mass.setter
    def fuel_mass(self, fuel_mass:float) -> None:
        return self._intf.set_property(State._metadata, State._set_fuel_mass_metadata, fuel_mass)

    _get_drag_area_metadata = { "offset" : _get_drag_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def drag_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of motion, used for atmospheric drag calculations. Uses SmallArea Dimension."""
        return self._intf.get_property(State._metadata, State._get_drag_area_metadata)

    _set_drag_area_metadata = { "offset" : _set_drag_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @drag_area.setter
    def drag_area(self, fuel_mass:float) -> None:
        return self._intf.set_property(State._metadata, State._set_drag_area_metadata, fuel_mass)

    _get_srp_area_metadata = { "offset" : _get_srp_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def srp_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of solar radiation, used for solar radiation calculations. Uses SmallArea Dimension."""
        return self._intf.get_property(State._metadata, State._get_srp_area_metadata)

    _set_srp_area_metadata = { "offset" : _set_srp_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @srp_area.setter
    def srp_area(self, srp_area:float) -> None:
        return self._intf.set_property(State._metadata, State._set_srp_area_metadata, srp_area)

    _get_tank_pressure_metadata = { "offset" : _get_tank_pressure_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tank_pressure(self) -> float:
        """Get or set the fuel tank pressure. Uses Pressure Dimension."""
        return self._intf.get_property(State._metadata, State._get_tank_pressure_metadata)

    _set_tank_pressure_metadata = { "offset" : _set_tank_pressure_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tank_pressure.setter
    def tank_pressure(self, tank_pressure:float) -> None:
        return self._intf.set_property(State._metadata, State._set_tank_pressure_metadata, tank_pressure)

    _get_tank_temperature_metadata = { "offset" : _get_tank_temperature_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tank_temperature(self) -> float:
        """Get or set the temperature of the fuel tank. Uses Temperature Dimension."""
        return self._intf.get_property(State._metadata, State._get_tank_temperature_metadata)

    _set_tank_temperature_metadata = { "offset" : _set_tank_temperature_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tank_temperature.setter
    def tank_temperature(self, tank_temperature:float) -> None:
        return self._intf.set_property(State._metadata, State._set_tank_temperature_metadata, tank_temperature)

    _get_fuel_density_metadata = { "offset" : _get_fuel_density_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fuel_density(self) -> float:
        """Get or set the density of the fuel tank. Uses SmallDensity Dimension."""
        return self._intf.get_property(State._metadata, State._get_fuel_density_metadata)

    _set_fuel_density_metadata = { "offset" : _set_fuel_density_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fuel_density.setter
    def fuel_density(self, fuel_density:float) -> None:
        return self._intf.set_property(State._metadata, State._set_fuel_density_metadata, fuel_density)

    _get_cr_metadata = { "offset" : _get_cr_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cr(self) -> float:
        """Get or set the reflectivity of the spacecraft used for solar radiation pressure calculations, where 2.0 is fully reflective and 1.0 is not reflective at all. Dimensionless."""
        return self._intf.get_property(State._metadata, State._get_cr_metadata)

    _set_cr_metadata = { "offset" : _set_cr_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cr.setter
    def cr(self, cr:float) -> None:
        return self._intf.set_property(State._metadata, State._set_cr_metadata, cr)

    _get_cd_metadata = { "offset" : _get_cd_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cd(self) -> float:
        """Get or set the dimensionless drag coefficient associated with the drag area. Dimensionless."""
        return self._intf.get_property(State._metadata, State._get_cd_metadata)

    _set_cd_metadata = { "offset" : _set_cd_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cd.setter
    def cd(self, cd:float) -> None:
        return self._intf.set_property(State._metadata, State._set_cd_metadata, cd)

    _get_radiation_pressure_coefficient_metadata = { "offset" : _get_radiation_pressure_coefficient_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radiation_pressure_coefficient(self) -> float:
        """Get or set the reflectivity of the spacecraft used for central body radiation pressure (albedo / thermal pressure) calculations, where 2.0 is fully reflective and 1.0 is not reflective at all. Dimensionless."""
        return self._intf.get_property(State._metadata, State._get_radiation_pressure_coefficient_metadata)

    _set_radiation_pressure_coefficient_metadata = { "offset" : _set_radiation_pressure_coefficient_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radiation_pressure_coefficient.setter
    def radiation_pressure_coefficient(self, radiation_pressure_coeff:float) -> None:
        return self._intf.set_property(State._metadata, State._set_radiation_pressure_coefficient_metadata, radiation_pressure_coeff)

    _get_radiation_pressure_area_metadata = { "offset" : _get_radiation_pressure_area_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radiation_pressure_area(self) -> float:
        """Get or set the cross-sectional area of the spacecraft assumed perpendicular to the direction of central body radiation, used for central body radiation (albedo / thermal pressure) calculations. Uses Small Area Dimension."""
        return self._intf.get_property(State._metadata, State._get_radiation_pressure_area_metadata)

    _set_radiation_pressure_area_metadata = { "offset" : _set_radiation_pressure_area_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radiation_pressure_area.setter
    def radiation_pressure_area(self, rad_pressure_area:float) -> None:
        return self._intf.set_property(State._metadata, State._set_radiation_pressure_area_metadata, rad_pressure_area)

    _get_k1_metadata = { "offset" : _get_k1_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def k1(self) -> float:
        """If you are using a non-spherical SRP model, this field defines the model's GPS solar radiation pressure K1 (scale) value. Dimensionless."""
        return self._intf.get_property(State._metadata, State._get_k1_metadata)

    _set_k1_metadata = { "offset" : _set_k1_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @k1.setter
    def k1(self, k1:float) -> None:
        return self._intf.set_property(State._metadata, State._set_k1_metadata, k1)

    _get_k2_metadata = { "offset" : _get_k2_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def k2(self) -> float:
        """If you are using a non-spherical SRP model, this field defines the model's GPS solar radiation pressure K2 (scale) value. Dimensionless."""
        return self._intf.get_property(State._metadata, State._get_k2_metadata)

    _set_k2_metadata = { "offset" : _set_k2_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @k2.setter
    def k2(self, k2:float) -> None:
        return self._intf.set_property(State._metadata, State._set_k2_metadata, k2)

    _get_in_frame_name_metadata = { "offset" : _get_in_frame_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_in_frame_name(self, frame_name:str) -> "State":
        """Get the orbit state in the specified frame."""
        return self._intf.invoke(State._metadata, State._get_in_frame_name_metadata, frame_name, OutArg())

    _property_names[element_type] = "element_type"
    _property_names[element] = "element"
    _property_names[epoch] = "epoch"
    _property_names[coord_system_name] = "coord_system_name"
    _property_names[dry_mass] = "dry_mass"
    _property_names[fuel_mass] = "fuel_mass"
    _property_names[drag_area] = "drag_area"
    _property_names[srp_area] = "srp_area"
    _property_names[tank_pressure] = "tank_pressure"
    _property_names[tank_temperature] = "tank_temperature"
    _property_names[fuel_density] = "fuel_density"
    _property_names[cr] = "cr"
    _property_names[cd] = "cd"
    _property_names[radiation_pressure_coefficient] = "radiation_pressure_coefficient"
    _property_names[radiation_pressure_area] = "radiation_pressure_area"
    _property_names[k1] = "k1"
    _property_names[k2] = "k2"

    def __init__(self, source_object=None):
        """Construct an object of type State."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, State)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, State, [State, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4795127631371666389, 13941908390852527797), State)
agcls.AgTypeNameMap["State"] = State

class StoppingConditionCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """The list of Stopping Conditions."""

    _num_methods = 10
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _add_method_offset = 2
    _remove_method_offset = 3
    _get__new_enum_method_offset = 4
    _get_count_method_offset = 5
    _cut_method_offset = 6
    _paste_method_offset = 7
    _insert_copy_method_offset = 8
    _get_item_by_index_method_offset = 9
    _get_item_by_name_method_offset = 10
    _metadata = {
        "iid_data" : (5630863724408399388, 12206717745624651652),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StoppingConditionCollection)
    def __iter__(self):
        """Create an iterator for the StoppingConditionCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "StoppingConditionElement":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "StoppingConditionElement":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._item_metadata, index_or_name, OutArg())

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add(self, condition_name:str) -> "StoppingConditionElement":
        """Add a stopping condition."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._add_metadata, condition_name, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def remove(self, index_or_name:typing.Any) -> None:
        """Remove a stopping condition."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._remove_metadata, index_or_name)

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(StoppingConditionCollection._metadata, StoppingConditionCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(StoppingConditionCollection._metadata, StoppingConditionCollection._get_count_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def cut(self, index_or_name:typing.Any) -> None:
        """Copy the stopping condition into the clipboard and removes the stopping condition from the list."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._cut_metadata, index_or_name)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def paste(self) -> "StoppingConditionElement":
        """Pastes the stopping condition from the clipboard and inserts into the list."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._paste_metadata, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("StoppingConditionElement"), agmarshall.InterfaceOutArg,) }
    def insert_copy(self, stop_cond:"StoppingConditionElement") -> "StoppingConditionElement":
        """Copy the stopping condition and inserts the copy into the list."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._insert_copy_metadata, stop_cond, OutArg())

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "StoppingConditionElement":
        """Retrieve a stopping condition in the collection by index."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "StoppingConditionElement":
        """Retrieve a stopping condition in the collection by name."""
        return self._intf.invoke(StoppingConditionCollection._metadata, StoppingConditionCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type StoppingConditionCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StoppingConditionCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StoppingConditionCollection, [StoppingConditionCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5312353168653404039, 12679003996478946741), StoppingConditionCollection)
agcls.AgTypeNameMap["StoppingConditionCollection"] = StoppingConditionCollection

class AccessStoppingCondition(IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for an access stopping condition."""

    _num_methods = 28
    _vtable_offset = IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods
    _get_time_convergence_method_offset = 1
    _set_time_convergence_method_offset = 2
    _get_repeat_count_method_offset = 3
    _set_repeat_count_method_offset = 4
    _get_inherited_method_offset = 5
    _set_inherited_method_offset = 6
    _get_max_trip_times_method_offset = 7
    _set_max_trip_times_method_offset = 8
    _get_sequence_method_offset = 9
    _set_sequence_method_offset = 10
    _get_constraints_method_offset = 11
    _get_criterion_method_offset = 12
    _set_criterion_method_offset = 13
    _get_before_conditions_method_offset = 14
    _get_aberration_type_method_offset = 15
    _set_aberration_type_method_offset = 16
    _set_base_selection_method_offset = 17
    _get_base_selection_type_method_offset = 18
    _get_base_selection_method_offset = 19
    _get_clock_host_method_offset = 20
    _set_clock_host_method_offset = 21
    _get_signal_sense_method_offset = 22
    _set_signal_sense_method_offset = 23
    _get_target_object_method_offset = 24
    _get_time_delay_convergence_tolerance_method_offset = 25
    _set_time_delay_convergence_tolerance_method_offset = 26
    _get_use_light_time_delay_method_offset = 27
    _set_use_light_time_delay_method_offset = 28
    _metadata = {
        "iid_data" : (4897277133588344594, 316659950229172406),
        "vtable_reference" : IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AccessStoppingCondition)

    _get_time_convergence_metadata = { "offset" : _get_time_convergence_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_convergence(self) -> float:
        """Get or set the time tolerance that is used by the access algorithms to determine the start times and stop times of access intervals. Uses Time dimension."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_time_convergence_metadata)

    _set_time_convergence_metadata = { "offset" : _set_time_convergence_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_convergence.setter
    def time_convergence(self, value:float) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_time_convergence_metadata, value)

    _get_repeat_count_metadata = { "offset" : _get_repeat_count_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def repeat_count(self) -> float:
        """Get or set the number of times the condition must be satisfied before the propagation ends or moves on to the designated automatic sequence. Dimensionless."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_repeat_count_metadata)

    _set_repeat_count_metadata = { "offset" : _set_repeat_count_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @repeat_count.setter
    def repeat_count(self, value:float) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_repeat_count_metadata, value)

    _get_inherited_metadata = { "offset" : _get_inherited_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def inherited(self) -> bool:
        """Condition Inherited by Automatic Sequences - if true, the stopping condition will be applied to any automatic sequences activated within the same segment."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_inherited_metadata)

    _set_inherited_metadata = { "offset" : _set_inherited_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @inherited.setter
    def inherited(self, value:bool) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_inherited_metadata, value)

    _get_max_trip_times_metadata = { "offset" : _get_max_trip_times_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_trip_times(self) -> float:
        """Get or set the maximum number of times that the stopping condition will be applied - and any resulting automatic sequences executed. Dimensionless."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_max_trip_times_metadata)

    _set_max_trip_times_metadata = { "offset" : _set_max_trip_times_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_trip_times.setter
    def max_trip_times(self, value:float) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_max_trip_times_metadata, value)

    _get_sequence_metadata = { "offset" : _get_sequence_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def sequence(self) -> str:
        """Get or set the automatic sequence to trigger if the highlighted stopping condition is satisfied."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_sequence_metadata)

    _set_sequence_metadata = { "offset" : _set_sequence_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @sequence.setter
    def sequence(self, value:str) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_sequence_metadata, value)

    _get_constraints_metadata = { "offset" : _get_constraints_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def constraints(self) -> "ConstraintCollection":
        """Further conditions that must be met in order for the stopping condition to be deemed satisfied."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_constraints_metadata)

    _get_criterion_metadata = { "offset" : _get_criterion_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AccessCriterion),) }
    @property
    def criterion(self) -> "AccessCriterion":
        """Specify the direction from which the stopping condition value must be achieved."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_criterion_metadata)

    _set_criterion_metadata = { "offset" : _set_criterion_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AccessCriterion),) }
    @criterion.setter
    def criterion(self, value:"AccessCriterion") -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_criterion_metadata, value)

    _get_before_conditions_metadata = { "offset" : _get_before_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def before_conditions(self) -> "StoppingConditionCollection":
        """A 'before' stopping condition is used to define a stopping condition that depends on two events. Astrogator will ignore a stopping condition until its 'before' conditions are met. Astrogator then interpolates backwards to the normal stopping condition."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_before_conditions_metadata)

    _get_aberration_type_metadata = { "offset" : _get_aberration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AberrationType),) }
    @property
    def aberration_type(self) -> "AberrationType":
        """Get or set the model of aberration to be used in access computations."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_aberration_type_metadata)

    _set_aberration_type_metadata = { "offset" : _set_aberration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AberrationType),) }
    @aberration_type.setter
    def aberration_type(self, value:"AberrationType") -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_aberration_type_metadata, value)

    _set_base_selection_metadata = { "offset" : _set_base_selection_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BaseSelection),) }
    def set_base_selection(self, selection:"BaseSelection") -> None:
        """BaseSelection."""
        return self._intf.invoke(AccessStoppingCondition._metadata, AccessStoppingCondition._set_base_selection_metadata, selection)

    _get_base_selection_type_metadata = { "offset" : _get_base_selection_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BaseSelection),) }
    @property
    def base_selection_type(self) -> "BaseSelection":
        """Get the base object for the access calculation."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_base_selection_type_metadata)

    _get_base_selection_metadata = { "offset" : _get_base_selection_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_selection(self) -> "ILinkToObject":
        """Return the base selection object."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_base_selection_metadata)

    _get_clock_host_metadata = { "offset" : _get_clock_host_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IvClockHost),) }
    @property
    def clock_host(self) -> "IvClockHost":
        """Time values are reported with a clock colocated with the clock host object."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_clock_host_metadata)

    _set_clock_host_metadata = { "offset" : _set_clock_host_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IvClockHost),) }
    @clock_host.setter
    def clock_host(self, value:"IvClockHost") -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_clock_host_metadata, value)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IvTimeSense),) }
    @property
    def signal_sense(self) -> "IvTimeSense":
        """Get or set the direction of the signal."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IvTimeSense),) }
    @signal_sense.setter
    def signal_sense(self, value:"IvTimeSense") -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_signal_sense_metadata, value)

    _get_target_object_metadata = { "offset" : _get_target_object_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_object(self) -> "ILinkToObject":
        """Get the target object for the access calculation."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_target_object_metadata)

    _get_time_delay_convergence_tolerance_metadata = { "offset" : _get_time_delay_convergence_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_delay_convergence_tolerance(self) -> float:
        """Get or set the tolerance used when iterating to determine the light time delay. The iteration stops when the improvement in the value is less than this tolerance. Uses Time Dimension."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_time_delay_convergence_tolerance_metadata)

    _set_time_delay_convergence_tolerance_metadata = { "offset" : _set_time_delay_convergence_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_delay_convergence_tolerance.setter
    def time_delay_convergence_tolerance(self, value:float) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_time_delay_convergence_tolerance_metadata, value)

    _get_use_light_time_delay_metadata = { "offset" : _get_use_light_time_delay_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_light_time_delay(self) -> bool:
        """Whether to consider light time delay in access computations."""
        return self._intf.get_property(AccessStoppingCondition._metadata, AccessStoppingCondition._get_use_light_time_delay_metadata)

    _set_use_light_time_delay_metadata = { "offset" : _set_use_light_time_delay_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_light_time_delay.setter
    def use_light_time_delay(self, value:bool) -> None:
        return self._intf.set_property(AccessStoppingCondition._metadata, AccessStoppingCondition._set_use_light_time_delay_metadata, value)

    _property_names[time_convergence] = "time_convergence"
    _property_names[repeat_count] = "repeat_count"
    _property_names[inherited] = "inherited"
    _property_names[max_trip_times] = "max_trip_times"
    _property_names[sequence] = "sequence"
    _property_names[constraints] = "constraints"
    _property_names[criterion] = "criterion"
    _property_names[before_conditions] = "before_conditions"
    _property_names[aberration_type] = "aberration_type"
    _property_names[base_selection_type] = "base_selection_type"
    _property_names[base_selection] = "base_selection"
    _property_names[clock_host] = "clock_host"
    _property_names[signal_sense] = "signal_sense"
    _property_names[target_object] = "target_object"
    _property_names[time_delay_convergence_tolerance] = "time_delay_convergence_tolerance"
    _property_names[use_light_time_delay] = "use_light_time_delay"

    def __init__(self, source_object=None):
        """Construct an object of type AccessStoppingCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AccessStoppingCondition)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
        IStoppingConditionComponent.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
        IStoppingConditionComponent._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AccessStoppingCondition, [AccessStoppingCondition, IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4811929967101649166, 10033959377873712801), AccessStoppingCondition)
agcls.AgTypeNameMap["AccessStoppingCondition"] = AccessStoppingCondition

class LightingStoppingCondition(IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a lighting stopping condition."""

    _num_methods = 18
    _vtable_offset = IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods
    _get_max_trip_times_method_offset = 1
    _set_max_trip_times_method_offset = 2
    _get_repeat_count_method_offset = 3
    _set_repeat_count_method_offset = 4
    _get_constraints_method_offset = 5
    _get_before_conditions_method_offset = 6
    _get_inherited_method_offset = 7
    _set_inherited_method_offset = 8
    _get_sequence_method_offset = 9
    _set_sequence_method_offset = 10
    _get_condition_method_offset = 11
    _set_condition_method_offset = 12
    _get_eclipsing_bodies_list_source_method_offset = 13
    _set_eclipsing_bodies_list_source_method_offset = 14
    _add_eclipsing_body_method_offset = 15
    _remove_eclipsing_body_method_offset = 16
    _get_eclipsing_bodies_method_offset = 17
    _get_available_eclipsing_bodies_method_offset = 18
    _metadata = {
        "iid_data" : (5629329659757080199, 7326471514331213698),
        "vtable_reference" : IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, LightingStoppingCondition)

    _get_max_trip_times_metadata = { "offset" : _get_max_trip_times_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_trip_times(self) -> float:
        """Rhe maximum number of times that the stopping condition will be applied - and any resulting automatic sequences executed. Dimensionless."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_max_trip_times_metadata)

    _set_max_trip_times_metadata = { "offset" : _set_max_trip_times_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_trip_times.setter
    def max_trip_times(self, value:float) -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_max_trip_times_metadata, value)

    _get_repeat_count_metadata = { "offset" : _get_repeat_count_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def repeat_count(self) -> float:
        """Get or set the number of times the condition must be satisfied before the propagation ends or moves on to the designated automatic sequence. Dimensionless."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_repeat_count_metadata)

    _set_repeat_count_metadata = { "offset" : _set_repeat_count_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @repeat_count.setter
    def repeat_count(self, value:float) -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_repeat_count_metadata, value)

    _get_constraints_metadata = { "offset" : _get_constraints_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def constraints(self) -> "ConstraintCollection":
        """Further conditions that must be met in order for the stopping condition to be deemed satisfied."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_constraints_metadata)

    _get_before_conditions_metadata = { "offset" : _get_before_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def before_conditions(self) -> "StoppingConditionCollection":
        """A 'before' stopping condition is used to define a stopping condition that depends on two events. Astrogator will ignore a stopping condition until its 'before' conditions are met. Astrogator then interpolates backwards to the normal stopping condition."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_before_conditions_metadata)

    _get_inherited_metadata = { "offset" : _get_inherited_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def inherited(self) -> bool:
        """Condition Inherited by Automatic Sequences - if true, the stopping condition will be applied to any automatic sequences activated within the same segment."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_inherited_metadata)

    _set_inherited_metadata = { "offset" : _set_inherited_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @inherited.setter
    def inherited(self, value:bool) -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_inherited_metadata, value)

    _get_sequence_metadata = { "offset" : _get_sequence_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def sequence(self) -> str:
        """Get or set the automatic sequence to trigger if the highlighted stopping condition is satisfied."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_sequence_metadata)

    _set_sequence_metadata = { "offset" : _set_sequence_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @sequence.setter
    def sequence(self, value:str) -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_sequence_metadata, value)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(LightingCondition),) }
    @property
    def condition(self) -> "LightingCondition":
        """Specify the direction from which the stopping condition value must be achieved."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_condition_metadata)

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(LightingCondition),) }
    @condition.setter
    def condition(self, value:"LightingCondition") -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_condition_metadata, value)

    _get_eclipsing_bodies_list_source_metadata = { "offset" : _get_eclipsing_bodies_list_source_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EclipsingBodiesSource),) }
    @property
    def eclipsing_bodies_list_source(self) -> "EclipsingBodiesSource":
        """Eclipsing Bodies List Source."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_eclipsing_bodies_list_source_metadata)

    _set_eclipsing_bodies_list_source_metadata = { "offset" : _set_eclipsing_bodies_list_source_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EclipsingBodiesSource),) }
    @eclipsing_bodies_list_source.setter
    def eclipsing_bodies_list_source(self, value:"EclipsingBodiesSource") -> None:
        return self._intf.set_property(LightingStoppingCondition._metadata, LightingStoppingCondition._set_eclipsing_bodies_list_source_metadata, value)

    _add_eclipsing_body_metadata = { "offset" : _add_eclipsing_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def add_eclipsing_body(self, eclipsing_body:str) -> None:
        """Add an eclipsing body."""
        return self._intf.invoke(LightingStoppingCondition._metadata, LightingStoppingCondition._add_eclipsing_body_metadata, eclipsing_body)

    _remove_eclipsing_body_metadata = { "offset" : _remove_eclipsing_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove_eclipsing_body(self, eclipsing_body:str) -> None:
        """Remove an eclipsing body."""
        return self._intf.invoke(LightingStoppingCondition._metadata, LightingStoppingCondition._remove_eclipsing_body_metadata, eclipsing_body)

    _get_eclipsing_bodies_metadata = { "offset" : _get_eclipsing_bodies_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def eclipsing_bodies(self) -> list:
        """Return a list of user selected eclipsing bodies."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_eclipsing_bodies_metadata)

    _get_available_eclipsing_bodies_metadata = { "offset" : _get_available_eclipsing_bodies_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_eclipsing_bodies(self) -> list:
        """Return a list of available eclipsing bodies."""
        return self._intf.get_property(LightingStoppingCondition._metadata, LightingStoppingCondition._get_available_eclipsing_bodies_metadata)

    _property_names[max_trip_times] = "max_trip_times"
    _property_names[repeat_count] = "repeat_count"
    _property_names[constraints] = "constraints"
    _property_names[before_conditions] = "before_conditions"
    _property_names[inherited] = "inherited"
    _property_names[sequence] = "sequence"
    _property_names[condition] = "condition"
    _property_names[eclipsing_bodies_list_source] = "eclipsing_bodies_list_source"
    _property_names[eclipsing_bodies] = "eclipsing_bodies"
    _property_names[available_eclipsing_bodies] = "available_eclipsing_bodies"

    def __init__(self, source_object=None):
        """Construct an object of type LightingStoppingCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, LightingStoppingCondition)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
        IStoppingConditionComponent.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
        IStoppingConditionComponent._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, LightingStoppingCondition, [LightingStoppingCondition, IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5265280262797051925, 13545129389778936992), LightingStoppingCondition)
agcls.AgTypeNameMap["LightingStoppingCondition"] = LightingStoppingCondition

class StoppingCondition(IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Basic properties for a stopping condition."""

    _num_methods = 30
    _vtable_offset = IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods
    _get_trip_method_offset = 1
    _set_trip_method_offset = 2
    _get_tolerance_method_offset = 3
    _set_tolerance_method_offset = 4
    _get_repeat_count_method_offset = 5
    _set_repeat_count_method_offset = 6
    _get_inherited_method_offset = 7
    _set_inherited_method_offset = 8
    _get_max_trip_times_method_offset = 9
    _set_max_trip_times_method_offset = 10
    _get_coord_system_method_offset = 11
    _set_coord_system_method_offset = 12
    _get_sequence_method_offset = 13
    _set_sequence_method_offset = 14
    _get_constraints_method_offset = 15
    _get_user_calculation_object_name_method_offset = 16
    _set_user_calculation_object_name_method_offset = 17
    _get_user_calculation_object_method_offset = 18
    _set_user_calculation_object_method_offset = 19
    _get_central_body_name_method_offset = 20
    _set_central_body_name_method_offset = 21
    _get_criterion_method_offset = 22
    _set_criterion_method_offset = 23
    _get_before_conditions_method_offset = 24
    _get_dimension_method_offset = 25
    _get_reference_point_method_offset = 26
    _set_reference_point_method_offset = 27
    _copy_user_calculation_object_to_clipboard_method_offset = 28
    _paste_user_calculation_object_from_clipboard_method_offset = 29
    _get_user_calculation_object_link_embed_control_method_offset = 30
    _metadata = {
        "iid_data" : (4986301151178184009, 6721367927442663319),
        "vtable_reference" : IStoppingConditionComponent._vtable_offset + IStoppingConditionComponent._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StoppingCondition)

    _get_trip_metadata = { "offset" : _get_trip_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def trip(self) -> typing.Any:
        """Get or set the desired value - the value at which the condition will be satisfied. Dimension depends on context."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_trip_metadata)

    _set_trip_metadata = { "offset" : _set_trip_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @trip.setter
    def trip(self, value:typing.Any) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_trip_metadata, value)

    _get_tolerance_metadata = { "offset" : _get_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tolerance(self) -> float:
        """Get or set the desired tolerance for achieving the stopping condition. Dimension depends on context."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_tolerance_metadata)

    _set_tolerance_metadata = { "offset" : _set_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tolerance.setter
    def tolerance(self, value:float) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_tolerance_metadata, value)

    _get_repeat_count_metadata = { "offset" : _get_repeat_count_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def repeat_count(self) -> float:
        """Get or set the number of times the condition must be satisfied before the propagation ends or moves on to the designated automatic sequence. Dimensionless."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_repeat_count_metadata)

    _set_repeat_count_metadata = { "offset" : _set_repeat_count_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @repeat_count.setter
    def repeat_count(self, value:float) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_repeat_count_metadata, value)

    _get_inherited_metadata = { "offset" : _get_inherited_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def inherited(self) -> bool:
        """Condition Inherited by Automatic Sequences - if true, the stopping condition will be applied to any automatic sequences activated within the same segment."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_inherited_metadata)

    _set_inherited_metadata = { "offset" : _set_inherited_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @inherited.setter
    def inherited(self, value:bool) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_inherited_metadata, value)

    _get_max_trip_times_metadata = { "offset" : _get_max_trip_times_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_trip_times(self) -> float:
        """Get or set the maximum number of times that the stopping condition will be applied - and any resulting automatic sequences executed. Dimensionless."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_max_trip_times_metadata)

    _set_max_trip_times_metadata = { "offset" : _set_max_trip_times_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_trip_times.setter
    def max_trip_times(self, value:float) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_max_trip_times_metadata, value)

    _get_coord_system_metadata = { "offset" : _get_coord_system_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system(self) -> str:
        """Get or set the coordinate system. The default coordinate system of a stopping condition is Earth Inertial."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_coord_system_metadata)

    _set_coord_system_metadata = { "offset" : _set_coord_system_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system.setter
    def coord_system(self, value:str) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_coord_system_metadata, value)

    _get_sequence_metadata = { "offset" : _get_sequence_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def sequence(self) -> str:
        """Get or set the automatic sequence to trigger if the highlighted stopping condition is satisfied."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_sequence_metadata)

    _set_sequence_metadata = { "offset" : _set_sequence_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @sequence.setter
    def sequence(self, value:str) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_sequence_metadata, value)

    _get_constraints_metadata = { "offset" : _get_constraints_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def constraints(self) -> "ConstraintCollection":
        """Further conditions that must be met in order for the stopping condition to be deemed satisfied."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_constraints_metadata)

    _get_user_calculation_object_name_metadata = { "offset" : _get_user_calculation_object_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def user_calculation_object_name(self) -> str:
        """User Calc Object - a User Calculation Object for the highlighted stopping condition. For user-defined stopping conditions, use this field to specify what kind of value you want to stop on."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_user_calculation_object_name_metadata)

    _set_user_calculation_object_name_metadata = { "offset" : _set_user_calculation_object_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @user_calculation_object_name.setter
    def user_calculation_object_name(self, value:str) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_user_calculation_object_name_metadata, value)

    _get_user_calculation_object_metadata = { "offset" : _get_user_calculation_object_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_calculation_object(self) -> "IComponentInfo":
        """A User Calculation Object for the highlighted stopping condition. For user-defined stopping conditions, use this field to specify what kind of value you want to stop on."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_user_calculation_object_metadata)

    _set_user_calculation_object_metadata = { "offset" : _set_user_calculation_object_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IComponentInfo"),) }
    @user_calculation_object.setter
    def user_calculation_object(self, new_value:"IComponentInfo") -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_user_calculation_object_metadata, new_value)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Get or set the central body. The default central body of a stopping condition is the Earth."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_central_body_name_metadata, value)

    _get_criterion_metadata = { "offset" : _get_criterion_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Criterion),) }
    @property
    def criterion(self) -> "Criterion":
        """Specify the direction from which the stopping condition value must be achieved."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_criterion_metadata)

    _set_criterion_metadata = { "offset" : _set_criterion_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(Criterion),) }
    @criterion.setter
    def criterion(self, value:"Criterion") -> None:
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_criterion_metadata, value)

    _get_before_conditions_metadata = { "offset" : _get_before_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def before_conditions(self) -> "StoppingConditionCollection":
        """A 'before' stopping condition is used to define a stopping condition that depends on two events. Astrogator will ignore a stopping condition until its 'before' conditions are met. Astrogator then interpolates backwards to the normal stopping condition."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_before_conditions_metadata)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Get the dimension of the stopping condition."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_dimension_metadata)

    _get_reference_point_metadata = { "offset" : _get_reference_point_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def reference_point(self) -> str:
        """Get or set the reference point used for calculation."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_reference_point_metadata)

    _set_reference_point_metadata = { "offset" : _set_reference_point_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @reference_point.setter
    def reference_point(self, value:str) -> None:
        """Get or set the reference point used for calculation."""
        return self._intf.set_property(StoppingCondition._metadata, StoppingCondition._set_reference_point_metadata, value)

    _copy_user_calculation_object_to_clipboard_metadata = { "offset" : _copy_user_calculation_object_to_clipboard_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_user_calculation_object_to_clipboard(self) -> None:
        """Copy the user-defined stopping condition calc object to the clipboard."""
        return self._intf.invoke(StoppingCondition._metadata, StoppingCondition._copy_user_calculation_object_to_clipboard_metadata, )

    _paste_user_calculation_object_from_clipboard_metadata = { "offset" : _paste_user_calculation_object_from_clipboard_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def paste_user_calculation_object_from_clipboard(self) -> None:
        """Replace the user-defined stopping condition calc object with the calc object in the clipboard."""
        return self._intf.invoke(StoppingCondition._metadata, StoppingCondition._paste_user_calculation_object_from_clipboard_metadata, )

    _get_user_calculation_object_link_embed_control_metadata = { "offset" : _get_user_calculation_object_link_embed_control_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_calculation_object_link_embed_control(self) -> "IComponentLinkEmbedControl":
        """Get the link / embed controller for managing the user calc. object."""
        return self._intf.get_property(StoppingCondition._metadata, StoppingCondition._get_user_calculation_object_link_embed_control_metadata)

    _property_names[trip] = "trip"
    _property_names[tolerance] = "tolerance"
    _property_names[repeat_count] = "repeat_count"
    _property_names[inherited] = "inherited"
    _property_names[max_trip_times] = "max_trip_times"
    _property_names[coord_system] = "coord_system"
    _property_names[sequence] = "sequence"
    _property_names[constraints] = "constraints"
    _property_names[user_calculation_object_name] = "user_calculation_object_name"
    _property_names[user_calculation_object] = "user_calculation_object"
    _property_names[central_body_name] = "central_body_name"
    _property_names[criterion] = "criterion"
    _property_names[before_conditions] = "before_conditions"
    _property_names[dimension] = "dimension"
    _property_names[reference_point] = "reference_point"
    _property_names[user_calculation_object_link_embed_control] = "user_calculation_object_link_embed_control"

    def __init__(self, source_object=None):
        """Construct an object of type StoppingCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StoppingCondition)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
        IStoppingConditionComponent.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
        IStoppingConditionComponent._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StoppingCondition, [StoppingCondition, IComponentInfo, ICloneable, IStoppingConditionComponent, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5288325747025247960, 14620341470789352850), StoppingCondition)
agcls.AgTypeNameMap["StoppingCondition"] = StoppingCondition

class StoppingConditionElement(IComponentInfo, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """The status of a stopping condition."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_active_method_offset = 1
    _set_active_method_offset = 2
    _enable_control_parameter_method_offset = 3
    _disable_control_parameter_method_offset = 4
    _is_control_parameter_enabled_method_offset = 5
    _get_control_parameters_available_method_offset = 6
    _get_properties_method_offset = 7
    _metadata = {
        "iid_data" : (4803961492800537644, 11540006383510258587),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StoppingConditionElement)

    _get_active_metadata = { "offset" : _get_active_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def active(self) -> bool:
        """If true, the stopping condition is active."""
        return self._intf.get_property(StoppingConditionElement._metadata, StoppingConditionElement._get_active_metadata)

    _set_active_metadata = { "offset" : _set_active_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @active.setter
    def active(self, value:bool) -> None:
        return self._intf.set_property(StoppingConditionElement._metadata, StoppingConditionElement._set_active_metadata, value)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlStoppingCondition),) }
    def enable_control_parameter(self, param:"ControlStoppingCondition") -> None:
        """Enable or disables the specified control parameter."""
        return self._intf.invoke(StoppingConditionElement._metadata, StoppingConditionElement._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlStoppingCondition),) }
    def disable_control_parameter(self, param:"ControlStoppingCondition") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(StoppingConditionElement._metadata, StoppingConditionElement._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlStoppingCondition), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlStoppingCondition") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(StoppingConditionElement._metadata, StoppingConditionElement._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(StoppingConditionElement._metadata, StoppingConditionElement._get_control_parameters_available_metadata)

    _get_properties_metadata = { "offset" : _get_properties_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def properties(self) -> "IStoppingConditionComponent":
        """Get the properties available to the stopping condition."""
        return self._intf.get_property(StoppingConditionElement._metadata, StoppingConditionElement._get_properties_metadata)

    _property_names[active] = "active"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[properties] = "properties"

    def __init__(self, source_object=None):
        """Construct an object of type StoppingConditionElement."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StoppingConditionElement)
        IComponentInfo.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StoppingConditionElement, [StoppingConditionElement, IComponentInfo, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5056737663241954080, 13082724610818700691), StoppingConditionElement)
agcls.AgTypeNameMap["StoppingConditionElement"] = StoppingConditionElement

class MCSSequence(IMCSSegment, IRuntimeTypeInfoProvider, IMCSSequence, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """The Sequence segment."""
    def __init__(self, source_object=None):
        """Construct an object of type MCSSequence."""
        SupportsDeleteCallback.__init__(self)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IMCSSequence.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IMCSSequence._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSSequence, [IMCSSegment, IRuntimeTypeInfoProvider, IMCSSequence, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4728221592052087246, 17245365907217799322), MCSSequence)
agcls.AgTypeNameMap["MCSSequence"] = MCSSequence

class MCSBackwardSequence(IMCSSegment, IRuntimeTypeInfoProvider, IMCSSequence, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Backward Sequence segment."""

    _num_methods = 0
    _vtable_offset = IMCSSequence._vtable_offset + IMCSSequence._num_methods
    _metadata = {
        "iid_data" : (5647154259170389577, 17278564796132019345),
        "vtable_reference" : IMCSSequence._vtable_offset + IMCSSequence._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSBackwardSequence)


    def __init__(self, source_object=None):
        """Construct an object of type MCSBackwardSequence."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSBackwardSequence)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IMCSSequence.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IMCSSequence._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSBackwardSequence, [MCSBackwardSequence, IMCSSegment, IRuntimeTypeInfoProvider, IMCSSequence, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5483461179733130724, 18354047637942609031), MCSBackwardSequence)
agcls.AgTypeNameMap["MCSBackwardSequence"] = MCSBackwardSequence

class MCSLaunch(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Launch segment."""

    _num_methods = 32
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_step_size_method_offset = 3
    _set_step_size_method_offset = 4
    _get_pre_launch_time_method_offset = 5
    _set_pre_launch_time_method_offset = 6
    _get_epoch_method_offset = 7
    _set_epoch_method_offset = 8
    _enable_control_parameter_method_offset = 9
    _disable_control_parameter_method_offset = 10
    _is_control_parameter_enabled_method_offset = 11
    _get_control_parameters_available_method_offset = 12
    _get_initial_acceleration_method_offset = 13
    _set_initial_acceleration_method_offset = 14
    _get_spacecraft_parameters_method_offset = 15
    _get_fuel_tank_method_offset = 16
    _get_display_system_type_method_offset = 17
    _set_display_system_type_method_offset = 18
    _get_display_system_method_offset = 19
    _get_ascent_type_method_offset = 20
    _set_ascent_type_method_offset = 21
    _get_time_of_flight_method_offset = 22
    _set_time_of_flight_method_offset = 23
    _get_burnout_type_method_offset = 24
    _set_burnout_type_method_offset = 25
    _get_burnout_method_offset = 26
    _get_burnout_velocity_method_offset = 27
    _get_use_previous_segment_state_method_offset = 28
    _set_use_previous_segment_state_method_offset = 29
    _get_set_mission_elapsed_time_epoch_method_offset = 30
    _set_set_mission_elapsed_time_epoch_method_offset = 31
    _get_user_variables_method_offset = 32
    _metadata = {
        "iid_data" : (5629359826497880460, 7376461998916133763),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSLaunch)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Get or set the central body."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_central_body_name_metadata, value)

    _get_step_size_metadata = { "offset" : _get_step_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step_size(self) -> float:
        """Get or set the time interval between calculated ephemeris output points. Uses Time Dimension."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_step_size_metadata)

    _set_step_size_metadata = { "offset" : _set_step_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step_size.setter
    def step_size(self, value:float) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_step_size_metadata, value)

    _get_pre_launch_time_metadata = { "offset" : _get_pre_launch_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def pre_launch_time(self) -> float:
        """Get or set the amount of time before the spacecraft's launch that it will be created in the scenario. The vehicle will remain at the launch position until beginning of the launch epoch. Uses Time Dimension."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_pre_launch_time_metadata)

    _set_pre_launch_time_metadata = { "offset" : _set_pre_launch_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @pre_launch_time.setter
    def pre_launch_time(self, value:float) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_pre_launch_time_metadata, value)

    _get_epoch_metadata = { "offset" : _get_epoch_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def epoch(self) -> typing.Any:
        """Get or set the date and time of the launch. Uses DateFormat Dimension."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_epoch_metadata)

    _set_epoch_metadata = { "offset" : _set_epoch_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @epoch.setter
    def epoch(self, value:typing.Any) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_epoch_metadata, value)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlLaunch),) }
    def enable_control_parameter(self, param:"ControlLaunch") -> None:
        """Enable the specified control parameter."""
        return self._intf.invoke(MCSLaunch._metadata, MCSLaunch._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlLaunch),) }
    def disable_control_parameter(self, param:"ControlLaunch") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSLaunch._metadata, MCSLaunch._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlLaunch), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlLaunch") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSLaunch._metadata, MCSLaunch._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_control_parameters_available_metadata)

    _get_initial_acceleration_metadata = { "offset" : _get_initial_acceleration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def initial_acceleration(self) -> float:
        """Get or set the initial acceleration, for the Quartic Motion Ascent Type. Uses Acceleration Dimension."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_initial_acceleration_metadata)

    _set_initial_acceleration_metadata = { "offset" : _set_initial_acceleration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @initial_acceleration.setter
    def initial_acceleration(self, value:float) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_initial_acceleration_metadata, value)

    _get_spacecraft_parameters_metadata = { "offset" : _get_spacecraft_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def spacecraft_parameters(self) -> "SpacecraftParameters":
        """Satellite Properties - the spacecraft's physical properties."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_spacecraft_parameters_metadata)

    _get_fuel_tank_metadata = { "offset" : _get_fuel_tank_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def fuel_tank(self) -> "FuelTank":
        """Get the spacecraft's fuel tank properties."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_fuel_tank_metadata)

    _get_display_system_type_metadata = { "offset" : _get_display_system_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(LaunchDisplaySystem),) }
    @property
    def display_system_type(self) -> "LaunchDisplaySystem":
        """Get the launch coordinate type."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_display_system_type_metadata)

    _set_display_system_type_metadata = { "offset" : _set_display_system_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(LaunchDisplaySystem),) }
    def set_display_system_type(self, display_system_type:"LaunchDisplaySystem") -> None:
        """Set the display system type."""
        return self._intf.invoke(MCSLaunch._metadata, MCSLaunch._set_display_system_type_metadata, display_system_type)

    _get_display_system_metadata = { "offset" : _get_display_system_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def display_system(self) -> "IDisplaySystem":
        """Get the current Display System."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_display_system_metadata)

    _get_ascent_type_metadata = { "offset" : _get_ascent_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AscentType),) }
    @property
    def ascent_type(self) -> "AscentType":
        """Get or set the order of the spline used to generate the motion along the ellipse."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_ascent_type_metadata)

    _set_ascent_type_metadata = { "offset" : _set_ascent_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AscentType),) }
    @ascent_type.setter
    def ascent_type(self, value:"AscentType") -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_ascent_type_metadata, value)

    _get_time_of_flight_metadata = { "offset" : _get_time_of_flight_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_of_flight(self) -> float:
        """Get or set the time of flight (the time from launch to burnout). Uses Time Dimension."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_time_of_flight_metadata)

    _set_time_of_flight_metadata = { "offset" : _set_time_of_flight_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_of_flight.setter
    def time_of_flight(self, value:float) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_time_of_flight_metadata, value)

    _get_burnout_type_metadata = { "offset" : _get_burnout_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BurnoutType),) }
    @property
    def burnout_type(self) -> "BurnoutType":
        """Get the burnout type."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_burnout_type_metadata)

    _set_burnout_type_metadata = { "offset" : _set_burnout_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BurnoutType),) }
    def set_burnout_type(self, burnout_type:"BurnoutType") -> None:
        """Set the burnout type."""
        return self._intf.invoke(MCSLaunch._metadata, MCSLaunch._set_burnout_type_metadata, burnout_type)

    _get_burnout_metadata = { "offset" : _get_burnout_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def burnout(self) -> "IBurnout":
        """Get the burnout point definition."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_burnout_metadata)

    _get_burnout_velocity_metadata = { "offset" : _get_burnout_velocity_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def burnout_velocity(self) -> "BurnoutVelocity":
        """Get the burnout velocity definition."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_burnout_velocity_metadata)

    _get_use_previous_segment_state_metadata = { "offset" : _get_use_previous_segment_state_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_previous_segment_state(self) -> bool:
        """If true, the previous segment state is used to define the launch location parameters."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_use_previous_segment_state_metadata)

    _set_use_previous_segment_state_metadata = { "offset" : _set_use_previous_segment_state_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_previous_segment_state.setter
    def use_previous_segment_state(self, value:bool) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_use_previous_segment_state_metadata, value)

    _get_set_mission_elapsed_time_epoch_metadata = { "offset" : _get_set_mission_elapsed_time_epoch_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def set_mission_elapsed_time_epoch(self) -> bool:
        """If true, the Mission Elapsed Time epoch will be set to the launch epoch."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_set_mission_elapsed_time_epoch_metadata)

    _set_set_mission_elapsed_time_epoch_metadata = { "offset" : _set_set_mission_elapsed_time_epoch_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @set_mission_elapsed_time_epoch.setter
    def set_mission_elapsed_time_epoch(self, value:bool) -> None:
        return self._intf.set_property(MCSLaunch._metadata, MCSLaunch._set_set_mission_elapsed_time_epoch_metadata, value)

    _get_user_variables_metadata = { "offset" : _get_user_variables_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_variables(self) -> "UserVariableCollection":
        """Interface used to modify user variables for the launch segment."""
        return self._intf.get_property(MCSLaunch._metadata, MCSLaunch._get_user_variables_metadata)

    _property_names[central_body_name] = "central_body_name"
    _property_names[step_size] = "step_size"
    _property_names[pre_launch_time] = "pre_launch_time"
    _property_names[epoch] = "epoch"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[initial_acceleration] = "initial_acceleration"
    _property_names[spacecraft_parameters] = "spacecraft_parameters"
    _property_names[fuel_tank] = "fuel_tank"
    _property_names[display_system_type] = "display_system_type"
    _property_names[display_system] = "display_system"
    _property_names[ascent_type] = "ascent_type"
    _property_names[time_of_flight] = "time_of_flight"
    _property_names[burnout_type] = "burnout_type"
    _property_names[burnout] = "burnout"
    _property_names[burnout_velocity] = "burnout_velocity"
    _property_names[use_previous_segment_state] = "use_previous_segment_state"
    _property_names[set_mission_elapsed_time_epoch] = "set_mission_elapsed_time_epoch"
    _property_names[user_variables] = "user_variables"

    def __init__(self, source_object=None):
        """Construct an object of type MCSLaunch."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSLaunch)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSLaunch, [MCSLaunch, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4943627088855875894, 16294750130809619880), MCSLaunch)
agcls.AgTypeNameMap["MCSLaunch"] = MCSLaunch

class DisplaySystemGeodetic(IDisplaySystem, SupportsDeleteCallback):
    """Properties for a geodetic launch coordinate system."""

    _num_methods = 6
    _vtable_offset = IDisplaySystem._vtable_offset + IDisplaySystem._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _metadata = {
        "iid_data" : (4956741510127523826, 11651406499667779456),
        "vtable_reference" : IDisplaySystem._vtable_offset + IDisplaySystem._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DisplaySystemGeodetic)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the latitude of the launch location. Uses Angle Dimension."""
        return self._intf.get_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the longitude of the launch location. Uses Angle Dimension."""
        return self._intf.get_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._set_longitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the altitude of the launch location. Uses Distance Dimension."""
        return self._intf.get_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        return self._intf.set_property(DisplaySystemGeodetic._metadata, DisplaySystemGeodetic._set_altitude_metadata, value)

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[altitude] = "altitude"

    def __init__(self, source_object=None):
        """Construct an object of type DisplaySystemGeodetic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DisplaySystemGeodetic)
        IDisplaySystem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplaySystem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DisplaySystemGeodetic, [DisplaySystemGeodetic, IDisplaySystem])

agcls.AgClassCatalog.add_catalog_entry((5185180578154751703, 12390255830257417611), DisplaySystemGeodetic)
agcls.AgTypeNameMap["DisplaySystemGeodetic"] = DisplaySystemGeodetic

class DisplaySystemGeocentric(IDisplaySystem, SupportsDeleteCallback):
    """Properties for a geocentric launch coordinate system."""

    _num_methods = 6
    _vtable_offset = IDisplaySystem._vtable_offset + IDisplaySystem._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_radius_method_offset = 5
    _set_radius_method_offset = 6
    _metadata = {
        "iid_data" : (5196042213365885690, 9963805608834460839),
        "vtable_reference" : IDisplaySystem._vtable_offset + IDisplaySystem._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DisplaySystemGeocentric)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the latitude of the launch location. Uses Angle Dimension."""
        return self._intf.get_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the longitude of the launch location. Uses Angle Dimension."""
        return self._intf.get_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._set_longitude_metadata, value)

    _get_radius_metadata = { "offset" : _get_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius(self) -> float:
        """Get or set the radius of the launch location. Uses Distance Dimension."""
        return self._intf.get_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._get_radius_metadata)

    _set_radius_metadata = { "offset" : _set_radius_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius.setter
    def radius(self, value:float) -> None:
        return self._intf.set_property(DisplaySystemGeocentric._metadata, DisplaySystemGeocentric._set_radius_metadata, value)

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[radius] = "radius"

    def __init__(self, source_object=None):
        """Construct an object of type DisplaySystemGeocentric."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DisplaySystemGeocentric)
        IDisplaySystem.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IDisplaySystem._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DisplaySystemGeocentric, [DisplaySystemGeocentric, IDisplaySystem])

agcls.AgClassCatalog.add_catalog_entry((5504948962407842285, 6237651898638065543), DisplaySystemGeocentric)
agcls.AgTypeNameMap["DisplaySystemGeocentric"] = DisplaySystemGeocentric

class BurnoutGeodetic(IBurnout, SupportsDeleteCallback):
    """Properties for a geodetic burnout point definition."""

    _num_methods = 6
    _vtable_offset = IBurnout._vtable_offset + IBurnout._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_altitude_method_offset = 5
    _set_altitude_method_offset = 6
    _metadata = {
        "iid_data" : (5576865739894481099, 2369937294263193020),
        "vtable_reference" : IBurnout._vtable_offset + IBurnout._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutGeodetic)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the latitude of the spacecraft burnout. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutGeodetic._metadata, BurnoutGeodetic._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutGeodetic._metadata, BurnoutGeodetic._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the longitude of the spacecraft burnout. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutGeodetic._metadata, BurnoutGeodetic._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutGeodetic._metadata, BurnoutGeodetic._set_longitude_metadata, value)

    _get_altitude_metadata = { "offset" : _get_altitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude(self) -> float:
        """Get or set the altitude of the spacecraft burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutGeodetic._metadata, BurnoutGeodetic._get_altitude_metadata)

    _set_altitude_metadata = { "offset" : _set_altitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude.setter
    def altitude(self, value:float) -> None:
        return self._intf.set_property(BurnoutGeodetic._metadata, BurnoutGeodetic._set_altitude_metadata, value)

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[altitude] = "altitude"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutGeodetic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutGeodetic)
        IBurnout.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBurnout._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutGeodetic, [BurnoutGeodetic, IBurnout])

agcls.AgClassCatalog.add_catalog_entry((5539865009872503013, 8668164257717054908), BurnoutGeodetic)
agcls.AgTypeNameMap["BurnoutGeodetic"] = BurnoutGeodetic

class BurnoutCBFCartesian(IBurnout, SupportsDeleteCallback):
    """Properties for a Cartesian CBF burnout state definition."""

    _num_methods = 12
    _vtable_offset = IBurnout._vtable_offset + IBurnout._num_methods
    _get_cartesian_burnout_x_method_offset = 1
    _set_cartesian_burnout_x_method_offset = 2
    _get_cartesian_burnout_y_method_offset = 3
    _set_cartesian_burnout_y_method_offset = 4
    _get_cartesian_burnout_z_method_offset = 5
    _set_cartesian_burnout_z_method_offset = 6
    _get_cartesian_burnout_vx_method_offset = 7
    _set_cartesian_burnout_vx_method_offset = 8
    _get_cartesian_burnout_vy_method_offset = 9
    _set_cartesian_burnout_vy_method_offset = 10
    _get_cartesian_burnout_vz_method_offset = 11
    _set_cartesian_burnout_vz_method_offset = 12
    _metadata = {
        "iid_data" : (5533696357243410767, 4114781725420291000),
        "vtable_reference" : IBurnout._vtable_offset + IBurnout._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutCBFCartesian)

    _get_cartesian_burnout_x_metadata = { "offset" : _get_cartesian_burnout_x_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_x(self) -> float:
        """The CBF x component of the spacecraft burnout location. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_x_metadata)

    _set_cartesian_burnout_x_metadata = { "offset" : _set_cartesian_burnout_x_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_x.setter
    def cartesian_burnout_x(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_x_metadata, value)

    _get_cartesian_burnout_y_metadata = { "offset" : _get_cartesian_burnout_y_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_y(self) -> float:
        """The CBF y component of the spacecraft burnout location. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_y_metadata)

    _set_cartesian_burnout_y_metadata = { "offset" : _set_cartesian_burnout_y_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_y.setter
    def cartesian_burnout_y(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_y_metadata, value)

    _get_cartesian_burnout_z_metadata = { "offset" : _get_cartesian_burnout_z_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_z(self) -> float:
        """The CBF z component of the spacecraft burnout location. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_z_metadata)

    _set_cartesian_burnout_z_metadata = { "offset" : _set_cartesian_burnout_z_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_z.setter
    def cartesian_burnout_z(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_z_metadata, value)

    _get_cartesian_burnout_vx_metadata = { "offset" : _get_cartesian_burnout_vx_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_vx(self) -> float:
        """The CBF x component of the spacecraft burnout velocity. Uses Rate Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_vx_metadata)

    _set_cartesian_burnout_vx_metadata = { "offset" : _set_cartesian_burnout_vx_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_vx.setter
    def cartesian_burnout_vx(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_vx_metadata, value)

    _get_cartesian_burnout_vy_metadata = { "offset" : _get_cartesian_burnout_vy_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_vy(self) -> float:
        """The CBF y component of the spacecraft burnout velocity. Uses Rate Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_vy_metadata)

    _set_cartesian_burnout_vy_metadata = { "offset" : _set_cartesian_burnout_vy_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_vy.setter
    def cartesian_burnout_vy(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_vy_metadata, value)

    _get_cartesian_burnout_vz_metadata = { "offset" : _get_cartesian_burnout_vz_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def cartesian_burnout_vz(self) -> float:
        """The CBF z component of the spacecraft burnout velocity. Uses Rate Dimension."""
        return self._intf.get_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._get_cartesian_burnout_vz_metadata)

    _set_cartesian_burnout_vz_metadata = { "offset" : _set_cartesian_burnout_vz_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @cartesian_burnout_vz.setter
    def cartesian_burnout_vz(self, value:float) -> None:
        return self._intf.set_property(BurnoutCBFCartesian._metadata, BurnoutCBFCartesian._set_cartesian_burnout_vz_metadata, value)

    _property_names[cartesian_burnout_x] = "cartesian_burnout_x"
    _property_names[cartesian_burnout_y] = "cartesian_burnout_y"
    _property_names[cartesian_burnout_z] = "cartesian_burnout_z"
    _property_names[cartesian_burnout_vx] = "cartesian_burnout_vx"
    _property_names[cartesian_burnout_vy] = "cartesian_burnout_vy"
    _property_names[cartesian_burnout_vz] = "cartesian_burnout_vz"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutCBFCartesian."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutCBFCartesian)
        IBurnout.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBurnout._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutCBFCartesian, [BurnoutCBFCartesian, IBurnout])

agcls.AgClassCatalog.add_catalog_entry((5317169324524563753, 10355185014748398739), BurnoutCBFCartesian)
agcls.AgTypeNameMap["BurnoutCBFCartesian"] = BurnoutCBFCartesian

class BurnoutGeocentric(IBurnout, SupportsDeleteCallback):
    """{Properties for a geocentric burnout point definition."""

    _num_methods = 6
    _vtable_offset = IBurnout._vtable_offset + IBurnout._num_methods
    _get_latitude_method_offset = 1
    _set_latitude_method_offset = 2
    _get_longitude_method_offset = 3
    _set_longitude_method_offset = 4
    _get_radius_method_offset = 5
    _set_radius_method_offset = 6
    _metadata = {
        "iid_data" : (5762718672652749616, 7859120436857445764),
        "vtable_reference" : IBurnout._vtable_offset + IBurnout._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutGeocentric)

    _get_latitude_metadata = { "offset" : _get_latitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def latitude(self) -> typing.Any:
        """Get or set the latitude of the spacecraft burnout. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutGeocentric._metadata, BurnoutGeocentric._get_latitude_metadata)

    _set_latitude_metadata = { "offset" : _set_latitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @latitude.setter
    def latitude(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutGeocentric._metadata, BurnoutGeocentric._set_latitude_metadata, value)

    _get_longitude_metadata = { "offset" : _get_longitude_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def longitude(self) -> typing.Any:
        """Get or set the longitude of the spacecraft burnout. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutGeocentric._metadata, BurnoutGeocentric._get_longitude_metadata)

    _set_longitude_metadata = { "offset" : _set_longitude_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @longitude.setter
    def longitude(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutGeocentric._metadata, BurnoutGeocentric._set_longitude_metadata, value)

    _get_radius_metadata = { "offset" : _get_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius(self) -> float:
        """Get or set the radius of the spacecraft burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutGeocentric._metadata, BurnoutGeocentric._get_radius_metadata)

    _set_radius_metadata = { "offset" : _set_radius_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius.setter
    def radius(self, value:float) -> None:
        return self._intf.set_property(BurnoutGeocentric._metadata, BurnoutGeocentric._set_radius_metadata, value)

    _property_names[latitude] = "latitude"
    _property_names[longitude] = "longitude"
    _property_names[radius] = "radius"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutGeocentric."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutGeocentric)
        IBurnout.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBurnout._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutGeocentric, [BurnoutGeocentric, IBurnout])

agcls.AgClassCatalog.add_catalog_entry((5595051361256152931, 155232765662894480), BurnoutGeocentric)
agcls.AgTypeNameMap["BurnoutGeocentric"] = BurnoutGeocentric

class BurnoutLaunchAzAltitude(IBurnout, SupportsDeleteCallback):
    """Properties for a launch azimuth / altitude burnout point definition."""

    _num_methods = 6
    _vtable_offset = IBurnout._vtable_offset + IBurnout._num_methods
    _get_azimuth_method_offset = 1
    _set_azimuth_method_offset = 2
    _get_down_range_dist_method_offset = 3
    _set_down_range_dist_method_offset = 4
    _get_altitude_radius_method_offset = 5
    _set_altitude_radius_method_offset = 6
    _metadata = {
        "iid_data" : (5725841762349009839, 190320382121745067),
        "vtable_reference" : IBurnout._vtable_offset + IBurnout._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutLaunchAzAltitude)

    _get_azimuth_metadata = { "offset" : _get_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def azimuth(self) -> typing.Any:
        """Get or set the azimuth of the launch trajectory. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._get_azimuth_metadata)

    _set_azimuth_metadata = { "offset" : _set_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @azimuth.setter
    def azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._set_azimuth_metadata, value)

    _get_down_range_dist_metadata = { "offset" : _get_down_range_dist_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def down_range_dist(self) -> float:
        """Get or set the downrange distance of the spacecraft from launch at burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._get_down_range_dist_metadata)

    _set_down_range_dist_metadata = { "offset" : _set_down_range_dist_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @down_range_dist.setter
    def down_range_dist(self, value:float) -> None:
        return self._intf.set_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._set_down_range_dist_metadata, value)

    _get_altitude_radius_metadata = { "offset" : _get_altitude_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def altitude_radius(self) -> float:
        """Get or set the altitude of the spacecraft at burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._get_altitude_radius_metadata)

    _set_altitude_radius_metadata = { "offset" : _set_altitude_radius_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @altitude_radius.setter
    def altitude_radius(self, value:float) -> None:
        return self._intf.set_property(BurnoutLaunchAzAltitude._metadata, BurnoutLaunchAzAltitude._set_altitude_radius_metadata, value)

    _property_names[azimuth] = "azimuth"
    _property_names[down_range_dist] = "down_range_dist"
    _property_names[altitude_radius] = "altitude_radius"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutLaunchAzAltitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutLaunchAzAltitude)
        IBurnout.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBurnout._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutLaunchAzAltitude, [BurnoutLaunchAzAltitude, IBurnout])

agcls.AgClassCatalog.add_catalog_entry((5762357320506376366, 9632671212585029549), BurnoutLaunchAzAltitude)
agcls.AgTypeNameMap["BurnoutLaunchAzAltitude"] = BurnoutLaunchAzAltitude

class BurnoutLaunchAzRadius(IBurnout, SupportsDeleteCallback):
    """Properties for a launch azimuth / radius burnout point definition."""

    _num_methods = 6
    _vtable_offset = IBurnout._vtable_offset + IBurnout._num_methods
    _get_azimuth_method_offset = 1
    _set_azimuth_method_offset = 2
    _get_down_range_dist_method_offset = 3
    _set_down_range_dist_method_offset = 4
    _get_radius_method_offset = 5
    _set_radius_method_offset = 6
    _metadata = {
        "iid_data" : (5453106663533698568, 1463398942728011174),
        "vtable_reference" : IBurnout._vtable_offset + IBurnout._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutLaunchAzRadius)

    _get_azimuth_metadata = { "offset" : _get_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def azimuth(self) -> typing.Any:
        """Get or set the azimuth of the launch trajectory. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._get_azimuth_metadata)

    _set_azimuth_metadata = { "offset" : _set_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @azimuth.setter
    def azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._set_azimuth_metadata, value)

    _get_down_range_dist_metadata = { "offset" : _get_down_range_dist_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def down_range_dist(self) -> float:
        """Get or set the downrange distance of the spacecraft from launch at burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._get_down_range_dist_metadata)

    _set_down_range_dist_metadata = { "offset" : _set_down_range_dist_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @down_range_dist.setter
    def down_range_dist(self, value:float) -> None:
        return self._intf.set_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._set_down_range_dist_metadata, value)

    _get_radius_metadata = { "offset" : _get_radius_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def radius(self) -> float:
        """Get or set the radius of the spacecraft at burnout. Uses Distance Dimension."""
        return self._intf.get_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._get_radius_metadata)

    _set_radius_metadata = { "offset" : _set_radius_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @radius.setter
    def radius(self, value:float) -> None:
        return self._intf.set_property(BurnoutLaunchAzRadius._metadata, BurnoutLaunchAzRadius._set_radius_metadata, value)

    _property_names[azimuth] = "azimuth"
    _property_names[down_range_dist] = "down_range_dist"
    _property_names[radius] = "radius"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutLaunchAzRadius."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutLaunchAzRadius)
        IBurnout.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IBurnout._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutLaunchAzRadius, [BurnoutLaunchAzRadius, IBurnout])

agcls.AgClassCatalog.add_catalog_entry((4790898737235059150, 10276049878661454781), BurnoutLaunchAzRadius)
agcls.AgTypeNameMap["BurnoutLaunchAzRadius"] = BurnoutLaunchAzRadius

class BurnoutVelocity(SupportsDeleteCallback):
    """Properties for the burnout velocity of a Launch segment."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_burnout_option_method_offset = 1
    _set_burnout_option_method_offset = 2
    _get_fixed_velocity_method_offset = 3
    _set_fixed_velocity_method_offset = 4
    _get_inertial_velocity_method_offset = 5
    _set_inertial_velocity_method_offset = 6
    _get_inertial_velocity_azimuth_method_offset = 7
    _set_inertial_velocity_azimuth_method_offset = 8
    _get_inertial_horizontal_flight_path_angle_method_offset = 9
    _set_inertial_horizontal_flight_path_angle_method_offset = 10
    _metadata = {
        "iid_data" : (4907025633322372581, 16846539782637522584),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BurnoutVelocity)

    _get_burnout_option_metadata = { "offset" : _get_burnout_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(BurnoutOptions),) }
    @property
    def burnout_option(self) -> "BurnoutOptions":
        """Select to use the fixed or inertial frame."""
        return self._intf.get_property(BurnoutVelocity._metadata, BurnoutVelocity._get_burnout_option_metadata)

    _set_burnout_option_metadata = { "offset" : _set_burnout_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(BurnoutOptions),) }
    @burnout_option.setter
    def burnout_option(self, value:"BurnoutOptions") -> None:
        return self._intf.set_property(BurnoutVelocity._metadata, BurnoutVelocity._set_burnout_option_metadata, value)

    _get_fixed_velocity_metadata = { "offset" : _get_fixed_velocity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def fixed_velocity(self) -> float:
        """Get or set the velocity magnitude . Uses Rate Dimension."""
        return self._intf.get_property(BurnoutVelocity._metadata, BurnoutVelocity._get_fixed_velocity_metadata)

    _set_fixed_velocity_metadata = { "offset" : _set_fixed_velocity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @fixed_velocity.setter
    def fixed_velocity(self, value:float) -> None:
        return self._intf.set_property(BurnoutVelocity._metadata, BurnoutVelocity._set_fixed_velocity_metadata, value)

    _get_inertial_velocity_metadata = { "offset" : _get_inertial_velocity_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def inertial_velocity(self) -> float:
        """Get or set the velocity magnitude. Uses Rate Dimension."""
        return self._intf.get_property(BurnoutVelocity._metadata, BurnoutVelocity._get_inertial_velocity_metadata)

    _set_inertial_velocity_metadata = { "offset" : _set_inertial_velocity_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @inertial_velocity.setter
    def inertial_velocity(self, value:float) -> None:
        return self._intf.set_property(BurnoutVelocity._metadata, BurnoutVelocity._set_inertial_velocity_metadata, value)

    _get_inertial_velocity_azimuth_metadata = { "offset" : _get_inertial_velocity_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def inertial_velocity_azimuth(self) -> typing.Any:
        """Get or set the inertial velocity azimuth. Inertial velocity azimuth is the angle from the projection of north in the local horizontal plane to the inertial velocity vector, right handed. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutVelocity._metadata, BurnoutVelocity._get_inertial_velocity_azimuth_metadata)

    _set_inertial_velocity_azimuth_metadata = { "offset" : _set_inertial_velocity_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @inertial_velocity_azimuth.setter
    def inertial_velocity_azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutVelocity._metadata, BurnoutVelocity._set_inertial_velocity_azimuth_metadata, value)

    _get_inertial_horizontal_flight_path_angle_metadata = { "offset" : _get_inertial_horizontal_flight_path_angle_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def inertial_horizontal_flight_path_angle(self) -> typing.Any:
        """Inertial horizontal flight path angle is the angle from the local horizontal to the inertial velocity vector, positive towards radius. It is also 90 degrees minus vertical flight path angle. Uses Angle Dimension."""
        return self._intf.get_property(BurnoutVelocity._metadata, BurnoutVelocity._get_inertial_horizontal_flight_path_angle_metadata)

    _set_inertial_horizontal_flight_path_angle_metadata = { "offset" : _set_inertial_horizontal_flight_path_angle_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @inertial_horizontal_flight_path_angle.setter
    def inertial_horizontal_flight_path_angle(self, value:typing.Any) -> None:
        return self._intf.set_property(BurnoutVelocity._metadata, BurnoutVelocity._set_inertial_horizontal_flight_path_angle_metadata, value)

    _property_names[burnout_option] = "burnout_option"
    _property_names[fixed_velocity] = "fixed_velocity"
    _property_names[inertial_velocity] = "inertial_velocity"
    _property_names[inertial_velocity_azimuth] = "inertial_velocity_azimuth"
    _property_names[inertial_horizontal_flight_path_angle] = "inertial_horizontal_flight_path_angle"

    def __init__(self, source_object=None):
        """Construct an object of type BurnoutVelocity."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BurnoutVelocity)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BurnoutVelocity, [BurnoutVelocity, ])

agcls.AgClassCatalog.add_catalog_entry((5357386615005395369, 9502758749122036118), BurnoutVelocity)
agcls.AgTypeNameMap["BurnoutVelocity"] = BurnoutVelocity

class MCSFollow(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Follow segment."""

    _num_methods = 22
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_leader_method_offset = 1
    _get_x_offset_method_offset = 2
    _set_x_offset_method_offset = 3
    _get_y_offset_method_offset = 4
    _set_y_offset_method_offset = 5
    _get_z_offset_method_offset = 6
    _set_z_offset_method_offset = 7
    _get_separation_conditions_method_offset = 8
    _get_spacecraft_parameters_method_offset = 9
    _get_fuel_tank_method_offset = 10
    _get_joining_type_method_offset = 11
    _set_joining_type_method_offset = 12
    _get_separation_type_method_offset = 13
    _set_separation_type_method_offset = 14
    _get_spacecraft_and_fuel_tank_type_method_offset = 15
    _set_spacecraft_and_fuel_tank_type_method_offset = 16
    _get_joining_conditions_method_offset = 17
    _enable_control_parameter_method_offset = 18
    _disable_control_parameter_method_offset = 19
    _is_control_parameter_enabled_method_offset = 20
    _get_control_parameters_available_method_offset = 21
    _get_user_variables_method_offset = 22
    _metadata = {
        "iid_data" : (5434070931037272692, 14909867834553447813),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSFollow)

    _get_leader_metadata = { "offset" : _get_leader_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def leader(self) -> "ILinkToObject":
        """Get the leader object."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_leader_metadata)

    _get_x_offset_metadata = { "offset" : _get_x_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def x_offset(self) -> float:
        """Get or set the distance that the spacecraft will be offset from the leader's body frame along the X axis. Uses Distance Dimension."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_x_offset_metadata)

    _set_x_offset_metadata = { "offset" : _set_x_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @x_offset.setter
    def x_offset(self, value:float) -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_x_offset_metadata, value)

    _get_y_offset_metadata = { "offset" : _get_y_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def y_offset(self) -> float:
        """Get or set the distance that the spacecraft will be offset from the leader's body frame along the Y axis. Uses Distance Dimension."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_y_offset_metadata)

    _set_y_offset_metadata = { "offset" : _set_y_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @y_offset.setter
    def y_offset(self, value:float) -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_y_offset_metadata, value)

    _get_z_offset_metadata = { "offset" : _get_z_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def z_offset(self) -> float:
        """Get or set the distance that the spacecraft will be offset from the leader's body frame along the Z axis. Uses Distance Dimension."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_z_offset_metadata)

    _set_z_offset_metadata = { "offset" : _set_z_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @z_offset.setter
    def z_offset(self, value:float) -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_z_offset_metadata, value)

    _get_separation_conditions_metadata = { "offset" : _get_separation_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def separation_conditions(self) -> "StoppingConditionCollection":
        """If separation conditions are specified, the list of separation conditions."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_separation_conditions_metadata)

    _get_spacecraft_parameters_metadata = { "offset" : _get_spacecraft_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def spacecraft_parameters(self) -> "SpacecraftParameters":
        """Get the spacecraft's physical properties."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_spacecraft_parameters_metadata)

    _get_fuel_tank_metadata = { "offset" : _get_fuel_tank_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def fuel_tank(self) -> "FuelTank":
        """Get the spacecraft's fuel tank properties."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_fuel_tank_metadata)

    _get_joining_type_metadata = { "offset" : _get_joining_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FollowJoin),) }
    @property
    def joining_type(self) -> "FollowJoin":
        """Get or set the joining type."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_joining_type_metadata)

    _set_joining_type_metadata = { "offset" : _set_joining_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FollowJoin),) }
    @joining_type.setter
    def joining_type(self, value:"FollowJoin") -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_joining_type_metadata, value)

    _get_separation_type_metadata = { "offset" : _get_separation_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FollowSeparation),) }
    @property
    def separation_type(self) -> "FollowSeparation":
        """Get or set the separation type."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_separation_type_metadata)

    _set_separation_type_metadata = { "offset" : _set_separation_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FollowSeparation),) }
    @separation_type.setter
    def separation_type(self, value:"FollowSeparation") -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_separation_type_metadata, value)

    _get_spacecraft_and_fuel_tank_type_metadata = { "offset" : _get_spacecraft_and_fuel_tank_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FollowSpacecraftAndFuelTank),) }
    @property
    def spacecraft_and_fuel_tank_type(self) -> "FollowSpacecraftAndFuelTank":
        """Get or set the spacecraft snd fuel tank configuration type."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_spacecraft_and_fuel_tank_type_metadata)

    _set_spacecraft_and_fuel_tank_type_metadata = { "offset" : _set_spacecraft_and_fuel_tank_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FollowSpacecraftAndFuelTank),) }
    @spacecraft_and_fuel_tank_type.setter
    def spacecraft_and_fuel_tank_type(self, value:"FollowSpacecraftAndFuelTank") -> None:
        return self._intf.set_property(MCSFollow._metadata, MCSFollow._set_spacecraft_and_fuel_tank_type_metadata, value)

    _get_joining_conditions_metadata = { "offset" : _get_joining_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def joining_conditions(self) -> "StoppingConditionCollection":
        """If joining conditions are specified, the list of joining conditions."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_joining_conditions_metadata)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlFollow),) }
    def enable_control_parameter(self, param:"ControlFollow") -> None:
        """Enable the specified control parameter."""
        return self._intf.invoke(MCSFollow._metadata, MCSFollow._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlFollow),) }
    def disable_control_parameter(self, param:"ControlFollow") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSFollow._metadata, MCSFollow._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlFollow), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlFollow") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSFollow._metadata, MCSFollow._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_control_parameters_available_metadata)

    _get_user_variables_metadata = { "offset" : _get_user_variables_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_variables(self) -> "UserVariableCollection":
        """Interface used to modify user variables for the follow segment."""
        return self._intf.get_property(MCSFollow._metadata, MCSFollow._get_user_variables_metadata)

    _property_names[leader] = "leader"
    _property_names[x_offset] = "x_offset"
    _property_names[y_offset] = "y_offset"
    _property_names[z_offset] = "z_offset"
    _property_names[separation_conditions] = "separation_conditions"
    _property_names[spacecraft_parameters] = "spacecraft_parameters"
    _property_names[fuel_tank] = "fuel_tank"
    _property_names[joining_type] = "joining_type"
    _property_names[separation_type] = "separation_type"
    _property_names[spacecraft_and_fuel_tank_type] = "spacecraft_and_fuel_tank_type"
    _property_names[joining_conditions] = "joining_conditions"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[user_variables] = "user_variables"

    def __init__(self, source_object=None):
        """Construct an object of type MCSFollow."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSFollow)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSFollow, [MCSFollow, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4651089075740067830, 16957833940556717219), MCSFollow)
agcls.AgTypeNameMap["MCSFollow"] = MCSFollow

class MCSManeuver(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """General properties for a Maneuver segment."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_maneuver_type_method_offset = 1
    _set_maneuver_type_method_offset = 2
    _get_maneuver_method_offset = 3
    _enable_control_parameter_method_offset = 4
    _disable_control_parameter_method_offset = 5
    _is_control_parameter_enabled_method_offset = 6
    _get_control_parameters_available_method_offset = 7
    _metadata = {
        "iid_data" : (5274901131733017230, 16553439757625640619),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSManeuver)

    _get_maneuver_type_metadata = { "offset" : _get_maneuver_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ManeuverType),) }
    @property
    def maneuver_type(self) -> "ManeuverType":
        """Get the maneuver type."""
        return self._intf.get_property(MCSManeuver._metadata, MCSManeuver._get_maneuver_type_metadata)

    _set_maneuver_type_metadata = { "offset" : _set_maneuver_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ManeuverType),) }
    def set_maneuver_type(self, maneuver_type:"ManeuverType") -> None:
        """Set the maneuver type."""
        return self._intf.invoke(MCSManeuver._metadata, MCSManeuver._set_maneuver_type_metadata, maneuver_type)

    _get_maneuver_metadata = { "offset" : _get_maneuver_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maneuver(self) -> "IManeuver":
        """Get the Maneuver properties."""
        return self._intf.get_property(MCSManeuver._metadata, MCSManeuver._get_maneuver_metadata)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlManeuver),) }
    def enable_control_parameter(self, param:"ControlManeuver") -> None:
        """Enable the specified control parameter."""
        return self._intf.invoke(MCSManeuver._metadata, MCSManeuver._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlManeuver),) }
    def disable_control_parameter(self, param:"ControlManeuver") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSManeuver._metadata, MCSManeuver._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlManeuver), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlManeuver") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSManeuver._metadata, MCSManeuver._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSManeuver._metadata, MCSManeuver._get_control_parameters_available_metadata)

    _property_names[maneuver_type] = "maneuver_type"
    _property_names[maneuver] = "maneuver"
    _property_names[control_parameters_available] = "control_parameters_available"

    def __init__(self, source_object=None):
        """Construct an object of type MCSManeuver."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSManeuver)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSManeuver, [MCSManeuver, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5197336180196135168, 17213101951654587804), MCSManeuver)
agcls.AgTypeNameMap["MCSManeuver"] = MCSManeuver

class ManeuverFinite(IManeuver, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Engine properties for a Finite Maneuver."""

    _num_methods = 7
    _vtable_offset = IManeuver._vtable_offset + IManeuver._num_methods
    _get_pressure_mode_method_offset = 1
    _set_pressure_mode_method_offset = 2
    _get_thrust_efficiency_method_offset = 3
    _set_thrust_efficiency_method_offset = 4
    _get_thrust_efficiency_mode_method_offset = 5
    _set_thrust_efficiency_mode_method_offset = 6
    _get_propagator_method_offset = 7
    _metadata = {
        "iid_data" : (5217603186191275445, 13780017606388512913),
        "vtable_reference" : IManeuver._vtable_offset + IManeuver._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ManeuverFinite)

    _get_pressure_mode_metadata = { "offset" : _get_pressure_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PressureMode),) }
    @property
    def pressure_mode(self) -> "PressureMode":
        """Get or set the pressure mode."""
        return self._intf.get_property(ManeuverFinite._metadata, ManeuverFinite._get_pressure_mode_metadata)

    _set_pressure_mode_metadata = { "offset" : _set_pressure_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(PressureMode),) }
    @pressure_mode.setter
    def pressure_mode(self, value:"PressureMode") -> None:
        return self._intf.set_property(ManeuverFinite._metadata, ManeuverFinite._set_pressure_mode_metadata, value)

    _get_thrust_efficiency_metadata = { "offset" : _get_thrust_efficiency_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def thrust_efficiency(self) -> float:
        """Get or set the thrust efficiency value. Any number above zero is valid, with typical values around 0.98 to 1.02. Dimensionless."""
        return self._intf.get_property(ManeuverFinite._metadata, ManeuverFinite._get_thrust_efficiency_metadata)

    _set_thrust_efficiency_metadata = { "offset" : _set_thrust_efficiency_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @thrust_efficiency.setter
    def thrust_efficiency(self, value:float) -> None:
        return self._intf.set_property(ManeuverFinite._metadata, ManeuverFinite._set_thrust_efficiency_metadata, value)

    _get_thrust_efficiency_mode_metadata = { "offset" : _get_thrust_efficiency_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ThrustType),) }
    @property
    def thrust_efficiency_mode(self) -> "ThrustType":
        """Thrust - the calculations that are effected by the thrust efficiency value."""
        return self._intf.get_property(ManeuverFinite._metadata, ManeuverFinite._get_thrust_efficiency_mode_metadata)

    _set_thrust_efficiency_mode_metadata = { "offset" : _set_thrust_efficiency_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ThrustType),) }
    @thrust_efficiency_mode.setter
    def thrust_efficiency_mode(self, value:"ThrustType") -> None:
        return self._intf.set_property(ManeuverFinite._metadata, ManeuverFinite._set_thrust_efficiency_mode_metadata, value)

    _get_propagator_metadata = { "offset" : _get_propagator_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def propagator(self) -> "ManeuverFinitePropagator":
        """Get the propagator."""
        return self._intf.get_property(ManeuverFinite._metadata, ManeuverFinite._get_propagator_metadata)

    _property_names[pressure_mode] = "pressure_mode"
    _property_names[thrust_efficiency] = "thrust_efficiency"
    _property_names[thrust_efficiency_mode] = "thrust_efficiency_mode"
    _property_names[propagator] = "propagator"

    def __init__(self, source_object=None):
        """Construct an object of type ManeuverFinite."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ManeuverFinite)
        IManeuver.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IManeuver._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ManeuverFinite, [ManeuverFinite, IManeuver, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5240610333708177546, 2799450710697962646), ManeuverFinite)
agcls.AgTypeNameMap["ManeuverFinite"] = ManeuverFinite

class ManeuverImpulsive(IManeuver, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for an Impulsive Maneuver."""

    _num_methods = 2
    _vtable_offset = IManeuver._vtable_offset + IManeuver._num_methods
    _get_update_mass_method_offset = 1
    _set_update_mass_method_offset = 2
    _metadata = {
        "iid_data" : (4750637454318981908, 11599245637405498752),
        "vtable_reference" : IManeuver._vtable_offset + IManeuver._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ManeuverImpulsive)

    _get_update_mass_metadata = { "offset" : _get_update_mass_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def update_mass(self) -> bool:
        """True if the mass of the spacecraft will be updated on the basis of fuel usage."""
        return self._intf.get_property(ManeuverImpulsive._metadata, ManeuverImpulsive._get_update_mass_metadata)

    _set_update_mass_metadata = { "offset" : _set_update_mass_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @update_mass.setter
    def update_mass(self, value:bool) -> None:
        return self._intf.set_property(ManeuverImpulsive._metadata, ManeuverImpulsive._set_update_mass_metadata, value)

    _property_names[update_mass] = "update_mass"

    def __init__(self, source_object=None):
        """Construct an object of type ManeuverImpulsive."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ManeuverImpulsive)
        IManeuver.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IManeuver._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ManeuverImpulsive, [ManeuverImpulsive, IManeuver, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5132300925433346245, 9116018589565227669), ManeuverImpulsive)
agcls.AgTypeNameMap["ManeuverImpulsive"] = ManeuverImpulsive

class AttitudeControlImpulsiveVelocityVector(IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Velocity Vector attitude control for an Impulsive Maneuver."""

    _num_methods = 3
    _vtable_offset = IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods
    _get_delta_v_magnitude_method_offset = 1
    _set_delta_v_magnitude_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _metadata = {
        "iid_data" : (5383729676911479635, 15831358368491906978),
        "vtable_reference" : IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlImpulsiveVelocityVector)

    _get_delta_v_magnitude_metadata = { "offset" : _get_delta_v_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delta_v_magnitude(self) -> float:
        """Get or set the size of the delta-V to be applied to the orbit along the velocity vector. Uses Rate Dimension."""
        return self._intf.get_property(AttitudeControlImpulsiveVelocityVector._metadata, AttitudeControlImpulsiveVelocityVector._get_delta_v_magnitude_metadata)

    _set_delta_v_magnitude_metadata = { "offset" : _set_delta_v_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delta_v_magnitude.setter
    def delta_v_magnitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveVelocityVector._metadata, AttitudeControlImpulsiveVelocityVector._set_delta_v_magnitude_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlImpulsiveVelocityVector._metadata, AttitudeControlImpulsiveVelocityVector._get_body_constraint_vector_metadata)

    _property_names[delta_v_magnitude] = "delta_v_magnitude"
    _property_names[body_constraint_vector] = "body_constraint_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlImpulsiveVelocityVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlImpulsiveVelocityVector)
        IAttitudeControlImpulsive.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlImpulsive._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlImpulsiveVelocityVector, [AttitudeControlImpulsiveVelocityVector, IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4884403392612853562, 4746486195757777540), AttitudeControlImpulsiveVelocityVector)
agcls.AgTypeNameMap["AttitudeControlImpulsiveVelocityVector"] = AttitudeControlImpulsiveVelocityVector

class AttitudeControlImpulsiveAntiVelocityVector(IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Anti-Velocity Vector attitude control for an Impulsive Maneuver."""

    _num_methods = 3
    _vtable_offset = IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods
    _get_delta_v_magnitude_method_offset = 1
    _set_delta_v_magnitude_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _metadata = {
        "iid_data" : (5478756088215709330, 17473176637230871450),
        "vtable_reference" : IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlImpulsiveAntiVelocityVector)

    _get_delta_v_magnitude_metadata = { "offset" : _get_delta_v_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delta_v_magnitude(self) -> float:
        """Get or set the size of the delta-V to be applied to the orbit along the velocity vector. Uses Rate Dimension."""
        return self._intf.get_property(AttitudeControlImpulsiveAntiVelocityVector._metadata, AttitudeControlImpulsiveAntiVelocityVector._get_delta_v_magnitude_metadata)

    _set_delta_v_magnitude_metadata = { "offset" : _set_delta_v_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delta_v_magnitude.setter
    def delta_v_magnitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveAntiVelocityVector._metadata, AttitudeControlImpulsiveAntiVelocityVector._set_delta_v_magnitude_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlImpulsiveAntiVelocityVector._metadata, AttitudeControlImpulsiveAntiVelocityVector._get_body_constraint_vector_metadata)

    _property_names[delta_v_magnitude] = "delta_v_magnitude"
    _property_names[body_constraint_vector] = "body_constraint_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlImpulsiveAntiVelocityVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlImpulsiveAntiVelocityVector)
        IAttitudeControlImpulsive.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlImpulsive._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlImpulsiveAntiVelocityVector, [AttitudeControlImpulsiveAntiVelocityVector, IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5209788070296914908, 11656929720675121292), AttitudeControlImpulsiveAntiVelocityVector)
agcls.AgTypeNameMap["AttitudeControlImpulsiveAntiVelocityVector"] = AttitudeControlImpulsiveAntiVelocityVector

class AttitudeControlImpulsiveAttitude(IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Attitude attitude control for an Impulsive Maneuver."""

    _num_methods = 5
    _vtable_offset = IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods
    _get_delta_v_magnitude_method_offset = 1
    _set_delta_v_magnitude_method_offset = 2
    _get_reference_axes_name_method_offset = 3
    _set_reference_axes_name_method_offset = 4
    _get_orientation_method_offset = 5
    _metadata = {
        "iid_data" : (5725077979125730187, 16239998713400669345),
        "vtable_reference" : IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlImpulsiveAttitude)

    _get_delta_v_magnitude_metadata = { "offset" : _get_delta_v_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delta_v_magnitude(self) -> float:
        """Get or set the size of the delta-V to be applied along the X axis - as defined by the selected axes and rotation. Uses Rate Dimension."""
        return self._intf.get_property(AttitudeControlImpulsiveAttitude._metadata, AttitudeControlImpulsiveAttitude._get_delta_v_magnitude_metadata)

    _set_delta_v_magnitude_metadata = { "offset" : _set_delta_v_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delta_v_magnitude.setter
    def delta_v_magnitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveAttitude._metadata, AttitudeControlImpulsiveAttitude._set_delta_v_magnitude_metadata, value)

    _get_reference_axes_name_metadata = { "offset" : _get_reference_axes_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def reference_axes_name(self) -> str:
        """Ref Axes - the reference axes to be used in modeling this maneuver."""
        return self._intf.get_property(AttitudeControlImpulsiveAttitude._metadata, AttitudeControlImpulsiveAttitude._get_reference_axes_name_metadata)

    _set_reference_axes_name_metadata = { "offset" : _set_reference_axes_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @reference_axes_name.setter
    def reference_axes_name(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveAttitude._metadata, AttitudeControlImpulsiveAttitude._set_reference_axes_name_metadata, value)

    _get_orientation_metadata = { "offset" : _get_orientation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def orientation(self) -> "IOrientation":
        """Get the orientation of the attitude."""
        return self._intf.get_property(AttitudeControlImpulsiveAttitude._metadata, AttitudeControlImpulsiveAttitude._get_orientation_metadata)

    _property_names[delta_v_magnitude] = "delta_v_magnitude"
    _property_names[reference_axes_name] = "reference_axes_name"
    _property_names[orientation] = "orientation"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlImpulsiveAttitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlImpulsiveAttitude)
        IAttitudeControlImpulsive.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlImpulsive._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlImpulsiveAttitude, [AttitudeControlImpulsiveAttitude, IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5745604571369372982, 2700112517730179510), AttitudeControlImpulsiveAttitude)
agcls.AgTypeNameMap["AttitudeControlImpulsiveAttitude"] = AttitudeControlImpulsiveAttitude

class AttitudeControlImpulsiveFile(IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the File attitude control for an Impulsive Maneuver."""

    _num_methods = 7
    _vtable_offset = IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods
    _get_delta_v_magnitude_method_offset = 1
    _set_delta_v_magnitude_method_offset = 2
    _get_filename_method_offset = 3
    _set_filename_method_offset = 4
    _get_file_time_offset_method_offset = 5
    _set_file_time_offset_method_offset = 6
    _get_full_filename_method_offset = 7
    _metadata = {
        "iid_data" : (4902519617822865926, 15504321543168614062),
        "vtable_reference" : IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlImpulsiveFile)

    _get_delta_v_magnitude_metadata = { "offset" : _get_delta_v_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def delta_v_magnitude(self) -> float:
        """Get or set the size of the delta-V to be applied to the orbit along the specified direction. Uses Rate Dimension."""
        return self._intf.get_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._get_delta_v_magnitude_metadata)

    _set_delta_v_magnitude_metadata = { "offset" : _set_delta_v_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @delta_v_magnitude.setter
    def delta_v_magnitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._set_delta_v_magnitude_metadata, value)

    _get_filename_metadata = { "offset" : _get_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def filename(self) -> str:
        """Get or set the attitude file to use."""
        return self._intf.get_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._get_filename_metadata)

    _set_filename_metadata = { "offset" : _set_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @filename.setter
    def filename(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._set_filename_metadata, value)

    _get_file_time_offset_metadata = { "offset" : _get_file_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def file_time_offset(self) -> float:
        """Get or set the time offset can be used to adjust the time stored in the attitude file. Dimensionless."""
        return self._intf.get_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._get_file_time_offset_metadata)

    _set_file_time_offset_metadata = { "offset" : _set_file_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @file_time_offset.setter
    def file_time_offset(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._set_file_time_offset_metadata, value)

    _get_full_filename_metadata = { "offset" : _get_full_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def full_filename(self) -> str:
        """Get the full path and name of the attitude file to use."""
        return self._intf.get_property(AttitudeControlImpulsiveFile._metadata, AttitudeControlImpulsiveFile._get_full_filename_metadata)

    _property_names[delta_v_magnitude] = "delta_v_magnitude"
    _property_names[filename] = "filename"
    _property_names[file_time_offset] = "file_time_offset"
    _property_names[full_filename] = "full_filename"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlImpulsiveFile."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlImpulsiveFile)
        IAttitudeControlImpulsive.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlImpulsive._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlImpulsiveFile, [AttitudeControlImpulsiveFile, IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5004932378358409643, 12572056464618414249), AttitudeControlImpulsiveFile)
agcls.AgTypeNameMap["AttitudeControlImpulsiveFile"] = AttitudeControlImpulsiveFile

class AttitudeControlImpulsiveThrustVector(IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Thrust Vector attitude control for an Impulsive Maneuver."""

    _num_methods = 23
    _vtable_offset = IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods
    _get_thrust_axes_name_method_offset = 1
    _set_thrust_axes_name_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _get_allow_negative_spherical_magnitude_method_offset = 4
    _set_allow_negative_spherical_magnitude_method_offset = 5
    _get_coord_type_method_offset = 6
    _set_coord_type_method_offset = 7
    _get_x_method_offset = 8
    _set_x_method_offset = 9
    _get_y_method_offset = 10
    _set_y_method_offset = 11
    _get_z_method_offset = 12
    _set_z_method_offset = 13
    _get_azimuth_method_offset = 14
    _set_azimuth_method_offset = 15
    _get_elevation_method_offset = 16
    _set_elevation_method_offset = 17
    _get_magnitude_method_offset = 18
    _set_magnitude_method_offset = 19
    _assign_cartesian_method_offset = 20
    _query_cartesian_method_offset = 21
    _assign_spherical_method_offset = 22
    _query_spherical_method_offset = 23
    _metadata = {
        "iid_data" : (5502906477986733323, 9761518986136370305),
        "vtable_reference" : IAttitudeControlImpulsive._vtable_offset + IAttitudeControlImpulsive._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlImpulsiveThrustVector)

    _get_thrust_axes_name_metadata = { "offset" : _get_thrust_axes_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def thrust_axes_name(self) -> str:
        """Get or set the thrust axes."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_thrust_axes_name_metadata)

    _set_thrust_axes_name_metadata = { "offset" : _set_thrust_axes_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @thrust_axes_name.setter
    def thrust_axes_name(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_thrust_axes_name_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_body_constraint_vector_metadata)

    _get_allow_negative_spherical_magnitude_metadata = { "offset" : _get_allow_negative_spherical_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def allow_negative_spherical_magnitude(self) -> bool:
        """True if the spherical magnitude should be allowed to be less than zero."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_allow_negative_spherical_magnitude_metadata)

    _set_allow_negative_spherical_magnitude_metadata = { "offset" : _set_allow_negative_spherical_magnitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @allow_negative_spherical_magnitude.setter
    def allow_negative_spherical_magnitude(self, value:bool) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_allow_negative_spherical_magnitude_metadata, value)

    _get_coord_type_metadata = { "offset" : _get_coord_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ImpulsiveDeltaVRepresentation),) }
    @property
    def coord_type(self) -> "ImpulsiveDeltaVRepresentation":
        """Get or set the coordinate representation for the DeltaV."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_coord_type_metadata)

    _set_coord_type_metadata = { "offset" : _set_coord_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ImpulsiveDeltaVRepresentation),) }
    @coord_type.setter
    def coord_type(self, value:"ImpulsiveDeltaVRepresentation") -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_coord_type_metadata, value)

    _get_x_metadata = { "offset" : _get_x_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def x(self) -> float:
        """Get or set the Cartesian X component of the impulsive DeltaV (dimension: SmallDistance/Time)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_x_metadata)

    _set_x_metadata = { "offset" : _set_x_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @x.setter
    def x(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_x_metadata, value)

    _get_y_metadata = { "offset" : _get_y_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def y(self) -> float:
        """Get or set the Cartesian Y component of the impulsive DeltaV (dimension: SmallDistance/Time)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_y_metadata)

    _set_y_metadata = { "offset" : _set_y_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @y.setter
    def y(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_y_metadata, value)

    _get_z_metadata = { "offset" : _get_z_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def z(self) -> float:
        """Get or set the Cartesian Z component of the impulsive DeltaV (dimension: SmallDistance/Time)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_z_metadata)

    _set_z_metadata = { "offset" : _set_z_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @z.setter
    def z(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_z_metadata, value)

    _get_azimuth_metadata = { "offset" : _get_azimuth_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def azimuth(self) -> typing.Any:
        """Get or set the spherical Azimuth angle of the impulsive DeltaV (dimension: Angle)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_azimuth_metadata)

    _set_azimuth_metadata = { "offset" : _set_azimuth_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @azimuth.setter
    def azimuth(self, value:typing.Any) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_azimuth_metadata, value)

    _get_elevation_metadata = { "offset" : _get_elevation_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def elevation(self) -> typing.Any:
        """Get or set the spherical Elevation angle of the impulsive DeltaV (dimension: Angle)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_elevation_metadata)

    _set_elevation_metadata = { "offset" : _set_elevation_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @elevation.setter
    def elevation(self, value:typing.Any) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_elevation_metadata, value)

    _get_magnitude_metadata = { "offset" : _get_magnitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def magnitude(self) -> float:
        """Get or set the spherical Magnitude of the impulsive DeltaV (dimension: SmallDistance/Time)."""
        return self._intf.get_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._get_magnitude_metadata)

    _set_magnitude_metadata = { "offset" : _set_magnitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @magnitude.setter
    def magnitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._set_magnitude_metadata, value)

    _assign_cartesian_metadata = { "offset" : _assign_cartesian_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def assign_cartesian(self, x_value:float, y_value:float, z_value:float) -> None:
        """Assign all three Cartesian components of the DeltaV vector (dimension: SmallDistance/Time)."""
        return self._intf.invoke(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._assign_cartesian_metadata, x_value, y_value, z_value)

    _query_cartesian_metadata = { "offset" : _query_cartesian_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def query_cartesian(self) -> list:
        """Get all three Cartesian components of the DeltaV vector as an array (dimension: SmallDistance/Time)."""
        return self._intf.invoke(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._query_cartesian_metadata, OutArg())

    _assign_spherical_metadata = { "offset" : _assign_spherical_method_offset,
            "arg_types" : (agcom.Variant, agcom.Variant, agcom.DOUBLE,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantArg, agmarshall.DoubleArg,) }
    def assign_spherical(self, azimuth_value:typing.Any, elevation_value:typing.Any, magnitude_value:float) -> None:
        """Assign all three spherical components of the DeltaV vector (order Az, El, Mag; dimensions: Angle, Angle, SmallDistance/Time)."""
        return self._intf.invoke(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._assign_spherical_metadata, azimuth_value, elevation_value, magnitude_value)

    _query_spherical_metadata = { "offset" : _query_spherical_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def query_spherical(self) -> list:
        """Get all three spherical components of the DeltaV vector as an array (order Az, El, Mag; dimensions: Angle, Angle, SmallDistance/Time)."""
        return self._intf.invoke(AttitudeControlImpulsiveThrustVector._metadata, AttitudeControlImpulsiveThrustVector._query_spherical_metadata, OutArg())

    _property_names[thrust_axes_name] = "thrust_axes_name"
    _property_names[body_constraint_vector] = "body_constraint_vector"
    _property_names[allow_negative_spherical_magnitude] = "allow_negative_spherical_magnitude"
    _property_names[coord_type] = "coord_type"
    _property_names[x] = "x"
    _property_names[y] = "y"
    _property_names[z] = "z"
    _property_names[azimuth] = "azimuth"
    _property_names[elevation] = "elevation"
    _property_names[magnitude] = "magnitude"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlImpulsiveThrustVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlImpulsiveThrustVector)
        IAttitudeControlImpulsive.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlImpulsive._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlImpulsiveThrustVector, [AttitudeControlImpulsiveThrustVector, IAttitudeControlImpulsive, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5575491667768589580, 12658580146838550400), AttitudeControlImpulsiveThrustVector)
agcls.AgTypeNameMap["AttitudeControlImpulsiveThrustVector"] = AttitudeControlImpulsiveThrustVector

class AttitudeControlFiniteAntiVelocityVector(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Anti-Velocity Vector attitude control for a Finite Maneuver."""

    _num_methods = 3
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_attitude_update_method_offset = 1
    _set_attitude_update_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _metadata = {
        "iid_data" : (4782013786436564730, 1569290256608416660),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteAntiVelocityVector)

    _get_attitude_update_metadata = { "offset" : _get_attitude_update_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @property
    def attitude_update(self) -> "AttitudeUpdate":
        """How and when the attitude will be updated."""
        return self._intf.get_property(AttitudeControlFiniteAntiVelocityVector._metadata, AttitudeControlFiniteAntiVelocityVector._get_attitude_update_metadata)

    _set_attitude_update_metadata = { "offset" : _set_attitude_update_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @attitude_update.setter
    def attitude_update(self, value:"AttitudeUpdate") -> None:
        return self._intf.set_property(AttitudeControlFiniteAntiVelocityVector._metadata, AttitudeControlFiniteAntiVelocityVector._set_attitude_update_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlFiniteAntiVelocityVector._metadata, AttitudeControlFiniteAntiVelocityVector._get_body_constraint_vector_metadata)

    _property_names[attitude_update] = "attitude_update"
    _property_names[body_constraint_vector] = "body_constraint_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteAntiVelocityVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteAntiVelocityVector)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteAntiVelocityVector, [AttitudeControlFiniteAntiVelocityVector, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4929934656189608976, 9775493696746436502), AttitudeControlFiniteAntiVelocityVector)
agcls.AgTypeNameMap["AttitudeControlFiniteAntiVelocityVector"] = AttitudeControlFiniteAntiVelocityVector

class AttitudeControlFiniteAttitude(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Attitude attitude control for a Finite Maneuver."""

    _num_methods = 5
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_attitude_update_method_offset = 1
    _set_attitude_update_method_offset = 2
    _get_reference_axes_name_method_offset = 3
    _set_reference_axes_name_method_offset = 4
    _get_orientation_method_offset = 5
    _metadata = {
        "iid_data" : (4631154172073222433, 9787163487429423038),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteAttitude)

    _get_attitude_update_metadata = { "offset" : _get_attitude_update_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @property
    def attitude_update(self) -> "AttitudeUpdate":
        """How and when the attitude will be updated."""
        return self._intf.get_property(AttitudeControlFiniteAttitude._metadata, AttitudeControlFiniteAttitude._get_attitude_update_metadata)

    _set_attitude_update_metadata = { "offset" : _set_attitude_update_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @attitude_update.setter
    def attitude_update(self, value:"AttitudeUpdate") -> None:
        return self._intf.set_property(AttitudeControlFiniteAttitude._metadata, AttitudeControlFiniteAttitude._set_attitude_update_metadata, value)

    _get_reference_axes_name_metadata = { "offset" : _get_reference_axes_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def reference_axes_name(self) -> str:
        """Ref Axes - the reference axes to be used in modeling this maneuver."""
        return self._intf.get_property(AttitudeControlFiniteAttitude._metadata, AttitudeControlFiniteAttitude._get_reference_axes_name_metadata)

    _set_reference_axes_name_metadata = { "offset" : _set_reference_axes_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @reference_axes_name.setter
    def reference_axes_name(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlFiniteAttitude._metadata, AttitudeControlFiniteAttitude._set_reference_axes_name_metadata, value)

    _get_orientation_metadata = { "offset" : _get_orientation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def orientation(self) -> "IOrientation":
        """Get the orientation of the attitude."""
        return self._intf.get_property(AttitudeControlFiniteAttitude._metadata, AttitudeControlFiniteAttitude._get_orientation_metadata)

    _property_names[attitude_update] = "attitude_update"
    _property_names[reference_axes_name] = "reference_axes_name"
    _property_names[orientation] = "orientation"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteAttitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteAttitude)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteAttitude, [AttitudeControlFiniteAttitude, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5543774537829345684, 6248581389650464156), AttitudeControlFiniteAttitude)
agcls.AgTypeNameMap["AttitudeControlFiniteAttitude"] = AttitudeControlFiniteAttitude

class AttitudeControlFiniteFile(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the File attitude control for a Finite Maneuver."""

    _num_methods = 5
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_filename_method_offset = 1
    _set_filename_method_offset = 2
    _get_file_time_offset_method_offset = 3
    _set_file_time_offset_method_offset = 4
    _get_full_filename_method_offset = 5
    _metadata = {
        "iid_data" : (5395533152266767476, 10947136709492955554),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteFile)

    _get_filename_metadata = { "offset" : _get_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def filename(self) -> str:
        """Get or set the attitude file to use."""
        return self._intf.get_property(AttitudeControlFiniteFile._metadata, AttitudeControlFiniteFile._get_filename_metadata)

    _set_filename_metadata = { "offset" : _set_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @filename.setter
    def filename(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlFiniteFile._metadata, AttitudeControlFiniteFile._set_filename_metadata, value)

    _get_file_time_offset_metadata = { "offset" : _get_file_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def file_time_offset(self) -> float:
        """Get or set the time offset can be used to adjust the time stored in the attitude file. Dimensionless."""
        return self._intf.get_property(AttitudeControlFiniteFile._metadata, AttitudeControlFiniteFile._get_file_time_offset_metadata)

    _set_file_time_offset_metadata = { "offset" : _set_file_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @file_time_offset.setter
    def file_time_offset(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteFile._metadata, AttitudeControlFiniteFile._set_file_time_offset_metadata, value)

    _get_full_filename_metadata = { "offset" : _get_full_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def full_filename(self) -> str:
        """Get the full path and name of the attitude file to use."""
        return self._intf.get_property(AttitudeControlFiniteFile._metadata, AttitudeControlFiniteFile._get_full_filename_metadata)

    _property_names[filename] = "filename"
    _property_names[file_time_offset] = "file_time_offset"
    _property_names[full_filename] = "full_filename"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteFile."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteFile)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteFile, [AttitudeControlFiniteFile, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5304200409859175388, 7105606624544560287), AttitudeControlFiniteFile)
agcls.AgTypeNameMap["AttitudeControlFiniteFile"] = AttitudeControlFiniteFile

class AttitudeControlFiniteThrustVector(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Thrust Vector attitude control for a Finite Maneuver."""

    _num_methods = 6
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_attitude_update_method_offset = 1
    _set_attitude_update_method_offset = 2
    _get_thrust_axes_name_method_offset = 3
    _set_thrust_axes_name_method_offset = 4
    _get_body_constraint_vector_method_offset = 5
    _get_thrust_vector_method_offset = 6
    _metadata = {
        "iid_data" : (5148522151278071898, 11453760759581062313),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteThrustVector)

    _get_attitude_update_metadata = { "offset" : _get_attitude_update_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @property
    def attitude_update(self) -> "AttitudeUpdate":
        """How and when the attitude will be updated."""
        return self._intf.get_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._get_attitude_update_metadata)

    _set_attitude_update_metadata = { "offset" : _set_attitude_update_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @attitude_update.setter
    def attitude_update(self, value:"AttitudeUpdate") -> None:
        return self._intf.set_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._set_attitude_update_metadata, value)

    _get_thrust_axes_name_metadata = { "offset" : _get_thrust_axes_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def thrust_axes_name(self) -> str:
        """Get or set the thrust axes."""
        return self._intf.get_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._get_thrust_axes_name_metadata)

    _set_thrust_axes_name_metadata = { "offset" : _set_thrust_axes_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @thrust_axes_name.setter
    def thrust_axes_name(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._set_thrust_axes_name_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._get_body_constraint_vector_metadata)

    _get_thrust_vector_metadata = { "offset" : _get_thrust_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def thrust_vector(self) -> "IDirection":
        """Define the thrust vector in the reference axes."""
        return self._intf.get_property(AttitudeControlFiniteThrustVector._metadata, AttitudeControlFiniteThrustVector._get_thrust_vector_metadata)

    _property_names[attitude_update] = "attitude_update"
    _property_names[thrust_axes_name] = "thrust_axes_name"
    _property_names[body_constraint_vector] = "body_constraint_vector"
    _property_names[thrust_vector] = "thrust_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteThrustVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteThrustVector)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteThrustVector, [AttitudeControlFiniteThrustVector, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5300112662321021295, 17392290777368689581), AttitudeControlFiniteThrustVector)
agcls.AgTypeNameMap["AttitudeControlFiniteThrustVector"] = AttitudeControlFiniteThrustVector

class AttitudeControlFiniteTimeVarying(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Time Varying attitude control for a Finite Maneuver."""

    _num_methods = 35
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_thrust_axes_name_method_offset = 1
    _set_thrust_axes_name_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _get_azimuth_polynomial_constant_term_method_offset = 4
    _set_azimuth_polynomial_constant_term_method_offset = 5
    _get_azimuth_polynomial_linear_term_method_offset = 6
    _set_azimuth_polynomial_linear_term_method_offset = 7
    _get_azimuth_polynomial_quadratic_term_method_offset = 8
    _set_azimuth_polynomial_quadratic_term_method_offset = 9
    _get_azimuth_polynomial_cubic_term_method_offset = 10
    _set_azimuth_polynomial_cubic_term_method_offset = 11
    _get_azimuth_polynomial_quartic_term_method_offset = 12
    _set_azimuth_polynomial_quartic_term_method_offset = 13
    _get_azimuth_sinusoidal_amplitude_method_offset = 14
    _set_azimuth_sinusoidal_amplitude_method_offset = 15
    _get_azimuth_sinusoidal_frequency_method_offset = 16
    _set_azimuth_sinusoidal_frequency_method_offset = 17
    _get_azimuth_sinusoidal_phase_method_offset = 18
    _set_azimuth_sinusoidal_phase_method_offset = 19
    _get_elevation_polynomial_constant_term_method_offset = 20
    _set_elevation_polynomial_constant_term_method_offset = 21
    _get_elevation_polynomial_linear_term_method_offset = 22
    _set_elevation_polynomial_linear_term_method_offset = 23
    _get_elevation_polynomial_quadratic_term_method_offset = 24
    _set_elevation_polynomial_quadratic_term_method_offset = 25
    _get_elevation_polynomial_cubic_term_method_offset = 26
    _set_elevation_polynomial_cubic_term_method_offset = 27
    _get_elevation_polynomial_quartic_term_method_offset = 28
    _set_elevation_polynomial_quartic_term_method_offset = 29
    _get_elevation_sinusoidal_amplitude_method_offset = 30
    _set_elevation_sinusoidal_amplitude_method_offset = 31
    _get_elevation_sinusoidal_frequency_method_offset = 32
    _set_elevation_sinusoidal_frequency_method_offset = 33
    _get_elevation_sinusoidal_phase_method_offset = 34
    _set_elevation_sinusoidal_phase_method_offset = 35
    _metadata = {
        "iid_data" : (5634020525553782637, 14867654061630199955),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteTimeVarying)

    _get_thrust_axes_name_metadata = { "offset" : _get_thrust_axes_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def thrust_axes_name(self) -> str:
        """Get or set the thrust axes."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_thrust_axes_name_metadata)

    _set_thrust_axes_name_metadata = { "offset" : _set_thrust_axes_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @thrust_axes_name.setter
    def thrust_axes_name(self, value:str) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_thrust_axes_name_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_body_constraint_vector_metadata)

    _get_azimuth_polynomial_constant_term_metadata = { "offset" : _get_azimuth_polynomial_constant_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_polynomial_constant_term(self) -> float:
        """Azimuth constant term (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_polynomial_constant_term_metadata)

    _set_azimuth_polynomial_constant_term_metadata = { "offset" : _set_azimuth_polynomial_constant_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_polynomial_constant_term.setter
    def azimuth_polynomial_constant_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_polynomial_constant_term_metadata, value)

    _get_azimuth_polynomial_linear_term_metadata = { "offset" : _get_azimuth_polynomial_linear_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_polynomial_linear_term(self) -> float:
        """Azimuth linear term (dimension: angle/time)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_polynomial_linear_term_metadata)

    _set_azimuth_polynomial_linear_term_metadata = { "offset" : _set_azimuth_polynomial_linear_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_polynomial_linear_term.setter
    def azimuth_polynomial_linear_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_polynomial_linear_term_metadata, value)

    _get_azimuth_polynomial_quadratic_term_metadata = { "offset" : _get_azimuth_polynomial_quadratic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_polynomial_quadratic_term(self) -> float:
        """Azimuth quadratic term (dimension: angle/time^2)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_polynomial_quadratic_term_metadata)

    _set_azimuth_polynomial_quadratic_term_metadata = { "offset" : _set_azimuth_polynomial_quadratic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_polynomial_quadratic_term.setter
    def azimuth_polynomial_quadratic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_polynomial_quadratic_term_metadata, value)

    _get_azimuth_polynomial_cubic_term_metadata = { "offset" : _get_azimuth_polynomial_cubic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_polynomial_cubic_term(self) -> float:
        """Azimuth cubic term (dimension: angle/time^3)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_polynomial_cubic_term_metadata)

    _set_azimuth_polynomial_cubic_term_metadata = { "offset" : _set_azimuth_polynomial_cubic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_polynomial_cubic_term.setter
    def azimuth_polynomial_cubic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_polynomial_cubic_term_metadata, value)

    _get_azimuth_polynomial_quartic_term_metadata = { "offset" : _get_azimuth_polynomial_quartic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_polynomial_quartic_term(self) -> float:
        """Azimuth quartic term (dimension: angle/time^4)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_polynomial_quartic_term_metadata)

    _set_azimuth_polynomial_quartic_term_metadata = { "offset" : _set_azimuth_polynomial_quartic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_polynomial_quartic_term.setter
    def azimuth_polynomial_quartic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_polynomial_quartic_term_metadata, value)

    _get_azimuth_sinusoidal_amplitude_metadata = { "offset" : _get_azimuth_sinusoidal_amplitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_sinusoidal_amplitude(self) -> float:
        """Azimuth sine term amplitude (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_sinusoidal_amplitude_metadata)

    _set_azimuth_sinusoidal_amplitude_metadata = { "offset" : _set_azimuth_sinusoidal_amplitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_sinusoidal_amplitude.setter
    def azimuth_sinusoidal_amplitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_sinusoidal_amplitude_metadata, value)

    _get_azimuth_sinusoidal_frequency_metadata = { "offset" : _get_azimuth_sinusoidal_frequency_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_sinusoidal_frequency(self) -> float:
        """Azimuth sine term frequency (dimension: angle/time)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_sinusoidal_frequency_metadata)

    _set_azimuth_sinusoidal_frequency_metadata = { "offset" : _set_azimuth_sinusoidal_frequency_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_sinusoidal_frequency.setter
    def azimuth_sinusoidal_frequency(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_sinusoidal_frequency_metadata, value)

    _get_azimuth_sinusoidal_phase_metadata = { "offset" : _get_azimuth_sinusoidal_phase_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def azimuth_sinusoidal_phase(self) -> float:
        """Azimuth sine term phase (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_azimuth_sinusoidal_phase_metadata)

    _set_azimuth_sinusoidal_phase_metadata = { "offset" : _set_azimuth_sinusoidal_phase_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @azimuth_sinusoidal_phase.setter
    def azimuth_sinusoidal_phase(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_azimuth_sinusoidal_phase_metadata, value)

    _get_elevation_polynomial_constant_term_metadata = { "offset" : _get_elevation_polynomial_constant_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_polynomial_constant_term(self) -> float:
        """Elevation constant term (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_polynomial_constant_term_metadata)

    _set_elevation_polynomial_constant_term_metadata = { "offset" : _set_elevation_polynomial_constant_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_polynomial_constant_term.setter
    def elevation_polynomial_constant_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_polynomial_constant_term_metadata, value)

    _get_elevation_polynomial_linear_term_metadata = { "offset" : _get_elevation_polynomial_linear_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_polynomial_linear_term(self) -> float:
        """Elevation linear term (dimension: angle/time)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_polynomial_linear_term_metadata)

    _set_elevation_polynomial_linear_term_metadata = { "offset" : _set_elevation_polynomial_linear_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_polynomial_linear_term.setter
    def elevation_polynomial_linear_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_polynomial_linear_term_metadata, value)

    _get_elevation_polynomial_quadratic_term_metadata = { "offset" : _get_elevation_polynomial_quadratic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_polynomial_quadratic_term(self) -> float:
        """Elevation quadratic term (dimension: angle/time^2)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_polynomial_quadratic_term_metadata)

    _set_elevation_polynomial_quadratic_term_metadata = { "offset" : _set_elevation_polynomial_quadratic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_polynomial_quadratic_term.setter
    def elevation_polynomial_quadratic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_polynomial_quadratic_term_metadata, value)

    _get_elevation_polynomial_cubic_term_metadata = { "offset" : _get_elevation_polynomial_cubic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_polynomial_cubic_term(self) -> float:
        """Elevation cubic term (dimension: angle/time^3)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_polynomial_cubic_term_metadata)

    _set_elevation_polynomial_cubic_term_metadata = { "offset" : _set_elevation_polynomial_cubic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_polynomial_cubic_term.setter
    def elevation_polynomial_cubic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_polynomial_cubic_term_metadata, value)

    _get_elevation_polynomial_quartic_term_metadata = { "offset" : _get_elevation_polynomial_quartic_term_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_polynomial_quartic_term(self) -> float:
        """Elevation quartic term (dimension: angle/time^4)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_polynomial_quartic_term_metadata)

    _set_elevation_polynomial_quartic_term_metadata = { "offset" : _set_elevation_polynomial_quartic_term_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_polynomial_quartic_term.setter
    def elevation_polynomial_quartic_term(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_polynomial_quartic_term_metadata, value)

    _get_elevation_sinusoidal_amplitude_metadata = { "offset" : _get_elevation_sinusoidal_amplitude_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_sinusoidal_amplitude(self) -> float:
        """Elevation sine term amplitude (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_sinusoidal_amplitude_metadata)

    _set_elevation_sinusoidal_amplitude_metadata = { "offset" : _set_elevation_sinusoidal_amplitude_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_sinusoidal_amplitude.setter
    def elevation_sinusoidal_amplitude(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_sinusoidal_amplitude_metadata, value)

    _get_elevation_sinusoidal_frequency_metadata = { "offset" : _get_elevation_sinusoidal_frequency_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_sinusoidal_frequency(self) -> float:
        """Elevation sine term frequency (dimension: angle/time)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_sinusoidal_frequency_metadata)

    _set_elevation_sinusoidal_frequency_metadata = { "offset" : _set_elevation_sinusoidal_frequency_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_sinusoidal_frequency.setter
    def elevation_sinusoidal_frequency(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_sinusoidal_frequency_metadata, value)

    _get_elevation_sinusoidal_phase_metadata = { "offset" : _get_elevation_sinusoidal_phase_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def elevation_sinusoidal_phase(self) -> float:
        """Elevation sine term phase (dimension: angle)."""
        return self._intf.get_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._get_elevation_sinusoidal_phase_metadata)

    _set_elevation_sinusoidal_phase_metadata = { "offset" : _set_elevation_sinusoidal_phase_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @elevation_sinusoidal_phase.setter
    def elevation_sinusoidal_phase(self, value:float) -> None:
        return self._intf.set_property(AttitudeControlFiniteTimeVarying._metadata, AttitudeControlFiniteTimeVarying._set_elevation_sinusoidal_phase_metadata, value)

    _property_names[thrust_axes_name] = "thrust_axes_name"
    _property_names[body_constraint_vector] = "body_constraint_vector"
    _property_names[azimuth_polynomial_constant_term] = "azimuth_polynomial_constant_term"
    _property_names[azimuth_polynomial_linear_term] = "azimuth_polynomial_linear_term"
    _property_names[azimuth_polynomial_quadratic_term] = "azimuth_polynomial_quadratic_term"
    _property_names[azimuth_polynomial_cubic_term] = "azimuth_polynomial_cubic_term"
    _property_names[azimuth_polynomial_quartic_term] = "azimuth_polynomial_quartic_term"
    _property_names[azimuth_sinusoidal_amplitude] = "azimuth_sinusoidal_amplitude"
    _property_names[azimuth_sinusoidal_frequency] = "azimuth_sinusoidal_frequency"
    _property_names[azimuth_sinusoidal_phase] = "azimuth_sinusoidal_phase"
    _property_names[elevation_polynomial_constant_term] = "elevation_polynomial_constant_term"
    _property_names[elevation_polynomial_linear_term] = "elevation_polynomial_linear_term"
    _property_names[elevation_polynomial_quadratic_term] = "elevation_polynomial_quadratic_term"
    _property_names[elevation_polynomial_cubic_term] = "elevation_polynomial_cubic_term"
    _property_names[elevation_polynomial_quartic_term] = "elevation_polynomial_quartic_term"
    _property_names[elevation_sinusoidal_amplitude] = "elevation_sinusoidal_amplitude"
    _property_names[elevation_sinusoidal_frequency] = "elevation_sinusoidal_frequency"
    _property_names[elevation_sinusoidal_phase] = "elevation_sinusoidal_phase"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteTimeVarying."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteTimeVarying)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteTimeVarying, [AttitudeControlFiniteTimeVarying, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4702333051035569318, 8055730207611007421), AttitudeControlFiniteTimeVarying)
agcls.AgTypeNameMap["AttitudeControlFiniteTimeVarying"] = AttitudeControlFiniteTimeVarying

class AttitudeControlFiniteVelocityVector(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Velocity Vector attitude control for a Finite Maneuver."""

    _num_methods = 3
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _get_attitude_update_method_offset = 1
    _set_attitude_update_method_offset = 2
    _get_body_constraint_vector_method_offset = 3
    _metadata = {
        "iid_data" : (5137071414965646431, 4057671418243972257),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFiniteVelocityVector)

    _get_attitude_update_metadata = { "offset" : _get_attitude_update_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @property
    def attitude_update(self) -> "AttitudeUpdate":
        """How and when the attitude will be updated."""
        return self._intf.get_property(AttitudeControlFiniteVelocityVector._metadata, AttitudeControlFiniteVelocityVector._get_attitude_update_metadata)

    _set_attitude_update_metadata = { "offset" : _set_attitude_update_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AttitudeUpdate),) }
    @attitude_update.setter
    def attitude_update(self, value:"AttitudeUpdate") -> None:
        return self._intf.set_property(AttitudeControlFiniteVelocityVector._metadata, AttitudeControlFiniteVelocityVector._set_attitude_update_metadata, value)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlFiniteVelocityVector._metadata, AttitudeControlFiniteVelocityVector._get_body_constraint_vector_metadata)

    _property_names[attitude_update] = "attitude_update"
    _property_names[body_constraint_vector] = "body_constraint_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFiniteVelocityVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFiniteVelocityVector)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFiniteVelocityVector, [AttitudeControlFiniteVelocityVector, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5340603337482853225, 4608082530829545350), AttitudeControlFiniteVelocityVector)
agcls.AgTypeNameMap["AttitudeControlFiniteVelocityVector"] = AttitudeControlFiniteVelocityVector

class AttitudeControlFinitePlugin(IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Plugin attitude control for a Finite Maneuver."""

    _num_methods = 3
    _vtable_offset = IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods
    _select_plugin_by_name_method_offset = 1
    _get_plugin_name_method_offset = 2
    _get_plugin_config_method_offset = 3
    _metadata = {
        "iid_data" : (4773268931231490560, 6457243350605167235),
        "vtable_reference" : IAttitudeControlFinite._vtable_offset + IAttitudeControlFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlFinitePlugin)

    _select_plugin_by_name_metadata = { "offset" : _select_plugin_by_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def select_plugin_by_name(self, name:str) -> None:
        """Select plugin using the plugin name."""
        return self._intf.invoke(AttitudeControlFinitePlugin._metadata, AttitudeControlFinitePlugin._select_plugin_by_name_metadata, name)

    _get_plugin_name_metadata = { "offset" : _get_plugin_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def plugin_name(self) -> str:
        """Get the selected plugin's name."""
        return self._intf.get_property(AttitudeControlFinitePlugin._metadata, AttitudeControlFinitePlugin._get_plugin_name_metadata)

    _get_plugin_config_metadata = { "offset" : _get_plugin_config_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def plugin_config(self) -> "PluginProperties":
        """Get the properties of the selected plugin."""
        return self._intf.get_property(AttitudeControlFinitePlugin._metadata, AttitudeControlFinitePlugin._get_plugin_config_metadata)

    _property_names[plugin_name] = "plugin_name"
    _property_names[plugin_config] = "plugin_config"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlFinitePlugin."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlFinitePlugin)
        IAttitudeControlFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlFinitePlugin, [AttitudeControlFinitePlugin, IAttitudeControlFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5751876833050268726, 10430523435175771071), AttitudeControlFinitePlugin)
agcls.AgTypeNameMap["AttitudeControlFinitePlugin"] = AttitudeControlFinitePlugin

class AttitudeControlOptimalFiniteLagrange(IAttitudeControlOptimalFinite, IAttitudeControl, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the Lagrange Interpolation attitude control for a Optimal Finite Maneuver."""

    _num_methods = 1
    _vtable_offset = IAttitudeControlOptimalFinite._vtable_offset + IAttitudeControlOptimalFinite._num_methods
    _get_body_constraint_vector_method_offset = 1
    _metadata = {
        "iid_data" : (5193424269214069960, 1358925997096785086),
        "vtable_reference" : IAttitudeControlOptimalFinite._vtable_offset + IAttitudeControlOptimalFinite._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, AttitudeControlOptimalFiniteLagrange)

    _get_body_constraint_vector_metadata = { "offset" : _get_body_constraint_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def body_constraint_vector(self) -> "IDirection":
        """Define a constraint vector in spacecraft body coordinates to complete the attitude definition."""
        return self._intf.get_property(AttitudeControlOptimalFiniteLagrange._metadata, AttitudeControlOptimalFiniteLagrange._get_body_constraint_vector_metadata)

    _property_names[body_constraint_vector] = "body_constraint_vector"

    def __init__(self, source_object=None):
        """Construct an object of type AttitudeControlOptimalFiniteLagrange."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, AttitudeControlOptimalFiniteLagrange)
        IAttitudeControlOptimalFinite.__init__(self, source_object)
        IAttitudeControl.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAttitudeControlOptimalFinite._private_init(self, intf)
        IAttitudeControl._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AttitudeControlOptimalFiniteLagrange, [AttitudeControlOptimalFiniteLagrange, IAttitudeControlOptimalFinite, IAttitudeControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4745590117958740244, 14748864079568769432), AttitudeControlOptimalFiniteLagrange)
agcls.AgTypeNameMap["AttitudeControlOptimalFiniteLagrange"] = AttitudeControlOptimalFiniteLagrange

class ManeuverFinitePropagator(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the propagation of a Finite Maneuver."""

    _num_methods = 21
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_propagator_name_method_offset = 1
    _set_propagator_name_method_offset = 2
    _get_stopping_conditions_method_offset = 3
    _get_min_propagation_time_method_offset = 4
    _set_min_propagation_time_method_offset = 5
    _get_max_propagation_time_method_offset = 6
    _set_max_propagation_time_method_offset = 7
    _get_enable_max_propagation_time_method_offset = 8
    _set_enable_max_propagation_time_method_offset = 9
    _get_enable_warning_message_method_offset = 10
    _set_enable_warning_message_method_offset = 11
    _get_enable_center_burn_method_offset = 12
    _set_enable_center_burn_method_offset = 13
    _get_bias_method_offset = 14
    _set_bias_method_offset = 15
    _get_override_max_propagation_time_method_offset = 16
    _set_override_max_propagation_time_method_offset = 17
    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 18
    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 19
    _get_should_reinitialize_stm_at_start_of_segment_propagation_method_offset = 20
    _set_should_reinitialize_stm_at_start_of_segment_propagation_method_offset = 21
    _metadata = {
        "iid_data" : (5566679304303890544, 11813338934988407462),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ManeuverFinitePropagator)

    _get_propagator_name_metadata = { "offset" : _get_propagator_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def propagator_name(self) -> str:
        """Get or set the propagator."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_propagator_name_metadata)

    _set_propagator_name_metadata = { "offset" : _set_propagator_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @propagator_name.setter
    def propagator_name(self, value:str) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_propagator_name_metadata, value)

    _get_stopping_conditions_metadata = { "offset" : _get_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def stopping_conditions(self) -> "StoppingConditionCollection":
        """Get the stopping conditions list."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_stopping_conditions_metadata)

    _get_min_propagation_time_metadata = { "offset" : _get_min_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_propagation_time(self) -> float:
        """Minimum Propagation Time - the minimum time that must elapse from the beginning of the segment until Astrogator will begin checking stopping conditions for satisfaction. Uses Time Dimension."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_min_propagation_time_metadata)

    _set_min_propagation_time_metadata = { "offset" : _set_min_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_propagation_time.setter
    def min_propagation_time(self, value:float) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_min_propagation_time_metadata, value)

    _get_max_propagation_time_metadata = { "offset" : _get_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_propagation_time(self) -> float:
        """Maximum Propagation Time - the maximum propagation time, after which the segment will end regardless of whether the stopping conditions have been satisfied. Uses Time Dimension."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_max_propagation_time_metadata)

    _set_max_propagation_time_metadata = { "offset" : _set_max_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_propagation_time.setter
    def max_propagation_time(self, value:float) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_max_propagation_time_metadata, value)

    _get_enable_max_propagation_time_metadata = { "offset" : _get_enable_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_max_propagation_time(self) -> bool:
        """Enable Maximum Propagation Time - apply the maximum propagation time."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_enable_max_propagation_time_metadata)

    _set_enable_max_propagation_time_metadata = { "offset" : _set_enable_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_max_propagation_time.setter
    def enable_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_enable_max_propagation_time_metadata, value)

    _get_enable_warning_message_metadata = { "offset" : _get_enable_warning_message_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_warning_message(self) -> bool:
        """Issue a warning message if propagation is stopped by the Maximum Propagation Time parameter."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_enable_warning_message_metadata)

    _set_enable_warning_message_metadata = { "offset" : _set_enable_warning_message_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_warning_message.setter
    def enable_warning_message(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_enable_warning_message_metadata, value)

    _get_enable_center_burn_metadata = { "offset" : _get_enable_center_burn_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_center_burn(self) -> bool:
        """Get or set the option to start the maneuver half the time before the previous segment ended. This property is only available for use with a duration stopping condition."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_enable_center_burn_metadata)

    _set_enable_center_burn_metadata = { "offset" : _set_enable_center_burn_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_center_burn.setter
    def enable_center_burn(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_enable_center_burn_metadata, value)

    _get_bias_metadata = { "offset" : _get_bias_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def bias(self) -> float:
        """Get or set the value by which to adjust the centering of the burn. A positive value will center the burn after the previous segment ends by the amount specified in the Burn Center Bias field. Uses Time Dimension."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_bias_metadata)

    _set_bias_metadata = { "offset" : _set_bias_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @bias.setter
    def bias(self, value:float) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_bias_metadata, value)

    _get_override_max_propagation_time_metadata = { "offset" : _get_override_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def override_max_propagation_time(self) -> bool:
        """Override Maximum Propagation Time - if there is a duration or epoch stopping condition that occurs after the maximum propagation time, ignore the maximum propagation time."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_override_max_propagation_time_metadata)

    _set_override_max_propagation_time_metadata = { "offset" : _set_override_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @override_max_propagation_time.setter
    def override_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_override_max_propagation_time_metadata, value)

    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self) -> bool:
        """Stop immediately if propagation begins beyond an active epoch stopping condition."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata)

    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @should_stop_for_initially_surpassed_epoch_stopping_conditions.setter
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata, value)

    _get_should_reinitialize_stm_at_start_of_segment_propagation_metadata = { "offset" : _get_should_reinitialize_stm_at_start_of_segment_propagation_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def should_reinitialize_stm_at_start_of_segment_propagation(self) -> bool:
        """If this segment is propagating the state transition matrix, reset it to the identity matrix at the start of the segment."""
        return self._intf.get_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._get_should_reinitialize_stm_at_start_of_segment_propagation_metadata)

    _set_should_reinitialize_stm_at_start_of_segment_propagation_metadata = { "offset" : _set_should_reinitialize_stm_at_start_of_segment_propagation_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @should_reinitialize_stm_at_start_of_segment_propagation.setter
    def should_reinitialize_stm_at_start_of_segment_propagation(self, value:bool) -> None:
        return self._intf.set_property(ManeuverFinitePropagator._metadata, ManeuverFinitePropagator._set_should_reinitialize_stm_at_start_of_segment_propagation_metadata, value)

    _property_names[propagator_name] = "propagator_name"
    _property_names[stopping_conditions] = "stopping_conditions"
    _property_names[min_propagation_time] = "min_propagation_time"
    _property_names[max_propagation_time] = "max_propagation_time"
    _property_names[enable_max_propagation_time] = "enable_max_propagation_time"
    _property_names[enable_warning_message] = "enable_warning_message"
    _property_names[enable_center_burn] = "enable_center_burn"
    _property_names[bias] = "bias"
    _property_names[override_max_propagation_time] = "override_max_propagation_time"
    _property_names[should_stop_for_initially_surpassed_epoch_stopping_conditions] = "should_stop_for_initially_surpassed_epoch_stopping_conditions"
    _property_names[should_reinitialize_stm_at_start_of_segment_propagation] = "should_reinitialize_stm_at_start_of_segment_propagation"

    def __init__(self, source_object=None):
        """Construct an object of type ManeuverFinitePropagator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ManeuverFinitePropagator)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ManeuverFinitePropagator, [ManeuverFinitePropagator, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5539664440505171113, 10732172948923783042), ManeuverFinitePropagator)
agcls.AgTypeNameMap["ManeuverFinitePropagator"] = ManeuverFinitePropagator

class MCSHold(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Hold segment."""

    _num_methods = 23
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_step_size_method_offset = 1
    _set_step_size_method_offset = 2
    _get_hold_frame_name_method_offset = 3
    _set_hold_frame_name_method_offset = 4
    _get_enable_hold_attitude_method_offset = 5
    _set_enable_hold_attitude_method_offset = 6
    _get_stopping_conditions_method_offset = 7
    _get_min_propagation_time_method_offset = 8
    _set_min_propagation_time_method_offset = 9
    _get_max_propagation_time_method_offset = 10
    _set_max_propagation_time_method_offset = 11
    _get_enable_max_propagation_time_method_offset = 12
    _set_enable_max_propagation_time_method_offset = 13
    _get_enable_warning_message_method_offset = 14
    _set_enable_warning_message_method_offset = 15
    _enable_control_parameter_method_offset = 16
    _disable_control_parameter_method_offset = 17
    _is_control_parameter_enabled_method_offset = 18
    _get_control_parameters_available_method_offset = 19
    _get_override_max_propagation_time_method_offset = 20
    _set_override_max_propagation_time_method_offset = 21
    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 22
    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset = 23
    _metadata = {
        "iid_data" : (4662680246288167963, 8721114741357391511),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSHold)

    _get_step_size_metadata = { "offset" : _get_step_size_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step_size(self) -> float:
        """Get or set the time interval between calculated ephemeris output points. Uses Time Dimension."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_step_size_metadata)

    _set_step_size_metadata = { "offset" : _set_step_size_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step_size.setter
    def step_size(self, value:float) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_step_size_metadata, value)

    _get_hold_frame_name_metadata = { "offset" : _get_hold_frame_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def hold_frame_name(self) -> str:
        """Hold Frame - the reference frame of the Hold segment."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_hold_frame_name_metadata)

    _set_hold_frame_name_metadata = { "offset" : _set_hold_frame_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @hold_frame_name.setter
    def hold_frame_name(self, value:str) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_hold_frame_name_metadata, value)

    _get_enable_hold_attitude_metadata = { "offset" : _get_enable_hold_attitude_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_hold_attitude(self) -> bool:
        """Enable Hold Altitude - if true, the spacecraft's attitude is fixed within the hold frame."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_enable_hold_attitude_metadata)

    _set_enable_hold_attitude_metadata = { "offset" : _set_enable_hold_attitude_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_hold_attitude.setter
    def enable_hold_attitude(self, value:bool) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_enable_hold_attitude_metadata, value)

    _get_stopping_conditions_metadata = { "offset" : _get_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def stopping_conditions(self) -> "StoppingConditionCollection":
        """Get the stopping conditions defined for the segment."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_stopping_conditions_metadata)

    _get_min_propagation_time_metadata = { "offset" : _get_min_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_propagation_time(self) -> float:
        """Minimum Propagation Time - the minimum time that must elapse from the beginning of the segment until Astrogator will begin checking stopping conditions for satisfaction. Uses Time Dimension."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_min_propagation_time_metadata)

    _set_min_propagation_time_metadata = { "offset" : _set_min_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_propagation_time.setter
    def min_propagation_time(self, value:float) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_min_propagation_time_metadata, value)

    _get_max_propagation_time_metadata = { "offset" : _get_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max_propagation_time(self) -> float:
        """Maximum Propagation Time - the maximum propagation time, after which the segment will end regardless of whether the stopping conditions have been satisfied. Uses Time Dimension."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_max_propagation_time_metadata)

    _set_max_propagation_time_metadata = { "offset" : _set_max_propagation_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max_propagation_time.setter
    def max_propagation_time(self, value:float) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_max_propagation_time_metadata, value)

    _get_enable_max_propagation_time_metadata = { "offset" : _get_enable_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_max_propagation_time(self) -> bool:
        """Enable Maximum Propagation Time - if true, the maximum propagation time will be enforced."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_enable_max_propagation_time_metadata)

    _set_enable_max_propagation_time_metadata = { "offset" : _set_enable_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_max_propagation_time.setter
    def enable_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_enable_max_propagation_time_metadata, value)

    _get_enable_warning_message_metadata = { "offset" : _get_enable_warning_message_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_warning_message(self) -> bool:
        """If true, Astrogator will issue a warning message if propagation is stopped by the Maximum Propagation Time parameter."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_enable_warning_message_metadata)

    _set_enable_warning_message_metadata = { "offset" : _set_enable_warning_message_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_warning_message.setter
    def enable_warning_message(self, value:bool) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_enable_warning_message_metadata, value)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced),) }
    def enable_control_parameter(self, param:"ControlAdvanced") -> None:
        """Enable or disables the specified control parameter."""
        return self._intf.invoke(MCSHold._metadata, MCSHold._enable_control_parameter_metadata, param)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced),) }
    def disable_control_parameter(self, param:"ControlAdvanced") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSHold._metadata, MCSHold._disable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlAdvanced), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlAdvanced") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSHold._metadata, MCSHold._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_control_parameters_available_metadata)

    _get_override_max_propagation_time_metadata = { "offset" : _get_override_max_propagation_time_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def override_max_propagation_time(self) -> bool:
        """Override Maximum Propagation Time - if there is a duration or epoch stopping condition that occurs after the maximum propagation time, ignore the maximum propagation time."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_override_max_propagation_time_metadata)

    _set_override_max_propagation_time_metadata = { "offset" : _set_override_max_propagation_time_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @override_max_propagation_time.setter
    def override_max_propagation_time(self, value:bool) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_override_max_propagation_time_metadata, value)

    _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _get_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self) -> bool:
        """Stop immediately if propagation begins beyond an active epoch stopping condition."""
        return self._intf.get_property(MCSHold._metadata, MCSHold._get_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata)

    _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata = { "offset" : _set_should_stop_for_initially_surpassed_epoch_stopping_conditions_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @should_stop_for_initially_surpassed_epoch_stopping_conditions.setter
    def should_stop_for_initially_surpassed_epoch_stopping_conditions(self, value:bool) -> None:
        return self._intf.set_property(MCSHold._metadata, MCSHold._set_should_stop_for_initially_surpassed_epoch_stopping_conditions_metadata, value)

    _property_names[step_size] = "step_size"
    _property_names[hold_frame_name] = "hold_frame_name"
    _property_names[enable_hold_attitude] = "enable_hold_attitude"
    _property_names[stopping_conditions] = "stopping_conditions"
    _property_names[min_propagation_time] = "min_propagation_time"
    _property_names[max_propagation_time] = "max_propagation_time"
    _property_names[enable_max_propagation_time] = "enable_max_propagation_time"
    _property_names[enable_warning_message] = "enable_warning_message"
    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[override_max_propagation_time] = "override_max_propagation_time"
    _property_names[should_stop_for_initially_surpassed_epoch_stopping_conditions] = "should_stop_for_initially_surpassed_epoch_stopping_conditions"

    def __init__(self, source_object=None):
        """Construct an object of type MCSHold."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSHold)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSHold, [MCSHold, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5036634056898744919, 16511560869260231339), MCSHold)
agcls.AgTypeNameMap["MCSHold"] = MCSHold

class MCSUpdate(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an Update segment."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _set_action_and_value_method_offset = 1
    _get_action_method_offset = 2
    _get_value_method_offset = 3
    _set_action_method_offset = 4
    _set_value_method_offset = 5
    _disable_control_parameter_method_offset = 6
    _enable_control_parameter_method_offset = 7
    _is_control_parameter_enabled_method_offset = 8
    _get_control_parameters_available_method_offset = 9
    _get_user_variables_method_offset = 10
    _metadata = {
        "iid_data" : (4612298704220394462, 16427056929712906381),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSUpdate)

    _set_action_and_value_metadata = { "offset" : _set_action_and_value_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(UpdateParam), agmarshall.EnumArg(UpdateAction), agmarshall.DoubleArg,) }
    def set_action_and_value(self, parameter_type:"UpdateParam", action_type:"UpdateAction", value:float) -> None:
        """Set an action and the new value."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._set_action_and_value_metadata, parameter_type, action_type, value)

    _get_action_metadata = { "offset" : _get_action_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(UpdateParam), agmarshall.EnumArg(UpdateAction),) }
    def get_action(self, parameter_type:"UpdateParam") -> "UpdateAction":
        """Get the action type for a parameter."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._get_action_metadata, parameter_type, OutArg())

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.EnumArg(UpdateParam), agmarshall.DoubleArg,) }
    def get_value(self, parameter_type:"UpdateParam") -> float:
        """Get the value type for a parameter."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._get_value_metadata, parameter_type, OutArg())

    _set_action_metadata = { "offset" : _set_action_method_offset,
            "arg_types" : (agcom.LONG, agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(UpdateParam), agmarshall.EnumArg(UpdateAction),) }
    def set_action(self, parameter_name:"UpdateParam", action_type:"UpdateAction") -> None:
        """Set the update action type."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._set_action_metadata, parameter_name, action_type)

    _set_value_metadata = { "offset" : _set_value_method_offset,
            "arg_types" : (agcom.LONG, agcom.DOUBLE,),
            "marshallers" : (agmarshall.EnumArg(UpdateParam), agmarshall.DoubleArg,) }
    def set_value(self, parameter_name:"UpdateParam", value:float) -> None:
        """Set the update value."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._set_value_metadata, parameter_name, value)

    _disable_control_parameter_metadata = { "offset" : _disable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlUpdate),) }
    def disable_control_parameter(self, param:"ControlUpdate") -> None:
        """Disables the specified control parameter."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._disable_control_parameter_metadata, param)

    _enable_control_parameter_metadata = { "offset" : _enable_control_parameter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ControlUpdate),) }
    def enable_control_parameter(self, param:"ControlUpdate") -> None:
        """Enable the specified control parameter."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._enable_control_parameter_metadata, param)

    _is_control_parameter_enabled_metadata = { "offset" : _is_control_parameter_enabled_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ControlUpdate), agmarshall.VariantBoolArg,) }
    def is_control_parameter_enabled(self, param:"ControlUpdate") -> bool:
        """Sees if the specified control is enabled."""
        return self._intf.invoke(MCSUpdate._metadata, MCSUpdate._is_control_parameter_enabled_metadata, param, OutArg())

    _get_control_parameters_available_metadata = { "offset" : _get_control_parameters_available_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def control_parameters_available(self) -> bool:
        """Return whether or not the control parameters can be set."""
        return self._intf.get_property(MCSUpdate._metadata, MCSUpdate._get_control_parameters_available_metadata)

    _get_user_variables_metadata = { "offset" : _get_user_variables_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_variables(self) -> "UserVariableUpdateCollection":
        """Interface used to modify user variables for the update segment."""
        return self._intf.get_property(MCSUpdate._metadata, MCSUpdate._get_user_variables_metadata)

    _property_names[control_parameters_available] = "control_parameters_available"
    _property_names[user_variables] = "user_variables"

    def __init__(self, source_object=None):
        """Construct an object of type MCSUpdate."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSUpdate)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSUpdate, [MCSUpdate, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4769851952476936711, 12743095332754810808), MCSUpdate)
agcls.AgTypeNameMap["MCSUpdate"] = MCSUpdate

class MCSReturn(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Return segment."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_return_control_to_parent_sequence_method_offset = 1
    _set_return_control_to_parent_sequence_method_offset = 2
    _metadata = {
        "iid_data" : (5086645245770904087, 7110397434433108120),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSReturn)

    _get_return_control_to_parent_sequence_metadata = { "offset" : _get_return_control_to_parent_sequence_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ReturnControl),) }
    @property
    def return_control_to_parent_sequence(self) -> "ReturnControl":
        """Get or set the state of the segment."""
        return self._intf.get_property(MCSReturn._metadata, MCSReturn._get_return_control_to_parent_sequence_metadata)

    _set_return_control_to_parent_sequence_metadata = { "offset" : _set_return_control_to_parent_sequence_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ReturnControl),) }
    @return_control_to_parent_sequence.setter
    def return_control_to_parent_sequence(self, value:"ReturnControl") -> None:
        return self._intf.set_property(MCSReturn._metadata, MCSReturn._set_return_control_to_parent_sequence_metadata, value)

    _property_names[return_control_to_parent_sequence] = "return_control_to_parent_sequence"

    def __init__(self, source_object=None):
        """Construct an object of type MCSReturn."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSReturn)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSReturn, [MCSReturn, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5338959801614235290, 674960884198980233), MCSReturn)
agcls.AgTypeNameMap["MCSReturn"] = MCSReturn

class MCSStop(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Stop segment."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_enabled_method_offset = 1
    _set_enabled_method_offset = 2
    _metadata = {
        "iid_data" : (5433661175555402357, 6615063642560173976),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSStop)

    _get_enabled_metadata = { "offset" : _get_enabled_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enabled(self) -> bool:
        """If true, the segment will stop the execution of the MCS."""
        return self._intf.get_property(MCSStop._metadata, MCSStop._get_enabled_metadata)

    _set_enabled_metadata = { "offset" : _set_enabled_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enabled.setter
    def enabled(self, value:bool) -> None:
        return self._intf.set_property(MCSStop._metadata, MCSStop._set_enabled_metadata, value)

    _property_names[enabled] = "enabled"

    def __init__(self, source_object=None):
        """Construct an object of type MCSStop."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSStop)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSStop, [MCSStop, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5575190769478601305, 3572350491614611646), MCSStop)
agcls.AgTypeNameMap["MCSStop"] = MCSStop

class MCSTargetSequence(IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable, SupportsDeleteCallback):
    """General properties of a TargetSequence segment."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_action_method_offset = 1
    _set_action_method_offset = 2
    _get_when_profiles_finish_method_offset = 3
    _set_when_profiles_finish_method_offset = 4
    _get_continue_on_failure_method_offset = 5
    _set_continue_on_failure_method_offset = 6
    _get_segments_method_offset = 7
    _get_profiles_method_offset = 8
    _apply_profiles_method_offset = 9
    _reset_profiles_method_offset = 10
    _apply_profile_method_offset = 11
    _reset_profile_method_offset = 12
    _apply_profile_by_name_method_offset = 13
    _reset_profile_by_name_method_offset = 14
    _get_reset_inner_targeters_method_offset = 15
    _set_reset_inner_targeters_method_offset = 16
    _metadata = {
        "iid_data" : (4754863831773773924, 14533412604548864682),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSTargetSequence)

    _get_action_metadata = { "offset" : _get_action_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TargetSequenceAction),) }
    @property
    def action(self) -> "TargetSequenceAction":
        """Whether to run the sequence nominally or using profiles."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_action_metadata)

    _set_action_metadata = { "offset" : _set_action_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TargetSequenceAction),) }
    @action.setter
    def action(self, value:"TargetSequenceAction") -> None:
        return self._intf.set_property(MCSTargetSequence._metadata, MCSTargetSequence._set_action_metadata, value)

    _get_when_profiles_finish_metadata = { "offset" : _get_when_profiles_finish_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ProfilesFinish),) }
    @property
    def when_profiles_finish(self) -> "ProfilesFinish":
        """When Profiles Converge - the action to be carried out if targeting has converged."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_when_profiles_finish_metadata)

    _set_when_profiles_finish_metadata = { "offset" : _set_when_profiles_finish_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ProfilesFinish),) }
    @when_profiles_finish.setter
    def when_profiles_finish(self, value:"ProfilesFinish") -> None:
        return self._intf.set_property(MCSTargetSequence._metadata, MCSTargetSequence._set_when_profiles_finish_metadata, value)

    _get_continue_on_failure_metadata = { "offset" : _get_continue_on_failure_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def continue_on_failure(self) -> bool:
        """Continue if profiles don't converge - if true, the target sequence continues if a profile fails to converge; otherwise, the MCS will stop upon the failure of a search profile."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_continue_on_failure_metadata)

    _set_continue_on_failure_metadata = { "offset" : _set_continue_on_failure_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @continue_on_failure.setter
    def continue_on_failure(self, value:bool) -> None:
        return self._intf.set_property(MCSTargetSequence._metadata, MCSTargetSequence._set_continue_on_failure_metadata, value)

    _get_segments_metadata = { "offset" : _get_segments_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def segments(self) -> "MCSSegmentCollection":
        """Return the segments contained within the target sequence."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_segments_metadata)

    _get_profiles_metadata = { "offset" : _get_profiles_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def profiles(self) -> "ProfileCollection":
        """Return the profiles used within the target sequence."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_profiles_metadata)

    _apply_profiles_metadata = { "offset" : _apply_profiles_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def apply_profiles(self) -> None:
        """Apply Changes - applies the current values of search profiles' controls and the changes specified by the segment configuration profiles to the segments within the target sequence."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._apply_profiles_metadata, )

    _reset_profiles_metadata = { "offset" : _reset_profiles_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reset_profiles(self) -> None:
        """Reset - resets the controls of the search profiles to the segments' values."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._reset_profiles_metadata, )

    _apply_profile_metadata = { "offset" : _apply_profile_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IProfile"),) }
    def apply_profile(self, profile:"IProfile") -> None:
        """Apply Changes - applies the current values of specified profile to the segments within the target sequence."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._apply_profile_metadata, profile)

    _reset_profile_metadata = { "offset" : _reset_profile_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IProfile"),) }
    def reset_profile(self, profile:"IProfile") -> None:
        """Reset - resets the current values of specified profile."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._reset_profile_metadata, profile)

    _apply_profile_by_name_metadata = { "offset" : _apply_profile_by_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def apply_profile_by_name(self, profile:str) -> None:
        """Apply Changes - applies the current values of specified profile to the segments within the target sequence."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._apply_profile_by_name_metadata, profile)

    _reset_profile_by_name_metadata = { "offset" : _reset_profile_by_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def reset_profile_by_name(self, profile:str) -> None:
        """Reset - resets the current values of specified profile."""
        return self._intf.invoke(MCSTargetSequence._metadata, MCSTargetSequence._reset_profile_by_name_metadata, profile)

    _get_reset_inner_targeters_metadata = { "offset" : _get_reset_inner_targeters_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def reset_inner_targeters(self) -> bool:
        """If true, inner target sequences will have their profiles reset before each run."""
        return self._intf.get_property(MCSTargetSequence._metadata, MCSTargetSequence._get_reset_inner_targeters_metadata)

    _set_reset_inner_targeters_metadata = { "offset" : _set_reset_inner_targeters_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @reset_inner_targeters.setter
    def reset_inner_targeters(self, value:bool) -> None:
        return self._intf.set_property(MCSTargetSequence._metadata, MCSTargetSequence._set_reset_inner_targeters_metadata, value)

    _property_names[action] = "action"
    _property_names[when_profiles_finish] = "when_profiles_finish"
    _property_names[continue_on_failure] = "continue_on_failure"
    _property_names[segments] = "segments"
    _property_names[profiles] = "profiles"
    _property_names[reset_inner_targeters] = "reset_inner_targeters"

    def __init__(self, source_object=None):
        """Construct an object of type MCSTargetSequence."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSTargetSequence)
        IMCSSegment.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IMCSSegment._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSTargetSequence, [MCSTargetSequence, IMCSSegment, IRuntimeTypeInfoProvider, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5275189599605526062, 2875478924429173914), MCSTargetSequence)
agcls.AgTypeNameMap["MCSTargetSequence"] = MCSTargetSequence

class ProfileCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a list of target sequence profiles."""

    _num_methods = 14
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _add_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _get_count_method_offset = 4
    _get_available_profiles_method_offset = 5
    _remove_method_offset = 6
    _remove_all_method_offset = 7
    _get_provide_runtime_type_info_method_offset = 8
    _cut_method_offset = 9
    _paste_method_offset = 10
    _insert_copy_method_offset = 11
    _add2_method_offset = 12
    _get_item_by_index_method_offset = 13
    _get_item_by_name_method_offset = 14
    _metadata = {
        "iid_data" : (5630196485976636884, 6542187252655594407),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileCollection)
    def __iter__(self):
        """Create an iterator for the ProfileCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IProfile":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add(self, profile_name:str) -> "IProfile":
        """Add a profile."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._add_metadata, profile_name, OutArg())

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "IProfile":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(ProfileCollection._metadata, ProfileCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(ProfileCollection._metadata, ProfileCollection._get_count_metadata)

    _get_available_profiles_metadata = { "offset" : _get_available_profiles_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_profiles(self) -> list:
        """Return a list of available profiles."""
        return self._intf.get_property(ProfileCollection._metadata, ProfileCollection._get_available_profiles_metadata)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def remove(self, index_or_profile_name:typing.Any) -> None:
        """Remove an item from the profile collection."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._remove_metadata, index_or_profile_name)

    _remove_all_metadata = { "offset" : _remove_all_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove_all(self) -> None:
        """Remove all profiles from the profile collection."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._remove_all_metadata, )

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(ProfileCollection._metadata, ProfileCollection._get_provide_runtime_type_info_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def cut(self, index_or_name:typing.Any) -> None:
        """Copy the profile into the clipboard and removes profile from the list."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._cut_metadata, index_or_name)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (agcom.Variant, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.EnumArg(ProfileInsertDirection), agmarshall.InterfaceOutArg,) }
    def paste(self, index_or_name:typing.Any, direction:"ProfileInsertDirection") -> "IProfile":
        """Pastes the profile from the clipboard and inserts into the list."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._paste_metadata, index_or_name, direction, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, agcom.Variant, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IProfile"), agmarshall.VariantArg, agmarshall.EnumArg(ProfileInsertDirection), agmarshall.InterfaceOutArg,) }
    def insert_copy(self, profile:"IProfile", index_or_name:typing.Any, direction:"ProfileInsertDirection") -> "IProfile":
        """Copy the profile and inserts the copy into the list."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._insert_copy_metadata, profile, index_or_name, direction, OutArg())

    _add2_metadata = { "offset" : _add2_method_offset,
            "arg_types" : (agcom.BSTR, agcom.Variant, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg, agmarshall.EnumArg(ProfileInsertDirection), agmarshall.InterfaceOutArg,) }
    def add2(self, profile_name:str, index_or_name:typing.Any, direction:"ProfileInsertDirection") -> "IProfile":
        """Add a profile."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._add2_metadata, profile_name, index_or_name, direction, OutArg())

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "IProfile":
        """Retrieve a profile from the collection by index."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "IProfile":
        """Retrieve a profile from the collection by name."""
        return self._intf.invoke(ProfileCollection._metadata, ProfileCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[available_profiles] = "available_profiles"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileCollection, [ProfileCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5143990449118478455, 3349457320014041742), ProfileCollection)
agcls.AgTypeNameMap["ProfileCollection"] = ProfileCollection

class MCSOptions(SupportsDeleteCallback):
    """Properties for the MCS Options."""

    _num_methods = 37
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_draw_trajectory_in_2d_method_offset = 1
    _set_draw_trajectory_in_2d_method_offset = 2
    _get_draw_trajectory_in_3d_method_offset = 3
    _set_draw_trajectory_in_3d_method_offset = 4
    _get_update_animation_time_for_all_objects_method_offset = 5
    _set_update_animation_time_for_all_objects_method_offset = 6
    _get_clear_draw_while_calculating_graphics_before_each_run_method_offset = 7
    _set_clear_draw_while_calculating_graphics_before_each_run_method_offset = 8
    _get_clear_additional_b_plane_points_method_offset = 9
    _set_clear_additional_b_plane_points_method_offset = 10
    _get_propagate_on_apply_method_offset = 11
    _set_propagate_on_apply_method_offset = 12
    _get_enable_trajectory_segment_colors_method_offset = 13
    _set_enable_trajectory_segment_colors_method_offset = 14
    _get_save_numbers_in_raw_format_method_offset = 15
    _set_save_numbers_in_raw_format_method_offset = 16
    _get_stopping_condition_time_tolerance_method_offset = 17
    _set_stopping_condition_time_tolerance_method_offset = 18
    _get_enable_segment_controls_method_offset = 19
    _set_enable_segment_controls_method_offset = 20
    _get_enable_segment_results_method_offset = 21
    _set_enable_segment_results_method_offset = 22
    _get_enable_logging_method_offset = 23
    _set_enable_logging_method_offset = 24
    _get_graphics_update_rate_method_offset = 25
    _set_graphics_update_rate_method_offset = 26
    _get_promote_controls_method_offset = 27
    _set_promote_controls_method_offset = 28
    _get_use_nominal_settings_method_offset = 29
    _set_use_nominal_settings_method_offset = 30
    _get_min_ephem_step_method_offset = 31
    _set_min_ephem_step_method_offset = 32
    _get_generate_ephemeris_method_offset = 33
    _set_generate_ephemeris_method_offset = 34
    _get_user_variables_method_offset = 35
    _get_smart_run_mode_method_offset = 36
    _set_smart_run_mode_method_offset = 37
    _metadata = {
        "iid_data" : (4995434399239591698, 15701662039751640989),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, MCSOptions)

    _get_draw_trajectory_in_2d_metadata = { "offset" : _get_draw_trajectory_in_2d_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def draw_trajectory_in_2d(self) -> bool:
        """If true, Astrogator will draw the trajectory in the 2D Graphics windows as the ephemeris is calculated during the current run."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_draw_trajectory_in_2d_metadata)

    _set_draw_trajectory_in_2d_metadata = { "offset" : _set_draw_trajectory_in_2d_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @draw_trajectory_in_2d.setter
    def draw_trajectory_in_2d(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_draw_trajectory_in_2d_metadata, value)

    _get_draw_trajectory_in_3d_metadata = { "offset" : _get_draw_trajectory_in_3d_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def draw_trajectory_in_3d(self) -> bool:
        """If true, Astrogator will draw the trajectory in the 3D Graphics windows as the ephemeris is calculated during the current run."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_draw_trajectory_in_3d_metadata)

    _set_draw_trajectory_in_3d_metadata = { "offset" : _set_draw_trajectory_in_3d_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @draw_trajectory_in_3d.setter
    def draw_trajectory_in_3d(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_draw_trajectory_in_3d_metadata, value)

    _get_update_animation_time_for_all_objects_metadata = { "offset" : _get_update_animation_time_for_all_objects_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def update_animation_time_for_all_objects(self) -> bool:
        """If true, all other objects will be animated so that they appear at the proper position for the time being computed as the trajectory is calculated."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_update_animation_time_for_all_objects_metadata)

    _set_update_animation_time_for_all_objects_metadata = { "offset" : _set_update_animation_time_for_all_objects_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @update_animation_time_for_all_objects.setter
    def update_animation_time_for_all_objects(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_update_animation_time_for_all_objects_metadata, value)

    _get_clear_draw_while_calculating_graphics_before_each_run_metadata = { "offset" : _get_clear_draw_while_calculating_graphics_before_each_run_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def clear_draw_while_calculating_graphics_before_each_run(self) -> bool:
        """If true, Astrogator will automatically clear all target iteration graphics of the previous run - in all graphics windows - before the current run draws new calculations."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_clear_draw_while_calculating_graphics_before_each_run_metadata)

    _set_clear_draw_while_calculating_graphics_before_each_run_metadata = { "offset" : _set_clear_draw_while_calculating_graphics_before_each_run_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @clear_draw_while_calculating_graphics_before_each_run.setter
    def clear_draw_while_calculating_graphics_before_each_run(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_clear_draw_while_calculating_graphics_before_each_run_metadata, value)

    _get_clear_additional_b_plane_points_metadata = { "offset" : _get_clear_additional_b_plane_points_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def clear_additional_b_plane_points(self) -> bool:
        """If true, Astrogator will clear all additional B-Plane points from any previous run - in all graphics windows - before the current run."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_clear_additional_b_plane_points_metadata)

    _set_clear_additional_b_plane_points_metadata = { "offset" : _set_clear_additional_b_plane_points_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @clear_additional_b_plane_points.setter
    def clear_additional_b_plane_points(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_clear_additional_b_plane_points_metadata, value)

    _get_propagate_on_apply_metadata = { "offset" : _get_propagate_on_apply_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def propagate_on_apply(self) -> bool:
        """If true, Astrogator will propagate trajectories whenever you click OK or Apply on the Orbit page. If you want to close the Properties Browser for the satellite without running the MCS, make certain this option is disabled."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_propagate_on_apply_metadata)

    _set_propagate_on_apply_metadata = { "offset" : _set_propagate_on_apply_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @propagate_on_apply.setter
    def propagate_on_apply(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_propagate_on_apply_metadata, value)

    _get_enable_trajectory_segment_colors_metadata = { "offset" : _get_enable_trajectory_segment_colors_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_trajectory_segment_colors(self) -> bool:
        """If true, trajectory segments are displayed in the 2D Graphics window in the colors selected for the respective segments; otherwise the color of the trajectory is defined by the 2D Graphics Attributes page."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_enable_trajectory_segment_colors_metadata)

    _set_enable_trajectory_segment_colors_metadata = { "offset" : _set_enable_trajectory_segment_colors_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_trajectory_segment_colors.setter
    def enable_trajectory_segment_colors(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_enable_trajectory_segment_colors_metadata, value)

    _get_save_numbers_in_raw_format_metadata = { "offset" : _get_save_numbers_in_raw_format_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def save_numbers_in_raw_format(self) -> bool:
        """If true, Astrogator will store satellite information in binary format - to preserve the maximum amount of precision."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_save_numbers_in_raw_format_metadata)

    _set_save_numbers_in_raw_format_metadata = { "offset" : _set_save_numbers_in_raw_format_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @save_numbers_in_raw_format.setter
    def save_numbers_in_raw_format(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_save_numbers_in_raw_format_metadata, value)

    _get_stopping_condition_time_tolerance_metadata = { "offset" : _get_stopping_condition_time_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stopping_condition_time_tolerance(self) -> float:
        """Get or set the time tolerance, which will be applied with respect to desired trip values throughout the MCS. If this value is set to zero, time tolerance will not be applied."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_stopping_condition_time_tolerance_metadata)

    _set_stopping_condition_time_tolerance_metadata = { "offset" : _set_stopping_condition_time_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stopping_condition_time_tolerance.setter
    def stopping_condition_time_tolerance(self, value:float) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_stopping_condition_time_tolerance_metadata, value)

    _get_enable_segment_controls_metadata = { "offset" : _get_enable_segment_controls_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_segment_controls(self) -> bool:
        """If true, Astrogator will automatically add independent variables to differential correctors."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_enable_segment_controls_metadata)

    _set_enable_segment_controls_metadata = { "offset" : _set_enable_segment_controls_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_segment_controls.setter
    def enable_segment_controls(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_enable_segment_controls_metadata, value)

    _get_enable_segment_results_metadata = { "offset" : _get_enable_segment_results_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_segment_results(self) -> bool:
        """If true, Astrogator will automatically add dependent variables to differential correctors."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_enable_segment_results_metadata)

    _set_enable_segment_results_metadata = { "offset" : _set_enable_segment_results_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_segment_results.setter
    def enable_segment_results(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_enable_segment_results_metadata, value)

    _get_enable_logging_metadata = { "offset" : _get_enable_logging_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_logging(self) -> bool:
        """If true, target sequences can be set to produce 'run history' log files for differential corrector profiles. By default, a target sequence does not produce a log file."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_enable_logging_metadata)

    _set_enable_logging_metadata = { "offset" : _set_enable_logging_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_logging.setter
    def enable_logging(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_enable_logging_metadata, value)

    _get_graphics_update_rate_metadata = { "offset" : _get_graphics_update_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def graphics_update_rate(self) -> float:
        """Get or set the rate (between 0 and 1) at which to update graphics. Dimensionless."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_graphics_update_rate_metadata)

    _set_graphics_update_rate_metadata = { "offset" : _set_graphics_update_rate_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @graphics_update_rate.setter
    def graphics_update_rate(self, value:float) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_graphics_update_rate_metadata, value)

    _get_promote_controls_metadata = { "offset" : _get_promote_controls_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def promote_controls(self) -> bool:
        """If true, a target sequence will be able to affect controls and results within a nested target sequence in addition to its own."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_promote_controls_metadata)

    _set_promote_controls_metadata = { "offset" : _set_promote_controls_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @promote_controls.setter
    def promote_controls(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_promote_controls_metadata, value)

    _get_use_nominal_settings_metadata = { "offset" : _get_use_nominal_settings_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_nominal_settings(self) -> bool:
        """If true, components or segments that are modified by a target sequence will be restored to their nominal values as soon as the target sequence completes its run."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_use_nominal_settings_metadata)

    _set_use_nominal_settings_metadata = { "offset" : _set_use_nominal_settings_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_nominal_settings.setter
    def use_nominal_settings(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_use_nominal_settings_metadata, value)

    _get_min_ephem_step_metadata = { "offset" : _get_min_ephem_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min_ephem_step(self) -> float:
        """Get or set the minimum step size for saving ephemeris. Uses Time Dimension."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_min_ephem_step_metadata)

    _set_min_ephem_step_metadata = { "offset" : _set_min_ephem_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min_ephem_step.setter
    def min_ephem_step(self, value:float) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_min_ephem_step_metadata, value)

    _get_generate_ephemeris_metadata = { "offset" : _get_generate_ephemeris_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def generate_ephemeris(self) -> bool:
        """If true, Astrogator will generate ephemeris."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_generate_ephemeris_metadata)

    _set_generate_ephemeris_metadata = { "offset" : _set_generate_ephemeris_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @generate_ephemeris.setter
    def generate_ephemeris(self, value:bool) -> None:
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_generate_ephemeris_metadata, value)

    _get_user_variables_metadata = { "offset" : _get_user_variables_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def user_variables(self) -> "UserVariableDefinitionCollection":
        """Interface used to add/remove user variables."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_user_variables_metadata)

    _get_smart_run_mode_metadata = { "offset" : _get_smart_run_mode_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SmartRunMode),) }
    @property
    def smart_run_mode(self) -> "SmartRunMode":
        """Control whether the run will attempt to only run changed segments."""
        return self._intf.get_property(MCSOptions._metadata, MCSOptions._get_smart_run_mode_metadata)

    _set_smart_run_mode_metadata = { "offset" : _set_smart_run_mode_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SmartRunMode),) }
    @smart_run_mode.setter
    def smart_run_mode(self, value:"SmartRunMode") -> None:
        """Control whether the run will attempt to only run changed segments."""
        return self._intf.set_property(MCSOptions._metadata, MCSOptions._set_smart_run_mode_metadata, value)

    _property_names[draw_trajectory_in_2d] = "draw_trajectory_in_2d"
    _property_names[draw_trajectory_in_3d] = "draw_trajectory_in_3d"
    _property_names[update_animation_time_for_all_objects] = "update_animation_time_for_all_objects"
    _property_names[clear_draw_while_calculating_graphics_before_each_run] = "clear_draw_while_calculating_graphics_before_each_run"
    _property_names[clear_additional_b_plane_points] = "clear_additional_b_plane_points"
    _property_names[propagate_on_apply] = "propagate_on_apply"
    _property_names[enable_trajectory_segment_colors] = "enable_trajectory_segment_colors"
    _property_names[save_numbers_in_raw_format] = "save_numbers_in_raw_format"
    _property_names[stopping_condition_time_tolerance] = "stopping_condition_time_tolerance"
    _property_names[enable_segment_controls] = "enable_segment_controls"
    _property_names[enable_segment_results] = "enable_segment_results"
    _property_names[enable_logging] = "enable_logging"
    _property_names[graphics_update_rate] = "graphics_update_rate"
    _property_names[promote_controls] = "promote_controls"
    _property_names[use_nominal_settings] = "use_nominal_settings"
    _property_names[min_ephem_step] = "min_ephem_step"
    _property_names[generate_ephemeris] = "generate_ephemeris"
    _property_names[user_variables] = "user_variables"
    _property_names[smart_run_mode] = "smart_run_mode"

    def __init__(self, source_object=None):
        """Construct an object of type MCSOptions."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, MCSOptions)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, MCSOptions, [MCSOptions, ])

agcls.AgClassCatalog.add_catalog_entry((5252079360558931739, 420969178435848603), MCSOptions)
agcls.AgTypeNameMap["MCSOptions"] = MCSOptions

class CalculationObjectCollection(SupportsDeleteCallback):
    """Collection of calculation objects."""

    _num_methods = 10
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _add_method_offset = 1
    _item_method_offset = 2
    _remove_method_offset = 3
    _get__new_enum_method_offset = 4
    _get_count_method_offset = 5
    _cut_method_offset = 6
    _paste_method_offset = 7
    _insert_copy_method_offset = 8
    _get_item_by_index_method_offset = 9
    _get_item_by_name_method_offset = 10
    _metadata = {
        "iid_data" : (4871526303947963367, 9557979883536836749),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationObjectCollection)
    def __iter__(self):
        """Create an iterator for the CalculationObjectCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "IComponentInfo":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add(self, name:str) -> "IComponentInfo":
        """Add a calc object to the collection."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._add_metadata, name, OutArg())

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "IComponentInfo":
        """Return a calc object."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._item_metadata, index_or_name, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def remove(self, index_or_name:typing.Any) -> None:
        """Remove a calc object from the collection."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._remove_metadata, index_or_name)

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(CalculationObjectCollection._metadata, CalculationObjectCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(CalculationObjectCollection._metadata, CalculationObjectCollection._get_count_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def cut(self, index_or_name:typing.Any) -> None:
        """Copy a calc object to the clipboard and removes the calc object from the list."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._cut_metadata, index_or_name)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def paste(self) -> "IComponentInfo":
        """Pastes a calc object from the clipboard into the list."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._paste_metadata, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("IComponentInfo"), agmarshall.InterfaceOutArg,) }
    def insert_copy(self, calc_obj:"IComponentInfo") -> "IComponentInfo":
        """Copy a calc object and inserts the copy into the list."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._insert_copy_metadata, calc_obj, OutArg())

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "IComponentInfo":
        """Retrieve a calc object found by the index."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "IComponentInfo":
        """Retrieve a calc object found by the name."""
        return self._intf.invoke(CalculationObjectCollection._metadata, CalculationObjectCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationObjectCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationObjectCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationObjectCollection, [CalculationObjectCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5427828449804903932, 12224863713293115564), CalculationObjectCollection)
agcls.AgTypeNameMap["CalculationObjectCollection"] = CalculationObjectCollection

class ConstraintCollection(SupportsDeleteCallback):
    """The list of constraints assigned to a stopping condition."""

    _num_methods = 10
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _add_method_offset = 1
    _item_method_offset = 2
    _remove_method_offset = 3
    _get__new_enum_method_offset = 4
    _get_count_method_offset = 5
    _cut_method_offset = 6
    _paste_method_offset = 7
    _insert_copy_method_offset = 8
    _get_item_by_index_method_offset = 9
    _get_item_by_name_method_offset = 10
    _metadata = {
        "iid_data" : (5396110646366435143, 4493176677956963970),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ConstraintCollection)
    def __iter__(self):
        """Create an iterator for the ConstraintCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "AsTriggerCondition":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def add(self, result_name:str) -> "AsTriggerCondition":
        """Add a constraint to the collection."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._add_metadata, result_name, OutArg())

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "AsTriggerCondition":
        """Iterate through the collection."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._item_metadata, index_or_name, OutArg())

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def remove(self, index_or_name:typing.Any) -> None:
        """Remove a specified constraint from the collection."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._remove_metadata, index_or_name)

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """A property that allows you to enumerate through the collection."""
        return self._intf.get_property(ConstraintCollection._metadata, ConstraintCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(ConstraintCollection._metadata, ConstraintCollection._get_count_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def cut(self, index_or_name:typing.Any) -> None:
        """Copy the constraint into the clipboard and removes the constraint from the list."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._cut_metadata, index_or_name)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def paste(self) -> "AsTriggerCondition":
        """Pastes the constraint from the clipboard and inserts into the list."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._paste_metadata, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("AsTriggerCondition"), agmarshall.InterfaceOutArg,) }
    def insert_copy(self, cond:"AsTriggerCondition") -> "AsTriggerCondition":
        """Copy the constraint and inserts the copy into the list."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._insert_copy_metadata, cond, OutArg())

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "AsTriggerCondition":
        """Retrieve a constraint from the collection by index."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "AsTriggerCondition":
        """Retrieve a constraint from the collection by name."""
        return self._intf.invoke(ConstraintCollection._metadata, ConstraintCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type ConstraintCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ConstraintCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ConstraintCollection, [ConstraintCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4630173644378793429, 7503241363423471795), ConstraintCollection)
agcls.AgTypeNameMap["ConstraintCollection"] = ConstraintCollection

class PluginProperties(SupportsDeleteCallback):
    """Properties of a plugin attitude control."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_property_method_offset = 1
    _set_property_method_offset = 2
    _get_available_properties_method_offset = 3
    _metadata = {
        "iid_data" : (5256416044241565756, 10938286104523749517),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, PluginProperties)

    _get_property_metadata = { "offset" : _get_property_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg,) }
    def get_property(self, path:str) -> typing.Any:
        """Get a property."""
        return self._intf.invoke(PluginProperties._metadata, PluginProperties._get_property_metadata, path, OutArg())

    _set_property_metadata = { "offset" : _set_property_method_offset,
            "arg_types" : (agcom.BSTR, agcom.Variant,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantArg,) }
    def set_property(self, path:str, property_value:typing.Any) -> None:
        """Set a property."""
        return self._intf.invoke(PluginProperties._metadata, PluginProperties._set_property_metadata, path, property_value)

    _get_available_properties_metadata = { "offset" : _get_available_properties_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_properties(self) -> list:
        """Return an array of all available properties."""
        return self._intf.get_property(PluginProperties._metadata, PluginProperties._get_available_properties_metadata)

    _property_names[available_properties] = "available_properties"

    def __init__(self, source_object=None):
        """Construct an object of type PluginProperties."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, PluginProperties)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, PluginProperties, [PluginProperties, ])

agcls.AgClassCatalog.add_catalog_entry((4832164181540462688, 9736795849403081643), PluginProperties)
agcls.AgTypeNameMap["PluginProperties"] = PluginProperties

class ProfileSearchPlugin(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties of a plugin search profile."""

    _num_methods = 9
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_controls_method_offset = 1
    _get_results_method_offset = 2
    _get_plugin_config_method_offset = 3
    _get_plugin_identifier_method_offset = 4
    _get_scripting_tool_method_offset = 5
    _get_reset_controls_before_run_method_offset = 6
    _set_reset_controls_before_run_method_offset = 7
    _get_targeter_graphs_method_offset = 8
    _get_log_file_method_offset = 9
    _metadata = {
        "iid_data" : (4989321574453629085, 18383769424312266673),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileSearchPlugin)

    _get_controls_metadata = { "offset" : _get_controls_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def controls(self) -> "SearchPluginControlCollection":
        """Get the selected control parameters."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_controls_metadata)

    _get_results_metadata = { "offset" : _get_results_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def results(self) -> "SearchPluginResultCollection":
        """Get the selected equality constraints."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_results_metadata)

    _get_plugin_config_metadata = { "offset" : _get_plugin_config_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def plugin_config(self) -> "PluginProperties":
        """Get the properties of the selected plugin."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_plugin_config_metadata)

    _get_plugin_identifier_metadata = { "offset" : _get_plugin_identifier_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def plugin_identifier(self) -> str:
        """Get the plugin identifier."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_plugin_identifier_metadata)

    _get_scripting_tool_metadata = { "offset" : _get_scripting_tool_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scripting_tool(self) -> "ScriptingTool":
        """Return the Scripting tool for the sequence."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_scripting_tool_metadata)

    _get_reset_controls_before_run_metadata = { "offset" : _get_reset_controls_before_run_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def reset_controls_before_run(self) -> bool:
        """Reset controls before each run."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_reset_controls_before_run_metadata)

    _set_reset_controls_before_run_metadata = { "offset" : _set_reset_controls_before_run_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @reset_controls_before_run.setter
    def reset_controls_before_run(self, value:bool) -> None:
        return self._intf.set_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._set_reset_controls_before_run_metadata, value)

    _get_targeter_graphs_metadata = { "offset" : _get_targeter_graphs_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def targeter_graphs(self) -> "TargeterGraphCollection":
        """Graphs."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_targeter_graphs_metadata)

    _get_log_file_metadata = { "offset" : _get_log_file_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def log_file(self) -> str:
        """Name of the log file for this profile."""
        return self._intf.get_property(ProfileSearchPlugin._metadata, ProfileSearchPlugin._get_log_file_metadata)

    _property_names[controls] = "controls"
    _property_names[results] = "results"
    _property_names[plugin_config] = "plugin_config"
    _property_names[plugin_identifier] = "plugin_identifier"
    _property_names[scripting_tool] = "scripting_tool"
    _property_names[reset_controls_before_run] = "reset_controls_before_run"
    _property_names[targeter_graphs] = "targeter_graphs"
    _property_names[log_file] = "log_file"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileSearchPlugin."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileSearchPlugin)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileSearchPlugin, [ProfileSearchPlugin, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5288551634889688931, 6097150253307945909), ProfileSearchPlugin)
agcls.AgTypeNameMap["ProfileSearchPlugin"] = ProfileSearchPlugin

class TargeterGraph(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Targeter Graph."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _set_name_method_offset = 2
    _get_generate_on_run_method_offset = 3
    _set_generate_on_run_method_offset = 4
    _get_user_comment_method_offset = 5
    _set_user_comment_method_offset = 6
    _get_show_label_iterations_method_offset = 7
    _set_show_label_iterations_method_offset = 8
    _get_show_desired_value_method_offset = 9
    _set_show_desired_value_method_offset = 10
    _get_show_tolerance_band_method_offset = 11
    _set_show_tolerance_band_method_offset = 12
    _get_independent_variable_method_offset = 13
    _set_independent_variable_method_offset = 14
    _get_active_controls_method_offset = 15
    _get_results_method_offset = 16
    _metadata = {
        "iid_data" : (5135924380442436999, 7678843553599670682),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraph)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get or set the name of the graph."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_name_metadata)

    _set_name_metadata = { "offset" : _set_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @name.setter
    def name(self, value:str) -> None:
        """Get or set the name of the graph."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_name_metadata, value)

    _get_generate_on_run_metadata = { "offset" : _get_generate_on_run_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def generate_on_run(self) -> bool:
        """Generate the graph while MCS is being run."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_generate_on_run_metadata)

    _set_generate_on_run_metadata = { "offset" : _set_generate_on_run_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @generate_on_run.setter
    def generate_on_run(self, value:bool) -> None:
        """Generate the graph while MCS is being run."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_generate_on_run_metadata, value)

    _get_user_comment_metadata = { "offset" : _get_user_comment_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def user_comment(self) -> str:
        """User Comment."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_user_comment_metadata)

    _set_user_comment_metadata = { "offset" : _set_user_comment_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @user_comment.setter
    def user_comment(self, value:str) -> None:
        """User Comment."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_user_comment_metadata, value)

    _get_show_label_iterations_metadata = { "offset" : _get_show_label_iterations_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_label_iterations(self) -> bool:
        """Label each of the points on the graph labeled according to its iteration number."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_show_label_iterations_metadata)

    _set_show_label_iterations_metadata = { "offset" : _set_show_label_iterations_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_label_iterations.setter
    def show_label_iterations(self, value:bool) -> None:
        """Label each of the points on the graph labeled according to its iteration number."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_show_label_iterations_metadata, value)

    _get_show_desired_value_metadata = { "offset" : _get_show_desired_value_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_desired_value(self) -> bool:
        """Show the desired equality constraint value(s) on the graph. Not applicable to SNOPT and IPOPT search profiles."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_show_desired_value_metadata)

    _set_show_desired_value_metadata = { "offset" : _set_show_desired_value_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_desired_value.setter
    def show_desired_value(self, value:bool) -> None:
        """Show the desired equality constraint value(s) on the graph. Not applicable to SNOPT and IPOPT search profiles."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_show_desired_value_metadata, value)

    _get_show_tolerance_band_metadata = { "offset" : _get_show_tolerance_band_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_tolerance_band(self) -> bool:
        """Show the tolerance band on the graph. Not applicable to SNOPT and IPOPT search profiles."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_show_tolerance_band_metadata)

    _set_show_tolerance_band_metadata = { "offset" : _set_show_tolerance_band_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_tolerance_band.setter
    def show_tolerance_band(self, value:bool) -> None:
        """Show the tolerance band on the graph. Not applicable to SNOPT and IPOPT search profiles."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_show_tolerance_band_metadata, value)

    _get_independent_variable_metadata = { "offset" : _get_independent_variable_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def independent_variable(self) -> str:
        """Select the graph's X axis."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_independent_variable_metadata)

    _set_independent_variable_metadata = { "offset" : _set_independent_variable_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @independent_variable.setter
    def independent_variable(self, value:str) -> None:
        """Select the graph's X axis."""
        return self._intf.set_property(TargeterGraph._metadata, TargeterGraph._set_independent_variable_metadata, value)

    _get_active_controls_metadata = { "offset" : _get_active_controls_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def active_controls(self) -> "TargeterGraphActiveControlCollection":
        """Active Controls."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_active_controls_metadata)

    _get_results_metadata = { "offset" : _get_results_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def results(self) -> "TargeterGraphResultCollection":
        """Results."""
        return self._intf.get_property(TargeterGraph._metadata, TargeterGraph._get_results_metadata)

    _property_names[name] = "name"
    _property_names[generate_on_run] = "generate_on_run"
    _property_names[user_comment] = "user_comment"
    _property_names[show_label_iterations] = "show_label_iterations"
    _property_names[show_desired_value] = "show_desired_value"
    _property_names[show_tolerance_band] = "show_tolerance_band"
    _property_names[independent_variable] = "independent_variable"
    _property_names[active_controls] = "active_controls"
    _property_names[results] = "results"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraph."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraph)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraph, [TargeterGraph, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4820292145119127894, 2993481562376072357), TargeterGraph)
agcls.AgTypeNameMap["TargeterGraph"] = TargeterGraph

class TargeterGraphCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """The list of User Variables accessed through the Driver."""

    _num_methods = 11
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _add_graph_method_offset = 4
    _remove_graph_method_offset = 5
    _get_provide_runtime_type_info_method_offset = 6
    _cut_method_offset = 7
    _paste_method_offset = 8
    _insert_copy_method_offset = 9
    _get_item_by_index_method_offset = 10
    _get_item_by_name_method_offset = 11
    _metadata = {
        "iid_data" : (4666281358055310374, 1442486553483265713),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraphCollection)
    def __iter__(self):
        """Create an iterator for the TargeterGraphCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "TargeterGraph":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "TargeterGraph":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(TargeterGraphCollection._metadata, TargeterGraphCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(TargeterGraphCollection._metadata, TargeterGraphCollection._get_count_metadata)

    _add_graph_metadata = { "offset" : _add_graph_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def add_graph(self) -> "TargeterGraph":
        """Add a new targeter graph."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._add_graph_metadata, OutArg())

    _remove_graph_metadata = { "offset" : _remove_graph_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def remove_graph(self, index_or_name:typing.Any) -> None:
        """Remove a targeter graph."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._remove_graph_metadata, index_or_name)

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(TargeterGraphCollection._metadata, TargeterGraphCollection._get_provide_runtime_type_info_metadata)

    _cut_metadata = { "offset" : _cut_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def cut(self, index_or_name:typing.Any) -> None:
        """Copy a targeter graph to the clipboard and removes the targeter graph from the list."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._cut_metadata, index_or_name)

    _paste_metadata = { "offset" : _paste_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def paste(self) -> "TargeterGraph":
        """Pastes a targeter graph from the clipboard into the list."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._paste_metadata, OutArg())

    _insert_copy_metadata = { "offset" : _insert_copy_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("TargeterGraph"), agmarshall.InterfaceOutArg,) }
    def insert_copy(self, graph:"TargeterGraph") -> "TargeterGraph":
        """Copy a targeter graph and inserts the copy into the list."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._insert_copy_metadata, graph, OutArg())

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "TargeterGraph":
        """Retrieve a targeter graph from the collection by index."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "TargeterGraph":
        """Retrieve a targeter graph from the collection by name."""
        return self._intf.invoke(TargeterGraphCollection._metadata, TargeterGraphCollection._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraphCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraphCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraphCollection, [TargeterGraphCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5713818234225934544, 4609064603796981650), TargeterGraphCollection)
agcls.AgTypeNameMap["TargeterGraphCollection"] = TargeterGraphCollection

class TargeterGraphResultCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Targeter graph results."""

    _num_methods = 4
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_provide_runtime_type_info_method_offset = 4
    _metadata = {
        "iid_data" : (5681214531064451260, 3542062518212208806),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraphResultCollection)
    def __iter__(self):
        """Create an iterator for the TargeterGraphResultCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "TargeterGraphResult":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "TargeterGraphResult":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(TargeterGraphResultCollection._metadata, TargeterGraphResultCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(TargeterGraphResultCollection._metadata, TargeterGraphResultCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(TargeterGraphResultCollection._metadata, TargeterGraphResultCollection._get_count_metadata)

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(TargeterGraphResultCollection._metadata, TargeterGraphResultCollection._get_provide_runtime_type_info_metadata)

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraphResultCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraphResultCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraphResultCollection, [TargeterGraphResultCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4675941918306799033, 16991213643032778938), TargeterGraphResultCollection)
agcls.AgTypeNameMap["TargeterGraphResultCollection"] = TargeterGraphResultCollection

class TargeterGraphActiveControlCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Targeter graph active controls."""

    _num_methods = 4
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_provide_runtime_type_info_method_offset = 4
    _metadata = {
        "iid_data" : (5718318377038943766, 11707308167199207819),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraphActiveControlCollection)
    def __iter__(self):
        """Create an iterator for the TargeterGraphActiveControlCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "TargeterGraphActiveControl":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "TargeterGraphActiveControl":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(TargeterGraphActiveControlCollection._metadata, TargeterGraphActiveControlCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(TargeterGraphActiveControlCollection._metadata, TargeterGraphActiveControlCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(TargeterGraphActiveControlCollection._metadata, TargeterGraphActiveControlCollection._get_count_metadata)

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(TargeterGraphActiveControlCollection._metadata, TargeterGraphActiveControlCollection._get_provide_runtime_type_info_metadata)

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraphActiveControlCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraphActiveControlCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraphActiveControlCollection, [TargeterGraphActiveControlCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5060958647911646068, 16483538695082057382), TargeterGraphActiveControlCollection)
agcls.AgTypeNameMap["TargeterGraphActiveControlCollection"] = TargeterGraphActiveControlCollection

class TargeterGraphActiveControl(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for targeter graph active control."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _get_parent_name_method_offset = 2
    _get_show_graph_value_method_offset = 3
    _set_show_graph_value_method_offset = 4
    _get_line_color_method_offset = 5
    _set_line_color_method_offset = 6
    _get_point_style_method_offset = 7
    _set_point_style_method_offset = 8
    _get_y_axis_method_offset = 9
    _set_y_axis_method_offset = 10
    _metadata = {
        "iid_data" : (5193288564717621701, 9284685105260980413),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraphActiveControl)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the active control."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_name_metadata)

    _get_parent_name_metadata = { "offset" : _get_parent_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_name(self) -> str:
        """Get the segment or component to which the element belongs."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_parent_name_metadata)

    _get_show_graph_value_metadata = { "offset" : _get_show_graph_value_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_graph_value(self) -> bool:
        """Show the value."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_show_graph_value_metadata)

    _set_show_graph_value_metadata = { "offset" : _set_show_graph_value_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_graph_value.setter
    def show_graph_value(self, value:bool) -> None:
        """Show the value."""
        return self._intf.set_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._set_show_graph_value_metadata, value)

    _get_line_color_metadata = { "offset" : _get_line_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def line_color(self) -> agcolor.Color:
        """Line Color."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_line_color_metadata)

    _set_line_color_metadata = { "offset" : _set_line_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @line_color.setter
    def line_color(self, value:agcolor.Color) -> None:
        """Line Color."""
        return self._intf.set_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._set_line_color_metadata, value)

    _get_point_style_metadata = { "offset" : _get_point_style_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def point_style(self) -> str:
        """Point Style."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_point_style_metadata)

    _set_point_style_metadata = { "offset" : _set_point_style_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @point_style.setter
    def point_style(self, value:str) -> None:
        """Point Style."""
        return self._intf.set_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._set_point_style_metadata, value)

    _get_y_axis_metadata = { "offset" : _get_y_axis_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def y_axis(self) -> str:
        """Select whether to display the control's value range on the left or right side of the graph."""
        return self._intf.get_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._get_y_axis_metadata)

    _set_y_axis_metadata = { "offset" : _set_y_axis_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @y_axis.setter
    def y_axis(self, value:str) -> None:
        """Select whether to display the control's value range on the left or right side of the graph."""
        return self._intf.set_property(TargeterGraphActiveControl._metadata, TargeterGraphActiveControl._set_y_axis_metadata, value)

    _property_names[name] = "name"
    _property_names[parent_name] = "parent_name"
    _property_names[show_graph_value] = "show_graph_value"
    _property_names[line_color] = "line_color"
    _property_names[point_style] = "point_style"
    _property_names[y_axis] = "y_axis"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraphActiveControl."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraphActiveControl)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraphActiveControl, [TargeterGraphActiveControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5123733805731250082, 10040985544979632029), TargeterGraphActiveControl)
agcls.AgTypeNameMap["TargeterGraphActiveControl"] = TargeterGraphActiveControl

class TargeterGraphResult(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for targeter graph result."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_name_method_offset = 1
    _get_parent_name_method_offset = 2
    _get_show_desired_value_method_offset = 3
    _set_show_desired_value_method_offset = 4
    _get_line_color_method_offset = 5
    _set_line_color_method_offset = 6
    _get_point_style_method_offset = 7
    _set_point_style_method_offset = 8
    _get_y_axis_method_offset = 9
    _set_y_axis_method_offset = 10
    _get_graph_option_method_offset = 11
    _set_graph_option_method_offset = 12
    _get_show_tolerance_band_method_offset = 13
    _set_show_tolerance_band_method_offset = 14
    _metadata = {
        "iid_data" : (5079886305028495981, 16269527803066934938),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TargeterGraphResult)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the result."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_name_metadata)

    _get_parent_name_metadata = { "offset" : _get_parent_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_name(self) -> str:
        """Get the segment or component for which this result has been selected."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_parent_name_metadata)

    _get_show_desired_value_metadata = { "offset" : _get_show_desired_value_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_desired_value(self) -> bool:
        """Show the desired value."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_show_desired_value_metadata)

    _set_show_desired_value_metadata = { "offset" : _set_show_desired_value_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_desired_value.setter
    def show_desired_value(self, value:bool) -> None:
        """Show the desired value."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_show_desired_value_metadata, value)

    _get_line_color_metadata = { "offset" : _get_line_color_method_offset,
            "arg_types" : (POINTER(agcom.OLE_COLOR),),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @property
    def line_color(self) -> agcolor.Color:
        """Line Color."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_line_color_metadata)

    _set_line_color_metadata = { "offset" : _set_line_color_method_offset,
            "arg_types" : (agcom.OLE_COLOR,),
            "marshallers" : (agmarshall.OLEColorArg,) }
    @line_color.setter
    def line_color(self, value:agcolor.Color) -> None:
        """Line Color."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_line_color_metadata, value)

    _get_point_style_metadata = { "offset" : _get_point_style_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def point_style(self) -> str:
        """Point Style."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_point_style_metadata)

    _set_point_style_metadata = { "offset" : _set_point_style_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @point_style.setter
    def point_style(self, value:str) -> None:
        """Point Style."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_point_style_metadata, value)

    _get_y_axis_metadata = { "offset" : _get_y_axis_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def y_axis(self) -> str:
        """Select whether to display the result's value range on the left or right side of the graph."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_y_axis_metadata)

    _set_y_axis_metadata = { "offset" : _set_y_axis_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @y_axis.setter
    def y_axis(self, value:str) -> None:
        """Select whether to display the result's value range on the left or right side of the graph."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_y_axis_metadata, value)

    _get_graph_option_metadata = { "offset" : _get_graph_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(GraphOption),) }
    @property
    def graph_option(self) -> "GraphOption":
        """Graph option."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_graph_option_metadata)

    _set_graph_option_metadata = { "offset" : _set_graph_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(GraphOption),) }
    @graph_option.setter
    def graph_option(self, value:"GraphOption") -> None:
        """Graph option."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_graph_option_metadata, value)

    _get_show_tolerance_band_metadata = { "offset" : _get_show_tolerance_band_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def show_tolerance_band(self) -> bool:
        """Show the tolerance band."""
        return self._intf.get_property(TargeterGraphResult._metadata, TargeterGraphResult._get_show_tolerance_band_metadata)

    _set_show_tolerance_band_metadata = { "offset" : _set_show_tolerance_band_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @show_tolerance_band.setter
    def show_tolerance_band(self, value:bool) -> None:
        """Show the tolerance band."""
        return self._intf.set_property(TargeterGraphResult._metadata, TargeterGraphResult._set_show_tolerance_band_metadata, value)

    _property_names[name] = "name"
    _property_names[parent_name] = "parent_name"
    _property_names[show_desired_value] = "show_desired_value"
    _property_names[line_color] = "line_color"
    _property_names[point_style] = "point_style"
    _property_names[y_axis] = "y_axis"
    _property_names[graph_option] = "graph_option"
    _property_names[show_tolerance_band] = "show_tolerance_band"

    def __init__(self, source_object=None):
        """Construct an object of type TargeterGraphResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TargeterGraphResult)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TargeterGraphResult, [TargeterGraphResult, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5372818770923136803, 2677805927587744941), TargeterGraphResult)
agcls.AgTypeNameMap["TargeterGraphResult"] = TargeterGraphResult

class ProfileDifferentialCorrector(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Differential Corrector profile."""

    _num_methods = 39
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_control_parameters_method_offset = 1
    _get_results_method_offset = 2
    _get_max_iterations_method_offset = 3
    _set_max_iterations_method_offset = 4
    _get_enable_display_status_method_offset = 5
    _set_enable_display_status_method_offset = 6
    _get_convergence_criteria_method_offset = 7
    _set_convergence_criteria_method_offset = 8
    _get_enable_line_search_method_offset = 9
    _set_enable_line_search_method_offset = 10
    _get_max_line_search_iterations_method_offset = 11
    _set_max_line_search_iterations_method_offset = 12
    _get_line_search_lower_bound_method_offset = 13
    _set_line_search_lower_bound_method_offset = 14
    _get_line_search_upper_bound_method_offset = 15
    _set_line_search_upper_bound_method_offset = 16
    _get_line_search_tolerance_method_offset = 17
    _set_line_search_tolerance_method_offset = 18
    _get_enable_homotopy_method_offset = 19
    _set_enable_homotopy_method_offset = 20
    _get_homotopy_steps_method_offset = 21
    _set_homotopy_steps_method_offset = 22
    _get_derivative_calc_method_method_offset = 23
    _set_derivative_calc_method_method_offset = 24
    _get_clear_corrections_before_run_method_offset = 25
    _set_clear_corrections_before_run_method_offset = 26
    _get_enable_b_plane_nominal_method_offset = 27
    _set_enable_b_plane_nominal_method_offset = 28
    _get_enable_b_plane_perturbations_method_offset = 29
    _set_enable_b_plane_perturbations_method_offset = 30
    _get_draw_perturbation_method_offset = 31
    _set_draw_perturbation_method_offset = 32
    _get_scripting_tool_method_offset = 33
    _get_root_finding_algorithm_method_offset = 34
    _set_root_finding_algorithm_method_offset = 35
    _get_number_of_iterations_method_offset = 36
    _get_targeter_graphs_method_offset = 37
    _get_stop_on_limit_cycle_detection_method_offset = 38
    _set_stop_on_limit_cycle_detection_method_offset = 39
    _metadata = {
        "iid_data" : (4651276936250783348, 17358117410539768209),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileDifferentialCorrector)

    _get_control_parameters_metadata = { "offset" : _get_control_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def control_parameters(self) -> "DifferentialCorrectorControlCollection":
        """Get the list of control parameters defined for the profile."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_control_parameters_metadata)

    _get_results_metadata = { "offset" : _get_results_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def results(self) -> "DifferentialCorrectorResultCollection":
        """Get the list of results defined for the profile."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_results_metadata)

    _get_max_iterations_metadata = { "offset" : _get_max_iterations_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def max_iterations(self) -> int:
        """Get or set the number of complete iterations of the profile to try before stopping. Dimensionless."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_max_iterations_metadata)

    _set_max_iterations_metadata = { "offset" : _set_max_iterations_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @max_iterations.setter
    def max_iterations(self, value:int) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_max_iterations_metadata, value)

    _get_enable_display_status_metadata = { "offset" : _get_enable_display_status_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_display_status(self) -> bool:
        """If true, a page will appear during the targeting run to report the status of the targeting effort in terms of proximity to the desired value for each dependent variable in the profile."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_enable_display_status_metadata)

    _set_enable_display_status_metadata = { "offset" : _set_enable_display_status_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_display_status.setter
    def enable_display_status(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_enable_display_status_metadata, value)

    _get_convergence_criteria_metadata = { "offset" : _get_convergence_criteria_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConvergenceCriteria),) }
    @property
    def convergence_criteria(self) -> "ConvergenceCriteria":
        """Get or set the convergence criteria."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_convergence_criteria_metadata)

    _set_convergence_criteria_metadata = { "offset" : _set_convergence_criteria_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ConvergenceCriteria),) }
    @convergence_criteria.setter
    def convergence_criteria(self, value:"ConvergenceCriteria") -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_convergence_criteria_metadata, value)

    _get_enable_line_search_metadata = { "offset" : _get_enable_line_search_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_line_search(self) -> bool:
        """If true, the profile will perform a line search."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_enable_line_search_metadata)

    _set_enable_line_search_metadata = { "offset" : _set_enable_line_search_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_line_search.setter
    def enable_line_search(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_enable_line_search_metadata, value)

    _get_max_line_search_iterations_metadata = { "offset" : _get_max_line_search_iterations_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def max_line_search_iterations(self) -> int:
        """Get or set the number of line search iterations to try before stopping. Dimensionless."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_max_line_search_iterations_metadata)

    _set_max_line_search_iterations_metadata = { "offset" : _set_max_line_search_iterations_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @max_line_search_iterations.setter
    def max_line_search_iterations(self, value:int) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_max_line_search_iterations_metadata, value)

    _get_line_search_lower_bound_metadata = { "offset" : _get_line_search_lower_bound_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def line_search_lower_bound(self) -> float:
        """Get or set the low boundary for the line search. Dimensionless."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_line_search_lower_bound_metadata)

    _set_line_search_lower_bound_metadata = { "offset" : _set_line_search_lower_bound_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @line_search_lower_bound.setter
    def line_search_lower_bound(self, value:float) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_line_search_lower_bound_metadata, value)

    _get_line_search_upper_bound_metadata = { "offset" : _get_line_search_upper_bound_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def line_search_upper_bound(self) -> float:
        """Get or set the high boundary for the line search. Dimensionless."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_line_search_upper_bound_metadata)

    _set_line_search_upper_bound_metadata = { "offset" : _set_line_search_upper_bound_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @line_search_upper_bound.setter
    def line_search_upper_bound(self, value:float) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_line_search_upper_bound_metadata, value)

    _get_line_search_tolerance_metadata = { "offset" : _get_line_search_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def line_search_tolerance(self) -> float:
        """Get or set the tolerance for the line search. Dimensionless."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_line_search_tolerance_metadata)

    _set_line_search_tolerance_metadata = { "offset" : _set_line_search_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @line_search_tolerance.setter
    def line_search_tolerance(self, value:float) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_line_search_tolerance_metadata, value)

    _get_enable_homotopy_metadata = { "offset" : _get_enable_homotopy_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_homotopy(self) -> bool:
        """If true, the profile will divide the problem into steps to solve it."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_enable_homotopy_metadata)

    _set_enable_homotopy_metadata = { "offset" : _set_enable_homotopy_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_homotopy.setter
    def enable_homotopy(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_enable_homotopy_metadata, value)

    _get_homotopy_steps_metadata = { "offset" : _get_homotopy_steps_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def homotopy_steps(self) -> int:
        """Get or set the number of steps to divide a problem into for a homotopic calculation. Dimensionless - ."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_homotopy_steps_metadata)

    _set_homotopy_steps_metadata = { "offset" : _set_homotopy_steps_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.LongArg,) }
    @homotopy_steps.setter
    def homotopy_steps(self, value:int) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_homotopy_steps_metadata, value)

    _get_derivative_calc_method_metadata = { "offset" : _get_derivative_calc_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DerivativeCalculationMethod),) }
    @property
    def derivative_calc_method(self) -> "DerivativeCalculationMethod":
        """Get or set the derivative calculation method."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_derivative_calc_method_metadata)

    _set_derivative_calc_method_metadata = { "offset" : _set_derivative_calc_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DerivativeCalculationMethod),) }
    @derivative_calc_method.setter
    def derivative_calc_method(self, value:"DerivativeCalculationMethod") -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_derivative_calc_method_metadata, value)

    _get_clear_corrections_before_run_metadata = { "offset" : _get_clear_corrections_before_run_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def clear_corrections_before_run(self) -> bool:
        """Clear Corrections Before Each Run - if true, the differential corrector is automatically reset each time that it is run, discarding information that was computed the last time it was run."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_clear_corrections_before_run_metadata)

    _set_clear_corrections_before_run_metadata = { "offset" : _set_clear_corrections_before_run_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @clear_corrections_before_run.setter
    def clear_corrections_before_run(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_clear_corrections_before_run_metadata, value)

    _get_enable_b_plane_nominal_metadata = { "offset" : _get_enable_b_plane_nominal_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_b_plane_nominal(self) -> bool:
        """If true, Astrogator will update the display of B-Planes for the nominal run of each iteration during the targeting process."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_enable_b_plane_nominal_metadata)

    _set_enable_b_plane_nominal_metadata = { "offset" : _set_enable_b_plane_nominal_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_b_plane_nominal.setter
    def enable_b_plane_nominal(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_enable_b_plane_nominal_metadata, value)

    _get_enable_b_plane_perturbations_metadata = { "offset" : _get_enable_b_plane_perturbations_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable_b_plane_perturbations(self) -> bool:
        """If true, Astrogator will update the display of B-Planes for both of the perturbations of each iteration during the targeting process."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_enable_b_plane_perturbations_metadata)

    _set_enable_b_plane_perturbations_metadata = { "offset" : _set_enable_b_plane_perturbations_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable_b_plane_perturbations.setter
    def enable_b_plane_perturbations(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_enable_b_plane_perturbations_metadata, value)

    _get_draw_perturbation_metadata = { "offset" : _get_draw_perturbation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DrawPerturbation),) }
    @property
    def draw_perturbation(self) -> "DrawPerturbation":
        """Define the display of perturbations in the 2D and 3D Graphics windows, if you have set Astrogator to draw while calculating."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_draw_perturbation_metadata)

    _set_draw_perturbation_metadata = { "offset" : _set_draw_perturbation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DrawPerturbation),) }
    @draw_perturbation.setter
    def draw_perturbation(self, value:"DrawPerturbation") -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_draw_perturbation_metadata, value)

    _get_scripting_tool_metadata = { "offset" : _get_scripting_tool_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scripting_tool(self) -> "ScriptingTool":
        """Return the Scripting tool for the sequence."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_scripting_tool_metadata)

    _get_root_finding_algorithm_metadata = { "offset" : _get_root_finding_algorithm_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(RootFindingAlgorithm),) }
    @property
    def root_finding_algorithm(self) -> "RootFindingAlgorithm":
        """Get or set the root-finding algorithm to use."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_root_finding_algorithm_metadata)

    _set_root_finding_algorithm_metadata = { "offset" : _set_root_finding_algorithm_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(RootFindingAlgorithm),) }
    @root_finding_algorithm.setter
    def root_finding_algorithm(self, value:"RootFindingAlgorithm") -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_root_finding_algorithm_metadata, value)

    _get_number_of_iterations_metadata = { "offset" : _get_number_of_iterations_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def number_of_iterations(self) -> int:
        """Get the number of iterations of the last run."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_number_of_iterations_metadata)

    _get_targeter_graphs_metadata = { "offset" : _get_targeter_graphs_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def targeter_graphs(self) -> "TargeterGraphCollection":
        """Graphs."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_targeter_graphs_metadata)

    _get_stop_on_limit_cycle_detection_metadata = { "offset" : _get_stop_on_limit_cycle_detection_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def stop_on_limit_cycle_detection(self) -> bool:
        """If true, Astrogator will stop targeting if a limit cycle is detected."""
        return self._intf.get_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._get_stop_on_limit_cycle_detection_metadata)

    _set_stop_on_limit_cycle_detection_metadata = { "offset" : _set_stop_on_limit_cycle_detection_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @stop_on_limit_cycle_detection.setter
    def stop_on_limit_cycle_detection(self, value:bool) -> None:
        return self._intf.set_property(ProfileDifferentialCorrector._metadata, ProfileDifferentialCorrector._set_stop_on_limit_cycle_detection_metadata, value)

    _property_names[control_parameters] = "control_parameters"
    _property_names[results] = "results"
    _property_names[max_iterations] = "max_iterations"
    _property_names[enable_display_status] = "enable_display_status"
    _property_names[convergence_criteria] = "convergence_criteria"
    _property_names[enable_line_search] = "enable_line_search"
    _property_names[max_line_search_iterations] = "max_line_search_iterations"
    _property_names[line_search_lower_bound] = "line_search_lower_bound"
    _property_names[line_search_upper_bound] = "line_search_upper_bound"
    _property_names[line_search_tolerance] = "line_search_tolerance"
    _property_names[enable_homotopy] = "enable_homotopy"
    _property_names[homotopy_steps] = "homotopy_steps"
    _property_names[derivative_calc_method] = "derivative_calc_method"
    _property_names[clear_corrections_before_run] = "clear_corrections_before_run"
    _property_names[enable_b_plane_nominal] = "enable_b_plane_nominal"
    _property_names[enable_b_plane_perturbations] = "enable_b_plane_perturbations"
    _property_names[draw_perturbation] = "draw_perturbation"
    _property_names[scripting_tool] = "scripting_tool"
    _property_names[root_finding_algorithm] = "root_finding_algorithm"
    _property_names[number_of_iterations] = "number_of_iterations"
    _property_names[targeter_graphs] = "targeter_graphs"
    _property_names[stop_on_limit_cycle_detection] = "stop_on_limit_cycle_detection"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileDifferentialCorrector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileDifferentialCorrector)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileDifferentialCorrector, [ProfileDifferentialCorrector, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5697793952066233454, 6970629070586778798), ProfileDifferentialCorrector)
agcls.AgTypeNameMap["ProfileDifferentialCorrector"] = ProfileDifferentialCorrector

class ProfileScriptingTool(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Scripting Tool profile."""

    _num_methods = 10
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_enable_method_offset = 1
    _set_enable_method_offset = 2
    _get_segment_properties_method_offset = 3
    _get_calculation_objects_method_offset = 4
    _get_parameters_method_offset = 5
    _get_language_type_method_offset = 6
    _set_language_type_method_offset = 7
    _script_text_method_offset = 8
    _copy_to_clipboard_method_offset = 9
    _paste_from_clipboard_method_offset = 10
    _metadata = {
        "iid_data" : (4932992111891026631, 13621962777049563066),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileScriptingTool)

    _get_enable_metadata = { "offset" : _get_enable_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable(self) -> bool:
        """If true, the scripting tool will be used."""
        return self._intf.get_property(ProfileScriptingTool._metadata, ProfileScriptingTool._get_enable_metadata)

    _set_enable_metadata = { "offset" : _set_enable_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable.setter
    def enable(self, value:bool) -> None:
        return self._intf.set_property(ProfileScriptingTool._metadata, ProfileScriptingTool._set_enable_metadata, value)

    _get_segment_properties_metadata = { "offset" : _get_segment_properties_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def segment_properties(self) -> "ScriptingSegmentCollection":
        """Return the collection of the segment properties."""
        return self._intf.get_property(ProfileScriptingTool._metadata, ProfileScriptingTool._get_segment_properties_metadata)

    _get_calculation_objects_metadata = { "offset" : _get_calculation_objects_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def calculation_objects(self) -> "ScriptingCalculationObjectCollection":
        """Return the collection of the calc objects."""
        return self._intf.get_property(ProfileScriptingTool._metadata, ProfileScriptingTool._get_calculation_objects_metadata)

    _get_parameters_metadata = { "offset" : _get_parameters_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def parameters(self) -> "ScriptingParameterCollection":
        """Return the collection of parameters."""
        return self._intf.get_property(ProfileScriptingTool._metadata, ProfileScriptingTool._get_parameters_metadata)

    _get_language_type_metadata = { "offset" : _get_language_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(Language),) }
    @property
    def language_type(self) -> "Language":
        """Get or set the scripting language being used."""
        return self._intf.get_property(ProfileScriptingTool._metadata, ProfileScriptingTool._get_language_type_metadata)

    _set_language_type_metadata = { "offset" : _set_language_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(Language),) }
    @language_type.setter
    def language_type(self, value:"Language") -> None:
        return self._intf.set_property(ProfileScriptingTool._metadata, ProfileScriptingTool._set_language_type_metadata, value)

    _script_text_metadata = { "offset" : _script_text_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def script_text(self, script:str) -> None:
        """Injects the script into the scripting tool."""
        return self._intf.invoke(ProfileScriptingTool._metadata, ProfileScriptingTool._script_text_metadata, script)

    _copy_to_clipboard_metadata = { "offset" : _copy_to_clipboard_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def copy_to_clipboard(self) -> None:
        """Copy entire scripting tool to clipboard."""
        return self._intf.invoke(ProfileScriptingTool._metadata, ProfileScriptingTool._copy_to_clipboard_metadata, )

    _paste_from_clipboard_metadata = { "offset" : _paste_from_clipboard_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def paste_from_clipboard(self) -> None:
        """Replace entire scripting tool with scripting tool in clipboard."""
        return self._intf.invoke(ProfileScriptingTool._metadata, ProfileScriptingTool._paste_from_clipboard_metadata, )

    _property_names[enable] = "enable"
    _property_names[segment_properties] = "segment_properties"
    _property_names[calculation_objects] = "calculation_objects"
    _property_names[parameters] = "parameters"
    _property_names[language_type] = "language_type"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileScriptingTool."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileScriptingTool)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileScriptingTool, [ProfileScriptingTool, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5081539591855218287, 5356479670344280495), ProfileScriptingTool)
agcls.AgTypeNameMap["ProfileScriptingTool"] = ProfileScriptingTool

class DifferentialCorrectorControl(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for control parameters of a differential corrector profile."""

    _num_methods = 25
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_enable_method_offset = 1
    _set_enable_method_offset = 2
    _get_name_method_offset = 3
    _get_final_value_method_offset = 4
    _get_last_update_method_offset = 5
    _get_parent_name_method_offset = 6
    _get_initial_value_method_offset = 7
    _get_perturbation_method_offset = 8
    _set_perturbation_method_offset = 9
    _get_correction_method_offset = 10
    _set_correction_method_offset = 11
    _get_tolerance_method_offset = 12
    _set_tolerance_method_offset = 13
    _get_max_step_method_offset = 14
    _set_max_step_method_offset = 15
    _get_scaling_method_method_offset = 16
    _set_scaling_method_method_offset = 17
    _get_scaling_value_method_offset = 18
    _set_scaling_value_method_offset = 19
    _get_dimension_method_offset = 20
    _get_use_custom_display_unit_method_offset = 21
    _set_use_custom_display_unit_method_offset = 22
    _get_custom_display_unit_method_offset = 23
    _set_custom_display_unit_method_offset = 24
    _get_values_method_offset = 25
    _metadata = {
        "iid_data" : (5253000958888501130, 7959737892745020576),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DifferentialCorrectorControl)

    _get_enable_metadata = { "offset" : _get_enable_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable(self) -> bool:
        """If true, the variable is being used."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_enable_metadata)

    _set_enable_metadata = { "offset" : _set_enable_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable.setter
    def enable(self, value:bool) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_enable_metadata, value)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the parameter."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_name_metadata)

    _get_final_value_metadata = { "offset" : _get_final_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def final_value(self) -> typing.Any:
        """Get the value of the independent variable after the last targeter run."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_final_value_metadata)

    _get_last_update_metadata = { "offset" : _get_last_update_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def last_update(self) -> typing.Any:
        """Get the amount by which the value of the independent variable changed during the last targeter run."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_last_update_metadata)

    _get_parent_name_metadata = { "offset" : _get_parent_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_name(self) -> str:
        """Get the name of the segment to which the parameter belongs."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_parent_name_metadata)

    _get_initial_value_metadata = { "offset" : _get_initial_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def initial_value(self) -> typing.Any:
        """Get the nominal value of the element selected as a parameter. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_initial_value_metadata)

    _get_perturbation_metadata = { "offset" : _get_perturbation_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def perturbation(self) -> typing.Any:
        """Get the value to be used in calculating numerical derivatives. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_perturbation_metadata)

    _set_perturbation_metadata = { "offset" : _set_perturbation_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @perturbation.setter
    def perturbation(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_perturbation_metadata, value)

    _get_correction_metadata = { "offset" : _get_correction_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def correction(self) -> typing.Any:
        """Get the amount by which the nominal value of the parameter should be corrected to achieve the selected goals. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_correction_metadata)

    _set_correction_metadata = { "offset" : _set_correction_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @correction.setter
    def correction(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_correction_metadata, value)

    _get_tolerance_metadata = { "offset" : _get_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def tolerance(self) -> typing.Any:
        """Get or set the smallest update to the parameter to be made before the targeter stops. Only used if the convergence criteria is set to 'Either equality constraints or last control parameter updates within tolerance'. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_tolerance_metadata)

    _set_tolerance_metadata = { "offset" : _set_tolerance_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @tolerance.setter
    def tolerance(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_tolerance_metadata, value)

    _get_max_step_metadata = { "offset" : _get_max_step_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def max_step(self) -> typing.Any:
        """Get or set the maximum increment to make to the value of the parameter in any one step. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_max_step_metadata)

    _set_max_step_metadata = { "offset" : _set_max_step_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @max_step.setter
    def max_step(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_max_step_metadata, value)

    _get_scaling_method_metadata = { "offset" : _get_scaling_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DifferentialCorrectorScalingMethod),) }
    @property
    def scaling_method(self) -> "DifferentialCorrectorScalingMethod":
        """Allow better numerical behavior if the parameters have very different magnitudes. The same scaling method is applied to all parameters."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_scaling_method_metadata)

    _set_scaling_method_metadata = { "offset" : _set_scaling_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DifferentialCorrectorScalingMethod),) }
    @scaling_method.setter
    def scaling_method(self, value:"DifferentialCorrectorScalingMethod") -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_scaling_method_metadata, value)

    _get_scaling_value_metadata = { "offset" : _get_scaling_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def scaling_value(self) -> typing.Any:
        """Apply to the Specified Value scaling method. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_scaling_value_metadata)

    _set_scaling_value_metadata = { "offset" : _set_scaling_value_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @scaling_value.setter
    def scaling_value(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_scaling_value_metadata, value)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Dimension of the constraint."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_dimension_metadata)

    _get_use_custom_display_unit_metadata = { "offset" : _get_use_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_display_unit(self) -> bool:
        """If true, allows display of values in another unit."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_use_custom_display_unit_metadata)

    _set_use_custom_display_unit_metadata = { "offset" : _set_use_custom_display_unit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_display_unit.setter
    def use_custom_display_unit(self, value:bool) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_use_custom_display_unit_metadata, value)

    _get_custom_display_unit_metadata = { "offset" : _get_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def custom_display_unit(self) -> str:
        """Get or set the unit in which the value will be displayed in the GUI."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_custom_display_unit_metadata)

    _set_custom_display_unit_metadata = { "offset" : _set_custom_display_unit_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @custom_display_unit.setter
    def custom_display_unit(self, value:str) -> None:
        return self._intf.set_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._set_custom_display_unit_metadata, value)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """List of values of this independent variable at each iteration, including nominal run. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorControl._metadata, DifferentialCorrectorControl._get_values_metadata)

    _property_names[enable] = "enable"
    _property_names[name] = "name"
    _property_names[final_value] = "final_value"
    _property_names[last_update] = "last_update"
    _property_names[parent_name] = "parent_name"
    _property_names[initial_value] = "initial_value"
    _property_names[perturbation] = "perturbation"
    _property_names[correction] = "correction"
    _property_names[tolerance] = "tolerance"
    _property_names[max_step] = "max_step"
    _property_names[scaling_method] = "scaling_method"
    _property_names[scaling_value] = "scaling_value"
    _property_names[dimension] = "dimension"
    _property_names[use_custom_display_unit] = "use_custom_display_unit"
    _property_names[custom_display_unit] = "custom_display_unit"
    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type DifferentialCorrectorControl."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DifferentialCorrectorControl)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DifferentialCorrectorControl, [DifferentialCorrectorControl, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5558465406678164076, 14489559354740820113), DifferentialCorrectorControl)
agcls.AgTypeNameMap["DifferentialCorrectorControl"] = DifferentialCorrectorControl

class DifferentialCorrectorResult(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for equality constraints of a differential corrector profile."""

    _num_methods = 22
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_enable_method_offset = 1
    _set_enable_method_offset = 2
    _get_name_method_offset = 3
    _get_desired_value_method_offset = 4
    _set_desired_value_method_offset = 5
    _get_current_value_method_offset = 6
    _get_parent_name_method_offset = 7
    _get_difference_method_offset = 8
    _get_tolerance_method_offset = 9
    _set_tolerance_method_offset = 10
    _get_scaling_method_method_offset = 11
    _set_scaling_method_method_offset = 12
    _get_scaling_value_method_offset = 13
    _set_scaling_value_method_offset = 14
    _get_weight_method_offset = 15
    _set_weight_method_offset = 16
    _get_dimension_method_offset = 17
    _get_use_custom_display_unit_method_offset = 18
    _set_use_custom_display_unit_method_offset = 19
    _get_custom_display_unit_method_offset = 20
    _set_custom_display_unit_method_offset = 21
    _get_values_method_offset = 22
    _metadata = {
        "iid_data" : (5087800593462724886, 6103903351099707307),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DifferentialCorrectorResult)

    _get_enable_metadata = { "offset" : _get_enable_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def enable(self) -> bool:
        """If true, the variable is being used."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_enable_metadata)

    _set_enable_metadata = { "offset" : _set_enable_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @enable.setter
    def enable(self, value:bool) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_enable_metadata, value)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Get the name of the parameter."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_name_metadata)

    _get_desired_value_metadata = { "offset" : _get_desired_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def desired_value(self) -> typing.Any:
        """Get or set the desired value. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_desired_value_metadata)

    _set_desired_value_metadata = { "offset" : _set_desired_value_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @desired_value.setter
    def desired_value(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_desired_value_metadata, value)

    _get_current_value_metadata = { "offset" : _get_current_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def current_value(self) -> typing.Any:
        """Get the value achieved for this dependent variable in the last targeter run. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_current_value_metadata)

    _get_parent_name_metadata = { "offset" : _get_parent_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_name(self) -> str:
        """Object - the name of the segment to which the parameter belongs."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_parent_name_metadata)

    _get_difference_metadata = { "offset" : _get_difference_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def difference(self) -> typing.Any:
        """Get the difference between the current and desired value for this dependent variable. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_difference_metadata)

    _get_tolerance_metadata = { "offset" : _get_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def tolerance(self) -> typing.Any:
        """How close the targeter should come to the desired value before stopping. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_tolerance_metadata)

    _set_tolerance_metadata = { "offset" : _set_tolerance_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @tolerance.setter
    def tolerance(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_tolerance_metadata, value)

    _get_scaling_method_metadata = { "offset" : _get_scaling_method_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DifferentialCorrectorScalingMethod),) }
    @property
    def scaling_method(self) -> "DifferentialCorrectorScalingMethod":
        """Allow better numerical behavior if the constraints have very different magnitudes. The same scaling method is applied to all constraints."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_scaling_method_metadata)

    _set_scaling_method_metadata = { "offset" : _set_scaling_method_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(DifferentialCorrectorScalingMethod),) }
    @scaling_method.setter
    def scaling_method(self, value:"DifferentialCorrectorScalingMethod") -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_scaling_method_metadata, value)

    _get_scaling_value_metadata = { "offset" : _get_scaling_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def scaling_value(self) -> typing.Any:
        """Apply to the Specified Value scaling method. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_scaling_value_metadata)

    _set_scaling_value_metadata = { "offset" : _set_scaling_value_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @scaling_value.setter
    def scaling_value(self, value:typing.Any) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_scaling_value_metadata, value)

    _get_weight_metadata = { "offset" : _get_weight_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def weight(self) -> float:
        """Get or set the factor by which the constraint error is to be multiplied. This is used to emphasize/de-emphasize the importance of one constraint relative to the others. Dimensionless."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_weight_metadata)

    _set_weight_metadata = { "offset" : _set_weight_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @weight.setter
    def weight(self, value:float) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_weight_metadata, value)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Get the dimension of the values."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_dimension_metadata)

    _get_use_custom_display_unit_metadata = { "offset" : _get_use_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_display_unit(self) -> bool:
        """If true, allows display of values in another unit."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_use_custom_display_unit_metadata)

    _set_use_custom_display_unit_metadata = { "offset" : _set_use_custom_display_unit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_display_unit.setter
    def use_custom_display_unit(self, value:bool) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_use_custom_display_unit_metadata, value)

    _get_custom_display_unit_metadata = { "offset" : _get_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def custom_display_unit(self) -> str:
        """Get or set the unit in which the value will be displayed in the GUI."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_custom_display_unit_metadata)

    _set_custom_display_unit_metadata = { "offset" : _set_custom_display_unit_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @custom_display_unit.setter
    def custom_display_unit(self, value:str) -> None:
        return self._intf.set_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._set_custom_display_unit_metadata, value)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """List of values of this dependent variable at each iteration, including nominal run. Dimension depends on context."""
        return self._intf.get_property(DifferentialCorrectorResult._metadata, DifferentialCorrectorResult._get_values_metadata)

    _property_names[enable] = "enable"
    _property_names[name] = "name"
    _property_names[desired_value] = "desired_value"
    _property_names[current_value] = "current_value"
    _property_names[parent_name] = "parent_name"
    _property_names[difference] = "difference"
    _property_names[tolerance] = "tolerance"
    _property_names[scaling_method] = "scaling_method"
    _property_names[scaling_value] = "scaling_value"
    _property_names[weight] = "weight"
    _property_names[dimension] = "dimension"
    _property_names[use_custom_display_unit] = "use_custom_display_unit"
    _property_names[custom_display_unit] = "custom_display_unit"
    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type DifferentialCorrectorResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DifferentialCorrectorResult)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DifferentialCorrectorResult, [DifferentialCorrectorResult, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4848528198033868470, 13072468517312703888), DifferentialCorrectorResult)
agcls.AgTypeNameMap["DifferentialCorrectorResult"] = DifferentialCorrectorResult

class DifferentialCorrectorControlCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for the list of control parameters for a differential corrector profile."""

    _num_methods = 5
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_control_by_paths_method_offset = 4
    _get_provide_runtime_type_info_method_offset = 5
    _metadata = {
        "iid_data" : (5331395578659733574, 4228290395270634151),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DifferentialCorrectorControlCollection)
    def __iter__(self):
        """Create an iterator for the DifferentialCorrectorControlCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "DifferentialCorrectorControl":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "DifferentialCorrectorControl":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(DifferentialCorrectorControlCollection._metadata, DifferentialCorrectorControlCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(DifferentialCorrectorControlCollection._metadata, DifferentialCorrectorControlCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(DifferentialCorrectorControlCollection._metadata, DifferentialCorrectorControlCollection._get_count_metadata)

    _get_control_by_paths_metadata = { "offset" : _get_control_by_paths_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_control_by_paths(self, object_path:str, control_path:str) -> "DifferentialCorrectorControl":
        """Return the control specified by the object/control path."""
        return self._intf.invoke(DifferentialCorrectorControlCollection._metadata, DifferentialCorrectorControlCollection._get_control_by_paths_metadata, object_path, control_path, OutArg())

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(DifferentialCorrectorControlCollection._metadata, DifferentialCorrectorControlCollection._get_provide_runtime_type_info_metadata)

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type DifferentialCorrectorControlCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DifferentialCorrectorControlCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DifferentialCorrectorControlCollection, [DifferentialCorrectorControlCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5052038889388285920, 16755659920294581388), DifferentialCorrectorControlCollection)
agcls.AgTypeNameMap["DifferentialCorrectorControlCollection"] = DifferentialCorrectorControlCollection

class DifferentialCorrectorResultCollection(IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Differential Corrector result collection."""

    _num_methods = 5
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_result_by_paths_method_offset = 4
    _get_provide_runtime_type_info_method_offset = 5
    _metadata = {
        "iid_data" : (4818834817002475399, 15340702828551915904),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, DifferentialCorrectorResultCollection)
    def __iter__(self):
        """Create an iterator for the DifferentialCorrectorResultCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "DifferentialCorrectorResult":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "DifferentialCorrectorResult":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(DifferentialCorrectorResultCollection._metadata, DifferentialCorrectorResultCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(DifferentialCorrectorResultCollection._metadata, DifferentialCorrectorResultCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(DifferentialCorrectorResultCollection._metadata, DifferentialCorrectorResultCollection._get_count_metadata)

    _get_result_by_paths_metadata = { "offset" : _get_result_by_paths_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_result_by_paths(self, object_path:str, result_path:str) -> "DifferentialCorrectorResult":
        """Return the result specified by the object/result path."""
        return self._intf.invoke(DifferentialCorrectorResultCollection._metadata, DifferentialCorrectorResultCollection._get_result_by_paths_metadata, object_path, result_path, OutArg())

    _get_provide_runtime_type_info_metadata = { "offset" : _get_provide_runtime_type_info_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def provide_runtime_type_info(self) -> "RuntimeTypeInfo":
        """Return the RuntimeTypeInfo interface to access properties at runtime."""
        return self._intf.get_property(DifferentialCorrectorResultCollection._metadata, DifferentialCorrectorResultCollection._get_provide_runtime_type_info_metadata)

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"
    _property_names[provide_runtime_type_info] = "provide_runtime_type_info"

    def __init__(self, source_object=None):
        """Construct an object of type DifferentialCorrectorResultCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, DifferentialCorrectorResultCollection)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, DifferentialCorrectorResultCollection, [DifferentialCorrectorResultCollection, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5420089016781372257, 3020197927996230316), DifferentialCorrectorResultCollection)
agcls.AgTypeNameMap["DifferentialCorrectorResultCollection"] = DifferentialCorrectorResultCollection

class SearchPluginControl(SupportsDeleteCallback):
    """Properties of search plugin control parameters."""

    _num_methods = 13
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_control_name_method_offset = 1
    _get_current_value_method_offset = 2
    _get_parent_segment_name_method_offset = 3
    _get_initial_value_method_offset = 4
    _get_plugin_identifier_method_offset = 5
    _get_plugin_config_method_offset = 6
    _get_dimension_method_offset = 7
    _get_use_custom_display_unit_method_offset = 8
    _set_use_custom_display_unit_method_offset = 9
    _get_custom_display_unit_method_offset = 10
    _set_custom_display_unit_method_offset = 11
    _get_values_method_offset = 12
    _set_current_value_method_offset = 13
    _metadata = {
        "iid_data" : (4867399211973024127, 7320608480965167759),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SearchPluginControl)

    _get_control_name_metadata = { "offset" : _get_control_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def control_name(self) -> str:
        """Get the name of the control parameter."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_control_name_metadata)

    _get_current_value_metadata = { "offset" : _get_current_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def current_value(self) -> typing.Any:
        """Get the current value of the control parameter."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_current_value_metadata)

    _get_parent_segment_name_metadata = { "offset" : _get_parent_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_segment_name(self) -> str:
        """Get the parent segment of the control parameter."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_parent_segment_name_metadata)

    _get_initial_value_metadata = { "offset" : _get_initial_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def initial_value(self) -> typing.Any:
        """Get the initial value of the control parameter. Dimension depends on context."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_initial_value_metadata)

    _get_plugin_identifier_metadata = { "offset" : _get_plugin_identifier_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def plugin_identifier(self) -> str:
        """Get the plugin identifier."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_plugin_identifier_metadata)

    _get_plugin_config_metadata = { "offset" : _get_plugin_config_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def plugin_config(self) -> "PluginProperties":
        """Get the properties of the selected plugin."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_plugin_config_metadata)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Get the dimension of the values."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_dimension_metadata)

    _get_use_custom_display_unit_metadata = { "offset" : _get_use_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_display_unit(self) -> bool:
        """If true, allows display of values in another unit."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_use_custom_display_unit_metadata)

    _set_use_custom_display_unit_metadata = { "offset" : _set_use_custom_display_unit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_display_unit.setter
    def use_custom_display_unit(self, value:bool) -> None:
        return self._intf.set_property(SearchPluginControl._metadata, SearchPluginControl._set_use_custom_display_unit_metadata, value)

    _get_custom_display_unit_metadata = { "offset" : _get_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def custom_display_unit(self) -> str:
        """Get or set the unit in which the value will be displayed in the GUI."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_custom_display_unit_metadata)

    _set_custom_display_unit_metadata = { "offset" : _set_custom_display_unit_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @custom_display_unit.setter
    def custom_display_unit(self, value:str) -> None:
        return self._intf.set_property(SearchPluginControl._metadata, SearchPluginControl._set_custom_display_unit_metadata, value)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """List of values of this independent variable at each iteration, including nominal run. Dimension depends on context."""
        return self._intf.get_property(SearchPluginControl._metadata, SearchPluginControl._get_values_metadata)

    _set_current_value_metadata = { "offset" : _set_current_value_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @current_value.setter
    def current_value(self, value:typing.Any) -> None:
        return self._intf.set_property(SearchPluginControl._metadata, SearchPluginControl._set_current_value_metadata, value)

    _property_names[control_name] = "control_name"
    _property_names[current_value] = "current_value"
    _property_names[parent_segment_name] = "parent_segment_name"
    _property_names[initial_value] = "initial_value"
    _property_names[plugin_identifier] = "plugin_identifier"
    _property_names[plugin_config] = "plugin_config"
    _property_names[dimension] = "dimension"
    _property_names[use_custom_display_unit] = "use_custom_display_unit"
    _property_names[custom_display_unit] = "custom_display_unit"
    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type SearchPluginControl."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SearchPluginControl)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SearchPluginControl, [SearchPluginControl, ])

agcls.AgClassCatalog.add_catalog_entry((5045474660547290749, 17356533529411786427), SearchPluginControl)
agcls.AgTypeNameMap["SearchPluginControl"] = SearchPluginControl

class SearchPluginControlCollection(SupportsDeleteCallback):
    """Properties for the list of search plugin control parameters."""

    _num_methods = 4
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_control_by_paths_method_offset = 4
    _metadata = {
        "iid_data" : (4624987267695215442, 5534608355596712125),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SearchPluginControlCollection)
    def __iter__(self):
        """Create an iterator for the SearchPluginControlCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "SearchPluginControl":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "SearchPluginControl":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(SearchPluginControlCollection._metadata, SearchPluginControlCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Property that allows you to enumerate through the collection."""
        return self._intf.get_property(SearchPluginControlCollection._metadata, SearchPluginControlCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return the size of the collection."""
        return self._intf.get_property(SearchPluginControlCollection._metadata, SearchPluginControlCollection._get_count_metadata)

    _get_control_by_paths_metadata = { "offset" : _get_control_by_paths_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_control_by_paths(self, object_path:str, control_path:str) -> "SearchPluginControl":
        """Return the control specified by the object/control path."""
        return self._intf.invoke(SearchPluginControlCollection._metadata, SearchPluginControlCollection._get_control_by_paths_metadata, object_path, control_path, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type SearchPluginControlCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SearchPluginControlCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SearchPluginControlCollection, [SearchPluginControlCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4848900109341529946, 9106163264584221337), SearchPluginControlCollection)
agcls.AgTypeNameMap["SearchPluginControlCollection"] = SearchPluginControlCollection

class SearchPluginResult(SupportsDeleteCallback):
    """Properties of search plugin equality constraints."""

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_result_name_method_offset = 1
    _get_current_value_method_offset = 2
    _get_parent_segment_name_method_offset = 3
    _get_plugin_identifier_method_offset = 4
    _get_plugin_config_method_offset = 5
    _get_dimension_method_offset = 6
    _get_use_custom_display_unit_method_offset = 7
    _set_use_custom_display_unit_method_offset = 8
    _get_custom_display_unit_method_offset = 9
    _set_custom_display_unit_method_offset = 10
    _get_values_method_offset = 11
    _metadata = {
        "iid_data" : (4895148609240169264, 4585343154220999064),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SearchPluginResult)

    _get_result_name_metadata = { "offset" : _get_result_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def result_name(self) -> str:
        """Get the name of the equality constraint."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_result_name_metadata)

    _get_current_value_metadata = { "offset" : _get_current_value_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def current_value(self) -> typing.Any:
        """Get the current value of the result variable."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_current_value_metadata)

    _get_parent_segment_name_metadata = { "offset" : _get_parent_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def parent_segment_name(self) -> str:
        """Get the parent segment of the result variable."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_parent_segment_name_metadata)

    _get_plugin_identifier_metadata = { "offset" : _get_plugin_identifier_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def plugin_identifier(self) -> str:
        """Get the plugin identifier."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_plugin_identifier_metadata)

    _get_plugin_config_metadata = { "offset" : _get_plugin_config_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def plugin_config(self) -> "PluginProperties":
        """Get the properties of the selected plugin."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_plugin_config_metadata)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Get the dimension of the values."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_dimension_metadata)

    _get_use_custom_display_unit_metadata = { "offset" : _get_use_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_display_unit(self) -> bool:
        """If true, allows display of values in another unit."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_use_custom_display_unit_metadata)

    _set_use_custom_display_unit_metadata = { "offset" : _set_use_custom_display_unit_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_display_unit.setter
    def use_custom_display_unit(self, value:bool) -> None:
        return self._intf.set_property(SearchPluginResult._metadata, SearchPluginResult._set_use_custom_display_unit_metadata, value)

    _get_custom_display_unit_metadata = { "offset" : _get_custom_display_unit_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def custom_display_unit(self) -> str:
        """Get or set the unit in which the value will be displayed in the GUI."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_custom_display_unit_metadata)

    _set_custom_display_unit_metadata = { "offset" : _set_custom_display_unit_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @custom_display_unit.setter
    def custom_display_unit(self, value:str) -> None:
        return self._intf.set_property(SearchPluginResult._metadata, SearchPluginResult._set_custom_display_unit_metadata, value)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """List of values of this dependent variable at each iteration, including nominal run. Dimension depends on context."""
        return self._intf.get_property(SearchPluginResult._metadata, SearchPluginResult._get_values_metadata)

    _property_names[result_name] = "result_name"
    _property_names[current_value] = "current_value"
    _property_names[parent_segment_name] = "parent_segment_name"
    _property_names[plugin_identifier] = "plugin_identifier"
    _property_names[plugin_config] = "plugin_config"
    _property_names[dimension] = "dimension"
    _property_names[use_custom_display_unit] = "use_custom_display_unit"
    _property_names[custom_display_unit] = "custom_display_unit"
    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type SearchPluginResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SearchPluginResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SearchPluginResult, [SearchPluginResult, ])

agcls.AgClassCatalog.add_catalog_entry((5546957583458452990, 17896374163675725444), SearchPluginResult)
agcls.AgTypeNameMap["SearchPluginResult"] = SearchPluginResult

class SearchPluginResultCollection(SupportsDeleteCallback):
    """Properties for the list of search plugin equality constraints."""

    _num_methods = 4
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _item_method_offset = 1
    _get__new_enum_method_offset = 2
    _get_count_method_offset = 3
    _get_result_by_paths_method_offset = 4
    _metadata = {
        "iid_data" : (4689816242179407525, 118837404913377457),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SearchPluginResultCollection)
    def __iter__(self):
        """Create an iterator for the SearchPluginResultCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "SearchPluginResult":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "SearchPluginResult":
        """Allow you to iterate through the collection."""
        return self._intf.invoke(SearchPluginResultCollection._metadata, SearchPluginResultCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Function to enumerate through the collection."""
        return self._intf.get_property(SearchPluginResultCollection._metadata, SearchPluginResultCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Size of the collection."""
        return self._intf.get_property(SearchPluginResultCollection._metadata, SearchPluginResultCollection._get_count_metadata)

    _get_result_by_paths_metadata = { "offset" : _get_result_by_paths_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_result_by_paths(self, object_path:str, result_path:str) -> "SearchPluginResult":
        """Return the result specified by the object/result path."""
        return self._intf.invoke(SearchPluginResultCollection._metadata, SearchPluginResultCollection._get_result_by_paths_metadata, object_path, result_path, OutArg())

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type SearchPluginResultCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SearchPluginResultCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SearchPluginResultCollection, [SearchPluginResultCollection, ])

agcls.AgClassCatalog.add_catalog_entry((4906992132230204902, 3876273479435196306), SearchPluginResultCollection)
agcls.AgTypeNameMap["SearchPluginResultCollection"] = SearchPluginResultCollection

class ProfileChangeManeuverType(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Change Maneuver Type profile."""

    _num_methods = 4
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_method_offset = 1
    _set_segment_method_offset = 2
    _get_maneuver_type_method_offset = 3
    _set_maneuver_type_method_offset = 4
    _metadata = {
        "iid_data" : (4638741168878062177, 15357931456736488104),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileChangeManeuverType)

    _get_segment_metadata = { "offset" : _get_segment_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def segment(self) -> "MCSManeuver":
        """Get or set the targeted maneuver segment."""
        return self._intf.get_property(ProfileChangeManeuverType._metadata, ProfileChangeManeuverType._get_segment_metadata)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("MCSManeuver"),) }
    @segment.setter
    def segment(self, segment:"MCSManeuver") -> None:
        return self._intf.set_property(ProfileChangeManeuverType._metadata, ProfileChangeManeuverType._set_segment_metadata, segment)

    _get_maneuver_type_metadata = { "offset" : _get_maneuver_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ManeuverType),) }
    @property
    def maneuver_type(self) -> "ManeuverType":
        """Get or set the new maneuver type for the targeted segment."""
        return self._intf.get_property(ProfileChangeManeuverType._metadata, ProfileChangeManeuverType._get_maneuver_type_metadata)

    _set_maneuver_type_metadata = { "offset" : _set_maneuver_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ManeuverType),) }
    @maneuver_type.setter
    def maneuver_type(self, value:"ManeuverType") -> None:
        return self._intf.set_property(ProfileChangeManeuverType._metadata, ProfileChangeManeuverType._set_maneuver_type_metadata, value)

    _property_names[segment] = "segment"
    _property_names[maneuver_type] = "maneuver_type"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileChangeManeuverType."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileChangeManeuverType)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileChangeManeuverType, [ProfileChangeManeuverType, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5514825464600670224, 11753580946685421731), ProfileChangeManeuverType)
agcls.AgTypeNameMap["ProfileChangeManeuverType"] = ProfileChangeManeuverType

class ProfileChangeReturnSegment(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Change Return Segment profile."""

    _num_methods = 5
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_name_method_offset = 1
    _set_segment_name_method_offset = 2
    _set_segment_method_offset = 3
    _get_state_method_offset = 4
    _set_state_method_offset = 5
    _metadata = {
        "iid_data" : (4705165520827975122, 8995277470071094949),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileChangeReturnSegment)

    _get_segment_name_metadata = { "offset" : _get_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def segment_name(self) -> str:
        """Get or set the targeted return segment."""
        return self._intf.get_property(ProfileChangeReturnSegment._metadata, ProfileChangeReturnSegment._get_segment_name_metadata)

    _set_segment_name_metadata = { "offset" : _set_segment_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @segment_name.setter
    def segment_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangeReturnSegment._metadata, ProfileChangeReturnSegment._set_segment_name_metadata, value)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("MCSReturn"),) }
    def set_segment(self, mcs_return:"MCSReturn") -> None:
        """Set the return segment to target."""
        return self._intf.invoke(ProfileChangeReturnSegment._metadata, ProfileChangeReturnSegment._set_segment_metadata, mcs_return)

    _get_state_metadata = { "offset" : _get_state_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ReturnControl),) }
    @property
    def state(self) -> "ReturnControl":
        """Get or set the new state for the targeted return segment."""
        return self._intf.get_property(ProfileChangeReturnSegment._metadata, ProfileChangeReturnSegment._get_state_metadata)

    _set_state_metadata = { "offset" : _set_state_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ReturnControl),) }
    @state.setter
    def state(self, value:"ReturnControl") -> None:
        return self._intf.set_property(ProfileChangeReturnSegment._metadata, ProfileChangeReturnSegment._set_state_metadata, value)

    _property_names[segment_name] = "segment_name"
    _property_names[state] = "state"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileChangeReturnSegment."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileChangeReturnSegment)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileChangeReturnSegment, [ProfileChangeReturnSegment, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4746376939470809221, 4289144284289568669), ProfileChangeReturnSegment)
agcls.AgTypeNameMap["ProfileChangeReturnSegment"] = ProfileChangeReturnSegment

class ProfileChangePropagator(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Change Propagator profile."""

    _num_methods = 5
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_name_method_offset = 1
    _set_segment_name_method_offset = 2
    _set_segment_method_offset = 3
    _get_propagator_name_method_offset = 4
    _set_propagator_name_method_offset = 5
    _metadata = {
        "iid_data" : (4634704878393296108, 17784937878159440318),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileChangePropagator)

    _get_segment_name_metadata = { "offset" : _get_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def segment_name(self) -> str:
        """Get or set the name of the profile."""
        return self._intf.get_property(ProfileChangePropagator._metadata, ProfileChangePropagator._get_segment_name_metadata)

    _set_segment_name_metadata = { "offset" : _set_segment_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @segment_name.setter
    def segment_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangePropagator._metadata, ProfileChangePropagator._set_segment_name_metadata, value)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IMCSSegment"),) }
    def set_segment(self, mcs_segment:"IMCSSegment") -> None:
        """Set the targeted segment."""
        return self._intf.invoke(ProfileChangePropagator._metadata, ProfileChangePropagator._set_segment_metadata, mcs_segment)

    _get_propagator_name_metadata = { "offset" : _get_propagator_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def propagator_name(self) -> str:
        """Get or set the new propagator's name."""
        return self._intf.get_property(ProfileChangePropagator._metadata, ProfileChangePropagator._get_propagator_name_metadata)

    _set_propagator_name_metadata = { "offset" : _set_propagator_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @propagator_name.setter
    def propagator_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangePropagator._metadata, ProfileChangePropagator._set_propagator_name_metadata, value)

    _property_names[segment_name] = "segment_name"
    _property_names[propagator_name] = "propagator_name"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileChangePropagator."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileChangePropagator)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileChangePropagator, [ProfileChangePropagator, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4917428364571480839, 14790385948966539179), ProfileChangePropagator)
agcls.AgTypeNameMap["ProfileChangePropagator"] = ProfileChangePropagator

class ProfileChangeStopSegment(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Change Stop Segment profile."""

    _num_methods = 5
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_name_method_offset = 1
    _set_segment_name_method_offset = 2
    _set_segment_method_offset = 3
    _get_state_method_offset = 4
    _set_state_method_offset = 5
    _metadata = {
        "iid_data" : (4939758405760454184, 7478107008343462300),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileChangeStopSegment)

    _get_segment_name_metadata = { "offset" : _get_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def segment_name(self) -> str:
        """Get or set the targeted stop segment."""
        return self._intf.get_property(ProfileChangeStopSegment._metadata, ProfileChangeStopSegment._get_segment_name_metadata)

    _set_segment_name_metadata = { "offset" : _set_segment_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @segment_name.setter
    def segment_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangeStopSegment._metadata, ProfileChangeStopSegment._set_segment_name_metadata, value)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("MCSStop"),) }
    def set_segment(self, mcs_stop:"MCSStop") -> None:
        """Set the stop segment to target."""
        return self._intf.invoke(ProfileChangeStopSegment._metadata, ProfileChangeStopSegment._set_segment_metadata, mcs_stop)

    _get_state_metadata = { "offset" : _get_state_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StateType),) }
    @property
    def state(self) -> "StateType":
        """Get or set the new state of the targeted stop segment."""
        return self._intf.get_property(ProfileChangeStopSegment._metadata, ProfileChangeStopSegment._get_state_metadata)

    _set_state_metadata = { "offset" : _set_state_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StateType),) }
    @state.setter
    def state(self, value:"StateType") -> None:
        return self._intf.set_property(ProfileChangeStopSegment._metadata, ProfileChangeStopSegment._set_state_metadata, value)

    _property_names[segment_name] = "segment_name"
    _property_names[state] = "state"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileChangeStopSegment."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileChangeStopSegment)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileChangeStopSegment, [ProfileChangeStopSegment, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5176652920498491223, 12779532205614345663), ProfileChangeStopSegment)
agcls.AgTypeNameMap["ProfileChangeStopSegment"] = ProfileChangeStopSegment

class ProfileChangeStoppingConditionState(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Change Stopping Condition State profile."""

    _num_methods = 8
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_name_method_offset = 1
    _set_segment_name_method_offset = 2
    _set_segment_method_offset = 3
    _get_state_method_offset = 4
    _set_state_method_offset = 5
    _set_trigger_method_offset = 6
    _get_trigger_name_method_offset = 7
    _set_trigger_name_method_offset = 8
    _metadata = {
        "iid_data" : (5430643143655333764, 9065193238726384518),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileChangeStoppingConditionState)

    _get_segment_name_metadata = { "offset" : _get_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def segment_name(self) -> str:
        """Get or set the segment that contains the targeted stopping condition."""
        return self._intf.get_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._get_segment_name_metadata)

    _set_segment_name_metadata = { "offset" : _set_segment_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @segment_name.setter
    def segment_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._set_segment_name_metadata, value)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IMCSSegment"),) }
    def set_segment(self, mcs_segment:"IMCSSegment") -> None:
        """Set the targeted segment."""
        return self._intf.invoke(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._set_segment_metadata, mcs_segment)

    _get_state_metadata = { "offset" : _get_state_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StateType),) }
    @property
    def state(self) -> "StateType":
        """Get or set the new state of the targeted stopping condition."""
        return self._intf.get_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._get_state_metadata)

    _set_state_metadata = { "offset" : _set_state_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StateType),) }
    @state.setter
    def state(self, value:"StateType") -> None:
        return self._intf.set_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._set_state_metadata, value)

    _set_trigger_metadata = { "offset" : _set_trigger_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("StoppingCondition"),) }
    def set_trigger(self, stopping_condition:"StoppingCondition") -> None:
        """Set the targeted stopping condition."""
        return self._intf.invoke(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._set_trigger_metadata, stopping_condition)

    _get_trigger_name_metadata = { "offset" : _get_trigger_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def trigger_name(self) -> str:
        """Get or set the name of the targeted stopping condition."""
        return self._intf.get_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._get_trigger_name_metadata)

    _set_trigger_name_metadata = { "offset" : _set_trigger_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @trigger_name.setter
    def trigger_name(self, value:str) -> None:
        return self._intf.set_property(ProfileChangeStoppingConditionState._metadata, ProfileChangeStoppingConditionState._set_trigger_name_metadata, value)

    _property_names[segment_name] = "segment_name"
    _property_names[state] = "state"
    _property_names[trigger_name] = "trigger_name"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileChangeStoppingConditionState."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileChangeStoppingConditionState)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileChangeStoppingConditionState, [ProfileChangeStoppingConditionState, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4678353966876850557, 14685284640460450435), ProfileChangeStoppingConditionState)
agcls.AgTypeNameMap["ProfileChangeStoppingConditionState"] = ProfileChangeStoppingConditionState

class ProfileSeedFiniteManeuver(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Seed Finite Maneuver segment."""

    _num_methods = 5
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _get_segment_name_method_offset = 1
    _set_segment_name_method_offset = 2
    _set_segment_method_offset = 3
    _get_leave_all_active_stopping_conditions_active_method_offset = 4
    _set_leave_all_active_stopping_conditions_active_method_offset = 5
    _metadata = {
        "iid_data" : (5161634194327496149, 14603357244715454085),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileSeedFiniteManeuver)

    _get_segment_name_metadata = { "offset" : _get_segment_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def segment_name(self) -> str:
        """Get or set the targeted maneuver segment."""
        return self._intf.get_property(ProfileSeedFiniteManeuver._metadata, ProfileSeedFiniteManeuver._get_segment_name_metadata)

    _set_segment_name_metadata = { "offset" : _set_segment_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @segment_name.setter
    def segment_name(self, value:str) -> None:
        return self._intf.set_property(ProfileSeedFiniteManeuver._metadata, ProfileSeedFiniteManeuver._set_segment_name_metadata, value)

    _set_segment_metadata = { "offset" : _set_segment_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("MCSManeuver"),) }
    def set_segment(self, mcs_maneuver:"MCSManeuver") -> None:
        """Set the maneuver segment to target."""
        return self._intf.invoke(ProfileSeedFiniteManeuver._metadata, ProfileSeedFiniteManeuver._set_segment_metadata, mcs_maneuver)

    _get_leave_all_active_stopping_conditions_active_metadata = { "offset" : _get_leave_all_active_stopping_conditions_active_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def leave_all_active_stopping_conditions_active(self) -> bool:
        """If true, all active stopping conditions on the seeded maneuver will remain active during run."""
        return self._intf.get_property(ProfileSeedFiniteManeuver._metadata, ProfileSeedFiniteManeuver._get_leave_all_active_stopping_conditions_active_metadata)

    _set_leave_all_active_stopping_conditions_active_metadata = { "offset" : _set_leave_all_active_stopping_conditions_active_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @leave_all_active_stopping_conditions_active.setter
    def leave_all_active_stopping_conditions_active(self, value:bool) -> None:
        return self._intf.set_property(ProfileSeedFiniteManeuver._metadata, ProfileSeedFiniteManeuver._set_leave_all_active_stopping_conditions_active_metadata, value)

    _property_names[segment_name] = "segment_name"
    _property_names[leave_all_active_stopping_conditions_active] = "leave_all_active_stopping_conditions_active"

    def __init__(self, source_object=None):
        """Construct an object of type ProfileSeedFiniteManeuver."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileSeedFiniteManeuver)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileSeedFiniteManeuver, [ProfileSeedFiniteManeuver, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((4619183164175431290, 12957669796538914955), ProfileSeedFiniteManeuver)
agcls.AgTypeNameMap["ProfileSeedFiniteManeuver"] = ProfileSeedFiniteManeuver

class ProfileRunOnce(IProfile, IRuntimeTypeInfoProvider, SupportsDeleteCallback):
    """Properties for a Run Once profile."""

    _num_methods = 0
    _vtable_offset = IProfile._vtable_offset + IProfile._num_methods
    _metadata = {
        "iid_data" : (5362986155356147363, 15814818936516716949),
        "vtable_reference" : IProfile._vtable_offset + IProfile._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, ProfileRunOnce)


    def __init__(self, source_object=None):
        """Construct an object of type ProfileRunOnce."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, ProfileRunOnce)
        IProfile.__init__(self, source_object)
        IRuntimeTypeInfoProvider.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IProfile._private_init(self, intf)
        IRuntimeTypeInfoProvider._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, ProfileRunOnce, [ProfileRunOnce, IProfile, IRuntimeTypeInfoProvider])

agcls.AgClassCatalog.add_catalog_entry((5722064001165018176, 12997517095273249469), ProfileRunOnce)
agcls.AgTypeNameMap["ProfileRunOnce"] = ProfileRunOnce

class BPlaneCollection(SupportsDeleteCallback):
    """Properties for the collection of B-Planes."""

    _num_methods = 6
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _add_method_offset = 1
    _remove_method_offset = 2
    _remove_all_method_offset = 3
    _item_method_offset = 4
    _get__new_enum_method_offset = 5
    _get_count_method_offset = 6
    _metadata = {
        "iid_data" : (5594941962752983119, 13562937374643146671),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, BPlaneCollection)
    def __iter__(self):
        """Create an iterator for the BPlaneCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> str:
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _add_metadata = { "offset" : _add_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def add(self, plane_name:str) -> None:
        """Add a BPlane."""
        return self._intf.invoke(BPlaneCollection._metadata, BPlaneCollection._add_metadata, plane_name)

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, plane_name:str) -> None:
        """Remove a BPlane."""
        return self._intf.invoke(BPlaneCollection._metadata, BPlaneCollection._remove_metadata, plane_name)

    _remove_all_metadata = { "offset" : _remove_all_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def remove_all(self) -> None:
        """Remove all BPlanes."""
        return self._intf.invoke(BPlaneCollection._metadata, BPlaneCollection._remove_all_metadata, )

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.LongArg, agmarshall.BStrArg,) }
    def item(self, index:int) -> str:
        """Allow you to iterate through the collection."""
        return self._intf.invoke(BPlaneCollection._metadata, BPlaneCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Allow you to enumerate through the collection."""
        return self._intf.get_property(BPlaneCollection._metadata, BPlaneCollection._get__new_enum_metadata)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Get the number of active BPlanes."""
        return self._intf.get_property(BPlaneCollection._metadata, BPlaneCollection._get_count_metadata)

    __getitem__ = item


    _property_names[_new_enum] = "_new_enum"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type BPlaneCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, BPlaneCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, BPlaneCollection, [BPlaneCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5110455434975895853, 10166370749291811978), BPlaneCollection)
agcls.AgTypeNameMap["BPlaneCollection"] = BPlaneCollection

class StateCalcDamageFlux(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcDamageFlux."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5064599181893483132, 838052118368786821),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcDamageFlux)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcDamageFlux."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcDamageFlux)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcDamageFlux, [StateCalcDamageFlux, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5061318997549054970, 5706412726880589720), StateCalcDamageFlux)
agcls.AgTypeNameMap["StateCalcDamageFlux"] = StateCalcDamageFlux

class StateCalcDamageMassFlux(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcDamageMassFlux."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5323133528960076198, 12570579641122910654),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcDamageMassFlux)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcDamageMassFlux."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcDamageMassFlux)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcDamageMassFlux, [StateCalcDamageMassFlux, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4973428904588242653, 6958212153847005885), StateCalcDamageMassFlux)
agcls.AgTypeNameMap["StateCalcDamageMassFlux"] = StateCalcDamageMassFlux

class StateCalcMagneticFieldDipoleL(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcMagFieldDipoleL."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5572993639986105434, 2150862437823850410),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcMagneticFieldDipoleL)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcMagneticFieldDipoleL."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcMagneticFieldDipoleL)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcMagneticFieldDipoleL, [StateCalcMagneticFieldDipoleL, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4679876560310607257, 8310578291428709278), StateCalcMagneticFieldDipoleL)
agcls.AgTypeNameMap["StateCalcMagneticFieldDipoleL"] = StateCalcMagneticFieldDipoleL

class StateCalcSEETMagneticFieldLineSeparationAngle(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a SEETMagFieldFieldLineSepAngle calculation object."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_target_object_method_offset = 1
    _metadata = {
        "iid_data" : (5335771638573946346, 7564082294565021117),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcSEETMagneticFieldLineSeparationAngle)

    _get_target_object_metadata = { "offset" : _get_target_object_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_object(self) -> "ILinkToObject":
        """Get the reference object."""
        return self._intf.get_property(StateCalcSEETMagneticFieldLineSeparationAngle._metadata, StateCalcSEETMagneticFieldLineSeparationAngle._get_target_object_metadata)

    _property_names[target_object] = "target_object"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcSEETMagneticFieldLineSeparationAngle."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcSEETMagneticFieldLineSeparationAngle)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcSEETMagneticFieldLineSeparationAngle, [StateCalcSEETMagneticFieldLineSeparationAngle, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5587815812935971298, 7525019620982063505), StateCalcSEETMagneticFieldLineSeparationAngle)
agcls.AgTypeNameMap["StateCalcSEETMagneticFieldLineSeparationAngle"] = StateCalcSEETMagneticFieldLineSeparationAngle

class StateCalcImpactFlux(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcImpactFlux."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4712475622720863446, 9874820785248486066),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcImpactFlux)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcImpactFlux."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcImpactFlux)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcImpactFlux, [StateCalcImpactFlux, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5205176986868350023, 2621073616466898844), StateCalcImpactFlux)
agcls.AgTypeNameMap["StateCalcImpactFlux"] = StateCalcImpactFlux

class StateCalcImpactMassFlux(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcImpactMassFlux."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5663867202832921241, 12848046706352279460),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcImpactMassFlux)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcImpactMassFlux."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcImpactMassFlux)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcImpactMassFlux, [StateCalcImpactMassFlux, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5270447181510537100, 9273433778213163966), StateCalcImpactMassFlux)
agcls.AgTypeNameMap["StateCalcImpactMassFlux"] = StateCalcImpactMassFlux

class StateCalcSEETSAAFlux(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcSEETSAAFlux."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5229848331502696195, 11640390802336049814),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcSEETSAAFlux)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcSEETSAAFlux."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcSEETSAAFlux)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcSEETSAAFlux, [StateCalcSEETSAAFlux, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5390130523806265182, 8748824487015767454), StateCalcSEETSAAFlux)
agcls.AgTypeNameMap["StateCalcSEETSAAFlux"] = StateCalcSEETSAAFlux

class StateCalcSEETVehTemp(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AgAsStateCalcSEETVehTemp."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5710340041284306171, 10643006267837267871),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcSEETVehTemp)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcSEETVehTemp."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcSEETVehTemp)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcSEETVehTemp, [StateCalcSEETVehTemp, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5450830750262536790, 298824803349435582), StateCalcSEETVehTemp)
agcls.AgTypeNameMap["StateCalcSEETVehTemp"] = StateCalcSEETVehTemp

class StateCalcEpoch(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an Epoch calculation object."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4735976942903942727, 11382373395637864086),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcEpoch)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcEpoch."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcEpoch)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcEpoch, [StateCalcEpoch, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5546769491034375944, 206659031291144346), StateCalcEpoch)
agcls.AgTypeNameMap["StateCalcEpoch"] = StateCalcEpoch

class StateCalcJacobiConstant(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Jacobi Constant calculation object."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5313931721597877856, 6320025579692878468),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcJacobiConstant)


    def __init__(self, source_object=None):
        """Construct an object of type StateCalcJacobiConstant."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcJacobiConstant)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcJacobiConstant, [StateCalcJacobiConstant, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4926408186651605609, 4266894203843328418), StateCalcJacobiConstant)
agcls.AgTypeNameMap["StateCalcJacobiConstant"] = StateCalcJacobiConstant

class StateCalcJacobiOsculating(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an osculating Jacobi integral calculation object."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_secondary_name_method_offset = 3
    _set_secondary_name_method_offset = 4
    _metadata = {
        "iid_data" : (4962445305111709194, 10572901812271851444),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcJacobiOsculating)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Get or set the primary central body of the three-body system."""
        return self._intf.get_property(StateCalcJacobiOsculating._metadata, StateCalcJacobiOsculating._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcJacobiOsculating._metadata, StateCalcJacobiOsculating._set_central_body_name_metadata, value)

    _get_secondary_name_metadata = { "offset" : _get_secondary_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def secondary_name(self) -> str:
        """Get or set the secondary central body of the three-body system."""
        return self._intf.get_property(StateCalcJacobiOsculating._metadata, StateCalcJacobiOsculating._get_secondary_name_metadata)

    _set_secondary_name_metadata = { "offset" : _set_secondary_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @secondary_name.setter
    def secondary_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcJacobiOsculating._metadata, StateCalcJacobiOsculating._set_secondary_name_metadata, value)

    _property_names[central_body_name] = "central_body_name"
    _property_names[secondary_name] = "secondary_name"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcJacobiOsculating."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcJacobiOsculating)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcJacobiOsculating, [StateCalcJacobiOsculating, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4893148544095721651, 9245737750026533311), StateCalcJacobiOsculating)
agcls.AgTypeNameMap["StateCalcJacobiOsculating"] = StateCalcJacobiOsculating

class StateCalcCartesianElem(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Cartesian Element calculation object."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_coord_system_name_method_offset = 1
    _set_coord_system_name_method_offset = 2
    _metadata = {
        "iid_data" : (5592104075436038276, 13656611350088774319),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcCartesianElem)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get or set the coordinate system within which the element is defined."""
        return self._intf.get_property(StateCalcCartesianElem._metadata, StateCalcCartesianElem._get_coord_system_name_metadata)

    _set_coord_system_name_metadata = { "offset" : _set_coord_system_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system_name.setter
    def coord_system_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcCartesianElem._metadata, StateCalcCartesianElem._set_coord_system_name_metadata, value)

    _property_names[coord_system_name] = "coord_system_name"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcCartesianElem."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcCartesianElem)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcCartesianElem, [StateCalcCartesianElem, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5671576764808850599, 13911282508128967061), StateCalcCartesianElem)
agcls.AgTypeNameMap["StateCalcCartesianElem"] = StateCalcCartesianElem

class StateCalcCartSTMElem(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for a Cartesian STM Element calculation object."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_coord_system_name_method_offset = 1
    _set_coord_system_name_method_offset = 2
    _get_final_state_component_method_offset = 3
    _set_final_state_component_method_offset = 4
    _get_initial_state_component_method_offset = 5
    _set_initial_state_component_method_offset = 6
    _metadata = {
        "iid_data" : (4663303186423105219, 10757831706775060869),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcCartSTMElem)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get or set the coordinate system within which the element is defined."""
        return self._intf.get_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._get_coord_system_name_metadata)

    _set_coord_system_name_metadata = { "offset" : _set_coord_system_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system_name.setter
    def coord_system_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._set_coord_system_name_metadata, value)

    _get_final_state_component_metadata = { "offset" : _get_final_state_component_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @property
    def final_state_component(self) -> "STMPerturbationVariables":
        """Get or set the final variation variable."""
        return self._intf.get_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._get_final_state_component_metadata)

    _set_final_state_component_metadata = { "offset" : _set_final_state_component_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @final_state_component.setter
    def final_state_component(self, value:"STMPerturbationVariables") -> None:
        return self._intf.set_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._set_final_state_component_metadata, value)

    _get_initial_state_component_metadata = { "offset" : _get_initial_state_component_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @property
    def initial_state_component(self) -> "STMPerturbationVariables":
        """Get or set the initial variation variable."""
        return self._intf.get_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._get_initial_state_component_metadata)

    _set_initial_state_component_metadata = { "offset" : _set_initial_state_component_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @initial_state_component.setter
    def initial_state_component(self, value:"STMPerturbationVariables") -> None:
        return self._intf.set_property(StateCalcCartSTMElem._metadata, StateCalcCartSTMElem._set_initial_state_component_metadata, value)

    _property_names[coord_system_name] = "coord_system_name"
    _property_names[final_state_component] = "final_state_component"
    _property_names[initial_state_component] = "initial_state_component"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcCartSTMElem."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcCartSTMElem)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcCartSTMElem, [StateCalcCartSTMElem, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4764427105086404315, 18352981306806488972), StateCalcCartSTMElem)
agcls.AgTypeNameMap["StateCalcCartSTMElem"] = StateCalcCartSTMElem

class StateCalcSTMEigenval(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an STM Eigenvalue calculation object."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_coord_system_name_method_offset = 1
    _set_coord_system_name_method_offset = 2
    _get_eigenvalue_number_method_offset = 3
    _set_eigenvalue_number_method_offset = 4
    _get_eigenvalue_complex_part_method_offset = 5
    _set_eigenvalue_complex_part_method_offset = 6
    _metadata = {
        "iid_data" : (4683699417864441378, 8882850736444958850),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcSTMEigenval)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get or set the coordinate system within which the element is defined."""
        return self._intf.get_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._get_coord_system_name_metadata)

    _set_coord_system_name_metadata = { "offset" : _set_coord_system_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system_name.setter
    def coord_system_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._set_coord_system_name_metadata, value)

    _get_eigenvalue_number_metadata = { "offset" : _get_eigenvalue_number_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STMEigenNumber),) }
    @property
    def eigenvalue_number(self) -> "STMEigenNumber":
        """Get or set the number identifying one of the six Eigenvalues."""
        return self._intf.get_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._get_eigenvalue_number_metadata)

    _set_eigenvalue_number_metadata = { "offset" : _set_eigenvalue_number_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STMEigenNumber),) }
    @eigenvalue_number.setter
    def eigenvalue_number(self, value:"STMEigenNumber") -> None:
        return self._intf.set_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._set_eigenvalue_number_metadata, value)

    _get_eigenvalue_complex_part_metadata = { "offset" : _get_eigenvalue_complex_part_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ComplexNumber),) }
    @property
    def eigenvalue_complex_part(self) -> "ComplexNumber":
        """Whether this value represents the real or imaginary part of the Eigenvalue."""
        return self._intf.get_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._get_eigenvalue_complex_part_metadata)

    _set_eigenvalue_complex_part_metadata = { "offset" : _set_eigenvalue_complex_part_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ComplexNumber),) }
    @eigenvalue_complex_part.setter
    def eigenvalue_complex_part(self, value:"ComplexNumber") -> None:
        return self._intf.set_property(StateCalcSTMEigenval._metadata, StateCalcSTMEigenval._set_eigenvalue_complex_part_metadata, value)

    _property_names[coord_system_name] = "coord_system_name"
    _property_names[eigenvalue_number] = "eigenvalue_number"
    _property_names[eigenvalue_complex_part] = "eigenvalue_complex_part"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcSTMEigenval."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcSTMEigenval)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcSTMEigenval, [StateCalcSTMEigenval, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4668678151179349103, 6834144943639096487), StateCalcSTMEigenval)
agcls.AgTypeNameMap["StateCalcSTMEigenval"] = StateCalcSTMEigenval

class StateCalcSTMEigenvecElem(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an STM Eigenvector element calculation object."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_coord_system_name_method_offset = 1
    _set_coord_system_name_method_offset = 2
    _get_eigenvector_number_method_offset = 3
    _set_eigenvector_number_method_offset = 4
    _get_state_variable_method_offset = 5
    _set_state_variable_method_offset = 6
    _get_eigenvector_complex_part_method_offset = 7
    _set_eigenvector_complex_part_method_offset = 8
    _metadata = {
        "iid_data" : (5656549373487126346, 16982121463062240904),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcSTMEigenvecElem)

    _get_coord_system_name_metadata = { "offset" : _get_coord_system_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def coord_system_name(self) -> str:
        """Get or set the coordinate system within which the element is defined."""
        return self._intf.get_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._get_coord_system_name_metadata)

    _set_coord_system_name_metadata = { "offset" : _set_coord_system_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @coord_system_name.setter
    def coord_system_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._set_coord_system_name_metadata, value)

    _get_eigenvector_number_metadata = { "offset" : _get_eigenvector_number_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STMEigenNumber),) }
    @property
    def eigenvector_number(self) -> "STMEigenNumber":
        """Get or set the number identifying one of the six Eigenvectors."""
        return self._intf.get_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._get_eigenvector_number_metadata)

    _set_eigenvector_number_metadata = { "offset" : _set_eigenvector_number_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STMEigenNumber),) }
    @eigenvector_number.setter
    def eigenvector_number(self, value:"STMEigenNumber") -> None:
        return self._intf.set_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._set_eigenvector_number_metadata, value)

    _get_state_variable_metadata = { "offset" : _get_state_variable_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @property
    def state_variable(self) -> "STMPerturbationVariables":
        """Get or set the variable identifying the component within an Eigenvector."""
        return self._intf.get_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._get_state_variable_metadata)

    _set_state_variable_metadata = { "offset" : _set_state_variable_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(STMPerturbationVariables),) }
    @state_variable.setter
    def state_variable(self, value:"STMPerturbationVariables") -> None:
        return self._intf.set_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._set_state_variable_metadata, value)

    _get_eigenvector_complex_part_metadata = { "offset" : _get_eigenvector_complex_part_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ComplexNumber),) }
    @property
    def eigenvector_complex_part(self) -> "ComplexNumber":
        """Whether this value represents the real or imaginary part of an Eigenvector element."""
        return self._intf.get_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._get_eigenvector_complex_part_metadata)

    _set_eigenvector_complex_part_metadata = { "offset" : _set_eigenvector_complex_part_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ComplexNumber),) }
    @eigenvector_complex_part.setter
    def eigenvector_complex_part(self, value:"ComplexNumber") -> None:
        return self._intf.set_property(StateCalcSTMEigenvecElem._metadata, StateCalcSTMEigenvecElem._set_eigenvector_complex_part_metadata, value)

    _property_names[coord_system_name] = "coord_system_name"
    _property_names[eigenvector_number] = "eigenvector_number"
    _property_names[state_variable] = "state_variable"
    _property_names[eigenvector_complex_part] = "eigenvector_complex_part"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcSTMEigenvecElem."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcSTMEigenvecElem)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcSTMEigenvecElem, [StateCalcSTMEigenvecElem, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5648792996863224538, 14390084934938091964), StateCalcSTMEigenvecElem)
agcls.AgTypeNameMap["StateCalcSTMEigenvecElem"] = StateCalcSTMEigenvecElem

class StateCalcEnvironment(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Properties for an Environment calculation object."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_atmosphere_model_name_method_offset = 3
    _set_atmosphere_model_name_method_offset = 4
    _metadata = {
        "iid_data" : (4823976028697900760, 927912681356112823),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcEnvironment)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Get or set the central body of the component."""
        return self._intf.get_property(StateCalcEnvironment._metadata, StateCalcEnvironment._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcEnvironment._metadata, StateCalcEnvironment._set_central_body_name_metadata, value)

    _get_atmosphere_model_name_metadata = { "offset" : _get_atmosphere_model_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def atmosphere_model_name(self) -> str:
        """Get or set the atmospheric density model of the component."""
        return self._intf.get_property(StateCalcEnvironment._metadata, StateCalcEnvironment._get_atmosphere_model_name_metadata)

    _set_atmosphere_model_name_metadata = { "offset" : _set_atmosphere_model_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @atmosphere_model_name.setter
    def atmosphere_model_name(self, value:str) -> None:
        return self._intf.set_property(StateCalcEnvironment._metadata, StateCalcEnvironment._set_atmosphere_model_name_metadata, value)

    _property_names[central_body_name] = "central_body_name"
    _property_names[atmosphere_model_name] = "atmosphere_model_name"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcEnvironment."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcEnvironment)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcEnvironment, [StateCalcEnvironment, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((5228943156068851180, 17757122866649733292), StateCalcEnvironment)
agcls.AgTypeNameMap["StateCalcEnvironment"] = StateCalcEnvironment

class StateCalcOrbitDelaunayG(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AsStateCalcOrbitDelaunayG."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_element_type_method_offset = 3
    _set_element_type_method_offset = 4
    _metadata = {
        "iid_data" : (4629954694119662350, 8751096521941454219),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcOrbitDelaunayG)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Central Body used for calculation."""
        return self._intf.get_property(StateCalcOrbitDelaunayG._metadata, StateCalcOrbitDelaunayG._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        """Central Body used for calculation."""
        return self._intf.set_property(StateCalcOrbitDelaunayG._metadata, StateCalcOrbitDelaunayG._set_central_body_name_metadata, value)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @property
    def element_type(self) -> "ElementType":
        """Choice of osculating or mean elements."""
        return self._intf.get_property(StateCalcOrbitDelaunayG._metadata, StateCalcOrbitDelaunayG._get_element_type_metadata)

    _set_element_type_metadata = { "offset" : _set_element_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @element_type.setter
    def element_type(self, value:"ElementType") -> None:
        """Choice of osculating or mean elements."""
        return self._intf.set_property(StateCalcOrbitDelaunayG._metadata, StateCalcOrbitDelaunayG._set_element_type_metadata, value)

    _property_names[central_body_name] = "central_body_name"
    _property_names[element_type] = "element_type"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcOrbitDelaunayG."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcOrbitDelaunayG)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcOrbitDelaunayG, [StateCalcOrbitDelaunayG, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4685955006802271629, 16763781113943652765), StateCalcOrbitDelaunayG)
agcls.AgTypeNameMap["StateCalcOrbitDelaunayG"] = StateCalcOrbitDelaunayG

class StateCalcOrbitDelaunayH(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AsStateCalcOrbitDelaunayH."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_element_type_method_offset = 3
    _set_element_type_method_offset = 4
    _metadata = {
        "iid_data" : (4828188022700843748, 13086359271346066049),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcOrbitDelaunayH)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Central Body used for calculation."""
        return self._intf.get_property(StateCalcOrbitDelaunayH._metadata, StateCalcOrbitDelaunayH._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        """Central Body used for calculation."""
        return self._intf.set_property(StateCalcOrbitDelaunayH._metadata, StateCalcOrbitDelaunayH._set_central_body_name_metadata, value)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @property
    def element_type(self) -> "ElementType":
        """Choice of osculating or mean elements."""
        return self._intf.get_property(StateCalcOrbitDelaunayH._metadata, StateCalcOrbitDelaunayH._get_element_type_metadata)

    _set_element_type_metadata = { "offset" : _set_element_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @element_type.setter
    def element_type(self, value:"ElementType") -> None:
        """Choice of osculating or mean elements."""
        return self._intf.set_property(StateCalcOrbitDelaunayH._metadata, StateCalcOrbitDelaunayH._set_element_type_metadata, value)

    _property_names[central_body_name] = "central_body_name"
    _property_names[element_type] = "element_type"

    def __init__(self, source_object=None):
        """Construct an object of type StateCalcOrbitDelaunayH."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, StateCalcOrbitDelaunayH)
        IComponentInfo.__init__(self, source_object)
        ICloneable.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IComponentInfo._private_init(self, intf)
        ICloneable._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, StateCalcOrbitDelaunayH, [StateCalcOrbitDelaunayH, IComponentInfo, ICloneable])

agcls.AgClassCatalog.add_catalog_entry((4695981083676433439, 17340207238014231457), StateCalcOrbitDelaunayH)
agcls.AgTypeNameMap["StateCalcOrbitDelaunayH"] = StateCalcOrbitDelaunayH

class StateCalcOrbitDelaunayL(IComponentInfo, ICloneable, SupportsDeleteCallback):
    """Interface for AsStateCalcOrbitDelaunayL."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_name_method_offset = 1
    _set_central_body_name_method_offset = 2
    _get_element_type_method_offset = 3
    _set_element_type_method_offset = 4
    _metadata = {
        "iid_data" : (5043781925077325770, 5014519046419148467),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, StateCalcOrbitDelaunayL)

    _get_central_body_name_metadata = { "offset" : _get_central_body_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body_name(self) -> str:
        """Central Body used for calculation."""
        return self._intf.get_property(StateCalcOrbitDelaunayL._metadata, StateCalcOrbitDelaunayL._get_central_body_name_metadata)

    _set_central_body_name_metadata = { "offset" : _set_central_body_name_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body_name.setter
    def central_body_name(self, value:str) -> None:
        """Central Body used for calculation."""
        return self._intf.set_property(StateCalcOrbitDelaunayL._metadata, StateCalcOrbitDelaunayL._set_central_body_name_metadata, value)

    _get_element_type_metadata = { "offset" : _get_element_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ElementType),) }
    @