# Copyright (C) 2022 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""
The Vector Geometry (VGT) API enables users define new or utilize existing geometric constructs such as coordinate systems, vectors, points, angles, axes and planes.

The geometric elements can be used to transform between coordinate
systems, compute first and second derivatives, or perform other types of
analysis.
"""

__all__ = ["AberrationModelType", "AnalysisWorkbenchAngleFindAngleResult", "AnalysisWorkbenchAngleFindAngleWithRateResult",
"AnalysisWorkbenchAngleFindResult", "AnalysisWorkbenchAngleFindWithRateResult", "AnalysisWorkbenchAxesFindInAxesResult",
"AnalysisWorkbenchAxesFindInAxesWithRateResult", "AnalysisWorkbenchAxesTransformResult",
"AnalysisWorkbenchAxesTransformWithRateResult", "AnalysisWorkbenchCentralBody",
"AnalysisWorkbenchCentralBodyCollection", "AnalysisWorkbenchCentralBodyReference", "AnalysisWorkbenchComponent",
"AnalysisWorkbenchComponentCollection", "AnalysisWorkbenchComponentInstance", "AnalysisWorkbenchComponentProvider",
"AnalysisWorkbenchComponentTemplate", "AnalysisWorkbenchComponentTypeInformation", "AnalysisWorkbenchConvergence",
"AnalysisWorkbenchDerivative", "AnalysisWorkbenchIntegral", "AnalysisWorkbenchInterpolator",
"AnalysisWorkbenchPlaneFindInAxesResult", "AnalysisWorkbenchPlaneFindInAxesWithRateResult",
"AnalysisWorkbenchPlaneFindInSystemResult", "AnalysisWorkbenchPlaneFindInSystemWithRateResult",
"AnalysisWorkbenchPointLocateInSystemResult", "AnalysisWorkbenchPointLocateInSystemWithRateResult",
"AnalysisWorkbenchPositionLLA", "AnalysisWorkbenchRoot", "AnalysisWorkbenchSampling", "AnalysisWorkbenchSignalDelay",
"AnalysisWorkbenchSystemFindInSystemResult", "AnalysisWorkbenchSystemTransformResult",
"AnalysisWorkbenchSystemTransformWithRateResult", "AnalysisWorkbenchVectorFindInAxesResult",
"AnalysisWorkbenchVectorFindInAxesWithRateResult", "AngleToLocationType", "AngleType", "AsymptoteDirectionType",
"AxesType", "CalculationScalarType", "CalculationToolCondition", "CalculationToolConditionCombined",
"CalculationToolConditionFactory", "CalculationToolConditionGroup", "CalculationToolConditionScalarBounds",
"CalculationToolConditionSet", "CalculationToolConditionSetEvaluateResult",
"CalculationToolConditionSetEvaluateWithRateResult", "CalculationToolConditionSetFactory",
"CalculationToolConditionSetGroup", "CalculationToolConditionSetScalarThresholds",
"CalculationToolConditionTrajectoryWithinVolume", "CalculationToolConvergeBasic", "CalculationToolDerivativeBasic",
"CalculationToolEvaluateResult", "CalculationToolEvaluateWithRateResult", "CalculationToolIntegralBasic",
"CalculationToolInterpolatorBasic", "CalculationToolParameterSet", "CalculationToolParameterSetAttitude",
"CalculationToolParameterSetFactory", "CalculationToolParameterSetGroundTrajectory", "CalculationToolParameterSetGroup",
"CalculationToolParameterSetOrbit", "CalculationToolParameterSetTrajectory", "CalculationToolParameterSetVector",
"CalculationToolSamplingBasic", "CalculationToolSamplingCurvatureTolerance", "CalculationToolSamplingFixedStep",
"CalculationToolSamplingMethod", "CalculationToolSamplingMethodFactory", "CalculationToolSamplingRelativeTolerance",
"CalculationToolScalar", "CalculationToolScalarAlongTrajectory", "CalculationToolScalarAngle",
"CalculationToolScalarAverage", "CalculationToolScalarConstant", "CalculationToolScalarCustom",
"CalculationToolScalarCustomInlineScript", "CalculationToolScalarDataElement", "CalculationToolScalarDerivative",
"CalculationToolScalarDotProduct", "CalculationToolScalarElapsedTime", "CalculationToolScalarFactory",
"CalculationToolScalarFile", "CalculationToolScalarFixedAtTimeInstant", "CalculationToolScalarFunction",
"CalculationToolScalarFunctionOf2Variables", "CalculationToolScalarGroup", "CalculationToolScalarIntegral",
"CalculationToolScalarPlugin", "CalculationToolScalarStandardDeviation",
"CalculationToolScalarSurfaceDistanceBetweenPoints", "CalculationToolScalarVectorComponent",
"CalculationToolScalarVectorMagnitude", "ClockHostType", "ConditionCombinedOperationType", "ConditionSetType",
"ConditionThresholdType", "ConditionType", "DistanceToLocationType", "EventArrayFilterType", "EventArrayType",
"EventIntervalCollectionType", "EventIntervalListType", "EventIntervalType", "EventListMergeOperation", "ExtremumType",
"FileInterpolatorType", "GridValuesMethodType", "IAnalysisWorkbenchComponent", "IAnalysisWorkbenchComponentContext",
"IAnalysisWorkbenchComponentReference", "IAnalysisWorkbenchComponentTimeProperties", "IAnalysisWorkbenchConvergence",
"IAnalysisWorkbenchDerivative", "IAnalysisWorkbenchIntegral", "IAnalysisWorkbenchInterpolator",
"IAnalysisWorkbenchMethodCallResult", "IAnalysisWorkbenchSampling", "IAnalysisWorkbenchSignalDelay",
"ICalculationToolCondition", "ICalculationToolConditionSet", "ICalculationToolParameterSet",
"ICalculationToolSamplingMethod", "ICalculationToolScalar", "ISpatialAnalysisToolGridValuesMethod",
"ISpatialAnalysisToolSpatialCalculation", "ISpatialAnalysisToolVolume", "ISpatialAnalysisToolVolumeGrid",
"ITimeToolInstant", "ITimeToolPruneFilter", "ITimeToolTimeArray", "ITimeToolTimeInterval",
"ITimeToolTimeIntervalCollection", "ITimeToolTimeIntervalList", "IVectorGeometryToolAngle", "IVectorGeometryToolAxes",
"IVectorGeometryToolPlane", "IVectorGeometryToolPoint", "IVectorGeometryToolSystem", "IVectorGeometryToolVector",
"InheritDimensionType", "IntegrationWindowType", "InterpolationMethodType", "IntersectionSurfaceType",
"IntervalDurationType", "IntervalFromIntervalListSelectionType", "IntervalPruneFilterType",
"LagrangeLibrationPointType", "LightingConditionsType", "MeanElementTheory", "ParameterSetType", "PlaneQuadrantType",
"PlaneType", "PointBPlaneType", "PointType", "PrincipalAxisOfRotationType", "QuadratureType", "RangeSpeedType",
"ResultVectorRequestType", "RotationSweepModeType", "SampleReferenceTimeType", "SatisfactionCrossing", "SaveDataType",
"SignalDirectionType", "SignalPathReferenceSystem", "SignedAngleType", "SmartEpochState", "SmartIntervalState",
"SpatialAnalysisToolAnalysisToolVolumeGridBearingAltitude", "SpatialAnalysisToolCalculationAltitude",
"SpatialAnalysisToolCalculationAngleToLocation", "SpatialAnalysisToolCalculationConditionSatisfactionMetric",
"SpatialAnalysisToolCalculationDistanceToLocation", "SpatialAnalysisToolCalculationFactory",
"SpatialAnalysisToolCalculationFile", "SpatialAnalysisToolCalculationFromCalculationScalar",
"SpatialAnalysisToolCalculationGroup", "SpatialAnalysisToolCalculationPropagationDelayToLocation",
"SpatialAnalysisToolCalculationSolarIntensity", "SpatialAnalysisToolConditionAccessToLocation",
"SpatialAnalysisToolConditionCombined", "SpatialAnalysisToolConditionConditionAtLocation",
"SpatialAnalysisToolConditionFactory", "SpatialAnalysisToolConditionGridBoundingVolume",
"SpatialAnalysisToolConditionGroup", "SpatialAnalysisToolConditionLighting", "SpatialAnalysisToolConditionOverTime",
"SpatialAnalysisToolConditionSpatialCalculationBounds", "SpatialAnalysisToolConditionValidTimeAtLocation",
"SpatialAnalysisToolGridCoordinateDefinition", "SpatialAnalysisToolGridValuesCustom",
"SpatialAnalysisToolGridValuesFixedNumberOfSteps", "SpatialAnalysisToolGridValuesFixedStep",
"SpatialAnalysisToolGridValuesMethod", "SpatialAnalysisToolSpatialCalculation", "SpatialAnalysisToolVolume",
"SpatialAnalysisToolVolumeGrid", "SpatialAnalysisToolVolumeGridCartesian", "SpatialAnalysisToolVolumeGridConstrained",
"SpatialAnalysisToolVolumeGridCylindrical", "SpatialAnalysisToolVolumeGridFactory",
"SpatialAnalysisToolVolumeGridGroup", "SpatialAnalysisToolVolumeGridLatitudeLongitudeAltitude",
"SpatialAnalysisToolVolumeGridResult", "SpatialAnalysisToolVolumeGridSpherical",
"SpatialCalculationAltitudeReferenceType", "SpatialCalculationType", "SpatialConditionOverTypeDurationType",
"SpeedType", "StartStopType", "SurfaceReferenceShapeType", "SurfaceShapeType", "SystemType",
"ThresholdConvergenceSenseType", "TimeEventType", "TimeSenseType", "TimeToolAxesSamplingInterval",
"TimeToolAxesSamplingIntervalCollection", "TimeToolAxesSamplingResult", "TimeToolInstant", "TimeToolInstantEpoch",
"TimeToolInstantExtremum", "TimeToolInstantFactory", "TimeToolInstantGroup", "TimeToolInstantOccurrenceResult",
"TimeToolInstantSignaled", "TimeToolInstantSmartEpoch", "TimeToolInstantStartStopTime", "TimeToolInstantTimeOffset",
"TimeToolInterval", "TimeToolIntervalCollection", "TimeToolIntervalListResult", "TimeToolIntervalVectorCollection",
"TimeToolIntervalsFilter", "TimeToolIntervalsVectorResult", "TimeToolLightTimeDelay", "TimeToolPointSamplingInterval",
"TimeToolPointSamplingIntervalCollection", "TimeToolPointSamplingResult", "TimeToolPruneFilter",
"TimeToolPruneFilterFactory", "TimeToolSignalDelayBasic", "TimeToolTimeArray", "TimeToolTimeArrayConditionCrossings",
"TimeToolTimeArrayExtrema", "TimeToolTimeArrayFactory", "TimeToolTimeArrayFiltered", "TimeToolTimeArrayFindTimesResult",
"TimeToolTimeArrayFixedStep", "TimeToolTimeArrayFixedTimes", "TimeToolTimeArrayGroup", "TimeToolTimeArrayMerged",
"TimeToolTimeArraySignaled", "TimeToolTimeArrayStartStopTimes", "TimeToolTimeInterval",
"TimeToolTimeIntervalBetweenTimeInstants", "TimeToolTimeIntervalCollection", "TimeToolTimeIntervalCollectionCondition",
"TimeToolTimeIntervalCollectionFactory", "TimeToolTimeIntervalCollectionGroup",
"TimeToolTimeIntervalCollectionLighting", "TimeToolTimeIntervalCollectionOccurredResult",
"TimeToolTimeIntervalCollectionSignaled", "TimeToolTimeIntervalFactory", "TimeToolTimeIntervalFirstIntervalsFilter",
"TimeToolTimeIntervalFixed", "TimeToolTimeIntervalFixedDuration", "TimeToolTimeIntervalFromIntervalList",
"TimeToolTimeIntervalGapsFilter", "TimeToolTimeIntervalGroup", "TimeToolTimeIntervalLastIntervalsFilter",
"TimeToolTimeIntervalList", "TimeToolTimeIntervalListCondition", "TimeToolTimeIntervalListFactory",
"TimeToolTimeIntervalListFile", "TimeToolTimeIntervalListFiltered", "TimeToolTimeIntervalListFixed",
"TimeToolTimeIntervalListGroup", "TimeToolTimeIntervalListMerged", "TimeToolTimeIntervalListScaled",
"TimeToolTimeIntervalListSignaled", "TimeToolTimeIntervalListTimeOffset",
"TimeToolTimeIntervalRelativeSatisfactionConditionFilter", "TimeToolTimeIntervalResult",
"TimeToolTimeIntervalSatisfactionConditionFilter", "TimeToolTimeIntervalScaled", "TimeToolTimeIntervalSignaled",
"TimeToolTimeIntervalSmartInterval", "TimeToolTimeIntervalTimeOffset", "TrajectoryAxesCoordinatesType",
"VectorComponentType", "VectorGeometryToolAngle", "VectorGeometryToolAngleBetweenPlanes",
"VectorGeometryToolAngleBetweenVectors", "VectorGeometryToolAngleDihedral", "VectorGeometryToolAngleFactory",
"VectorGeometryToolAngleGroup", "VectorGeometryToolAngleReference", "VectorGeometryToolAngleRotation",
"VectorGeometryToolAngleToPlane", "VectorGeometryToolAxes", "VectorGeometryToolAxesAlignedAndConstrained",
"VectorGeometryToolAxesAngularOffset", "VectorGeometryToolAxesAtTimeInstant", "VectorGeometryToolAxesAttitudeFile",
"VectorGeometryToolAxesBPlane", "VectorGeometryToolAxesCommonTasks", "VectorGeometryToolAxesCustomScript",
"VectorGeometryToolAxesFactory", "VectorGeometryToolAxesFixed", "VectorGeometryToolAxesFixedAtEpoch",
"VectorGeometryToolAxesGroup", "VectorGeometryToolAxesLabels", "VectorGeometryToolAxesLagrangeLibration",
"VectorGeometryToolAxesModelAttachment", "VectorGeometryToolAxesOnSurface", "VectorGeometryToolAxesPlugin",
"VectorGeometryToolAxesReference", "VectorGeometryToolAxesSpinning", "VectorGeometryToolAxesTrajectory",
"VectorGeometryToolComponentType", "VectorGeometryToolPlane", "VectorGeometryToolPlaneFactory",
"VectorGeometryToolPlaneGroup", "VectorGeometryToolPlaneLabels", "VectorGeometryToolPlaneNormal",
"VectorGeometryToolPlaneQuadrant", "VectorGeometryToolPlaneReference", "VectorGeometryToolPlaneTrajectory",
"VectorGeometryToolPlaneTriad", "VectorGeometryToolPlaneTwoVector", "VectorGeometryToolPoint",
"VectorGeometryToolPointAtTimeInstant", "VectorGeometryToolPointBPlane",
"VectorGeometryToolPointCentralBodyFixedOffset", "VectorGeometryToolPointCentralBodyIntersect",
"VectorGeometryToolPointCommonTasks", "VectorGeometryToolPointCovarianceGrazing", "VectorGeometryToolPointFactory",
"VectorGeometryToolPointFile", "VectorGeometryToolPointFixedInSystem", "VectorGeometryToolPointGlint",
"VectorGeometryToolPointGrazing", "VectorGeometryToolPointGroup", "VectorGeometryToolPointLagrangeLibration",
"VectorGeometryToolPointModelAttachment", "VectorGeometryToolPointOnSurface",
"VectorGeometryToolPointPlaneIntersection", "VectorGeometryToolPointPlaneProjection", "VectorGeometryToolPointPlugin",
"VectorGeometryToolPointReference", "VectorGeometryToolPointSatelliteCollectionEntry",
"VectorGeometryToolSamplingMethod", "VectorGeometryToolScaledVectorDimensionInheritanceOptionType",
"VectorGeometryToolSystem", "VectorGeometryToolSystemAssembled", "VectorGeometryToolSystemCommonTasks",
"VectorGeometryToolSystemFactory", "VectorGeometryToolSystemGroup", "VectorGeometryToolSystemOnSurface",
"VectorGeometryToolSystemReference", "VectorGeometryToolVector", "VectorGeometryToolVectorAngleRate",
"VectorGeometryToolVectorAngularVelocity", "VectorGeometryToolVectorApoapsis", "VectorGeometryToolVectorConing",
"VectorGeometryToolVectorCross", "VectorGeometryToolVectorCustomScript", "VectorGeometryToolVectorDerivative",
"VectorGeometryToolVectorDirectionToStar", "VectorGeometryToolVectorDisplacement",
"VectorGeometryToolVectorEccentricity", "VectorGeometryToolVectorFactory", "VectorGeometryToolVectorFile",
"VectorGeometryToolVectorFixedAtEpoch", "VectorGeometryToolVectorFixedAtTimeInstant",
"VectorGeometryToolVectorFixedInAxes", "VectorGeometryToolVectorGroup", "VectorGeometryToolVectorLineOfNodes",
"VectorGeometryToolVectorLinearCombination", "VectorGeometryToolVectorModelAttachment",
"VectorGeometryToolVectorOrbitAngularMomentum", "VectorGeometryToolVectorOrbitNormal",
"VectorGeometryToolVectorPeriapsis", "VectorGeometryToolVectorPlugin", "VectorGeometryToolVectorProjection",
"VectorGeometryToolVectorProjectionAlongVector", "VectorGeometryToolVectorReference",
"VectorGeometryToolVectorReflection", "VectorGeometryToolVectorRotationVector",
"VectorGeometryToolVectorScalarLinearCombination", "VectorGeometryToolVectorScalarScaled",
"VectorGeometryToolVectorScaled", "VectorGeometryToolVectorSurfaceDisplacement",
"VectorGeometryToolVectorTwoPlanesIntersection", "VectorGeometryToolVectorVelocityAcceleration",
"VectorGeometryToolWellKnownAxes", "VectorGeometryToolWellKnownEarthAxes", "VectorGeometryToolWellKnownEarthSystems",
"VectorGeometryToolWellKnownSunAxes", "VectorGeometryToolWellKnownSunSystems", "VectorGeometryToolWellKnownSystems",
"VectorType", "VolumeCombinedOperationType", "VolumeFromGridEdgeType", "VolumeGridType",
"VolumeSatisfactionAccumulationType", "VolumeSatisfactionDurationType", "VolumeSatisfactionFilterType",
"VolumeSatisfactionMetricType", "VolumeType"]

from ctypes import POINTER
from enum import IntEnum, IntFlag
import typing

from .internal import coclassutil as agcls, comutil as agcom, marshall as agmarshall
from .internal.apiutil import (
    EnumeratorProxy,
    InterfaceProxy,
    OutArg,
    SupportsDeleteCallback,
    get_interface_property,
    initialize_from_source_object,
    set_class_attribute,
    set_interface_attribute,
)
from .internal.comutil import IDispatch, IUnknown
from .stkutil import EulerOrientationSequenceType, ICartesian3Vector, IDirection, IOrientation, IPosition, Quantity
from .utilities.exceptions import STKRuntimeError


def _raise_uninitialized_error(*args):
    raise STKRuntimeError("Valid STK object model classes are returned from STK methods and should not be created independently.")

class CalculationScalarType(IntEnum):
    """Define available calculation scalar types."""

    UNKNOWN = -1
    """Unknown or unsupported calculation scalar types."""
    ANGLE = 0
    """Scalar equal to angular displacement obtained from any angle in VGT."""
    FIXED_AT_TIME_INSTANT = 1
    """Constant scalar created by evaluating input scalar calculation at specified reference time instant."""
    CONSTANT = 2
    """Constant scalar value of specified dimension."""
    DATA_ELEMENT = 3
    """Any time-dependent data element from STK data providers available for parent STK object."""
    DERIVATIVE = 4
    """Derivative of input scalar calculation."""
    ELAPSED_TIME = 5
    """Time elapsed since reference time instant."""
    FILE = 6
    """Tabulated scalar calculation data loaded from specified file."""
    FUNCTION = 7
    """Defined by performing one of specified functions on input scalar."""
    INTEGRAL = 8
    """Integral of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."""
    FUNCTION_OF_2_VARIABLES = 9
    """Defined by performing one of specified binary operations on two scalar arguments."""
    VECTOR_MAGNITUDE = 10
    """Scalar equal to magnitude of specified vector."""
    PLUGIN = 11
    """A calc scalar plugin based on a COM object."""
    CUSTOM_SCRIPT = 12
    """A calc scalar uses scripted algorithm in MATLAB (.m or .dll), Perl or VBScript to define its value and rate."""
    SURFACE_DISTANCE_BETWEEN_POINTS = 13
    """Surface distance along the specified central body ellipsoid between two points (or their respective projections if specified at altitude)."""
    DOT_PRODUCT = 14
    """Scalar equal to the dot product between two vectors."""
    VECTOR_COMPONENT = 15
    """Scalar equal to the specified component of a vector when resolved in the specified axes."""
    AVERAGE = 16
    """Average of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."""
    STANDARD_DEVIATION = 17
    """Standard deviation of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."""
    CALCULATION_ALONG_TRAJECTORY = 18
    """Calculations Along Trajectory. Use CalculationToolScalarAlongTrajectory to access it."""
    CUSTOM_INLINE_SCRIPT = 19
    """Custom inline script scalar."""

CalculationScalarType.UNKNOWN.__doc__ = "Unknown or unsupported calculation scalar types."
CalculationScalarType.ANGLE.__doc__ = "Scalar equal to angular displacement obtained from any angle in VGT."
CalculationScalarType.FIXED_AT_TIME_INSTANT.__doc__ = "Constant scalar created by evaluating input scalar calculation at specified reference time instant."
CalculationScalarType.CONSTANT.__doc__ = "Constant scalar value of specified dimension."
CalculationScalarType.DATA_ELEMENT.__doc__ = "Any time-dependent data element from STK data providers available for parent STK object."
CalculationScalarType.DERIVATIVE.__doc__ = "Derivative of input scalar calculation."
CalculationScalarType.ELAPSED_TIME.__doc__ = "Time elapsed since reference time instant."
CalculationScalarType.FILE.__doc__ = "Tabulated scalar calculation data loaded from specified file."
CalculationScalarType.FUNCTION.__doc__ = "Defined by performing one of specified functions on input scalar."
CalculationScalarType.INTEGRAL.__doc__ = "Integral of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."
CalculationScalarType.FUNCTION_OF_2_VARIABLES.__doc__ = "Defined by performing one of specified binary operations on two scalar arguments."
CalculationScalarType.VECTOR_MAGNITUDE.__doc__ = "Scalar equal to magnitude of specified vector."
CalculationScalarType.PLUGIN.__doc__ = "A calc scalar plugin based on a COM object."
CalculationScalarType.CUSTOM_SCRIPT.__doc__ = "A calc scalar uses scripted algorithm in MATLAB (.m or .dll), Perl or VBScript to define its value and rate."
CalculationScalarType.SURFACE_DISTANCE_BETWEEN_POINTS.__doc__ = "Surface distance along the specified central body ellipsoid between two points (or their respective projections if specified at altitude)."
CalculationScalarType.DOT_PRODUCT.__doc__ = "Scalar equal to the dot product between two vectors."
CalculationScalarType.VECTOR_COMPONENT.__doc__ = "Scalar equal to the specified component of a vector when resolved in the specified axes."
CalculationScalarType.AVERAGE.__doc__ = "Average of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."
CalculationScalarType.STANDARD_DEVIATION.__doc__ = "Standard deviation of input scalar computed with respect to time using one of specified numerical methods and using one of specified accumulation types."
CalculationScalarType.CALCULATION_ALONG_TRAJECTORY.__doc__ = "Calculations Along Trajectory. Use CalculationToolScalarAlongTrajectory to access it."
CalculationScalarType.CUSTOM_INLINE_SCRIPT.__doc__ = "Custom inline script scalar."

agcls.AgTypeNameMap["CalculationScalarType"] = CalculationScalarType

class ConditionCombinedOperationType(IntEnum):
    """Define scalar condition combined operation types."""

    AND = 1
    """Scalar condition combined AND operation."""
    OR = 2
    """Scalar condition combined OR operation."""
    XOR = 3
    """Scalar condition combined XOR operation."""
    MINUS = 4
    """Scalar condition combined MINUS operation."""

ConditionCombinedOperationType.AND.__doc__ = "Scalar condition combined AND operation."
ConditionCombinedOperationType.OR.__doc__ = "Scalar condition combined OR operation."
ConditionCombinedOperationType.XOR.__doc__ = "Scalar condition combined XOR operation."
ConditionCombinedOperationType.MINUS.__doc__ = "Scalar condition combined MINUS operation."

agcls.AgTypeNameMap["ConditionCombinedOperationType"] = ConditionCombinedOperationType

class ConditionSetType(IntEnum):
    """Define available condition set types."""

    UNKNOWN = -1
    """Unknown or unsupported condition set types."""
    SCALAR_THRESHOLDS = 0
    """Condition set placing multiple thresholds on specified scalar."""

ConditionSetType.UNKNOWN.__doc__ = "Unknown or unsupported condition set types."
ConditionSetType.SCALAR_THRESHOLDS.__doc__ = "Condition set placing multiple thresholds on specified scalar."

agcls.AgTypeNameMap["ConditionSetType"] = ConditionSetType

class ConditionThresholdType(IntEnum):
    """Operations for Scalar Bounds Condition."""

    ABOVE_MINIMUM = 1
    """Bound is above a minimum value."""
    BELOW_MAXIMUM = 2
    """Bound is below a maximum value."""
    BETWEEN_MINIMUM_AND_MAXIMUM = 3
    """Bound is between a minimum and maximum value."""
    NOT_BETWEEN_MINIMUM_AND_MAXIMUM = 4
    """Bound is outside a minimum and maximum value."""

ConditionThresholdType.ABOVE_MINIMUM.__doc__ = "Bound is above a minimum value."
ConditionThresholdType.BELOW_MAXIMUM.__doc__ = "Bound is below a maximum value."
ConditionThresholdType.BETWEEN_MINIMUM_AND_MAXIMUM.__doc__ = "Bound is between a minimum and maximum value."
ConditionThresholdType.NOT_BETWEEN_MINIMUM_AND_MAXIMUM.__doc__ = "Bound is outside a minimum and maximum value."

agcls.AgTypeNameMap["ConditionThresholdType"] = ConditionThresholdType

class ConditionType(IntEnum):
    """Define available condition types."""

    UNKNOWN = -1
    """Unknown or unsupported condition type."""
    SCALAR_BOUNDS = 0
    """Condition placing bounds on specified scalar."""
    COMBINED = 1
    """Multiple conditiones on specified scalar."""
    TRAJECTORY_WITHIN_VOLUME = 2
    """Condition placing point in volume."""

ConditionType.UNKNOWN.__doc__ = "Unknown or unsupported condition type."
ConditionType.SCALAR_BOUNDS.__doc__ = "Condition placing bounds on specified scalar."
ConditionType.COMBINED.__doc__ = "Multiple conditiones on specified scalar."
ConditionType.TRAJECTORY_WITHIN_VOLUME.__doc__ = "Condition placing point in volume."

agcls.AgTypeNameMap["ConditionType"] = ConditionType

class InheritDimensionType(IntEnum):
    """Define how dimension is inherited."""

    NONE = 0
    """Do not inherit dimension."""
    FROM_X = 1
    """Inherit dimension from X scalar."""
    FROM_Y = 2
    """Inherit dimension from Y scalar."""

InheritDimensionType.NONE.__doc__ = "Do not inherit dimension."
InheritDimensionType.FROM_X.__doc__ = "Inherit dimension from X scalar."
InheritDimensionType.FROM_Y.__doc__ = "Inherit dimension from Y scalar."

agcls.AgTypeNameMap["InheritDimensionType"] = InheritDimensionType

class EventArrayFilterType(IntEnum):
    """Event array filter types."""

    SKIP_TIME_STEP = 0
    """Skip time step."""
    SKIP_COUNT = 1
    """Skip count."""
    INTERVALS = 2
    """Intervals."""

EventArrayFilterType.SKIP_TIME_STEP.__doc__ = "Skip time step."
EventArrayFilterType.SKIP_COUNT.__doc__ = "Skip count."
EventArrayFilterType.INTERVALS.__doc__ = "Intervals."

agcls.AgTypeNameMap["EventArrayFilterType"] = EventArrayFilterType

class EventArrayType(IntEnum):
    """Define available time array types."""

    UNKNOWN = -1
    """Unknown or unsupported time array types."""
    EXTREMA = 0
    """Determine time of local minimum and/or maximum of specified scalar calculation."""
    START_STOP_TIMES = 1
    """Defined by taking start and/or stop times of every interval in specified reference interval list and adding them to array."""
    MERGED = 2
    """Defined by merging times from two other arrays by creating union of bounding intervals from two constituent arrays."""
    FILTERED = 3
    """Defined by filtering times from original time array according to specified filtering method."""
    FIXED_STEP = 4
    """Defined by taking fixed time steps from specified time reference and adding sampled times to array if they fall within specified bounding interval list."""
    CONDITION_CROSSINGS = 5
    """Time array containing times at which specified condition changes its satisfaction status."""
    SIGNALED = 6
    """Determine what time array is recorded at target clock location by performing signal transmission of original time array between base and target clock locations."""
    FIXED_TIMES = 7
    """Time array containing specific times."""

EventArrayType.UNKNOWN.__doc__ = "Unknown or unsupported time array types."
EventArrayType.EXTREMA.__doc__ = "Determine time of local minimum and/or maximum of specified scalar calculation."
EventArrayType.START_STOP_TIMES.__doc__ = "Defined by taking start and/or stop times of every interval in specified reference interval list and adding them to array."
EventArrayType.MERGED.__doc__ = "Defined by merging times from two other arrays by creating union of bounding intervals from two constituent arrays."
EventArrayType.FILTERED.__doc__ = "Defined by filtering times from original time array according to specified filtering method."
EventArrayType.FIXED_STEP.__doc__ = "Defined by taking fixed time steps from specified time reference and adding sampled times to array if they fall within specified bounding interval list."
EventArrayType.CONDITION_CROSSINGS.__doc__ = "Time array containing times at which specified condition changes its satisfaction status."
EventArrayType.SIGNALED.__doc__ = "Determine what time array is recorded at target clock location by performing signal transmission of original time array between base and target clock locations."
EventArrayType.FIXED_TIMES.__doc__ = "Time array containing specific times."

agcls.AgTypeNameMap["EventArrayType"] = EventArrayType

class EventIntervalCollectionType(IntEnum):
    """Define available interval collection types."""

    UNKNOWN = -1
    """Unknown or unsupported interval collection types."""
    LIGHTING = 0
    """Defined by computing sunlight, penumbra and umbra intervals as seen at specified location using specified selection of eclipsing bodies."""
    SIGNALED = 1
    """Determine what interval list collection is recorded at target clock location by performing signal transmission of original interval list collection between base and target clock locations."""
    CONDITION = 2
    """Interval collection containing intervals during which condition set is satisfied."""

EventIntervalCollectionType.UNKNOWN.__doc__ = "Unknown or unsupported interval collection types."
EventIntervalCollectionType.LIGHTING.__doc__ = "Defined by computing sunlight, penumbra and umbra intervals as seen at specified location using specified selection of eclipsing bodies."
EventIntervalCollectionType.SIGNALED.__doc__ = "Determine what interval list collection is recorded at target clock location by performing signal transmission of original interval list collection between base and target clock locations."
EventIntervalCollectionType.CONDITION.__doc__ = "Interval collection containing intervals during which condition set is satisfied."

agcls.AgTypeNameMap["EventIntervalCollectionType"] = EventIntervalCollectionType

class EventIntervalListType(IntEnum):
    """Define available interval list types."""

    UNKNOWN = -1
    """Unknown or unsupported interval list types."""
    MERGED = 1
    """Interval list created by merging two constituent interval lists using specified logical operation."""
    FILTERED = 2
    """Defined by filtering intervals from original interval list using specified filtering method."""
    CONDITION = 3
    """Interval list containing intervals during which specified condition is satisfied (UI type name is Satisfaction)."""
    SCALED = 4
    """Interval List defined by scaling every interval in original interval list using either absolute or relative scale."""
    SIGNALED = 5
    """Determine what interval list is recorded at target clock location by performing signal transmission of original interval list between base and target clock locations."""
    TIME_OFFSET = 6
    """Interval List defined by shifting specified reference interval list by fixed time offset."""
    FILE = 0
    """Interval list loaded from specified interval file."""
    FIXED = 7
    """Interval list with individual intervals defined between explicitly specified start and stop times."""

EventIntervalListType.UNKNOWN.__doc__ = "Unknown or unsupported interval list types."
EventIntervalListType.MERGED.__doc__ = "Interval list created by merging two constituent interval lists using specified logical operation."
EventIntervalListType.FILTERED.__doc__ = "Defined by filtering intervals from original interval list using specified filtering method."
EventIntervalListType.CONDITION.__doc__ = "Interval list containing intervals during which specified condition is satisfied (UI type name is Satisfaction)."
EventIntervalListType.SCALED.__doc__ = "Interval List defined by scaling every interval in original interval list using either absolute or relative scale."
EventIntervalListType.SIGNALED.__doc__ = "Determine what interval list is recorded at target clock location by performing signal transmission of original interval list between base and target clock locations."
EventIntervalListType.TIME_OFFSET.__doc__ = "Interval List defined by shifting specified reference interval list by fixed time offset."
EventIntervalListType.FILE.__doc__ = "Interval list loaded from specified interval file."
EventIntervalListType.FIXED.__doc__ = "Interval list with individual intervals defined between explicitly specified start and stop times."

agcls.AgTypeNameMap["EventIntervalListType"] = EventIntervalListType

class EventIntervalType(IntEnum):
    """Define available interval types."""

    UNKNOWN = -1
    """Unknown or unsupported interval types."""
    FIXED = 0
    """Interval defined between two explicitly specified start and stop times."""
    FIXED_DURATION = 1
    """Interval of fixed duration specified using start and stop offsets relative to specified reference time instant."""
    BETWEEN_TIME_INSTANTS = 2
    """Interval between specified start and stop time instants."""
    FROM_INTERVAL_LIST = 3
    """Interval created from specified interval list by using one of several selection methods."""
    SCALED = 4
    """Interval defined by scaling original interval using either absolute or relative scale."""
    SIGNALED = 5
    """Determine an interval recorded at a target clock location by performing signal transmission."""
    TIME_OFFSET = 6
    """Interval defined by shifting specified reference interval by fixed time offset."""
    SMART_INTERVAL = 7
    """A smart interval."""

EventIntervalType.UNKNOWN.__doc__ = "Unknown or unsupported interval types."
EventIntervalType.FIXED.__doc__ = "Interval defined between two explicitly specified start and stop times."
EventIntervalType.FIXED_DURATION.__doc__ = "Interval of fixed duration specified using start and stop offsets relative to specified reference time instant."
EventIntervalType.BETWEEN_TIME_INSTANTS.__doc__ = "Interval between specified start and stop time instants."
EventIntervalType.FROM_INTERVAL_LIST.__doc__ = "Interval created from specified interval list by using one of several selection methods."
EventIntervalType.SCALED.__doc__ = "Interval defined by scaling original interval using either absolute or relative scale."
EventIntervalType.SIGNALED.__doc__ = "Determine an interval recorded at a target clock location by performing signal transmission."
EventIntervalType.TIME_OFFSET.__doc__ = "Interval defined by shifting specified reference interval by fixed time offset."
EventIntervalType.SMART_INTERVAL.__doc__ = "A smart interval."

agcls.AgTypeNameMap["EventIntervalType"] = EventIntervalType

class EventListMergeOperation(IntEnum):
    """Define merge operations for interval lists."""

    AND = 0
    """Intervals both in A and B."""
    OR = 1
    """Intervals in A or B."""
    XOR = 2
    """Intervals in A or B but not in both."""
    MINUS = 3
    """Intervals in A and not in B."""

EventListMergeOperation.AND.__doc__ = "Intervals both in A and B."
EventListMergeOperation.OR.__doc__ = "Intervals in A or B."
EventListMergeOperation.XOR.__doc__ = "Intervals in A or B but not in both."
EventListMergeOperation.MINUS.__doc__ = "Intervals in A and not in B."

agcls.AgTypeNameMap["EventListMergeOperation"] = EventListMergeOperation

class TimeEventType(IntEnum):
    """Define available time instant types."""

    UNKNOWN = -1
    """Unknown or unsupported time instant types."""
    EPOCH = 0
    """Time instant set at specified date/time."""
    EXTREMUM = 1
    """Determine time of global minimum or maximum of specified scalar calculation."""
    FROM_INTERVAL = 2
    """Start or stop time of selected reference interval."""
    SIGNALED = 3
    """Determine what time is recorded at target clock location by performing signal transmission of original time instant between base and target clock locations."""
    TIME_OFFSET = 4
    """Time instant at fixed offset from specified reference time instant."""
    SMART_EPOCH = 5
    """A smart epoch."""

TimeEventType.UNKNOWN.__doc__ = "Unknown or unsupported time instant types."
TimeEventType.EPOCH.__doc__ = "Time instant set at specified date/time."
TimeEventType.EXTREMUM.__doc__ = "Determine time of global minimum or maximum of specified scalar calculation."
TimeEventType.FROM_INTERVAL.__doc__ = "Start or stop time of selected reference interval."
TimeEventType.SIGNALED.__doc__ = "Determine what time is recorded at target clock location by performing signal transmission of original time instant between base and target clock locations."
TimeEventType.TIME_OFFSET.__doc__ = "Time instant at fixed offset from specified reference time instant."
TimeEventType.SMART_EPOCH.__doc__ = "A smart epoch."

agcls.AgTypeNameMap["TimeEventType"] = TimeEventType

class ExtremumType(IntFlag):
    """These constants are utilized when finding a local or global minimum or maximum, or the threshold crossing."""

    MINIMUM = 1
    """Find the minimum value."""
    MAXIMUM = 2
    """Find the maximum value."""

ExtremumType.MINIMUM.__doc__ = "Find the minimum value."
ExtremumType.MAXIMUM.__doc__ = "Find the maximum value."

agcls.AgTypeNameMap["ExtremumType"] = ExtremumType

class FileInterpolatorType(IntEnum):
    """Interpolator types."""

    INVALID = -1
    """Unknown or invalid interpolator."""
    LAGRANGE = 1
    """Lagrange interpolation."""
    HERMITE = 2
    """Hermite interpolation."""
    HOLD_PREVIOUS = 3
    """Holds the value at the closest previous sample time to any requested time."""
    HOLD_NEXT = 4
    """Holds the value at the closest next sample time to any requested time."""
    HOLD_NEAREST = 5
    """Holds the value at the closest sample time (either the previous sample or the next sample) to any requested time."""

FileInterpolatorType.INVALID.__doc__ = "Unknown or invalid interpolator."
FileInterpolatorType.LAGRANGE.__doc__ = "Lagrange interpolation."
FileInterpolatorType.HERMITE.__doc__ = "Hermite interpolation."
FileInterpolatorType.HOLD_PREVIOUS.__doc__ = "Holds the value at the closest previous sample time to any requested time."
FileInterpolatorType.HOLD_NEXT.__doc__ = "Holds the value at the closest next sample time to any requested time."
FileInterpolatorType.HOLD_NEAREST.__doc__ = "Holds the value at the closest sample time (either the previous sample or the next sample) to any requested time."

agcls.AgTypeNameMap["FileInterpolatorType"] = FileInterpolatorType

class QuadratureType(IntEnum):
    """Integral types."""

    FIXED_STEP_SIMPSON = 2
    """Simpson integral method."""
    FIXED_STEP_TRAPEZOID = 1
    """Trapezoidal integral method."""
    ADAPTIVE_STEP = 3
    """Adaptive Lobatto integral method."""

QuadratureType.FIXED_STEP_SIMPSON.__doc__ = "Simpson integral method."
QuadratureType.FIXED_STEP_TRAPEZOID.__doc__ = "Trapezoidal integral method."
QuadratureType.ADAPTIVE_STEP.__doc__ = "Adaptive Lobatto integral method."

agcls.AgTypeNameMap["QuadratureType"] = QuadratureType

class IntegrationWindowType(IntEnum):
    """Define the interval of times during which an integral is evaluated."""

    TOTAL = 0
    """Define the integral's window as the entire available interval list which effectively makes the value of the integral constant."""
    CUMULATIVE_TO_CURRENT = 1
    """Define the integral's window as the window of time from the beginning of the available interval until the current time, i.e. window duration grows over time."""
    CUMULATIVE_FROM_CURRENT = 2
    """Define the integral's window as window of time from the current time until the end of the available interval, i.e. window duration decreases over time."""
    SLIDING_WINDOW = 3
    """Define the integral's window as the interval of times centered around the current time with the specified front and back durations."""

IntegrationWindowType.TOTAL.__doc__ = "Define the integral's window as the entire available interval list which effectively makes the value of the integral constant."
IntegrationWindowType.CUMULATIVE_TO_CURRENT.__doc__ = "Define the integral's window as the window of time from the beginning of the available interval until the current time, i.e. window duration grows over time."
IntegrationWindowType.CUMULATIVE_FROM_CURRENT.__doc__ = "Define the integral's window as window of time from the current time until the end of the available interval, i.e. window duration decreases over time."
IntegrationWindowType.SLIDING_WINDOW.__doc__ = "Define the integral's window as the interval of times centered around the current time with the specified front and back durations."

agcls.AgTypeNameMap["IntegrationWindowType"] = IntegrationWindowType

class InterpolationMethodType(IntEnum):
    """Interpolator types."""

    INVALID = -1
    """Unknown or invalid interpolator."""
    LAGRANGE = 1
    """Lagrange interpolation."""
    HERMITE = 2
    """Hermite interpolation."""

InterpolationMethodType.INVALID.__doc__ = "Unknown or invalid interpolator."
InterpolationMethodType.LAGRANGE.__doc__ = "Lagrange interpolation."
InterpolationMethodType.HERMITE.__doc__ = "Hermite interpolation."

agcls.AgTypeNameMap["InterpolationMethodType"] = InterpolationMethodType

class IntervalDurationType(IntEnum):
    """Duration for filtering intervals or gaps from interval lists or time arrays."""

    AT_LEAST = 0
    """Filter by at least a specified number of seconds."""
    AT_MOST = 1
    """Filter by at most a specified number of seconds."""

IntervalDurationType.AT_LEAST.__doc__ = "Filter by at least a specified number of seconds."
IntervalDurationType.AT_MOST.__doc__ = "Filter by at most a specified number of seconds."

agcls.AgTypeNameMap["IntervalDurationType"] = IntervalDurationType

class IntervalFromIntervalListSelectionType(IntEnum):
    """Select the method to choose an interval from an interval list."""

    FROM_START = 1
    """Select an interval by counting a specified number from the first interval."""
    FROM_END = 2
    """Select an interval by counting a specified number back from the last interval."""
    MAXIMUM_DURATION = 4
    """Select the interval with the largest duration."""
    MINIMUM_DURATION = 5
    """Select the interval with the smallest duration."""
    MAXIMUM_GAP = 9
    """Select the largest gap between intervals."""
    MIN_GAP = 10
    """Select the largest gap between intervals."""
    SPAN = 100
    """Select the interval that is the span of the interval list."""

IntervalFromIntervalListSelectionType.FROM_START.__doc__ = "Select an interval by counting a specified number from the first interval."
IntervalFromIntervalListSelectionType.FROM_END.__doc__ = "Select an interval by counting a specified number back from the last interval."
IntervalFromIntervalListSelectionType.MAXIMUM_DURATION.__doc__ = "Select the interval with the largest duration."
IntervalFromIntervalListSelectionType.MINIMUM_DURATION.__doc__ = "Select the interval with the smallest duration."
IntervalFromIntervalListSelectionType.MAXIMUM_GAP.__doc__ = "Select the largest gap between intervals."
IntervalFromIntervalListSelectionType.MIN_GAP.__doc__ = "Select the largest gap between intervals."
IntervalFromIntervalListSelectionType.SPAN.__doc__ = "Select the interval that is the span of the interval list."

agcls.AgTypeNameMap["IntervalFromIntervalListSelectionType"] = IntervalFromIntervalListSelectionType

class ParameterSetType(IntEnum):
    """Define parameter set types."""

    UNKNOWN = -1
    """Unknown or unsupported parameter set."""
    ATTITUDE = 0
    """A parameter set type is defined by identifying one set of axes in reference to another."""
    GROUND_TRAJECTORY = 1
    """A parameter set type is defined by identifying location in reference central body."""
    TRAJECTORY = 2
    """A parameter set type is defined by identifying location in reference coordinate system."""
    ORBIT = 3
    """A parameter set type is defined by identifying orbiting point and its central body."""
    VECTOR = 4
    """A parameter set type is defined by identifying vector in reference axes."""

ParameterSetType.UNKNOWN.__doc__ = "Unknown or unsupported parameter set."
ParameterSetType.ATTITUDE.__doc__ = "A parameter set type is defined by identifying one set of axes in reference to another."
ParameterSetType.GROUND_TRAJECTORY.__doc__ = "A parameter set type is defined by identifying location in reference central body."
ParameterSetType.TRAJECTORY.__doc__ = "A parameter set type is defined by identifying location in reference coordinate system."
ParameterSetType.ORBIT.__doc__ = "A parameter set type is defined by identifying orbiting point and its central body."
ParameterSetType.VECTOR.__doc__ = "A parameter set type is defined by identifying vector in reference axes."

agcls.AgTypeNameMap["ParameterSetType"] = ParameterSetType

class IntervalPruneFilterType(IntEnum):
    """Specify the filter for filtering interval lists or time arrays."""

    UNKNOWN = 0
    """Unknown or unsupported prune filter."""
    FIRST_INTERVALS = 1
    """Select specified number of first intervals from original list."""
    LAST_INTERVALS = 2
    """Select specified number of last intervals from original list."""
    INTERVALS = 3
    """Select intervals which satisfy additional duration condition."""
    GAPS = 4
    """Select gaps between intervals which satisfy additional duration condition."""
    SATISFACTION_INTERVALS = 5
    """Satisfaction Intervals selects intervals which satisfy additional condition and duration."""
    RELATIVE_SATISFACTION_INTERVALS = 6
    """Relative Satisfaction Intervals selects intervals which satisfy additional condition and duration."""

IntervalPruneFilterType.UNKNOWN.__doc__ = "Unknown or unsupported prune filter."
IntervalPruneFilterType.FIRST_INTERVALS.__doc__ = "Select specified number of first intervals from original list."
IntervalPruneFilterType.LAST_INTERVALS.__doc__ = "Select specified number of last intervals from original list."
IntervalPruneFilterType.INTERVALS.__doc__ = "Select intervals which satisfy additional duration condition."
IntervalPruneFilterType.GAPS.__doc__ = "Select gaps between intervals which satisfy additional duration condition."
IntervalPruneFilterType.SATISFACTION_INTERVALS.__doc__ = "Satisfaction Intervals selects intervals which satisfy additional condition and duration."
IntervalPruneFilterType.RELATIVE_SATISFACTION_INTERVALS.__doc__ = "Relative Satisfaction Intervals selects intervals which satisfy additional condition and duration."

agcls.AgTypeNameMap["IntervalPruneFilterType"] = IntervalPruneFilterType

class SampleReferenceTimeType(IntEnum):
    """Event array reference type."""

    TIME_INSTANT = 0
    """Use a Time Instant as the reference time."""
    START_OF_EACH_INTERVAL = 1
    """Use the start of each interval as the reference time."""
    STOP_OF_EACH_INTERVAL = 2
    """Use the stop of each interval as the reference time."""
    START_OF_INTERVAL_LIST = 3
    """Use the start of each interval list as the reference time."""
    STOP_OF_INTERVAL_LIST = 4
    """Use the stop of each interval list as the reference time."""

SampleReferenceTimeType.TIME_INSTANT.__doc__ = "Use a Time Instant as the reference time."
SampleReferenceTimeType.START_OF_EACH_INTERVAL.__doc__ = "Use the start of each interval as the reference time."
SampleReferenceTimeType.STOP_OF_EACH_INTERVAL.__doc__ = "Use the stop of each interval as the reference time."
SampleReferenceTimeType.START_OF_INTERVAL_LIST.__doc__ = "Use the start of each interval list as the reference time."
SampleReferenceTimeType.STOP_OF_INTERVAL_LIST.__doc__ = "Use the stop of each interval list as the reference time."

agcls.AgTypeNameMap["SampleReferenceTimeType"] = SampleReferenceTimeType

class VectorGeometryToolSamplingMethod(IntEnum):
    """Define the Sampling Method."""

    UNKNOWN = 0
    """Unknown or unsupported sampling method."""
    FIXED_STEP = 1
    """Fixed step sampling method."""
    RELATIVE_TOLERANCE = 2
    """Relative tolerance method uses a combination of relative and absolute tolerance changes in scalar values between samples."""
    CURVATURE_TOLERANCE = 3
    """Curvature tolerance also uses changes in slope between samples."""

VectorGeometryToolSamplingMethod.UNKNOWN.__doc__ = "Unknown or unsupported sampling method."
VectorGeometryToolSamplingMethod.FIXED_STEP.__doc__ = "Fixed step sampling method."
VectorGeometryToolSamplingMethod.RELATIVE_TOLERANCE.__doc__ = "Relative tolerance method uses a combination of relative and absolute tolerance changes in scalar values between samples."
VectorGeometryToolSamplingMethod.CURVATURE_TOLERANCE.__doc__ = "Curvature tolerance also uses changes in slope between samples."

agcls.AgTypeNameMap["VectorGeometryToolSamplingMethod"] = VectorGeometryToolSamplingMethod

class SatisfactionCrossing(IntEnum):
    """Direction crossing flags."""

    NONE = 0
    """Use either off-to-on or on-to-off condition to determine satisfaction."""
    IN = 1
    """Satisfaction occurs when crossing from off to on condition."""
    OUT = 2
    """Satisfaction occurs when crossing from on to off condition."""

SatisfactionCrossing.NONE.__doc__ = "Use either off-to-on or on-to-off condition to determine satisfaction."
SatisfactionCrossing.IN.__doc__ = "Satisfaction occurs when crossing from off to on condition."
SatisfactionCrossing.OUT.__doc__ = "Satisfaction occurs when crossing from on to off condition."

agcls.AgTypeNameMap["SatisfactionCrossing"] = SatisfactionCrossing

class SaveDataType(IntEnum):
    """Method for saving computed data."""

    APPLICATION_SETTINGS = -1
    """Use the application setting to determine whether computed data should be saved/loaded."""
    YES = 100
    """Save/load computed data."""
    NO = 10
    """Don't save computed data, recompute data on load."""

SaveDataType.APPLICATION_SETTINGS.__doc__ = "Use the application setting to determine whether computed data should be saved/loaded."
SaveDataType.YES.__doc__ = "Save/load computed data."
SaveDataType.NO.__doc__ = "Don't save computed data, recompute data on load."

agcls.AgTypeNameMap["SaveDataType"] = SaveDataType

class SignalPathReferenceSystem(IntEnum):
    """Signal path reference system types."""

    USE_ACCESS_DEFAULT = -1
    """Use Access default system."""
    CENTRAL_BODY_INERTIAL = 0
    """Use central body inertial system."""
    SOLAR_SYSTEM_BARYCENTER = 1
    """Use solar system barycenter system."""
    CUSTOM = 2
    """User will specify a system."""

SignalPathReferenceSystem.USE_ACCESS_DEFAULT.__doc__ = "Use Access default system."
SignalPathReferenceSystem.CENTRAL_BODY_INERTIAL.__doc__ = "Use central body inertial system."
SignalPathReferenceSystem.SOLAR_SYSTEM_BARYCENTER.__doc__ = "Use solar system barycenter system."
SignalPathReferenceSystem.CUSTOM.__doc__ = "User will specify a system."

agcls.AgTypeNameMap["SignalPathReferenceSystem"] = SignalPathReferenceSystem

class SmartEpochState(IntEnum):
    """Smart epoch states."""

    EXPLICIT = 0
    """Smart epoch is specified explicitly using a time."""
    IMPLICIT = 3
    """Smart epoch is specified implicitly."""

SmartEpochState.EXPLICIT.__doc__ = "Smart epoch is specified explicitly using a time."
SmartEpochState.IMPLICIT.__doc__ = "Smart epoch is specified implicitly."

agcls.AgTypeNameMap["SmartEpochState"] = SmartEpochState

class SmartIntervalState(IntEnum):
    """Smart interval states."""

    EXPLICIT = 0
    """Smart interval is specified explicitly using start/stop times."""
    IMPLICIT = 1
    """Smart interval is specified implicitly using start/stop times."""
    START_STOP = 2
    """Smart interval is specified using smart epochs."""
    START_DURATION = 4
    """Smart interval is specified using a start epoch and duration."""
    EXPLICIT_DURATION = 3
    """Smart interval is specified using a start time and explicit duration."""

SmartIntervalState.EXPLICIT.__doc__ = "Smart interval is specified explicitly using start/stop times."
SmartIntervalState.IMPLICIT.__doc__ = "Smart interval is specified implicitly using start/stop times."
SmartIntervalState.START_STOP.__doc__ = "Smart interval is specified using smart epochs."
SmartIntervalState.START_DURATION.__doc__ = "Smart interval is specified using a start epoch and duration."
SmartIntervalState.EXPLICIT_DURATION.__doc__ = "Smart interval is specified using a start time and explicit duration."

agcls.AgTypeNameMap["SmartIntervalState"] = SmartIntervalState

class SpeedType(IntEnum):
    """Define various speed options."""

    LIGHT_TRANSMISSION_SPEED = 1
    """Light transmission speed."""
    CUSTOM_TRANSMISSION_SPEED = 2
    """Custom transmission speed."""

SpeedType.LIGHT_TRANSMISSION_SPEED.__doc__ = "Light transmission speed."
SpeedType.CUSTOM_TRANSMISSION_SPEED.__doc__ = "Custom transmission speed."

agcls.AgTypeNameMap["SpeedType"] = SpeedType

class StartStopType(IntEnum):
    """Start/stop options."""

    COUNT_START_ONLY = 1
    """Use the start time of the interval."""
    COUNT_STOP_ONLY = 2
    """Use the stop time of the interval."""
    COUNT_START_STOP = 3
    """Use the start time and stop time of the interval."""

StartStopType.COUNT_START_ONLY.__doc__ = "Use the start time of the interval."
StartStopType.COUNT_STOP_ONLY.__doc__ = "Use the stop time of the interval."
StartStopType.COUNT_START_STOP.__doc__ = "Use the start time and stop time of the interval."

agcls.AgTypeNameMap["StartStopType"] = StartStopType

class ThresholdConvergenceSenseType(IntEnum):
    """Specify the desired sense of the results from threshold crossing computations."""

    SIMPLE = 0
    """Just converge within tolerance."""
    ABOVE = 1
    """Result above or at threshold."""
    BELOW = 2
    """Result below or at threshold."""

ThresholdConvergenceSenseType.SIMPLE.__doc__ = "Just converge within tolerance."
ThresholdConvergenceSenseType.ABOVE.__doc__ = "Result above or at threshold."
ThresholdConvergenceSenseType.BELOW.__doc__ = "Result below or at threshold."

agcls.AgTypeNameMap["ThresholdConvergenceSenseType"] = ThresholdConvergenceSenseType

class VectorComponentType(IntEnum):
    """Define component directions for a vector."""

    X = 0
    """X component."""
    Y = 1
    """Y component."""
    Z = 2
    """Z component."""
    NEGATIVE_X = 3
    """-X component."""
    NEGATIVE_Y = 4
    """-Y component."""
    NEGATIVE_Z = 5
    """-Z component."""

VectorComponentType.X.__doc__ = "X component."
VectorComponentType.Y.__doc__ = "Y component."
VectorComponentType.Z.__doc__ = "Z component."
VectorComponentType.NEGATIVE_X.__doc__ = "-X component."
VectorComponentType.NEGATIVE_Y.__doc__ = "-Y component."
VectorComponentType.NEGATIVE_Z.__doc__ = "-Z component."

agcls.AgTypeNameMap["VectorComponentType"] = VectorComponentType

class SpatialCalculationAltitudeReferenceType(IntEnum):
    """Define volume calc altitude reference types."""

    ELLIPSOID = 0
    """Volume calc altitude above reference ellipsoid."""
    TERRAIN = 1
    """Volume calc altitude reference above terrain."""
    MSL = 2
    """Volume calc altitude reference above mean sea level."""

SpatialCalculationAltitudeReferenceType.ELLIPSOID.__doc__ = "Volume calc altitude above reference ellipsoid."
SpatialCalculationAltitudeReferenceType.TERRAIN.__doc__ = "Volume calc altitude reference above terrain."
SpatialCalculationAltitudeReferenceType.MSL.__doc__ = "Volume calc altitude reference above mean sea level."

agcls.AgTypeNameMap["SpatialCalculationAltitudeReferenceType"] = SpatialCalculationAltitudeReferenceType

class AngleToLocationType(IntEnum):
    """Define volume calc angle off vector reference types."""

    FROM_PLANE_SIGNED = 1
    """Volume calc angle off plane signed."""
    FROM_PLANE_UNSIGNED = 2
    """Volume calc angle off plane unsigned."""
    ABOUT_VECTOR_SIGNED = 4
    """Volume calc angle about vector signed."""
    ABOUT_VECTOR_UNSIGNED = 8
    """Volume calc angle about vector unsigned."""
    FROM_REFERENCE_VECTOR = 16
    """Volume calc angle off vector."""

AngleToLocationType.FROM_PLANE_SIGNED.__doc__ = "Volume calc angle off plane signed."
AngleToLocationType.FROM_PLANE_UNSIGNED.__doc__ = "Volume calc angle off plane unsigned."
AngleToLocationType.ABOUT_VECTOR_SIGNED.__doc__ = "Volume calc angle about vector signed."
AngleToLocationType.ABOUT_VECTOR_UNSIGNED.__doc__ = "Volume calc angle about vector unsigned."
AngleToLocationType.FROM_REFERENCE_VECTOR.__doc__ = "Volume calc angle off vector."

agcls.AgTypeNameMap["AngleToLocationType"] = AngleToLocationType

class DistanceToLocationType(IntEnum):
    """Define volume calc range distance types."""

    FROM_POINT = 0
    """Volume calc range distance type from point."""
    ALONG_VECTOR_SIGNED = 1
    """Volume calc range distance type from point."""
    ALONG_VECTOR_UNSIGNED = 2
    """Volume calc range distance type along vector unsigned."""
    FROM_PLANE_SIGNED = 4
    """Volume calc range distance type plane signed."""
    FROM_PLANE_UNSIGNED = 8
    """Volume calc range distance type plane signed."""

DistanceToLocationType.FROM_POINT.__doc__ = "Volume calc range distance type from point."
DistanceToLocationType.ALONG_VECTOR_SIGNED.__doc__ = "Volume calc range distance type from point."
DistanceToLocationType.ALONG_VECTOR_UNSIGNED.__doc__ = "Volume calc range distance type along vector unsigned."
DistanceToLocationType.FROM_PLANE_SIGNED.__doc__ = "Volume calc range distance type plane signed."
DistanceToLocationType.FROM_PLANE_UNSIGNED.__doc__ = "Volume calc range distance type plane signed."

agcls.AgTypeNameMap["DistanceToLocationType"] = DistanceToLocationType

class RangeSpeedType(IntEnum):
    """Define volume calc range distance types."""

    LIGHT_SPEED = 1
    """Volume calc range distance type from point."""
    CUSTOM = 2
    """Volume calc range distance type from point."""

RangeSpeedType.LIGHT_SPEED.__doc__ = "Volume calc range distance type from point."
RangeSpeedType.CUSTOM.__doc__ = "Volume calc range distance type from point."

agcls.AgTypeNameMap["RangeSpeedType"] = RangeSpeedType

class SpatialCalculationType(IntEnum):
    """Define volume calc types."""

    UNKNOWN = -1
    """Unknown or unsupported volume calc."""
    ALTITUDE_AT_LOCATION = 0
    """volume calc type altitude to location."""
    ANGLE_TO_LOCATION = 1
    """volume calc type angle to location."""
    FILE = 2
    """volume calc type File."""
    FROM_CALCULATION_SCALAR = 3
    """volume calc type scalar at location."""
    SOLAR_INTENSITY = 4
    """volume calc type Solar Intensity."""
    SPATIAL_CONDITION_METRIC = 5
    """volume calc type Spatial Condition Satisfaction Metric."""
    RANGE = 6
    """volume calc type Distance to Location."""
    PROPAGATION_DELAY_TO_LOCATION = 7
    """volume calc type Propagation Delay to Location."""

SpatialCalculationType.UNKNOWN.__doc__ = "Unknown or unsupported volume calc."
SpatialCalculationType.ALTITUDE_AT_LOCATION.__doc__ = "volume calc type altitude to location."
SpatialCalculationType.ANGLE_TO_LOCATION.__doc__ = "volume calc type angle to location."
SpatialCalculationType.FILE.__doc__ = "volume calc type File."
SpatialCalculationType.FROM_CALCULATION_SCALAR.__doc__ = "volume calc type scalar at location."
SpatialCalculationType.SOLAR_INTENSITY.__doc__ = "volume calc type Solar Intensity."
SpatialCalculationType.SPATIAL_CONDITION_METRIC.__doc__ = "volume calc type Spatial Condition Satisfaction Metric."
SpatialCalculationType.RANGE.__doc__ = "volume calc type Distance to Location."
SpatialCalculationType.PROPAGATION_DELAY_TO_LOCATION.__doc__ = "volume calc type Propagation Delay to Location."

agcls.AgTypeNameMap["SpatialCalculationType"] = SpatialCalculationType

class VolumeSatisfactionAccumulationType(IntEnum):
    """Define volume calc spatial condition accumulation types."""

    UP_TO_CURRENT_TIME = -1
    """Volume calc spatial condition satisfaction accumulation type up to current time."""
    CURRENT_TIME = 0
    """Volume calc spatial condition satisfaction accumulation type current time."""
    FROM_CURRENT_TIME = 1
    """Volume calc spatial condition satisfaction accumulation type from current time."""
    TOTAL = 10
    """Volume calc spatial condition satisfaction accumulation type total."""

VolumeSatisfactionAccumulationType.UP_TO_CURRENT_TIME.__doc__ = "Volume calc spatial condition satisfaction accumulation type up to current time."
VolumeSatisfactionAccumulationType.CURRENT_TIME.__doc__ = "Volume calc spatial condition satisfaction accumulation type current time."
VolumeSatisfactionAccumulationType.FROM_CURRENT_TIME.__doc__ = "Volume calc spatial condition satisfaction accumulation type from current time."
VolumeSatisfactionAccumulationType.TOTAL.__doc__ = "Volume calc spatial condition satisfaction accumulation type total."

agcls.AgTypeNameMap["VolumeSatisfactionAccumulationType"] = VolumeSatisfactionAccumulationType

class VolumeSatisfactionDurationType(IntEnum):
    """Define volume calc spatial condition duration types."""

    MINIMUM = -1
    """Volume calc spatial condition satisfaction duration type minimum."""
    SUM = 0
    """Volume calc spatial condition satisfaction duration type sum."""
    MAXIMUM = 1
    """Volume calc spatial condition satisfaction duration type maximum."""

VolumeSatisfactionDurationType.MINIMUM.__doc__ = "Volume calc spatial condition satisfaction duration type minimum."
VolumeSatisfactionDurationType.SUM.__doc__ = "Volume calc spatial condition satisfaction duration type sum."
VolumeSatisfactionDurationType.MAXIMUM.__doc__ = "Volume calc spatial condition satisfaction duration type maximum."

agcls.AgTypeNameMap["VolumeSatisfactionDurationType"] = VolumeSatisfactionDurationType

class VolumeSatisfactionFilterType(IntEnum):
    """Define volume calc spatial condition filter types."""

    FIRST_INTERVALS = -10
    """Volume calc spatial condition satisfaction filter type first intervals."""
    LAST_INTERVALS = -20
    """Volume calc spatial condition satisfaction filter type last intervals."""
    NONE = 0
    """Volume calc spatial condition satisfaction filter type none."""
    GAP_DURATION = 1
    """Volume calc spatial condition satisfaction filter type gap duration."""
    INTERVAL_DURATION = 2
    """Volume calc spatial condition satisfaction filter type interval duration."""

VolumeSatisfactionFilterType.FIRST_INTERVALS.__doc__ = "Volume calc spatial condition satisfaction filter type first intervals."
VolumeSatisfactionFilterType.LAST_INTERVALS.__doc__ = "Volume calc spatial condition satisfaction filter type last intervals."
VolumeSatisfactionFilterType.NONE.__doc__ = "Volume calc spatial condition satisfaction filter type none."
VolumeSatisfactionFilterType.GAP_DURATION.__doc__ = "Volume calc spatial condition satisfaction filter type gap duration."
VolumeSatisfactionFilterType.INTERVAL_DURATION.__doc__ = "Volume calc spatial condition satisfaction filter type interval duration."

agcls.AgTypeNameMap["VolumeSatisfactionFilterType"] = VolumeSatisfactionFilterType

class VolumeSatisfactionMetricType(IntEnum):
    """Define volume calc spatial condition satisfaction metric types."""

    NUMBER_OF_GAPS = -2
    """Volume calc spatial condition satisfaction metric type number of gaps."""
    NUMBER_OF_INTERVALS = -1
    """Volume calc spatial condition satisfaction metric type number of intervals."""
    TIME_SINCE_LAST_SATISFACTION = 1
    """Volume calc spatial condition satisfaction metric type time since last satisfaction."""
    TIME_UNTIL_NEXT_SATISFACTION = 2
    """Volume calc spatial condition satisfaction metric type time until next satisfaction."""
    INTERVAL_DURATION = 10
    """Volume calc spatial condition satisfaction metric type interbnal duration."""
    GAP_DURATION = 20
    """Volume calc spatial condition satisfaction metric type gap duration."""

VolumeSatisfactionMetricType.NUMBER_OF_GAPS.__doc__ = "Volume calc spatial condition satisfaction metric type number of gaps."
VolumeSatisfactionMetricType.NUMBER_OF_INTERVALS.__doc__ = "Volume calc spatial condition satisfaction metric type number of intervals."
VolumeSatisfactionMetricType.TIME_SINCE_LAST_SATISFACTION.__doc__ = "Volume calc spatial condition satisfaction metric type time since last satisfaction."
VolumeSatisfactionMetricType.TIME_UNTIL_NEXT_SATISFACTION.__doc__ = "Volume calc spatial condition satisfaction metric type time until next satisfaction."
VolumeSatisfactionMetricType.INTERVAL_DURATION.__doc__ = "Volume calc spatial condition satisfaction metric type interbnal duration."
VolumeSatisfactionMetricType.GAP_DURATION.__doc__ = "Volume calc spatial condition satisfaction metric type gap duration."

agcls.AgTypeNameMap["VolumeSatisfactionMetricType"] = VolumeSatisfactionMetricType

class VolumeGridType(IntEnum):
    """Define volume grid types."""

    UNKNOWN = -1
    """Unknown or unsupported volume grid."""
    CARTESIAN = 0
    """volume grid type cartesian."""
    CYLINDRICAL = 1
    """volume grid type Cylindrical."""
    SPHERICAL = 2
    """volume grid type Spherical."""
    CONSTRAINED = 3
    """volume grid type Constrained."""
    LATITUDE_LONGITUDE_ALTITUDE = 4
    """volume grid type LatLonAlt (Cartographic)."""
    BEARING_ALTITUDE = 5
    """volume grid type BearingAlt (Surface Bearing)."""

VolumeGridType.UNKNOWN.__doc__ = "Unknown or unsupported volume grid."
VolumeGridType.CARTESIAN.__doc__ = "volume grid type cartesian."
VolumeGridType.CYLINDRICAL.__doc__ = "volume grid type Cylindrical."
VolumeGridType.SPHERICAL.__doc__ = "volume grid type Spherical."
VolumeGridType.CONSTRAINED.__doc__ = "volume grid type Constrained."
VolumeGridType.LATITUDE_LONGITUDE_ALTITUDE.__doc__ = "volume grid type LatLonAlt (Cartographic)."
VolumeGridType.BEARING_ALTITUDE.__doc__ = "volume grid type BearingAlt (Surface Bearing)."

agcls.AgTypeNameMap["VolumeGridType"] = VolumeGridType

class ResultVectorRequestType(IntEnum):
    """Define volume result vector request types."""

    POSITION = 1
    """volume result vector request pos type."""
    NATIVE_POSITION = 2
    """volume result vector request native pos type."""
    METRIC = 16
    """volume result vector request metric type."""
    SATISFACTION = 32
    """volume result vector request satisfaction type."""
    GRADIENT = 256
    """volume result vector request gradient type."""

ResultVectorRequestType.POSITION.__doc__ = "volume result vector request pos type."
ResultVectorRequestType.NATIVE_POSITION.__doc__ = "volume result vector request native pos type."
ResultVectorRequestType.METRIC.__doc__ = "volume result vector request metric type."
ResultVectorRequestType.SATISFACTION.__doc__ = "volume result vector request satisfaction type."
ResultVectorRequestType.GRADIENT.__doc__ = "volume result vector request gradient type."

agcls.AgTypeNameMap["ResultVectorRequestType"] = ResultVectorRequestType

class VolumeType(IntEnum):
    """Define volume grid types."""

    UNKNOWN = -1
    """Unknown or unsupported volume."""
    COMBINED = 0
    """volume type combined."""
    LIGHTING = 1
    """volume type lighting."""
    OVER_TIME = 2
    """volume type over time."""
    GRID_BOUNDING_VOLUME = 3
    """volume type from grid (Grid Bounding Volume)."""
    SPATIAL_CALCULATION_BOUNDS = 4
    """volume type from calc (Spatial Calculation Bounds)."""
    VALID_TIME_AT_LOCATION = 5
    """volume type from time satisfaction (Valid Time At Location)."""
    CONDITION_AT_LOCATION = 6
    """volume type from condition (Condition At Location)."""
    ACCESS_TO_LOCATION = 7
    """volume type Inview (Access To Location)."""

VolumeType.UNKNOWN.__doc__ = "Unknown or unsupported volume."
VolumeType.COMBINED.__doc__ = "volume type combined."
VolumeType.LIGHTING.__doc__ = "volume type lighting."
VolumeType.OVER_TIME.__doc__ = "volume type over time."
VolumeType.GRID_BOUNDING_VOLUME.__doc__ = "volume type from grid (Grid Bounding Volume)."
VolumeType.SPATIAL_CALCULATION_BOUNDS.__doc__ = "volume type from calc (Spatial Calculation Bounds)."
VolumeType.VALID_TIME_AT_LOCATION.__doc__ = "volume type from time satisfaction (Valid Time At Location)."
VolumeType.CONDITION_AT_LOCATION.__doc__ = "volume type from condition (Condition At Location)."
VolumeType.ACCESS_TO_LOCATION.__doc__ = "volume type Inview (Access To Location)."

agcls.AgTypeNameMap["VolumeType"] = VolumeType

class AberrationModelType(IntEnum):
    """Define the model of aberration to use."""

    UNKNOWN = -1
    """Aberration type unknown."""
    TOTAL = 0
    """Aberration type total."""
    ANNUAL = 1
    """SAberration type annual."""
    NONE = 2
    """Aberration type none."""

AberrationModelType.UNKNOWN.__doc__ = "Aberration type unknown."
AberrationModelType.TOTAL.__doc__ = "Aberration type total."
AberrationModelType.ANNUAL.__doc__ = "SAberration type annual."
AberrationModelType.NONE.__doc__ = "Aberration type none."

agcls.AgTypeNameMap["AberrationModelType"] = AberrationModelType

class ClockHostType(IntEnum):
    """Define whether base or target of an Access instance holds the clock for Access times."""

    UNKNOWN = -1
    """Unknown."""
    BASE = 0
    """Base object holds time instance."""
    TARGET = 1
    """Target object holds time instance."""

ClockHostType.UNKNOWN.__doc__ = "Unknown."
ClockHostType.BASE.__doc__ = "Base object holds time instance."
ClockHostType.TARGET.__doc__ = "Target object holds time instance."

agcls.AgTypeNameMap["ClockHostType"] = ClockHostType

class VolumeCombinedOperationType(IntEnum):
    """Define spatial condition combined operation types."""

    AND = 1
    """Spatial condition combined AND operation."""
    OR = 2
    """Spatial condition combined OR operation."""
    XOR = 3
    """Spatial condition combined XOR operation."""
    MINUS = 4
    """Spatial condition combined MINUS operation."""

VolumeCombinedOperationType.AND.__doc__ = "Spatial condition combined AND operation."
VolumeCombinedOperationType.OR.__doc__ = "Spatial condition combined OR operation."
VolumeCombinedOperationType.XOR.__doc__ = "Spatial condition combined XOR operation."
VolumeCombinedOperationType.MINUS.__doc__ = "Spatial condition combined MINUS operation."

agcls.AgTypeNameMap["VolumeCombinedOperationType"] = VolumeCombinedOperationType

class VolumeFromGridEdgeType(IntEnum):
    """Define spatial condition from grid edge type."""

    MASK_POINTS = 16
    """Spatial condition over time from grid edge type mask points."""
    MASK_VOXELS = 32
    """Spatial condition over time from grid edge type mask voxels."""

VolumeFromGridEdgeType.MASK_POINTS.__doc__ = "Spatial condition over time from grid edge type mask points."
VolumeFromGridEdgeType.MASK_VOXELS.__doc__ = "Spatial condition over time from grid edge type mask voxels."

agcls.AgTypeNameMap["VolumeFromGridEdgeType"] = VolumeFromGridEdgeType

class LightingConditionsType(IntFlag):
    """Define spatial condition lighting conditions types."""

    UNDEFINED = 0
    """Spatial condition lighting undefined."""
    SUNLIGHT = 1
    """Spatial condition lighting sun light."""
    PENUMBRA = 2
    """Spatial condition lighting penumbra."""
    TYPE_UMBRA = 4
    """Spatial condition lighting umbra."""

LightingConditionsType.UNDEFINED.__doc__ = "Spatial condition lighting undefined."
LightingConditionsType.SUNLIGHT.__doc__ = "Spatial condition lighting sun light."
LightingConditionsType.PENUMBRA.__doc__ = "Spatial condition lighting penumbra."
LightingConditionsType.TYPE_UMBRA.__doc__ = "Spatial condition lighting umbra."

agcls.AgTypeNameMap["LightingConditionsType"] = LightingConditionsType

class SpatialConditionOverTypeDurationType(IntEnum):
    """Define spatial condition over time duration type."""

    STATIC = 0
    """Spatial condition over time duration type Static."""
    CUMULATIVE_TO_CURRENT_TIME = 1
    """Spatial condition over time duration type CumulativeToCurrent."""
    CUMULATIVE_FROM_CURRENT_TIME = 2
    """Spatial condition over time duration type CumulativeFromCurrent."""
    SLIDING_WINDOW = 3
    """Spatial condition over time duration type SlidingWindow."""

SpatialConditionOverTypeDurationType.STATIC.__doc__ = "Spatial condition over time duration type Static."
SpatialConditionOverTypeDurationType.CUMULATIVE_TO_CURRENT_TIME.__doc__ = "Spatial condition over time duration type CumulativeToCurrent."
SpatialConditionOverTypeDurationType.CUMULATIVE_FROM_CURRENT_TIME.__doc__ = "Spatial condition over time duration type CumulativeFromCurrent."
SpatialConditionOverTypeDurationType.SLIDING_WINDOW.__doc__ = "Spatial condition over time duration type SlidingWindow."

agcls.AgTypeNameMap["SpatialConditionOverTypeDurationType"] = SpatialConditionOverTypeDurationType

class TimeSenseType(IntEnum):
    """Define whether object1 or object2 of an Access instance holds the clock for Access times."""

    UNKNOWN = -1
    """Unklnown."""
    TRANSMIT = 0
    """Position is computed in Time Sense."""
    RECEIVE = 1
    """Position is computed in Receive Sense."""

TimeSenseType.UNKNOWN.__doc__ = "Unklnown."
TimeSenseType.TRANSMIT.__doc__ = "Position is computed in Time Sense."
TimeSenseType.RECEIVE.__doc__ = "Position is computed in Receive Sense."

agcls.AgTypeNameMap["TimeSenseType"] = TimeSenseType

class GridValuesMethodType(IntEnum):
    """Define volumetric grid values method types."""

    UNKNOWN = -1
    """Unknown or unsupportedgrid values method."""
    FIXED_NUMBER_OF_STEPS = 0
    """Fixed number steps grid values method for volumetric grid."""
    FIXED_STEP_SIZE = 1
    """Fixed step size grid values method for volumetric grid."""
    CUSTOM_VALUES = 2
    """Custom grid values method for volumetric grid."""

GridValuesMethodType.UNKNOWN.__doc__ = "Unknown or unsupportedgrid values method."
GridValuesMethodType.FIXED_NUMBER_OF_STEPS.__doc__ = "Fixed number steps grid values method for volumetric grid."
GridValuesMethodType.FIXED_STEP_SIZE.__doc__ = "Fixed step size grid values method for volumetric grid."
GridValuesMethodType.CUSTOM_VALUES.__doc__ = "Custom grid values method for volumetric grid."

agcls.AgTypeNameMap["GridValuesMethodType"] = GridValuesMethodType

class VectorGeometryToolComponentType(IntEnum):
    """Represents kinds of vectory geometry components."""

    UNKNOWN = -1
    """Unsupported component kind."""
    INVALID = 0
    """Invalid component."""
    AXES = 1
    """Axes component."""
    ANGLE = 2
    """Angle component."""
    VECTOR = 3
    """Vector component."""
    POINT = 4
    """Point component."""
    PLANE = 5
    """Plane component."""
    SYSTEM = 6
    """System component."""
    TIME_INSTANT = 7
    """An event."""
    TIME_ARRAY = 8
    """An event array."""
    TIME_INTERVAL = 9
    """An event interval."""
    TIME_INTERVAL_COLLECTION = 10
    """An event interval collection."""
    TIME_INTERVAL_LIST = 11
    """A list of event intervals."""
    PARAMETER_SET = 12
    """A parameter set."""
    CALCULATION_SCALAR = 13
    """A scalar."""
    CONDITION = 14
    """A condition."""
    CONDITION_SET = 15
    """A condition set."""
    SPATIAL_VOLUME_GRID = 16
    """A volume grid."""
    VOLUME = 17
    """A volume."""
    SPATIAL_CALCULATION = 18
    """A volume calc."""

VectorGeometryToolComponentType.UNKNOWN.__doc__ = "Unsupported component kind."
VectorGeometryToolComponentType.INVALID.__doc__ = "Invalid component."
VectorGeometryToolComponentType.AXES.__doc__ = "Axes component."
VectorGeometryToolComponentType.ANGLE.__doc__ = "Angle component."
VectorGeometryToolComponentType.VECTOR.__doc__ = "Vector component."
VectorGeometryToolComponentType.POINT.__doc__ = "Point component."
VectorGeometryToolComponentType.PLANE.__doc__ = "Plane component."
VectorGeometryToolComponentType.SYSTEM.__doc__ = "System component."
VectorGeometryToolComponentType.TIME_INSTANT.__doc__ = "An event."
VectorGeometryToolComponentType.TIME_ARRAY.__doc__ = "An event array."
VectorGeometryToolComponentType.TIME_INTERVAL.__doc__ = "An event interval."
VectorGeometryToolComponentType.TIME_INTERVAL_COLLECTION.__doc__ = "An event interval collection."
VectorGeometryToolComponentType.TIME_INTERVAL_LIST.__doc__ = "A list of event intervals."
VectorGeometryToolComponentType.PARAMETER_SET.__doc__ = "A parameter set."
VectorGeometryToolComponentType.CALCULATION_SCALAR.__doc__ = "A scalar."
VectorGeometryToolComponentType.CONDITION.__doc__ = "A condition."
VectorGeometryToolComponentType.CONDITION_SET.__doc__ = "A condition set."
VectorGeometryToolComponentType.SPATIAL_VOLUME_GRID.__doc__ = "A volume grid."
VectorGeometryToolComponentType.VOLUME.__doc__ = "A volume."
VectorGeometryToolComponentType.SPATIAL_CALCULATION.__doc__ = "A volume calc."

agcls.AgTypeNameMap["VectorGeometryToolComponentType"] = VectorGeometryToolComponentType

class AngleType(IntEnum):
    """Represents angle types."""

    UNKNOWN = -1
    """Unknown or unsupported type."""
    BETWEEN_VECTORS = 0
    """An angle between two vectors."""
    BETWEEN_PLANES = 1
    """An angle between two planes."""
    DIHEDRAL_ANGLE = 2
    """An angle between two vectors about an axis."""
    ROTATION_ANGLE = 3
    """Angle of the shortest rotation between the two specified axes."""
    TO_PLANE = 4
    """An angle between a vector and a plane."""
    TEMPLATE = 5
    """Represents a VGT angle created from a template. This type of angle is not creatable."""

AngleType.UNKNOWN.__doc__ = "Unknown or unsupported type."
AngleType.BETWEEN_VECTORS.__doc__ = "An angle between two vectors."
AngleType.BETWEEN_PLANES.__doc__ = "An angle between two planes."
AngleType.DIHEDRAL_ANGLE.__doc__ = "An angle between two vectors about an axis."
AngleType.ROTATION_ANGLE.__doc__ = "Angle of the shortest rotation between the two specified axes."
AngleType.TO_PLANE.__doc__ = "An angle between a vector and a plane."
AngleType.TEMPLATE.__doc__ = "Represents a VGT angle created from a template. This type of angle is not creatable."

agcls.AgTypeNameMap["AngleType"] = AngleType

class AxesType(IntEnum):
    """Represents vector types."""

    UNKNOWN = -1
    """Unknown or unsupported type."""
    LAGRANGE_LIBRATION = 0
    """Libration point axes using one primary and multiple secondary central bodies. Set primary and secondary bodies, and point type."""
    ANGULAR_OFFSET = 1
    """Axes created by rotating the Reference axes about the Spin vector through the specified rotation angle plus the additional rotational offset."""
    FIXED_AT_EPOCH = 2
    """Axes based on another set fixed at a specified epoch."""
    B_PLANE = 3
    """B-Plane axes using the selected target body and reference vector."""
    CUSTOM_SCRIPT = 4
    """Customized axes offset with respect to a set of reference Axes."""
    FIXED = 6
    """Axes fixed in reference axes."""
    ALIGNED_AND_CONSTRAINED = 7
    """Axes aligned using two pairs of vectors. One vector in each pair is fixed in these axes and the other vector serves as an independent reference."""
    MODEL_ATTACHMENT = 8
    """Axes aligned with the specified pointable element of the object's 3D model. The axes follow the model as well as any articulations that affect the specified pointable element."""
    SPINNING = 9
    """Axes created by spinning the Reference axes about the Spin vector with the specified rate. The axes are aligned with the Reference axes at the specified epoch plus the additional rotational offset."""
    ON_SURFACE = 10
    """Projection of the reference point onto the central body."""
    TRAJECTORY = 11
    """Axes based on trajectory of the point relative to the reference coordinate system."""
    TEMPLATE = 12
    """Represents a VGT axes created from a template. This type of axes is not creatable."""
    AT_TIME_INSTANT = 13
    """Axes orientation fixed relative to reference axes based on orientation of another set of axes evaluated at specified time instant."""
    PLUGIN = 14
    """An axes plugin point."""
    FILE = 5
    """Axes specified by data from a file."""

AxesType.UNKNOWN.__doc__ = "Unknown or unsupported type."
AxesType.LAGRANGE_LIBRATION.__doc__ = "Libration point axes using one primary and multiple secondary central bodies. Set primary and secondary bodies, and point type."
AxesType.ANGULAR_OFFSET.__doc__ = "Axes created by rotating the Reference axes about the Spin vector through the specified rotation angle plus the additional rotational offset."
AxesType.FIXED_AT_EPOCH.__doc__ = "Axes based on another set fixed at a specified epoch."
AxesType.B_PLANE.__doc__ = "B-Plane axes using the selected target body and reference vector."
AxesType.CUSTOM_SCRIPT.__doc__ = "Customized axes offset with respect to a set of reference Axes."
AxesType.FIXED.__doc__ = "Axes fixed in reference axes."
AxesType.ALIGNED_AND_CONSTRAINED.__doc__ = "Axes aligned using two pairs of vectors. One vector in each pair is fixed in these axes and the other vector serves as an independent reference."
AxesType.MODEL_ATTACHMENT.__doc__ = "Axes aligned with the specified pointable element of the object's 3D model. The axes follow the model as well as any articulations that affect the specified pointable element."
AxesType.SPINNING.__doc__ = "Axes created by spinning the Reference axes about the Spin vector with the specified rate. The axes are aligned with the Reference axes at the specified epoch plus the additional rotational offset."
AxesType.ON_SURFACE.__doc__ = "Projection of the reference point onto the central body."
AxesType.TRAJECTORY.__doc__ = "Axes based on trajectory of the point relative to the reference coordinate system."
AxesType.TEMPLATE.__doc__ = "Represents a VGT axes created from a template. This type of axes is not creatable."
AxesType.AT_TIME_INSTANT.__doc__ = "Axes orientation fixed relative to reference axes based on orientation of another set of axes evaluated at specified time instant."
AxesType.PLUGIN.__doc__ = "An axes plugin point."
AxesType.FILE.__doc__ = "Axes specified by data from a file."

agcls.AgTypeNameMap["AxesType"] = AxesType

class PlaneType(IntEnum):
    """Represents plane types."""

    UNKNOWN = -1
    """Unknown or unsupported type."""
    NORMAL = 0
    """A plane normal to a vector at a given point."""
    QUADRANT = 1
    """A plane is defined by the quadrant from a Reference System (e.g., XY, XZ, YZ, YX, ZX, ZY). The reference point in all cases is the origin of the coordinate system."""
    TRAJECTORY = 2
    """A plane is defined on the basis of a trajectory of a selected point with respect to a reference point."""
    TRIAD = 3
    """A plane is defined by the three points."""
    TEMPLATE = 4
    """Represents a VGT plane created from a template. This type of plane is not creatable."""
    TWO_VECTOR = 5
    """A plane passing through point and containing two given vectors."""

PlaneType.UNKNOWN.__doc__ = "Unknown or unsupported type."
PlaneType.NORMAL.__doc__ = "A plane normal to a vector at a given point."
PlaneType.QUADRANT.__doc__ = "A plane is defined by the quadrant from a Reference System (e.g., XY, XZ, YZ, YX, ZX, ZY). The reference point in all cases is the origin of the coordinate system."
PlaneType.TRAJECTORY.__doc__ = "A plane is defined on the basis of a trajectory of a selected point with respect to a reference point."
PlaneType.TRIAD.__doc__ = "A plane is defined by the three points."
PlaneType.TEMPLATE.__doc__ = "Represents a VGT plane created from a template. This type of plane is not creatable."
PlaneType.TWO_VECTOR.__doc__ = "A plane passing through point and containing two given vectors."

agcls.AgTypeNameMap["PlaneType"] = PlaneType

class PointType(IntEnum):
    """Represents point types."""

    UNKNOWN = -1
    """Unknown or unsupported type."""
    B_PLANE = 0
    """B-Plane point using the selected target body."""
    GRAZING = 1
    """The grazing point is the point of closest approach to the surface of the selected central body along a defined direction."""
    COVARIANCE_GRAZING = 2
    """The point of closest approach to the surface of the specified position covariance ellipsoid surface along a defined direction. Position covariance must be available for a vehicle object to be considered a possible target for this option."""
    FIXED_IN_SYSTEM = 4
    """Point fixed in a reference coordinate system."""
    GLINT = 5
    """Point on central body surface that reflects from source to observer."""
    PLANE_INTERSECTION = 6
    """Point on a plane located along a given direction looking from a given origin."""
    MODEL_ATTACHMENT = 7
    """Point placed at the specified attachment point of the object's 3D model. The point follows the model as well as any articulations that affect the specified attachment point."""
    PLANE_PROJECTION = 8
    """The projection of a point onto a reference plane."""
    ON_SURFACE = 9
    """The detic subpoint of the reference point as projected onto the central body."""
    LAGRANGE_LIBRATION = 10
    """Libration point using one primary and multiple secondary central bodies."""
    TEMPLATE = 11
    """Represents a VGT point created from a template. This type of point is not creatable."""
    CENTRAL_BODY_INTERSECTION = 12
    """Point on central body surface along direction vector originating at source point."""
    AT_TIME_INSTANT = 13
    """Point fixed relative to reference system based on another point evaluated at specified time instant."""
    PLUGIN = 14
    """A point plugin point."""
    FILE = 3
    """Point specified by data from a file."""
    FIXED_ON_CENTRAL_BODY = 15
    """Point fixed on a central body."""
    SATELLITE_COLLECTION_ENTRY = 16
    """A point placed at the center of mass of a specified satellite of the satellite collection."""

PointType.UNKNOWN.__doc__ = "Unknown or unsupported type."
PointType.B_PLANE.__doc__ = "B-Plane point using the selected target body."
PointType.GRAZING.__doc__ = "The grazing point is the point of closest approach to the surface of the selected central body along a defined direction."
PointType.COVARIANCE_GRAZING.__doc__ = "The point of closest approach to the surface of the specified position covariance ellipsoid surface along a defined direction. Position covariance must be available for a vehicle object to be considered a possible target for this option."
PointType.FIXED_IN_SYSTEM.__doc__ = "Point fixed in a reference coordinate system."
PointType.GLINT.__doc__ = "Point on central body surface that reflects from source to observer."
PointType.PLANE_INTERSECTION.__doc__ = "Point on a plane located along a given direction looking from a given origin."
PointType.MODEL_ATTACHMENT.__doc__ = "Point placed at the specified attachment point of the object's 3D model. The point follows the model as well as any articulations that affect the specified attachment point."
PointType.PLANE_PROJECTION.__doc__ = "The projection of a point onto a reference plane."
PointType.ON_SURFACE.__doc__ = "The detic subpoint of the reference point as projected onto the central body."
PointType.LAGRANGE_LIBRATION.__doc__ = "Libration point using one primary and multiple secondary central bodies."
PointType.TEMPLATE.__doc__ = "Represents a VGT point created from a template. This type of point is not creatable."
PointType.CENTRAL_BODY_INTERSECTION.__doc__ = "Point on central body surface along direction vector originating at source point."
PointType.AT_TIME_INSTANT.__doc__ = "Point fixed relative to reference system based on another point evaluated at specified time instant."
PointType.PLUGIN.__doc__ = "A point plugin point."
PointType.FILE.__doc__ = "Point specified by data from a file."
PointType.FIXED_ON_CENTRAL_BODY.__doc__ = "Point fixed on a central body."
PointType.SATELLITE_COLLECTION_ENTRY.__doc__ = "A point placed at the center of mass of a specified satellite of the satellite collection."

agcls.AgTypeNameMap["PointType"] = PointType

class SystemType(IntEnum):
    """Represents system types."""

    UNKNOWN = -1
    """Unknown or unsupported system type."""
    ASSEMBLED = 0
    """A system assembled from an origin point and a set of reference axes."""
    ON_SURFACE = 1
    """A system with an origin on the surface of the central body with topocentric axes rotated on a clock angle."""
    TEMPLATE = 2
    """Represents a VGT system created from a template. This type of system is not creatable."""

SystemType.UNKNOWN.__doc__ = "Unknown or unsupported system type."
SystemType.ASSEMBLED.__doc__ = "A system assembled from an origin point and a set of reference axes."
SystemType.ON_SURFACE.__doc__ = "A system with an origin on the surface of the central body with topocentric axes rotated on a clock angle."
SystemType.TEMPLATE.__doc__ = "Represents a VGT system created from a template. This type of system is not creatable."

agcls.AgTypeNameMap["SystemType"] = SystemType

class VectorType(IntEnum):
    """Represents vector types."""

    UNKNOWN = -1
    """Unknown or unsupported vector type."""
    DISPLACEMENT = 0
    """Vector defined by its start and end points."""
    APOAPSIS = 1
    """Vector from the center of the specified central body to the farthest point of an elliptical orbit created from the motion of the specified point."""
    FIXED_AT_EPOCH = 2
    """Based on another vector fixed at a specified epoch."""
    ANGULAR_VELOCITY = 3
    """Angular velocity vector of one set of axes computed with respect to the reference set."""
    CONING = 4
    """Vector created by revolving the Reference vector around the About vector with the specified rate. The vector is aligned with Reference vector at specified epoch. After that it revolves between start/stop angles using either uni- or bi-directional mode."""
    CROSS_PRODUCT = 5
    """The vector cross product of two vectors."""
    CUSTOM_SCRIPT = 6
    """Customized vector components defined with respect to reference axes."""
    DERIVATIVE = 7
    """Derivative of a vector computed with respect to specified axes."""
    ANGLE_RATE = 8
    """Angle rate vector perpendicular to the plane in which the angle is defined."""
    ECCENTRICITY = 9
    """Vector directed from the center of the specified central body toward the nearest point of an elliptical orbit created from the motion of the specified point."""
    FIXED_IN_AXES = 10
    """Vector fixed in reference axes."""
    TWO_PLANES_INTERSECTION = 12
    """Defined along the intersection of two planes."""
    LINE_OF_NODES = 13
    """Unit vector along the line of nodes - the line of intersection of the osculating orbit plane and the inertial equator of the specified central body."""
    MODEL_ATTACHMENT = 14
    """Unit vector along the specified pointable element of the object's 3D model. The vector's direction follows the model as well as any articulations that affect the specified pointable element."""
    ORBIT_ANGULAR_MOMENTUM = 15
    """Vector perpendicular to the plane of an elliptical orbit created from the motion of the specified point with respect to the center of the specified central body."""
    ORBIT_NORMAL = 16
    """Unit vector perpendicular to the plane of an elliptical orbit created from the motion of the specified point with respect to the center of the specified central body."""
    PERIAPSIS = 17
    """Vector from the center of the specified central body to the nearest point of an elliptical orbit created from the motion of the specified point."""
    PROJECTION = 18
    """A projection of a vector computed with respect to a reference plane."""
    REFLECTION = 19
    """Incident vector reflected using a plane whose normal is the normal vector, scaled by a factor. The selected vector or its opposite can be reflected on just one or on both sides of the plane."""
    SCALED = 20
    """Scaled version of the input vector."""
    DIRECTION_TO_STAR = 21
    """Defined with respect to a star object."""
    TEMPLATE = 22
    """Represents a VGT vector created from a template. This type of vector is not creatable."""
    AT_TIME_INSTANT = 23
    """Vector fixed relative to reference axes based on another vector evaluated at specified time instant."""
    LINEAR_COMBINATION = 24
    """Linear combination of two input vectors."""
    PROJECT_ALONG = 25
    """A projection of a source vector in the direction of another vector."""
    SCALAR_LINEAR_COMBINATION = 26
    """Linear combination of two input vectors using scalars."""
    SCALAR_SCALED = 27
    """Scaled version of the input vector using scalar."""
    VELOCITY = 28
    """Velocity vector of a point in a coordinate system."""
    PLUGIN = 29
    """A vector plugin point."""
    ROTATION_VECTOR = 30
    """Rotation vector representing the rotation of one axes relative to reference axes, expressed as angle*rotationAxis."""
    DISPLACEMENT_ON_SURFACE = 31
    """Displacement between origin and destination points using surface distance and altitude difference."""
    FILE = 32
    """Vector interpolated from tabulated data from file."""

VectorType.UNKNOWN.__doc__ = "Unknown or unsupported vector type."
VectorType.DISPLACEMENT.__doc__ = "Vector defined by its start and end points."
VectorType.APOAPSIS.__doc__ = "Vector from the center of the specified central body to the farthest point of an elliptical orbit created from the motion of the specified point."
VectorType.FIXED_AT_EPOCH.__doc__ = "Based on another vector fixed at a specified epoch."
VectorType.ANGULAR_VELOCITY.__doc__ = "Angular velocity vector of one set of axes computed with respect to the reference set."
VectorType.CONING.__doc__ = "Vector created by revolving the Reference vector around the About vector with the specified rate. The vector is aligned with Reference vector at specified epoch. After that it revolves between start/stop angles using either uni- or bi-directional mode."
VectorType.CROSS_PRODUCT.__doc__ = "The vector cross product of two vectors."
VectorType.CUSTOM_SCRIPT.__doc__ = "Customized vector components defined with respect to reference axes."
VectorType.DERIVATIVE.__doc__ = "Derivative of a vector computed with respect to specified axes."
VectorType.ANGLE_RATE.__doc__ = "Angle rate vector perpendicular to the plane in which the angle is defined."
VectorType.ECCENTRICITY.__doc__ = "Vector directed from the center of the specified central body toward the nearest point of an elliptical orbit created from the motion of the specified point."
VectorType.FIXED_IN_AXES.__doc__ = "Vector fixed in reference axes."
VectorType.TWO_PLANES_INTERSECTION.__doc__ = "Defined along the intersection of two planes."
VectorType.LINE_OF_NODES.__doc__ = "Unit vector along the line of nodes - the line of intersection of the osculating orbit plane and the inertial equator of the specified central body."
VectorType.MODEL_ATTACHMENT.__doc__ = "Unit vector along the specified pointable element of the object's 3D model. The vector's direction follows the model as well as any articulations that affect the specified pointable element."
VectorType.ORBIT_ANGULAR_MOMENTUM.__doc__ = "Vector perpendicular to the plane of an elliptical orbit created from the motion of the specified point with respect to the center of the specified central body."
VectorType.ORBIT_NORMAL.__doc__ = "Unit vector perpendicular to the plane of an elliptical orbit created from the motion of the specified point with respect to the center of the specified central body."
VectorType.PERIAPSIS.__doc__ = "Vector from the center of the specified central body to the nearest point of an elliptical orbit created from the motion of the specified point."
VectorType.PROJECTION.__doc__ = "A projection of a vector computed with respect to a reference plane."
VectorType.REFLECTION.__doc__ = "Incident vector reflected using a plane whose normal is the normal vector, scaled by a factor. The selected vector or its opposite can be reflected on just one or on both sides of the plane."
VectorType.SCALED.__doc__ = "Scaled version of the input vector."
VectorType.DIRECTION_TO_STAR.__doc__ = "Defined with respect to a star object."
VectorType.TEMPLATE.__doc__ = "Represents a VGT vector created from a template. This type of vector is not creatable."
VectorType.AT_TIME_INSTANT.__doc__ = "Vector fixed relative to reference axes based on another vector evaluated at specified time instant."
VectorType.LINEAR_COMBINATION.__doc__ = "Linear combination of two input vectors."
VectorType.PROJECT_ALONG.__doc__ = "A projection of a source vector in the direction of another vector."
VectorType.SCALAR_LINEAR_COMBINATION.__doc__ = "Linear combination of two input vectors using scalars."
VectorType.SCALAR_SCALED.__doc__ = "Scaled version of the input vector using scalar."
VectorType.VELOCITY.__doc__ = "Velocity vector of a point in a coordinate system."
VectorType.PLUGIN.__doc__ = "A vector plugin point."
VectorType.ROTATION_VECTOR.__doc__ = "Rotation vector representing the rotation of one axes relative to reference axes, expressed as angle*rotationAxis."
VectorType.DISPLACEMENT_ON_SURFACE.__doc__ = "Displacement between origin and destination points using surface distance and altitude difference."
VectorType.FILE.__doc__ = "Vector interpolated from tabulated data from file."

agcls.AgTypeNameMap["VectorType"] = VectorType

class MeanElementTheory(IntEnum):
    """Mean element theory types for approximating motion."""

    OSCULATING_ELEMENTS = 1
    """Osculating elements (six standard Keplerian orbital elements)."""
    KOZAI = 2
    """The Kozai-Iszak (KI) mean elements are based upon the paper \"The Motion of a Close earth satellite,\" Y. Kozai, The Astronomical Journal, Nov 1959, pp.367-377."""
    BROUWER_LYDDANE = 3
    """Refers to the BL mean elements considering both the short and long period terms (resulting from averaging over the rotation of periapse). The perturbation terms are the J2, J3, J4 and J5 oblateness terms and it considers the term involving J2^2."""
    BROUWER_LYDDANE_SHORT_PERIODIC_TERMS_ONLY = 4
    """Refers to the BL mean elements considering only the short period terms (i.e. those involving averaging over the period of the orbit) where the only perturbation force is the oblateness arising from the J2 gravity term."""

MeanElementTheory.OSCULATING_ELEMENTS.__doc__ = "Osculating elements (six standard Keplerian orbital elements)."
MeanElementTheory.KOZAI.__doc__ = "The Kozai-Iszak (KI) mean elements are based upon the paper ``The Motion of a Close earth satellite,`` Y. Kozai, The Astronomical Journal, Nov 1959, pp.367-377."
MeanElementTheory.BROUWER_LYDDANE.__doc__ = "Refers to the BL mean elements considering both the short and long period terms (resulting from averaging over the rotation of periapse). The perturbation terms are the J2, J3, J4 and J5 oblateness terms and it considers the term involving J2^2."
MeanElementTheory.BROUWER_LYDDANE_SHORT_PERIODIC_TERMS_ONLY.__doc__ = "Refers to the BL mean elements considering only the short period terms (i.e. those involving averaging over the period of the orbit) where the only perturbation force is the oblateness arising from the J2 gravity term."

agcls.AgTypeNameMap["MeanElementTheory"] = MeanElementTheory

class AsymptoteDirectionType(IntEnum):
    """Direction options."""

    INCOMING = 1
    """Incoming direction."""
    OUTGOING = 2
    """Outgoing direction."""

AsymptoteDirectionType.INCOMING.__doc__ = "Incoming direction."
AsymptoteDirectionType.OUTGOING.__doc__ = "Outgoing direction."

agcls.AgTypeNameMap["AsymptoteDirectionType"] = AsymptoteDirectionType

class LagrangeLibrationPointType(IntEnum):
    """Types of the Lagrange points, also known as libration points. Lagrange points are points in space where gravitational forces and the orbital motion of a body balance each other."""

    L1 = 1
    """A point between the Sun and Earth."""
    L2 = 2
    """Similar to L2, but on the night side of Earth, further away from the Sun, but about the same distane from Earth."""
    L3 = 3
    """L3 lies on a line defined by the Sun and Earth, on the opposite side of the Sun, just outside the orbit of Earth. L3 remains hidden behind the Sun at all times."""
    L4 = 4
    """The L4 and L5 points lie at 60 degrees ahead of and behind Earth in its orbit as seen from the Sun. L4 and L5 are \"stable\" points."""
    L5 = 5
    """The L4 and L5 points lie at 60 degrees ahead of and behind Earth in its orbit as seen from the Sun. L4 and L5 are \"stable\" points."""

LagrangeLibrationPointType.L1.__doc__ = "A point between the Sun and Earth."
LagrangeLibrationPointType.L2.__doc__ = "Similar to L2, but on the night side of Earth, further away from the Sun, but about the same distane from Earth."
LagrangeLibrationPointType.L3.__doc__ = "L3 lies on a line defined by the Sun and Earth, on the opposite side of the Sun, just outside the orbit of Earth. L3 remains hidden behind the Sun at all times."
LagrangeLibrationPointType.L4.__doc__ = "The L4 and L5 points lie at 60 degrees ahead of and behind Earth in its orbit as seen from the Sun. L4 and L5 are ``stable`` points."
LagrangeLibrationPointType.L5.__doc__ = "The L4 and L5 points lie at 60 degrees ahead of and behind Earth in its orbit as seen from the Sun. L4 and L5 are ``stable`` points."

agcls.AgTypeNameMap["LagrangeLibrationPointType"] = LagrangeLibrationPointType

class PlaneQuadrantType(IntEnum):
    """Quadrants from a reference system (e.g., XY, XZ, YZ, YX, ZX, ZY)."""

    XY = 1
    """XY quadrant."""
    YX = 2
    """YX quadrant."""
    XZ = 3
    """XZ quadrant."""
    ZX = 4
    """ZX quadrant."""
    YZ = 5
    """YZ quadrant."""
    ZY = 6
    """ZY quadrant."""

PlaneQuadrantType.XY.__doc__ = "XY quadrant."
PlaneQuadrantType.YX.__doc__ = "YX quadrant."
PlaneQuadrantType.XZ.__doc__ = "XZ quadrant."
PlaneQuadrantType.ZX.__doc__ = "ZX quadrant."
PlaneQuadrantType.YZ.__doc__ = "YZ quadrant."
PlaneQuadrantType.ZY.__doc__ = "ZY quadrant."

agcls.AgTypeNameMap["PlaneQuadrantType"] = PlaneQuadrantType

class TrajectoryAxesCoordinatesType(IntEnum):
    """Trajectory axes coordinate types."""

    ICR = 0
    """Intrack Crosstrack Radial Axes. The Z axis is outward along the position vector (radial); the Y axis is along the cross product of the position and velocity (crosstrack); the X axis is in the direction of motion and constructed as Y x Z (intrack)."""
    VNC = 1
    """Velocity - Normal - Co-normal Axes. The X axis is along the velocity vector; the Y axis is along the cross product of the position and velocity (normal); the Z axis is constructed as X x Y (co-normal)."""
    RIC = 2
    """Radial Intrack Crosstrack Axes. The X axis is outward along the position vector (radial); the Z axis is along the cross product of the position and velocity (crosstrack); the Y axis is in the direction of motion and is constructed as Z x X (intrack)."""
    LVLH = 3
    """Local Vertical, Local Horizontal Axes. The X axis is along the position vector (local vertical); the Z axis is along the cross product of the position and velocity; the Y axis is in the direction of motion and constructed as Z x X (local horizontal)."""
    VVLH = 4
    """Vehicle Velocity, Local Horizontal Axes. The Z axis is along the negative position vector; the Y axis is along the negative cross product of the position and velocity (local horizontal); the X axis is constructed as Z x Y (toward velocity)."""
    BODY_BODY_ROTATING = 5
    """Body-to-body Rotating Axes. The X axis is along the negative position vector; the Z axis is along the cross product of the position and velocity; the Y axis is constructed as Z x X."""
    EQUINOCTIAL = 6
    """Equinoctial Axes. The Z axis is along the orbit normal; the X axis is along the fiducial direction located by rotating about Z-axis by negative of RAAN value; the Y axis is constructed as Z x X."""
    NTC = 7
    """Normal - Tangential - Crosstrack Axes. The Y axis is along the velocity vector (tangential); the Z axis is along the cross product of the position and velocity (crosstrack); the X axis is constructed as Y x Z (normal)."""

TrajectoryAxesCoordinatesType.ICR.__doc__ = "Intrack Crosstrack Radial Axes. The Z axis is outward along the position vector (radial); the Y axis is along the cross product of the position and velocity (crosstrack); the X axis is in the direction of motion and constructed as Y x Z (intrack)."
TrajectoryAxesCoordinatesType.VNC.__doc__ = "Velocity - Normal - Co-normal Axes. The X axis is along the velocity vector; the Y axis is along the cross product of the position and velocity (normal); the Z axis is constructed as X x Y (co-normal)."
TrajectoryAxesCoordinatesType.RIC.__doc__ = "Radial Intrack Crosstrack Axes. The X axis is outward along the position vector (radial); the Z axis is along the cross product of the position and velocity (crosstrack); the Y axis is in the direction of motion and is constructed as Z x X (intrack)."
TrajectoryAxesCoordinatesType.LVLH.__doc__ = "Local Vertical, Local Horizontal Axes. The X axis is along the position vector (local vertical); the Z axis is along the cross product of the position and velocity; the Y axis is in the direction of motion and constructed as Z x X (local horizontal)."
TrajectoryAxesCoordinatesType.VVLH.__doc__ = "Vehicle Velocity, Local Horizontal Axes. The Z axis is along the negative position vector; the Y axis is along the negative cross product of the position and velocity (local horizontal); the X axis is constructed as Z x Y (toward velocity)."
TrajectoryAxesCoordinatesType.BODY_BODY_ROTATING.__doc__ = "Body-to-body Rotating Axes. The X axis is along the negative position vector; the Z axis is along the cross product of the position and velocity; the Y axis is constructed as Z x X."
TrajectoryAxesCoordinatesType.EQUINOCTIAL.__doc__ = "Equinoctial Axes. The Z axis is along the orbit normal; the X axis is along the fiducial direction located by rotating about Z-axis by negative of RAAN value; the Y axis is constructed as Z x X."
TrajectoryAxesCoordinatesType.NTC.__doc__ = "Normal - Tangential - Crosstrack Axes. The Y axis is along the velocity vector (tangential); the Z axis is along the cross product of the position and velocity (crosstrack); the X axis is constructed as Y x Z (normal)."

agcls.AgTypeNameMap["TrajectoryAxesCoordinatesType"] = TrajectoryAxesCoordinatesType

class PrincipalAxisOfRotationType(IntEnum):
    """Rotation directions."""

    X = 0
    """Rotate about Axis X."""
    Y = 1
    """Rotate about Axis Y."""
    Z = 2
    """Rotate about Axis Z."""

PrincipalAxisOfRotationType.X.__doc__ = "Rotate about Axis X."
PrincipalAxisOfRotationType.Y.__doc__ = "Rotate about Axis Y."
PrincipalAxisOfRotationType.Z.__doc__ = "Rotate about Axis Z."

agcls.AgTypeNameMap["PrincipalAxisOfRotationType"] = PrincipalAxisOfRotationType

class SignedAngleType(IntEnum):
    """Define options for computing an angle."""

    NONE = 0
    """Choose the option to use unsigned angle."""
    POSITIVE = 1
    """Choose the option to measure angles as positive when the reference Vector is directed toward the plane's normal."""
    NEGATIVE = 2
    """Choose the option to measure angles as negative when the reference Vector is directed toward the plane's normal."""

SignedAngleType.NONE.__doc__ = "Choose the option to use unsigned angle."
SignedAngleType.POSITIVE.__doc__ = "Choose the option to measure angles as positive when the reference Vector is directed toward the plane's normal."
SignedAngleType.NEGATIVE.__doc__ = "Choose the option to measure angles as negative when the reference Vector is directed toward the plane's normal."

agcls.AgTypeNameMap["SignedAngleType"] = SignedAngleType

class PointBPlaneType(IntEnum):
    """B-Plane point types."""

    ASYMPTOTE = 1
    """Asymptote."""
    TWO_BODY = 2
    """Two body."""

PointBPlaneType.ASYMPTOTE.__doc__ = "Asymptote."
PointBPlaneType.TWO_BODY.__doc__ = "Two body."

agcls.AgTypeNameMap["PointBPlaneType"] = PointBPlaneType

class SurfaceReferenceShapeType(IntEnum):
    """Surface shape types."""

    ELLIPSOID = 1
    """An ellipsoid reference shape as defined by the central body (by default, it is WSG84)."""
    TERRAIN = 2
    """Terrain as the terrain reference."""
    MSL = 3
    """Mean Sea Level as the terrain reference."""

SurfaceReferenceShapeType.ELLIPSOID.__doc__ = "An ellipsoid reference shape as defined by the central body (by default, it is WSG84)."
SurfaceReferenceShapeType.TERRAIN.__doc__ = "Terrain as the terrain reference."
SurfaceReferenceShapeType.MSL.__doc__ = "Mean Sea Level as the terrain reference."

agcls.AgTypeNameMap["SurfaceReferenceShapeType"] = SurfaceReferenceShapeType

class SurfaceShapeType(IntEnum):
    """Surface types."""

    DETIC = 1
    """Detic surface model."""
    CENTRIC = 2
    """Centric surface model."""

SurfaceShapeType.DETIC.__doc__ = "Detic surface model."
SurfaceShapeType.CENTRIC.__doc__ = "Centric surface model."

agcls.AgTypeNameMap["SurfaceShapeType"] = SurfaceShapeType

class RotationSweepModeType(IntEnum):
    """The rotation sweeping modes."""

    BIDIRECTIONAL = 1
    """Bidirectional sweeping mode."""
    UNIDIRECTIONAL = 2
    """Unidirectional sweeping mode."""

RotationSweepModeType.BIDIRECTIONAL.__doc__ = "Bidirectional sweeping mode."
RotationSweepModeType.UNIDIRECTIONAL.__doc__ = "Unidirectional sweeping mode."

agcls.AgTypeNameMap["RotationSweepModeType"] = RotationSweepModeType

class SignalDirectionType(IntEnum):
    """Signal sense transmission options."""

    RECEIVE = 1
    """Signal receive."""
    TRANSMIT = 2
    """Signal transmit."""

SignalDirectionType.RECEIVE.__doc__ = "Signal receive."
SignalDirectionType.TRANSMIT.__doc__ = "Signal transmit."

agcls.AgTypeNameMap["SignalDirectionType"] = SignalDirectionType

class IntersectionSurfaceType(IntEnum):
    """Intersection surface flags."""

    ON_CENTRAL_BODY_ELLIPSOID = 0
    """Intersection with central body ellipsoid."""
    AT_ALTITUDE_ABOVE_ELLIPSOID = 1
    """Intersection at altitude."""
    AT_TERRAIN = 2
    """Use terrain as intersection surface."""

IntersectionSurfaceType.ON_CENTRAL_BODY_ELLIPSOID.__doc__ = "Intersection with central body ellipsoid."
IntersectionSurfaceType.AT_ALTITUDE_ABOVE_ELLIPSOID.__doc__ = "Intersection at altitude."
IntersectionSurfaceType.AT_TERRAIN.__doc__ = "Use terrain as intersection surface."

agcls.AgTypeNameMap["IntersectionSurfaceType"] = IntersectionSurfaceType

class VectorGeometryToolScaledVectorDimensionInheritanceOptionType(IntEnum):
    """Dimension inheritance constants used to configure the dimension inheritance of a vector scaled by a scalar."""

    NONE = 0
    """Do not inherit dimension."""
    FROM_CALCULATION_SCALAR = 1
    """Inherit dimension from scalar."""
    FROM_VECTOR = 2
    """Inherit dimension from vector."""

VectorGeometryToolScaledVectorDimensionInheritanceOptionType.NONE.__doc__ = "Do not inherit dimension."
VectorGeometryToolScaledVectorDimensionInheritanceOptionType.FROM_CALCULATION_SCALAR.__doc__ = "Inherit dimension from scalar."
VectorGeometryToolScaledVectorDimensionInheritanceOptionType.FROM_VECTOR.__doc__ = "Inherit dimension from vector."

agcls.AgTypeNameMap["VectorGeometryToolScaledVectorDimensionInheritanceOptionType"] = VectorGeometryToolScaledVectorDimensionInheritanceOptionType


class IVectorGeometryToolPoint(object):
    """The interface defines methods and properties common to all points."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _locate_in_system_with_rate_method_offset = 2
    _locate_in_system_method_offset = 3
    _metadata = {
        "iid_data" : (4624590286464293543, 3973676715389699987),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolPoint."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolPoint)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolPoint)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolPoint, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PointType),) }
    @property
    def type(self) -> "PointType":
        """Return a type of the point object."""
        return self._intf.get_property(IVectorGeometryToolPoint._metadata, IVectorGeometryToolPoint._get_type_metadata)

    _locate_in_system_with_rate_metadata = { "offset" : _locate_in_system_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceOutArg,) }
    def locate_in_system_with_rate(self, epoch:typing.Any, system:"IVectorGeometryToolSystem") -> "AnalysisWorkbenchPointLocateInSystemWithRateResult":
        """Locates the point's position and velocity in a specified coordinate system."""
        return self._intf.invoke(IVectorGeometryToolPoint._metadata, IVectorGeometryToolPoint._locate_in_system_with_rate_metadata, epoch, system, OutArg())

    _locate_in_system_metadata = { "offset" : _locate_in_system_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceOutArg,) }
    def locate_in_system(self, epoch:typing.Any, system:"IVectorGeometryToolSystem") -> "AnalysisWorkbenchPointLocateInSystemResult":
        """Locates the point's position in a specified coordinate system."""
        return self._intf.invoke(IVectorGeometryToolPoint._metadata, IVectorGeometryToolPoint._locate_in_system_metadata, epoch, system, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((4624590286464293543, 3973676715389699987), IVectorGeometryToolPoint)
agcls.AgTypeNameMap["IVectorGeometryToolPoint"] = IVectorGeometryToolPoint

class IVectorGeometryToolVector(object):
    """The interface defines methods and properties common to all vectors."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_in_axes_method_offset = 2
    _find_in_axes_with_rate_method_offset = 3
    _metadata = {
        "iid_data" : (4994471793862230398, 8121306839707329686),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolVector."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolVector)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolVector)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolVector, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VectorType),) }
    @property
    def type(self) -> "VectorType":
        """Return a type of the vector object."""
        return self._intf.get_property(IVectorGeometryToolVector._metadata, IVectorGeometryToolVector._get_type_metadata)

    _find_in_axes_metadata = { "offset" : _find_in_axes_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchVectorFindInAxesResult":
        """Compute the vector in the specified axes."""
        return self._intf.invoke(IVectorGeometryToolVector._metadata, IVectorGeometryToolVector._find_in_axes_metadata, epoch, axes, OutArg())

    _find_in_axes_with_rate_metadata = { "offset" : _find_in_axes_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes_with_rate(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchVectorFindInAxesWithRateResult":
        """Compute the vector and its rate in the specified axes."""
        return self._intf.invoke(IVectorGeometryToolVector._metadata, IVectorGeometryToolVector._find_in_axes_with_rate_metadata, epoch, axes, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((4994471793862230398, 8121306839707329686), IVectorGeometryToolVector)
agcls.AgTypeNameMap["IVectorGeometryToolVector"] = IVectorGeometryToolVector

class IVectorGeometryToolSystem(object):
    """The interface contains methods and properties shared by all VGT systems."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_in_system_method_offset = 2
    _transform_method_offset = 3
    _transform_with_rate_method_offset = 4
    _metadata = {
        "iid_data" : (4788837751689081889, 7300266131152037277),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolSystem."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolSystem)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolSystem)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolSystem, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SystemType),) }
    @property
    def type(self) -> "SystemType":
        """Return a type of the system object."""
        return self._intf.get_property(IVectorGeometryToolSystem._metadata, IVectorGeometryToolSystem._get_type_metadata)

    _find_in_system_metadata = { "offset" : _find_in_system_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceOutArg,) }
    def find_in_system(self, epoch:typing.Any, system:"IVectorGeometryToolSystem") -> "AnalysisWorkbenchSystemFindInSystemResult":
        """Find position, velocity, rate and orientation using the specified system."""
        return self._intf.invoke(IVectorGeometryToolSystem._metadata, IVectorGeometryToolSystem._find_in_system_metadata, epoch, system, OutArg())

    _transform_metadata = { "offset" : _transform_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceOutArg,) }
    def transform(self, epoch:typing.Any, output_system:"IVectorGeometryToolSystem", position_in_my_system:"ICartesian3Vector") -> "AnalysisWorkbenchSystemTransformResult":
        """Translate the position vector from this system into the output system."""
        return self._intf.invoke(IVectorGeometryToolSystem._metadata, IVectorGeometryToolSystem._transform_metadata, epoch, output_system, position_in_my_system, OutArg())

    _transform_with_rate_metadata = { "offset" : _transform_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, agcom.PVOID, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceOutArg,) }
    def transform_with_rate(self, epoch:typing.Any, output_system:"IVectorGeometryToolSystem", position_in_my_system:"ICartesian3Vector", velocity_in_my_system:"ICartesian3Vector") -> "AnalysisWorkbenchSystemTransformWithRateResult":
        """Translate the position and rate vectors from this system into the output system."""
        return self._intf.invoke(IVectorGeometryToolSystem._metadata, IVectorGeometryToolSystem._transform_with_rate_metadata, epoch, output_system, position_in_my_system, velocity_in_my_system, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((4788837751689081889, 7300266131152037277), IVectorGeometryToolSystem)
agcls.AgTypeNameMap["IVectorGeometryToolSystem"] = IVectorGeometryToolSystem

class IVectorGeometryToolAxes(object):
    """The interface defines methods and properties common to all axes."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_in_axes_with_rate_method_offset = 2
    _find_in_axes_method_offset = 3
    _get_labels_method_offset = 4
    _get_x_axis_method_offset = 5
    _get_y_axis_method_offset = 6
    _get_z_axis_method_offset = 7
    _transform_method_offset = 8
    _transform_with_rate_method_offset = 9
    _metadata = {
        "iid_data" : (4641684088358424472, 9856172197771500675),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolAxes."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolAxes)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolAxes)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolAxes, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AxesType),) }
    @property
    def type(self) -> "AxesType":
        """Return a type of the axes object."""
        return self._intf.get_property(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._get_type_metadata)

    _find_in_axes_with_rate_metadata = { "offset" : _find_in_axes_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes_with_rate(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchAxesFindInAxesWithRateResult":
        """Find an angular velocity and orientation in the specified axes."""
        return self._intf.invoke(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._find_in_axes_with_rate_metadata, epoch, axes, OutArg())

    _find_in_axes_metadata = { "offset" : _find_in_axes_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchAxesFindInAxesResult":
        """Find an orientation in the specified axes."""
        return self._intf.invoke(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._find_in_axes_metadata, epoch, axes, OutArg())

    _get_labels_metadata = { "offset" : _get_labels_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def labels(self) -> "VectorGeometryToolAxesLabels":
        """Return an object that allows modifying the axes labels."""
        return self._intf.get_property(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._get_labels_metadata)

    _get_x_axis_metadata = { "offset" : _get_x_axis_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def x_axis(self) -> "IVectorGeometryToolVector":
        """Return the X axis of the component."""
        return self._intf.get_property(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._get_x_axis_metadata)

    _get_y_axis_metadata = { "offset" : _get_y_axis_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def y_axis(self) -> "IVectorGeometryToolVector":
        """Return the Y axis of the component."""
        return self._intf.get_property(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._get_y_axis_metadata)

    _get_z_axis_metadata = { "offset" : _get_z_axis_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def z_axis(self) -> "IVectorGeometryToolVector":
        """Return the Z axis of the component."""
        return self._intf.get_property(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._get_z_axis_metadata)

    _transform_metadata = { "offset" : _transform_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceOutArg,) }
    def transform(self, epoch:typing.Any, output_axes:"IVectorGeometryToolAxes", vector_in_my_axes:"ICartesian3Vector") -> "AnalysisWorkbenchAxesTransformResult":
        """Transform the input vector from this axes into the output axes."""
        return self._intf.invoke(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._transform_metadata, epoch, output_axes, vector_in_my_axes, OutArg())

    _transform_with_rate_metadata = { "offset" : _transform_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, agcom.PVOID, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceInArg("ICartesian3Vector"), agmarshall.InterfaceOutArg,) }
    def transform_with_rate(self, epoch:typing.Any, output_axes:"IVectorGeometryToolAxes", vector_in_my_axes:"ICartesian3Vector", rate_in_my_axes:"ICartesian3Vector") -> "AnalysisWorkbenchAxesTransformWithRateResult":
        """Transform the input vector and vector's rate from this axes into the output axes."""
        return self._intf.invoke(IVectorGeometryToolAxes._metadata, IVectorGeometryToolAxes._transform_with_rate_metadata, epoch, output_axes, vector_in_my_axes, rate_in_my_axes, OutArg())

    _property_names[type] = "type"
    _property_names[labels] = "labels"
    _property_names[x_axis] = "x_axis"
    _property_names[y_axis] = "y_axis"
    _property_names[z_axis] = "z_axis"


agcls.AgClassCatalog.add_catalog_entry((4641684088358424472, 9856172197771500675), IVectorGeometryToolAxes)
agcls.AgTypeNameMap["IVectorGeometryToolAxes"] = IVectorGeometryToolAxes

class IVectorGeometryToolAngle(object):
    """The interface defines methods and properties common to all angles."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_angle_method_offset = 2
    _find_angle_with_rate_method_offset = 3
    _find_coordinates_method_offset = 4
    _find_coordinates_with_rate_method_offset = 5
    _metadata = {
        "iid_data" : (5576499434980039356, 3250766286789469325),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolAngle."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolAngle)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolAngle)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolAngle, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AngleType),) }
    @property
    def type(self) -> "AngleType":
        """Return a type of the angle object."""
        return self._intf.get_property(IVectorGeometryToolAngle._metadata, IVectorGeometryToolAngle._get_type_metadata)

    _find_angle_metadata = { "offset" : _find_angle_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def find_angle(self, epoch:typing.Any) -> "AnalysisWorkbenchAngleFindAngleResult":
        """Find an angle at the specified epoch."""
        return self._intf.invoke(IVectorGeometryToolAngle._metadata, IVectorGeometryToolAngle._find_angle_metadata, epoch, OutArg())

    _find_angle_with_rate_metadata = { "offset" : _find_angle_with_rate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def find_angle_with_rate(self, epoch:typing.Any) -> "AnalysisWorkbenchAngleFindAngleWithRateResult":
        """Find an angle and angle rate."""
        return self._intf.invoke(IVectorGeometryToolAngle._metadata, IVectorGeometryToolAngle._find_angle_with_rate_metadata, epoch, OutArg())

    _find_coordinates_metadata = { "offset" : _find_coordinates_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_coordinates(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchAngleFindResult":
        """Find the angle value and three vectors that define the angle in a specified input axes."""
        return self._intf.invoke(IVectorGeometryToolAngle._metadata, IVectorGeometryToolAngle._find_coordinates_metadata, epoch, axes, OutArg())

    _find_coordinates_with_rate_metadata = { "offset" : _find_coordinates_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_coordinates_with_rate(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchAngleFindWithRateResult":
        """Find the angle value, the angle rate and three vectors that define the angle in a specified input axes."""
        return self._intf.invoke(IVectorGeometryToolAngle._metadata, IVectorGeometryToolAngle._find_coordinates_with_rate_metadata, epoch, axes, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((5576499434980039356, 3250766286789469325), IVectorGeometryToolAngle)
agcls.AgTypeNameMap["IVectorGeometryToolAngle"] = IVectorGeometryToolAngle

class IVectorGeometryToolPlane(object):
    """The interface defines methods and properties common to all VGT planes."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_in_axes_method_offset = 2
    _find_in_axes_with_rate_method_offset = 3
    _find_in_system_method_offset = 4
    _find_in_system_with_rate_method_offset = 5
    _get_labels_method_offset = 6
    _metadata = {
        "iid_data" : (4649944266039171678, 14893747184173212307),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IVectorGeometryToolPlane."""
        initialize_from_source_object(self, source_object, IVectorGeometryToolPlane)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IVectorGeometryToolPlane)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IVectorGeometryToolPlane, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(PlaneType),) }
    @property
    def type(self) -> "PlaneType":
        """Return a type of the plane object."""
        return self._intf.get_property(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._get_type_metadata)

    _find_in_axes_metadata = { "offset" : _find_in_axes_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchPlaneFindInAxesResult":
        """Compute the plane's axes vectors in a specified reference axes."""
        return self._intf.invoke(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._find_in_axes_metadata, epoch, axes, OutArg())

    _find_in_axes_with_rate_metadata = { "offset" : _find_in_axes_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolAxes"), agmarshall.InterfaceOutArg,) }
    def find_in_axes_with_rate(self, epoch:typing.Any, axes:"IVectorGeometryToolAxes") -> "AnalysisWorkbenchPlaneFindInAxesWithRateResult":
        """Compute the plane's axes vectors and their rates in a specified reference axes."""
        return self._intf.invoke(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._find_in_axes_with_rate_metadata, epoch, axes, OutArg())

    _find_in_system_metadata = { "offset" : _find_in_system_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceOutArg,) }
    def find_in_system(self, epoch:typing.Any, system:"IVectorGeometryToolSystem") -> "AnalysisWorkbenchPlaneFindInSystemResult":
        """Compute the position and X and Y axes in the specified coordinate system."""
        return self._intf.invoke(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._find_in_system_metadata, epoch, system, OutArg())

    _find_in_system_with_rate_metadata = { "offset" : _find_in_system_with_rate_method_offset,
            "arg_types" : (agcom.Variant, agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceInArg("IVectorGeometryToolSystem"), agmarshall.InterfaceOutArg,) }
    def find_in_system_with_rate(self, epoch:typing.Any, system:"IVectorGeometryToolSystem") -> "AnalysisWorkbenchPlaneFindInSystemWithRateResult":
        """Compute the position, X and Y axes and their rates of change in the specified coordinate system."""
        return self._intf.invoke(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._find_in_system_with_rate_metadata, epoch, system, OutArg())

    _get_labels_metadata = { "offset" : _get_labels_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def labels(self) -> "VectorGeometryToolPlaneLabels":
        """Allow configuring the plane's X and Y axes labels."""
        return self._intf.get_property(IVectorGeometryToolPlane._metadata, IVectorGeometryToolPlane._get_labels_metadata)

    _property_names[type] = "type"
    _property_names[labels] = "labels"


agcls.AgClassCatalog.add_catalog_entry((4649944266039171678, 14893747184173212307), IVectorGeometryToolPlane)
agcls.AgTypeNameMap["IVectorGeometryToolPlane"] = IVectorGeometryToolPlane

class IAnalysisWorkbenchComponentContext(object):
    """The interface represents a context associated with a VGT component. All VGT components are associated with a valid context. A context can represent a VGT instance or a VGT template."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_template_method_offset = 1
    _metadata = {
        "iid_data" : (4867448151487010560, 8329384048399829653),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchComponentContext."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchComponentContext)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchComponentContext)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchComponentContext, None)

    _get_is_template_metadata = { "offset" : _get_is_template_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_template(self) -> bool:
        """Return whether the current instance is a VGT template."""
        return self._intf.get_property(IAnalysisWorkbenchComponentContext._metadata, IAnalysisWorkbenchComponentContext._get_is_template_metadata)

    _property_names[is_template] = "is_template"


agcls.AgClassCatalog.add_catalog_entry((4867448151487010560, 8329384048399829653), IAnalysisWorkbenchComponentContext)
agcls.AgTypeNameMap["IAnalysisWorkbenchComponentContext"] = IAnalysisWorkbenchComponentContext

class IAnalysisWorkbenchComponent(object):
    """A base interface implemented by all VGT components. The methods and properties of the interface provide type information about the VGT component."""

    _num_methods = 19
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_component_type_method_offset = 1
    _get_category_method_offset = 2
    _set_category_method_offset = 3
    _get_name_method_offset = 4
    _get_description_method_offset = 5
    _get_path_method_offset = 6
    _get_is_duplicable_method_offset = 7
    _get_context_method_offset = 8
    _get_type_information_method_offset = 9
    _get_qualified_path_method_offset = 10
    _get_is_valid_method_offset = 11
    _get_is_ready_method_offset = 12
    _get_is_read_only_method_offset = 13
    _duplicate_method_offset = 14
    _anonymous_duplicate_method_offset = 15
    _depends_on_method_offset = 16
    _get_embedded_components_method_offset = 17
    _export_method_offset = 18
    _rename_method_offset = 19
    _metadata = {
        "iid_data" : (5666204506719628304, 9309708717126950559),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchComponent."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchComponent)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchComponent)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchComponent, None)

    _get_component_type_metadata = { "offset" : _get_component_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VectorGeometryToolComponentType),) }
    @property
    def component_type(self) -> "VectorGeometryToolComponentType":
        """Return the component kind."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_component_type_metadata)

    _get_category_metadata = { "offset" : _get_category_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def category(self) -> str:
        """Allow the user to access or change the component category (Folder)."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_category_metadata)

    _set_category_metadata = { "offset" : _set_category_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @category.setter
    def category(self, category:str) -> None:
        return self._intf.set_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._set_category_metadata, category)

    _get_name_metadata = { "offset" : _get_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def name(self) -> str:
        """Return the component name."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_name_metadata)

    _get_description_metadata = { "offset" : _get_description_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def description(self) -> str:
        """Return the component description."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_description_metadata)

    _get_path_metadata = { "offset" : _get_path_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def path(self) -> str:
        """Return the component's fully qualified path (ie. ``CentralBody/Earth Body``, etc.)."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_path_metadata)

    _get_is_duplicable_metadata = { "offset" : _get_is_duplicable_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_duplicable(self) -> bool:
        """Return whether the VGT component can be duplicated."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_is_duplicable_metadata)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return the context object associated with the instance. The returned object is either an instance of AnalysisWorkbenchComponentInstance or IAnalysisWorkbenchComponentTemplate interface."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_context_metadata)

    _get_type_information_metadata = { "offset" : _get_type_information_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def type_information(self) -> "AnalysisWorkbenchComponentTypeInformation":
        """Return the component type information."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_type_information_metadata)

    _get_qualified_path_metadata = { "offset" : _get_qualified_path_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def qualified_path(self) -> str:
        """An STK-conformant path to the VGT component that can be used to visualize the VGT components in 3D (i.e. ``CentralBody/Earth Body Vector``, etc.)."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_qualified_path_metadata)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Return whether the component is valid."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_is_valid_metadata)

    _get_is_ready_metadata = { "offset" : _get_is_ready_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_ready(self) -> bool:
        """Return whether the component is ready. The component is ready if it's been fully initialized."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_is_ready_metadata)

    _get_is_read_only_metadata = { "offset" : _get_is_read_only_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_read_only(self) -> bool:
        """Return whether the component is modifiable."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_is_read_only_metadata)

    _duplicate_metadata = { "offset" : _duplicate_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def duplicate(self, new_name:str, description:str) -> "IAnalysisWorkbenchComponent":
        """Create a copy of the instance of a VGT component. The new component is automatically registered and will be persisted or restored when a scenario is saved or loaded."""
        return self._intf.invoke(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._duplicate_metadata, new_name, description, OutArg())

    _anonymous_duplicate_metadata = { "offset" : _anonymous_duplicate_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def anonymous_duplicate(self) -> "IAnalysisWorkbenchComponent":
        """Create an anonymous copy of the instance of a VGT component. The new component is not registered and will not be persisted nor restored when a scenario is saved or loaded."""
        return self._intf.invoke(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._anonymous_duplicate_metadata, OutArg())

    _depends_on_metadata = { "offset" : _depends_on_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchComponent"), agmarshall.VariantBoolArg,) }
    def depends_on(self, component:"IAnalysisWorkbenchComponent") -> bool:
        """Test if the instance depends on another component."""
        return self._intf.invoke(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._depends_on_metadata, component, OutArg())

    _get_embedded_components_metadata = { "offset" : _get_embedded_components_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def embedded_components(self) -> "AnalysisWorkbenchComponentCollection":
        """Return a collection of embedded components."""
        return self._intf.get_property(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._get_embedded_components_metadata)

    _export_metadata = { "offset" : _export_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def export(self, filename:str, comments:str) -> None:
        """Export the component to a file."""
        return self._intf.invoke(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._export_metadata, filename, comments)

    _rename_metadata = { "offset" : _rename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def rename(self, new_name:str) -> None:
        """Rename the component."""
        return self._intf.invoke(IAnalysisWorkbenchComponent._metadata, IAnalysisWorkbenchComponent._rename_metadata, new_name)

    _property_names[component_type] = "component_type"
    _property_names[category] = "category"
    _property_names[name] = "name"
    _property_names[description] = "description"
    _property_names[path] = "path"
    _property_names[is_duplicable] = "is_duplicable"
    _property_names[context] = "context"
    _property_names[type_information] = "type_information"
    _property_names[qualified_path] = "qualified_path"
    _property_names[is_valid] = "is_valid"
    _property_names[is_ready] = "is_ready"
    _property_names[is_read_only] = "is_read_only"
    _property_names[embedded_components] = "embedded_components"


agcls.AgClassCatalog.add_catalog_entry((5666204506719628304, 9309708717126950559), IAnalysisWorkbenchComponent)
agcls.AgTypeNameMap["IAnalysisWorkbenchComponent"] = IAnalysisWorkbenchComponent

class ICalculationToolScalar(object):
    """Any scalar calculation that is not constant by construction."""

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _evaluate_method_offset = 2
    _quick_evaluate_method_offset = 3
    _evaluate_with_rate_method_offset = 4
    _quick_evaluate_with_rate_method_offset = 5
    _get_availability_method_offset = 6
    _get_unit_of_measure_method_offset = 7
    _quick_evaluate_array_method_offset = 8
    _quick_evaluate_with_rate_array_method_offset = 9
    _quick_evaluate_time_array_method_offset = 10
    _quick_evaluate_with_rate_event_array_method_offset = 11
    _metadata = {
        "iid_data" : (4703695914540684039, 8102226453412927140),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICalculationToolScalar."""
        initialize_from_source_object(self, source_object, ICalculationToolScalar)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICalculationToolScalar)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICalculationToolScalar, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(CalculationScalarType),) }
    @property
    def type(self) -> "CalculationScalarType":
        """Return the scalar calculation type."""
        return self._intf.get_property(ICalculationToolScalar._metadata, ICalculationToolScalar._get_type_metadata)

    _evaluate_metadata = { "offset" : _evaluate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate(self, epoch:typing.Any) -> "CalculationToolEvaluateResult":
        """Evaluate the scalar calculation at the specified time instant."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._evaluate_metadata, epoch, OutArg())

    _quick_evaluate_metadata = { "offset" : _quick_evaluate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.LPSafearrayArg,) }
    def quick_evaluate(self, epoch:typing.Any) -> list:
        """Evaluate the scalar calculation at the specified time instant and returns the results as an array with two elements, the first element being of boolean type indicating whether the computation succeeded, followed by a double-precision value representing..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_metadata, epoch, OutArg())

    _evaluate_with_rate_metadata = { "offset" : _evaluate_with_rate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate_with_rate(self, epoch:typing.Any) -> "CalculationToolEvaluateWithRateResult":
        """Evaluate the scalar calculation at the specified time instant. The result is a scalar value and its rate of change."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._evaluate_with_rate_metadata, epoch, OutArg())

    _quick_evaluate_with_rate_metadata = { "offset" : _quick_evaluate_with_rate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.LPSafearrayArg,) }
    def quick_evaluate_with_rate(self, epoch:typing.Any) -> list:
        """Evaluate the scalar calculation at the specified time instant and returns the results as an array with three elements, the first element being of boolean type indicating whether the computation succeeded, followed by two double-precision values one rep..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_with_rate_metadata, epoch, OutArg())

    _get_availability_metadata = { "offset" : _get_availability_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_availability(self) -> "TimeToolIntervalCollection":
        """Return a list of availability intervals."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._get_availability_metadata, OutArg())

    _get_unit_of_measure_metadata = { "offset" : _get_unit_of_measure_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def unit_of_measure(self) -> str:
        """Return calc scalar's unit of measure, i.e. 'AngleUnit', 'DistanceUnit', etc."""
        return self._intf.get_property(ICalculationToolScalar._metadata, ICalculationToolScalar._get_unit_of_measure_metadata)

    _quick_evaluate_array_metadata = { "offset" : _quick_evaluate_array_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def quick_evaluate_array(self, times:list) -> list:
        """Evaluate the scalar calculation, and rate, over an array of times, entered as strings in the Scenario date unit. It returns an array corresponding to the input times..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_array_metadata, times, OutArg())

    _quick_evaluate_with_rate_array_metadata = { "offset" : _quick_evaluate_with_rate_array_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def quick_evaluate_with_rate_array(self, times:list) -> list:
        """Evaluate the scalar calculation over the array of times provided by an Event Array component. It returns an array corresponding to the input times..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_with_rate_array_metadata, times, OutArg())

    _quick_evaluate_time_array_metadata = { "offset" : _quick_evaluate_time_array_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"), agmarshall.LPSafearrayArg,) }
    def quick_evaluate_time_array(self, ref_array:"ITimeToolTimeArray") -> list:
        """Evaluate the scalar calculation, and rate, over the array of times provided by an Event Array component. It returns an array corresponding to the input times..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_time_array_metadata, ref_array, OutArg())

    _quick_evaluate_with_rate_event_array_metadata = { "offset" : _quick_evaluate_with_rate_event_array_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"), agmarshall.LPSafearrayArg,) }
    def quick_evaluate_with_rate_event_array(self, ref_array:"ITimeToolTimeArray") -> list:
        """Evaluate the scalar calculation, and rate, over the array of times provided by an Event Array component. It returns an array corresponding to the input times..."""
        return self._intf.invoke(ICalculationToolScalar._metadata, ICalculationToolScalar._quick_evaluate_with_rate_event_array_metadata, ref_array, OutArg())

    _property_names[type] = "type"
    _property_names[unit_of_measure] = "unit_of_measure"


agcls.AgClassCatalog.add_catalog_entry((4703695914540684039, 8102226453412927140), ICalculationToolScalar)
agcls.AgTypeNameMap["ICalculationToolScalar"] = ICalculationToolScalar

class ICalculationToolCondition(object):
    """Condition returns a non-dimensional metric that is positive if satisfied, negative if not satisfied and 0 if on boundary; this provides computational methods needed for accurate detection of condition crossings."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _evaluate_method_offset = 2
    _evaluate_with_rate_method_offset = 3
    _metadata = {
        "iid_data" : (5735161800843082622, 14427694074333278871),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICalculationToolCondition."""
        initialize_from_source_object(self, source_object, ICalculationToolCondition)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICalculationToolCondition)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICalculationToolCondition, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConditionType),) }
    @property
    def type(self) -> "ConditionType":
        """Return the type of condition."""
        return self._intf.get_property(ICalculationToolCondition._metadata, ICalculationToolCondition._get_type_metadata)

    _evaluate_metadata = { "offset" : _evaluate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate(self, epoch:typing.Any) -> "CalculationToolEvaluateResult":
        """Return result of evaluating continuously varying condition metric at the specified time, used for detecting condition crossings."""
        return self._intf.invoke(ICalculationToolCondition._metadata, ICalculationToolCondition._evaluate_metadata, epoch, OutArg())

    _evaluate_with_rate_metadata = { "offset" : _evaluate_with_rate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate_with_rate(self, epoch:typing.Any) -> "CalculationToolEvaluateWithRateResult":
        """Return result of evaluating continuously varying condition metric and its rate of change at the specified time, used for detecting condition crossings."""
        return self._intf.invoke(ICalculationToolCondition._metadata, ICalculationToolCondition._evaluate_with_rate_metadata, epoch, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((5735161800843082622, 14427694074333278871), ICalculationToolCondition)
agcls.AgTypeNameMap["ICalculationToolCondition"] = ICalculationToolCondition

class ICalculationToolConditionSet(object):
    """Condition set returns an array of non-dimensional metrics, one for each condition in the set; each metric is positive if corresponding condition is satisfied, negative if not satisfied and 0 if on boundary; this provides computational methods needed for..."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _evaluate_method_offset = 2
    _evaluate_with_rate_method_offset = 3
    _metadata = {
        "iid_data" : (4655106964357171455, 14573822137210863791),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICalculationToolConditionSet."""
        initialize_from_source_object(self, source_object, ICalculationToolConditionSet)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICalculationToolConditionSet)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICalculationToolConditionSet, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConditionSetType),) }
    @property
    def type(self) -> "ConditionSetType":
        """Return the type of condition set."""
        return self._intf.get_property(ICalculationToolConditionSet._metadata, ICalculationToolConditionSet._get_type_metadata)

    _evaluate_metadata = { "offset" : _evaluate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate(self, epoch:typing.Any) -> "CalculationToolConditionSetEvaluateResult":
        """Return an array of results of evaluating continuously varying condition metrics, one for each condition in the set, at the specified time, used for detecting condition crossings."""
        return self._intf.invoke(ICalculationToolConditionSet._metadata, ICalculationToolConditionSet._evaluate_metadata, epoch, OutArg())

    _evaluate_with_rate_metadata = { "offset" : _evaluate_with_rate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def evaluate_with_rate(self, epoch:typing.Any) -> "CalculationToolConditionSetEvaluateWithRateResult":
        """Return an array of results of evaluating continuously varying condition metrics and their rates of change, one for each condition in the set, at the specified time, used for detecting condition crossings."""
        return self._intf.invoke(ICalculationToolConditionSet._metadata, ICalculationToolConditionSet._evaluate_with_rate_metadata, epoch, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((4655106964357171455, 14573822137210863791), ICalculationToolConditionSet)
agcls.AgTypeNameMap["ICalculationToolConditionSet"] = ICalculationToolConditionSet

class IAnalysisWorkbenchConvergence(object):
    """Represents a base class for convergence definitions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4915568931125686564, 3473459645036490151),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchConvergence."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchConvergence)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchConvergence)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchConvergence, None)



agcls.AgClassCatalog.add_catalog_entry((4915568931125686564, 3473459645036490151), IAnalysisWorkbenchConvergence)
agcls.AgTypeNameMap["IAnalysisWorkbenchConvergence"] = IAnalysisWorkbenchConvergence

class IAnalysisWorkbenchDerivative(object):
    """Represents a base class for derivative definitions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5545857981522920125, 18254938137483007413),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchDerivative."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchDerivative)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchDerivative)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchDerivative, None)



agcls.AgClassCatalog.add_catalog_entry((5545857981522920125, 18254938137483007413), IAnalysisWorkbenchDerivative)
agcls.AgTypeNameMap["IAnalysisWorkbenchDerivative"] = IAnalysisWorkbenchDerivative

class ITimeToolInstant(object):
    """Define an event (time instant)."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_today_method_offset = 2
    _get_tomorrow_method_offset = 3
    _get_noon_today_method_offset = 4
    _get_noon_tomorrow_method_offset = 5
    _find_occurrence_method_offset = 6
    _occurs_before_method_offset = 7
    _metadata = {
        "iid_data" : (5052226534501425744, 2346000977674024115),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolInstant."""
        initialize_from_source_object(self, source_object, ITimeToolInstant)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolInstant)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolInstant, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TimeEventType),) }
    @property
    def type(self) -> "TimeEventType":
        """Return the type of time instant."""
        return self._intf.get_property(ITimeToolInstant._metadata, ITimeToolInstant._get_type_metadata)

    _get_today_metadata = { "offset" : _get_today_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def today(self) -> "ITimeToolInstant":
        """Return time instant that corresponds to today's GMT midnight."""
        return self._intf.get_property(ITimeToolInstant._metadata, ITimeToolInstant._get_today_metadata)

    _get_tomorrow_metadata = { "offset" : _get_tomorrow_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def tomorrow(self) -> "ITimeToolInstant":
        """Return time instant that corresponds to tomorrow's GMT midnight."""
        return self._intf.get_property(ITimeToolInstant._metadata, ITimeToolInstant._get_tomorrow_metadata)

    _get_noon_today_metadata = { "offset" : _get_noon_today_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def noon_today(self) -> "ITimeToolInstant":
        """Return time instant that corresponds to today's GMT noon."""
        return self._intf.get_property(ITimeToolInstant._metadata, ITimeToolInstant._get_noon_today_metadata)

    _get_noon_tomorrow_metadata = { "offset" : _get_noon_tomorrow_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def noon_tomorrow(self) -> "ITimeToolInstant":
        """Return time instant that corresponds to tomorrow's GMT noon."""
        return self._intf.get_property(ITimeToolInstant._metadata, ITimeToolInstant._get_noon_tomorrow_metadata)

    _find_occurrence_metadata = { "offset" : _find_occurrence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def find_occurrence(self) -> "TimeToolInstantOccurrenceResult":
        """Return computed time instance if it occurs."""
        return self._intf.invoke(ITimeToolInstant._metadata, ITimeToolInstant._find_occurrence_metadata, OutArg())

    _occurs_before_metadata = { "offset" : _occurs_before_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantBoolArg,) }
    def occurs_before(self, epoch:typing.Any) -> bool:
        """Return true if computed time instance occurs before or at specified time, return false otherwise."""
        return self._intf.invoke(ITimeToolInstant._metadata, ITimeToolInstant._occurs_before_metadata, epoch, OutArg())

    _property_names[type] = "type"
    _property_names[today] = "today"
    _property_names[tomorrow] = "tomorrow"
    _property_names[noon_today] = "noon_today"
    _property_names[noon_tomorrow] = "noon_tomorrow"


agcls.AgClassCatalog.add_catalog_entry((5052226534501425744, 2346000977674024115), ITimeToolInstant)
agcls.AgTypeNameMap["ITimeToolInstant"] = ITimeToolInstant

class ITimeToolTimeArray(object):
    """An ordered array of times, which may or may not be evenly spaced."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _find_times_method_offset = 2
    _metadata = {
        "iid_data" : (5676478022059654098, 10082373008458811814),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolTimeArray."""
        initialize_from_source_object(self, source_object, ITimeToolTimeArray)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolTimeArray)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolTimeArray, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventArrayType),) }
    @property
    def type(self) -> "EventArrayType":
        """Return the type of time array."""
        return self._intf.get_property(ITimeToolTimeArray._metadata, ITimeToolTimeArray._get_type_metadata)

    _find_times_metadata = { "offset" : _find_times_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def find_times(self) -> "TimeToolTimeArrayFindTimesResult":
        """Return computed array of times."""
        return self._intf.invoke(ITimeToolTimeArray._metadata, ITimeToolTimeArray._find_times_metadata, OutArg())

    _property_names[type] = "type"


agcls.AgClassCatalog.add_catalog_entry((5676478022059654098, 10082373008458811814), ITimeToolTimeArray)
agcls.AgTypeNameMap["ITimeToolTimeArray"] = ITimeToolTimeArray

class ITimeToolTimeInterval(object):
    """A single time interval."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_label_start_description_method_offset = 2
    _get_label_stop_description_method_offset = 3
    _get_label_start_method_offset = 4
    _get_label_stop_method_offset = 5
    _find_interval_method_offset = 6
    _occurred_method_offset = 7
    _metadata = {
        "iid_data" : (5014335522660337085, 11558391138117110205),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolTimeInterval."""
        initialize_from_source_object(self, source_object, ITimeToolTimeInterval)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolTimeInterval)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolTimeInterval, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalType),) }
    @property
    def type(self) -> "EventIntervalType":
        """Return the type of interval."""
        return self._intf.get_property(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._get_type_metadata)

    _get_label_start_description_metadata = { "offset" : _get_label_start_description_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def label_start_description(self) -> str:
        """The start description."""
        return self._intf.get_property(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._get_label_start_description_metadata)

    _get_label_stop_description_metadata = { "offset" : _get_label_stop_description_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def label_stop_description(self) -> str:
        """The stop description."""
        return self._intf.get_property(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._get_label_stop_description_metadata)

    _get_label_start_metadata = { "offset" : _get_label_start_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def label_start(self) -> str:
        """A label associated with the interval start."""
        return self._intf.get_property(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._get_label_start_metadata)

    _get_label_stop_metadata = { "offset" : _get_label_stop_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def label_stop(self) -> str:
        """A label associated with the interval stop."""
        return self._intf.get_property(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._get_label_stop_metadata)

    _find_interval_metadata = { "offset" : _find_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def find_interval(self) -> "TimeToolTimeIntervalResult":
        """Return computed interval if it exists."""
        return self._intf.invoke(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._find_interval_metadata, OutArg())

    _occurred_metadata = { "offset" : _occurred_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantBoolArg,) }
    def occurred(self, epoch:typing.Any) -> bool:
        """Determine if specified time falls within computed interval if it exists."""
        return self._intf.invoke(ITimeToolTimeInterval._metadata, ITimeToolTimeInterval._occurred_metadata, epoch, OutArg())

    _property_names[type] = "type"
    _property_names[label_start_description] = "label_start_description"
    _property_names[label_stop_description] = "label_stop_description"
    _property_names[label_start] = "label_start"
    _property_names[label_stop] = "label_stop"


agcls.AgClassCatalog.add_catalog_entry((5014335522660337085, 11558391138117110205), ITimeToolTimeInterval)
agcls.AgTypeNameMap["ITimeToolTimeInterval"] = ITimeToolTimeInterval

class ITimeToolTimeIntervalCollection(object):
    """A collection of related interval lists."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_labels_method_offset = 2
    _find_interval_collection_method_offset = 3
    _occurred_method_offset = 4
    _metadata = {
        "iid_data" : (4802578054528125388, 3471729172666103957),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolTimeIntervalCollection."""
        initialize_from_source_object(self, source_object, ITimeToolTimeIntervalCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolTimeIntervalCollection)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolTimeIntervalCollection, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalCollectionType),) }
    @property
    def type(self) -> "EventIntervalCollectionType":
        """Return the type of collection of interval lists."""
        return self._intf.get_property(ITimeToolTimeIntervalCollection._metadata, ITimeToolTimeIntervalCollection._get_type_metadata)

    _get_labels_metadata = { "offset" : _get_labels_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def labels(self) -> list:
        """Get the labels associated with the interval lists in the collection."""
        return self._intf.get_property(ITimeToolTimeIntervalCollection._metadata, ITimeToolTimeIntervalCollection._get_labels_metadata)

    _find_interval_collection_metadata = { "offset" : _find_interval_collection_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def find_interval_collection(self) -> "TimeToolIntervalsVectorResult":
        """Return computed collection of interval lists."""
        return self._intf.invoke(ITimeToolTimeIntervalCollection._metadata, ITimeToolTimeIntervalCollection._find_interval_collection_metadata, OutArg())

    _occurred_metadata = { "offset" : _occurred_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def occurred(self, epoch:typing.Any) -> "TimeToolTimeIntervalCollectionOccurredResult":
        """Determine if specified time falls within any of the computed interval lists in the collection."""
        return self._intf.invoke(ITimeToolTimeIntervalCollection._metadata, ITimeToolTimeIntervalCollection._occurred_metadata, epoch, OutArg())

    _property_names[type] = "type"
    _property_names[labels] = "labels"


agcls.AgClassCatalog.add_catalog_entry((4802578054528125388, 3471729172666103957), ITimeToolTimeIntervalCollection)
agcls.AgTypeNameMap["ITimeToolTimeIntervalCollection"] = ITimeToolTimeIntervalCollection

class ITimeToolTimeIntervalList(object):
    """An ordered list of time intervals."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_labels_method_offset = 2
    _get_descriptions_method_offset = 3
    _find_intervals_method_offset = 4
    _occurred_method_offset = 5
    _metadata = {
        "iid_data" : (4964372786095352688, 17150178243890460291),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolTimeIntervalList."""
        initialize_from_source_object(self, source_object, ITimeToolTimeIntervalList)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolTimeIntervalList)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolTimeIntervalList, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalListType),) }
    @property
    def type(self) -> "EventIntervalListType":
        """Return the type of interval list."""
        return self._intf.get_property(ITimeToolTimeIntervalList._metadata, ITimeToolTimeIntervalList._get_type_metadata)

    _get_labels_metadata = { "offset" : _get_labels_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def labels(self) -> list:
        """Get the label descriptions associated with the interval list."""
        return self._intf.get_property(ITimeToolTimeIntervalList._metadata, ITimeToolTimeIntervalList._get_labels_metadata)

    _get_descriptions_metadata = { "offset" : _get_descriptions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def descriptions(self) -> list:
        """Get the labels associated with the interval list."""
        return self._intf.get_property(ITimeToolTimeIntervalList._metadata, ITimeToolTimeIntervalList._get_descriptions_metadata)

    _find_intervals_metadata = { "offset" : _find_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def find_intervals(self) -> "TimeToolIntervalListResult":
        """Return computed interval list that can be empty."""
        return self._intf.invoke(ITimeToolTimeIntervalList._metadata, ITimeToolTimeIntervalList._find_intervals_metadata, OutArg())

    _occurred_metadata = { "offset" : _occurred_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantBoolArg,) }
    def occurred(self, epoch:typing.Any) -> bool:
        """Determine if specified time falls within computed interval list."""
        return self._intf.invoke(ITimeToolTimeIntervalList._metadata, ITimeToolTimeIntervalList._occurred_metadata, epoch, OutArg())

    _property_names[type] = "type"
    _property_names[labels] = "labels"
    _property_names[descriptions] = "descriptions"


agcls.AgClassCatalog.add_catalog_entry((4964372786095352688, 17150178243890460291), ITimeToolTimeIntervalList)
agcls.AgTypeNameMap["ITimeToolTimeIntervalList"] = ITimeToolTimeIntervalList

class IAnalysisWorkbenchIntegral(object):
    """Represents a base class for integral definitions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4882407970628127095, 17356507896756110225),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchIntegral."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchIntegral)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchIntegral)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchIntegral, None)



agcls.AgClassCatalog.add_catalog_entry((4882407970628127095, 17356507896756110225), IAnalysisWorkbenchIntegral)
agcls.AgTypeNameMap["IAnalysisWorkbenchIntegral"] = IAnalysisWorkbenchIntegral

class IAnalysisWorkbenchInterpolator(object):
    """Represents a base class for interpolation definitions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5176727654361637888, 17197923576813057165),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchInterpolator."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchInterpolator)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchInterpolator)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchInterpolator, None)



agcls.AgClassCatalog.add_catalog_entry((5176727654361637888, 17197923576813057165), IAnalysisWorkbenchInterpolator)
agcls.AgTypeNameMap["IAnalysisWorkbenchInterpolator"] = IAnalysisWorkbenchInterpolator

class ICalculationToolParameterSet(object):
    """Parameter set contains various sets of scalar computations."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _get_labels_method_offset = 2
    _get_dimensions_method_offset = 3
    _get_scalar_names_method_offset = 4
    _calculate_method_offset = 5
    _calculate_with_derivative_method_offset = 6
    _metadata = {
        "iid_data" : (5479297772565193960, 11336582244767551646),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICalculationToolParameterSet."""
        initialize_from_source_object(self, source_object, ICalculationToolParameterSet)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICalculationToolParameterSet)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICalculationToolParameterSet, None)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ParameterSetType),) }
    @property
    def type(self) -> "ParameterSetType":
        """Get the type of parameter set."""
        return self._intf.get_property(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._get_type_metadata)

    _get_labels_metadata = { "offset" : _get_labels_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def labels(self) -> list:
        """Get the labels identifying hierarchy of representations within parameter set."""
        return self._intf.get_property(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._get_labels_metadata)

    _get_dimensions_metadata = { "offset" : _get_dimensions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def dimensions(self) -> list:
        """Get the names identifying types of dimensions of individual scalars within parameter set."""
        return self._intf.get_property(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._get_dimensions_metadata)

    _get_scalar_names_metadata = { "offset" : _get_scalar_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def scalar_names(self) -> list:
        """Get the names identifying individual scalars within parameter set."""
        return self._intf.get_property(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._get_scalar_names_metadata)

    _calculate_metadata = { "offset" : _calculate_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.LPSafearrayArg,) }
    def calculate(self, epoch:typing.Any) -> list:
        """Return results of computing individual scalars within parameter set at the specified time."""
        return self._intf.invoke(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._calculate_metadata, epoch, OutArg())

    _calculate_with_derivative_metadata = { "offset" : _calculate_with_derivative_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.LPSafearrayArg,) }
    def calculate_with_derivative(self, epoch:typing.Any) -> list:
        """Return results of computing individual scalars and their time derivatives within parameter set at the specified time."""
        return self._intf.invoke(ICalculationToolParameterSet._metadata, ICalculationToolParameterSet._calculate_with_derivative_metadata, epoch, OutArg())

    _property_names[type] = "type"
    _property_names[labels] = "labels"
    _property_names[dimensions] = "dimensions"
    _property_names[scalar_names] = "scalar_names"


agcls.AgClassCatalog.add_catalog_entry((5479297772565193960, 11336582244767551646), ICalculationToolParameterSet)
agcls.AgTypeNameMap["ICalculationToolParameterSet"] = ICalculationToolParameterSet

class ITimeToolPruneFilter(object):
    """A filter used with event interval list pruned class to prune interval lists..."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_filter_type_method_offset = 1
    _metadata = {
        "iid_data" : (5411019644010585303, 8356519221695235998),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ITimeToolPruneFilter."""
        initialize_from_source_object(self, source_object, ITimeToolPruneFilter)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ITimeToolPruneFilter)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ITimeToolPruneFilter, None)

    _get_filter_type_metadata = { "offset" : _get_filter_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalPruneFilterType),) }
    @property
    def filter_type(self) -> "IntervalPruneFilterType":
        """Return a type of the filter."""
        return self._intf.get_property(ITimeToolPruneFilter._metadata, ITimeToolPruneFilter._get_filter_type_metadata)

    _property_names[filter_type] = "filter_type"


agcls.AgClassCatalog.add_catalog_entry((5411019644010585303, 8356519221695235998), ITimeToolPruneFilter)
agcls.AgTypeNameMap["ITimeToolPruneFilter"] = ITimeToolPruneFilter

class IAnalysisWorkbenchSampling(object):
    """Base sampling interface."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (5454263766504862323, 6397342263310433470),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchSampling."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchSampling)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchSampling)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchSampling, None)



agcls.AgClassCatalog.add_catalog_entry((5454263766504862323, 6397342263310433470), IAnalysisWorkbenchSampling)
agcls.AgTypeNameMap["IAnalysisWorkbenchSampling"] = IAnalysisWorkbenchSampling

class ICalculationToolSamplingMethod(object):
    """A sampling method."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_type_method_offset = 1
    _metadata = {
        "iid_data" : (4827843609729710366, 8894356402950175411),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ICalculationToolSamplingMethod."""
        initialize_from_source_object(self, source_object, ICalculationToolSamplingMethod)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ICalculationToolSamplingMethod)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ICalculationToolSamplingMethod, None)

    _get_method_type_metadata = { "offset" : _get_method_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VectorGeometryToolSamplingMethod),) }
    @property
    def method_type(self) -> "VectorGeometryToolSamplingMethod":
        """A sampling method type."""
        return self._intf.get_property(ICalculationToolSamplingMethod._metadata, ICalculationToolSamplingMethod._get_method_type_metadata)

    _property_names[method_type] = "method_type"


agcls.AgClassCatalog.add_catalog_entry((4827843609729710366, 8894356402950175411), ICalculationToolSamplingMethod)
agcls.AgTypeNameMap["ICalculationToolSamplingMethod"] = ICalculationToolSamplingMethod

class IAnalysisWorkbenchSignalDelay(object):
    """Signal delay definition determines how long it takes for a signal to propagate from one location to another."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4836052727854610771, 2658301614566846344),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchSignalDelay."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchSignalDelay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchSignalDelay)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchSignalDelay, None)



agcls.AgClassCatalog.add_catalog_entry((4836052727854610771, 2658301614566846344), IAnalysisWorkbenchSignalDelay)
agcls.AgTypeNameMap["IAnalysisWorkbenchSignalDelay"] = IAnalysisWorkbenchSignalDelay

class ISpatialAnalysisToolGridValuesMethod(object):
    """A grid values method."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_type_method_offset = 1
    _metadata = {
        "iid_data" : (4739060787290770026, 755991415117168800),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISpatialAnalysisToolGridValuesMethod."""
        initialize_from_source_object(self, source_object, ISpatialAnalysisToolGridValuesMethod)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISpatialAnalysisToolGridValuesMethod)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISpatialAnalysisToolGridValuesMethod, None)

    _get_method_type_metadata = { "offset" : _get_method_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(GridValuesMethodType),) }
    @property
    def method_type(self) -> "GridValuesMethodType":
        """Grid values method type."""
        return self._intf.get_property(ISpatialAnalysisToolGridValuesMethod._metadata, ISpatialAnalysisToolGridValuesMethod._get_method_type_metadata)

    _property_names[method_type] = "method_type"


agcls.AgClassCatalog.add_catalog_entry((4739060787290770026, 755991415117168800), ISpatialAnalysisToolGridValuesMethod)
agcls.AgTypeNameMap["ISpatialAnalysisToolGridValuesMethod"] = ISpatialAnalysisToolGridValuesMethod

class ISpatialAnalysisToolVolume(object):
    """A volume interface. The methods and properties of the interface provide Volume functions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4841356890486692076, 2175121878908514689),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISpatialAnalysisToolVolume."""
        initialize_from_source_object(self, source_object, ISpatialAnalysisToolVolume)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISpatialAnalysisToolVolume)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISpatialAnalysisToolVolume, None)



agcls.AgClassCatalog.add_catalog_entry((4841356890486692076, 2175121878908514689), ISpatialAnalysisToolVolume)
agcls.AgTypeNameMap["ISpatialAnalysisToolVolume"] = ISpatialAnalysisToolVolume

class ISpatialAnalysisToolSpatialCalculation(object):
    """A volume calc interface. The methods and properties of the interface provide Volumetric calc functions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4788088731026599523, 14975002087762860939),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISpatialAnalysisToolSpatialCalculation."""
        initialize_from_source_object(self, source_object, ISpatialAnalysisToolSpatialCalculation)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISpatialAnalysisToolSpatialCalculation)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISpatialAnalysisToolSpatialCalculation, None)



agcls.AgClassCatalog.add_catalog_entry((4788088731026599523, 14975002087762860939), ISpatialAnalysisToolSpatialCalculation)
agcls.AgTypeNameMap["ISpatialAnalysisToolSpatialCalculation"] = ISpatialAnalysisToolSpatialCalculation

class ISpatialAnalysisToolVolumeGrid(object):
    """A volume grid interface. The methods and properties of the interface provide Volumetric Grid functions."""

    _num_methods = 0
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _metadata = {
        "iid_data" : (4872197438153942913, 17020680151074665636),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type ISpatialAnalysisToolVolumeGrid."""
        initialize_from_source_object(self, source_object, ISpatialAnalysisToolVolumeGrid)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, ISpatialAnalysisToolVolumeGrid)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, ISpatialAnalysisToolVolumeGrid, None)



agcls.AgClassCatalog.add_catalog_entry((4872197438153942913, 17020680151074665636), ISpatialAnalysisToolVolumeGrid)
agcls.AgTypeNameMap["ISpatialAnalysisToolVolumeGrid"] = ISpatialAnalysisToolVolumeGrid

class IAnalysisWorkbenchComponentTimeProperties(object):
    """Define methods to compute time properties such as availability and special times."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_availability_method_offset = 1
    _metadata = {
        "iid_data" : (4693587397954990762, 1841444941110468780),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchComponentTimeProperties."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchComponentTimeProperties)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchComponentTimeProperties)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchComponentTimeProperties, None)

    _get_availability_metadata = { "offset" : _get_availability_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_availability(self) -> "TimeToolIntervalCollection":
        """Return a collection of availability intervals."""
        return self._intf.invoke(IAnalysisWorkbenchComponentTimeProperties._metadata, IAnalysisWorkbenchComponentTimeProperties._get_availability_metadata, OutArg())



agcls.AgClassCatalog.add_catalog_entry((4693587397954990762, 1841444941110468780), IAnalysisWorkbenchComponentTimeProperties)
agcls.AgTypeNameMap["IAnalysisWorkbenchComponentTimeProperties"] = IAnalysisWorkbenchComponentTimeProperties

class IAnalysisWorkbenchComponentReference(object):
    """A base interface for all VGT component references."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_path_method_offset = 1
    _metadata = {
        "iid_data" : (5000476646516070310, 1217044254048263356),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchComponentReference."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchComponentReference)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchComponentReference)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchComponentReference, None)

    _get_path_metadata = { "offset" : _get_path_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def path(self) -> str:
        """Return a path to the referenced VGT component."""
        return self._intf.get_property(IAnalysisWorkbenchComponentReference._metadata, IAnalysisWorkbenchComponentReference._get_path_metadata)

    _property_names[path] = "path"


agcls.AgClassCatalog.add_catalog_entry((5000476646516070310, 1217044254048263356), IAnalysisWorkbenchComponentReference)
agcls.AgTypeNameMap["IAnalysisWorkbenchComponentReference"] = IAnalysisWorkbenchComponentReference

class IAnalysisWorkbenchMethodCallResult(object):
    """Instances of the interface are used to return the result of a computation."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _metadata = {
        "iid_data" : (4645374652506169535, 9723345654066007957),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def __init__(self, source_object=None):
        """Construct an object of type IAnalysisWorkbenchMethodCallResult."""
        initialize_from_source_object(self, source_object, IAnalysisWorkbenchMethodCallResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def _get_property(self, attrname):
        return get_interface_property(attrname, IAnalysisWorkbenchMethodCallResult)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_interface_attribute(self, attrname, value, IAnalysisWorkbenchMethodCallResult, None)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """True indicates the method call was successful."""
        return self._intf.get_property(IAnalysisWorkbenchMethodCallResult._metadata, IAnalysisWorkbenchMethodCallResult._get_is_valid_metadata)

    _property_names[is_valid] = "is_valid"


agcls.AgClassCatalog.add_catalog_entry((4645374652506169535, 9723345654066007957), IAnalysisWorkbenchMethodCallResult)
agcls.AgTypeNameMap["IAnalysisWorkbenchMethodCallResult"] = IAnalysisWorkbenchMethodCallResult



class CalculationToolEvaluateResult(SupportsDeleteCallback):
    """Represents the results of evaluating a scalar component using ICalculationToolScalar.Evaluate method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_value_method_offset = 2
    _metadata = {
        "iid_data" : (5614154425131203652, 5944516947886916776),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolEvaluateResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(CalculationToolEvaluateResult._metadata, CalculationToolEvaluateResult._get_is_valid_metadata)

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def value(self) -> float:
        """The scalar value."""
        return self._intf.get_property(CalculationToolEvaluateResult._metadata, CalculationToolEvaluateResult._get_value_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[value] = "value"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolEvaluateResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolEvaluateResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolEvaluateResult, [CalculationToolEvaluateResult, ])

agcls.AgClassCatalog.add_catalog_entry((5522853299729796607, 10609423862588038055), CalculationToolEvaluateResult)
agcls.AgTypeNameMap["CalculationToolEvaluateResult"] = CalculationToolEvaluateResult

class CalculationToolEvaluateWithRateResult(SupportsDeleteCallback):
    """Represents the results of evaluating a scalar component using ICalculationToolScalar.Evaluate method."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_value_method_offset = 2
    _get_rate_method_offset = 3
    _metadata = {
        "iid_data" : (4891493138703768112, 5257715458728871821),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolEvaluateWithRateResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(CalculationToolEvaluateWithRateResult._metadata, CalculationToolEvaluateWithRateResult._get_is_valid_metadata)

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def value(self) -> float:
        """Computed scalar value."""
        return self._intf.get_property(CalculationToolEvaluateWithRateResult._metadata, CalculationToolEvaluateWithRateResult._get_value_metadata)

    _get_rate_metadata = { "offset" : _get_rate_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def rate(self) -> float:
        """A rate of change of the computed scalar value."""
        return self._intf.get_property(CalculationToolEvaluateWithRateResult._metadata, CalculationToolEvaluateWithRateResult._get_rate_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[value] = "value"
    _property_names[rate] = "rate"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolEvaluateWithRateResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolEvaluateWithRateResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolEvaluateWithRateResult, [CalculationToolEvaluateWithRateResult, ])

agcls.AgClassCatalog.add_catalog_entry((5662589128398381369, 18228667425859790763), CalculationToolEvaluateWithRateResult)
agcls.AgTypeNameMap["CalculationToolEvaluateWithRateResult"] = CalculationToolEvaluateWithRateResult

class TimeToolTimeIntervalResult(SupportsDeleteCallback):
    """Contains the results returned with ITimeToolTimeIntervalList.FindIntervals method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_interval_method_offset = 2
    _metadata = {
        "iid_data" : (4746291031874526350, 12635609897701193394),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolTimeIntervalResult._metadata, TimeToolTimeIntervalResult._get_is_valid_metadata)

    _get_interval_metadata = { "offset" : _get_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval(self) -> "TimeToolInterval":
        """An interval."""
        return self._intf.get_property(TimeToolTimeIntervalResult._metadata, TimeToolTimeIntervalResult._get_interval_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[interval] = "interval"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalResult, [TimeToolTimeIntervalResult, ])

agcls.AgClassCatalog.add_catalog_entry((5440144273098326668, 11989278598177568391), TimeToolTimeIntervalResult)
agcls.AgTypeNameMap["TimeToolTimeIntervalResult"] = TimeToolTimeIntervalResult

class TimeToolInstantOccurrenceResult(SupportsDeleteCallback):
    """Contains the results returned with ITimeToolInstant.FindOccurrence method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_epoch_method_offset = 2
    _metadata = {
        "iid_data" : (4710347834450863344, 12752006829468697747),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantOccurrenceResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolInstantOccurrenceResult._metadata, TimeToolInstantOccurrenceResult._get_is_valid_metadata)

    _get_epoch_metadata = { "offset" : _get_epoch_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def epoch(self) -> typing.Any:
        """The epoch at which the event occurs."""
        return self._intf.get_property(TimeToolInstantOccurrenceResult._metadata, TimeToolInstantOccurrenceResult._get_epoch_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[epoch] = "epoch"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantOccurrenceResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantOccurrenceResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantOccurrenceResult, [TimeToolInstantOccurrenceResult, ])

agcls.AgClassCatalog.add_catalog_entry((4920266892008173400, 17346109474033320069), TimeToolInstantOccurrenceResult)
agcls.AgTypeNameMap["TimeToolInstantOccurrenceResult"] = TimeToolInstantOccurrenceResult

class TimeToolTimeArrayFindTimesResult(SupportsDeleteCallback):
    """Return a collection of intervals and an array of times."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_intervals_method_offset = 2
    _get_start_method_offset = 3
    _get_stop_method_offset = 4
    _get_times_method_offset = 5
    _metadata = {
        "iid_data" : (4849540246283896048, 6942407106662663820),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayFindTimesResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolTimeArrayFindTimesResult._metadata, TimeToolTimeArrayFindTimesResult._get_is_valid_metadata)

    _get_intervals_metadata = { "offset" : _get_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def intervals(self) -> "TimeToolIntervalCollection":
        """A collection of found intervals."""
        return self._intf.get_property(TimeToolTimeArrayFindTimesResult._metadata, TimeToolTimeArrayFindTimesResult._get_intervals_metadata)

    _get_start_metadata = { "offset" : _get_start_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def start(self) -> typing.Any:
        """The start time of the entire interval span."""
        return self._intf.get_property(TimeToolTimeArrayFindTimesResult._metadata, TimeToolTimeArrayFindTimesResult._get_start_metadata)

    _get_stop_metadata = { "offset" : _get_stop_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def stop(self) -> typing.Any:
        """The stop time of the entire interval span."""
        return self._intf.get_property(TimeToolTimeArrayFindTimesResult._metadata, TimeToolTimeArrayFindTimesResult._get_stop_metadata)

    _get_times_metadata = { "offset" : _get_times_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def times(self) -> list:
        """An array of found times."""
        return self._intf.get_property(TimeToolTimeArrayFindTimesResult._metadata, TimeToolTimeArrayFindTimesResult._get_times_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[intervals] = "intervals"
    _property_names[start] = "start"
    _property_names[stop] = "stop"
    _property_names[times] = "times"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayFindTimesResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayFindTimesResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayFindTimesResult, [TimeToolTimeArrayFindTimesResult, ])

agcls.AgClassCatalog.add_catalog_entry((5536753132893401039, 8791447751690390407), TimeToolTimeArrayFindTimesResult)
agcls.AgTypeNameMap["TimeToolTimeArrayFindTimesResult"] = TimeToolTimeArrayFindTimesResult

class TimeToolIntervalsVectorResult(SupportsDeleteCallback):
    """Contains the results returned with ITimeToolTimeIntervalCollection.FindIntervalCollection method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_interval_collections_method_offset = 2
    _metadata = {
        "iid_data" : (5620318592927493953, 3800513619111014055),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolIntervalsVectorResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolIntervalsVectorResult._metadata, TimeToolIntervalsVectorResult._get_is_valid_metadata)

    _get_interval_collections_metadata = { "offset" : _get_interval_collections_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval_collections(self) -> "TimeToolIntervalVectorCollection":
        """A collection of interval collections."""
        return self._intf.get_property(TimeToolIntervalsVectorResult._metadata, TimeToolIntervalsVectorResult._get_interval_collections_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[interval_collections] = "interval_collections"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolIntervalsVectorResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolIntervalsVectorResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolIntervalsVectorResult, [TimeToolIntervalsVectorResult, ])

agcls.AgClassCatalog.add_catalog_entry((5214899444003868340, 2439419900054653882), TimeToolIntervalsVectorResult)
agcls.AgTypeNameMap["TimeToolIntervalsVectorResult"] = TimeToolIntervalsVectorResult

class TimeToolTimeIntervalCollectionOccurredResult(SupportsDeleteCallback):
    """Contains the results returned with ITimeToolTimeIntervalCollection.Occurred method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_index_method_offset = 2
    _metadata = {
        "iid_data" : (4984827063323662366, 10431628503200399788),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionOccurredResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionOccurredResult._metadata, TimeToolTimeIntervalCollectionOccurredResult._get_is_valid_metadata)

    _get_index_metadata = { "offset" : _get_index_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def index(self) -> int:
        """Index of an interval in the collection."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionOccurredResult._metadata, TimeToolTimeIntervalCollectionOccurredResult._get_index_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[index] = "index"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionOccurredResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionOccurredResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionOccurredResult, [TimeToolTimeIntervalCollectionOccurredResult, ])

agcls.AgClassCatalog.add_catalog_entry((5449426388650809590, 4955022891265085365), TimeToolTimeIntervalCollectionOccurredResult)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionOccurredResult"] = TimeToolTimeIntervalCollectionOccurredResult

class TimeToolIntervalListResult(SupportsDeleteCallback):
    """Contains the results returned with ITimeToolTimeIntervalList.FindIntervals method."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_intervals_method_offset = 2
    _metadata = {
        "iid_data" : (4634920377665823855, 14964140121783584654),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolIntervalListResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(TimeToolIntervalListResult._metadata, TimeToolIntervalListResult._get_is_valid_metadata)

    _get_intervals_metadata = { "offset" : _get_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def intervals(self) -> "TimeToolIntervalCollection":
        """A list of intervals."""
        return self._intf.get_property(TimeToolIntervalListResult._metadata, TimeToolIntervalListResult._get_intervals_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[intervals] = "intervals"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolIntervalListResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolIntervalListResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolIntervalListResult, [TimeToolIntervalListResult, ])

agcls.AgClassCatalog.add_catalog_entry((4749115582065684196, 2274990611771605949), TimeToolIntervalListResult)
agcls.AgTypeNameMap["TimeToolIntervalListResult"] = TimeToolIntervalListResult

class TimeToolIntervalVectorCollection(SupportsDeleteCallback):
    """A collection of interval collections."""

    _num_methods = 3
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _get_count_method_offset = 1
    _item_method_offset = 2
    _get__new_enum_method_offset = 3
    _metadata = {
        "iid_data" : (5575174488488139361, 17888026257103832750),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolIntervalVectorCollection)
    def __iter__(self):
        """Create an iterator for the TimeToolIntervalVectorCollection object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "TimeToolIntervalCollection":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Number of elements in the collection."""
        return self._intf.get_property(TimeToolIntervalVectorCollection._metadata, TimeToolIntervalVectorCollection._get_count_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LongArg, agmarshall.InterfaceOutArg,) }
    def item(self, index:int) -> "TimeToolIntervalCollection":
        """Access an element at the specified position."""
        return self._intf.invoke(TimeToolIntervalVectorCollection._metadata, TimeToolIntervalVectorCollection._item_metadata, index, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolIntervalVectorCollection._metadata, TimeToolIntervalVectorCollection._get__new_enum_metadata)

    __getitem__ = item


    _property_names[count] = "count"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolIntervalVectorCollection."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolIntervalVectorCollection)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolIntervalVectorCollection, [TimeToolIntervalVectorCollection, ])

agcls.AgClassCatalog.add_catalog_entry((5203412792180924882, 14239251967220039073), TimeToolIntervalVectorCollection)
agcls.AgTypeNameMap["TimeToolIntervalVectorCollection"] = TimeToolIntervalVectorCollection

class TimeToolInstantGroup(SupportsDeleteCallback):
    """Access or create VGT events associated with an object."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5743339348524930415, 13668044487410794916),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantGroup)
    def __iter__(self):
        """Create an iterator for the TimeToolInstantGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITimeToolInstant":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(TimeToolInstantGroup._metadata, TimeToolInstantGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(TimeToolInstantGroup._metadata, TimeToolInstantGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "TimeToolInstantFactory":
        """Return a Factory object used to create custom events."""
        return self._intf.get_property(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ITimeToolInstant":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(TimeToolInstantGroup._metadata, TimeToolInstantGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ITimeToolInstant":
        """Retrieve an event from the collection by index."""
        return self._intf.invoke(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ITimeToolInstant":
        """Retrieve an event from the collection by name."""
        return self._intf.invoke(TimeToolInstantGroup._metadata, TimeToolInstantGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantGroup, [TimeToolInstantGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5740305381890470273, 8415316688318852744), TimeToolInstantGroup)
agcls.AgTypeNameMap["TimeToolInstantGroup"] = TimeToolInstantGroup

class TimeToolTimeIntervalGroup(SupportsDeleteCallback):
    """Access or create VGT event intervals associated with an object."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5142922729546224873, 11946726056227879853),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalGroup)
    def __iter__(self):
        """Create an iterator for the TimeToolTimeIntervalGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITimeToolTimeInterval":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_interval_name:str) -> None:
        """Remove an element by name."""
        return self._intf.invoke(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._remove_metadata, event_interval_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "TimeToolTimeIntervalFactory":
        """Return a Factory object used to create custom event intervals."""
        return self._intf.get_property(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ITimeToolTimeInterval":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ITimeToolTimeInterval":
        """Retrieve an event interval from the collection by index."""
        return self._intf.invoke(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ITimeToolTimeInterval":
        """Retrieve an event interval from the collection by name."""
        return self._intf.invoke(TimeToolTimeIntervalGroup._metadata, TimeToolTimeIntervalGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalGroup, [TimeToolTimeIntervalGroup, ])

agcls.AgClassCatalog.add_catalog_entry((4639595004900791506, 2670686152777183627), TimeToolTimeIntervalGroup)
agcls.AgTypeNameMap["TimeToolTimeIntervalGroup"] = TimeToolTimeIntervalGroup

class TimeToolTimeIntervalListGroup(SupportsDeleteCallback):
    """Access or create VGT event interval lists associated with an object."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (4938951633900015673, 7195780011008496024),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListGroup)
    def __iter__(self):
        """Create an iterator for the TimeToolTimeIntervalListGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITimeToolTimeIntervalList":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "TimeToolTimeIntervalListFactory":
        """Return a factory object used to create custom event interval lists."""
        return self._intf.get_property(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ITimeToolTimeIntervalList":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ITimeToolTimeIntervalList":
        """Retrieve an event interval list from the collection by index."""
        return self._intf.invoke(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ITimeToolTimeIntervalList":
        """Retrieve an event interval list from the collection by name."""
        return self._intf.invoke(TimeToolTimeIntervalListGroup._metadata, TimeToolTimeIntervalListGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListGroup, [TimeToolTimeIntervalListGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5001426302640578234, 7402032539944503439), TimeToolTimeIntervalListGroup)
agcls.AgTypeNameMap["TimeToolTimeIntervalListGroup"] = TimeToolTimeIntervalListGroup

class TimeToolTimeArrayGroup(SupportsDeleteCallback):
    """Access or create VGT event arrays associated with an object."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5362443867455049139, 5221206169826244786),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayGroup)
    def __iter__(self):
        """Create an iterator for the TimeToolTimeArrayGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITimeToolTimeArray":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "TimeToolTimeArrayFactory":
        """Return a Factory object used to create event arrays."""
        return self._intf.get_property(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ITimeToolTimeArray":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ITimeToolTimeArray":
        """Retrieve an event array from the collection by index."""
        return self._intf.invoke(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ITimeToolTimeArray":
        """Retrieve an event array from the collection by name."""
        return self._intf.invoke(TimeToolTimeArrayGroup._metadata, TimeToolTimeArrayGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayGroup, [TimeToolTimeArrayGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5225191702089452142, 4839624242436753046), TimeToolTimeArrayGroup)
agcls.AgTypeNameMap["TimeToolTimeArrayGroup"] = TimeToolTimeArrayGroup

class CalculationToolScalarGroup(SupportsDeleteCallback):
    """Access or create VGT calculation scalars associated with an object or a central body."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (4659785540148584367, 12814880882912016044),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarGroup)
    def __iter__(self):
        """Create an iterator for the CalculationToolScalarGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ICalculationToolScalar":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "CalculationToolScalarFactory":
        """Return a factory object used to create calc scalar components."""
        return self._intf.get_property(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ICalculationToolScalar":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ICalculationToolScalar":
        """Retrieve an element from the collection by index."""
        return self._intf.invoke(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ICalculationToolScalar":
        """Retrieve an element from the collection by name."""
        return self._intf.invoke(CalculationToolScalarGroup._metadata, CalculationToolScalarGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarGroup, [CalculationToolScalarGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5048329027604808808, 12927501014310075801), CalculationToolScalarGroup)
agcls.AgTypeNameMap["CalculationToolScalarGroup"] = CalculationToolScalarGroup

class TimeToolTimeIntervalCollectionGroup(SupportsDeleteCallback):
    """Access or create VGT event interval collections associated with an object."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5628253959361021565, 6774093585548149417),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionGroup)
    def __iter__(self):
        """Create an iterator for the TimeToolTimeIntervalCollectionGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ITimeToolTimeIntervalCollection":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "TimeToolTimeIntervalCollectionFactory":
        """Return a factory object used to create calc scalar components."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ITimeToolTimeIntervalCollection":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ITimeToolTimeIntervalCollection":
        """Retrieve an event interval from the collection by index."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ITimeToolTimeIntervalCollection":
        """Retrieve an event interval from the collection by name."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionGroup._metadata, TimeToolTimeIntervalCollectionGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionGroup, [TimeToolTimeIntervalCollectionGroup, ])

agcls.AgClassCatalog.add_catalog_entry((4676956353340798946, 4132032579773110454), TimeToolTimeIntervalCollectionGroup)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionGroup"] = TimeToolTimeIntervalCollectionGroup

class CalculationToolParameterSetGroup(SupportsDeleteCallback):
    """Access or create VGT parameter sets associated with an object or a central body."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5625575995100372233, 5247817078545365909),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetGroup)
    def __iter__(self):
        """Create an iterator for the CalculationToolParameterSetGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ICalculationToolParameterSet":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "CalculationToolParameterSetFactory":
        """Return a factory object used to create calc scalar components."""
        return self._intf.get_property(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ICalculationToolParameterSet":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ICalculationToolParameterSet":
        """Retrieve an element from the collection by index."""
        return self._intf.invoke(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ICalculationToolParameterSet":
        """Retrieve an element from the collection by name."""
        return self._intf.invoke(CalculationToolParameterSetGroup._metadata, CalculationToolParameterSetGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetGroup, [CalculationToolParameterSetGroup, ])

agcls.AgClassCatalog.add_catalog_entry((4675360958547478665, 15542114842964845204), CalculationToolParameterSetGroup)
agcls.AgTypeNameMap["CalculationToolParameterSetGroup"] = CalculationToolParameterSetGroup

class CalculationToolConditionGroup(SupportsDeleteCallback):
    """Access or create VGT conditions associated with an object or a central body."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5727353733426007755, 13885664073261871009),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionGroup)
    def __iter__(self):
        """Create an iterator for the CalculationToolConditionGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ICalculationToolCondition":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "CalculationToolConditionFactory":
        """Return a factory object used to create calc scalar components."""
        return self._intf.get_property(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ICalculationToolCondition":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ICalculationToolCondition":
        """Retrieve a condition from the collection by index."""
        return self._intf.invoke(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ICalculationToolCondition":
        """Retrieve a condition from the collection by name."""
        return self._intf.invoke(CalculationToolConditionGroup._metadata, CalculationToolConditionGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionGroup, [CalculationToolConditionGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5750765973699081321, 14571179364062773916), CalculationToolConditionGroup)
agcls.AgTypeNameMap["CalculationToolConditionGroup"] = CalculationToolConditionGroup

class CalculationToolConditionSetGroup(SupportsDeleteCallback):
    """Allow accessing and creating condition set components."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5611826300224457159, 16584276783501574569),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionSetGroup)
    def __iter__(self):
        """Create an iterator for the CalculationToolConditionSetGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ICalculationToolConditionSet":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "CalculationToolConditionSetFactory":
        """Return a factory object used to create condition set components."""
        return self._intf.get_property(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ICalculationToolConditionSet":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ICalculationToolConditionSet":
        """Retrieve a condition set from the collection by index."""
        return self._intf.invoke(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ICalculationToolConditionSet":
        """Retrieve a condition set from the collection by name."""
        return self._intf.invoke(CalculationToolConditionSetGroup._metadata, CalculationToolConditionSetGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSetGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionSetGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSetGroup, [CalculationToolConditionSetGroup, ])

agcls.AgClassCatalog.add_catalog_entry((4707405303890402417, 10690097794579297951), CalculationToolConditionSetGroup)
agcls.AgTypeNameMap["CalculationToolConditionSetGroup"] = CalculationToolConditionSetGroup

class CalculationToolConditionSetEvaluateResult(SupportsDeleteCallback):
    """Represents the results returned by ConditionSet.Evaluate."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_values_method_offset = 2
    _metadata = {
        "iid_data" : (5427191482923494608, 3628611338351778977),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionSetEvaluateResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(CalculationToolConditionSetEvaluateResult._metadata, CalculationToolConditionSetEvaluateResult._get_is_valid_metadata)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """Computed values."""
        return self._intf.get_property(CalculationToolConditionSetEvaluateResult._metadata, CalculationToolConditionSetEvaluateResult._get_values_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSetEvaluateResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionSetEvaluateResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSetEvaluateResult, [CalculationToolConditionSetEvaluateResult, ])

agcls.AgClassCatalog.add_catalog_entry((5222756833012542291, 4920051115777198524), CalculationToolConditionSetEvaluateResult)
agcls.AgTypeNameMap["CalculationToolConditionSetEvaluateResult"] = CalculationToolConditionSetEvaluateResult

class CalculationToolConditionSetEvaluateWithRateResult(SupportsDeleteCallback):
    """Represents the results returned by ConditionSet.EvaluateWithRate."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_is_valid_method_offset = 1
    _get_values_method_offset = 2
    _get_rates_method_offset = 3
    _metadata = {
        "iid_data" : (4805407826333748553, 13212141293468544164),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionSetEvaluateWithRateResult)

    _get_is_valid_metadata = { "offset" : _get_is_valid_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_valid(self) -> bool:
        """Indicate whether the result object is valid."""
        return self._intf.get_property(CalculationToolConditionSetEvaluateWithRateResult._metadata, CalculationToolConditionSetEvaluateWithRateResult._get_is_valid_metadata)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """Computed values."""
        return self._intf.get_property(CalculationToolConditionSetEvaluateWithRateResult._metadata, CalculationToolConditionSetEvaluateWithRateResult._get_values_metadata)

    _get_rates_metadata = { "offset" : _get_rates_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def rates(self) -> list:
        """Computed rates."""
        return self._intf.get_property(CalculationToolConditionSetEvaluateWithRateResult._metadata, CalculationToolConditionSetEvaluateWithRateResult._get_rates_metadata)

    _property_names[is_valid] = "is_valid"
    _property_names[values] = "values"
    _property_names[rates] = "rates"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSetEvaluateWithRateResult."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionSetEvaluateWithRateResult)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSetEvaluateWithRateResult, [CalculationToolConditionSetEvaluateWithRateResult, ])

agcls.AgClassCatalog.add_catalog_entry((4840393028332232055, 13403874806573712528), CalculationToolConditionSetEvaluateWithRateResult)
agcls.AgTypeNameMap["CalculationToolConditionSetEvaluateWithRateResult"] = CalculationToolConditionSetEvaluateWithRateResult

class SpatialAnalysisToolVolumeGridGroup(SupportsDeleteCallback):
    """Access or create VGT volume grids associated with an object or a central body."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5652804191328005584, 15020715553548321674),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolVolumeGridGroup)
    def __iter__(self):
        """Create an iterator for the SpatialAnalysisToolVolumeGridGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ISpatialAnalysisToolVolumeGrid":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "SpatialAnalysisToolVolumeGridFactory":
        """Return a factory object used to create volume grid components."""
        return self._intf.get_property(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ISpatialAnalysisToolVolumeGrid":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ISpatialAnalysisToolVolumeGrid":
        """Retrieve a volume grid from the collection by index."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Retrieve a volume grid from the collection by name."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridGroup._metadata, SpatialAnalysisToolVolumeGridGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolVolumeGridGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolVolumeGridGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolVolumeGridGroup, [SpatialAnalysisToolVolumeGridGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5434667114735017104, 247084353934700168), SpatialAnalysisToolVolumeGridGroup)
agcls.AgTypeNameMap["SpatialAnalysisToolVolumeGridGroup"] = SpatialAnalysisToolVolumeGridGroup

class SpatialAnalysisToolConditionGroup(SupportsDeleteCallback):
    """Access or create spatial conditions associated with a volume grid."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5105009531150323546, 886691369936677802),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolConditionGroup)
    def __iter__(self):
        """Create an iterator for the SpatialAnalysisToolConditionGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ISpatialAnalysisToolVolume":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "SpatialAnalysisToolConditionFactory":
        """Return a factory object used to create spatial condition components."""
        return self._intf.get_property(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ISpatialAnalysisToolVolume":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ISpatialAnalysisToolVolume":
        """Retrieve an volume from the collection by index."""
        return self._intf.invoke(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ISpatialAnalysisToolVolume":
        """Retrieve volume from the collection by name."""
        return self._intf.invoke(SpatialAnalysisToolConditionGroup._metadata, SpatialAnalysisToolConditionGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolConditionGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolConditionGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolConditionGroup, [SpatialAnalysisToolConditionGroup, ])

agcls.AgClassCatalog.add_catalog_entry((4930145607069579046, 10312486085103245186), SpatialAnalysisToolConditionGroup)
agcls.AgTypeNameMap["SpatialAnalysisToolConditionGroup"] = SpatialAnalysisToolConditionGroup

class SpatialAnalysisToolCalculationGroup(SupportsDeleteCallback):
    """Access or create VGT volume calcs associated with an object or a central body."""

    _num_methods = 9
    _vtable_offset = IDispatch._vtable_offset + IDispatch._num_methods
    _remove_method_offset = 1
    _get_context_method_offset = 2
    _contains_method_offset = 3
    _get_count_method_offset = 4
    _get_factory_method_offset = 5
    _item_method_offset = 6
    _get__new_enum_method_offset = 7
    _get_item_by_index_method_offset = 8
    _get_item_by_name_method_offset = 9
    _metadata = {
        "iid_data" : (5218042406034281573, 13159514031507062161),
        "vtable_reference" : IDispatch._vtable_offset + IDispatch._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationGroup)
    def __iter__(self):
        """Create an iterator for the SpatialAnalysisToolCalculationGroup object."""
        self.__dict__["_enumerator"] = self._new_enum
        self._enumerator.reset()
        return self
    def __next__(self) -> "ISpatialAnalysisToolSpatialCalculation":
        """Return the next element in the collection."""
        if self._enumerator is None:
            raise StopIteration
        nextval = self._enumerator.next()
        if nextval is None:
            raise StopIteration
        return nextval

    _remove_metadata = { "offset" : _remove_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    def remove(self, event_name:str) -> None:
        """Remove a specified element."""
        return self._intf.invoke(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._remove_metadata, event_name)

    _get_context_metadata = { "offset" : _get_context_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def context(self) -> "IAnalysisWorkbenchComponentContext":
        """Return a context object. The context can be used to find out which central body or STK object this instance is associated with."""
        return self._intf.get_property(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get_context_metadata)

    _contains_metadata = { "offset" : _contains_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.VariantBoolArg,) }
    def contains(self, name:str) -> bool:
        """Search for a an element with a given name. Returns false if the specified element does not exist."""
        return self._intf.invoke(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._contains_metadata, name, OutArg())

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.LongArg,) }
    @property
    def count(self) -> int:
        """Return a number of elements in the group."""
        return self._intf.get_property(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get_count_metadata)

    _get_factory_metadata = { "offset" : _get_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def factory(self) -> "SpatialAnalysisToolCalculationFactory":
        """Return a factory object used to create volume calc components."""
        return self._intf.get_property(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get_factory_metadata)

    _item_metadata = { "offset" : _item_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def item(self, index_or_name:typing.Any) -> "ISpatialAnalysisToolSpatialCalculation":
        """Return an element by name or at a specified position."""
        return self._intf.invoke(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._item_metadata, index_or_name, OutArg())

    _get__new_enum_metadata = { "offset" : _get__new_enum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IEnumVariantArg,) }
    @property
    def _new_enum(self) -> EnumeratorProxy:
        """Return a COM enumerator."""
        return self._intf.get_property(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get__new_enum_metadata)

    _get_item_by_index_metadata = { "offset" : _get_item_by_index_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_index(self, index:int) -> "ISpatialAnalysisToolSpatialCalculation":
        """Retrieve a volume calculation from the collection by index."""
        return self._intf.invoke(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get_item_by_index_metadata, index, OutArg())

    _get_item_by_name_metadata = { "offset" : _get_item_by_name_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def get_item_by_name(self, name:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Retrieve a volume calculation from the collection by name."""
        return self._intf.invoke(SpatialAnalysisToolCalculationGroup._metadata, SpatialAnalysisToolCalculationGroup._get_item_by_name_metadata, name, OutArg())

    __getitem__ = item


    _property_names[context] = "context"
    _property_names[count] = "count"
    _property_names[factory] = "factory"
    _property_names[_new_enum] = "_new_enum"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolCalculationGroup."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolCalculationGroup)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolCalculationGroup, [SpatialAnalysisToolCalculationGroup, ])

agcls.AgClassCatalog.add_catalog_entry((5357586751106003383, 7462844834759351198), SpatialAnalysisToolCalculationGroup)
agcls.AgTypeNameMap["SpatialAnalysisToolCalculationGroup"] = SpatialAnalysisToolCalculationGroup

class CalculationToolScalar(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Any scalar calculation that is not constant by construction."""
    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalar."""
        SupportsDeleteCallback.__init__(self)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalar, [ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5230397928531433355, 10269566397212942242), CalculationToolScalar)
agcls.AgTypeNameMap["CalculationToolScalar"] = CalculationToolScalar

class CalculationToolScalarAngle(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Scalar equal to angular displacement obtained from any angle in VGT."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_angle_method_offset = 1
    _set_input_angle_method_offset = 2
    _metadata = {
        "iid_data" : (4959187984038204723, 1310938748632609426),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarAngle)

    _get_input_angle_metadata = { "offset" : _get_input_angle_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_angle(self) -> "IVectorGeometryToolAngle":
        """The input angle, which is a VGT angle component. Note angle computation in VGT may involve more than just angular displacement value: in VGT angles may be drawn in 3D which requires knowledge and evaluation of supporting vectors."""
        return self._intf.get_property(CalculationToolScalarAngle._metadata, CalculationToolScalarAngle._get_input_angle_metadata)

    _set_input_angle_metadata = { "offset" : _set_input_angle_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAngle"),) }
    @input_angle.setter
    def input_angle(self, input_angle:"IVectorGeometryToolAngle") -> None:
        return self._intf.set_property(CalculationToolScalarAngle._metadata, CalculationToolScalarAngle._set_input_angle_metadata, input_angle)

    _property_names[input_angle] = "input_angle"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarAngle."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarAngle)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarAngle, [CalculationToolScalarAngle, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4904462145790055437, 5156190887739896966), CalculationToolScalarAngle)
agcls.AgTypeNameMap["CalculationToolScalarAngle"] = CalculationToolScalarAngle

class CalculationToolScalarAverage(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Integral of input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""

    _num_methods = 25
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_scalar_method_offset = 1
    _set_input_scalar_method_offset = 2
    _get_compute_as_average_method_offset = 3
    _set_compute_as_average_method_offset = 4
    _get_integration_window_type_method_offset = 5
    _set_integration_window_type_method_offset = 6
    _get_start_offset_method_offset = 7
    _set_start_offset_method_offset = 8
    _get_stop_offset_method_offset = 9
    _set_stop_offset_method_offset = 10
    _get_use_custom_time_limits_method_offset = 11
    _set_use_custom_time_limits_method_offset = 12
    _get_custom_time_limits_method_offset = 13
    _set_custom_time_limits_method_offset = 14
    _get_save_data_option_method_offset = 15
    _set_save_data_option_method_offset = 16
    _get_interpolation_method_offset = 17
    _set_interpolation_method_offset = 18
    _get_sampling_method_offset = 19
    _set_sampling_method_offset = 20
    _get_integral_method_offset = 21
    _set_integral_method_offset = 22
    _get_keep_constant_outside_time_limits_method_offset = 23
    _set_keep_constant_outside_time_limits_method_offset = 24
    _set_offsets_method_offset = 25
    _metadata = {
        "iid_data" : (5376371496447394548, 11623923147254190485),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarAverage)

    _get_input_scalar_metadata = { "offset" : _get_input_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_scalar(self) -> "ICalculationToolScalar":
        """The input scalar calculation."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_input_scalar_metadata)

    _set_input_scalar_metadata = { "offset" : _set_input_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @input_scalar.setter
    def input_scalar(self, input_scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_input_scalar_metadata, input_scalar)

    _get_compute_as_average_metadata = { "offset" : _get_compute_as_average_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def compute_as_average(self) -> bool:
        """Specify whether the resulting integral value is divided by its time span to generate average value instead of integral."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_compute_as_average_metadata)

    _set_compute_as_average_metadata = { "offset" : _set_compute_as_average_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @compute_as_average.setter
    def compute_as_average(self, compute_as_average:bool) -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_compute_as_average_metadata, compute_as_average)

    _get_integration_window_type_metadata = { "offset" : _get_integration_window_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @property
    def integration_window_type(self) -> "IntegrationWindowType":
        """The integration window, or accumulation, type."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_integration_window_type_metadata)

    _set_integration_window_type_metadata = { "offset" : _set_integration_window_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @integration_window_type.setter
    def integration_window_type(self, integration_window_type:"IntegrationWindowType") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_integration_window_type_metadata, integration_window_type)

    _get_start_offset_metadata = { "offset" : _get_start_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_offset(self) -> float:
        """Set the offset with respect to current time to define the start of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_start_offset_metadata)

    _set_start_offset_metadata = { "offset" : _set_start_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_offset.setter
    def start_offset(self, start_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_start_offset_metadata, start_offset)

    _get_stop_offset_metadata = { "offset" : _get_stop_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_offset(self) -> float:
        """Set the offset with respect to current time to define the stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_stop_offset_metadata)

    _set_stop_offset_metadata = { "offset" : _set_stop_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_offset.setter
    def stop_offset(self, stop_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_stop_offset_metadata, stop_offset)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Specify whether to use custom interval list (CustomTimeLimits)."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """The interval list within which the global minimum or maximum is sought. The default is the overall availability of host object."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_custom_time_limits_metadata, custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Set the value to determine if computed time of extremum is saved/loaded, or recomputed on load if necessary."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_save_data_option_metadata, save_data_option)

    _get_interpolation_metadata = { "offset" : _get_interpolation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interpolation(self) -> "IAnalysisWorkbenchInterpolator":
        """Specify whether to use Lagrange or Hermite interpolation. See STK help on interpolation."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_interpolation_metadata)

    _set_interpolation_metadata = { "offset" : _set_interpolation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchInterpolator"),) }
    @interpolation.setter
    def interpolation(self, interpolation:"IAnalysisWorkbenchInterpolator") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_interpolation_metadata, interpolation)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_sampling_metadata, sampling)

    _get_integral_metadata = { "offset" : _get_integral_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def integral(self) -> "IAnalysisWorkbenchIntegral":
        """The numerical integration method."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_integral_metadata)

    _set_integral_metadata = { "offset" : _set_integral_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchIntegral"),) }
    @integral.setter
    def integral(self, integral:"IAnalysisWorkbenchIntegral") -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_integral_metadata, integral)

    _get_keep_constant_outside_time_limits_metadata = { "offset" : _get_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def keep_constant_outside_time_limits(self) -> bool:
        """If true, the integral's integrand value is replaced by a constant 0 so that the integral remains constant over the gaps in integration."""
        return self._intf.get_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._get_keep_constant_outside_time_limits_metadata)

    _set_keep_constant_outside_time_limits_metadata = { "offset" : _set_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @keep_constant_outside_time_limits.setter
    def keep_constant_outside_time_limits(self, keep_constant_outside_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_keep_constant_outside_time_limits_metadata, keep_constant_outside_time_limits)

    _set_offsets_metadata = { "offset" : _set_offsets_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def set_offsets(self, start_offset:float, stop_offset:float) -> None:
        """Set the offsets with respect to current time to define the start and stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.invoke(CalculationToolScalarAverage._metadata, CalculationToolScalarAverage._set_offsets_metadata, start_offset, stop_offset)

    _property_names[input_scalar] = "input_scalar"
    _property_names[compute_as_average] = "compute_as_average"
    _property_names[integration_window_type] = "integration_window_type"
    _property_names[start_offset] = "start_offset"
    _property_names[stop_offset] = "stop_offset"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[interpolation] = "interpolation"
    _property_names[sampling] = "sampling"
    _property_names[integral] = "integral"
    _property_names[keep_constant_outside_time_limits] = "keep_constant_outside_time_limits"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarAverage."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarAverage)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarAverage, [CalculationToolScalarAverage, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5436686424833363725, 875804097364919458), CalculationToolScalarAverage)
agcls.AgTypeNameMap["CalculationToolScalarAverage"] = CalculationToolScalarAverage

class CalculationToolScalarConstant(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Constant scalar value of specified dimension."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_value_method_offset = 1
    _set_value_method_offset = 2
    _get_dimension_method_offset = 3
    _set_dimension_method_offset = 4
    _metadata = {
        "iid_data" : (5533910452542745513, 14034208071237068448),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarConstant)

    _get_value_metadata = { "offset" : _get_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def value(self) -> float:
        """A value which can be in any STK supported unit available for selected dimension."""
        return self._intf.get_property(CalculationToolScalarConstant._metadata, CalculationToolScalarConstant._get_value_metadata)

    _set_value_metadata = { "offset" : _set_value_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @value.setter
    def value(self, value:float) -> None:
        return self._intf.set_property(CalculationToolScalarConstant._metadata, CalculationToolScalarConstant._set_value_metadata, value)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """The dimension of the constant value, this can be any of the STK supported dimensions."""
        return self._intf.get_property(CalculationToolScalarConstant._metadata, CalculationToolScalarConstant._get_dimension_metadata)

    _set_dimension_metadata = { "offset" : _set_dimension_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @dimension.setter
    def dimension(self, dimension:str) -> None:
        return self._intf.set_property(CalculationToolScalarConstant._metadata, CalculationToolScalarConstant._set_dimension_metadata, dimension)

    _property_names[value] = "value"
    _property_names[dimension] = "dimension"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarConstant."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarConstant)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarConstant, [CalculationToolScalarConstant, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5761343657246828391, 12742771671638689976), CalculationToolScalarConstant)
agcls.AgTypeNameMap["CalculationToolScalarConstant"] = CalculationToolScalarConstant

class CalculationToolScalarCustom(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A calc scalar based on a scripted algorithm in MATLAB (.m or .dll), Perl or VBScript to define its value and rate."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_filename_method_offset = 1
    _set_filename_method_offset = 2
    _reload_method_offset = 3
    _get_invalidate_on_execution_error_method_offset = 4
    _set_invalidate_on_execution_error_method_offset = 5
    _metadata = {
        "iid_data" : (5409179971415946885, 10112159419728251554),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarCustom)

    _get_filename_metadata = { "offset" : _get_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def filename(self) -> str:
        """A path to MATLAB (.m or .dll), Perl or VBScript file."""
        return self._intf.get_property(CalculationToolScalarCustom._metadata, CalculationToolScalarCustom._get_filename_metadata)

    _set_filename_metadata = { "offset" : _set_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @filename.setter
    def filename(self, filename:str) -> None:
        return self._intf.set_property(CalculationToolScalarCustom._metadata, CalculationToolScalarCustom._set_filename_metadata, filename)

    _reload_metadata = { "offset" : _reload_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reload(self) -> None:
        """Reload the file specified with Filename property."""
        return self._intf.invoke(CalculationToolScalarCustom._metadata, CalculationToolScalarCustom._reload_metadata, )

    _get_invalidate_on_execution_error_metadata = { "offset" : _get_invalidate_on_execution_error_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def invalidate_on_execution_error(self) -> bool:
        """Specify InvalidOnExecError flag for a custom scalar."""
        return self._intf.get_property(CalculationToolScalarCustom._metadata, CalculationToolScalarCustom._get_invalidate_on_execution_error_metadata)

    _set_invalidate_on_execution_error_metadata = { "offset" : _set_invalidate_on_execution_error_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @invalidate_on_execution_error.setter
    def invalidate_on_execution_error(self, invalidate_on_exec_error:bool) -> None:
        return self._intf.set_property(CalculationToolScalarCustom._metadata, CalculationToolScalarCustom._set_invalidate_on_execution_error_metadata, invalidate_on_exec_error)

    _property_names[filename] = "filename"
    _property_names[invalidate_on_execution_error] = "invalidate_on_execution_error"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarCustom."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarCustom)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarCustom, [CalculationToolScalarCustom, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5177995385747099113, 10709837510467460012), CalculationToolScalarCustom)
agcls.AgTypeNameMap["CalculationToolScalarCustom"] = CalculationToolScalarCustom

class CalculationToolScalarCustomInlineScript(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A calc scalar based on using an inline scripted algorithm in MATLAB, Perl, VBScript or JScript to define its value and rate."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_script_type_method_offset = 1
    _set_script_type_method_offset = 2
    _get_value_function_method_offset = 3
    _set_value_function_method_offset = 4
    _get_derivative_function_method_offset = 5
    _set_derivative_function_method_offset = 6
    _get_dimension_method_offset = 7
    _set_dimension_method_offset = 8
    _get_all_arguments_method_offset = 9
    _set_all_arguments_method_offset = 10
    _metadata = {
        "iid_data" : (4764386451504013000, 3625316982411733925),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarCustomInlineScript)

    _get_script_type_metadata = { "offset" : _get_script_type_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def script_type(self) -> str:
        """Script type allowed {JScript | Matlab | VBScript}."""
        return self._intf.get_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._get_script_type_metadata)

    _set_script_type_metadata = { "offset" : _set_script_type_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @script_type.setter
    def script_type(self, script_type:str) -> None:
        return self._intf.set_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._set_script_type_metadata, script_type)

    _get_value_function_metadata = { "offset" : _get_value_function_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def value_function(self) -> str:
        """The Value function to be evaluated."""
        return self._intf.get_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._get_value_function_metadata)

    _set_value_function_metadata = { "offset" : _set_value_function_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @value_function.setter
    def value_function(self, value_function:str) -> None:
        return self._intf.set_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._set_value_function_metadata, value_function)

    _get_derivative_function_metadata = { "offset" : _get_derivative_function_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def derivative_function(self) -> str:
        """The Derivative function to be evaluated."""
        return self._intf.get_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._get_derivative_function_metadata)

    _set_derivative_function_metadata = { "offset" : _set_derivative_function_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @derivative_function.setter
    def derivative_function(self, derivative_function:str) -> None:
        return self._intf.set_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._set_derivative_function_metadata, derivative_function)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Dimension name."""
        return self._intf.get_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._get_dimension_metadata)

    _set_dimension_metadata = { "offset" : _set_dimension_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @dimension.setter
    def dimension(self, dimension:str) -> None:
        return self._intf.set_property(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._set_dimension_metadata, dimension)

    _get_all_arguments_metadata = { "offset" : _get_all_arguments_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_arguments(self) -> list:
        """Return the list of arguments."""
        return self._intf.invoke(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._get_all_arguments_metadata, OutArg())

    _set_all_arguments_metadata = { "offset" : _set_all_arguments_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_all_arguments(self, calc_list:list) -> None:
        """Set the list of arguments."""
        return self._intf.invoke(CalculationToolScalarCustomInlineScript._metadata, CalculationToolScalarCustomInlineScript._set_all_arguments_metadata, calc_list)

    _property_names[script_type] = "script_type"
    _property_names[value_function] = "value_function"
    _property_names[derivative_function] = "derivative_function"
    _property_names[dimension] = "dimension"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarCustomInlineScript."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarCustomInlineScript)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarCustomInlineScript, [CalculationToolScalarCustomInlineScript, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5758632709114625099, 1419255989428412582), CalculationToolScalarCustomInlineScript)
agcls.AgTypeNameMap["CalculationToolScalarCustomInlineScript"] = CalculationToolScalarCustomInlineScript

class CalculationToolScalarDataElement(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Any time-dependent data element from STK data providers available for parent STK object."""

    _num_methods = 15
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_data_provider_method_offset = 1
    _get_element_name_method_offset = 2
    _get_group_method_offset = 3
    _get_interpolation_method_offset = 4
    _set_interpolation_method_offset = 5
    _get_sampling_method_offset = 6
    _set_sampling_method_offset = 7
    _get_use_samples_method_offset = 8
    _set_use_samples_method_offset = 9
    _get_save_data_option_method_offset = 10
    _set_save_data_option_method_offset = 11
    _set_method_offset = 12
    _set_with_group_method_offset = 13
    _get_invalid_data_indicator_method_offset = 14
    _set_invalid_data_indicator_method_offset = 15
    _metadata = {
        "iid_data" : (5684166753534512598, 14848638846349344702),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarDataElement)

    _get_data_provider_metadata = { "offset" : _get_data_provider_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def data_provider(self) -> str:
        """The name of the data provider."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_data_provider_metadata)

    _get_element_name_metadata = { "offset" : _get_element_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def element_name(self) -> str:
        """The name of the data element within the data provider."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_element_name_metadata)

    _get_group_metadata = { "offset" : _get_group_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def group(self) -> str:
        """A group name the data element is a part of. If the element is not a part of a group, the property will return an empty string."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_group_metadata)

    _get_interpolation_metadata = { "offset" : _get_interpolation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interpolation(self) -> "IAnalysisWorkbenchInterpolator":
        """Specify whether to use Lagrange or Hermite interpolation. See STK help on interpolation."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_interpolation_metadata)

    _set_interpolation_metadata = { "offset" : _set_interpolation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchInterpolator"),) }
    @interpolation.setter
    def interpolation(self, interpolation:"IAnalysisWorkbenchInterpolator") -> None:
        return self._intf.set_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_interpolation_metadata, interpolation)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """Relative tolerance uses a combination of relative and absolute changes in scalar values between samples. Curvature tolerance also uses changes in slope between samples."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_sampling_metadata, sampling)

    _get_use_samples_metadata = { "offset" : _get_use_samples_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_samples(self) -> bool:
        """If set to true, selected data provider is presampled over its entire availability span using sampling method specified in Advanced options..."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_use_samples_metadata)

    _set_use_samples_metadata = { "offset" : _set_use_samples_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_samples.setter
    def use_samples(self, use_samples:bool) -> None:
        return self._intf.set_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_use_samples_metadata, use_samples)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Determine if computed samples are saved/loaded, otherwise if using samples they are recomputed on load."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_save_data_option_metadata, save_data_option)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set(self, data_provider:str, element_name:str) -> None:
        """Set the data provider and the element name."""
        return self._intf.invoke(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_metadata, data_provider, element_name)

    _set_with_group_metadata = { "offset" : _set_with_group_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set_with_group(self, data_provider:str, type_name:str, element_name:str) -> None:
        """Set the data provider name, the element name, and data provider type name."""
        return self._intf.invoke(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_with_group_metadata, data_provider, type_name, element_name)

    _get_invalid_data_indicator_metadata = { "offset" : _get_invalid_data_indicator_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def invalid_data_indicator(self) -> float:
        """Set the value to display in a report or graph when the actual value is not a valid real number."""
        return self._intf.get_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._get_invalid_data_indicator_metadata)

    _set_invalid_data_indicator_metadata = { "offset" : _set_invalid_data_indicator_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @invalid_data_indicator.setter
    def invalid_data_indicator(self, invalid_data_indicator:float) -> None:
        return self._intf.set_property(CalculationToolScalarDataElement._metadata, CalculationToolScalarDataElement._set_invalid_data_indicator_metadata, invalid_data_indicator)

    _property_names[data_provider] = "data_provider"
    _property_names[element_name] = "element_name"
    _property_names[group] = "group"
    _property_names[interpolation] = "interpolation"
    _property_names[sampling] = "sampling"
    _property_names[use_samples] = "use_samples"
    _property_names[save_data_option] = "save_data_option"
    _property_names[invalid_data_indicator] = "invalid_data_indicator"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarDataElement."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarDataElement)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarDataElement, [CalculationToolScalarDataElement, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5321856052550789231, 6123381429176273038), CalculationToolScalarDataElement)
agcls.AgTypeNameMap["CalculationToolScalarDataElement"] = CalculationToolScalarDataElement

class CalculationToolScalarDerivative(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Derivative of an input scalar calculation."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_scalar_method_offset = 1
    _set_scalar_method_offset = 2
    _get_differencing_time_step_method_offset = 3
    _set_differencing_time_step_method_offset = 4
    _get_force_use_of_numerical_differences_method_offset = 5
    _set_force_use_of_numerical_differences_method_offset = 6
    _metadata = {
        "iid_data" : (5439408000144674159, 16540285713240218546),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarDerivative)

    _get_scalar_metadata = { "offset" : _get_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scalar(self) -> "ICalculationToolScalar":
        """The input scalar component used to compute the derivative."""
        return self._intf.get_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._get_scalar_metadata)

    _set_scalar_metadata = { "offset" : _set_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @scalar.setter
    def scalar(self, scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._set_scalar_metadata, scalar)

    _get_differencing_time_step_metadata = { "offset" : _get_differencing_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def differencing_time_step(self) -> float:
        """The time step used, if necessary, in numerical evaluation of derivatives using central differencing."""
        return self._intf.get_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._get_differencing_time_step_metadata)

    _set_differencing_time_step_metadata = { "offset" : _set_differencing_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @differencing_time_step.setter
    def differencing_time_step(self, differencing_time_step:float) -> None:
        return self._intf.set_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._set_differencing_time_step_metadata, differencing_time_step)

    _get_force_use_of_numerical_differences_metadata = { "offset" : _get_force_use_of_numerical_differences_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def force_use_of_numerical_differences(self) -> bool:
        """Force the use of numerical differences even if the derivative can be computed analytically."""
        return self._intf.get_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._get_force_use_of_numerical_differences_metadata)

    _set_force_use_of_numerical_differences_metadata = { "offset" : _set_force_use_of_numerical_differences_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @force_use_of_numerical_differences.setter
    def force_use_of_numerical_differences(self, force_use_of_numerical_differences:bool) -> None:
        return self._intf.set_property(CalculationToolScalarDerivative._metadata, CalculationToolScalarDerivative._set_force_use_of_numerical_differences_metadata, force_use_of_numerical_differences)

    _property_names[scalar] = "scalar"
    _property_names[differencing_time_step] = "differencing_time_step"
    _property_names[force_use_of_numerical_differences] = "force_use_of_numerical_differences"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarDerivative."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarDerivative)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarDerivative, [CalculationToolScalarDerivative, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5415006546127808424, 17328826830833065386), CalculationToolScalarDerivative)
agcls.AgTypeNameMap["CalculationToolScalarDerivative"] = CalculationToolScalarDerivative

class CalculationToolScalarDotProduct(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Dot product between two vectors."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_vector_a_method_offset = 1
    _set_vector_a_method_offset = 2
    _get_normalize_vector_a_method_offset = 3
    _set_normalize_vector_a_method_offset = 4
    _get_vector_b_method_offset = 5
    _set_vector_b_method_offset = 6
    _get_normalize_vector_b_method_offset = 7
    _set_normalize_vector_b_method_offset = 8
    _get_dimension_method_offset = 9
    _set_dimension_method_offset = 10
    _metadata = {
        "iid_data" : (4701425013565595554, 16894540468255929531),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarDotProduct)

    _get_vector_a_metadata = { "offset" : _get_vector_a_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vector_a(self) -> "IVectorGeometryToolVector":
        """First vector."""
        return self._intf.get_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._get_vector_a_metadata)

    _set_vector_a_metadata = { "offset" : _set_vector_a_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @vector_a.setter
    def vector_a(self, vector_a:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._set_vector_a_metadata, vector_a)

    _get_normalize_vector_a_metadata = { "offset" : _get_normalize_vector_a_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def normalize_vector_a(self) -> bool:
        """Whether to normalize vector A."""
        return self._intf.get_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._get_normalize_vector_a_metadata)

    _set_normalize_vector_a_metadata = { "offset" : _set_normalize_vector_a_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @normalize_vector_a.setter
    def normalize_vector_a(self, normalize_vector_a:bool) -> None:
        return self._intf.set_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._set_normalize_vector_a_metadata, normalize_vector_a)

    _get_vector_b_metadata = { "offset" : _get_vector_b_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vector_b(self) -> "IVectorGeometryToolVector":
        """Second vector."""
        return self._intf.get_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._get_vector_b_metadata)

    _set_vector_b_metadata = { "offset" : _set_vector_b_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @vector_b.setter
    def vector_b(self, vector_b:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._set_vector_b_metadata, vector_b)

    _get_normalize_vector_b_metadata = { "offset" : _get_normalize_vector_b_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def normalize_vector_b(self) -> bool:
        """Whether to normalize vector B."""
        return self._intf.get_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._get_normalize_vector_b_metadata)

    _set_normalize_vector_b_metadata = { "offset" : _set_normalize_vector_b_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @normalize_vector_b.setter
    def normalize_vector_b(self, normalize_vector_b:bool) -> None:
        return self._intf.set_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._set_normalize_vector_b_metadata, normalize_vector_b)

    _get_dimension_metadata = { "offset" : _get_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def dimension(self) -> str:
        """Return a unit of measure, i.e. 'Angle', 'Distance', etc."""
        return self._intf.get_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._get_dimension_metadata)

    _set_dimension_metadata = { "offset" : _set_dimension_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @dimension.setter
    def dimension(self, dimension:str) -> None:
        return self._intf.set_property(CalculationToolScalarDotProduct._metadata, CalculationToolScalarDotProduct._set_dimension_metadata, dimension)

    _property_names[vector_a] = "vector_a"
    _property_names[normalize_vector_a] = "normalize_vector_a"
    _property_names[vector_b] = "vector_b"
    _property_names[normalize_vector_b] = "normalize_vector_b"
    _property_names[dimension] = "dimension"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarDotProduct."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarDotProduct)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarDotProduct, [CalculationToolScalarDotProduct, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4792387092728447134, 3158224088086022029), CalculationToolScalarDotProduct)
agcls.AgTypeNameMap["CalculationToolScalarDotProduct"] = CalculationToolScalarDotProduct

class CalculationToolScalarElapsedTime(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Time elapsed since the reference time instant. Negative if in the past."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_time_instant_method_offset = 1
    _set_reference_time_instant_method_offset = 2
    _metadata = {
        "iid_data" : (5303668177545430544, 4808828999605306252),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarElapsedTime)

    _get_reference_time_instant_metadata = { "offset" : _get_reference_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_time_instant(self) -> "ITimeToolInstant":
        """The reference time instant."""
        return self._intf.get_property(CalculationToolScalarElapsedTime._metadata, CalculationToolScalarElapsedTime._get_reference_time_instant_metadata)

    _set_reference_time_instant_metadata = { "offset" : _set_reference_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @reference_time_instant.setter
    def reference_time_instant(self, reference_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(CalculationToolScalarElapsedTime._metadata, CalculationToolScalarElapsedTime._set_reference_time_instant_metadata, reference_time_instant)

    _property_names[reference_time_instant] = "reference_time_instant"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarElapsedTime."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarElapsedTime)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarElapsedTime, [CalculationToolScalarElapsedTime, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4848170193154715426, 2061643609355890078), CalculationToolScalarElapsedTime)
agcls.AgTypeNameMap["CalculationToolScalarElapsedTime"] = CalculationToolScalarElapsedTime

class CalculationToolScalarFactory(SupportsDeleteCallback):
    """The factory creates scalar calculation components."""

    _num_methods = 24
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_available_plugin_display_names_method_offset = 1
    _create_method_offset = 2
    _create_angle_method_offset = 3
    _create_fixed_at_time_instant_method_offset = 4
    _create_constant_method_offset = 5
    _create_data_element_method_offset = 6
    _create_data_element_within_group_method_offset = 7
    _create_derivative_method_offset = 8
    _create_elapsed_time_method_offset = 9
    _create_from_file_method_offset = 10
    _create_function_method_offset = 11
    _create_integral_method_offset = 12
    _create_function_of_2_variables_method_offset = 13
    _create_vector_magnitude_method_offset = 14
    _create_plugin_from_display_name_method_offset = 15
    _is_type_supported_method_offset = 16
    _create_from_custom_script_method_offset = 17
    _create_surface_distance_between_points_method_offset = 18
    _create_dot_product_method_offset = 19
    _create_vector_component_method_offset = 20
    _create_average_method_offset = 21
    _create_standard_deviation_method_offset = 22
    _create_calculation_along_trajectory_method_offset = 23
    _create_custom_inline_script_method_offset = 24
    _metadata = {
        "iid_data" : (5499352364829801528, 7117583500936800935),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarFactory)

    _get_available_plugin_display_names_metadata = { "offset" : _get_available_plugin_display_names_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_plugin_display_names(self) -> list:
        """An array of display names associated with available scalar calculation plugins. The elements of the array are strings. Display names are used to create Calc scalars based on COM plugins using CreateCalcScalarPluginFromDisplayName method."""
        return self._intf.get_property(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._get_available_plugin_display_names_metadata)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(CalculationScalarType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"CalculationScalarType") -> "ICalculationToolScalar":
        """Create and registers a scalar calculation using specified name, description, and type."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_metadata, name, description, type, OutArg())

    _create_angle_metadata = { "offset" : _create_angle_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_angle(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation equal to angular displacement obtained from any angle in VGT."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_angle_metadata, name, description, OutArg())

    _create_fixed_at_time_instant_metadata = { "offset" : _create_fixed_at_time_instant_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed_at_time_instant(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation defined by evaluating the input scalar calculation at the specified reference time instant."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_fixed_at_time_instant_metadata, name, description, OutArg())

    _create_constant_metadata = { "offset" : _create_constant_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_constant(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation of constant value of the specified dimension."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_constant_metadata, name, description, OutArg())

    _create_data_element_metadata = { "offset" : _create_data_element_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_data_element(self, name:str, description:str, data_provider:str, element_name:str) -> "ICalculationToolScalar":
        """Create a scalar calculation defined from a time-dependent data element from STK data providers available for parent STK object."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_data_element_metadata, name, description, data_provider, element_name, OutArg())

    _create_data_element_within_group_metadata = { "offset" : _create_data_element_within_group_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_data_element_within_group(self, name:str, description:str, data_provider:str, group_name:str, element_name:str) -> "ICalculationToolScalar":
        """Create a scalar calculation defined from a time-dependent data element from STK data providers available for parent STK object."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_data_element_within_group_metadata, name, description, data_provider, group_name, element_name, OutArg())

    _create_derivative_metadata = { "offset" : _create_derivative_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_derivative(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is the derivative of an input scalar calculation."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_derivative_metadata, name, description, OutArg())

    _create_elapsed_time_metadata = { "offset" : _create_elapsed_time_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_elapsed_time(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is the time elapsed since a reference time instant."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_elapsed_time_metadata, name, description, OutArg())

    _create_from_file_metadata = { "offset" : _create_from_file_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_file(self, name:str, description:str, filepath:str) -> "ICalculationToolScalar":
        """Create scalar calculation specified by external data file."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_from_file_metadata, name, description, filepath, OutArg())

    _create_function_metadata = { "offset" : _create_function_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_function(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is defined by performing the specified function on the input scalar or time instant."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_function_metadata, name, description, OutArg())

    _create_integral_metadata = { "offset" : _create_integral_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_integral(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is the integral of an input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_integral_metadata, name, description, OutArg())

    _create_function_of_2_variables_metadata = { "offset" : _create_function_of_2_variables_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_function_of_2_variables(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is defined by performing a function(x,y) on two scalar arguments."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_function_of_2_variables_metadata, name, description, OutArg())

    _create_vector_magnitude_metadata = { "offset" : _create_vector_magnitude_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_vector_magnitude(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation equal to the magnitude of a specified vector."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_vector_magnitude_metadata, name, description, OutArg())

    _create_plugin_from_display_name_metadata = { "offset" : _create_plugin_from_display_name_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_plugin_from_display_name(self, name:str, description:str, display_name:str) -> "ICalculationToolScalar":
        """Create a scalar calculation based on a COM plugin. For information how to implement and register VGT plugins, see <topic name='Engine Plugins: COM-based Engine Plugin Components'>COM-based Engine Plugins.</topic>."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_plugin_from_display_name_metadata, name, description, display_name, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(CalculationScalarType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"CalculationScalarType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._is_type_supported_metadata, type, OutArg())

    _create_from_custom_script_metadata = { "offset" : _create_from_custom_script_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_custom_script(self, name:str, description:str, filepath:str) -> "ICalculationToolScalar":
        """Create a calc scalar calculation that uses scripted algorithm in MATLAB (.m or .dll), Perl or VBScript to define its value and rate."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_from_custom_script_metadata, name, description, filepath, OutArg())

    _create_surface_distance_between_points_metadata = { "offset" : _create_surface_distance_between_points_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_surface_distance_between_points(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a calc scalar calculation that is surface distance along the specified central body ellipsoid between two points (or their respective projections if specified at altitude)."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_surface_distance_between_points_metadata, name, description, OutArg())

    _create_dot_product_metadata = { "offset" : _create_dot_product_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_dot_product(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is defined by a dot product between two vectors."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_dot_product_metadata, name, description, OutArg())

    _create_vector_component_metadata = { "offset" : _create_vector_component_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_vector_component(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is defined by a specified component of a vector when resolved in specified axes."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_vector_component_metadata, name, description, OutArg())

    _create_average_metadata = { "offset" : _create_average_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_average(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is the average of an input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_average_metadata, name, description, OutArg())

    _create_standard_deviation_metadata = { "offset" : _create_standard_deviation_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_standard_deviation(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation that is the standard deviation of an input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_standard_deviation_metadata, name, description, OutArg())

    _create_calculation_along_trajectory_metadata = { "offset" : _create_calculation_along_trajectory_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_calculation_along_trajectory(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a scalar calculation along trajectory."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_calculation_along_trajectory_metadata, name, description, OutArg())

    _create_custom_inline_script_metadata = { "offset" : _create_custom_inline_script_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_custom_inline_script(self, name:str, description:str) -> "ICalculationToolScalar":
        """Create a custom inline script scalar."""
        return self._intf.invoke(CalculationToolScalarFactory._metadata, CalculationToolScalarFactory._create_custom_inline_script_metadata, name, description, OutArg())

    _property_names[available_plugin_display_names] = "available_plugin_display_names"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarFactory, [CalculationToolScalarFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5603183957251412242, 16723870511898521754), CalculationToolScalarFactory)
agcls.AgTypeNameMap["CalculationToolScalarFactory"] = CalculationToolScalarFactory

class CalculationToolScalarFile(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Tabulated scalar calculation data loaded from specified file - a file with .csc extension."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_filename_method_offset = 1
    _set_filename_method_offset = 2
    _reload_method_offset = 3
    _get_file_span_method_offset = 4
    _get_file_interpolation_type_method_offset = 5
    _set_file_interpolation_type_method_offset = 6
    _get_file_interpolation_order_method_offset = 7
    _set_file_interpolation_order_method_offset = 8
    _get_use_native_file_interpolation_settings_method_offset = 9
    _set_use_native_file_interpolation_settings_method_offset = 10
    _metadata = {
        "iid_data" : (5153202141983971278, 8810191830175553723),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarFile)

    _get_filename_metadata = { "offset" : _get_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def filename(self) -> str:
        """The path to an ASCII file with .csc extension."""
        return self._intf.get_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._get_filename_metadata)

    _set_filename_metadata = { "offset" : _set_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @filename.setter
    def filename(self, filename:str) -> None:
        return self._intf.set_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._set_filename_metadata, filename)

    _reload_metadata = { "offset" : _reload_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reload(self) -> None:
        """Reload the file specified with Filename property."""
        return self._intf.invoke(CalculationToolScalarFile._metadata, CalculationToolScalarFile._reload_metadata, )

    _get_file_span_metadata = { "offset" : _get_file_span_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_file_span(self) -> "TimeToolTimeIntervalResult":
        """Compute the interval time span of the file."""
        return self._intf.invoke(CalculationToolScalarFile._metadata, CalculationToolScalarFile._get_file_span_metadata, OutArg())

    _get_file_interpolation_type_metadata = { "offset" : _get_file_interpolation_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(FileInterpolatorType),) }
    @property
    def file_interpolation_type(self) -> "FileInterpolatorType":
        """The interpolation method used with the data."""
        return self._intf.get_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._get_file_interpolation_type_metadata)

    _set_file_interpolation_type_metadata = { "offset" : _set_file_interpolation_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(FileInterpolatorType),) }
    @file_interpolation_type.setter
    def file_interpolation_type(self, file_interpolation_type:"FileInterpolatorType") -> None:
        return self._intf.set_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._set_file_interpolation_type_metadata, file_interpolation_type)

    _get_file_interpolation_order_metadata = { "offset" : _get_file_interpolation_order_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def file_interpolation_order(self) -> int:
        """The interpolation order used with the interpolation method to interrogate the data."""
        return self._intf.get_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._get_file_interpolation_order_metadata)

    _set_file_interpolation_order_metadata = { "offset" : _set_file_interpolation_order_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @file_interpolation_order.setter
    def file_interpolation_order(self, file_interpolation_order:int) -> None:
        return self._intf.set_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._set_file_interpolation_order_metadata, file_interpolation_order)

    _get_use_native_file_interpolation_settings_metadata = { "offset" : _get_use_native_file_interpolation_settings_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_native_file_interpolation_settings(self) -> bool:
        """Flag indicating whether the interpolation method and order settings specified within the file, if any, will be honored."""
        return self._intf.get_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._get_use_native_file_interpolation_settings_metadata)

    _set_use_native_file_interpolation_settings_metadata = { "offset" : _set_use_native_file_interpolation_settings_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_native_file_interpolation_settings.setter
    def use_native_file_interpolation_settings(self, use_native_file_interpolation_settings:bool) -> None:
        return self._intf.set_property(CalculationToolScalarFile._metadata, CalculationToolScalarFile._set_use_native_file_interpolation_settings_metadata, use_native_file_interpolation_settings)

    _property_names[filename] = "filename"
    _property_names[file_interpolation_type] = "file_interpolation_type"
    _property_names[file_interpolation_order] = "file_interpolation_order"
    _property_names[use_native_file_interpolation_settings] = "use_native_file_interpolation_settings"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarFile."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarFile)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarFile, [CalculationToolScalarFile, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5148097650078253983, 10233293468354629821), CalculationToolScalarFile)
agcls.AgTypeNameMap["CalculationToolScalarFile"] = CalculationToolScalarFile

class CalculationToolScalarFixedAtTimeInstant(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Constant scalar created by evaluating the input scalar calculation at the specified reference time instant. Undefined if original scalar is not available at specified time or if reference time instant is undefined."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_scalar_method_offset = 1
    _set_input_scalar_method_offset = 2
    _get_reference_time_instant_method_offset = 3
    _set_reference_time_instant_method_offset = 4
    _metadata = {
        "iid_data" : (5116980743969883807, 14308846120555093929),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarFixedAtTimeInstant)

    _get_input_scalar_metadata = { "offset" : _get_input_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_scalar(self) -> "ICalculationToolScalar":
        """The input scalar component."""
        return self._intf.get_property(CalculationToolScalarFixedAtTimeInstant._metadata, CalculationToolScalarFixedAtTimeInstant._get_input_scalar_metadata)

    _set_input_scalar_metadata = { "offset" : _set_input_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @input_scalar.setter
    def input_scalar(self, input_scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarFixedAtTimeInstant._metadata, CalculationToolScalarFixedAtTimeInstant._set_input_scalar_metadata, input_scalar)

    _get_reference_time_instant_metadata = { "offset" : _get_reference_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_time_instant(self) -> "ITimeToolInstant":
        """The reference time instant."""
        return self._intf.get_property(CalculationToolScalarFixedAtTimeInstant._metadata, CalculationToolScalarFixedAtTimeInstant._get_reference_time_instant_metadata)

    _set_reference_time_instant_metadata = { "offset" : _set_reference_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @reference_time_instant.setter
    def reference_time_instant(self, reference_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(CalculationToolScalarFixedAtTimeInstant._metadata, CalculationToolScalarFixedAtTimeInstant._set_reference_time_instant_metadata, reference_time_instant)

    _property_names[input_scalar] = "input_scalar"
    _property_names[reference_time_instant] = "reference_time_instant"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarFixedAtTimeInstant."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarFixedAtTimeInstant)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarFixedAtTimeInstant, [CalculationToolScalarFixedAtTimeInstant, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4932699577219865707, 13915785772631940010), CalculationToolScalarFixedAtTimeInstant)
agcls.AgTypeNameMap["CalculationToolScalarFixedAtTimeInstant"] = CalculationToolScalarFixedAtTimeInstant

class CalculationToolScalarFunction(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by performing the specified function on the input scalar or time instant."""

    _num_methods = 31
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_calculation_scalar_method_offset = 1
    _set_use_calculation_scalar_method_offset = 2
    _get_input_scalar_method_offset = 3
    _set_input_scalar_method_offset = 4
    _get_input_time_method_offset = 5
    _set_input_time_method_offset = 6
    _get_input_time_units_method_offset = 7
    _set_input_time_units_method_offset = 8
    _get_coefficient_a_method_offset = 9
    _set_coefficient_a_method_offset = 10
    _get_coefficient_b_method_offset = 11
    _set_coefficient_b_method_offset = 12
    _get_coefficient_c_method_offset = 13
    _set_coefficient_c_method_offset = 14
    _get_coefficient_d_method_offset = 15
    _set_coefficient_d_method_offset = 16
    _get_coefficients_method_offset = 17
    _set_coefficients_method_offset = 18
    _get_selected_function_method_offset = 19
    _set_selected_function_method_offset = 20
    _get_available_functions_method_offset = 21
    _get_inherit_dimension_from_input_method_offset = 22
    _set_inherit_dimension_from_input_method_offset = 23
    _get_output_dimension_method_offset = 24
    _set_output_dimension_method_offset = 25
    _get_output_units_method_offset = 26
    _set_output_units_method_offset = 27
    _get_sampling_method_offset = 28
    _set_sampling_method_offset = 29
    _get_convergence_method_offset = 30
    _set_convergence_method_offset = 31
    _metadata = {
        "iid_data" : (5290175361666189224, 17475337594760058546),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarFunction)

    _get_use_calculation_scalar_metadata = { "offset" : _get_use_calculation_scalar_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_calculation_scalar(self) -> bool:
        """Specify whether to use the input scalar calculation or the time elapsed from the input time instant. Set to true to use the scalar."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_use_calculation_scalar_metadata)

    _set_use_calculation_scalar_metadata = { "offset" : _set_use_calculation_scalar_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_calculation_scalar.setter
    def use_calculation_scalar(self, use_scalar:bool) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_use_calculation_scalar_metadata, use_scalar)

    _get_input_scalar_metadata = { "offset" : _get_input_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_scalar(self) -> "ICalculationToolScalar":
        """The input scalar calculation (used if UseScalar is true). The UseScalar property should be set to true before this property can be set."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_input_scalar_metadata)

    _set_input_scalar_metadata = { "offset" : _set_input_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @input_scalar.setter
    def input_scalar(self, input_scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_input_scalar_metadata, input_scalar)

    _get_input_time_metadata = { "offset" : _get_input_time_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_time(self) -> "ITimeToolInstant":
        """The input time instant (used if UseScalar is false)."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_input_time_metadata)

    _set_input_time_metadata = { "offset" : _set_input_time_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @input_time.setter
    def input_time(self, input_time:"ITimeToolInstant") -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_input_time_metadata, input_time)

    _get_input_time_units_metadata = { "offset" : _get_input_time_units_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def input_time_units(self) -> str:
        """The input time unit to interpret input time."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_input_time_units_metadata)

    _set_input_time_units_metadata = { "offset" : _set_input_time_units_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @input_time_units.setter
    def input_time_units(self, input_unit:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_input_time_units_metadata, input_unit)

    _get_coefficient_a_metadata = { "offset" : _get_coefficient_a_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_a(self) -> float:
        """The constant coefficient A."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_coefficient_a_metadata)

    _set_coefficient_a_metadata = { "offset" : _set_coefficient_a_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_a.setter
    def coefficient_a(self, a:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_coefficient_a_metadata, a)

    _get_coefficient_b_metadata = { "offset" : _get_coefficient_b_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_b(self) -> float:
        """The constant coefficient B."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_coefficient_b_metadata)

    _set_coefficient_b_metadata = { "offset" : _set_coefficient_b_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_b.setter
    def coefficient_b(self, b:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_coefficient_b_metadata, b)

    _get_coefficient_c_metadata = { "offset" : _get_coefficient_c_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_c(self) -> float:
        """The constant coefficient C."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_coefficient_c_metadata)

    _set_coefficient_c_metadata = { "offset" : _set_coefficient_c_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_c.setter
    def coefficient_c(self, c:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_coefficient_c_metadata, c)

    _get_coefficient_d_metadata = { "offset" : _get_coefficient_d_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_d(self) -> float:
        """The constant coefficient D."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_coefficient_d_metadata)

    _set_coefficient_d_metadata = { "offset" : _set_coefficient_d_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_d.setter
    def coefficient_d(self, d:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_coefficient_d_metadata, d)

    _get_coefficients_metadata = { "offset" : _get_coefficients_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def coefficients(self) -> list:
        """The array of constant coefficients, whose dimension and units are determined by those of input and output."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_coefficients_metadata)

    _set_coefficients_metadata = { "offset" : _set_coefficients_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @coefficients.setter
    def coefficients(self, coefficients:list) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_coefficients_metadata, coefficients)

    _get_selected_function_metadata = { "offset" : _get_selected_function_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def selected_function(self) -> str:
        """The function which will use the input scalar or time instant in some combination with the constant coefficients A, B, C, D."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_selected_function_metadata)

    _set_selected_function_metadata = { "offset" : _set_selected_function_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @selected_function.setter
    def selected_function(self, selected_function:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_selected_function_metadata, selected_function)

    _get_available_functions_metadata = { "offset" : _get_available_functions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_functions(self) -> list:
        """Get the available function names."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_available_functions_metadata)

    _get_inherit_dimension_from_input_metadata = { "offset" : _get_inherit_dimension_from_input_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def inherit_dimension_from_input(self) -> bool:
        """Specify whether to inherit the output dimension from the input scalar or time instant."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_inherit_dimension_from_input_metadata)

    _set_inherit_dimension_from_input_metadata = { "offset" : _set_inherit_dimension_from_input_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @inherit_dimension_from_input.setter
    def inherit_dimension_from_input(self, inherit_dimension_from_input:bool) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_inherit_dimension_from_input_metadata, inherit_dimension_from_input)

    _get_output_dimension_metadata = { "offset" : _get_output_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def output_dimension(self) -> str:
        """The output dimension. Use any of STK supported dimensions. This value will be used if InheritDimensionFromInput is false. The InheritDimensionFromInput property should be set to false before this property can be fixed."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_output_dimension_metadata)

    _set_output_dimension_metadata = { "offset" : _set_output_dimension_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @output_dimension.setter
    def output_dimension(self, output_dimension:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_output_dimension_metadata, output_dimension)

    _get_output_units_metadata = { "offset" : _get_output_units_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def output_units(self) -> str:
        """Specify a unit for the selected output dimension. This is not used for internal computations or reporting/graphing but is needed to unambiguously interpret units of associated coefficients."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_output_units_metadata)

    _set_output_units_metadata = { "offset" : _set_output_units_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @output_units.setter
    def output_units(self, output_unit:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_output_units_metadata, output_unit)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """The Convergence definition, which uses time tolerance to determine when time of extremum is found."""
        return self._intf.get_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(CalculationToolScalarFunction._metadata, CalculationToolScalarFunction._set_convergence_metadata, convergence)

    _property_names[use_calculation_scalar] = "use_calculation_scalar"
    _property_names[input_scalar] = "input_scalar"
    _property_names[input_time] = "input_time"
    _property_names[input_time_units] = "input_time_units"
    _property_names[coefficient_a] = "coefficient_a"
    _property_names[coefficient_b] = "coefficient_b"
    _property_names[coefficient_c] = "coefficient_c"
    _property_names[coefficient_d] = "coefficient_d"
    _property_names[coefficients] = "coefficients"
    _property_names[selected_function] = "selected_function"
    _property_names[available_functions] = "available_functions"
    _property_names[inherit_dimension_from_input] = "inherit_dimension_from_input"
    _property_names[output_dimension] = "output_dimension"
    _property_names[output_units] = "output_units"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarFunction."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarFunction)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarFunction, [CalculationToolScalarFunction, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4712789712859139993, 16973698019291607179), CalculationToolScalarFunction)
agcls.AgTypeNameMap["CalculationToolScalarFunction"] = CalculationToolScalarFunction

class CalculationToolScalarFunctionOf2Variables(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by performing a function(x,y) on two scalar arguments."""

    _num_methods = 23
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_x_method_offset = 1
    _set_x_method_offset = 2
    _get_units_x_method_offset = 3
    _set_units_x_method_offset = 4
    _get_coefficient_a_method_offset = 5
    _set_coefficient_a_method_offset = 6
    _get_y_method_offset = 7
    _set_y_method_offset = 8
    _get_units_y_method_offset = 9
    _set_units_y_method_offset = 10
    _get_coefficient_b_method_offset = 11
    _set_coefficient_b_method_offset = 12
    _get_coefficient_c_method_offset = 13
    _set_coefficient_c_method_offset = 14
    _get_output_dimension_inheritance_method_offset = 15
    _set_output_dimension_inheritance_method_offset = 16
    _get_output_dimension_method_offset = 17
    _set_output_dimension_method_offset = 18
    _get_available_functions_method_offset = 19
    _get_selected_function_method_offset = 20
    _set_selected_function_method_offset = 21
    _get_output_units_method_offset = 22
    _set_output_units_method_offset = 23
    _metadata = {
        "iid_data" : (5385933067987517982, 1513144451347807877),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarFunctionOf2Variables)

    _get_x_metadata = { "offset" : _get_x_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def x(self) -> "ICalculationToolScalar":
        """The scalar argument X."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_x_metadata)

    _set_x_metadata = { "offset" : _set_x_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @x.setter
    def x(self, x:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_x_metadata, x)

    _get_units_x_metadata = { "offset" : _get_units_x_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def units_x(self) -> str:
        """The unit used to interpret numerical values of scalar argument X."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_units_x_metadata)

    _set_units_x_metadata = { "offset" : _set_units_x_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @units_x.setter
    def units_x(self, unit_x:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_units_x_metadata, unit_x)

    _get_coefficient_a_metadata = { "offset" : _get_coefficient_a_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_a(self) -> float:
        """The constant coefficient A."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_coefficient_a_metadata)

    _set_coefficient_a_metadata = { "offset" : _set_coefficient_a_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_a.setter
    def coefficient_a(self, a:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_coefficient_a_metadata, a)

    _get_y_metadata = { "offset" : _get_y_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def y(self) -> "ICalculationToolScalar":
        """The scalar argument Y."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_y_metadata)

    _set_y_metadata = { "offset" : _set_y_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @y.setter
    def y(self, y:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_y_metadata, y)

    _get_units_y_metadata = { "offset" : _get_units_y_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def units_y(self) -> str:
        """The unit used to interpret numerical values of scalar argument Y."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_units_y_metadata)

    _set_units_y_metadata = { "offset" : _set_units_y_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @units_y.setter
    def units_y(self, unit_y:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_units_y_metadata, unit_y)

    _get_coefficient_b_metadata = { "offset" : _get_coefficient_b_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_b(self) -> float:
        """The constant coefficient B."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_coefficient_b_metadata)

    _set_coefficient_b_metadata = { "offset" : _set_coefficient_b_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_b.setter
    def coefficient_b(self, b:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_coefficient_b_metadata, b)

    _get_coefficient_c_metadata = { "offset" : _get_coefficient_c_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def coefficient_c(self) -> float:
        """The constant coefficient C."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_coefficient_c_metadata)

    _set_coefficient_c_metadata = { "offset" : _set_coefficient_c_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @coefficient_c.setter
    def coefficient_c(self, c:float) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_coefficient_c_metadata, c)

    _get_output_dimension_inheritance_metadata = { "offset" : _get_output_dimension_inheritance_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(InheritDimensionType),) }
    @property
    def output_dimension_inheritance(self) -> "InheritDimensionType":
        """Specify whether the output dimension is inherited or explicitly specified using OutputDimension."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_output_dimension_inheritance_metadata)

    _set_output_dimension_inheritance_metadata = { "offset" : _set_output_dimension_inheritance_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(InheritDimensionType),) }
    @output_dimension_inheritance.setter
    def output_dimension_inheritance(self, output_dimension_inheritance:"InheritDimensionType") -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_output_dimension_inheritance_metadata, output_dimension_inheritance)

    _get_output_dimension_metadata = { "offset" : _get_output_dimension_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def output_dimension(self) -> str:
        """The output dimension. Use any of STK supported dimensions. This value will be used if OutputDimensionInheritance is false."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_output_dimension_metadata)

    _set_output_dimension_metadata = { "offset" : _set_output_dimension_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @output_dimension.setter
    def output_dimension(self, output_dimension:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_output_dimension_metadata, output_dimension)

    _get_available_functions_metadata = { "offset" : _get_available_functions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_functions(self) -> list:
        """The available functions. A function(x,y) uses some combination of two scalar arguments x and y as well as one to three constant coefficients a, b, c."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_available_functions_metadata)

    _get_selected_function_metadata = { "offset" : _get_selected_function_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def selected_function(self) -> str:
        """The selected function."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_selected_function_metadata)

    _set_selected_function_metadata = { "offset" : _set_selected_function_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @selected_function.setter
    def selected_function(self, selected_function:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_selected_function_metadata, selected_function)

    _get_output_units_metadata = { "offset" : _get_output_units_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def output_units(self) -> str:
        """The unit for the selected dimension. The unit is not used for internal computations or reporting/graphing but is needed to unambiguously interpret units of associated coefficients."""
        return self._intf.get_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._get_output_units_metadata)

    _set_output_units_metadata = { "offset" : _set_output_units_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @output_units.setter
    def output_units(self, output_unit:str) -> None:
        return self._intf.set_property(CalculationToolScalarFunctionOf2Variables._metadata, CalculationToolScalarFunctionOf2Variables._set_output_units_metadata, output_unit)

    _property_names[x] = "x"
    _property_names[units_x] = "units_x"
    _property_names[coefficient_a] = "coefficient_a"
    _property_names[y] = "y"
    _property_names[units_y] = "units_y"
    _property_names[coefficient_b] = "coefficient_b"
    _property_names[coefficient_c] = "coefficient_c"
    _property_names[output_dimension_inheritance] = "output_dimension_inheritance"
    _property_names[output_dimension] = "output_dimension"
    _property_names[available_functions] = "available_functions"
    _property_names[selected_function] = "selected_function"
    _property_names[output_units] = "output_units"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarFunctionOf2Variables."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarFunctionOf2Variables)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarFunctionOf2Variables, [CalculationToolScalarFunctionOf2Variables, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4967368611246770084, 41349991072973977), CalculationToolScalarFunctionOf2Variables)
agcls.AgTypeNameMap["CalculationToolScalarFunctionOf2Variables"] = CalculationToolScalarFunctionOf2Variables

class CalculationToolScalarIntegral(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Integral of input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""

    _num_methods = 25
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_scalar_method_offset = 1
    _set_input_scalar_method_offset = 2
    _get_compute_as_average_method_offset = 3
    _set_compute_as_average_method_offset = 4
    _get_integration_window_type_method_offset = 5
    _set_integration_window_type_method_offset = 6
    _get_start_offset_method_offset = 7
    _set_start_offset_method_offset = 8
    _get_stop_offset_method_offset = 9
    _set_stop_offset_method_offset = 10
    _get_use_custom_time_limits_method_offset = 11
    _set_use_custom_time_limits_method_offset = 12
    _get_custom_time_limits_method_offset = 13
    _set_custom_time_limits_method_offset = 14
    _get_save_data_option_method_offset = 15
    _set_save_data_option_method_offset = 16
    _get_interpolation_method_offset = 17
    _set_interpolation_method_offset = 18
    _get_sampling_method_offset = 19
    _set_sampling_method_offset = 20
    _get_integral_method_offset = 21
    _set_integral_method_offset = 22
    _get_keep_constant_outside_time_limits_method_offset = 23
    _set_keep_constant_outside_time_limits_method_offset = 24
    _set_offsets_method_offset = 25
    _metadata = {
        "iid_data" : (5056256084423666488, 2219291357061330108),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarIntegral)

    _get_input_scalar_metadata = { "offset" : _get_input_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_scalar(self) -> "ICalculationToolScalar":
        """The input scalar calculation."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_input_scalar_metadata)

    _set_input_scalar_metadata = { "offset" : _set_input_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @input_scalar.setter
    def input_scalar(self, input_scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_input_scalar_metadata, input_scalar)

    _get_compute_as_average_metadata = { "offset" : _get_compute_as_average_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def compute_as_average(self) -> bool:
        """Specify whether the resulting integral value is divided by its time span to generate average value instead of integral."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_compute_as_average_metadata)

    _set_compute_as_average_metadata = { "offset" : _set_compute_as_average_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @compute_as_average.setter
    def compute_as_average(self, compute_as_average:bool) -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_compute_as_average_metadata, compute_as_average)

    _get_integration_window_type_metadata = { "offset" : _get_integration_window_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @property
    def integration_window_type(self) -> "IntegrationWindowType":
        """The integration window, or accumulation, type."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_integration_window_type_metadata)

    _set_integration_window_type_metadata = { "offset" : _set_integration_window_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @integration_window_type.setter
    def integration_window_type(self, integration_window_type:"IntegrationWindowType") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_integration_window_type_metadata, integration_window_type)

    _get_start_offset_metadata = { "offset" : _get_start_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_offset(self) -> float:
        """Set the offset with respect to current time to define the start of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_start_offset_metadata)

    _set_start_offset_metadata = { "offset" : _set_start_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_offset.setter
    def start_offset(self, start_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_start_offset_metadata, start_offset)

    _get_stop_offset_metadata = { "offset" : _get_stop_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_offset(self) -> float:
        """Set the offset with respect to current time to define the stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_stop_offset_metadata)

    _set_stop_offset_metadata = { "offset" : _set_stop_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_offset.setter
    def stop_offset(self, stop_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_stop_offset_metadata, stop_offset)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Specify whether to use custom interval list (CustomTimeLimits)."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """The interval list within which the global minimum or maximum is sought. The default is the overall availability of host object."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_custom_time_limits_metadata, custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Set the value to determine if computed time of extremum is saved/loaded, or recomputed on load if necessary."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_save_data_option_metadata, save_data_option)

    _get_interpolation_metadata = { "offset" : _get_interpolation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interpolation(self) -> "IAnalysisWorkbenchInterpolator":
        """Specify whether to use Lagrange or Hermite interpolation. See STK help on interpolation."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_interpolation_metadata)

    _set_interpolation_metadata = { "offset" : _set_interpolation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchInterpolator"),) }
    @interpolation.setter
    def interpolation(self, interpolation:"IAnalysisWorkbenchInterpolator") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_interpolation_metadata, interpolation)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_sampling_metadata, sampling)

    _get_integral_metadata = { "offset" : _get_integral_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def integral(self) -> "IAnalysisWorkbenchIntegral":
        """The numerical integration method."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_integral_metadata)

    _set_integral_metadata = { "offset" : _set_integral_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchIntegral"),) }
    @integral.setter
    def integral(self, integral:"IAnalysisWorkbenchIntegral") -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_integral_metadata, integral)

    _get_keep_constant_outside_time_limits_metadata = { "offset" : _get_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def keep_constant_outside_time_limits(self) -> bool:
        """If true, the integral's integrand value is replaced by a constant 0 so that the integral remains constant over the gaps in integration."""
        return self._intf.get_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._get_keep_constant_outside_time_limits_metadata)

    _set_keep_constant_outside_time_limits_metadata = { "offset" : _set_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @keep_constant_outside_time_limits.setter
    def keep_constant_outside_time_limits(self, keep_constant_outside_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_keep_constant_outside_time_limits_metadata, keep_constant_outside_time_limits)

    _set_offsets_metadata = { "offset" : _set_offsets_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def set_offsets(self, start_offset:float, stop_offset:float) -> None:
        """Set the offsets with respect to current time to define the start and stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.invoke(CalculationToolScalarIntegral._metadata, CalculationToolScalarIntegral._set_offsets_metadata, start_offset, stop_offset)

    _property_names[input_scalar] = "input_scalar"
    _property_names[compute_as_average] = "compute_as_average"
    _property_names[integration_window_type] = "integration_window_type"
    _property_names[start_offset] = "start_offset"
    _property_names[stop_offset] = "stop_offset"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[interpolation] = "interpolation"
    _property_names[sampling] = "sampling"
    _property_names[integral] = "integral"
    _property_names[keep_constant_outside_time_limits] = "keep_constant_outside_time_limits"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarIntegral."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarIntegral)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarIntegral, [CalculationToolScalarIntegral, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5398899900186130554, 246171721776597151), CalculationToolScalarIntegral)
agcls.AgTypeNameMap["CalculationToolScalarIntegral"] = CalculationToolScalarIntegral

class CalculationToolScalarPlugin(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Use a scalar calculation plugin."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_prog_id_method_offset = 1
    _get_display_name_method_offset = 2
    _get_available_properties_method_offset = 3
    _reset_method_offset = 4
    _set_property_method_offset = 5
    _get_property_method_offset = 6
    _metadata = {
        "iid_data" : (4827675994189541692, 5493812121129913759),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarPlugin)

    _get_prog_id_metadata = { "offset" : _get_prog_id_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def prog_id(self) -> str:
        """A programmatic ID associated with the component."""
        return self._intf.get_property(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._get_prog_id_metadata)

    _get_display_name_metadata = { "offset" : _get_display_name_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def display_name(self) -> str:
        """The plugin's Display Name associated with the COM plugin."""
        return self._intf.get_property(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._get_display_name_metadata)

    _get_available_properties_metadata = { "offset" : _get_available_properties_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def available_properties(self) -> list:
        """An array of names of the properties that can be used to configure the plugin."""
        return self._intf.get_property(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._get_available_properties_metadata)

    _reset_metadata = { "offset" : _reset_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reset(self) -> None:
        """Reset the plugin."""
        return self._intf.invoke(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._reset_metadata, )

    _set_property_metadata = { "offset" : _set_property_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def set_property(self, name:str, value:str) -> None:
        """Set the plugin properties. This method throws an exception if the specified property does not exist, an invalid value was specified or the specified property is read-only."""
        return self._intf.invoke(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._set_property_metadata, name, value)

    _get_property_metadata = { "offset" : _get_property_method_offset,
            "arg_types" : (agcom.BSTR, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg,) }
    def get_property(self, name:str) -> str:
        """Read a value of the specified plugin property. This method throws an exception if the property does not exist."""
        return self._intf.invoke(CalculationToolScalarPlugin._metadata, CalculationToolScalarPlugin._get_property_metadata, name, OutArg())

    _property_names[prog_id] = "prog_id"
    _property_names[display_name] = "display_name"
    _property_names[available_properties] = "available_properties"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarPlugin."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarPlugin)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarPlugin, [CalculationToolScalarPlugin, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5691400509382320907, 9517433290752141957), CalculationToolScalarPlugin)
agcls.AgTypeNameMap["CalculationToolScalarPlugin"] = CalculationToolScalarPlugin

class CalculationToolScalarAlongTrajectory(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Scalar value of spatial calculation evaluated along trajectory of point."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_trajectory_point_method_offset = 1
    _set_trajectory_point_method_offset = 2
    _get_spatial_calculation_method_offset = 3
    _set_spatial_calculation_method_offset = 4
    _metadata = {
        "iid_data" : (5169596386408591497, 10944394653958111876),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarAlongTrajectory)

    _get_trajectory_point_metadata = { "offset" : _get_trajectory_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def trajectory_point(self) -> "IVectorGeometryToolPoint":
        """Trajectory point."""
        return self._intf.get_property(CalculationToolScalarAlongTrajectory._metadata, CalculationToolScalarAlongTrajectory._get_trajectory_point_metadata)

    _set_trajectory_point_metadata = { "offset" : _set_trajectory_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @trajectory_point.setter
    def trajectory_point(self, trajectory_point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolScalarAlongTrajectory._metadata, CalculationToolScalarAlongTrajectory._set_trajectory_point_metadata, trajectory_point)

    _get_spatial_calculation_metadata = { "offset" : _get_spatial_calculation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def spatial_calculation(self) -> "ISpatialAnalysisToolSpatialCalculation":
        """Spatial Calculation."""
        return self._intf.get_property(CalculationToolScalarAlongTrajectory._metadata, CalculationToolScalarAlongTrajectory._get_spatial_calculation_metadata)

    _set_spatial_calculation_metadata = { "offset" : _set_spatial_calculation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ISpatialAnalysisToolSpatialCalculation"),) }
    @spatial_calculation.setter
    def spatial_calculation(self, spatial_calculation:"ISpatialAnalysisToolSpatialCalculation") -> None:
        return self._intf.set_property(CalculationToolScalarAlongTrajectory._metadata, CalculationToolScalarAlongTrajectory._set_spatial_calculation_metadata, spatial_calculation)

    _property_names[trajectory_point] = "trajectory_point"
    _property_names[spatial_calculation] = "spatial_calculation"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarAlongTrajectory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarAlongTrajectory)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarAlongTrajectory, [CalculationToolScalarAlongTrajectory, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4648048658061411939, 12881727305934950065), CalculationToolScalarAlongTrajectory)
agcls.AgTypeNameMap["CalculationToolScalarAlongTrajectory"] = CalculationToolScalarAlongTrajectory

class CalculationToolScalarStandardDeviation(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Integral of input scalar computed with respect to time using one of the specified numerical methods and using one of the specified accumulation types."""

    _num_methods = 25
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_scalar_method_offset = 1
    _set_input_scalar_method_offset = 2
    _get_compute_as_average_method_offset = 3
    _set_compute_as_average_method_offset = 4
    _get_integration_window_type_method_offset = 5
    _set_integration_window_type_method_offset = 6
    _get_start_offset_method_offset = 7
    _set_start_offset_method_offset = 8
    _get_stop_offset_method_offset = 9
    _set_stop_offset_method_offset = 10
    _get_use_custom_time_limits_method_offset = 11
    _set_use_custom_time_limits_method_offset = 12
    _get_custom_time_limits_method_offset = 13
    _set_custom_time_limits_method_offset = 14
    _get_save_data_option_method_offset = 15
    _set_save_data_option_method_offset = 16
    _get_interpolation_method_offset = 17
    _set_interpolation_method_offset = 18
    _get_sampling_method_offset = 19
    _set_sampling_method_offset = 20
    _get_integral_method_offset = 21
    _set_integral_method_offset = 22
    _get_keep_constant_outside_time_limits_method_offset = 23
    _set_keep_constant_outside_time_limits_method_offset = 24
    _set_offsets_method_offset = 25
    _metadata = {
        "iid_data" : (4867937565024961078, 7959695188043257759),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarStandardDeviation)

    _get_input_scalar_metadata = { "offset" : _get_input_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_scalar(self) -> "ICalculationToolScalar":
        """The input scalar calculation."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_input_scalar_metadata)

    _set_input_scalar_metadata = { "offset" : _set_input_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @input_scalar.setter
    def input_scalar(self, input_scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_input_scalar_metadata, input_scalar)

    _get_compute_as_average_metadata = { "offset" : _get_compute_as_average_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def compute_as_average(self) -> bool:
        """Specify whether the resulting integral value is divided by its time span to generate StandardDeviation value instead of integral."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_compute_as_average_metadata)

    _set_compute_as_average_metadata = { "offset" : _set_compute_as_average_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @compute_as_average.setter
    def compute_as_average(self, compute_as_average:bool) -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_compute_as_average_metadata, compute_as_average)

    _get_integration_window_type_metadata = { "offset" : _get_integration_window_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @property
    def integration_window_type(self) -> "IntegrationWindowType":
        """The integration window, or accumulation, type."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_integration_window_type_metadata)

    _set_integration_window_type_metadata = { "offset" : _set_integration_window_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntegrationWindowType),) }
    @integration_window_type.setter
    def integration_window_type(self, integration_window_type:"IntegrationWindowType") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_integration_window_type_metadata, integration_window_type)

    _get_start_offset_metadata = { "offset" : _get_start_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_offset(self) -> float:
        """Set the offset with respect to current time to define the start of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_start_offset_metadata)

    _set_start_offset_metadata = { "offset" : _set_start_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_offset.setter
    def start_offset(self, start_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_start_offset_metadata, start_offset)

    _get_stop_offset_metadata = { "offset" : _get_stop_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_offset(self) -> float:
        """Set the offset with respect to current time to define the stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_stop_offset_metadata)

    _set_stop_offset_metadata = { "offset" : _set_stop_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_offset.setter
    def stop_offset(self, stop_offset:float) -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_stop_offset_metadata, stop_offset)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Specify whether to use custom interval list (CustomTimeLimits)."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """The interval list within which the global minimum or maximum is sought. The default is the overall availability of host object."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_custom_time_limits_metadata, custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Set the value to determine if computed time of extremum is saved/loaded, or recomputed on load if necessary."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_save_data_option_metadata, save_data_option)

    _get_interpolation_metadata = { "offset" : _get_interpolation_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interpolation(self) -> "IAnalysisWorkbenchInterpolator":
        """Specify whether to use Lagrange or Hermite interpolation. See STK help on interpolation."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_interpolation_metadata)

    _set_interpolation_metadata = { "offset" : _set_interpolation_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchInterpolator"),) }
    @interpolation.setter
    def interpolation(self, interpolation:"IAnalysisWorkbenchInterpolator") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_interpolation_metadata, interpolation)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_sampling_metadata, sampling)

    _get_integral_metadata = { "offset" : _get_integral_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def integral(self) -> "IAnalysisWorkbenchIntegral":
        """The numerical integration method."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_integral_metadata)

    _set_integral_metadata = { "offset" : _set_integral_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchIntegral"),) }
    @integral.setter
    def integral(self, integral:"IAnalysisWorkbenchIntegral") -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_integral_metadata, integral)

    _get_keep_constant_outside_time_limits_metadata = { "offset" : _get_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def keep_constant_outside_time_limits(self) -> bool:
        """If true, the integral's integrand value is replaced by a constant 0 so that the integral remains constant over the gaps in integration."""
        return self._intf.get_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._get_keep_constant_outside_time_limits_metadata)

    _set_keep_constant_outside_time_limits_metadata = { "offset" : _set_keep_constant_outside_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @keep_constant_outside_time_limits.setter
    def keep_constant_outside_time_limits(self, keep_constant_outside_time_limits:bool) -> None:
        return self._intf.set_property(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_keep_constant_outside_time_limits_metadata, keep_constant_outside_time_limits)

    _set_offsets_metadata = { "offset" : _set_offsets_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def set_offsets(self, start_offset:float, stop_offset:float) -> None:
        """Set the offsets with respect to current time to define the start and stop of the sliding window, used when IntegrationWindowType is set to Sliding Window."""
        return self._intf.invoke(CalculationToolScalarStandardDeviation._metadata, CalculationToolScalarStandardDeviation._set_offsets_metadata, start_offset, stop_offset)

    _property_names[input_scalar] = "input_scalar"
    _property_names[compute_as_average] = "compute_as_average"
    _property_names[integration_window_type] = "integration_window_type"
    _property_names[start_offset] = "start_offset"
    _property_names[stop_offset] = "stop_offset"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[interpolation] = "interpolation"
    _property_names[sampling] = "sampling"
    _property_names[integral] = "integral"
    _property_names[keep_constant_outside_time_limits] = "keep_constant_outside_time_limits"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarStandardDeviation."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarStandardDeviation)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarStandardDeviation, [CalculationToolScalarStandardDeviation, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4728970761222061821, 15128931790916762790), CalculationToolScalarStandardDeviation)
agcls.AgTypeNameMap["CalculationToolScalarStandardDeviation"] = CalculationToolScalarStandardDeviation

class CalculationToolScalarSurfaceDistanceBetweenPoints(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Surface distance along the specified central body ellipsoid between two points (or their respective projections if specified at altitude)."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_point_1_method_offset = 1
    _set_point_1_method_offset = 2
    _get_point_2_method_offset = 3
    _set_point_2_method_offset = 4
    _get_surface_central_body_method_offset = 5
    _set_surface_central_body_method_offset = 6
    _get_differencing_time_step_method_offset = 7
    _set_differencing_time_step_method_offset = 8
    _metadata = {
        "iid_data" : (5659354710283125094, 1259219978826109078),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarSurfaceDistanceBetweenPoints)

    _get_point_1_metadata = { "offset" : _get_point_1_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point_1(self) -> "IVectorGeometryToolPoint":
        """Starting point on the central body ellipsoid (or projection of point at altitude onto the ellipsoid)."""
        return self._intf.get_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._get_point_1_metadata)

    _set_point_1_metadata = { "offset" : _set_point_1_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @point_1.setter
    def point_1(self, point1:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._set_point_1_metadata, point1)

    _get_point_2_metadata = { "offset" : _get_point_2_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point_2(self) -> "IVectorGeometryToolPoint":
        """Terminating point on the central body ellipsoid (or projection of point at altitude onto the ellipsoid)."""
        return self._intf.get_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._get_point_2_metadata)

    _set_point_2_metadata = { "offset" : _set_point_2_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @point_2.setter
    def point_2(self, point2:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._set_point_2_metadata, point2)

    _get_surface_central_body_metadata = { "offset" : _get_surface_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def surface_central_body(self) -> str:
        """Central body on which the surface distance between points is to be calculated."""
        return self._intf.get_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._get_surface_central_body_metadata)

    _set_surface_central_body_metadata = { "offset" : _set_surface_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @surface_central_body.setter
    def surface_central_body(self, surface_central_body:str) -> None:
        return self._intf.set_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._set_surface_central_body_metadata, surface_central_body)

    _get_differencing_time_step_metadata = { "offset" : _get_differencing_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def differencing_time_step(self) -> float:
        """Time step used in numerical evaluation of scalar calculation time rate of change (derivatives using central differencing)."""
        return self._intf.get_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._get_differencing_time_step_metadata)

    _set_differencing_time_step_metadata = { "offset" : _set_differencing_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @differencing_time_step.setter
    def differencing_time_step(self, differencing_time_step:float) -> None:
        return self._intf.set_property(CalculationToolScalarSurfaceDistanceBetweenPoints._metadata, CalculationToolScalarSurfaceDistanceBetweenPoints._set_differencing_time_step_metadata, differencing_time_step)

    _property_names[point_1] = "point_1"
    _property_names[point_2] = "point_2"
    _property_names[surface_central_body] = "surface_central_body"
    _property_names[differencing_time_step] = "differencing_time_step"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarSurfaceDistanceBetweenPoints."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarSurfaceDistanceBetweenPoints)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarSurfaceDistanceBetweenPoints, [CalculationToolScalarSurfaceDistanceBetweenPoints, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5458481980404184200, 1068397851801295271), CalculationToolScalarSurfaceDistanceBetweenPoints)
agcls.AgTypeNameMap["CalculationToolScalarSurfaceDistanceBetweenPoints"] = CalculationToolScalarSurfaceDistanceBetweenPoints

class CalculationToolScalarVectorComponent(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """The specified component of a vector when resolved in the specified axes."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_vector_method_offset = 1
    _set_input_vector_method_offset = 2
    _get_reference_axes_method_offset = 3
    _set_reference_axes_method_offset = 4
    _get_component_method_offset = 5
    _set_component_method_offset = 6
    _metadata = {
        "iid_data" : (5021364283583990465, 9706837447278064055),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarVectorComponent)

    _get_input_vector_metadata = { "offset" : _get_input_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_vector(self) -> "IVectorGeometryToolVector":
        """Vector."""
        return self._intf.get_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._get_input_vector_metadata)

    _set_input_vector_metadata = { "offset" : _set_input_vector_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @input_vector.setter
    def input_vector(self, input_vector:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._set_input_vector_metadata, input_vector)

    _get_reference_axes_metadata = { "offset" : _get_reference_axes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_axes(self) -> "IVectorGeometryToolAxes":
        """Axes used to resolve the vector's components."""
        return self._intf.get_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._get_reference_axes_metadata)

    _set_reference_axes_metadata = { "offset" : _set_reference_axes_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    @reference_axes.setter
    def reference_axes(self, reference_axes:"IVectorGeometryToolAxes") -> None:
        return self._intf.set_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._set_reference_axes_metadata, reference_axes)

    _get_component_metadata = { "offset" : _get_component_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VectorComponentType),) }
    @property
    def component(self) -> "VectorComponentType":
        """The component of the vector to return as the value of the scalar."""
        return self._intf.get_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._get_component_metadata)

    _set_component_metadata = { "offset" : _set_component_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VectorComponentType),) }
    @component.setter
    def component(self, component:"VectorComponentType") -> None:
        return self._intf.set_property(CalculationToolScalarVectorComponent._metadata, CalculationToolScalarVectorComponent._set_component_metadata, component)

    _property_names[input_vector] = "input_vector"
    _property_names[reference_axes] = "reference_axes"
    _property_names[component] = "component"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarVectorComponent."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarVectorComponent)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarVectorComponent, [CalculationToolScalarVectorComponent, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5434681583091997855, 8195345628688496798), CalculationToolScalarVectorComponent)
agcls.AgTypeNameMap["CalculationToolScalarVectorComponent"] = CalculationToolScalarVectorComponent

class CalculationToolScalarVectorMagnitude(ICalculationToolScalar, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Scalar equal to the magnitude of a specified vector."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_input_vector_method_offset = 1
    _set_input_vector_method_offset = 2
    _metadata = {
        "iid_data" : (5342859323934275233, 16435141540105004707),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolScalarVectorMagnitude)

    _get_input_vector_metadata = { "offset" : _get_input_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def input_vector(self) -> "IVectorGeometryToolVector":
        """Specify any vector in VGT. Note that its magnitude is reference axes independent which is why it is not specified."""
        return self._intf.get_property(CalculationToolScalarVectorMagnitude._metadata, CalculationToolScalarVectorMagnitude._get_input_vector_metadata)

    _set_input_vector_metadata = { "offset" : _set_input_vector_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @input_vector.setter
    def input_vector(self, input_vector:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(CalculationToolScalarVectorMagnitude._metadata, CalculationToolScalarVectorMagnitude._set_input_vector_metadata, input_vector)

    _property_names[input_vector] = "input_vector"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolScalarVectorMagnitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolScalarVectorMagnitude)
        ICalculationToolScalar.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolScalar._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolScalarVectorMagnitude, [CalculationToolScalarVectorMagnitude, ICalculationToolScalar, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5718680998699535592, 13838468874888393908), CalculationToolScalarVectorMagnitude)
agcls.AgTypeNameMap["CalculationToolScalarVectorMagnitude"] = CalculationToolScalarVectorMagnitude

class CalculationToolCondition(ICalculationToolCondition, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Condition returns a non-dimensional metric that is positive if satisfied, negative if not satisfied and 0 if on boundary; this provides computational methods needed for accurate detection of condition crossings."""
    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolCondition."""
        SupportsDeleteCallback.__init__(self)
        ICalculationToolCondition.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolCondition._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolCondition, [ICalculationToolCondition, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4627729760611390020, 5273234787457922734), CalculationToolCondition)
agcls.AgTypeNameMap["CalculationToolCondition"] = CalculationToolCondition

class CalculationToolConditionCombined(ICalculationToolCondition, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Define a condition which combines multiple conditions."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_boolean_operation_method_offset = 1
    _set_boolean_operation_method_offset = 2
    _get_count_method_offset = 3
    _get_all_conditions_method_offset = 4
    _set_all_conditions_method_offset = 5
    _get_condition_method_offset = 6
    _set_condition_method_offset = 7
    _remove_condition_method_offset = 8
    _add_condition_method_offset = 9
    _metadata = {
        "iid_data" : (5384348199380707650, 11046299175776141490),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionCombined)

    _get_boolean_operation_metadata = { "offset" : _get_boolean_operation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConditionCombinedOperationType),) }
    @property
    def boolean_operation(self) -> "ConditionCombinedOperationType":
        """Get the operation from the condition that determines how the conditions are combined. The operation can be set to AND, OR, XOR, MINUS."""
        return self._intf.get_property(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._get_boolean_operation_metadata)

    _set_boolean_operation_metadata = { "offset" : _set_boolean_operation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ConditionCombinedOperationType),) }
    @boolean_operation.setter
    def boolean_operation(self, combine_operation:"ConditionCombinedOperationType") -> None:
        return self._intf.set_property(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._set_boolean_operation_metadata, combine_operation)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Get the number of conditions in the combined condition."""
        return self._intf.get_property(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._get_count_metadata)

    _get_all_conditions_metadata = { "offset" : _get_all_conditions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_all_conditions(self) -> list:
        """Get all conditions that are being combined."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._get_all_conditions_metadata, OutArg())

    _set_all_conditions_metadata = { "offset" : _set_all_conditions_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_all_conditions(self, conditions:list) -> None:
        """Set all conditions to be combined."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._set_all_conditions_metadata, conditions)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def get_condition(self, pos:int) -> "ICalculationToolCondition":
        """Get the condition at the position specified."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._get_condition_metadata, pos, OutArg())

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.PVOID, agcom.INT,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"), agmarshall.IntArg,) }
    def set_condition(self, ref:"ICalculationToolCondition", pos:int) -> None:
        """Set the condition at the position specified."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._set_condition_metadata, ref, pos)

    _remove_condition_metadata = { "offset" : _remove_condition_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def remove_condition(self, pos:int) -> None:
        """Remove the condition at the position specified."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._remove_condition_metadata, pos)

    _add_condition_metadata = { "offset" : _add_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"),) }
    def add_condition(self, ref:"ICalculationToolCondition") -> None:
        """Add a condition at the end of the list."""
        return self._intf.invoke(CalculationToolConditionCombined._metadata, CalculationToolConditionCombined._add_condition_metadata, ref)

    _property_names[boolean_operation] = "boolean_operation"
    _property_names[count] = "count"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionCombined."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionCombined)
        ICalculationToolCondition.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolCondition._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionCombined, [CalculationToolConditionCombined, ICalculationToolCondition, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4767400244892944523, 13373757026237479857), CalculationToolConditionCombined)
agcls.AgTypeNameMap["CalculationToolConditionCombined"] = CalculationToolConditionCombined

class CalculationToolConditionFactory(SupportsDeleteCallback):
    """The factory creates condition components."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_scalar_bounds_method_offset = 2
    _is_type_supported_method_offset = 3
    _create_combined_method_offset = 4
    _create_trajectory_within_volume_method_offset = 5
    _metadata = {
        "iid_data" : (4908222899827327698, 13471708632999045260),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(ConditionType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"ConditionType") -> "ICalculationToolCondition":
        """Create and registers a condition using specified name, description and type."""
        return self._intf.invoke(CalculationToolConditionFactory._metadata, CalculationToolConditionFactory._create_metadata, name, description, type, OutArg())

    _create_scalar_bounds_metadata = { "offset" : _create_scalar_bounds_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_scalar_bounds(self, name:str, description:str) -> "ICalculationToolCondition":
        """Create a condition placing bounds on specified scalar."""
        return self._intf.invoke(CalculationToolConditionFactory._metadata, CalculationToolConditionFactory._create_scalar_bounds_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ConditionType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"ConditionType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(CalculationToolConditionFactory._metadata, CalculationToolConditionFactory._is_type_supported_metadata, type, OutArg())

    _create_combined_metadata = { "offset" : _create_combined_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_combined(self, name:str, description:str) -> "ICalculationToolCondition":
        """Create a condition which combines multiple conditions."""
        return self._intf.invoke(CalculationToolConditionFactory._metadata, CalculationToolConditionFactory._create_combined_metadata, name, description, OutArg())

    _create_trajectory_within_volume_metadata = { "offset" : _create_trajectory_within_volume_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_trajectory_within_volume(self, name:str, description:str) -> "ICalculationToolCondition":
        """Create a condition for point in volume."""
        return self._intf.invoke(CalculationToolConditionFactory._metadata, CalculationToolConditionFactory._create_trajectory_within_volume_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionFactory, [CalculationToolConditionFactory, ])

agcls.AgClassCatalog.add_catalog_entry((4995204375913048056, 16018337718719146118), CalculationToolConditionFactory)
agcls.AgTypeNameMap["CalculationToolConditionFactory"] = CalculationToolConditionFactory

class CalculationToolConditionTrajectoryWithinVolume(ICalculationToolCondition, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by determining if input trajectory poiny is within extents of specified volume grid coordinate."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_point_method_offset = 1
    _set_point_method_offset = 2
    _get_constraint_method_offset = 3
    _set_constraint_method_offset = 4
    _metadata = {
        "iid_data" : (4990807610198674937, 11839537235231187358),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionTrajectoryWithinVolume)

    _get_point_metadata = { "offset" : _get_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point(self) -> "IVectorGeometryToolPoint":
        """Get the trajectory point from the condition."""
        return self._intf.get_property(CalculationToolConditionTrajectoryWithinVolume._metadata, CalculationToolConditionTrajectoryWithinVolume._get_point_metadata)

    _set_point_metadata = { "offset" : _set_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @point.setter
    def point(self, point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolConditionTrajectoryWithinVolume._metadata, CalculationToolConditionTrajectoryWithinVolume._set_point_metadata, point)

    _get_constraint_metadata = { "offset" : _get_constraint_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def constraint(self) -> "ISpatialAnalysisToolVolume":
        """Get the volume constraint on trajectory point."""
        return self._intf.get_property(CalculationToolConditionTrajectoryWithinVolume._metadata, CalculationToolConditionTrajectoryWithinVolume._get_constraint_metadata)

    _set_constraint_metadata = { "offset" : _set_constraint_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ISpatialAnalysisToolVolume"),) }
    @constraint.setter
    def constraint(self, constraint:"ISpatialAnalysisToolVolume") -> None:
        return self._intf.set_property(CalculationToolConditionTrajectoryWithinVolume._metadata, CalculationToolConditionTrajectoryWithinVolume._set_constraint_metadata, constraint)

    _property_names[point] = "point"
    _property_names[constraint] = "constraint"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionTrajectoryWithinVolume."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionTrajectoryWithinVolume)
        ICalculationToolCondition.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolCondition._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionTrajectoryWithinVolume, [CalculationToolConditionTrajectoryWithinVolume, ICalculationToolCondition, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4949089905743702745, 12125459046762185093), CalculationToolConditionTrajectoryWithinVolume)
agcls.AgTypeNameMap["CalculationToolConditionTrajectoryWithinVolume"] = CalculationToolConditionTrajectoryWithinVolume

class CalculationToolConditionScalarBounds(ICalculationToolCondition, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by determining if input scalar is within specified bounds; returns +1 if satisfied, -1 if not satisfied and 0 if on boundary."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_scalar_method_offset = 1
    _set_scalar_method_offset = 2
    _get_operation_method_offset = 3
    _set_operation_method_offset = 4
    _get_minimum_method_offset = 5
    _set_minimum_method_offset = 6
    _get_maximum_method_offset = 7
    _set_maximum_method_offset = 8
    _set_method_offset = 9
    _get_minimum_unitless_method_offset = 10
    _set_minimum_unitless_method_offset = 11
    _get_maximum_unitless_method_offset = 12
    _set_maximum_unitless_method_offset = 13
    _set_unitless_method_offset = 14
    _metadata = {
        "iid_data" : (5118226098986923307, 15094161467708565122),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionScalarBounds)

    _get_scalar_metadata = { "offset" : _get_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scalar(self) -> "ICalculationToolScalar":
        """Get the scalar calculation from the condition."""
        return self._intf.get_property(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_scalar_metadata)

    _set_scalar_metadata = { "offset" : _set_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @scalar.setter
    def scalar(self, scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_scalar_metadata, scalar)

    _get_operation_metadata = { "offset" : _get_operation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ConditionThresholdType),) }
    @property
    def operation(self) -> "ConditionThresholdType":
        """Get the operation from the condition that determines how the bounds are considered. The operation can be set to define satisfaction when the scalar is above minimum, below maximum, between minimum and maximum or outside minimum and maximum."""
        return self._intf.get_property(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_operation_metadata)

    _set_operation_metadata = { "offset" : _set_operation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ConditionThresholdType),) }
    @operation.setter
    def operation(self, operation:"ConditionThresholdType") -> None:
        return self._intf.set_property(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_operation_metadata, operation)

    _get_minimum_metadata = { "offset" : _get_minimum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_minimum(self) -> "Quantity":
        """Get the minimum bound value from the condition. Call SetMinimum to apply changes."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_minimum_metadata, OutArg())

    _set_minimum_metadata = { "offset" : _set_minimum_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"),) }
    def set_minimum(self, value:"Quantity") -> None:
        """Set the minimum bound value for the condition."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_minimum_metadata, value)

    _get_maximum_metadata = { "offset" : _get_maximum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_maximum(self) -> "Quantity":
        """Get the maximum bound value from the condition. Call SetMaximum to apply changes."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_maximum_metadata, OutArg())

    _set_maximum_metadata = { "offset" : _set_maximum_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"),) }
    def set_maximum(self, value:"Quantity") -> None:
        """Set the maximum bound value for the condition."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_maximum_metadata, value)

    _set_metadata = { "offset" : _set_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"), agmarshall.InterfaceInArg("Quantity"),) }
    def set(self, min:"Quantity", max:"Quantity") -> None:
        """Set the min/max bounds. Throws an exception if the minimum is greater than maximum."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_metadata, min, max)

    _get_minimum_unitless_metadata = { "offset" : _get_minimum_unitless_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    def get_minimum_unitless(self) -> float:
        """Get the unitless minimum bound value from the condition. Call SetMinimum to apply changes."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_minimum_unitless_metadata, OutArg())

    _set_minimum_unitless_metadata = { "offset" : _set_minimum_unitless_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    def set_minimum_unitless(self, value:float) -> None:
        """Set the unitless minimum bound value for the condition."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_minimum_unitless_metadata, value)

    _get_maximum_unitless_metadata = { "offset" : _get_maximum_unitless_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    def get_maximum_unitless(self) -> float:
        """Get the unitless maximum bound value from the condition. Call SetMaximum to apply changes."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._get_maximum_unitless_metadata, OutArg())

    _set_maximum_unitless_metadata = { "offset" : _set_maximum_unitless_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    def set_maximum_unitless(self, value:float) -> None:
        """Set the unitless maximum bound value for the condition."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_maximum_unitless_metadata, value)

    _set_unitless_metadata = { "offset" : _set_unitless_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg,) }
    def set_unitless(self, min:float, max:float) -> None:
        """Set the unitless min/max bounds. Throws an exception if the minimum is greater than maximum."""
        return self._intf.invoke(CalculationToolConditionScalarBounds._metadata, CalculationToolConditionScalarBounds._set_unitless_metadata, min, max)

    _property_names[scalar] = "scalar"
    _property_names[operation] = "operation"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionScalarBounds."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionScalarBounds)
        ICalculationToolCondition.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolCondition._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionScalarBounds, [CalculationToolConditionScalarBounds, ICalculationToolCondition, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5647988579309068664, 11530525339333461889), CalculationToolConditionScalarBounds)
agcls.AgTypeNameMap["CalculationToolConditionScalarBounds"] = CalculationToolConditionScalarBounds

class CalculationToolConditionSet(ICalculationToolConditionSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Condition set returns an array of non-dimensional metrics, one for each condition in the set; each metric is positive if corresponding condition is satisfied, negative if not satisfied and 0 if on boundary; this provides computational methods needed for..."""
    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSet."""
        SupportsDeleteCallback.__init__(self)
        ICalculationToolConditionSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolConditionSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSet, [ICalculationToolConditionSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5331250789123914523, 14006917373386840708), CalculationToolConditionSet)
agcls.AgTypeNameMap["CalculationToolConditionSet"] = CalculationToolConditionSet

class CalculationToolConditionSetFactory(SupportsDeleteCallback):
    """The factory creates condition set components."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_scalar_thresholds_method_offset = 2
    _is_type_supported_method_offset = 3
    _metadata = {
        "iid_data" : (4786678123302050092, 4967157609886676869),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionSetFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(ConditionSetType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"ConditionSetType") -> "ICalculationToolConditionSet":
        """Create and registers a condition set using specified name, description, and type."""
        return self._intf.invoke(CalculationToolConditionSetFactory._metadata, CalculationToolConditionSetFactory._create_metadata, name, description, type, OutArg())

    _create_scalar_thresholds_metadata = { "offset" : _create_scalar_thresholds_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_scalar_thresholds(self, name:str, description:str) -> "ICalculationToolConditionSet":
        """Create a scalar thresholds condition set."""
        return self._intf.invoke(CalculationToolConditionSetFactory._metadata, CalculationToolConditionSetFactory._create_scalar_thresholds_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ConditionSetType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"ConditionSetType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(CalculationToolConditionSetFactory._metadata, CalculationToolConditionSetFactory._is_type_supported_metadata, type, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSetFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionSetFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSetFactory, [CalculationToolConditionSetFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5012399328141270168, 11113123877685848731), CalculationToolConditionSetFactory)
agcls.AgTypeNameMap["CalculationToolConditionSetFactory"] = CalculationToolConditionSetFactory

class CalculationToolConditionSetScalarThresholds(ICalculationToolConditionSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Condition set based on single scalar calculation compared to set of threshold values."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_scalar_method_offset = 1
    _set_scalar_method_offset = 2
    _get_thresholds_method_offset = 3
    _get_threshold_labels_method_offset = 4
    _get_include_above_highest_threshold_method_offset = 5
    _set_include_above_highest_threshold_method_offset = 6
    _get_include_below_lowest_threshold_method_offset = 7
    _set_include_below_lowest_threshold_method_offset = 8
    _set_thresholds_and_labels_method_offset = 9
    _metadata = {
        "iid_data" : (5264840048988725368, 15687740403874149559),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConditionSetScalarThresholds)

    _get_scalar_metadata = { "offset" : _get_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def scalar(self) -> "ICalculationToolScalar":
        """The input scalar calculation."""
        return self._intf.get_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._get_scalar_metadata)

    _set_scalar_metadata = { "offset" : _set_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @scalar.setter
    def scalar(self, scalar:"ICalculationToolScalar") -> None:
        return self._intf.set_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._set_scalar_metadata, scalar)

    _get_thresholds_metadata = { "offset" : _get_thresholds_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def thresholds(self) -> list:
        """The input threshold values, flags indicating whether to include conditions above the highest and below the lowest threhsolds, and corresponding labels."""
        return self._intf.get_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._get_thresholds_metadata)

    _get_threshold_labels_metadata = { "offset" : _get_threshold_labels_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def threshold_labels(self) -> list:
        """The input threshold values, flags indicating whether to include conditions above the highest and below the lowest threhsolds, and corresponding labels."""
        return self._intf.get_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._get_threshold_labels_metadata)

    _get_include_above_highest_threshold_metadata = { "offset" : _get_include_above_highest_threshold_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def include_above_highest_threshold(self) -> bool:
        """The threshold indicates whether to include conditions above the highest threhsold."""
        return self._intf.get_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._get_include_above_highest_threshold_metadata)

    _set_include_above_highest_threshold_metadata = { "offset" : _set_include_above_highest_threshold_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @include_above_highest_threshold.setter
    def include_above_highest_threshold(self, include_above_highest_threshold:bool) -> None:
        return self._intf.set_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._set_include_above_highest_threshold_metadata, include_above_highest_threshold)

    _get_include_below_lowest_threshold_metadata = { "offset" : _get_include_below_lowest_threshold_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def include_below_lowest_threshold(self) -> bool:
        """The threshold indicates whether to include conditions below the lowest threhsolds."""
        return self._intf.get_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._get_include_below_lowest_threshold_metadata)

    _set_include_below_lowest_threshold_metadata = { "offset" : _set_include_below_lowest_threshold_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @include_below_lowest_threshold.setter
    def include_below_lowest_threshold(self, include_below_lowest_threshold:bool) -> None:
        return self._intf.set_property(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._set_include_below_lowest_threshold_metadata, include_below_lowest_threshold)

    _set_thresholds_and_labels_metadata = { "offset" : _set_thresholds_and_labels_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.LPSafearrayArg,) }
    def set_thresholds_and_labels(self, thresholds:list, threshold_labels:list) -> None:
        """Set thresholds and threshold labels."""
        return self._intf.invoke(CalculationToolConditionSetScalarThresholds._metadata, CalculationToolConditionSetScalarThresholds._set_thresholds_and_labels_metadata, thresholds, threshold_labels)

    _property_names[scalar] = "scalar"
    _property_names[thresholds] = "thresholds"
    _property_names[threshold_labels] = "threshold_labels"
    _property_names[include_above_highest_threshold] = "include_above_highest_threshold"
    _property_names[include_below_lowest_threshold] = "include_below_lowest_threshold"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConditionSetScalarThresholds."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConditionSetScalarThresholds)
        ICalculationToolConditionSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolConditionSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConditionSetScalarThresholds, [CalculationToolConditionSetScalarThresholds, ICalculationToolConditionSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5572445424587947883, 5673353751277832377), CalculationToolConditionSetScalarThresholds)
agcls.AgTypeNameMap["CalculationToolConditionSetScalarThresholds"] = CalculationToolConditionSetScalarThresholds

class AnalysisWorkbenchConvergence(IAnalysisWorkbenchConvergence, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Represents a base class for convergence definitions."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchConvergence."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchConvergence.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchConvergence._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchConvergence, [IAnalysisWorkbenchConvergence, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4882492916629755513, 15853984640358895501), AnalysisWorkbenchConvergence)
agcls.AgTypeNameMap["AnalysisWorkbenchConvergence"] = AnalysisWorkbenchConvergence

class CalculationToolConvergeBasic(IAnalysisWorkbenchConvergence, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Convergence definition includes parameters that determine criteria for accurate detection of extrema or condition crossings for scalar calculations."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_sense_method_offset = 1
    _set_sense_method_offset = 2
    _get_time_tolerance_method_offset = 3
    _set_time_tolerance_method_offset = 4
    _get_absolute_tolerance_method_offset = 5
    _set_absolute_tolerance_method_offset = 6
    _get_relative_tolerance_method_offset = 7
    _set_relative_tolerance_method_offset = 8
    _metadata = {
        "iid_data" : (5085862782014627789, 13842813656478947762),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolConvergeBasic)

    _get_sense_metadata = { "offset" : _get_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ThresholdConvergenceSenseType),) }
    @property
    def sense(self) -> "ThresholdConvergenceSenseType":
        """Get the convergence sense which determines whether the converged value should be limited to just within or just outside of condition boundaries."""
        return self._intf.get_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._get_sense_metadata)

    _set_sense_metadata = { "offset" : _set_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ThresholdConvergenceSenseType),) }
    @sense.setter
    def sense(self, sense:"ThresholdConvergenceSenseType") -> None:
        return self._intf.set_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._set_sense_metadata, sense)

    _get_time_tolerance_metadata = { "offset" : _get_time_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_tolerance(self) -> float:
        """Get the time tolerance which determines the time accuracy of the converged value."""
        return self._intf.get_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._get_time_tolerance_metadata)

    _set_time_tolerance_metadata = { "offset" : _set_time_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_tolerance.setter
    def time_tolerance(self, time_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._set_time_tolerance_metadata, time_tolerance)

    _get_absolute_tolerance_metadata = { "offset" : _get_absolute_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def absolute_tolerance(self) -> float:
        """Get the absolute tolerance which determines the distance between the value and the boundaries within which the value is considered converged."""
        return self._intf.get_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._get_absolute_tolerance_metadata)

    _set_absolute_tolerance_metadata = { "offset" : _set_absolute_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @absolute_tolerance.setter
    def absolute_tolerance(self, absolute_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._set_absolute_tolerance_metadata, absolute_tolerance)

    _get_relative_tolerance_metadata = { "offset" : _get_relative_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_tolerance(self) -> float:
        """Get the relative tolerance which determines the relative distance between the value and the boundaries within which the value is considered converged."""
        return self._intf.get_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._get_relative_tolerance_metadata)

    _set_relative_tolerance_metadata = { "offset" : _set_relative_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_tolerance.setter
    def relative_tolerance(self, relative_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolConvergeBasic._metadata, CalculationToolConvergeBasic._set_relative_tolerance_metadata, relative_tolerance)

    _property_names[sense] = "sense"
    _property_names[time_tolerance] = "time_tolerance"
    _property_names[absolute_tolerance] = "absolute_tolerance"
    _property_names[relative_tolerance] = "relative_tolerance"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolConvergeBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolConvergeBasic)
        IAnalysisWorkbenchConvergence.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchConvergence._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolConvergeBasic, [CalculationToolConvergeBasic, IAnalysisWorkbenchConvergence, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5514954824464931267, 18409877581449644180), CalculationToolConvergeBasic)
agcls.AgTypeNameMap["CalculationToolConvergeBasic"] = CalculationToolConvergeBasic

class AnalysisWorkbenchDerivative(IAnalysisWorkbenchDerivative, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Represents a base class for derivative definitions."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchDerivative."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchDerivative.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchDerivative._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchDerivative, [IAnalysisWorkbenchDerivative, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5587768146652831139, 5043452618809804181), AnalysisWorkbenchDerivative)
agcls.AgTypeNameMap["AnalysisWorkbenchDerivative"] = AnalysisWorkbenchDerivative

class CalculationToolDerivativeBasic(IAnalysisWorkbenchDerivative, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Derivative definition determines how numerical differencing is used to compute derivatives."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_time_step_method_offset = 1
    _set_time_step_method_offset = 2
    _metadata = {
        "iid_data" : (5698636090077154234, 12182121597687392176),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolDerivativeBasic)

    _get_time_step_metadata = { "offset" : _get_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_step(self) -> float:
        """Get the time step used for numerical evaluation of derivatives using central differencing."""
        return self._intf.get_property(CalculationToolDerivativeBasic._metadata, CalculationToolDerivativeBasic._get_time_step_metadata)

    _set_time_step_metadata = { "offset" : _set_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_step.setter
    def time_step(self, time_step:float) -> None:
        return self._intf.set_property(CalculationToolDerivativeBasic._metadata, CalculationToolDerivativeBasic._set_time_step_metadata, time_step)

    _property_names[time_step] = "time_step"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolDerivativeBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolDerivativeBasic)
        IAnalysisWorkbenchDerivative.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchDerivative._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolDerivativeBasic, [CalculationToolDerivativeBasic, IAnalysisWorkbenchDerivative, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4922997640042415299, 3689582741653722007), CalculationToolDerivativeBasic)
agcls.AgTypeNameMap["CalculationToolDerivativeBasic"] = CalculationToolDerivativeBasic

class TimeToolInstant(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Define an event (time instant)."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstant."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstant, [ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5398793057979554597, 7356352219388702598), TimeToolInstant)
agcls.AgTypeNameMap["TimeToolInstant"] = TimeToolInstant

class TimeToolTimeArray(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """An ordered array of times, which may or may not be evenly spaced."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArray."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArray, [ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5332777741257833242, 4202084266511321478), TimeToolTimeArray)
agcls.AgTypeNameMap["TimeToolTimeArray"] = TimeToolTimeArray

class TimeToolTimeArrayConditionCrossings(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Time array containing times at which the specified condition will change its satisfaction status. Determination is performed within the interval list using Sampling and Convergence parameters."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_satisfaction_crossing_method_offset = 1
    _set_satisfaction_crossing_method_offset = 2
    _get_condition_method_offset = 3
    _set_condition_method_offset = 4
    _get_custom_time_limits_method_offset = 5
    _set_custom_time_limits_method_offset = 6
    _get_use_custom_time_limits_method_offset = 7
    _set_use_custom_time_limits_method_offset = 8
    _get_save_data_option_method_offset = 9
    _set_save_data_option_method_offset = 10
    _get_sampling_method_offset = 11
    _set_sampling_method_offset = 12
    _get_convergence_method_offset = 13
    _set_convergence_method_offset = 14
    _metadata = {
        "iid_data" : (5613889423687348587, 15384622870918658997),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayConditionCrossings)

    _get_satisfaction_crossing_metadata = { "offset" : _get_satisfaction_crossing_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SatisfactionCrossing),) }
    @property
    def satisfaction_crossing(self) -> "SatisfactionCrossing":
        """The direction of interest for satisfaction crossing."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_satisfaction_crossing_metadata)

    _set_satisfaction_crossing_metadata = { "offset" : _set_satisfaction_crossing_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SatisfactionCrossing),) }
    @satisfaction_crossing.setter
    def satisfaction_crossing(self, satisfaction_crossing:"SatisfactionCrossing") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_satisfaction_crossing_metadata, satisfaction_crossing)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def condition(self) -> "ICalculationToolCondition":
        """The condition component."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_condition_metadata)

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"),) }
    @condition.setter
    def condition(self, condition:"ICalculationToolCondition") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_condition_metadata, condition)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """Specify the interval list within which satisfaction crossing times are sought. The default is set to overall availability of host object. The time limits will be used if UseCustomTimeLimits is set to true."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_custom_time_limits_metadata, custom_time_limits)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Indicate whether to use specified custom time limits (see CustomTimeLimits)."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Determine if computed satisfaction crossing times are saved/loaded, or recomputed on load if necessary."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_save_data_option_metadata, save_data_option)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """The Convergence definition, which uses time tolerance to determine when crossing times are found."""
        return self._intf.get_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(TimeToolTimeArrayConditionCrossings._metadata, TimeToolTimeArrayConditionCrossings._set_convergence_metadata, convergence)

    _property_names[satisfaction_crossing] = "satisfaction_crossing"
    _property_names[condition] = "condition"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayConditionCrossings."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayConditionCrossings)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayConditionCrossings, [TimeToolTimeArrayConditionCrossings, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5396215197440132353, 5751529864914655407), TimeToolTimeArrayConditionCrossings)
agcls.AgTypeNameMap["TimeToolTimeArrayConditionCrossings"] = TimeToolTimeArrayConditionCrossings

class TimeToolTimeArrayExtrema(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine times of local minimum and/or maximum of specified scalar calculation. Determination is performed within interval list using Sampling and Convergence parameters."""

    _num_methods = 16
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_extremum_type_method_offset = 1
    _set_extremum_type_method_offset = 2
    _get_is_global_method_offset = 3
    _set_is_global_method_offset = 4
    _get_calculation_scalar_method_offset = 5
    _set_calculation_scalar_method_offset = 6
    _get_custom_time_limits_method_offset = 7
    _set_custom_time_limits_method_offset = 8
    _get_use_custom_time_limits_method_offset = 9
    _set_use_custom_time_limits_method_offset = 10
    _get_save_data_option_method_offset = 11
    _set_save_data_option_method_offset = 12
    _get_sampling_method_offset = 13
    _set_sampling_method_offset = 14
    _get_convergence_method_offset = 15
    _set_convergence_method_offset = 16
    _metadata = {
        "iid_data" : (5206648834428252232, 967315087763540632),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayExtrema)

    _get_extremum_type_metadata = { "offset" : _get_extremum_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ExtremumType),) }
    @property
    def extremum_type(self) -> "ExtremumType":
        """The extremum type of interest (either minimum or maximum) for scalar calculation."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_extremum_type_metadata)

    _set_extremum_type_metadata = { "offset" : _set_extremum_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ExtremumType),) }
    @extremum_type.setter
    def extremum_type(self, extremum_type:"ExtremumType") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_extremum_type_metadata, extremum_type)

    _get_is_global_metadata = { "offset" : _get_is_global_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def is_global(self) -> bool:
        """Indicate whether to perform local or global search. The default is false."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_is_global_metadata)

    _set_is_global_metadata = { "offset" : _set_is_global_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @is_global.setter
    def is_global(self, is_global:bool) -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_is_global_metadata, is_global)

    _get_calculation_scalar_metadata = { "offset" : _get_calculation_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def calculation_scalar(self) -> "ICalculationToolScalar":
        """The scalar calculation."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_calculation_scalar_metadata)

    _set_calculation_scalar_metadata = { "offset" : _set_calculation_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @calculation_scalar.setter
    def calculation_scalar(self, calculation:"ICalculationToolScalar") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_calculation_scalar_metadata, calculation)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """A custom interval list or a single interval. It is by default set to overall availability of host object. This determines time limits within extrema are sought. The time limits will be used if UseCustomTimeLimits is set to true."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_custom_time_limits_metadata, custom_time_limits)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Indicate whether to use specified custom time limits (see CustomTimeLimits)."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Specify whether computed times of extrema are saved/loaded, otherwise it is recomputed on load if necessary."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_save_data_option_metadata, save_data_option)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """The Convergence definition, which uses time tolerance to determine when times of extrema are found."""
        return self._intf.get_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(TimeToolTimeArrayExtrema._metadata, TimeToolTimeArrayExtrema._set_convergence_metadata, convergence)

    _property_names[extremum_type] = "extremum_type"
    _property_names[is_global] = "is_global"
    _property_names[calculation_scalar] = "calculation_scalar"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayExtrema."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayExtrema)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayExtrema, [TimeToolTimeArrayExtrema, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4895652447947847564, 10544338283664157372), TimeToolTimeArrayExtrema)
agcls.AgTypeNameMap["TimeToolTimeArrayExtrema"] = TimeToolTimeArrayExtrema

class TimeToolTimeArrayFactory(SupportsDeleteCallback):
    """The factory creates event arrays."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_extrema_method_offset = 2
    _create_start_stop_times_method_offset = 3
    _create_merged_method_offset = 4
    _create_filtered_method_offset = 5
    _create_fixed_step_method_offset = 6
    _create_condition_crossings_method_offset = 7
    _create_signaled_method_offset = 8
    _is_type_supported_method_offset = 9
    _create_fixed_times_method_offset = 10
    _metadata = {
        "iid_data" : (5542319423916064894, 16004686220202616230),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(EventArrayType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"EventArrayType") -> "ITimeToolTimeArray":
        """Create and register an event array using specified name, description, and type."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_metadata, name, description, type, OutArg())

    _create_extrema_metadata = { "offset" : _create_extrema_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_extrema(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array by determining times of local minimum and/or maximum of specified scalar calculation."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_extrema_metadata, name, description, OutArg())

    _create_start_stop_times_metadata = { "offset" : _create_start_stop_times_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_start_stop_times(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array by taking start and/or stop times of every interval in the specified reference interval list and adding them to array."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_start_stop_times_metadata, name, description, OutArg())

    _create_merged_metadata = { "offset" : _create_merged_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_merged(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array by merging times from two other arrays by creating a union of bounding intervals from two constituent arrays."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_merged_metadata, name, description, OutArg())

    _create_filtered_metadata = { "offset" : _create_filtered_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_filtered(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array by filtering times from an original time array according to specified filtering method."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_filtered_metadata, name, description, OutArg())

    _create_fixed_step_metadata = { "offset" : _create_fixed_step_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed_step(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array using fixed time steps from the specified time reference and adding sampled times to array if they fall within specified bounding interval list."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_fixed_step_metadata, name, description, OutArg())

    _create_condition_crossings_metadata = { "offset" : _create_condition_crossings_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_condition_crossings(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array containing times at which the specified condition will change its satisfaction status."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_condition_crossings_metadata, name, description, OutArg())

    _create_signaled_metadata = { "offset" : _create_signaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_signaled(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array recorded at target clock location by performing signal transmission of original time array between base and target clock locations."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_signaled_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(EventArrayType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"EventArrayType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._is_type_supported_metadata, type, OutArg())

    _create_fixed_times_metadata = { "offset" : _create_fixed_times_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed_times(self, name:str, description:str) -> "ITimeToolTimeArray":
        """Create an event array using specified times."""
        return self._intf.invoke(TimeToolTimeArrayFactory._metadata, TimeToolTimeArrayFactory._create_fixed_times_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayFactory, [TimeToolTimeArrayFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5458075538004504061, 2067516564496506015), TimeToolTimeArrayFactory)
agcls.AgTypeNameMap["TimeToolTimeArrayFactory"] = TimeToolTimeArrayFactory

class TimeToolTimeArrayFiltered(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by filtering times from original time array according to specified filtering method."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_time_array_method_offset = 1
    _set_original_time_array_method_offset = 2
    _get_filter_type_method_offset = 3
    _set_filter_type_method_offset = 4
    _get_count_method_offset = 5
    _set_count_method_offset = 6
    _get_step_method_offset = 7
    _set_step_method_offset = 8
    _get_include_interval_stop_times_method_offset = 9
    _set_include_interval_stop_times_method_offset = 10
    _get_filter_interval_list_method_offset = 11
    _set_filter_interval_list_method_offset = 12
    _metadata = {
        "iid_data" : (4807678415758810560, 13966026461879800744),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayFiltered)

    _get_original_time_array_metadata = { "offset" : _get_original_time_array_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_time_array(self) -> "ITimeToolTimeArray":
        """The original time array."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_original_time_array_metadata)

    _set_original_time_array_metadata = { "offset" : _set_original_time_array_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"),) }
    @original_time_array.setter
    def original_time_array(self, original_time_array:"ITimeToolTimeArray") -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_original_time_array_metadata, original_time_array)

    _get_filter_type_metadata = { "offset" : _get_filter_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventArrayFilterType),) }
    @property
    def filter_type(self) -> "EventArrayFilterType":
        """Skip Time Steps filter type omits from filtered time array any times that fall within specified time step of last accepted time sample. Skip Count filter type omits specified number of time samples since last accepted time sample..."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_filter_type_metadata)

    _set_filter_type_metadata = { "offset" : _set_filter_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EventArrayFilterType),) }
    @filter_type.setter
    def filter_type(self, filter_type:"EventArrayFilterType") -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_filter_type_metadata, filter_type)

    _get_count_metadata = { "offset" : _get_count_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def count(self) -> int:
        """Specify the number of times skipped between accepted samples when FilterType is set to Skip Count..."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_count_metadata)

    _set_count_metadata = { "offset" : _set_count_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @count.setter
    def count(self, count:int) -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_count_metadata, count)

    _get_step_metadata = { "offset" : _get_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step(self) -> float:
        """The number of steps skipped between accepted samples when FilterType is set to Skip Time Steps."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_step_metadata)

    _set_step_metadata = { "offset" : _set_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step.setter
    def step(self, step:float) -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_step_metadata, step)

    _get_include_interval_stop_times_metadata = { "offset" : _get_include_interval_stop_times_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def include_interval_stop_times(self) -> bool:
        """If set to true, includes stop times of each interval from original time array."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_include_interval_stop_times_metadata)

    _set_include_interval_stop_times_metadata = { "offset" : _set_include_interval_stop_times_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @include_interval_stop_times.setter
    def include_interval_stop_times(self, include_interval_stop_times:bool) -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_include_interval_stop_times_metadata, include_interval_stop_times)

    _get_filter_interval_list_metadata = { "offset" : _get_filter_interval_list_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def filter_interval_list(self) -> "ITimeToolTimeIntervalList":
        """The interval list used to filter samples when FilterType is set to Skip Intervals."""
        return self._intf.get_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._get_filter_interval_list_metadata)

    _set_filter_interval_list_metadata = { "offset" : _set_filter_interval_list_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @filter_interval_list.setter
    def filter_interval_list(self, filter_interval_list:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeArrayFiltered._metadata, TimeToolTimeArrayFiltered._set_filter_interval_list_metadata, filter_interval_list)

    _property_names[original_time_array] = "original_time_array"
    _property_names[filter_type] = "filter_type"
    _property_names[count] = "count"
    _property_names[step] = "step"
    _property_names[include_interval_stop_times] = "include_interval_stop_times"
    _property_names[filter_interval_list] = "filter_interval_list"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayFiltered."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayFiltered)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayFiltered, [TimeToolTimeArrayFiltered, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4642904597198643175, 4377576397480008330), TimeToolTimeArrayFiltered)
agcls.AgTypeNameMap["TimeToolTimeArrayFiltered"] = TimeToolTimeArrayFiltered

class TimeToolTimeArrayFixedStep(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by taking fixed time steps from specified time reference and adding sampled times to array if they fall within specified bounding interval list."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_bounding_interval_list_method_offset = 1
    _set_bounding_interval_list_method_offset = 2
    _get_sampling_time_step_method_offset = 3
    _set_sampling_time_step_method_offset = 4
    _get_include_interval_edges_method_offset = 5
    _set_include_interval_edges_method_offset = 6
    _get_reference_type_method_offset = 7
    _set_reference_type_method_offset = 8
    _get_reference_time_instant_method_offset = 9
    _set_reference_time_instant_method_offset = 10
    _metadata = {
        "iid_data" : (5206395693562101053, 5734909382665571514),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayFixedStep)

    _get_bounding_interval_list_metadata = { "offset" : _get_bounding_interval_list_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def bounding_interval_list(self) -> "ITimeToolTimeIntervalList":
        """The bounding interval list."""
        return self._intf.get_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._get_bounding_interval_list_metadata)

    _set_bounding_interval_list_metadata = { "offset" : _set_bounding_interval_list_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @bounding_interval_list.setter
    def bounding_interval_list(self, bounding_interval_list:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._set_bounding_interval_list_metadata, bounding_interval_list)

    _get_sampling_time_step_metadata = { "offset" : _get_sampling_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def sampling_time_step(self) -> float:
        """The sampling time step."""
        return self._intf.get_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._get_sampling_time_step_metadata)

    _set_sampling_time_step_metadata = { "offset" : _set_sampling_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @sampling_time_step.setter
    def sampling_time_step(self, sampling_time_step:float) -> None:
        return self._intf.set_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._set_sampling_time_step_metadata, sampling_time_step)

    _get_include_interval_edges_metadata = { "offset" : _get_include_interval_edges_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def include_interval_edges(self) -> bool:
        """Specify whether to include interval edges."""
        return self._intf.get_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._get_include_interval_edges_metadata)

    _set_include_interval_edges_metadata = { "offset" : _set_include_interval_edges_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @include_interval_edges.setter
    def include_interval_edges(self, include_interval_edges:bool) -> None:
        return self._intf.set_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._set_include_interval_edges_metadata, include_interval_edges)

    _get_reference_type_metadata = { "offset" : _get_reference_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SampleReferenceTimeType),) }
    @property
    def reference_type(self) -> "SampleReferenceTimeType":
        """Specify the time reference from which fixed sampling time steps are taken. Note: selecting Start/Stop of each Interval resets the time reference for each interval, whereas other types maintain single reference for entire array."""
        return self._intf.get_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._get_reference_type_metadata)

    _set_reference_type_metadata = { "offset" : _set_reference_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SampleReferenceTimeType),) }
    @reference_type.setter
    def reference_type(self, reference_type:"SampleReferenceTimeType") -> None:
        return self._intf.set_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._set_reference_type_metadata, reference_type)

    _get_reference_time_instant_metadata = { "offset" : _get_reference_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_time_instant(self) -> "ITimeToolInstant":
        """The reference time instant. Only applicable if the ReferenceType is set to time instant."""
        return self._intf.get_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._get_reference_time_instant_metadata)

    _set_reference_time_instant_metadata = { "offset" : _set_reference_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @reference_time_instant.setter
    def reference_time_instant(self, reference_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolTimeArrayFixedStep._metadata, TimeToolTimeArrayFixedStep._set_reference_time_instant_metadata, reference_time_instant)

    _property_names[bounding_interval_list] = "bounding_interval_list"
    _property_names[sampling_time_step] = "sampling_time_step"
    _property_names[include_interval_edges] = "include_interval_edges"
    _property_names[reference_type] = "reference_type"
    _property_names[reference_time_instant] = "reference_time_instant"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayFixedStep."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayFixedStep)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayFixedStep, [TimeToolTimeArrayFixedStep, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5291295331531848140, 1885669106906320519), TimeToolTimeArrayFixedStep)
agcls.AgTypeNameMap["TimeToolTimeArrayFixedStep"] = TimeToolTimeArrayFixedStep

class TimeToolTimeArrayFixedTimes(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Array defined by time ordered instants each explicitly specified."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_array_times_method_offset = 1
    _set_array_times_method_offset = 2
    _metadata = {
        "iid_data" : (5074323306179123922, 16054451061673617558),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayFixedTimes)

    _get_array_times_metadata = { "offset" : _get_array_times_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def array_times(self) -> list:
        """The array of times."""
        return self._intf.get_property(TimeToolTimeArrayFixedTimes._metadata, TimeToolTimeArrayFixedTimes._get_array_times_metadata)

    _set_array_times_metadata = { "offset" : _set_array_times_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_array_times(self, times:list) -> None:
        """Set array of times."""
        return self._intf.invoke(TimeToolTimeArrayFixedTimes._metadata, TimeToolTimeArrayFixedTimes._set_array_times_metadata, times)

    _property_names[array_times] = "array_times"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayFixedTimes."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayFixedTimes)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayFixedTimes, [TimeToolTimeArrayFixedTimes, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5593791433450696394, 1539321884068845490), TimeToolTimeArrayFixedTimes)
agcls.AgTypeNameMap["TimeToolTimeArrayFixedTimes"] = TimeToolTimeArrayFixedTimes

class TimeToolTimeArrayMerged(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by merging times from two other arrays by creating a union of bounding intervals from two constituent arrays. If some intervals overlap, then within overlap times from both arrays are merged together."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_time_array_a_method_offset = 1
    _set_time_array_a_method_offset = 2
    _get_time_array_b_method_offset = 3
    _set_time_array_b_method_offset = 4
    _metadata = {
        "iid_data" : (4780019232320281739, 780076398979475114),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayMerged)

    _get_time_array_a_metadata = { "offset" : _get_time_array_a_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def time_array_a(self) -> "ITimeToolTimeArray":
        """The first time array."""
        return self._intf.get_property(TimeToolTimeArrayMerged._metadata, TimeToolTimeArrayMerged._get_time_array_a_metadata)

    _set_time_array_a_metadata = { "offset" : _set_time_array_a_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"),) }
    @time_array_a.setter
    def time_array_a(self, time_array_a:"ITimeToolTimeArray") -> None:
        return self._intf.set_property(TimeToolTimeArrayMerged._metadata, TimeToolTimeArrayMerged._set_time_array_a_metadata, time_array_a)

    _get_time_array_b_metadata = { "offset" : _get_time_array_b_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def time_array_b(self) -> "ITimeToolTimeArray":
        """The second time array."""
        return self._intf.get_property(TimeToolTimeArrayMerged._metadata, TimeToolTimeArrayMerged._get_time_array_b_metadata)

    _set_time_array_b_metadata = { "offset" : _set_time_array_b_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"),) }
    @time_array_b.setter
    def time_array_b(self, time_array_b:"ITimeToolTimeArray") -> None:
        return self._intf.set_property(TimeToolTimeArrayMerged._metadata, TimeToolTimeArrayMerged._set_time_array_b_metadata, time_array_b)

    _property_names[time_array_a] = "time_array_a"
    _property_names[time_array_b] = "time_array_b"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayMerged."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayMerged)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayMerged, [TimeToolTimeArrayMerged, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4782774864999860286, 3889233035658327469), TimeToolTimeArrayMerged)
agcls.AgTypeNameMap["TimeToolTimeArrayMerged"] = TimeToolTimeArrayMerged

class TimeToolTimeArraySignaled(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine what time array is recorded at target clock location by performing signal transmission of original time array between base and target clock locations..."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_time_array_method_offset = 1
    _set_original_time_array_method_offset = 2
    _get_signal_sense_method_offset = 3
    _set_signal_sense_method_offset = 4
    _get_base_clock_location_method_offset = 5
    _set_base_clock_location_method_offset = 6
    _get_target_clock_location_method_offset = 7
    _set_target_clock_location_method_offset = 8
    _get_signal_delay_method_offset = 9
    _set_signal_delay_method_offset = 10
    _metadata = {
        "iid_data" : (5436916865848414632, 14446293692646829208),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArraySignaled)

    _get_original_time_array_metadata = { "offset" : _get_original_time_array_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_time_array(self) -> "ITimeToolTimeArray":
        """The original time array."""
        return self._intf.get_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._get_original_time_array_metadata)

    _set_original_time_array_metadata = { "offset" : _set_original_time_array_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeArray"),) }
    @original_time_array.setter
    def original_time_array(self, original_time_array:"ITimeToolTimeArray") -> None:
        return self._intf.set_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._set_original_time_array_metadata, original_time_array)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @property
    def signal_sense(self) -> "SignalDirectionType":
        """The direction of the signal, whether you are Transmitting or Receiving from the Base Clock Location."""
        return self._intf.get_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @signal_sense.setter
    def signal_sense(self, signal_sense:"SignalDirectionType") -> None:
        return self._intf.set_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._set_signal_sense_metadata, signal_sense)

    _get_base_clock_location_metadata = { "offset" : _get_base_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_clock_location(self) -> "IVectorGeometryToolPoint":
        """The base clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._get_base_clock_location_metadata)

    _set_base_clock_location_metadata = { "offset" : _set_base_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @base_clock_location.setter
    def base_clock_location(self, base_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._set_base_clock_location_metadata, base_clock_location)

    _get_target_clock_location_metadata = { "offset" : _get_target_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_clock_location(self) -> "IVectorGeometryToolPoint":
        """The target clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._get_target_clock_location_metadata)

    _set_target_clock_location_metadata = { "offset" : _set_target_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @target_clock_location.setter
    def target_clock_location(self, target_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._set_target_clock_location_metadata, target_clock_location)

    _get_signal_delay_metadata = { "offset" : _get_signal_delay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def signal_delay(self) -> "IAnalysisWorkbenchSignalDelay":
        """The Signal delay definition, which includes signal transmission, time delay convergence and signal path reference system."""
        return self._intf.get_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._get_signal_delay_metadata)

    _set_signal_delay_metadata = { "offset" : _set_signal_delay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSignalDelay"),) }
    @signal_delay.setter
    def signal_delay(self, signal_delay:"IAnalysisWorkbenchSignalDelay") -> None:
        return self._intf.set_property(TimeToolTimeArraySignaled._metadata, TimeToolTimeArraySignaled._set_signal_delay_metadata, signal_delay)

    _property_names[original_time_array] = "original_time_array"
    _property_names[signal_sense] = "signal_sense"
    _property_names[base_clock_location] = "base_clock_location"
    _property_names[target_clock_location] = "target_clock_location"
    _property_names[signal_delay] = "signal_delay"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArraySignaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArraySignaled)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArraySignaled, [TimeToolTimeArraySignaled, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5435814862915860050, 12317878345385448122), TimeToolTimeArraySignaled)
agcls.AgTypeNameMap["TimeToolTimeArraySignaled"] = TimeToolTimeArraySignaled

class TimeToolTimeArrayStartStopTimes(ITimeToolTimeArray, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by taking start and/or stop times of every interval in specified reference interval list and adding them to array. The array is then bounded by single interval spanning specified reference interval list..."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_start_stop_option_method_offset = 1
    _set_start_stop_option_method_offset = 2
    _get_reference_intervals_method_offset = 3
    _set_reference_intervals_method_offset = 4
    _metadata = {
        "iid_data" : (4846567064350463042, 16499803777902134457),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeArrayStartStopTimes)

    _get_start_stop_option_metadata = { "offset" : _get_start_stop_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(StartStopType),) }
    @property
    def start_stop_option(self) -> "StartStopType":
        """The edge type. At least one of the two edge types must be selected."""
        return self._intf.get_property(TimeToolTimeArrayStartStopTimes._metadata, TimeToolTimeArrayStartStopTimes._get_start_stop_option_metadata)

    _set_start_stop_option_metadata = { "offset" : _set_start_stop_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(StartStopType),) }
    @start_stop_option.setter
    def start_stop_option(self, start_stop_option:"StartStopType") -> None:
        return self._intf.set_property(TimeToolTimeArrayStartStopTimes._metadata, TimeToolTimeArrayStartStopTimes._set_start_stop_option_metadata, start_stop_option)

    _get_reference_intervals_metadata = { "offset" : _get_reference_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_intervals(self) -> "ITimeToolTimeIntervalList":
        """The reference interval list."""
        return self._intf.get_property(TimeToolTimeArrayStartStopTimes._metadata, TimeToolTimeArrayStartStopTimes._get_reference_intervals_metadata)

    _set_reference_intervals_metadata = { "offset" : _set_reference_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @reference_intervals.setter
    def reference_intervals(self, reference_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeArrayStartStopTimes._metadata, TimeToolTimeArrayStartStopTimes._set_reference_intervals_metadata, reference_intervals)

    _property_names[start_stop_option] = "start_stop_option"
    _property_names[reference_intervals] = "reference_intervals"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeArrayStartStopTimes."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeArrayStartStopTimes)
        ITimeToolTimeArray.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeArray._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeArrayStartStopTimes, [TimeToolTimeArrayStartStopTimes, ITimeToolTimeArray, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4860958271402795436, 3117480943035526832), TimeToolTimeArrayStartStopTimes)
agcls.AgTypeNameMap["TimeToolTimeArrayStartStopTimes"] = TimeToolTimeArrayStartStopTimes

class TimeToolInstantEpoch(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Event set at specified date/time."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_epoch_method_offset = 1
    _set_epoch_method_offset = 2
    _metadata = {
        "iid_data" : (5361126161618754498, 11546813527429934240),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantEpoch)

    _get_epoch_metadata = { "offset" : _get_epoch_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def epoch(self) -> typing.Any:
        """An explicit date/time, set using any STK date format."""
        return self._intf.get_property(TimeToolInstantEpoch._metadata, TimeToolInstantEpoch._get_epoch_metadata)

    _set_epoch_metadata = { "offset" : _set_epoch_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    @epoch.setter
    def epoch(self, epoch:typing.Any) -> None:
        return self._intf.set_property(TimeToolInstantEpoch._metadata, TimeToolInstantEpoch._set_epoch_metadata, epoch)

    _property_names[epoch] = "epoch"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantEpoch."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantEpoch)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantEpoch, [TimeToolInstantEpoch, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5435356060854789844, 4067587155681333929), TimeToolInstantEpoch)
agcls.AgTypeNameMap["TimeToolInstantEpoch"] = TimeToolInstantEpoch

class TimeToolInstantExtremum(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine time of global minimum or maximum of specified scalar calculation. Determination is performed within interval list using Sampling and Convergence parameters."""

    _num_methods = 14
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_extremum_type_method_offset = 1
    _set_extremum_type_method_offset = 2
    _get_calculation_scalar_method_offset = 3
    _set_calculation_scalar_method_offset = 4
    _get_custom_time_limits_method_offset = 5
    _set_custom_time_limits_method_offset = 6
    _get_use_custom_time_limits_method_offset = 7
    _set_use_custom_time_limits_method_offset = 8
    _get_save_data_option_method_offset = 9
    _set_save_data_option_method_offset = 10
    _get_sampling_method_offset = 11
    _set_sampling_method_offset = 12
    _get_convergence_method_offset = 13
    _set_convergence_method_offset = 14
    _metadata = {
        "iid_data" : (5490767592740683220, 17574270972815238058),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantExtremum)

    _get_extremum_type_metadata = { "offset" : _get_extremum_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ExtremumType),) }
    @property
    def extremum_type(self) -> "ExtremumType":
        """The extremum type of interest (either minimum or maximum) for scalar calculation."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_extremum_type_metadata)

    _set_extremum_type_metadata = { "offset" : _set_extremum_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ExtremumType),) }
    @extremum_type.setter
    def extremum_type(self, extremum_type:"ExtremumType") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_extremum_type_metadata, extremum_type)

    _get_calculation_scalar_metadata = { "offset" : _get_calculation_scalar_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def calculation_scalar(self) -> "ICalculationToolScalar":
        """The scalar calculation."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_calculation_scalar_metadata)

    _set_calculation_scalar_metadata = { "offset" : _set_calculation_scalar_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolScalar"),) }
    @calculation_scalar.setter
    def calculation_scalar(self, calculation:"ICalculationToolScalar") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_calculation_scalar_metadata, calculation)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """A custom interval list or a single interval. By default it is set to overall availability of host object. This determines time limits within which global minimum or maximum is sought. The time limits will be used if UseCustomTimeLimits is set to true."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_custom_time_limits_metadata, custom_time_limits)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Indicate whether to use specified custom time limits (see CustomTimeLimits)."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Determine if computed time of extremum is saved/loaded, otherwise it is recomputed on load if necessary."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_save_data_option_metadata, save_data_option)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """A Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """A Convergence definition, which uses time tolerance to determine when time of extremum is found."""
        return self._intf.get_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(TimeToolInstantExtremum._metadata, TimeToolInstantExtremum._set_convergence_metadata, convergence)

    _property_names[extremum_type] = "extremum_type"
    _property_names[calculation_scalar] = "calculation_scalar"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantExtremum."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantExtremum)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantExtremum, [TimeToolInstantExtremum, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5389385592881205397, 339728145862133895), TimeToolInstantExtremum)
agcls.AgTypeNameMap["TimeToolInstantExtremum"] = TimeToolInstantExtremum

class TimeToolInstantFactory(SupportsDeleteCallback):
    """The factory creates events."""

    _num_methods = 11
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_today_method_offset = 1
    _get_tomorrow_method_offset = 2
    _create_method_offset = 3
    _create_epoch_method_offset = 4
    _create_extremum_method_offset = 5
    _create_start_stop_time_method_offset = 6
    _create_signaled_method_offset = 7
    _create_time_offset_method_offset = 8
    _create_smart_epoch_from_time_method_offset = 9
    _create_smart_epoch_from_event_method_offset = 10
    _is_type_supported_method_offset = 11
    _metadata = {
        "iid_data" : (5177435815935857475, 9742177550060253581),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantFactory)

    _get_today_metadata = { "offset" : _get_today_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def today(self) -> "ITimeToolInstant":
        """Return Today time instant."""
        return self._intf.get_property(TimeToolInstantFactory._metadata, TimeToolInstantFactory._get_today_metadata)

    _get_tomorrow_metadata = { "offset" : _get_tomorrow_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def tomorrow(self) -> "ITimeToolInstant":
        """Return Tomorrow time instant."""
        return self._intf.get_property(TimeToolInstantFactory._metadata, TimeToolInstantFactory._get_tomorrow_metadata)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(TimeEventType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"TimeEventType") -> "ITimeToolInstant":
        """Create and registers an event using specified name, description, and type."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_metadata, name, description, type, OutArg())

    _create_epoch_metadata = { "offset" : _create_epoch_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_epoch(self, name:str, description:str) -> "ITimeToolInstant":
        """Create an event set at a specified date/time."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_epoch_metadata, name, description, OutArg())

    _create_extremum_metadata = { "offset" : _create_extremum_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_extremum(self, name:str, description:str) -> "ITimeToolInstant":
        """Create an event that determines the time of global minimum or maximum of specified scalar calculation."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_extremum_metadata, name, description, OutArg())

    _create_start_stop_time_metadata = { "offset" : _create_start_stop_time_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_start_stop_time(self, name:str, description:str) -> "ITimeToolInstant":
        """Create an event that is either the start or stop time selected from a reference interval."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_start_stop_time_metadata, name, description, OutArg())

    _create_signaled_metadata = { "offset" : _create_signaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_signaled(self, name:str, description:str) -> "ITimeToolInstant":
        """Create an event recorded on a specified clock via signal transmission from an original time instant recorded on different clock."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_signaled_metadata, name, description, OutArg())

    _create_time_offset_metadata = { "offset" : _create_time_offset_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_time_offset(self, name:str, description:str) -> "ITimeToolInstant":
        """Create an event at fixed offset from specified reference event."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_time_offset_metadata, name, description, OutArg())

    _create_smart_epoch_from_time_metadata = { "offset" : _create_smart_epoch_from_time_method_offset,
            "arg_types" : (agcom.Variant, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.VariantArg, agmarshall.InterfaceOutArg,) }
    def create_smart_epoch_from_time(self, epoch:typing.Any) -> "TimeToolInstantSmartEpoch":
        """Create a smart epoch from STK epoch."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_smart_epoch_from_time_metadata, epoch, OutArg())

    _create_smart_epoch_from_event_metadata = { "offset" : _create_smart_epoch_from_event_method_offset,
            "arg_types" : (agcom.PVOID, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"), agmarshall.InterfaceOutArg,) }
    def create_smart_epoch_from_event(self, ref_event:"ITimeToolInstant") -> "TimeToolInstantSmartEpoch":
        """Create a smart epoch from an event."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._create_smart_epoch_from_event_metadata, ref_event, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(TimeEventType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"TimeEventType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(TimeToolInstantFactory._metadata, TimeToolInstantFactory._is_type_supported_metadata, type, OutArg())

    _property_names[today] = "today"
    _property_names[tomorrow] = "tomorrow"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantFactory, [TimeToolInstantFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5272300824979797026, 16498667286970460562), TimeToolInstantFactory)
agcls.AgTypeNameMap["TimeToolInstantFactory"] = TimeToolInstantFactory

class TimeToolTimeInterval(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A single time interval."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeInterval."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeInterval, [ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4907362365384322524, 7697029984187487375), TimeToolTimeInterval)
agcls.AgTypeNameMap["TimeToolTimeInterval"] = TimeToolTimeInterval

class TimeToolTimeIntervalBetweenTimeInstants(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval between specified start and stop time instants. If start instant occurs after stop, then interval is undefined."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_start_time_instant_method_offset = 1
    _set_start_time_instant_method_offset = 2
    _get_stop_time_instant_method_offset = 3
    _set_stop_time_instant_method_offset = 4
    _metadata = {
        "iid_data" : (5763963832395591361, 8213926450025946009),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalBetweenTimeInstants)

    _get_start_time_instant_metadata = { "offset" : _get_start_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def start_time_instant(self) -> "ITimeToolInstant":
        """The start time instant of the interval."""
        return self._intf.get_property(TimeToolTimeIntervalBetweenTimeInstants._metadata, TimeToolTimeIntervalBetweenTimeInstants._get_start_time_instant_metadata)

    _set_start_time_instant_metadata = { "offset" : _set_start_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @start_time_instant.setter
    def start_time_instant(self, start_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolTimeIntervalBetweenTimeInstants._metadata, TimeToolTimeIntervalBetweenTimeInstants._set_start_time_instant_metadata, start_time_instant)

    _get_stop_time_instant_metadata = { "offset" : _get_stop_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def stop_time_instant(self) -> "ITimeToolInstant":
        """The stop time instant of the interval."""
        return self._intf.get_property(TimeToolTimeIntervalBetweenTimeInstants._metadata, TimeToolTimeIntervalBetweenTimeInstants._get_stop_time_instant_metadata)

    _set_stop_time_instant_metadata = { "offset" : _set_stop_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @stop_time_instant.setter
    def stop_time_instant(self, stop_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolTimeIntervalBetweenTimeInstants._metadata, TimeToolTimeIntervalBetweenTimeInstants._set_stop_time_instant_metadata, stop_time_instant)

    _property_names[start_time_instant] = "start_time_instant"
    _property_names[stop_time_instant] = "stop_time_instant"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalBetweenTimeInstants."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalBetweenTimeInstants)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalBetweenTimeInstants, [TimeToolTimeIntervalBetweenTimeInstants, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5441730547251256494, 11273521191384897433), TimeToolTimeIntervalBetweenTimeInstants)
agcls.AgTypeNameMap["TimeToolTimeIntervalBetweenTimeInstants"] = TimeToolTimeIntervalBetweenTimeInstants

class TimeToolTimeIntervalCollection(ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A collection of related interval lists."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollection."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolTimeIntervalCollection.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalCollection._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollection, [ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5320390972149596139, 11612889387071709314), TimeToolTimeIntervalCollection)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollection"] = TimeToolTimeIntervalCollection

class TimeToolTimeIntervalCollectionCondition(ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval list containing intervals during which specified condition is satisfied. Determination is performed within interval list using Sampling and Convergence parameters."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_condition_set_method_offset = 1
    _set_condition_set_method_offset = 2
    _get_custom_time_limits_method_offset = 3
    _set_custom_time_limits_method_offset = 4
    _get_use_custom_time_limits_method_offset = 5
    _set_use_custom_time_limits_method_offset = 6
    _get_save_data_option_method_offset = 7
    _set_save_data_option_method_offset = 8
    _get_sampling_method_offset = 9
    _set_sampling_method_offset = 10
    _get_convergence_method_offset = 11
    _set_convergence_method_offset = 12
    _metadata = {
        "iid_data" : (5695491852784106006, 14645292488236401817),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionCondition)

    _get_condition_set_metadata = { "offset" : _get_condition_set_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def condition_set(self) -> "ICalculationToolConditionSet":
        """Get/set the condition set object."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_condition_set_metadata)

    _set_condition_set_metadata = { "offset" : _set_condition_set_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolConditionSet"),) }
    @condition_set.setter
    def condition_set(self, condition_set:"ICalculationToolConditionSet") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_condition_set_metadata, condition_set)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """A custom interval list or a single interval. By default it is set to overall availability of host object. This determines time limits within which global minimum or maximum is sought. The time limits will be used if UseCustomTimeLimits is set to true."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_custom_time_limits_metadata, custom_time_limits)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Specify whether to use specified custom interval list (see CustomTimeLimits)."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Determine if computed time of extremum is saved/loaded, otherwise it is recomputed on load if necessary."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_save_data_option_metadata, save_data_option)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """A Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """A Convergence definition, which uses time tolerance to determine when time of extremum is found."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionCondition._metadata, TimeToolTimeIntervalCollectionCondition._set_convergence_metadata, convergence)

    _property_names[condition_set] = "condition_set"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionCondition)
        ITimeToolTimeIntervalCollection.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalCollection._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionCondition, [TimeToolTimeIntervalCollectionCondition, ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5096431107182997469, 15497237714408753540), TimeToolTimeIntervalCollectionCondition)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionCondition"] = TimeToolTimeIntervalCollectionCondition

class TimeToolTimeIntervalCollectionFactory(SupportsDeleteCallback):
    """The factory creates collections of event interval lists."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_lighting_method_offset = 2
    _create_signaled_method_offset = 3
    _is_type_supported_method_offset = 4
    _create_satisfaction_method_offset = 5
    _metadata = {
        "iid_data" : (4929297514598945487, 10137873554628665765),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(EventIntervalCollectionType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"EventIntervalCollectionType") -> "ITimeToolTimeIntervalCollection":
        """Create and register an event interval collection using specified name, description, and type."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionFactory._metadata, TimeToolTimeIntervalCollectionFactory._create_metadata, name, description, type, OutArg())

    _create_lighting_metadata = { "offset" : _create_lighting_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_lighting(self, name:str, description:str) -> "ITimeToolTimeIntervalCollection":
        """Create an event interval collection defined by computing sunlight, penumbra and umbra intervals as seen at specified location using specified selection of eclipsing bodies."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionFactory._metadata, TimeToolTimeIntervalCollectionFactory._create_lighting_metadata, name, description, OutArg())

    _create_signaled_metadata = { "offset" : _create_signaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_signaled(self, name:str, description:str) -> "ITimeToolTimeIntervalCollection":
        """Create an event interval collection recorded at target clock location by performing signal transmission of original interval list collection between base and target clock locations."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionFactory._metadata, TimeToolTimeIntervalCollectionFactory._create_signaled_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalCollectionType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"EventIntervalCollectionType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionFactory._metadata, TimeToolTimeIntervalCollectionFactory._is_type_supported_metadata, type, OutArg())

    _create_satisfaction_metadata = { "offset" : _create_satisfaction_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_satisfaction(self, name:str, description:str) -> "ITimeToolTimeIntervalCollection":
        """Create an event interval collection containing intervals during which condition set is satisfied."""
        return self._intf.invoke(TimeToolTimeIntervalCollectionFactory._metadata, TimeToolTimeIntervalCollectionFactory._create_satisfaction_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionFactory, [TimeToolTimeIntervalCollectionFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5176155607638442896, 7177596217140786077), TimeToolTimeIntervalCollectionFactory)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionFactory"] = TimeToolTimeIntervalCollectionFactory

class TimeToolTimeIntervalCollectionLighting(ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by computing sunlight, penumbra and umbra intervals as seen at specified location using specified selection of eclipsing bodies."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_location_method_offset = 1
    _set_location_method_offset = 2
    _get_eclipsing_bodies_method_offset = 3
    _set_eclipsing_bodies_method_offset = 4
    _get_use_object_eclipsing_bodies_method_offset = 5
    _set_use_object_eclipsing_bodies_method_offset = 6
    _metadata = {
        "iid_data" : (5361006311311084485, 10468446585511132859),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionLighting)

    _get_location_metadata = { "offset" : _get_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def location(self) -> "IVectorGeometryToolPoint":
        """The location point to compute sunlight, penumbra and umbra."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._get_location_metadata)

    _set_location_metadata = { "offset" : _set_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @location.setter
    def location(self, location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._set_location_metadata, location)

    _get_eclipsing_bodies_metadata = { "offset" : _get_eclipsing_bodies_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def eclipsing_bodies(self) -> list:
        """A custom list of eclipsing bodies. This list is used if UseObjectEclipsingBodies is set to false."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._get_eclipsing_bodies_metadata)

    _set_eclipsing_bodies_metadata = { "offset" : _set_eclipsing_bodies_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @eclipsing_bodies.setter
    def eclipsing_bodies(self, eclipsing_bodies:list) -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._set_eclipsing_bodies_metadata, eclipsing_bodies)

    _get_use_object_eclipsing_bodies_metadata = { "offset" : _get_use_object_eclipsing_bodies_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_object_eclipsing_bodies(self) -> bool:
        """When true, configure eclipsing bodies list based on that of parent STK Object."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._get_use_object_eclipsing_bodies_metadata)

    _set_use_object_eclipsing_bodies_metadata = { "offset" : _set_use_object_eclipsing_bodies_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_object_eclipsing_bodies.setter
    def use_object_eclipsing_bodies(self, use_object_eclipsing_bodies:bool) -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionLighting._metadata, TimeToolTimeIntervalCollectionLighting._set_use_object_eclipsing_bodies_metadata, use_object_eclipsing_bodies)

    _property_names[location] = "location"
    _property_names[eclipsing_bodies] = "eclipsing_bodies"
    _property_names[use_object_eclipsing_bodies] = "use_object_eclipsing_bodies"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionLighting."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionLighting)
        ITimeToolTimeIntervalCollection.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalCollection._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionLighting, [TimeToolTimeIntervalCollectionLighting, ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4778372489419407290, 1841925492488396976), TimeToolTimeIntervalCollectionLighting)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionLighting"] = TimeToolTimeIntervalCollectionLighting

class TimeToolTimeIntervalCollectionSignaled(ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine what interval list collection is recorded at target clock location by performing signal transmission of original interval list collection between base and target clock locations..."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_collection_method_offset = 1
    _set_original_collection_method_offset = 2
    _get_signal_sense_method_offset = 3
    _set_signal_sense_method_offset = 4
    _get_base_clock_location_method_offset = 5
    _set_base_clock_location_method_offset = 6
    _get_target_clock_location_method_offset = 7
    _set_target_clock_location_method_offset = 8
    _get_signal_delay_method_offset = 9
    _set_signal_delay_method_offset = 10
    _metadata = {
        "iid_data" : (5102559603137595011, 6446915665441949320),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalCollectionSignaled)

    _get_original_collection_metadata = { "offset" : _get_original_collection_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_collection(self) -> "ITimeToolTimeIntervalCollection":
        """The original interval list collection."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._get_original_collection_metadata)

    _set_original_collection_metadata = { "offset" : _set_original_collection_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalCollection"),) }
    @original_collection.setter
    def original_collection(self, original_collection:"ITimeToolTimeIntervalCollection") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._set_original_collection_metadata, original_collection)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @property
    def signal_sense(self) -> "SignalDirectionType":
        """The direction of the signal, whether you are Transmitting or Receiving from the Base Clock Location."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @signal_sense.setter
    def signal_sense(self, signal_sense:"SignalDirectionType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._set_signal_sense_metadata, signal_sense)

    _get_base_clock_location_metadata = { "offset" : _get_base_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_clock_location(self) -> "IVectorGeometryToolPoint":
        """The base clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._get_base_clock_location_metadata)

    _set_base_clock_location_metadata = { "offset" : _set_base_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @base_clock_location.setter
    def base_clock_location(self, base_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._set_base_clock_location_metadata, base_clock_location)

    _get_target_clock_location_metadata = { "offset" : _get_target_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_clock_location(self) -> "IVectorGeometryToolPoint":
        """The target clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._get_target_clock_location_metadata)

    _set_target_clock_location_metadata = { "offset" : _set_target_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @target_clock_location.setter
    def target_clock_location(self, target_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._set_target_clock_location_metadata, target_clock_location)

    _get_signal_delay_metadata = { "offset" : _get_signal_delay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def signal_delay(self) -> "IAnalysisWorkbenchSignalDelay":
        """The Signal delay definition, which includes signal transmission, time delay convergence and signal path reference system."""
        return self._intf.get_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._get_signal_delay_metadata)

    _set_signal_delay_metadata = { "offset" : _set_signal_delay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSignalDelay"),) }
    @signal_delay.setter
    def signal_delay(self, signal_delay:"IAnalysisWorkbenchSignalDelay") -> None:
        return self._intf.set_property(TimeToolTimeIntervalCollectionSignaled._metadata, TimeToolTimeIntervalCollectionSignaled._set_signal_delay_metadata, signal_delay)

    _property_names[original_collection] = "original_collection"
    _property_names[signal_sense] = "signal_sense"
    _property_names[base_clock_location] = "base_clock_location"
    _property_names[target_clock_location] = "target_clock_location"
    _property_names[signal_delay] = "signal_delay"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalCollectionSignaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalCollectionSignaled)
        ITimeToolTimeIntervalCollection.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalCollection._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalCollectionSignaled, [TimeToolTimeIntervalCollectionSignaled, ITimeToolTimeIntervalCollection, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4681123355224258665, 3519235201891138198), TimeToolTimeIntervalCollectionSignaled)
agcls.AgTypeNameMap["TimeToolTimeIntervalCollectionSignaled"] = TimeToolTimeIntervalCollectionSignaled

class TimeToolTimeIntervalFactory(SupportsDeleteCallback):
    """The factory creates event intervals."""

    _num_methods = 9
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_fixed_method_offset = 2
    _create_fixed_duration_method_offset = 3
    _create_between_time_instants_method_offset = 4
    _create_from_interval_list_method_offset = 5
    _create_scaled_method_offset = 6
    _create_signaled_method_offset = 7
    _create_time_offset_method_offset = 8
    _is_type_supported_method_offset = 9
    _metadata = {
        "iid_data" : (4781822647992058058, 10338186051262684820),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(EventIntervalType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"EventIntervalType") -> "ITimeToolTimeInterval":
        """Create and register an interval using specified name, description, and type."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_metadata, name, description, type, OutArg())

    _create_fixed_metadata = { "offset" : _create_fixed_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval defined between two explicitly specified start and stop times."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_fixed_metadata, name, description, OutArg())

    _create_fixed_duration_metadata = { "offset" : _create_fixed_duration_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed_duration(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval of fixed duration specified using start and stop offsets relative to specified reference time instant."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_fixed_duration_metadata, name, description, OutArg())

    _create_between_time_instants_metadata = { "offset" : _create_between_time_instants_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_between_time_instants(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval using specified start and stop time instants."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_between_time_instants_metadata, name, description, OutArg())

    _create_from_interval_list_metadata = { "offset" : _create_from_interval_list_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_interval_list(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval from a specified interval list by using one of several selection methods."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_from_interval_list_metadata, name, description, OutArg())

    _create_scaled_metadata = { "offset" : _create_scaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_scaled(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval by scaling an original interval using either absolute or relative scale."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_scaled_metadata, name, description, OutArg())

    _create_signaled_metadata = { "offset" : _create_signaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_signaled(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval that is recorded at target clock location by performing signal transmission of original interval between base and target clock locations."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_signaled_metadata, name, description, OutArg())

    _create_time_offset_metadata = { "offset" : _create_time_offset_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_time_offset(self, name:str, description:str) -> "ITimeToolTimeInterval":
        """Create an interval defined by shifting the specified reference interval by a fixed time offset."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._create_time_offset_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"EventIntervalType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(TimeToolTimeIntervalFactory._metadata, TimeToolTimeIntervalFactory._is_type_supported_metadata, type, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalFactory, [TimeToolTimeIntervalFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5271144977203828184, 14422681085791257527), TimeToolTimeIntervalFactory)
agcls.AgTypeNameMap["TimeToolTimeIntervalFactory"] = TimeToolTimeIntervalFactory

class TimeToolTimeIntervalFixed(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval defined between two explicitly specified start and stop times. Stop date/time is required to be at or after start."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_start_time_method_offset = 1
    _get_stop_time_method_offset = 2
    _set_interval_method_offset = 3
    _metadata = {
        "iid_data" : (4700629706653582627, 17112466619207988153),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalFixed)

    _get_start_time_metadata = { "offset" : _get_start_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def start_time(self) -> typing.Any:
        """The start time of the interval."""
        return self._intf.get_property(TimeToolTimeIntervalFixed._metadata, TimeToolTimeIntervalFixed._get_start_time_metadata)

    _get_stop_time_metadata = { "offset" : _get_stop_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def stop_time(self) -> typing.Any:
        """The stop time of the interval."""
        return self._intf.get_property(TimeToolTimeIntervalFixed._metadata, TimeToolTimeIntervalFixed._get_stop_time_metadata)

    _set_interval_metadata = { "offset" : _set_interval_method_offset,
            "arg_types" : (agcom.Variant, agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantArg,) }
    def set_interval(self, start_epoch:typing.Any, stop_epoch:typing.Any) -> None:
        """Set interval's start and stop times."""
        return self._intf.invoke(TimeToolTimeIntervalFixed._metadata, TimeToolTimeIntervalFixed._set_interval_metadata, start_epoch, stop_epoch)

    _property_names[start_time] = "start_time"
    _property_names[stop_time] = "stop_time"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalFixed."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalFixed)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalFixed, [TimeToolTimeIntervalFixed, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5434382701560354571, 17399214976131420314), TimeToolTimeIntervalFixed)
agcls.AgTypeNameMap["TimeToolTimeIntervalFixed"] = TimeToolTimeIntervalFixed

class TimeToolTimeIntervalFixedDuration(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval of fixed duration specified using start and stop offsets relative to specified reference time instant."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_time_instant_method_offset = 1
    _set_reference_time_instant_method_offset = 2
    _get_start_offset_method_offset = 3
    _set_start_offset_method_offset = 4
    _get_stop_offset_method_offset = 5
    _set_stop_offset_method_offset = 6
    _metadata = {
        "iid_data" : (5616504913797350155, 4594426271094609308),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalFixedDuration)

    _get_reference_time_instant_metadata = { "offset" : _get_reference_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_time_instant(self) -> "ITimeToolInstant":
        """The reference time instant."""
        return self._intf.get_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._get_reference_time_instant_metadata)

    _set_reference_time_instant_metadata = { "offset" : _set_reference_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @reference_time_instant.setter
    def reference_time_instant(self, reference_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._set_reference_time_instant_metadata, reference_time_instant)

    _get_start_offset_metadata = { "offset" : _get_start_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def start_offset(self) -> float:
        """The start time offset value."""
        return self._intf.get_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._get_start_offset_metadata)

    _set_start_offset_metadata = { "offset" : _set_start_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @start_offset.setter
    def start_offset(self, start_offset:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._set_start_offset_metadata, start_offset)

    _get_stop_offset_metadata = { "offset" : _get_stop_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def stop_offset(self) -> float:
        """The stop time offset value."""
        return self._intf.get_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._get_stop_offset_metadata)

    _set_stop_offset_metadata = { "offset" : _set_stop_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @stop_offset.setter
    def stop_offset(self, stop_offset:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalFixedDuration._metadata, TimeToolTimeIntervalFixedDuration._set_stop_offset_metadata, stop_offset)

    _property_names[reference_time_instant] = "reference_time_instant"
    _property_names[start_offset] = "start_offset"
    _property_names[stop_offset] = "stop_offset"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalFixedDuration."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalFixedDuration)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalFixedDuration, [TimeToolTimeIntervalFixedDuration, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5617013323472138033, 5900512052836164536), TimeToolTimeIntervalFixedDuration)
agcls.AgTypeNameMap["TimeToolTimeIntervalFixedDuration"] = TimeToolTimeIntervalFixedDuration

class TimeToolTimeIntervalFromIntervalList(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval created from specified interval list by using one of several selection methods."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_intervals_method_offset = 1
    _set_reference_intervals_method_offset = 2
    _get_interval_selection_method_offset = 3
    _set_interval_selection_method_offset = 4
    _get_interval_number_method_offset = 5
    _set_interval_number_method_offset = 6
    _metadata = {
        "iid_data" : (5212467572212542000, 10268755993903263125),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalFromIntervalList)

    _get_reference_intervals_metadata = { "offset" : _get_reference_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_intervals(self) -> "ITimeToolTimeIntervalList":
        """The reference interval list."""
        return self._intf.get_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._get_reference_intervals_metadata)

    _set_reference_intervals_metadata = { "offset" : _set_reference_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @reference_intervals.setter
    def reference_intervals(self, reference_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._set_reference_intervals_metadata, reference_intervals)

    _get_interval_selection_metadata = { "offset" : _get_interval_selection_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalFromIntervalListSelectionType),) }
    @property
    def interval_selection(self) -> "IntervalFromIntervalListSelectionType":
        """The method used to select an interval from the reference interval list."""
        return self._intf.get_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._get_interval_selection_metadata)

    _set_interval_selection_metadata = { "offset" : _set_interval_selection_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntervalFromIntervalListSelectionType),) }
    @interval_selection.setter
    def interval_selection(self, interval_selection:"IntervalFromIntervalListSelectionType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._set_interval_selection_metadata, interval_selection)

    _get_interval_number_metadata = { "offset" : _get_interval_number_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def interval_number(self) -> int:
        """An interval number. Applicable only if IntervalSelection is IntervalSelectionFromStart or IntervalSelectionFromEnd."""
        return self._intf.get_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._get_interval_number_metadata)

    _set_interval_number_metadata = { "offset" : _set_interval_number_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @interval_number.setter
    def interval_number(self, interval_number:int) -> None:
        return self._intf.set_property(TimeToolTimeIntervalFromIntervalList._metadata, TimeToolTimeIntervalFromIntervalList._set_interval_number_metadata, interval_number)

    _property_names[reference_intervals] = "reference_intervals"
    _property_names[interval_selection] = "interval_selection"
    _property_names[interval_number] = "interval_number"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalFromIntervalList."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalFromIntervalList)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalFromIntervalList, [TimeToolTimeIntervalFromIntervalList, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5605488631452751831, 17424609706075019906), TimeToolTimeIntervalFromIntervalList)
agcls.AgTypeNameMap["TimeToolTimeIntervalFromIntervalList"] = TimeToolTimeIntervalFromIntervalList

class TimeToolTimeIntervalList(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """An ordered list of time intervals."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalList."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalList, [ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4931610229651409517, 5176676340500202392), TimeToolTimeIntervalList)
agcls.AgTypeNameMap["TimeToolTimeIntervalList"] = TimeToolTimeIntervalList

class TimeToolTimeIntervalListCondition(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval list containing intervals during which specified condition is satisfied. Determination is performed within interval list using Sampling and Convergence parameters."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_condition_method_offset = 1
    _set_condition_method_offset = 2
    _get_custom_time_limits_method_offset = 3
    _set_custom_time_limits_method_offset = 4
    _get_use_custom_time_limits_method_offset = 5
    _set_use_custom_time_limits_method_offset = 6
    _get_save_data_option_method_offset = 7
    _set_save_data_option_method_offset = 8
    _get_sampling_method_offset = 9
    _set_sampling_method_offset = 10
    _get_convergence_method_offset = 11
    _set_convergence_method_offset = 12
    _metadata = {
        "iid_data" : (5026707234268337802, 16142642421116537497),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListCondition)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def condition(self) -> "ICalculationToolCondition":
        """The condition component."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_condition_metadata)

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"),) }
    @condition.setter
    def condition(self, condition:"ICalculationToolCondition") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_condition_metadata, condition)

    _get_custom_time_limits_metadata = { "offset" : _get_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def custom_time_limits(self) -> "ITimeToolTimeIntervalList":
        """The interval list or single interval within which intervals of satisfaction are sought. The specified value is used if UseCustomTimeLimits is true. The default is set to overall availability of host object..."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_custom_time_limits_metadata)

    _set_custom_time_limits_metadata = { "offset" : _set_custom_time_limits_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @custom_time_limits.setter
    def custom_time_limits(self, custom_time_limits:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_custom_time_limits_metadata, custom_time_limits)

    _get_use_custom_time_limits_metadata = { "offset" : _get_use_custom_time_limits_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_time_limits(self) -> bool:
        """Indicate whether to use specified custom time limits (see CustomTimeLimits)."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_use_custom_time_limits_metadata)

    _set_use_custom_time_limits_metadata = { "offset" : _set_use_custom_time_limits_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_time_limits.setter
    def use_custom_time_limits(self, use_custom_time_limits:bool) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_use_custom_time_limits_metadata, use_custom_time_limits)

    _get_save_data_option_metadata = { "offset" : _get_save_data_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @property
    def save_data_option(self) -> "SaveDataType":
        """Determine if computed intervals of satisfaction are saved/loaded, or recomputed on load if necessary."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_save_data_option_metadata)

    _set_save_data_option_metadata = { "offset" : _set_save_data_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SaveDataType),) }
    @save_data_option.setter
    def save_data_option(self, save_data_option:"SaveDataType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_save_data_option_metadata, save_data_option)

    _get_sampling_metadata = { "offset" : _get_sampling_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling(self) -> "IAnalysisWorkbenchSampling":
        """The Sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_sampling_metadata)

    _set_sampling_metadata = { "offset" : _set_sampling_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSampling"),) }
    @sampling.setter
    def sampling(self, sampling:"IAnalysisWorkbenchSampling") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_sampling_metadata, sampling)

    _get_convergence_metadata = { "offset" : _get_convergence_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def convergence(self) -> "IAnalysisWorkbenchConvergence":
        """The Convergence definition, which uses time tolerance to determine when times for intervals of satisfaction are found."""
        return self._intf.get_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._get_convergence_metadata)

    _set_convergence_metadata = { "offset" : _set_convergence_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchConvergence"),) }
    @convergence.setter
    def convergence(self, convergence:"IAnalysisWorkbenchConvergence") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListCondition._metadata, TimeToolTimeIntervalListCondition._set_convergence_metadata, convergence)

    _property_names[condition] = "condition"
    _property_names[custom_time_limits] = "custom_time_limits"
    _property_names[use_custom_time_limits] = "use_custom_time_limits"
    _property_names[save_data_option] = "save_data_option"
    _property_names[sampling] = "sampling"
    _property_names[convergence] = "convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListCondition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListCondition)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListCondition, [TimeToolTimeIntervalListCondition, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5695291326310786302, 3161624783042523292), TimeToolTimeIntervalListCondition)
agcls.AgTypeNameMap["TimeToolTimeIntervalListCondition"] = TimeToolTimeIntervalListCondition

class TimeToolTimeIntervalListFactory(SupportsDeleteCallback):
    """The factory creates event interval lists."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_merged_method_offset = 2
    _create_filtered_method_offset = 3
    _create_from_condition_method_offset = 4
    _create_scaled_method_offset = 5
    _create_signaled_method_offset = 6
    _create_time_offset_method_offset = 7
    _is_type_supported_method_offset = 8
    _create_from_file_method_offset = 9
    _create_fixed_method_offset = 10
    _metadata = {
        "iid_data" : (4817877923312545831, 15527202984233884858),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(EventIntervalListType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"EventIntervalListType") -> "ITimeToolTimeIntervalList":
        """Create and register an interval list using specified name, description, and type."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_metadata, name, description, type, OutArg())

    _create_merged_metadata = { "offset" : _create_merged_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_merged(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list by merging two constituent interval lists using specified logical operation."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_merged_metadata, name, description, OutArg())

    _create_filtered_metadata = { "offset" : _create_filtered_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_filtered(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list by filtering intervals from original interval list using specified filtering method."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_filtered_metadata, name, description, OutArg())

    _create_from_condition_metadata = { "offset" : _create_from_condition_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_condition(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list containing intervals during which specified condition is satisfied."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_from_condition_metadata, name, description, OutArg())

    _create_scaled_metadata = { "offset" : _create_scaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_scaled(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list defined by scaling every interval in original interval list using either absolute or relative scale."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_scaled_metadata, name, description, OutArg())

    _create_signaled_metadata = { "offset" : _create_signaled_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_signaled(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list recorded at the target clock location by performing signal transmission of original interval list between base and target clock locations."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_signaled_metadata, name, description, OutArg())

    _create_time_offset_metadata = { "offset" : _create_time_offset_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_time_offset(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list defined by shifting the specified reference interval list by a fixed time offset."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_time_offset_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(EventIntervalListType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"EventIntervalListType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._is_type_supported_metadata, type, OutArg())

    _create_from_file_metadata = { "offset" : _create_from_file_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_file(self, name:str, description:str, file_path:str) -> "ITimeToolTimeIntervalList":
        """Create an interval list based on specified interval file."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_from_file_metadata, name, description, file_path, OutArg())

    _create_fixed_metadata = { "offset" : _create_fixed_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_fixed(self, name:str, description:str) -> "ITimeToolTimeIntervalList":
        """Interval list defined by time ordered non-overlapping intervals each explicitly specified by its start and stop times. Stop date/time is required to be at or after start for each interval."""
        return self._intf.invoke(TimeToolTimeIntervalListFactory._metadata, TimeToolTimeIntervalListFactory._create_fixed_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListFactory, [TimeToolTimeIntervalListFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5520594195372287577, 9965002346799811229), TimeToolTimeIntervalListFactory)
agcls.AgTypeNameMap["TimeToolTimeIntervalListFactory"] = TimeToolTimeIntervalListFactory

class TimeToolTimeIntervalListFile(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval list loaded from specified interval file - ASCII file with .int extension. See STK help."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_filename_method_offset = 1
    _set_filename_method_offset = 2
    _reload_method_offset = 3
    _get_file_span_method_offset = 4
    _metadata = {
        "iid_data" : (4969943485407135988, 4215037281840912568),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListFile)

    _get_filename_metadata = { "offset" : _get_filename_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def filename(self) -> str:
        """The path of an external file that contains the time interval list."""
        return self._intf.get_property(TimeToolTimeIntervalListFile._metadata, TimeToolTimeIntervalListFile._get_filename_metadata)

    _set_filename_metadata = { "offset" : _set_filename_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @filename.setter
    def filename(self, filename:str) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListFile._metadata, TimeToolTimeIntervalListFile._set_filename_metadata, filename)

    _reload_metadata = { "offset" : _reload_method_offset,
            "arg_types" : (),
            "marshallers" : () }
    def reload(self) -> None:
        """Reload the interval list file."""
        return self._intf.invoke(TimeToolTimeIntervalListFile._metadata, TimeToolTimeIntervalListFile._reload_metadata, )

    _get_file_span_metadata = { "offset" : _get_file_span_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_file_span(self) -> "TimeToolTimeIntervalResult":
        """Compute the interval list file span."""
        return self._intf.invoke(TimeToolTimeIntervalListFile._metadata, TimeToolTimeIntervalListFile._get_file_span_metadata, OutArg())

    _property_names[filename] = "filename"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListFile."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListFile)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListFile, [TimeToolTimeIntervalListFile, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4939698509197330113, 12995431996803870356), TimeToolTimeIntervalListFile)
agcls.AgTypeNameMap["TimeToolTimeIntervalListFile"] = TimeToolTimeIntervalListFile

class TimeToolTimeIntervalListFiltered(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Defined by filtering intervals from original interval list using specified filtering method."""

    _num_methods = 5
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_intervals_method_offset = 1
    _set_original_intervals_method_offset = 2
    _get_filter_factory_method_offset = 3
    _get_filter_method_offset = 4
    _set_filter_method_offset = 5
    _metadata = {
        "iid_data" : (5362704445766881861, 6912538458780151178),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListFiltered)

    _get_original_intervals_metadata = { "offset" : _get_original_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_intervals(self) -> "ITimeToolTimeIntervalList":
        """The original interval list."""
        return self._intf.get_property(TimeToolTimeIntervalListFiltered._metadata, TimeToolTimeIntervalListFiltered._get_original_intervals_metadata)

    _set_original_intervals_metadata = { "offset" : _set_original_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @original_intervals.setter
    def original_intervals(self, original_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListFiltered._metadata, TimeToolTimeIntervalListFiltered._set_original_intervals_metadata, original_intervals)

    _get_filter_factory_metadata = { "offset" : _get_filter_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def filter_factory(self) -> "TimeToolPruneFilterFactory":
        """Get the prune filter factory."""
        return self._intf.get_property(TimeToolTimeIntervalListFiltered._metadata, TimeToolTimeIntervalListFiltered._get_filter_factory_metadata)

    _get_filter_metadata = { "offset" : _get_filter_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def filter(self) -> "ITimeToolPruneFilter":
        """The pruning filter."""
        return self._intf.get_property(TimeToolTimeIntervalListFiltered._metadata, TimeToolTimeIntervalListFiltered._get_filter_metadata)

    _set_filter_metadata = { "offset" : _set_filter_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolPruneFilter"),) }
    @filter.setter
    def filter(self, filter:"ITimeToolPruneFilter") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListFiltered._metadata, TimeToolTimeIntervalListFiltered._set_filter_metadata, filter)

    _property_names[original_intervals] = "original_intervals"
    _property_names[filter_factory] = "filter_factory"
    _property_names[filter] = "filter"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListFiltered."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListFiltered)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListFiltered, [TimeToolTimeIntervalListFiltered, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4625969561133636137, 1394880749826965914), TimeToolTimeIntervalListFiltered)
agcls.AgTypeNameMap["TimeToolTimeIntervalListFiltered"] = TimeToolTimeIntervalListFiltered

class TimeToolTimeIntervalListFixed(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval list defined by time ordered non-overlapping intervals each explicitly specified by its start and stop times. Stop date/time is required to be at or after start for each interval."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_intervals_method_offset = 1
    _set_intervals_method_offset = 2
    _metadata = {
        "iid_data" : (5035651772834666939, 4266315594091686530),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListFixed)

    _get_intervals_metadata = { "offset" : _get_intervals_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def get_intervals(self) -> list:
        """Get intervals with explicitly specified start and stop times from interval list. The method returns a one-dimensional array which elements are 2-tuples of intervals' start/stop times converted according to the current unit preferences."""
        return self._intf.invoke(TimeToolTimeIntervalListFixed._metadata, TimeToolTimeIntervalListFixed._get_intervals_metadata, OutArg())

    _set_intervals_metadata = { "offset" : _set_intervals_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    def set_intervals(self, intervals:list) -> None:
        """Set interval list from intervals with explicitly specified start and stop times. The method takes a one-dimensional array which elements are 2-tuples of intervals' start/stop times converted according to the current unit preferences."""
        return self._intf.invoke(TimeToolTimeIntervalListFixed._metadata, TimeToolTimeIntervalListFixed._set_intervals_metadata, intervals)


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListFixed."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListFixed)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListFixed, [TimeToolTimeIntervalListFixed, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5035876072746351075, 10259654735031477925), TimeToolTimeIntervalListFixed)
agcls.AgTypeNameMap["TimeToolTimeIntervalListFixed"] = TimeToolTimeIntervalListFixed

class TimeToolTimeIntervalListMerged(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval list created by merging two constituent interval lists using specified logical operation. It is possible to select either interval list or interval types for either or both constituents."""

    _num_methods = 15
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_interval_list_or_interval_a_method_offset = 1
    _get_interval_list_or_interval_b_method_offset = 2
    _get_merge_operation_method_offset = 3
    _set_merge_operation_method_offset = 4
    _set_interval_list_a_method_offset = 5
    _set_interval_a_method_offset = 6
    _set_interval_list_b_method_offset = 7
    _set_interval_b_method_offset = 8
    _add_interval_method_offset = 9
    _add_interval_list_method_offset = 10
    _set_interval_method_offset = 11
    _set_interval_list_method_offset = 12
    _get_time_component_method_offset = 13
    _get_time_component_size_method_offset = 14
    _remove_time_component_method_offset = 15
    _metadata = {
        "iid_data" : (5336865813588725345, 16270761028594191753),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListMerged)

    _get_interval_list_or_interval_a_metadata = { "offset" : _get_interval_list_or_interval_a_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval_list_or_interval_a(self) -> "IAnalysisWorkbenchComponent":
        """The interval list or interval A."""
        return self._intf.get_property(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._get_interval_list_or_interval_a_metadata)

    _get_interval_list_or_interval_b_metadata = { "offset" : _get_interval_list_or_interval_b_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def interval_list_or_interval_b(self) -> "IAnalysisWorkbenchComponent":
        """The interval list or interval B."""
        return self._intf.get_property(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._get_interval_list_or_interval_b_metadata)

    _get_merge_operation_metadata = { "offset" : _get_merge_operation_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(EventListMergeOperation),) }
    @property
    def merge_operation(self) -> "EventListMergeOperation":
        """The merge operation."""
        return self._intf.get_property(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._get_merge_operation_metadata)

    _set_merge_operation_metadata = { "offset" : _set_merge_operation_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(EventListMergeOperation),) }
    @merge_operation.setter
    def merge_operation(self, merge_operation:"EventListMergeOperation") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_merge_operation_metadata, merge_operation)

    _set_interval_list_a_metadata = { "offset" : _set_interval_list_a_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    def set_interval_list_a(self, ref_intervals:"ITimeToolTimeIntervalList") -> None:
        """Set the interval list A."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_list_a_metadata, ref_intervals)

    _set_interval_a_metadata = { "offset" : _set_interval_a_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    def set_interval_a(self, ref_intervals:"ITimeToolTimeInterval") -> None:
        """Set the interval A."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_a_metadata, ref_intervals)

    _set_interval_list_b_metadata = { "offset" : _set_interval_list_b_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    def set_interval_list_b(self, ref_intervals:"ITimeToolTimeIntervalList") -> None:
        """Set the interval list B."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_list_b_metadata, ref_intervals)

    _set_interval_b_metadata = { "offset" : _set_interval_b_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    def set_interval_b(self, ref_intervals:"ITimeToolTimeInterval") -> None:
        """Set the interval B."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_b_metadata, ref_intervals)

    _add_interval_metadata = { "offset" : _add_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    def add_interval(self, ref_intervals:"ITimeToolTimeInterval") -> None:
        """Add interval."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._add_interval_metadata, ref_intervals)

    _add_interval_list_metadata = { "offset" : _add_interval_list_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    def add_interval_list(self, ref_intervals:"ITimeToolTimeIntervalList") -> None:
        """Add interval list."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._add_interval_list_metadata, ref_intervals)

    _set_interval_metadata = { "offset" : _set_interval_method_offset,
            "arg_types" : (agcom.PVOID, agcom.INT,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"), agmarshall.IntArg,) }
    def set_interval(self, ref_intervals:"ITimeToolTimeInterval", pos:int) -> None:
        """Set the interval at given index."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_metadata, ref_intervals, pos)

    _set_interval_list_metadata = { "offset" : _set_interval_list_method_offset,
            "arg_types" : (agcom.PVOID, agcom.INT,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"), agmarshall.IntArg,) }
    def set_interval_list(self, ref_intervals:"ITimeToolTimeIntervalList", pos:int) -> None:
        """Set the interval list at given index."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._set_interval_list_metadata, ref_intervals, pos)

    _get_time_component_metadata = { "offset" : _get_time_component_method_offset,
            "arg_types" : (agcom.INT, POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.IntArg, agmarshall.BStrArg,) }
    def get_time_component(self, pos:int) -> str:
        """Get time component at given position."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._get_time_component_metadata, pos, OutArg())

    _get_time_component_size_metadata = { "offset" : _get_time_component_size_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    def get_time_component_size(self) -> int:
        """Get time component list size."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._get_time_component_size_metadata, OutArg())

    _remove_time_component_metadata = { "offset" : _remove_time_component_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    def remove_time_component(self, pos:int) -> None:
        """Remove time component at given position."""
        return self._intf.invoke(TimeToolTimeIntervalListMerged._metadata, TimeToolTimeIntervalListMerged._remove_time_component_metadata, pos)

    _property_names[interval_list_or_interval_a] = "interval_list_or_interval_a"
    _property_names[interval_list_or_interval_b] = "interval_list_or_interval_b"
    _property_names[merge_operation] = "merge_operation"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListMerged."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListMerged)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListMerged, [TimeToolTimeIntervalListMerged, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5271487708003070052, 15898448361769664948), TimeToolTimeIntervalListMerged)
agcls.AgTypeNameMap["TimeToolTimeIntervalListMerged"] = TimeToolTimeIntervalListMerged

class TimeToolTimeIntervalListScaled(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval List defined by scaling every interval in original interval list using either absolute or relative scale. If resulting interval's start becomes after its stop, the interval is removed from scaled list..."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_intervals_method_offset = 1
    _set_original_intervals_method_offset = 2
    _get_absolute_increment_method_offset = 3
    _set_absolute_increment_method_offset = 4
    _get_relative_increment_method_offset = 5
    _set_relative_increment_method_offset = 6
    _get_use_absolute_increment_method_offset = 7
    _set_use_absolute_increment_method_offset = 8
    _metadata = {
        "iid_data" : (5066170491909867768, 15012607590605642908),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListScaled)

    _get_original_intervals_metadata = { "offset" : _get_original_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_intervals(self) -> "ITimeToolTimeIntervalList":
        """The original interval list."""
        return self._intf.get_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._get_original_intervals_metadata)

    _set_original_intervals_metadata = { "offset" : _set_original_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @original_intervals.setter
    def original_intervals(self, original_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._set_original_intervals_metadata, original_intervals)

    _get_absolute_increment_metadata = { "offset" : _get_absolute_increment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def absolute_increment(self) -> float:
        """The absolute increment value which creates a new interval list by expanding (or shortening if negative) every interval in the original interval list by shifting interval's start/stop times equally by half of specified increment value."""
        return self._intf.get_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._get_absolute_increment_metadata)

    _set_absolute_increment_metadata = { "offset" : _set_absolute_increment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @absolute_increment.setter
    def absolute_increment(self, absolute_increment:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._set_absolute_increment_metadata, absolute_increment)

    _get_relative_increment_metadata = { "offset" : _get_relative_increment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_increment(self) -> float:
        """The relative increment value from which absolute increment is obtained by multiplying relative value by interval duration..."""
        return self._intf.get_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._get_relative_increment_metadata)

    _set_relative_increment_metadata = { "offset" : _set_relative_increment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_increment.setter
    def relative_increment(self, relative_increment:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._set_relative_increment_metadata, relative_increment)

    _get_use_absolute_increment_metadata = { "offset" : _get_use_absolute_increment_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_absolute_increment(self) -> bool:
        """Specify whether to use absolute or relative increment."""
        return self._intf.get_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._get_use_absolute_increment_metadata)

    _set_use_absolute_increment_metadata = { "offset" : _set_use_absolute_increment_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_absolute_increment.setter
    def use_absolute_increment(self, use_absolute_increment:bool) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListScaled._metadata, TimeToolTimeIntervalListScaled._set_use_absolute_increment_metadata, use_absolute_increment)

    _property_names[original_intervals] = "original_intervals"
    _property_names[absolute_increment] = "absolute_increment"
    _property_names[relative_increment] = "relative_increment"
    _property_names[use_absolute_increment] = "use_absolute_increment"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListScaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListScaled)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListScaled, [TimeToolTimeIntervalListScaled, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5135112674452953850, 13376159879454619046), TimeToolTimeIntervalListScaled)
agcls.AgTypeNameMap["TimeToolTimeIntervalListScaled"] = TimeToolTimeIntervalListScaled

class TimeToolTimeIntervalListSignaled(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine what interval list is recorded at target clock location by performing signal transmission of original interval list between base and target clock locations..."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_intervals_method_offset = 1
    _set_original_intervals_method_offset = 2
    _get_signal_sense_method_offset = 3
    _set_signal_sense_method_offset = 4
    _get_base_clock_location_method_offset = 5
    _set_base_clock_location_method_offset = 6
    _get_target_clock_location_method_offset = 7
    _set_target_clock_location_method_offset = 8
    _get_signal_delay_method_offset = 9
    _set_signal_delay_method_offset = 10
    _metadata = {
        "iid_data" : (5408509073392837198, 4033135434938257290),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListSignaled)

    _get_original_intervals_metadata = { "offset" : _get_original_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_intervals(self) -> "ITimeToolTimeIntervalList":
        """The original time interval list."""
        return self._intf.get_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._get_original_intervals_metadata)

    _set_original_intervals_metadata = { "offset" : _set_original_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @original_intervals.setter
    def original_intervals(self, original_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._set_original_intervals_metadata, original_intervals)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @property
    def signal_sense(self) -> "SignalDirectionType":
        """The direction of the signal, whether you are Transmitting or Receiving from the Base Clock Location."""
        return self._intf.get_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @signal_sense.setter
    def signal_sense(self, signal_sense:"SignalDirectionType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._set_signal_sense_metadata, signal_sense)

    _get_base_clock_location_metadata = { "offset" : _get_base_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_clock_location(self) -> "IVectorGeometryToolPoint":
        """The base clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._get_base_clock_location_metadata)

    _set_base_clock_location_metadata = { "offset" : _set_base_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @base_clock_location.setter
    def base_clock_location(self, base_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._set_base_clock_location_metadata, base_clock_location)

    _get_target_clock_location_metadata = { "offset" : _get_target_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_clock_location(self) -> "IVectorGeometryToolPoint":
        """The target clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._get_target_clock_location_metadata)

    _set_target_clock_location_metadata = { "offset" : _set_target_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @target_clock_location.setter
    def target_clock_location(self, target_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._set_target_clock_location_metadata, target_clock_location)

    _get_signal_delay_metadata = { "offset" : _get_signal_delay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def signal_delay(self) -> "IAnalysisWorkbenchSignalDelay":
        """The Signal delay definition, which includes signal transmission, time delay convergence and signal path reference system."""
        return self._intf.get_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._get_signal_delay_metadata)

    _set_signal_delay_metadata = { "offset" : _set_signal_delay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSignalDelay"),) }
    @signal_delay.setter
    def signal_delay(self, signal_delay:"IAnalysisWorkbenchSignalDelay") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListSignaled._metadata, TimeToolTimeIntervalListSignaled._set_signal_delay_metadata, signal_delay)

    _property_names[original_intervals] = "original_intervals"
    _property_names[signal_sense] = "signal_sense"
    _property_names[base_clock_location] = "base_clock_location"
    _property_names[target_clock_location] = "target_clock_location"
    _property_names[signal_delay] = "signal_delay"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListSignaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListSignaled)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListSignaled, [TimeToolTimeIntervalListSignaled, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5092160389376989615, 317323019564379040), TimeToolTimeIntervalListSignaled)
agcls.AgTypeNameMap["TimeToolTimeIntervalListSignaled"] = TimeToolTimeIntervalListSignaled

class TimeToolTimeIntervalListTimeOffset(ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval List defined by shifting the specified reference interval list by a fixed time offset."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_intervals_method_offset = 1
    _set_reference_intervals_method_offset = 2
    _get_time_offset_method_offset = 3
    _set_time_offset_method_offset = 4
    _metadata = {
        "iid_data" : (4727529051409948883, 15038220645091515569),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalListTimeOffset)

    _get_reference_intervals_metadata = { "offset" : _get_reference_intervals_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_intervals(self) -> "ITimeToolTimeIntervalList":
        """The reference interval list."""
        return self._intf.get_property(TimeToolTimeIntervalListTimeOffset._metadata, TimeToolTimeIntervalListTimeOffset._get_reference_intervals_metadata)

    _set_reference_intervals_metadata = { "offset" : _set_reference_intervals_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeIntervalList"),) }
    @reference_intervals.setter
    def reference_intervals(self, reference_intervals:"ITimeToolTimeIntervalList") -> None:
        return self._intf.set_property(TimeToolTimeIntervalListTimeOffset._metadata, TimeToolTimeIntervalListTimeOffset._set_reference_intervals_metadata, reference_intervals)

    _get_time_offset_metadata = { "offset" : _get_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_offset(self) -> float:
        """The time offset."""
        return self._intf.get_property(TimeToolTimeIntervalListTimeOffset._metadata, TimeToolTimeIntervalListTimeOffset._get_time_offset_metadata)

    _set_time_offset_metadata = { "offset" : _set_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_offset.setter
    def time_offset(self, time_offset:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalListTimeOffset._metadata, TimeToolTimeIntervalListTimeOffset._set_time_offset_metadata, time_offset)

    _property_names[reference_intervals] = "reference_intervals"
    _property_names[time_offset] = "time_offset"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalListTimeOffset."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalListTimeOffset)
        ITimeToolTimeIntervalList.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeIntervalList._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalListTimeOffset, [TimeToolTimeIntervalListTimeOffset, ITimeToolTimeIntervalList, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5150624900174320789, 5541852546161415593), TimeToolTimeIntervalListTimeOffset)
agcls.AgTypeNameMap["TimeToolTimeIntervalListTimeOffset"] = TimeToolTimeIntervalListTimeOffset

class TimeToolTimeIntervalScaled(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval defined by scaling original interval using either absolute or relative scale. If resulting interval's start becomes after its stop, the interval becomes undefined."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_interval_method_offset = 1
    _set_original_interval_method_offset = 2
    _get_absolute_increment_method_offset = 3
    _set_absolute_increment_method_offset = 4
    _get_relative_increment_method_offset = 5
    _set_relative_increment_method_offset = 6
    _get_use_absolute_increment_method_offset = 7
    _set_use_absolute_increment_method_offset = 8
    _metadata = {
        "iid_data" : (5153652075809608699, 13548435864393192877),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalScaled)

    _get_original_interval_metadata = { "offset" : _get_original_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_interval(self) -> "ITimeToolTimeInterval":
        """The original interval."""
        return self._intf.get_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._get_original_interval_metadata)

    _set_original_interval_metadata = { "offset" : _set_original_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    @original_interval.setter
    def original_interval(self, original_interval:"ITimeToolTimeInterval") -> None:
        return self._intf.set_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._set_original_interval_metadata, original_interval)

    _get_absolute_increment_metadata = { "offset" : _get_absolute_increment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def absolute_increment(self) -> float:
        """The absolute increment value which creates a interval by expanding (or shortening if negative) the original interval by shifting its start/stop times equally by half of specified increment value."""
        return self._intf.get_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._get_absolute_increment_metadata)

    _set_absolute_increment_metadata = { "offset" : _set_absolute_increment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @absolute_increment.setter
    def absolute_increment(self, absolute_increment:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._set_absolute_increment_metadata, absolute_increment)

    _get_relative_increment_metadata = { "offset" : _get_relative_increment_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_increment(self) -> float:
        """The relative increment value from which absolute increment is obtained by multiplying relative value by interval duration..."""
        return self._intf.get_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._get_relative_increment_metadata)

    _set_relative_increment_metadata = { "offset" : _set_relative_increment_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_increment.setter
    def relative_increment(self, relative_increment:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._set_relative_increment_metadata, relative_increment)

    _get_use_absolute_increment_metadata = { "offset" : _get_use_absolute_increment_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_absolute_increment(self) -> bool:
        """Specify whether to use absolute or relative increment."""
        return self._intf.get_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._get_use_absolute_increment_metadata)

    _set_use_absolute_increment_metadata = { "offset" : _set_use_absolute_increment_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_absolute_increment.setter
    def use_absolute_increment(self, use_absolute_increment:bool) -> None:
        return self._intf.set_property(TimeToolTimeIntervalScaled._metadata, TimeToolTimeIntervalScaled._set_use_absolute_increment_metadata, use_absolute_increment)

    _property_names[original_interval] = "original_interval"
    _property_names[absolute_increment] = "absolute_increment"
    _property_names[relative_increment] = "relative_increment"
    _property_names[use_absolute_increment] = "use_absolute_increment"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalScaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalScaled)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalScaled, [TimeToolTimeIntervalScaled, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5407853737749222524, 4515930083547027092), TimeToolTimeIntervalScaled)
agcls.AgTypeNameMap["TimeToolTimeIntervalScaled"] = TimeToolTimeIntervalScaled

class TimeToolTimeIntervalSignaled(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Determine what interval is recorded at target clock location by performing signal transmission of original interval between base and target clock locations."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_interval_method_offset = 1
    _set_original_interval_method_offset = 2
    _get_signal_sense_method_offset = 3
    _set_signal_sense_method_offset = 4
    _get_base_clock_location_method_offset = 5
    _set_base_clock_location_method_offset = 6
    _get_target_clock_location_method_offset = 7
    _set_target_clock_location_method_offset = 8
    _get_signal_delay_method_offset = 9
    _set_signal_delay_method_offset = 10
    _metadata = {
        "iid_data" : (4661949231789629063, 5033190062846728106),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalSignaled)

    _get_original_interval_metadata = { "offset" : _get_original_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_interval(self) -> "ITimeToolTimeInterval":
        """The original interval."""
        return self._intf.get_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._get_original_interval_metadata)

    _set_original_interval_metadata = { "offset" : _set_original_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    @original_interval.setter
    def original_interval(self, original_interval:"ITimeToolTimeInterval") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._set_original_interval_metadata, original_interval)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @property
    def signal_sense(self) -> "SignalDirectionType":
        """The direction of the signal, whether you are Transmitting or Receiving from the BaseClockLocation."""
        return self._intf.get_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @signal_sense.setter
    def signal_sense(self, signal_sense:"SignalDirectionType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._set_signal_sense_metadata, signal_sense)

    _get_base_clock_location_metadata = { "offset" : _get_base_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_clock_location(self) -> "IVectorGeometryToolPoint":
        """The base clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._get_base_clock_location_metadata)

    _set_base_clock_location_metadata = { "offset" : _set_base_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @base_clock_location.setter
    def base_clock_location(self, base_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._set_base_clock_location_metadata, base_clock_location)

    _get_target_clock_location_metadata = { "offset" : _get_target_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_clock_location(self) -> "IVectorGeometryToolPoint":
        """The target clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._get_target_clock_location_metadata)

    _set_target_clock_location_metadata = { "offset" : _set_target_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @target_clock_location.setter
    def target_clock_location(self, target_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._set_target_clock_location_metadata, target_clock_location)

    _get_signal_delay_metadata = { "offset" : _get_signal_delay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def signal_delay(self) -> "IAnalysisWorkbenchSignalDelay":
        """The Signal delay definition, which includes signal transmission, time delay convergence and signal path reference system."""
        return self._intf.get_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._get_signal_delay_metadata)

    _set_signal_delay_metadata = { "offset" : _set_signal_delay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSignalDelay"),) }
    @signal_delay.setter
    def signal_delay(self, signal_delay:"IAnalysisWorkbenchSignalDelay") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSignaled._metadata, TimeToolTimeIntervalSignaled._set_signal_delay_metadata, signal_delay)

    _property_names[original_interval] = "original_interval"
    _property_names[signal_sense] = "signal_sense"
    _property_names[base_clock_location] = "base_clock_location"
    _property_names[target_clock_location] = "target_clock_location"
    _property_names[signal_delay] = "signal_delay"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalSignaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalSignaled)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalSignaled, [TimeToolTimeIntervalSignaled, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5146912718400089076, 7977322097241354916), TimeToolTimeIntervalSignaled)
agcls.AgTypeNameMap["TimeToolTimeIntervalSignaled"] = TimeToolTimeIntervalSignaled

class TimeToolTimeIntervalSmartInterval(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A smart interval."""

    _num_methods = 17
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_interval_method_offset = 1
    _get_duration_as_string_method_offset = 2
    _set_duration_as_string_method_offset = 3
    _get_state_method_offset = 4
    _set_state_method_offset = 5
    _set_implicit_interval_method_offset = 6
    _find_start_time_method_offset = 7
    _find_stop_time_method_offset = 8
    _get_start_epoch_method_offset = 9
    _set_start_epoch_method_offset = 10
    _get_stop_epoch_method_offset = 11
    _set_stop_epoch_method_offset = 12
    _set_explicit_interval_method_offset = 13
    _set_start_and_stop_epochs_method_offset = 14
    _set_start_and_stop_times_method_offset = 15
    _set_start_epoch_and_duration_method_offset = 16
    _set_start_time_and_duration_method_offset = 17
    _metadata = {
        "iid_data" : (4625607466421251572, 4331861431364410030),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalSmartInterval)

    _get_reference_interval_metadata = { "offset" : _get_reference_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_interval(self) -> "ITimeToolTimeInterval":
        """The reference interval used to compute start/stop times of this interval if the state of the interval is set to implicit."""
        return self._intf.get_property(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._get_reference_interval_metadata)

    _get_duration_as_string_metadata = { "offset" : _get_duration_as_string_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def duration_as_string(self) -> str:
        """The duration of the interval."""
        return self._intf.get_property(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._get_duration_as_string_metadata)

    _set_duration_as_string_metadata = { "offset" : _set_duration_as_string_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @duration_as_string.setter
    def duration_as_string(self, duration_as_string:str) -> None:
        return self._intf.set_property(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_duration_as_string_metadata, duration_as_string)

    _get_state_metadata = { "offset" : _get_state_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SmartIntervalState),) }
    @property
    def state(self) -> "SmartIntervalState":
        """A state of the smart interval."""
        return self._intf.get_property(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._get_state_metadata)

    _set_state_metadata = { "offset" : _set_state_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SmartIntervalState),) }
    @state.setter
    def state(self, state:"SmartIntervalState") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_state_metadata, state)

    _set_implicit_interval_metadata = { "offset" : _set_implicit_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    def set_implicit_interval(self, event_interval:"ITimeToolTimeInterval") -> None:
        """Set the reference interval and changes the state to Implicit."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_implicit_interval_metadata, event_interval)

    _find_start_time_metadata = { "offset" : _find_start_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    def find_start_time(self) -> typing.Any:
        """Find a start time of the interval. An exception is thrown if the start time cannot be determined from the interval's current state."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._find_start_time_metadata, OutArg())

    _find_stop_time_metadata = { "offset" : _find_stop_time_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    def find_stop_time(self) -> typing.Any:
        """Find a stop time of the interval. An exception is thrown if the stop time cannot be determined from the interval's current state."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._find_stop_time_metadata, OutArg())

    _get_start_epoch_metadata = { "offset" : _get_start_epoch_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_start_epoch(self) -> "TimeToolInstantSmartEpoch":
        """Return a copy of the start epoch. Changes to the epoch will not affect the state of the interval."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._get_start_epoch_metadata, OutArg())

    _set_start_epoch_metadata = { "offset" : _set_start_epoch_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TimeToolInstantSmartEpoch"),) }
    def set_start_epoch(self, start_epoch:"TimeToolInstantSmartEpoch") -> None:
        """Set a start of the interval using specified epoch component."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_start_epoch_metadata, start_epoch)

    _get_stop_epoch_metadata = { "offset" : _get_stop_epoch_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    def get_stop_epoch(self) -> "TimeToolInstantSmartEpoch":
        """Return a copy of the stop epoch. Changes to the epoch will not affect the state of the interval."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._get_stop_epoch_metadata, OutArg())

    _set_stop_epoch_metadata = { "offset" : _set_stop_epoch_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TimeToolInstantSmartEpoch"),) }
    def set_stop_epoch(self, stop_epoch:"TimeToolInstantSmartEpoch") -> None:
        """Set a stop of the interval using specified epoch component."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_stop_epoch_metadata, stop_epoch)

    _set_explicit_interval_metadata = { "offset" : _set_explicit_interval_method_offset,
            "arg_types" : (agcom.Variant, agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantArg,) }
    def set_explicit_interval(self, start:typing.Any, stop:typing.Any) -> None:
        """Set the interval's start and the stop times changes the interval's state to explicit. Exception is thrown if specified start time is greater than stop time."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_explicit_interval_metadata, start, stop)

    _set_start_and_stop_epochs_metadata = { "offset" : _set_start_and_stop_epochs_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("TimeToolInstantSmartEpoch"), agmarshall.InterfaceInArg("TimeToolInstantSmartEpoch"),) }
    def set_start_and_stop_epochs(self, ref_start_epoch:"TimeToolInstantSmartEpoch", ref_stop_epoch:"TimeToolInstantSmartEpoch") -> None:
        """Set the interval's start and stop epochs as two smart epoch components. Exception is thrown if specified start time is greater than stop time."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_start_and_stop_epochs_metadata, ref_start_epoch, ref_stop_epoch)

    _set_start_and_stop_times_metadata = { "offset" : _set_start_and_stop_times_method_offset,
            "arg_types" : (agcom.Variant, agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.VariantArg,) }
    def set_start_and_stop_times(self, start_time:typing.Any, stop_time:typing.Any) -> None:
        """Set the interval's start and stop epochs as explicit times. Exception is thrown if specified start time is greater than stop time."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_start_and_stop_times_metadata, start_time, stop_time)

    _set_start_epoch_and_duration_metadata = { "offset" : _set_start_epoch_and_duration_method_offset,
            "arg_types" : (agcom.PVOID, agcom.BSTR,),
            "marshallers" : (agmarshall.InterfaceInArg("TimeToolInstantSmartEpoch"), agmarshall.BStrArg,) }
    def set_start_epoch_and_duration(self, ref_start_epoch:"TimeToolInstantSmartEpoch", duration_str:str) -> None:
        """Set the interval's start epoch and the interval's duration."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_start_epoch_and_duration_metadata, ref_start_epoch, duration_str)

    _set_start_time_and_duration_metadata = { "offset" : _set_start_time_and_duration_method_offset,
            "arg_types" : (agcom.Variant, agcom.BSTR,),
            "marshallers" : (agmarshall.VariantArg, agmarshall.BStrArg,) }
    def set_start_time_and_duration(self, epoch:typing.Any, duration_str:str) -> None:
        """Set the interval's start time and the interval's duration."""
        return self._intf.invoke(TimeToolTimeIntervalSmartInterval._metadata, TimeToolTimeIntervalSmartInterval._set_start_time_and_duration_metadata, epoch, duration_str)

    _property_names[reference_interval] = "reference_interval"
    _property_names[duration_as_string] = "duration_as_string"
    _property_names[state] = "state"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalSmartInterval."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalSmartInterval)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalSmartInterval, [TimeToolTimeIntervalSmartInterval, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5123187924205363884, 5573303980465739693), TimeToolTimeIntervalSmartInterval)
agcls.AgTypeNameMap["TimeToolTimeIntervalSmartInterval"] = TimeToolTimeIntervalSmartInterval

class TimeToolTimeIntervalTimeOffset(ITimeToolTimeInterval, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interval defined by shifting specified reference interval by fixed time offset."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_interval_method_offset = 1
    _set_reference_interval_method_offset = 2
    _get_time_offset_method_offset = 3
    _set_time_offset_method_offset = 4
    _metadata = {
        "iid_data" : (4829788478522900189, 18194221658270060953),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalTimeOffset)

    _get_reference_interval_metadata = { "offset" : _get_reference_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_interval(self) -> "ITimeToolTimeInterval":
        """The reference interval."""
        return self._intf.get_property(TimeToolTimeIntervalTimeOffset._metadata, TimeToolTimeIntervalTimeOffset._get_reference_interval_metadata)

    _set_reference_interval_metadata = { "offset" : _set_reference_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    @reference_interval.setter
    def reference_interval(self, reference_interval:"ITimeToolTimeInterval") -> None:
        return self._intf.set_property(TimeToolTimeIntervalTimeOffset._metadata, TimeToolTimeIntervalTimeOffset._set_reference_interval_metadata, reference_interval)

    _get_time_offset_metadata = { "offset" : _get_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_offset(self) -> float:
        """The time offset value."""
        return self._intf.get_property(TimeToolTimeIntervalTimeOffset._metadata, TimeToolTimeIntervalTimeOffset._get_time_offset_metadata)

    _set_time_offset_metadata = { "offset" : _set_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_offset.setter
    def time_offset(self, time_offset:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalTimeOffset._metadata, TimeToolTimeIntervalTimeOffset._set_time_offset_metadata, time_offset)

    _property_names[reference_interval] = "reference_interval"
    _property_names[time_offset] = "time_offset"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalTimeOffset."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalTimeOffset)
        ITimeToolTimeInterval.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolTimeInterval._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalTimeOffset, [TimeToolTimeIntervalTimeOffset, ITimeToolTimeInterval, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5594696185291586304, 7469384033316253831), TimeToolTimeIntervalTimeOffset)
agcls.AgTypeNameMap["TimeToolTimeIntervalTimeOffset"] = TimeToolTimeIntervalTimeOffset

class TimeToolInstantSignaled(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Event recorded on specified clock via signal transmission from original time instant recorded on different clock."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_original_time_instant_method_offset = 1
    _set_original_time_instant_method_offset = 2
    _get_signal_sense_method_offset = 3
    _set_signal_sense_method_offset = 4
    _get_base_clock_location_method_offset = 5
    _set_base_clock_location_method_offset = 6
    _get_target_clock_location_method_offset = 7
    _set_target_clock_location_method_offset = 8
    _get_signal_delay_method_offset = 9
    _set_signal_delay_method_offset = 10
    _metadata = {
        "iid_data" : (5072103158722034880, 4341972865712102555),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantSignaled)

    _get_original_time_instant_metadata = { "offset" : _get_original_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def original_time_instant(self) -> "ITimeToolInstant":
        """The original time instant."""
        return self._intf.get_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._get_original_time_instant_metadata)

    _set_original_time_instant_metadata = { "offset" : _set_original_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @original_time_instant.setter
    def original_time_instant(self, original_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._set_original_time_instant_metadata, original_time_instant)

    _get_signal_sense_metadata = { "offset" : _get_signal_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @property
    def signal_sense(self) -> "SignalDirectionType":
        """The direction of the signal, whether you are Transmitting or Receiving from the BaseClockLocation."""
        return self._intf.get_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._get_signal_sense_metadata)

    _set_signal_sense_metadata = { "offset" : _set_signal_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalDirectionType),) }
    @signal_sense.setter
    def signal_sense(self, signal_sense:"SignalDirectionType") -> None:
        return self._intf.set_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._set_signal_sense_metadata, signal_sense)

    _get_base_clock_location_metadata = { "offset" : _get_base_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def base_clock_location(self) -> "IVectorGeometryToolPoint":
        """The base clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._get_base_clock_location_metadata)

    _set_base_clock_location_metadata = { "offset" : _set_base_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @base_clock_location.setter
    def base_clock_location(self, base_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._set_base_clock_location_metadata, base_clock_location)

    _get_target_clock_location_metadata = { "offset" : _get_target_clock_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def target_clock_location(self) -> "IVectorGeometryToolPoint":
        """The target clock location, which is a point from VGT."""
        return self._intf.get_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._get_target_clock_location_metadata)

    _set_target_clock_location_metadata = { "offset" : _set_target_clock_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @target_clock_location.setter
    def target_clock_location(self, target_clock_location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._set_target_clock_location_metadata, target_clock_location)

    _get_signal_delay_metadata = { "offset" : _get_signal_delay_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def signal_delay(self) -> "IAnalysisWorkbenchSignalDelay":
        """The Signal delay definition, which includes signal transmission, time delay convergence and signal path reference system."""
        return self._intf.get_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._get_signal_delay_metadata)

    _set_signal_delay_metadata = { "offset" : _set_signal_delay_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IAnalysisWorkbenchSignalDelay"),) }
    @signal_delay.setter
    def signal_delay(self, signal_delay:"IAnalysisWorkbenchSignalDelay") -> None:
        return self._intf.set_property(TimeToolInstantSignaled._metadata, TimeToolInstantSignaled._set_signal_delay_metadata, signal_delay)

    _property_names[original_time_instant] = "original_time_instant"
    _property_names[signal_sense] = "signal_sense"
    _property_names[base_clock_location] = "base_clock_location"
    _property_names[target_clock_location] = "target_clock_location"
    _property_names[signal_delay] = "signal_delay"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantSignaled."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantSignaled)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantSignaled, [TimeToolInstantSignaled, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5693916528422022973, 3263658904170695863), TimeToolInstantSignaled)
agcls.AgTypeNameMap["TimeToolInstantSignaled"] = TimeToolInstantSignaled

class TimeToolInstantSmartEpoch(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A smart epoch."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_time_instant_method_offset = 1
    _get_reference_epoch_method_offset = 2
    _get_state_method_offset = 3
    _set_state_method_offset = 4
    _set_explicit_time_method_offset = 5
    _set_implicit_time_method_offset = 6
    _metadata = {
        "iid_data" : (5299430079025464053, 2983292493577895056),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantSmartEpoch)

    _get_time_instant_metadata = { "offset" : _get_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.Variant),),
            "marshallers" : (agmarshall.VariantArg,) }
    @property
    def time_instant(self) -> typing.Any:
        """Represents the time instant if the state is set to explicit."""
        return self._intf.get_property(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._get_time_instant_metadata)

    _get_reference_epoch_metadata = { "offset" : _get_reference_epoch_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_epoch(self) -> "ITimeToolInstant":
        """A reference event object used to compute time instant if the state is set to implicit."""
        return self._intf.get_property(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._get_reference_epoch_metadata)

    _get_state_metadata = { "offset" : _get_state_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SmartEpochState),) }
    @property
    def state(self) -> "SmartEpochState":
        """State of the event."""
        return self._intf.get_property(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._get_state_metadata)

    _set_state_metadata = { "offset" : _set_state_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SmartEpochState),) }
    @state.setter
    def state(self, state:"SmartEpochState") -> None:
        return self._intf.set_property(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._set_state_metadata, state)

    _set_explicit_time_metadata = { "offset" : _set_explicit_time_method_offset,
            "arg_types" : (agcom.Variant,),
            "marshallers" : (agmarshall.VariantArg,) }
    def set_explicit_time(self, epoch:typing.Any) -> None:
        """Set explicit time instant and the smart epoch's state to Explicit."""
        return self._intf.invoke(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._set_explicit_time_metadata, epoch)

    _set_implicit_time_metadata = { "offset" : _set_implicit_time_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    def set_implicit_time(self, event_epoch:"ITimeToolInstant") -> None:
        """Set the reference event and the smart epoch's state to Implicit."""
        return self._intf.invoke(TimeToolInstantSmartEpoch._metadata, TimeToolInstantSmartEpoch._set_implicit_time_metadata, event_epoch)

    _property_names[time_instant] = "time_instant"
    _property_names[reference_epoch] = "reference_epoch"
    _property_names[state] = "state"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantSmartEpoch."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantSmartEpoch)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantSmartEpoch, [TimeToolInstantSmartEpoch, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5284572681912631891, 10588360420919622552), TimeToolInstantSmartEpoch)
agcls.AgTypeNameMap["TimeToolInstantSmartEpoch"] = TimeToolInstantSmartEpoch

class TimeToolInstantStartStopTime(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Event is either start or stop time selected from a reference interval."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_start_method_offset = 1
    _set_use_start_method_offset = 2
    _get_reference_interval_method_offset = 3
    _set_reference_interval_method_offset = 4
    _metadata = {
        "iid_data" : (4785383463426884876, 6466418643845985677),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantStartStopTime)

    _get_use_start_metadata = { "offset" : _get_use_start_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_start(self) -> bool:
        """Indicate whether to use start (true) or stop (false)."""
        return self._intf.get_property(TimeToolInstantStartStopTime._metadata, TimeToolInstantStartStopTime._get_use_start_metadata)

    _set_use_start_metadata = { "offset" : _set_use_start_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_start.setter
    def use_start(self, use_start:bool) -> None:
        return self._intf.set_property(TimeToolInstantStartStopTime._metadata, TimeToolInstantStartStopTime._set_use_start_metadata, use_start)

    _get_reference_interval_metadata = { "offset" : _get_reference_interval_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_interval(self) -> "ITimeToolTimeInterval":
        """The reference interval."""
        return self._intf.get_property(TimeToolInstantStartStopTime._metadata, TimeToolInstantStartStopTime._get_reference_interval_metadata)

    _set_reference_interval_metadata = { "offset" : _set_reference_interval_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolTimeInterval"),) }
    @reference_interval.setter
    def reference_interval(self, reference_event_interval:"ITimeToolTimeInterval") -> None:
        return self._intf.set_property(TimeToolInstantStartStopTime._metadata, TimeToolInstantStartStopTime._set_reference_interval_metadata, reference_event_interval)

    _property_names[use_start] = "use_start"
    _property_names[reference_interval] = "reference_interval"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantStartStopTime."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantStartStopTime)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantStartStopTime, [TimeToolInstantStartStopTime, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5286281606334682631, 1156813953277887129), TimeToolInstantStartStopTime)
agcls.AgTypeNameMap["TimeToolInstantStartStopTime"] = TimeToolInstantStartStopTime

class TimeToolInstantTimeOffset(ITimeToolInstant, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Event at fixed offset from specified reference event."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_reference_time_instant_method_offset = 1
    _set_reference_time_instant_method_offset = 2
    _get_time_offset_method_offset = 3
    _set_time_offset_method_offset = 4
    _metadata = {
        "iid_data" : (4849369508245529166, 13064483229774346402),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolInstantTimeOffset)

    _get_reference_time_instant_metadata = { "offset" : _get_reference_time_instant_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_time_instant(self) -> "ITimeToolInstant":
        """The reference time instant."""
        return self._intf.get_property(TimeToolInstantTimeOffset._metadata, TimeToolInstantTimeOffset._get_reference_time_instant_metadata)

    _set_reference_time_instant_metadata = { "offset" : _set_reference_time_instant_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ITimeToolInstant"),) }
    @reference_time_instant.setter
    def reference_time_instant(self, reference_time_instant:"ITimeToolInstant") -> None:
        return self._intf.set_property(TimeToolInstantTimeOffset._metadata, TimeToolInstantTimeOffset._set_reference_time_instant_metadata, reference_time_instant)

    _get_time_offset_metadata = { "offset" : _get_time_offset_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_offset(self) -> float:
        """The time offset from the ReferenceTimeInstant. The value is in ``TimeUnit`` dimension."""
        return self._intf.get_property(TimeToolInstantTimeOffset._metadata, TimeToolInstantTimeOffset._get_time_offset_metadata)

    _set_time_offset_metadata = { "offset" : _set_time_offset_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_offset.setter
    def time_offset(self, time_offset2:float) -> None:
        return self._intf.set_property(TimeToolInstantTimeOffset._metadata, TimeToolInstantTimeOffset._set_time_offset_metadata, time_offset2)

    _property_names[reference_time_instant] = "reference_time_instant"
    _property_names[time_offset] = "time_offset"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolInstantTimeOffset."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolInstantTimeOffset)
        ITimeToolInstant.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolInstant._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolInstantTimeOffset, [TimeToolInstantTimeOffset, ITimeToolInstant, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4631038835821414856, 7748244684488556208), TimeToolInstantTimeOffset)
agcls.AgTypeNameMap["TimeToolInstantTimeOffset"] = TimeToolInstantTimeOffset

class TimeToolTimeIntervalFirstIntervalsFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter selects a portion of first intervals."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_maximum_number_of_intervals_method_offset = 1
    _set_maximum_number_of_intervals_method_offset = 2
    _metadata = {
        "iid_data" : (5238555741080133400, 9243782524213090747),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalFirstIntervalsFilter)

    _get_maximum_number_of_intervals_metadata = { "offset" : _get_maximum_number_of_intervals_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_number_of_intervals(self) -> int:
        """Maximum number of intervals."""
        return self._intf.get_property(TimeToolTimeIntervalFirstIntervalsFilter._metadata, TimeToolTimeIntervalFirstIntervalsFilter._get_maximum_number_of_intervals_metadata)

    _set_maximum_number_of_intervals_metadata = { "offset" : _set_maximum_number_of_intervals_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_number_of_intervals.setter
    def maximum_number_of_intervals(self, maximum_number_of_intervals:int) -> None:
        return self._intf.set_property(TimeToolTimeIntervalFirstIntervalsFilter._metadata, TimeToolTimeIntervalFirstIntervalsFilter._set_maximum_number_of_intervals_metadata, maximum_number_of_intervals)

    _property_names[maximum_number_of_intervals] = "maximum_number_of_intervals"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalFirstIntervalsFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalFirstIntervalsFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalFirstIntervalsFilter, [TimeToolTimeIntervalFirstIntervalsFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((4841739463115949456, 12213426980424696479), TimeToolTimeIntervalFirstIntervalsFilter)
agcls.AgTypeNameMap["TimeToolTimeIntervalFirstIntervalsFilter"] = TimeToolTimeIntervalFirstIntervalsFilter

class TimeToolTimeIntervalGapsFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter merges intervals unless they are separated by gaps of at least/most certain duration."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_duration_type_method_offset = 1
    _set_duration_type_method_offset = 2
    _get_gap_duration_method_offset = 3
    _set_gap_duration_method_offset = 4
    _metadata = {
        "iid_data" : (5577530501196125174, 17516988796283437440),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalGapsFilter)

    _get_duration_type_metadata = { "offset" : _get_duration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @property
    def duration_type(self) -> "IntervalDurationType":
        """Choose a duration type (at least/at most)."""
        return self._intf.get_property(TimeToolTimeIntervalGapsFilter._metadata, TimeToolTimeIntervalGapsFilter._get_duration_type_metadata)

    _set_duration_type_metadata = { "offset" : _set_duration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @duration_type.setter
    def duration_type(self, duration_kind:"IntervalDurationType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalGapsFilter._metadata, TimeToolTimeIntervalGapsFilter._set_duration_type_metadata, duration_kind)

    _get_gap_duration_metadata = { "offset" : _get_gap_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def gap_duration(self) -> float:
        """Duration of the gap."""
        return self._intf.get_property(TimeToolTimeIntervalGapsFilter._metadata, TimeToolTimeIntervalGapsFilter._get_gap_duration_metadata)

    _set_gap_duration_metadata = { "offset" : _set_gap_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @gap_duration.setter
    def gap_duration(self, gap_duration:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalGapsFilter._metadata, TimeToolTimeIntervalGapsFilter._set_gap_duration_metadata, gap_duration)

    _property_names[duration_type] = "duration_type"
    _property_names[gap_duration] = "gap_duration"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalGapsFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalGapsFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalGapsFilter, [TimeToolTimeIntervalGapsFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((4915234003613464599, 13847403613333364110), TimeToolTimeIntervalGapsFilter)
agcls.AgTypeNameMap["TimeToolTimeIntervalGapsFilter"] = TimeToolTimeIntervalGapsFilter

class AnalysisWorkbenchIntegral(IAnalysisWorkbenchIntegral, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Represents a base class for integral definitions."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchIntegral."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchIntegral.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchIntegral._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchIntegral, [IAnalysisWorkbenchIntegral, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5141699313388952889, 439554261460355240), AnalysisWorkbenchIntegral)
agcls.AgTypeNameMap["AnalysisWorkbenchIntegral"] = AnalysisWorkbenchIntegral

class CalculationToolIntegralBasic(IAnalysisWorkbenchIntegral, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Integral definition determines how scalar calculation is numerically integrated."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _set_type_method_offset = 2
    _get_tolerance_method_offset = 3
    _set_tolerance_method_offset = 4
    _get_maximum_iterations_method_offset = 5
    _set_maximum_iterations_method_offset = 6
    _metadata = {
        "iid_data" : (5223039038232914469, 8592596877327925126),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolIntegralBasic)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(QuadratureType),) }
    @property
    def type(self) -> "QuadratureType":
        """Get the integral type which determines the method of integration and can be set to trapezoidal, Simplson or adaptive Lobatto."""
        return self._intf.get_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._get_type_metadata)

    _set_type_metadata = { "offset" : _set_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(QuadratureType),) }
    @type.setter
    def type(self, type:"QuadratureType") -> None:
        return self._intf.set_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._set_type_metadata, type)

    _get_tolerance_metadata = { "offset" : _get_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def tolerance(self) -> float:
        """Get the tolerance which determines how accurate integral is computed by finding relative difference between refined and unrefined integral evaluations. Only available if Adaptive Lobatto is selected as the integral type."""
        return self._intf.get_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._get_tolerance_metadata)

    _set_tolerance_metadata = { "offset" : _set_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @tolerance.setter
    def tolerance(self, tolerance:float) -> None:
        return self._intf.set_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._set_tolerance_metadata, tolerance)

    _get_maximum_iterations_metadata = { "offset" : _get_maximum_iterations_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_iterations(self) -> int:
        """Get the number of iteration which determines how many refinement iterations are allowed. Only available if Adaptive Lobatto is selected as the integral type."""
        return self._intf.get_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._get_maximum_iterations_metadata)

    _set_maximum_iterations_metadata = { "offset" : _set_maximum_iterations_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_iterations.setter
    def maximum_iterations(self, maximum_iterations:int) -> None:
        return self._intf.set_property(CalculationToolIntegralBasic._metadata, CalculationToolIntegralBasic._set_maximum_iterations_metadata, maximum_iterations)

    _property_names[type] = "type"
    _property_names[tolerance] = "tolerance"
    _property_names[maximum_iterations] = "maximum_iterations"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolIntegralBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolIntegralBasic)
        IAnalysisWorkbenchIntegral.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchIntegral._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolIntegralBasic, [CalculationToolIntegralBasic, IAnalysisWorkbenchIntegral, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4728253546109001966, 9993082615726594997), CalculationToolIntegralBasic)
agcls.AgTypeNameMap["CalculationToolIntegralBasic"] = CalculationToolIntegralBasic

class AnalysisWorkbenchInterpolator(IAnalysisWorkbenchInterpolator, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Represents a base class for interpolation definitions."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchInterpolator."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchInterpolator.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchInterpolator._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchInterpolator, [IAnalysisWorkbenchInterpolator, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4634326977035843189, 527318962673378965), AnalysisWorkbenchInterpolator)
agcls.AgTypeNameMap["AnalysisWorkbenchInterpolator"] = AnalysisWorkbenchInterpolator

class CalculationToolInterpolatorBasic(IAnalysisWorkbenchInterpolator, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Interpolation definition determines how to obtain values in between tabulated samples. See STK help on interpolation for further details."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_type_method_offset = 1
    _set_type_method_offset = 2
    _get_order_method_offset = 3
    _set_order_method_offset = 4
    _metadata = {
        "iid_data" : (5619549411845612202, 16364875609665918339),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolInterpolatorBasic)

    _get_type_metadata = { "offset" : _get_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(InterpolationMethodType),) }
    @property
    def type(self) -> "InterpolationMethodType":
        """Get the interpolation type, which can be Lagrange or Hermite interpolation. See STK help on interpolation for further details."""
        return self._intf.get_property(CalculationToolInterpolatorBasic._metadata, CalculationToolInterpolatorBasic._get_type_metadata)

    _set_type_metadata = { "offset" : _set_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(InterpolationMethodType),) }
    @type.setter
    def type(self, type:"InterpolationMethodType") -> None:
        return self._intf.set_property(CalculationToolInterpolatorBasic._metadata, CalculationToolInterpolatorBasic._set_type_metadata, type)

    _get_order_metadata = { "offset" : _get_order_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def order(self) -> int:
        """Get the interpolation order, which determines the order of interpolation polynomial and is related to how many samples are used during interpolation. See STK help on interpolation for further details."""
        return self._intf.get_property(CalculationToolInterpolatorBasic._metadata, CalculationToolInterpolatorBasic._get_order_metadata)

    _set_order_metadata = { "offset" : _set_order_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @order.setter
    def order(self, order:int) -> None:
        return self._intf.set_property(CalculationToolInterpolatorBasic._metadata, CalculationToolInterpolatorBasic._set_order_metadata, order)

    _property_names[type] = "type"
    _property_names[order] = "order"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolInterpolatorBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolInterpolatorBasic)
        IAnalysisWorkbenchInterpolator.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchInterpolator._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolInterpolatorBasic, [CalculationToolInterpolatorBasic, IAnalysisWorkbenchInterpolator, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5710680427636003903, 2230165650911561406), CalculationToolInterpolatorBasic)
agcls.AgTypeNameMap["CalculationToolInterpolatorBasic"] = CalculationToolInterpolatorBasic

class TimeToolIntervalsFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter selects intervals of at least/most certain duration."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_duration_type_method_offset = 1
    _set_duration_type_method_offset = 2
    _get_interval_duration_method_offset = 3
    _set_interval_duration_method_offset = 4
    _metadata = {
        "iid_data" : (5450995978684978217, 16753011116458334649),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolIntervalsFilter)

    _get_duration_type_metadata = { "offset" : _get_duration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @property
    def duration_type(self) -> "IntervalDurationType":
        """Choose a duration type (at least/at most)."""
        return self._intf.get_property(TimeToolIntervalsFilter._metadata, TimeToolIntervalsFilter._get_duration_type_metadata)

    _set_duration_type_metadata = { "offset" : _set_duration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @duration_type.setter
    def duration_type(self, duration_kind:"IntervalDurationType") -> None:
        return self._intf.set_property(TimeToolIntervalsFilter._metadata, TimeToolIntervalsFilter._set_duration_type_metadata, duration_kind)

    _get_interval_duration_metadata = { "offset" : _get_interval_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def interval_duration(self) -> float:
        """The interval duration."""
        return self._intf.get_property(TimeToolIntervalsFilter._metadata, TimeToolIntervalsFilter._get_interval_duration_metadata)

    _set_interval_duration_metadata = { "offset" : _set_interval_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @interval_duration.setter
    def interval_duration(self, interval_duration:float) -> None:
        return self._intf.set_property(TimeToolIntervalsFilter._metadata, TimeToolIntervalsFilter._set_interval_duration_metadata, interval_duration)

    _property_names[duration_type] = "duration_type"
    _property_names[interval_duration] = "interval_duration"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolIntervalsFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolIntervalsFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolIntervalsFilter, [TimeToolIntervalsFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((5615056885707293863, 4921065534759074228), TimeToolIntervalsFilter)
agcls.AgTypeNameMap["TimeToolIntervalsFilter"] = TimeToolIntervalsFilter

class TimeToolTimeIntervalLastIntervalsFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter selects a portion of last intervals."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_maximum_number_of_intervals_method_offset = 1
    _set_maximum_number_of_intervals_method_offset = 2
    _metadata = {
        "iid_data" : (5298781469380646786, 4200496564274675643),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalLastIntervalsFilter)

    _get_maximum_number_of_intervals_metadata = { "offset" : _get_maximum_number_of_intervals_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_number_of_intervals(self) -> int:
        """Maximum number of intervals."""
        return self._intf.get_property(TimeToolTimeIntervalLastIntervalsFilter._metadata, TimeToolTimeIntervalLastIntervalsFilter._get_maximum_number_of_intervals_metadata)

    _set_maximum_number_of_intervals_metadata = { "offset" : _set_maximum_number_of_intervals_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_number_of_intervals.setter
    def maximum_number_of_intervals(self, maximum_number_of_intervals:int) -> None:
        return self._intf.set_property(TimeToolTimeIntervalLastIntervalsFilter._metadata, TimeToolTimeIntervalLastIntervalsFilter._set_maximum_number_of_intervals_metadata, maximum_number_of_intervals)

    _property_names[maximum_number_of_intervals] = "maximum_number_of_intervals"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalLastIntervalsFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalLastIntervalsFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalLastIntervalsFilter, [TimeToolTimeIntervalLastIntervalsFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((5370173919127882269, 172199397625736352), TimeToolTimeIntervalLastIntervalsFilter)
agcls.AgTypeNameMap["TimeToolTimeIntervalLastIntervalsFilter"] = TimeToolTimeIntervalLastIntervalsFilter

class CalculationToolParameterSet(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Parameter set contains various sets of scalar computations."""
    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSet."""
        SupportsDeleteCallback.__init__(self)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSet, [ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5211800288647545596, 7543106923046475154), CalculationToolParameterSet)
agcls.AgTypeNameMap["CalculationToolParameterSet"] = CalculationToolParameterSet

class CalculationToolParameterSetAttitude(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Attitude parameter set contains various representations of attitude of one set of axes relative to another."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_axes_method_offset = 1
    _set_axes_method_offset = 2
    _get_reference_axes_method_offset = 3
    _set_reference_axes_method_offset = 4
    _metadata = {
        "iid_data" : (5157814150658641714, 4846008862630832260),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetAttitude)

    _get_axes_metadata = { "offset" : _get_axes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def axes(self) -> "IVectorGeometryToolAxes":
        """Get the axes for which attitude representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetAttitude._metadata, CalculationToolParameterSetAttitude._get_axes_metadata)

    _set_axes_metadata = { "offset" : _set_axes_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    @axes.setter
    def axes(self, axes:"IVectorGeometryToolAxes") -> None:
        return self._intf.set_property(CalculationToolParameterSetAttitude._metadata, CalculationToolParameterSetAttitude._set_axes_metadata, axes)

    _get_reference_axes_metadata = { "offset" : _get_reference_axes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_axes(self) -> "IVectorGeometryToolAxes":
        """Get the reference axes relative to which attitude representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetAttitude._metadata, CalculationToolParameterSetAttitude._get_reference_axes_metadata)

    _set_reference_axes_metadata = { "offset" : _set_reference_axes_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    @reference_axes.setter
    def reference_axes(self, reference_axes:"IVectorGeometryToolAxes") -> None:
        return self._intf.set_property(CalculationToolParameterSetAttitude._metadata, CalculationToolParameterSetAttitude._set_reference_axes_metadata, reference_axes)

    _property_names[axes] = "axes"
    _property_names[reference_axes] = "reference_axes"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetAttitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetAttitude)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetAttitude, [CalculationToolParameterSetAttitude, ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5089636213011434244, 12709160544851939744), CalculationToolParameterSetAttitude)
agcls.AgTypeNameMap["CalculationToolParameterSetAttitude"] = CalculationToolParameterSetAttitude

class CalculationToolParameterSetFactory(SupportsDeleteCallback):
    """The factory is used to create instances of available parameter set types."""

    _num_methods = 7
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_attitude_method_offset = 2
    _create_ground_trajectory_method_offset = 3
    _create_trajectory_method_offset = 4
    _create_orbit_method_offset = 5
    _create_vector_method_offset = 6
    _is_type_supported_method_offset = 7
    _metadata = {
        "iid_data" : (5288103279616649311, 6790554156469523601),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(ParameterSetType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"ParameterSetType") -> "ICalculationToolParameterSet":
        """Create and registers a parameter set using specified name and description."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_metadata, name, description, type, OutArg())

    _create_attitude_metadata = { "offset" : _create_attitude_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_attitude(self, name:str, description:str) -> "ICalculationToolParameterSet":
        """Create a parameter set defined by identifying one set of axes in reference to another."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_attitude_metadata, name, description, OutArg())

    _create_ground_trajectory_metadata = { "offset" : _create_ground_trajectory_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_ground_trajectory(self, name:str, description:str) -> "ICalculationToolParameterSet":
        """Create a parameter set defined by identifying location in reference central body."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_ground_trajectory_metadata, name, description, OutArg())

    _create_trajectory_metadata = { "offset" : _create_trajectory_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_trajectory(self, name:str, description:str) -> "ICalculationToolParameterSet":
        """Create a parameter set defined by identifying location in reference coordinate system."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_trajectory_metadata, name, description, OutArg())

    _create_orbit_metadata = { "offset" : _create_orbit_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_orbit(self, name:str, description:str) -> "ICalculationToolParameterSet":
        """Create a parameter set defined by identifying orbiting point and its central body."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_orbit_metadata, name, description, OutArg())

    _create_vector_metadata = { "offset" : _create_vector_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_vector(self, name:str, description:str) -> "ICalculationToolParameterSet":
        """Create a parameter set defined by identifying vector in reference axes."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._create_vector_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(ParameterSetType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"ParameterSetType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(CalculationToolParameterSetFactory._metadata, CalculationToolParameterSetFactory._is_type_supported_metadata, type, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetFactory, [CalculationToolParameterSetFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5191271020123231030, 13615349136520883352), CalculationToolParameterSetFactory)
agcls.AgTypeNameMap["CalculationToolParameterSetFactory"] = CalculationToolParameterSetFactory

class CalculationToolParameterSetGroundTrajectory(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Ground trajectory parameter set contains various representations of trajectory of a point relative to central body reference shape."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_location_method_offset = 1
    _set_location_method_offset = 2
    _get_central_body_method_offset = 3
    _set_central_body_method_offset = 4
    _metadata = {
        "iid_data" : (5004898066850638741, 11511760438073722810),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetGroundTrajectory)

    _get_location_metadata = { "offset" : _get_location_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def location(self) -> "IVectorGeometryToolPoint":
        """Get the point for which ground trajectory representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetGroundTrajectory._metadata, CalculationToolParameterSetGroundTrajectory._get_location_metadata)

    _set_location_metadata = { "offset" : _set_location_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @location.setter
    def location(self, location:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolParameterSetGroundTrajectory._metadata, CalculationToolParameterSetGroundTrajectory._set_location_metadata, location)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get the central body relative to which ground trajectory representations are computed. Both the central body reference shape and its CBF (central body centered fixed) system are used by this parameter set."""
        return self._intf.get_property(CalculationToolParameterSetGroundTrajectory._metadata, CalculationToolParameterSetGroundTrajectory._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(CalculationToolParameterSetGroundTrajectory._metadata, CalculationToolParameterSetGroundTrajectory._set_central_body_metadata, central_body)

    _property_names[location] = "location"
    _property_names[central_body] = "central_body"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetGroundTrajectory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetGroundTrajectory)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetGroundTrajectory, [CalculationToolParameterSetGroundTrajectory, ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5476359557675953687, 12818036812419277696), CalculationToolParameterSetGroundTrajectory)
agcls.AgTypeNameMap["CalculationToolParameterSetGroundTrajectory"] = CalculationToolParameterSetGroundTrajectory

class CalculationToolParameterSetOrbit(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Orbit parameter set contains various trajectory representations of an orbiting point."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_orbiting_point_method_offset = 1
    _set_orbiting_point_method_offset = 2
    _get_reference_system_method_offset = 3
    _set_reference_system_method_offset = 4
    _get_gravitational_parameter_method_offset = 5
    _set_gravitational_parameter_method_offset = 6
    _get_central_body_method_offset = 7
    _set_central_body_method_offset = 8
    _get_use_central_body_gravitational_parameter_method_offset = 9
    _set_use_central_body_gravitational_parameter_method_offset = 10
    _get_use_central_body_inertial_method_offset = 11
    _set_use_central_body_inertial_method_offset = 12
    _metadata = {
        "iid_data" : (4631037522197801655, 2468098686994820286),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetOrbit)

    _get_orbiting_point_metadata = { "offset" : _get_orbiting_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def orbiting_point(self) -> "IVectorGeometryToolPoint":
        """Get the point for which orbital parameters are computed."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_orbiting_point_metadata)

    _set_orbiting_point_metadata = { "offset" : _set_orbiting_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @orbiting_point.setter
    def orbiting_point(self, orbiting_point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_orbiting_point_metadata, orbiting_point)

    _get_reference_system_metadata = { "offset" : _get_reference_system_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_system(self) -> "IVectorGeometryToolSystem":
        """Get the reference system in which orbital parameters are computed. Only used if the option to specify reference system is selected."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_reference_system_metadata)

    _set_reference_system_metadata = { "offset" : _set_reference_system_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"),) }
    @reference_system.setter
    def reference_system(self, reference_system:"IVectorGeometryToolSystem") -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_reference_system_metadata, reference_system)

    _get_gravitational_parameter_metadata = { "offset" : _get_gravitational_parameter_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def gravitational_parameter(self) -> float:
        """Get the gravitational parameter for the mass relative to which orbital parameters are computed. Only used if the option to specify gravitational parameter is selected."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_gravitational_parameter_metadata)

    _set_gravitational_parameter_metadata = { "offset" : _set_gravitational_parameter_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @gravitational_parameter.setter
    def gravitational_parameter(self, gravitational_parameter:float) -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_gravitational_parameter_metadata, gravitational_parameter)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get the central body relative to which orbital parameters are computed."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_central_body_metadata, central_body)

    _get_use_central_body_gravitational_parameter_metadata = { "offset" : _get_use_central_body_gravitational_parameter_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_central_body_gravitational_parameter(self) -> bool:
        """Get the option that determines whether to specify the gravitational parameter value or to inherit it from the central body."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_use_central_body_gravitational_parameter_metadata)

    _set_use_central_body_gravitational_parameter_metadata = { "offset" : _set_use_central_body_gravitational_parameter_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_central_body_gravitational_parameter.setter
    def use_central_body_gravitational_parameter(self, use_central_body_gravitational_parameter:bool) -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_use_central_body_gravitational_parameter_metadata, use_central_body_gravitational_parameter)

    _get_use_central_body_inertial_metadata = { "offset" : _get_use_central_body_inertial_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_central_body_inertial(self) -> bool:
        """Get the option that determines whether to specify the reference coordinate system or to the inherit inertial reference system from the central body."""
        return self._intf.get_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._get_use_central_body_inertial_metadata)

    _set_use_central_body_inertial_metadata = { "offset" : _set_use_central_body_inertial_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_central_body_inertial.setter
    def use_central_body_inertial(self, use_central_body_inertial:bool) -> None:
        return self._intf.set_property(CalculationToolParameterSetOrbit._metadata, CalculationToolParameterSetOrbit._set_use_central_body_inertial_metadata, use_central_body_inertial)

    _property_names[orbiting_point] = "orbiting_point"
    _property_names[reference_system] = "reference_system"
    _property_names[gravitational_parameter] = "gravitational_parameter"
    _property_names[central_body] = "central_body"
    _property_names[use_central_body_gravitational_parameter] = "use_central_body_gravitational_parameter"
    _property_names[use_central_body_inertial] = "use_central_body_inertial"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetOrbit."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetOrbit)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetOrbit, [CalculationToolParameterSetOrbit, ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4611801354393510048, 12183196691739788168), CalculationToolParameterSetOrbit)
agcls.AgTypeNameMap["CalculationToolParameterSetOrbit"] = CalculationToolParameterSetOrbit

class CalculationToolParameterSetTrajectory(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Trajectory parameter set contains various representations of trajectory of a point relative to a reference coordinate system."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_point_method_offset = 1
    _set_point_method_offset = 2
    _get_reference_system_method_offset = 3
    _set_reference_system_method_offset = 4
    _metadata = {
        "iid_data" : (5006139916852774401, 13019652784963788729),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetTrajectory)

    _get_point_metadata = { "offset" : _get_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def point(self) -> "IVectorGeometryToolPoint":
        """Get the point for which trajectory representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetTrajectory._metadata, CalculationToolParameterSetTrajectory._get_point_metadata)

    _set_point_metadata = { "offset" : _set_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @point.setter
    def point(self, point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(CalculationToolParameterSetTrajectory._metadata, CalculationToolParameterSetTrajectory._set_point_metadata, point)

    _get_reference_system_metadata = { "offset" : _get_reference_system_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_system(self) -> "IVectorGeometryToolSystem":
        """Get the reference system relative to which trajectory representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetTrajectory._metadata, CalculationToolParameterSetTrajectory._get_reference_system_metadata)

    _set_reference_system_metadata = { "offset" : _set_reference_system_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"),) }
    @reference_system.setter
    def reference_system(self, reference_system:"IVectorGeometryToolSystem") -> None:
        return self._intf.set_property(CalculationToolParameterSetTrajectory._metadata, CalculationToolParameterSetTrajectory._set_reference_system_metadata, reference_system)

    _property_names[point] = "point"
    _property_names[reference_system] = "reference_system"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetTrajectory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetTrajectory)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetTrajectory, [CalculationToolParameterSetTrajectory, ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5733434664696181485, 15500898216521877646), CalculationToolParameterSetTrajectory)
agcls.AgTypeNameMap["CalculationToolParameterSetTrajectory"] = CalculationToolParameterSetTrajectory

class CalculationToolParameterSetVector(ICalculationToolParameterSet, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Vector parameter set contains various representations of a vector in a reference set of axes."""

    _num_methods = 4
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_vector_method_offset = 1
    _set_vector_method_offset = 2
    _get_reference_axes_method_offset = 3
    _set_reference_axes_method_offset = 4
    _metadata = {
        "iid_data" : (5239130838684790465, 17380030367494480022),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolParameterSetVector)

    _get_vector_metadata = { "offset" : _get_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def vector(self) -> "IVectorGeometryToolVector":
        """Get the vector for which representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetVector._metadata, CalculationToolParameterSetVector._get_vector_metadata)

    _set_vector_metadata = { "offset" : _set_vector_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @vector.setter
    def vector(self, vector:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(CalculationToolParameterSetVector._metadata, CalculationToolParameterSetVector._set_vector_metadata, vector)

    _get_reference_axes_metadata = { "offset" : _get_reference_axes_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_axes(self) -> "IVectorGeometryToolAxes":
        """Get the reference axes relative to which representations are computed."""
        return self._intf.get_property(CalculationToolParameterSetVector._metadata, CalculationToolParameterSetVector._get_reference_axes_metadata)

    _set_reference_axes_metadata = { "offset" : _set_reference_axes_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolAxes"),) }
    @reference_axes.setter
    def reference_axes(self, reference_axes:"IVectorGeometryToolAxes") -> None:
        return self._intf.set_property(CalculationToolParameterSetVector._metadata, CalculationToolParameterSetVector._set_reference_axes_metadata, reference_axes)

    _property_names[vector] = "vector"
    _property_names[reference_axes] = "reference_axes"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolParameterSetVector."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolParameterSetVector)
        ICalculationToolParameterSet.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolParameterSet._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolParameterSetVector, [CalculationToolParameterSetVector, ICalculationToolParameterSet, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5342518829739884757, 14433235443967563942), CalculationToolParameterSetVector)
agcls.AgTypeNameMap["CalculationToolParameterSetVector"] = CalculationToolParameterSetVector

class TimeToolPruneFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """A filter used with event interval list pruned class to prune interval lists..."""
    def __init__(self, source_object=None):
        """Construct an object of type TimeToolPruneFilter."""
        SupportsDeleteCallback.__init__(self)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolPruneFilter, [ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((4753774320714343401, 2720215011582395827), TimeToolPruneFilter)
agcls.AgTypeNameMap["TimeToolPruneFilter"] = TimeToolPruneFilter

class TimeToolPruneFilterFactory(SupportsDeleteCallback):
    """The factory creates pruning filters."""

    _num_methods = 1
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _metadata = {
        "iid_data" : (5042513438281501086, 13689364354501709749),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolPruneFilterFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.EnumArg(IntervalPruneFilterType), agmarshall.InterfaceOutArg,) }
    def create(self, filter:"IntervalPruneFilterType") -> "ITimeToolPruneFilter":
        """Create and initializes a new prune filter using default configuration."""
        return self._intf.invoke(TimeToolPruneFilterFactory._metadata, TimeToolPruneFilterFactory._create_metadata, filter, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type TimeToolPruneFilterFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolPruneFilterFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolPruneFilterFactory, [TimeToolPruneFilterFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5226485936241673410, 17861777124087912090), TimeToolPruneFilterFactory)
agcls.AgTypeNameMap["TimeToolPruneFilterFactory"] = TimeToolPruneFilterFactory

class TimeToolTimeIntervalRelativeSatisfactionConditionFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter selects intervals if certain side condition is satisfied at least/most certain percentage of time."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_condition_method_offset = 1
    _set_condition_method_offset = 2
    _get_duration_type_method_offset = 3
    _set_duration_type_method_offset = 4
    _get_relative_interval_duration_method_offset = 5
    _set_relative_interval_duration_method_offset = 6
    _metadata = {
        "iid_data" : (4708733116692114035, 8453241099873031043),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalRelativeSatisfactionConditionFilter)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def condition(self) -> "ICalculationToolCondition":
        """Get or set the additional condition to be satisfied At Most or At Least specified duration within any interval for it to be considered in filtered list."""
        return self._intf.get_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._get_condition_metadata)

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"),) }
    @condition.setter
    def condition(self, condition:"ICalculationToolCondition") -> None:
        return self._intf.set_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._set_condition_metadata, condition)

    _get_duration_type_metadata = { "offset" : _get_duration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @property
    def duration_type(self) -> "IntervalDurationType":
        """Choose a duration type (at least/at most)."""
        return self._intf.get_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._get_duration_type_metadata)

    _set_duration_type_metadata = { "offset" : _set_duration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @duration_type.setter
    def duration_type(self, duration_kind:"IntervalDurationType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._set_duration_type_metadata, duration_kind)

    _get_relative_interval_duration_metadata = { "offset" : _get_relative_interval_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_interval_duration(self) -> float:
        """A percentage of time the condition must be satisfied."""
        return self._intf.get_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._get_relative_interval_duration_metadata)

    _set_relative_interval_duration_metadata = { "offset" : _set_relative_interval_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_interval_duration.setter
    def relative_interval_duration(self, relative_interval_duration:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalRelativeSatisfactionConditionFilter._metadata, TimeToolTimeIntervalRelativeSatisfactionConditionFilter._set_relative_interval_duration_metadata, relative_interval_duration)

    _property_names[condition] = "condition"
    _property_names[duration_type] = "duration_type"
    _property_names[relative_interval_duration] = "relative_interval_duration"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalRelativeSatisfactionConditionFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalRelativeSatisfactionConditionFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalRelativeSatisfactionConditionFilter, [TimeToolTimeIntervalRelativeSatisfactionConditionFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((5599006480069411520, 196698249444219283), TimeToolTimeIntervalRelativeSatisfactionConditionFilter)
agcls.AgTypeNameMap["TimeToolTimeIntervalRelativeSatisfactionConditionFilter"] = TimeToolTimeIntervalRelativeSatisfactionConditionFilter

class AnalysisWorkbenchSampling(IAnalysisWorkbenchSampling, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Base sampling interface."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchSampling."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchSampling.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchSampling._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchSampling, [IAnalysisWorkbenchSampling, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5764026328462147817, 7669428216318596746), AnalysisWorkbenchSampling)
agcls.AgTypeNameMap["AnalysisWorkbenchSampling"] = AnalysisWorkbenchSampling

class CalculationToolSamplingBasic(IAnalysisWorkbenchSampling, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Sampling definition determines how scalar data should be sampled in order to adequately capture trends in that data."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_sampling_method_method_offset = 1
    _set_sampling_method_method_offset = 2
    _get_method_factory_method_offset = 3
    _metadata = {
        "iid_data" : (4878478022758053645, 16585190424563731886),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolSamplingBasic)

    _get_sampling_method_metadata = { "offset" : _get_sampling_method_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def sampling_method(self) -> "ICalculationToolSamplingMethod":
        """Get the sampling definition, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolSamplingBasic._metadata, CalculationToolSamplingBasic._get_sampling_method_metadata)

    _set_sampling_method_metadata = { "offset" : _set_sampling_method_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolSamplingMethod"),) }
    @sampling_method.setter
    def sampling_method(self, sampling_method:"ICalculationToolSamplingMethod") -> None:
        return self._intf.set_property(CalculationToolSamplingBasic._metadata, CalculationToolSamplingBasic._set_sampling_method_metadata, sampling_method)

    _get_method_factory_metadata = { "offset" : _get_method_factory_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def method_factory(self) -> "CalculationToolSamplingMethodFactory":
        """Create sampling definitions, which can use a fixed step, relative tolerance or curvature tolerance. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples..."""
        return self._intf.get_property(CalculationToolSamplingBasic._metadata, CalculationToolSamplingBasic._get_method_factory_metadata)

    _property_names[sampling_method] = "sampling_method"
    _property_names[method_factory] = "method_factory"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolSamplingBasic)
        IAnalysisWorkbenchSampling.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchSampling._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingBasic, [CalculationToolSamplingBasic, IAnalysisWorkbenchSampling, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4842369728921496656, 433140158192125353), CalculationToolSamplingBasic)
agcls.AgTypeNameMap["CalculationToolSamplingBasic"] = CalculationToolSamplingBasic

class CalculationToolSamplingCurvatureTolerance(ICalculationToolSamplingMethod, SupportsDeleteCallback):
    """Curvature tolerance definition includes parameters that determine how scalar data should be sampled based on limits on slope changes between samples."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_time_step_method_offset = 1
    _set_minimum_time_step_method_offset = 2
    _get_maximum_time_step_method_offset = 3
    _set_maximum_time_step_method_offset = 4
    _get_step_at_boundaries_method_offset = 5
    _set_step_at_boundaries_method_offset = 6
    _get_relative_tolerance_method_offset = 7
    _set_relative_tolerance_method_offset = 8
    _get_absolute_tolerance_method_offset = 9
    _set_absolute_tolerance_method_offset = 10
    _get_curvature_tolerance_method_offset = 11
    _set_curvature_tolerance_method_offset = 12
    _metadata = {
        "iid_data" : (5542979227852921429, 14660439002322066049),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolSamplingCurvatureTolerance)

    _get_minimum_time_step_metadata = { "offset" : _get_minimum_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_time_step(self) -> float:
        """Get the minimum allowed time step."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_minimum_time_step_metadata)

    _set_minimum_time_step_metadata = { "offset" : _set_minimum_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_time_step.setter
    def minimum_time_step(self, minimum_time_step:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_minimum_time_step_metadata, minimum_time_step)

    _get_maximum_time_step_metadata = { "offset" : _get_maximum_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_time_step(self) -> float:
        """Get the maximum allowed time step."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_maximum_time_step_metadata)

    _set_maximum_time_step_metadata = { "offset" : _set_maximum_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_time_step.setter
    def maximum_time_step(self, maximum_time_step:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_maximum_time_step_metadata, maximum_time_step)

    _get_step_at_boundaries_metadata = { "offset" : _get_step_at_boundaries_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step_at_boundaries(self) -> float:
        """Get the step taken at boundaries of discontinuity or availability."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_step_at_boundaries_metadata)

    _set_step_at_boundaries_metadata = { "offset" : _set_step_at_boundaries_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step_at_boundaries.setter
    def step_at_boundaries(self, step_at_boundaries:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_step_at_boundaries_metadata, step_at_boundaries)

    _get_relative_tolerance_metadata = { "offset" : _get_relative_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_tolerance(self) -> float:
        """Get the relative tolerance which determines acceptable difference between predicted and actual changes in values of sampled data over a step relative to the sampled values."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_relative_tolerance_metadata)

    _set_relative_tolerance_metadata = { "offset" : _set_relative_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_tolerance.setter
    def relative_tolerance(self, relative_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_relative_tolerance_metadata, relative_tolerance)

    _get_absolute_tolerance_metadata = { "offset" : _get_absolute_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def absolute_tolerance(self) -> float:
        """Get the absolute tolerance which determines acceptable difference between predicted and actual changes in values of sampled data over a step."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_absolute_tolerance_metadata)

    _set_absolute_tolerance_metadata = { "offset" : _set_absolute_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @absolute_tolerance.setter
    def absolute_tolerance(self, absolute_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_absolute_tolerance_metadata, absolute_tolerance)

    _get_curvature_tolerance_metadata = { "offset" : _get_curvature_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def curvature_tolerance(self) -> float:
        """Get the curvature tolerance which determines acceptable angular difference between slopes over consecutive steps."""
        return self._intf.get_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._get_curvature_tolerance_metadata)

    _set_curvature_tolerance_metadata = { "offset" : _set_curvature_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @curvature_tolerance.setter
    def curvature_tolerance(self, curvature_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolSamplingCurvatureTolerance._metadata, CalculationToolSamplingCurvatureTolerance._set_curvature_tolerance_metadata, curvature_tolerance)

    _property_names[minimum_time_step] = "minimum_time_step"
    _property_names[maximum_time_step] = "maximum_time_step"
    _property_names[step_at_boundaries] = "step_at_boundaries"
    _property_names[relative_tolerance] = "relative_tolerance"
    _property_names[absolute_tolerance] = "absolute_tolerance"
    _property_names[curvature_tolerance] = "curvature_tolerance"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingCurvatureTolerance."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolSamplingCurvatureTolerance)
        ICalculationToolSamplingMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolSamplingMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingCurvatureTolerance, [CalculationToolSamplingCurvatureTolerance, ICalculationToolSamplingMethod])

agcls.AgClassCatalog.add_catalog_entry((5387756316129866756, 2484055402072399547), CalculationToolSamplingCurvatureTolerance)
agcls.AgTypeNameMap["CalculationToolSamplingCurvatureTolerance"] = CalculationToolSamplingCurvatureTolerance

class CalculationToolSamplingFixedStep(ICalculationToolSamplingMethod, SupportsDeleteCallback):
    """Fixed step definition includes parameters that determine how scalar data should be sampled based on fixed steps between samples."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_time_step_method_offset = 1
    _set_time_step_method_offset = 2
    _metadata = {
        "iid_data" : (4760999063576923422, 2852563063428767411),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolSamplingFixedStep)

    _get_time_step_metadata = { "offset" : _get_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_step(self) -> float:
        """Get the fixed time step."""
        return self._intf.get_property(CalculationToolSamplingFixedStep._metadata, CalculationToolSamplingFixedStep._get_time_step_metadata)

    _set_time_step_metadata = { "offset" : _set_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_step.setter
    def time_step(self, time_step:float) -> None:
        return self._intf.set_property(CalculationToolSamplingFixedStep._metadata, CalculationToolSamplingFixedStep._set_time_step_metadata, time_step)

    _property_names[time_step] = "time_step"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingFixedStep."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolSamplingFixedStep)
        ICalculationToolSamplingMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolSamplingMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingFixedStep, [CalculationToolSamplingFixedStep, ICalculationToolSamplingMethod])

agcls.AgClassCatalog.add_catalog_entry((5657779203521466895, 699813618720941244), CalculationToolSamplingFixedStep)
agcls.AgTypeNameMap["CalculationToolSamplingFixedStep"] = CalculationToolSamplingFixedStep

class CalculationToolSamplingMethod(ICalculationToolSamplingMethod, SupportsDeleteCallback):
    """A sampling method."""
    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingMethod."""
        SupportsDeleteCallback.__init__(self)
        ICalculationToolSamplingMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolSamplingMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingMethod, [ICalculationToolSamplingMethod])

agcls.AgClassCatalog.add_catalog_entry((5520509056155849796, 13270445498439148177), CalculationToolSamplingMethod)
agcls.AgTypeNameMap["CalculationToolSamplingMethod"] = CalculationToolSamplingMethod

class CalculationToolSamplingMethodFactory(SupportsDeleteCallback):
    """The factory creates sampling method components."""

    _num_methods = 3
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_fixed_step_method_offset = 1
    _create_curvature_tolerance_method_offset = 2
    _create_relative_tolerance_method_offset = 3
    _metadata = {
        "iid_data" : (4621614585912529563, 1951602854656295068),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolSamplingMethodFactory)

    _create_fixed_step_metadata = { "offset" : _create_fixed_step_method_offset,
            "arg_types" : (agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def create_fixed_step(self, fixed_step:float) -> "ICalculationToolSamplingMethod":
        """Create a fixed time step sampling definition."""
        return self._intf.invoke(CalculationToolSamplingMethodFactory._metadata, CalculationToolSamplingMethodFactory._create_fixed_step_metadata, fixed_step, OutArg())

    _create_curvature_tolerance_metadata = { "offset" : _create_curvature_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def create_curvature_tolerance(self, curvature_tolerance:float) -> "ICalculationToolSamplingMethod":
        """Create a curvature tolerance sampling definition. Curvature tolerance uses changes in slope between samples."""
        return self._intf.invoke(CalculationToolSamplingMethodFactory._metadata, CalculationToolSamplingMethodFactory._create_curvature_tolerance_metadata, curvature_tolerance, OutArg())

    _create_relative_tolerance_metadata = { "offset" : _create_relative_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def create_relative_tolerance(self, relative_tolerance:float) -> "ICalculationToolSamplingMethod":
        """Create a relative tolerance sampling definition. Relative tolerance uses a combination of relative and absolute changes in scalar values between samples."""
        return self._intf.invoke(CalculationToolSamplingMethodFactory._metadata, CalculationToolSamplingMethodFactory._create_relative_tolerance_metadata, relative_tolerance, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingMethodFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolSamplingMethodFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingMethodFactory, [CalculationToolSamplingMethodFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5108812290465688521, 9237146689306794658), CalculationToolSamplingMethodFactory)
agcls.AgTypeNameMap["CalculationToolSamplingMethodFactory"] = CalculationToolSamplingMethodFactory

class CalculationToolSamplingRelativeTolerance(ICalculationToolSamplingMethod, SupportsDeleteCallback):
    """Relative tolerance definition includes parameters that determine how scalar data should be sampled based on limits on difference between actual changes between samples and changes predicted by dead reckoning."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_time_step_method_offset = 1
    _set_minimum_time_step_method_offset = 2
    _get_maximum_time_step_method_offset = 3
    _set_maximum_time_step_method_offset = 4
    _get_step_at_boundaries_method_offset = 5
    _set_step_at_boundaries_method_offset = 6
    _get_relative_tolerance_method_offset = 7
    _set_relative_tolerance_method_offset = 8
    _get_absolute_tolerance_method_offset = 9
    _set_absolute_tolerance_method_offset = 10
    _metadata = {
        "iid_data" : (4753454007727116326, 13557865281292527789),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, CalculationToolSamplingRelativeTolerance)

    _get_minimum_time_step_metadata = { "offset" : _get_minimum_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_time_step(self) -> float:
        """Get the minimum allowed time step."""
        return self._intf.get_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._get_minimum_time_step_metadata)

    _set_minimum_time_step_metadata = { "offset" : _set_minimum_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_time_step.setter
    def minimum_time_step(self, minimum_time_step:float) -> None:
        return self._intf.set_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._set_minimum_time_step_metadata, minimum_time_step)

    _get_maximum_time_step_metadata = { "offset" : _get_maximum_time_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_time_step(self) -> float:
        """Get the maximum allowed time step."""
        return self._intf.get_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._get_maximum_time_step_metadata)

    _set_maximum_time_step_metadata = { "offset" : _set_maximum_time_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_time_step.setter
    def maximum_time_step(self, maximum_time_step:float) -> None:
        return self._intf.set_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._set_maximum_time_step_metadata, maximum_time_step)

    _get_step_at_boundaries_metadata = { "offset" : _get_step_at_boundaries_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step_at_boundaries(self) -> float:
        """Get the step taken at boundaries of discontinuity or availability."""
        return self._intf.get_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._get_step_at_boundaries_metadata)

    _set_step_at_boundaries_metadata = { "offset" : _set_step_at_boundaries_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step_at_boundaries.setter
    def step_at_boundaries(self, step_at_boundaries:float) -> None:
        return self._intf.set_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._set_step_at_boundaries_metadata, step_at_boundaries)

    _get_relative_tolerance_metadata = { "offset" : _get_relative_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def relative_tolerance(self) -> float:
        """Get the relative tolerance which determines acceptable difference between predicted and actual changes in values of sampled data over a step relative to the sampled values."""
        return self._intf.get_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._get_relative_tolerance_metadata)

    _set_relative_tolerance_metadata = { "offset" : _set_relative_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @relative_tolerance.setter
    def relative_tolerance(self, relative_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._set_relative_tolerance_metadata, relative_tolerance)

    _get_absolute_tolerance_metadata = { "offset" : _get_absolute_tolerance_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def absolute_tolerance(self) -> float:
        """Get the absolute tolerance which determines acceptable difference between predicted and actual changes in values of sampled data over a step."""
        return self._intf.get_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._get_absolute_tolerance_metadata)

    _set_absolute_tolerance_metadata = { "offset" : _set_absolute_tolerance_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @absolute_tolerance.setter
    def absolute_tolerance(self, absolute_tolerance:float) -> None:
        return self._intf.set_property(CalculationToolSamplingRelativeTolerance._metadata, CalculationToolSamplingRelativeTolerance._set_absolute_tolerance_metadata, absolute_tolerance)

    _property_names[minimum_time_step] = "minimum_time_step"
    _property_names[maximum_time_step] = "maximum_time_step"
    _property_names[step_at_boundaries] = "step_at_boundaries"
    _property_names[relative_tolerance] = "relative_tolerance"
    _property_names[absolute_tolerance] = "absolute_tolerance"

    def __init__(self, source_object=None):
        """Construct an object of type CalculationToolSamplingRelativeTolerance."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, CalculationToolSamplingRelativeTolerance)
        ICalculationToolSamplingMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ICalculationToolSamplingMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, CalculationToolSamplingRelativeTolerance, [CalculationToolSamplingRelativeTolerance, ICalculationToolSamplingMethod])

agcls.AgClassCatalog.add_catalog_entry((4936649529186074219, 353883352016272033), CalculationToolSamplingRelativeTolerance)
agcls.AgTypeNameMap["CalculationToolSamplingRelativeTolerance"] = CalculationToolSamplingRelativeTolerance

class TimeToolTimeIntervalSatisfactionConditionFilter(ITimeToolPruneFilter, SupportsDeleteCallback):
    """The filter selects intervals if certain side condition is satisfied at least/most certain duration."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_condition_method_offset = 1
    _set_condition_method_offset = 2
    _get_duration_type_method_offset = 3
    _set_duration_type_method_offset = 4
    _get_interval_duration_method_offset = 5
    _set_interval_duration_method_offset = 6
    _metadata = {
        "iid_data" : (5756195466743196097, 11251701537728199608),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolTimeIntervalSatisfactionConditionFilter)

    _get_condition_metadata = { "offset" : _get_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def condition(self) -> "ICalculationToolCondition":
        """Get or set the additional condition to be satisfied At Most or At Least specified duration within any interval for it to be considered in filtered list."""
        return self._intf.get_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._get_condition_metadata)

    _set_condition_metadata = { "offset" : _set_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ICalculationToolCondition"),) }
    @condition.setter
    def condition(self, condition:"ICalculationToolCondition") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._set_condition_metadata, condition)

    _get_duration_type_metadata = { "offset" : _get_duration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @property
    def duration_type(self) -> "IntervalDurationType":
        """Choose a duration type (at least/at most)."""
        return self._intf.get_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._get_duration_type_metadata)

    _set_duration_type_metadata = { "offset" : _set_duration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(IntervalDurationType),) }
    @duration_type.setter
    def duration_type(self, duration_kind:"IntervalDurationType") -> None:
        return self._intf.set_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._set_duration_type_metadata, duration_kind)

    _get_interval_duration_metadata = { "offset" : _get_interval_duration_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def interval_duration(self) -> float:
        """A duration of time the condition must be satisfied."""
        return self._intf.get_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._get_interval_duration_metadata)

    _set_interval_duration_metadata = { "offset" : _set_interval_duration_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @interval_duration.setter
    def interval_duration(self, interval_duration:float) -> None:
        return self._intf.set_property(TimeToolTimeIntervalSatisfactionConditionFilter._metadata, TimeToolTimeIntervalSatisfactionConditionFilter._set_interval_duration_metadata, interval_duration)

    _property_names[condition] = "condition"
    _property_names[duration_type] = "duration_type"
    _property_names[interval_duration] = "interval_duration"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolTimeIntervalSatisfactionConditionFilter."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolTimeIntervalSatisfactionConditionFilter)
        ITimeToolPruneFilter.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ITimeToolPruneFilter._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolTimeIntervalSatisfactionConditionFilter, [TimeToolTimeIntervalSatisfactionConditionFilter, ITimeToolPruneFilter])

agcls.AgClassCatalog.add_catalog_entry((4851832596060338991, 14965342242796491937), TimeToolTimeIntervalSatisfactionConditionFilter)
agcls.AgTypeNameMap["TimeToolTimeIntervalSatisfactionConditionFilter"] = TimeToolTimeIntervalSatisfactionConditionFilter

class AnalysisWorkbenchSignalDelay(IAnalysisWorkbenchSignalDelay, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Signal delay definition determines how long it takes for a signal to propagate from one location to another."""
    def __init__(self, source_object=None):
        """Construct an object of type AnalysisWorkbenchSignalDelay."""
        SupportsDeleteCallback.__init__(self)
        IAnalysisWorkbenchSignalDelay.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchSignalDelay._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, AnalysisWorkbenchSignalDelay, [IAnalysisWorkbenchSignalDelay, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((4953635476598966337, 4896658474318640771), AnalysisWorkbenchSignalDelay)
agcls.AgTypeNameMap["AnalysisWorkbenchSignalDelay"] = AnalysisWorkbenchSignalDelay

class TimeToolSignalDelayBasic(IAnalysisWorkbenchSignalDelay, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """Signal delay definition determines how long it takes for a signal to propagate from one location to another."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_signal_path_reference_system_method_offset = 1
    _set_signal_path_reference_system_method_offset = 2
    _get_reference_system_method_offset = 3
    _set_reference_system_method_offset = 4
    _get_speed_option_method_offset = 5
    _set_speed_option_method_offset = 6
    _get_transfer_speed_method_offset = 7
    _set_transfer_speed_method_offset = 8
    _get_time_delay_convergence_method_offset = 9
    _set_time_delay_convergence_method_offset = 10
    _metadata = {
        "iid_data" : (5705418358163775767, 12397507959819827338),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolSignalDelayBasic)

    _get_signal_path_reference_system_metadata = { "offset" : _get_signal_path_reference_system_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SignalPathReferenceSystem),) }
    @property
    def signal_path_reference_system(self) -> "SignalPathReferenceSystem":
        """Get the type of signal path reference system which can be set to use STK Access default (see STK Help for further details), Solar system barycenter inertial reference, central body inertial reference or custom reference system..."""
        return self._intf.get_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._get_signal_path_reference_system_metadata)

    _set_signal_path_reference_system_metadata = { "offset" : _set_signal_path_reference_system_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SignalPathReferenceSystem),) }
    @signal_path_reference_system.setter
    def signal_path_reference_system(self, signal_path_reference_system:"SignalPathReferenceSystem") -> None:
        return self._intf.set_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._set_signal_path_reference_system_metadata, signal_path_reference_system)

    _get_reference_system_metadata = { "offset" : _get_reference_system_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_system(self) -> "IVectorGeometryToolSystem":
        """Get the custom reference system which is used as a reference for signal path if the signal path reference option is set to Custom."""
        return self._intf.get_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._get_reference_system_metadata)

    _set_reference_system_metadata = { "offset" : _set_reference_system_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolSystem"),) }
    @reference_system.setter
    def reference_system(self, reference_system:"IVectorGeometryToolSystem") -> None:
        return self._intf.set_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._set_reference_system_metadata, reference_system)

    _get_speed_option_metadata = { "offset" : _get_speed_option_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SpeedType),) }
    @property
    def speed_option(self) -> "SpeedType":
        """Get the speed option which determines whether to use the speed of light or a custom speed value."""
        return self._intf.get_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._get_speed_option_metadata)

    _set_speed_option_metadata = { "offset" : _set_speed_option_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SpeedType),) }
    @speed_option.setter
    def speed_option(self, speed_option:"SpeedType") -> None:
        return self._intf.set_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._set_speed_option_metadata, speed_option)

    _get_transfer_speed_metadata = { "offset" : _get_transfer_speed_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def transfer_speed(self) -> float:
        """Get the signal propagation speed value which is used if the speed option is set to Custom."""
        return self._intf.get_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._get_transfer_speed_metadata)

    _set_transfer_speed_metadata = { "offset" : _set_transfer_speed_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @transfer_speed.setter
    def transfer_speed(self, transfer_speed:float) -> None:
        return self._intf.set_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._set_transfer_speed_metadata, transfer_speed)

    _get_time_delay_convergence_metadata = { "offset" : _get_time_delay_convergence_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_delay_convergence(self) -> float:
        """Get the time delay convergence which determines the accuracy of computed propagation time between the two locations."""
        return self._intf.get_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._get_time_delay_convergence_metadata)

    _set_time_delay_convergence_metadata = { "offset" : _set_time_delay_convergence_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_delay_convergence.setter
    def time_delay_convergence(self, time_delay_convergence:float) -> None:
        return self._intf.set_property(TimeToolSignalDelayBasic._metadata, TimeToolSignalDelayBasic._set_time_delay_convergence_metadata, time_delay_convergence)

    _property_names[signal_path_reference_system] = "signal_path_reference_system"
    _property_names[reference_system] = "reference_system"
    _property_names[speed_option] = "speed_option"
    _property_names[transfer_speed] = "transfer_speed"
    _property_names[time_delay_convergence] = "time_delay_convergence"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolSignalDelayBasic."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolSignalDelayBasic)
        IAnalysisWorkbenchSignalDelay.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        IAnalysisWorkbenchSignalDelay._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolSignalDelayBasic, [TimeToolSignalDelayBasic, IAnalysisWorkbenchSignalDelay, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5642525835138409264, 10528515051945010859), TimeToolSignalDelayBasic)
agcls.AgTypeNameMap["TimeToolSignalDelayBasic"] = TimeToolSignalDelayBasic

class SpatialAnalysisToolCalculationFactory(SupportsDeleteCallback):
    """The factory is used to create instances of volume calcs."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _is_type_supported_method_offset = 1
    _create_method_offset = 2
    _create_altitude_method_offset = 3
    _create_angle_to_location_method_offset = 4
    _create_from_file_method_offset = 5
    _create_from_calculation_scalar_method_offset = 6
    _create_solar_intensity_method_offset = 7
    _create_spatial_condition_satisfaction_metrics_method_offset = 8
    _create_distance_to_location_method_offset = 9
    _create_propagation_delay_to_location_method_offset = 10
    _metadata = {
        "iid_data" : (5124917152380054858, 1631627205173804441),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationFactory)

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(SpatialCalculationType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"SpatialCalculationType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._is_type_supported_metadata, type, OutArg())

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(SpatialCalculationType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"SpatialCalculationType") -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a volume calc using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_metadata, name, description, type, OutArg())

    _create_altitude_metadata = { "offset" : _create_altitude_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_altitude(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a altitude to location volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_altitude_metadata, name, description, OutArg())

    _create_angle_to_location_metadata = { "offset" : _create_angle_to_location_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_angle_to_location(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a angle to location volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_angle_to_location_metadata, name, description, OutArg())

    _create_from_file_metadata = { "offset" : _create_from_file_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_file(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a file volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_from_file_metadata, name, description, OutArg())

    _create_from_calculation_scalar_metadata = { "offset" : _create_from_calculation_scalar_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_calculation_scalar(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a scalar to location volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_from_calculation_scalar_metadata, name, description, OutArg())

    _create_solar_intensity_metadata = { "offset" : _create_solar_intensity_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_solar_intensity(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a solar intensity volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_solar_intensity_metadata, name, description, OutArg())

    _create_spatial_condition_satisfaction_metrics_metadata = { "offset" : _create_spatial_condition_satisfaction_metrics_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_spatial_condition_satisfaction_metrics(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a spatial condition satisfaction metric volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_spatial_condition_satisfaction_metrics_metadata, name, description, OutArg())

    _create_distance_to_location_metadata = { "offset" : _create_distance_to_location_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_distance_to_location(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a distance to location volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_distance_to_location_metadata, name, description, OutArg())

    _create_propagation_delay_to_location_metadata = { "offset" : _create_propagation_delay_to_location_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_propagation_delay_to_location(self, name:str, description:str) -> "ISpatialAnalysisToolSpatialCalculation":
        """Create and registers a distance to location volume calc type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolCalculationFactory._metadata, SpatialAnalysisToolCalculationFactory._create_propagation_delay_to_location_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolCalculationFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolCalculationFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolCalculationFactory, [SpatialAnalysisToolCalculationFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5540830022443821797, 17040724212397248403), SpatialAnalysisToolCalculationFactory)
agcls.AgTypeNameMap["SpatialAnalysisToolCalculationFactory"] = SpatialAnalysisToolCalculationFactory

class SpatialAnalysisToolConditionFactory(SupportsDeleteCallback):
    """The factory is used to create instances of volumes."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _is_type_supported_method_offset = 2
    _create_combined_method_offset = 3
    _create_lighting_method_offset = 4
    _create_volume_over_time_method_offset = 5
    _create_from_grid_method_offset = 6
    _create_from_spatial_calculation_method_offset = 7
    _create_from_time_satisfaction_method_offset = 8
    _create_from_condition_method_offset = 9
    _create_from_access_method_offset = 10
    _metadata = {
        "iid_data" : (4718823887408175374, 1421370750746771080),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolConditionFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(VolumeType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"VolumeType") -> "ISpatialAnalysisToolVolume":
        """Create and registers a volume using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_metadata, name, description, type, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(VolumeType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"VolumeType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._is_type_supported_metadata, type, OutArg())

    _create_combined_metadata = { "offset" : _create_combined_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_combined(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type combined."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_combined_metadata, name, description, OutArg())

    _create_lighting_metadata = { "offset" : _create_lighting_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_lighting(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type lighting."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_lighting_metadata, name, description, OutArg())

    _create_volume_over_time_metadata = { "offset" : _create_volume_over_time_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_volume_over_time(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type over time."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_volume_over_time_metadata, name, description, OutArg())

    _create_from_grid_metadata = { "offset" : _create_from_grid_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_grid(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type from grid."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_from_grid_metadata, name, description, OutArg())

    _create_from_spatial_calculation_metadata = { "offset" : _create_from_spatial_calculation_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_spatial_calculation(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type from calc."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_from_spatial_calculation_metadata, name, description, OutArg())

    _create_from_time_satisfaction_metadata = { "offset" : _create_from_time_satisfaction_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_time_satisfaction(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type from time satisfaction."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_from_time_satisfaction_metadata, name, description, OutArg())

    _create_from_condition_metadata = { "offset" : _create_from_condition_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_condition(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type condition."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_from_condition_metadata, name, description, OutArg())

    _create_from_access_metadata = { "offset" : _create_from_access_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_from_access(self, name:str, description:str) -> "ISpatialAnalysisToolVolume":
        """Create a volume type Inview."""
        return self._intf.invoke(SpatialAnalysisToolConditionFactory._metadata, SpatialAnalysisToolConditionFactory._create_from_access_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolConditionFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolConditionFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolConditionFactory, [SpatialAnalysisToolConditionFactory, ])

agcls.AgClassCatalog.add_catalog_entry((4635516309349170136, 6850651087969509002), SpatialAnalysisToolConditionFactory)
agcls.AgTypeNameMap["SpatialAnalysisToolConditionFactory"] = SpatialAnalysisToolConditionFactory

class SpatialAnalysisToolVolumeGridFactory(SupportsDeleteCallback):
    """The factory is used to create instances of volume grids."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _create_method_offset = 1
    _create_cartesian_method_offset = 2
    _is_type_supported_method_offset = 3
    _create_cylindrical_method_offset = 4
    _create_spherical_method_offset = 5
    _create_constrained_method_offset = 6
    _create_latitude_longitude_altitude_method_offset = 7
    _create_bearing_altitude_method_offset = 8
    _metadata = {
        "iid_data" : (5006269217737268897, 4165198783253421479),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolVolumeGridFactory)

    _create_metadata = { "offset" : _create_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, agcom.LONG, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.EnumArg(VolumeGridType), agmarshall.InterfaceOutArg,) }
    def create(self, name:str, description:str, type:"VolumeGridType") -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a volume grid using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_metadata, name, description, type, OutArg())

    _create_cartesian_metadata = { "offset" : _create_cartesian_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_cartesian(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a cartesian volume grid type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_cartesian_metadata, name, description, OutArg())

    _is_type_supported_metadata = { "offset" : _is_type_supported_method_offset,
            "arg_types" : (agcom.LONG, POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.EnumArg(VolumeGridType), agmarshall.VariantBoolArg,) }
    def is_type_supported(self, type:"VolumeGridType") -> bool:
        """Return whether the specified type is supported."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._is_type_supported_metadata, type, OutArg())

    _create_cylindrical_metadata = { "offset" : _create_cylindrical_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_cylindrical(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a cylindrical volume grid type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_cylindrical_metadata, name, description, OutArg())

    _create_spherical_metadata = { "offset" : _create_spherical_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_spherical(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a spherical volume grid type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_spherical_metadata, name, description, OutArg())

    _create_constrained_metadata = { "offset" : _create_constrained_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_constrained(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a volume grid of type that can be constrained by conditions using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_constrained_metadata, name, description, OutArg())

    _create_latitude_longitude_altitude_metadata = { "offset" : _create_latitude_longitude_altitude_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_latitude_longitude_altitude(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers cartographic volume grid type using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_latitude_longitude_altitude_metadata, name, description, OutArg())

    _create_bearing_altitude_metadata = { "offset" : _create_bearing_altitude_method_offset,
            "arg_types" : (agcom.BSTR, agcom.BSTR, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.BStrArg, agmarshall.BStrArg, agmarshall.InterfaceOutArg,) }
    def create_bearing_altitude(self, name:str, description:str) -> "ISpatialAnalysisToolVolumeGrid":
        """Create and registers a volume grid of type surface bearing using specified name and description."""
        return self._intf.invoke(SpatialAnalysisToolVolumeGridFactory._metadata, SpatialAnalysisToolVolumeGridFactory._create_bearing_altitude_metadata, name, description, OutArg())


    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolVolumeGridFactory."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolVolumeGridFactory)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolVolumeGridFactory, [SpatialAnalysisToolVolumeGridFactory, ])

agcls.AgClassCatalog.add_catalog_entry((5404595843631519249, 16534451179732670352), SpatialAnalysisToolVolumeGridFactory)
agcls.AgTypeNameMap["SpatialAnalysisToolVolumeGridFactory"] = SpatialAnalysisToolVolumeGridFactory

class SpatialAnalysisToolGridCoordinateDefinition(SupportsDeleteCallback):
    """Define a set of coordinate values."""

    _num_methods = 6
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_method_type_method_offset = 1
    _get_grid_values_method_method_offset = 2
    _set_fixed_step_method_offset = 3
    _set_grid_values_fixed_number_of_steps_method_offset = 4
    _set_custom_method_offset = 5
    _set_fixed_number_of_steps_method_offset = 6
    _metadata = {
        "iid_data" : (5130335259321654556, 16843626956261718182),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolGridCoordinateDefinition)

    _get_method_type_metadata = { "offset" : _get_method_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(GridValuesMethodType),) }
    @property
    def method_type(self) -> "GridValuesMethodType":
        """Grid values method type."""
        return self._intf.get_property(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._get_method_type_metadata)

    _get_grid_values_method_metadata = { "offset" : _get_grid_values_method_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def grid_values_method(self) -> "ISpatialAnalysisToolGridValuesMethod":
        """Get or set the grid values interface."""
        return self._intf.get_property(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._get_grid_values_method_metadata)

    _set_fixed_step_metadata = { "offset" : _set_fixed_step_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE, agcom.VARIANT_BOOL, agcom.DOUBLE, agcom.DOUBLE, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.VariantBoolArg, agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.InterfaceOutArg,) }
    def set_fixed_step(self, min:float, max:float, include_min_max:bool, ref_value:float, fixed_step:float) -> "SpatialAnalysisToolGridValuesFixedStep":
        """Set grid values type to fixed step."""
        return self._intf.invoke(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._set_fixed_step_metadata, min, max, include_min_max, ref_value, fixed_step, OutArg())

    _set_grid_values_fixed_number_of_steps_metadata = { "offset" : _set_grid_values_fixed_number_of_steps_method_offset,
            "arg_types" : (agcom.DOUBLE, agcom.DOUBLE, agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.DoubleArg, agmarshall.DoubleArg, agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def set_grid_values_fixed_number_of_steps(self, min:float, max:float, num_steps:int) -> "SpatialAnalysisToolGridValuesFixedNumberOfSteps":
        """Do not use this method, as it is deprecated. Use SetGridValuesFixedNumberOfStepsEx."""
        return self._intf.invoke(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._set_grid_values_fixed_number_of_steps_metadata, min, max, num_steps, OutArg())

    _set_custom_metadata = { "offset" : _set_custom_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY), POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.LPSafearrayArg, agmarshall.InterfaceOutArg,) }
    def set_custom(self, values:list) -> "SpatialAnalysisToolGridValuesCustom":
        """Set grid values type to custom values."""
        return self._intf.invoke(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._set_custom_metadata, values, OutArg())

    _set_fixed_number_of_steps_metadata = { "offset" : _set_fixed_number_of_steps_method_offset,
            "arg_types" : (agcom.PVOID, agcom.PVOID, agcom.INT, POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"), agmarshall.InterfaceInArg("Quantity"), agmarshall.IntArg, agmarshall.InterfaceOutArg,) }
    def set_fixed_number_of_steps(self, min:"Quantity", max:"Quantity", num_steps:int) -> "SpatialAnalysisToolGridValuesFixedNumberOfSteps":
        """Set grid values type to fixed number of steps with min and max as Quantity."""
        return self._intf.invoke(SpatialAnalysisToolGridCoordinateDefinition._metadata, SpatialAnalysisToolGridCoordinateDefinition._set_fixed_number_of_steps_metadata, min, max, num_steps, OutArg())

    _property_names[method_type] = "method_type"
    _property_names[grid_values_method] = "grid_values_method"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolGridCoordinateDefinition."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolGridCoordinateDefinition)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolGridCoordinateDefinition, [SpatialAnalysisToolGridCoordinateDefinition, ])

agcls.AgClassCatalog.add_catalog_entry((4908749182944525328, 3082335439345659541), SpatialAnalysisToolGridCoordinateDefinition)
agcls.AgTypeNameMap["SpatialAnalysisToolGridCoordinateDefinition"] = SpatialAnalysisToolGridCoordinateDefinition

class SpatialAnalysisToolGridValuesCustom(ISpatialAnalysisToolGridValuesMethod, SupportsDeleteCallback):
    """Fixed step grid values."""

    _num_methods = 2
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_values_method_offset = 1
    _set_values_method_offset = 2
    _metadata = {
        "iid_data" : (5079967470133000247, 1440374130877273486),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolGridValuesCustom)

    _get_values_metadata = { "offset" : _get_values_method_offset,
            "arg_types" : (POINTER(agcom.LPSAFEARRAY),),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @property
    def values(self) -> list:
        """Custom set of coordinate values."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesCustom._metadata, SpatialAnalysisToolGridValuesCustom._get_values_metadata)

    _set_values_metadata = { "offset" : _set_values_method_offset,
            "arg_types" : (agcom.LPSAFEARRAY,),
            "marshallers" : (agmarshall.LPSafearrayArg,) }
    @values.setter
    def values(self, values:list) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesCustom._metadata, SpatialAnalysisToolGridValuesCustom._set_values_metadata, values)

    _property_names[values] = "values"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolGridValuesCustom."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolGridValuesCustom)
        ISpatialAnalysisToolGridValuesMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolGridValuesMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolGridValuesCustom, [SpatialAnalysisToolGridValuesCustom, ISpatialAnalysisToolGridValuesMethod])

agcls.AgClassCatalog.add_catalog_entry((4675204159661460389, 16403186152470131590), SpatialAnalysisToolGridValuesCustom)
agcls.AgTypeNameMap["SpatialAnalysisToolGridValuesCustom"] = SpatialAnalysisToolGridValuesCustom

class SpatialAnalysisToolGridValuesFixedNumberOfSteps(ISpatialAnalysisToolGridValuesMethod, SupportsDeleteCallback):
    """Fixed step grid values."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_min_method_offset = 1
    _set_min_method_offset = 2
    _get_max_method_offset = 3
    _set_max_method_offset = 4
    _get_number_of_steps_method_offset = 5
    _set_number_of_steps_method_offset = 6
    _get_minimum_method_offset = 7
    _set_minimum_method_offset = 8
    _get_maximum_method_offset = 9
    _set_maximum_method_offset = 10
    _metadata = {
        "iid_data" : (5139481416355729146, 10630847657046153914),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolGridValuesFixedNumberOfSteps)

    _get_min_metadata = { "offset" : _get_min_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def min(self) -> float:
        """Do not use this property, as it is deprecated. Use MinEx."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._get_min_metadata)

    _set_min_metadata = { "offset" : _set_min_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @min.setter
    def min(self, min:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._set_min_metadata, min)

    _get_max_metadata = { "offset" : _get_max_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def max(self) -> float:
        """Do not use this property, as it is deprecated. Use MaxEx."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._get_max_metadata)

    _set_max_metadata = { "offset" : _set_max_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @max.setter
    def max(self, max:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._set_max_metadata, max)

    _get_number_of_steps_metadata = { "offset" : _get_number_of_steps_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def number_of_steps(self) -> int:
        """The number of steps between coordinate values."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._get_number_of_steps_metadata)

    _set_number_of_steps_metadata = { "offset" : _set_number_of_steps_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @number_of_steps.setter
    def number_of_steps(self, number_of_steps:int) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._set_number_of_steps_metadata, number_of_steps)

    _get_minimum_metadata = { "offset" : _get_minimum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def minimum(self) -> "Quantity":
        """Minimum coordinate value as Quantity."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._get_minimum_metadata)

    _set_minimum_metadata = { "offset" : _set_minimum_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"),) }
    @minimum.setter
    def minimum(self, min_ex:"Quantity") -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._set_minimum_metadata, min_ex)

    _get_maximum_metadata = { "offset" : _get_maximum_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def maximum(self) -> "Quantity":
        """Maximum coordinate value as Quantity."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._get_maximum_metadata)

    _set_maximum_metadata = { "offset" : _set_maximum_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("Quantity"),) }
    @maximum.setter
    def maximum(self, max_ex:"Quantity") -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedNumberOfSteps._metadata, SpatialAnalysisToolGridValuesFixedNumberOfSteps._set_maximum_metadata, max_ex)

    _property_names[min] = "min"
    _property_names[max] = "max"
    _property_names[number_of_steps] = "number_of_steps"
    _property_names[minimum] = "minimum"
    _property_names[maximum] = "maximum"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolGridValuesFixedNumberOfSteps."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolGridValuesFixedNumberOfSteps)
        ISpatialAnalysisToolGridValuesMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolGridValuesMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolGridValuesFixedNumberOfSteps, [SpatialAnalysisToolGridValuesFixedNumberOfSteps, ISpatialAnalysisToolGridValuesMethod])

agcls.AgClassCatalog.add_catalog_entry((4906412158379095950, 18165921868605089162), SpatialAnalysisToolGridValuesFixedNumberOfSteps)
agcls.AgTypeNameMap["SpatialAnalysisToolGridValuesFixedNumberOfSteps"] = SpatialAnalysisToolGridValuesFixedNumberOfSteps

class SpatialAnalysisToolGridValuesFixedStep(ISpatialAnalysisToolGridValuesMethod, SupportsDeleteCallback):
    """Fixed step grid values."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_minimum_method_offset = 1
    _set_minimum_method_offset = 2
    _get_maximum_method_offset = 3
    _set_maximum_method_offset = 4
    _get_include_minimum_maximum_method_offset = 5
    _set_include_minimum_maximum_method_offset = 6
    _get_reference_value_method_offset = 7
    _set_reference_value_method_offset = 8
    _get_step_method_offset = 9
    _set_step_method_offset = 10
    _metadata = {
        "iid_data" : (5343659644451078017, 5778378386165949108),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolGridValuesFixedStep)

    _get_minimum_metadata = { "offset" : _get_minimum_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum(self) -> float:
        """Minimum coordinate value."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._get_minimum_metadata)

    _set_minimum_metadata = { "offset" : _set_minimum_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum.setter
    def minimum(self, min:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._set_minimum_metadata, min)

    _get_maximum_metadata = { "offset" : _get_maximum_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum(self) -> float:
        """Maximum coordinate value."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._get_maximum_metadata)

    _set_maximum_metadata = { "offset" : _set_maximum_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum.setter
    def maximum(self, max:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._set_maximum_metadata, max)

    _get_include_minimum_maximum_metadata = { "offset" : _get_include_minimum_maximum_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def include_minimum_maximum(self) -> bool:
        """Flag indicating whether to include minimum and maximum coordinate values in the defined set of values."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._get_include_minimum_maximum_metadata)

    _set_include_minimum_maximum_metadata = { "offset" : _set_include_minimum_maximum_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @include_minimum_maximum.setter
    def include_minimum_maximum(self, include_min_max:bool) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._set_include_minimum_maximum_metadata, include_min_max)

    _get_reference_value_metadata = { "offset" : _get_reference_value_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def reference_value(self) -> float:
        """Reference coordinate value from which steps are taken."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._get_reference_value_metadata)

    _set_reference_value_metadata = { "offset" : _set_reference_value_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @reference_value.setter
    def reference_value(self, reference_value:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._set_reference_value_metadata, reference_value)

    _get_step_metadata = { "offset" : _get_step_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def step(self) -> float:
        """Step between subsequent coordinate values."""
        return self._intf.get_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._get_step_metadata)

    _set_step_metadata = { "offset" : _set_step_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @step.setter
    def step(self, step:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolGridValuesFixedStep._metadata, SpatialAnalysisToolGridValuesFixedStep._set_step_metadata, step)

    _property_names[minimum] = "minimum"
    _property_names[maximum] = "maximum"
    _property_names[include_minimum_maximum] = "include_minimum_maximum"
    _property_names[reference_value] = "reference_value"
    _property_names[step] = "step"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolGridValuesFixedStep."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolGridValuesFixedStep)
        ISpatialAnalysisToolGridValuesMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolGridValuesMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolGridValuesFixedStep, [SpatialAnalysisToolGridValuesFixedStep, ISpatialAnalysisToolGridValuesMethod])

agcls.AgClassCatalog.add_catalog_entry((4734747611981081133, 2146139380126170776), SpatialAnalysisToolGridValuesFixedStep)
agcls.AgTypeNameMap["SpatialAnalysisToolGridValuesFixedStep"] = SpatialAnalysisToolGridValuesFixedStep

class SpatialAnalysisToolGridValuesMethod(ISpatialAnalysisToolGridValuesMethod, SupportsDeleteCallback):
    """A grid values method."""
    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolGridValuesMethod."""
        SupportsDeleteCallback.__init__(self)
        ISpatialAnalysisToolGridValuesMethod.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolGridValuesMethod._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolGridValuesMethod, [ISpatialAnalysisToolGridValuesMethod])

agcls.AgClassCatalog.add_catalog_entry((4916369434262782655, 3989956864512378539), SpatialAnalysisToolGridValuesMethod)
agcls.AgTypeNameMap["SpatialAnalysisToolGridValuesMethod"] = SpatialAnalysisToolGridValuesMethod

class TimeToolLightTimeDelay(SupportsDeleteCallback):
    """Manage Light Time Delay options.."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_use_light_time_delay_method_offset = 1
    _set_use_light_time_delay_method_offset = 2
    _get_time_delay_convergence_method_offset = 3
    _set_time_delay_convergence_method_offset = 4
    _get_aberration_type_method_offset = 5
    _set_aberration_type_method_offset = 6
    _get_clock_host_method_offset = 7
    _set_clock_host_method_offset = 8
    _get_time_sense_method_offset = 9
    _set_time_sense_method_offset = 10
    _metadata = {
        "iid_data" : (5571719835290065762, 15016412572713881232),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, TimeToolLightTimeDelay)

    _get_use_light_time_delay_metadata = { "offset" : _get_use_light_time_delay_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_light_time_delay(self) -> bool:
        """Flag indicating whether to include minimum and maximum coordinate values in the defined set of values."""
        return self._intf.get_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._get_use_light_time_delay_metadata)

    _set_use_light_time_delay_metadata = { "offset" : _set_use_light_time_delay_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_light_time_delay.setter
    def use_light_time_delay(self, use_light_time_delay:bool) -> None:
        return self._intf.set_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._set_use_light_time_delay_metadata, use_light_time_delay)

    _get_time_delay_convergence_metadata = { "offset" : _get_time_delay_convergence_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def time_delay_convergence(self) -> float:
        """Set light Time Delay Convergence."""
        return self._intf.get_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._get_time_delay_convergence_metadata)

    _set_time_delay_convergence_metadata = { "offset" : _set_time_delay_convergence_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @time_delay_convergence.setter
    def time_delay_convergence(self, time_delay_convergence:float) -> None:
        return self._intf.set_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._set_time_delay_convergence_metadata, time_delay_convergence)

    _get_aberration_type_metadata = { "offset" : _get_aberration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AberrationModelType),) }
    @property
    def aberration_type(self) -> "AberrationModelType":
        """Method used to Aberration Type."""
        return self._intf.get_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._get_aberration_type_metadata)

    _set_aberration_type_metadata = { "offset" : _set_aberration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AberrationModelType),) }
    @aberration_type.setter
    def aberration_type(self, aberration_type:"AberrationModelType") -> None:
        return self._intf.set_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._set_aberration_type_metadata, aberration_type)

    _get_clock_host_metadata = { "offset" : _get_clock_host_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(ClockHostType),) }
    @property
    def clock_host(self) -> "ClockHostType":
        """Indicate whether object1 or object2 of an Access instance holds the clock for Access times."""
        return self._intf.get_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._get_clock_host_metadata)

    _set_clock_host_metadata = { "offset" : _set_clock_host_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(ClockHostType),) }
    @clock_host.setter
    def clock_host(self, clock_host:"ClockHostType") -> None:
        return self._intf.set_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._set_clock_host_metadata, clock_host)

    _get_time_sense_metadata = { "offset" : _get_time_sense_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(TimeSenseType),) }
    @property
    def time_sense(self) -> "TimeSenseType":
        """Indicate whether apparent position is computed in a transmit or receive sense."""
        return self._intf.get_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._get_time_sense_metadata)

    _set_time_sense_metadata = { "offset" : _set_time_sense_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(TimeSenseType),) }
    @time_sense.setter
    def time_sense(self, time_sense:"TimeSenseType") -> None:
        return self._intf.set_property(TimeToolLightTimeDelay._metadata, TimeToolLightTimeDelay._set_time_sense_metadata, time_sense)

    _property_names[use_light_time_delay] = "use_light_time_delay"
    _property_names[time_delay_convergence] = "time_delay_convergence"
    _property_names[aberration_type] = "aberration_type"
    _property_names[clock_host] = "clock_host"
    _property_names[time_sense] = "time_sense"

    def __init__(self, source_object=None):
        """Construct an object of type TimeToolLightTimeDelay."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, TimeToolLightTimeDelay)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, TimeToolLightTimeDelay, [TimeToolLightTimeDelay, ])

agcls.AgClassCatalog.add_catalog_entry((5707542397246644802, 8133195754214030981), TimeToolLightTimeDelay)
agcls.AgTypeNameMap["TimeToolLightTimeDelay"] = TimeToolLightTimeDelay

class SpatialAnalysisToolVolume(ISpatialAnalysisToolVolume, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume interface. The methods and properties of the interface provide Volume functions."""
    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolVolume."""
        SupportsDeleteCallback.__init__(self)
        ISpatialAnalysisToolVolume.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolVolume._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolVolume, [ISpatialAnalysisToolVolume, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5136212516512030015, 17210711229734835135), SpatialAnalysisToolVolume)
agcls.AgTypeNameMap["SpatialAnalysisToolVolume"] = SpatialAnalysisToolVolume

class SpatialAnalysisToolSpatialCalculation(ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume calc interface. The methods and properties of the interface provide Volumetric calc functions."""
    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolSpatialCalculation."""
        SupportsDeleteCallback.__init__(self)
        ISpatialAnalysisToolSpatialCalculation.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolSpatialCalculation._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolSpatialCalculation, [ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5603624786402262993, 11281447092007354792), SpatialAnalysisToolSpatialCalculation)
agcls.AgTypeNameMap["SpatialAnalysisToolSpatialCalculation"] = SpatialAnalysisToolSpatialCalculation

class SpatialAnalysisToolCalculationAltitude(ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume calc altitude interface."""

    _num_methods = 8
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_central_body_method_offset = 1
    _set_central_body_method_offset = 2
    _get_shape_model_method_offset = 3
    _set_shape_model_method_offset = 4
    _get_use_custom_reference_method_offset = 5
    _set_use_custom_reference_method_offset = 6
    _get_reference_point_method_offset = 7
    _set_reference_point_method_offset = 8
    _metadata = {
        "iid_data" : (5267149886846502538, 15246089171375647638),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationAltitude)

    _get_central_body_metadata = { "offset" : _get_central_body_method_offset,
            "arg_types" : (POINTER(agcom.BSTR),),
            "marshallers" : (agmarshall.BStrArg,) }
    @property
    def central_body(self) -> str:
        """Get the central body for the volume calc. Both the central body reference shape and its CBF (central body centered fixed) system are used by this volume calc."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._get_central_body_metadata)

    _set_central_body_metadata = { "offset" : _set_central_body_method_offset,
            "arg_types" : (agcom.BSTR,),
            "marshallers" : (agmarshall.BStrArg,) }
    @central_body.setter
    def central_body(self, central_body:str) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._set_central_body_metadata, central_body)

    _get_shape_model_metadata = { "offset" : _get_shape_model_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(SpatialCalculationAltitudeReferenceType),) }
    @property
    def shape_model(self) -> "SpatialCalculationAltitudeReferenceType":
        """The Volume Calc Altitude Reference Type."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._get_shape_model_metadata)

    _set_shape_model_metadata = { "offset" : _set_shape_model_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(SpatialCalculationAltitudeReferenceType),) }
    @shape_model.setter
    def shape_model(self, shape_model:"SpatialCalculationAltitudeReferenceType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._set_shape_model_metadata, shape_model)

    _get_use_custom_reference_metadata = { "offset" : _get_use_custom_reference_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_custom_reference(self) -> bool:
        """Whether to use custom reference."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._get_use_custom_reference_metadata)

    _set_use_custom_reference_metadata = { "offset" : _set_use_custom_reference_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_custom_reference.setter
    def use_custom_reference(self, use_custom_reference:bool) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._set_use_custom_reference_metadata, use_custom_reference)

    _get_reference_point_metadata = { "offset" : _get_reference_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_point(self) -> "IVectorGeometryToolPoint":
        """A reference point. Can be any point from VGT."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._get_reference_point_metadata)

    _set_reference_point_metadata = { "offset" : _set_reference_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @reference_point.setter
    def reference_point(self, reference_point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAltitude._metadata, SpatialAnalysisToolCalculationAltitude._set_reference_point_metadata, reference_point)

    _property_names[central_body] = "central_body"
    _property_names[shape_model] = "shape_model"
    _property_names[use_custom_reference] = "use_custom_reference"
    _property_names[reference_point] = "reference_point"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolCalculationAltitude."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolCalculationAltitude)
        ISpatialAnalysisToolSpatialCalculation.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolSpatialCalculation._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolCalculationAltitude, [SpatialAnalysisToolCalculationAltitude, ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5246343411400455836, 13693915155416519566), SpatialAnalysisToolCalculationAltitude)
agcls.AgTypeNameMap["SpatialAnalysisToolCalculationAltitude"] = SpatialAnalysisToolCalculationAltitude

class SpatialAnalysisToolCalculationAngleToLocation(ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume calc angle off vector interface."""

    _num_methods = 10
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_angle_method_offset = 1
    _set_angle_method_offset = 2
    _get_reference_plane_method_offset = 3
    _set_reference_plane_method_offset = 4
    _get_reference_point_method_offset = 5
    _set_reference_point_method_offset = 6
    _get_reference_vector_method_offset = 7
    _set_reference_vector_method_offset = 8
    _get_about_vector_method_offset = 9
    _set_about_vector_method_offset = 10
    _metadata = {
        "iid_data" : (5168636768733921460, 8889143688558765241),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationAngleToLocation)

    _get_angle_metadata = { "offset" : _get_angle_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(AngleToLocationType),) }
    @property
    def angle(self) -> "AngleToLocationType":
        """The Volume Calc Angle Off Vector Type."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._get_angle_metadata)

    _set_angle_metadata = { "offset" : _set_angle_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(AngleToLocationType),) }
    @angle.setter
    def angle(self, angle:"AngleToLocationType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._set_angle_metadata, angle)

    _get_reference_plane_metadata = { "offset" : _get_reference_plane_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_plane(self) -> "IVectorGeometryToolPlane":
        """The Volume Calc Angle Off Vector reference plane."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._get_reference_plane_metadata)

    _set_reference_plane_metadata = { "offset" : _set_reference_plane_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPlane"),) }
    @reference_plane.setter
    def reference_plane(self, reference_plane:"IVectorGeometryToolPlane") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._set_reference_plane_metadata, reference_plane)

    _get_reference_point_metadata = { "offset" : _get_reference_point_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_point(self) -> "IVectorGeometryToolPoint":
        """The Volume Calc Angle Off Vector reference point."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._get_reference_point_metadata)

    _set_reference_point_metadata = { "offset" : _set_reference_point_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolPoint"),) }
    @reference_point.setter
    def reference_point(self, reference_point:"IVectorGeometryToolPoint") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._set_reference_point_metadata, reference_point)

    _get_reference_vector_metadata = { "offset" : _get_reference_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def reference_vector(self) -> "IVectorGeometryToolVector":
        """The Volume Calc Angle Off Vector reference vector."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._get_reference_vector_metadata)

    _set_reference_vector_metadata = { "offset" : _set_reference_vector_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @reference_vector.setter
    def reference_vector(self, reference_vector:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._set_reference_vector_metadata, reference_vector)

    _get_about_vector_metadata = { "offset" : _get_about_vector_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def about_vector(self) -> "IVectorGeometryToolVector":
        """The Volume Calc Angle Off Vector reference about vector."""
        return self._intf.get_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._get_about_vector_metadata)

    _set_about_vector_metadata = { "offset" : _set_about_vector_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("IVectorGeometryToolVector"),) }
    @about_vector.setter
    def about_vector(self, about_vector:"IVectorGeometryToolVector") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationAngleToLocation._metadata, SpatialAnalysisToolCalculationAngleToLocation._set_about_vector_metadata, about_vector)

    _property_names[angle] = "angle"
    _property_names[reference_plane] = "reference_plane"
    _property_names[reference_point] = "reference_point"
    _property_names[reference_vector] = "reference_vector"
    _property_names[about_vector] = "about_vector"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolCalculationAngleToLocation."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolCalculationAngleToLocation)
        ISpatialAnalysisToolSpatialCalculation.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolSpatialCalculation._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolCalculationAngleToLocation, [SpatialAnalysisToolCalculationAngleToLocation, ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5561615505447228747, 2307594500702910097), SpatialAnalysisToolCalculationAngleToLocation)
agcls.AgTypeNameMap["SpatialAnalysisToolCalculationAngleToLocation"] = SpatialAnalysisToolCalculationAngleToLocation

class SpatialAnalysisToolCalculationConditionSatisfactionMetric(ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume calc condition satisfaction interface."""

    _num_methods = 20
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_spatial_condition_method_offset = 1
    _set_spatial_condition_method_offset = 2
    _get_satisfaction_metric_method_offset = 3
    _set_satisfaction_metric_method_offset = 4
    _get_accumulation_type_method_offset = 5
    _set_accumulation_type_method_offset = 6
    _get_duration_type_method_offset = 7
    _set_duration_type_method_offset = 8
    _get_filter_method_offset = 9
    _set_filter_method_offset = 10
    _get_maximum_number_of_intervals_method_offset = 11
    _set_maximum_number_of_intervals_method_offset = 12
    _get_use_minimum_duration_method_offset = 13
    _set_use_minimum_duration_method_offset = 14
    _get_use_maximum_duration_method_offset = 15
    _set_use_maximum_duration_method_offset = 16
    _get_minimum_duration_time_method_offset = 17
    _set_minimum_duration_time_method_offset = 18
    _get_maximum_duration_time_method_offset = 19
    _set_maximum_duration_time_method_offset = 20
    _metadata = {
        "iid_data" : (4773877820582895346, 18047239333924917439),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationConditionSatisfactionMetric)

    _get_spatial_condition_metadata = { "offset" : _get_spatial_condition_method_offset,
            "arg_types" : (POINTER(agcom.PVOID),),
            "marshallers" : (agmarshall.InterfaceOutArg,) }
    @property
    def spatial_condition(self) -> "ISpatialAnalysisToolVolume":
        """A spatial condition for satisfaction metric."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_spatial_condition_metadata)

    _set_spatial_condition_metadata = { "offset" : _set_spatial_condition_method_offset,
            "arg_types" : (agcom.PVOID,),
            "marshallers" : (agmarshall.InterfaceInArg("ISpatialAnalysisToolVolume"),) }
    @spatial_condition.setter
    def spatial_condition(self, spatial_condition:"ISpatialAnalysisToolVolume") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_spatial_condition_metadata, spatial_condition)

    _get_satisfaction_metric_metadata = { "offset" : _get_satisfaction_metric_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionMetricType),) }
    @property
    def satisfaction_metric(self) -> "VolumeSatisfactionMetricType":
        """Spatial condition satisfaction metric types."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_satisfaction_metric_metadata)

    _set_satisfaction_metric_metadata = { "offset" : _set_satisfaction_metric_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionMetricType),) }
    @satisfaction_metric.setter
    def satisfaction_metric(self, satisfaction_metric:"VolumeSatisfactionMetricType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_satisfaction_metric_metadata, satisfaction_metric)

    _get_accumulation_type_metadata = { "offset" : _get_accumulation_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionAccumulationType),) }
    @property
    def accumulation_type(self) -> "VolumeSatisfactionAccumulationType":
        """Spatial condition satisfaction accumulation types."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_accumulation_type_metadata)

    _set_accumulation_type_metadata = { "offset" : _set_accumulation_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionAccumulationType),) }
    @accumulation_type.setter
    def accumulation_type(self, accumulation_type:"VolumeSatisfactionAccumulationType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_accumulation_type_metadata, accumulation_type)

    _get_duration_type_metadata = { "offset" : _get_duration_type_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionDurationType),) }
    @property
    def duration_type(self) -> "VolumeSatisfactionDurationType":
        """Spatial condition satisfaction duration types."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_duration_type_metadata)

    _set_duration_type_metadata = { "offset" : _set_duration_type_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionDurationType),) }
    @duration_type.setter
    def duration_type(self, duration_type:"VolumeSatisfactionDurationType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_duration_type_metadata, duration_type)

    _get_filter_metadata = { "offset" : _get_filter_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionFilterType),) }
    @property
    def filter(self) -> "VolumeSatisfactionFilterType":
        """Spatial condition satisfaction duration types."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_filter_metadata)

    _set_filter_metadata = { "offset" : _set_filter_method_offset,
            "arg_types" : (agcom.LONG,),
            "marshallers" : (agmarshall.EnumArg(VolumeSatisfactionFilterType),) }
    @filter.setter
    def filter(self, filter:"VolumeSatisfactionFilterType") -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_filter_metadata, filter)

    _get_maximum_number_of_intervals_metadata = { "offset" : _get_maximum_number_of_intervals_method_offset,
            "arg_types" : (POINTER(agcom.INT),),
            "marshallers" : (agmarshall.IntArg,) }
    @property
    def maximum_number_of_intervals(self) -> int:
        """Spatial condition satisfaction Maximum number of intervals."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_maximum_number_of_intervals_metadata)

    _set_maximum_number_of_intervals_metadata = { "offset" : _set_maximum_number_of_intervals_method_offset,
            "arg_types" : (agcom.INT,),
            "marshallers" : (agmarshall.IntArg,) }
    @maximum_number_of_intervals.setter
    def maximum_number_of_intervals(self, maximum_number_of_intervals:int) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_maximum_number_of_intervals_metadata, maximum_number_of_intervals)

    _get_use_minimum_duration_metadata = { "offset" : _get_use_minimum_duration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_minimum_duration(self) -> bool:
        """Spatial condition satisfaction enable minimum duration."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_use_minimum_duration_metadata)

    _set_use_minimum_duration_metadata = { "offset" : _set_use_minimum_duration_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_minimum_duration.setter
    def use_minimum_duration(self, use_minimum_duration:bool) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_use_minimum_duration_metadata, use_minimum_duration)

    _get_use_maximum_duration_metadata = { "offset" : _get_use_maximum_duration_method_offset,
            "arg_types" : (POINTER(agcom.VARIANT_BOOL),),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @property
    def use_maximum_duration(self) -> bool:
        """Spatial condition satisfaction enable maximum duration."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_use_maximum_duration_metadata)

    _set_use_maximum_duration_metadata = { "offset" : _set_use_maximum_duration_method_offset,
            "arg_types" : (agcom.VARIANT_BOOL,),
            "marshallers" : (agmarshall.VariantBoolArg,) }
    @use_maximum_duration.setter
    def use_maximum_duration(self, use_maximum_duration:bool) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_use_maximum_duration_metadata, use_maximum_duration)

    _get_minimum_duration_time_metadata = { "offset" : _get_minimum_duration_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def minimum_duration_time(self) -> float:
        """Spatial condition satisfaction minimum duration time."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_minimum_duration_time_metadata)

    _set_minimum_duration_time_metadata = { "offset" : _set_minimum_duration_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @minimum_duration_time.setter
    def minimum_duration_time(self, minimum_duration_time:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_minimum_duration_time_metadata, minimum_duration_time)

    _get_maximum_duration_time_metadata = { "offset" : _get_maximum_duration_time_method_offset,
            "arg_types" : (POINTER(agcom.DOUBLE),),
            "marshallers" : (agmarshall.DoubleArg,) }
    @property
    def maximum_duration_time(self) -> float:
        """Spatial condition satisfaction maximum duration time."""
        return self._intf.get_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._get_maximum_duration_time_metadata)

    _set_maximum_duration_time_metadata = { "offset" : _set_maximum_duration_time_method_offset,
            "arg_types" : (agcom.DOUBLE,),
            "marshallers" : (agmarshall.DoubleArg,) }
    @maximum_duration_time.setter
    def maximum_duration_time(self, maximum_duration_time:float) -> None:
        return self._intf.set_property(SpatialAnalysisToolCalculationConditionSatisfactionMetric._metadata, SpatialAnalysisToolCalculationConditionSatisfactionMetric._set_maximum_duration_time_metadata, maximum_duration_time)

    _property_names[spatial_condition] = "spatial_condition"
    _property_names[satisfaction_metric] = "satisfaction_metric"
    _property_names[accumulation_type] = "accumulation_type"
    _property_names[duration_type] = "duration_type"
    _property_names[filter] = "filter"
    _property_names[maximum_number_of_intervals] = "maximum_number_of_intervals"
    _property_names[use_minimum_duration] = "use_minimum_duration"
    _property_names[use_maximum_duration] = "use_maximum_duration"
    _property_names[minimum_duration_time] = "minimum_duration_time"
    _property_names[maximum_duration_time] = "maximum_duration_time"

    def __init__(self, source_object=None):
        """Construct an object of type SpatialAnalysisToolCalculationConditionSatisfactionMetric."""
        SupportsDeleteCallback.__init__(self)
        initialize_from_source_object(self, source_object, SpatialAnalysisToolCalculationConditionSatisfactionMetric)
        ISpatialAnalysisToolSpatialCalculation.__init__(self, source_object)
        IAnalysisWorkbenchComponent.__init__(self, source_object)
    def _private_init(self, intf:InterfaceProxy):
        self.__dict__["_intf"] = intf
        ISpatialAnalysisToolSpatialCalculation._private_init(self, intf)
        IAnalysisWorkbenchComponent._private_init(self, intf)
    def __eq__(self, other):
        """Check equality of the underlying STK references."""
        return agcls.compare_com_objects(self, other)
    def __setattr__(self, attrname, value):
        """Attempt to assign an attribute."""
        set_class_attribute(self, attrname, value, SpatialAnalysisToolCalculationConditionSatisfactionMetric, [SpatialAnalysisToolCalculationConditionSatisfactionMetric, ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent])

agcls.AgClassCatalog.add_catalog_entry((5757987867521137107, 16897264315178726031), SpatialAnalysisToolCalculationConditionSatisfactionMetric)
agcls.AgTypeNameMap["SpatialAnalysisToolCalculationConditionSatisfactionMetric"] = SpatialAnalysisToolCalculationConditionSatisfactionMetric

class SpatialAnalysisToolCalculationPropagationDelayToLocation(ISpatialAnalysisToolSpatialCalculation, IAnalysisWorkbenchComponent, SupportsDeleteCallback):
    """A volume calc propagation delay to location interface."""

    _num_methods = 12
    _vtable_offset = IUnknown._vtable_offset + IUnknown._num_methods
    _get_distance_method_offset = 1
    _set_distance_method_offset = 2
    _get_reference_point_method_offset = 3
    _set_reference_point_method_offset = 4
    _get_reference_plane_method_offset = 5
    _set_reference_plane_method_offset = 6
    _get_along_vector_method_offset = 7
    _set_along_vector_method_offset = 8
    _get_speed_type_method_offset = 9
    _set_speed_type_method_offset = 10
    _get_speed_method_offset = 11
    _set_speed_method_offset = 12
    _metadata = {
        "iid_data" : (5218582178275522249, 11135030506945546897),
        "vtable_reference" : IUnknown._vtable_offset + IUnknown._num_methods - 1,
    }
    _property_names = {}
    def _get_property(self, attrname):
        return get_interface_property(attrname, SpatialAnalysisToolCalculationPropagationDelayToLocation)

    _get_distance_metadata = { "offset" : _get_distance_method_offset,
            "arg_types" : (POINTER(agcom.LONG),),
            "marshallers" : (agmarshall.EnumArg(DistanceToLocationType),) }
    @property
    def distance(self) -> "DistanceToLocationType":
        """The Volume Calc range distance types."""
        return self._intf.get_property(SpatialAnalysisToolCalculationPropagationDelayToLocation._metadata, SpatialAnalysisToolCalculationPropagationDelayToLocation._get_distance_metadata)

    _set_distance_metadata = { "offset" : _set_distance_method_offse